#! /usr/bin/env python

import os

from lab.environments import BaselSlurmEnvironment, LocalEnvironment

from downward.reports.scatter import ScatterPlotReport

import project


REPO = project.get_repo_base()
BENCHMARKS_DIR = "sas-tasks/"
if project.REMOTE:
    SUITE = ["rubiks-cube"]
    ENV = BaselSlurmEnvironment(email="clemens.buechner@unibas.ch",
                                partition="infai_2")
else:
    SUITE = [
        "rubiks-cube:s3-t1-p0.sas",
        "rubiks-cube:s3-t2-p0.sas",
        "rubiks-cube:s3-t3-p0.sas",
    ]
    ENV = LocalEnvironment(processes=2)

pattern1 = "[ 0,  1,  2,  3]"  # first half of corner cubies
pattern2 = "[ 4,  5,  6,  7]"  # second half of corner cubies
pattern3 = "[ 8,  9, 10, 11]"  # first third of edge cubies
pattern4 = "[12, 13, 14, 15]"  # second third of edge cubies
pattern5 = "[16, 17, 18, 19]"  # remaining edge cubies
korf_patterns = ("[[0, 1, 2, 3, 4, 5, 6, 7], [8, 9, 10, 11, 12, 13], " +
                 "[14, 15, 16, 17, 18, 19]]")
corner_edge_patterns = ("[[0, 1, 2, 3, 4, 5, 6, 7], " +
                        "[8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19]]")
abstractions = (f"[projections(manual_patterns([{pattern1}])," +
                " create_complete_transition_system=true)," +
                f" projections(manual_patterns([{pattern2}])," +
                " create_complete_transition_system=true)," +
                f" projections(manual_patterns([{pattern3}])," +
                " create_complete_transition_system=true)," +
                f" projections(manual_patterns([{pattern4}])," +
                " create_complete_transition_system=true)," +
                f" projections(manual_patterns([{pattern5}])," +
                " create_complete_transition_system=true)]")
corner_abstraction = (f"[projections(manual_patterns([{pattern1}])," +
                      " create_complete_transition_system=true)]")
edge_abstraction = (f"[projections(manual_patterns([{pattern3}])," +
                    " create_complete_transition_system=true)]")


CONFIGS = [
    ("blind", ["--search", "astar(blind())"]),
    ("max-manual-pdb", ["--search", f"astar(maximize({abstractions}))"]),
    ("max-systematic-pdb",
     ["--search", "astar(maximize([projections(systematic(3), " +
      "create_complete_transition_system=true)]))"]),
    ("cegar-original-900s",
     ["--search", "astar(cegar(subtasks=[original()], " +
      "max_transitions=infinity, max_time=900))"]),
    ("merge-and-shrink",
     ["--search",
      "astar(merge_and_shrink(shrink_strategy=shrink_bisimulation(greedy=false)," +
      "merge_strategy=merge_sccs(order_of_sccs=topological, " +
      "merge_selector=score_based_filtering(scoring_functions=[goal_relevance,dfp,total_order]))," +
      "label_reduction=exact(before_shrinking=true,before_merging=false)," +
      "max_states=50000,threshold_before_merge=1))"]),
]

BUILD_OPTIONS = []
DRIVER_OPTIONS = []
REVS = [
    ("cdc1df0dc", "rubiks-cube"),
]
ATTRIBUTES = [
    "error",
    "run_dir",
    "search_start_time",
    "search_start_memory",
    "total_time",
    "initial_h_value",
    "coverage",
    "cost",
    "expansions_until_last_jump",
    "memory",
    "cartesian_states",
    "cartesian_unreachable_states",
    "cartesian_unsolvable_states",
    "cartesian_transitions",
    "cartesian_loops_single",
    "cartesian_transitions_single",
    project.EVALUATIONS_PER_TIME,
]

exp = project.CommonExperiment(environment=ENV)
exp.add_parse_again_step()
for config_nick, config in CONFIGS:
    for rev, rev_nick in REVS:
        algo_name = f"{rev_nick}:{config_nick}" if rev_nick else config_nick
        exp.add_algorithm(
            algo_name,
            REPO,
            rev,
            config,
            build_options=BUILD_OPTIONS,
            driver_options=DRIVER_OPTIONS,
        )
exp.add_suite(BENCHMARKS_DIR, SUITE)

project.add_absolute_report(
    exp,
    attributes=ATTRIBUTES,
    filter=[project.add_evaluations_per_time])

attributes = [
    "expansions_until_last_jump",
    "total_time",
]

pairs = []
for c1 in CONFIGS:
    for c2 in CONFIGS:
        if c1 != c2:
            pairs.append((f"rubiks-cube:{c1[0]}", f"rubiks-cube:{c2[0]}"))

for algo1, algo2 in pairs:
    for attr in attributes:
        exp.add_report(ScatterPlotReport(
                relative=project.RELATIVE,
                get_category=None if project.TEX else lambda run1, run2: run1["domain"],
                attributes=[attr],
                filter_algorithm=[algo1, algo2],
                filter=[project.add_evaluations_per_time],
                format="tex" if project.TEX else "png",
            ),
            name=f'{exp.name}-{algo1}-vs-{algo2}-{attr}{"-relative" if project.RELATIVE else ""}')

exp.run_steps()
