#! /usr/bin/env python

import logging
import re

from lab.parser import Parser


class CommonParser(Parser):
    def add_repeated_pattern(self, name, regex, file="run.log", required=False, type=int):

        def find_all_occurences(content, props):
            matches = re.findall(regex, content)
            if required and not matches:
                logging.error(f"Pattern {regex} not found in file {file}")
            props[name] = [type(m) for m in matches]

        self.add_function(find_all_occurences, file=file)

    def add_bottom_up_pattern(self, name, regex, file="run.log", required=False, type=int):

        def search_from_bottom(content, props):
            reversed_content = "\n".join(reversed(content.splitlines()))
            match = re.search(regex, reversed_content)
            if required and not match:
                logging.error(f"Pattern {regex} not found in file {file}")
            if match:
                props[name] = type(match.group(1))

        self.add_function(search_from_bottom, file=file)


def no_search(content, props):
    if "search_start_time" not in props:
        error = props.get("error")
        if error is not None and error != "incomplete-search-found-no-plan":
            props["error"] = "no-search-due-to-" + error


def find_cegar_termination_criterion(content, props):
    outcomes = {
        "cegar_found_concrete_solution": "Found concrete solution during refinement.",
        "cegar_proved_unsolvability": "Abstract task is unsolvable.",
        "cegar_reached_states_limit": "Reached maximum number of states.",
        "cegar_reached_transitions_limit": "Reached maximum number of transitions.",
        "cegar_reached_time_limit": "Reached time limit.",
        "cegar_reached_memory_limit": "Reached memory limit.",
    }

    for outcome, text in outcomes.items():
        props[outcome] = int(text in content)
        if props[outcome]:
            if "cegar_outcome" in props:
                props["cegar_outcome"] = "mixed"
            else:
                props["cegar_outcome"] = outcome


def main():
    parser = CommonParser()
    parser.add_pattern("search_start_time", r"\[t=(.+)s, \d+ KB\] g=0, 1 evaluated, 0 expanded", type=float)
    parser.add_pattern("search_start_memory", r"\[t=.+s, (\d+) KB\] g=0, 1 evaluated, 0 expanded", type=int)
    parser.add_pattern("initial_h_value", r"f = (\d+) \[1 evaluated, 0 expanded, t=.+s, \d+ KB\]", type=int)

    parser.add_pattern("time_for_computing_cartesian_abstractions", r"Time for initializing additive Cartesian heuristic: (.+)s\n", type=float)
    parser.add_pattern("time_for_computing_cartesian_abstractions", r"Time for building Cartesian abstractions: (.+)s\n", type=float)
    parser.add_bottom_up_pattern("cartesian_states", r"Cartesian states: (\d+)\n", type=int)
    parser.add_bottom_up_pattern("cartesian_unreachable_states", r"Unreachable Cartesian states: (.+)\n", type=int)
    parser.add_bottom_up_pattern("cartesian_unsolvable_states", r"Unsolvable Cartesian states: (.+)\n", type=int)
    parser.add_bottom_up_pattern("cartesian_states_total", r"Total number of Cartesian states: (.+)\n", type=int)
    parser.add_pattern("cartesian_transitions", r"Total number of non-looping transitions: (.+)\n", type=int)
    parser.add_pattern("cartesian_transitions", r"Total number of transitions in Cartesian abstractions: (.+)\n", type=int)
    parser.add_pattern("cartesian_loops_single", r"Looping transitions: (.+)\n", type=int)
    parser.add_pattern("cartesian_transitions_single", r"Non-looping transitions: (.+)\n", type=int)

    parser.add_function(no_search)
    parser.add_function(find_cegar_termination_criterion)

    parser.parse()


if __name__ == "__main__":
    main()
