// __BEGIN_LICENSE__
//  Copyright (c) 2009-2013, United States Government as represented by the
//  Administrator of the National Aeronautics and Space Administration. All
//  rights reserved.
//
//  The NGT platform is licensed under the Apache License, Version 2.0 (the
//  "License"); you may not use this file except in compliance with the
//  License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
// __END_LICENSE__


/// \file wv_correct.cc
///

// Correct CCD artifacts in WorldView 1 and 2 images with given TDI.

// The problem: A WV image is obtained by mosaicking from left to
// right image blocks which are as tall is the entire image (each
// block comes from an individual CCD image sensor). Blocks are
// slightly misplaced in respect to each other by some unknown
// subpixel offsets. We use tabulated values for the offsets and their
// locations to undo them.

#include <vw/Image/Algorithms2.h>
#include <vw/Image/Interpolation.h>
#include <vw/Image/Filter.h>
#include <vw/FileIO/DiskImageUtils.h>
#include <asp/Core/Macros.h>
#include <asp/Core/Common.h>
#include <asp/Camera/RPC_XML.h>
#include <asp/Camera/RPCModel.h>
#include <xercesc/parsers/XercesDOMParser.hpp>
#include <xercesc/sax/HandlerBase.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <boost/dll.hpp>

namespace po = boost::program_options;
namespace fs = boost::filesystem;
using namespace vw;
using namespace asp;
using namespace vw::cartography;
using namespace xercesc;
using namespace std;


struct Options : vw::cartography::GdalWriteOptions {
  int band;
  std::string image_file, camera_file, output_image, output_type,
    dx, dy;
  bool print_per_column_corrections;
};

// See which table has the corrections for given MS data,
// find the right row in that table, and read the values
void parse_ms_correction_table(std::string const& sat_id, int band,
                               int tdi, std::string const& scan_dir,
                               std::vector<double> & corr_x,
                               std::vector<double> & corr_y) {

  // Clear the outputs
  corr_x.clear();
  corr_y.clear();

  std::string data_path = boost::dll::program_location().parent_path().parent_path().string()
    + "/share/wv_correct";

  std::string lookup_path = data_path + "/ms_correction_lookup.txt";

  std::ifstream handle;
  handle.open(lookup_path.c_str());
  if (handle.fail()) 
    vw_throw(vw::IOErr() << "Unable to open file \"" << lookup_path << "\"");

  std::string line;
  while (getline(handle, line, '\n')){

    if (line.size() == 0 || line[0] == '#')
      continue; // skip comment and empty line

    std::string l_sat_id, l_band, l_tdi, l_scan_dir, table_path, row_str;
    std::istringstream is(line);

    if (!(is >> l_sat_id >> l_band >> l_tdi >> l_scan_dir >> table_path >> row_str)) 
      continue;

    if (sat_id != l_sat_id || atoi(l_band.c_str()) != band || atoi(l_tdi.c_str()) != tdi ||
        scan_dir != l_scan_dir) {
      continue;
    }

    table_path = data_path + "/" + table_path;
    DiskImageView<float> lookup_table(table_path);
    int row = atoi(row_str.c_str());
    if (lookup_table.rows() <= row) 
      vw_throw(ArgumentErr() << "Could not find at least " << row + 1 << " rows in "
                << table_path << "\n");

    int num_vals = lookup_table.cols() / 2;
    if (2*num_vals != lookup_table.cols())
      vw_throw(ArgumentErr() << "Expecting an even number of columns in table "
                << table_path << "\n");
    
    corr_x.resize(num_vals);
    corr_y.resize(num_vals);
    
    for (int it = 0; it < num_vals; it++) {
      corr_x[it] = lookup_table(it,            row);
      corr_y[it] = lookup_table(it + num_vals, row);
    }
    
    break;
  }
  
}

void handle_arguments(int argc, char *argv[], Options& opt) {
  
  po::options_description general_options("");
  general_options.add_options()
    ("ot",  po::value(&opt.output_type)->default_value("Float32"), "Output data type. Supported types: Byte, UInt16, Int16, UInt32, Int32, Float32. If the output type is a kind of integer, values are rounded and then clamped to the limits of that type.")
    ("band",   po::value<int>(&opt.band)->default_value(0),
     "For multi-spectral images, specify the band to correct. Required unless --dx and --dy are set.")
    ("dx",  po::value(&opt.dx)->default_value(""), "For PAN or multi-spectral images, specify the plain text file having per-column corrections in the x direction, one per line, overriding the pre-computed table.")
    ("dy",  po::value(&opt.dy)->default_value(""), "As above, but for the y direction.")
    ("print-per-column-corrections", po::bool_switch(&opt.print_per_column_corrections)->default_value(false), "Print on standard output the per-column corrections about to apply (for multispectral images).");
    
  general_options.add(vw::cartography::GdalWriteOptionsDescription(opt));
  
  po::options_description positional("");
  positional.add_options()
    ("camera-image", po::value(&opt.image_file))
    ("camera-model", po::value(&opt.camera_file))
    ("output-image", po::value(&opt.output_image));
  
  po::positional_options_description positional_desc;
  positional_desc.add("camera-image",1);
  positional_desc.add("camera-model",1);
  positional_desc.add("output-image",1);
  
  std::string usage("[options] <camera-image> <camera-model> <output-image>");
  bool allow_unregistered = false;
  std::vector<std::string> unregistered;
  po::variables_map vm =
    asp::check_command_line(argc, argv, opt, general_options, general_options,
                             positional, positional_desc, usage,
                             allow_unregistered, unregistered);

  if (!vm.count("camera-image") || !vm.count("camera-model") ||
      !vm.count("output-image") )
    vw_throw(ArgumentErr() << "Requires <camera-image>, <camera-model> "
             << "and <output-image> in order to proceed.\n\n"
             << usage << general_options);
  
  vw::create_out_dir(opt.output_image);
}

void arr_to_vec(double arr[], int len, vector<double> & vec){
  vec.clear();
  for (int i = 0; i < len; i++) vec.push_back(arr[i]);
}

void get_offsets(int tdi, bool is_wv01, bool is_forward,
                 std::vector<double> & posx, std::vector<double> & ccdx,
                 std::vector<double> & posy, std::vector<double> & ccdy){

  posx.clear();
  ccdx.clear();
  posy.clear();
  ccdy.clear();
  
  // Here we tabulate all ccds offsets and their column pixel positions.
  
  if (!is_wv01){
    
    // Do WV02
    
    if (is_forward){

      // Forward scan direction
      
      if (tdi == 8){

        double posx_arr[] = {686,1389,2091,2796,3499,4203,4905,5608,6311,7018,7721,8425,9130,9833,10540,11246,11948,12653,13359,14060,14771,15476,16179,16879,17591,18296,18997,19704,20413,21119,21828,22528,23236,23938,24642,25347,26050,26757,27459,28165,28871,29571,30277,30980,31686,32388,33090,33795,34497};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {0.61,-0.48,0.514877822,-0.42,0.34,-0.410974812,0.42,-0.407028856,0.351330898,-0.269074758,0.446291377,-0.352892991,0.422367495,-0.265900548,0.510878629,-0.356820535,0.418892319,-0.39445358,0.412768582,-0.448113279,0.402712242,-0.35025355,0.242418592,-0.361696305,0.575444199,-0.3860937,0.226599683,-0.469802842,0.516676937,-0.505506755,0.365395904,-0.381782384,0.359518867,-0.391200765,0.60056376,-0.582558262,0.474938955,-0.494872156,0.508981417,-0.596565804,0.449049865,-0.522175341,0.475590295,-0.507063514,0.690495494,-0.548603608,0.578229037,-0.52393122,0.522313869};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] = {684,1387,2092,2794,3499,4200,4905,5609,6315,7017,7722,8425,9134,9836,10539,11245,11949,12655,13359,14064,14771,15474,16180,16886,17590,18296,19001,19707,20412,21117,21822,22527,23231,23937,24642,25346,26052,26757,27460,28164,28868,29573,30280,30982,31684,32390,33091,33847,34491};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {-0.200805296,0.163312385,-0.303400156,0.190147254,-0.198682571,0.120534513,-0.165163269,0.260918209,-0.194916111,0.297224582,-0.341403643,0.293191264,-0.308569359,0.228048922,-0.275457494,0.281969391,-0.342623543,0.33506075,-0.373224866,0.309935892,-0.317169892,0.311384632,-0.337586901,0.304615757,-0.357728907,0.309898182,-0.348613341,0.321742737,-0.32319268,0.31302694,-0.351948867,0.323089155,-0.305167385,0.295114824,-0.373661156,0.299014159,-0.23710711,0.26189909,-0.197283063,0.252105538,-0.182082587,0.269410094,-0.204998056,0.108369017,-0.13,0.15,-0.12,0.1,-0.135375447};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      } else if (tdi == 16){

        double posx_arr[] = {6.8408061504396665e+02,1.3900000000000000e+03,2.0900000000000000e+03,2.7950000000000000e+03,3.4970000000000000e+03,4.2020000000000000e+03,4.9040000000000000e+03,5.6080000000000000e+03,6.3110000000000000e+03,7.0170000000000000e+03,7.7210000000000000e+03,8.4250000000000000e+03,9.1310000000000000e+03,9.8350000000000000e+03,1.0540000000000000e+04,1.1247000000000000e+04,1.1949000000000000e+04,1.2652000000000000e+04,1.3360000000000000e+04,1.4064000000000000e+04,1.4769000000000000e+04,1.5473000000000000e+04,1.6181000000000000e+04,1.6884000000000000e+04,1.7590000000000000e+04,1.8296000000000000e+04,1.8999000000000000e+04,1.9705000000000000e+04,2.0410000000000000e+04,2.1116000000000000e+04,2.1822000000000000e+04,2.2528000000000000e+04,2.3232000000000000e+04,2.3936000000000000e+04,2.4641000000000000e+04,2.5346000000000000e+04,2.6051000000000000e+04,2.6755000000000000e+04,2.7460000000000000e+04,2.8164000000000000e+04,2.8869000000000000e+04,2.9573000000000000e+04,3.0277000000000000e+04,3.0979000000000000e+04,3.1684000000000000e+04,3.2387000000000000e+04,3.3092000000000000e+04,3.3795000000000000e+04,3.4498000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {2.5390684887464610e-01,-3.5534887400077469e-01,3.5868058677238895e-01,-2.6279478351582913e-01,3.3640332758089347e-01,-2.8099414838325443e-01,2.7610787186160823e-01,-2.9878722178904282e-01,2.9693940932888430e-01,-2.7079842371083113e-01,2.1755335800347692e-01,-2.1726904398138858e-01,2.1541342670419822e-01,-2.4604265224190178e-01,3.0929860826737643e-01,-1.7972099657116622e-01,2.3823069349951079e-01,-2.7729730756480198e-01,2.4415317254242136e-01,-2.1686339721048109e-01,2.5411074470725709e-01,-2.0571384826445988e-01,1.7800440971482070e-01,-2.5430628778962583e-01,4.2017888553467031e-01,-2.3486773623023255e-01,2.0955271241184922e-01,-2.8602396363998506e-01,3.7096035067307642e-01,-3.5297017540846010e-01,2.9983024563792071e-01,-2.6074063994628921e-01,2.5778353693466827e-01,-3.3136385583421091e-01,4.3976987720553201e-01,-4.2001754859752821e-01,3.4814040067367447e-01,-3.6874250907110984e-01,3.8041983620849318e-01,-3.7355204481815735e-01,3.8370800707015251e-01,-3.9642530983566426e-01,4.3131666926206813e-01,-4.0424239752634772e-01,4.5564270925189820e-01,-3.7277358779385250e-01,4.2647796915821801e-01,-4.5096831816509353e-01,4.1310425985455690e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.8408061504396665e+02,1.3900000000000000e+03,2.0900000000000000e+03,2.7950000000000000e+03,3.4970000000000000e+03,4.2020000000000000e+03,4.9040000000000000e+03,5.6080000000000000e+03,6.3110000000000000e+03,7.0170000000000000e+03,7.7210000000000000e+03,8.4250000000000000e+03,9.1310000000000000e+03,9.8350000000000000e+03,1.0540000000000000e+04,1.1247000000000000e+04,1.1949000000000000e+04,1.2652000000000000e+04,1.3360000000000000e+04,1.4064000000000000e+04,1.4769000000000000e+04,1.5473000000000000e+04,1.6181000000000000e+04,1.6884000000000000e+04,1.7590000000000000e+04,1.8296000000000000e+04,1.8999000000000000e+04,1.9705000000000000e+04,2.0410000000000000e+04,2.1116000000000000e+04,2.1822000000000000e+04,2.2528000000000000e+04,2.3232000000000000e+04,2.3936000000000000e+04,2.4641000000000000e+04,2.5346000000000000e+04,2.6051000000000000e+04,2.6755000000000000e+04,2.7460000000000000e+04,2.8164000000000000e+04,2.8869000000000000e+04,2.9573000000000000e+04,3.0277000000000000e+04,3.0979000000000000e+04,3.1680099466622523e+04,3.2387000000000000e+04,3.3104012536780050e+04,3.3757337357205266e+04,3.4502797729228914e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {-1.4234418776605290e-01,2.0019542891130560e-01,-9.3196388852414311e-02,1.7714720253179089e-01,-2.3677412189404023e-01,1.4049177941600352e-01,-1.6861234263992159e-01,1.3685256177186800e-01,-1.5420364173682916e-01,1.2573054439204184e-01,-1.4400807716044600e-01,1.6999871227638616e-01,-1.6074734680673120e-01,1.4612135414520888e-01,-2.1672541026902764e-01,1.4724229848437181e-01,-2.1245257639792839e-01,1.9820095871647861e-01,-2.0126392418829669e-01,1.8868895909998856e-01,-1.9320276320211682e-01,1.9147336195991457e-01,-2.2624648578705292e-01,2.2829898658168546e-01,-2.8937659996897863e-01,2.2356864832573883e-01,-2.1086667379933821e-01,2.3843004515655380e-01,-2.5730142894125391e-01,2.3721948745093918e-01,-2.2286322028088668e-01,1.9492167752978881e-01,-1.9502127190666094e-01,1.9297124691620351e-01,-2.6490728340707226e-01,2.1718335242684822e-01,-1.8953228514405335e-01,1.6616123278351685e-01,-1.4978268915551907e-01,1.6141450520704365e-01,-9.9302156372836348e-02,1.2756970494274872e-01,-9.9998093884728467e-02,8.8737727682353107e-02,-5.6616895743647838e-02,5.4919615855584025e-02,-4.3574116086978226e-02,6.7289510994713364e-02,-4.0313421172810865e-02};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);

      }else if (tdi == 32){ // WV02, TDI 32, forward scan direction
          
        double posx_arr[] = {687,1386,2098,2795,3497,4204,4903,5602,6307,7011,7721,8435,9130,9845,10539,11239,11948,12656,13355,14061,14772,15476,16181,16893,17589,18296,19001,19704,20411,21119,21818,22526,23224,23936,24645,25346,26048,26752,27459,28171,28870,29572,30275,30982,31686,32392,33089,33794,34494};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {0.23,-0.200870907,0.193869541,-0.252935963,0.156866694,-0.323477584,0.254999801,-0.164044311,0.137935927,-0.134198529,0.251196321,-0.127770636,0.192743363,-0.0817029972,0.279175392,-0.195373178,0.160454246,-0.187902874,0.185485025,-0.222631894,0.262762182,-0.183260502,0.153832087,-0.169862389,0.280372366,-0.178701578,0.165044079,-0.195968767,0.272389729,-0.198909335,0.22742834,-0.2793403,0.306560026,-0.284292223,0.376212463,-0.285305848,0.292147837,-0.414304838,0.342276557,-0.347423176,0.280371648,-0.36,0.32,-0.347601526,0.3,-0.21,0.19,-0.2,0.13};
        
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] = {599,1450,2786,3502,4206,4911,5633,6318,7018,7723,8426,9130,9835,10537,11245,11948,12656,13359,14066,14771,15470,16181,16895,17590,18296,19002,19706,20411,21118,21818,22525,23231,23937,24643,25346,26045,26756,27455,28165,28876,29534,30286,30946,31685,32381,33092,33794,34495};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {-0.179676025,0.15,0.0853180693,-0.164617487,0.0665333363,-0.0874214203,0.064026293,-0.0874982203,0.0785140517,-0.0802015105,0.0743760818,-0.138005942,0.110733611,-0.166057087,0.121906995,-0.140371862,0.179980823,-0.197258988,0.145376309,-0.128783256,0.163833408,-0.152976763,0.132341989,-0.153580946,0.105734243,-0.148236936,0.124184802,-0.19458627,0.102229409,-0.130355208,0.118469822,-0.125463552,0.146726178,-0.174620896,0.140067683,-0.128724518,0.152752943,-0.107566015,0.17,-0.0716320264,0.0552220226,-0.0432558069,0,0.0880390533,-0.102243106,0.11,-0.0928345189,0.19};
        
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 48){

        double posx_arr[] = {6.8500000000000000e+02,1.3870000000000000e+03,2.0900000000000000e+03,2.7833949228956494e+03,3.5000000000000000e+03,4.2153146077502679e+03,4.9070000000000000e+03,5.6090000000000000e+03,6.3110000000000000e+03,6.9768739999698882e+03,7.7074452677528570e+03,8.3649594087575297e+03,9.7850000000000000e+03,1.0541000000000000e+04,1.1246000000000000e+04,1.2613000000000000e+04,1.3367000000000000e+04,1.3991000000000000e+04,1.4777000000000000e+04,1.5481000000000000e+04,1.6180000000000000e+04,1.6885000000000000e+04,1.7589000000000000e+04,1.8296000000000000e+04,1.8999000000000000e+04,1.9704000000000000e+04,2.0411000000000000e+04,2.1115000000000000e+04,2.1820000000000000e+04,2.2526000000000000e+04,2.3232000000000000e+04,2.3936000000000000e+04,2.4641000000000000e+04,2.5347000000000000e+04,2.6051000000000000e+04,2.6756000000000000e+04,2.7460000000000000e+04,2.8164000000000000e+04,2.8868000000000000e+04,2.9573000000000000e+04,3.0276000000000000e+04,3.0979000000000000e+04,3.1684000000000000e+04,3.2389000000000000e+04,3.3091000000000000e+04,3.3796000000000000e+04,3.4497000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {1.0341629610890941e-01,-6.6928763297078031e-02,3.0155477757583198e-02,-3.3155368927938744e-02,7.1943250755293967e-02,-1.7703590512652495e-02,5.8348735237027369e-02,-4.4654367132889700e-02,4.2694728692775158e-02,5.5760215995853679e-04,-3.6565192260286372e-03,-8.4710496870385454e-04,2.0735001573248923e-02,5.9200091656747472e-02,3.3390829775972596e-02,1.5924222690679547e-02,1.2433486962539811e-02,-1.2337818612879543e-02,1.4265422619318227e-02,-2.4921764890428022e-02,3.5826987996452911e-02,-3.3723843893140447e-02,1.1804641267464354e-01,-4.6969602098551944e-02,5.4881044253378072e-02,-7.9167766164993283e-02,1.0218507666437618e-01,-9.9476998959238588e-02,7.8008080879757566e-02,-8.6823563217757277e-02,8.1529385384335268e-02,-1.0772228088639324e-01,2.0563126095007117e-01,-1.5357029259986685e-01,1.4227549664685765e-01,-1.1902502159597986e-01,1.5687627887139308e-01,-1.2839514357461734e-01,1.2210711747626858e-01,-1.4178815963954489e-01,1.3029067012278395e-01,-1.2834516060064524e-01,1.7226295441862252e-01,-1.4154131492681182e-01,1.2301976757233268e-01,-5.5304505685741294e-02,9.3226657293811480e-02};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={5.7700000000000000e+02,1.3560000000000000e+03,2.0930000000000000e+03,2.7930000000000000e+03,3.4990000000000000e+03,4.2020000000000000e+03,4.9030000000000000e+03,5.6070000000000000e+03,6.3130000000000000e+03,7.0170000000000000e+03,7.7190000000000000e+03,8.4270000000000000e+03,9.1310000000000000e+03,9.8350000000000000e+03,1.0520000000000000e+04,1.1242000000000000e+04,1.1884000000000000e+04,1.3358000000000000e+04,1.4770000000000000e+04,1.5475000000000000e+04,1.6181000000000000e+04,1.6888000000000000e+04,1.7590000000000000e+04,1.9001000000000000e+04,1.9708000000000000e+04,2.0411000000000000e+04,2.1120000000000000e+04,2.1822000000000000e+04,2.2528000000000000e+04,2.3234000000000000e+04,2.3937000000000000e+04,2.4641000000000000e+04,2.5350000000000000e+04,2.7395000000000000e+04,2.8871000000000000e+04,2.9570000000000000e+04,3.0277000000000000e+04,3.0981000000000000e+04,3.1683000000000000e+04,3.2388000000000000e+04,3.3092000000000000e+04,3.3794000000000000e+04,3.4497000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {1.0570213953611338e-02,2.0266889413527507e-02,-1.6386162678859423e-02,-2.9828468059933534e-02,-4.4952491140354663e-02,-4.9370914196828715e-02,2.7910792395426747e-02,-3.1050112108440567e-02,2.1360409733297950e-02,-2.7667853483971227e-02,1.1549886985667888e-02,-3.1661894928171418e-02,1.4036738189459722e-02,-2.2861716759744022e-02,1.0418560250149094e-02,-1.8484118071735597e-02,1.3002966471000757e-02,-2.7214589983848736e-02,-1.2675350294268143e-02,1.4253982497216024e-02,-2.8780593727833606e-02,-1.0814691554799964e-02,-6.0026795038763472e-02,-3.5757666784038064e-02,1.7968678091565892e-02,-4.1231799339921868e-02,1.4761135692591550e-02,-3.5891104296970983e-02,1.4837565362896002e-02,-3.4655284413776737e-02,2.5964357449015010e-02,-6.9484098821118209e-02,2.3409295762523191e-02,-1.1549734582821873e-02,2.8135367674027995e-02,-2.6530298882852830e-02,4.9870709188203688e-02,-5.9792391099903953e-02,1.5634482362369645e-01,-1.0730615249765883e-01,1.5157463886466002e-01,-1.5249469561113649e-01,1.9909194292256033e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);

      }else if (tdi == 64){

        double posx_arr[] = {5.8200000000000000e+02,1.3880000000000000e+03,2.0950000000000000e+03,2.7240000000000000e+03,3.4980000000000000e+03,4.2020000000000000e+03,4.9030000000000000e+03,5.6070000000000000e+03,6.3120000000000000e+03,7.0170000000000000e+03,7.7200000000000000e+03,8.4250000000000000e+03,9.1320000000000000e+03,9.8350000000000000e+03,1.0583443614712753e+04,1.1244000000000000e+04,1.1948000000000000e+04,1.2654000000000000e+04,1.3361000000000000e+04,1.4067000000000000e+04,1.4770000000000000e+04,1.5477000000000000e+04,1.6179000000000000e+04,1.6885000000000000e+04,1.7574494738325688e+04,1.8298000000000000e+04,1.8998000000000000e+04,1.9716000000000000e+04,2.0411000000000000e+04,2.1120000000000000e+04,2.1818089972892296e+04,2.2483000000000000e+04,2.3180000000000000e+04,2.3888000000000000e+04,2.4641000000000000e+04,2.6054000000000000e+04,2.6756000000000000e+04,2.7459000000000000e+04,2.8166000000000000e+04,2.8917951230724892e+04,2.9548000000000000e+04,3.0278077908470597e+04,3.1683000000000000e+04,3.2303000000000000e+04,3.3896014871274172e+04,3.4494470609482283e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-2.5956471759746354e-02,4.4578766394764630e-02,-8.9827031879944549e-02,5.5791806377529674e-02,-9.1842370052754557e-02,9.8046038365362242e-02,-9.4535764013415610e-02,9.6096236241295041e-02,-1.1027807070090942e-01,1.2103349723362071e-01,-1.2288374951142525e-01,1.1183981058653035e-01,-1.2351319972822833e-01,1.1857903334290790e-01,-3.8374302604157862e-02,1.0431820384662552e-01,-1.0837656303530220e-01,1.1648716601192505e-01,-9.9990725363775013e-02,1.0167088854860895e-01,-7.2622679909949028e-02,6.9660056393154771e-02,-5.7251999767388762e-02,8.5497410837256149e-02,-5.4978509301553680e-04,3.1398926776652236e-02,-3.3534823655966488e-02,3.4747869881864457e-02,-1.8199410864362924e-02,1.6133228098641383e-02,-6.6017078947983032e-03,1.5989338758809264e-02,1.4495383888490173e-02,2.1650921936725430e-02,9.1564466855612342e-02,2.8398021789331582e-02,-1.7661309009325159e-02,3.2708593858028739e-02,-1.6400978423544536e-02,5.5021377087672574e-03,2.4145354413039066e-02,9.6319560743016175e-04,2.2218246784993852e-02,-1.0658027785738437e-02,5.3917520123029417e-02,-9.4354588520648486e-02}; 
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={5.9200000000000000e+02,1.3170420560108732e+03,2.0930000000000000e+03,2.7950000000000000e+03,3.4603191398928402e+03,4.2020000000000000e+03,4.9060000000000000e+03,5.6100000000000000e+03,6.3130000000000000e+03,7.0150000000000000e+03,7.7210000000000000e+03,8.4250000000000000e+03,9.1280000000000000e+03,9.8350000000000000e+03,1.0535000000000000e+04,1.1245000000000000e+04,1.1946000000000000e+04,1.2656000000000000e+04,1.3354000000000000e+04,1.4064000000000000e+04,1.4772000000000000e+04,1.5472000000000000e+04,1.6178976571876621e+04,1.6886000000000000e+04,1.8296000000000000e+04,1.9703000000000000e+04,2.1109000000000000e+04,2.1816000000000000e+04,2.2524000000000000e+04,2.3941000000000000e+04,2.4498275779050047e+04,2.5345000000000000e+04,2.6050000000000000e+04,2.6757000000000000e+04,2.7463000000000000e+04,2.8164000000000000e+04,2.8871000000000000e+04,2.9573000000000000e+04,3.0277000000000000e+04,3.0980000000000000e+04,3.1684000000000000e+04,3.2389000000000000e+04,3.3091000000000000e+04,3.3795000000000000e+04,3.4499000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {2.0735639339753642e-02,-1.0545200904867574e-02,3.2849740270865541e-02,-4.6063848303733043e-02,1.1393361549998066e-02,-8.5227048213953785e-02,5.4846313905793621e-02,-7.0830913458622441e-02,5.8791095145657932e-02,-7.8803486624469218e-02,5.6202540000042336e-02,-8.0624083769682137e-02,7.2853151923653520e-02,-8.0857103112995279e-02,3.2073179099094629e-02,-8.2158082652905695e-02,3.6528864089062682e-02,-5.0344724044001858e-02,3.4218553836270697e-02,-6.7790889988649877e-02,3.8643824272193844e-02,-5.3116461931800592e-02,3.1664006294234437e-03,-4.6859464270032986e-02,-4.9473511205148687e-02,-4.3014298467437297e-02,-2.5493164396813435e-02,1.4764731428306827e-02,-2.4538664981630438e-02,-3.6412522449148846e-02,1.1393361549998066e-02,-3.2952895768327932e-02,3.0337303721746686e-02,-2.3391491162823966e-02,4.6542314256068334e-02,-5.4906963835417488e-02,3.6904469948493231e-02,-5.9839920680011705e-02,1.0706325427970893e-01,-1.1398690327458148e-01,2.0053486276127494e-01,-8.5213294686183508e-02,1.6964861601916248e-01,-1.5316083931449867e-01,1.6627374271198914e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);

      }
      
    }else{
      
      // Reverse scan direction for WV02
      
      if (tdi == 8){

        double posx_arr[] = {686,1390,2092,2796,3499,4203,4906,5611,6312,7019,7723,8428,9130,9836,10541,11247,11950,12655,13361,14064,14771,15478,16181,16882,17595,18298,19001,19707,20413,21118,21824,22526,23233,23939,24641,25349,26052,26759,27462,28165,28868,29576,30279,30980,31686,32389,33092,33797,34496};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-0.540629455,0.554006581,-0.704852099,0.357144541,-0.569714243,0.623382347,-0.363965624,0.505035172,-0.406574415,0.412216009,-0.588707536,0.46903767,-0.578869621,0.379630326,-0.481771777,0.542391759,-0.52123352,0.496852667,-0.568846495,0.557942156,-0.579475723,0.340925965,-0.278236397,0.38,-0.453814948,0.355614951,-0.270314932,0.404056203,-0.466882034,0.393031857,-0.335709262,0.224423693,-0.264509576,0.312079312,-0.265481695,0.402870354,-0.364294626,0.27,-0.46065139,0.410212171,-0.452024642,0.45,-0.562202119,0.563251952,-0.413924053,0.585742931,-0.601223106,0.440437984,-0.537681541};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] = {685,1393,2093,2795,3500,4201,4907,5610,6319,7018,7723,8427,9133,9835,10541,11246,11951,12654,13360,14065,14770,15476,16181,16884,17593,18297,19002,19708,20412,21117,21823,22528,23233,23937,24643,25348,26053,26757,27462,28166,28870,29574,30278,30982,31685,32389,33092,33795,34500};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {0.371317023,-0.425311806,0.421083454,-0.297024981,0.50172029,-0.618824514,0.361561999,-0.530163527,0.526409295,-0.558253425,0.525169894,-0.546126186,0.531177043,-0.558707992,0.487727402,-0.559893807,0.542065996,-0.564670797,0.529775034,-0.561113219,0.499900361,-0.513486006,0.57099431,-0.602553861,0.411532484,-0.52,0.514348735,-0.486603015,0.455814867,-0.488906582,0.44,-0.469926687,0.41,-0.479074967,0.380329556,-0.475231234,0.463364119,-0.441574787,0.429709868,-0.431819883,0.418603812,-0.427429724,0.463059469,-0.45427966,0.503635063,-0.401902633,0.415867899,-0.472805098,0.35};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
      
      }else if (tdi == 16){
        
        double posx_arr[] = {6.8500000000000000e+02,1.3890000000000000e+03,2.0920000000000000e+03,2.7940000000000000e+03,3.4970000000000000e+03,4.2010000000000000e+03,4.9040000000000000e+03,5.6090000000000000e+03,6.3130000000000000e+03,7.0170000000000000e+03,7.7200000000000000e+03,8.4250000000000000e+03,9.1300000000000000e+03,9.8340000000000000e+03,1.0540000000000000e+04,1.1244000000000000e+04,1.1949000000000000e+04,1.2653000000000000e+04,1.3358000000000000e+04,1.4064000000000000e+04,1.4770000000000000e+04,1.5475000000000000e+04,1.6179000000000000e+04,1.6884000000000000e+04,1.7591000000000000e+04,1.8298000000000000e+04,1.9001000000000000e+04,1.9705000000000000e+04,2.0410000000000000e+04,2.1118000000000000e+04,2.1819000000000000e+04,2.2526000000000000e+04,2.3232000000000000e+04,2.3936000000000000e+04,2.4640000000000000e+04,2.5346000000000000e+04,2.6051000000000000e+04,2.6756000000000000e+04,2.7459000000000000e+04,2.8164000000000000e+04,2.8869000000000000e+04,2.9572000000000000e+04,3.0276000000000000e+04,3.0981000000000000e+04,3.1684000000000000e+04,3.2387000000000000e+04,3.3091000000000000e+04,3.3794000000000000e+04,3.4499000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-3.4574537886232626e-01,3.9787575470308928e-01,-4.6062745420999512e-01,4.1955176689302121e-01,-3.5614274665743240e-01,3.6729158428131403e-01,-3.8279502434997981e-01,3.8759439974250887e-01,-3.7326604349382575e-01,3.7313704319410057e-01,-3.9898415187989034e-01,3.6354729632866983e-01,-4.4133561450997227e-01,3.5214448130969339e-01,-2.6272163619996458e-01,4.6827596342248690e-01,-4.1290438039039595e-01,3.7016763294562566e-01,-3.6930953429008129e-01,3.9268416731026762e-01,-3.3817431093249156e-01,2.5585431813799275e-01,-2.5643817416222225e-01,3.5080143260171215e-01,-2.5976588367099607e-01,2.4713269956144188e-01,-2.3592859495433446e-01,3.1520335038929820e-01,-2.8307849084683534e-01,2.4136258106225661e-01,-2.0850772101202214e-01,1.9512720700781175e-01,-2.3093516146406431e-01,2.1325824410642835e-01,-1.4171210365838610e-01,2.8824308471069221e-01,-2.6423225693096053e-01,2.7277241665840968e-01,-2.8048664133071755e-01,2.7473492239417063e-01,-3.1488995425124794e-01,3.0609214859229761e-01,-3.2952946007350564e-01,3.5987606178036530e-01,-2.9850936546380741e-01,3.8190937729923380e-01,-4.3392059335259014e-01,4.5578053058836221e-01,-4.7502083692902891e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.8544485229142265e+02,1.3890000000000000e+03,2.0900000000000000e+03,2.7940000000000000e+03,3.4960000000000000e+03,4.2020000000000000e+03,4.9050000000000000e+03,5.6080000000000000e+03,6.3120000000000000e+03,7.0170000000000000e+03,7.7220000000000000e+03,8.4260000000000000e+03,9.1300000000000000e+03,9.8350000000000000e+03,1.0539000000000000e+04,1.1245000000000000e+04,1.1949000000000000e+04,1.2654000000000000e+04,1.3359000000000000e+04,1.4063000000000000e+04,1.4769000000000000e+04,1.5474000000000000e+04,1.6179000000000000e+04,1.6885000000000000e+04,1.7590000000000000e+04,1.8295000000000000e+04,1.9000000000000000e+04,1.9706000000000000e+04,2.0410000000000000e+04,2.1116000000000000e+04,2.1821000000000000e+04,2.2526000000000000e+04,2.3232000000000000e+04,2.3936000000000000e+04,2.4642000000000000e+04,2.5347000000000000e+04,2.6053000000000000e+04,2.6755000000000000e+04,2.7460000000000000e+04,2.8163000000000000e+04,2.8869000000000000e+04,2.9573000000000000e+04,3.0278000000000000e+04,3.0980000000000000e+04,3.1685000000000000e+04,3.2387000000000000e+04,3.3090000000000000e+04,3.3794000000000000e+04,3.4499000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {1.9189392751473355e-01,-2.5614129853806417e-01,3.0251687519198145e-01,-2.9742905281500542e-01,2.3671530170053012e-01,-3.4195103110980063e-01,3.5047636063573390e-01,-3.3739605133187045e-01,3.7027086489076466e-01,-3.6044455719675084e-01,3.4698390037376997e-01,-3.5331234102904263e-01,3.8280024278642000e-01,-3.8942022189477016e-01,3.3714093789523197e-01,-4.5248161663109643e-01,3.8389515519454398e-01,-3.6857860900251949e-01,3.4077281393335995e-01,-3.8652428467339339e-01,3.9057754518403609e-01,-3.5429366047249111e-01,3.1780820554455869e-01,-3.5706957583120041e-01,3.0014031223542564e-01,-3.4309023336086203e-01,3.1749209942510243e-01,-3.3333939125132828e-01,2.7640180833171257e-01,-3.0957752851321163e-01,3.0612309509314800e-01,-3.2342055745103887e-01,3.2908296306796858e-01,-3.1985521355344892e-01,2.3474495447902263e-01,-2.8828951456743407e-01,2.7538596857288555e-01,-2.7423470759589069e-01,2.7389183018973773e-01,-2.2677078495215713e-01,2.8624912447649808e-01,-2.7582565144260940e-01,2.5757900445335125e-01,-2.7787936224331067e-01,3.5449683027495543e-01,-2.7473552591979988e-01,2.8519062432246800e-01,-2.8286009130175926e-01,3.2715512617647990e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);

      }else if (tdi == 32){ // WV02, TDI 32, reverse scan direction

        double posx_arr[] = {699,1397,2103,2793,3494,4200,4899,5603,6308,7011,7716,8440,9125,9848,10545,11256,11944,12647,13355,14059,14766,15487,16187,16887,17585,18307,19012,19700,20406,21131,21827,22539,23226,23934,24652,25340,26045,26768,27455,28160,28876,29577,30271,30989,31679,32382,33099,33793,34509};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-0.335309154,0.38905987,-0.393934819,0.389559906,-0.28075494,0.406487641,-0.309601194,0.253437631,-0.246713878,0.326511227,-0.376625986,0.338550253,-0.365883965,0.284085525,-0.272803025,0.338328635,-0.34147743,0.294963716,-0.331045579,0.360597886,-0.336537411,0.288312408,-0.245688743,0.364856611,-0.22505144,0.270050828,-0.164612022,0.217985028,-0.239825283,0.251399055,-0.209751979,0.178159777,-0.157302691,0.144229942,-0.117036312,0.264435954,-0.1853822,0.219754018,-0.208041899,0.279225317,-0.250305903,0.218536,-0.259987289,0.312066587,-0.217581913,0.247862225,-0.329537891,0.43173011,-0.244665742};
        
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] = {698,1380,2103,2792,3488,4195,4908,5619,6307,7018,7727,8421,9126,9848,10536,11256,11945,12649,13371,14068,14766,15473,16183,16888,17585,18306,19012,19716,20425,21131,21816,22539,23226,23935,24653,25341,26051,26767,27456,28159,28877,29577,30287,30989,31679,32382,33100,33789,34492};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {0.0733006458,-0.191287907,0.169869401,-0.267955801,0.140960055,-0.219522246,0.226332504,-0.247717521,0.214147217,-0.246025264,0.230311045,-0.25224611,0.222460444,-0.293111714,0.212547912,-0.265809622,0.240435936,-0.241445645,0.226134242,-0.258601247,0.180333976,-0.270246202,0.291984931,-0.311268718,0.12243298,-0.197870885,0.204639961,-0.197677915,0.161499083,-0.182979548,0.167472594,-0.191336091,0.157932791,-0.172758497,0.111660405,-0.151111619,0.173739603,-0.167923426,0.166282476,-0.140580905,0.156935996,-0.175825537,0.167537666,-0.120138715,0.252559378,-0.209173921,0.221983862,-0.171931521,0.281420508};
        
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
    }else if (tdi == 48){

        double posx_arr[] = {6.8500000000000000e+02,1.3890000000000000e+03,2.0910000000000000e+03,2.7930000000000000e+03,3.4970000000000000e+03,4.2010000000000000e+03,4.9050000000000000e+03,5.6080000000000000e+03,6.3130000000000000e+03,7.0160000000000000e+03,7.7210000000000000e+03,8.4260000000000000e+03,9.1300000000000000e+03,9.8340000000000000e+03,1.0539000000000000e+04,1.1244000000000000e+04,1.1949000000000000e+04,1.2654000000000000e+04,1.3359000000000000e+04,1.4064000000000000e+04,1.4769000000000000e+04,1.5475000000000000e+04,1.6180000000000000e+04,1.6885000000000000e+04,1.7589000000000000e+04,1.8295000000000000e+04,1.9001000000000000e+04,1.9706000000000000e+04,2.0411000000000000e+04,2.1117000000000000e+04,2.1820000000000000e+04,2.2527000000000000e+04,2.3232000000000000e+04,2.3936000000000000e+04,2.4643343208360497e+04,2.5345000000000000e+04,2.6053000000000000e+04,2.6756000000000000e+04,2.7461000000000000e+04,2.8164000000000000e+04,2.8868000000000000e+04,2.9573000000000000e+04,3.0276000000000000e+04,3.0981000000000000e+04,3.1681000000000000e+04,3.2387000000000000e+04,3.3092000000000000e+04,3.3794000000000000e+04,3.4496000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-1.4300903327522471e-01,2.0244675851091773e-01,-1.7349218693053359e-01,1.8441668247656365e-01,-1.6516872608831479e-01,2.0092489294579371e-01,-2.1374568239690162e-01,1.9302217649944564e-01,-1.9495975909431931e-01,2.0062676355539916e-01,-2.2852962586985318e-01,1.6918596435176431e-01,-2.5012980361716841e-01,1.6248836817833354e-01,-7.5417410227573797e-02,2.2721227942885155e-01,-2.1968709262038072e-01,2.1202713814221866e-01,-1.9925209546088707e-01,2.0865906272043860e-01,-1.5452351199499684e-01,1.0852612751691323e-01,-1.1781600600694060e-01,1.5421549725393000e-01,-5.8999133115210607e-02,9.7238843433336622e-02,-8.2315444423758141e-02,1.1486116105909630e-01,-7.2037554538535831e-02,8.1788260499833967e-02,-1.0345192232339379e-01,6.9105658972570649e-02,-1.1157433062684349e-01,8.6787583321675174e-02,-2.1530306295830109e-03,4.7213600110020820e-02,-7.4601222881290508e-02,7.7415377695255416e-02,-6.2153530739940316e-02,8.2001767463922959e-02,-1.4291810497966040e-01,1.1760993227419325e-01,-1.1830233978026980e-01,1.3852895797281584e-01,-5.5702429300845215e-02,1.5620163835022466e-01,-1.6079049958998287e-01,1.6156756005608655e-01,-1.6949649184840465e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.1900000000000000e+02,1.3880000000000000e+03,2.0910000000000000e+03,2.7940000000000000e+03,3.4980000000000000e+03,4.2010000000000000e+03,4.9040000000000000e+03,5.6090000000000000e+03,6.3120000000000000e+03,7.0170000000000000e+03,7.7200000000000000e+03,8.4250000000000000e+03,9.1300000000000000e+03,9.8340000000000000e+03,1.0540000000000000e+04,1.1244000000000000e+04,1.1950000000000000e+04,1.2653000000000000e+04,1.3359000000000000e+04,1.4065000000000000e+04,1.4770000000000000e+04,1.5476000000000000e+04,1.6181000000000000e+04,1.6886000000000000e+04,1.7591000000000000e+04,1.8294000000000000e+04,1.8998000000000000e+04,1.9705000000000000e+04,2.0409000000000000e+04,2.1117000000000000e+04,2.1821000000000000e+04,2.2526000000000000e+04,2.3231000000000000e+04,2.3936000000000000e+04,2.4639000000000000e+04,2.5345000000000000e+04,2.6051000000000000e+04,2.6754000000000000e+04,2.7460000000000000e+04,2.8165000000000000e+04,2.8868000000000000e+04,2.9574000000000000e+04,3.0279000000000000e+04,3.0982000000000000e+04,3.1684000000000000e+04,3.2386000000000000e+04,3.3092000000000000e+04,3.3796000000000000e+04,3.4498000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {1.3107372288166254e-02,-4.4774223393362671e-02,3.0076638024842117e-02,-8.6135161020642934e-02,3.0369194236788175e-02,-1.2818010979865727e-01,9.4688714202052965e-02,-1.0540202465113455e-01,1.1731023589284088e-01,-1.0675122704840387e-01,9.3172407761417331e-02,-1.1473527861399281e-01,9.9088061518403103e-02,-1.2304067434790292e-01,1.0756213297575871e-01,-1.3819201579833401e-01,9.2013249937297772e-02,-1.1514096785202740e-01,7.3097984221020360e-02,-1.0907100257198538e-01,1.0834244803526043e-01,-8.8411909107346920e-02,7.9168787031505469e-02,-1.1231197490097258e-01,5.2334907575884340e-02,-1.3078536132023377e-01,5.1879883070260188e-02,-1.0376250633410414e-01,4.1507445777280015e-02,-8.8539222248880847e-02,9.0152758809091224e-02,-9.2523410067961318e-02,5.3738510279886756e-02,-5.9296263353887768e-02,3.7139553575117376e-02,-5.4500145534886857e-02,5.4769082338792012e-02,-4.4249066821494085e-02,5.2929184630848433e-02,-8.4648068287386760e-02,6.2385656069781491e-02,-9.4806448475208582e-02,9.2817019204868553e-02,-1.1126504107212210e-01,1.7820375403635824e-01,-1.0663810718022593e-01,1.6995684162678765e-01,-1.2078158881427919e-01,1.7499221706103876e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 64){

        double posx_arr[] = {6.2699186227874088e+02,1.3461675875707542e+03,2.0930000000000000e+03,2.7950000000000000e+03,3.4950000000000000e+03,4.2030000000000000e+03,4.9040000000000000e+03,5.6100000000000000e+03,6.3120000000000000e+03,7.0140000000000000e+03,7.7210000000000000e+03,8.4240000000000000e+03,9.1300000000000000e+03,9.8340000000000000e+03,1.1245000000000000e+04,1.1950000000000000e+04,1.2654000000000000e+04,1.3359000000000000e+04,1.4067000000000000e+04,1.4770000000000000e+04,1.6188000000000000e+04,1.6885000000000000e+04,1.7548000000000000e+04,1.8257000000000000e+04,1.8992000000000000e+04,1.9680000000000000e+04,2.0344000000000000e+04,2.1062000000000000e+04,2.1831000000000000e+04,2.3235000000000000e+04,2.3842000000000000e+04,2.4644000000000000e+04,2.5348000000000000e+04,2.6030000000000000e+04,2.6766000000000000e+04,2.7460000000000000e+04,2.8166000000000000e+04,2.8825000000000000e+04,2.9535000000000000e+04,3.0237000000000000e+04,3.0912280738464615e+04,3.1687000000000000e+04,3.3056000000000000e+04,3.3788983639632665e+04,3.4519000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-1.2792609435887725e-02,4.0494240728929487e-02,-1.0273352691430539e-01,8.2229471277714114e-02,-5.4475529897062357e-02,7.2915037399568755e-02,-1.0103524435446153e-01,9.5778667499021697e-02,-9.5304881062583668e-02,1.0309000552364080e-01,-1.2828111750968413e-01,8.9488905322020015e-02,-1.3170504889741538e-01,9.4227250600038351e-02,1.1822042900765461e-01,-1.1334921334805237e-01,1.0119421868702914e-01,-6.4727334822871962e-02,8.0871507398435433e-02,-6.9872252666780432e-02,-6.0274378318823513e-02,3.9940351814570278e-02,-3.7475294553143516e-02,-3.6557268782466311e-02,-1.7750634661306372e-02,2.2263518250025437e-02,-2.0724443503512890e-02,-2.5508351624193539e-02,-2.1584991651836631e-02,-3.3704752318420475e-02,2.8146249683458420e-02,1.1343238652058059e-01,-9.3495349110702036e-02,2.6950548118281847e-02,-5.1586284783127122e-02,4.7866630814835179e-02,-4.3834106753798716e-02,3.5011613858667877e-02,-2.3395009628591333e-02,2.6682662730749450e-02,-9.8322288711756856e-03,9.5464566142484233e-02,-2.7065840532238712e-02,-9.8322288711756856e-03,-5.4894710994844831e-02};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.3500000000000000e+02,1.2820000000000000e+03,2.1007299424320090e+03,2.7720000000000000e+03,3.3920000000000000e+03,4.2030000000000000e+03,4.8730000000000000e+03,5.6070000000000000e+03,6.3110000000000000e+03,7.0160000000000000e+03,7.7190000000000000e+03,8.4190000000000000e+03,9.1310000000000000e+03,9.8360000000000000e+03,1.0536000000000000e+04,1.1245000000000000e+04,1.1949000000000000e+04,1.2656000000000000e+04,1.3358000000000000e+04,1.4063000000000000e+04,1.5471000000000000e+04,1.6148000000000000e+04,1.6876000000000000e+04,1.7590000000000000e+04,1.8213000000000000e+04,1.9007000000000000e+04,1.9700000000000000e+04,2.1816000000000000e+04,2.2540000000000000e+04,2.3235000000000000e+04,2.3936000000000000e+04,2.4639000000000000e+04,2.5346000000000000e+04,2.6054000000000000e+04,2.6756000000000000e+04,2.7459000000000000e+04,2.8162000000000000e+04,2.8867000000000000e+04,2.9572000000000000e+04,3.0277000000000000e+04,3.0980000000000000e+04,3.1683000000000000e+04,3.2385000000000000e+04,3.3092000000000000e+04,3.3795000000000000e+04,3.4498000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {4.7837093818003780e-02,-4.1131383205127936e-02,1.5291105530093801e-02,2.6406219479828064e-02,1.7038892246400386e-02,-8.3893616118013131e-02,-3.0798558429673883e-02,-7.1296297311020024e-02,7.0237957959390321e-02,-8.8849543754364088e-02,5.3731177621167692e-02,-6.2127535561531590e-02,6.7979180999753724e-02,-7.1084761864264878e-02,2.0915654083241642e-02,-1.0161360190451210e-01,6.8181658529722286e-02,-6.8670109770611248e-02,5.8230917196948596e-02,-9.0710391178991531e-02,-6.6201593855869656e-02,-1.8387554893765402e-02,-2.5298629591160960e-02,-4.9787072084544734e-02,-1.3793659780506999e-02,2.9405784476530958e-02,-1.9392190296183390e-02,-1.3920797534325183e-02,1.2248721915121240e-02,2.0476225781268360e-02,-5.2621159173484582e-02,-6.9400960299884795e-02,-2.6735049295340201e-02,2.4508151834901338e-02,-2.9419616554539875e-02,4.6921057682914893e-02,-2.4728786996870374e-02,5.8759661611343217e-02,-7.0148392770484819e-02,6.9844448140139487e-02,-7.6779042671121636e-02,1.5075498789185876e-01,-1.1342676258290449e-01,1.3176756444011312e-01,-1.2380488764138615e-01,2.1154165394809354e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }
      
    } // end reverse scan
    
  }else{

    // Do WV01
    
    if (is_forward){

      // Forward scan
      
      if (tdi == 8){

        double posx_arr[] = {6.9000000000000000e+02,1.3820000000000000e+03,2.0770000000000000e+03,2.7720000000000000e+03,3.4670000000000000e+03,4.1680000000000000e+03,4.8670000000000000e+03,5.5670000000000000e+03,6.2680000000000000e+03,6.9720000000000000e+03,7.6750000000000000e+03,8.3790000000000000e+03,9.0860000000000000e+03,9.7910000000000000e+03,1.0497000000000000e+04,1.1204000000000000e+04,1.1913000000000000e+04,1.2622000000000000e+04,1.3331000000000000e+04,1.4041000000000000e+04,1.4750000000000000e+04,1.5459000000000000e+04,1.6169000000000000e+04,1.6878000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9009000000000000e+04,1.9725000000000000e+04,2.0435000000000000e+04,2.1141000000000000e+04,2.1875000000000000e+04,2.2579000000000000e+04,2.3267000000000000e+04,2.3975000000000000e+04,2.4683000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6799000000000000e+04,2.7503000000000000e+04,2.8207000000000000e+04,2.8908000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1010000000000000e+04,3.1707000000000000e+04,3.2404000000000000e+04,3.3098000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {0,0,0,0,3.9572899122610738e-01,-0.4,1.3571345876690324e-01,-1.3599568525285227e-01,2.1848597840209738e-01,-1.9302532491443239e-01,1.5373607383106744e-01,-2.7948565111193063e-01,1.8470511522813510e-01,-8.8010620304032103e-02,4.1712154427482312e-01,-1.6133955851477760e-01,1.6782370523021301e-01,-1.9071066562455274e-01,9.4139947899578386e-02,-1.8390507569972631e-01,1.0458337107311372e-01,-1.3184212217192476e-01,1.3748731085111202e-01,-1.6997707746371654e-01,4.0533794015827662e-01,-1.8414617591004279e-01,1.5844241125556013e-01,-1.3291686600523619e-01,1.0204979703149097e-01,-2.0664117331352239e-01,1.8094775320573228e-01,-2.0731801920342768e-01,7.1547167910899323e-02,-1.3458151098218407e-01,1.4946741303774508e-01,-9.1924110765654321e-02,6.8305981753759237e-02,0,0,0,0,0,0,0,0,1.9511982760486032e-01,-2.6781463372201086e-01,2.2508286764347213e-01,-2.1325412224027540e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9000000000000000e+02,1.3820000000000000e+03,2.0770000000000000e+03,2.7720000000000000e+03,3.4670000000000000e+03,4.1680000000000000e+03,4.8670000000000000e+03,5.5670000000000000e+03,6.2680000000000000e+03,6.9720000000000000e+03,7.6750000000000000e+03,8.3790000000000000e+03,9.0860000000000000e+03,9.7910000000000000e+03,1.0497000000000000e+04,1.1204000000000000e+04,1.1913000000000000e+04,1.2622000000000000e+04,1.3331000000000000e+04,1.4041000000000000e+04,1.4750000000000000e+04,1.5459000000000000e+04,1.6169000000000000e+04,1.6878000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9009000000000000e+04,1.9725000000000000e+04,2.0435000000000000e+04,2.1141000000000000e+04,2.1875000000000000e+04,2.2579000000000000e+04,2.3267000000000000e+04,2.3975000000000000e+04,2.4683000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6799000000000000e+04,2.7503000000000000e+04,2.8207000000000000e+04,2.8908000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1010000000000000e+04,3.1707000000000000e+04,3.2404000000000000e+04,3.3098000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {0,0,0,0,2.7737184873656524e-01,-2.5199396985728040e-01,9.2651619425092591e-02,-1.2484085009254375e-01,9.1624371368929466e-02,-1.0163420477660505e-01,-8.5087919020240044e-02,-5.3278003593145037e-02,1.9474828198101091e-01,-5.5653320997682228e-02,1.4105791229907522e-01,-9.7756436532447491e-02,2.3515271743956914e-01,-2.4064621068353220e-01,1.9793293593058281e-01,-3.0417359707178432e-01,3.0974742641785702e-01,-2.8392516950893354e-01,4.3358586437052404e-01,-2.1286311036139788e-01,2.4607083739631108e-01,-2.8247520147602145e-01,3.5719728474354340e-01,-3.6362100039496115e-01,3.7045268711965218e-01,-3.3826985178248126e-01,4.2964452311943979e-01,-3.7778034342534478e-01,3.5082720318556498e-01,-4.6804694273970554e-01,3.2346002285296005e-01,-3.1529610028369914e-01,4.7169816490584404e-01,-4.7766860316648618e-01,4.1307927109971820e-01,-3.4825254544933831e-01,3.6562789909663967e-01,-4.8754456032180638e-01,3.9579963026130471e-01,-3.2165596442784983e-01,3.7812760195170941e-01,-3.8101306006307878e-01,4.9930246080447915e-01,-5.7700807932152287e-01,4.8434239761391673e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);

      }else if (tdi == 16){
        
        double posx_arr[] = {6.2900000000000000e+02,1.3570000000000000e+03,2.0190000000000000e+03,2.7690000000000000e+03,3.4720000000000000e+03,4.1790000000000000e+03,4.8670000000000000e+03,5.5680000000000000e+03,6.2730000000000000e+03,6.9730000000000000e+03,7.6250000000000000e+03,8.3790000000000000e+03,9.0910000000000000e+03,9.7920000000000000e+03,1.0499000000000000e+04,1.1204000000000000e+04,1.1911000000000000e+04,1.2620000000000000e+04,1.3332000000000000e+04,1.4040000000000000e+04,1.4750000000000000e+04,1.5459000000000000e+04,1.6170000000000000e+04,1.6879000000000000e+04,1.7591000000000000e+04,1.8300000000000000e+04,1.9008000000000000e+04,1.9720000000000000e+04,2.0426000000000000e+04,2.1141000000000000e+04,2.1848000000000000e+04,2.2559000000000000e+04,2.3271000000000000e+04,2.3977000000000000e+04,2.4682000000000000e+04,2.5419000000000000e+04,2.6095000000000000e+04,2.6803000000000000e+04,2.7446000000000000e+04,2.8302000000000000e+04,2.8908000000000000e+04,2.9608000000000000e+04,3.0313000000000000e+04,3.1009000000000000e+04,3.1705000000000000e+04,3.2406000000000000e+04,3.3100000000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {6.7322551801454480e-02,5.5941279131803634e-02,-6.3757124794674186e-02,-6.8350222655965132e-02,1.7691659140427704e-01,-6.5546000611674385e-02,6.3907733085781446e-02,-9.4966625926268172e-02,9.4161850926099605e-02,-1.0542259530055144e-01,-8.1558061694499709e-02,-6.3956300595019416e-02,4.1564490770397769e-02,-6.0005436499402910e-02,2.7687811096602116e-01,-6.7650615566507560e-02,8.9531829438824359e-02,-7.5787800987529685e-02,6.0918450645685199e-02,-4.9435252430723518e-02,7.8217062959282352e-02,-9.4972312086564092e-02,6.8771948144368461e-02,-5.5916563953937647e-02,2.5576397858317801e-01,-6.2953016311800947e-02,4.5986274236245760e-02,-5.2151312380337433e-02,3.5461672386768096e-02,-7.8840968976700190e-02,5.8279888571948685e-02,-5.6318493405969880e-02,-2.3130788594668883e-02,-2.5727892445735229e-02,6.7618551931374907e-02,-3.2635798235672329e-02,-6.7421171442540700e-02,8.6289010551310996e-02,-4.2940282592809562e-02,-6.7487297331515511e-02,-1.6954569447987194e-01,1.5768945188280239e-01,-1.3197137444641416e-01,2.2044168881490209e-01,-1.7686345973293444e-01,2.8571073037673689e-01,-2.5204743971722710e-01,2.9477986650443111e-01,-3.1357641656875035e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={5.5700000000000000e+02,1.3840000000000000e+03,2.0790000000000000e+03,2.7720000000000000e+03,4.9010000000000000e+03,5.6020000000000000e+03,6.8740000000000000e+03,7.6810000000000000e+03,8.3770000000000000e+03,9.0860000000000000e+03,9.7910000000000000e+03,1.0500000000000000e+04,1.1206000000000000e+04,1.1914000000000000e+04,1.2623000000000000e+04,1.3331000000000000e+04,1.4041000000000000e+04,1.4749000000000000e+04,1.5459000000000000e+04,1.6170000000000000e+04,1.6881000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9722000000000000e+04,2.0431000000000000e+04,2.1140000000000000e+04,2.1851000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3976000000000000e+04,2.4682000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6800000000000000e+04,2.7503000000000000e+04,2.8206000000000000e+04,2.8909000000000000e+04,2.9611000000000000e+04,3.0309000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2401000000000000e+04,3.3098000000000000e+04,3.3791000000000000e+04,3.4482000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {5.2874163366164850e-02,1.4813114779561809e-01,-1.3228522060801995e-01,5.1229840789382905e-02,4.1734055216619194e-02,-3.4922349728371403e-02,2.3489411511399848e-02,8.0756339158257071e-02,-4.1430846518602682e-02,1.3825891634375403e-01,-1.0281921942927558e-01,1.8102049531293241e-01,-1.4505660180914340e-01,1.8186344173432878e-01,-1.8620316464758269e-01,1.9090650984864999e-01,-2.8086787970093174e-01,2.7066109799278149e-01,-2.2564204861733023e-01,3.2153128507658019e-01,-2.5708859257733496e-01,2.6364737894970347e-01,-2.8693208660602365e-01,2.7970516529790845e-01,-3.5311023659542107e-01,3.4701280163120907e-01,-3.1032664634310730e-01,3.5125263768893017e-01,-3.3322268782288833e-01,3.6955614181826035e-01,-3.3683473663931718e-01,3.2182908581461295e-01,-2.8261547868377229e-01,3.4742873160768417e-01,-2.9435215444692314e-01,3.0697747027458971e-01,-2.9547216462701609e-01,3.2267414829365171e-01,-3.4291155301037418e-01,2.8631768612226638e-01,-2.6625338683543259e-01,3.4282658200878574e-01,-2.5735046181265991e-01,2.6016063755950836e-01,-2.7861330621270186e-01,2.4248759221995142e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);

      }else if (tdi == 32){
        
         double posx_arr[] = {6.9100000000000000e+02,1.3830000000000000e+03,2.0770000000000000e+03,2.7730000000000000e+03,3.4580000000000000e+03,4.1690000000000000e+03,4.8670000000000000e+03,5.5680000000000000e+03,6.2700000000000000e+03,6.9720000000000000e+03,7.6770000000000000e+03,8.3800000000000000e+03,9.0860000000000000e+03,9.7920000000000000e+03,1.0505000000000000e+04,1.1206000000000000e+04,1.1913000000000000e+04,1.2622000000000000e+04,1.3331000000000000e+04,1.4041000000000000e+04,1.4750000000000000e+04,1.5460000000000000e+04,1.6170000000000000e+04,1.6881000000000000e+04,1.7591000000000000e+04,1.8301000000000000e+04,1.9012000000000000e+04,1.9720000000000000e+04,2.0432000000000000e+04,2.1140000000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3266000000000000e+04,2.3974000000000000e+04,2.4682000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6799000000000000e+04,2.7503000000000000e+04,2.8207000000000000e+04,2.8910000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2402000000000000e+04,3.3099000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
         arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
         double ccdx_arr[] = {-1.5815654418563752e-01,1.3826350530962050e-01,-1.6016106928636276e-01,1.7292351786151797e-01,-8.4251284556116038e-02,1.3523902369157348e-01,-1.3545886295780737e-01,1.3984628576111927e-01,-8.1532674525840704e-02,9.4697680166625067e-02,-1.0592758885259967e-01,4.6579834981628057e-02,-8.2313118665729212e-02,1.2002910740140957e-01,1.5536100300284500e-01,9.5570826053090252e-02,-8.5935714547473396e-02,7.1701504496924306e-02,-4.8261551539294886e-02,1.0887016101633357e-01,-6.0376640338178925e-02,5.0426384764782284e-02,-7.4629150917886147e-02,7.3333337683582994e-02,1.3473731928967686e-01,5.9902243380629411e-02,-1.0637719830921741e-01,1.2901431398998836e-01,-6.7279387139093816e-02,8.6465589898355108e-02,-7.9828628987704930e-02,1.0291510265774768e-01,-1.0434449874621080e-01,1.2679075861184977e-01,-4.1769763304420532e-02,1.8653481213510253e-01,-2.0958544403438775e-01,1.9791028244010830e-01,-2.4696221386197473e-01,2.2895773820335324e-01,-1.7850291944712204e-01,1.8711950573480507e-01,-3.3754429964928234e-01,2.5821884129030215e-01,-2.0434256714964005e-01,4.1418566014699210e-01,-3.9931658359979100e-01,3.7510088116850077e-01,-4.1403543054361425e-01};
         arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
         double posy_arr[] ={6.9100000000000000e+02,1.3830000000000000e+03,2.0770000000000000e+03,2.7740000000000000e+03,3.4575000000000000e+03,4.1680000000000000e+03,4.8660000000000000e+03,5.5670000000000000e+03,6.2710000000000000e+03,6.9730000000000000e+03,7.4430000000000000e+03,8.4860000000000000e+03,9.1100000000000000e+03,9.7920000000000000e+03,1.0498000000000000e+04,1.1207000000000000e+04,1.1914000000000000e+04,1.2622000000000000e+04,1.3329000000000000e+04,1.4041000000000000e+04,1.4750000000000000e+04,1.5460000000000000e+04,1.6170000000000000e+04,1.6880000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9720000000000000e+04,2.0430000000000000e+04,2.1141000000000000e+04,2.1850000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3974000000000000e+04,2.4681000000000000e+04,2.5389000000000000e+04,2.6094000000000000e+04,2.6797000000000000e+04,2.7501000000000000e+04,2.8238000000000000e+04,2.8896000000000000e+04,2.9710000000000000e+04,3.0106000000000000e+04,3.1007000000000000e+04,3.1552000000000000e+04,3.2402000000000000e+04,3.3097000000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
         arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
         double ccdy_arr[] = {-2.5679384780785375e-01,2.1132524108212025e-01,-1.9890929041258171e-01,1.6743466645971244e-01,-3.0290900802832363e-02,1.2520084986428626e-01,-6.6310853481927404e-02,9.7787759335348293e-02,-4.7026742606632993e-02,-4.0719397722168357e-02,3.3620546875873569e-02,-3.0102129563891310e-02,5.1037911282746602e-02,-4.0388786002217299e-02,4.8061714461599923e-02,-5.5895642605816913e-02,7.1314129810702370e-02,-7.6604178475669069e-02,9.7449063613589945e-02,-1.0828765871885951e-01,1.1363625166513155e-01,-1.2841434371486909e-01,1.1913396821174335e-01,-1.6093971470699187e-01,1.6679734496194806e-01,-1.9561240122110690e-01,1.4561399603445646e-01,-1.8834366060830782e-01,1.4779996646048937e-01,-1.6694224556665363e-01,1.6887551340374002e-01,-1.6896325192069545e-01,1.7280986149540695e-01,-1.7481150806231555e-01,1.8974131057032312e-01,-1.6203523746941276e-01,1.2895698177790360e-01,-1.2814173545280660e-01,1.3614836986578066e-01,-1.5894266097567594e-01,1.4865193671756302e-01,-1.1977251781868413e-01,1.3054339999410874e-01,-9.6750535696703138e-02,1.6842720636109526e-01,-8.2351842567259040e-02,-5.9741317509331537e-02,-7.4004189893254774e-02,-6.0371436981168827e-02};
         arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 48){
        
        double posx_arr[] = {6.9100000000000000e+02,1.3830000000000000e+03,2.0770000000000000e+03,2.7730000000000000e+03,3.4580000000000000e+03,4.1690000000000000e+03,4.8670000000000000e+03,5.5680000000000000e+03,6.2700000000000000e+03,6.9720000000000000e+03,7.6770000000000000e+03,8.3800000000000000e+03,9.0860000000000000e+03,9.7920000000000000e+03,1.0505000000000000e+04,1.1206000000000000e+04,1.1913000000000000e+04,1.2622000000000000e+04,1.3331000000000000e+04,1.4041000000000000e+04,1.4750000000000000e+04,1.5460000000000000e+04,1.6170000000000000e+04,1.6881000000000000e+04,1.7591000000000000e+04,1.8301000000000000e+04,1.9012000000000000e+04,1.9720000000000000e+04,2.0432000000000000e+04,2.1140000000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3266000000000000e+04,2.3974000000000000e+04,2.4682000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6799000000000000e+04,2.7503000000000000e+04,2.8207000000000000e+04,2.8910000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2402000000000000e+04,3.3099000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-3.2946263385876073e-01,2.6822672303060707e-01,-2.7219800277644979e-01,3.2114706103684659e-01,-1.1264051193844726e-01,2.1650365243094197e-01,-2.9850534257863964e-01,2.4690262924579376e-01,-3.0264558267838221e-01,1.9002513807042659e-01,-2.2269004121580999e-01,1.9360526493221386e-01,-2.4333024293404595e-01,2.3253426572925068e-01,2.5156939346654091e-02,2.3240995046557339e-01,-2.2672903386601889e-01,1.5590477088799856e-01,-1.5002686187069883e-01,2.0344831295675106e-01,-1.5000864041927442e-01,1.8004430049333014e-01,-1.8236695292080785e-01,1.8325510525277885e-01,3.3572343456822654e-02,1.6768958115164090e-01,-1.7656399543377982e-01,1.9586089774158649e-01,-1.7984311406510889e-01,1.9926780777821917e-01,-2.0358163540196328e-01,1.8788400881417389e-01,-2.6839931133109096e-01,2.4275915958519964e-01,-1.9419370826317078e-01,2.3213640634391663e-01,-2.0322722705876156e-01,2.5395023204178102e-01,-3.1352589198440872e-01,2.8813130768645712e-01,-3.2571760506100766e-01,3.7344440543148194e-01,-3.2440667961683478e-01,4.4966322933102876e-01,-3.5284854392228138e-01,4.5498566236612847e-01,-4.6890882370639664e-01,5.3253439420167337e-01,-5.1690187379933494e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9100000000000000e+02,1.3830000000000000e+03,2.0770000000000000e+03,2.7740000000000000e+03,3.4575000000000000e+03,4.1680000000000000e+03,4.8660000000000000e+03,5.5670000000000000e+03,6.2710000000000000e+03,6.9730000000000000e+03,7.4430000000000000e+03,8.4860000000000000e+03,9.1100000000000000e+03,9.7920000000000000e+03,1.0498000000000000e+04,1.1207000000000000e+04,1.1914000000000000e+04,1.2622000000000000e+04,1.3329000000000000e+04,1.4041000000000000e+04,1.4750000000000000e+04,1.5460000000000000e+04,1.6170000000000000e+04,1.6880000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9720000000000000e+04,2.0430000000000000e+04,2.1141000000000000e+04,2.1850000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3974000000000000e+04,2.4681000000000000e+04,2.5389000000000000e+04,2.6094000000000000e+04,2.6797000000000000e+04,2.7501000000000000e+04,2.8238000000000000e+04,2.8896000000000000e+04,2.9710000000000000e+04,3.0106000000000000e+04,3.1007000000000000e+04,3.1552000000000000e+04,3.2402000000000000e+04,3.3097000000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {-1.9855267108725011e-01,1.7955082108595999e-01,-1.8827745225040018e-01,1.3932648815497944e-01,-6.0352542704901455e-02,8.3598970550751980e-02,-8.1175104051077959e-02,6.1423375459795058e-02,-4.5004799966324879e-02,5.1255915991506043e-02,1.5851812214634477e-02,1.5587369184767868e-02,1.2194157289131528e-02,-3.3985901939564599e-02,7.0922061628214295e-02,-5.3223665244117681e-02,3.8875501918710656e-02,-5.5471377905873670e-02,5.3105759125309210e-02,-9.9237321842864795e-02,6.2515982392321895e-02,-8.4411396949582967e-02,8.7660527700112317e-02,-9.7615657244219792e-02,1.1887807394615302e-01,-1.0116151698982925e-01,9.2884382992062239e-02,-9.0252420387025509e-02,8.8929836687679620e-02,-1.0833431823799911e-01,8.0427361987862905e-02,-8.8225813952202234e-02,6.7822547934654032e-02,-8.6349198150212400e-02,8.7246760279810182e-02,-7.0883233135712959e-02,4.1376134151766455e-02,-3.5761138416796644e-02,3.7183085174551947e-02,-1.8063316353517003e-02,2.0949603842599852e-02,-2.4268409874204040e-02,-2.2141072979198307e-02,3.0889019160442014e-02,3.4287549363534349e-02,7.0188151962212772e-02,-6.9597035784645517e-02,9.1561840117775795e-02,-1.4346661758051732e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 56){
        
        double posx_arr[] = {6.9050000000000000e+02,1.3830000000000000e+03,2.0775000000000000e+03,2.7725000000000000e+03,3.4635000000000000e+03,4.1680000000000000e+03,4.8670000000000000e+03,5.5680000000000000e+03,6.2695000000000000e+03,6.9720000000000000e+03,7.6765000000000000e+03,8.3800000000000000e+03,9.0845000000000000e+03,9.7915000000000000e+03,1.0502500000000000e+04,1.1205500000000000e+04,1.1913500000000000e+04,1.2621500000000000e+04,1.3330500000000000e+04,1.4040500000000000e+04,1.4749500000000000e+04,1.5460000000000000e+04,1.6169500000000000e+04,1.6880500000000000e+04,1.7590500000000000e+04,1.8301000000000000e+04,1.9011500000000000e+04,1.9720500000000000e+04,2.0431500000000000e+04,2.1140000000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3266500000000000e+04,2.3974000000000000e+04,2.4682000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6798500000000000e+04,2.7502500000000000e+04,2.8207000000000000e+04,2.8910000000000000e+04,2.9610500000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2402000000000000e+04,3.3098000000000000e+04,3.3792000000000000e+04,3.4483500000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-3.5011442399307202e-01,3.1186407063727184e-01,-3.2335982284254117e-01,3.5688803627847732e-01,-1.5016902336398030e-01,2.7091922018898873e-01,-3.4524579685389989e-01,3.1697476178278094e-01,-3.3427642010528191e-01,2.4348731667372364e-01,-2.5459010039954394e-01,2.2227864965341840e-01,-2.4963564133255550e-01,2.6519143229617087e-01,-1.2640033051156893e-02,2.6926418565530419e-01,-2.6413094069459742e-01,1.7933663392775823e-01,-1.7857720325097262e-01,2.3227298026241555e-01,-1.8833606658120045e-01,2.0690430915056696e-01,-2.2153185513515222e-01,2.2470157361209891e-01,-1.0775858639518222e-02,1.9409607914016191e-01,-1.9017563481638641e-01,2.0941570156418010e-01,-1.9959115270537459e-01,2.3450585103367219e-01,-2.4346167898401047e-01,2.3507302665020746e-01,-2.9839629178583321e-01,2.8796840809065738e-01,-2.2339992370188613e-01,2.6676508093280049e-01,-2.2822730588765003e-01,2.7316283112057416e-01,-3.4718648441520028e-01,3.2886139038062245e-01,-3.1213617802392812e-01,4.1015850839081253e-01,-3.6843147363566531e-01,4.4417245109948189e-01,-3.4664171603617239e-01,4.8513123485069182e-01,-4.9177442050887177e-01,5.5564212715312777e-01,-5.1330089584066618e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9100000000000000e+02,1.3835000000000000e+03,2.0765000000000000e+03,2.7740000000000000e+03,3.3962500000000000e+03,4.1665000000000000e+03,4.8670000000000000e+03,5.5665000000000000e+03,6.2720000000000000e+03,6.9150000000000000e+03,7.4840000000000000e+03,8.3960000000000000e+03,9.0795000000000000e+03,9.8765000000000000e+03,1.0498500000000000e+04,1.1205500000000000e+04,1.1913500000000000e+04,1.2623000000000000e+04,1.3330500000000000e+04,1.4041500000000000e+04,1.4750000000000000e+04,1.5460000000000000e+04,1.6170500000000000e+04,1.6880500000000000e+04,1.7590500000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9722000000000000e+04,2.0428500000000000e+04,2.1140500000000000e+04,2.1901000000000000e+04,2.2493500000000000e+04,2.3267000000000000e+04,2.3976000000000000e+04,2.4678500000000000e+04,2.5387500000000000e+04,2.6094500000000000e+04,2.6797000000000000e+04,2.7501000000000000e+04,2.8222500000000000e+04,2.8901500000000000e+04,2.9660500000000000e+04,3.0208000000000000e+04,3.1008000000000000e+04,3.1629500000000000e+04,3.2402500000000000e+04,3.3098500000000000e+04,3.3792000000000000e+04,3.4483500000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {-1.6583445664923085e-01,1.5749516507363931e-01,-1.6399364833748065e-01,1.3444012169169411e-01,-1.4218531165367934e-02,8.7605123753737107e-02,-7.6127025966082007e-02,6.4299120414989461e-02,-4.3959138934048909e-02,1.0176549144186326e-02,2.0660621136971577e-02,1.5891891134220865e-02,-6.6450679959886225e-03,-2.5258926562893420e-02,5.0562514191529004e-02,-3.9414471676108662e-02,2.9733409728897639e-02,-4.6981472589652754e-02,4.8151836469005942e-02,-7.6412104777085432e-02,5.6004981001094828e-02,-6.9293794016547533e-02,6.0390792185470138e-02,-7.1416824888125160e-02,8.7665159376180662e-02,-7.9976193812377444e-02,5.8036691718802436e-02,-5.9601458599666465e-02,5.6422514716840351e-02,-6.5824108186222058e-02,3.3418567527584750e-02,-3.6893476936227472e-02,1.7823520772154627e-02,-5.1921130252393144e-02,3.3106020550451989e-02,-1.4647343638039951e-02,-5.5192217889821343e-03,8.3853995859731918e-03,-1.4399925983834681e-02,3.3745059837696025e-02,-5.4753302855168780e-02,4.9745691197937855e-02,-9.3580681077980274e-02,1.1683790680589558e-01,-3.0054266039368501e-02,1.5181256524274175e-01,-1.4862946648500802e-01,1.7373902256820783e-01,-1.9462470532068496e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 64){
        
        double posx_arr[] = {6.9000000000000000e+02,1.3830000000000000e+03,2.0780000000000000e+03,2.7720000000000000e+03,3.4690000000000000e+03,4.1670000000000000e+03,4.8670000000000000e+03,5.5680000000000000e+03,6.2690000000000000e+03,6.9720000000000000e+03,7.6760000000000000e+03,8.3800000000000000e+03,9.0830000000000000e+03,9.7910000000000000e+03,1.0500000000000000e+04,1.1205000000000000e+04,1.1914000000000000e+04,1.2621000000000000e+04,1.3330000000000000e+04,1.4040000000000000e+04,1.4749000000000000e+04,1.5460000000000000e+04,1.6169000000000000e+04,1.6880000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9721000000000000e+04,2.0431000000000000e+04,2.1140000000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3974000000000000e+04,2.4682000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6798000000000000e+04,2.7502000000000000e+04,2.8207000000000000e+04,2.8910000000000000e+04,2.9611000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2402000000000000e+04,3.3097000000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-3.7076621412738331e-01,3.5550141824393666e-01,-3.7452164290863260e-01,3.9262901152010804e-01,-1.8769753478951337e-01,3.2533478794703546e-01,-3.9198625112916013e-01,3.8704689431976808e-01,-3.6590725753218167e-01,2.9694949527702069e-01,-2.8649015958327789e-01,2.5095203437462293e-01,-2.5594103973106508e-01,2.9784859886309106e-01,-5.0437005448967877e-02,3.0611842084503493e-01,-3.0153284752317600e-01,2.0276849696751786e-01,-2.0712754463124641e-01,2.6109764756808007e-01,-2.2666349274312650e-01,2.3376431780780377e-01,-2.6069675734949660e-01,2.6614804197141900e-01,-5.5124060735859098e-02,2.2050257712868293e-01,-2.0378727419899301e-01,2.2297050538677374e-01,-2.1933919134564028e-01,2.6974389428912521e-01,-2.8334172256605766e-01,2.8226204448624104e-01,-3.2839327224057546e-01,3.3317765659611515e-01,-2.5260613914060148e-01,3.0139375552168429e-01,-2.5322738471653850e-01,2.9237543019936735e-01,-3.8084707684599189e-01,3.6959147307478779e-01,-2.9855475098684858e-01,4.4687261135014311e-01,-4.1245626765449583e-01,4.3868167286793502e-01,-3.4043488815006340e-01,5.1527680733525516e-01,-5.1464001731134690e-01,5.7874986010458207e-01,-5.0969991788199742e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9100000000000000e+02,1.3840000000000000e+03,2.0760000000000000e+03,2.7740000000000000e+03,3.3350000000000000e+03,4.1650000000000000e+03,4.8680000000000000e+03,5.5660000000000000e+03,6.2730000000000000e+03,6.8570000000000000e+03,7.5250000000000000e+03,8.3060000000000000e+03,9.0490000000000000e+03,9.9610000000000000e+03,1.0499000000000000e+04,1.1204000000000000e+04,1.1913000000000000e+04,1.2624000000000000e+04,1.3332000000000000e+04,1.4042000000000000e+04,1.4750000000000000e+04,1.5460000000000000e+04,1.6171000000000000e+04,1.6881000000000000e+04,1.7591000000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9724000000000000e+04,2.0427000000000000e+04,2.1140000000000000e+04,2.1952000000000000e+04,2.2429000000000000e+04,2.3267000000000000e+04,2.3978000000000000e+04,2.4676000000000000e+04,2.5386000000000000e+04,2.6095000000000000e+04,2.6797000000000000e+04,2.7501000000000000e+04,2.8207000000000000e+04,2.8907000000000000e+04,2.9611000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2403000000000000e+04,3.3100000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {-1.3311624221121157e-01,1.3543950906131863e-01,-1.3970984442456114e-01,1.2955375522840876e-01,3.1915480374165583e-02,9.1611276956722235e-02,-7.1078947881086069e-02,6.7174865370183856e-02,-4.2913477901772940e-02,-3.0902817703133391e-02,2.5469430059308677e-02,1.6196413083673863e-02,-2.5484293281108773e-02,-1.6531951186222246e-02,3.0202966754843706e-02,-2.5605278108099649e-02,2.0591317539084622e-02,-3.8491567273431844e-02,4.3197913812702673e-02,-5.3586887711306076e-02,4.9493979609867761e-02,-5.4176191083512099e-02,3.3121056670827966e-02,-4.5217992532030522e-02,5.6452244806208293e-02,-5.8790870634925628e-02,2.3189000445542639e-02,-2.8950496812307429e-02,2.3915192746001082e-02,-2.3313898134445021e-02,-1.3590226932693400e-02,1.4438860079747286e-02,-3.2175506390344777e-02,-1.7493062354573888e-02,-2.1034719178906199e-02,4.1588545859633057e-02,-5.2414577729730724e-02,5.2531937588743027e-02,-6.5982937142221310e-02,8.5553436028909052e-02,-1.3045620955293741e-01,1.2375979227007976e-01,-1.6502028917676223e-01,2.0278679445134914e-01,-9.4396081442271351e-02,2.3343697852327072e-01,-2.2766189718537050e-01,2.5591620501863988e-01,-2.4578279306085260e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }
      
    }else{

      // Reverse scan for WV01

      if (tdi == 8){

        double posx_arr[] = {690,1384,2077,2773,3485,4167,4867,5567,6267,6971,7676,8381,9085,9790,10497,11199,11912,12622,13333,14039,14750,15461,16173,16836,17592,18302,19004,19714,20430,21093,21847,22560,23267,23976,24680,25390,26095,26805,27502,28207,28911,29611,30308,31012,31708,32404,33099,33792,34483};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-0.471342233,0.406662467,-0.3,0.25,-0.18,0.414554245,-0.329970496,0.300464941,-0.245331098,0.326201617,-0.268883885,0.247085176,-0.260431946,0.253758738,-0.07,0.104311742,-0.0972457403,0.236106458,-0.230207262,0.119667833,-0.181168112,0.165534201,-0.13,-0.1,0.146001801,0.187365551,-0.0809334415,0.0915021807,-0.197808712,-0.15,-0.219994213,0.174653647,-0.131106811,0.194498965,0.1,0.172245035,-0.174624177,0.175056576,-0.149723933,0.293603786,-0.31,0.227724287,-0.247815188,0.278231209,-0.15,0.290767364,-0.371329954,0.36,-0.34};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] = {687,1384,2083,2769,3470,4168,4864,5569,6268,6971,7673,8384,9085,9792,10499,11207,11912,12623,13327,14040,14749,15460,16170,16890,17594,18303,19013,19723,20436,21141,21836,22583,23261,23977,24681,25415,26095,26804,27510,28209,28878,29609,30310,31010,31707,32404,33101,33793,34485};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {0.33021674,-0.510332286,0.322879066,-0.506351785,0.532988964,-0.468325768,0.416633933,-0.291694647,0.319217952,-0.455454692,0.398153166,-0.239204978,0.383470053,-0.278574718,0.403614749,-0.34,0.346153912,-0.195547767,0.245805935,-0.36,0.234872778,-0.207773434,0.34,-0.0736518249,0.142476915,-0.131084283,0.210383644,-0.21,0.205289073,-0.17,0.19,-0.0540550958,0.127,0.13,0.185869231,-0.05,-0.226153494,0.1130555,-0.119997255,0.177305054,0.05,0.3,-0.348494401,0.223737041,-0.331650795,0.538082299,-0.471742139,0.590081627,-0.683604722};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
      
      
      } else if (tdi == 16){
        
        double posx_arr[] = {6.9000000000000000e+02,1.3820000000000000e+03,2.0760000000000000e+03,2.7720000000000000e+03,3.4650000000000000e+03,4.1670000000000000e+03,4.8670000000000000e+03,5.5680000000000000e+03,6.2680000000000000e+03,6.9710000000000000e+03,7.6770000000000000e+03,8.3780000000000000e+03,9.0850000000000000e+03,9.7910000000000000e+03,1.0503000000000000e+04,1.1206000000000000e+04,1.1913000000000000e+04,1.2622000000000000e+04,1.3330000000000000e+04,1.4040000000000000e+04,1.4749000000000000e+04,1.5459000000000000e+04,1.6170000000000000e+04,1.6881000000000000e+04,1.7591000000000000e+04,1.8299000000000000e+04,1.9008000000000000e+04,1.9721000000000000e+04,2.0429000000000000e+04,2.1140000000000000e+04,2.1849000000000000e+04,2.2557000000000000e+04,2.3265000000000000e+04,2.3974000000000000e+04,2.4682000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6798000000000000e+04,2.7503000000000000e+04,2.8206000000000000e+04,2.8909000000000000e+04,2.9611000000000000e+04,3.0310000000000000e+04,3.1010000000000000e+04,3.1707000000000000e+04,3.2404000000000000e+04,3.3098000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-3.6563699517936904e-01,3.2374257439446741e-01,-3.0665689416227304e-01,2.8779817609507674e-01,-1.4391974546538006e-01,2.5213647269676465e-01,-2.7431622419889917e-01,2.2048856906422215e-01,-2.6590706716919182e-01,1.9957802948970854e-01,-2.7095596052316773e-01,2.3435367631109566e-01,-2.5315591932358317e-01,1.9856962666552852e-01,-3.5920061119356986e-02,1.5980621898278752e-01,-1.6159921442794922e-01,1.9744720254044340e-01,-1.9378974298741097e-01,1.3105906502192419e-01,-1.5230954549604461e-01,1.7447700273627750e-01,-1.4295214614100249e-01,1.6062348332303306e-01,4.8050851400649242e-02,1.3750807699320705e-01,-7.3113271749859196e-02,7.1346049905560857e-02,-9.9108950389247746e-02,1.9229689329792210e-01,-1.7243313894497606e-01,1.7966640657714811e-01,-1.5356877017491991e-01,1.7351553213620843e-01,-1.0112703267223394e-01,1.6278473565663124e-01,-1.2930576980595232e-01,1.4836628451840536e-01,-2.0982163366775036e-01,2.6205776957416932e-01,-2.0192748468363880e-01,2.0991625990584037e-01,-2.3878054471338728e-01,3.2054730743521237e-01,-2.2902300508023066e-01,3.5625055719764054e-01,-3.2197370046832202e-01,3.8652575820766899e-01,-3.6059189451754942e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9000000000000000e+02,1.3820000000000000e+03,2.0770000000000000e+03,2.7720000000000000e+03,3.4670000000000000e+03,4.1680000000000000e+03,4.8670000000000000e+03,5.5670000000000000e+03,6.2680000000000000e+03,6.9720000000000000e+03,7.6750000000000000e+03,8.3790000000000000e+03,9.0860000000000000e+03,9.7910000000000000e+03,1.0497000000000000e+04,1.1204000000000000e+04,1.1913000000000000e+04,1.2622000000000000e+04,1.3331000000000000e+04,1.4041000000000000e+04,1.4750000000000000e+04,1.5459000000000000e+04,1.6169000000000000e+04,1.6878000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9009000000000000e+04,1.9725000000000000e+04,2.0435000000000000e+04,2.1141000000000000e+04,2.1875000000000000e+04,2.2579000000000000e+04,2.3267000000000000e+04,2.3975000000000000e+04,2.4683000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6799000000000000e+04,2.7503000000000000e+04,2.8207000000000000e+04,2.8908000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1010000000000000e+04,3.1707000000000000e+04,3.2404000000000000e+04,3.3098000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {1.6682712498426239e-01,-1.8658497293945167e-01,1.8716800259901073e-01,-2.0080511706846771e-01,2.5776226096916355e-01,-1.8014397580283278e-01,2.3187602202653296e-01,-1.8888815612376539e-01,2.3825312684105868e-01,-1.9764347168914259e-01,2.1303867405775190e-01,-2.1864997247176085e-01,1.9060281719060299e-01,-2.1095297916034275e-01,2.1385786807211804e-01,-2.0243697148649251e-01,1.6808369230471218e-01,-1.8247654347570377e-01,1.6321496740677194e-01,-1.6534240035866782e-01,1.3358190259939087e-01,-1.3762340408262302e-01,1.2517965805100878e-01,-1.2516492091413561e-01,9.9679467259256083e-02,-8.5525125627462728e-02,7.0833977291224953e-02,-4.1071933536127661e-02,2.4648656550307783e-02,-6.2804188860494159e-02,-1.4575088847986997e-02,2.2637929246120417e-02,-5.1578725614456332e-02,3.1892413968958873e-02,-4.9620043941112163e-02,1.0853263900788856e-01,-1.3861588547237652e-01,1.5996598004665616e-01,-1.5596844034381802e-01,2.1230696422520279e-01,-2.2320393916806852e-01,2.4987976967881814e-01,-2.7165466145699768e-01,3.4162810522906795e-01,-2.9219664119329974e-01,3.7353398140209576e-01,-4.0425355608407010e-01,4.8807824720156195e-01,-4.9484352711504931e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 32){
        
        double posx_arr[] = {6.9050000000000000e+02,1.3830000000000000e+03,2.0775000000000000e+03,2.7725000000000000e+03,3.4695000000000000e+03,4.1670000000000000e+03,4.8665000000000000e+03,5.5675000000000000e+03,6.2690000000000000e+03,6.9715000000000000e+03,7.6760000000000000e+03,8.3800000000000000e+03,9.0840000000000000e+03,9.7910000000000000e+03,1.0500000000000000e+04,1.1205000000000000e+04,1.1913500000000000e+04,1.2621500000000000e+04,1.3330500000000000e+04,1.4040000000000000e+04,1.4749500000000000e+04,1.5460500000000000e+04,1.6169500000000000e+04,1.6880000000000000e+04,1.7610000000000000e+04,1.8300500000000000e+04,1.9011000000000000e+04,1.9721000000000000e+04,2.0430500000000000e+04,2.1139500000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3974000000000000e+04,2.4681500000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6798500000000000e+04,2.7502500000000000e+04,2.8206500000000000e+04,2.8909500000000000e+04,2.9611000000000000e+04,3.0310500000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2402500000000000e+04,3.3097500000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-4.1811326057304959e-01,4.0436492773618415e-01,-3.8874308492672749e-01,4.0319109386388913e-01,-2.0327103754554171e-01,3.6356001096171148e-01,-3.7765652249194748e-01,3.7329236241614483e-01,-3.7302405347673778e-01,3.4574137566161223e-01,-2.9179442912111209e-01,3.1284562913875691e-01,-2.7828282578299496e-01,3.0090163652936064e-01,-5.3931494662861360e-02,3.2946480423954322e-01,-3.2313106631976390e-01,2.3003129440224190e-01,-2.3202222584287224e-01,2.9234690769238858e-01,-2.5747661959347629e-01,2.6450356695147847e-01,-2.8755358957872401e-01,2.9324437626200378e-01,6.0839298404533892e-03,2.1002995323592552e-01,-2.1271467179468675e-01,2.4112648011153520e-01,-2.2923991816791242e-01,2.9696748336603973e-01,-2.7624265659087388e-01,2.7938818218620237e-01,-3.3947674252676047e-01,3.3579165778069131e-01,-2.5225719514600858e-01,3.0966256137775544e-01,-2.7309471282926978e-01,3.2397963446313971e-01,-3.8460087104211044e-01,3.6046929186990184e-01,-3.1279656600468553e-01,4.2412178858616717e-01,-4.3334390546529999e-01,4.4002898166945115e-01,-3.6492039414093347e-01,5.0437629078566248e-01,-5.1841469206586543e-01,5.8386079370471022e-01,-5.2279820127807264e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9200000000000000e+02,1.4320000000000000e+03,2.2430000000000000e+03,2.8660000000000000e+03,3.3100000000000000e+03,4.3560000000000000e+03,4.7940000000000000e+03,5.7250000000000000e+03,6.0800000000000000e+03,6.9650000000000000e+03,7.6720000000000000e+03,8.3850000000000000e+03,9.0850000000000000e+03,9.7920000000000000e+03,1.0496000000000000e+04,1.1209000000000000e+04,1.1916000000000000e+04,1.2618000000000000e+04,1.3338000000000000e+04,1.4040000000000000e+04,1.4743000000000000e+04,1.5466000000000000e+04,1.6167000000000000e+04,1.6877000000000000e+04,1.7590000000000000e+04,1.8300000000000000e+04,1.9012000000000000e+04,1.9716000000000000e+04,2.0438000000000000e+04,2.1146000000000000e+04,2.1822000000000000e+04,2.2582000000000000e+04,2.3213000000000000e+04,2.4001000000000000e+04,2.4548000000000000e+04,2.5388000000000000e+04,2.6096000000000000e+04,2.6800000000000000e+04,2.7703000000000000e+04,2.8206000000000000e+04,2.8909000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2403000000000000e+04,3.3099000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {9.1817015907751640e-02,-7.4558880902590441e-02,1.0021750106569254e-01,-8.1255540914656865e-02,7.8758824849761530e-02,-6.5480982872218318e-02,4.9927752031917731e-02,-5.2293862125859278e-02,4.4319011950036923e-02,-7.0037937681126455e-02,5.5525684225796956e-02,-6.2678806301854101e-02,8.0150315988021165e-02,-7.7939607398843286e-02,9.0760334939195914e-02,-9.0941081179475858e-02,6.7621657629514137e-02,-5.2973098857163559e-02,2.9068364004368498e-02,-1.3508175778356052e-01,4.5802374906223992e-02,-4.9637495860299065e-02,7.1341710312362985e-02,-7.3468022284351439e-02,7.0095118508753287e-02,-4.6981720901255804e-02,3.4671327473436653e-02,-2.2220436013734590e-02,2.2689540013475777e-02,-4.3354894358314626e-02,-3.0374701550523554e-02,2.7394663867156926e-02,3.1546584927592602e-02,2.8535779885519252e-02,3.9224566635710702e-02,3.2352361968647471e-02,-8.3530325319985055e-02,7.3884253966537444e-02,-7.3927772892873569e-02,1.6861565469596540e-01,-8.8302634804034108e-02,1.9476882214716010e-01,-1.2955016456739019e-01,2.1302034574583958e-01,-1.3312539090262687e-01,2.5720164159618186e-01,-2.5450605018449951e-01,3.4096561624091770e-01,-3.2846345420828316e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 48){
        
        double posx_arr[] = {6.9100000000000000e+02,1.3830000000000000e+03,2.0770000000000000e+03,2.7730000000000000e+03,3.4700000000000000e+03,4.1670000000000000e+03,4.8660000000000000e+03,5.5670000000000000e+03,6.2690000000000000e+03,6.9710000000000000e+03,7.6760000000000000e+03,8.3800000000000000e+03,9.0850000000000000e+03,9.7910000000000000e+03,1.0500000000000000e+04,1.1205000000000000e+04,1.1913000000000000e+04,1.2622000000000000e+04,1.3331000000000000e+04,1.4040000000000000e+04,1.4750000000000000e+04,1.5461000000000000e+04,1.6170000000000000e+04,1.6880000000000000e+04,1.7630000000000000e+04,1.8300000000000000e+04,1.9011000000000000e+04,1.9721000000000000e+04,2.0430000000000000e+04,2.1139000000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3974000000000000e+04,2.4681000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6799000000000000e+04,2.7503000000000000e+04,2.8206000000000000e+04,2.8909000000000000e+04,2.9611000000000000e+04,3.0311000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2403000000000000e+04,3.3098000000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-4.6546030701871588e-01,4.5322843722843159e-01,-4.0296452694482243e-01,4.1375317620767021e-01,-2.1884454030157008e-01,4.0178523397638749e-01,-3.6332679385473488e-01,3.5953783051252153e-01,-3.8014084942129389e-01,3.9453325604620371e-01,-2.9709869865894628e-01,3.7473922390289083e-01,-3.0062461183492484e-01,3.0395467419563021e-01,-5.7425983876754835e-02,3.5281118763405150e-01,-3.4472928511635181e-01,2.5729409183696594e-01,-2.5691690705449804e-01,3.2359616781669703e-01,-2.8828974644382610e-01,2.9524281609515324e-01,-3.1441042180795142e-01,3.2034071055258856e-01,6.7291920416765877e-02,1.9955732934316811e-01,-2.2164206939038050e-01,2.5928245483629664e-01,-2.3914064499018456e-01,3.2419107244295431e-01,-2.6914359061569004e-01,2.7651431988616371e-01,-3.5056021281294547e-01,3.3840565896526742e-01,-2.5190825115141569e-01,3.1793136723382659e-01,-2.9296204094200112e-01,3.5558383872691207e-01,-3.8835466523822898e-01,3.5134711066501589e-01,-3.2703838102252242e-01,4.0137096582219123e-01,-4.5423154327610415e-01,4.4137629047096727e-01,-3.8940590013180354e-01,4.9347577423606981e-01,-5.2218936682038408e-01,5.8897172730483849e-01,-5.3589648467414785e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9200000000000000e+02,1.4320000000000000e+03,2.2430000000000000e+03,2.8660000000000000e+03,3.3100000000000000e+03,4.3560000000000000e+03,4.7940000000000000e+03,5.7250000000000000e+03,6.0800000000000000e+03,6.9650000000000000e+03,7.6720000000000000e+03,8.3850000000000000e+03,9.0850000000000000e+03,9.7920000000000000e+03,1.0496000000000000e+04,1.1209000000000000e+04,1.1916000000000000e+04,1.2618000000000000e+04,1.3338000000000000e+04,1.4040000000000000e+04,1.4743000000000000e+04,1.5466000000000000e+04,1.6167000000000000e+04,1.6877000000000000e+04,1.7590000000000000e+04,1.8300000000000000e+04,1.9012000000000000e+04,1.9716000000000000e+04,2.0438000000000000e+04,2.1146000000000000e+04,2.1822000000000000e+04,2.2582000000000000e+04,2.3213000000000000e+04,2.4001000000000000e+04,2.4548000000000000e+04,2.5388000000000000e+04,2.6096000000000000e+04,2.6800000000000000e+04,2.7703000000000000e+04,2.8206000000000000e+04,2.8909000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2403000000000000e+04,3.3099000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {-9.1817015907751640e-02,7.4558880902590441e-02,-1.0021750106569254e-01,-8.1255540914656865e-02,-7.8758824849761530e-02,6.5480982872218318e-02,-4.9927752031917731e-02,-5.2293862125859278e-02,4.4319011950036923e-02,-7.0037937681126455e-02,5.5525684225796956e-02,-6.2678806301854101e-02,8.0150315988021165e-02,-7.7939607398843286e-02,9.0760334939195914e-02,-9.0941081179475858e-02,6.7621657629514137e-02,-5.2973098857163559e-02,2.9068364004368498e-02,-1.3508175778356052e-01,4.5802374906223992e-02,-4.9637495860299065e-02,7.1341710312362985e-02,-7.3468022284351439e-02,7.0095118508753287e-02,-4.6981720901255804e-02,3.4671327473436653e-02,-2.2220436013734590e-02,2.2689540013475777e-02,-4.3354894358314626e-02,-3.0374701550523554e-02,2.7394663867156926e-02,3.1546584927592602e-02,2.8535779885519252e-02,3.9224566635710702e-02,3.2352361968647471e-02,-8.3530325319985055e-02,7.3884253966537444e-02,-7.3927772892873569e-02,1.6861565469596540e-01,-8.8302634804034108e-02,1.9476882214716010e-01,-1.2955016456739019e-01,2.1302034574583958e-01,-1.3312539090262687e-01,2.5720164159618186e-01,-2.5450605018449951e-01,3.4096561624091770e-01,-3.2846345420828316e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 56){
        
        double posx_arr[] = {6.9050000000000000e+02,1.3830000000000000e+03,2.0775000000000000e+03,2.7725000000000000e+03,3.4695000000000000e+03,4.1670000000000000e+03,4.8665000000000000e+03,5.5675000000000000e+03,6.2690000000000000e+03,6.9715000000000000e+03,7.6760000000000000e+03,8.3800000000000000e+03,9.0840000000000000e+03,9.7910000000000000e+03,1.0500000000000000e+04,1.1205000000000000e+04,1.1913500000000000e+04,1.2621500000000000e+04,1.3330500000000000e+04,1.4040000000000000e+04,1.4749500000000000e+04,1.5460500000000000e+04,1.6169500000000000e+04,1.6880000000000000e+04,1.7610000000000000e+04,1.8300500000000000e+04,1.9011000000000000e+04,1.9721000000000000e+04,2.0430500000000000e+04,2.1139500000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3974000000000000e+04,2.4681500000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6798500000000000e+04,2.7502500000000000e+04,2.8206500000000000e+04,2.8909500000000000e+04,2.9611000000000000e+04,3.0310500000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2402500000000000e+04,3.3097500000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-4.1811326057304959e-01,4.0436492773618415e-01,-3.8874308492672749e-01,4.0319109386388913e-01,-2.0327103754554171e-01,3.6356001096171148e-01,-3.7765652249194748e-01,3.7329236241614483e-01,-3.7302405347673778e-01,3.4574137566161223e-01,-2.9179442912111209e-01,3.1284562913875691e-01,-2.7828282578299496e-01,3.0090163652936064e-01,-5.3931494662861360e-02,3.2946480423954322e-01,-3.2313106631976390e-01,2.3003129440224190e-01,-2.3202222584287224e-01,2.9234690769238858e-01,-2.5747661959347629e-01,2.6450356695147847e-01,-2.8755358957872401e-01,2.9324437626200378e-01,6.0839298404533892e-03,2.1002995323592552e-01,-2.1271467179468675e-01,2.4112648011153520e-01,-2.2923991816791242e-01,2.9696748336603973e-01,-2.7624265659087388e-01,2.7938818218620237e-01,-3.3947674252676047e-01,3.3579165778069131e-01,-2.5225719514600858e-01,3.0966256137775544e-01,-2.7309471282926978e-01,3.2397963446313971e-01,-3.8460087104211044e-01,3.6046929186990184e-01,-3.1279656600468553e-01,4.2412178858616717e-01,-4.3334390546529999e-01,4.4002898166945115e-01,-3.6492039414093347e-01,5.0437629078566248e-01,-5.1841469206586543e-01,5.8386079370471022e-01,-5.2279820127807264e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9200000000000000e+02,1.4320000000000000e+03,2.2430000000000000e+03,2.8660000000000000e+03,3.3100000000000000e+03,4.3560000000000000e+03,4.7940000000000000e+03,5.7250000000000000e+03,6.0800000000000000e+03,6.9650000000000000e+03,7.6720000000000000e+03,8.3850000000000000e+03,9.0850000000000000e+03,9.7920000000000000e+03,1.0496000000000000e+04,1.1209000000000000e+04,1.1916000000000000e+04,1.2618000000000000e+04,1.3338000000000000e+04,1.4040000000000000e+04,1.4743000000000000e+04,1.5466000000000000e+04,1.6167000000000000e+04,1.6877000000000000e+04,1.7590000000000000e+04,1.8300000000000000e+04,1.9012000000000000e+04,1.9716000000000000e+04,2.0438000000000000e+04,2.1146000000000000e+04,2.1822000000000000e+04,2.2582000000000000e+04,2.3213000000000000e+04,2.4001000000000000e+04,2.4548000000000000e+04,2.5388000000000000e+04,2.6096000000000000e+04,2.6800000000000000e+04,2.7703000000000000e+04,2.8206000000000000e+04,2.8909000000000000e+04,2.9610000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2403000000000000e+04,3.3099000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        double ccdy_arr[] = {9.1817015907751640e-02,-7.4558880902590441e-02,1.0021750106569254e-01,-8.1255540914656865e-02,7.8758824849761530e-02,-6.5480982872218318e-02,4.9927752031917731e-02,-5.2293862125859278e-02,4.4319011950036923e-02,-7.0037937681126455e-02,5.5525684225796956e-02,-6.2678806301854101e-02,8.0150315988021165e-02,-7.7939607398843286e-02,9.0760334939195914e-02,-9.0941081179475858e-02,6.7621657629514137e-02,-5.2973098857163559e-02,2.9068364004368498e-02,-1.3508175778356052e-01,4.5802374906223992e-02,-4.9637495860299065e-02,7.1341710312362985e-02,-7.3468022284351439e-02,7.0095118508753287e-02,-4.6981720901255804e-02,3.4671327473436653e-02,-2.2220436013734590e-02,2.2689540013475777e-02,-4.3354894358314626e-02,-3.0374701550523554e-02,2.7394663867156926e-02,3.1546584927592602e-02,2.8535779885519252e-02,3.9224566635710702e-02,3.2352361968647471e-02,-8.3530325319985055e-02,7.3884253966537444e-02,-7.3927772892873569e-02,1.6861565469596540e-01,-8.8302634804034108e-02,1.9476882214716010e-01,-1.2955016456739019e-01,2.1302034574583958e-01,-1.3312539090262687e-01,2.5720164159618186e-01,-2.5450605018449951e-01,3.4096561624091770e-01,-3.2846345420828316e-01};
        arr_to_vec(ccdy_arr, sizeof(ccdy_arr)/sizeof(double), ccdy);
        
      }else if (tdi == 64){
        
        double posx_arr[] = {6.9000000000000000e+02,1.3830000000000000e+03,2.0780000000000000e+03,2.7720000000000000e+03,3.4690000000000000e+03,4.1670000000000000e+03,4.8670000000000000e+03,5.5680000000000000e+03,6.2690000000000000e+03,6.9720000000000000e+03,7.6760000000000000e+03,8.3800000000000000e+03,9.0830000000000000e+03,9.7910000000000000e+03,1.0500000000000000e+04,1.1205000000000000e+04,1.1914000000000000e+04,1.2621000000000000e+04,1.3330000000000000e+04,1.4040000000000000e+04,1.4749000000000000e+04,1.5460000000000000e+04,1.6169000000000000e+04,1.6880000000000000e+04,1.7590000000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9721000000000000e+04,2.0431000000000000e+04,2.1140000000000000e+04,2.1849000000000000e+04,2.2558000000000000e+04,2.3267000000000000e+04,2.3974000000000000e+04,2.4682000000000000e+04,2.5388000000000000e+04,2.6094000000000000e+04,2.6798000000000000e+04,2.7502000000000000e+04,2.8207000000000000e+04,2.8910000000000000e+04,2.9611000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2402000000000000e+04,3.3097000000000000e+04,3.3792000000000000e+04,3.4483000000000000e+04};
        arr_to_vec(posx_arr, sizeof(posx_arr)/sizeof(double), posx);
        
        double ccdx_arr[] = {-3.7076621412738331e-01,3.5550141824393666e-01,-3.7452164290863260e-01,3.9262901152010804e-01,-1.8769753478951337e-01,3.2533478794703546e-01,-3.9198625112916013e-01,3.8704689431976808e-01,-3.6590725753218167e-01,2.9694949527702069e-01,-2.8649015958327789e-01,2.5095203437462293e-01,-2.5594103973106508e-01,2.9784859886309106e-01,-5.0437005448967877e-02,3.0611842084503493e-01,-3.0153284752317600e-01,2.0276849696751786e-01,-2.0712754463124641e-01,2.6109764756808007e-01,-2.2666349274312650e-01,2.3376431780780377e-01,-2.6069675734949660e-01,2.6614804197141900e-01,-5.5124060735859098e-02,2.2050257712868293e-01,-2.0378727419899301e-01,2.2297050538677374e-01,-2.1933919134564028e-01,2.6974389428912521e-01,-2.8334172256605766e-01,2.8226204448624104e-01,-3.2839327224057546e-01,3.3317765659611515e-01,-2.5260613914060148e-01,3.0139375552168429e-01,-2.5322738471653850e-01,2.9237543019936735e-01,-3.8084707684599189e-01,3.6959147307478779e-01,-2.9855475098684858e-01,4.4687261135014311e-01,-4.1245626765449583e-01,4.3868167286793502e-01,-3.4043488815006340e-01,5.1527680733525516e-01,-5.1464001731134690e-01,5.7874986010458207e-01,-5.0969991788199742e-01};
        arr_to_vec(ccdx_arr, sizeof(ccdx_arr)/sizeof(double), ccdx);
        
        double posy_arr[] ={6.9100000000000000e+02,1.3840000000000000e+03,2.0760000000000000e+03,2.7740000000000000e+03,3.3350000000000000e+03,4.1650000000000000e+03,4.8680000000000000e+03,5.5660000000000000e+03,6.2730000000000000e+03,6.8570000000000000e+03,7.5250000000000000e+03,8.3060000000000000e+03,9.0490000000000000e+03,9.9610000000000000e+03,1.0499000000000000e+04,1.1204000000000000e+04,1.1913000000000000e+04,1.2624000000000000e+04,1.3332000000000000e+04,1.4042000000000000e+04,1.4750000000000000e+04,1.5460000000000000e+04,1.6171000000000000e+04,1.6881000000000000e+04,1.7591000000000000e+04,1.8301000000000000e+04,1.9011000000000000e+04,1.9724000000000000e+04,2.0427000000000000e+04,2.1140000000000000e+04,2.1952000000000000e+04,2.2429000000000000e+04,2.3267000000000000e+04,2.3978000000000000e+04,2.4676000000000000e+04,2.5386000000000000e+04,2.6095000000000000e+04,2.6797000000000000e+04,2.7501000000000000e+04,2.8207000000000000e+04,2.8907000000000000e+04,2.9611000000000000e+04,3.0310000000000000e+04,3.1009000000000000e+04,3.1707000000000000e+04,3.2403000000000000e+04,3.3100000000000000e+04,3.3792000000000000e+04,3.4484000000000000e+04};
        arr_to_vec(posy_arr, sizeof(posy_arr)/sizeof(double), posy);
        
        // There was not enough good data to reliably estimate ccdy
        // offsets. Some faint CCD artifacts are still visible after
        // correction. Need to revisit this case when more data is
        // available.
        ccdy = vector<double>(posy.size(), 0);
      }
    }
  }
  
  std::string wv = "WV01";
  if (!is_wv01) wv = "WV02";
  if (posx.empty()){
    vw_out(WarningMessage)
      << "wv_correct: Corrections not implemented for camera model "
      << wv << " for TDI " << tdi << ". A copy of the input image will be created."
      << std::endl;
  }
}

// Apply WorldView corrections to each vertical block as high as the image
// corresponding to one CCD sensor.
template <class ImageT>
class WVPerBlockCorrectView: public ImageViewBase<WVPerBlockCorrectView<ImageT>>{
  ImageT m_img;
  int m_tdi;
  bool m_is_wv01, m_is_forward;
  double m_pitch_ratio;
  std::vector<double> m_posx, m_ccdx, m_posy, m_ccdy;
  
  typedef typename ImageT::pixel_type PixelT;

public:
  WVPerBlockCorrectView(ImageT const& img, int tdi, bool is_wv01, bool is_forward,
                 double pitch_ratio):
    m_img(img), m_tdi(tdi), m_is_wv01(is_wv01), m_is_forward(is_forward),
    m_pitch_ratio(pitch_ratio){
    
    get_offsets(m_tdi, m_is_wv01, m_is_forward,  
                m_posx, m_ccdx, m_posy, m_ccdy);

    // Compensate for the variable pitch ratio
    for (int i = 0; i < (int)m_posx.size(); i++) m_posx[i] *= m_pitch_ratio;
    for (int i = 0; i < (int)m_posy.size(); i++) m_posy[i] *= m_pitch_ratio;
    
    VW_ASSERT(m_posx.size() == m_ccdx.size() &&
              m_posy.size() == m_ccdy.size(),
              ArgumentErr() << "wv_correct: Expecting the arrays of positions "
              << "and offsets to have the same sizes.");
  }
  
  typedef PixelT pixel_type;
  typedef PixelT result_type;
  typedef ProceduralPixelAccessor<WVPerBlockCorrectView> pixel_accessor;

  inline int32 cols() const { return m_img.cols(); }
  inline int32 rows() const { return m_img.rows(); }
  inline int32 planes() const { return 1; }

  inline pixel_accessor origin() const { return pixel_accessor(*this, 0, 0); }

  inline pixel_type operator()( double/*i*/, double/*j*/, int32/*p*/ = 0 ) const {
    vw_throw(NoImplErr() << "WVPerBlockCorrectView::operator()(...) is not implemented");
    return pixel_type();
  }

  typedef CropView<ImageView<pixel_type> > prerasterize_type;
  inline prerasterize_type prerasterize(BBox2i const& bbox) const {

    // Need to see a bit more of the input image for the purpose
    // of interpolation.
    int max_offset = 5; // CCD offsets are always under 1 pix
    int bias = BilinearInterpolation::pixel_buffer + max_offset;
    BBox2i biased_box = bbox;
    biased_box.expand(bias);
    biased_box.crop(bounding_box(m_img));
    
    ImageView<result_type> cropped_img = crop(m_img, biased_box);
    InterpolationView<EdgeExtensionView< ImageView<result_type>,
      ConstantEdgeExtension >, BilinearInterpolation> interp_img
      = interpolate(cropped_img, BilinearInterpolation(),
                    ConstantEdgeExtension());

    ImageView<result_type> tile(bbox.width(), bbox.height());
    for (int col = bbox.min().x(); col < bbox.max().x(); col++){

      // Accumulate the corrections up to the current column
      double valx = 0, valy = 0;
      if (m_ccdx.size() > 0){
        valx = 0.0;
        for (size_t t = 0; t < m_ccdx.size(); t++){
          if (m_posx[t] < col){
            valx -= m_ccdx[t];
          }
        }
        valy = 0.0;
        for (size_t t = 0; t < m_ccdy.size(); t++){
          if (m_posy[t] < col){
            valy -= m_ccdy[t];
          }
        }
      }
      
      for (int row = bbox.min().y(); row < bbox.max().y(); row++){
        tile(col - bbox.min().x(), row - bbox.min().y() )
          = interp_img(col - biased_box.min().x() + valx,
                       row - biased_box.min().y() + valy);
      }
    }
    
    return prerasterize_type(tile, -bbox.min().x(), -bbox.min().y(),
                             cols(), rows() );
  }

  template <class DestT>
  inline void rasterize(DestT const& dest, BBox2i bbox) const {
    vw::rasterize(prerasterize(bbox), dest, bbox);
  }
};
template <class ImageT>
WVPerBlockCorrectView<ImageT> wv_correct(ImageT const& img,
                                 int tdi, bool is_wv01, bool is_forward,
                                 double pitch_ratio){
  return WVPerBlockCorrectView<ImageT>(img, tdi, is_wv01, is_forward, pitch_ratio);
}

// Apply WorldView corrections to each column individually. This is more precise
// but works only for some select World View-3 multispectral images.
template <class ImageT>
class WVPerColumnCorrectView: public ImageViewBase< WVPerColumnCorrectView<ImageT> >{
  ImageT m_img;
  std::vector<double> m_dx, m_dy;
  typedef typename ImageT::pixel_type PixelT;

public:
  WVPerColumnCorrectView(ImageT const& img,
                         std::vector<double> const& dx,
                         std::vector<double> const& dy):
    m_img(img), m_dx(dx), m_dy(dy){

    if (m_dx.size() != m_dy.size()) {
      vw_throw( ArgumentErr() << "The x and y corrections must have the same dimensions.\n" );
    }

    // Add more zeros if the corrections are too short.
    // This is needed because different WV images can have
    // different widths.
    // TODO(oalexan1): This is not safe long-term.
    while (m_dx.size() < m_img.cols()) 
      m_dx.push_back(0);
    while (m_dy.size() < m_img.cols()) 
      m_dy.push_back(0);
    
    if (m_img.cols() != m_dx.size() || m_img.cols() != m_dy.size()) {
      // vw_throw( ArgumentErr() << "Expecting as many corrections as columns.\n" );
    }

  }
  
  typedef PixelT pixel_type;
  typedef PixelT result_type;
  typedef ProceduralPixelAccessor<WVPerColumnCorrectView> pixel_accessor;

  inline int32 cols() const { return m_img.cols(); }
  inline int32 rows() const { return m_img.rows(); }
  inline int32 planes() const { return 1; }

  inline pixel_accessor origin() const { return pixel_accessor( *this, 0, 0 ); }

  inline pixel_type operator()( double/*i*/, double/*j*/, int32/*p*/ = 0 ) const {
    vw_throw(NoImplErr() << "WVPerColumnCorrectView::operator()(...) is not implemented");
    return pixel_type();
  }

  typedef CropView<ImageView<pixel_type> > prerasterize_type;
  inline prerasterize_type prerasterize(BBox2i const& bbox) const {
    // Need to see a bit more of the input image for the purpose
    // of interpolation.
    int max_offset = 5; // CCD offsets are always under 1 pix
    int bias = BilinearInterpolation::pixel_buffer + max_offset;
    BBox2i biased_box = bbox;
    biased_box.expand(bias);
    biased_box.crop(bounding_box(m_img));
    
    ImageView<result_type> cropped_img = crop(m_img, biased_box);
    InterpolationView<EdgeExtensionView< ImageView<result_type>,
      ConstantEdgeExtension >, BilinearInterpolation> interp_img
      = interpolate(cropped_img, BilinearInterpolation(),
                    ConstantEdgeExtension());

    
    ImageView<result_type> tile(bbox.width(), bbox.height());

    for (int col = bbox.min().x(); col < bbox.max().x(); col++) {
      for (int row = bbox.min().y(); row < bbox.max().y(); row++) {
        // Note that the same correction is used for an entire column
        tile(col - bbox.min().x(), row - bbox.min().y() )
          = interp_img(col - biased_box.min().x() - m_dx[col],
                       row - biased_box.min().y() - m_dy[col]);
      }
    }

    return prerasterize_type(tile, -bbox.min().x(), -bbox.min().y(),
                             cols(), rows() );
  }

  template <class DestT>
  inline void rasterize(DestT const& dest, BBox2i bbox) const {
    vw::rasterize(prerasterize(bbox), dest, bbox);
  }
};
template <class ImageT>
WVPerColumnCorrectView<ImageT> wv_correct(ImageT const& img,
                                          std::vector<double> const& dx,
                                          std::vector<double> const& dy){
  return WVPerColumnCorrectView<ImageT>(img, dx, dy);
}

int main(int argc, char *argv[]) {

  Options opt;
  try {
    handle_arguments(argc, argv, opt);

    GeometricXML geo;
    AttitudeXML att;
    EphemerisXML eph;
    ImageXML img;
    RPCXML rpc;
    std::string scan_dir, sat_id, band_id;
    double det_pitch = 0.0;
    int tdi = 0;

    // Use per-column corrections (most recent) or per block of columns
    // corrections, which is coarser-grained (the older approach).
    bool per_column_correction = false;
    
    try{
      XMLPlatformUtils::Initialize();
      read_xml(opt.camera_file, geo, att, eph, img, rpc);
      
      scan_dir = boost::to_lower_copy(img.scan_direction);
      if (scan_dir != "forward" && scan_dir != "reverse")
        vw_throw( ArgumentErr() << "XML file \"" << opt.camera_file
                  << "\" is lacking a valid image scan direction.\n" );

      // Read XML parameters and sanity checks
      sat_id = img.sat_id;
      band_id = img.band_id;

      if (band_id == "P") {
        
        tdi = img.tdi;
        if (opt.dx != "" || opt.dy != "") 
          per_column_correction = true;
        
      } else if (band_id == "Multi") {

        per_column_correction = true;
        
        if (img.tdi_multi.size() != 8) 
          vw_throw( ArgumentErr() << "Expecting 8 TDI values, one per multispectral band.\n" );

      }
      
      det_pitch = geo.detector_pixel_pitch;
      if (det_pitch <= 0.0)
        vw_throw( ArgumentErr() << "XML file \"" << opt.camera_file
                  << "\" has a non-positive pixel pitch.\n" );

      
    } catch ( const std::exception& e ) {                
      vw_throw( ArgumentErr() << e.what() );
    }

    if (sat_id != "WV01" && sat_id != "WV02" && sat_id != "WV03") 
      vw_throw( ArgumentErr() << "Only WV01, WV02, and WV03 satellites are supported." );
    
    if ( (opt.dx != "" && opt.dy == "") || (opt.dx == "" && opt.dy != "") )
      vw_throw( ArgumentErr() << "Only one of --dx and --dy was specified." );
              
    bool is_wv01 = (sat_id == "WV01");
    bool is_forward = (scan_dir == "forward");

    // Adjust for detector pitch
    double pitch_ratio = 8.0e-3/det_pitch;

    int num_bands = vw::get_num_channels(opt.image_file);
    if (num_bands != 1) 
      vw_throw( ArgumentErr() << "Unsupported image with " << num_bands
                << " bands. Use gdal_translate to pick the desired band.\n\n");
    
    DiskImageView<float> input_img(opt.image_file);
    if (input_img.planes() != 1) 
      vw_throw( ArgumentErr() << "Only single-band images are supported. "
                << "For multispectral data, please extract first the desired "
                << "band using gdal_translate per the documentation.\n");
    
    bool has_nodata = false;
    double nodata = numeric_limits<double>::quiet_NaN();
    boost::shared_ptr<DiskImageResource> img_rsrc
      ( new DiskImageResourceGDAL(opt.image_file) );
    if (img_rsrc->has_nodata_read()){
      has_nodata = true;
      nodata = img_rsrc->nodata_read();
    }

    vw::cartography::GeoReference georef;
    bool has_georef = false;
    
    ImageViewRef<float> corr_img;
    std::vector<double> dx, dy;
    if (!per_column_correction) {
      // Per block correction
      if (has_nodata) 
        corr_img = apply_mask(wv_correct(create_mask(input_img, nodata),
                                         tdi, is_wv01, is_forward,
                                         pitch_ratio), nodata);
      else
        corr_img = wv_correct(input_img, tdi, is_wv01, is_forward, pitch_ratio);
    } else {
      // Per column correction
      
      if (band_id != "Multi" && (opt.dx == "" || opt.dy == "")) 
        vw_throw( ArgumentErr() << "Per column corrections were not specified.\n" );
      
      if (band_id == "Multi" && (opt.dx == "" || opt.dy == "")) {
        
        if (opt.band <= 0 || opt.band > 8) 
          vw_throw( ArgumentErr() << "For multispectral images, the band must "
                    << "be between 1 and 8.\n" );
        
        tdi = img.tdi_multi[opt.band - 1];
        parse_ms_correction_table(sat_id, opt.band, tdi, scan_dir,
                                  // outputs
                                  dx, dy);
        
        
        if (dx.size() == 0 || dy.size() == 0)
          vw_out(WarningMessage)
            << "wv_correct: Corrections not implemented for satellite "
            << sat_id << " for TDI " << tdi << ", band " << opt.band << ", "
            << "and scan direction " << scan_dir << "."
            << " A copy of the input image will be created." << std::endl;
        
      } else {
        vw_out() << "Reading per column corrections from: "
                  << opt.dx << ' ' << opt.dy << std::endl;
        asp::read_vec(opt.dx, dx);
        asp::read_vec(opt.dy, dy);
      }
      
      if (opt.print_per_column_corrections) {
        vw_out() << "Printing the x and y corrections as two columns." << std::endl;
        vw_out().precision(17);
        for (size_t it = 0; it < dx.size(); it++) {
          vw_out() << dx[it] << ' ' << dy[it] << std::endl;
        }
      }
      
      if (has_nodata) 
        corr_img = apply_mask(wv_correct(create_mask(input_img, nodata), dx, dy), nodata);
      else
        corr_img = wv_correct(input_img, dx, dy);
    }
    
    vw_out() << "Writing: " << opt.output_image << std::endl;
    TerminalProgressCallback tpc("asp", "\t--> ");
    if (opt.output_type == "Float32") 
      vw::cartography::block_write_gdal_image(opt.output_image, corr_img,
                                              has_georef, georef, has_nodata, nodata, opt, tpc);
    else if (opt.output_type == "Byte") 
      vw::cartography::block_write_gdal_image(opt.output_image,
					      per_pixel_filter(corr_img,
							       RoundAndClamp<uint8, float>()),
					      has_georef, georef, has_nodata,
                                              vw::round_and_clamp<uint8>(nodata),
                                              opt, tpc);
    else if (opt.output_type == "UInt16") 
      vw::cartography::block_write_gdal_image(opt.output_image,
					      per_pixel_filter(corr_img,
							       RoundAndClamp<uint16, float>()),
                                              has_georef, georef, has_nodata,
                                              vw::round_and_clamp<uint16>(nodata),
                                              opt, tpc);
    else if (opt.output_type == "Int16") 
      vw::cartography::block_write_gdal_image(opt.output_image,
					      per_pixel_filter(corr_img,
							       RoundAndClamp<int16, float>()),
                                              has_georef, georef, has_nodata,
                                              vw::round_and_clamp<int16>(nodata),
                                              opt, tpc);

    else if (opt.output_type == "UInt32") 
      vw::cartography::block_write_gdal_image(opt.output_image,
					      per_pixel_filter(corr_img,
							       RoundAndClamp<uint32, float>()),
                                              has_georef, georef, has_nodata,
                                              vw::round_and_clamp<uint32>(nodata),
                                              opt, tpc);
    else if (opt.output_type == "Int32") 
      vw::cartography::block_write_gdal_image(opt.output_image,
					      per_pixel_filter(corr_img,
							       RoundAndClamp<int32, float>()),
                                              has_georef, georef, has_nodata,
                                              vw::round_and_clamp<int32>(nodata),
                                              opt, tpc);
    else
      vw_throw( NoImplErr() << "Unsupported output type: " << opt.output_type << ".\n" );
    
    
  } ASP_STANDARD_CATCHES;
  
  return 0;
}
