/* Copyright 2019 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#ifndef TENSORFLOW_COMPILER_XLA_SERVICE_GPU_CHOLESKY_THUNK_H_
#define TENSORFLOW_COMPILER_XLA_SERVICE_GPU_CHOLESKY_THUNK_H_

#include "absl/types/optional.h"
#include "tensorflow/compiler/xla/service/buffer_assignment.h"
#include "tensorflow/compiler/xla/service/gpu/buffer_allocations.h"
#include "tensorflow/compiler/xla/service/gpu/cusolver_context.h"
#include "tensorflow/compiler/xla/service/gpu/gpu_executable.h"
#include "tensorflow/compiler/xla/service/gpu/thunk.h"
#include "tensorflow/compiler/xla/service/hlo_instruction.h"
#include "tensorflow/compiler/xla/types.h"
#include "tensorflow/compiler/xla/xla_data.pb.h"
#include "tensorflow/core/lib/core/status.h"
#include "tensorflow/core/platform/stream_executor_no_cuda.h"
#include "tensorflow/stream_executor/blas.h"
#include "tensorflow/stream_executor/gpu/gpu_asm_opts.h"

namespace xla {
namespace gpu {

// This class stores everything that StreamExecutor needs to launch a Cholesky
// decomposition (LAPACK potrf). It is generated by IrEmitter.
//
// As an implementation detail, we may run potrf (potentially in a loop, if
// batch_size >1), or potrfBatched.
//
// Thread-compatible.
class CholeskyThunk : public Thunk {
 public:
  CholeskyThunk(ThunkInfo thunk_info, const CholeskyOptions& options,
                se::GpuAsmOpts asm_opts, BufferAllocation::Slice a_buffer,
                BufferAllocation::Slice workspace_buffer,
                BufferAllocation::Slice info_buffer, PrimitiveType type,
                int64_t batch_size, int64_t n);

  CholeskyThunk(const CholeskyThunk&) = delete;
  CholeskyThunk& operator=(const CholeskyThunk&) = delete;

  Status ExecuteOnStream(const ExecuteParams& params) override;

 private:
  template <typename T>
  Status DoPotrfBatched(const ExecuteParams& params, GpuSolverContext* context);

  template <typename T>
  Status DoPotrfUnbatched(const ExecuteParams& params,
                          GpuSolverContext* context);

  se::GpuAsmOpts asm_opts_;
  se::blas::UpperLower uplo_;

  const BufferAllocation::Slice a_buffer_;
  const BufferAllocation::Slice workspace_buffer_;
  const BufferAllocation::Slice info_buffer_;

  const PrimitiveType type_;
  const int64_t batch_size_;
  const int64_t n_;
};

}  // namespace gpu
}  // namespace xla

#endif  // TENSORFLOW_COMPILER_XLA_SERVICE_GPU_CHOLESKY_THUNK_H_
