from __future__ import print_function

import os
from sys import exit
from sys import platform

import tarfile
import numpy as np
import multiprocessing as mp

from grid_plot_mpl import plot_scatter, plot_contourf, plot_path


#
# Plotting parameters
#

plot_params = {
    "scatter": {
        "filenames": [
            "grid_codes",
            "grid_codes_macrophage_alveolar_resting",
            "grid_codes_macrophage_alveolar_active",
            "grid_codes_macrophage_alveolar_apoptotic"
        ],
        "combined_plot": True,
        "fn": plot_scatter
    },
    "contourf": {
        "filenames": [
            "rd_phago_ck"
        ],
        "combined_plot": False,
        "fn": plot_contourf
    },
    "path": {
        "filenames": [
            "path_macrophage"
        ],
        "combined_plot": False,
        "fn": plot_path
    }
}


#
# Plotting specific data
#

plot_data = [
    "random_migration",
    "chemotactic_migration",
    "recruitment"
]


#
# Read in the constants file
#

def read_consts(fn):
    try:
        consts = np.loadtxt(fn, delimiter='\t', dtype='i4, i4, f8, f8, f8, i4', unpack=True)
    except:
        try:
            consts = np.loadtxt(fn, delimiter=' ', dtype='i4, i4, f8, f8, f8, i4', unpack=True)
        except:
            raise

    return consts


#
# Plotting routine
#

def main():
    from argparse import ArgumentParser

    #===========================================================================
    # Command line arguments
    #===========================================================================

    parser = ArgumentParser(description='Plot simulations')

    parser.add_argument('-n', '--nplots', type=int, dest='num_plots', default=1,
                        required=False, help='number of plots')

    _plot_types = ('scatter', 'contourf', 'path')
    parser.add_argument('-p', '--plot', type=str, dest='plot_t', required=True,
                        choices=_plot_types, help='type of plot')

    # Parse args
    args = parser.parse_args()

    #===========================================================================


    #===========================================================================
    # Store local vars from command line args
    #===========================================================================

    # Number of plots
    num_plots = args.num_plots

    # Requested plot
    plot_t = args.plot_t
    print("\nplot type: {0}".format(plot_t))

    # Format for the plotting number in the file name
    fmt = '%0{0}d'.format(len('{0}'.format(num_plots)))

    #===========================================================================


    # Solutions to plot
    soln = plot_params[plot_t]["filenames"]
    print("\nSolution(s): ", end='')
    for s in soln:
        print(" {0}  ".format(s), end='')
    print("\n", end='')


    #
    # construct plot
    #

    for directory in plot_data:
        # skip path plot for recruitment
        if "recruitment" in directory and "path" in plot_t:
            continue

        # set the tar file
        odir = "output"
        tarf = "{0}/{1}/raw.tar.gz".format(directory, odir)

        # Print to screen
        print('\nTarfile: \n\t{0}'.format(tarf))

        # Read in the constants and solution files
        data = {}
        with tarfile.open(tarf, "r:gz") as tar:
            consts = read_consts(tar.extractfile("consts.txt").readlines())

            try:
                for s in soln:
                    data[s] = np.loadtxt(tar.extractfile(s+".txt").readlines())
            except:
                raise

        # set the constants
        seed, nout, time_limit, timestep, output_interval, grid_size = [consts[i].item() for i in np.arange(len(consts))]

        # sanity check on solutions
        for s in soln:
            if (np.shape(data[s])[0] != nout):
                print('nout = {0}'.format(nout))
                print('np.shape(data[{0}]) = {1}'.format(s,np.shape(data[s])))
                raise ValueError('Number of data rows not equal to number of outputs')

        # Output directory for figures
        sdir = "{0}/figs".format(directory)

        # Create directory if it does not exist
        if not os.path.isdir(sdir):
            try:
                os.makedirs(sdir)
            except:
                raise

        # Print to screen
        print('\nSaving to: \n\t{0}'.format(sdir))

        # define the plotting options
        plot_options = {
            "combined_plot": plot_params[plot_t]["combined_plot"],
            "nout": nout,
            "time_limit": time_limit,
            "timestep": timestep,
            "output_interval": output_interval,
            "grid_size": grid_size,
            "dot_size": 2.0,
            "xlim": (0,grid_size),
            "ylim": (0,grid_size),
            "xticks": None,
            "yticks": None,
            "lbl": None,
            "lblfontsize": 13
        }

        #
        # Define the range of plots
        #

        if num_plots > nout:
            num_plots = nout

        # Set the step for plotting frames
        slice_div = np.floor_divide(nout, num_plots)

        # Define plotting frames
        pframes = [i for i in np.arange(1, nout+1, slice_div)]

        if not nout in pframes:
            pframes.append(nout)

        print(
            '\nPlotting every {0} frame(s) => {1} plot(s)'.format(
                slice_div, len(pframes)
            )
        )

        #===================================================================
        # Construct the requested plots
        #===================================================================

        # Output file name
        if plot_t == "scatter":
            fname = plot_t + '_grid_codes_' + fmt + ".png"
            print("\nOutput file format: {0}".format(fname))

        elif plot_t == "contourf":
            fname = {}
            for s in soln:
                fname[s] = plot_t + '_' + s + "_" + fmt + ".png"
            print("\nOutput file format: {0}_SOLUTION_{1}.png".format(plot_t, fmt))

        elif plot_t == "path":
            fname = plot_t + '_macrophage_' + fmt + ".png"
            print("\nOutput file format: {0}".format(fname))

        else:
            raise ValueError("unknown plot type")

        # Create multiprocessing pool
        nproc = int(mp.cpu_count()/2.0) if mp.cpu_count() > 1 else 1
        if nproc > len(pframes):
            nproc = len(pframes)

        print(
            '\nCreating {0} plots using a pool of {1} worker processes ... '.format(len(pframes), nproc),
            end=''
        )

        pool = mp.Pool(nproc)

        # Zip the arguments for each worker in the multiprocessing pool
        #   data, sdir, fname, fr, _plt, count
        if plot_t == 'path':
            plot_dat = tuple({s: data[s][0:i] for s in soln} for i in pframes)
        else:
            plot_dat = tuple({s: data[s][i-1] for s in soln} for i in pframes)

        pargs = zip(
            plot_dat,
            tuple(sdir for i in range(len(pframes))),
            tuple(fname for i in range(len(pframes))),
            pframes,
            tuple(plot_options for i in range(len(pframes))),
            range(1,len(pframes)+1)
        )

        # Submit plots to worker processes
        pool.map(plot_params[plot_t]["fn"], pargs)

        print('done\n')
        pool.close()
        pool.join()

        #===================================================================

if __name__ == '__main__':
    main()
