from scipy.integrate import solve_ivp
import numpy as np


#
# Viral entry and replication ODE model
#

def ode_model_solve(t, z, par):
  r_eu, r_eb, r_ib, r_iu, v, u, r, p, a, nvir = z
  r_bind, r_recycle, r_endo, r_release, r_U, r_P, lam_R, r_S, r_A, lam_P, r_E, r_scale, a_export, nvir_threshold, nv_scale, repmax, rephalf = par

  tnh = 0.0
  if ode_model_solve.internalising:
    tnh = 1.0
  elif nvir >= nvir_threshold:
    ode_model_solve.internalising = True
    tnh = 1.0

  r_b = r_bind * tnh

  tnh_e = 0.0
  if ode_model_solve.exporting:
    tnh_e = 1.0
  elif a >= a_export:
    ode_model_solve.exporting = True
    tnh_e = 1.0

  r_export = r_E * tnh_e

  ## ACE2 receptor and replication model
  return [
    -r_b * nvir * r_eu + r_recycle * r_iu,                                          # external unbound ACE2 receptor (R_eu: y[0])
    r_b * nvir * r_eu - r_endo * r_eb,                                              # external bound ACE2 receptor (R_eb: y[1])
    r_endo * r_eb - r_release * r_ib,                                               # internal bound ACE2 receptor (R_ib: y[2])
    r_release * r_ib - r_recycle * r_iu,                                            # internal unbound ACE2 receptor (R_iu: y[3])
    r_release * r_ib - r_U * v,                                                     # intracellular virus (V: y[4])
    r_U * v - r_P * u,                                                              # uncoated viral components (U: y[5])
    r_P * u + repmax * (rephalf*r / (rephalf + r)) - lam_R * r,                     # viral RNA ready for protein synthesis (R: y[6])
    r_S * r - lam_P * p - r_A * p,                                                  # synthesized viral proteins (P: y[7])
    r_A * p - r_export * a,                                                         # assembled virus (A: y[8])
    r_export * a - r_b * nvir * r_eu                                                # extracellular virus (VIR_E: y[9])
  ]

# set functions static variables
ode_model_solve.internalising = False
ode_model_solve.exporting = False


#
# Solve the viral entry and replication models
#

def entry_repl_solve(sim_time):
  # initial conditions
  u0 = [
    10000.0, # R_eu
    0.0,     # R_eb
    0.0,     # R_ib
    0.0,     # R_iu
    0.0,     # V
    0.0,     # U
    0.0,     # R
    0.0,     # P
    0.0,     # A
    1.0      # Virus
  ]


  # parameters (units: hr^{-1})
  p_hr = (
    0.06,  # r_bind
    0.6,   # r_recycle
    0.6,   # r_endo
    0.06,  # r_release
    0.6,   # r_U
    0.6,   # r_P
    0.021, # lam_R
    0.6,   # r_S
    0.6,   # r_A
    0.021, # lam_P
    0.6,   # r_E
    100.0, # r_scale
    1.0,   # a_export
    1.0,   # nvir_threshold
    100.0, # nv_scale
    0.3,   # repmax
    2000   # rephalf
  )

  tstart = 0
  tend = sim_time
  tspan = [tstart, tend]
  tint = 1

  sol = solve_ivp(ode_model_solve, tspan, u0, args=[p_hr], t_eval=[val for val in range(tstart,tend+1,tint)], dense_output=False)

  return sol.t, sol.y
