#######################################################################################
# 4. Reconstructing pollination histories in 11 trees by stochastic character mapping
#######################################################################################

library(phytools)
library(corHMM)
library(beepr)
library(dplyr)

# Load models
load("results_uncertain.RData")

# Load pollinator and clade data from Kriebel et al.
all_salvia <- read.csv("Salvia_flower_evol_May2020/Data/groups.csv", header=T)
row.names(all_salvia) <- all_salvia$species

# Erase species not in tree
all_salvia <- all_salvia[tree_list[[1]]$tip.label, ]

# Recode pollinator syndromes 
# 1 code for bee pollination
# 2 code for bird pollination
# 1&2 indicates ambiguity
all_salvia$pollinator2 <- gsub("bee&bird", "intermediate", all_salvia$pollinator)
all_salvia$pollinator2 <- gsub("sister", "bee", all_salvia$pollinator)
all_salvia$pollinator2<-as.factor(all_salvia$pollinator2)
all_salvia$pollinator2<-factor(all_salvia$pollinator2,labels = c("1","2","1&2"))

# Final base
b0 <- all_salvia[,c("species","pollinator2")]

# FUNCTIONS NEEDED

## function *detect_origin* takes one tip and one simmap reconstruction,
## and extracts character states at the nodes from the root to the tip.
## Finally, it classifies each tip in two categories: from bee or from bird

detect_origin <- function(tip, phylo){
  root <- Nnode(phylo) + 2
  np <- nodepath(phylo, from = root, to = tip)
  nstates <- c(getStates(phylo, type = "nodes"))
  seq_nodes <- c(nstates[names(nstates) %in% np])
  seq_nodes <- replace(seq_nodes, seq_nodes == "3", "1")
  seq_nodes <- replace(seq_nodes, seq_nodes == "4", "2")
  N <- length(seq_nodes)
  seq_changes <- if_else(seq_nodes == lag(seq_nodes, 1), 0, 1)
  seq_changes[1] <- 0
  changes <- sum(seq_changes)
  res <- character()
  if(seq_nodes[1] == "1" & changes == 0) res <- "from_bee" else res <- "from_bird"
  return(res)
}

# detect_classify function apply detect origin 
# to all the tips in a simmap tree

detect_classify <- function(phylo, FUN){
  X <- c(1:(phylo$Nnode+1))
  states <- sapply(X, FUN = FUN, phylo)
  res <- data.frame(states = states, species = phylo$tip.label)
  return(res)
}


# largest remainder rounding
LRM <- function(x){
  D <- sum(x) - sum(trunc(x))
  E <- trunc(x)
  R <- x - E
  o <- order(R, decreasing = TRUE)
  E[o[1:D]]<-E[o[1:D]]+1
  return(E)
}

# matrix with roundings per tree (row) and model (column)
rounding <- list()
for(i in 1:11){
  rounding[[i]] <- LRM(res_unc$aiccwt[res_unc$tree == i]*1000) 
}

rounding <- do.call(rbind, rounding)
rounding

########################################################
# 4.1 Stochastic character mapping on tree 1 (mcc tree)
########################################################

#-----------------------------------------------------------------
# WARNING 1: This could take a while
# WARNING 2: adjust the number of nCores according to your system
#-----------------------------------------------------------------

# 4.1.1 smm_ard
simmap_smm_ard_t1 <- makeSimmap(tree = tree_list[[1]], data = b0, 
                                model =  unc_smm_ard[[1]]$solution,
                                rate.cat = 1, nSim = rounding[1,1],
                                nCores = 7)

# 4.1.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.1.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t1 <- makeSimmap(tree = tree_list[[1]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[1]]$solution,
                                          rate.cat = 2, nSim = rounding[1,3],
                                          nCores = 7)

# 4.1.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t1 <- makeSimmap(tree = tree_list[[1]], data = b0, 
                                       model = unc_hmm_ard_sym_ctard[[1]]$solution,
                                       rate.cat = 2, nSim = rounding[1,4],
                                       nCores = 7)

# 4.1.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t1 <- makeSimmap(tree = tree_list[[1]], data = b0, 
                                       model = unc_hmm_sym_sym_ctard[[1]]$solution,
                                       rate.cat = 2, nSim = rounding[1,5],
                                       nCores = 7)

# 4.1.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t1 <- makeSimmap(tree = tree_list[[1]], data = b0, 
                                       model = unc_hmm_ard_ard_ctsym[[1]]$solution,
                                       rate.cat = 2, nSim = rounding[1,6],
                                       nCores = 7)

# 4.1.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t1 <- makeSimmap(tree = tree_list[[1]], data = b0, 
                                       model = unc_hmm_ard_sym_ctsym[[1]]$solution,
                                       rate.cat = 2, nSim = rounding[1,7],
                                       nCores = 7)

# 4.1.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t1 <- makeSimmap(tree = tree_list[[1]], data = b0, 
                                       model = unc_hmm_sym_sym_ctsym[[1]]$solution,
                                       rate.cat = 2, nSim = rounding[1,8],
                                       nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t1, 
                 simmap_hmm_ard_ard_ctard_t1,
                 simmap_hmm_ard_sym_ctard_t1, 
                 simmap_hmm_sym_sym_ctard_t1,
                 simmap_hmm_ard_ard_ctsym_t1, 
                 simmap_hmm_ard_sym_ctsym_t1,
                 simmap_hmm_sym_sym_ctsym_t1)
simmap_t01<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t01) <- c("multiSimmap","multiPhylo") 

# CHECK (only in the mcc tree): comparing posterior distribution of pollination states at nodes
# with the marginal probability of pollination states at nodes
# categories 2 and 4 from ASR using HMM were merged as bird pollination (1)
# categories 1 and 3 from ASR using HMM were merged as bee pollination (0)

bird_state_node <- function(phylo){
  birdx <- if_else(getStates(phylo, type = "nodes") == "2"|
                     getStates(phylo, type = "nodes") == "4", 1, 0)
  return(birdx)
}

list_bird_states <- list()
for(i in 1: length(simmap_t01)) list_bird_states[[i]] <- bird_state_node(simmap_t01[[i]])
list_birds <- do.call(cbind, list_bird_states)
post_state <- data.frame(bee = 1-rowSums(list_birds)/999,
                         bird = rowSums(list_birds)/999)
post_state <- as.matrix(post_state)
plot(res[, 2], post_state[, 2], xlab = "marginal prob. of bird pollination",
     ylab = "posterior prob. of bird pollination", col = rgb(0,0,0,0.2), pch = 19)
summary(lm(post_state[, 2]~res[, 2]))
abline(lm(post_state[, 2]~res[, 2]), col = "red")
abline(0, 1, col = "blue")


# 4.1.9 Classify species according to their ancestors
#----------------------------------
# WARNING: This could take a while
#----------------------------------

ORIGINS_01 <- list(length(simmap_t01)) 
for(i in 1:length(simmap_t01)) ORIGINS_01[[i]] <- detect_classify(simmap_t01[[i]], detect_origin)
beep(5)

origins_t01 <- do.call(cbind, ORIGINS_01)[, seq(1, 2000, 2)]
row.names(origins_t01) <- ORIGINS_01[[1]]$species
rm(ORIGINS_01)
origins_01 <- apply(origins_t01, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

# CHECK (only in mcc tree)
# rows are 528 species ordered according to tip number
# columns are reconstruction of pollinator histories
# yellow is "from bird", purple is "from bee"
# the horizontal cyan line separtates Calosphace (top) from 
# the rest of Salvia (bottom)
image(t(origins_01), col = hcl.colors(2))
abline(h = 263/528, col = "cyan")

#-----------------------------------------------------------------------------------
save(origins_t01, origins_01, file = "results_simmap_origin_t01.RData")
rm(simmap_t01, origins_t01, origins_01, 
   simmap_smm_ard_t1,
   simmap_hmm_ard_ard_ctard_t1, 
   simmap_hmm_ard_sym_ctard_t1,
   simmap_hmm_sym_sym_ctard_t1,
   simmap_hmm_ard_ard_ctsym_t1,
   simmap_hmm_ard_sym_ctsym_t1,
   simmap_hmm_sym_sym_ctsym_t1)
gc()
#------------------------------------------------------------------------------------

########################################################
# 4.2 Stochastic character mapping on tree cluster 1
########################################################

# 4.2.1 smm_ard
simmap_smm_ard_t2 <- makeSimmap(tree = tree_list[[2]], data = b0, 
                                model =  unc_smm_ard[[2]]$solution,
                                rate.cat = 1, nSim = rounding[2,1],
                                nCores = 7)

# 4.2.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.2.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t2 <- makeSimmap(tree = tree_list[[2]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[2]]$solution,
                                          rate.cat = 2, nSim = rounding[2,3],
                                          nCores = 7)

# 4.2.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t2 <- makeSimmap(tree = tree_list[[2]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[2]]$solution,
                                          rate.cat = 2, nSim = rounding[2,4],
                                          nCores = 7)

# 4.2.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t2 <- makeSimmap(tree = tree_list[[2]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[2]]$solution,
                                          rate.cat = 2, nSim = rounding[2,5],
                                          nCores = 7)

# 4.2.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t2 <- makeSimmap(tree = tree_list[[2]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[2]]$solution,
                                          rate.cat = 2, nSim = rounding[2,6],
                                          nCores = 7)

# 4.2.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t2 <- makeSimmap(tree = tree_list[[2]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[2]]$solution,
                                          rate.cat = 2, nSim = rounding[2,7],
                                          nCores = 7)

# 4.2.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t2 <- makeSimmap(tree = tree_list[[2]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[2]]$solution,
                                          rate.cat = 2, nSim = rounding[2,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t2, 
                 simmap_hmm_ard_ard_ctard_t2,
                 simmap_hmm_ard_sym_ctard_t2, 
                 simmap_hmm_sym_sym_ctard_t2,
                 simmap_hmm_ard_ard_ctsym_t2, 
                 simmap_hmm_ard_sym_ctsym_t2,
                 simmap_hmm_sym_sym_ctsym_t2)
simmap_t02<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t02) <- c("multiSimmap","multiPhylo") 

# 4.2.9 Classify species according to their ancestors
ORIGINS_02 <- list(length(simmap_t02)) 
for(i in 1:length(simmap_t02)) ORIGINS_02[[i]] <- detect_classify(simmap_t02[[i]], detect_origin)
beep(5)

origins_t02 <- do.call(cbind, ORIGINS_02)[, seq(1, 2000, 2)]
row.names(origins_t02) <- ORIGINS_02[[1]]$species
rm(ORIGINS_02)
origins_02 <- apply(origins_t02, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#------------------------------------------------------------------------------
save(origins_t02, origins_02, file = "results_simmap_origin_t02.RData")
rm(simmap_t02, origins_t02, origins_02,
   simmap_smm_ard_t2,
   simmap_hmm_ard_ard_ctard_t2, 
   simmap_hmm_ard_sym_ctard_t2,
   simmap_hmm_sym_sym_ctard_t2,
   simmap_hmm_ard_ard_ctsym_t2,
   simmap_hmm_ard_sym_ctsym_t2,
   simmap_hmm_sym_sym_ctsym_t2)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.3 Stochastic character mapping on tree cluster 2
########################################################

# 4.3.1 smm_ard no simulation were done (AICc weigth < 0.0001)

# 4.3.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.3.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t3 <- makeSimmap(tree = tree_list[[3]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[3]]$solution,
                                          rate.cat = 2, nSim = rounding[3,3],
                                          nCores = 7)

# 4.3.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t3 <- makeSimmap(tree = tree_list[[3]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[3]]$solution,
                                          rate.cat = 2, nSim = rounding[3,4],
                                          nCores = 7)

# 4.3.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t3 <- makeSimmap(tree = tree_list[[3]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[3]]$solution,
                                          rate.cat = 2, nSim = rounding[3,5],
                                          nCores = 7)

# 4.3.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t3 <- makeSimmap(tree = tree_list[[3]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[3]]$solution,
                                          rate.cat = 2, nSim = rounding[3,6],
                                          nCores = 7)

# 4.3.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t3 <- makeSimmap(tree = tree_list[[3]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[3]]$solution,
                                          rate.cat = 2, nSim = rounding[3,7],
                                          nCores = 7)

# 4.3.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t3 <- makeSimmap(tree = tree_list[[3]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[3]]$solution,
                                          rate.cat = 2, nSim = rounding[3,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_hmm_ard_ard_ctard_t3,
                 simmap_hmm_ard_sym_ctard_t3, 
                 simmap_hmm_sym_sym_ctard_t3,
                 simmap_hmm_ard_ard_ctsym_t3, 
                 simmap_hmm_ard_sym_ctsym_t3,
                 simmap_hmm_sym_sym_ctsym_t3)
simmap_t03<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t03) <- c("multiSimmap","multiPhylo") 

# 4.3.9 Classify species according to their ancestors
ORIGINS_03 <- list(length(simmap_t03)) 
for(i in 1:length(simmap_t03)) ORIGINS_03[[i]] <- detect_classify(simmap_t03[[i]], detect_origin)
beep(5)

origins_t03 <- do.call(cbind, ORIGINS_03)[, seq(1, 2000, 2)]
row.names(origins_t03) <- ORIGINS_03[[1]]$species
rm(ORIGINS_03)
origins_03 <- apply(origins_t03, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t03, origins_03, file = "results_simmap_origin_t03.RData")
rm(simmap_t03, origins_t03, origins_03,
   simmap_hmm_ard_ard_ctard_t3, 
   simmap_hmm_ard_sym_ctard_t3,
   simmap_hmm_sym_sym_ctard_t3,
   simmap_hmm_ard_ard_ctsym_t3,
   simmap_hmm_ard_sym_ctsym_t3,
   simmap_hmm_sym_sym_ctsym_t3)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.4 Stochastic character mapping on tree cluster 3
########################################################

# 4.4.1 smm_ard
simmap_smm_ard_t4 <- makeSimmap(tree = tree_list[[4]], data = b0, 
                                model =  unc_smm_ard[[4]]$solution,
                                rate.cat = 1, nSim = rounding[4,1],
                                nCores = 7)

# 4.4.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.4.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t4 <- makeSimmap(tree = tree_list[[4]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[4]]$solution,
                                          rate.cat = 2, nSim = rounding[4,3],
                                          nCores = 7)

# 4.4.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t4 <- makeSimmap(tree = tree_list[[4]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[4]]$solution,
                                          rate.cat = 2, nSim = rounding[4,4],
                                          nCores = 7)

# 4.4.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t4 <- makeSimmap(tree = tree_list[[4]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[4]]$solution,
                                          rate.cat = 2, nSim = rounding[4,5],
                                          nCores = 7)

# 4.4.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t4 <- makeSimmap(tree = tree_list[[4]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[4]]$solution,
                                          rate.cat = 2, nSim = rounding[4,6],
                                          nCores = 7)

# 4.4.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t4 <- makeSimmap(tree = tree_list[[4]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[4]]$solution,
                                          rate.cat = 2, nSim = rounding[4,7],
                                          nCores = 7)

# 4.4.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t4 <- makeSimmap(tree = tree_list[[4]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[4]]$solution,
                                          rate.cat = 2, nSim = rounding[4,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t4, 
                 simmap_hmm_ard_ard_ctard_t4,
                 simmap_hmm_ard_sym_ctard_t4, 
                 simmap_hmm_sym_sym_ctard_t4,
                 simmap_hmm_ard_ard_ctsym_t4, 
                 simmap_hmm_ard_sym_ctsym_t4,
                 simmap_hmm_sym_sym_ctsym_t4)
simmap_t04<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t04) <- c("multiSimmap","multiPhylo") 

# 4.4.9 Classify species according to their ancestors
ORIGINS_04 <- list(length(simmap_t04)) 
for(i in 1:length(simmap_t04)) ORIGINS_04[[i]] <- detect_classify(simmap_t04[[i]], detect_origin)
beep(5)

origins_t04 <- do.call(cbind, ORIGINS_04)[, seq(1, 2000, 2)]
row.names(origins_t04) <- ORIGINS_04[[1]]$species
rm(ORIGINS_04)
origins_04 <- apply(origins_t04, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t04, origins_04, file = "results_simmap_origin_t04.RData")
rm(simmap_t04, origins_t04, origins_04,
   simmap_smm_ard_t4,
   simmap_hmm_ard_ard_ctard_t4, 
   simmap_hmm_ard_sym_ctard_t4,
   simmap_hmm_sym_sym_ctard_t4,
   simmap_hmm_ard_ard_ctsym_t4,
   simmap_hmm_ard_sym_ctsym_t4,
   simmap_hmm_sym_sym_ctsym_t4)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.5 Stochastic character mapping on tree cluster 4
########################################################

# 4.5.1 smm_ard no simulation were done (AICc weigth < 0.0001)

# 4.5.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.5.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t5 <- makeSimmap(tree = tree_list[[5]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[5]]$solution,
                                          rate.cat = 2, nSim = rounding[5,3],
                                          nCores = 7)

# 4.5.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t5 <- makeSimmap(tree = tree_list[[5]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[5]]$solution,
                                          rate.cat = 2, nSim = rounding[5,4],
                                          nCores = 7)

# 4.5.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t5 <- makeSimmap(tree = tree_list[[5]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[5]]$solution,
                                          rate.cat = 2, nSim = rounding[5,5],
                                          nCores = 7)

# 4.5.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t5 <- makeSimmap(tree = tree_list[[5]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[5]]$solution,
                                          rate.cat = 2, nSim = rounding[5,6],
                                          nCores = 7)

# 4.5.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t5 <- makeSimmap(tree = tree_list[[5]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[5]]$solution,
                                          rate.cat = 2, nSim = rounding[5,7],
                                          nCores = 7)

# 4.5.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t5 <- makeSimmap(tree = tree_list[[5]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[5]]$solution,
                                          rate.cat = 2, nSim = rounding[5,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_hmm_ard_ard_ctard_t5,
                 simmap_hmm_ard_sym_ctard_t5, 
                 simmap_hmm_sym_sym_ctard_t5,
                 simmap_hmm_ard_ard_ctsym_t5, 
                 simmap_hmm_ard_sym_ctsym_t5,
                 simmap_hmm_sym_sym_ctsym_t5)
simmap_t05<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t05) <- c("multiSimmap","multiPhylo") 

# 4.5.9 Classify species according to their ancestors
ORIGINS_05 <- list(length(simmap_t05)) 
for(i in 1:length(simmap_t05)) ORIGINS_05[[i]] <- detect_classify(simmap_t05[[i]], detect_origin)
beep(5)

origins_t05 <- do.call(cbind, ORIGINS_05)[, seq(1, 2000, 2)]
row.names(origins_t05) <- ORIGINS_05[[1]]$species
rm(ORIGINS_05)
origins_05 <- apply(origins_t05, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t05, origins_05, file = "results_simmap_origin_t05.RData")
rm(simmap_t05, origins_t05, origins_05, 
   simmap_hmm_ard_ard_ctard_t5, 
   simmap_hmm_ard_sym_ctard_t5,
   simmap_hmm_sym_sym_ctard_t5,
   simmap_hmm_ard_ard_ctsym_t5,
   simmap_hmm_ard_sym_ctsym_t5,
   simmap_hmm_sym_sym_ctsym_t5)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.6 Stochastic character mapping on tree cluster 5
########################################################

# 4.6.1 smm_ard
simmap_smm_ard_t6 <- makeSimmap(tree = tree_list[[6]], data = b0, 
                                model =  unc_smm_ard[[6]]$solution,
                                rate.cat = 1, nSim = rounding[6,1],
                                nCores = 7)

# 4.6.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.6.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t6 <- makeSimmap(tree = tree_list[[6]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[6]]$solution,
                                          rate.cat = 2, nSim = rounding[6,3],
                                          nCores = 7)

# 4.6.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t6 <- makeSimmap(tree = tree_list[[6]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[6]]$solution,
                                          rate.cat = 2, nSim = rounding[6,4],
                                          nCores = 7)

# 4.6.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t6 <- makeSimmap(tree = tree_list[[6]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[6]]$solution,
                                          rate.cat = 2, nSim = rounding[6,5],
                                          nCores = 7)

# 4.6.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t6 <- makeSimmap(tree = tree_list[[6]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[6]]$solution,
                                          rate.cat = 2, nSim = rounding[6,6],
                                          nCores = 7)

# 4.6.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t6 <- makeSimmap(tree = tree_list[[6]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[6]]$solution,
                                          rate.cat = 2, nSim = rounding[6,7],
                                          nCores = 7)

# 4.6.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t6 <- makeSimmap(tree = tree_list[[6]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[6]]$solution,
                                          rate.cat = 2, nSim = rounding[6,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t6, 
                 simmap_hmm_ard_ard_ctard_t6,
                 simmap_hmm_ard_sym_ctard_t6, 
                 simmap_hmm_sym_sym_ctard_t6,
                 simmap_hmm_ard_ard_ctsym_t6, 
                 simmap_hmm_ard_sym_ctsym_t6,
                 simmap_hmm_sym_sym_ctsym_t6)
simmap_t06<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t06) <- c("multiSimmap","multiPhylo") 

# 4.6.9 Classify species according to their ancestors
ORIGINS_06 <- list(length(simmap_t06)) 
for(i in 1:length(simmap_t06)) ORIGINS_06[[i]] <- detect_classify(simmap_t06[[i]], detect_origin)
beep(5)

origins_t06 <- do.call(cbind, ORIGINS_06)[, seq(1, 2000, 2)]
row.names(origins_t06) <- ORIGINS_06[[1]]$species
rm(ORIGINS_06)
origins_06 <- apply(origins_t06, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t06, origins_06, file = "results_simmap_origin_t06.RData")
rm(simmap_t06, origins_t06, origins_06,
   simmap_smm_ard_t6,
   simmap_hmm_ard_ard_ctard_t6, 
   simmap_hmm_ard_sym_ctard_t6,
   simmap_hmm_sym_sym_ctard_t6,
   simmap_hmm_ard_ard_ctsym_t6,
   simmap_hmm_ard_sym_ctsym_t6,
   simmap_hmm_sym_sym_ctsym_t6)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.7 Stochastic character mapping on tree cluster 6
########################################################

# 4.7.1 smm_ard no simulation were done (AICc weigth < 0.0001)

# 4.7.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.7.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t7 <- makeSimmap(tree = tree_list[[7]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[7]]$solution,
                                          rate.cat = 2, nSim = rounding[7,3],
                                          nCores = 7)

# 4.7.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t7 <- makeSimmap(tree = tree_list[[7]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[7]]$solution,
                                          rate.cat = 2, nSim = rounding[7,4],
                                          nCores = 7)

# 4.7.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t7 <- makeSimmap(tree = tree_list[[7]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[7]]$solution,
                                          rate.cat = 2, nSim = rounding[7,5],
                                          nCores = 7)

# 4.7.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t7 <- makeSimmap(tree = tree_list[[7]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[7]]$solution,
                                          rate.cat = 2, nSim = rounding[7,6],
                                          nCores = 7)

# 4.7.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t7 <- makeSimmap(tree = tree_list[[7]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[7]]$solution,
                                          rate.cat = 2, nSim = rounding[7,7],
                                          nCores = 7)

# 4.7.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t7 <- makeSimmap(tree = tree_list[[7]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[7]]$solution,
                                          rate.cat = 2, nSim = rounding[7,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_hmm_ard_ard_ctard_t7,
                 simmap_hmm_ard_sym_ctard_t7, 
                 simmap_hmm_sym_sym_ctard_t7,
                 simmap_hmm_ard_ard_ctsym_t7, 
                 simmap_hmm_ard_sym_ctsym_t7,
                 simmap_hmm_sym_sym_ctsym_t7)
simmap_t07<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t07) <- c("multiSimmap","multiPhylo") 

# 4.5.9 Classify species according to their ancestors
ORIGINS_07 <- list(length(simmap_t07)) 
for(i in 1:length(simmap_t07)) ORIGINS_07[[i]] <- detect_classify(simmap_t07[[i]], detect_origin)
beep(5)

origins_t07 <- do.call(cbind, ORIGINS_07)[, seq(1, 2000, 2)]
row.names(origins_t07) <- ORIGINS_07[[1]]$species
rm(ORIGINS_07)
origins_07 <- apply(origins_t07, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t07, origins_07, file = "results_simmap_origin_t07.RData")
rm(simmap_t07, origins_t07, origins_07, 
   simmap_hmm_ard_ard_ctard_t7, 
   simmap_hmm_ard_sym_ctard_t7,
   simmap_hmm_sym_sym_ctard_t7,
   simmap_hmm_ard_ard_ctsym_t7,
   simmap_hmm_ard_sym_ctsym_t7,
   simmap_hmm_sym_sym_ctsym_t7)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.8 Stochastic character mapping on tree cluster 7
########################################################

# 4.8.1 smm_ard
simmap_smm_ard_t8 <- makeSimmap(tree = tree_list[[8]], data = b0, 
                                model =  unc_smm_ard[[8]]$solution,
                                rate.cat = 1, nSim = rounding[8,1],
                                nCores = 7)

# 4.8.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.8.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t8 <- makeSimmap(tree = tree_list[[8]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[8]]$solution,
                                          rate.cat = 2, nSim = rounding[8,3],
                                          nCores = 7)

# 4.8.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t8 <- makeSimmap(tree = tree_list[[8]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[8]]$solution,
                                          rate.cat = 2, nSim = rounding[8,4],
                                          nCores = 7)

# 4.8.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t8 <- makeSimmap(tree = tree_list[[8]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[8]]$solution,
                                          rate.cat = 2, nSim = rounding[8,5],
                                          nCores = 7)

# 4.8.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t8 <- makeSimmap(tree = tree_list[[8]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[8]]$solution,
                                          rate.cat = 2, nSim = rounding[8,6],
                                          nCores = 7)

# 4.8.8 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t8 <- makeSimmap(tree = tree_list[[8]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[8]]$solution,
                                          rate.cat = 2, nSim = rounding[8,7],
                                          nCores = 7)

# 4.8.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t8 <- makeSimmap(tree = tree_list[[8]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[8]]$solution,
                                          rate.cat = 2, nSim = rounding[8,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t8, 
                 simmap_hmm_ard_ard_ctard_t8,
                 simmap_hmm_ard_sym_ctard_t8, 
                 simmap_hmm_sym_sym_ctard_t8,
                 simmap_hmm_ard_ard_ctsym_t8, 
                 simmap_hmm_ard_sym_ctsym_t8,
                 simmap_hmm_sym_sym_ctsym_t8)
simmap_t08<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t08) <- c("multiSimmap","multiPhylo") 

# 4.8.9 Classify species according to their ancestors
ORIGINS_08 <- list(length(simmap_t08)) 
for(i in 1:length(simmap_t08)) ORIGINS_08[[i]] <- detect_classify(simmap_t08[[i]], detect_origin)
beep(5)

origins_t08 <- do.call(cbind, ORIGINS_08)[, seq(1, 2000, 2)]
row.names(origins_t08) <- ORIGINS_08[[1]]$species
rm(ORIGINS_08)
origins_08 <- apply(origins_t08, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t08, origins_08, file = "results_simmap_origin_t08.RData")
rm(simmap_t08, origins_t08, origins_08,
   simmap_smm_ard_t8,
   simmap_hmm_ard_ard_ctard_t8, 
   simmap_hmm_ard_sym_ctard_t8,
   simmap_hmm_sym_sym_ctard_t8,
   simmap_hmm_ard_ard_ctsym_t8,
   simmap_hmm_ard_sym_ctsym_t8,
   simmap_hmm_sym_sym_ctsym_t8)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.9 Stochastic character mapping on tree cluster 8
########################################################

# 4.9.1 smm_ard
simmap_smm_ard_t9 <- makeSimmap(tree = tree_list[[9]], data = b0, 
                                model =  unc_smm_ard[[9]]$solution,
                                rate.cat = 1, nSim = rounding[9,1],
                                nCores = 7)

# 4.9.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.9.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t9 <- makeSimmap(tree = tree_list[[9]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[9]]$solution,
                                          rate.cat = 2, nSim = rounding[9,3],
                                          nCores = 7)

# 4.9.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t9 <- makeSimmap(tree = tree_list[[9]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[9]]$solution,
                                          rate.cat = 2, nSim = rounding[9,4],
                                          nCores = 7)

# 4.9.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t9 <- makeSimmap(tree = tree_list[[9]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[9]]$solution,
                                          rate.cat = 2, nSim = rounding[9,5],
                                          nCores = 7)

# 4.9.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t9 <- makeSimmap(tree = tree_list[[9]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[9]]$solution,
                                          rate.cat = 2, nSim = rounding[9,6],
                                          nCores = 7)

# 4.9.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t9 <- makeSimmap(tree = tree_list[[9]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[9]]$solution,
                                          rate.cat = 2, nSim = rounding[9,7],
                                          nCores = 7)

# 4.9.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t9 <- makeSimmap(tree = tree_list[[9]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[9]]$solution,
                                          rate.cat = 2, nSim = rounding[9,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t9, 
                 simmap_hmm_ard_ard_ctard_t9,
                 simmap_hmm_ard_sym_ctard_t9, 
                 simmap_hmm_sym_sym_ctard_t9,
                 simmap_hmm_ard_ard_ctsym_t9, 
                 simmap_hmm_ard_sym_ctsym_t9,
                 simmap_hmm_sym_sym_ctsym_t9)
simmap_t09<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t09) <- c("multiSimmap","multiPhylo") 

# 4.9.9 Classify species according to their ancestors
ORIGINS_09 <- list(length(simmap_t09)) 
for(i in 1:length(simmap_t09)) ORIGINS_09[[i]] <- detect_classify(simmap_t09[[i]], detect_origin)
beep(5)

origins_t09 <- do.call(cbind, ORIGINS_09)[, seq(1, 2000, 2)]
row.names(origins_t09) <- ORIGINS_09[[1]]$species
rm(ORIGINS_09)
origins_09 <- apply(origins_t09, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t09, origins_09, file = "results_simmap_origin_t09.RData")
rm(simmap_t09, origins_t09, origins_09,
   simmap_smm_ard_t9,
   simmap_hmm_ard_ard_ctard_t9, 
   simmap_hmm_ard_sym_ctard_t9,
   simmap_hmm_sym_sym_ctard_t9,
   simmap_hmm_ard_ard_ctsym_t9,
   simmap_hmm_ard_sym_ctsym_t9,
   simmap_hmm_sym_sym_ctsym_t9)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.10 Stochastic character mapping on tree cluster 9
########################################################

# 4.10.1 smm_ard
simmap_smm_ard_t10 <- makeSimmap(tree = tree_list[[10]], data = b0, 
                                model =  unc_smm_ard[[10]]$solution,
                                rate.cat = 1, nSim = rounding[10,1],
                                nCores = 7)

# 4.10.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.10.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t10 <- makeSimmap(tree = tree_list[[10]], data = b0, 
                                          model = unc_hmm_ard_ard_ctard[[10]]$solution,
                                          rate.cat = 2, nSim = rounding[10,3],
                                          nCores = 7)
# 4.10.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t10 <- makeSimmap(tree = tree_list[[10]], data = b0, 
                                          model = unc_hmm_ard_sym_ctard[[10]]$solution,
                                          rate.cat = 2, nSim = rounding[10,4],
                                          nCores = 7)

# 4.10.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t10 <- makeSimmap(tree = tree_list[[10]], data = b0, 
                                          model = unc_hmm_sym_sym_ctard[[10]]$solution,
                                          rate.cat = 2, nSim = rounding[10,5],
                                          nCores = 7)

# 4.10.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t10 <- makeSimmap(tree = tree_list[[10]], data = b0, 
                                          model = unc_hmm_ard_ard_ctsym[[10]]$solution,
                                          rate.cat = 2, nSim = rounding[10,6],
                                          nCores = 7)

# 4.10.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t10 <- makeSimmap(tree = tree_list[[10]], data = b0, 
                                          model = unc_hmm_ard_sym_ctsym[[10]]$solution,
                                          rate.cat = 2, nSim = rounding[10,7],
                                          nCores = 7)

# 4.10.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t10 <- makeSimmap(tree = tree_list[[10]], data = b0, 
                                          model = unc_hmm_sym_sym_ctsym[[10]]$solution,
                                          rate.cat = 2, nSim = rounding[10,8],
                                          nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t10, 
                 simmap_hmm_ard_ard_ctard_t10,
                 simmap_hmm_ard_sym_ctard_t10, 
                 simmap_hmm_sym_sym_ctard_t10,
                 simmap_hmm_ard_ard_ctsym_t10, 
                 simmap_hmm_ard_sym_ctsym_t10,
                 simmap_hmm_sym_sym_ctsym_t10)
simmap_t10<- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t10) <- c("multiSimmap","multiPhylo") 

# 4.10.9 Classify species according to their ancestors
ORIGINS_10 <- list(length(simmap_t10)) 
for(i in 1:length(simmap_t10)) ORIGINS_10[[i]] <- detect_classify(simmap_t10[[i]], detect_origin)
beep(5)

origins_t10 <- do.call(cbind, ORIGINS_10)[, seq(1, 2000, 2)]
row.names(origins_t10) <- ORIGINS_10[[1]]$species
rm(ORIGINS_10)
origins_10 <- apply(origins_t10, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t10, origins_10, file = "results_simmap_origin_t10.RData")
rm(simmap_t10, origins_t10, origins_10,
   simmap_smm_ard_t10,
   simmap_hmm_ard_ard_ctard_t10, 
   simmap_hmm_ard_sym_ctard_t10,
   simmap_hmm_sym_sym_ctard_t10,
   simmap_hmm_ard_ard_ctsym_t10,
   simmap_hmm_ard_sym_ctsym_t10,
   simmap_hmm_sym_sym_ctsym_t10)
gc()
#-------------------------------------------------------------------------------

########################################################
# 4.11 Stochastic character mapping on tree cluster 10
########################################################

# 4.11.1 smm_ard
simmap_smm_ard_t11 <- makeSimmap(tree = tree_list[[11]], data = b0, 
                                 model =  unc_smm_ard[[11]]$solution,
                                 rate.cat = 1, nSim = rounding[11,1],
                                 nCores = 7)

# 4.11.2 smm_sym no simulation were done (AICc weigth < 0.0001)

# 4.11.3 hmm ard ard ctard
simmap_hmm_ard_ard_ctard_t11 <- makeSimmap(tree = tree_list[[11]], data = b0, 
                                           model = unc_hmm_ard_ard_ctard[[11]]$solution,
                                           rate.cat = 2, nSim = rounding[11,3],
                                           nCores = 7)
# 4.11.4 hmm ard sym ctard
simmap_hmm_ard_sym_ctard_t11 <- makeSimmap(tree = tree_list[[11]], data = b0, 
                                           model = unc_hmm_ard_sym_ctard[[11]]$solution,
                                           rate.cat = 2, nSim = rounding[11,4],
                                           nCores = 7)

# 4.11.5 hmm sym sym ctard
simmap_hmm_sym_sym_ctard_t11 <- makeSimmap(tree = tree_list[[11]], data = b0, 
                                           model = unc_hmm_sym_sym_ctard[[11]]$solution,
                                           rate.cat = 2, nSim = rounding[11,5],
                                           nCores = 7)

# 4.11.6 hmm ard ard ctsym
simmap_hmm_ard_ard_ctsym_t11 <- makeSimmap(tree = tree_list[[11]], data = b0, 
                                           model = unc_hmm_ard_ard_ctsym[[11]]$solution,
                                           rate.cat = 2, nSim = rounding[11,6],
                                           nCores = 7)

# 4.11.7 hmm ard sym ctsym
simmap_hmm_ard_sym_ctsym_t11 <- makeSimmap(tree = tree_list[[11]], data = b0, 
                                           model = unc_hmm_ard_sym_ctsym[[11]]$solution,
                                           rate.cat = 2, nSim = rounding[11,7],
                                           nCores = 7)

# 4.11.8 hmm sym sym ctsym
simmap_hmm_sym_sym_ctsym_t11 <- makeSimmap(tree = tree_list[[11]], data = b0, 
                                           model = unc_hmm_sym_sym_ctsym[[11]]$solution,
                                           rate.cat = 2, nSim = rounding[11,8],
                                           nCores = 7)
beep(5)

simmap_0 <- list(simmap_smm_ard_t11, 
                 simmap_hmm_ard_ard_ctard_t11,
                 simmap_hmm_ard_sym_ctard_t11, 
                 simmap_hmm_sym_sym_ctard_t11,
                 simmap_hmm_ard_ard_ctsym_t11, 
                 simmap_hmm_ard_sym_ctsym_t11,
                 simmap_hmm_sym_sym_ctsym_t11)
simmap_t11 <- unlist(simmap_0, recursive = F)
rm(simmap_0)
class(simmap_t11) <- c("multiSimmap","multiPhylo") 

# 4.11.9 Classify species according to their ancestors
ORIGINS_11 <- list(length(simmap_t11)) 
for(i in 1:length(simmap_t11)) ORIGINS_11[[i]] <- detect_classify(simmap_t11[[i]], detect_origin)
beep(5)

origins_t11 <- do.call(cbind, ORIGINS_11)[, seq(1, 2000, 2)]
row.names(origins_t11) <- ORIGINS_11[[1]]$species
rm(ORIGINS_11)
origins_11 <- apply(origins_t11, 2, function(x) as.numeric(gsub("from_bee", 0, gsub("from_bird", 1, x))))

#-------------------------------------------------------------------------------
save(origins_t11, origins_11, file = "results_simmap_origin_t11.RData")
rm(simmap_t11, origins_t11, origins_11,
   simmap_smm_ard_t11,
   simmap_hmm_ard_ard_ctard_t11, 
   simmap_hmm_ard_sym_ctard_t11,
   simmap_hmm_sym_sym_ctard_t11,
   simmap_hmm_ard_ard_ctsym_t11,
   simmap_hmm_ard_sym_ctsym_t11,
   simmap_hmm_sym_sym_ctsym_t11)
gc()
#-------------------------------------------------------------------------------

