
### R script for analyses in Wieczynski et al. 2022, Ecography

# Clear data cache
rm(list=ls())


### Define data directory and load packages/functions/data -----

data_dir<-"~/Wieczynski_etal_2022_Ecography_Master/"

packages = c("tidyverse","raster", "RColorBrewer","colorRamps","scales", "ggmap", "maps", "car", "yhat", 
             "mapdata", "ggnewscale", "colorspace", "gtools", "data.table", "rgdal", "rhdf5") 

package.check <- lapply(packages, FUN = function(x) {
  if (!require(x, character.only = TRUE)) {
    install.packages(x, dependencies = TRUE)
    library(x, character.only = TRUE)
  }
})


# photosynthesis eqns
Amax<-function(P){10.25 + P*15.51}
K<-function(P,LMA,N){175.51 - 0.612*LMA - 104.53*N + 1412.23*P + 0.785*LMA*N - 8.395*LMA*P}
R<-function(P,LMA,N){-1.004 + 0.017*LMA +  2.363*N -   16.41*P - 0.015*LMA*N + 0.133*LMA*P} 


# load raw data
metadata_reduced<-fread(paste0(data_dir,"Data/ground_Metadata.csv"))
plot_list<-metadata_reduced$Plot
groundPlot_rawTreeData<-fread(paste0(data_dir,"Data/ground_RawTreeData.csv")) 
groundPlot_prod_rawdata<-fread(paste0(data_dir,"Data/ground_ProductivityOutput.csv"))

# load RS test data (These data are just for test running the RSCFM, they include pixel metadata and random trait data, but do not include the RS trait data used for the main analysis)
rs_rawdata<-fread(paste0(data_dir,"Data/rs_test_data.csv")) %>%
  rowwise %>%
  mutate(LMA=rnorm(1, 150, 20),
         N=rnorm(1, 20, 2),
         P=rnorm(1, 1.25, .4),
         Canopy_height=rnorm(1, 17.5, 5)) %>%
  ungroup


# ground plot summary
plot_BA_summary<-rename(groundPlot_rawTreeData, Plot=PlotCode) %>%
  filter(!Plot=="PAN_03") %>%
  mutate(BA=0.00007854*dbh^2, ba_WD=BA*WD) %>%
  group_by(Plot) %>%
  summarize(plot_BA=sum(BA), plot_WD=mean(WD), plot_BA_WD=sum(ba_WD)/sum(BA), n_trees=n())


# RS landscape regions
plot_regions<-data.frame(
  region_id=c(rep.int(1, 6), rep.int(2, 2), 3),
  Plot=c("WAY_01", "TRU_04", "SPD_02", "SPD_01", "ESP_01", "ACJ_01", "TAM_05", "TAM_06", "PAN_02")
)


#
### GROUND-BASED NPP estimates -----

tree_traits_1<-rename(groundPlot_rawTreeData, Plot=PlotCode) %>%
  filter(!Plot=="PAN_03") %>%
  left_join(metadata_reduced[, c("Plot", "MAT_C")]) %>%
  left_join(plot_BA_summary) %>%
  mutate(id=row_number(),
         Height=exp(1.51+log10(dbh)*0.84),
         M_a=0.0673*(WD*(dbh^2)*Height)^0.976,
         M_l=0.158*M_a^0.707,
         CA=0.695*dbh^1.305, 
         foliage_area=M_l*1000*(1/LMA),
         tree_lai=foliage_area/CA)

PPA_layers_1<-tree_traits_1 %>%
  group_by(Plot) %>%
  arrange(Plot, desc(Height)) %>%
  mutate(cumsum_CA=cumsum(CA), cumsum_FA=cumsum(foliage_area), cumsum_M_a=cumsum(M_a), cumsum_M_l=cumsum(M_l), 
         ppa_layer_1=floor(cumsum_CA/10000), ppa_layer_1=lag(ppa_layer_1, default=0)) 

tree_traits_2<-PPA_layers_1 %>%
  group_by(Plot) %>%
  mutate(n_layers=length(unique(ppa_layer_1)), total_lai=sum(foliage_area, na.rm=T)/10000) %>%
  rowwise %>%
  mutate(layer_lai=total_lai/n_layers, SLAI=ppa_layer_1*layer_lai)

tree_traits<-tree_traits_2

tree_prod<-tree_traits %>%
  left_join(metadata_reduced) %>%
  rowwise() %>%
  mutate(I_mol=0.75*0.48*4.6*(I_mean)*exp(-0.5*SLAI),
         N_area=N/1000*LMA,
         P_area=P/1000*LMA,
         k=K(P_area,LMA,N_area),
         r=R(P_area,LMA,N_area),
         c=rnorm(1, 0.33, 0.04),
         T_resp=-0.242+0.0937*MAT_C-0.00177*MAT_C^2,
         A=max(0, ((12.13*I_mol)/(k+I_mol)-r)), 
         dMdt=((0.33/0.5)*(A*0.0378*(12/24))*((1/LMA)*10000)*M_l)/1000*0.5, #  *0.0378 <- converts micromol/m^2/s to gC/cm^2/year
         dMdt_gross=dMdt/0.33)

tree_prod<-dplyr::select(tree_prod, Plot, Lon, Lat, x, y, id) %>%
  left_join(tree_prod)

tree_plot_prod<-tree_prod %>% 
  group_by(Plot, Lon, Lat, x, y) %>%
  summarize(CFM_NPP_ground=sum(dMdt, na.rm=T)) %>% 
  ungroup() %>%
  left_join(metadata_reduced) %>%
  dplyr::select(Plot, Elevation, GPP, NPP, CFM_NPP_ground) %>%
  rename(empirical_NPP=NPP, empirical_GPP=GPP) %>%
  gather(productivity_measure, value, -c(Plot, Elevation)) %>%
  mutate(dataset_id="ground") %>%
  dplyr::select(Plot, Elevation, dataset_id, productivity_measure, value)

tree_prod_trait_stats<-tree_prod[c("Plot", "LMA", "N", "P")] %>%
  group_by(Plot) %>%
  summarize(mean_LMA=mean(LMA), mean_N=mean(N), mean_P=mean(P))


## export raw ground-based productivity data files

# ground-based tree productivity
tree_prod_export<-tree_prod %>%
  dplyr::select(Plot, id, dbh, LMA, N_area, P_area, M_l, I_mol, k, r, c, A, dMdt)
fwrite(tree_prod_export, paste0(data_dir,"Data/ground_prod.csv"))

# ground-based plot productivity
fwrite(tree_plot_prod, paste0(data_dir,"Data/ground_plot_prod.csv"))



#
### REMOTE SENSING pixel growth and NPP estimates -----

## Choose RS data scale and resolution

data_scale="plot" # Can be called anything, e.g., "plot" or "landscape"
px_size=100 # Length (m) along one side of a square pixel. Must match resolution of RS data.
shading="RSCFM" # For normal RSCFM: set shading="RSCFM" | For average shading: shading="average"   |   For no shading: shading="none"


# record time (for timing computations)
ptm<-proc.time()

## Ground data for rs estimates

# for ground-based canopy structure
ground_layer_func<-function(plot){filter(tree_traits_2_forRS, Plot==plot)$Height}
rs_layer_func<-function(plot){filter(aggregate_biomass_px, Plot==plot)$Canopy_height}

tree_traits_1_forRS<-rename(groundPlot_rawTreeData, Plot=PlotCode) %>%
  filter(!Plot=="PAN_03") %>%
  left_join(metadata_reduced[, .(Plot, MAT_C)]) %>%
  left_join(plot_BA_summary) %>%
  mutate(id=row_number(),
         Height=exp(2.5182 + 0.0971 + 0.012*(plot_BA-32.4) + 0.336*(0.48-0.6) - 0.0449*(3.1-3.7) + 0.0191*(MAT_C-24.7) + (0.5736-0.107)*log(dbh/10)),
         M_a=0.0673*(WD*(dbh^2)*Height)^0.976,
         M_l=0.158*M_a^0.707,
         CA=0.695*dbh^1.305, 
         foliage_area=M_l*1000*(1/LMA),
         tree_lai=foliage_area/CA) %>%
  filter(!id %in% c(4721,4790,3880,189,862,3441,2735)) %>%
  group_by(Plot) %>%
  mutate(Height_rescaled_01=rescale(Height), Height_rescaled_PercentRank=percent_rank(Height), Height_rescaled_CumeDist=cume_dist(Height)) %>%
  ungroup() %>%
  left_join(metadata_reduced)

PPA_layers_1_forRS<-tree_traits_1_forRS %>%
  group_by(Plot) %>%
  arrange(Plot, desc(Height)) %>%
  mutate(cumsum_CA=cumsum(CA), 
         ppa_layer_1=floor(cumsum_CA/10000), ppa_layer_1=lag(ppa_layer_1, default=0))


tree_traits_2_forRS<-PPA_layers_1_forRS %>%
  group_by(Plot) %>%
  arrange(Plot, Height) %>%
  mutate(n_layers=length(unique(ppa_layer_1)), total_lai=sum(foliage_area, na.rm=T)/10000,
         Height_percentile=cume_dist(Height),
         cum_Ma=cumsum(M_a), cum_Ml=cumsum(M_l)) %>%
  rowwise() %>%
  mutate(layer_lai=total_lai/n_layers, SLAI=ppa_layer_1*layer_lai)


avg_slai_forRS<-tree_traits_2_forRS %>%
  group_by(Plot) %>%
  summarize(avg_SLAI=mean(SLAI, na.rm=T))


min_heights_forRS<-tree_traits_2_forRS %>%
  group_by(Plot) %>%
  summarize(min_h=min(Height, na.rm=T)) %>%
  mutate(min_h=floor(min_h))


for(plot in unique(tree_traits_2_forRS$Plot)){
  assign(paste0("heights_", plot), filter(tree_traits_2_forRS, Plot==plot)$Height)
}

# calculate total M_l up to a certian height
Ma_Ml_data_forRS_1<-tree_traits_1_forRS %>%
  group_by(Plot) %>%
  arrange(Plot, Height) %>%
  mutate(M_a_plot=sum(M_a), M_l_plot=sum(M_l), cumsum_n_trees=row_number(), cumsum_M_a=cumsum(M_a), cumsum_M_l=cumsum(M_l),
         cum_prop_M_a=cumsum_M_a/M_a_plot, cum_prop_M_l=cumsum_M_l/M_l_plot, cum_prop_Ma_Ml=cumsum_M_l/cumsum_M_a) %>%
  ungroup()

stats_lm_propMl_h<-Ma_Ml_data_forRS_1 %>%
  group_by(Plot, Elevation) %>% 
  do(fit = lm(log10(cum_prop_Ma_Ml) ~ log10(Height), data = ., na.action=na.omit)) %>%
  mutate(r2=round(summary(fit)$r.squared,5),p_val=round(pf(summary(fit)$fstatistic[1], summary(fit)$fstatistic[2], summary(fit)$fstatistic[3], lower=FALSE)[[1]],5), slope=data.frame(fit[1])[2,1], intercept=data.frame(fit[1])[1,1]) %>%
  dplyr::select(-fit) %>% 
  rowwise() %>%
  mutate(sig=ifelse(p_val<=0.05, "*", "ns")) %>%
  data.frame()

propMl_func<-function(x, plot_id){
  line_params<-filter(stats_lm_propMl_h, Plot==plot_id)
  slope<-line_params[[5]]; intercept<-line_params[[6]]
  predicted_val<-slope*log10(x)+intercept
  val<-10^(predicted_val)
  return(val)
}


# for M_l within each 1m slice of a pixel 
Ma_Ml_data_forRS_2<-tree_traits_1_forRS %>%
  group_by(Plot) %>%
  arrange(Plot, Height) %>%
  mutate(M_a_plot=sum(M_a), M_l_plot=sum(M_l),
         cume_dist_M_a=cume_dist(M_a), cume_dist_M_l=cume_dist(M_l)) %>%
  rowwise() %>%
  mutate(cumsum_M_a=cume_dist_M_a*M_a_plot, cumsum_M_l=cume_dist_M_l*M_l_plot)

min_max_h_df<-Ma_Ml_data_forRS_2 %>%
  group_by(Plot) %>%
  summarize(min_h=min(Height, na.rm=T), max_h=max(Height, na.rm=T)) %>%
  mutate(min_h=floor(min_h), max_h=ceiling(max_h))

trees_px<-tree_traits_2_forRS %>%
  group_by(Plot) %>%
  summarize(n_trees=n()) %>%
  mutate(n_trees_px=n_trees/(10000/px_size^2))

ground_biomass_summary<-Ma_Ml_data_forRS_2 %>%
  dplyr::select(Plot, Elevation, M_a_plot, M_l_plot) %>%
  unique %>%
  mutate(Ml_Ma_ratio=M_l_plot/M_a_plot,
         M_a_px_ground=M_a_plot/(10000/px_size^2), M_l_px_ground=M_l_plot/(10000/px_size^2))

# combine logistic functions below
logistic_Ma_func<-function(x, plot_id){
  total_Ma<-filter(ground_biomass_summary, Plot==plot_id)$M_a_plot
  predicted_func<-ecdf(eval(parse(text=paste0("heights_", plot_id))))
  return(predicted_func(x)*total_Ma)
}

logistic_Ml_func<-function(x, plot_id){
  total_Ml<-filter(ground_biomass_summary, Plot==plot_id)$M_l_plot
  predicted_func<-ecdf(eval(parse(text=paste0("heights_", plot_id))))
  return(predicted_func(x)*total_Ml)
}

propM_pixel_func_percentile<-function(px_pct, plot_id){
  Ma_plot<-filter(ground_biomass_summary, Plot==plot_id)$M_a_plot
  Ml_plot<-filter(ground_biomass_summary, Plot==plot_id)$M_l_plot
  px_h_val<-quantile(eval(parse(text=paste0("heights_", plot_id))), px_pct, na.rm=T, type=4)[[1]]
  temp_max_Ma<-logistic_Ma_func(px_h_val, plot_id)
  temp_max_Ml<-logistic_Ml_func(px_h_val, plot_id)
  return(c(px_h_val, temp_max_Ma/Ma_plot, temp_max_Ml/temp_max_Ma))
}


## RS productivity estimates for each plot

rs_prod<-data.frame()
rs_px_prod<-data.frame()

for(temp_plot in plot_list){
  # temp_plot<-plot_list[[2]]
  
  rs_trait_data_p_raw<-rs_rawdata %>%
    data.table() %>%
    filter(Plot==temp_plot)
  
  rs_trait_data_p<-rs_trait_data_p_raw %>%
    mutate(id=row_number(), original_CH=Canopy_height, Canopy_height=round(Canopy_height)) %>%
    left_join(min_heights_forRS) %>%
    filter(Canopy_height>=min_h)  %>%
    group_by(Plot) %>%
    mutate(Canopy_height_rescaled_CumeDist=cume_dist(Canopy_height)) %>%
    group_by(Plot) %>%
    mutate(n_ha=length(unique(xy_ha_id))) %>%
    ungroup()
  
  rm(rs_trait_data_p_raw)
  
  n_ha_temp<-unique(rs_trait_data_p$n_ha)
  
  px_count<-rs_trait_data_p %>% 
    group_by(Plot) %>%
    summarize(n_px=n(), px_mult=(10000/px_size^2*n_ha_temp)/n_px, px_div=n_px/(10000/px_size^2*n_ha_temp))
  
  aggregate_biomass_px<-rs_trait_data_p %>% 
    left_join(min_max_h_df) %>%
    filter(Canopy_height>=min_h) %>%
    group_by(Plot) %>%
    mutate(max_CH=max(Canopy_height), total_CH=sum(Canopy_height-min_h+1),  
           px_h_quant=(Canopy_height-min_h)/(max_CH-min_h)
    ) %>%
    ungroup() %>%
    left_join(ground_biomass_summary) %>%
    left_join(within(metadata_reduced, rm(Lon, Lat, x, y))) %>%
    left_join(px_count) %>%
    rowwise() %>%
    mutate(
      agg_px_BA=1.5056*Canopy_height,
      agg_px_WD=ifelse(Elevation_px<=400, 0.5881-0.0037*Canopy_height, 0.5881-0.0037*Canopy_height+((Elevation_px-400)*0.00005)), 
      M_a_px=3.8358*Canopy_height^0.2807*agg_px_BA^0.9721*agg_px_WD^1.3763/0.48*1000/(10000/px_size^2), 
      M_l_px=M_a_px*Ml_Ma_ratio
    ) %>%
    ungroup()
  
  rm(rs_trait_data_p)
  
  # leaf biomass distribution across slices
  
  # can combine the propMa/Ml functions below
  propMa_slice_func_percentile<-function(px_pct, slice_pct_1, slice_pct_2, plot_id){
    px_h_val<-quantile(eval(parse(text=paste0("heights_", plot_id))), px_pct, na.rm=T, type=4)[[1]]
    slice_h_val_1<-quantile(eval(parse(text=paste0("heights_", plot_id))), slice_pct_1, na.rm=T, type=4)[[1]]
    slice_h_val_2<-quantile(eval(parse(text=paste0("heights_", plot_id))), slice_pct_2, na.rm=T, type=4)[[1]]
    temp_max_Ma<-logistic_Ma_func(px_h_val, plot_id)
    temp_Ma<-ifelse(slice_pct_2==0,
                    logistic_Ma_func(slice_h_val_1, plot_id),
                    logistic_Ma_func(slice_h_val_1, plot_id)-logistic_Ma_func(slice_h_val_2, plot_id))
    return(temp_Ma/temp_max_Ma)
  }
  
  propMl_slice_func_percentile<-function(px_pct, slice_pct_1, slice_pct_2, plot_id){
    px_h_val<-quantile(eval(parse(text=paste0("heights_", plot_id))), px_pct, na.rm=T, type=4)[[1]]
    slice_h_val_1<-quantile(eval(parse(text=paste0("heights_", plot_id))), slice_pct_1, na.rm=T, type=4)[[1]]
    slice_h_val_2<-quantile(eval(parse(text=paste0("heights_", plot_id))), slice_pct_2, na.rm=T, type=4)[[1]]
    temp_max_Ma<-logistic_Ma_func(px_h_val, plot_id)
    temp_max_Ml<-logistic_Ml_func(px_h_val, plot_id)
    temp_Ma<-ifelse(slice_pct_2==0,
                    logistic_Ma_func(slice_h_val_1, plot_id),
                    logistic_Ma_func(slice_h_val_1, plot_id)-logistic_Ma_func(slice_h_val_2, plot_id))
    temp_Ml<-ifelse(slice_pct_2==0,
                    logistic_Ml_func(slice_h_val_1, plot_id),
                    logistic_Ml_func(slice_h_val_1, plot_id)-logistic_Ml_func(slice_h_val_2, plot_id))
    return(temp_Ml/temp_Ma)
  }
  
  # LAI distribution across slices
  
  stats_nls1<-tree_traits_2_forRS[c("Plot", "Height", "SLAI")] %>% 
    group_by(Plot) %>%
    do(fit_slai=nls(SLAI ~ SSasymp(Height, Asym, R0, lrc), data=., na.action=na.omit)) %>%
    mutate(Asym=coef(fit_slai)[[1]], R0=coef(fit_slai)[[2]], lrc=coef(fit_slai)[[3]]) %>%
    dplyr::select(-fit_slai) %>%
    data.frame()
  
  slai_func<-function(x, plot_id){
    curve_params<-filter(stats_nls1, Plot==plot_id)
    Asym<-curve_params[[2]]; R0<-curve_params[[3]]; lrc<-curve_params[[4]];
    predicted_val<-Asym+(R0-Asym)*exp(-exp(lrc)*x)
    val<-ifelse(predicted_val<=0, 0, predicted_val)
    return(val)
  }
  
  
  # continuous version
  slai_func_percentile<-function(x, plot_id){
    curve_params<-filter(stats_nls1, Plot==plot_id)
    Asym<-curve_params[[2]]; R0<-curve_params[[3]]; lrc<-curve_params[[4]];
    h_val<-quantile(eval(parse(text=paste0("heights_", plot_id))), x, na.rm=T, type=4)[[1]]
    predicted_val<-Asym+(R0-Asym)*exp(-exp(lrc)*h_val)
    val<-ifelse(predicted_val<=0, 0, predicted_val)
    return(val)
  }
  
  # LAI and biomass slice dataframes
  
  rs_slice_base_df<-aggregate_biomass_px %>%
    mutate(CH_rounded=round(Canopy_height)) %>%
    group_by(Plot)
  
  rm(aggregate_biomass_px)
  
  rs_slice_df_1<-rs_slice_base_df %>% 
    mutate(min_CH=min_h, max_CH=max(CH_rounded)) %>% 
    rowwise() %>%
    mutate(n_slices=CH_rounded+1-min_CH) %>%
    filter(n_slices>0) %>%
    uncount(n_slices, .remove=F) %>%
    group_by(Plot, id) %>%
    mutate(CH_slice=row_number()+min_CH-1) %>%
    ungroup()
  
  rs_slice_quantiles_SLAI<-rs_slice_df_1[c("Plot", "CH_rounded", "max_CH", "CH_slice")] %>%  
    unique() %>%
    group_by(Plot, CH_rounded, max_CH) %>%
    mutate(CH_slice_2=max_CH-(CH_rounded-CH_slice), CH_slice_quantile_SLAI=round(CH_slice_2/max_CH, 3)) %>%
    ungroup() %>%
    dplyr::select(Plot, CH_rounded, CH_slice, CH_slice_quantile_SLAI)
  
  rs_slice_quantiles<-rs_slice_base_df %>%
    summarize(min_CH=unique(min_h), max_CH=max(CH_rounded)) %>%
    rowwise() %>%
    mutate(n_slices=max_CH+1-min_CH) %>% 
    filter(n_slices>0) %>%
    uncount(n_slices, .remove=F) %>%
    group_by(Plot) %>%
    mutate(CH_slice=row_number()+min_CH-1,
           CH_slice_quantile_1=cume_dist(CH_slice),
           CH_slice_quantile_2=lag(CH_slice_quantile_1, default=0)) %>%
    dplyr::select(Plot, CH_slice, CH_slice_quantile_1, CH_slice_quantile_2)
  
  rm(rs_slice_base_df)
  
  rs_slices_df<-rs_slice_df_1 %>%
    left_join(dplyr::select(rename(rs_slice_quantiles, CH_rounded=CH_slice, CH_quantile=CH_slice_quantile_1), -CH_slice_quantile_2)) %>%
    left_join(rs_slice_quantiles) %>%
    left_join(rs_slice_quantiles_SLAI)
  
  rm(rs_slice_df_1)
  rm(rs_slice_quantiles_SLAI)
  rm(rs_slice_quantiles)
  
  rs_SLAI_slices_unique<-rs_slices_df[c("Plot", "CH_slice_quantile_SLAI")] %>%
    unique() %>%
    rowwise() %>%
    mutate(SLAI=slai_func_percentile(CH_slice_quantile_SLAI, Plot))
  
  rs_Ml_slices_unique<-rs_slices_df[c("Plot", "CH_slice", "CH_quantile", "CH_slice_quantile_1", "CH_slice_quantile_2")] %>%
    unique() %>%
    rowwise() %>%
    mutate(propMa_slice=propMa_slice_func_percentile(CH_quantile, CH_slice_quantile_1, CH_slice_quantile_2, Plot), # CH_quantile
           propMl_slice=propMl_slice_func_percentile(CH_quantile, CH_slice_quantile_1, CH_slice_quantile_2, Plot))
  
  
  rs_traits_p<-rs_slices_df %>%
    left_join(rs_Ml_slices_unique) %>%
    left_join(rs_SLAI_slices_unique) %>%
    left_join(avg_slai_forRS) %>%
    rowwise() %>%
    mutate(
      SLAI=case_when(shading=="RSCFM" ~ SLAI,
                     shading=="average" ~ avg_SLAI,
                     shading=="none" ~ 0),
      M_a_px_layer=M_a_px*propMa_slice, M_l_px_layer=M_a_px_layer*propMl_slice
    ) %>%
    left_join(within(metadata_reduced, rm(Lon, Lat, x, y)))
  
  rm(rs_slices_df)
  rm(rs_SLAI_slices_unique)
  rm(rs_Ml_slices_unique)
  
  rs_prod_p_1<-rs_traits_p %>%
    rowwise() %>%
    mutate(scale=paste0("rs_df_", data_scale),
           I_mol=0.75*0.48*4.6*(I_mean)*exp(-0.5*SLAI), 
           N_area=N/1000*LMA,
           P_area=P/1000*LMA,
           k=K(P_area,LMA,N_area),
           r=R(P_area,LMA,N_area),
           T_resp=-0.242+0.0937*MAT_C-0.00177*MAT_C^2,
           A=max(0, ((12.13*I_mol)/(k+I_mol)-r))) %>% 
    filter(k>=0, k<=400,
           r>=0, r<=4.5) 
  
  rm(rs_traits_p)
  
  rs_prod_temp<-rs_prod_p_1 %>%
    left_join(px_count) %>%
    mutate(dMdt=((0.33/0.48)*(A*0.0378*(12/24))*((1/LMA)*10000)*M_l_px_layer)/1000*0.48,
           dMdt_gross=dMdt/.33,
           dataset_id=paste(data_scale, px_size, sep="_")) %>%
    mutate(M_l_px_layer=M_l_px_layer/px_size^2, dMdt=dMdt*(10000/px_size^2))
  
  rm(rs_prod_p_1)
  
  rs_px_prod_temp<-rs_prod_temp %>%
    group_by(Plot, Lon, Lat, x, y, Elevation, id, dataset_id, xy_ha_id, LMA, N, P, N_area, P_area) %>%
    summarize(A=mean(A, na.rm=T), M_a_px=sum(M_a_px_layer, na.rm=T), M_l_px=sum(M_l_px_layer, na.rm=T), dMdt_px=sum(dMdt, na.rm=T)) %>%
    ungroup %>%
    mutate(n_px=n(), n_ha=length(unique(xy_ha_id))) 
  
  # append data
  if(!ncol(rs_prod)==0){
    rs_prod_temp<-rs_prod_temp[colnames(rs_prod)]
    rs_px_prod_temp<-rs_px_prod_temp[colnames(rs_px_prod)]
  }
  
  rs_prod<-rbind(rs_prod, rs_prod_temp)
  rs_px_prod<-rbind(rs_px_prod, rs_px_prod_temp)
  
  print(paste0(temp_plot, " in:"))
  print(plot_list)
}
proc.time()-ptm

rs_px_prod
## export raw RS productivity data files

# RS pixel slice growth data
fwrite(rs_prod, paste0(data_dir,"Data/test_data_output/rs_slice_prod_test.csv"))

# RS pixel growth data
fwrite(rs_px_prod, paste0(data_dir,"Data/test_data_output/rs_px_prod_test.csv"))



#
### Remove outlier pixels -----

rs_px_prod_temp<-fread(paste0(data_dir,"Data/test_data_output/rs_px_prod_test.csv"))


# find outliers using quantiles and standard errors at whole-dataset and individual-plot levels
rs_plot_outlier_summary<-rs_px_prod_temp %>%
  group_by(dataset_id, Elevation) %>%
  summarize(mean_dMdt_px=mean(dMdt_px, na.rm=T), sd_dMdt_px=sd(dMdt_px, na.rm=T), lower_dMdt_px=quantile(dMdt_px, 0.01, na.rm=T), upper_dMdt_px=quantile(dMdt_px, 0.99, na.rm=T)) %>%
  rowwise() %>%
  mutate(se1=mean_dMdt_px-3*sd_dMdt_px, se2=mean_dMdt_px+3*sd_dMdt_px) %>%
  ungroup() %>%
  dplyr::select(-sd_dMdt_px) %>%
  gather(variable, value, 3:7)
rs_plot_outlier_summary$variable<-factor(rs_plot_outlier_summary$variable, levels=c("mean_dMdt_px", "se1", "se2", "lower_dMdt_px", "upper_dMdt_px"))

rs_dataset_outlier_summary<-rs_px_prod_temp %>%
  group_by(dataset_id) %>%
  summarize(mean_dMdt_px=mean(dMdt_px, na.rm=T), sd_dMdt_px=sd(dMdt_px, na.rm=T), lower_dMdt_px=quantile(dMdt_px, 0.01, na.rm=T), upper_dMdt_px=quantile(dMdt_px, 0.99, na.rm=T)) %>%
  rowwise() %>%
  mutate(se1=mean_dMdt_px-3*sd_dMdt_px, se2=mean_dMdt_px+3*sd_dMdt_px) %>%
  ungroup() %>%
  dplyr::select(-sd_dMdt_px) %>%
  gather(variable, value, 2:6)
rs_dataset_outlier_summary$variable<-factor(rs_dataset_outlier_summary$variable, levels=c("mean_dMdt_px", "se1", "se2", "lower_dMdt_px", "upper_dMdt_px"))


# plot outlier summary | red = plot stats, black = whole dataset stats
ggplot()+
  geom_histogram(data=rs_px_prod_temp, aes(dMdt_px, ..ndensity..), fill="gray50", bins=100)+ # 
  geom_vline(data=rs_plot_outlier_summary, aes(xintercept=value, linetype=variable), color="red")+
  geom_vline(data=rs_dataset_outlier_summary, aes(xintercept=value, linetype=variable), color="black")+
  scale_linetype_manual(values=c(1,5,5,3,3))+
  facet_grid(Elevation~dataset_id) +
  theme(panel.background = element_blank(), panel.border = element_rect(color = "black", fill=NA)) 


# trim outliers from data (whole-dataset version using 3*standard deviation)
rs_px_prod_trimmed<-rs_px_prod_temp %>%
  group_by(dataset_id) %>%  # dataset_id, Elevation
  filter(dMdt_px>=mean(dMdt_px, na.rm=T)-3*sd(dMdt_px, na.rm=T), dMdt_px<=mean(dMdt_px, na.rm=T)+3*sd(dMdt_px, na.rm=T)) %>%
  #filter(dMdt_px>=quantile(dMdt_px, 0.01, na.rm=T), dMdt_px<=quantile(dMdt_px, 0.99, na.rm=T)) %>%
  mutate(n_px=n(), n_ha=length(unique(xy_ha_id))) %>% 
  ungroup()

# trim outliers from data (within-plot version)
rs_plot_prod_trimmed<-rs_px_prod_trimmed %>%
  #mutate(px_size=as.numeric(word(dataset_id, 2, sep="_")), dMdt_px=dMdt_px*(10000/px_size^2)) %>%
  group_by(Plot, Elevation, dataset_id, n_ha) %>% # , px_mult, M_a_ha, Ml_Ma_ratio
  summarize(RSCFM_NPP_RS=mean(dMdt_px, na.rm=T), RSCFM_NPP_RS_sd=sd(dMdt_px, na.rm=T)) %>%
  mutate(RSCFM_NPP_RS_se1=RSCFM_NPP_RS+RSCFM_NPP_RS_sd, RSCFM_NPP_RS_se2=RSCFM_NPP_RS-RSCFM_NPP_RS_sd) %>%
  ungroup() %>%
  dplyr::select(Plot, Elevation, dataset_id, RSCFM_NPP_RS, RSCFM_NPP_RS_se1, RSCFM_NPP_RS_se2) %>% 
  gather(productivity_measure, value, -c(Plot, Elevation, dataset_id)) %>%
  mutate(productivity_measure=paste(productivity_measure, dataset_id, sep="_")) %>%
  data.frame()


# export trimmed datasets
fwrite(rs_px_prod_trimmed,paste0(data_dir,"Data/test_data_output/rs_px_prod_trimmed_test.csv"))
fwrite(rs_plot_prod_trimmed,paste0(data_dir,"Data/test_data_output/rs_plot_prod_trimmed_test.csv"))



#
### NPP and pixel growth figures (Figure 2) -----

# import processed data

tree_plot_prod<-fread(paste0(data_dir,"Data/ground_plot_prod.csv"))
rs_plot_prod<-fread(paste0(data_dir,"Data/rs_plot_prod_trimmed.csv"))
rs_plot_prod_MODIS<-fread(paste0(data_dir,"Data/rs_plot_prod_MODIS.csv"))
rs_plot_prod_BESS<-fread(paste0(data_dir,"Data/rs_plot_prod_BESS.csv"))

rs_px_prod_long<-fread(paste0(data_dir,"Data/rs_px_prod_trimmed.csv"))
rs_px_prod<-rs_px_prod_long %>%
  dplyr::select(Plot, Lon, Lat, x, y, Elevation, id, dataset_id, A, M_a_px, M_l_px, dMdt_px, n_px, n_ha)
fwrite(rs_px_prod, paste0(data_dir,"Data/rs_px_prod_trimmed_short.csv"))
rs_px_prod_MODIS<-fread(paste0(data_dir,"Data/rs_px_prod_MODIS.csv"))
rs_px_prod_BESS<-fread(paste0(data_dir,"Data/rs_px_prod_BESS.csv"))





# combine NPP estimates
gdata_main<-tree_plot_prod %>%
  filter(!grepl("GPP", productivity_measure)) %>%
  rbind(rs_plot_prod) %>%
  rbind(rs_plot_prod_MODIS) %>%
  rbind(rs_plot_prod_BESS) %>%
  mutate(scale=ifelse(dataset_id %in% c("l_10", "l_100", "l_1000", "MODIS_l_10", "BESS_l_10"), "Landscape", "Plot")) %>%
  rbind(mutate(filter(tree_plot_prod, productivity_measure=="empirical_NPP"), scale="Landscape"))
gdata_main$scale<-factor(gdata_main$scale, levels=c("Plot", "Landscape")) 

plot_NPP<-gdata_main %>%
  filter(!grepl("_se", productivity_measure))
plot_NPP$productivity_measure<-factor(plot_NPP$productivity_measure, levels=c("empirical_NPP", "CFM_NPP_ground", 
                                                                              "MODIS_NPP_RS_l_1000", "BESS_NPP_RS_l_1000",
                                                                              "RSCFM_NPP_RS_p_10", "RSCFM_NPP_RS_p_100", "RSCFM_NPP_RS_l_10", "RSCFM_NPP_RS_l_100", "RSCFM_NPP_RS_l_1000")) 

color_palette<-c("gray30", "olivedrab3", "red2", "darkorchid3", rep.int("dodgerblue2",2), rep.int("darkorange1", 3))


## Plot and Landscape NPP (Figure 2)

svg(file=paste(paste0(data_dir,"Figures_and_Tables/NPP_estimates_PlotAndLandscape.svg", sep=""), width=14, height=6))
ggplot() + 
  geom_point(data=plot_NPP, aes(x=Elevation, y=value, shape=productivity_measure, color=productivity_measure), size=2, stroke=.8, alpha=0) +
  geom_smooth(data=plot_NPP, aes(x=Elevation, y=value, color=productivity_measure, linetype=productivity_measure), method="loess", se=F, alpha=1, size=1, span=.9) + 
  geom_point(data=plot_NPP, aes(x=Elevation, y=value, shape=productivity_measure, color=productivity_measure), fill=NA, size=3, stroke=1, alpha=1) +
  scale_color_manual(values=color_palette) +
  scale_shape_manual(values=c(19,19,24,24,21,23,21,23,24)) +
  scale_linetype_manual(values=c(1,1,3,3,1,2,1,2,3)) +
  scale_x_continuous(breaks=c(0, 1000, 2000, 3000)) +
  labs(x=Elevation~(m), y=Net~Primary~Productivity~(NPP)~(MgC~yr^-1~ha^-1)) +
  facet_wrap(~scale, scales="free_x") +
  theme(panel.background = element_blank(), panel.border = element_rect(color = "black", fill=NA),
        plot.background=element_blank(), panel.grid=element_blank(),
        strip.background=element_rect(color=NA, fill=NA),
        strip.text=element_text(size=22),
        axis.text=element_text(size=18), axis.title=element_text(size=20), legend.position="none")
dev.off()



# predicted vs observed (all NPP values)
empirical_NPP_data<-filter(plot_NPP, productivity_measure=="empirical_NPP") %>%
  rename(empirical_NPP=value) %>%
  dplyr::select(Plot, empirical_NPP) %>%
  distinct()

RS_gdata<-rs_px_prod %>% 
  mutate(productivity_measure=paste0("RSCFM_NPP_RS_", dataset_id)) %>% 
  rename(value=dMdt_px) %>%
  dplyr::select(Plot, Elevation, dataset_id, productivity_measure, value)


deviation_data<-RS_gdata %>%
  bind_rows(rs_px_prod_MODIS) %>%
  bind_rows(rs_px_prod_BESS) %>%
  bind_rows(dplyr::select(filter(plot_NPP, productivity_measure=="CFM_NPP_ground"), -scale)) %>%
  bind_rows(dplyr::select(filter(gdata_main, productivity_measure=="empirical_NPP", scale=="Plot"), -scale)) %>%
  left_join(empirical_NPP_data) %>%
  mutate(deviation=value-empirical_NPP, abs_deviation=abs(deviation),
         scale=ifelse(dataset_id %in% c("l_10", "l_100", "l_1000", "MODIS_l_10", "BESS_l_10"), "Landscape", "Plot"))
deviation_data$productivity_measure<-factor(deviation_data$productivity_measure, levels=c("empirical_NPP", "CFM_NPP_ground", "RSCFM_NPP_RS_p_10", "RSCFM_NPP_RS_p_100", "RSCFM_NPP_RS_l_10", "RSCFM_NPP_RS_l_100", "RSCFM_NPP_RS_l_1000", "MODIS_NPP_RS_l_1000", "BESS_NPP_RS_l_1000"))
deviation_data$scale<-factor(deviation_data$scale, levels=c("Plot", "Landscape"))


#  deviations (all elevations)
avg_deviations<-deviation_data %>% 
  group_by(productivity_measure, scale) %>%
  summarize(n_samples=n(), n_samples_text=paste0("n = ", n()), mean_value=round(mean(value, na.rm=T), 2), sd_value=round(sd(value, na.rm=T), 2), 
            mean_deviation=round(mean(deviation, na.rm=T), 2), sd_deviation=round(sd(deviation, na.rm=T), 2), max_dev=round(max(deviation, na.rm=T), 2), 
            mean_abs_deviation=round(mean(abs_deviation, na.rm=T), 2)) %>%
  ungroup() %>%
  mutate(nudge=ifelse(scale=="Plot", max_dev+2, max_dev+2), mean_dev_text=paste0("µ = ", mean_deviation),
         export_text=paste0(n_samples_text, "\n", mean_value," ± ", sd_value, "\n", "(", mean_deviation," ± ", sd_deviation, ")"))
avg_deviations$productivity_measure<-factor(avg_deviations$productivity_measure, levels=c("empirical_NPP", "CFM_NPP_ground", "RSCFM_NPP_RS_p_10", "RSCFM_NPP_RS_p_100", "RSCFM_NPP_RS_l_10", "RSCFM_NPP_RS_l_100", "RSCFM_NPP_RS_l_1000", "MODIS_NPP_RS_l_1000", "BESS_NPP_RS_l_1000"))
avg_deviations$scale<-factor(avg_deviations$scale, levels=c("Plot", "Landscape"))
avg_deviations

fwrite(avg_deviations, paste0(data_dir,"Figures_and_Tables/NPP_deviations_all.csv"))





# deviation violin plot

color_palette<-c("gray30", "olivedrab3", rep.int("dodgerblue2",2), rep.int("darkorange1", 3), "red2", "darkorchid3")

deviation_gdata<-filter(deviation_data, !productivity_measure=="empirical_NPP")
avg_deviations_gdata<-filter(avg_deviations, !productivity_measure=="empirical_NPP")

y_min<-floor(min(deviation_gdata$deviation))
y_max<-ceiling(max(avg_deviations_gdata$nudge))+1.75



## Plot and Landscape separated

g1<-
  ggplot()+
  geom_hline(yintercept=0, linetype="dotted", color="gray30") +
  geom_jitter(data=filter(deviation_gdata, scale=="Plot"), aes(x=productivity_measure, y=deviation, alpha=productivity_measure), color="gray30", shape=20, width=0.1, size=0.1) +
  geom_violin(data=filter(deviation_gdata, scale=="Plot"), aes(x=productivity_measure, y=deviation, color=productivity_measure, fill=productivity_measure, linetype=productivity_measure), size=0.6, alpha=0.2) + # , scale="width"
  geom_point(data=filter(avg_deviations_gdata, scale=="Plot"), aes(x=productivity_measure, y=mean_deviation, color=productivity_measure), shape="-", size=15, stroke=1, alpha=1) +
  geom_text(data=filter(avg_deviations_gdata, scale=="Plot"), aes(x=productivity_measure, y=nudge+1.75, label=n_samples_text)) +
  geom_text(data=filter(avg_deviations_gdata, scale=="Plot"), aes(x=productivity_measure, y=nudge, label=mean_dev_text)) +
  labs(x="Model", y=expression(atop("Pixel growth rate deviation",~(M[pixel]-"Measured NPP")))) +
  scale_color_manual(values=color_palette[-1]) +
  scale_fill_manual(values=color_palette[-1]) +
  scale_alpha_manual(values=c(0.8, 0.2, 0.8, 0.01, 0.1, 0.4)) +
  scale_shape_manual(values=c(21,21,23,21,23,24,24)) +
  scale_linetype_manual(values=c(1,1,2,1,2,3,3)) +
  scale_x_discrete(labels=c("CFM_NPP_ground"="Ground\n(1ha)", "RSCFM_NPP_RS_p_10"="RS-CFM\n(0.01ha)", "RSCFM_NPP_RS_p_100"="RS-CFM\n(1ha)", 
                            "RSCFM_NPP_RS_l_10"="RS-CFM\n(0.01ha)", "RSCFM_NPP_RS_l_100"="RS-CFM\n(1ha)", "RSCFM_NPP_RS_l_1000"="RS-CFM\n(100ha)", "MODIS_NPP_RS_l_1000"="MODIS\n(100ha)", "BESS_NPP_RS_l_1000"="BESS\n(100ha)")) +
  ylim(y_min, y_max) +
  facet_wrap(~scale, scales="free_x") +
  theme(panel.background=element_blank(), panel.border=element_rect(color="black", fill=NA),
        strip.background=element_rect(color=NA, fill=NA),
        strip.text=element_text(size=16),
        axis.text.x=element_text(size=13), axis.text.y=element_text(size=14),
        axis.title.x=element_blank(),
        axis.title=element_text(size=18), 
        legend.position="none")


png(file=paste(paste0(data_dir,"Figures_and_Tables/NPP_deviation_violin_Plot.png", sep=""), width=4, height=3.7, units="in", res=300))
g1
dev.off()




g2<-
  ggplot()+
  geom_hline(yintercept=0, linetype="dotted", color="gray30") +
  geom_jitter(data=filter(deviation_gdata, scale=="Landscape"), aes(x=productivity_measure, y=deviation, alpha=productivity_measure), color="gray30", shape=20, width=0.1, size=0.1) +
  geom_violin(data=filter(deviation_gdata, scale=="Landscape"), aes(x=productivity_measure, y=deviation, color=productivity_measure, fill=productivity_measure, linetype=productivity_measure), size=0.6, alpha=0.2) + # , scale="width"
  geom_point(data=filter(avg_deviations_gdata, scale=="Landscape"), aes(x=productivity_measure, y=mean_deviation, color=productivity_measure), shape="-", size=15, stroke=1, alpha=1) +
  geom_text(data=filter(avg_deviations_gdata, scale=="Landscape"), aes(x=productivity_measure, y=nudge+1.75, label=n_samples_text)) +
  geom_text(data=filter(avg_deviations_gdata, scale=="Landscape"), aes(x=productivity_measure, y=nudge, label=mean_dev_text)) +
  labs(x="Model", y=expression(atop("Growth estimate deviation",~(M[pixel]-"Measured NPP")))) + 
  scale_color_manual(values=color_palette[-c(1:4)]) +
  scale_fill_manual(values=color_palette[-c(1:4)]) +
  scale_alpha_manual(values=c(0.01, 0.1, 0.4, 0.4, 0.4)) +
  scale_shape_manual(values=c(21,23,24,24,24)) +
  scale_linetype_manual(values=c(1,2,3,3,3)) +
  scale_x_discrete(labels=c("CFM_NPP_ground"="Ground\n(1ha)", "RSCFM_NPP_RS_p_10"="RS-CFM\n(0.01ha)", "RSCFM_NPP_RS_p_100"="RS-CFM\n(1ha)", 
                            "RSCFM_NPP_RS_l_10"="RS-CFM\n(0.01ha)", "RSCFM_NPP_RS_l_100"="RS-CFM\n(1ha)", "RSCFM_NPP_RS_l_1000"="RS-CFM\n(100ha)", "MODIS_NPP_RS_l_1000"="MODIS\n(100ha)", "BESS_NPP_RS_l_1000"="BESS\n(100ha)")) +  
  ylim(y_min, y_max) +
  facet_wrap(~scale, scales="free_x") +
  theme(panel.background=element_blank(), panel.border=element_rect(color="black", fill=NA),
        strip.background=element_rect(color=NA, fill=NA),
        strip.text=element_text(size=16),
        axis.text.x=element_text(size=13),
        axis.text.y=element_blank(),
        axis.title.x=element_blank(),
        axis.title.y=element_blank(),
        axis.title=element_text(size=18), 
        axis.ticks.y=element_blank(),
        legend.position="none")


png(file=paste(paste0(data_dir,"Figures_and_Tables/NPP_deviation_violin_Landscape.png", sep=""), width=4.25, height=3.7, units="in", res=300))
g2
dev.off()




# RMSD
empirical_NPP_data<-filter(plot_NPP, productivity_measure=="empirical_NPP") %>% 
  rename(empirical_NPP=value) %>%
  dplyr::select(Plot, empirical_NPP)

PvsO_data<-filter(plot_NPP, !productivity_measure %in% c("empirical_NPP")) %>% 
  left_join(empirical_NPP_data) 
PvsO_data$productivity_measure<-factor(PvsO_data$productivity_measure, levels=c("CFM_NPP_ground", "RSCFM_NPP_RS_p_10", "RSCFM_NPP_RS_p_100", "RSCFM_NPP_RS_l_10", "RSCFM_NPP_RS_l_100",
                                                                                "RSCFM_NPP_RS_l_1000", "MODIS_NPP_RS_l_1000", "BESS_NPP_RS_l_1000"))

RMSD_summary<-PvsO_data %>%
  group_by(productivity_measure) %>%
  summarize(RMSD=sqrt(mean((empirical_NPP-value)^2)))
RMSD_summary


g_min<-min(plot_NPP$value)-0.3
g_max<-max(plot_NPP$value)+0.3


# Predicted vs Observed NPP plot

svg(file=paste(paste0(data_dir,"Figures_and_Tables/Predicted_vs_Observed_NPP.svg", sep=""), width=6, height=6))
ggplot() +
  geom_abline(intercept=0, slope=1, color="gray", size=1, linetype=3)+
  geom_point(data=PvsO_data, aes(empirical_NPP, value, color=productivity_measure, shape=productivity_measure), size=3, stroke=1) +
  scale_color_manual(values=color_palette[-1]) +
  scale_shape_manual(values=c(19,21,23,21,23,24,24,24)) +
  xlim(g_min, g_max)+
  ylim(g_min, g_max)+
  labs(x=Measured~NPP~(MgC~yr^-1~ha^-1), y=Predicted~NPP~(MgC~yr^-1~ha^-1))+
  theme(panel.background=element_blank(), panel.border=element_rect(color="black", fill=NA),
        axis.text=element_text(size=17),
        axis.title=element_text(size=19), 
        aspect.ratio=1, legend.position="none")
dev.off()



#
### High-resolution NPP maps (Figure 4) -----

# import pixel NPP data  (includes plot AND landscape data)

rs_px_prod<-fread(paste0(data_dir,"Data/rs_px_prod_trimmed_short.csv")) %>%
  filter(dataset_id %in% c("p_10", "l_100"))


## plot level maps

plotMap_gdata<-filter(rs_px_prod, dataset_id=="p_10") %>%
  group_by(Elevation) %>%
  mutate(x=x-min(x), y=y-min(y)) %>% 
  ungroup

min_z<-0 
max_z<-ceiling(max(rs_px_prod$dMdt_px)) 
mid_z<-(max_z-min_z)/2


svg(file=paste(paste0(data_dir,"Figures_and_Tables/raw/plot_NPP_map_a.svg", sep="")), width=5, height=6, bg=NA)
ggplot()+
  geom_tile(data=plotMap_gdata, aes(x, y, fill=dMdt_px), color="black", size=0, alpha=1)+
  scale_fill_gradientn(colors=matlab.like2(100), limits=c(min_z, max_z), na.value="black")+
  scale_y_continuous(expand=c(0,0))+
  scale_x_continuous(expand=c(0,0))+
  coord_fixed(ratio=1)+
  labs(fill=M[pixel]~(MgC~yr^-1~ha^-1))+
  guides(fill = guide_colourbar(barwidth = 8, barheight = 0.5), color=F)+
  facet_wrap(~Elevation, labeller=labeller(Elevation=c("215"="TAM_06 (215m)", "223"="TAM_05 (223m)", "595"="PAN_02 (595m)", "848"="PAN_03 (848m)", "1527"="SPD_02 (1527m)", "1776"="SPD_01 (1776m)", 
                                                       "2758"="TRU_04 (2758m)", "2863"="ESP_01 (2863m)", "3045"="WAY_01 (3045m)", "3537"="ACJ_01 (3537m)"))) +
  theme(panel.background = element_rect(color=NA, fill="black"), panel.border = element_rect(color = "black", fill=NA), panel.grid=element_blank(),
        strip.background=element_blank(),
        axis.text=element_blank(), axis.ticks=element_blank(), axis.title=element_blank(),
        strip.text=element_text(size=10),
        legend.position="none")
dev.off()




## landscape level

elev_df<-fread(paste0(data_dir,"Data/elev_df.csv"))

landscapeMap_gdata<-rs_px_prod %>% 
  filter(dataset_id=="l_100") %>%
  left_join(plot_regions)


rs_plot_coords<-rs_px_prod %>% 
  left_join(plot_regions) %>%
  filter(dataset_id=="p_10") %>%
  group_by(region_id, Plot, Elevation) %>%
  summarize(x=mean(x, na.rm=T), y=mean(y, na.rm=T)) %>%
  ungroup() %>%
  mutate(plot_text=paste0(Plot, " (", Elevation, "m)"))
rs_plot_coords$Elevation<-factor(rs_plot_coords$Elevation, levels=sort(rs_plot_coords$Elevation))



# choose region to plot


region<-1

landscapeMap_gdata_region<-filter(landscapeMap_gdata, region_id==region)

# (fix to include Lon and Lat)
min_x<-min(landscapeMap_gdata_region$x)-200
max_x<-max(landscapeMap_gdata_region$x)+200
min_y<-min(landscapeMap_gdata_region$y)-200
max_y<-max(landscapeMap_gdata_region$y)+200
min_z<-0 
max_z<-ceiling(max(rs_px_prod$dMdt_px))
mid_z<-(max_z-min_z)/2

elev_gdata<-elev_df %>%
  filter(ha_x>=min_x, ha_x<=max_x, ha_y>=min_y, ha_y<=max_y) 

rs_plot_coords_gdata<-rs_plot_coords %>%
  filter(region_id==region)

shape_vals<-if(region==3){2} else if(region==2){0:1} else if(region==1){3:8}  


fig<-
  ggplot()+
  geom_raster(data=elev_gdata, aes(ha_x, ha_y, fill=ha_Elevation_px), interpolate=F)+
  scale_fill_gradient(low="gray90", high="black", limits=c(0, 4300), breaks=c(0, 2000, 4000))+
  labs(fill="Elevation")+
  new_scale_fill() + 
  geom_raster(data=landscapeMap_gdata_region, aes(x, y, fill=dMdt_px))+
  scale_color_gradientn(colors=matlab.like2(100), limits=c(min_z, max_z))+
  scale_fill_gradientn(colors=matlab.like2(100), limits=c(min_z, max_z))+
  geom_point(data=rs_plot_coords_gdata, aes(x, y, shape=factor(Elevation)), color="red2", size=2.25, stroke=1.5)+
  scale_shape_manual(values=shape_vals)+
  scale_x_continuous(expand=expand_scale(c(0, 0)), limits=c(min_x, max_x))+
  scale_y_continuous(expand=expand_scale(c(0, 0)), limits=c(min_y, max_y))+
  coord_fixed(ratio=1)+
  labs(fill=M[pixel]~(MgC~yr^-1~ha^-1))+
  guides(shape = FALSE)+
  theme(panel.background = element_blank(), 
        panel.border = element_rect(color="black", fill=NA),
        axis.text=element_blank(), axis.ticks=element_blank(), axis.title=element_blank(), plot.margin=unit(c(0,0,0,0), "mm"), 
        panel.grid=element_blank(),
        legend.title=element_text(size=14, vjust=0.5), legend.text=element_text(size=12),
        legend.key.height=unit(3, "mm"), legend.key.width=unit(8, "mm"),
        legend.position="bottom", legend.justification="right", legend.box.just = "top",
        legend.margin = margin(0, 0, 0, 0))



svg(file=paste(paste0(data_dir,"Figures_and_Tables/landscape_NPP_map_", region, ".svg", sep="")), width=ifelse(region==1, 5, 0.8), bg=NA)
fig
dev.off()

#
### Extract Alternative NPP Estimates (MODIS & BESS) -----

rs_px_prod<-fread(paste0(data_dir,"Data/rs_px_prod_trimmed_short.csv"))
modis_raster<-raster(paste0(data_dir,"Data/MODIS/MOD17A3_Science_NPP_2013.tif"))
modis_raster_crop<-crop(modis_raster, extent(min(rs_px_prod$Lon), max(rs_px_prod$Lon), min(rs_px_prod$Lat), max(rs_px_prod$Lat)))


## MODIS NPP estimates 

temp_extract<-raster::extract(modis_raster_crop, SpatialPoints(rs_px_prod[, .(Lon, Lat)]), cellnumbers=T) %>%
  data.frame() %>%
  rename(cell_id=cells, MODIS_NPP=MOD17A3_Science_NPP_2013) 

temp_coords<-coordinates(modis_raster_crop)[temp_extract[,1],] %>%
  as.data.frame() %>%
  rename(MODIS_Lon=x, MODIS_Lat=y)

rs_px_prod_MODIS_1<-cbind(rs_px_prod, temp_extract, temp_coords) %>%
  mutate(MODIS_NPP=MODIS_NPP*10*0.0001) %>%
  dplyr::select(-c(id, xy_ha_id, n_px, n_ha))


MODIS_npp_landscape<-rs_px_prod_MODIS_1 %>%
  dplyr::select(Plot, Elevation, dataset_id, cell_id, dMdt_px, MODIS_NPP) %>%
  filter(dataset_id %in% c("l_10")) %>%
  group_by(Plot, Elevation, dataset_id, cell_id) %>%
  summarize(dMdt_px=mean(dMdt_px, na.rm=T), MODIS_NPP_px=mean(MODIS_NPP, na.rm=T)) %>%
  drop_na(MODIS_NPP_px) %>% 
  mutate(n_px=length(unique(cell_id))) %>%
  ungroup() %>%
  mutate(dataset_id=paste0("MODIS_", dataset_id))
length(unique(MODIS_npp_landscape$cell_id))


rs_px_prod_MODIS<-MODIS_npp_landscape %>%
  mutate(productivity_measure="MODIS_NPP_RS_l_1000") %>%
  dplyr::select(Plot, Elevation, dataset_id, productivity_measure, MODIS_NPP_px) %>%
  rename(value=MODIS_NPP_px)


rs_plot_prod_MODIS<-MODIS_npp_landscape %>%
  mutate(productivity_measure="MODIS_NPP_RS_l_1000") %>%
  group_by(Plot, Elevation, dataset_id, productivity_measure) %>%
  summarize(value=mean(MODIS_NPP_px, na.rm=T)) %>%
  ungroup


# fwrite(rs_px_prod_MODIS, paste0(data_dir,"Data/rs_px_prod_MODIS.csv"))
# fwrite(rs_plot_prod_MODIS, paste0(data_dir,"Data/rs_plot_prod_MODIS.csv"))






## BESS NPP estimates

temp_df_modis<-rs_px_prod_MODIS_1 %>%
  filter(dataset_id=="l_10") %>%
  dplyr::select(contains("MODIS")) %>%
  unique()
temp_df_modis<-round(temp_df_modis, 8)

modis_range<-c(min(temp_df_modis$MODIS_Lon, na.rm=T), max(temp_df_modis$MODIS_Lon, na.rm=T), 
               min(temp_df_modis$MODIS_Lat, na.rm=T), max(temp_df_modis$MODIS_Lat, na.rm=T))

modis_grid<-expand.grid(MODIS_Lon=seq(modis_range[[1]]-10, modis_range[[2]]+10, 0.00833333),
                        MODIS_Lat=seq(modis_range[[3]]-10, modis_range[[4]]+10, 0.00833333))

modis_grid_1<-modis_grid %>% 
  filter(MODIS_Lon > -72, MODIS_Lon < -62, MODIS_Lat > -20, MODIS_Lat < -10)

modis_grid_2<-modis_grid_1 %>%
  group_by(MODIS_Lat) %>%
  mutate(h=row_number(MODIS_Lon)) %>%
  group_by(MODIS_Lon) %>%
  mutate(v=row_number(MODIS_Lat)) %>%
  ungroup()


file_names<-list.files(paste0(data_dir,"Data/BESS"))

BESS_df<-modis_grid_2[,1:2]
for(i in file_names){
  BESS_import<-h5dump(H5Fopen(paste0(data_dir,"Data/BESS/",i)))$GPP_1km
  id<-paste0("B_", str_sub(word(i, 2, sep=fixed(".")), -3))
  
  temp_BESS_df<-as.data.table(BESS_import) %>% 
    tibble::rownames_to_column(var="v") %>%
    pivot_longer(!v, names_to="h", values_to="UI", names_prefix = "V") %>%
    mutate(v=as.numeric(v), v=1201-v, h=as.numeric(h), DN=as.numeric(UI), DN_drop_na=ifelse(DN==255, NA, DN), BESS_NPP=8*0.33*0.1*0.01*DN_drop_na) %>%
    left_join(modis_grid_2) %>%
    dplyr::select(MODIS_Lon, MODIS_Lat, BESS_NPP)
    colnames(temp_BESS_df)[3]<-id
    
  BESS_df<-full_join(BESS_df, temp_BESS_df)
}


annual_BESS_df<-BESS_df %>%
  drop_na() %>%
  pivot_longer(cols=starts_with("B_")) %>%
  group_by(MODIS_Lon, MODIS_Lat) %>%
  summarize(BESS_NPP=sum(value, na.rm=T)) %>%
  ungroup()


BESS_raster<-rasterFromXYZ(annual_BESS_df)
BESS_raster_crop<-crop(BESS_raster, extent(min(modis_grid_2$MODIS_Lon), max(modis_grid_2$MODIS_Lon), min(modis_grid_2$MODIS_Lat), max(modis_grid_2$MODIS_Lat)))

temp_extract<-raster::extract(BESS_raster_crop, SpatialPoints(rs_px_prod[, .(Lon, Lat)]), cellnumbers=T) %>%
  data.frame() %>%
  rename(cell_id=cells) 

temp_coords<-coordinates(BESS_raster_crop)[temp_extract[,1],] %>%
  as.data.frame() %>%
  rename(MODIS_Lon=x, MODIS_Lat=y)

rs_px_prod_BESS_1<-cbind(rs_px_prod, temp_extract, temp_coords) %>%
  dplyr::select(-c(id, xy_ha_id, n_px, n_ha))

BESS_npp_landscape<-rs_px_prod_BESS_1 %>%
  dplyr::select(Plot, Elevation, dataset_id, cell_id, dMdt_px, BESS_NPP) %>%
  filter(dataset_id %in% c("l_10")) %>%
  group_by(Plot, Elevation, dataset_id, cell_id) %>%
  summarize(dMdt_px=mean(dMdt_px, na.rm=T), BESS_NPP_px=mean(BESS_NPP, na.rm=T)) %>%
  drop_na(BESS_NPP_px) %>% 
  mutate(n_px=length(unique(cell_id))) %>%
  ungroup() %>%
  mutate(dataset_id=paste0("BESS_", dataset_id))
length(unique(BESS_npp_landscape$cell_id))


rs_px_prod_BESS<-BESS_npp_landscape %>%
  mutate(productivity_measure="BESS_NPP_RS_l_1000") %>%
  dplyr::select(Plot, Elevation, dataset_id, productivity_measure, BESS_NPP_px) %>%
  rename(value=BESS_NPP_px)


rs_plot_prod_BESS<-BESS_npp_landscape %>%
  mutate(productivity_measure="BESS_NPP_RS_l_1000") %>%
  group_by(Plot, Elevation, dataset_id, productivity_measure) %>%
  summarize(value=mean(BESS_NPP_px, na.rm=T)) %>%
  ungroup


# fwrite(rs_px_prod_BESS, paste0(data_dir,"Data/CHAMBASA_RSdata/chambasa_DW/rs_px_prod_BESS.csv"))
# fwrite(rs_plot_prod_BESS, paste0(data_dir,"Data/CHAMBASA_RSdata/chambasa_DW/rs_plot_prod_BESS.csv"))

#
### end -----





