! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************

! ADDTION OF ARTIFICIAL VISCOSITY
! Used to compute models presented in Moyano et al. (2022): 
! Asteroseismology of evolved stars to constrain the internal transport of angular momentum
! V. Efficiency of the transport on the red giant branch and in the red clump

module run_star_extras

  use star_lib
  use star_def
  use const_def
  use math_lib

  implicit none
  real(dp), parameter:: visc_add=8.d3

contains

  subroutine extras_controls(id, ierr)
    integer, intent(in) :: id
    integer, intent(out) :: ierr
    type (star_info), pointer :: s
    ierr = 0
    call star_ptr(id, s, ierr)
    if (ierr /= 0) return

    s% how_many_extra_history_columns => how_many_extra_history_columns
    s% data_for_extra_history_columns => data_for_extra_history_columns
    s% how_many_extra_profile_columns => how_many_extra_profile_columns
    s% data_for_extra_profile_columns => data_for_extra_profile_columns  
    s% job% warn_run_star_extras = .false.     

    !New routines to add

    s% other_am_mixing => add_nu

  end subroutine extras_controls

  subroutine add_nu(id, ierr)
    ! Additional viscosity included in the diffusion coefficient of the equation of angular momentum transport
    integer, intent(in) :: id
    integer, intent(out) :: ierr
    type (star_info), pointer :: s
    integer :: k,j,op_err

    call star_ptr(id,s,ierr)
    if (ierr /= 0) return

    do k= s %nz , 1, -1
       s% am_nu_omega(k) = s% am_nu_omega(k) + visc_add
    end do

  end subroutine add_nu


  integer function how_many_extra_profile_columns(id)
    use star_def, only: star_info
    integer, intent(in) :: id
    integer :: ierr
    type (star_info), pointer :: s
    ierr = 0
    call star_ptr(id, s, ierr)
    if (ierr /= 0) return
    how_many_extra_profile_columns = 1
  end function how_many_extra_profile_columns


  subroutine data_for_extra_profile_columns(id, n, nz, names, vals, ierr)
    use star_def, only: star_info, maxlen_profile_column_name
    use const_def, only: dp
    integer, intent(in) :: id, n, nz
    character (len=maxlen_profile_column_name) :: names(n)

    real(dp) :: vals(nz,n)

    integer, intent(out) :: ierr
    type (star_info), pointer :: s
    integer :: k,j,op_err,nsmooth,nsmootham
    ierr = 0
    call star_ptr(id, s, ierr)
    if (ierr /= 0) return

    names(1)='D_am_tot'
    vals(:,1)= 1d-50
    !Total diffusion coefficient including the artificial viscosity
    do k=1,s%nz
       vals(k,1)= s% am_nu_omega(k) + visc_add
    enddo

  end subroutine data_for_extra_profile_columns


  integer function how_many_extra_history_columns(id)
    integer, intent(in) :: id
    integer :: ierr
    type (star_info), pointer :: s
    ierr = 0
    call star_ptr(id, s, ierr)
    if (ierr /= 0) return
    how_many_extra_history_columns = 5
  end function how_many_extra_history_columns

  subroutine data_for_extra_history_columns(id, n, names, vals, ierr)
    use const_def, only: pi
    integer, intent(in) :: id, n
    character (len=maxlen_history_column_name) :: names(n)
    real(dp) :: vals(n)
    real(dp) :: ominteg,ninteg,neff,dr,J1,J2,J3,nu_pulse,delta_pi1,mixmod_freq
    integer :: j,k,op_err
    integer, intent(out) :: ierr
    type (star_info), pointer :: s
    ierr = 0
    call star_ptr(id, s, ierr)
    if (ierr /= 0) return

    op_err = 0

    names(1) = 'nu_max[uhz]' !Aseteroseismic nu_max
    names(2) = 'om_g' !Core rotation as sensed by g-modes  
    names(3) = 'delta_nu[uhz]' !Large separation
    names(4) = 'delta_pi1[s]' !Period spacing of g-modes
    names(5) = 'mixmod_freq' ! Mixed-mode density

    vals(1) = 1d-50
    vals(2) = 1d-50
    vals(3) = 1d-50
    vals(4) = 1d-50
    vals(5) = 1d-50

    vals(1) = s% nu_max
    nu_pulse = s% nu_max !Set pulsation period via scaling relations

    ominteg=1d-99
    ninteg=1d-50
    J1=1d-50
    J2=1d-50
    J3=1d-50

    delta_pi1=0.d0

    do k = 2, s% nz-1

       if (s% brunt_N2(k) < 1d-14) neff = 1d-14 
       if (s% brunt_N2(k) > 1d-14) neff = s% brunt_N2(k)

       if (2d0*pi*nu_pulse/1d6<sqrt(neff)) then
          if (2d0*pi*nu_pulse/1d6<sqrt(2d0)*s% csound(k)/s% r(k)) then
             dr = (s% r(k+1)-s% r(k-1))/2d0
             ominteg = ominteg + sqrt(neff)*(dr/s% r(k))*s% omega(k)
             ninteg = ninteg + sqrt(neff)*(dr/s% r(k))
          end if
       end if

       !Calculate the period spacing of g-modes
       if(s% brunt_N2(k) > 0.d0) then
          dr = abs((s% r(k+1)-s% r(k-1))/2d0)
          delta_pi1=delta_pi1 + sqrt(s% brunt_N2(k)) * (dr/s% r(k))
          if(delta_pi1 < 0.d0) then
             print*, "delta_pi1",delta_pi1
             stop
          endif
       endif

    end do
    delta_pi1=sqrt(2.d0)*pow2(pi)/delta_pi1

    vals(2) = ominteg/ninteg
    vals(3) = s% delta_nu_sun*sqrt(s% star_mass)*pow3(s% Teff/s% Teff_sun) / &
         pow(s% L_phot,0.75d0)
    vals(4) = delta_pi1
    vals(5) = vals(3)/(vals(4)*pow2(s% nu_max))*1d6

  end subroutine data_for_extra_history_columns

end module run_star_extras

