# Copyright Contributors to the Pyro-Cov project.
# SPDX-License-Identifier: Apache-2.0

import datetime
import functools
import logging
import math
import pickle
import re
import warnings
from collections import Counter, OrderedDict, defaultdict
from timeit import default_timer
from typing import List

import numpy as np
import pyro
import pyro.distributions as dist
import torch
import tqdm
from pyro import poutine
from pyro.infer import SVI, JitTrace_ELBO, Trace_ELBO
from pyro.infer.autoguide import (
    AutoDelta,
    AutoGuideList,
    AutoLowRankMultivariateNormal,
    AutoNormal,
)
from pyro.infer.autoguide.initialization import InitMessenger
from pyro.infer.reparam import LocScaleReparam
from pyro.ops.streaming import CountMeanVarianceStats, StatsOfDict
from pyro.optim import ClippedAdam
from pyro.poutine.util import site_is_subsample

import pyrocov.geo

from . import pangolin, sarscov2
from .ops import sparse_multinomial_likelihood
from .util import pearson_correlation

logger = logging.getLogger(__name__)

# Reasonable values might be week (7), fortnight (14), or month (28)
TIMESTEP = 14  # in days
GENERATION_TIME = 5.5  # in days
START_DATE = "2019-12-01"


def get_fine_regions(columns, min_samples):
    """
    Select regions that have at least ``min_samples`` samples.
    Remaining regions will be coarsely aggregated up to country level.
    """
    # Count number of samples in each subregion.
    counts = Counter()
    for location in columns["location"]:
        parts = location.split("/")
        if len(parts) < 2:
            continue
        parts = tuple(p.strip() for p in parts[:3])
        counts[parts] += 1

    # Select fine countries.
    return frozenset(parts for parts, count in counts.items() if count >= min_samples)


def rank_loo_lineages(full_dataset: dict, min_samples: int = 50) -> List[str]:
    """
    Compute a list of lineages ranked in descending order of cut size.
    This is used in growth rate leave-one-out prediction experiments.
    """

    def get_parent(lineage):
        lineage = pangolin.decompress(lineage)
        parent = pangolin.get_parent(lineage)
        if parent is not None:
            parent = pangolin.compress(parent)
        return parent

    # Compute sample counts.
    lineage_id_inv = full_dataset["lineage_id_inv"]
    lineage_id = full_dataset["lineage_id"]
    lineage_counts = full_dataset["weekly_counts"].sum([0, 1])
    descendent_counts = lineage_counts.clone()
    for c, lineage in enumerate(lineage_id_inv):
        ancestor = get_parent(lineage)
        while ancestor is not None:
            a = lineage_id.get(ancestor)
            if a is not None:
                descendent_counts[a] += lineage_counts[c]
            ancestor = get_parent(ancestor)
    total = lineage_counts.sum().item()
    cut_size = torch.min(descendent_counts, total - descendent_counts)

    # Filter and sort lineages by cut size.
    ranked_lineages = [
        (size, lineage)
        for size, lineage in zip(cut_size.tolist(), lineage_id_inv)
        if lineage not in ("A", "B", "B.1")
        if size >= min_samples
    ]
    ranked_lineages.sort(reverse=True)
    return [name for gap, name in ranked_lineages]


def dense_to_sparse(x):
    index = x.nonzero(as_tuple=False).T.contiguous()
    value = x[tuple(index)]
    total = x.sum(-1)
    return {"index": index, "value": value, "total": total}


def load_nextstrain_data(
    device="cpu",
    min_region_size=50,
    include={},
    exclude={},
    end_day=None,
    columns_filename="results/nextstrain.columns.pkl",
    features_filename="results/nextstrain.features.pt",
) -> dict:
    """
    Loads the two files columns_filename and features_filename,
    converts the input to PyTorch tensors and truncates the data according to
    ``include`` and ``exclude``.

    :param str device: torch device to use
    :param dict include: filters of data to include
    :param dict exclude: filters of data to exclude
    :param end_day: last day to include
    :param str columns_filename: generated by scripts/preprocess_nextstrain.py
    :param str features_filename: generated by scripts/preprocess_nextstrain.py
    :returns: A dataset dict
    :rtype: dict
    """
    logger.info("Loading data")
    include = include.copy()
    exclude = exclude.copy()

    if end_day:
        logger.info(f"Load nextstrain data end_day: {end_day}")

    # Load column data.
    with open(columns_filename, "rb") as f:
        columns = pickle.load(f)
    # Clean up location ids.
    columns["location"] = list(map(pyrocov.geo.gisaid_normalize, columns["location"]))
    logger.info(f"Training on {len(columns['day'])} rows with columns:")
    logger.info(", ".join(columns.keys()))

    # Aggregate regions smaller than min_region_size to country level.
    fine_regions = get_fine_regions(columns, min_region_size)

    # Filter features into numbers of mutations and possibly genes.
    raw_features = torch.load(features_filename)
    mutations = raw_features["aa_mutations"]
    features = raw_features["aa_features"].to(
        device=device, dtype=torch.get_default_dtype()
    )
    keep = [True] * len(mutations)
    if include.get("gene"):
        re_gene = re.compile(include.pop("gene"))
        keep = [k and bool(re_gene.search(m)) for k, m in zip(keep, mutations)]
    if exclude.get("gene"):
        re_gene = re.compile(exclude.pop("gene"))
        keep = [k and not re_gene.search(m) for k, m in zip(keep, mutations)]
    if include.get("region"):
        gene, region = include.pop("region")
        lb, ub = sarscov2.GENE_STRUCTURE[gene][region]
        for i, m in enumerate(mutations):
            g, m = m.split(":")
            if g != gene:
                keep[i] = False
                continue
            match = re.search("[0-9]+", m)
            assert match is not None
            pos = int(match.group())
            if not (lb < pos <= ub):
                keep[i] = False
    mutations = [m for k, m in zip(keep, mutations) if k]
    if mutations:
        features = features[:, keep]
    else:
        warnings.warn("No mutations selected; using empty features")
        mutations = ["S:D614G"]  # bogus
        features = features[:, :1] * 0
    logger.info("Loaded {} feature matrix".format(" x ".join(map(str, features.shape))))

    # Construct the list of lineages.
    lineage_id_inv = raw_features["lineages"]
    lineage_id = {k: i for i, k in enumerate(lineage_id_inv)}
    lineages = columns["lineage"]

    # Generate sparse_data.
    sparse_data: dict = Counter()
    location_id: dict = OrderedDict()
    skipped_lineages = set()
    num_obs = 0
    for day, location, lineage in zip(columns["day"], columns["location"], lineages):
        if lineage not in lineage_id:
            if lineage not in skipped_lineages:
                skipped_lineages.add(lineage)
                if not lineage.startswith("fine"):
                    logger.warning(f"WARNING skipping unsampled lineage {lineage}")
            continue

        # Filter by include/exclude
        row = {"location": location, "day": day, "lineage": lineage}
        if not all(re.search(v, row[k]) for k, v in include.items()):
            continue
        if any(re.search(v, row[k]) for k, v in exclude.items()):
            continue

        # Filter by day
        if end_day is not None:
            if day > end_day:
                continue

        # preprocess parts
        parts = location.split("/")
        if len(parts) < 2:
            continue
        parts = tuple(p.strip() for p in parts[:3])
        if len(parts) == 3 and parts not in fine_regions:
            parts = parts[:2]
        location = " / ".join(parts)
        p = location_id.setdefault(location, len(location_id))

        # Save sparse data.
        num_obs += 1
        t = day // TIMESTEP
        l = lineage_id[lineage]
        sparse_data[t, p, l] += 1
    logger.warning(f"WARNING skipped {len(skipped_lineages)} unsampled lineages")
    logger.info(f"Found {len(location_id)} regions")
    location_id_inv = [None] * len(location_id)
    for k, i in location_id.items():
        location_id_inv[i] = k
    assert all(location_id_inv)

    # Generate weekly_counts tensor from sparse_data.
    if end_day is not None:
        T = 1 + end_day // TIMESTEP
    else:
        T = 1 + max(columns["day"]) // TIMESTEP
    P = len(location_id)
    L = len(lineage_id)
    weekly_counts = torch.zeros(T, P, L)
    for tps, n in sparse_data.items():
        weekly_counts[tps] = n
    logger.info(f"Dataset size [T x P x L] {T} x {P} x {L}")

    logger.info(
        f"Keeping {num_obs}/{len(lineages)} rows "
        f"(dropped {len(lineages) - int(num_obs)})"
    )

    # Construct sparse representation.
    place_lineage_index = (
        weekly_counts.ne(0).any(0).reshape(-1).nonzero(as_tuple=True)[0]
    )
    sparse_counts = dense_to_sparse(weekly_counts)

    # Construct time scales centered around observations.
    time = torch.arange(float(T)) * TIMESTEP / GENERATION_TIME
    time -= time.mean()

    dataset = {
        "features": features,
        "lineage_id": lineage_id,
        "lineage_id_inv": lineage_id_inv,
        "location_id": location_id,
        "location_id_inv": location_id_inv,
        "mutations": mutations,
        "place_lineage_index": place_lineage_index,
        "sparse_counts": sparse_counts,
        "time": time,
        "weekly_counts": weekly_counts,
    }
    return dataset


def load_jhu_data(nextstrain_data: dict) -> dict:
    """
    Load case count time series.

    This is used for plotting but is not used for fitting a model.
    """
    # Load raw JHU case count data.
    us_cases_df = pyrocov.geo.read_csv("time_series_covid19_confirmed_US.csv")
    global_cases_df = pyrocov.geo.read_csv("time_series_covid19_confirmed_global.csv")
    daily_cases = torch.cat(
        [
            pyrocov.geo.pd_to_torch(us_cases_df, columns=slice(11, None)),
            pyrocov.geo.pd_to_torch(global_cases_df, columns=slice(4, None)),
        ]
    ).T
    logger.info(
        "Loaded {} x {} daily case data, totaling {}".format(
            *daily_cases.shape, daily_cases[-1].sum().item()
        )
    )

    # Convert JHU locations to GISAID locations.
    locations = list(nextstrain_data["location_id"])
    matrix = pyrocov.geo.nextstrain_to_jhu_location(
        locations, us_cases_df, global_cases_df
    )
    assert matrix.shape == (len(locations), daily_cases.shape[-1])
    daily_cases = daily_cases @ matrix.T
    daily_cases[1:] -= daily_cases[:-1].clone()  # cumulative -> density
    daily_cases.clamp_(min=0)
    assert daily_cases.shape[1] == len(nextstrain_data["location_id"])

    # Convert daily counts to TIMESTEP counts (e.g. weekly).
    start_date = datetime.datetime.strptime(START_DATE, "%Y-%m-%d")
    jhu_start_date = pyrocov.geo.parse_date(us_cases_df.columns[11])
    assert start_date < jhu_start_date
    dt = (jhu_start_date - start_date).days
    T = len(nextstrain_data["weekly_clades"])
    weekly_cases = daily_cases.new_zeros(T, len(locations))
    for w in range(TIMESTEP):
        t0 = (w + dt) // TIMESTEP
        source = daily_cases[w::TIMESTEP]
        destin = weekly_cases[t0 : t0 + len(source)]
        destin[:] += source[: len(destin)]
    assert weekly_cases.sum() > 0

    return {
        "daily_cases": daily_cases.clamp(min=0),
        "weekly_cases": weekly_cases.clamp(min=0),
    }


def model(dataset, model_type, *, forecast_steps=None):
    """
    Bayesian regression model of lineage portions as a function of mutation features.

    This function can be run in two different modes:
    - During training, ``forecast_steps=None`` and the model is conditioned on
      observed data.
    - During prediction (after training), the likelihood statement is omitted
      and instead a ``probs`` tensor is recorded; this is the predicted lineage
      portions in each (time, region) bin.
    """
    # Tensor shapes are commented at at the end of some lines.
    features = dataset["features"]
    time = dataset["time"]  # [T]
    weekly_counts = dataset["weekly_counts"]
    sparse_counts = dataset["sparse_counts"]
    pl_index = dataset["place_lineage_index"]
    T, P, L = weekly_counts.shape
    L, F = features.shape
    PL = len(pl_index)
    assert PL <= P * L
    assert time.shape == (T,)

    # Optionally extend time axis.
    if forecast_steps is not None:  # During prediction.
        T = T + forecast_steps
        t0 = time[0]
        dt = time[1] - time[0]
        time = t0 + dt * torch.arange(float(T))
        assert time.shape == (T,)

    time_plate = pyro.plate("time", T, dim=-3)
    place_plate = pyro.plate("place", P, dim=-2)
    lineage_plate = pyro.plate("lineage", L, dim=-1)
    pl_plate = pyro.plate("place_lineage", PL, dim=-1)

    # Configure reparametrization (which does not affect model density).
    reparam = {}
    if "reparam" in model_type:
        reparam["coef"] = LocScaleReparam()
        reparam["pl_rate"] = LocScaleReparam()
        reparam["pl_init"] = LocScaleReparam()
    with poutine.reparam(config=reparam):

        # Sample global random variables.
        coef_scale = pyro.sample("coef_scale", dist.LogNormal(-4, 2))
        rate_scale = pyro.sample("rate_scale", dist.LogNormal(-4, 2))
        init_scale = pyro.sample("init_scale", dist.LogNormal(0, 2))

        # Assume relative growth rate depends strongly on mutations and weakly
        # on lineage and place. Assume initial infections depend strongly on
        # lineage and place.
        coef = pyro.sample(
            "coef", dist.Laplace(torch.zeros(F), coef_scale).to_event(1)
        )  # [F]
        with lineage_plate:
            rate_loc = pyro.deterministic("rate_loc", 0.01 * coef @ features.T)  # [L]
        with pl_plate:
            pl_rate_loc = rate_loc.expand(P, L).reshape(-1)
            pl_rate = pyro.sample(
                "pl_rate", dist.Normal(pl_rate_loc[pl_index], rate_scale)
            )  # [PL]
            pl_init = pyro.sample("pl_init", dist.Normal(0, init_scale))  # [PL]
        with place_plate, lineage_plate:
            rate = pyro.deterministic(
                "rate",
                pl_rate_loc.scatter(0, pl_index, pl_rate).reshape(P, L),
            )  # [P, L]
            init = pyro.deterministic(
                "init",
                torch.full((P * L,), -1e2).scatter(0, pl_index, pl_init).reshape(P, L),
            )  # [P, L]
        logits = init + rate * time[:, None, None]  # [T, P, L]

        # Optionally predict probabilities (during prediction).
        if forecast_steps is not None:
            probs = logits.softmax(-1)
            with time_plate, place_plate, lineage_plate:
                pyro.deterministic("probs", probs)
            return

        # Finally observe counts (during inference).
        if "dense" in model_type:  # equivalent either way
            # Compute a dense likelihood (more expensive).
            with time_plate, place_plate:
                pyro.sample(
                    "obs",
                    dist.Multinomial(logits=logits.unsqueeze(-2), validate_args=False),
                    obs=weekly_counts.unsqueeze(-2),
                )  # [T, P, 1, L]
            return
        # Compute a sparse likelihood (cheaper).
        logits = logits.log_softmax(-1)
        t, p, c = sparse_counts["index"]
        pyro.factor(
            "obs",
            sparse_multinomial_likelihood(
                sparse_counts["total"], logits[t, p, c], sparse_counts["value"]
            ),
        )


class InitLocFn:
    """
    Initializer for latent variables.

    This is passed as the ``init_loc_fn`` to guides.
    """

    def __init__(self, dataset):
        # Initialize init.
        init = dataset["weekly_counts"].sum(0)  # [P, L]
        init.add_(1 / init.size(-1)).div_(init.sum(-1, True))
        init.log_().sub_(init.median(-1, True).values).add_(torch.randn(init.shape))
        self.init = init  # [P, L]
        self.init_decentered = init / 2
        self.init_loc = init.mean(0)  # [L]
        self.init_loc_decentered = self.init_loc / 2
        self.pl_init = self.init.reshape(-1)[dataset["place_lineage_index"]]
        self.pl_init = self.pl_init / 2
        assert not torch.isnan(self.init).any()
        logger.info(f"init stddev = {self.init.std():0.3g}")

    def __call__(self, site):
        name = site["name"]
        shape = site["fn"].shape()
        if hasattr(self, name):
            result = getattr(self, name)
            assert result.shape == shape
            return result
        if name in (
            "coef_scale",
            "init_scale",
            "init_loc_scale",
        ):
            return torch.ones(shape)
        if name == "logits_scale":
            return torch.full(shape, 0.002)
        if name in (
            "rate_scale",
            "place_scale",
            "lineage_scale",
        ):
            return torch.full(shape, 0.01)
        if name in (
            "rate_loc",
            "rate_loc_decentered",
            "coef",
            "coef_decentered",
            "rate",
            "rate_decentered",
            "pl_rate",
            "pl_rate_decentered",
        ):
            return torch.rand(shape).sub_(0.5).mul_(0.01)
        if name == "coef_loc":
            return torch.rand(shape).sub_(0.5).mul_(0.01).add_(1.0)
        raise ValueError(f"InitLocFn found unhandled site {repr(name)}; please update.")


class Guide(AutoGuideList):
    """
    Custom guide for large-scale inference.

    This combines a low-rank multivariate normal guide over small variables
    with a mean field guide over remaining latent variables.
    """

    def __init__(self, model, init_loc_fn, init_scale, rank):
        super().__init__(InitMessenger(init_loc_fn)(model))

        # Jointly estimate globals, mutation coefficients, and lineage coefficients.
        mvn = [
            "coef_scale",
            "rate_loc_scale",
            "init_loc_scale",
            "rate_scale",
            "init_scale",
            "coef",
            "coef_decentered",
            "rate_loc",
            "rate_loc_decentered",
            "init_loc",
            "init_loc_decentered",
        ]
        self.append(
            AutoLowRankMultivariateNormal(
                poutine.block(model, expose=mvn),
                init_loc_fn=init_loc_fn,
                init_scale=init_scale,
                rank=rank,
            )
        )
        model = poutine.block(model, hide=mvn)

        # Mean-field estimate all remaining latent variables.
        self.append(AutoNormal(model, init_loc_fn=init_loc_fn, init_scale=init_scale))


@torch.no_grad()
@poutine.mask(mask=False)
def predict(
    model,
    guide,
    dataset,
    model_type,
    *,
    num_samples=1000,
    vectorize=None,
    save_params=("rate", "init", "probs"),
    forecast_steps=0,
) -> dict:
    def get_conditionals(data):
        trace = poutine.trace(poutine.condition(model, data)).get_trace(
            dataset, model_type, forecast_steps=forecast_steps
        )
        return {
            name: site["value"].detach()
            for name, site in trace.nodes.items()
            if site["type"] == "sample" and not site_is_subsample(site)
            if not name.startswith("obs")
        }

    # Compute median point estimate.
    result: dict = defaultdict(dict)
    for name, value in get_conditionals(guide.median(dataset)).items():
        if value.numel() < 1e5 or name in save_params:
            result["median"][name] = value

    # Compute moments.
    save_params = {
        k for k, v in result["median"].items() if v.numel() < 1e5 or k in save_params
    }
    if vectorize is None:
        vectorize = result["median"]["probs"].numel() < 1e6
    if vectorize:
        with pyro.plate("particles", num_samples, dim=-4):
            samples = get_conditionals(guide())
        for k, v in samples.items():
            if k in save_params:
                result["mean"][k] = v.mean(0).squeeze()
                result["std"][k] = v.std(0).squeeze()
    else:
        stats = StatsOfDict({k: CountMeanVarianceStats for k in save_params})
        for _ in tqdm.tqdm(range(num_samples)):
            stats.update(get_conditionals(guide()))
        for name, stats_ in stats.get().items():
            if "mean" in stats_:
                result["mean"][name] = stats_["mean"]
            if "variance" in stats_:
                result["std"][name] = stats_["variance"].sqrt()
    return dict(result)


def fit_svi(
    dataset: dict,
    *,
    model_type: str,
    guide_type: str,
    cond_data={},
    forecast_steps=0,
    learning_rate=0.05,
    learning_rate_decay=0.1,
    num_steps=3001,
    num_samples=1000,
    clip_norm=10.0,
    rank=200,
    jit=True,
    log_every=50,
    seed=20210319,
    check_loss=False,
) -> dict:
    """
    Fits a variational posterior using stochastic variational inference (SVI).
    """
    start_time = default_timer()

    logger.info(f"Fitting {guide_type} guide via SVI")
    pyro.set_rng_seed(seed)
    pyro.clear_param_store()
    param_store = pyro.get_param_store()

    # Initialize guide so we can count parameters and register hooks.
    cond_data = {k: torch.as_tensor(v) for k, v in cond_data.items()}
    model_ = poutine.condition(model, cond_data)
    init_loc_fn = InitLocFn(dataset)
    Elbo = JitTrace_ELBO if jit else Trace_ELBO
    if guide_type == "map":
        guide = AutoDelta(model_, init_loc_fn=init_loc_fn)
    elif guide_type == "normal":
        guide = AutoNormal(model_, init_loc_fn=init_loc_fn, init_scale=0.01)
    elif guide_type == "full":
        guide = AutoLowRankMultivariateNormal(
            model_, init_loc_fn=init_loc_fn, init_scale=0.01, rank=rank
        )
    else:
        guide = Guide(model_, init_loc_fn=init_loc_fn, init_scale=0.01, rank=rank)
    # This initializes the guide:
    latent_shapes = {k: v.shape for k, v in guide(dataset, model_type).items()}
    latent_numel = {k: v.numel() for k, v in latent_shapes.items()}
    logger.info(
        "\n".join(
            [f"Model has {sum(latent_numel.values())} latent variables of shapes:"]
            + [f" {k} {tuple(v)}" for k, v in latent_shapes.items()]
        )
    )
    param_shapes = {k: v.shape for k, v in pyro.get_param_store().named_parameters()}
    param_numel = {k: v.numel() for k, v in param_shapes.items()}
    logger.info(
        "\n".join(
            [f"Guide has {sum(param_numel.values())} parameters of shapes:"]
            + [f" {k} {tuple(v)}" for k, v in param_shapes.items()]
        )
    )

    # Log gradient norms during inference.
    series: dict = defaultdict(list)

    def hook(g, series):
        series.append(torch.linalg.norm(g.reshape(-1), math.inf).item())

    for name, value in pyro.get_param_store().named_parameters():
        value.register_hook(functools.partial(hook, series=series[name]))

    def optim_config(param_name):
        config: dict = {
            "lr": learning_rate,
            "lrd": learning_rate_decay ** (1 / num_steps),
            "clip_norm": clip_norm,
        }
        scalars = [k for k, v in latent_numel.items() if v == 1]
        if any("locs." + s in name for s in scalars):
            config["lr"] *= 0.2
        elif "scales" in param_name:
            config["lr"] *= 0.1
        elif "scale_tril" in param_name:
            config["lr"] *= 0.05
        elif "factors" in param_name or "prec_sqrts" in param_name:
            config["lr"] *= 0.05
        elif "weight_" in param_name:
            config["lr"] *= 0.01
        elif "weight" in param_name:
            config["lr"] *= 0.03
        elif "_centered" in param_name:
            config["lr"] *= 0.1
        return config

    optim = ClippedAdam(optim_config)
    elbo = Elbo(max_plate_nesting=3, ignore_jit_warnings=True)
    svi = SVI(model_, guide, optim, elbo)
    losses = []
    num_obs = dataset["weekly_counts"].count_nonzero()
    for step in range(num_steps):
        loss = svi.step(dataset=dataset, model_type=model_type)
        assert not math.isnan(loss)
        losses.append(loss)
        median = guide.median()
        for name, value in median.items():
            if value.numel() == 1:
                series[name].append(float(value))
        if log_every and step % log_every == 0:
            logger.info(
                " ".join(
                    [f"step {step: >4d} L={loss / num_obs:0.6g}"]
                    + [
                        "{}={:0.3g}".format(
                            "".join(p[0] for p in k.split("_")).upper(), v.item()
                        )
                        for k, v in median.items()
                        if v.numel() == 1
                    ]
                )
            )
        if check_loss and step >= 50:
            prev = torch.tensor(losses[-50:-25], device="cpu").median().item()
            curr = torch.tensor(losses[-25:], device="cpu").median().item()
            assert (curr - prev) < num_obs, "loss is increasing"

    result = predict(
        model_,
        guide,
        dataset,
        model_type,
        num_samples=num_samples,
        forecast_steps=forecast_steps,
    )
    result["losses"] = losses
    series["loss"] = losses
    result["series"] = dict(series)
    result["params"] = {
        k: v.detach().float().cpu().clone()
        for k, v in param_store.items()
        if v.numel() < 1e8
    }
    result["walltime"] = default_timer() - start_time
    return result


@torch.no_grad()
def log_stats(dataset: dict, result: dict) -> dict:
    """
    Logs statistics of predictions and model fit in the ``result`` of
    ``fit_svi()``.

    :param dict dataset: The dataset dictionary.
    :param dict result: The output of :func:`fit_svi`.
    :returns: A dictionary of statistics.
    """
    stats = {k: float(v) for k, v in result["median"].items() if v.numel() == 1}
    stats["loss"] = float(np.median(result["losses"][-100:]))
    mutations = dataset["mutations"]
    mean = result["mean"]["coef"].cpu()
    if not mean.shape:
        return stats  # Work around error in map estimation.

    # Statistical significance.
    std = result["std"]["coef"].cpu()
    sig = mean.abs() / std
    logger.info(f"|μ|/σ [median,max] = [{sig.median():0.3g},{sig.max():0.3g}]")
    stats["|μ|/σ median"] = sig.median()
    stats["|μ|/σ max"] = sig.max()

    # Effects of individual mutations.
    for name in ["S:D614G", "S:N501Y", "S:E484K", "S:L452R"]:
        if name not in mutations:
            continue
        i = mutations.index(name)
        m = mean[i] * 0.01
        s = std[i] * 0.01
        logger.info(f"ΔlogR({name}) = {m:0.3g} ± {s:0.2f}")
        stats[f"ΔlogR({name}) mean"] = m
        stats[f"ΔlogR({name}) std"] = s

    # Growth rates of individual lineages.
    rate = result["mean"]["rate"].mean(0)
    rate = rate - rate[dataset["lineage_id"]["A"]]
    for lineage in ["B.1.1.7", "B.1.617.2", "AY.23.1"]:
        R_RA = float(rate[dataset["lineage_id"][lineage]].exp())
        logger.info(f"R({lineage})/R(A) = {R_RA:0.3g}")
        stats[f"R({lineage})/R(A)"] = R_RA

    # Posterior predictive error.
    true = dataset["weekly_counts"] + 1e-20  # avoid nans
    counts = true.sum(-1, True)
    true_probs = true / counts
    pred = result["median"]["probs"][: len(true)] + 1e-20  # truncate, avoid nans
    kl = true.mul(true_probs.log() - pred.log()).sum([0, -1])
    error = (pred - true_probs) * counts ** 0.5  # scaled by Poisson stddev
    mae = error.abs().mean(0)  # average over time
    mse = error.square().mean(0)  # average over time
    stats["MAE"] = float(mae.sum(-1).mean())  # average over region
    stats["RMSE"] = float(mse.sum(-1).mean().sqrt())  # root average over region
    stats["KL"] = float(kl.sum() / counts.sum())  # in units of nats / observation
    logger.info("KL = {KL:0.4g}, MAE = {MAE:0.4g}, RMSE = {RMSE:0.4g}".format(**stats))

    # Examine the MSE and RMSE over a few regions of interest.
    queries = {
        "England": ["B.1.1.7"],
        # "England": ["B.1.1.7", "B.1.177", "B.1.1", "B.1"],
        # "USA / California": ["B.1.1.7", "B.1.429", "B.1.427", "B.1.2", "B.1", "P.1"],
    }
    for place, lineages in queries.items():
        matches = [p for name, p in dataset["location_id"].items() if place in name]
        if not matches:
            continue
        assert len(matches) == 1, matches
        p = matches[0]
        stats[f"{place} KL"] = float(kl[p].sum() / true[:, p].sum())
        stats[f"{place} MAE"] = float(mae[p].sum())
        stats[f"{place} RMSE"] = float(mse[p].sum().sqrt())
        logger.info(
            "{}\tKL = {:0.3g}, MAE = {:0.3g}, RMSE = {:0.3g}".format(
                place,
                stats[f"{place} KL"],
                stats[f"{place} MAE"],
                stats[f"{place} RMSE"],
            )
        )

        for lineage in lineages:
            i = dataset["lineage_id"][lineage]
            stats[f"{place} {lineage} MAE"] = mae[p, i]
            stats[f"{place} {lineage} RMSE"] = mse[p, i].sqrt()
            logger.info(
                "{} {}\tMAE = {:0.3g}, RMSE = {:0.3g}".format(
                    place,
                    lineage,
                    stats[f"{place} {lineage} MAE"],
                    stats[f"{place} {lineage} RMSE"],
                )
            )

    return {k: float(v) for k, v in stats.items()}


@torch.no_grad()
def log_holdout_stats(fits: dict) -> dict:
    """
    Logs statistics comparing multiple results from ``fit_svi``.
    """
    assert len(fits) > 1
    fits = list(fits.items())
    stats = {}
    for i, (name1, fit1) in enumerate(fits[:-1]):
        for name2, fit2 in fits[i + 1 :]:
            # Compute mutation similarity.
            mutations = sorted(set(fit1["mutations"]) & set(fit2["mutations"]))
            medians = []
            for fit in (fit1, fit2):
                mutation_id = {m: i for i, m in enumerate(fit["mutations"])}
                idx = torch.tensor([mutation_id[m] for m in mutations])
                medians.append(fit["median"]["coef"][idx] * 0.01)
            error = medians[0] - medians[1]
            mutation_std = torch.cat(medians).std().item()
            mutation_rmse = error.square().mean().sqrt().item()
            mutation_mae = error.abs().mean().item()
            mutation_correlation = pearson_correlation(medians[0], medians[1]).item()

            # Compute lineage similarity.
            means = []
            for fit in (fit1, fit2):
                rate = fit["mean"]["rate"]
                if rate.dim() == 2:
                    rate = rate.mean(0)
                means.append(rate)
            error = means[0] - means[1]
            lineage_std = torch.cat(means).std().item()
            lineage_rmse = error.square().mean().sqrt().item()
            lineage_mae = error.abs().mean().item()
            lineage_correlation = pearson_correlation(means[0], means[1]).item()

            # Print stats.
            logger.info(
                f"{name1} vs {name2} mutations: "
                f"ρ = {mutation_correlation:0.3g}, "
                f"RMSE = {mutation_rmse:0.3g}, "
                f"MAE = {mutation_mae:0.3g}"
            )
            logger.info(
                f"{name1} vs {name2} lineages: "
                f"ρ = {lineage_correlation:0.3g}, "
                f"RMSE = {lineage_rmse:0.3g}, "
                f"MAE = {lineage_mae:0.3g}"
            )

            # Save stats.
            stats["mutation_corr"] = mutation_correlation
            stats["mutation_rmse"] = mutation_rmse
            stats["mutation_mae"] = mutation_mae
            stats["mutation_stddev"] = mutation_std
            stats["lineage_corr"] = lineage_correlation
            stats["lineage_rmse"] = lineage_rmse
            stats["lineage_mae"] = lineage_mae
            stats["lineage_stdev"] = lineage_std

    return {k: float(v) for k, v in stats.items()}
