#!/usr/bin/env python3

__copyright__ = "Copyright (C) 2021 Davide Rossi"
__license__ = "GPL-3.0-or-later"

import csv
import sys
import os
import click
from collections import defaultdict

@click.command(
    help = f"""
    A script to output in tabular form the result of a query
    """
)
@click.option(
    "-t", 
    "--threshold",
    type = int,
    default = 0,
    help = "minimum threshold for a value"
)
@click.option(
    "-s",
    "--summary_only",
    is_flag = True,
    help = "only output summary table"
)
@click.option(
    "-x",
    "--sort_by_total",
    is_flag = True,
    default = False,
    help = "sort output zones by total commits"
)
@click.option(
    "-c",
    "--columns",
    type = str,
    default = None,
    help = "columns to output (comma-separated list)"
)
@click.option(
    "-r",
    "--ratio_line",
    is_flag = True,
    default = False,
    help = "add a ratio line to the summary"
)

def main(summary_only, threshold, sort_by_total, columns, ratio_line):
    zones = []
    years = []
    data = {}
    csv_reader = csv.reader(sys.stdin, delimiter='\t')
    for csv_record in csv_reader:
        year, zone, cnt, m, f = csv_record
        data[(year, zone)] = (int(cnt), int(m), int(f))
        zones.append(zone)
        years.append(year)
    years = list(set(years))
    years.sort()
    zones = list(set(zones))
    zones.sort()
    f_perc = {}
    zones_cnt = defaultdict(int)
    total_cnt = 0
    for year in years:
        for zone in zones:
            if (year, zone) in data:
                cnt, m, f = data[(year, zone)]
                f_perc[(year, zone)] = float(f)/(f+m) if f+m > threshold else 0
            else:
                cnt, m, f = (0, 0, 0)
                f_perc[(year, zone)] = 0.0
            zones_cnt[zone] += cnt
            total_cnt += cnt
            if not summary_only:
                print(f'{year}\t{zone}\t{cnt}\t{m}\t{f}')

    if not summary_only:
        print()

    if sort_by_total:
        zones = sorted(zones_cnt, key=zones_cnt.get, reverse=True)
    if columns is not None:
        zones = columns.split(',')
    for zone in zone:
        print(f'\t{zone}', end='')
    print()
    for year in years:
        print(f'{year}\t', end='')
        sep = ''
        for zone in zones:
            print(f'{sep}{f_perc[(year, zone)]}', end='')
            sep = '\t'
        print()
    if ratio_line:
        print(f'ratio', end='')
        for zone in zones:
            print(f'\t{zones_cnt[zone]/float(total_cnt) if total_cnt != 0 else 0}', end='')
        print()

if __name__ == "__main__":
    main() #pylint: disable=no-value-for-parameter
