#!/usr/bin/env python3

__copyright__ = "Copyright (C) 2020-2022 Stefano Zacchiroli"
__license__ = "GPL-3.0-or-later"

import click
import matplotlib.pyplot as plt
import numpy as np
import pandas as pd


COLOR_all = "0.6"  # all commits
COLOR_fit = "0.1"  # fit lines
FIT_STYLE = {"linestyle": ":", "marker": "", "color": COLOR_fit}


def year_plot(df, *args, **kwargs):
    plt.figure()
    ax = plt.axes()
    plot = df.plot(ax=ax, *args, **kwargs)

    xlabels = [str(y) if y % 5 == 0 else "" for y in df["year"]]
    ax.set_xticklabels(xlabels)

    return plot


def bump_xlim(plot, adjustment=0.5):
    xlim_l, xlim_r = plot.get_xlim()
    plot.set_xlim(xlim_l - adjustment, xlim_r + adjustment)


@click.command(
    help="""\
Plot yearly commit (or author) totals

Example: ./plot_yearly_totals.py -i yearly-commits.csv -o figures/yearly-commits.pdf

"""
)
@click.option(
    "--input-csv",
    "-i",
    type=click.Path(exists=True, dir_okay=False, readable=True),
    default="yearly-commits.csv",
    show_default=True,
    help="input CSV file with <year, count> columns and header",
)
@click.option(
    "--output-pdf",
    "-o",
    type=click.Path(dir_okay=False, writable=True),
    default="figures/yearly-commits.pdf",
    show_default=True,
    help="output PDF file",
)
@click.option(
    "--label",
    "-l",
    type=str,
    default="commits",
    show_default=True,
    help="label for data series",
)
@click.option(
    "--min-year",
    type=int,
    default=1971,
    show_default=True,
    help="minimum year to consider",
)
@click.option(
    "--max-year",
    type=int,
    default=2020,
    show_default=True,
    help="maximum year to consider",
)
def main(input_csv, output_pdf, label, min_year, max_year):
    df = pd.read_csv(input_csv)
    df = df[df["year"] >= min_year][df["year"] <= max_year]  # year selection

    total_fit_params = np.polyfit(df["year"], np.log(df["count"]), 1)
    df["total_fit"] = np.exp(total_fit_params[1]) * np.exp(
        total_fit_params[0] * df["year"]
    )
    total_plot = year_plot(
        df,
        kind="bar",
        x="year",
        y="count",
        logy=True,
        label="trend (exp. fit)",
        color=COLOR_all,
    )
    df.plot(
        ax=total_plot.axes,
        y="total_fit",
        label="commits",
        **FIT_STYLE,
    )
    bump_xlim(total_plot)
    plt.gcf().savefig(output_pdf)


if __name__ == "__main__":
    main()
