/*
 * File:   util.h
 *
 * Contains some useful defines and auxiliary methods.
 *
 * Author: Jan Goedgebeur
 *
 */

#ifndef _UTIL_H
#define	_UTIL_H

/****************************Defines for timing********************************/

#include <time.h>
#if !defined(CLK_TCK) && !defined(_SC_CLK_TCK)
#include <unistd.h>
#endif
#if !defined(CLK_TCK) && defined(_SC_CLK_TCK)
#define CLK_TCK sysconf(_SC_CLK_TCK)
#endif
#if !defined(CLK_TCK) && defined(CLOCKS_PER_SEC)
#define CLK_TCK CLOCKS_PER_SEC
#endif
#if !defined(CLK_TCK)
#define CLK_TCK 60     /* If the CPU time stated by the program appears
                      to be out by a constant ratio, the most likely
                      explanation is that the code got to this point but
                      60 is the wrong guess.  Another common value is 100. */
#endif


/**********************************Defines*************************************/

//#define MAXN 255
//#define MAXN 64

//Max 42 vertices, since we also use edge labels (42*3/2=63)
#define MAXN 42

#define REG 3

//Actually MAXN-1 is enough for the number of neighbours
typedef unsigned char GRAPH[MAXN][MAXN];
typedef unsigned char ADJACENCY[MAXN];

typedef unsigned char EDGE[2];

//Slightly faster when storing this in a global variable instead of passing as a param
GRAPH current_graph;
ADJACENCY adj;

//nv of the graph
unsigned char nv = 0;


/* Some marks */
unsigned int marks[MAXN];

#define MAXVAL INT_MAX - 1
static int markvalue = MAXVAL;
#define RESETMARKS {int mki; if ((markvalue += 1) > MAXVAL) \
      { markvalue = 1; for (mki=0;mki<MAXN;++mki) marks[mki]=0;}}
#define MARK(v) marks[v] = markvalue
#define UNMARK(v) marks[v] = markvalue - 1
#define ISMARKED(v) (marks[v] == markvalue)


/***************************Useful auxiliary methods***************************/


static void
add_edge(unsigned char adj[], unsigned char v, unsigned char w) {
    current_graph[v][adj[v]] = w;
    current_graph[w][adj[w]] = v;
    adj[v]++;
    adj[w]++;
}

/******************************************************************************/

/**
 * Decodes the code (which is in multicode format) of a graph.
 */
static void
decode_multicode(unsigned char code[], int codelength) {
    int i;

    if(nv != code[0]) {
        fprintf(stderr, "Error: Wrong number of vertices: expected %d while found %d vertices \n", nv, code[0]);
        exit(1);
    }
    //graph[0][0] = number_of_vertices;

    for(i = 0; i < nv; i++) {
        adj[i] = 0;
    }

    int currentvertex = 1;
    //Codelength
    for(i = 1; i < codelength; i++) {
        if(code[i] == 0) {
            currentvertex++;
        } else {
            add_edge(adj, currentvertex - 1, code[i] - 1);
        }
    }

    for(i = 0; i < nv; i++) {
        if(adj[i] != REG) {
            fprintf(stderr, "Error: every vertex of the graph should have degree %d (while found %d)\n", REG, adj[i]);
            exit(1);
        }

    }

}

/******************************************************************************/

void printgraph(GRAPH graph, ADJACENCY adj) {
    int i, j;
    fprintf(stderr, "Printing graph: \n");
    for(i = 0; i < nv; i++) {
        fprintf(stderr, "%d :", i);
        for(j = 0; j < adj[i]; j++)
            fprintf(stderr, " %d", graph[i][j]);
        fprintf(stderr, "\n");
    }
}


#endif	/* _UTIL_H */