/**
 * edge-kempe-switching.c
 *
 * Edge-kempe-switching is a program which, for every cubic input graph G,
 * first determines all 3-edge-colourings of G, then performs all edge-Kempe switchings
 * and finally outputs statistics on the number of edge-Kempe equivalence classes.
 *
 * Author: Jan Goedgebeur
 * In collaboration with Patric Ostergard
 *
 * ----------------------------------------
 *
 * Copyright (c) 2021-2022 KU Leuven
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/**
 * Compile with:
 *
 * cc -O3 -mpopcnt edge-kempe-switching.c -o edge-kempe-switching
 *
 */

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/times.h>
#include "util.h"

/*************************Defines and global variables*************************/

#define POPC(x) (__builtin_popcountll(x))
#define BIT(i) (1ULL << (i))

unsigned char edge_index[MAXN][MAXN];

//upper bound max num colourings: 2^{n/2+1} for a cubic graph on n vertices
//upper bound max num colourings (without permutations): (9/4*2^n/2)/6
//this was proven by Bessy and Havet in 2013
//in practice the max num colourings is even significantly slower
// n    bound
//12	24
//14	48
//16	96
//18	192
//20	384
//22	768
//24	1,536
//26	3,072
//28	6,144
//30	12,288
//32	24,576
#define MAX_NUM_COLOURINGS 50000
//#define MAX_NUM_COLOURINGS 100000

//Use malloc instead!
//unsigned long long int edge_bitvector_colour1[MAX_NUM_COLOURINGS];
//unsigned long long int edge_bitvector_colour2[MAX_NUM_COLOURINGS];
//unsigned long long int edge_bitvector_colour3[MAX_NUM_COLOURINGS];

//unsigned long long int (*edge_bitvector_colour1)[MAX_NUM_COLOURINGS];
//unsigned long long int (*edge_bitvector_colour2)[MAX_NUM_COLOURINGS];
//unsigned long long int (*edge_bitvector_colour3)[MAX_NUM_COLOURINGS];

//For union find
//Use malloc to be safe
//int colouring_orbits[MAX_NUM_COLOURINGS];
//int colouring_orbits_size[MAX_NUM_COLOURINGS];
int *colouring_orbits;
int *colouring_orbits_size;
int number_of_colouring_orbits;


//0,1,2,3 and 0 is not used
unsigned long long int current_colouring_bitvector[4];

int num_colourings = 0;
int max_num_colourings_found = 0;

typedef struct {
    unsigned long long int bitvector[3];
} EDGE_COLOUR_STRUCT;

EDGE_COLOUR_STRUCT *edge_bitvector_colours;


/* Some variables and marks used to test the colourability of graphs */
//#define REG 3

unsigned char number_of_colours_snarks[MAXN];
unsigned char colours_snarks[MAXN][REG];
unsigned char neighbour_index[MAXN][MAXN];

static int markvalue_snarks = MAXVAL;
unsigned int marks_snarks[MAXN][REG];
#define RESETMARKS_SNARKS {int mki, mkj; if ((markvalue_snarks += 1) > MAXVAL) \
      { markvalue_snarks = 1; for(mki=0;mki<MAXN;++mki) for(mkj=0;mkj<REG;++mkj) marks_snarks[mki][mkj]=0;}}
#define MARK_SNARKS(v, w) marks_snarks[v][w] = markvalue_snarks
#define UNMARK_SNARKS(v, w) marks_snarks[v][w] = markvalue_snarks - 1
#define ISMARKED_SNARKS(v, w) (marks_snarks[v][w] == markvalue_snarks)

int current_number_of_edges = 0;

/* Marks for the colour cycles */
static int markvalue_cycle_colour1 = MAXVAL;
unsigned int marks_cycle_colour1[MAXN][REG];
#define RESETMARKS_CYCLE_COLOUR1 {int mki, mkj; if ((markvalue_cycle_colour1 += 1) > MAXVAL) \
      { markvalue_cycle_colour1 = 1; for(mki=0;mki<MAXN;++mki) for(mkj=0;mkj<REG;++mkj) marks_cycle_colour1[mki][mkj]=0;}}
#define MARK_CYCLE_COLOUR1(v, w) marks_cycle_colour1[v][w] = markvalue_cycle_colour1
#define ISMARKED_CYCLE_COLOUR1(v, w) (marks_cycle_colour1[v][w] == markvalue_cycle_colour1)

static int markvalue_cycle_colour2 = MAXVAL;
unsigned int marks_cycle_colour2[MAXN][REG];
#define RESETMARKS_CYCLE_COLOUR2 {int mki, mkj; if ((markvalue_cycle_colour2 += 1) > MAXVAL) \
      { markvalue_cycle_colour2 = 1; for(mki=0;mki<MAXN;++mki) for(mkj=0;mkj<REG;++mkj) marks_cycle_colour2[mki][mkj]=0;}}
#define MARK_CYCLE_COLOUR2(v, w) marks_cycle_colour2[v][w] = markvalue_cycle_colour2
#define ISMARKED_CYCLE_COLOUR2(v, w) (marks_cycle_colour2[v][w] == markvalue_cycle_colour2)

static int markvalue_cycle_colour3 = MAXVAL;
unsigned int marks_cycle_colour3[MAXN][REG];
#define RESETMARKS_CYCLE_COLOUR3 {int mki, mkj; if ((markvalue_cycle_colour3 += 1) > MAXVAL) \
      { markvalue_cycle_colour3 = 1; for(mki=0;mki<MAXN;++mki) for(mkj=0;mkj<REG;++mkj) marks_cycle_colour3[mki][mkj]=0;}}
#define MARK_CYCLE_COLOUR3(v, w) marks_cycle_colour3[v][w] = markvalue_cycle_colour3
#define ISMARKED_CYCLE_COLOUR3(v, w) (marks_cycle_colour3[v][w] == markvalue_cycle_colour3)


#define find_next_vertex(v, colour) (colours_snarks[v][0] == colour ? current_graph[v][0] : colours_snarks[v][1] == colour ? current_graph[v][1] : current_graph[v][2])

#define TWOFACTOR_12 0
#define TWOFACTOR_13 1
#define TWOFACTOR_23 2

// two_factor_combo, cycle number, elements of cycle
int stored_twofactor_cycles[3][MAXN][MAXN];
int stored_twofactor_cycles_sizes[3][MAXN];
int num_stored_twofactor_cycles[3];


//For debugging

//#define _DEBUG //Comment to disable checks (should be slighly faster)

//Macros for debugging
#ifdef _DEBUG

#define DEBUGMSG(msg) fprintf(stderr, "%s\n", msg);

#define DEBUGASSERT(assertion) if(!(assertion)) {fprintf(stderr, "%s:%u Assertion failed: %s\n", __FILE__, __LINE__, #assertion); exit(1);}

#else

#define DEBUGMSG(msg)

#define DEBUGASSERT(assertion)

#endif


/******************************************************************************/

int comparator_edgecolours(const void *p, const void *q) {
    //unsigned long long int l = ((EDGE_COLOUR_STRUCT *)p)->bitvector[0];
    //unsigned long long int r = ((EDGE_COLOUR_STRUCT *)q)->bitvector[0];
    //return (l - r);

    int i;
    for(i = 0; i < 3; i++) {
        unsigned long long int l = ((EDGE_COLOUR_STRUCT *)p)->bitvector[i];
        unsigned long long int r = ((EDGE_COLOUR_STRUCT *)q)->bitvector[i];

        if(l < r)
            return -1;
        else if(l > r)
            return 1;
    }

    //fprintf(stderr, "equal\n");
    //Equal
    return 0;
}

/******************************************************************************/

int comparator_edgecolours_struct(const unsigned long long int bitvector0[3], const unsigned long long int bitvector1[3]) {
    int i;
    for(i = 0; i < 3; i++) {
        unsigned long long int l = bitvector0[i];
        unsigned long long int r = bitvector1[i];

        if(l < r)
            return -1;
        else if(l > r)
            return 1;
    }

    //fprintf(stderr, "equal\n");
    //Equal
    return 0;
}

/******************************************************************************/


/**
 * Returns 1 if current_vertex already has a coloured edge with colour "colour",
 * else returns 0.
 */
int is_conflicting_colouring(unsigned char colours[][REG], int current_vertex, int colour) {
    int i;
    for(i = 0; i < REG; i++) {
        if(ISMARKED_SNARKS(current_vertex, i) && colours[current_vertex][i] == colour)
            return 1;
    }
    return 0;

}

/**
 * Determines the 2 available colours which are different from used_colour.
 */
void determine_available_colours(int used_colour, EDGE available_colours) {
    switch(used_colour) {
        case 1:
            available_colours[0] = 2;
            available_colours[1] = 3;
            break;
        case 2:
            available_colours[0] = 1;
            available_colours[1] = 3;
            break;
        case 3:
            available_colours[0] = 1;
            available_colours[1] = 2;
            break;
        default:
            fprintf(stderr, "Error: invalid previous colour\n");
            exit(1);
    }
}

/**
 * It is assumed that vertex already has 2 coloured incident edges. This method
 * determines the uncoloured neighbour of vertex and the missing colour.
 */
void determine_uncoloured_vertex(int vertex, int *uncoloured_vertex, int *missing_colour) {
    DEBUGASSERT(number_of_colours_snarks[vertex] == 2)

    int i;
    int sum_colours = 0;
    for(i = 0; i < REG; i++) {
        if(!ISMARKED_SNARKS(vertex, i)) {
            *uncoloured_vertex = current_graph[vertex][i];
        } else {
            sum_colours += colours_snarks[vertex][i];
        }
    }
    switch(sum_colours) {
        case 3:
            *missing_colour = 3;
            break;
        case 4:
            *missing_colour = 2;
            break;
        case 5:
            *missing_colour = 1;
            break;
        default:
            fprintf(stdout, "Error: invalid sum_colours\n");
            exit(1);
    }

}

/**
 * Resets the colour of the edges which were coloured by label_nonfree_choices.
 */
void unmark_colours(EDGE nonfree_labelled[], int nonfree_labelled_size) {
    int i;
    int vertex0, vertex1;
    for(i = 0; i < nonfree_labelled_size; i++) {
        vertex0 = nonfree_labelled[i][0];
        vertex1 = nonfree_labelled[i][1];
        UNMARK_SNARKS(vertex0, neighbour_index[vertex0][vertex1]);
        UNMARK_SNARKS(vertex1, neighbour_index[vertex1][vertex0]);

        current_colouring_bitvector[colours_snarks[vertex0][neighbour_index[vertex0][vertex1]]] &= ~BIT(edge_index[vertex0][vertex1]);

        number_of_colours_snarks[vertex0]--;
        number_of_colours_snarks[vertex1]--;
    }
}

/**
 * Labels all edges with the colour that was imposed by the current colouring
 * (i.e. if a vertex has 2 colours, it's third neighbour must be coloured with the
 * missing colour).
 *
 * The edges which were coloured like this, are put in nonfree_labelled.
 */
int label_nonfree_choices(int current_vertex, EDGE nonfree_labelled[], int *nonfree_labelled_size) {
    int uncoloured_vertex;
    int missing_colour;
    while(number_of_colours_snarks[current_vertex] == 2) {
        determine_uncoloured_vertex(current_vertex, &uncoloured_vertex, &missing_colour);
        DEBUGASSERT(missing_colour > 0 && missing_colour < 4)
        if(!is_conflicting_colouring(colours_snarks, uncoloured_vertex, missing_colour)) {
            int index_uncoloured_vertex = neighbour_index[current_vertex][uncoloured_vertex];
            int index_current_vertex = neighbour_index[uncoloured_vertex][current_vertex];
            colours_snarks[current_vertex][index_uncoloured_vertex] = missing_colour;
            colours_snarks[uncoloured_vertex][index_current_vertex] = missing_colour;

            current_colouring_bitvector[missing_colour] |= BIT(edge_index[current_vertex][uncoloured_vertex]);

            MARK_SNARKS(current_vertex, index_uncoloured_vertex);
            MARK_SNARKS(uncoloured_vertex, index_current_vertex);
            number_of_colours_snarks[current_vertex] = 3;
            number_of_colours_snarks[uncoloured_vertex]++;

            nonfree_labelled[*nonfree_labelled_size][0] = current_vertex;
            nonfree_labelled[*nonfree_labelled_size][1] = uncoloured_vertex;
            (*nonfree_labelled_size)++;

            current_vertex = uncoloured_vertex;
        } else {
            unmark_colours(nonfree_labelled, *nonfree_labelled_size);
            return 0;
        }
    }
    return 1;

}

/******************************************************************************/

//Canonical form: ascending order
void transform_triple_into_canonical_form(unsigned long long int bitvector_array[3]) {
    unsigned long long int temp;
    if(bitvector_array[0] > bitvector_array[1]) {
        temp = bitvector_array[0];
        bitvector_array[0] = bitvector_array[1];
        bitvector_array[1] = temp;
    }

    if(bitvector_array[1] > bitvector_array[2]) {
        temp = bitvector_array[1];
        bitvector_array[1] = bitvector_array[2];
        bitvector_array[2] = temp;
    }

    if(bitvector_array[0] > bitvector_array[1]) {
        temp = bitvector_array[0];
        bitvector_array[0] = bitvector_array[1];
        bitvector_array[1] = temp;
    }
}

/******************************************************************************/

void store_colouring() {
    //First run some checks
    int total_edges_coloured = 0;
    unsigned long long int union_bitvectors = 0ULL;
    int i;
    for (i = 0; i < REG; i++) {
        total_edges_coloured += POPC(current_colouring_bitvector[i + 1]);
        union_bitvectors |= current_colouring_bitvector[i + 1];
    }
    if(total_edges_coloured != 3*nv / 2) {
        fprintf(stderr, "Error: invalid num edges coloured! %d vs %d\n", total_edges_coloured, 3*nv/2);
        exit(1);
    }

    unsigned long long int expected_bitvector = BIT(current_number_of_edges) - 1;

    if(union_bitvectors != expected_bitvector) {
        fprintf(stderr, "Error: problem with colouring bitvector: %llu vs %llu\n", union_bitvectors, expected_bitvector);
        exit(1);
    }

    if(num_colourings == MAX_NUM_COLOURINGS) {
        fprintf(stderr, "Error: too many colourings, please increase MAX_NUM_COLOURINGS (%d)\n", MAX_NUM_COLOURINGS);
        exit(1);
    }


    //Store colouring in canonical form
    edge_bitvector_colours[num_colourings].bitvector[0] = current_colouring_bitvector[1];
    edge_bitvector_colours[num_colourings].bitvector[1] = current_colouring_bitvector[2];
    edge_bitvector_colours[num_colourings].bitvector[2] = current_colouring_bitvector[3];

    //fprintf(stderr, "Original colouring:\n");
    //fprintf(stderr, "Colouring: %llu %llu %llu\n", edge_bitvector_colours[num_colourings].bitvector[0], edge_bitvector_colours[num_colourings].bitvector[1], edge_bitvector_colours[num_colourings].bitvector[2]);

    transform_triple_into_canonical_form(edge_bitvector_colours[num_colourings].bitvector);

    //fprintf(stderr, "canon form:\n");
    //fprintf(stderr, "Colouring: %llu %llu %llu\n", edge_bitvector_colours[num_colourings].bitvector[0], edge_bitvector_colours[num_colourings].bitvector[1], edge_bitvector_colours[num_colourings].bitvector[2]);

    num_colourings++;
}

/******************************************************************************/

/**
 * Main (recursive) method of the colour routine.
 */
int colour_next_free_choice(int number_of_coloured_edges) {
    if(number_of_coloured_edges != current_number_of_edges) {
        int current_vertex;
        for(current_vertex = 1; current_vertex < nv; current_vertex++) {
            if(number_of_colours_snarks[current_vertex] == 1) {
                break;
            }
            DEBUGASSERT(number_of_colours_snarks[current_vertex] != 2)
        }
        DEBUGASSERT(current_vertex < nv)

        int used_colour;
        int i;
        for(i = 0; i < REG; i++) {
            if(ISMARKED_SNARKS(current_vertex, i)) {
                used_colour = colours_snarks[current_vertex][i];
                break;
            }
        }
        DEBUGASSERT(i < REG)

        EDGE available_vertices;
        int index_available_vertex0 = (i + 1) % 3;
        int index_available_vertex1 = (i + 2) % 3;
        available_vertices[0] = current_graph[current_vertex][index_available_vertex0];
        available_vertices[1] = current_graph[current_vertex][index_available_vertex1];

        EDGE available_colours;
        determine_available_colours(used_colour, available_colours);

        DEBUGASSERT((available_colours[0] > 0 && available_colours[0] < 4) && (available_colours[1] > 0 && available_colours[1] < 4))

        EDGE nonfree_labelled[current_number_of_edges - number_of_coloured_edges];
        int j;
        for(i = 0; i < 2; i++) {
            if(is_conflicting_colouring(colours_snarks, available_vertices[0], available_colours[i]) ||
               is_conflicting_colouring(colours_snarks, available_vertices[1], available_colours[(i + 1) % 2])) {
                continue;
            }

            int index_current_vertex0 = neighbour_index[available_vertices[0]][current_vertex];
            int index_current_vertex1 = neighbour_index[available_vertices[1]][current_vertex];

            colours_snarks[available_vertices[0]][index_current_vertex0] = available_colours[i];
            colours_snarks[available_vertices[1]][index_current_vertex1] = available_colours[(i + 1) % 2];

            MARK_SNARKS(available_vertices[0], index_current_vertex0);
            MARK_SNARKS(available_vertices[1], index_current_vertex1);

            //number_of_colours_snarks[current_vertex] = 3;
            number_of_colours_snarks[available_vertices[0]]++;
            number_of_colours_snarks[available_vertices[1]]++;

            int nonfree_labelled_size = 0;
            int abort = 0;
            for(j = 0; j < 2;j++) {
                if(!label_nonfree_choices(available_vertices[j], nonfree_labelled, &nonfree_labelled_size)) {
                    DEBUGASSERT(nonfree_labelled_size <= current_number_of_edges - number_of_coloured_edges)
                    abort = 1;
                    break;
                }
                DEBUGASSERT(nonfree_labelled_size <= current_number_of_edges - number_of_coloured_edges)
            }

            if(!abort) {
                colours_snarks[current_vertex][index_available_vertex0] = available_colours[i];
                colours_snarks[current_vertex][index_available_vertex1] = available_colours[(i + 1) % 2];
                MARK_SNARKS(current_vertex, index_available_vertex0);
                MARK_SNARKS(current_vertex, index_available_vertex1);
                number_of_colours_snarks[current_vertex] = 3;

                current_colouring_bitvector[available_colours[i]] |= BIT(edge_index[current_vertex][available_vertices[0]]);
                current_colouring_bitvector[available_colours[(i + 1) % 2]] |= BIT(edge_index[current_vertex][available_vertices[1]]);

//                if(colour_next_free_choice(number_of_coloured_edges + nonfree_labelled_size + 2)) {
//                    return 1;
//                } else {
//                    unmark_colours(nonfree_labelled, nonfree_labelled_size);
//                }

                //Simply continue colouring, even when a colouring was already found
                colour_next_free_choice(number_of_coloured_edges + nonfree_labelled_size + 2);

                unmark_colours(nonfree_labelled, nonfree_labelled_size);

                current_colouring_bitvector[available_colours[i]] &= ~BIT(edge_index[current_vertex][available_vertices[0]]);
                current_colouring_bitvector[available_colours[(i + 1) % 2]] &= ~BIT(edge_index[current_vertex][available_vertices[1]]);

                UNMARK_SNARKS(current_vertex, index_available_vertex0);
                UNMARK_SNARKS(current_vertex, index_available_vertex1);
                number_of_colours_snarks[current_vertex] = 1;
            }
            UNMARK_SNARKS(available_vertices[0], index_current_vertex0);
            UNMARK_SNARKS(available_vertices[1], index_current_vertex1);

            //number_of_colours_snarks[current_vertex] = 1;
            number_of_colours_snarks[available_vertices[0]]--;
            number_of_colours_snarks[available_vertices[1]]--;
        }
        return 0;
    } else {
        //3-edge-colouring found, store it in canonical form!
        store_colouring();

        return 1;
    }
}

/******************************************************************************/

void init_is_colourable(unsigned char number_of_colours[]) {
    RESETMARKS_SNARKS
    int i, j;
    for (i = 0; i < nv; i++) {
        number_of_colours[i] = 0;
        for(j = 0; j < adj[i]; j++) {
            neighbour_index[i][current_graph[i][j]] = j;
        }
    }

}

/******************************************************************************/

void determine_and_store_all_3_edge_colourings() {
    num_colourings = 0;

    init_is_colourable(number_of_colours_snarks);

    //We can assume wlog that the first three edges have colours 1, 2, 3
    //Since else we will receive duplicate colourings
    //All colourings are found since we store the colourings in canonical form

    int current_vertex = 0;
    int i, neighbour, current_index;
    for(i = 0; i < REG; i++) {
        colours_snarks[current_vertex][i] = i + 1;
        neighbour = current_graph[current_vertex][i];
        current_index = neighbour_index[neighbour][current_vertex];
        colours_snarks[neighbour][current_index] = i + 1;

        current_colouring_bitvector[i + 1] = BIT(edge_index[current_vertex][neighbour]);

        MARK_SNARKS(current_vertex, i);
        MARK_SNARKS(neighbour, current_index);
        number_of_colours_snarks[neighbour] = 1;
    }
    number_of_colours_snarks[current_vertex] = 3;

    //Info: still ok when there is a triangle, only checks for forced colours if an endpoint of an edge already has 2 coloured edges

    colour_next_free_choice(3);

    for (i = 0; i < REG; i++) {
        if(POPC(current_colouring_bitvector[i + 1]) != 1) {
            fprintf(stderr, "Error: edge colour bitvector was not correctly reset!\n");
            exit(1);
        }
    }

    //fprintf(stderr, "num_colourings: %d\n", num_colourings);

    if(num_colourings > max_num_colourings_found)
        max_num_colourings_found = num_colourings;

}

/******************************************************************************/

int determine_colour_of_edge(unsigned long long int edge_bitvector_colours[3], int v1, int v2) {
    int edge_lab = edge_index[v1][v2];
    for (int i = 0; i < 3; ++i) {
        if((edge_bitvector_colours[i] & BIT(edge_lab)) != 0ULL)
            return i+1;
    }

    fprintf(stderr, "Error: edge not found in edge bitvector!\n");
    exit(1);
}

//#define determine_colour_of_edge_macro(bitvector, edge_lab) ((bitvector[0] & BIT(edge_lab)) != 0ULL ? 1 : (bitvector[1] & BIT(edge_lab)) != 0ULL ? 2 : 3)


/******************************************************************************/

//Important: it is assumed that neighbour_index and edge_index were already set!

void fill_colours_snarks_based_on_bitvector_colouring(unsigned long long int edge_bitvector_colours[3]) {
    //edge_index + neighbour_index[][] are still valid since it's the same graph

    for (int i = 0; i < nv; ++i) {
        for (int j = 0; j < adj[i]; ++j) {
            int neighbour = current_graph[i][j];
            if(i < neighbour) {
                //Determine and store colour of edge {i, neibour}
                //Is call by reference, ok!
                int edge_colour = determine_colour_of_edge(edge_bitvector_colours, i, neighbour);
                //Macro is only a bit faster
//                int edge_colour2 = determine_colour_of_edge_macro(edge_bitvector_colours, edge_index[i][neighbour]);
//                if(edge_colour != edge_colour2) {
//                    fprintf(stderr, "Error: different colours\n");
//                    exit(1);
//                }
                colours_snarks[i][j] = edge_colour; //vertex nbr_index
                colours_snarks[neighbour][neighbour_index[neighbour][i]] = edge_colour;
            }
        }
    }

}

/******************************************************************************/

/**
 * Returns 1 if the edge was already marked by a certain colour,
 * else returns 0.
 */
int ismarked_colour(int from, int index_other, int colour) {
    if(colour == 1) {
        return ISMARKED_CYCLE_COLOUR1(from, index_other);
    } else if(colour == 2) {
        return ISMARKED_CYCLE_COLOUR2(from, index_other);
    } else if(colour == 3) {
        return ISMARKED_CYCLE_COLOUR3(from, index_other);
    } else {
        fprintf(stderr, "Error: invalid colour: %d\n", colour);
        exit(1);
    }

}

/******************************************************************************/

/**
 * Determines an even colour cycle of colours[0] and colours[1] which contains
 * start_edge.
 * Important: it is assumed that start_edge has colour colours[0].
 */
void determine_even_cycle(EDGE start_edge, EDGE colours, int cycle[], int *cycle_size) {
    DEBUGASSERT(colours_snarks[start_edge[0]][neighbour_index[start_edge[0]][start_edge[1]]] == colours[0]);

    cycle[0] = start_edge[0];
    //cycle[1] = start_edge[1];
    *cycle_size = 1;
    int current_vertex = start_edge[1];
    int current_colour_index = 1;
    while(current_vertex != start_edge[0]) {
        cycle[*cycle_size] = current_vertex;
        (*cycle_size)++;
        current_vertex = find_next_vertex(current_vertex, colours[current_colour_index]);
        current_colour_index = (current_colour_index + 1) % 2;
    }

    //Cycle will always be even because of alternating colours
    DEBUGASSERT(*cycle_size % 2 == 0);
}

/******************************************************************************/

/**
 * Marks the colour of an edge.
 * Remark: it doesnt matter if from is > or < to.
 */
void mark_colour(int from, int to, int colour) {
    int to_index = neighbour_index[from][to];
    int from_index = neighbour_index[to][from];
    if(colour == 1) {
        MARK_CYCLE_COLOUR1(from, to_index);
        MARK_CYCLE_COLOUR1(to, from_index);
    } else if(colour == 2) {
        MARK_CYCLE_COLOUR2(from, to_index);
        MARK_CYCLE_COLOUR2(to, from_index);
    } else if(colour == 3) {
        MARK_CYCLE_COLOUR3(from, to_index);
        MARK_CYCLE_COLOUR3(to, from_index);
    } else {
        fprintf(stderr, "Error: invalid colour: %d\n", colour);
        exit(1);
    }
}

/******************************************************************************/

int determine_twofactor_colour(int colour1, int colour2) {
    if((colour1 == 1 && colour2 == 2) || (colour1 == 2 && colour2 == 1))
        return TWOFACTOR_12;
    else if((colour1 == 1 && colour2 == 3) || (colour1 == 3 && colour2 == 1))
        return TWOFACTOR_13;
    else if((colour1 == 2 && colour2 == 3) || (colour1 == 3 && colour2 == 2))
        return TWOFACTOR_23;

    fprintf(stderr, "Error: invalid colour combination!\n");
    exit(1);
}

/******************************************************************************/

/**
 * Forms all colour cycles of current_graph.
 *
 * Important: current_graph must already be coloured before calling this method
 */
 //Cycles are stored into stored_twofactor_cycles[3][MAXN][MAXN];
 //Important: hamiltonian cycles are not stored, since swapping them does not yield a different colouring!
void search_and_store_all_even_colour_cycles() {
    //unsigned char cycle[nv];
    int cycle[nv];
    int neighbour, colour, cycle_size;
    EDGE available_colours, start_edge, temp_edge, colours_even_cycle;
    //EDGE temp_edge2;
    int i, j, k;

    RESETMARKS_CYCLE_COLOUR1;
    RESETMARKS_CYCLE_COLOUR2;
    RESETMARKS_CYCLE_COLOUR3;

     num_stored_twofactor_cycles[TWOFACTOR_12] = 0;
     num_stored_twofactor_cycles[TWOFACTOR_13] = 0;
     num_stored_twofactor_cycles[TWOFACTOR_23] = 0;

     for(i = 0; i < nv - 1; i++) {
        for(j = 0; j < REG; j++) {
            neighbour = current_graph[i][j];
            if(i < neighbour) {
                start_edge[0] = i;
                start_edge[1] = neighbour;
                colour = colours_snarks[i][j];
                colours_even_cycle[0] = colour;
                determine_available_colours(colour, available_colours);
                for(k = 0; k < 2; k++) {
                    if(!ismarked_colour(i, j, available_colours[k])) {
                        //i.e. this colour cycle with colours colour and available_colours[k] was not processed before
                        cycle_size = 0;
                        colours_even_cycle[1] = available_colours[k];
                        determine_even_cycle(start_edge, colours_even_cycle, cycle, &cycle_size);

                        //fprintf(stderr, "cycle found with length %d\n", cycle_size);
                        //fprintf(stderr, "colour cycle %d found: ", current_cycle_number);
                        int l;
                        for(l = 0; l < cycle_size; l++) {
                            temp_edge[0] = cycle[l];
                            temp_edge[1] = cycle[(l + 1) % cycle_size];
                            //transform_edge_into_canonical_form(temp_edge);
                            mark_colour(temp_edge[0], temp_edge[1], colours_even_cycle[(l + 1) % 2]);

                            //fprintf(stderr, "%d %d (col: %d) ", temp_edge[0], temp_edge[1], colours_even_cycle[l % 2]);

                            //Not important to know
                            //edge_in_cycles[edge_labels[temp_edge[0]][temp_edge[1]]] |= BIT(current_cycle_number);
                        }
                        //current_cycle_number++;

                        //Useless to swap if cycle is a ham cycle since this is exactly the a colouring with the same canon form!!!
                        if(cycle_size < nv) {
                            int twofactortype = determine_twofactor_colour(colour, available_colours[k]);

                            if(num_stored_twofactor_cycles[twofactortype] == MAXN) {
                                fprintf(stderr, "Error: too many cycles for stored_twofactor_cycles!\n");
                                exit(1);
                            }

                            memcpy(stored_twofactor_cycles[twofactortype][num_stored_twofactor_cycles[twofactortype]], cycle, sizeof (int) * cycle_size);
                            stored_twofactor_cycles_sizes[twofactortype][num_stored_twofactor_cycles[twofactortype]] = cycle_size;
                            num_stored_twofactor_cycles[twofactortype]++;

                        }

                    }
                }
            }
        }
    }
}

/******************************************************************************/

void swap_colours(int cycle[], int cycle_size, unsigned long long int colour_bitvectors_new[3], int update_bitvector) {

//    fprintf(stderr, "Cycle:\n");
//    for (int i = 0; i < cycle_size; i++) {
//        fprintf(stderr, "%d ", cycle[i]);
//    }
//    fprintf(stderr, "\n");
//
//    for (int i = 0; i < cycle_size; i++) {
//        fprintf(stderr, "colour of edge %d %d: %d\n", cycle[i], cycle[(i + 1) % cycle_size], colours_snarks[cycle[i]][neighbour_index[cycle[i]][cycle[(i + 1) % cycle_size]]]);
//    }

    EDGE colours;
    colours[0] = colours_snarks[cycle[0]][neighbour_index[cycle[0]][cycle[1]]];
    colours[1] = colours_snarks[cycle[1]][neighbour_index[cycle[1]][cycle[2]]];

    //fprintf(stderr, "Swapping colour %d and %d\n", colours[0], colours[1]);

    int i;
    for(i = 0; i < cycle_size; i++) {
        if(update_bitvector) {
            int index = edge_index[cycle[i]][cycle[(i + 1) % cycle_size]];
//            if((colour_bitvectors_new[colours[i % 2] - 1] & BIT(index)) == 0ULL) {
//                fprintf(stderr, "Error: colour was not present!\n");
//                exit(1);
//            }
//            if((colour_bitvectors_new[colours[(i + 1) % 2] - 1] & BIT(index)) != 0ULL) {
//                fprintf(stderr, "Error: colour was already present!\n");
//                exit(1);
//            }
            colour_bitvectors_new[colours[i % 2] - 1] &= ~BIT(index);
            colour_bitvectors_new[colours[(i + 1) % 2] - 1] |= BIT(index);
        }

//        if(colours_snarks[cycle[i]][neighbour_index[cycle[i]][cycle[(i + 1) % cycle_size]]] != colours[i % 2]) {
//            fprintf(stderr, "Error: wrong colour!\n");
//            exit(1);
//        }

        colours_snarks[cycle[i]][neighbour_index[cycle[i]][cycle[(i + 1) % cycle_size]]] = colours[(i + 1) % 2];
        colours_snarks[cycle[(i + 1) % cycle_size]][neighbour_index[cycle[(i + 1) % cycle_size]][cycle[i]]] = colours[(i + 1) % 2];
    }

//    fprintf(stderr, "New colours:\n");
//    for (int i = 0; i < cycle_size; i++) {
//        fprintf(stderr, "colour of edge %d %d: %d\n", cycle[i], cycle[(i + 1) % cycle_size], colours_snarks[cycle[i]][neighbour_index[cycle[i]][cycle[(i + 1) % cycle_size]]]);
//    }

}

/******************************************************************************/

//Colouring bitvector is assumed to be in canonical form!
int search_colouring_index_slow(unsigned long long int bitvector_array[3]) {
    for(int i = 0; i < num_colourings; i++)
        if (comparator_edgecolours_struct(edge_bitvector_colours[i].bitvector, bitvector_array) == 0) {
            return i;
        }

    fprintf(stderr, "Error: colouring not found!\n");
    exit(1);

}

int search_colouring_index_binary_search(unsigned long long int bitvector_array[3], int from, int to) {
    //Now while loop instead of recursion
    while(from <= to) {
        int mid = (from+to)/2;
        int compareresult = comparator_edgecolours_struct(edge_bitvector_colours[mid].bitvector, bitvector_array);
        if(compareresult == -1) {
            //return search_colouring_index_binary_search(bitvector_array, mid+1, to);
            from = mid + 1;
        } else if(compareresult == 1) {
            //return search_colouring_index_binary_search(bitvector_array, from, mid - 1);
            to = mid - 1;
        } else {
            return mid; //Equal to element at position mid
        }
    }
    fprintf(stderr, "Error: colouring not found2!\n");
    exit(1);

//    if(from > to) {
//             fprintf(stderr, "Error: colouring not found2!\n");
//            exit(1);
//    } else {
//        int mid = (from+to)/2;
//        int compareresult = comparator_edgecolours_struct(edge_bitvector_colours[mid].bitvector, bitvector_array);
//        if(compareresult == -1) {
//            return search_colouring_index_binary_search(bitvector_array, mid+1, to);
//        } else if(compareresult == 1) {
//            return search_colouring_index_binary_search(bitvector_array, from, mid - 1);
//        } else {
//            return mid; //Equal to element at position mid
//        }
//    }
    //Alternative:
//    if(from == to) {
//        if (comparator_edgecolours_struct(edge_bitvector_colours[from].bitvector, bitvector_array) == 0) {
//            return from;
//        } else {
//            fprintf(stderr, "Error: colouring not found2!\n");
//            exit(1);
//        }
//    } else {
//        int mid = (from+to)/2;
//        if(comparator_edgecolours_struct(edge_bitvector_colours[mid].bitvector, bitvector_array) == -1) {
//            return search_colouring_index_binary_search(bitvector_array, mid+1, to);
//        } else {
//            return search_colouring_index_binary_search(bitvector_array, from, mid);
//        }
//    }
}

/******************************************************************************/

/**
 * Find with simple path compression.
 */
int find_root(int orbits[], int i) {
    while(i != orbits[i]) {
        orbits[i] = orbits[orbits[i]]; //Simple variant of path compression
        i = orbits[i];
    }
    return i;

}

/**
 * Unions the roots of a and b (if they weren't already equal) using union by size.
 */
void union_elements(int orbits[], int root_orbits_size[], int *number_of_orbits, int a, int b) {
    //DEBUGASSERT(a != b);
    int root_a = find_root(orbits, a);
    int root_b = find_root(orbits, b);
    //fprintf(stderr, "Union %d and %d (roots: %d and %d)\n", a, b, root_a, root_b);
    if(root_a != root_b) { //Union by size
        if(root_orbits_size[root_a] < root_orbits_size[root_b]) {
            orbits[root_a] = root_b;
            root_orbits_size[root_b] += root_orbits_size[root_a];
        } else {
            orbits[root_b] = root_a;
            root_orbits_size[root_a] += root_orbits_size[root_b];
        }
        (*number_of_orbits)--;
    }
/*
    else {
        fprintf(stderr, "root of %d and %d are equal: %d\n", a, b, root_a);
    }
*/
}



/******************************************************************************/

//Important: colours_snarks and neighbour_index must be valid!
void recursively_switch_colour_cycles_of_2factor_in_all_possible_ways(int index_original_colouring,
                                                                      int twofactortype, int current_cycle) {

    //No need to swap all cycles since this will be the same colouring (permutation of colours)
    //Will be avoided when keeping first cycle constant

    for (int i = current_cycle; i < num_stored_twofactor_cycles[twofactortype]; i++) {

        //fprintf(stderr, "twofactortype %d\n", twofactortype);
        //fprintf(stderr, "Swapping cycle nr %d with size %d\n", i, stored_twofactor_cycles_sizes[twofactortype][i]);

        unsigned long long int colour_bitvectors_new[3];
        colour_bitvectors_new[0] = edge_bitvector_colours[index_original_colouring].bitvector[0];
        colour_bitvectors_new[1] = edge_bitvector_colours[index_original_colouring].bitvector[1];
        colour_bitvectors_new[2] = edge_bitvector_colours[index_original_colouring].bitvector[2];

        //fprintf(stderr, "Original colouring:\n");
        //fprintf(stderr, "Bitvector: %llu %llu %llu\n", colour_bitvectors_new[0], colour_bitvectors_new[1], colour_bitvectors_new[2]);

        //Switch
        swap_colours(stored_twofactor_cycles[twofactortype][i], stored_twofactor_cycles_sizes[twofactortype][i],
                     colour_bitvectors_new, 1);

        //fprintf(stderr, "Swapped colouring:\n");
        //fprintf(stderr, "Bitvector: %llu %llu %llu\n", colour_bitvectors_new[0], colour_bitvectors_new[1], colour_bitvectors_new[2]);

        //Transform colouring into canon form
        transform_triple_into_canonical_form(colour_bitvectors_new);

        //fprintf(stderr, "Canon form colouring:\n");
        //fprintf(stderr, "Bitvector: %llu %llu %llu\n", colour_bitvectors_new[0], colour_bitvectors_new[1], colour_bitvectors_new[2]);

        //Swap en restore swap werkt perfect! Gecheckt!

        //Using binary search is not faster than linear search since the number of colourings is still very small so far
        //int index_swapped_colouring = search_colouring_index_slow(colour_bitvectors_new);
        int index_swapped_colouring = search_colouring_index_binary_search(colour_bitvectors_new, 0, num_colourings);
//        if(index_swapped_colouring != index_swapped_colouring2) {
//            fprintf(stderr, "Error: different results normal vs binary search!\n");
//            exit(1);
//        }

        //Else the reverse swap was alread performed or will still performed later
        //Checked: is correct (but hardly any faster since if it already points to the root it is fast anyway...)
        if(index_original_colouring < index_swapped_colouring) {
            //Union find
            //each time when union -> num_colour_classes--

            //Important: init_union_find() must already have been called before!
            union_elements(colouring_orbits, colouring_orbits_size, &number_of_colouring_orbits,
                           index_original_colouring, index_swapped_colouring);

        }

        //Recursion
        recursively_switch_colour_cycles_of_2factor_in_all_possible_ways(index_original_colouring, twofactortype, i+1);

        //fprintf(stderr, "Reswapping to restore original colouring:\n");

        //Undo swap by swapping again (checked: works correctly)
        //No need to update bitvector, since it is not stored
        swap_colours(stored_twofactor_cycles[twofactortype][i], stored_twofactor_cycles_sizes[twofactortype][i],
                     colour_bitvectors_new, 0);
        //swap_colours(stored_twofactor_cycles[twofactortype][i], stored_twofactor_cycles_sizes[twofactortype][i],
          //           colour_bitvectors_new, 1);

        //fprintf(stderr, "Restored colouring:\n");
        //fprintf(stderr, "Bitvector: %llu %llu %llu\n", colour_bitvectors_new[0], colour_bitvectors_new[1], colour_bitvectors_new[2]);

    }

}

/******************************************************************************/

void init_union_find() {
    for (int i = 0; i < num_colourings; i++) {
        colouring_orbits[i] = i; //Every element initially points to itself
        colouring_orbits_size[i] = 1;
    }
    number_of_colouring_orbits = num_colourings;
}

/******************************************************************************/

void perform_edge_kempe_switchings() {

    init_union_find();

    //For every 3-edge-colouring
    for (int i = 0; i < num_colourings; i++) {
        //Takes some time 4.8 vs 3.9 sec
        fill_colours_snarks_based_on_bitvector_colouring(edge_bitvector_colours[i].bitvector);

        //Warning: HC's are not stored since this yields exactly the same colouring
        search_and_store_all_even_colour_cycles();

//        for (int j = 0; j < 3; j++) {
//            fprintf(stderr, "Printing cycles of twofactortype %d\n", j);
//            fprintf(stderr, "num_stored_twofactor_cycles[%d] = %d\n", j, num_stored_twofactor_cycles[j]);
//
//            for (int k = 0; k < num_stored_twofactor_cycles[j]; k++) {
//                fprintf(stderr, "stored_twofactor_cycles_sizes[j][k]: %d\n", stored_twofactor_cycles_sizes[j][k]);
//                for (int l = 0; l < stored_twofactor_cycles_sizes[j][k]; l++) {
//                    fprintf(stderr, "%d ", stored_twofactor_cycles[j][k][l]);
//                }
//                fprintf(stderr, "\n");
//            }
//        }


        //For every even 2-factor in colouring: cols 12, 13, and 23
        for (int j = 0; j < 3; j++) { //For every colour combination 12 13 23
            //recursively_switch_colour_cycles_of_2factor_in_all_possible_ways(i, j, 0);
            //Now not swapping first cycle:
            //Can wlog keep the first cycle fixed (switching all cycles of a 2-factor does not change a 1-factorisation)
            //Is faster
            recursively_switch_colour_cycles_of_2factor_in_all_possible_ways(i, j, 1);
        }

    }

    //Now store number of non-Kempe-equivalent colour classes in frequency table

    //Warning: it is possible that the number of equivalence classes is zero (i.e. uncolourable graph)


}

/******************************************************************************/

int determine_num_edge_kempe_equivalence_classes() {

    //Label the edges of the graph
    int edge_label = 0;
    int i, j;
    for(i = 0; i < nv; i++) {
        if(adj[i] != REG) {
            fprintf(stderr, "Error: graphs are expected to be cubic!\n");
            exit(1);
        }
        for (j = 0; j < adj[i]; j++) {
            if (i < current_graph[i][j]) {
                edge_index[i][current_graph[i][j]] = edge_label;
                edge_index[current_graph[i][j]][i] = edge_label;
                //fprintf(stderr, "edge %d %d has label %d\n", i, current_graph[i][j], edge_label);
                edge_label++;
            }
        }
    }
    if(edge_label != 3 * nv / 2) {
        fprintf(stderr, "Error: invalid number of edges \n");
        exit(1);
    }

    current_number_of_edges = edge_label;

    //TODO: alternative (indep algo): compute all perfect matchings and check for which ones the corresponding 2-factor
    //only consists of even cycles
    //Note: determining and storing all 3-edge-colourings is a not a bottleneck
    determine_and_store_all_3_edge_colourings();

    //Sort using quicksort


//    for(i = 0; i < num_colourings; i++) {
//        fprintf(stderr, "Colouring %d: %llu %llu %llu\n", i, edge_bitvector_colours[i].bitvector[0], edge_bitvector_colours[i].bitvector[1], edge_bitvector_colours[i].bitvector[2]);
//    }

    qsort(edge_bitvector_colours, num_colourings, sizeof(EDGE_COLOUR_STRUCT), comparator_edgecolours);

//    fprintf(stderr, "Colouring after sort:\n");
//    for(int i = 0; i < num_colourings; i++) {
//        fprintf(stderr, "Colouring %d: %llu %llu %llu\n", i, edge_bitvector_colours[i].bitvector[0], edge_bitvector_colours[i].bitvector[1], edge_bitvector_colours[i].bitvector[2]);
//    }

    //TODO: temp debugging check to make sure that there are no duplicate colourings
    for(i = 0; i < num_colourings - 1; i++) {
        if(comparator_edgecolours_struct(edge_bitvector_colours[i].bitvector, edge_bitvector_colours[i+1].bitvector) == 0) {
            fprintf(stderr, "Error: should not be same colour!\n");
            exit(1);
        }
    }

    perform_edge_kempe_switchings();

    //fprintf(stderr, "Number of colourings: %d and number of edge-Kempe equivalence classes: %d\n",
      //      num_colourings, number_of_colouring_orbits);

    //To make sure that every orbit points to its root, but this is not necessary (only for debugging purposes)
//    for(i = 0; i < num_colourings; i++) {
//        colouring_orbits[i] = find_root(colouring_orbits, i);
//        fprintf(stderr, "colouring_orbits[%d] = %d\n", i, colouring_orbits[i]);
//    }
//    fprintf(stderr, "-----\n");

    return number_of_colouring_orbits; //Return number of equivalence classes

}

void print_help(char * argv0) {
    fprintf(stderr, "Usage: %s [options]\n", argv0);
    fprintf(stderr, "Valid options are:\n");
    fprintf(stderr, "  out <x>: Output the graphs which have exactly x edge-Kempe equivalence classes.\n");
    fprintf(stderr, "  m <rest> <modulo>: Splits the computation in <modulo> (more or less equally big) parts. Here part <rest> will be executed.\n");
}

/******************************************************************************/

/*
 *
 */
int main(int argc, char** argv) {


    if(sizeof(unsigned long long int) != 8) {
        fprintf(stderr, "Error: this version relies on 64 bit unsigned long long ints -- sorry.\n");
        exit(1);
    }

    DEBUGMSG("Info: Debug is on")

    //For the timing
    struct tms TMS;

    int codelength = 0;
    unsigned char code[MAXN * (MAXN - 1) / 2 + MAXN];
    unsigned long long int number_of_graphs_read = 0;
    unsigned long long int number_of_graphs_processed = 0;
    unsigned long long int number_of_graphs_written = 0;

    /* Command line parameters */
    static int modulo = 1;
    static int rest = 0;

    int output_with_given_num_classes = 0;
    int desired_num_classes = 0;

    if(argc == 2 && argv[1][0] == '-' && argv[1][1] == 'h') {
        print_help(argv[0]);
        exit(1);
    }

    int i;
    for (i = 1; i < argc; i++) {
        switch (argv[i][0]) {
            case 'm':
            {
                if (strcmp(argv[i], "mod") == 0) {
                    if (i + 2 < argc) {
                        i++;
                        rest = atoi(argv[i]);
                        i++;
                        modulo = atoi(argv[i]);
                        if (rest >= modulo) {
                            fprintf(stderr, "Error: rest (%d) must be smaller than modulo (%d).\n", rest, modulo);
                            exit(1);
                        }
                    } else {
                        fprintf(stderr, "Error: option 'mod': missing rest or modulo\n");
                        exit(1);
                    }
                } else {
                    fprintf(stderr, "Error: invalid option: %s\n", argv[i]);
                    fprintf(stderr, "Execute '%s -h' for a list of possible options.\n", argv[0]);
                    exit(1);
                }
                break;
            }
            case 'o':
            {
                if (strcmp(argv[i], "out") == 0) {
                    if(i + 1 < argc) {
                        i++;
                        desired_num_classes = atoi(argv[i]);

                        if(desired_num_classes < 0) {
                            fprintf(stderr, "Error: desired_num_classes must be positive! (%d)\n", desired_num_classes);
                            exit(1);
                        }
                        output_with_given_num_classes = 1;
                        fprintf(stderr, "Info: writing graphs with exactly %d equivalence classes\n", desired_num_classes);
                    } else {
                        fprintf(stderr, "Error: option 'max': missing number\n");
                        exit(1);
                    }
                } else {
                    fprintf(stderr, "Error: invalid option: %s\n", argv[i]);
                    fprintf(stderr, "Execute '%s -h' for a list of possible options.\n", argv[0]);
                    exit(1);
                }
                break;
            }
            default:
            {
                fprintf(stderr, "Error: invalid option: %s\n", argv[i]);
                fprintf(stderr, "Execute '%s -h' for a list of possible options.\n", argv[0]);
                exit(1);
            }
        }
    }


    //Malloc
//    edge_bitvector_colour1 = malloc(sizeof(unsigned long long int) * MAX_NUM_COLOURINGS);
//    if(edge_bitvector_colour1 == NULL) {
//        fprintf(stderr, "Error: Can't get enough memory while creating edge_bitvector_colour1\n");
//        exit(1);
//    }
//    edge_bitvector_colour2 = malloc(sizeof(unsigned long long int) * MAX_NUM_COLOURINGS);
//    if(edge_bitvector_colour2 == NULL) {
//        fprintf(stderr, "Error: Can't get enough memory while creating edge_bitvector_colour2\n");
//        exit(1);
//    }
//    edge_bitvector_colour3 = malloc(sizeof(unsigned long long int) * MAX_NUM_COLOURINGS);
//    if(edge_bitvector_colour3 == NULL) {
//        fprintf(stderr, "Error: Can't get enough memory while creating edge_bitvector_colour3\n");
//        exit(1);
//    }

    //(*edge_bitvector_colour1)[123] = 0ULL;

    edge_bitvector_colours = (EDGE_COLOUR_STRUCT *) malloc(sizeof(EDGE_COLOUR_STRUCT) * MAX_NUM_COLOURINGS);
    if(edge_bitvector_colours == NULL) {
        fprintf(stderr, "Error: out of memory while creating edge_bitvector_colours\n");
        exit(1);
    }

    colouring_orbits = (int *) malloc(sizeof(int) * MAX_NUM_COLOURINGS);
    if(colouring_orbits == NULL) {
        fprintf(stderr, "Error: out of memory while creating colouring_orbits\n");
        exit(1);
    }
    colouring_orbits_size = (int *) malloc(sizeof(int) * MAX_NUM_COLOURINGS);
    if(colouring_orbits_size == NULL) {
        fprintf(stderr, "Error: out of memory while creating colouring_orbits_size\n");
        exit(1);
    }


    //unsigned long long int frequency_table_num_colouring_classes[MAX_NUM_COLOURINGS+1] = {0ULL};

    unsigned long long int * frequency_table_num_colouring_classes = (unsigned long long int *) malloc(sizeof(unsigned long long int) * (MAX_NUM_COLOURINGS + 1));
    if(frequency_table_num_colouring_classes == NULL) {
        fprintf(stderr, "Error: out of memory while creating frequency_table_num_colouring_classes\n");
        exit(1);
    }
    for (i = 0; i <= MAX_NUM_COLOURINGS; i++)
        frequency_table_num_colouring_classes[i] = 0ULL;

    max_num_colourings_found = 0;

    int nuller;
    for(; fread(code, sizeof (unsigned char), 1, stdin);) {
        nv = code[0];
        if(nv > MAXN) {
            fprintf(stderr, "Number of vertices is too big (limit is %d) \n", MAXN);
            exit(0);
        }
        if(code[0] == 0)
            nuller = 1;
        else
            nuller = 0;
        codelength = 1;
        while(nuller < nv - 1) {
            code[codelength] = getc(stdin);
            if(code[codelength] == 0) nuller++;
            codelength++;
        }

        number_of_graphs_read++;
        decode_multicode(code, codelength);

        if((number_of_graphs_read % modulo) == rest) {
            number_of_graphs_processed++;

            //fprintf(stderr, "Processing graph %llu\n", number_of_graphs_read);
            //printgraph(current_graph, adj);

            int num_classes = determine_num_edge_kempe_equivalence_classes();

            frequency_table_num_colouring_classes[num_classes]++;

            if(output_with_given_num_classes && desired_num_classes == num_classes) {
                if(fwrite(code, sizeof (unsigned char), codelength, stdout) < codelength) {
                    fprintf(stderr, "Error: couldn't write code!\n");
                    exit(1);
                }
                number_of_graphs_written++;
            }

        }
    }

    //FREE
    //free(edge_bitvector_colour1);
    //free(edge_bitvector_colour2);
    //free(edge_bitvector_colour3);
    free(edge_bitvector_colours);

    free(colouring_orbits);
    free(colouring_orbits_size);

    times(&TMS);
    unsigned int savetime = (unsigned int) TMS.tms_utime;

    fprintf(stderr, "Max num colourings found: %d\n", max_num_colourings_found);

    unsigned long long int total_check = 0ULL;
    for(i = 0; i <= MAX_NUM_COLOURINGS; i++) {
        if(frequency_table_num_colouring_classes[i] > 0) {
            fprintf(stderr, "Num graphs with %d colour classes: %llu\n", i, frequency_table_num_colouring_classes[i]);
            total_check += frequency_table_num_colouring_classes[i];
        }
    }

    if(total_check != number_of_graphs_processed) {
        fprintf(stderr, "Error: wrong number of graphs processed?!\n");
        exit(1);
    }

    free(frequency_table_num_colouring_classes);

    fprintf(stderr, "Read %lld graphs\n", number_of_graphs_read);
    fprintf(stderr, "Processed %lld graphs\n", number_of_graphs_processed);

    if(output_with_given_num_classes)
        fprintf(stderr, "Written %lld graphs with %d equivalence classes\n", number_of_graphs_written, desired_num_classes);
    fprintf(stderr, "CPU time filter: %.1f seconds.\n", (double) savetime / (double) CLK_TCK);

    return (EXIT_SUCCESS);
}
