use chrono::{DateTime, Datelike, NaiveDate, TimeZone, Utc};
use git2::Time;
use walkdir::{DirEntry, WalkDir};
pub fn convert_time(time: &Time) -> DateTime<Utc> {
    let tz = chrono::FixedOffset::east(time.offset_minutes() * 60);
    tz.timestamp(time.seconds(), 0).with_timezone(&Utc)
}

pub fn is_hidden(entry: &DirEntry) -> bool {
    entry
        .file_name()
        .to_str()
        .map(|s| s.starts_with("."))
        .unwrap_or(false)
}

pub fn directories(path: &str) -> usize {
    let walker = WalkDir::new(path).into_iter();
    let count = walker
        .filter_entry(|e| !is_hidden(e) && e.file_type().is_dir())
        .count();
    if count == 1 {
        // only the path we gave is a dir
        0
    } else {
        count - 1
    }
}

pub fn top_level_directories(path: &str) -> usize {
    let mut count = 0;
    let walker = WalkDir::new(path).into_iter();
    for entry in walker.filter_entry(|e| !is_hidden(e)) {
        if let Ok(entry) = entry {
            if entry.file_type().is_dir() && entry.depth() == 1 {
                count += 1;
            }
        } else {
            log::error!("{} - directory entry is not parsable?", path);
        }
    }

    count
}

fn _incubation_days(start_date: &str, end_date: &str) -> i64 {
    let sd = chrono::NaiveDate::parse_from_str(start_date, "%Y-%m-%d").unwrap();
    let ed = chrono::NaiveDate::parse_from_str(end_date, "%Y-%m-%d").unwrap();

    let days = ed.signed_duration_since(sd);
    days.num_days()
}

fn _incubation_month(start_date: &str, date: &str) -> i64 {
    let sd = chrono::NaiveDate::parse_from_str(start_date, "%Y-%m-%d").unwrap();
    let tz_offset = chrono::FixedOffset::east(1 * 3600);
    let time = chrono::NaiveTime::from_hms(12, 0, 0);
    let datetime = chrono::NaiveDateTime::new(sd, time);
    let dt_with_tz: chrono::DateTime<chrono::FixedOffset> =
        tz_offset.from_local_datetime(&datetime).unwrap();

    let ed = chrono::DateTime::parse_from_rfc3339(date).unwrap();

    let days = ed.signed_duration_since(dt_with_tz);
    // println!("{}", days.num_days());

    let month = (days.num_days() as f64 / 30 as f64).ceil();
    month as i64
}

pub fn inc_month_to_date(start_date: &str, inc_month: usize) -> String {
    let sd = NaiveDate::parse_from_str(start_date, "%Y-%m-%d").unwrap();
    let mut year = sd.year();
    let mut month = sd.month();

    let mut desired_month = inc_month;

    while desired_month > 1 {
        if month == 12 {
            year += 1;
            month = 1;
        } else {
            month += 1;
        }
        desired_month -= 1
    }
    if month < 10 {
        format!("{}-0{}", year, month)
    } else {
        format!("{}-{}", year, month)
    }
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_directories() {
        assert_eq!(3, directories("test_resources/test_directories"));
        assert_eq!(0, directories("test_resources/test_directories/dir2"));
        assert_eq!(1, directories("test_resources/test_directories/dir1"));
    }

    #[test]
    fn test_top_level_directories() {
        assert_eq!(2, top_level_directories("test_resources/test_directories"));
        assert_eq!(
            0,
            top_level_directories("test_resources/test_directories/dir2")
        );
        assert_eq!(
            1,
            top_level_directories("test_resources/test_directories/dir1")
        );
    }
    #[test]
    fn test_incubation_months() {
        assert_eq!(
            1,
            _incubation_month("2010-10-30", "2010-11-01T16:39:57+00:00")
        );

        assert_eq!(
            2,
            _incubation_month("2010-10-30", "2010-11-30T16:39:57+00:00")
        );

        assert_ne!(
            1,
            _incubation_month("2010-10-30", "2010-11-30T16:39:57+00:00")
        );

        assert_eq!(
            14,
            _incubation_month("2010-10-30", "2011-11-30T16:39:57+00:00")
        );
    }
    #[test]
    fn test_inc_month_to_date() {
        let start_date = "2010-10-15";
        let month = 3;
        let expected = "2010-12";

        assert_eq!(inc_month_to_date(start_date, month), expected.to_string());

        assert_eq!(inc_month_to_date(start_date, 1), "2010-10".to_string());
        assert_eq!(inc_month_to_date(start_date, 4), "2011-01".to_string());

        assert_eq!(inc_month_to_date(start_date, 20), "2012-05".to_string());
    }
}
