use crate::commits_metrics::CommitsMetrics;
use crate::emails::EmailsMetrics;
use crate::metrics::Metrics;
use crate::repo::Repo;
use crate::sokrates_metrics::{Sokrates, SokratesMetrics};
use crate::tokei_metrics::TokeiMetrics;
use crate::{utils::*, Args};
use git2::Error;
use serde::Serialize;
#[derive(Serialize)]
pub struct Stats<'a> {
    /// Project's name
    project: &'a str,
    /// Project's start date - usually formatted as Y-m-d
    start_date: &'a str,
    /// Project's end date - usually formatted as Y-m-d
    end_date: &'a str,
    /// Project status: graduated, retired
    status: &'a str,
    /// Git repo
    #[serde(skip_serializing)]
    repo: &'a Repo<'a>,
    /// Java path
    #[serde(skip_serializing)]
    java_path: &'a str,
    /// Metrics
    #[serde(flatten)]
    metrics: Metrics,
}

impl<'a> Stats<'a> {
    pub fn new(project: &'a str, repo: &'a Repo, java_path: &'a str) -> Self {
        Self {
            project,
            start_date: repo.start_date,
            end_date: repo.end_date,
            status: repo.status,
            metrics: Metrics::default(),
            java_path: java_path,
            repo,
        }
    }

    /// The main function for getting the statistics
    /// Return a hashmap with incubation months as keys and metrics as values
    pub fn compute_statistics(&mut self, args: &Args) -> Result<Vec<Stats>, Error> {
        //Result<IndexMap<usize, Metrics>, Error> {
        log::info!("{}", format!("{} - computing stats", self.project));

        let _projects_names_fix_emails = indexmap::IndexMap::from([
            ("apex-core", "apex"),
            ("blur", "incubator-blur"),
            ("derby", "db-derby"),
            ("empire-db", "empire"),
            ("ftpserver", "incubator-ftpserver"),
            ("hcatalog", "incubator-hcatalog"),
            ("ant-ivy", "incubator-ivy"),
            ("kalumet", "incubator-kalumet"),
            ("lucene.net", "lucenenet"),
            ("mynewt-core", "mynewt"),
            ("npanday", "incubator-npanday"),
            ("nuvem", "incubator-nuvem"),
            ("odftoolkit", "incubator-odf"),
            ("photark", "incubator-photark"),
            ("pluto", "portals-pluto"),
            ("creadur-rat", "creadur"),
            ("s4", "incubator-s4"),
            ("sanselan", "incubator-sanselan"),
            ("servicecomb-java-chassis", "servicecomb"),
            ("tashi", "incubator-tashi"),
            ("warble-server", "warble"),
            ("wave", "incubator-wave"),
            ("zetacomponents", "incubator-zeta"),
        ]);

        let inc_months_commits = &self.repo.inc_month_commits;

        let months = self.repo.dates_to_months();
        log::info!(
            "{} - found {} commits",
            self.project,
            inc_months_commits
                .values()
                .fold(0, |sum, val| sum + val.len())
        );
        let mut last_metrics = Metrics {
            active_days: 0,
            added_lines: 0,
            authors: 0,
            avg_files_modified_commit: 0.0,
            blanks: 0,
            code: 0,
            comments: 0,
            commits: 0,
            committers: 0,
            deleted_lines: 0,
            directories: 0,
            emails: 0,
            emails_devs: 0,
            files: 0,
            files_added: 0,
            files_deleted: 0,
            files_modified: 0,
            files_renamed: 0,
            incubation_month: 1,
            lines: 0,
            major_contributors: 0,
            minor_contributors: 0,
            new_contributors: 0,
            releases: 0,
            top_level_dirs: 0,
            sokrates_metrics: SokratesMetrics::default(),
            programming_lang: "".to_string(),
        };
        let mut output: Vec<Stats> = vec![];
        let mut existing_contributors = indexmap::IndexSet::<String>::new();
        for (month, commits) in inc_months_commits.iter() {
            log::info!(
                "{}",
                format!("{} month: {} - extracting stats", self.project, month)
            );
            // we have no commits this month, we need to use the last know commits data
            if commits.is_empty() {
                last_metrics.incubation_month = *month;
                // we might not have commits, but we might have emails
                let email_data = EmailsMetrics::metrics(format!(
                    "../../projects/emails/{}-dev-{}.mbox",
                    self.project.to_lowercase(),
                    months.get(month).unwrap()
                ));

                let emails = email_data.emails;
                let emails_devs = email_data.devs;
                last_metrics.emails = emails;
                last_metrics.emails_devs = emails_devs;

                // process metrics are 0 because we did not have any activity
                last_metrics.active_days = 0;
                last_metrics.added_lines = 0;
                last_metrics.authors = 0;
                last_metrics.avg_files_modified_commit = 0.0;
                last_metrics.commits = 0;
                last_metrics.committers = 0;
                last_metrics.deleted_lines = 0;
                last_metrics.files_added = 0;
                last_metrics.files_deleted = 0;
                last_metrics.files_renamed = 0;
                last_metrics.files_modified = 0;
                last_metrics.major_contributors = 0;
                last_metrics.minor_contributors = 0;
                last_metrics.new_contributors = 0;
                last_metrics.releases = 0;
                // last_metrics.programming_lang = "".to_string();

                output.push(Stats {
                    project: self.project,
                    start_date: self.start_date,
                    end_date: self.end_date,
                    status: self.status,
                    metrics: last_metrics.clone(),
                    repo: self.repo,
                    java_path: self.java_path,
                })
            } else {
                let commit_last_month = &commits.last();
                log::debug!("{:?}", &commits);
                let month_metrics = CommitsMetrics::new(self.repo, &commits)?;
                // ***** COMMIT METRICS ***** //
                let active_days = month_metrics.active_days();
                let added_lines = month_metrics.added_lines();
                let authors = month_metrics.authors_emails().len();
                let commits = commits.len();
                let committers = month_metrics.committers_emails().len();
                let deleted_lines = month_metrics.deleted_lines();
                let files_added = month_metrics.files_added();
                let files_deleted = month_metrics.files_deleted();
                let files_renamed = month_metrics.files_renamed();
                let files_modified = month_metrics.files_modified();

                let avg_files_modified_commit = files_modified as f64 / commits as f64;

                let prev_contributors = existing_contributors
                    .iter()
                    .cloned()
                    .collect::<indexmap::IndexSet<String>>();
                let current_month_contributors = month_metrics.authors_emails();

                let contributors = current_month_contributors
                    .difference(&prev_contributors)
                    .collect::<Vec<_>>();
                let mut new_contributors = 0;
                for e in contributors {
                    new_contributors += 1;
                    existing_contributors.insert(e.to_string());
                }

                let (minor_contributors, major_contributors) =
                    month_metrics.major_minor_contributors();

                let email_data = EmailsMetrics::metrics(format!(
                    "../../projects/emails/{}-dev-{}.mbox",
                    self.project.to_lowercase().as_str(),
                    months.get(month).unwrap()
                ));

                let emails = email_data.emails;
                let emails_devs = email_data.devs;

                // we checkout at the last commit of this month as we need the source code analysis for this month
                if let Some(hash) = commit_last_month {
                    let hash = hash.id().to_string();

                    let checkout = self.repo.checkout_commit(&hash);

                    if let Ok(_checkout) = checkout {
                        // ***** CODE METRICS ***** //
                        let tokei_metrics = TokeiMetrics::new(self.repo);
                        let (code, files, lines, blanks, comments, programming_lang) =
                            match tokei_metrics {
                                Some(tm) => (
                                    tm.code(),
                                    tm.files(),
                                    tm.lines(),
                                    tm.blanks(),
                                    tm.comments(),
                                    tm.programming_language(),
                                ),
                                None => (0, 0, 0, 0, 0, "".to_string()),
                            };

                        let directories =
                            directories(self.repo.repo.path().parent().unwrap().to_str().unwrap());

                        let top_level_dirs = top_level_directories(
                            self.repo.repo.path().parent().unwrap().to_str().unwrap(),
                        );

                        // ***** SOKRATES METRICS ***** //
                        let mut sokrates = Sokrates::new(
                            self.repo
                                .repo
                                .path()
                                .parent()
                                .unwrap()
                                .to_str()
                                .unwrap_or(""),
                                self.java_path.to_string()
                        );

                        if !args.flag_skip_sokrates {
                            let history_output =
                                sokrates.extract_history(self.project, month, &hash);

                            if history_output.is_ok() {
                                let init_output = sokrates.init(self.project, month, &hash);

                                if init_output.is_ok() {
                                    let reports_output =
                                        sokrates.generate_reports(self.project, month, &hash);

                                    if reports_output.is_ok() {
                                        let metrics = sokrates.metrics();
                                        if let Ok(m) = metrics {
                                            // sokrates_metrics = m;
                                            sokrates.metrics = m;
                                        }
                                    } else {
                                        log::error!(
                                            "{} month: {} - sokrates failed to generate reports",
                                            self.project,
                                            month
                                        );
                                        continue;
                                    }
                                } else {
                                    log::error!(
                                        "{} month: {} - sokrates failed initialization",
                                        self.project,
                                        month
                                    );
                                    continue;
                                }
                            } else {
                                log::error!(
                                    "{} month: {} - sokrates failed to extract history",
                                    self.project,
                                    month
                                );
                                log::error!("{:?}", history_output.err());
                                continue;
                            };

                            let cleanup = sokrates.cleanup(self.project, month, &hash);
                            if let Err(_cleanup) = cleanup {
                                log::error!(
                                    "{} month: {} - cannot cleanup after sokrates",
                                    self.project,
                                    month
                                );
                            }
                        }

                        let current_metrics = Metrics {
                            active_days,
                            added_lines,
                            authors,
                            avg_files_modified_commit,
                            blanks,
                            code,
                            comments,
                            commits,
                            committers,
                            deleted_lines,
                            directories,
                            emails,
                            emails_devs,
                            files,
                            files_added,
                            files_deleted,
                            files_modified,
                            files_renamed,
                            incubation_month: *month,
                            lines,
                            major_contributors,
                            minor_contributors,
                            new_contributors,
                            programming_lang,
                            releases: 0,
                            sokrates_metrics: sokrates.metrics,
                            top_level_dirs,
                        };
                        last_metrics = current_metrics;

                        output.push(Stats {
                            project: self.project,
                            start_date: self.start_date,
                            end_date: self.end_date,
                            status: self.status,
                            metrics: last_metrics.clone(),
                            repo: self.repo,
                            java_path: self.java_path,
                        });
                    } else {
                        log::error!(
                            "{} month: {} - cannot do a checkout at hash {}",
                            self.project,
                            month,
                            hash
                        );
                        continue;
                    }
                } else {
                    // if we cannot do a checkout, even though we know we should, we skip this month
                }
            }
        }
        // reset repository to main/master/trunk
        self.repo.checkout_master_main_trunk()?;

        Ok(output)
    }
}
