use crate::convert_time;
use chrono::Datelike;
use git2::{Commit, Error, Repository};
use indexmap::map::Entry;
use indexmap::{IndexMap, IndexSet};
use serde::Serialize;

#[derive(Serialize, PartialEq, Eq, Hash, Debug)]
pub struct CommitMessage {
    pub project: String,
    pub status: String,
    pub inc_month: usize,
    pub sha: String,
    pub message: String,
}

pub struct Repo<'a> {
    pub repo: &'a Repository,
    pub project: &'a str,
    pub start_date: &'a str,
    pub end_date: &'a str,
    pub status: &'a str,
    pub commits: Vec<Commit<'a>>,
    pub inc_month_commits: IndexMap<usize, Vec<Commit<'a>>>,
}

impl<'a> Repo<'a> {
    /// Create a new repository object with all the metadata
    pub fn new(
        repo: &'a Repository,
        project: &'a str,
        start_date: &'a str,
        end_date: &'a str,
        status: &'a str,
    ) -> Result<Self, Error> {
        let commits = Self::commits(repo, start_date, end_date)?;
        let inc_month_commits = Self::commits_to_inc_months(start_date, end_date, &commits)?;
        Ok(Self {
            repo,
            project,
            start_date,
            end_date,
            status,
            commits,
            inc_month_commits,
        })
    }

    /// Retrives commits from the repository between project's start date and project's end date, excluding merge commits
    fn commits(
        repo: &'a Repository,
        start_date: &str,
        end_date: &str,
    ) -> Result<Vec<Commit<'a>>, Error> {
        // starting timestamp for dropping commits previous to incubation start
        #[allow(clippy::unwrap_used)]
        let final_timestamp = chrono::DateTime::parse_from_rfc3339(
            format!("{}{}", end_date, "T23:59:59+00:00").as_str(),
        )
        .unwrap()
        .timestamp();
        // ending timestamp for dropping commits after incubation ended
        #[allow(clippy::unwrap_used)]
        let start_timestamp = chrono::DateTime::parse_from_rfc3339(
            format!("{}{}", start_date, "T00:00:00+00:00").as_str(),
        )
        .unwrap()
        .timestamp();

        let mut revwalk = repo.revwalk()?;
        // start from the top
        let reverse_sorting = revwalk.set_sorting(git2::Sort::REVERSE);
        if let Err(_reverse_sorting) = reverse_sorting {
            log::error!("cannot iterate commits in reverse order")
        }
        let mut first_commit = true;

        revwalk.push_head()?;
        let commits: Vec<Commit<'a>> = revwalk
            .filter_map(|r| {
                match r {
                    Err(_) => None,
                    Ok(r) => repo.find_commit(r).ok().filter(|commit| {
                        // exclude merge commits
                        let commit_time = convert_time(&commit.committer().when()).timestamp();
                        // if this is first commit, then we keep it as it might not have parents
                        if first_commit {
                            first_commit = false;
                            (commit.parent_count() == 0
                                || commit.parent_count() == 1) && commit_time >= start_timestamp //drop any commits that are before project start
                            && commit_time <= final_timestamp // drop any commits that are after project start
                        } else {
                            commit.parent_count() == 1 // drop any merge commits
                            && commit_time >= start_timestamp //drop any commits that are before project start
                            && commit_time <= final_timestamp // drop any commits that are after project start
                        }
                    }),
                }
            })
            .collect();

        Ok(commits)
    }

    /// Returns an index map of commits per incubation month; the key is the incubation month as an integer
    /// and the value is the vector of commits for that respective incubation month
    fn commits_to_inc_months(
        start_date: &str,
        end_date: &str,
        commits: &Vec<Commit<'a>>,
    ) -> Result<IndexMap<usize, Vec<Commit<'a>>>, Error> {
        let incubation_months = Self::parse_dates_to_inc_months(start_date, end_date);
        // println!("{:?}", incubation_months);
        let mut output = IndexMap::<usize, Vec<Commit<'a>>>::new();

        // we add all our incubation months with no commits yet to the output
        for (_, v) in &incubation_months {
            output.insert(*v, vec![]);
        }

        let all_commits = commits.clone();

        for commit in all_commits {
            let commit_time = convert_time(&commit.committer().when());
            let commit_inc_month = incubation_months
                .get(format!("{}{}", commit_time.year(), commit_time.month()).as_str());

            // if we find commits in a particular incubation month, then we add it to our output list
            match output.entry(*commit_inc_month.unwrap()) {
                Entry::Occupied(mut entry) => {
                    entry.get_mut().push(commit);
                }
                Entry::Vacant(entry) => {
                    entry.insert(vec![commit]);
                }
            }
        }
        Ok(output)
    }

    /// Checkout the repository at the given commit
    pub fn checkout_commit(&self, refname: &str) -> Result<(), Error> {
        let revparse = self.repo.revparse_ext(refname);

        if let Ok((object, reference)) = revparse {
            let try_checkout = self.repo.checkout_tree(&object, None);
            if let Ok(_checkout) = try_checkout {
                log::info!("{} - succesfully checked out at {}", self.project, &refname);
                match reference {
                    // gref is an actual reference like branches or tags
                    Some(gref) => self.repo.set_head(gref.name().unwrap())?,
                    // this is a commit, not a reference
                    None => self.repo.set_head_detached(object.id())?,
                }
            } else {
                log::error!("{} - failed to checkout at {}", self.project, &refname);
            }
        } else {
            log::error!("{} - failed to checkout at {}", self.project, &refname);
        }

        Ok(())
    }

    /// Try to checkout the repository to its main branch (master, main, trunk)
    pub fn checkout_master_main_trunk(&self) -> Result<(), Error> {
        let repo_branches = self.repo.branches(Some(git2::BranchType::Local));

        let repo_branches = if let Ok(branches) = repo_branches {
            branches
                .into_iter()
                .filter_map(|x| {
                    if let Ok(x) = x {
                        if let Ok(name) = x.0.name() {
                            if let Some(name) = name {
                                Some(name.to_string())
                            } else {
                                None
                            }
                        } else {
                            None
                        }
                    } else {
                        None
                    }
                })
                .collect::<IndexSet<String>>()
        } else {
            IndexSet::<String>::new()
        };
        log::info!(
            "{} - found the following branches {}",
            self.project,
            &repo_branches
                .clone()
                .into_iter()
                .collect::<Vec<_>>()
                .join(",")
        );

        if self.project == "FreeMarker" {
            match self.checkout_commit("2.3-gae") {
                Ok(()) => {
                    return Ok(());
                }
                Err(_e) => {
                    log::error!(
                        "{} - has a branch named 2.3-gae, but I cannot check it out",
                        self.project
                    );
                    return Ok(());
                }
            };
        } else if self.project == "Dubbo" {
            match self.checkout_commit("3.0") {
                Ok(()) => {
                    return Ok(());
                }
                Err(_e) => {
                    log::error!(
                        "{} - has a branch named 3.0, but I cannot check it out",
                        self.project
                    );
                    return Ok(());
                }
            };
        } else if self.project == "DolphinScheduler" {
            match self.checkout_commit("dev") {
                Ok(()) => {
                    return Ok(());
                }
                Err(_e) => {
                    log::error!(
                        "{} - has a branch named dev, but I cannot check it out",
                        self.project
                    );
                    return Ok(());
                }
            };
        } else if repo_branches.contains("master") {
            match self.checkout_commit("master") {
                Ok(()) => {
                    return Ok(());
                }
                Err(_e) => {
                    log::error!(
                        "{} - has a branch named master, but I cannot check it out",
                        self.project
                    );
                    return Ok(());
                }
            };
        } else if repo_branches.contains("main") {
            match self.checkout_commit("main") {
                Ok(()) => {
                    return Ok(());
                }
                Err(_e) => {
                    log::error!(
                        "{} - has a branch named main, but I cannot check it out",
                        self.project
                    );
                    return Ok(());
                }
            };
        } else if repo_branches.contains("trunk") {
            match self.checkout_commit("trunk") {
                Ok(()) => {
                    return Ok(());
                }
                Err(_e) => {
                    log::error!(
                        "{} - has a branch named main, but I cannot check it out",
                        self.project
                    );
                    return Ok(());
                }
            };
        } else if repo_branches.contains("develop") {
            match self.checkout_commit("develop") {
                Ok(()) => {
                    return Ok(());
                }
                Err(_e) => {
                    log::error!(
                        "{} - has a branch named develop, but I cannot check it out",
                        self.project
                    );
                    return Ok(());
                }
            };
        } else {
            log::error!(
                "{} - has no branch named master/main/trunk/develop... cannot reset to main branch",
                self.project
            );
        }

        Ok(())
    }

    /// Transform the start date and end date into a map of incubation months and date
    ///
    /// E.g., 2010-01-01 to 2010-03-05 => 1 -> 201001, 2 -> 201002, 3 -> 201003
    pub fn dates_to_months(&self) -> IndexMap<usize, String> {
        let mut result = IndexMap::<usize, String>::new();

        let sd = chrono::NaiveDate::parse_from_str(self.start_date, "%Y-%m-%d").unwrap();
        let ed = chrono::NaiveDate::parse_from_str(self.end_date, "%Y-%m-%d").unwrap();

        fn next_month(year: i32, month: u32) -> (i32, u32) {
            assert!(month >= 1 && month <= 12);

            if month == 12 {
                (year + 1, 1)
            } else {
                (year, month + 1)
            }
        }

        let mut year = sd.year();
        let mut month = sd.month();
        let mut inc_month = 0;

        loop {
            let (next_year, next_month) = next_month(year, month);

            if year < ed.year() || (year == ed.year() && month <= ed.month()) {
                inc_month += 1;
                if month < 10 {
                    result.insert(inc_month, format!("{}0{}", year, month));
                } else {
                    result.insert(inc_month, format!("{}{}", year, month));
                }

                year = next_year;
                month = next_month;
            } else {
                break;
            }
        }
        result
    }

    /// Parses the incubation start and end dates to a list of incubation months.
    /// The returned data is a hash map with the date as 20101 - Jan 2010, as keys
    /// and integers (incubation month) as values
    /// This is a ordered HashMap
    pub fn parse_dates_to_inc_months(start_date: &str, end_date: &str) -> IndexMap<String, usize> {
        let mut result = IndexMap::<String, usize>::new();

        let sd = chrono::NaiveDate::parse_from_str(start_date, "%Y-%m-%d").unwrap();
        let ed = chrono::NaiveDate::parse_from_str(end_date, "%Y-%m-%d").unwrap();

        fn next_month(year: i32, month: u32) -> (i32, u32) {
            assert!(month >= 1 && month <= 12);

            if month == 12 {
                (year + 1, 1)
            } else {
                (year, month + 1)
            }
        }

        let mut year = sd.year();
        let mut month = sd.month();
        let mut inc_month = 0;

        loop {
            let (next_year, next_month) = next_month(year, month);

            if year < ed.year() || (year == ed.year() && month <= ed.month()) {
                inc_month += 1;
                result.insert(format!("{}{}", year, month), inc_month);

                year = next_year;
                month = next_month;
            } else {
                break;
            }
        }
        result
    }
}

#[cfg(test)]
mod test {
    use super::{Repo, Repository};

    #[test]
    fn test_parse_date_to_inc_months() {
        let months = Repo::parse_dates_to_inc_months("2010-10-30", "2010-10-31");
        assert!(months.contains_key("201010"));
        assert!(!months.contains_key("201011"));

        assert_eq!(Some(&1), months.get("201010"));
        assert_eq!(None, months.get("201011"));

        let months = Repo::parse_dates_to_inc_months("2010-10-30", "2011-01-30");
        println!("{:?}", months);
        assert!(months.contains_key("201010"));
        assert!(months.contains_key("201011"));
        assert!(months.contains_key("20111"));
        assert!(!months.contains_key("20112"));
        assert!(!months.contains_key("20102"));

        assert_eq!(Some(&1), months.get("201010"));
        assert_eq!(Some(&2), months.get("201011"));
        assert_eq!(Some(&4), months.get("20111"));
    }

    #[test]
    fn test_dates_to_months() {
        let repo = Repository::open("test_resources/git_repo").unwrap();
        let actual = Repo::new(&repo, "test", "2010-01-01", "2010-03-05", "graduated");

        if let Ok(actual) = actual {
            assert_eq!(
                actual.dates_to_months(),
                indexmap::IndexMap::from([(1, "201001"), (2, "201002"), (3, "201003")])
            );
        } else {
            assert!(false)
        }

        let actual = Repo::new(&repo, "test", "2010-12-30", "2011-01-05", "graduated");

        if let Ok(actual) = actual {
            assert_eq!(
                actual.dates_to_months(),
                indexmap::IndexMap::from([(1, "201012"), (2, "201101")])
            );
        } else {
            assert!(false)
        }
    }
}
