use std::fs::File;

use crate::utils::inc_month_to_date;
use mail_parser::HeaderValue;
use mail_parser::{parsers::mbox::MBoxParser, Message};
use scraper::{Html, Selector};
use serde::Serialize;
use ureq::{Agent};
#[derive(Clone, Debug, Serialize)]
pub struct EmailsMetrics {
    pub emails: usize,
    pub devs: usize,
}

impl EmailsMetrics {
    pub fn metrics(path: String) -> Self {
        Self::parse_mbox_file(path)
    }

    pub fn parse_mbox_file(path: String) -> EmailsMetrics {
        let mbox_file = std::fs::File::open(&path);
        let mut emails = 0;
        let mut emails_devs = indexmap::IndexSet::<String>::new();
        log::info!("Analyzing email: {}", &path);
        match mbox_file {
            Ok(val) => {
                for raw_message in MBoxParser::new(val) {
                    let parsed_email = Message::parse(&raw_message);
                    match parsed_email {
                        Some(e) => {
                            let from = match e.get_from() {
                                HeaderValue::Address(x) => x.name.as_deref().unwrap_or(""),
                                _ => "",
                            };
                            let from = from.replace(",", "");

                            let from_email = match e.get_from() {
                                HeaderValue::Address(x) => x.address.as_deref().unwrap_or(""),
                                _ => "",
                            };
                            let from_email = from_email.replace(",", "");

                            if from_email == "jira@apache.org" && !from.contains("(Commented)") {
                                continue;
                            } else {
                                let dev = from.replace("(Commented) (JIRA)", "").trim().to_string();
                                let dev = dev.replace("(JIRA)", "").trim().to_string();
                                if dev == "jiraposter@reviews.apache.org" {
                                    // we cannot differentiate between devs, so let's skip them
                                    continue;
                                }
                                emails += 1;
                                emails_devs.insert(dev);
                            }
                        }
                        None => {
                            log::error!("Cannot parse an email {}", &path);
                        }
                    }
                }
            }
            Err(_) => {
                log::error!("Cannot open file {}", &path);
            }
        }
        EmailsMetrics {
            emails,
            devs: emails_devs.len(),
        }
    }
}

pub fn _local_mboxes_ids(project: &str, emails_storage_folder: &str) -> Vec<String> {
    let mbox_path = emails_storage_folder;
    let files = std::fs::read_dir(&mbox_path);
    let paths = match files {
        Ok(f) => f
            .into_iter()
            .filter_map(|x| if let Ok(p) = x { Some(p.path()) } else { None })
            .collect::<Vec<_>>(),
        Err(ref _e) => {
            // println!("Cannot parse filename, {:?}", &files);
            log::error!("Cannot read mbox directory {mbox_path}");
            vec![]
        }
    };

    #[allow(clippy::unwrap_used)]
    let path_ids = paths
        .into_iter()
        .filter(|x| {
            x.file_name()
                .unwrap()
                .to_str()
                .unwrap()
                .starts_with(&project.to_lowercase())
        })
        .collect::<Vec<_>>();
    #[allow(clippy::unwrap_used)]
    let mut ids = path_ids
        .into_iter()
        .map(|x| x.as_path().to_str().unwrap().to_string())
        .collect::<Vec<_>>();

    ids.sort();
    let mut output = ids
        .into_iter()
        .map(|x| x.chars().rev().take(11).collect::<String>())
        .collect::<Vec<_>>()
        .into_iter()
        .map(|x| x.chars().rev().take(6).collect::<String>())
        .collect::<Vec<_>>();

    output.reverse();
    output.to_vec()
}

/// Downloads an email from the apache mailing lists, https://lists.apache.org/api/mbox.lua?list=dev&domain={}.apache.org&d={}
///
/// The new apache mailing list server returns an empty file if the file does not exist on their server. Therefore, this function
/// will still create an empty mbox file if there is no mail archive for that month on the server.
pub fn _download_email<'a>(
    project: &'a str,
    start_date: &'a str,
    month: usize,
    emails_folder: &'a str,
    agent: &Agent,
) -> Option<String> {
    let date = inc_month_to_date(start_date, month);
    let url = format!(
        // "http://mail-archives.apache.org/mod_mbox/{}-dev/{}.mbox",
        "https://lists.apache.org/api/mbox.lua?list=dev&domain={}.apache.org&d={}",
        project.to_lowercase(),
        &date,
    );

    let filename = format!(
        "{}/{}-dev-{}.mbox",
        emails_folder,
        project.to_lowercase(),
        date.replace('-', "") // we store all the emails as project-dev-yearmonth.mbox, so we need to remove the dash between year and month
    );
    let res = agent.get(&url).call();

    if let Ok(res) = res {
        if res.status() == 200 {
            let mut file = File::create(&filename).expect("Cannot create file {filename}");
            std::io::copy(&mut res.into_reader(), &mut file);

            Some(filename)
        } else {
            None
        }
    } else {
        None
    }
}

/// This used to work with the older mail archives of apache. As of Jan 2022, the way they store/serve the mbox files has changed, and thus this does not work anymore
pub fn _mboxes_ids(project: &str) -> Vec<String> {
    let url = format!("http://mail-archives.apache.org/mod_mbox/{}-dev", project);

    let res = ureq::get(&url).call();

    let html = match res {
        Ok(e) => e.into_string().unwrap_or_else(|_| "".to_string()),
        Err(_) => "".to_string(),
    };

    let fragment = Html::parse_fragment(&html);
    let spans = Selector::parse("span");
    if let Ok(spans) = spans {
        let mut ids = vec![];
        for s in fragment
            .select(&spans)
            .into_iter()
            .filter(|s| s.value().classes().any(|a| a == "links"))
        {
            if let Some(id) = s.value().attr("id") {
                ids.push(id.to_string());
            };
        }
        ids.to_vec()
    } else {
        log::error!("Cannot parse span for mboxes ids");
        vec![]
    }
}

#[cfg(test)]
mod test {
    use super::*;
    #[test]
    fn test_download_email() {
        let agent = AgentBuilder::new()
            .timeout_read(Duration::from_secs(15))
            .timeout_write(Duration::from_secs(300))
            .build();
        let emails_folder = "../../projects/emails";
        let project = "samoa";
        let start_date = "2014-12-15";
        let month = 2;

        assert_eq!(
            Some(format!("{}/{}-dev-201501.mbox", emails_folder, project)),
            crate::download_email(project, start_date, month, emails_folder, &agent)
        );

        assert_eq!(
            Some(format!("{}/{}-dev-200202.mbox", emails_folder, project)),
            crate::download_email(project, "2002-01-01", month, emails_folder, &agent)
        )
    }
}

// fn parse_email(args: &Args) -> Result<(), std::io::Error> {
//     let all_projects = list_projects();

//     match std::fs::create_dir_all("../data/emails-csvs") {
//         Ok(()) => println!("Created folder data/emails-csvs"),
//         Err(e) => println!("Cannot create folder data/emails-csvs... {}", e),
//     }

//     let projects_names_fix = IndexMap::from([
//         ("apex-core", "apex"),
//         ("blur", "incubator-blur"),
//         ("derby", "db-derby"),
//         ("empire-db", "empire"),
//         ("ftpserver", "incubator-ftpserver"),
//         ("hcatalog", "incubator-hcatalog"),
//         ("ant-ivy", "incubator-ivy"),
//         ("kalumet", "incubator-kalumet"),
//         ("lucene.net", "lucenenet"),
//         ("mynewt-core", "mynewt"),
//         ("npanday", "incubator-npanday"),
//         ("nuvem", "incubator-nuvem"),
//         ("odftoolkit", "incubator-odf"),
//         ("photark", "incubator-photark"),
//         ("pluto", "portals-pluto"),
//         ("creadur-rat", "creadur"),
//         ("s4", "incubator-s4"),
//         ("sanselan", "incubator-sanselan"),
//         ("servicecomb-java-chassis", "servicecomb"),
//         ("tashi", "incubator-tashi"),
//         ("warble-server", "warble"),
//         ("wave", "incubator-wave"),
//         ("zetacomponents", "incubator-zeta"),
//     ]);

//     let mut project_list: Vec<String> = vec![];
//     // we fixed the naming issue with some projects
//     for p in all_projects {
//         let mut split = p.split(",").collect::<Vec<_>>();
//         let name = split[0];
//         if projects_names_fix.contains_key(name.to_lowercase().as_str()) {
//             let fixed_name = projects_names_fix
//                 .get(name.to_lowercase().as_str())
//                 .unwrap();
//             split[0] = fixed_name;
//             // println!("{:?}", split);
//             let new_val = split.join(",");
//             project_list.push(new_val);
//         } else {
//             project_list.push(p);
//         }
//     }
//     // if we want only a single project, we need to filter it
//     let projects = if let Some(n) = &args.flag_parse_single_project {
//         let project_name = if projects_names_fix.contains_key(n.to_lowercase().as_str()) {
//             projects_names_fix
//                 .get(n.to_lowercase().as_str())
//                 .unwrap()
//                 .to_string()
//         } else {
//             n.to_string()
//         };
//         project_list
//             .into_iter()
//             .filter(|x| {
//                 let split = x.split(",").collect::<Vec<_>>(); // extract the repo name
//                 let repo_name = split[0];
//                 repo_name == project_name
//             })
//             .collect::<Vec<_>>()
//     } else {
//         project_list
//     };
//     if args.flag_show_projects {
//         println!("{:?}", projects);
//     }

//     let mut cwd = std::env::current_dir()
//         .unwrap()
//         .parent()
//         .unwrap()
//         .parent()
//         .unwrap()
//         .to_path_buf();
//     cwd.push("projects");
//     cwd.push("emails");
//     let emails_storage_folder = cwd.as_path().to_str().unwrap();

// println!("{:?}", emails_storage_folder);
// let outputs = projects
//     .par_iter()
//     .map(|p| {
//         // let p = &s.clone();
//         let split = p.split(",").collect::<Vec<_>>();
//         let repo_name = split[0];
//         let _repo_path = split[1];
//         let start_date = split[2];
//         let end_date = split[3];
//         let status = split[4];
//         // download email archives

//         // let ids = mboxes_ids(&repo_name.to_lowercase());
//         // we already have the mboxes locally
//         let ids = local_mboxes_ids(&repo_name, &emails_storage_folder);
//         let mut available_mboxes = vec![];
//         println!("Checking for a local mail archive for {}", repo_name);
//         for id in ids.into_iter() {
//             let path = format!(
//                 "{}/{}-dev-{}.mbox",
//                 emails_storage_folder,
//                 repo_name.to_lowercase(),
//                 id
//             );
//             let file_path = path.replace("\\", "/");
//             let file_path_on_disk = file_path.clone();
//             // if we don't have the mbox file, need to download it
//             let file_exists = Path::new(&file_path).is_file();

//             if file_exists {
//                 available_mboxes.push(file_path.to_string());
//             }
//             if file_exists == false && args.flag_skip_emails_download == true {
//                 continue;
//             }
//             if file_exists == false && args.flag_skip_emails_download == false {
//                 let url = format!(
//                     "http://mail-archives.apache.org/mod_mbox/{}-dev/{}.mbox",
//                     repo_name.to_lowercase(),
//                     id
//                 );
//                 println!(
//                     "{}: {} does not exist locally. Trying to download from {}",
//                     repo_name, &file_path, &url
//                 );
//                 let val = ureq::get(&url)
//                     .timeout(std::time::Duration::new(20, 0))
//                     .call();

//                 match val {
//                     Ok(s) => {
//                         let resp = s.into_string().unwrap();
//                         let f = File::create(file_path_on_disk);
//                         if let Ok(mut ff) = f {
//                             ff.write(resp.as_bytes());
//                             ff.flush();
//                             available_mboxes.push(file_path.to_string());
//                         }
//                     }
//                     Err(_) => println!("Error: {} - request timed out {}", repo_name, url),
//                 }
//             }
//         }
//         println!("Finished downloading mail archive for {}", repo_name);
// println!("{:?}", available_mboxes.clone());
//     available_mboxes.reverse();
//     let output = parse_mbox_files(
//         args,
//         repo_name,
//         status,
//         start_date,
//         end_date,
//         available_mboxes,
//     );

//     let mut f = File::create(format!(
//         "../data/emails-csvs/{}.csv",
//         repo_name.to_lowercase()
//     ))
//     .expect("Unable to create file");
//     match writeln!(f, "{}", output.join("\n")) {
//         Ok(()) => {}
//         Err(e) => println!("{} error occured during writing the file. {}", repo_name, e),
//     }
//     let r = repo_name.to_string().clone();
//     (r, output.len())
// })
// .collect::<Vec<_>>();

// let mut f = File::create("../data/emails-stats.csv").expect("Unable to create file");

// for (key, value) in &outputs {
//     write!(f, "{},{}\n", key, value);
// }
// match f.flush() {
//     Ok(()) => {}
//     Err(e) => println!(
//         "Writing the emails-stats.csv: error occured during flushing the file. {}",
//         e
//     ),
// }

// let mut f = File::create("../data/emails-data.csv").expect("Unable to create file");

// write!(
//     f,
//     "project,status,start_date,end_date,string_date,from,from_email,sender,sender_email,to_name,to_email\n"
// );

// match f.flush() {
//     Ok(()) => {}
//     Err(e) => println!(
//         "Writing the emails-stats.csv: error occured during flushing the file. {}",
//         e
//     ),
// }

// println!(
//     "Total emails analyzed: {:?}",
//     outputs
//         .into_iter()
//         .map(|s| s.1)
//         .collect::<Vec<_>>()
//         .iter()
//         .sum::<usize>()
// );
// Ok(())
// }
