#ifndef lcForces_H
#define lcForces_H
#include "std_include.h"
/*! \file lcForces.h */

#ifdef __NVCC__
#define HOSTDEVICE __host__ __device__ inline
#else
#define HOSTDEVICE inline __attribute__((always_inline))
#endif

/*!
Note that all of these "forces" actually return the negative of a force
*/
namespace lcForce
    {
    HOSTDEVICE int getBoundaryCase(int xd, int xu, int yd, int yu, int zd, int zu)
        {
        int xx = 0;
        int yy = 0;
        int zz = 0;
        if(xu >0) xx = 1;
        if(xd >0) xx = 2;
        if(yu >0) yy = 1;
        if(yd >0) yy = 2;
        if(zu >0) zz = 1;
        if(zd >0) zz = 2;
        return xx + 4*yy+16*zz;
        }

    HOSTDEVICE void bulkL1Force(const scalar L1, const dVec &qCurrent,
                const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
                dVec &spatialTerm)
        {
        spatialTerm = L1*(6.0*qCurrent-xDown-xUp-yDown-yUp-zDown-zUp);
        scalar AxxAyy = spatialTerm[0]+spatialTerm[3];
        spatialTerm[0] += AxxAyy;
        spatialTerm[1] *= 2.0;
        spatialTerm[2] *= 2.0;
        spatialTerm[3] += AxxAyy;
        spatialTerm[4] *= 2.0;
        };

    HOSTDEVICE void boundaryL1Force(const scalar L1, const dVec &qCurrent,
                const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
                const int &ixd,const int &ixu,const int &iyd,const int &iyu,const int &izd,const int &izu,
                dVec &spatialTerm)
        {
        if(ixd >0)//xDown is a boundary
            spatialTerm -= (xUp-qCurrent);
        if(ixu >0)//xUp is a boundary
            spatialTerm -= (xDown-qCurrent);//negative derivative and negative nu_x cancel
        if(iyd >0)//ydown is a boundary
            spatialTerm -= (yUp-qCurrent);
        if(iyu >0)
            spatialTerm -= (yDown-qCurrent);//negative derivative and negative nu_y cancel
        if(izd >0)//zDown is boundary
            spatialTerm -= (zUp-qCurrent);
        if(izu >0)
            spatialTerm -= (zDown-qCurrent);//negative derivative and negative nu_z cancel
        spatialTerm = spatialTerm*L1;
        scalar crossTerm = spatialTerm[0]+spatialTerm[3];
        spatialTerm[0] += crossTerm;
        spatialTerm[1] *= 2.0;
        spatialTerm[2] *= 2.0;
        spatialTerm[3] += crossTerm;
        spatialTerm[4] *= 2.0;
        };

    HOSTDEVICE void bulkL2Force(const scalar L2, const dVec &qCurrent,
            const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
            const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
            const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
            const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
            dVec &spatialTerm)
        {
            spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] - xUpDerivative[6] - xUpDerivative[12] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
    		spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 4*yDown[1] + yDownDerivative[0] - 4*yUp[1] - yUpDerivative[0] - yUpDerivative[12])/2.;
    		spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*zDown[2] + zDownDerivative[0] + zDownDerivative[6] - 4*zUp[2] - zUpDerivative[0] - zUpDerivative[6])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] + xDownDerivative[14] - 4*yDown[3] + yDownDerivative[1] - 4*yUp[3] - yUpDerivative[1] - yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
    		spatialTerm[4] = (16*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] - 4*yDown[4] + yDownDerivative[2] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
            spatialTerm = (0.5*L2)*spatialTerm;
        };

    HOSTDEVICE void boundaryL2Force(const scalar L2, int boundaryCase, const dVec &qCurrent,
        const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
        const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
        const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
        const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
        dVec &spatialTerm)
        {
            if(boundaryCase == 0)
        		{
        		spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] - xUpDerivative[6] - xUpDerivative[12] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 4*yDown[1] + yDownDerivative[0] - 4*yUp[1] - yUpDerivative[0] - yUpDerivative[12])/2.;
        		spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*zDown[2] + zDownDerivative[0] + zDownDerivative[6] - 4*zUp[2] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] + xDownDerivative[14] - 4*yDown[3] + yDownDerivative[1] - 4*yUp[3] - yUpDerivative[1] - yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (16*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] - 4*yDown[4] + yDownDerivative[2] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 1)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 8*qCurrent[3] - 6*xDown[0] + xDownDerivative[6] + xDownDerivative[12] - 2*yDown[1] + 2*yUp[1] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 4*yDown[1] - 2*yDown[3] + yDownDerivative[0] - 4*yUp[1] + 2*yUp[3] - yUpDerivative[0] - yUpDerivative[12] - 2*zDown[4] + 2*zUp[4])/2.;
        		spatialTerm[2] = (14*qCurrent[2] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 2*yDown[4] + 2*yUp[4] + 2*zDown[0] - 4*zDown[2] + 2*zDown[3] + zDownDerivative[0] + zDownDerivative[6] - 2*zUp[0] - 4*zUp[2] - 2*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] + xDownDerivative[14] - 4*yDown[3] + yDownDerivative[1] - 4*yUp[3] - yUpDerivative[1] - yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (16*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] - 4*yDown[4] + yDownDerivative[2] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 2)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 8*qCurrent[3] - 6*xUp[0] - xUpDerivative[6] - xUpDerivative[12] + 2*yDown[1] - 2*yUp[1] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] + xDownDerivative[12] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 4*yDown[1] + 2*yDown[3] + yDownDerivative[0] - 4*yUp[1] - 2*yUp[3] - yUpDerivative[0] - yUpDerivative[12] + 2*zDown[4] - 2*zUp[4])/2.;
        		spatialTerm[2] = (14*qCurrent[2] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 2*yDown[4] - 2*yUp[4] - 2*zDown[0] - 4*zDown[2] - 2*zDown[3] + zDownDerivative[0] + zDownDerivative[6] + 2*zUp[0] - 4*zUp[2] + 2*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] + xDownDerivative[14] - 4*yDown[3] + yDownDerivative[1] - 4*yUp[3] - yUpDerivative[1] - yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (16*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] - 4*yDown[4] + yDownDerivative[2] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 4)
        		{
        		spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] - xUpDerivative[6] - xUpDerivative[12] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] - 2*xDown[0] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] + 2*xUp[0] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yDown[1] + yDownDerivative[0] - 2*zDown[2] + 2*zUp[2])/2.;
        		spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*zDown[2] + zDownDerivative[0] + zDownDerivative[6] - 4*zUp[2] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] + 14*qCurrent[3] - 2*xDown[1] + xDownDerivative[14] + 2*xUp[1] - 6*yDown[3] + yDownDerivative[1] - 4*zDown[0] - 4*zDown[3] - 2*zDown[4] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + 2*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (14*qCurrent[4] - 2*xDown[2] - xDownDerivative[10] - xDownDerivative[13] + 2*xUp[2] - 6*yDown[4] + yDownDerivative[2] + 2*zDown[0] + 2*zDown[3] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] - 2*zUp[0] - 2*zUp[3] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 5)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 4*qCurrent[1] + 8*qCurrent[3] - 6*xDown[0] + xDownDerivative[6] + xDownDerivative[12] - 4*yDown[1] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[3] - 4*xDown[0] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 6*yDown[1] - 4*yDown[3] + yDownDerivative[0] - 2*zDown[2] - 2*zDown[4] + 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (14*qCurrent[2] + 4*qCurrent[4] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 4*yDown[4] + 2*zDown[0] - 4*zDown[2] + 2*zDown[3] + zDownDerivative[0] + zDownDerivative[6] - 2*zUp[0] - 4*zUp[2] - 2*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] + 4*qCurrent[1] + 14*qCurrent[3] - 4*xDown[1] + xDownDerivative[14] - 6*yDown[3] + yDownDerivative[1] - 4*zDown[0] - 4*zDown[3] - 2*zDown[4] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + 2*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[2] + 14*qCurrent[4] - 4*xDown[2] - xDownDerivative[10] - xDownDerivative[13] - 6*yDown[4] + yDownDerivative[2] + 2*zDown[0] + 2*zDown[3] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] - 2*zUp[0] - 2*zUp[3] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 6)
        		{
        		spatialTerm[0] = (14*qCurrent[0] - 4*qCurrent[1] + 8*qCurrent[3] - 6*xUp[0] - xUpDerivative[6] - xUpDerivative[12] + 4*yDown[1] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[3] + xDownDerivative[12] + 4*xUp[0] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yDown[1] + 4*yDown[3] + yDownDerivative[0] - 2*zDown[2] + 2*zDown[4] + 2*zUp[2] - 2*zUp[4])/2.;
        		spatialTerm[2] = (14*qCurrent[2] - 4*qCurrent[4] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 4*yDown[4] - 2*zDown[0] - 4*zDown[2] - 2*zDown[3] + zDownDerivative[0] + zDownDerivative[6] + 2*zUp[0] - 4*zUp[2] + 2*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] - 4*qCurrent[1] + 14*qCurrent[3] + xDownDerivative[14] + 4*xUp[1] - 6*yDown[3] + yDownDerivative[1] - 4*zDown[0] - 4*zDown[3] - 2*zDown[4] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + 2*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[2] + 14*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] + 4*xUp[2] - 6*yDown[4] + yDownDerivative[2] + 2*zDown[0] + 2*zDown[3] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] - 2*zUp[0] - 2*zUp[3] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 8)
        		{
        		spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] - xUpDerivative[6] - xUpDerivative[12] - 4*zDown[0] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] + 2*xDown[0] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[14] - 2*xUp[0] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yUp[1] - yUpDerivative[0] - yUpDerivative[12] + 2*zDown[2] - 2*zUp[2])/2.;
        		spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*zDown[2] + zDownDerivative[0] + zDownDerivative[6] - 4*zUp[2] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] + 14*qCurrent[3] + 2*xDown[1] - 2*xUp[1] - 6*yUp[3] - yUpDerivative[1] - yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] + 2*zDown[4] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] - 2*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (14*qCurrent[4] + 2*xDown[2] - 2*xUp[2] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 2*zDown[0] - 2*zDown[3] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] + 2*zUp[0] + 2*zUp[3] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 9)
        		{
        		spatialTerm[0] = (14*qCurrent[0] - 4*qCurrent[1] + 8*qCurrent[3] - 6*xDown[0] + xDownDerivative[6] + xDownDerivative[12] + 4*yUp[1] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[3] + 4*xDown[0] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[14] - 6*yUp[1] + 4*yUp[3] - yUpDerivative[0] - yUpDerivative[12] + 2*zDown[2] - 2*zDown[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (14*qCurrent[2] - 4*qCurrent[4] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] + 4*yUp[4] + 2*zDown[0] - 4*zDown[2] + 2*zDown[3] + zDownDerivative[0] + zDownDerivative[6] - 2*zUp[0] - 4*zUp[2] - 2*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] - 4*qCurrent[1] + 14*qCurrent[3] + 4*xDown[1] - 6*yUp[3] - yUpDerivative[1] - yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] + 2*zDown[4] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] - 2*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[2] + 14*qCurrent[4] + 4*xDown[2] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 2*zDown[0] - 2*zDown[3] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] + 2*zUp[0] + 2*zUp[3] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 10)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 4*qCurrent[1] + 8*qCurrent[3] - 6*xUp[0] - xUpDerivative[6] - xUpDerivative[12] - 4*yUp[1] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = 2*qCurrent[0] + 6*qCurrent[1] + 2*qCurrent[3] - 2*xUp[0] - 3*xUp[1] - xUpDerivative[8]/2. - xUpDerivative[14]/2. - 3*yUp[1] - 2*yUp[3] - yUpDerivative[0]/2. - yUpDerivative[12]/2. + zDown[2] + zDown[4] - zUp[2] - zUp[4];
        		spatialTerm[2] = (14*qCurrent[2] + 4*qCurrent[4] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*yUp[4] - 2*zDown[0] - 4*zDown[2] - 2*zDown[3] + zDownDerivative[0] + zDownDerivative[6] + 2*zUp[0] - 4*zUp[2] + 2*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (8*qCurrent[0] + 4*qCurrent[1] + 14*qCurrent[3] - 4*xUp[1] - 6*yUp[3] - yUpDerivative[1] - yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] + 2*zDown[4] - zDownDerivative[2] - zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] - 2*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[2] + 14*qCurrent[4] - 4*xUp[2] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 2*zDown[0] - 2*zDown[3] - 4*zDown[4] + zDownDerivative[1] + zDownDerivative[8] + 2*zUp[0] + 2*zUp[3] - 4*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 16)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*xDown[0] + 2*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] - 2*xUp[2] - xUpDerivative[6] - xUpDerivative[12] + 2*yDown[4] - 2*yUp[4] - 6*zDown[0] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 4*yDown[1] + yDownDerivative[0] - 4*yUp[1] - yUpDerivative[0] - yUpDerivative[12])/2.;
        		spatialTerm[2] = (14*qCurrent[2] - 2*xDown[0] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] + 2*xUp[0] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 2*yDown[1] + 2*yUp[1] - 6*zDown[2] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 14*qCurrent[3] + 2*xDown[2] + xDownDerivative[14] - 2*xUp[2] - 4*yDown[3] + 2*yDown[4] + yDownDerivative[1] - 4*yUp[3] - 2*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (14*qCurrent[4] - 2*xDown[1] - xDownDerivative[10] - xDownDerivative[13] + 2*xUp[1] - 2*yDown[3] - 4*yDown[4] + yDownDerivative[2] + 2*yUp[3] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 17)
        		{
        		spatialTerm[0] = (12*qCurrent[0] + 6*qCurrent[3] - 6*xDown[0] + 4*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 2*yDown[1] + 2*yDown[4] + 2*yUp[1] - 2*yUp[4] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[4] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 4*yDown[1] - 2*yDown[3] + yDownDerivative[0] - 4*yUp[1] + 2*yUp[3] - yUpDerivative[0] - yUpDerivative[12] - 4*zDown[4])/2.;
        		spatialTerm[2] = (12*qCurrent[2] - 4*qCurrent[3] - 4*xDown[0] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 2*yDown[1] - 2*yDown[4] + 2*yUp[1] + 2*yUp[4] + 4*zDown[0] - 6*zDown[2] + 4*zDown[3] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[2] + 14*qCurrent[3] + 4*xDown[2] + xDownDerivative[14] - 4*yDown[3] + 2*yDown[4] + yDownDerivative[1] - 4*yUp[3] - 2*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[1] + 14*qCurrent[4] - 4*xDown[1] - xDownDerivative[10] - xDownDerivative[13] - 2*yDown[3] - 4*yDown[4] + yDownDerivative[2] + 2*yUp[3] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 18)
        		{
        		spatialTerm[0] = 6*qCurrent[0] + 3*qCurrent[3] - 3*xUp[0] - 2*xUp[2] - xUpDerivative[6]/2. - xUpDerivative[12]/2. + yDown[1] + yDown[4] - yUp[1] - yUp[4] - 3*zDown[0] + 2*zDown[2] - 3*zDown[3] - zDownDerivative[2]/2. - zDownDerivative[9]/2.;
        		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[4] + xDownDerivative[12] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 4*yDown[1] + 2*yDown[3] + yDownDerivative[0] - 4*yUp[1] - 2*yUp[3] - yUpDerivative[0] - yUpDerivative[12] + 4*zDown[4])/2.;
        		spatialTerm[2] = (12*qCurrent[2] + 4*qCurrent[3] + 4*xUp[0] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 2*yDown[1] + 2*yDown[4] + 2*yUp[1] - 2*yUp[4] - 4*zDown[0] - 6*zDown[2] - 4*zDown[3] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[2] + 14*qCurrent[3] + xDownDerivative[14] - 4*xUp[2] - 4*yDown[3] + 2*yDown[4] + yDownDerivative[1] - 4*yUp[3] - 2*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[1] + 14*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] + 4*xUp[1] - 2*yDown[3] - 4*yDown[4] + yDownDerivative[2] + 2*yUp[3] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 20)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*qCurrent[4] - 4*xDown[0] + 2*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] - 2*xUp[2] - xUpDerivative[6] - xUpDerivative[12] + 4*yDown[4] - 6*zDown[0] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[2] - 2*xDown[0] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] + 2*xUp[0] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yDown[1] + yDownDerivative[0] - 4*zDown[2])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 14*qCurrent[2] - 2*xDown[0] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] + 2*xUp[0] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*yDown[1] - 6*zDown[2] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 12*qCurrent[3] - 2*xDown[1] + 2*xDown[2] + xDownDerivative[14] + 2*xUp[1] - 2*xUp[2] - 6*yDown[3] + 4*yDown[4] + yDownDerivative[1] - 6*zDown[0] - 6*zDown[3] - 4*zDown[4] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[0] + 12*qCurrent[4] - 2*xDown[1] - 2*xDown[2] - xDownDerivative[10] - xDownDerivative[13] + 2*xUp[1] + 2*xUp[2] - 4*yDown[3] - 6*yDown[4] + yDownDerivative[2] + 4*zDown[0] + 4*zDown[3] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 21)
        		{
        		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - 6*xDown[0] + 4*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*yDown[1] + 4*yDown[4] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] - 4*xDown[0] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 6*yDown[1] - 4*yDown[3] + yDownDerivative[0] - 4*zDown[2] - 4*zDown[4])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] - 4*xDown[0] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 4*yDown[1] - 4*yDown[4] + 4*zDown[0] - 6*zDown[2] + 4*zDown[3] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] - 4*xDown[1] + 4*xDown[2] + xDownDerivative[14] - 6*yDown[3] + 4*yDown[4] + yDownDerivative[1] - 6*zDown[0] - 6*zDown[3] - 4*zDown[4] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] - 4*xDown[1] - 4*xDown[2] - xDownDerivative[10] - xDownDerivative[13] - 4*yDown[3] - 6*yDown[4] + yDownDerivative[2] + 4*zDown[0] + 4*zDown[3] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 22)
        		{
        		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - 6*xUp[0] - 4*xUp[2] - xUpDerivative[6] - xUpDerivative[12] + 4*yDown[1] + 4*yDown[4] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] + xDownDerivative[12] + 4*xUp[0] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yDown[1] + 4*yDown[3] + yDownDerivative[0] - 4*zDown[2] + 4*zDown[4])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] + 4*xUp[0] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*yDown[1] + 4*yDown[4] - 4*zDown[0] - 6*zDown[2] - 4*zDown[3] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] + xDownDerivative[14] + 4*xUp[1] - 4*xUp[2] - 6*yDown[3] + 4*yDown[4] + yDownDerivative[1] - 6*zDown[0] - 6*zDown[3] - 4*zDown[4] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] + 4*xUp[1] + 4*xUp[2] - 4*yDown[3] - 6*yDown[4] + yDownDerivative[2] + 4*zDown[0] + 4*zDown[3] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 24)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] + 4*qCurrent[4] - 4*xDown[0] + 2*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] - 2*xUp[2] - xUpDerivative[6] - xUpDerivative[12] - 4*yUp[4] - 6*zDown[0] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[2] + 2*xDown[0] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[14] - 2*xUp[0] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yUp[1] - yUpDerivative[0] - yUpDerivative[12] + 4*zDown[2])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] + 14*qCurrent[2] - 2*xDown[0] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] + 2*xUp[0] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 4*yUp[1] - 6*zDown[2] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = 3*qCurrent[0] + 6*qCurrent[3] + xDown[1] + xDown[2] - xUp[1] - xUp[2] - 3*yUp[3] - 2*yUp[4] - yUpDerivative[1]/2. - yUpDerivative[14]/2. - 3*zDown[0] - 3*zDown[3] + 2*zDown[4] - zDownDerivative[2]/2. - zDownDerivative[9]/2.;
        		spatialTerm[4] = (4*qCurrent[0] + 12*qCurrent[4] - 2*xDown[1] + 2*xDown[2] + 2*xUp[1] - 2*xUp[2] + 4*yUp[3] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 4*zDown[0] - 4*zDown[3] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 25)
        		{
        		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - 6*xDown[0] + 4*xDown[2] + xDownDerivative[6] + xDownDerivative[12] + 4*yUp[1] - 4*yUp[4] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] + 4*xDown[0] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[14] - 6*yUp[1] + 4*yUp[3] - yUpDerivative[0] - yUpDerivative[12] + 4*zDown[2] - 4*zDown[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] - 4*xDown[0] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] + 4*yUp[1] + 4*yUp[4] + 4*zDown[0] - 6*zDown[2] + 4*zDown[3] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] + 4*xDown[1] + 4*xDown[2] - 6*yUp[3] - 4*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] + 4*zDown[4] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] - 4*xDown[1] + 4*xDown[2] + 4*yUp[3] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 4*zDown[0] - 4*zDown[3] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 26)
        		{
        		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - 6*xUp[0] - 4*xUp[2] - xUpDerivative[6] - xUpDerivative[12] - 4*yUp[1] - 4*yUp[4] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] - 4*xUp[0] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yUp[1] - 4*yUp[3] - yUpDerivative[0] - yUpDerivative[12] + 4*zDown[2] + 4*zDown[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] + 4*xUp[0] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 4*yUp[1] - 4*yUp[4] - 4*zDown[0] - 6*zDown[2] - 4*zDown[3] + zDownDerivative[0] + zDownDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] - 4*xUp[1] - 4*xUp[2] - 6*yUp[3] - 4*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] + 4*zDown[4] - zDownDerivative[2] - zDownDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] + 4*xUp[1] - 4*xUp[2] + 4*yUp[3] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 4*zDown[0] - 4*zDown[3] - 6*zDown[4] + zDownDerivative[1] + zDownDerivative[8])/2.;
        		};
        	if(boundaryCase == 32)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*xDown[0] - 2*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] + 2*xUp[2] - xUpDerivative[6] - xUpDerivative[12] - 2*yDown[4] + 2*yUp[4] - 6*zUp[0] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 4*yDown[1] + yDownDerivative[0] - 4*yUp[1] - yUpDerivative[0] - yUpDerivative[12])/2.;
        		spatialTerm[2] = (14*qCurrent[2] + 2*xDown[0] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 2*xUp[0] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 2*yDown[1] - 2*yUp[1] - 6*zUp[2] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 14*qCurrent[3] - 2*xDown[2] + xDownDerivative[14] + 2*xUp[2] - 4*yDown[3] - 2*yDown[4] + yDownDerivative[1] - 4*yUp[3] + 2*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (14*qCurrent[4] + 2*xDown[1] - xDownDerivative[10] - xDownDerivative[13] - 2*xUp[1] + 2*yDown[3] - 4*yDown[4] + yDownDerivative[2] - 2*yUp[3] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 33)
        		{
        		spatialTerm[0] = (12*qCurrent[0] + 6*qCurrent[3] - 6*xDown[0] - 4*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 2*yDown[1] - 2*yDown[4] + 2*yUp[1] + 2*yUp[4] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[4] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 4*yDown[1] - 2*yDown[3] + yDownDerivative[0] - 4*yUp[1] + 2*yUp[3] - yUpDerivative[0] - yUpDerivative[12] + 4*zUp[4])/2.;
        		spatialTerm[2] = (12*qCurrent[2] + 4*qCurrent[3] + 4*xDown[0] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] + 2*yDown[1] - 2*yDown[4] - 2*yUp[1] + 2*yUp[4] - 4*zUp[0] - 6*zUp[2] - 4*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[2] + 14*qCurrent[3] - 4*xDown[2] + xDownDerivative[14] - 4*yDown[3] - 2*yDown[4] + yDownDerivative[1] - 4*yUp[3] + 2*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[1] + 14*qCurrent[4] + 4*xDown[1] - xDownDerivative[10] - xDownDerivative[13] + 2*yDown[3] - 4*yDown[4] + yDownDerivative[2] - 2*yUp[3] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 34)
        		{
        		spatialTerm[0] = (12*qCurrent[0] + 6*qCurrent[3] - 6*xUp[0] + 4*xUp[2] - xUpDerivative[6] - xUpDerivative[12] + 2*yDown[1] - 2*yDown[4] - 2*yUp[1] + 2*yUp[4] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[4] + xDownDerivative[12] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 4*yDown[1] + 2*yDown[3] + yDownDerivative[0] - 4*yUp[1] - 2*yUp[3] - yUpDerivative[0] - yUpDerivative[12] - 4*zUp[4])/2.;
        		spatialTerm[2] = (12*qCurrent[2] - 4*qCurrent[3] - 4*xUp[0] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 2*yDown[1] + 2*yDown[4] - 2*yUp[1] - 2*yUp[4] + 4*zUp[0] - 6*zUp[2] + 4*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[2] + 14*qCurrent[3] + xDownDerivative[14] + 4*xUp[2] - 4*yDown[3] - 2*yDown[4] + yDownDerivative[1] - 4*yUp[3] + 2*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[1] + 14*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] - 4*xUp[1] + 2*yDown[3] - 4*yDown[4] + yDownDerivative[2] - 2*yUp[3] - 4*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 36)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] + 4*qCurrent[4] - 4*xDown[0] - 2*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] + 2*xUp[2] - xUpDerivative[6] - xUpDerivative[12] - 4*yDown[4] - 6*zUp[0] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[2] - 2*xDown[0] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] + 2*xUp[0] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yDown[1] + yDownDerivative[0] + 4*zUp[2])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] + 14*qCurrent[2] + 2*xDown[0] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 2*xUp[0] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 4*yDown[1] - 6*zUp[2] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 12*qCurrent[3] - 2*xDown[1] - 2*xDown[2] + xDownDerivative[14] + 2*xUp[1] + 2*xUp[2] - 6*yDown[3] - 4*yDown[4] + yDownDerivative[1] - 6*zUp[0] - 6*zUp[3] + 4*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[0] + 12*qCurrent[4] + 2*xDown[1] - 2*xDown[2] - xDownDerivative[10] - xDownDerivative[13] - 2*xUp[1] + 2*xUp[2] + 4*yDown[3] - 6*yDown[4] + yDownDerivative[2] - 4*zUp[0] - 4*zUp[3] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 37)
        		{
        		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - 6*xDown[0] - 4*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*yDown[1] - 4*yDown[4] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] - 4*xDown[0] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[12] + xDownDerivative[14] - 6*yDown[1] - 4*yDown[3] + yDownDerivative[0] + 4*zUp[2] + 4*zUp[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] + 4*xDown[0] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] + 4*yDown[1] - 4*yDown[4] - 4*zUp[0] - 6*zUp[2] - 4*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] - 4*xDown[1] - 4*xDown[2] + xDownDerivative[14] - 6*yDown[3] - 4*yDown[4] + yDownDerivative[1] - 6*zUp[0] - 6*zUp[3] + 4*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] + 4*xDown[1] - 4*xDown[2] - xDownDerivative[10] - xDownDerivative[13] + 4*yDown[3] - 6*yDown[4] + yDownDerivative[2] - 4*zUp[0] - 4*zUp[3] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 38)
        		{
        		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - 6*xUp[0] + 4*xUp[2] - xUpDerivative[6] - xUpDerivative[12] + 4*yDown[1] - 4*yDown[4] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] + xDownDerivative[12] + 4*xUp[0] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yDown[1] + 4*yDown[3] + yDownDerivative[0] + 4*zUp[2] - 4*zUp[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] - 4*xUp[0] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] + 4*yDown[1] + 4*yDown[4] + 4*zUp[0] - 6*zUp[2] + 4*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] + xDownDerivative[14] + 4*xUp[1] + 4*xUp[2] - 6*yDown[3] - 4*yDown[4] + yDownDerivative[1] - 6*zUp[0] - 6*zUp[3] + 4*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (4*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] - xDownDerivative[10] - xDownDerivative[13] - 4*xUp[1] + 4*xUp[2] + 4*yDown[3] - 6*yDown[4] + yDownDerivative[2] - 4*zUp[0] - 4*zUp[3] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 40)
        		{
        		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*qCurrent[4] - 4*xDown[0] - 2*xDown[2] + xDownDerivative[6] + xDownDerivative[12] - 4*xUp[0] + 2*xUp[2] - xUpDerivative[6] - xUpDerivative[12] + 4*yUp[4] - 6*zUp[0] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[2] + 2*xDown[0] - 4*xDown[1] + xDownDerivative[8] + xDownDerivative[14] - 2*xUp[0] - 4*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yUp[1] - yUpDerivative[0] - yUpDerivative[12] - 4*zUp[2])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 14*qCurrent[2] + 2*xDown[0] - 4*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 2*xUp[0] - 4*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*yUp[1] - 6*zUp[2] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 12*qCurrent[3] + 2*xDown[1] - 2*xDown[2] - 2*xUp[1] + 2*xUp[2] - 6*yUp[3] + 4*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] - 4*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[0] + 12*qCurrent[4] + 2*xDown[1] + 2*xDown[2] - 2*xUp[1] - 2*xUp[2] - 4*yUp[3] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] + 4*zUp[0] + 4*zUp[3] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 41)
        		{
        		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - 6*xDown[0] - 4*xDown[2] + xDownDerivative[6] + xDownDerivative[12] + 4*yUp[1] + 4*yUp[4] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] + 4*xDown[0] - 6*xDown[1] + xDownDerivative[8] + xDownDerivative[14] - 6*yUp[1] + 4*yUp[3] - yUpDerivative[0] - yUpDerivative[12] - 4*zUp[2] + 4*zUp[4])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] + 4*xDown[0] - 6*xDown[2] + xDownDerivative[9] - xDownDerivative[10] - xDownDerivative[13] - 4*yUp[1] + 4*yUp[4] - 4*zUp[0] - 6*zUp[2] - 4*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] + 4*xDown[1] - 4*xDown[2] - 6*yUp[3] + 4*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] - 4*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] + 4*xDown[1] + 4*xDown[2] - 4*yUp[3] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] + 4*zUp[0] + 4*zUp[3] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};
        	if(boundaryCase == 42)
        		{
        		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - 6*xUp[0] + 4*xUp[2] - xUpDerivative[6] - xUpDerivative[12] - 4*yUp[1] + 4*yUp[4] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] - 4*xUp[0] - 6*xUp[1] - xUpDerivative[8] - xUpDerivative[14] - 6*yUp[1] - 4*yUp[3] - yUpDerivative[0] - yUpDerivative[12] - 4*zUp[2] - 4*zUp[4])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] - 4*xUp[0] - 6*xUp[2] - xUpDerivative[9] + xUpDerivative[10] + xUpDerivative[13] - 4*yUp[1] - 4*yUp[4] + 4*zUp[0] - 6*zUp[2] + 4*zUp[3] - zUpDerivative[0] - zUpDerivative[6])/2.;
        		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] - 4*xUp[1] + 4*xUp[2] - 6*yUp[3] + 4*yUp[4] - yUpDerivative[1] - yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] - 4*zUp[4] + zUpDerivative[2] + zUpDerivative[9])/2.;
        		spatialTerm[4] = (-4*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] - 4*xUp[1] - 4*xUp[2] - 4*yUp[3] - 6*yUp[4] - yUpDerivative[2] + yUpDerivative[10] + yUpDerivative[13] + 4*zUp[0] + 4*zUp[3] - 6*zUp[4] - zUpDerivative[1] - zUpDerivative[8])/2.;
        		};


        spatialTerm = (0.5*L2)*spatialTerm;
        };

    HOSTDEVICE void bulkL3Force(const scalar L3, const dVec &qCurrent,
            const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
            const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
            const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
            const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
            dVec &spatialTerm)
        {
        spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] - xDownDerivative[12] - xDownDerivative[14] - 4*xUp[0] + xUpDerivative[12] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] + zDownDerivative[2] - 4*zUp[0] - 4*zUp[3] - zUpDerivative[2])/2.;
        spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] + xDownDerivative[5] - 4*xUp[1] - xUpDerivative[5] - 4*yDown[1] + yDownDerivative[3] - 4*yUp[1] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
        spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] + xDownDerivative[10] + xDownDerivative[11] - 4*xUp[2] - xUpDerivative[10] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
        spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] - 4*yDown[3] - 4*yUp[3] + yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] + zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] - zUpDerivative[9])/2.;
        spatialTerm[4] = (16*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] - 4*yDown[4] - 4*yUp[4] - yUpDerivative[13] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
        spatialTerm = (0.5*L3)*spatialTerm;
        };

    HOSTDEVICE void boundaryL3Force(const scalar L3, int boundaryCase, const dVec &qCurrent,
        const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
        const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
        const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
        const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
        dVec &spatialTerm)
        {
        if(boundaryCase == 0)
    		{
    		spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] - xDownDerivative[12] - xDownDerivative[14] - 4*xUp[0] + xUpDerivative[12] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] + zDownDerivative[2] - 4*zUp[0] - 4*zUp[3] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] + xDownDerivative[5] - 4*xUp[1] - xUpDerivative[5] - 4*yDown[1] + yDownDerivative[3] - 4*yUp[1] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] + xDownDerivative[10] + xDownDerivative[11] - 4*xUp[2] - xUpDerivative[10] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] - 4*yDown[3] - 4*yUp[3] + yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] + zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (16*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] - 4*yDown[4] - 4*yUp[4] - yUpDerivative[13] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 1)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 8*qCurrent[3] - 6*xDown[0] - xDownDerivative[12] - xDownDerivative[14] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] + zDownDerivative[2] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 6*xDown[1] + xDownDerivative[5] - 2*yDown[0] - 4*yDown[1] + yDownDerivative[3] + 2*yUp[0] - 4*yUp[1] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] - 6*xDown[2] + xDownDerivative[10] + xDownDerivative[11] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] - 2*zDown[0] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] + 2*zUp[0] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - 2*yDown[1] - 4*yDown[3] + 2*yUp[1] - 4*yUp[3] + yUpDerivative[14] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] + zDownDerivative[9] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (16*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - 2*yDown[2] - 4*yDown[4] + 2*yUp[2] - 4*yUp[4] - yUpDerivative[13] - 2*zDown[1] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] + 2*zUp[1] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 2)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 8*qCurrent[3] - xDownDerivative[14] - 6*xUp[0] + xUpDerivative[12] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] + zDownDerivative[2] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 6*xUp[1] - xUpDerivative[5] + 2*yDown[0] - 4*yDown[1] + yDownDerivative[3] - 2*yUp[0] - 4*yUp[1] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] + xDownDerivative[11] - 6*xUp[2] - xUpDerivative[10] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] + 2*zDown[0] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] - 2*zUp[0] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 16*qCurrent[3] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] + 2*yDown[1] - 4*yDown[3] - 2*yUp[1] - 4*yUp[3] + yUpDerivative[14] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] + zDownDerivative[9] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (16*qCurrent[4] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] + 2*yDown[2] - 4*yDown[4] - 2*yUp[2] - 4*yUp[4] - yUpDerivative[13] + 2*zDown[1] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] - 2*zUp[1] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 4)
    		{
    		spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] - 2*xDown[1] - xDownDerivative[12] - xDownDerivative[14] - 4*xUp[0] + 2*xUp[1] + xUpDerivative[12] + yDownDerivative[1] - 4*zDown[0] - 4*zDown[3] + 2*zDown[4] + zDownDerivative[2] - 4*zUp[0] - 4*zUp[3] - 2*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 4*xDown[1] - 2*xDown[3] + xDownDerivative[5] - 4*xUp[1] + 2*xUp[3] - xUpDerivative[5] - 6*yDown[1] + yDownDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] - 2*xDown[4] + xDownDerivative[10] + xDownDerivative[11] - 4*xUp[2] + 2*xUp[4] - xUpDerivative[10] + yDownDerivative[4] - 2*zDown[1] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] + 2*zUp[1] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] - 6*yDown[3] - 4*zDown[0] - 4*zDown[3] + 2*zDown[4] + zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] - 2*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] - 6*yDown[4] - 2*zDown[3] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] + 2*zUp[3] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 5)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 4*qCurrent[1] + 8*qCurrent[3] - 6*xDown[0] - 4*xDown[1] - xDownDerivative[12] - xDownDerivative[14] + yDownDerivative[1] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] + 2*zDown[4] + zDownDerivative[2] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] - 2*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[3] - 6*xDown[1] - 4*xDown[3] + xDownDerivative[5] - 4*yDown[0] - 6*yDown[1] + yDownDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] + 4*qCurrent[4] - 6*xDown[2] - 4*xDown[4] + xDownDerivative[10] + xDownDerivative[11] + yDownDerivative[4] - 2*zDown[0] - 2*zDown[1] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] + 2*zUp[0] + 2*zUp[1] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 4*qCurrent[1] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - 4*yDown[1] - 6*yDown[3] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] + 2*zDown[4] + zDownDerivative[9] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] - 2*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[2] + 14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - 4*yDown[2] - 6*yDown[4] - 2*zDown[1] - 2*zDown[3] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] + 2*zUp[1] + 2*zUp[3] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 6)
    		{
    		spatialTerm[0] = (14*qCurrent[0] - 4*qCurrent[1] + 8*qCurrent[3] - xDownDerivative[14] - 6*xUp[0] + 4*xUp[1] + xUpDerivative[12] + yDownDerivative[1] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] + 2*zDown[4] + zDownDerivative[2] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] - 2*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[3] - 6*xUp[1] + 4*xUp[3] - xUpDerivative[5] + 4*yDown[0] - 6*yDown[1] + yDownDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] - 4*qCurrent[4] + xDownDerivative[11] - 6*xUp[2] + 4*xUp[4] - xUpDerivative[10] + yDownDerivative[4] + 2*zDown[0] - 2*zDown[1] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] - 2*zUp[0] + 2*zUp[1] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] - 4*qCurrent[1] + 14*qCurrent[3] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] + 4*yDown[1] - 6*yDown[3] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] + 2*zDown[4] + zDownDerivative[9] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] - 2*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[2] + 14*qCurrent[4] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] + 4*yDown[2] - 6*yDown[4] + 2*zDown[1] - 2*zDown[3] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] - 2*zUp[1] + 2*zUp[3] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 8)
    		{
    		spatialTerm[0] = (16*qCurrent[0] + 8*qCurrent[3] - 4*xDown[0] + 2*xDown[1] - xDownDerivative[12] - 4*xUp[0] - 2*xUp[1] + xUpDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] - 2*zDown[4] + zDownDerivative[2] - 4*zUp[0] - 4*zUp[3] + 2*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 4*xDown[1] + 2*xDown[3] + xDownDerivative[5] - 4*xUp[1] - 2*xUp[3] - xUpDerivative[5] - 6*yUp[1] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (16*qCurrent[2] - 4*xDown[2] + 2*xDown[4] + xDownDerivative[10] - 4*xUp[2] - 2*xUp[4] - xUpDerivative[10] - yUpDerivative[4] - yUpDerivative[11] + 2*zDown[1] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] - 2*zUp[1] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xUpDerivative[6] + xUpDerivative[12] - 6*yUp[3] + yUpDerivative[14] - 4*zDown[0] - 4*zDown[3] - 2*zDown[4] + zDownDerivative[9] - 4*zUp[0] - 4*zUp[3] + 2*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] - xUpDerivative[7] - xUpDerivative[11] - 6*yUp[4] - yUpDerivative[13] + 2*zDown[3] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] - 2*zUp[3] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 9)
    		{
    		spatialTerm[0] = (14*qCurrent[0] - 4*qCurrent[1] + 8*qCurrent[3] - 6*xDown[0] + 4*xDown[1] - xDownDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] - 2*zDown[4] + zDownDerivative[2] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] + 2*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[3] - 6*xDown[1] + 4*xDown[3] + xDownDerivative[5] + 4*yUp[0] - 6*yUp[1] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] - 4*qCurrent[4] - 6*xDown[2] + 4*xDown[4] + xDownDerivative[10] - yUpDerivative[4] - yUpDerivative[11] - 2*zDown[0] + 2*zDown[1] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] + 2*zUp[0] - 2*zUp[1] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] - 4*qCurrent[1] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] + 4*yUp[1] - 6*yUp[3] + yUpDerivative[14] - 4*zDown[0] + 2*zDown[2] - 4*zDown[3] - 2*zDown[4] + zDownDerivative[9] - 4*zUp[0] - 2*zUp[2] - 4*zUp[3] + 2*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[2] + 14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + 4*yUp[2] - 6*yUp[4] - yUpDerivative[13] - 2*zDown[1] + 2*zDown[3] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] + 2*zUp[1] - 2*zUp[3] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 10)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 4*qCurrent[1] + 8*qCurrent[3] - 6*xUp[0] - 4*xUp[1] + xUpDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] - 2*zDown[4] + zDownDerivative[2] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] + 2*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[3] - 6*xUp[1] - 4*xUp[3] - xUpDerivative[5] - 4*yUp[0] - 6*yUp[1] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] + 4*qCurrent[4] - 6*xUp[2] - 4*xUp[4] - xUpDerivative[10] - yUpDerivative[4] - yUpDerivative[11] + 2*zDown[0] + 2*zDown[1] - 4*zDown[2] - zDownDerivative[0] - zDownDerivative[3] - 2*zUp[0] - 2*zUp[1] - 4*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (8*qCurrent[0] + 4*qCurrent[1] + 14*qCurrent[3] - xUpDerivative[6] + xUpDerivative[12] - 4*yUp[1] - 6*yUp[3] + yUpDerivative[14] - 4*zDown[0] - 2*zDown[2] - 4*zDown[3] - 2*zDown[4] + zDownDerivative[9] - 4*zUp[0] + 2*zUp[2] - 4*zUp[3] + 2*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[2] + 14*qCurrent[4] - xUpDerivative[7] - xUpDerivative[11] - 4*yUp[2] - 6*yUp[4] - yUpDerivative[13] + 2*zDown[1] + 2*zDown[3] - 4*zDown[4] - zDownDerivative[5] - zDownDerivative[8] - 2*zUp[1] - 2*zUp[3] - 4*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 16)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*xDown[0] - 2*xDown[2] - xDownDerivative[12] - xDownDerivative[14] - 4*xUp[0] + 2*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] - 2*xDown[4] + xDownDerivative[5] - 4*xUp[1] + 2*xUp[4] - xUpDerivative[5] - 4*yDown[1] - 2*yDown[2] + yDownDerivative[3] - 4*yUp[1] + 2*yUp[2] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] + 2*xDown[0] - 4*xDown[2] + 2*xDown[3] + xDownDerivative[10] + xDownDerivative[11] - 2*xUp[0] - 4*xUp[2] - 2*xUp[3] - xUpDerivative[10] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] - 4*yDown[3] - 2*yDown[4] - 4*yUp[3] + 2*yUp[4] + yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] + 2*yDown[0] + 2*yDown[3] - 4*yDown[4] - 2*yUp[0] - 2*yUp[3] - 4*yUp[4] - yUpDerivative[13] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 17)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 6*qCurrent[3] - 6*xDown[0] - 4*xDown[2] - xDownDerivative[12] - xDownDerivative[14] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[4] - 6*xDown[1] - 4*xDown[4] + xDownDerivative[5] - 2*yDown[0] - 4*yDown[1] - 2*yDown[2] + yDownDerivative[3] + 2*yUp[0] - 4*yUp[1] + 2*yUp[2] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (12*qCurrent[2] - 4*qCurrent[3] + 4*xDown[0] - 6*xDown[2] + 4*xDown[3] + xDownDerivative[10] + xDownDerivative[11] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] - 4*zDown[0] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[2] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - 2*yDown[1] - 4*yDown[3] - 2*yDown[4] + 2*yUp[1] - 4*yUp[3] + 2*yUp[4] + yUpDerivative[14] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[1] + 14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] + 2*yDown[0] - 2*yDown[2] + 2*yDown[3] - 4*yDown[4] - 2*yUp[0] + 2*yUp[2] - 2*yUp[3] - 4*yUp[4] - yUpDerivative[13] - 4*zDown[1] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 18)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 6*qCurrent[3] - xDownDerivative[14] - 6*xUp[0] + 4*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[4] - 6*xUp[1] + 4*xUp[4] - xUpDerivative[5] + 2*yDown[0] - 4*yDown[1] - 2*yDown[2] + yDownDerivative[3] - 2*yUp[0] - 4*yUp[1] + 2*yUp[2] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (12*qCurrent[2] + 4*qCurrent[3] + xDownDerivative[11] - 4*xUp[0] - 6*xUp[2] - 4*xUp[3] - xUpDerivative[10] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] + 4*zDown[0] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[2] + 14*qCurrent[3] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] + 2*yDown[1] - 4*yDown[3] - 2*yDown[4] - 2*yUp[1] - 4*yUp[3] + 2*yUp[4] + yUpDerivative[14] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[1] + 14*qCurrent[4] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] + 2*yDown[0] + 2*yDown[2] + 2*yDown[3] - 4*yDown[4] - 2*yUp[0] - 2*yUp[2] - 2*yUp[3] - 4*yUp[4] - yUpDerivative[13] + 4*zDown[1] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 20)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*qCurrent[4] - 4*xDown[0] - 2*xDown[1] - 2*xDown[2] - xDownDerivative[12] - xDownDerivative[14] - 4*xUp[0] + 2*xUp[1] + 2*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - 6*zDown[0] - 6*zDown[3] + 4*zDown[4] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[2] - 4*xDown[1] - 2*xDown[3] - 2*xDown[4] + xDownDerivative[5] - 4*xUp[1] + 2*xUp[3] + 2*xUp[4] - xUpDerivative[5] - 6*yDown[1] - 4*yDown[2] + yDownDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (4*qCurrent[1] + 14*qCurrent[2] + 2*xDown[0] - 4*xDown[2] + 2*xDown[3] - 2*xDown[4] + xDownDerivative[10] + xDownDerivative[11] - 2*xUp[0] - 4*xUp[2] - 2*xUp[3] + 2*xUp[4] - xUpDerivative[10] + yDownDerivative[4] - 4*zDown[1] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] - 6*yDown[3] - 4*yDown[4] - 6*zDown[0] - 6*zDown[3] + 4*zDown[4] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[0] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] + 4*yDown[0] + 4*yDown[3] - 6*yDown[4] - 4*zDown[3] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 21)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - 6*xDown[0] - 4*xDown[1] - 4*xDown[2] - xDownDerivative[12] - xDownDerivative[14] + yDownDerivative[1] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] + 4*zDown[4] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] - 6*xDown[1] - 4*xDown[3] - 4*xDown[4] + xDownDerivative[5] - 4*yDown[0] - 6*yDown[1] - 4*yDown[2] + yDownDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] + 4*xDown[0] - 6*xDown[2] + 4*xDown[3] - 4*xDown[4] + xDownDerivative[10] + xDownDerivative[11] + yDownDerivative[4] - 4*zDown[0] - 4*zDown[1] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - 4*yDown[1] - 6*yDown[3] - 4*yDown[4] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] + 4*zDown[4] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] + 4*yDown[0] - 4*yDown[2] + 4*yDown[3] - 6*yDown[4] - 4*zDown[1] - 4*zDown[3] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 22)
    		{
    		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - xDownDerivative[14] - 6*xUp[0] + 4*xUp[1] + 4*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] + 4*zDown[4] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] - 6*xUp[1] + 4*xUp[3] + 4*xUp[4] - xUpDerivative[5] + 4*yDown[0] - 6*yDown[1] - 4*yDown[2] + yDownDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] + xDownDerivative[11] - 4*xUp[0] - 6*xUp[2] - 4*xUp[3] + 4*xUp[4] - xUpDerivative[10] + yDownDerivative[4] + 4*zDown[0] - 4*zDown[1] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] + 4*yDown[1] - 6*yDown[3] - 4*yDown[4] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] + 4*zDown[4] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] + 4*yDown[0] + 4*yDown[2] + 4*yDown[3] - 6*yDown[4] + 4*zDown[1] - 4*zDown[3] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 24)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] + 4*qCurrent[4] - 4*xDown[0] + 2*xDown[1] - 2*xDown[2] - xDownDerivative[12] - 4*xUp[0] - 2*xUp[1] + 2*xUp[2] + xUpDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] - 4*zDown[4] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[2] - 4*xDown[1] + 2*xDown[3] - 2*xDown[4] + xDownDerivative[5] - 4*xUp[1] - 2*xUp[3] + 2*xUp[4] - xUpDerivative[5] - 6*yUp[1] + 4*yUp[2] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (-4*qCurrent[1] + 14*qCurrent[2] + 2*xDown[0] - 4*xDown[2] + 2*xDown[3] + 2*xDown[4] + xDownDerivative[10] - 2*xUp[0] - 4*xUp[2] - 2*xUp[3] - 2*xUp[4] - xUpDerivative[10] - yUpDerivative[4] - yUpDerivative[11] + 4*zDown[1] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xUpDerivative[6] + xUpDerivative[12] - 6*yUp[3] + 4*yUp[4] + yUpDerivative[14] - 6*zDown[0] - 6*zDown[3] - 4*zDown[4] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[0] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] - xUpDerivative[7] - xUpDerivative[11] - 4*yUp[0] - 4*yUp[3] - 6*yUp[4] - yUpDerivative[13] + 4*zDown[3] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 25)
    		{
    		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - 6*xDown[0] + 4*xDown[1] - 4*xDown[2] - xDownDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] - 4*zDown[4] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] - 6*xDown[1] + 4*xDown[3] - 4*xDown[4] + xDownDerivative[5] + 4*yUp[0] - 6*yUp[1] + 4*yUp[2] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] + 4*xDown[0] - 6*xDown[2] + 4*xDown[3] + 4*xDown[4] + xDownDerivative[10] - yUpDerivative[4] - yUpDerivative[11] - 4*zDown[0] + 4*zDown[1] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] + 4*yUp[1] - 6*yUp[3] + 4*yUp[4] + yUpDerivative[14] - 6*zDown[0] + 4*zDown[2] - 6*zDown[3] - 4*zDown[4] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] - 4*yUp[0] + 4*yUp[2] - 4*yUp[3] - 6*yUp[4] - yUpDerivative[13] - 4*zDown[1] + 4*zDown[3] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 26)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - 6*xUp[0] - 4*xUp[1] + 4*xUp[2] + xUpDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] - 4*zDown[4] + zDownDerivative[2])/2.;
    		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] - 6*xUp[1] - 4*xUp[3] + 4*xUp[4] - xUpDerivative[5] - 4*yUp[0] - 6*yUp[1] + 4*yUp[2] - yUpDerivative[3] + zDownDerivative[4] + zDownDerivative[7])/2.;
    		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] - 4*xUp[0] - 6*xUp[2] - 4*xUp[3] - 4*xUp[4] - xUpDerivative[10] - yUpDerivative[4] - yUpDerivative[11] + 4*zDown[0] + 4*zDown[1] - 6*zDown[2] - zDownDerivative[0] - zDownDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] - xUpDerivative[6] + xUpDerivative[12] - 4*yUp[1] - 6*yUp[3] + 4*yUp[4] + yUpDerivative[14] - 6*zDown[0] - 4*zDown[2] - 6*zDown[3] - 4*zDown[4] + zDownDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] - xUpDerivative[7] - xUpDerivative[11] - 4*yUp[0] - 4*yUp[2] - 4*yUp[3] - 6*yUp[4] - yUpDerivative[13] + 4*zDown[1] + 4*zDown[3] - 6*zDown[4] - zDownDerivative[5] - zDownDerivative[8])/2.;
    		};
    	if(boundaryCase == 32)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*xDown[0] + 2*xDown[2] - xDownDerivative[12] - xDownDerivative[14] - 4*xUp[0] - 2*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (16*qCurrent[1] - 4*xDown[1] + 2*xDown[4] + xDownDerivative[5] - 4*xUp[1] - 2*xUp[4] - xUpDerivative[5] - 4*yDown[1] + 2*yDown[2] + yDownDerivative[3] - 4*yUp[1] - 2*yUp[2] - yUpDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (14*qCurrent[2] - 2*xDown[0] - 4*xDown[2] - 2*xDown[3] + xDownDerivative[10] + xDownDerivative[11] + 2*xUp[0] - 4*xUp[2] + 2*xUp[3] - xUpDerivative[10] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] - 4*yDown[3] + 2*yDown[4] - 4*yUp[3] - 2*yUp[4] + yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] - 2*yDown[0] - 2*yDown[3] - 4*yDown[4] + 2*yUp[0] + 2*yUp[3] - 4*yUp[4] - yUpDerivative[13] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 33)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 6*qCurrent[3] - 6*xDown[0] + 4*xDown[2] - xDownDerivative[12] - xDownDerivative[14] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[4] - 6*xDown[1] + 4*xDown[4] + xDownDerivative[5] - 2*yDown[0] - 4*yDown[1] + 2*yDown[2] + yDownDerivative[3] + 2*yUp[0] - 4*yUp[1] - 2*yUp[2] - yUpDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (12*qCurrent[2] + 4*qCurrent[3] - 4*xDown[0] - 6*xDown[2] - 4*xDown[3] + xDownDerivative[10] + xDownDerivative[11] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] + 4*zUp[0] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[2] + 14*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - 2*yDown[1] - 4*yDown[3] + 2*yDown[4] + 2*yUp[1] - 4*yUp[3] - 2*yUp[4] + yUpDerivative[14] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[1] + 14*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - 2*yDown[0] - 2*yDown[2] - 2*yDown[3] - 4*yDown[4] + 2*yUp[0] + 2*yUp[2] + 2*yUp[3] - 4*yUp[4] - yUpDerivative[13] + 4*zUp[1] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 34)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 6*qCurrent[3] - xDownDerivative[14] - 6*xUp[0] - 4*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - yUpDerivative[1] + yUpDerivative[14] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[4] - 6*xUp[1] - 4*xUp[4] - xUpDerivative[5] + 2*yDown[0] - 4*yDown[1] + 2*yDown[2] + yDownDerivative[3] - 2*yUp[0] - 4*yUp[1] - 2*yUp[2] - yUpDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (12*qCurrent[2] - 4*qCurrent[3] + xDownDerivative[11] + 4*xUp[0] - 6*xUp[2] + 4*xUp[3] - xUpDerivative[10] + yDownDerivative[4] - yUpDerivative[4] - yUpDerivative[11] - 4*zUp[0] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[2] + 14*qCurrent[3] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] + 2*yDown[1] - 4*yDown[3] + 2*yDown[4] - 2*yUp[1] - 4*yUp[3] - 2*yUp[4] + yUpDerivative[14] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[1] + 14*qCurrent[4] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] - 2*yDown[0] + 2*yDown[2] - 2*yDown[3] - 4*yDown[4] + 2*yUp[0] - 2*yUp[2] + 2*yUp[3] - 4*yUp[4] - yUpDerivative[13] - 4*zUp[1] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 36)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] + 4*qCurrent[4] - 4*xDown[0] - 2*xDown[1] + 2*xDown[2] - xDownDerivative[12] - xDownDerivative[14] - 4*xUp[0] + 2*xUp[1] - 2*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - 6*zUp[0] - 6*zUp[3] - 4*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] - 4*qCurrent[2] - 4*xDown[1] - 2*xDown[3] + 2*xDown[4] + xDownDerivative[5] - 4*xUp[1] + 2*xUp[3] - 2*xUp[4] - xUpDerivative[5] - 6*yDown[1] + 4*yDown[2] + yDownDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (-4*qCurrent[1] + 14*qCurrent[2] - 2*xDown[0] - 4*xDown[2] - 2*xDown[3] - 2*xDown[4] + xDownDerivative[10] + xDownDerivative[11] + 2*xUp[0] - 4*xUp[2] + 2*xUp[3] + 2*xUp[4] - xUpDerivative[10] + yDownDerivative[4] + 4*zUp[1] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] - 6*yDown[3] + 4*yDown[4] - 6*zUp[0] - 6*zUp[3] - 4*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[0] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] - 4*yDown[0] - 4*yDown[3] - 6*yDown[4] + 4*zUp[3] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 37)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - 6*xDown[0] - 4*xDown[1] + 4*xDown[2] - xDownDerivative[12] - xDownDerivative[14] + yDownDerivative[1] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] - 4*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] - 6*xDown[1] - 4*xDown[3] + 4*xDown[4] + xDownDerivative[5] - 4*yDown[0] - 6*yDown[1] + 4*yDown[2] + yDownDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] - 4*xDown[0] - 6*xDown[2] - 4*xDown[3] - 4*xDown[4] + xDownDerivative[10] + xDownDerivative[11] + yDownDerivative[4] + 4*zUp[0] + 4*zUp[1] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xDownDerivative[14] - 4*yDown[1] - 6*yDown[3] + 4*yDown[4] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] - 4*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + xDownDerivative[13] - 4*yDown[0] - 4*yDown[2] - 4*yDown[3] - 6*yDown[4] + 4*zUp[1] + 4*zUp[3] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 38)
    		{
    		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] + 4*qCurrent[4] - xDownDerivative[14] - 6*xUp[0] + 4*xUp[1] - 4*xUp[2] + xUpDerivative[12] + yDownDerivative[1] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] - 4*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] - 6*xUp[1] + 4*xUp[3] - 4*xUp[4] - xUpDerivative[5] + 4*yDown[0] - 6*yDown[1] + 4*yDown[2] + yDownDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (-4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] + xDownDerivative[11] + 4*xUp[0] - 6*xUp[2] + 4*xUp[3] + 4*xUp[4] - xUpDerivative[10] + yDownDerivative[4] - 4*zUp[0] + 4*zUp[1] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] - xDownDerivative[14] - xUpDerivative[6] + xUpDerivative[12] + 4*yDown[1] - 6*yDown[3] + 4*yDown[4] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] - 4*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (4*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] + xDownDerivative[13] - xUpDerivative[7] - xUpDerivative[11] - 4*yDown[0] + 4*yDown[2] - 4*yDown[3] - 6*yDown[4] - 4*zUp[1] + 4*zUp[3] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 40)
    		{
    		spatialTerm[0] = (14*qCurrent[0] + 6*qCurrent[3] - 4*qCurrent[4] - 4*xDown[0] + 2*xDown[1] + 2*xDown[2] - xDownDerivative[12] - 4*xUp[0] - 2*xUp[1] - 2*xUp[2] + xUpDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] + 4*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (14*qCurrent[1] + 4*qCurrent[2] - 4*xDown[1] + 2*xDown[3] + 2*xDown[4] + xDownDerivative[5] - 4*xUp[1] - 2*xUp[3] - 2*xUp[4] - xUpDerivative[5] - 6*yUp[1] - 4*yUp[2] - yUpDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (4*qCurrent[1] + 14*qCurrent[2] - 2*xDown[0] - 4*xDown[2] - 2*xDown[3] + 2*xDown[4] + xDownDerivative[10] + 2*xUp[0] - 4*xUp[2] + 2*xUp[3] - 2*xUp[4] - xUpDerivative[10] - yUpDerivative[4] - yUpDerivative[11] - 4*zUp[1] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] - xUpDerivative[6] + xUpDerivative[12] - 6*yUp[3] - 4*yUp[4] + yUpDerivative[14] - 6*zUp[0] - 6*zUp[3] + 4*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[0] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] - xUpDerivative[7] - xUpDerivative[11] + 4*yUp[0] + 4*yUp[3] - 6*yUp[4] - yUpDerivative[13] - 4*zUp[3] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 41)
    		{
    		spatialTerm[0] = (12*qCurrent[0] - 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - 6*xDown[0] + 4*xDown[1] + 4*xDown[2] - xDownDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] + 4*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (-4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[3] - 4*qCurrent[4] - 6*xDown[1] + 4*xDown[3] + 4*xDown[4] + xDownDerivative[5] + 4*yUp[0] - 6*yUp[1] - 4*yUp[2] - yUpDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] + 4*qCurrent[3] - 4*qCurrent[4] - 4*xDown[0] - 6*xDown[2] - 4*xDown[3] + 4*xDown[4] + xDownDerivative[10] - yUpDerivative[4] - yUpDerivative[11] + 4*zUp[0] - 4*zUp[1] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] - 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[3] + xDownDerivative[6] - xDownDerivative[12] + 4*yUp[1] - 6*yUp[3] - 4*yUp[4] + yUpDerivative[14] - 6*zUp[0] - 4*zUp[2] - 6*zUp[3] + 4*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[0] - 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[4] + xDownDerivative[7] + xDownDerivative[11] + 4*yUp[0] + 4*yUp[2] + 4*yUp[3] - 6*yUp[4] - yUpDerivative[13] + 4*zUp[1] - 4*zUp[3] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
    	if(boundaryCase == 42)
    		{
    		spatialTerm[0] = (12*qCurrent[0] + 4*qCurrent[1] + 6*qCurrent[3] - 4*qCurrent[4] - 6*xUp[0] - 4*xUp[1] - 4*xUp[2] + xUpDerivative[12] - yUpDerivative[1] + yUpDerivative[14] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] + 4*zUp[4] - zUpDerivative[2])/2.;
    		spatialTerm[1] = (4*qCurrent[0] + 12*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[3] + 4*qCurrent[4] - 6*xUp[1] - 4*xUp[3] - 4*xUp[4] - xUpDerivative[5] - 4*yUp[0] - 6*yUp[1] - 4*yUp[2] - yUpDerivative[3] - zUpDerivative[4] - zUpDerivative[7])/2.;
    		spatialTerm[2] = (4*qCurrent[1] + 12*qCurrent[2] - 4*qCurrent[3] + 4*qCurrent[4] + 4*xUp[0] - 6*xUp[2] + 4*xUp[3] - 4*xUp[4] - xUpDerivative[10] - yUpDerivative[4] - yUpDerivative[11] - 4*zUp[0] - 4*zUp[1] - 6*zUp[2] + zUpDerivative[0] + zUpDerivative[3])/2.;
    		spatialTerm[3] = (6*qCurrent[0] + 4*qCurrent[1] - 4*qCurrent[2] + 12*qCurrent[3] - xUpDerivative[6] + xUpDerivative[12] - 4*yUp[1] - 6*yUp[3] - 4*yUp[4] + yUpDerivative[14] - 6*zUp[0] + 4*zUp[2] - 6*zUp[3] + 4*zUp[4] - zUpDerivative[9])/2.;
    		spatialTerm[4] = (-4*qCurrent[0] + 4*qCurrent[1] + 4*qCurrent[2] + 12*qCurrent[4] - xUpDerivative[7] - xUpDerivative[11] + 4*yUp[0] - 4*yUp[2] + 4*yUp[3] - 6*yUp[4] - yUpDerivative[13] - 4*zUp[1] - 4*zUp[3] - 6*zUp[4] + zUpDerivative[5] + zUpDerivative[8])/2.;
    		};
        spatialTerm = (0.5*L3)*spatialTerm;
        };

    HOSTDEVICE void bulkL4Force(const scalar L4, const dVec &qCurrent,
            const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
            const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
            const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
            const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
            dVec &spatialTerm)
        {
        spatialTerm[0] = (-xDown[1] + xUp[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] - zDown[4] + zUp[4])/2.;
        spatialTerm[1] = (xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        spatialTerm[2] = (2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        spatialTerm[3] = (-xDown[1] - xDown[2] + xUp[1] + xUp[2] + yDown[1] - yUp[1] + zDown[2] - zUp[2])/2.;
        spatialTerm[4] = (xDown[1] - xDown[2] - xUp[1] + xUp[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        spatialTerm = (0.5*L4)*spatialTerm;
        };

    HOSTDEVICE void boundaryL4Force(const scalar L4, int boundaryCase, const dVec &qCurrent,
        const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
        const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
        const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
        const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
        dVec &spatialTerm)
        {
            if(boundaryCase == 0)
        		{
        		spatialTerm[0] = (-xDown[1] + xUp[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] - zDown[4] + zUp[4])/2.;
        		spatialTerm[1] = (xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = (-xDown[1] - xDown[2] + xUp[1] + xUp[2] + yDown[1] - yUp[1] + zDown[2] - zUp[2])/2.;
        		spatialTerm[4] = (xDown[1] - xDown[2] - xUp[1] + xUp[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 1)
        		{
        		spatialTerm[0] = (2*qCurrent[1] - 2*xDown[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] - zDown[4] + zUp[4])/2.;
        		spatialTerm[1] = (2*qCurrent[0] + 2*qCurrent[3] - 2*qCurrent[4] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[3] + 2*qCurrent[4] + 3*xDown[1] + xDown[3] + xDown[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = (2*qCurrent[1] + 2*qCurrent[2] - 2*xDown[1] - 2*xDown[2] + yDown[1] - yUp[1] + zDown[2] - zUp[2])/2.;
        		spatialTerm[4] = (-2*qCurrent[1] + 2*qCurrent[2] + 2*xDown[1] - 2*xDown[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 2)
        		{
        		spatialTerm[0] = (-2*qCurrent[1] + 2*xUp[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] - zDown[4] + zUp[4])/2.;
        		spatialTerm[1] = (-2*qCurrent[0] - 2*qCurrent[3] + 2*qCurrent[4] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[3] - 2*qCurrent[4] - 3*xUp[1] - xUp[3] - xUp[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = (-2*qCurrent[1] - 2*qCurrent[2] + 2*xUp[1] + 2*xUp[2] + yDown[1] - yUp[1] + zDown[2] - zUp[2])/2.;
        		spatialTerm[4] = (2*qCurrent[1] - 2*qCurrent[2] - 2*xUp[1] + 2*xUp[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 4)
        		{
        		spatialTerm[0] = (-2*qCurrent[1] - 2*qCurrent[4] - xDown[1] + xUp[1] + 2*yDown[1] + 2*yDown[4] - zDown[4] + zUp[4])/2.;
        		spatialTerm[1] = (-2*qCurrent[0] + 2*qCurrent[2] - 2*qCurrent[3] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[1] - 2*qCurrent[4] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] - 2*yDown[1] + 2*yDown[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = (-2*qCurrent[1] - xDown[1] - xDown[2] + xUp[1] + xUp[2] + 2*yDown[1] + zDown[2] - zUp[2])/2.;
        		spatialTerm[4] = (-2*qCurrent[0] - 2*qCurrent[2] + xDown[1] - xDown[2] - xUp[1] + xUp[2] - yDown[0] - 3*yDown[1] - yDown[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 5)
        		{
        		spatialTerm[0] = -qCurrent[4] - xDown[1] + yDown[1] + yDown[4] - zDown[4]/2. + zUp[4]/2.;
        		spatialTerm[1] = (2*qCurrent[2] - 2*qCurrent[4] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[1] + 2*qCurrent[3] + 3*xDown[1] + xDown[3] + xDown[4] - 2*yDown[1] + 2*yDown[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = qCurrent[2] - xDown[1] - xDown[2] + yDown[1] + zDown[2]/2. - zUp[2]/2.;
        		spatialTerm[4] = (-2*qCurrent[0] - 2*qCurrent[1] + 2*xDown[1] - 2*xDown[2] - yDown[0] - 3*yDown[1] - yDown[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 6)
        		{
        		spatialTerm[0] = -2*qCurrent[1] - qCurrent[4] + xUp[1] + yDown[1] + yDown[4] - zDown[4]/2. + zUp[4]/2.;
        		spatialTerm[1] = (-4*qCurrent[0] + 2*qCurrent[2] - 4*qCurrent[3] + 2*qCurrent[4] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[1] - 2*qCurrent[3] - 4*qCurrent[4] - 3*xUp[1] - xUp[3] - xUp[4] - 2*yDown[1] + 2*yDown[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = -2*qCurrent[1] - qCurrent[2] + xUp[1] + xUp[2] + yDown[1] + zDown[2]/2. - zUp[2]/2.;
        		spatialTerm[4] = (-2*qCurrent[0] + 2*qCurrent[1] - 4*qCurrent[2] - 2*xUp[1] + 2*xUp[2] - yDown[0] - 3*yDown[1] - yDown[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 8)
        		{
        		spatialTerm[0] = (2*qCurrent[1] + 2*qCurrent[4] - xDown[1] + xUp[1] - 2*yUp[1] - 2*yUp[4] - zDown[4] + zUp[4])/2.;
        		spatialTerm[1] = (2*qCurrent[0] - 2*qCurrent[2] + 2*qCurrent[3] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[1] + 2*qCurrent[4] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] + 2*yUp[1] - 2*yUp[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = (2*qCurrent[1] - xDown[1] - xDown[2] + xUp[1] + xUp[2] - 2*yUp[1] + zDown[2] - zUp[2])/2.;
        		spatialTerm[4] = (2*qCurrent[0] + 2*qCurrent[2] + xDown[1] - xDown[2] - xUp[1] + xUp[2] + yUp[0] + 3*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 9)
        		{
        		spatialTerm[0] = (4*qCurrent[1] + 2*qCurrent[4] - 2*xDown[1] - 2*yUp[1] - 2*yUp[4] - zDown[4] + zUp[4])/2.;
        		spatialTerm[1] = (4*qCurrent[0] - 2*qCurrent[2] + 4*qCurrent[3] - 2*qCurrent[4] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[1] + 2*qCurrent[3] + 4*qCurrent[4] + 3*xDown[1] + xDown[3] + xDown[4] + 2*yUp[1] - 2*yUp[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = (4*qCurrent[1] + 2*qCurrent[2] - 2*xDown[1] - 2*xDown[2] - 2*yUp[1] + zDown[2] - zUp[2])/2.;
        		spatialTerm[4] = (2*qCurrent[0] - 2*qCurrent[1] + 4*qCurrent[2] + 2*xDown[1] - 2*xDown[2] + yUp[0] + 3*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 10)
        		{
        		spatialTerm[0] = qCurrent[4] + xUp[1] - yUp[1] - yUp[4] - zDown[4]/2. + zUp[4]/2.;
        		spatialTerm[1] = (-2*qCurrent[2] + 2*qCurrent[4] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] + zDown[2] - zDown[4] - zUp[2] + zUp[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[1] - 2*qCurrent[3] - 3*xUp[1] - xUp[3] - xUp[4] + 2*yUp[1] - 2*yUp[4] - zDown[1] - zDown[3] - 2*zDown[4] + zUp[1] + zUp[3] + 2*zUp[4])/2.;
        		spatialTerm[3] = -qCurrent[2] + xUp[1] + xUp[2] - yUp[1] + zDown[2]/2. - zUp[2]/2.;
        		spatialTerm[4] = (2*qCurrent[0] + 2*qCurrent[1] - 2*xUp[1] + 2*xUp[2] + yUp[0] + 3*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 2*zDown[2] - zUp[0] - zUp[1] - 2*zUp[2])/2.;
        		};
        	if(boundaryCase == 16)
        		{
        		spatialTerm[0] = (2*qCurrent[4] - xDown[1] + xUp[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] - 2*zDown[4])/2.;
        		spatialTerm[1] = (-2*qCurrent[2] + 2*qCurrent[4] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[1] - 2*qCurrent[3] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = (-2*qCurrent[2] - xDown[1] - xDown[2] + xUp[1] + xUp[2] + yDown[1] - yUp[1] + 2*zDown[2])/2.;
        		spatialTerm[4] = (2*qCurrent[0] + 2*qCurrent[1] + xDown[1] - xDown[2] - xUp[1] + xUp[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 17)
        		{
        		spatialTerm[0] = (2*qCurrent[1] + 2*qCurrent[4] - 2*xDown[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] - 2*zDown[4])/2.;
        		spatialTerm[1] = (2*qCurrent[0] - 2*qCurrent[2] + 2*qCurrent[3] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[1] + 2*qCurrent[4] + 3*xDown[1] + xDown[3] + xDown[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = qCurrent[1] - xDown[1] - xDown[2] + yDown[1]/2. - yUp[1]/2. + zDown[2];
        		spatialTerm[4] = (2*qCurrent[0] + 2*qCurrent[2] + 2*xDown[1] - 2*xDown[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 18)
        		{
        		spatialTerm[0] = (-2*qCurrent[1] + 2*qCurrent[4] + 2*xUp[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] - 2*zDown[4])/2.;
        		spatialTerm[1] = (-2*qCurrent[0] - 2*qCurrent[2] - 2*qCurrent[3] + 4*qCurrent[4] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[1] - 4*qCurrent[3] - 2*qCurrent[4] - 3*xUp[1] - xUp[3] - xUp[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = -qCurrent[1] - 2*qCurrent[2] + xUp[1] + xUp[2] + yDown[1]/2. - yUp[1]/2. + zDown[2];
        		spatialTerm[4] = (2*qCurrent[0] + 4*qCurrent[1] - 2*qCurrent[2] - 2*xUp[1] + 2*xUp[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 20)
        		{
        		spatialTerm[0] = -qCurrent[1] - xDown[1]/2. + xUp[1]/2. + yDown[1] + yDown[4] - zDown[4];
        		spatialTerm[1] = (-2*qCurrent[0] - 2*qCurrent[3] + 2*qCurrent[4] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-2*qCurrent[3] - 2*qCurrent[4] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] - 2*yDown[1] + 2*yDown[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = (-2*qCurrent[1] - 2*qCurrent[2] - xDown[1] - xDown[2] + xUp[1] + xUp[2] + 2*yDown[1] + 2*zDown[2])/2.;
        		spatialTerm[4] = (2*qCurrent[1] - 2*qCurrent[2] + xDown[1] - xDown[2] - xUp[1] + xUp[2] - yDown[0] - 3*yDown[1] - yDown[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 21)
        		{
        		spatialTerm[0] = -xDown[1] + yDown[1] + yDown[4] - zDown[4];
        		spatialTerm[1] = (xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (3*xDown[1] + xDown[3] + xDown[4] - 2*yDown[1] + 2*yDown[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = -xDown[1] - xDown[2] + yDown[1] + zDown[2];
        		spatialTerm[4] = (2*xDown[1] - 2*xDown[2] - yDown[0] - 3*yDown[1] - yDown[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 22)
        		{
        		spatialTerm[0] = -2*qCurrent[1] + xUp[1] + yDown[1] + yDown[4] - zDown[4];
        		spatialTerm[1] = (-4*qCurrent[0] - 4*qCurrent[3] + 4*qCurrent[4] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[3] - 4*qCurrent[4] - 3*xUp[1] - xUp[3] - xUp[4] - 2*yDown[1] + 2*yDown[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = -2*qCurrent[1] - 2*qCurrent[2] + xUp[1] + xUp[2] + yDown[1] + zDown[2];
        		spatialTerm[4] = (4*qCurrent[1] - 4*qCurrent[2] - 2*xUp[1] + 2*xUp[2] - yDown[0] - 3*yDown[1] - yDown[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 24)
        		{
        		spatialTerm[0] = (2*qCurrent[1] + 4*qCurrent[4] - xDown[1] + xUp[1] - 2*yUp[1] - 2*yUp[4] - 2*zDown[4])/2.;
        		spatialTerm[1] = (2*qCurrent[0] - 4*qCurrent[2] + 2*qCurrent[3] + 2*qCurrent[4] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] - 2*qCurrent[3] + 2*qCurrent[4] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] + 2*yUp[1] - 2*yUp[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = (2*qCurrent[1] - 2*qCurrent[2] - xDown[1] - xDown[2] + xUp[1] + xUp[2] - 2*yUp[1] + 2*zDown[2])/2.;
        		spatialTerm[4] = (4*qCurrent[0] + 2*qCurrent[1] + 2*qCurrent[2] + xDown[1] - xDown[2] - xUp[1] + xUp[2] + yUp[0] + 3*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 25)
        		{
        		spatialTerm[0] = 2*qCurrent[1] + 2*qCurrent[4] - xDown[1] - yUp[1] - yUp[4] - zDown[4];
        		spatialTerm[1] = (4*qCurrent[0] - 4*qCurrent[2] + 4*qCurrent[3] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] + 4*qCurrent[4] + 3*xDown[1] + xDown[3] + xDown[4] + 2*yUp[1] - 2*yUp[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = 2*qCurrent[1] - xDown[1] - xDown[2] - yUp[1] + zDown[2];
        		spatialTerm[4] = (4*qCurrent[0] + 4*qCurrent[2] + 2*xDown[1] - 2*xDown[2] + yUp[0] + 3*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 26)
        		{
        		spatialTerm[0] = 2*qCurrent[4] + xUp[1] - yUp[1] - yUp[4] - zDown[4];
        		spatialTerm[1] = (-4*qCurrent[2] + 4*qCurrent[4] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] + 2*zDown[2] - 2*zDown[4])/2.;
        		spatialTerm[2] = (-4*qCurrent[1] - 4*qCurrent[3] - 3*xUp[1] - xUp[3] - xUp[4] + 2*yUp[1] - 2*yUp[4] - zDown[1] - zDown[3] - 3*zDown[4])/2.;
        		spatialTerm[3] = -2*qCurrent[2] + xUp[1] + xUp[2] - yUp[1] + zDown[2];
        		spatialTerm[4] = (4*qCurrent[0] + 4*qCurrent[1] - 2*xUp[1] + 2*xUp[2] + yUp[0] + 3*yUp[1] + yUp[2] + zDown[0] + zDown[1] + 3*zDown[2])/2.;
        		};
        	if(boundaryCase == 32)
        		{
        		spatialTerm[0] = (-2*qCurrent[4] - xDown[1] + xUp[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] + 2*zUp[4])/2.;
        		spatialTerm[1] = (2*qCurrent[2] - 2*qCurrent[4] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[1] + 2*qCurrent[3] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = (2*qCurrent[2] - xDown[1] - xDown[2] + xUp[1] + xUp[2] + yDown[1] - yUp[1] - 2*zUp[2])/2.;
        		spatialTerm[4] = (-2*qCurrent[0] - 2*qCurrent[1] + xDown[1] - xDown[2] - xUp[1] + xUp[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 33)
        		{
        		spatialTerm[0] = (2*qCurrent[1] - 2*qCurrent[4] - 2*xDown[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] + 2*zUp[4])/2.;
        		spatialTerm[1] = (2*qCurrent[0] + 2*qCurrent[2] + 2*qCurrent[3] - 4*qCurrent[4] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[1] + 4*qCurrent[3] + 2*qCurrent[4] + 3*xDown[1] + xDown[3] + xDown[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = (2*qCurrent[1] + 4*qCurrent[2] - 2*xDown[1] - 2*xDown[2] + yDown[1] - yUp[1] - 2*zUp[2])/2.;
        		spatialTerm[4] = (-2*qCurrent[0] - 4*qCurrent[1] + 2*qCurrent[2] + 2*xDown[1] - 2*xDown[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 34)
        		{
        		spatialTerm[0] = (-2*qCurrent[1] - 2*qCurrent[4] + 2*xUp[1] + yDown[1] + yDown[4] - yUp[1] - yUp[4] + 2*zUp[4])/2.;
        		spatialTerm[1] = (-2*qCurrent[0] + 2*qCurrent[2] - 2*qCurrent[3] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 2*yDown[4] + yUp[0] - yUp[2] + yUp[3] - 2*yUp[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[1] - 2*qCurrent[4] - 3*xUp[1] - xUp[3] - xUp[4] - yDown[1] + yDown[4] + yUp[1] - yUp[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = -qCurrent[1] + xUp[1] + xUp[2] + yDown[1]/2. - yUp[1]/2. - zUp[2];
        		spatialTerm[4] = (-2*qCurrent[0] - 2*qCurrent[2] - 2*xUp[1] + 2*xUp[2] - yDown[0] - 2*yDown[1] - yDown[2] + yUp[0] + 2*yUp[1] + yUp[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 36)
        		{
        		spatialTerm[0] = -qCurrent[1] - 2*qCurrent[4] - xDown[1]/2. + xUp[1]/2. + yDown[1] + yDown[4] + zUp[4];
        		spatialTerm[1] = (-2*qCurrent[0] + 4*qCurrent[2] - 2*qCurrent[3] - 2*qCurrent[4] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 2*qCurrent[3] - 2*qCurrent[4] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] - 2*yDown[1] + 2*yDown[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = (-2*qCurrent[1] + 2*qCurrent[2] - xDown[1] - xDown[2] + xUp[1] + xUp[2] + 2*yDown[1] - 2*zUp[2])/2.;
        		spatialTerm[4] = (-4*qCurrent[0] - 2*qCurrent[1] - 2*qCurrent[2] + xDown[1] - xDown[2] - xUp[1] + xUp[2] - yDown[0] - 3*yDown[1] - yDown[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 37)
        		{
        		spatialTerm[0] = -2*qCurrent[4] - xDown[1] + yDown[1] + yDown[4] + zUp[4];
        		spatialTerm[1] = (4*qCurrent[2] - 4*qCurrent[4] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (4*qCurrent[1] + 4*qCurrent[3] + 3*xDown[1] + xDown[3] + xDown[4] - 2*yDown[1] + 2*yDown[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = 2*qCurrent[2] - xDown[1] - xDown[2] + yDown[1] - zUp[2];
        		spatialTerm[4] = (-4*qCurrent[0] - 4*qCurrent[1] + 2*xDown[1] - 2*xDown[2] - yDown[0] - 3*yDown[1] - yDown[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 38)
        		{
        		spatialTerm[0] = -2*qCurrent[1] - 2*qCurrent[4] + xUp[1] + yDown[1] + yDown[4] + zUp[4];
        		spatialTerm[1] = (-4*qCurrent[0] + 4*qCurrent[2] - 4*qCurrent[3] - xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] - yDown[0] + yDown[2] - yDown[3] + 3*yDown[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (4*qCurrent[1] - 4*qCurrent[4] - 3*xUp[1] - xUp[3] - xUp[4] - 2*yDown[1] + 2*yDown[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = -2*qCurrent[1] + xUp[1] + xUp[2] + yDown[1] - zUp[2];
        		spatialTerm[4] = (-4*qCurrent[0] - 4*qCurrent[2] - 2*xUp[1] + 2*xUp[2] - yDown[0] - 3*yDown[1] - yDown[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 40)
        		{
        		spatialTerm[0] = qCurrent[1] - xDown[1]/2. + xUp[1]/2. - yUp[1] - yUp[4] + zUp[4];
        		spatialTerm[1] = (2*qCurrent[0] + 2*qCurrent[3] - 2*qCurrent[4] + xDown[0] - 2*xDown[2] + xDown[3] - xDown[4] - xUp[0] + 2*xUp[2] - xUp[3] + xUp[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (2*qCurrent[3] + 2*qCurrent[4] + 2*xDown[1] + xDown[3] + xDown[4] - 2*xUp[1] - xUp[3] - xUp[4] + 2*yUp[1] - 2*yUp[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = (2*qCurrent[1] + 2*qCurrent[2] - xDown[1] - xDown[2] + xUp[1] + xUp[2] - 2*yUp[1] - 2*zUp[2])/2.;
        		spatialTerm[4] = (-2*qCurrent[1] + 2*qCurrent[2] + xDown[1] - xDown[2] - xUp[1] + xUp[2] + yUp[0] + 3*yUp[1] + yUp[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 41)
        		{
        		spatialTerm[0] = 2*qCurrent[1] - xDown[1] - yUp[1] - yUp[4] + zUp[4];
        		spatialTerm[1] = (4*qCurrent[0] + 4*qCurrent[3] - 4*qCurrent[4] + xDown[0] - 3*xDown[2] + xDown[3] - xDown[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (4*qCurrent[3] + 4*qCurrent[4] + 3*xDown[1] + xDown[3] + xDown[4] + 2*yUp[1] - 2*yUp[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = 2*qCurrent[1] + 2*qCurrent[2] - xDown[1] - xDown[2] - yUp[1] - zUp[2];
        		spatialTerm[4] = (-4*qCurrent[1] + 4*qCurrent[2] + 2*xDown[1] - 2*xDown[2] + yUp[0] + 3*yUp[1] + yUp[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};
        	if(boundaryCase == 42)
        		{
        		spatialTerm[0] = xUp[1] - yUp[1] - yUp[4] + zUp[4];
        		spatialTerm[1] = (-xUp[0] + 3*xUp[2] - xUp[3] + xUp[4] + yUp[0] - yUp[2] + yUp[3] - 3*yUp[4] - 2*zUp[2] + 2*zUp[4])/2.;
        		spatialTerm[2] = (-3*xUp[1] - xUp[3] - xUp[4] + 2*yUp[1] - 2*yUp[4] + zUp[1] + zUp[3] + 3*zUp[4])/2.;
        		spatialTerm[3] = xUp[1] + xUp[2] - yUp[1] - zUp[2];
        		spatialTerm[4] = (-2*xUp[1] + 2*xUp[2] + yUp[0] + 3*yUp[1] + yUp[2] - zUp[0] - zUp[1] - 3*zUp[2])/2.;
        		};

        spatialTerm = (0.5*L4)*spatialTerm;
        };

    HOSTDEVICE void bulkL6Force(const scalar L6, const dVec &qCurrent,
            const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
            const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
            const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
            const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
            dVec &spatialTerm)
        {
        spatialTerm[0] = (-2*(xDown[0]*xDown[0]) + 2*(xDown[1]*xDown[1]) + 2*(xDown[2]*xDown[2]) + 2*(xDown[3]*xDown[3]) + 2*(xDown[4]*xDown[4]) - 2*(xUp[0]*xUp[0]) + 2*(xUp[1]*xUp[1]) + 2*(xUp[2]*xUp[2]) + 2*(xUp[3]*xUp[3]) + 2*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 4*qCurrent[1]*xDown[1] - 4*qCurrent[2]*xDown[2] - 4*qCurrent[3]*xDown[3] - 4*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 4*qCurrent[1]*xUp[1] - 4*qCurrent[2]*xUp[2] - 4*qCurrent[3]*xUp[3] - 4*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 4*qCurrent[3]*yDown[0] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 4*qCurrent[3]*yUp[0] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 4*qCurrent[3]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[3]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] - 4*qCurrent[0]*(xDown[0] + xDown[3] + xUp[0] + xUp[3] - yDown[3] - yUp[3] - zDown[0] - zUp[0]) + 4*qCurrent[3]*zUp[0] + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] + 4*qCurrent[3]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*xUp[0] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*qCurrent[1]*yDown[3] + xDownDerivative[11]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) + 2*qCurrent[1]*yUp[3] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*zUp[0]*zUp[1] + (xDown[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 4*qCurrent[3]*yDown[1] + 2*xDown[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*xDown[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*xDown[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] + 2*xDown[4]*yDown[4] - 2*xUp[4]*yDown[4] - xDown[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] - 2*xDown[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*xDown[2]*yUp[2] + 2*xUp[2]*yUp[2] + xDown[0]*(2*yDown[0] + yDown[3] - 2*yUp[0] - yUp[3]) - 2*xDown[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*xDown[4]*yUp[4] + 2*xUp[4]*yUp[4] + 4*qCurrent[3]*zDown[1] - 4*qCurrent[0]*(xDown[1] + xUp[1] - zDown[1] - zUp[1]) + 4*qCurrent[3]*zUp[1])/2. + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zDown[0]*zDown[2] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] + 2*zUp[0]*zUp[2] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + (2*xDown[0]*zDown[0] + xDown[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] + 2*xDown[1]*zDown[1] - 2*xUp[1]*zDown[1] + 2*xDown[2]*zDown[2] - 2*xUp[2]*zDown[2] + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*xDown[4]*zDown[4] - 2*xUp[4]*zDown[4] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*xUp[1]*zUp[1] - 4*qCurrent[0]*(xDown[2] + xUp[2] - zDown[2] - zUp[2]) - 4*qCurrent[3]*(yDown[2] + yUp[2] - zDown[2] - zUp[2]) - 2*xDown[2]*zUp[2] + 2*xUp[2]*zUp[2] - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] - 2*xDown[4]*zUp[4] + 2*xUp[4]*zUp[4])/2. - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 2*(yDown[0]*yDown[0]) + 2*(yDown[1]*yDown[1]) + 2*(yDown[2]*yDown[2]) - 2*(yDown[3]*yDown[3]) + 2*(yDown[4]*yDown[4]) + 2*(yUp[0]*yUp[0]) + 2*(yUp[1]*yUp[1]) + 2*(yUp[2]*yUp[2]) - 2*(yUp[3]*yUp[3]) + 2*(yUp[4]*yUp[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) - 4*qCurrent[0]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 4*qCurrent[0]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 4*qCurrent[0]*yDown[0] - 4*qCurrent[1]*yDown[1] - 4*qCurrent[2]*yDown[2] - 4*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] - 4*qCurrent[0]*yUp[0] - 4*qCurrent[1]*yUp[1] - 4*qCurrent[2]*yUp[2] - 4*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 4*qCurrent[0]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[0]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] + 4*qCurrent[0]*zUp[0] + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] + 4*qCurrent[0]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[3]*(xDown[0] + xUp[0] - yDown[0] - yDown[3] - yUp[0] - yUp[3] + zDown[3] + zUp[3]) + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yDown[3] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] + 2*qCurrent[4]*yUp[3] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[4]*(zDown[0] + zDown[3]) + 2*zDown[0]*zDown[4] +   2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] - 2*qCurrent[4]*(zUp[0] + zUp[3]) + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + (2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*yDown[2]*zDown[2] - 2*yUp[2]*zDown[2] + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 2*yDown[4]*zDown[4] - 2*yUp[4]*zDown[4] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*yDown[2]*zUp[2] + 2*yUp[2]*zUp[2] - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] - 4*qCurrent[0]*(xDown[4] + xUp[4] - zDown[4] - zUp[4]) - 4*qCurrent[3]*(yDown[4] + yUp[4] - zDown[4] - zUp[4]) - 2*yDown[4]*zUp[4] + 2*yUp[4]*zUp[4])/2. - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];

        spatialTerm = (0.5*L6)*spatialTerm;
        };

    HOSTDEVICE void boundaryL6Force(const scalar L6, int boundaryCase, const dVec &qCurrent,
        const dVec &xDown, const dVec &xUp, const dVec &yDown, const dVec &yUp, const dVec &zDown, const dVec &zUp,
        const cubicLatticeDerivativeVector &xDownDerivative, const cubicLatticeDerivativeVector &xUpDerivative,
        const cubicLatticeDerivativeVector &yDownDerivative, const cubicLatticeDerivativeVector &yUpDerivative,
        const cubicLatticeDerivativeVector &zDownDerivative, const cubicLatticeDerivativeVector &zUpDerivative,
        dVec &spatialTerm)
        {
            if(boundaryCase == 0)
        		{
        		spatialTerm[0] = (-2*(xDown[0]*xDown[0]) + 2*(xDown[1]*xDown[1]) + 2*(xDown[2]*xDown[2]) + 2*(xDown[3]*xDown[3]) + 2*(xDown[4]*xDown[4]) - 2*(xUp[0]*xUp[0]) + 2*(xUp[1]*xUp[1]) + 2*(xUp[2]*xUp[2]) + 2*(xUp[3]*xUp[3]) + 2*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 4*qCurrent[1]*xDown[1] - 4*qCurrent[2]*xDown[2] - 4*qCurrent[3]*xDown[3] - 4*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 4*qCurrent[1]*xUp[1] - 4*qCurrent[2]*xUp[2] - 4*qCurrent[3]*xUp[3] - 4*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 4*qCurrent[3]*yDown[0] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 4*qCurrent[3]*yUp[0] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 4*qCurrent[3]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[3]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] - 4*qCurrent[0]*(xDown[0] + xDown[3] + xUp[0] + xUp[3] - yDown[3] - yUp[3] - zDown[0] - zUp[0]) + 4*qCurrent[3]*zUp[0] + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] + 4*qCurrent[3]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*xUp[0] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*qCurrent[1]*yDown[3] + xDownDerivative[11]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) + 2*qCurrent[1]*yUp[3] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*zUp[0]*zUp[1] + (xDown[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 4*qCurrent[3]*yDown[1] + 2*xDown[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*xDown[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*xDown[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] + 2*xDown[4]*yDown[4] - 2*xUp[4]*yDown[4] - xDown[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] - 2*xDown[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*xDown[2]*yUp[2] + 2*xUp[2]*yUp[2] + xDown[0]*(2*yDown[0] + yDown[3] - 2*yUp[0] - yUp[3]) - 2*xDown[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*xDown[4]*yUp[4] + 2*xUp[4]*yUp[4] + 4*qCurrent[3]*zDown[1] - 4*qCurrent[0]*(xDown[1] + xUp[1] - zDown[1] - zUp[1]) + 4*qCurrent[3]*zUp[1])/2. + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zDown[0]*zDown[2] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] + 2*zUp[0]*zUp[2] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + (2*xDown[0]*zDown[0] + xDown[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] + 2*xDown[1]*zDown[1] - 2*xUp[1]*zDown[1] + 2*xDown[2]*zDown[2] - 2*xUp[2]*zDown[2] + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*xDown[4]*zDown[4] - 2*xUp[4]*zDown[4] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*xUp[1]*zUp[1] - 4*qCurrent[0]*(xDown[2] + xUp[2] - zDown[2] - zUp[2]) - 4*qCurrent[3]*(yDown[2] + yUp[2] - zDown[2] - zUp[2]) - 2*xDown[2]*zUp[2] + 2*xUp[2]*zUp[2] - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] - 2*xDown[4]*zUp[4] + 2*xUp[4]*zUp[4])/2. - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 2*(yDown[0]*yDown[0]) + 2*(yDown[1]*yDown[1]) + 2*(yDown[2]*yDown[2]) - 2*(yDown[3]*yDown[3]) + 2*(yDown[4]*yDown[4]) + 2*(yUp[0]*yUp[0]) + 2*(yUp[1]*yUp[1]) + 2*(yUp[2]*yUp[2]) - 2*(yUp[3]*yUp[3]) + 2*(yUp[4]*yUp[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) - 4*qCurrent[0]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 4*qCurrent[0]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 4*qCurrent[0]*yDown[0] - 4*qCurrent[1]*yDown[1] - 4*qCurrent[2]*yDown[2] - 4*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] - 4*qCurrent[0]*yUp[0] - 4*qCurrent[1]*yUp[1] - 4*qCurrent[2]*yUp[2] - 4*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 4*qCurrent[0]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[0]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] + 4*qCurrent[0]*zUp[0] + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] + 4*qCurrent[0]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[3]*(xDown[0] + xUp[0] - yDown[0] - yDown[3] - yUp[0] - yUp[3] + zDown[3] + zUp[3]) + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yDown[3] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] + 2*qCurrent[4]*yUp[3] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[4]*(zDown[0] + zDown[3]) + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] - 2*qCurrent[4]*(zUp[0] + zUp[3]) + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + (2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*yDown[2]*zDown[2] - 2*yUp[2]*zDown[2] + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 2*yDown[4]*zDown[4] - 2*yUp[4]*zDown[4] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*yDown[2]*zUp[2] + 2*yUp[2]*zUp[2] - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] - 4*qCurrent[0]*(xDown[4] + xUp[4] - zDown[4] - zUp[4]) - 4*qCurrent[3]*(yDown[4] + yUp[4] - zDown[4] - zUp[4]) - 2*yDown[4]*zUp[4] + 2*yUp[4]*zUp[4])/2. - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 1)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 8*qCurrent[1]*xDown[1] - 8*qCurrent[2]*xDown[2] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 4*qCurrent[1]*yDown[0] - 2*qCurrent[1]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] + 4*qCurrent[1]*yUp[0] + 2*qCurrent[1]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] - 4*qCurrent[2]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] - 2*qCurrent[2]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] + 4*qCurrent[2]*zUp[0] + 4*qCurrent[0]*(-3*xDown[0] - 2*xDown[3] + yDown[3] + yUp[3] + zDown[0] + zUp[0]) + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] - 2*qCurrent[3]*(xDown[0] + 2*(2*xDown[3] + yDown[0] + yUp[0] - zDown[0] - zDown[3] - zUp[0] - zUp[3])) + 2*qCurrent[2]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 4*qCurrent[1]*yDown[1] + 2*xDown[1]*yDown[1] - 2*qCurrent[2]*yDown[2] + 2*xDown[2]*yDown[2] + 2*qCurrent[1]*yDown[3] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - 2*qCurrent[4]*yDown[4] + 2*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] + 4*qCurrent[1]*yUp[1] - 2*xDown[1]*yUp[1] + 2*qCurrent[2]*yUp[2] - 2*xDown[2]*yUp[2] + 2*qCurrent[1]*yUp[3] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + 2*qCurrent[4]*yUp[4] - 2*xDown[4]*yUp[4] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] - 2*qCurrent[2]*zDown[1] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*qCurrent[2]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[0]*(-4*xDown[1] - 2*yDown[0] - yDown[3] + 2*yUp[0] + yUp[3] + 2*zDown[1] + 2*zUp[1]) + qCurrent[3]*(-yDown[0] - 2*yDown[1] - 2*yDown[3] + yUp[0] - 2*yUp[1] + 2*yUp[3] + 2*zDown[1] + 2*zUp[1]) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = -2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - 2*qCurrent[1]*yDown[2] - 2*qCurrent[3]*yDown[2] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*qCurrent[1]*yUp[2] - 2*qCurrent[3]*yUp[2] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] - qCurrent[3]*zDown[0] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] - 2*qCurrent[1]*zDown[1] + 2*xDown[1]*zDown[1] + 2*qCurrent[3]*zDown[2] + 2*xDown[2]*zDown[2] + 2*zDown[0]*zDown[2] - 2*qCurrent[3]*zDown[3] + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] + 2*zDown[2]*zDown[3] - 2*qCurrent[4]*zDown[4] + 2*xDown[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] + qCurrent[3]*zUp[0] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] + 2*qCurrent[1]*zUp[1] - 2*xDown[1]*zUp[1] + 2*qCurrent[3]*zUp[2] - 2*xDown[2]*zUp[2] + 2*zUp[0]*zUp[2] + 2*qCurrent[2]*(xDown[0] + yDown[3] + yUp[3] - zDown[0] - 2*zDown[2] - zDown[3] - zUp[0] + 2*zUp[2] - zUp[3]) + 2*qCurrent[3]*zUp[3] - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + 2*zUp[2]*zUp[3] + qCurrent[0]*(-4*xDown[2] - 2*zDown[0] + 2*zDown[2] - zDown[3] + 2*zUp[0] + 2*zUp[2] + zUp[3]) + 2*qCurrent[4]*zUp[4] - 2*xDown[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = -(xDown[0]*xDown[0]) + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zDown[1]*zDown[1] - zDown[2]*zDown[2] + zDown[3]*zDown[3] - zDown[4]*zDown[4] - zUp[1]*zUp[1] - zUp[2]*zUp[2] + zUp[3]*zUp[3] - zUp[4]*zUp[4] - 2*xDown[0]*xDown[3] + (xDown[1]*xDownDerivative[5])/2. + xDown[1]*xDownDerivative[8] + (xDown[2]*xDownDerivative[10])/2. + xDown[2]*xDownDerivative[13] - qCurrent[1]*yDown[0] - 2*qCurrent[1]*yDown[1] - 2*qCurrent[2]*yDown[2] - 2*qCurrent[1]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] + qCurrent[1]*yUp[0] - 2*qCurrent[1]*yUp[1] - 2*qCurrent[2]*yUp[2] + 2*qCurrent[1]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] - qCurrent[2]*zDown[0] + 2*qCurrent[1]*zDown[1] + 2*qCurrent[2]*zDown[2] - 2*qCurrent[2]*zDown[3] + 2*zDown[0]*zDown[3] + 2*qCurrent[4]*zDown[4] + (zDown[2]*zDownDerivative[0])/2. + zDown[2]*zDownDerivative[3] + (zDown[4]*zDownDerivative[5])/2. + zDown[4]*zDownDerivative[8] + qCurrent[2]*zUp[0] + 2*qCurrent[1]*zUp[1] + 2*qCurrent[2]*zUp[2] - qCurrent[0]*(xDown[0] + 2*(2*xDown[3] + yDown[0] + yUp[0] - zDown[0] - zDown[3] - zUp[0] - zUp[3])) + 2*qCurrent[2]*zUp[3] + 2*zUp[0]*zUp[3] + 2*qCurrent[3]*(xDown[0] - yDown[0] - yDown[3] - yUp[0] - yUp[3] + zDown[3] + zUp[3]) + 2*qCurrent[4]*zUp[4] - (zUp[2]*zUpDerivative[0])/2. - zUp[2]*zUpDerivative[3] - (zUp[4]*zUpDerivative[5])/2. - zUp[4]*zUpDerivative[8];
        		spatialTerm[4] = (-4*xDown[0]*xDown[4] + 2*xDown[1]*xDownDerivative[9] + 2*xDown[2]*xDownDerivative[14] - 4*qCurrent[1]*yDown[4] - 4*qCurrent[3]*yDown[4] + 2*xDownDerivative[14]*yDown[4] - 4*yDown[3]*yDown[4] + 2*yDown[1]*yDownDerivative[4] + 4*qCurrent[1]*yUp[4] - 4*qCurrent[3]*yUp[4] - 4*yUp[3]*yUp[4] - 2*yUp[1]*yUpDerivative[4] - 2*yUp[4]*yUpDerivative[14] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*yDown[2]*zDown[2] - 2*yUp[2]*zDown[2] + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] - 4*qCurrent[2]*zDown[4] + 4*qCurrent[3]*zDown[4] + 2*yDown[4]*zDown[4] - 2*yUp[4]*zDown[4] + 4*zDown[0]*zDown[4] + 4*zDown[3]*zDown[4] + 2*zDown[2]*zDownDerivative[4] + 2*zDown[4]*zDownDerivative[9] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*yDown[2]*zUp[2] + 2*yUp[2]*zUp[2] + 4*qCurrent[4]*(xDown[0] + yDown[3] + yUp[3] - zDown[0] - zDown[3] - zUp[0] - zUp[3]) - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] + 4*qCurrent[2]*zUp[4] + 4*qCurrent[3]*zUp[4] - 2*yDown[4]*zUp[4] + 2*yUp[4]*zUp[4] + 4*zUp[0]*zUp[4] + 4*zUp[3]*zUp[4] + 4*qCurrent[0]*(-2*xDown[4] + zDown[4] + zUp[4]) - 2*zUp[2]*zUpDerivative[4] - 2*zUp[4]*zUpDerivative[9])/2.;
        		};
        	if(boundaryCase == 2)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 8*qCurrent[1]*xUp[1] - 8*qCurrent[2]*xUp[2] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] + 4*qCurrent[1]*yDown[0] + 2*qCurrent[1]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 4*qCurrent[1]*yUp[0] - 2*qCurrent[1]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 4*qCurrent[2]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] + 2*qCurrent[2]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] - 4*qCurrent[2]*zUp[0] + 4*qCurrent[0]*(-3*xUp[0] - 2*xUp[3] + yDown[3] + yUp[3] + zDown[0] + zUp[0]) + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] - 2*qCurrent[3]*(xUp[0] + 2*(2*xUp[3] + yDown[0] + yUp[0] - zDown[0] - zDown[3] - zUp[0] - zUp[3])) - 2*qCurrent[2]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xUp[0] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] + 4*qCurrent[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*qCurrent[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*qCurrent[1]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] + 2*qCurrent[4]*yDown[4] + xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 4*qCurrent[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*qCurrent[2]*yUp[2] + 2*xUp[2]*yUp[2] + 2*qCurrent[1]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*qCurrent[4]*yUp[4] + 2*xUp[4]*yUp[4] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*qCurrent[2]*zDown[1] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] - 2*qCurrent[2]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[3]*(yDown[0] - 2*yDown[1] + 2*yDown[3] - yUp[0] - 2*yUp[1] - 2*yUp[3] + 2*zDown[1] + 2*zUp[1]) + qCurrent[0]*(-4*xUp[1] + 2*yDown[0] + yDown[3] - 2*yUp[0] - yUp[3] + 2*zDown[1] + 2*zUp[1]) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = -2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[1]*yDown[2] - 2*qCurrent[3]*yDown[2] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] - 2*qCurrent[1]*yUp[2] - 2*qCurrent[3]*yUp[2] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + qCurrent[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] + 2*qCurrent[1]*zDown[1] - 2*xUp[1]*zDown[1] + 2*qCurrent[3]*zDown[2] - 2*xUp[2]*zDown[2] + 2*zDown[0]*zDown[2] + 2*qCurrent[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*zDown[2]*zDown[3] + 2*qCurrent[4]*zDown[4] - 2*xUp[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] - qCurrent[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*qCurrent[1]*zUp[1] + 2*xUp[1]*zUp[1] + 2*qCurrent[3]*zUp[2] + 2*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] + 2*qCurrent[2]*(xUp[0] + yDown[3] + yUp[3] - zDown[0] + 2*zDown[2] - zDown[3] - zUp[0] - 2*zUp[2] - zUp[3]) + qCurrent[0]*(-4*xUp[2] + 2*zDown[0] + 2*zDown[2] + zDown[3] - 2*zUp[0] + 2*zUp[2] - zUp[3]) - 2*qCurrent[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*qCurrent[4]*zUp[4] + 2*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = -(xUp[0]*xUp[0]) + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zDown[1]*zDown[1] - zDown[2]*zDown[2] + zDown[3]*zDown[3] - zDown[4]*zDown[4] - zUp[1]*zUp[1] - zUp[2]*zUp[2] + zUp[3]*zUp[3] - zUp[4]*zUp[4] - 2*xUp[0]*xUp[3] - (xUp[1]*xUpDerivative[5])/2. - xUp[1]*xUpDerivative[8] - (xUp[2]*xUpDerivative[10])/2. - xUp[2]*xUpDerivative[13] + qCurrent[1]*yDown[0] - 2*qCurrent[1]*yDown[1] - 2*qCurrent[2]*yDown[2] + 2*qCurrent[1]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] - qCurrent[1]*yUp[0] - 2*qCurrent[1]*yUp[1] - 2*qCurrent[2]*yUp[2] - 2*qCurrent[1]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] + qCurrent[2]*zDown[0] + 2*qCurrent[1]*zDown[1] + 2*qCurrent[2]*zDown[2] + 2*qCurrent[2]*zDown[3] + 2*zDown[0]*zDown[3] + 2*qCurrent[4]*zDown[4] + (zDown[2]*zDownDerivative[0])/2. + zDown[2]*zDownDerivative[3] + (zDown[4]*zDownDerivative[5])/2. + zDown[4]*zDownDerivative[8] - qCurrent[2]*zUp[0] + 2*qCurrent[1]*zUp[1] + 2*qCurrent[2]*zUp[2] - qCurrent[0]*(xUp[0] + 2*(2*xUp[3] + yDown[0] + yUp[0] - zDown[0] - zDown[3] - zUp[0] - zUp[3])) - 2*qCurrent[2]*zUp[3] + 2*zUp[0]*zUp[3] + 2*qCurrent[3]*(xUp[0] - yDown[0] - yDown[3] - yUp[0] - yUp[3] + zDown[3] + zUp[3]) + 2*qCurrent[4]*zUp[4] - (zUp[2]*zUpDerivative[0])/2. - zUp[2]*zUpDerivative[3] - (zUp[4]*zUpDerivative[5])/2. - zUp[4]*zUpDerivative[8];
        		spatialTerm[4] = (-4*xUp[0]*xUp[4] - 2*xUp[1]*xUpDerivative[9] - 2*xUp[2]*xUpDerivative[14] + 4*qCurrent[1]*yDown[4] - 4*qCurrent[3]*yDown[4] + 2*xDownDerivative[14]*yDown[4] - 4*yDown[3]*yDown[4] + 2*yDown[1]*yDownDerivative[4] - 4*qCurrent[1]*yUp[4] - 4*qCurrent[3]*yUp[4] - 4*yUp[3]*yUp[4] - 2*yUp[1]*yUpDerivative[4] - 2*yUp[4]*yUpDerivative[14] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*yDown[2]*zDown[2] - 2*yUp[2]*zDown[2] + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 4*qCurrent[2]*zDown[4] + 4*qCurrent[3]*zDown[4] + 2*yDown[4]*zDown[4] - 2*yUp[4]*zDown[4] + 4*zDown[0]*zDown[4] + 4*zDown[3]*zDown[4] + 2*zDown[2]*zDownDerivative[4] + 2*zDown[4]*zDownDerivative[9] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*yDown[2]*zUp[2] + 2*yUp[2]*zUp[2] + 4*qCurrent[4]*(xUp[0] + yDown[3] + yUp[3] - zDown[0] - zDown[3] - zUp[0] - zUp[3]) - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] - 4*qCurrent[2]*zUp[4] + 4*qCurrent[3]*zUp[4] - 2*yDown[4]*zUp[4] + 2*yUp[4]*zUp[4] + 4*zUp[0]*zUp[4] + 4*zUp[3]*zUp[4] + 4*qCurrent[0]*(-2*xUp[4] + zDown[4] + zUp[4]) - 2*zUp[2]*zUpDerivative[4] - 2*zUp[4]*zUpDerivative[9])/2.;
        		};
        	if(boundaryCase == 4)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yDown[3]*yDown[3] + zDown[0]*zDown[0] - zDown[1]*zDown[1] - zDown[2]*zDown[2] - zDown[4]*zDown[4] + zUp[0]*zUp[0] - zUp[1]*zUp[1] - zUp[2]*zUp[2] - zUp[4]*zUp[4] - 2*qCurrent[2]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. - 2*qCurrent[2]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. - 4*qCurrent[3]*yDown[0] - qCurrent[3]*yDown[3] - 2*yDown[0]*yDown[3] + xDownDerivative[10]*yDown[4] + (xDownDerivative[13]*yDown[4])/2. + yDown[1]*yDownDerivative[0] + (yDown[1]*yDownDerivative[3])/2. + 2*qCurrent[3]*zDown[0] - 2*qCurrent[4]*zDown[0] + 2*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 2*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + (zDown[2]*zDownDerivative[3])/2. + zDown[4]*zDownDerivative[5] + (zDown[4]*zDownDerivative[8])/2. - 2*qCurrent[0]*(xDown[0] + xDown[3] + xUp[0] + xUp[3] - yDown[3] - zDown[0] - zUp[0]) + 2*qCurrent[3]*zUp[0] + 2*qCurrent[4]*zUp[0] + qCurrent[1]*(-2*xDown[0] - 2*xDown[1] - xDown[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*zDown[1] + 2*zUp[1]) + 2*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[3] + qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 2*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - (zUp[2]*zUpDerivative[3])/2. - zUp[4]*zUpDerivative[5] - (zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 4*qCurrent[1]*xDown[1] - 2*xDown[0]*xDown[1] - 2*qCurrent[2]*xDown[2] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*xUp[0] + 4*qCurrent[1]*xUp[1] - 2*xUp[0]*xUp[1] + 2*qCurrent[2]*xUp[2] + 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] + 2*xDown[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*xDown[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*qCurrent[1]*yDown[3] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] + 2*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) - 2*qCurrent[4]*zDown[1] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*qCurrent[4]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[0]*(-2*xDown[0] - 2*xDown[1] - xDown[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*zDown[1] + 2*zUp[1]) + qCurrent[3]*(-xDown[0] - 2*xDown[3] + xUp[0] + 2*xUp[3] - 4*yDown[1] + 2*zDown[1] + 2*zUp[1]) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*zDown[0]*zDown[2] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] + 2*zUp[0]*zUp[2] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + (-4*qCurrent[0]*xDown[2] - 4*qCurrent[1]*xDown[2] - 4*qCurrent[0]*xUp[2] + 4*qCurrent[1]*xUp[2] - 8*qCurrent[3]*yDown[2] + (xDown[3] - 2*xUp[0] - xUp[3])*zDown[0] + 2*xDown[1]*zDown[1] - 2*xUp[1]*zDown[1] + 4*qCurrent[0]*zDown[2] + 4*qCurrent[3]*zDown[2] - 4*qCurrent[4]*zDown[2] + 2*xDown[2]*zDown[2] - 2*xUp[2]*zDown[2] + 2*xDown[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*xDown[4]*zDown[4] - 2*xUp[4]*zDown[4] - xDown[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*xUp[1]*zUp[1] + 4*qCurrent[0]*zUp[2] + 4*qCurrent[3]*zUp[2] + 4*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*xUp[2]*zUp[2] + xDown[0]*(2*zDown[0] + zDown[3] - 2*zUp[0] - zUp[3]) - 2*xDown[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] - 2*xDown[4]*zUp[4] + 2*xUp[4]*zUp[4])/2. - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) - 4*qCurrent[0]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 4*qCurrent[0]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[0]*yDown[0] - 8*qCurrent[2]*yDown[2] - 2*qCurrent[0]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] + 4*qCurrent[0]*zDown[0] - 2*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[0]*zDown[3] - 4*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] + 4*qCurrent[0]*zUp[0] + 2*qCurrent[4]*zUp[0] + qCurrent[1]*(-2*xDown[0] - 4*xDown[3] + 2*xUp[0] + 4*xUp[3] - 8*yDown[1] + 4*zDown[1] + 4*zUp[1]) + 4*qCurrent[2]*zUp[2] + 4*qCurrent[0]*zUp[3] + 4*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[3]*(xDown[0] + xUp[0] - 2*yDown[0] - 3*yDown[3] + zDown[3] + zUp[3]) + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = -2*qCurrent[1]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[1]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - 4*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - qCurrent[3]*zDown[0] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*qCurrent[1]*zDown[1] + 2*yDown[1]*zDown[1] - 2*qCurrent[2]*zDown[2] + 2*yDown[2]*zDown[2] - 2*qCurrent[3]*zDown[3] + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] + 2*qCurrent[3]*zDown[4] + 2*yDown[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] + qCurrent[3]*zUp[0] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*qCurrent[1]*zUp[1] - 2*yDown[1]*zUp[1] + 2*qCurrent[2]*zUp[2] - 2*yDown[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + 2*qCurrent[3]*zUp[4] - 2*yDown[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + 2*qCurrent[4]*(xDown[0] + xUp[0] + yDown[3] - zDown[0] - zDown[3] - 2*zDown[4] - zUp[0] - zUp[3] + 2*zUp[4]) + qCurrent[0]*(-2*xDown[4] - 2*xUp[4] - 2*zDown[0] - zDown[3] + 2*zDown[4] + 2*zUp[0] + zUp[3] + 2*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 5)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yDown[3]*yDown[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yDown[0] - 2*qCurrent[3]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 4*qCurrent[2]*zDown[0] + 4*qCurrent[3]*zDown[0] - 4*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] - 2*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] - 2*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] + 4*qCurrent[2]*zUp[0] + 4*qCurrent[3]*zUp[0] + 4*qCurrent[4]*zUp[0] + 4*qCurrent[0]*(4*qCurrent[1] - 3*xDown[0] - 2*xDown[3] + yDown[3] + zDown[0] + zUp[0]) + 4*qCurrent[1]*(2*qCurrent[3] - 2*xDown[0] - 2*xDown[1] - xDown[3] - 2*yDown[0] - yDown[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] + 2*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] + 2*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*yDown[3] + xDownDerivative[11]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*zUp[0]*zUp[1] + 2*(2*(qCurrent[0]*qCurrent[0]) + 6*(qCurrent[1]*qCurrent[1]) + 2*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 2*(qCurrent[4]*qCurrent[4]) - qCurrent[3]*xDown[0] - 2*qCurrent[2]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] - qCurrent[3]*yDown[0] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*qCurrent[3]*yDown[1] + 2*xDown[1]*yDown[1] - 4*qCurrent[1]*(xDown[1] + yDown[1]) - 2*qCurrent[2]*yDown[2] + 2*xDown[2]*yDown[2] - 2*qCurrent[3]*yDown[3] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - 2*qCurrent[4]*yDown[4] + 2*xDown[4]*yDown[4] - qCurrent[2]*zDown[1] + qCurrent[3]*zDown[1] - qCurrent[4]*zDown[1] + qCurrent[2]*zUp[1] + qCurrent[3]*zUp[1] + qCurrent[4]*zUp[1] + qCurrent[0]*(2*qCurrent[3] - 2*xDown[0] - 2*xDown[1] - xDown[3] - 2*yDown[0] - yDown[3] + zDown[1] + zUp[1])) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - 4*qCurrent[3]*yDown[2] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] - qCurrent[3]*zDown[0] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] + 2*xDown[1]*zDown[1] - 4*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[2] - 2*qCurrent[4]*zDown[2] + 2*xDown[2]*zDown[2] + 2*zDown[0]*zDown[2] - 2*qCurrent[3]*zDown[3] + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) - 2*qCurrent[4]*zDown[4] + 2*xDown[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] + qCurrent[3]*zUp[0] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*qCurrent[1]*(4*qCurrent[2] - 2*xDown[2] - 2*yDown[2] - zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[2] + 2*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*zUp[0]*zUp[2] + 2*qCurrent[3]*zUp[3] - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + qCurrent[0]*(-4*xDown[2] - 2*zDown[0] + 2*zDown[2] - zDown[3] + 2*zUp[0] + 2*zUp[2] + zUp[3]) + 2*qCurrent[4]*zUp[4] - 2*xDown[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) + 4*qCurrent[3]*xDown[0] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yDown[0] - 8*qCurrent[2]*yDown[2] - 12*qCurrent[3]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] - 2*qCurrent[2]*zDown[0] - 2*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] - 4*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] - 4*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] + 2*qCurrent[2]*zUp[0] + 2*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(4*qCurrent[3] - xDown[0] - 2*xDown[3] - yDown[0] - 2*yDown[1] - 2*yDown[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] + qCurrent[0]*(8*qCurrent[1] - 2*(xDown[0] + 4*xDown[3] + 4*yDown[0] + yDown[3] - 2*zDown[0] - 2*zDown[3] - 2*zUp[0] - 2*zUp[3])) + 4*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] + 4*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*yDown[3] - 4*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - qCurrent[3]*zDown[0] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*qCurrent[2]*zDown[2] + 2*yDown[2]*zDown[2] - 2*qCurrent[3]*zDown[3] + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - 2*qCurrent[4]*(zDown[0] + zDown[3]) - 2*qCurrent[2]*zDown[4] + 2*qCurrent[3]*zDown[4] - 4*qCurrent[4]*zDown[4] + 2*yDown[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] + qCurrent[3]*zUp[0] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*qCurrent[1]*(4*qCurrent[4] - 2*xDown[4] - 2*yDown[4] - zDown[1] + zUp[1]) + 2*qCurrent[2]*zUp[2] - 2*yDown[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] - 2*qCurrent[4]*(zUp[0] + zUp[3]) + 2*qCurrent[2]*zUp[4] + 2*qCurrent[3]*zUp[4] + 4*qCurrent[4]*zUp[4] - 2*yDown[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + qCurrent[0]*(-4*xDown[4] - 2*zDown[0] - zDown[3] + 2*zDown[4] + 2*zUp[0] + zUp[3] + 2*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 6)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yDown[0] - 2*qCurrent[3]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] + 4*qCurrent[2]*zDown[0] + 4*qCurrent[3]*zDown[0] - 4*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] + 2*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] - 2*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] - 4*qCurrent[0]*(4*qCurrent[1] + 3*xUp[0] + 2*xUp[3] - yDown[3] - zDown[0] - zUp[0]) - 4*qCurrent[2]*zUp[0] + 4*qCurrent[3]*zUp[0] + 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*yDown[0] + yDown[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] - 2*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] + 2*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xUp[0] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*qCurrent[1]*yDown[3] + xDownDerivative[11]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*zUp[0]*zUp[1] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 6*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) + qCurrent[3]*xUp[0] + 2*qCurrent[2]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] + qCurrent[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 2*qCurrent[3]*yDown[1] - 2*xUp[1]*yDown[1] + 4*qCurrent[1]*(xUp[1] + yDown[1]) + 2*qCurrent[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*qCurrent[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] + 2*qCurrent[4]*yDown[4] - 2*xUp[4]*yDown[4] + qCurrent[2]*zDown[1] + qCurrent[3]*zDown[1] - qCurrent[4]*zDown[1] - qCurrent[2]*zUp[1] + qCurrent[3]*zUp[1] + qCurrent[4]*zUp[1] + qCurrent[0]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*yDown[0] + yDown[3] + zDown[1] + zUp[1])) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yDown[2] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + qCurrent[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] - 2*xUp[1]*zDown[1] + 4*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[2] - 2*qCurrent[4]*zDown[2] - 2*xUp[2]*zDown[2] + 2*zDown[0]*zDown[2] + 2*qCurrent[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + 2*qCurrent[4]*zDown[4] - 2*xUp[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] - qCurrent[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] + qCurrent[1]*(-8*qCurrent[2] + 4*xUp[2] + 4*yDown[2] + 2*zDown[1] - 2*zUp[1]) + 2*xUp[1]*zUp[1] - 4*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[2] + 2*qCurrent[4]*zUp[2] + 2*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] + qCurrent[0]*(-4*xUp[2] + 2*zDown[0] + 2*zDown[2] + zDown[3] - 2*zUp[0] + 2*zUp[2] - zUp[3]) - 2*qCurrent[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) - 2*qCurrent[4]*zUp[4] + 2*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xUp[0]*xUp[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) + 4*qCurrent[3]*xUp[0] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yDown[0] - 8*qCurrent[2]*yDown[2] - 12*qCurrent[3]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] + 2*qCurrent[2]*zDown[0] - 2*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] - 4*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] - 2*qCurrent[2]*zUp[0] + 2*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-4*qCurrent[3] + xUp[0] + 2*xUp[3] + yDown[0] - 2*yDown[1] + 2*yDown[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] - 2*qCurrent[0]*(4*qCurrent[1] + xUp[0] + 4*xUp[3] + 4*yDown[0] + yDown[3] - 2*zDown[0] - 2*zDown[3] - 2*zUp[0] - 2*zUp[3]) - 4*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] + 4*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yDown[3] - 4*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - qCurrent[3]*zDown[0] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*qCurrent[2]*zDown[2] + 2*yDown[2]*zDown[2] - 2*qCurrent[3]*zDown[3] + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - 2*qCurrent[4]*(zDown[0] + zDown[3]) + 2*qCurrent[2]*zDown[4] + 2*qCurrent[3]*zDown[4] - 4*qCurrent[4]*zDown[4] + 2*yDown[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] + qCurrent[3]*zUp[0] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] - 2*yDown[1]*zUp[1] + qCurrent[1]*(-8*qCurrent[4] + 4*xUp[4] + 4*yDown[4] - 2*zDown[1] + 2*zUp[1]) + 2*qCurrent[2]*zUp[2] - 2*yDown[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] - 2*qCurrent[4]*(zUp[0] + zUp[3]) - 2*qCurrent[2]*zUp[4] + 2*qCurrent[3]*zUp[4] + 4*qCurrent[4]*zUp[4] - 2*yDown[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + qCurrent[0]*(-4*xUp[4] - 2*zDown[0] - zDown[3] + 2*zDown[4] + 2*zUp[0] + zUp[3] + 2*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 8)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yUp[3]*yUp[3] + zDown[0]*zDown[0] - zDown[1]*zDown[1] - zDown[2]*zDown[2] - zDown[4]*zDown[4] + zUp[0]*zUp[0] - zUp[1]*zUp[1] - zUp[2]*zUp[2] - zUp[4]*zUp[4] - 2*qCurrent[2]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. - 2*qCurrent[2]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. - 4*qCurrent[3]*yUp[0] - qCurrent[3]*yUp[3] - 2*yUp[0]*yUp[3] - yUp[1]*yUpDerivative[0] - (yUp[1]*yUpDerivative[3])/2. - yUp[4]*yUpDerivative[10] - (yUp[4]*yUpDerivative[13])/2. + 2*qCurrent[3]*zDown[0] + 2*qCurrent[4]*zDown[0] + 2*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[3] + qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 2*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + (zDown[2]*zDownDerivative[3])/2. + zDown[4]*zDownDerivative[5] + (zDown[4]*zDownDerivative[8])/2. - 2*qCurrent[0]*(xDown[0] + xDown[3] + xUp[0] + xUp[3] - yUp[3] - zDown[0] - zUp[0]) + 2*qCurrent[3]*zUp[0] - 2*qCurrent[4]*zUp[0] + qCurrent[1]*(2*xDown[0] - 2*xDown[1] + xDown[3] - 2*xUp[0] - 2*xUp[1] - xUp[3] + 2*zDown[1] + 2*zUp[1]) + 2*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 2*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - (zUp[2]*zUpDerivative[3])/2. - zUp[4]*zUpDerivative[5] - (zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] + 4*qCurrent[1]*xDown[1] - 2*xDown[0]*xDown[1] + 2*qCurrent[2]*xDown[2] + 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*xUp[0] - 4*qCurrent[1]*xUp[1] - 2*xUp[0]*xUp[1] - 2*qCurrent[2]*xUp[2] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 2*xDown[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*xDown[2]*yUp[2] + 2*xUp[2]*yUp[2] + 2*qCurrent[1]*yUp[3] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*xDown[4]*yUp[4] + 2*xUp[4]*yUp[4] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*qCurrent[4]*zDown[1] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] - 2*qCurrent[4]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[0]*(2*xDown[0] - 2*xDown[1] + xDown[3] - 2*xUp[0] - 2*xUp[1] - xUp[3] + 2*zDown[1] + 2*zUp[1]) + qCurrent[3]*(xDown[0] + 2*xDown[3] - xUp[0] - 2*xUp[3] - 4*yUp[1] + 2*zDown[1] + 2*zUp[1]) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = (4*qCurrent[1]*xDown[2] - 4*xDown[0]*xDown[2] + 2*xDown[1]*xDownDerivative[7] + 2*xDown[2]*xDownDerivative[12] - 4*qCurrent[1]*xUp[2] - 4*xUp[0]*xUp[2] - 2*xUp[1]*xUpDerivative[7] - 2*xUp[2]*xUpDerivative[12] - 8*qCurrent[3]*yUp[2] - 4*yUp[2]*yUp[3] - 2*yUp[1]*yUpDerivative[2] - 2*yUp[4]*yUpDerivative[12] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] + 2*xDown[1]*zDown[1] - 2*xUp[1]*zDown[1] + 4*qCurrent[3]*zDown[2] + 4*qCurrent[4]*zDown[2] + 2*xDown[2]*zDown[2] - 2*xUp[2]*zDown[2] + 4*zDown[0]*zDown[2] + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 4*zDown[2]*zDown[3] + 2*xDown[4]*zDown[4] - 2*xUp[4]*zDown[4] + 2*zDown[2]*zDownDerivative[2] + 2*zDown[4]*zDownDerivative[7] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*xUp[1]*zUp[1] - 4*qCurrent[0]*(xDown[2] + xUp[2] - zDown[2] - zUp[2]) + 4*qCurrent[3]*zUp[2] - 4*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*xUp[2]*zUp[2] + 4*zUp[0]*zUp[2] + 4*qCurrent[2]*(xDown[0] + xUp[0] + yUp[3] - zDown[0] - zDown[3] - zUp[0] - zUp[3]) - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] + 4*zUp[2]*zUp[3] - 2*xDown[4]*zUp[4] + 2*xUp[4]*zUp[4] - 2*zUp[2]*zUpDerivative[2] - 2*zUp[4]*zUpDerivative[7])/2.;
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) - 4*qCurrent[0]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 4*qCurrent[0]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[0]*yUp[0] - 8*qCurrent[2]*yUp[2] - 2*qCurrent[0]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 4*qCurrent[0]*zDown[0] + 2*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[0]*zDown[3] + 4*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] + 4*qCurrent[0]*zUp[0] - 2*qCurrent[4]*zUp[0] + 2*qCurrent[1]*(xDown[0] + 2*xDown[3] - xUp[0] - 2*xUp[3] - 4*yUp[1] + 2*zDown[1] + 2*zUp[1]) + 4*qCurrent[2]*zUp[2] + 4*qCurrent[0]*zUp[3] - 4*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[3]*(xDown[0] + xUp[0] - 2*yUp[0] - 3*yUp[3] + zDown[3] + zUp[3]) + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[1]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] - 2*qCurrent[1]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - 4*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] + qCurrent[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*qCurrent[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*qCurrent[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 2*qCurrent[3]*zDown[4] - 2*yUp[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] - qCurrent[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*qCurrent[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*qCurrent[2]*zUp[2] + 2*yUp[2]*zUp[2] - 2*qCurrent[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] + 2*qCurrent[4]*(xDown[0] + xUp[0] + yUp[3] - zDown[0] - zDown[3] + 2*zDown[4] - zUp[0] - zUp[3] - 2*zUp[4]) + 2*qCurrent[3]*zUp[4] + 2*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + qCurrent[0]*(-2*xDown[4] - 2*xUp[4] + 2*zDown[0] + zDown[3] + 2*zDown[4] - 2*zUp[0] - zUp[3] + 2*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 9)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yUp[3]*yUp[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yUp[0] - 2*qCurrent[3]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] - 4*qCurrent[2]*zDown[0] + 4*qCurrent[3]*zDown[0] + 4*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] - 2*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] + 2*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] - 4*qCurrent[0]*(4*qCurrent[1] + 3*xDown[0] + 2*xDown[3] - yUp[3] - zDown[0] - zUp[0]) + 4*qCurrent[2]*zUp[0] + 4*qCurrent[3]*zUp[0] - 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[1] + xDown[3] + 2*yUp[0] + yUp[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] + 2*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] - 2*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*yUp[3] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*zUp[0]*zUp[1] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 6*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) + qCurrent[3]*xDown[0] + 2*qCurrent[2]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + qCurrent[3]*yUp[0] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] - 2*qCurrent[3]*yUp[1] - 2*xDown[1]*yUp[1] + 4*qCurrent[1]*(xDown[1] + yUp[1]) + 2*qCurrent[2]*yUp[2] - 2*xDown[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + 2*qCurrent[4]*yUp[4] - 2*xDown[4]*yUp[4] - qCurrent[2]*zDown[1] + qCurrent[3]*zDown[1] + qCurrent[4]*zDown[1] + qCurrent[2]*zUp[1] + qCurrent[3]*zUp[1] - qCurrent[4]*zUp[1] + qCurrent[0]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[1] + xDown[3] + 2*yUp[0] + yUp[3] + zDown[1] + zUp[1])) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - 4*qCurrent[3]*yUp[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] - qCurrent[3]*zDown[0] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] + 2*xDown[1]*zDown[1] - 4*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[2] + 2*qCurrent[4]*zDown[2] + 2*xDown[2]*zDown[2] + 2*zDown[0]*zDown[2] - 2*qCurrent[3]*zDown[3] + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) - 2*qCurrent[4]*zDown[4] + 2*xDown[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] + qCurrent[3]*zUp[0] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] - 2*xDown[1]*zUp[1] + qCurrent[1]*(-8*qCurrent[2] + 4*xDown[2] + 4*yUp[2] - 2*zDown[1] + 2*zUp[1]) + 4*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[2] - 2*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*zUp[0]*zUp[2] + 2*qCurrent[3]*zUp[3] - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + qCurrent[0]*(-4*xDown[2] - 2*zDown[0] + 2*zDown[2] - zDown[3] + 2*zUp[0] + 2*zUp[2] + zUp[3]) + 2*qCurrent[4]*zUp[4] - 2*xDown[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) + 4*qCurrent[3]*xDown[0] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yUp[0] - 8*qCurrent[2]*yUp[2] - 12*qCurrent[3]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] - 2*qCurrent[2]*zDown[0] + 2*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] - 4*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] + 4*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] + 2*qCurrent[2]*zUp[0] - 2*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-4*qCurrent[3] + xDown[0] + 2*xDown[3] + yUp[0] - 2*yUp[1] + 2*yUp[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] - 2*qCurrent[0]*(4*qCurrent[1] + xDown[0] + 4*xDown[3] + 4*yUp[0] + yUp[3] - 2*zDown[0] - 2*zDown[3] - 2*zUp[0] - 2*zUp[3]) + 4*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] - 4*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*yUp[3] - 4*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] + qCurrent[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] - 2*yUp[1]*zDown[1] + 2*qCurrent[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] - 2*qCurrent[4]*(zDown[0] + zDown[3]) - 2*qCurrent[2]*zDown[4] + 2*qCurrent[3]*zDown[4] + 4*qCurrent[4]*zDown[4] - 2*yUp[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] - qCurrent[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] + qCurrent[1]*(-8*qCurrent[4] + 4*xDown[4] + 4*yUp[4] + 2*zDown[1] - 2*zUp[1]) + 2*yUp[1]*zUp[1] - 2*qCurrent[2]*zUp[2] + 2*yUp[2]*zUp[2] - 2*qCurrent[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] - 2*qCurrent[4]*(zUp[0] + zUp[3]) + 2*qCurrent[2]*zUp[4] + 2*qCurrent[3]*zUp[4] - 4*qCurrent[4]*zUp[4] + 2*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + qCurrent[0]*(-4*xDown[4] + 2*zDown[0] + zDown[3] + 2*zDown[4] - 2*zUp[0] - zUp[3] + 2*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 10)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yUp[3]*yUp[3]) + 2*(zDown[0]*zDown[0]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*(zUp[0]*zUp[0]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yUp[0] - 2*qCurrent[3]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 4*qCurrent[2]*zDown[0] + 4*qCurrent[3]*zDown[0] + 4*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] + 2*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] + 2*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8] - 4*qCurrent[2]*zUp[0] + 4*qCurrent[3]*zUp[0] - 4*qCurrent[4]*zUp[0] + 4*qCurrent[0]*(4*qCurrent[1] - 3*xUp[0] - 2*xUp[3] + yUp[3] + zDown[0] + zUp[0]) + 4*qCurrent[1]*(2*qCurrent[3] - 2*xUp[0] - 2*xUp[1] - xUp[3] - 2*yUp[0] - yUp[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] - 2*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] - 2*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xUp[0] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*qCurrent[1]*yUp[3] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6] + 2*zUp[0]*zUp[1] + 2*(2*(qCurrent[0]*qCurrent[0]) + 6*(qCurrent[1]*qCurrent[1]) + 2*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 2*(qCurrent[4]*qCurrent[4]) - qCurrent[3]*xUp[0] - 2*qCurrent[2]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - qCurrent[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 2*qCurrent[3]*yUp[1] + 2*xUp[1]*yUp[1] - 4*qCurrent[1]*(xUp[1] + yUp[1]) - 2*qCurrent[2]*yUp[2] + 2*xUp[2]*yUp[2] - 2*qCurrent[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*qCurrent[4]*yUp[4] + 2*xUp[4]*yUp[4] + qCurrent[2]*zDown[1] + qCurrent[3]*zDown[1] + qCurrent[4]*zDown[1] - qCurrent[2]*zUp[1] + qCurrent[3]*zUp[1] - qCurrent[4]*zUp[1] + qCurrent[0]*(2*qCurrent[3] - 2*xUp[0] - 2*xUp[1] - xUp[3] - 2*yUp[0] - yUp[3] + zDown[1] + zUp[1])) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yUp[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + qCurrent[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] - 2*xUp[1]*zDown[1] + 4*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[2] + 2*qCurrent[4]*zDown[2] - 2*xUp[2]*zDown[2] + 2*zDown[0]*zDown[2] + 2*qCurrent[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + 2*qCurrent[4]*zDown[4] - 2*xUp[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7] - qCurrent[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] + 2*qCurrent[1]*(4*qCurrent[2] - 2*xUp[2] - 2*yUp[2] + zDown[1] - zUp[1]) + 2*xUp[1]*zUp[1] - 4*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[2] - 2*qCurrent[4]*zUp[2] + 2*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] + qCurrent[0]*(-4*xUp[2] + 2*zDown[0] + 2*zDown[2] + zDown[3] - 2*zUp[0] + 2*zUp[2] - zUp[3]) - 2*qCurrent[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) - 2*qCurrent[4]*zUp[4] + 2*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xUp[0]*xUp[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) + 2*(zDown[3]*zDown[3]) - 2*(zDown[4]*zDown[4]) - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) + 2*(zUp[3]*zUp[3]) - 2*(zUp[4]*zUp[4]) + 4*qCurrent[3]*xUp[0] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yUp[0] - 8*qCurrent[2]*yUp[2] - 12*qCurrent[3]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 2*qCurrent[2]*zDown[0] + 2*qCurrent[4]*zDown[0] + 4*qCurrent[2]*zDown[2] + 4*qCurrent[2]*zDown[3] + 4*qCurrent[3]*zDown[3] + 4*qCurrent[4]*zDown[3] + 4*zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8] - 2*qCurrent[2]*zUp[0] - 2*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(4*qCurrent[3] - xUp[0] - 2*xUp[3] - yUp[0] - 2*yUp[1] - 2*yUp[3] + zDown[1] + zUp[1]) + 4*qCurrent[2]*zUp[2] + qCurrent[0]*(8*qCurrent[1] - 2*(xUp[0] + 4*xUp[3] + 4*yUp[0] + yUp[3] - 2*zDown[0] - 2*zDown[3] - 2*zUp[0] - 2*zUp[3])) - 4*qCurrent[2]*zUp[3] + 4*qCurrent[3]*zUp[3] - 4*qCurrent[4]*zUp[3] + 4*zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yUp[3] - 4*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] + qCurrent[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] - 2*yUp[1]*zDown[1] + 2*qCurrent[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] - 2*qCurrent[4]*(zDown[0] + zDown[3]) + 2*qCurrent[2]*zDown[4] + 2*qCurrent[3]*zDown[4] + 4*qCurrent[4]*zDown[4] - 2*yUp[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9] - qCurrent[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] + 2*qCurrent[1]*(4*qCurrent[4] - 2*xUp[4] - 2*yUp[4] + zDown[1] - zUp[1]) + 2*yUp[1]*zUp[1] - 2*qCurrent[2]*zUp[2] + 2*yUp[2]*zUp[2] - 2*qCurrent[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] - 2*qCurrent[4]*(zUp[0] + zUp[3]) - 2*qCurrent[2]*zUp[4] + 2*qCurrent[3]*zUp[4] - 4*qCurrent[4]*zUp[4] + 2*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + qCurrent[0]*(-4*xUp[4] + 2*zDown[0] + zDown[3] + 2*zDown[4] - 2*zUp[0] - zUp[3] + 2*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 16)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yDown[3]*yDown[3] - yUp[3]*yUp[3] - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - zDown[3]*zDown[3] - 2*(zDown[4]*zDown[4]) - 2*qCurrent[1]*xDown[1] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. - 2*qCurrent[1]*xUp[1] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. - 2*qCurrent[3]*yDown[0] - 2*qCurrent[4]*yDown[0] - qCurrent[4]*yDown[3] - 2*yDown[0]*yDown[3] + xDownDerivative[10]*yDown[4] + (xDownDerivative[13]*yDown[4])/2. + yDown[1]*yDownDerivative[0] + (yDown[1]*yDownDerivative[3])/2. - 2*qCurrent[3]*yUp[0] + 2*qCurrent[4]*yUp[0] + qCurrent[4]*yUp[3] - 2*yUp[0]*yUp[3] - yUp[1]*yUpDerivative[0] - (yUp[1]*yUpDerivative[3])/2. - yUp[4]*yUpDerivative[10] - (yUp[4]*yUpDerivative[13])/2. + 5*qCurrent[3]*zDown[0] + 4*qCurrent[1]*zDown[1] + qCurrent[2]*(-2*xDown[0] - 2*xDown[2] - xDown[3] + 2*xUp[0] - 2*xUp[2] + xUp[3] + 4*zDown[2]) - 2*qCurrent[0]*(xDown[0] + xDown[3] + xUp[0] + xUp[3] - yDown[3] - yUp[3] - 3*zDown[0] - zDown[3]) + 5*qCurrent[3]*zDown[3] + zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + (zDown[2]*zDownDerivative[3])/2. + zDown[4]*zDownDerivative[5] + (zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = (-4*qCurrent[2]*xDown[1] - 4*xDown[0]*xDown[1] + 2*xDown[1]*xDownDerivative[6] + 2*xDown[2]*xDownDerivative[11] + 4*qCurrent[2]*xUp[1] - 4*xUp[0]*xUp[1] - 2*xUp[1]*xUpDerivative[6] - 2*xUp[2]*xUpDerivative[11] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 4*qCurrent[3]*yDown[1] - 4*qCurrent[4]*yDown[1] + 2*xDown[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*xDown[2]*yDown[2] - 2*xUp[2]*yDown[2] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] - 4*yDown[1]*yDown[3] + 2*xDown[4]*yDown[4] + 2*xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + 2*yDown[1]*yDownDerivative[1] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] + 4*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*xDown[2]*yUp[2] + 2*xUp[2]*yUp[2] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 4*yUp[1]*yUp[3] - 2*xDown[4]*yUp[4] + 2*xUp[4]*yUp[4] - 2*yUp[1]*yUpDerivative[1] - 2*yUp[4]*yUpDerivative[11] - 4*qCurrent[0]*(xDown[1] + xUp[1] - 2*zDown[1]) + 8*qCurrent[3]*zDown[1] + 4*zDown[0]*zDown[1] + 4*qCurrent[1]*(xDown[0] + xUp[0] + yDown[3] + yUp[3] - zDown[0] - zDown[3]) + 4*zDown[1]*zDown[3] + 2*zDown[2]*zDownDerivative[1] + 2*zDown[4]*zDownDerivative[6])/2.;
        		spatialTerm[2] = -(qCurrent[3]*xDown[0]) - 2*qCurrent[1]*xDown[1] - 2*xDown[0]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + qCurrent[3]*xUp[0] + 2*qCurrent[1]*xUp[1] - 2*xUp[0]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 2*qCurrent[3]*yDown[2] - 2*qCurrent[4]*yDown[2] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] - 2*qCurrent[3]*yUp[2] + 2*qCurrent[4]*yUp[2] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] + 2*xDown[1]*zDown[1] - 2*xUp[1]*zDown[1] + 4*qCurrent[3]*zDown[2] + 2*xDown[2]*zDown[2] - 2*xUp[2]*zDown[2] + 2*zDown[0]*zDown[2] + qCurrent[0]*(-2*xDown[0] - 2*xDown[2] - xDown[3] + 2*xUp[0] - 2*xUp[2] + xUp[3] + 4*zDown[2]) + 2*qCurrent[2]*(xDown[0] - 2*xDown[2] + xUp[0] + 2*xUp[2] + yDown[3] + yUp[3] - zDown[0] - zDown[3]) + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*zDown[2]*zDown[3] + 2*xDown[4]*zDown[4] - 2*xUp[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = -(xDown[0]*xDown[0]) - xUp[0]*xUp[0] + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zDown[0]*zDown[0] - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) - 2*qCurrent[0]*xDown[3] - 2*xDown[0]*xDown[3] + (xDown[1]*xDownDerivative[5])/2. + xDown[1]*xDownDerivative[8] + (xDown[2]*xDownDerivative[10])/2. + xDown[2]*xDownDerivative[13] - 2*qCurrent[0]*xUp[3] - 2*xUp[0]*xUp[3] - (xUp[1]*xUpDerivative[5])/2. - xUp[1]*xUpDerivative[8] - (xUp[2]*xUpDerivative[10])/2. - xUp[2]*xUpDerivative[13] - 2*qCurrent[0]*yDown[0] - qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[1] - 2*qCurrent[4]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] - 2*qCurrent[0]*yUp[0] + qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[1] + 2*qCurrent[4]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] + 5*qCurrent[0]*zDown[0] + 4*qCurrent[1]*zDown[1] + qCurrent[2]*(-xDown[0] - 2*xDown[3] + xUp[0] + 2*xUp[3] - 2*yDown[2] - 2*yUp[2] + 4*zDown[2]) + 5*qCurrent[0]*zDown[3] + zDown[0]*zDown[3] + 2*qCurrent[3]*(xDown[0] + xUp[0] - yDown[0] - yDown[3] - yUp[0] - yUp[3] + zDown[0] + 3*zDown[3]) + 4*qCurrent[4]*zDown[4] + (zDown[2]*zDownDerivative[0])/2. + zDown[2]*zDownDerivative[3] + (zDown[4]*zDownDerivative[5])/2. + zDown[4]*zDownDerivative[8];
        		spatialTerm[4] = -2*qCurrent[2]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[2]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - qCurrent[3]*yDown[0] - 2*qCurrent[1]*yDown[1] - 2*qCurrent[2]*yDown[2] - 2*qCurrent[3]*yDown[3] - 2*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] + qCurrent[3]*yUp[0] + 2*qCurrent[1]*yUp[1] + 2*qCurrent[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - 2*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*yDown[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[4]*(xDown[0] + xUp[0] + yDown[3] - 2*yDown[4] + yUp[3] + 2*yUp[4] - zDown[0] - zDown[3]) + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 4*qCurrent[3]*zDown[4] + 2*yDown[4]*zDown[4] - 2*yUp[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + qCurrent[0]*(-2*xDown[4] - 2*xUp[4] - 2*yDown[0] - yDown[3] + 2*yUp[0] + yUp[3] + 4*zDown[4]) + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 17)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 2*(zDown[3]*zDown[3]) - 4*(zDown[4]*zDown[4]) - 2*qCurrent[3]*xDown[0] - 8*qCurrent[1]*xDown[1] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 4*qCurrent[1]*yDown[0] - 4*qCurrent[3]*yDown[0] - 4*qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[3] - 2*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] + 4*qCurrent[1]*yUp[0] - 4*qCurrent[3]*yUp[0] + 4*qCurrent[4]*yUp[0] + 2*qCurrent[1]*yUp[3] + 2*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 10*qCurrent[3]*zDown[0] + 8*qCurrent[1]*zDown[1] + 10*qCurrent[3]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[0]*(4*qCurrent[2] - 3*xDown[0] - 2*xDown[3] + yDown[3] + yUp[3] + 3*zDown[0] + zDown[3]) + qCurrent[2]*(8*qCurrent[3] - 4*(2*xDown[0] + 2*xDown[2] + xDown[3] + 2*zDown[0] - 2*zDown[2] + zDown[3])) + 8*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = -4*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] - qCurrent[3]*yDown[0] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*qCurrent[3]*yDown[1] - 2*qCurrent[4]*yDown[1] + 2*xDown[1]*yDown[1] - 2*qCurrent[2]*yDown[2] + 2*xDown[2]*yDown[2] - 2*qCurrent[3]*yDown[3] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - 2*yDown[1]*yDown[3] - 2*qCurrent[4]*yDown[4] + 2*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] + yDown[1]*yDownDerivative[1] + qCurrent[3]*yUp[0] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] - 2*qCurrent[3]*yUp[1] + 2*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 2*qCurrent[2]*yUp[2] - 2*xDown[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] - 2*yUp[1]*yUp[3] + 2*qCurrent[4]*yUp[4] - 2*xDown[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] - 4*qCurrent[2]*zDown[1] + 4*qCurrent[3]*zDown[1] + 2*zDown[0]*zDown[1] + qCurrent[0]*(-4*xDown[1] - 2*yDown[0] - yDown[3] + 2*yUp[0] + yUp[3] + 4*zDown[1]) + 2*qCurrent[1]*(4*qCurrent[2] + xDown[0] - 2*yDown[1] + yDown[3] + 2*yUp[1] + yUp[3] - zDown[0] - zDown[3]) + 2*zDown[1]*zDown[3] + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zDown[0]*zDown[2] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + 2*(2*(qCurrent[0]*qCurrent[0]) + 2*(qCurrent[1]*qCurrent[1]) + 6*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 2*(qCurrent[4]*qCurrent[4]) - qCurrent[3]*xDown[0] - 4*qCurrent[2]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] - qCurrent[3]*yDown[2] - qCurrent[4]*yDown[2] - qCurrent[3]*yUp[2] + qCurrent[4]*yUp[2] - qCurrent[3]*zDown[0] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] + 2*xDown[1]*zDown[1] - qCurrent[1]*(2*xDown[1] + yDown[2] - yUp[2] + 2*zDown[1]) - 4*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[2] + 2*xDown[2]*zDown[2] + qCurrent[0]*(2*qCurrent[3] - 2*xDown[0] - 2*xDown[2] - xDown[3] - 2*zDown[0] + 2*zDown[2] - zDown[3]) - 2*qCurrent[3]*zDown[3] + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] - 2*qCurrent[4]*zDown[4] + 2*xDown[4]*zDown[4]) + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = -(xDown[0]*xDown[0]) + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zDown[0]*zDown[0] - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*qCurrent[3]*xDown[0] - 2*xDown[0]*xDown[3] + (xDown[1]*xDownDerivative[5])/2. + xDown[1]*xDownDerivative[8] + (xDown[2]*xDownDerivative[10])/2. + xDown[2]*xDownDerivative[13] - qCurrent[1]*yDown[0] - 2*qCurrent[3]*yDown[0] - qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[1] - 2*qCurrent[1]*yDown[3] - 2*qCurrent[3]*yDown[3] - 2*qCurrent[4]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] + qCurrent[1]*yUp[0] - 2*qCurrent[3]*yUp[0] + qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[1] + 2*qCurrent[1]*yUp[3] - 2*qCurrent[3]*yUp[3] + 2*qCurrent[4]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] + 2*qCurrent[3]*zDown[0] + 4*qCurrent[1]*zDown[1] + 6*qCurrent[3]*zDown[3] + zDown[0]*zDown[3] + qCurrent[0]*(4*qCurrent[2] - xDown[0] - 4*xDown[3] - 2*yDown[0] - 2*yUp[0] + 5*zDown[0] + 5*zDown[3]) + qCurrent[2]*(8*qCurrent[3] - 2*(xDown[0] + 2*xDown[3] + yDown[2] + yUp[2] + zDown[0] - 2*zDown[2] + 2*zDown[3])) + 4*qCurrent[4]*zDown[4] + (zDown[2]*zDownDerivative[0])/2. + zDown[2]*zDownDerivative[3] + (zDown[4]*zDownDerivative[5])/2. + zDown[4]*zDownDerivative[8];
        		spatialTerm[4] = -4*qCurrent[0]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] - 2*qCurrent[0]*yDown[0] - qCurrent[3]*yDown[0] - 2*qCurrent[1]*yDown[1] - qCurrent[0]*yDown[3] - 2*qCurrent[3]*yDown[3] - 2*qCurrent[1]*yDown[4] - 2*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] + 2*qCurrent[0]*yUp[0] + qCurrent[3]*yUp[0] + 2*qCurrent[1]*yUp[1] + qCurrent[0]*yUp[3] + 2*qCurrent[3]*yUp[3] + 2*qCurrent[1]*yUp[4] - 2*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*yDown[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[4]*(xDown[0] + yDown[3] - 2*yDown[4] + yUp[3] + 2*yUp[4] - zDown[0] - zDown[3]) + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 2*qCurrent[2]*(4*qCurrent[4] - 2*xDown[4] - yDown[2] + yUp[2] - 2*zDown[4]) + 4*qCurrent[0]*zDown[4] + 4*qCurrent[3]*zDown[4] + 2*yDown[4]*zDown[4] - 2*yUp[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 18)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 2*(zDown[3]*zDown[3]) - 4*(zDown[4]*zDown[4]) - 2*qCurrent[3]*xUp[0] - 8*qCurrent[1]*xUp[1] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] + 4*qCurrent[1]*yDown[0] - 4*qCurrent[3]*yDown[0] - 4*qCurrent[4]*yDown[0] + 2*qCurrent[1]*yDown[3] - 2*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 4*qCurrent[1]*yUp[0] - 4*qCurrent[3]*yUp[0] + 4*qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[3] + 2*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 10*qCurrent[3]*zDown[0] + 8*qCurrent[1]*zDown[1] - 4*qCurrent[0]*(4*qCurrent[2] + 3*xUp[0] + 2*xUp[3] - yDown[3] - yUp[3] - 3*zDown[0] - zDown[3]) + 10*qCurrent[3]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[2]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[2] + xUp[3] + 2*zDown[0] + 2*zDown[2] + zDown[3]) + 8*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = 4*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + qCurrent[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 2*qCurrent[3]*yDown[1] - 2*qCurrent[4]*yDown[1] - 2*xUp[1]*yDown[1] + 2*qCurrent[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*qCurrent[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] - 2*yDown[1]*yDown[3] + 2*qCurrent[4]*yDown[4] + xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + yDown[1]*yDownDerivative[1] - qCurrent[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 2*qCurrent[3]*yUp[1] + 2*qCurrent[4]*yUp[1] + 2*xUp[1]*yUp[1] - 2*qCurrent[2]*yUp[2] + 2*xUp[2]*yUp[2] - 2*qCurrent[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*yUp[1]*yUp[3] - 2*qCurrent[4]*yUp[4] + 2*xUp[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] + 4*qCurrent[2]*zDown[1] + 4*qCurrent[3]*zDown[1] + 2*zDown[0]*zDown[1] + qCurrent[0]*(-4*xUp[1] + 2*yDown[0] + yDown[3] - 2*yUp[0] - yUp[3] + 4*zDown[1]) + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(4*qCurrent[2] - xUp[0] - 2*yDown[1] - yDown[3] + 2*yUp[1] - yUp[3] + zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zDown[0]*zDown[2] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) - 2*(2*(qCurrent[0]*qCurrent[0]) + 2*(qCurrent[1]*qCurrent[1]) + 6*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 2*(qCurrent[4]*qCurrent[4]) - qCurrent[3]*xUp[0] - 4*qCurrent[2]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] + qCurrent[3]*yDown[2] + qCurrent[4]*yDown[2] + qCurrent[3]*yUp[2] - qCurrent[4]*yUp[2] - qCurrent[3]*zDown[0] + 2*xUp[0]*zDown[0] + xUp[3]*zDown[0] + 2*xUp[1]*zDown[1] - qCurrent[1]*(2*xUp[1] + yDown[2] - yUp[2] + 2*zDown[1]) - 4*qCurrent[2]*zDown[2] - 2*qCurrent[3]*zDown[2] + 2*xUp[2]*zDown[2] + qCurrent[0]*(2*qCurrent[3] - 2*xUp[0] + 2*xUp[2] - xUp[3] - 2*zDown[0] - 2*zDown[2] - zDown[3]) - 2*qCurrent[3]*zDown[3] + xUp[0]*zDown[3] + 2*xUp[3]*zDown[3] - 2*qCurrent[4]*zDown[4] + 2*xUp[4]*zDown[4]) + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = -(xUp[0]*xUp[0]) + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zDown[0]*zDown[0] - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - 2*(zDown[4]*zDown[4]) + 2*qCurrent[3]*xUp[0] - 2*xUp[0]*xUp[3] - (xUp[1]*xUpDerivative[5])/2. - xUp[1]*xUpDerivative[8] - (xUp[2]*xUpDerivative[10])/2. - xUp[2]*xUpDerivative[13] + qCurrent[1]*yDown[0] - 2*qCurrent[3]*yDown[0] - qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[1] + 2*qCurrent[1]*yDown[3] - 2*qCurrent[3]*yDown[3] - 2*qCurrent[4]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] - qCurrent[1]*yUp[0] - 2*qCurrent[3]*yUp[0] + qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[1] - 2*qCurrent[1]*yUp[3] - 2*qCurrent[3]*yUp[3] + 2*qCurrent[4]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] + 2*qCurrent[3]*zDown[0] + 4*qCurrent[1]*zDown[1] - qCurrent[0]*(4*qCurrent[2] + xUp[0] + 4*xUp[3] + 2*yDown[0] + 2*yUp[0] - 5*zDown[0] - 5*zDown[3]) - 2*qCurrent[2]*(4*qCurrent[3] - xUp[0] - 2*xUp[3] + yDown[2] + yUp[2] - zDown[0] - 2*zDown[2] - 2*zDown[3]) + 6*qCurrent[3]*zDown[3] + zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + (zDown[2]*zDownDerivative[0])/2. + zDown[2]*zDownDerivative[3] + (zDown[4]*zDownDerivative[5])/2. + zDown[4]*zDownDerivative[8];
        		spatialTerm[4] = -4*qCurrent[0]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - 2*qCurrent[0]*yDown[0] - qCurrent[3]*yDown[0] - 2*qCurrent[1]*yDown[1] - qCurrent[0]*yDown[3] - 2*qCurrent[3]*yDown[3] + 2*qCurrent[1]*yDown[4] - 2*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] + 2*qCurrent[0]*yUp[0] + qCurrent[3]*yUp[0] + 2*qCurrent[1]*yUp[1] + qCurrent[0]*yUp[3] + 2*qCurrent[3]*yUp[3] - 2*qCurrent[1]*yUp[4] - 2*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] + 2*yDown[0]*zDown[0] + yDown[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] + 2*yDown[1]*zDown[1] - 2*yUp[1]*zDown[1] + 2*yDown[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[4]*(xUp[0] + yDown[3] - 2*yDown[4] + yUp[3] + 2*yUp[4] - zDown[0] - zDown[3]) + yDown[0]*zDown[3] + 2*yDown[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 4*qCurrent[0]*zDown[4] + 4*qCurrent[3]*zDown[4] + 2*yDown[4]*zDown[4] - 2*yUp[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + qCurrent[2]*(-8*qCurrent[4] + 4*xUp[4] - 2*yDown[2] + 2*yUp[2] + 4*zDown[4]) + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 20)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yDown[3]*yDown[3] - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - zDown[3]*zDown[3] - 2*(zDown[4]*zDown[4]) - 2*qCurrent[1]*xDown[0] - 2*qCurrent[2]*xDown[0] - 2*qCurrent[1]*xDown[1] - 2*qCurrent[2]*xDown[2] - qCurrent[1]*xDown[3] - qCurrent[2]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. + 2*qCurrent[1]*xUp[0] + 2*qCurrent[2]*xUp[0] - 2*qCurrent[1]*xUp[1] - 2*qCurrent[2]*xUp[2] + qCurrent[1]*xUp[3] + qCurrent[2]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. - 4*qCurrent[4]*yDown[0] - 2*qCurrent[4]*yDown[3] - 2*yDown[0]*yDown[3] + xDownDerivative[10]*yDown[4] + (xDownDerivative[13]*yDown[4])/2. + yDown[1]*yDownDerivative[0] + (yDown[1]*yDownDerivative[3])/2. - 4*qCurrent[4]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] - 2*qCurrent[4]*zDown[3] + zDown[0]*zDown[3] + 2*qCurrent[0]*(4*qCurrent[4] - xDown[0] - xDown[3] - xUp[0] - xUp[3] + yDown[3] + 3*zDown[0] + zDown[3]) + qCurrent[3]*(4*qCurrent[4] - 2*xDown[3] - 2*xUp[3] - 4*yDown[0] - yDown[3] + 5*zDown[0] + 5*zDown[3]) + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + (zDown[2]*zDownDerivative[3])/2. + zDown[4]*zDownDerivative[5] + (zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = -(qCurrent[3]*xDown[0]) - 2*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] - 2*qCurrent[2]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + qCurrent[3]*xUp[0] + 2*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] + 2*qCurrent[2]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 4*qCurrent[3]*yDown[1] - 4*qCurrent[4]*yDown[1] + 2*xDown[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*xDown[2]*yDown[2] - 2*xUp[2]*yDown[2] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] - 2*yDown[1]*yDown[3] + 2*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + yDown[1]*yDownDerivative[1] + 4*qCurrent[3]*zDown[1] - 4*qCurrent[4]*zDown[1] + 2*zDown[0]*zDown[1] + qCurrent[0]*(-2*xDown[0] - 2*xDown[1] - xDown[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 4*zDown[1]) + 2*qCurrent[1]*(4*qCurrent[4] + xDown[0] - 2*xDown[1] + xUp[0] + 2*xUp[1] + yDown[3] - zDown[0] - zDown[3]) + 2*zDown[1]*zDown[3] + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = -(qCurrent[3]*xDown[0]) - 2*qCurrent[1]*xDown[1] - 2*qCurrent[1]*xDown[2] - 2*xDown[0]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + qCurrent[3]*xUp[0] + 2*qCurrent[1]*xUp[1] + 2*qCurrent[1]*xUp[2] - 2*xUp[0]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yDown[2] - 4*qCurrent[4]*yDown[2] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] + 2*xDown[1]*zDown[1] - 2*xUp[1]*zDown[1] + 4*qCurrent[3]*zDown[2] - 4*qCurrent[4]*zDown[2] + 2*xDown[2]*zDown[2] - 2*xUp[2]*zDown[2] + 2*zDown[0]*zDown[2] + qCurrent[0]*(-2*xDown[0] - 2*xDown[2] - xDown[3] + 2*xUp[0] - 2*xUp[2] + xUp[3] + 4*zDown[2]) + 2*qCurrent[2]*(4*qCurrent[4] + xDown[0] - 2*xDown[2] + xUp[0] + 2*xUp[2] + yDown[3] - zDown[0] - zDown[3]) + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*zDown[2]*zDown[3] + 2*xDown[4]*zDown[4] - 2*xUp[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zDown[0]*zDown[0]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 4*(zDown[4]*zDown[4]) - 2*qCurrent[1]*xDown[0] - 2*qCurrent[2]*xDown[0] - 4*qCurrent[1]*xDown[3] - 4*qCurrent[2]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] + 2*qCurrent[1]*xUp[0] + 2*qCurrent[2]*xUp[0] + 4*qCurrent[1]*xUp[3] + 4*qCurrent[2]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 4*qCurrent[4]*yDown[0] - 8*qCurrent[1]*yDown[1] - 8*qCurrent[2]*yDown[2] - 8*qCurrent[4]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] - 4*qCurrent[4]*zDown[0] + 8*qCurrent[1]*zDown[1] + 8*qCurrent[2]*zDown[2] - 8*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[3]*(4*qCurrent[4] + xDown[0] + xUp[0] - 2*yDown[0] - 3*yDown[3] + zDown[0] + 3*zDown[3]) + 2*qCurrent[0]*(4*qCurrent[4] - 2*xDown[3] - 2*xUp[3] - 4*yDown[0] - yDown[3] + 5*zDown[0] + 5*zDown[3]) + 8*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[4] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 12*(qCurrent[4]*qCurrent[4]) + 2*qCurrent[4]*xDown[0] - 2*qCurrent[2]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*xUp[0] + 2*qCurrent[2]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - 2*qCurrent[3]*yDown[0] - 4*qCurrent[2]*yDown[2] - 4*qCurrent[3]*yDown[3] + 2*qCurrent[4]*yDown[3] - 4*qCurrent[3]*yDown[4] - 8*qCurrent[4]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[3]*zDown[0] - 2*qCurrent[4]*zDown[0] + 4*yDown[0]*zDown[0] + 2*yDown[3]*zDown[0] + 4*yDown[1]*zDown[1] - 2*qCurrent[1]*(xDown[4] - xUp[4] + 2*(yDown[1] + zDown[1])) - 4*qCurrent[2]*zDown[2] + 4*yDown[2]*zDown[2] - 4*qCurrent[3]*zDown[3] - 2*qCurrent[4]*zDown[3] + 2*yDown[0]*zDown[3] + 4*yDown[3]*zDown[3] + qCurrent[0]*(4*qCurrent[3] - 2*(xDown[4] + xUp[4] + 2*yDown[0] + yDown[3] + 2*zDown[0] + zDown[3] - 2*zDown[4])) + 4*qCurrent[3]*zDown[4] - 8*qCurrent[4]*zDown[4] + 4*yDown[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 21)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yDown[3]*yDown[3]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 2*(zDown[3]*zDown[3]) - 4*(zDown[4]*zDown[4]) + 8*qCurrent[2]*qCurrent[3] + 8*qCurrent[3]*qCurrent[4] - 8*qCurrent[2]*xDown[0] - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] - 4*qCurrent[2]*xDown[3] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yDown[0] - 8*qCurrent[4]*yDown[0] - 2*qCurrent[3]*yDown[3] - 4*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 8*qCurrent[2]*zDown[0] + 10*qCurrent[3]*zDown[0] - 8*qCurrent[4]*zDown[0] + qCurrent[1]*(8*qCurrent[3] - 4*(2*xDown[0] + 2*xDown[1] + xDown[3] + 2*yDown[0] + yDown[3] - 2*zDown[1])) + 8*qCurrent[2]*zDown[2] - 4*qCurrent[2]*zDown[3] + 10*qCurrent[3]*zDown[3] - 4*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[0]*(4*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[4] - 3*xDown[0] - 2*xDown[3] + yDown[3] + 3*zDown[0] + zDown[3]) + 8*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = 4*(qCurrent[0]*qCurrent[0]) + 12*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) - 2*qCurrent[3]*xDown[0] - 4*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] - 4*qCurrent[2]*xDown[2] - 4*qCurrent[3]*xDown[3] - 4*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] - 2*qCurrent[3]*yDown[0] + 4*xDown[0]*yDown[0] + 2*xDown[3]*yDown[0] - 4*qCurrent[3]*yDown[1] - 4*qCurrent[4]*yDown[1] + 4*xDown[1]*yDown[1] - 4*qCurrent[2]*yDown[2] + 4*xDown[2]*yDown[2] - 4*qCurrent[3]*yDown[3] + 2*xDown[0]*yDown[3] + 4*xDown[3]*yDown[3] - 2*yDown[1]*yDown[3] - 4*qCurrent[4]*yDown[4] + 4*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] + yDown[1]*yDownDerivative[1] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xDown[0] + 2*xDown[1] + xDown[3] + 2*yDown[0] + yDown[3] - 2*zDown[1])) - 4*qCurrent[2]*zDown[1] + 4*qCurrent[3]*zDown[1] - 4*qCurrent[4]*zDown[1] + 2*zDown[0]*zDown[1] + 2*qCurrent[1]*(4*qCurrent[2] + 4*qCurrent[4] + xDown[0] - 4*xDown[1] - 4*yDown[1] + yDown[3] - zDown[0] - zDown[3]) + 2*zDown[1]*zDown[3] + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 12*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) + 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[2]*xDown[0] - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] - 2*xDown[0]*xDown[2] - 4*qCurrent[3]*xDown[3] - 4*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - 4*qCurrent[3]*yDown[2] - 4*qCurrent[4]*yDown[2] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] - 2*qCurrent[2]*zDown[0] - 2*qCurrent[3]*zDown[0] + 4*xDown[0]*zDown[0] + 2*xDown[3]*zDown[0] + 4*xDown[1]*zDown[1] + qCurrent[1]*(8*qCurrent[2] - 4*(xDown[1] + xDown[2] + yDown[2] + zDown[1])) - 8*qCurrent[2]*zDown[2] + 4*qCurrent[3]*zDown[2] - 4*qCurrent[4]*zDown[2] + 4*xDown[2]*zDown[2] + 2*zDown[0]*zDown[2] - 2*qCurrent[2]*zDown[3] - 4*qCurrent[3]*zDown[3] + 2*xDown[0]*zDown[3] + 4*xDown[3]*zDown[3] + 2*zDown[2]*zDown[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xDown[0] + 2*xDown[2] + xDown[3] + 2*zDown[0] - 2*zDown[2] + zDown[3])) - 4*qCurrent[4]*zDown[4] + 4*xDown[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = qCurrent[0]*(4*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[4] - xDown[0] - 4*xDown[3] - 4*yDown[0] - yDown[3] + 5*zDown[0] + 5*zDown[3]) + (-2*(xDown[0]*xDown[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zDown[0]*zDown[0]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 4*(zDown[4]*zDown[4]) + 16*qCurrent[3]*qCurrent[4] + 4*qCurrent[3]*xDown[0] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yDown[0] - 4*qCurrent[4]*yDown[0] - 12*qCurrent[3]*yDown[3] - 8*qCurrent[4]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] + 4*qCurrent[3]*zDown[0] - 4*qCurrent[4]*zDown[0] + 4*qCurrent[1]*(4*qCurrent[3] - xDown[0] - 2*xDown[3] - yDown[0] - 2*yDown[1] - 2*yDown[3] + 2*zDown[1]) + 4*qCurrent[2]*(4*qCurrent[3] - xDown[0] - 2*xDown[3] - 2*yDown[2] - zDown[0] + 2*zDown[2] - 2*zDown[3]) + 12*qCurrent[3]*zDown[3] - 8*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 8*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[4] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 12*(qCurrent[4]*qCurrent[4]) + 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[4]*xDown[0] - 4*qCurrent[2]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] - 2*qCurrent[3]*yDown[0] - 4*qCurrent[2]*yDown[2] - 4*qCurrent[3]*yDown[3] + 2*qCurrent[4]*yDown[3] - 4*qCurrent[3]*yDown[4] - 8*qCurrent[4]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[3]*zDown[0] - 2*qCurrent[4]*zDown[0] + 4*yDown[0]*zDown[0] + 2*yDown[3]*zDown[0] + 4*yDown[1]*zDown[1] + qCurrent[1]*(8*qCurrent[4] - 4*(xDown[4] + yDown[1] + yDown[4] + zDown[1])) - 4*qCurrent[2]*zDown[2] + 4*yDown[2]*zDown[2] - 4*qCurrent[3]*zDown[3] - 2*qCurrent[4]*zDown[3] + 2*yDown[0]*zDown[3] + 4*yDown[3]*zDown[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xDown[4] + 2*yDown[0] + yDown[3] + 2*zDown[0] + zDown[3] - 2*zDown[4])) - 4*qCurrent[2]*zDown[4] + 4*qCurrent[3]*zDown[4] - 8*qCurrent[4]*zDown[4] + 4*yDown[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 22)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 2*(zDown[3]*zDown[3]) - 4*(zDown[4]*zDown[4]) - 8*qCurrent[2]*qCurrent[3] + 8*qCurrent[3]*qCurrent[4] + 8*qCurrent[2]*xUp[0] - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] + 4*qCurrent[2]*xUp[3] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yDown[0] - 8*qCurrent[4]*yDown[0] - 2*qCurrent[3]*yDown[3] - 4*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] + 8*qCurrent[2]*zDown[0] + 10*qCurrent[3]*zDown[0] - 8*qCurrent[4]*zDown[0] + 4*qCurrent[1]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*yDown[0] + yDown[3] + 2*zDown[1]) + 8*qCurrent[2]*zDown[2] + 4*qCurrent[2]*zDown[3] + 10*qCurrent[3]*zDown[3] - 4*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[0]*(-4*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[4] - 3*xUp[0] - 2*xUp[3] + yDown[3] + 3*zDown[0] + zDown[3]) + 8*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xUp[0] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*qCurrent[1]*yDown[3] + xDownDerivative[11]*yDown[4] + yDown[1]*(-2*yDown[3] + yDownDerivative[1]) + 2*zDown[0]*zDown[1] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 6*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) + qCurrent[3]*xUp[0] + 2*qCurrent[2]*xUp[1] + 2*qCurrent[2]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] + qCurrent[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 2*qCurrent[3]*yDown[1] - 2*qCurrent[4]*yDown[1] - 2*xUp[1]*yDown[1] + 4*qCurrent[1]*(-qCurrent[2] + qCurrent[4] + xUp[1] + yDown[1]) + 2*qCurrent[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*qCurrent[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] + 2*qCurrent[4]*yDown[4] - 2*xUp[4]*yDown[4] + 2*qCurrent[2]*zDown[1] + 2*qCurrent[3]*zDown[1] - 2*qCurrent[4]*zDown[1] + qCurrent[0]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*yDown[0] + yDown[3] + 2*zDown[1])) + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*zDown[0]*zDown[2] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 6*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) + 4*qCurrent[2]*qCurrent[4] + qCurrent[3]*xUp[0] + 4*qCurrent[2]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] - 2*qCurrent[3]*yDown[2] - 2*qCurrent[4]*yDown[2] + qCurrent[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] - 2*xUp[1]*zDown[1] + 2*qCurrent[1]*(-2*qCurrent[2] + xUp[1] + xUp[2] + yDown[2] + zDown[1]) + 4*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[2] - 2*qCurrent[4]*zDown[2] - 2*xUp[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + qCurrent[0]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[2] + xUp[3] + 2*zDown[0] + 2*zDown[2] + zDown[3]) + 2*qCurrent[4]*zDown[4] - 2*xUp[4]*zDown[4]) + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = (-2*(xUp[0]*xUp[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zDown[0]*zDown[0]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 4*(zDown[4]*zDown[4]) - 16*qCurrent[2]*qCurrent[3] + 16*qCurrent[3]*qCurrent[4] + 4*qCurrent[2]*xUp[0] + 4*qCurrent[3]*xUp[0] + 8*qCurrent[2]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yDown[0] - 4*qCurrent[4]*yDown[0] - 8*qCurrent[2]*yDown[2] - 12*qCurrent[3]*yDown[3] - 8*qCurrent[4]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] + 4*qCurrent[2]*zDown[0] + 4*qCurrent[3]*zDown[0] - 4*qCurrent[4]*zDown[0] + 4*qCurrent[1]*(-4*qCurrent[3] + xUp[0] + 2*xUp[3] + yDown[0] - 2*yDown[1] + 2*yDown[3] + 2*zDown[1]) + 8*qCurrent[2]*zDown[2] - 2*qCurrent[0]*(4*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[4] + xUp[0] + 4*xUp[3] + 4*yDown[0] + yDown[3] - 5*zDown[0] - 5*zDown[3]) + 8*qCurrent[2]*zDown[3] + 12*qCurrent[3]*zDown[3] - 8*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 8*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[4] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 12*(qCurrent[4]*qCurrent[4]) - 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[4]*xUp[0] + 4*qCurrent[2]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - 2*qCurrent[3]*yDown[0] - 4*qCurrent[2]*yDown[2] - 4*qCurrent[3]*yDown[3] + 2*qCurrent[4]*yDown[3] - 4*qCurrent[3]*yDown[4] - 8*qCurrent[4]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[3]*zDown[0] - 2*qCurrent[4]*zDown[0] + 4*yDown[0]*zDown[0] + 2*yDown[3]*zDown[0] + 4*yDown[1]*zDown[1] - 4*qCurrent[1]*(2*qCurrent[4] - xUp[4] + yDown[1] - yDown[4] + zDown[1]) - 4*qCurrent[2]*zDown[2] + 4*yDown[2]*zDown[2] - 4*qCurrent[3]*zDown[3] - 2*qCurrent[4]*zDown[3] + 2*yDown[0]*zDown[3] + 4*yDown[3]*zDown[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xUp[4] + 2*yDown[0] + yDown[3] + 2*zDown[0] + zDown[3] - 2*zDown[4])) + 4*qCurrent[2]*zDown[4] + 4*qCurrent[3]*zDown[4] - 8*qCurrent[4]*zDown[4] + 4*yDown[4]*zDown[4] + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 24)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yUp[3]*yUp[3] - 2*(zDown[1]*zDown[1]) - 2*(zDown[2]*zDown[2]) - zDown[3]*zDown[3] - 2*(zDown[4]*zDown[4]) + 2*qCurrent[1]*xDown[0] - 2*qCurrent[2]*xDown[0] - 2*qCurrent[1]*xDown[1] - 2*qCurrent[2]*xDown[2] + qCurrent[1]*xDown[3] - qCurrent[2]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. - 2*qCurrent[1]*xUp[0] + 2*qCurrent[2]*xUp[0] - 2*qCurrent[1]*xUp[1] - 2*qCurrent[2]*xUp[2] - qCurrent[1]*xUp[3] + qCurrent[2]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. + 4*qCurrent[4]*yUp[0] + 2*qCurrent[4]*yUp[3] - 2*yUp[0]*yUp[3] - yUp[1]*yUpDerivative[0] - (yUp[1]*yUpDerivative[3])/2. - yUp[4]*yUpDerivative[10] - (yUp[4]*yUpDerivative[13])/2. + 4*qCurrent[4]*zDown[0] + 4*qCurrent[1]*zDown[1] + 4*qCurrent[2]*zDown[2] - qCurrent[3]*(4*qCurrent[4] + 2*xDown[3] + 2*xUp[3] + 4*yUp[0] + yUp[3] - 5*zDown[0] - 5*zDown[3]) - 2*qCurrent[0]*(4*qCurrent[4] + xDown[0] + xDown[3] + xUp[0] + xUp[3] - yUp[3] - 3*zDown[0] - zDown[3]) + 2*qCurrent[4]*zDown[3] + zDown[0]*zDown[3] + 4*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + (zDown[2]*zDownDerivative[3])/2. + zDown[4]*zDownDerivative[5] + (zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = qCurrent[3]*xDown[0] - 2*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] + 2*qCurrent[2]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] - qCurrent[3]*xUp[0] + 2*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] - 2*qCurrent[2]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] + 4*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*xDown[2]*yUp[2] + 2*xUp[2]*yUp[2] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*yUp[1]*yUp[3] - 2*xDown[4]*yUp[4] + 2*xUp[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] + 4*qCurrent[3]*zDown[1] + 4*qCurrent[4]*zDown[1] + 2*zDown[0]*zDown[1] + qCurrent[0]*(2*xDown[0] - 2*xDown[1] + xDown[3] - 2*xUp[0] - 2*xUp[1] - xUp[3] + 4*zDown[1]) + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(4*qCurrent[4] - xDown[0] - 2*xDown[1] - xUp[0] + 2*xUp[1] - yUp[3] + zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = -(qCurrent[3]*xDown[0]) - 2*qCurrent[1]*xDown[1] + 2*qCurrent[1]*xDown[2] - 2*xDown[0]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + qCurrent[3]*xUp[0] + 2*qCurrent[1]*xUp[1] - 2*qCurrent[1]*xUp[2] - 2*xUp[0]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yUp[2] + 4*qCurrent[4]*yUp[2] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*xDown[0]*zDown[0] + xDown[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] + 2*xDown[1]*zDown[1] - 2*xUp[1]*zDown[1] + 4*qCurrent[3]*zDown[2] + 4*qCurrent[4]*zDown[2] + 2*xDown[2]*zDown[2] - 2*xUp[2]*zDown[2] + 2*zDown[0]*zDown[2] + qCurrent[0]*(-2*xDown[0] - 2*xDown[2] - xDown[3] + 2*xUp[0] - 2*xUp[2] + xUp[3] + 4*zDown[2]) + xDown[0]*zDown[3] + 2*xDown[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(4*qCurrent[4] - xDown[0] + 2*xDown[2] - xUp[0] - 2*xUp[2] - yUp[3] + zDown[0] + zDown[3]) + 2*xDown[4]*zDown[4] - 2*xUp[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zDown[0]*zDown[0]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 4*(zDown[4]*zDown[4]) + 2*qCurrent[1]*xDown[0] - 2*qCurrent[2]*xDown[0] + 4*qCurrent[1]*xDown[3] - 4*qCurrent[2]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 2*qCurrent[1]*xUp[0] + 2*qCurrent[2]*xUp[0] - 4*qCurrent[1]*xUp[3] + 4*qCurrent[2]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] + 4*qCurrent[4]*yUp[0] - 8*qCurrent[1]*yUp[1] - 8*qCurrent[2]*yUp[2] + 8*qCurrent[4]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 4*qCurrent[4]*zDown[0] + 8*qCurrent[1]*zDown[1] + 8*qCurrent[2]*zDown[2] - 2*qCurrent[0]*(4*qCurrent[4] + 2*xDown[3] + 2*xUp[3] + 4*yUp[0] + yUp[3] - 5*zDown[0] - 5*zDown[3]) - 4*qCurrent[3]*(4*qCurrent[4] - xDown[0] - xUp[0] + 2*yUp[0] + 3*yUp[3] - zDown[0] - 3*zDown[3]) + 8*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 8*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yUp[3] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[4]*(zDown[0] + zDown[3]) + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] - 2*(2*(qCurrent[0]*qCurrent[0]) + 2*(qCurrent[1]*qCurrent[1]) + 2*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 6*(qCurrent[4]*qCurrent[4]) + qCurrent[2]*xDown[4] - qCurrent[2]*xUp[4] - qCurrent[3]*yUp[0] - 2*qCurrent[2]*yUp[2] - 2*qCurrent[3]*yUp[3] + 2*qCurrent[3]*yUp[4] - 4*qCurrent[4]*yUp[4] - qCurrent[3]*zDown[0] + 2*yUp[0]*zDown[0] + yUp[3]*zDown[0] + 2*yUp[1]*zDown[1] - qCurrent[1]*(xDown[4] - xUp[4] + 2*(yUp[1] + zDown[1])) - 2*qCurrent[2]*zDown[2] + 2*yUp[2]*zDown[2] - 2*qCurrent[3]*zDown[3] + yUp[0]*zDown[3] + 2*yUp[3]*zDown[3] + qCurrent[0]*(2*qCurrent[3] + xDown[4] + xUp[4] - 2*yUp[0] - yUp[3] - 2*zDown[0] - zDown[3] - 2*zDown[4]) - 2*qCurrent[3]*zDown[4] - 4*qCurrent[4]*zDown[4] + 2*yUp[4]*zDown[4]) + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 25)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yUp[3]*yUp[3]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 2*(zDown[3]*zDown[3]) - 4*(zDown[4]*zDown[4]) + 8*qCurrent[2]*qCurrent[3] - 8*qCurrent[3]*qCurrent[4] - 8*qCurrent[2]*xDown[0] - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] - 4*qCurrent[2]*xDown[3] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yUp[0] + 8*qCurrent[4]*yUp[0] - 2*qCurrent[3]*yUp[3] + 4*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] - 8*qCurrent[2]*zDown[0] + 10*qCurrent[3]*zDown[0] + 8*qCurrent[4]*zDown[0] + 4*qCurrent[1]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[1] + xDown[3] + 2*yUp[0] + yUp[3] + 2*zDown[1]) + 8*qCurrent[2]*zDown[2] - 4*qCurrent[2]*zDown[3] + 10*qCurrent[3]*zDown[3] + 4*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[0]*(-4*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[4] - 3*xDown[0] - 2*xDown[3] + yUp[3] + 3*zDown[0] + zDown[3]) + 8*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*yUp[3] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*zDown[0]*zDown[1] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 6*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) + qCurrent[3]*xDown[0] - 2*qCurrent[2]*xDown[1] + 2*qCurrent[2]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + qCurrent[3]*yUp[0] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] - 2*qCurrent[3]*yUp[1] + 2*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 4*qCurrent[1]*(qCurrent[2] - qCurrent[4] + xDown[1] + yUp[1]) + 2*qCurrent[2]*yUp[2] - 2*xDown[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + 2*qCurrent[4]*yUp[4] - 2*xDown[4]*yUp[4] - 2*qCurrent[2]*zDown[1] + 2*qCurrent[3]*zDown[1] + 2*qCurrent[4]*zDown[1] + qCurrent[0]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[1] + xDown[3] + 2*yUp[0] + yUp[3] + 2*zDown[1])) + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 12*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) - 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[2]*xDown[0] - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] - 2*xDown[0]*xDown[2] - 4*qCurrent[3]*xDown[3] - 4*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - 4*qCurrent[3]*yUp[2] + 4*qCurrent[4]*yUp[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] - 2*qCurrent[2]*zDown[0] - 2*qCurrent[3]*zDown[0] + 4*xDown[0]*zDown[0] + 2*xDown[3]*zDown[0] + 4*xDown[1]*zDown[1] - 4*qCurrent[1]*(2*qCurrent[2] + xDown[1] - xDown[2] - yUp[2] + zDown[1]) - 8*qCurrent[2]*zDown[2] + 4*qCurrent[3]*zDown[2] + 4*qCurrent[4]*zDown[2] + 4*xDown[2]*zDown[2] + 2*zDown[0]*zDown[2] - 2*qCurrent[2]*zDown[3] - 4*qCurrent[3]*zDown[3] + 2*xDown[0]*zDown[3] + 4*xDown[3]*zDown[3] + 2*zDown[2]*zDown[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xDown[0] + 2*xDown[2] + xDown[3] + 2*zDown[0] - 2*zDown[2] + zDown[3])) - 4*qCurrent[4]*zDown[4] + 4*xDown[4]*zDown[4] + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zDown[0]*zDown[0]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 4*(zDown[4]*zDown[4]) + 16*qCurrent[2]*qCurrent[3] - 16*qCurrent[3]*qCurrent[4] - 4*qCurrent[2]*xDown[0] + 4*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yUp[0] + 4*qCurrent[4]*yUp[0] - 8*qCurrent[2]*yUp[2] - 12*qCurrent[3]*yUp[3] + 8*qCurrent[4]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] - 4*qCurrent[2]*zDown[0] + 4*qCurrent[3]*zDown[0] + 4*qCurrent[4]*zDown[0] + 4*qCurrent[1]*(-4*qCurrent[3] + xDown[0] + 2*xDown[3] + yUp[0] - 2*yUp[1] + 2*yUp[3] + 2*zDown[1]) + 8*qCurrent[2]*zDown[2] - 2*qCurrent[0]*(4*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[4] + xDown[0] + 4*xDown[3] + 4*yUp[0] + yUp[3] - 5*zDown[0] - 5*zDown[3]) - 8*qCurrent[2]*zDown[3] + 12*qCurrent[3]*zDown[3] + 8*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 8*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*yUp[3] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[4]*(zDown[0] + zDown[3]) + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 6*(qCurrent[4]*qCurrent[4]) + 4*qCurrent[2]*qCurrent[4] - 2*qCurrent[2]*xDown[4] + qCurrent[3]*yUp[0] + 2*qCurrent[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - 2*qCurrent[3]*yUp[4] + 4*qCurrent[4]*yUp[4] + qCurrent[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] - 2*yUp[1]*zDown[1] + 2*qCurrent[1]*(-2*qCurrent[4] + xDown[4] + yUp[1] + yUp[4] + zDown[1]) + 2*qCurrent[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] - 2*qCurrent[2]*zDown[4] + 2*qCurrent[3]*zDown[4] + 4*qCurrent[4]*zDown[4] - 2*yUp[4]*zDown[4] + qCurrent[0]*(-2*qCurrent[3] - 2*xDown[4] + 2*yUp[0] + yUp[3] + 2*zDown[0] + zDown[3] + 2*zDown[4])) + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 26)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yUp[3]*yUp[3]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 2*(zDown[3]*zDown[3]) - 4*(zDown[4]*zDown[4]) - 8*qCurrent[2]*qCurrent[3] - 8*qCurrent[3]*qCurrent[4] + 8*qCurrent[2]*xUp[0] - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] + 4*qCurrent[2]*xUp[3] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yUp[0] + 8*qCurrent[4]*yUp[0] - 2*qCurrent[3]*yUp[3] + 4*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 8*qCurrent[2]*zDown[0] + 10*qCurrent[3]*zDown[0] + 8*qCurrent[4]*zDown[0] + qCurrent[1]*(8*qCurrent[3] - 4*(2*xUp[0] + 2*xUp[1] + xUp[3] + 2*yUp[0] + yUp[3] - 2*zDown[1])) + 8*qCurrent[2]*zDown[2] + 4*qCurrent[2]*zDown[3] + 10*qCurrent[3]*zDown[3] + 4*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[0]*(4*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[4] - 3*xUp[0] - 2*xUp[3] + yUp[3] + 3*zDown[0] + zDown[3]) + 8*qCurrent[4]*zDown[4] + 2*zDown[2]*zDownDerivative[0] + zDown[2]*zDownDerivative[3] + 2*zDown[4]*zDownDerivative[5] + zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[1] = 4*(qCurrent[0]*qCurrent[0]) + 12*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) - 2*qCurrent[3]*xUp[0] + 4*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] - 4*qCurrent[2]*xUp[2] - 4*qCurrent[3]*xUp[3] - 4*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] - 2*qCurrent[3]*yUp[0] + 4*xUp[0]*yUp[0] + 2*xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] + 4*qCurrent[4]*yUp[1] + 4*xUp[1]*yUp[1] - 4*qCurrent[2]*yUp[2] + 4*xUp[2]*yUp[2] - 4*qCurrent[3]*yUp[3] + 2*xUp[0]*yUp[3] + 4*xUp[3]*yUp[3] - 2*yUp[1]*yUp[3] - 4*qCurrent[4]*yUp[4] + 4*xUp[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xUp[0] + 2*xUp[1] + xUp[3] + 2*yUp[0] + yUp[3] - 2*zDown[1])) + 4*qCurrent[2]*zDown[1] + 4*qCurrent[3]*zDown[1] + 4*qCurrent[4]*zDown[1] + 2*zDown[0]*zDown[1] + 2*zDown[1]*zDown[3] - 2*qCurrent[1]*(4*qCurrent[2] + 4*qCurrent[4] - xUp[0] + 4*xUp[1] + 4*yUp[1] - yUp[3] + zDown[0] + zDown[3]) + zDown[2]*zDownDerivative[1] + zDown[4]*zDownDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zDown[0]*zDown[2] + 2*zDown[2]*zDown[3] - 2*qCurrent[2]*(zDown[0] + zDown[3]) + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 6*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) - 4*qCurrent[2]*qCurrent[4] + qCurrent[3]*xUp[0] + 4*qCurrent[2]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] - 2*qCurrent[3]*yUp[2] + 2*qCurrent[4]*yUp[2] + qCurrent[3]*zDown[0] - 2*xUp[0]*zDown[0] - xUp[3]*zDown[0] - 2*xUp[1]*zDown[1] + 2*qCurrent[1]*(2*qCurrent[2] + xUp[1] - xUp[2] - yUp[2] + zDown[1]) + 4*qCurrent[2]*zDown[2] + 2*qCurrent[3]*zDown[2] + 2*qCurrent[4]*zDown[2] - 2*xUp[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - xUp[0]*zDown[3] - 2*xUp[3]*zDown[3] + qCurrent[0]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[2] + xUp[3] + 2*zDown[0] + 2*zDown[2] + zDown[3]) + 2*qCurrent[4]*zDown[4] - 2*xUp[4]*zDown[4]) + zDown[2]*zDownDerivative[2] + zDown[4]*zDownDerivative[7];
        		spatialTerm[3] = qCurrent[0]*(4*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[4] - xUp[0] - 4*xUp[3] - 4*yUp[0] - yUp[3] + 5*zDown[0] + 5*zDown[3]) + (-2*(xUp[0]*xUp[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zDown[0]*zDown[0]) - 4*(zDown[1]*zDown[1]) - 4*(zDown[2]*zDown[2]) - 4*(zDown[4]*zDown[4]) - 16*qCurrent[3]*qCurrent[4] + 4*qCurrent[3]*xUp[0] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yUp[0] + 4*qCurrent[4]*yUp[0] - 12*qCurrent[3]*yUp[3] + 8*qCurrent[4]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 4*qCurrent[3]*zDown[0] + 4*qCurrent[4]*zDown[0] + 4*qCurrent[1]*(4*qCurrent[3] - xUp[0] - 2*xUp[3] - yUp[0] - 2*yUp[1] - 2*yUp[3] + 2*zDown[1]) + 12*qCurrent[3]*zDown[3] + 8*qCurrent[4]*zDown[3] + 2*zDown[0]*zDown[3] + 4*qCurrent[2]*(-4*qCurrent[3] + xUp[0] + 2*xUp[3] - 2*yUp[2] + zDown[0] + 2*zDown[2] + 2*zDown[3]) + 8*qCurrent[4]*zDown[4] + zDown[2]*zDownDerivative[0] + 2*zDown[2]*zDownDerivative[3] + zDown[4]*zDownDerivative[5] + 2*zDown[4]*zDownDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yUp[3] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[4]*(zDown[0] + zDown[3]) + 2*zDown[0]*zDown[4] + 2*zDown[3]*zDown[4] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 6*(qCurrent[4]*qCurrent[4]) - 4*qCurrent[2]*qCurrent[4] + 2*qCurrent[2]*xUp[4] + qCurrent[3]*yUp[0] + 2*qCurrent[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - 2*qCurrent[3]*yUp[4] + 4*qCurrent[4]*yUp[4] + qCurrent[3]*zDown[0] - 2*yUp[0]*zDown[0] - yUp[3]*zDown[0] - 2*yUp[1]*zDown[1] + 2*qCurrent[1]*(2*qCurrent[4] - xUp[4] + yUp[1] - yUp[4] + zDown[1]) + 2*qCurrent[2]*zDown[2] - 2*yUp[2]*zDown[2] + 2*qCurrent[3]*zDown[3] - yUp[0]*zDown[3] - 2*yUp[3]*zDown[3] + 2*qCurrent[2]*zDown[4] + 2*qCurrent[3]*zDown[4] + 4*qCurrent[4]*zDown[4] - 2*yUp[4]*zDown[4] + qCurrent[0]*(-2*qCurrent[3] - 2*xUp[4] + 2*yUp[0] + yUp[3] + 2*zDown[0] + zDown[3] + 2*zDown[4])) + zDown[2]*zDownDerivative[4] + zDown[4]*zDownDerivative[9];
        		};
        	if(boundaryCase == 32)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yDown[3]*yDown[3] - yUp[3]*yUp[3] - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - zUp[3]*zUp[3] - 2*(zUp[4]*zUp[4]) - 2*qCurrent[1]*xDown[1] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. - 2*qCurrent[1]*xUp[1] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. - 2*qCurrent[3]*yDown[0] + 2*qCurrent[4]*yDown[0] + qCurrent[4]*yDown[3] - 2*yDown[0]*yDown[3] + xDownDerivative[10]*yDown[4] + (xDownDerivative[13]*yDown[4])/2. + yDown[1]*yDownDerivative[0] + (yDown[1]*yDownDerivative[3])/2. - 2*qCurrent[3]*yUp[0] - 2*qCurrent[4]*yUp[0] - qCurrent[4]*yUp[3] - 2*yUp[0]*yUp[3] - yUp[1]*yUpDerivative[0] - (yUp[1]*yUpDerivative[3])/2. - yUp[4]*yUpDerivative[10] - (yUp[4]*yUpDerivative[13])/2. + 5*qCurrent[3]*zUp[0] + 4*qCurrent[1]*zUp[1] + qCurrent[2]*(2*xDown[0] - 2*xDown[2] + xDown[3] - 2*xUp[0] - 2*xUp[2] - xUp[3] + 4*zUp[2]) - 2*qCurrent[0]*(xDown[0] + xDown[3] + xUp[0] + xUp[3] - yDown[3] - yUp[3] - 3*zUp[0] - zUp[3]) + 5*qCurrent[3]*zUp[3] + zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - (zUp[2]*zUpDerivative[3])/2. - zUp[4]*zUpDerivative[5] - (zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = (4*qCurrent[2]*xDown[1] - 4*xDown[0]*xDown[1] + 2*xDown[1]*xDownDerivative[6] + 2*xDown[2]*xDownDerivative[11] - 4*qCurrent[2]*xUp[1] - 4*xUp[0]*xUp[1] - 2*xUp[1]*xUpDerivative[6] - 2*xUp[2]*xUpDerivative[11] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 4*qCurrent[3]*yDown[1] + 4*qCurrent[4]*yDown[1] + 2*xDown[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*xDown[2]*yDown[2] - 2*xUp[2]*yDown[2] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] - 4*yDown[1]*yDown[3] + 2*xDown[4]*yDown[4] + 2*xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + 2*yDown[1]*yDownDerivative[1] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] - 4*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*xDown[2]*yUp[2] + 2*xUp[2]*yUp[2] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 4*yUp[1]*yUp[3] - 2*xDown[4]*yUp[4] + 2*xUp[4]*yUp[4] - 2*yUp[1]*yUpDerivative[1] - 2*yUp[4]*yUpDerivative[11] - 4*qCurrent[0]*(xDown[1] + xUp[1] - 2*zUp[1]) + 8*qCurrent[3]*zUp[1] + 4*zUp[0]*zUp[1] + 4*qCurrent[1]*(xDown[0] + xUp[0] + yDown[3] + yUp[3] - zUp[0] - zUp[3]) + 4*zUp[1]*zUp[3] - 2*zUp[2]*zUpDerivative[1] - 2*zUp[4]*zUpDerivative[6])/2.;
        		spatialTerm[2] = qCurrent[3]*xDown[0] + 2*qCurrent[1]*xDown[1] - 2*xDown[0]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - qCurrent[3]*xUp[0] - 2*qCurrent[1]*xUp[1] - 2*xUp[0]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 2*qCurrent[3]*yDown[2] + 2*qCurrent[4]*yDown[2] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] - 2*qCurrent[3]*yUp[2] - 2*qCurrent[4]*yUp[2] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*xUp[1]*zUp[1] + 4*qCurrent[3]*zUp[2] - 2*xDown[2]*zUp[2] + 2*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] + qCurrent[0]*(2*xDown[0] - 2*xDown[2] + xDown[3] - 2*xUp[0] - 2*xUp[2] - xUp[3] + 4*zUp[2]) + 2*qCurrent[2]*(xDown[0] + 2*xDown[2] + xUp[0] - 2*xUp[2] + yDown[3] + yUp[3] - zUp[0] - zUp[3]) - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*xDown[4]*zUp[4] + 2*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = -(xDown[0]*xDown[0]) - xUp[0]*xUp[0] + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zUp[0]*zUp[0] - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) - 2*qCurrent[0]*xDown[3] - 2*xDown[0]*xDown[3] + (xDown[1]*xDownDerivative[5])/2. + xDown[1]*xDownDerivative[8] + (xDown[2]*xDownDerivative[10])/2. + xDown[2]*xDownDerivative[13] - 2*qCurrent[0]*xUp[3] - 2*xUp[0]*xUp[3] - (xUp[1]*xUpDerivative[5])/2. - xUp[1]*xUpDerivative[8] - (xUp[2]*xUpDerivative[10])/2. - xUp[2]*xUpDerivative[13] - 2*qCurrent[0]*yDown[0] + qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[1] + 2*qCurrent[4]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] - 2*qCurrent[0]*yUp[0] - qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[1] - 2*qCurrent[4]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] + 5*qCurrent[0]*zUp[0] + 4*qCurrent[1]*zUp[1] + qCurrent[2]*(xDown[0] + 2*xDown[3] - xUp[0] - 2*xUp[3] - 2*yDown[2] - 2*yUp[2] + 4*zUp[2]) + 5*qCurrent[0]*zUp[3] + zUp[0]*zUp[3] + 2*qCurrent[3]*(xDown[0] + xUp[0] - yDown[0] - yDown[3] - yUp[0] - yUp[3] + zUp[0] + 3*zUp[3]) + 4*qCurrent[4]*zUp[4] - (zUp[2]*zUpDerivative[0])/2. - zUp[2]*zUpDerivative[3] - (zUp[4]*zUpDerivative[5])/2. - zUp[4]*zUpDerivative[8];
        		spatialTerm[4] = 2*qCurrent[2]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] - 2*qCurrent[2]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + qCurrent[3]*yDown[0] + 2*qCurrent[1]*yDown[1] + 2*qCurrent[2]*yDown[2] + 2*qCurrent[3]*yDown[3] - 2*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - qCurrent[3]*yUp[0] - 2*qCurrent[1]*yUp[1] - 2*qCurrent[2]*yUp[2] - 2*qCurrent[3]*yUp[3] - 2*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*yDown[2]*zUp[2] + 2*yUp[2]*zUp[2] + 2*qCurrent[4]*(xDown[0] + xUp[0] + yDown[3] + 2*yDown[4] + yUp[3] - 2*yUp[4] - zUp[0] - zUp[3]) - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] + 4*qCurrent[3]*zUp[4] - 2*yDown[4]*zUp[4] + 2*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + qCurrent[0]*(-2*xDown[4] - 2*xUp[4] + 2*yDown[0] + yDown[3] - 2*yUp[0] - yUp[3] + 4*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 33)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 2*(zUp[3]*zUp[3]) - 4*(zUp[4]*zUp[4]) - 2*qCurrent[3]*xDown[0] - 8*qCurrent[1]*xDown[1] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 4*qCurrent[1]*yDown[0] - 4*qCurrent[3]*yDown[0] + 4*qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[3] + 2*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] + 4*qCurrent[1]*yUp[0] - 4*qCurrent[3]*yUp[0] - 4*qCurrent[4]*yUp[0] + 2*qCurrent[1]*yUp[3] - 2*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 10*qCurrent[3]*zUp[0] + 8*qCurrent[1]*zUp[1] - 4*qCurrent[0]*(4*qCurrent[2] + 3*xDown[0] + 2*xDown[3] - yDown[3] - yUp[3] - 3*zUp[0] - zUp[3]) + 10*qCurrent[3]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[2]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[2] + xDown[3] + 2*zUp[0] + 2*zUp[2] + zUp[3]) + 8*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 4*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] - qCurrent[3]*yDown[0] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*qCurrent[3]*yDown[1] + 2*qCurrent[4]*yDown[1] + 2*xDown[1]*yDown[1] - 2*qCurrent[2]*yDown[2] + 2*xDown[2]*yDown[2] - 2*qCurrent[3]*yDown[3] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - 2*yDown[1]*yDown[3] - 2*qCurrent[4]*yDown[4] + 2*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] + yDown[1]*yDownDerivative[1] + qCurrent[3]*yUp[0] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] - 2*qCurrent[3]*yUp[1] - 2*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 2*qCurrent[2]*yUp[2] - 2*xDown[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] - 2*yUp[1]*yUp[3] + 2*qCurrent[4]*yUp[4] - 2*xDown[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] + 4*qCurrent[2]*zUp[1] + 4*qCurrent[3]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[0]*(-4*xDown[1] - 2*yDown[0] - yDown[3] + 2*yUp[0] + yUp[3] + 4*zUp[1]) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(4*qCurrent[2] - xDown[0] + 2*yDown[1] - yDown[3] - 2*yUp[1] - yUp[3] + zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zUp[0]*zUp[2] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) - 2*(2*(qCurrent[0]*qCurrent[0]) + 2*(qCurrent[1]*qCurrent[1]) + 6*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 2*(qCurrent[4]*qCurrent[4]) - qCurrent[3]*xDown[0] - 4*qCurrent[2]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + qCurrent[3]*yDown[2] - qCurrent[4]*yDown[2] + qCurrent[3]*yUp[2] + qCurrent[4]*yUp[2] - qCurrent[3]*zUp[0] + 2*xDown[0]*zUp[0] + xDown[3]*zUp[0] + 2*xDown[1]*zUp[1] - qCurrent[1]*(2*xDown[1] - yDown[2] + yUp[2] + 2*zUp[1]) - 4*qCurrent[2]*zUp[2] - 2*qCurrent[3]*zUp[2] + 2*xDown[2]*zUp[2] + qCurrent[0]*(2*qCurrent[3] - 2*xDown[0] + 2*xDown[2] - xDown[3] - 2*zUp[0] - 2*zUp[2] - zUp[3]) - 2*qCurrent[3]*zUp[3] + xDown[0]*zUp[3] + 2*xDown[3]*zUp[3] - 2*qCurrent[4]*zUp[4] + 2*xDown[4]*zUp[4]) - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = -(xDown[0]*xDown[0]) + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zUp[0]*zUp[0] - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) + 2*qCurrent[3]*xDown[0] - 2*xDown[0]*xDown[3] + (xDown[1]*xDownDerivative[5])/2. + xDown[1]*xDownDerivative[8] + (xDown[2]*xDownDerivative[10])/2. + xDown[2]*xDownDerivative[13] - qCurrent[1]*yDown[0] - 2*qCurrent[3]*yDown[0] + qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[1] - 2*qCurrent[1]*yDown[3] - 2*qCurrent[3]*yDown[3] + 2*qCurrent[4]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] + qCurrent[1]*yUp[0] - 2*qCurrent[3]*yUp[0] - qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[1] + 2*qCurrent[1]*yUp[3] - 2*qCurrent[3]*yUp[3] - 2*qCurrent[4]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] + 2*qCurrent[3]*zUp[0] + 4*qCurrent[1]*zUp[1] - qCurrent[0]*(4*qCurrent[2] + xDown[0] + 4*xDown[3] + 2*yDown[0] + 2*yUp[0] - 5*zUp[0] - 5*zUp[3]) - 2*qCurrent[2]*(4*qCurrent[3] - xDown[0] - 2*xDown[3] + yDown[2] + yUp[2] - zUp[0] - 2*zUp[2] - 2*zUp[3]) + 6*qCurrent[3]*zUp[3] + zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - (zUp[2]*zUpDerivative[0])/2. - zUp[2]*zUpDerivative[3] - (zUp[4]*zUpDerivative[5])/2. - zUp[4]*zUpDerivative[8];
        		spatialTerm[4] = -4*qCurrent[0]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[0]*yDown[0] + qCurrent[3]*yDown[0] + 2*qCurrent[1]*yDown[1] + qCurrent[0]*yDown[3] + 2*qCurrent[3]*yDown[3] - 2*qCurrent[1]*yDown[4] - 2*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[0]*yUp[0] - qCurrent[3]*yUp[0] - 2*qCurrent[1]*yUp[1] - qCurrent[0]*yUp[3] - 2*qCurrent[3]*yUp[3] + 2*qCurrent[1]*yUp[4] - 2*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*yDown[2]*zUp[2] + 2*yUp[2]*zUp[2] + 2*qCurrent[4]*(xDown[0] + yDown[3] + 2*yDown[4] + yUp[3] - 2*yUp[4] - zUp[0] - zUp[3]) - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] + 4*qCurrent[0]*zUp[4] + 4*qCurrent[3]*zUp[4] - 2*yDown[4]*zUp[4] + 2*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + qCurrent[2]*(-8*qCurrent[4] + 4*xDown[4] + 2*yDown[2] - 2*yUp[2] + 4*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 34)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) - 2*(yUp[3]*yUp[3]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 2*(zUp[3]*zUp[3]) - 4*(zUp[4]*zUp[4]) - 2*qCurrent[3]*xUp[0] - 8*qCurrent[1]*xUp[1] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] + 4*qCurrent[1]*yDown[0] - 4*qCurrent[3]*yDown[0] + 4*qCurrent[4]*yDown[0] + 2*qCurrent[1]*yDown[3] + 2*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 4*qCurrent[1]*yUp[0] - 4*qCurrent[3]*yUp[0] - 4*qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[3] - 2*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 10*qCurrent[3]*zUp[0] + 8*qCurrent[1]*zUp[1] + 10*qCurrent[3]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[0]*(4*qCurrent[2] - 3*xUp[0] - 2*xUp[3] + yDown[3] + yUp[3] + 3*zUp[0] + zUp[3]) + qCurrent[2]*(8*qCurrent[3] - 4*(2*xUp[0] + 2*xUp[2] + xUp[3] + 2*zUp[0] - 2*zUp[2] + zUp[3])) + 8*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = -4*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + qCurrent[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 2*qCurrent[3]*yDown[1] + 2*qCurrent[4]*yDown[1] - 2*xUp[1]*yDown[1] + 2*qCurrent[2]*yDown[2] - 2*xUp[2]*yDown[2] + 2*qCurrent[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] - 2*yDown[1]*yDown[3] + 2*qCurrent[4]*yDown[4] + xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + yDown[1]*yDownDerivative[1] - qCurrent[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 2*qCurrent[3]*yUp[1] - 2*qCurrent[4]*yUp[1] + 2*xUp[1]*yUp[1] - 2*qCurrent[2]*yUp[2] + 2*xUp[2]*yUp[2] - 2*qCurrent[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*yUp[1]*yUp[3] - 2*qCurrent[4]*yUp[4] + 2*xUp[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] - 4*qCurrent[2]*zUp[1] + 4*qCurrent[3]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[0]*(-4*xUp[1] + 2*yDown[0] + yDown[3] - 2*yUp[0] - yUp[3] + 4*zUp[1]) + 2*qCurrent[1]*(4*qCurrent[2] + xUp[0] + 2*yDown[1] + yDown[3] - 2*yUp[1] + yUp[3] - zUp[0] - zUp[3]) + 2*zUp[1]*zUp[3] - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xUp[0] - 2*xUp[0]*xUp[2] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zUp[0]*zUp[2] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + 2*(2*(qCurrent[0]*qCurrent[0]) + 2*(qCurrent[1]*qCurrent[1]) + 6*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 2*(qCurrent[4]*qCurrent[4]) - qCurrent[3]*xUp[0] - 4*qCurrent[2]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - qCurrent[3]*yDown[2] + qCurrent[4]*yDown[2] - qCurrent[3]*yUp[2] - qCurrent[4]*yUp[2] - qCurrent[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] + 2*xUp[1]*zUp[1] - qCurrent[1]*(2*xUp[1] - yDown[2] + yUp[2] + 2*zUp[1]) - 4*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[2] + 2*xUp[2]*zUp[2] + qCurrent[0]*(2*qCurrent[3] - 2*xUp[0] - 2*xUp[2] - xUp[3] - 2*zUp[0] + 2*zUp[2] - zUp[3]) - 2*qCurrent[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] - 2*qCurrent[4]*zUp[4] + 2*xUp[4]*zUp[4]) - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = -(xUp[0]*xUp[0]) + yDown[0]*yDown[0] + yDown[1]*yDown[1] + yDown[2]*yDown[2] - yDown[3]*yDown[3] + yDown[4]*yDown[4] + yUp[0]*yUp[0] + yUp[1]*yUp[1] + yUp[2]*yUp[2] - yUp[3]*yUp[3] + yUp[4]*yUp[4] - zUp[0]*zUp[0] - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - 2*(zUp[4]*zUp[4]) + 2*qCurrent[3]*xUp[0] - 2*xUp[0]*xUp[3] - (xUp[1]*xUpDerivative[5])/2. - xUp[1]*xUpDerivative[8] - (xUp[2]*xUpDerivative[10])/2. - xUp[2]*xUpDerivative[13] + qCurrent[1]*yDown[0] - 2*qCurrent[3]*yDown[0] + qCurrent[4]*yDown[0] - 2*qCurrent[1]*yDown[1] + 2*qCurrent[1]*yDown[3] - 2*qCurrent[3]*yDown[3] + 2*qCurrent[4]*yDown[3] - 2*qCurrent[4]*yDown[4] + (xDownDerivative[10]*yDown[4])/2. + xDownDerivative[13]*yDown[4] + (yDown[1]*yDownDerivative[0])/2. + yDown[1]*yDownDerivative[3] - qCurrent[1]*yUp[0] - 2*qCurrent[3]*yUp[0] - qCurrent[4]*yUp[0] - 2*qCurrent[1]*yUp[1] - 2*qCurrent[1]*yUp[3] - 2*qCurrent[3]*yUp[3] - 2*qCurrent[4]*yUp[3] - 2*qCurrent[4]*yUp[4] - (yUp[1]*yUpDerivative[0])/2. - yUp[1]*yUpDerivative[3] - (yUp[4]*yUpDerivative[10])/2. - yUp[4]*yUpDerivative[13] + 2*qCurrent[3]*zUp[0] + 4*qCurrent[1]*zUp[1] + 6*qCurrent[3]*zUp[3] + zUp[0]*zUp[3] + qCurrent[0]*(4*qCurrent[2] - xUp[0] - 4*xUp[3] - 2*yDown[0] - 2*yUp[0] + 5*zUp[0] + 5*zUp[3]) + qCurrent[2]*(8*qCurrent[3] - 2*(xUp[0] + 2*xUp[3] + yDown[2] + yUp[2] + zUp[0] - 2*zUp[2] + 2*zUp[3])) + 4*qCurrent[4]*zUp[4] - (zUp[2]*zUpDerivative[0])/2. - zUp[2]*zUpDerivative[3] - (zUp[4]*zUpDerivative[5])/2. - zUp[4]*zUpDerivative[8];
        		spatialTerm[4] = -4*qCurrent[0]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[0]*yDown[0] + qCurrent[3]*yDown[0] + 2*qCurrent[1]*yDown[1] + qCurrent[0]*yDown[3] + 2*qCurrent[3]*yDown[3] + 2*qCurrent[1]*yDown[4] - 2*qCurrent[3]*yDown[4] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[0]*yUp[0] - qCurrent[3]*yUp[0] - 2*qCurrent[1]*yUp[1] - qCurrent[0]*yUp[3] - 2*qCurrent[3]*yUp[3] - 2*qCurrent[1]*yUp[4] - 2*qCurrent[3]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] + 2*yUp[0]*zUp[0] + yUp[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*yUp[1]*zUp[1] - 2*yDown[2]*zUp[2] + 2*yUp[2]*zUp[2] + 2*qCurrent[4]*(xUp[0] + yDown[3] + 2*yDown[4] + yUp[3] - 2*yUp[4] - zUp[0] - zUp[3]) - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + yUp[0]*zUp[3] + 2*yUp[3]*zUp[3] + 2*qCurrent[2]*(4*qCurrent[4] - 2*xUp[4] + yDown[2] - yUp[2] - 2*zUp[4]) + 4*qCurrent[0]*zUp[4] + 4*qCurrent[3]*zUp[4] - 2*yDown[4]*zUp[4] + 2*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 36)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yDown[3]*yDown[3] - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - zUp[3]*zUp[3] - 2*(zUp[4]*zUp[4]) - 2*qCurrent[1]*xDown[0] + 2*qCurrent[2]*xDown[0] - 2*qCurrent[1]*xDown[1] - 2*qCurrent[2]*xDown[2] - qCurrent[1]*xDown[3] + qCurrent[2]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. + 2*qCurrent[1]*xUp[0] - 2*qCurrent[2]*xUp[0] - 2*qCurrent[1]*xUp[1] - 2*qCurrent[2]*xUp[2] + qCurrent[1]*xUp[3] - qCurrent[2]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. + 4*qCurrent[4]*yDown[0] + 2*qCurrent[4]*yDown[3] - 2*yDown[0]*yDown[3] + xDownDerivative[10]*yDown[4] + (xDownDerivative[13]*yDown[4])/2. + yDown[1]*yDownDerivative[0] + (yDown[1]*yDownDerivative[3])/2. + 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] - qCurrent[3]*(4*qCurrent[4] + 2*xDown[3] + 2*xUp[3] + 4*yDown[0] + yDown[3] - 5*zUp[0] - 5*zUp[3]) - 2*qCurrent[0]*(4*qCurrent[4] + xDown[0] + xDown[3] + xUp[0] + xUp[3] - yDown[3] - 3*zUp[0] - zUp[3]) + 2*qCurrent[4]*zUp[3] + zUp[0]*zUp[3] + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - (zUp[2]*zUpDerivative[3])/2. - zUp[4]*zUpDerivative[5] - (zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = -(qCurrent[3]*xDown[0]) + 2*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] - 2*qCurrent[2]*xDown[2] - 2*qCurrent[3]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + qCurrent[3]*xUp[0] - 2*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] + 2*qCurrent[2]*xUp[2] + 2*qCurrent[3]*xUp[3] + 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*xDown[0]*yDown[0] + xDown[3]*yDown[0] - 2*xUp[0]*yDown[0] - xUp[3]*yDown[0] - 4*qCurrent[3]*yDown[1] + 4*qCurrent[4]*yDown[1] + 2*xDown[1]*yDown[1] - 2*xUp[1]*yDown[1] + 2*xDown[2]*yDown[2] - 2*xUp[2]*yDown[2] + xDown[0]*yDown[3] + 2*xDown[3]*yDown[3] - xUp[0]*yDown[3] - 2*xUp[3]*yDown[3] - 2*yDown[1]*yDown[3] + 2*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] - 2*xUp[4]*yDown[4] + yDown[1]*yDownDerivative[1] + 4*qCurrent[3]*zUp[1] + 4*qCurrent[4]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[0]*(-2*xDown[0] - 2*xDown[1] - xDown[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 4*zUp[1]) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(4*qCurrent[4] - xDown[0] + 2*xDown[1] - xUp[0] - 2*xUp[1] - yDown[3] + zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = qCurrent[3]*xDown[0] + 2*qCurrent[1]*xDown[1] - 2*qCurrent[1]*xDown[2] - 2*xDown[0]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - qCurrent[3]*xUp[0] - 2*qCurrent[1]*xUp[1] + 2*qCurrent[1]*xUp[2] - 2*xUp[0]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yDown[2] + 4*qCurrent[4]*yDown[2] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*xUp[1]*zUp[1] + 4*qCurrent[3]*zUp[2] + 4*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] + qCurrent[0]*(2*xDown[0] - 2*xDown[2] + xDown[3] - 2*xUp[0] - 2*xUp[2] - xUp[3] + 4*zUp[2]) - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(4*qCurrent[4] - xDown[0] - 2*xDown[2] - xUp[0] + 2*xUp[2] - yDown[3] + zUp[0] + zUp[3]) - 2*xDown[4]*zUp[4] + 2*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zUp[0]*zUp[0]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 4*(zUp[4]*zUp[4]) - 2*qCurrent[1]*xDown[0] + 2*qCurrent[2]*xDown[0] - 4*qCurrent[1]*xDown[3] + 4*qCurrent[2]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] + 2*qCurrent[1]*xUp[0] - 2*qCurrent[2]*xUp[0] + 4*qCurrent[1]*xUp[3] - 4*qCurrent[2]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] + 4*qCurrent[4]*yDown[0] - 8*qCurrent[1]*yDown[1] - 8*qCurrent[2]*yDown[2] + 8*qCurrent[4]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] + 4*qCurrent[4]*zUp[0] + 8*qCurrent[1]*zUp[1] + 8*qCurrent[2]*zUp[2] - 2*qCurrent[0]*(4*qCurrent[4] + 2*xDown[3] + 2*xUp[3] + 4*yDown[0] + yDown[3] - 5*zUp[0] - 5*zUp[3]) - 4*qCurrent[3]*(4*qCurrent[4] - xDown[0] - xUp[0] + 2*yDown[0] + 3*yDown[3] - zUp[0] - 3*zUp[3]) + 8*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 8*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yDown[3] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[4]*(zUp[0] + zUp[3]) + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] - 2*(2*(qCurrent[0]*qCurrent[0]) + 2*(qCurrent[1]*qCurrent[1]) + 2*(qCurrent[2]*qCurrent[2]) + 2*(qCurrent[3]*qCurrent[3]) + 6*(qCurrent[4]*qCurrent[4]) - qCurrent[2]*xDown[4] + qCurrent[2]*xUp[4] - qCurrent[3]*yDown[0] - 2*qCurrent[2]*yDown[2] - 2*qCurrent[3]*yDown[3] + 2*qCurrent[3]*yDown[4] - 4*qCurrent[4]*yDown[4] - qCurrent[3]*zUp[0] + 2*yDown[0]*zUp[0] + yDown[3]*zUp[0] + 2*yDown[1]*zUp[1] + qCurrent[1]*(xDown[4] - xUp[4] - 2*(yDown[1] + zUp[1])) - 2*qCurrent[2]*zUp[2] + 2*yDown[2]*zUp[2] - 2*qCurrent[3]*zUp[3] + yDown[0]*zUp[3] + 2*yDown[3]*zUp[3] + qCurrent[0]*(2*qCurrent[3] + xDown[4] + xUp[4] - 2*yDown[0] - yDown[3] - 2*zUp[0] - zUp[3] - 2*zUp[4]) - 2*qCurrent[3]*zUp[4] - 4*qCurrent[4]*zUp[4] + 2*yDown[4]*zUp[4]) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 37)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yDown[3]*yDown[3]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 2*(zUp[3]*zUp[3]) - 4*(zUp[4]*zUp[4]) - 8*qCurrent[2]*qCurrent[3] - 8*qCurrent[3]*qCurrent[4] + 8*qCurrent[2]*xDown[0] - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] + 4*qCurrent[2]*xDown[3] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yDown[0] + 8*qCurrent[4]*yDown[0] - 2*qCurrent[3]*yDown[3] + 4*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] + 8*qCurrent[2]*zUp[0] + 10*qCurrent[3]*zUp[0] + 8*qCurrent[4]*zUp[0] + qCurrent[1]*(8*qCurrent[3] - 4*(2*xDown[0] + 2*xDown[1] + xDown[3] + 2*yDown[0] + yDown[3] - 2*zUp[1])) + 8*qCurrent[2]*zUp[2] + 4*qCurrent[2]*zUp[3] + 10*qCurrent[3]*zUp[3] + 4*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[0]*(4*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[4] - 3*xDown[0] - 2*xDown[3] + yDown[3] + 3*zUp[0] + zUp[3]) + 8*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 4*(qCurrent[0]*qCurrent[0]) + 12*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) - 2*qCurrent[3]*xDown[0] + 4*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] - 4*qCurrent[2]*xDown[2] - 4*qCurrent[3]*xDown[3] - 4*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] - 2*qCurrent[3]*yDown[0] + 4*xDown[0]*yDown[0] + 2*xDown[3]*yDown[0] - 4*qCurrent[3]*yDown[1] + 4*qCurrent[4]*yDown[1] + 4*xDown[1]*yDown[1] - 4*qCurrent[2]*yDown[2] + 4*xDown[2]*yDown[2] - 4*qCurrent[3]*yDown[3] + 2*xDown[0]*yDown[3] + 4*xDown[3]*yDown[3] - 2*yDown[1]*yDown[3] - 4*qCurrent[4]*yDown[4] + 4*xDown[4]*yDown[4] + xDownDerivative[11]*yDown[4] + yDown[1]*yDownDerivative[1] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xDown[0] + 2*xDown[1] + xDown[3] + 2*yDown[0] + yDown[3] - 2*zUp[1])) + 4*qCurrent[2]*zUp[1] + 4*qCurrent[3]*zUp[1] + 4*qCurrent[4]*zUp[1] + 2*zUp[0]*zUp[1] + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(4*qCurrent[2] + 4*qCurrent[4] - xDown[0] + 4*xDown[1] + 4*yDown[1] - yDown[3] + zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] + 2*zUp[0]*zUp[2] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 6*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) - 4*qCurrent[2]*qCurrent[4] + qCurrent[3]*xDown[0] + 4*qCurrent[2]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] - 2*qCurrent[3]*yDown[2] + 2*qCurrent[4]*yDown[2] + qCurrent[3]*zUp[0] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*qCurrent[1]*(2*qCurrent[2] + xDown[1] - xDown[2] - yDown[2] + zUp[1]) + 4*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[2] + 2*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + qCurrent[0]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[2] + xDown[3] + 2*zUp[0] + 2*zUp[2] + zUp[3]) + 2*qCurrent[4]*zUp[4] - 2*xDown[4]*zUp[4]) - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = qCurrent[0]*(4*qCurrent[1] - 4*qCurrent[2] - 4*qCurrent[4] - xDown[0] - 4*xDown[3] - 4*yDown[0] - yDown[3] + 5*zUp[0] + 5*zUp[3]) + (-2*(xDown[0]*xDown[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zUp[0]*zUp[0]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 4*(zUp[4]*zUp[4]) - 16*qCurrent[3]*qCurrent[4] + 4*qCurrent[3]*xDown[0] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yDown[0] + 4*qCurrent[4]*yDown[0] - 12*qCurrent[3]*yDown[3] + 8*qCurrent[4]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] + 4*qCurrent[3]*zUp[0] + 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(4*qCurrent[3] - xDown[0] - 2*xDown[3] - yDown[0] - 2*yDown[1] - 2*yDown[3] + 2*zUp[1]) + 12*qCurrent[3]*zUp[3] + 8*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[2]*(-4*qCurrent[3] + xDown[0] + 2*xDown[3] - 2*yDown[2] + zUp[0] + 2*zUp[2] + 2*zUp[3]) + 8*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xDown[0] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*yDown[3] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[4]*(zUp[0] + zUp[3]) + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 6*(qCurrent[4]*qCurrent[4]) - 4*qCurrent[2]*qCurrent[4] + 2*qCurrent[2]*xDown[4] + qCurrent[3]*yDown[0] + 2*qCurrent[2]*yDown[2] + 2*qCurrent[3]*yDown[3] - 2*qCurrent[3]*yDown[4] + 4*qCurrent[4]*yDown[4] + qCurrent[3]*zUp[0] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*qCurrent[1]*(2*qCurrent[4] - xDown[4] + yDown[1] - yDown[4] + zUp[1]) + 2*qCurrent[2]*zUp[2] - 2*yDown[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] + 2*qCurrent[2]*zUp[4] + 2*qCurrent[3]*zUp[4] + 4*qCurrent[4]*zUp[4] - 2*yDown[4]*zUp[4] + qCurrent[0]*(-2*qCurrent[3] - 2*xDown[4] + 2*yDown[0] + yDown[3] + 2*zUp[0] + zUp[3] + 2*zUp[4])) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 38)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yDown[3]*yDown[3]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 2*(zUp[3]*zUp[3]) - 4*(zUp[4]*zUp[4]) + 8*qCurrent[2]*qCurrent[3] - 8*qCurrent[3]*qCurrent[4] - 8*qCurrent[2]*xUp[0] - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] - 4*qCurrent[2]*xUp[3] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yDown[0] + 8*qCurrent[4]*yDown[0] - 2*qCurrent[3]*yDown[3] + 4*qCurrent[4]*yDown[3] - 4*yDown[0]*yDown[3] + 2*xDownDerivative[10]*yDown[4] + xDownDerivative[13]*yDown[4] + 2*yDown[1]*yDownDerivative[0] + yDown[1]*yDownDerivative[3] - 8*qCurrent[2]*zUp[0] + 10*qCurrent[3]*zUp[0] + 8*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*yDown[0] + yDown[3] + 2*zUp[1]) + 8*qCurrent[2]*zUp[2] - 4*qCurrent[2]*zUp[3] + 10*qCurrent[3]*zUp[3] + 4*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[0]*(-4*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[4] - 3*xUp[0] - 2*xUp[3] + yDown[3] + 3*zUp[0] + zUp[3]) + 8*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = -4*(qCurrent[0]*qCurrent[0]) - 12*(qCurrent[1]*qCurrent[1]) - 4*(qCurrent[2]*qCurrent[2]) - 4*(qCurrent[3]*qCurrent[3]) - 4*(qCurrent[4]*qCurrent[4]) + 2*qCurrent[3]*xUp[0] - 4*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] + 4*qCurrent[2]*xUp[2] + 4*qCurrent[3]*xUp[3] + 4*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] + 2*qCurrent[3]*yDown[0] - 4*xUp[0]*yDown[0] - 2*xUp[3]*yDown[0] - 4*qCurrent[3]*yDown[1] + 4*qCurrent[4]*yDown[1] - 4*xUp[1]*yDown[1] + 4*qCurrent[2]*yDown[2] - 4*xUp[2]*yDown[2] + 4*qCurrent[3]*yDown[3] - 2*xUp[0]*yDown[3] - 4*xUp[3]*yDown[3] - 2*yDown[1]*yDown[3] + 4*qCurrent[4]*yDown[4] + xDownDerivative[11]*yDown[4] - 4*xUp[4]*yDown[4] + yDown[1]*yDownDerivative[1] - 4*qCurrent[2]*zUp[1] + 4*qCurrent[3]*zUp[1] + 4*qCurrent[4]*zUp[1] + 2*zUp[0]*zUp[1] + 2*qCurrent[0]*(-2*qCurrent[3] + 2*xUp[0] - 2*xUp[1] + xUp[3] + 2*yDown[0] + yDown[3] + 2*zUp[1]) + 2*qCurrent[1]*(4*qCurrent[2] - 4*qCurrent[4] + xUp[0] + 4*xUp[1] + 4*yDown[1] + yDown[3] - zUp[0] - zUp[3]) + 2*zUp[1]*zUp[3] - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 12*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) - 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[2]*xUp[0] - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] - 2*xUp[0]*xUp[2] - 4*qCurrent[3]*xUp[3] - 4*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yDown[2] + 4*qCurrent[4]*yDown[2] + 2*qCurrent[2]*yDown[3] - 2*yDown[2]*yDown[3] + xDownDerivative[12]*yDown[4] + yDown[1]*yDownDerivative[2] - 2*qCurrent[2]*zUp[0] - 2*qCurrent[3]*zUp[0] + 4*xUp[0]*zUp[0] + 2*xUp[3]*zUp[0] + 4*xUp[1]*zUp[1] - 4*qCurrent[1]*(2*qCurrent[2] + xUp[1] - xUp[2] - yDown[2] + zUp[1]) - 8*qCurrent[2]*zUp[2] + 4*qCurrent[3]*zUp[2] + 4*qCurrent[4]*zUp[2] + 4*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] - 2*qCurrent[2]*zUp[3] - 4*qCurrent[3]*zUp[3] + 2*xUp[0]*zUp[3] + 4*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xUp[0] + 2*xUp[2] + xUp[3] + 2*zUp[0] - 2*zUp[2] + zUp[3])) - 4*qCurrent[4]*zUp[4] + 4*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xUp[0]*xUp[0]) + 4*(yDown[0]*yDown[0]) + 4*(yDown[1]*yDown[1]) + 4*(yDown[2]*yDown[2]) + 4*(yDown[4]*yDown[4]) - 2*(zUp[0]*zUp[0]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 4*(zUp[4]*zUp[4]) + 16*qCurrent[2]*qCurrent[3] - 16*qCurrent[3]*qCurrent[4] - 4*qCurrent[2]*xUp[0] + 4*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yDown[0] + 4*qCurrent[4]*yDown[0] - 8*qCurrent[2]*yDown[2] - 12*qCurrent[3]*yDown[3] + 8*qCurrent[4]*yDown[3] + 2*yDown[0]*yDown[3] - 8*qCurrent[4]*yDown[4] + xDownDerivative[10]*yDown[4] + 2*xDownDerivative[13]*yDown[4] + yDown[1]*yDownDerivative[0] + 2*yDown[1]*yDownDerivative[3] - 4*qCurrent[2]*zUp[0] + 4*qCurrent[3]*zUp[0] + 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-4*qCurrent[3] + xUp[0] + 2*xUp[3] + yDown[0] - 2*yDown[1] + 2*yDown[3] + 2*zUp[1]) + 8*qCurrent[2]*zUp[2] - 2*qCurrent[0]*(4*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[4] + xUp[0] + 4*xUp[3] + 4*yDown[0] + yDown[3] - 5*zUp[0] - 5*zUp[3]) - 8*qCurrent[2]*zUp[3] + 12*qCurrent[3]*zUp[3] + 8*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 8*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 2*qCurrent[4]*xUp[0] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] + 2*qCurrent[4]*yDown[3] + xDownDerivative[14]*yDown[4] - 2*yDown[3]*yDown[4] + yDown[1]*yDownDerivative[4] - 2*qCurrent[4]*(zUp[0] + zUp[3]) + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 6*(qCurrent[4]*qCurrent[4]) + 4*qCurrent[2]*qCurrent[4] - 2*qCurrent[2]*xUp[4] + qCurrent[3]*yDown[0] + 2*qCurrent[2]*yDown[2] + 2*qCurrent[3]*yDown[3] - 2*qCurrent[3]*yDown[4] + 4*qCurrent[4]*yDown[4] + qCurrent[3]*zUp[0] - 2*yDown[0]*zUp[0] - yDown[3]*zUp[0] - 2*yDown[1]*zUp[1] + 2*qCurrent[1]*(-2*qCurrent[4] + xUp[4] + yDown[1] + yDown[4] + zUp[1]) + 2*qCurrent[2]*zUp[2] - 2*yDown[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - yDown[0]*zUp[3] - 2*yDown[3]*zUp[3] - 2*qCurrent[2]*zUp[4] + 2*qCurrent[3]*zUp[4] + 4*qCurrent[4]*zUp[4] - 2*yDown[4]*zUp[4] + qCurrent[0]*(-2*qCurrent[3] - 2*xUp[4] + 2*yDown[0] + yDown[3] + 2*zUp[0] + zUp[3] + 2*zUp[4])) - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 40)
        		{
        		spatialTerm[0] = -(xDown[0]*xDown[0]) + xDown[1]*xDown[1] + xDown[2]*xDown[2] + xDown[3]*xDown[3] + xDown[4]*xDown[4] - xUp[0]*xUp[0] + xUp[1]*xUp[1] + xUp[2]*xUp[2] + xUp[3]*xUp[3] + xUp[4]*xUp[4] - yUp[3]*yUp[3] - 2*(zUp[1]*zUp[1]) - 2*(zUp[2]*zUp[2]) - zUp[3]*zUp[3] - 2*(zUp[4]*zUp[4]) + 2*qCurrent[1]*xDown[0] + 2*qCurrent[2]*xDown[0] - 2*qCurrent[1]*xDown[1] - 2*qCurrent[2]*xDown[2] + qCurrent[1]*xDown[3] + qCurrent[2]*xDown[3] - 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[5] + (xDown[1]*xDownDerivative[8])/2. + xDown[2]*xDownDerivative[10] + (xDown[2]*xDownDerivative[13])/2. - 2*qCurrent[1]*xUp[0] - 2*qCurrent[2]*xUp[0] - 2*qCurrent[1]*xUp[1] - 2*qCurrent[2]*xUp[2] - qCurrent[1]*xUp[3] - qCurrent[2]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[5] - (xUp[1]*xUpDerivative[8])/2. - xUp[2]*xUpDerivative[10] - (xUp[2]*xUpDerivative[13])/2. - 4*qCurrent[4]*yUp[0] - 2*qCurrent[4]*yUp[3] - 2*yUp[0]*yUp[3] - yUp[1]*yUpDerivative[0] - (yUp[1]*yUpDerivative[3])/2. - yUp[4]*yUpDerivative[10] - (yUp[4]*yUpDerivative[13])/2. - 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*zUp[1] + 4*qCurrent[2]*zUp[2] - 2*qCurrent[4]*zUp[3] + zUp[0]*zUp[3] + 2*qCurrent[0]*(4*qCurrent[4] - xDown[0] - xDown[3] - xUp[0] - xUp[3] + yUp[3] + 3*zUp[0] + zUp[3]) + qCurrent[3]*(4*qCurrent[4] - 2*xDown[3] - 2*xUp[3] - 4*yUp[0] - yUp[3] + 5*zUp[0] + 5*zUp[3]) + 4*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - (zUp[2]*zUpDerivative[3])/2. - zUp[4]*zUpDerivative[5] - (zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = qCurrent[3]*xDown[0] + 2*qCurrent[2]*xDown[1] - 2*xDown[0]*xDown[1] + 2*qCurrent[2]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] - qCurrent[3]*xUp[0] - 2*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] - 2*qCurrent[2]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] + 2*xUp[0]*yUp[0] + xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] - 4*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 2*xUp[1]*yUp[1] - 2*xDown[2]*yUp[2] + 2*xUp[2]*yUp[2] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + xUp[0]*yUp[3] + 2*xUp[3]*yUp[3] - 2*yUp[1]*yUp[3] - 2*xDown[4]*yUp[4] + 2*xUp[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] + 4*qCurrent[3]*zUp[1] - 4*qCurrent[4]*zUp[1] + 2*zUp[0]*zUp[1] + qCurrent[0]*(2*xDown[0] - 2*xDown[1] + xDown[3] - 2*xUp[0] - 2*xUp[1] - xUp[3] + 4*zUp[1]) + 2*qCurrent[1]*(4*qCurrent[4] + xDown[0] + 2*xDown[1] + xUp[0] - 2*xUp[1] + yUp[3] - zUp[0] - zUp[3]) + 2*zUp[1]*zUp[3] - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = qCurrent[3]*xDown[0] + 2*qCurrent[1]*xDown[1] + 2*qCurrent[1]*xDown[2] - 2*xDown[0]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] - qCurrent[3]*xUp[0] - 2*qCurrent[1]*xUp[1] - 2*qCurrent[1]*xUp[2] - 2*xUp[0]*xUp[2] - 2*qCurrent[3]*xUp[3] - 2*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yUp[2] - 4*qCurrent[4]*yUp[2] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] + 2*xUp[0]*zUp[0] + xUp[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*xUp[1]*zUp[1] + 4*qCurrent[3]*zUp[2] - 4*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] + qCurrent[0]*(2*xDown[0] - 2*xDown[2] + xDown[3] - 2*xUp[0] - 2*xUp[2] - xUp[3] + 4*zUp[2]) + 2*qCurrent[2]*(4*qCurrent[4] + xDown[0] + 2*xDown[2] + xUp[0] - 2*xUp[2] + yUp[3] - zUp[0] - zUp[3]) - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + xUp[0]*zUp[3] + 2*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] - 2*xDown[4]*zUp[4] + 2*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) - 2*(xUp[0]*xUp[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zUp[0]*zUp[0]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 4*(zUp[4]*zUp[4]) + 2*qCurrent[1]*xDown[0] + 2*qCurrent[2]*xDown[0] + 4*qCurrent[1]*xDown[3] + 4*qCurrent[2]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 2*qCurrent[1]*xUp[0] - 2*qCurrent[2]*xUp[0] - 4*qCurrent[1]*xUp[3] - 4*qCurrent[2]*xUp[3] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 4*qCurrent[4]*yUp[0] - 8*qCurrent[1]*yUp[1] - 8*qCurrent[2]*yUp[2] - 8*qCurrent[4]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] - 4*qCurrent[4]*zUp[0] + 8*qCurrent[1]*zUp[1] + 8*qCurrent[2]*zUp[2] - 8*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[3]*(4*qCurrent[4] + xDown[0] + xUp[0] - 2*yUp[0] - 3*yUp[3] + zUp[0] + 3*zUp[3]) + 2*qCurrent[0]*(4*qCurrent[4] - 2*xDown[3] - 2*xUp[3] - 4*yUp[0] - yUp[3] + 5*zUp[0] + 5*zUp[3]) + 8*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 12*(qCurrent[4]*qCurrent[4]) + 2*qCurrent[4]*xDown[0] + 2*qCurrent[2]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] + 2*qCurrent[4]*xUp[0] - 2*qCurrent[2]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - 2*qCurrent[3]*yUp[0] - 4*qCurrent[2]*yUp[2] - 4*qCurrent[3]*yUp[3] + 2*qCurrent[4]*yUp[3] - 4*qCurrent[3]*yUp[4] - 8*qCurrent[4]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[3]*zUp[0] - 2*qCurrent[4]*zUp[0] + 4*yUp[0]*zUp[0] + 2*yUp[3]*zUp[0] + 4*yUp[1]*zUp[1] + 2*qCurrent[1]*(xDown[4] - xUp[4] - 2*(yUp[1] + zUp[1])) - 4*qCurrent[2]*zUp[2] + 4*yUp[2]*zUp[2] - 4*qCurrent[3]*zUp[3] - 2*qCurrent[4]*zUp[3] + 2*yUp[0]*zUp[3] + 4*yUp[3]*zUp[3] + qCurrent[0]*(4*qCurrent[3] - 2*(xDown[4] + xUp[4] + 2*yUp[0] + yUp[3] + 2*zUp[0] + zUp[3] - 2*zUp[4])) + 4*qCurrent[3]*zUp[4] - 8*qCurrent[4]*zUp[4] + 4*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 41)
        		{
        		spatialTerm[0] = (4*(xDown[1]*xDown[1]) + 4*(xDown[2]*xDown[2]) + 4*(xDown[3]*xDown[3]) + 4*(xDown[4]*xDown[4]) - 2*(yUp[3]*yUp[3]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 2*(zUp[3]*zUp[3]) - 4*(zUp[4]*zUp[4]) - 8*qCurrent[2]*qCurrent[3] + 8*qCurrent[3]*qCurrent[4] + 8*qCurrent[2]*xDown[0] - 2*qCurrent[3]*xDown[0] - 8*qCurrent[2]*xDown[2] + 4*qCurrent[2]*xDown[3] - 8*qCurrent[3]*xDown[3] + 2*xDown[0]*xDown[3] - 8*qCurrent[4]*xDown[4] + 2*xDown[1]*xDownDerivative[5] + xDown[1]*xDownDerivative[8] + 2*xDown[2]*xDownDerivative[10] + xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yUp[0] - 8*qCurrent[4]*yUp[0] - 2*qCurrent[3]*yUp[3] - 4*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] + 8*qCurrent[2]*zUp[0] + 10*qCurrent[3]*zUp[0] - 8*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[1] + xDown[3] + 2*yUp[0] + yUp[3] + 2*zUp[1]) + 8*qCurrent[2]*zUp[2] + 4*qCurrent[2]*zUp[3] + 10*qCurrent[3]*zUp[3] - 4*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[0]*(-4*qCurrent[1] - 4*qCurrent[2] + 4*qCurrent[4] - 3*xDown[0] - 2*xDown[3] + yUp[3] + 3*zUp[0] + zUp[3]) + 8*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 2*qCurrent[1]*xDown[0] - 2*xDown[0]*xDown[1] + xDown[1]*xDownDerivative[6] + xDown[2]*xDownDerivative[11] + 2*qCurrent[1]*yUp[3] - yUp[1]*(2*yUp[3] + yUpDerivative[1]) - yUp[4]*yUpDerivative[11] + 2*zUp[0]*zUp[1] + 2*(-2*(qCurrent[0]*qCurrent[0]) - 6*(qCurrent[1]*qCurrent[1]) - 2*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) + qCurrent[3]*xDown[0] + 2*qCurrent[2]*xDown[1] + 2*qCurrent[2]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] + qCurrent[3]*yUp[0] - 2*xDown[0]*yUp[0] - xDown[3]*yUp[0] - 2*qCurrent[3]*yUp[1] - 2*qCurrent[4]*yUp[1] - 2*xDown[1]*yUp[1] + 4*qCurrent[1]*(-qCurrent[2] + qCurrent[4] + xDown[1] + yUp[1]) + 2*qCurrent[2]*yUp[2] - 2*xDown[2]*yUp[2] + 2*qCurrent[3]*yUp[3] - xDown[0]*yUp[3] - 2*xDown[3]*yUp[3] + 2*qCurrent[4]*yUp[4] - 2*xDown[4]*yUp[4] + 2*qCurrent[2]*zUp[1] + 2*qCurrent[3]*zUp[1] - 2*qCurrent[4]*zUp[1] + qCurrent[0]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[1] + xDown[3] + 2*yUp[0] + yUp[3] + 2*zUp[1])) + 2*zUp[1]*zUp[3] - 2*qCurrent[1]*(zUp[0] + zUp[3]) - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 2*qCurrent[2]*xDown[0] - 2*xDown[0]*xDown[2] + xDown[1]*xDownDerivative[7] + xDown[2]*xDownDerivative[12] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] + 2*zUp[0]*zUp[2] + 2*zUp[2]*zUp[3] - 2*qCurrent[2]*(zUp[0] + zUp[3]) + 2*(-2*(qCurrent[0]*qCurrent[0]) - 2*(qCurrent[1]*qCurrent[1]) - 6*(qCurrent[2]*qCurrent[2]) - 2*(qCurrent[3]*qCurrent[3]) - 2*(qCurrent[4]*qCurrent[4]) + 4*qCurrent[2]*qCurrent[4] + qCurrent[3]*xDown[0] + 4*qCurrent[2]*xDown[2] + 2*qCurrent[3]*xDown[3] + 2*qCurrent[4]*xDown[4] - 2*qCurrent[3]*yUp[2] - 2*qCurrent[4]*yUp[2] + qCurrent[3]*zUp[0] - 2*xDown[0]*zUp[0] - xDown[3]*zUp[0] - 2*xDown[1]*zUp[1] + 2*qCurrent[1]*(-2*qCurrent[2] + xDown[1] + xDown[2] + yUp[2] + zUp[1]) + 4*qCurrent[2]*zUp[2] + 2*qCurrent[3]*zUp[2] - 2*qCurrent[4]*zUp[2] - 2*xDown[2]*zUp[2] + 2*qCurrent[3]*zUp[3] - xDown[0]*zUp[3] - 2*xDown[3]*zUp[3] + qCurrent[0]*(-2*qCurrent[3] + 2*xDown[0] - 2*xDown[2] + xDown[3] + 2*zUp[0] + 2*zUp[2] + zUp[3]) + 2*qCurrent[4]*zUp[4] - 2*xDown[4]*zUp[4]) - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = (-2*(xDown[0]*xDown[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zUp[0]*zUp[0]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 4*(zUp[4]*zUp[4]) - 16*qCurrent[2]*qCurrent[3] + 16*qCurrent[3]*qCurrent[4] + 4*qCurrent[2]*xDown[0] + 4*qCurrent[3]*xDown[0] + 8*qCurrent[2]*xDown[3] - 4*xDown[0]*xDown[3] + xDown[1]*xDownDerivative[5] + 2*xDown[1]*xDownDerivative[8] + xDown[2]*xDownDerivative[10] + 2*xDown[2]*xDownDerivative[13] - 8*qCurrent[3]*yUp[0] - 4*qCurrent[4]*yUp[0] - 8*qCurrent[2]*yUp[2] - 12*qCurrent[3]*yUp[3] - 8*qCurrent[4]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 4*qCurrent[2]*zUp[0] + 4*qCurrent[3]*zUp[0] - 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(-4*qCurrent[3] + xDown[0] + 2*xDown[3] + yUp[0] - 2*yUp[1] + 2*yUp[3] + 2*zUp[1]) + 8*qCurrent[2]*zUp[2] - 2*qCurrent[0]*(4*qCurrent[1] + 4*qCurrent[2] - 4*qCurrent[4] + xDown[0] + 4*xDown[3] + 4*yUp[0] + yUp[3] - 5*zUp[0] - 5*zUp[3]) + 8*qCurrent[2]*zUp[3] + 12*qCurrent[3]*zUp[3] - 8*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 8*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 12*(qCurrent[4]*qCurrent[4]) - 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[4]*xDown[0] + 4*qCurrent[2]*xDown[4] - 2*xDown[0]*xDown[4] + xDown[1]*xDownDerivative[9] + xDown[2]*xDownDerivative[14] - 2*qCurrent[3]*yUp[0] - 4*qCurrent[2]*yUp[2] - 4*qCurrent[3]*yUp[3] + 2*qCurrent[4]*yUp[3] - 4*qCurrent[3]*yUp[4] - 8*qCurrent[4]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[3]*zUp[0] - 2*qCurrent[4]*zUp[0] + 4*yUp[0]*zUp[0] + 2*yUp[3]*zUp[0] + 4*yUp[1]*zUp[1] - 4*qCurrent[1]*(2*qCurrent[4] - xDown[4] + yUp[1] - yUp[4] + zUp[1]) - 4*qCurrent[2]*zUp[2] + 4*yUp[2]*zUp[2] - 4*qCurrent[3]*zUp[3] - 2*qCurrent[4]*zUp[3] + 2*yUp[0]*zUp[3] + 4*yUp[3]*zUp[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xDown[4] + 2*yUp[0] + yUp[3] + 2*zUp[0] + zUp[3] - 2*zUp[4])) + 4*qCurrent[2]*zUp[4] + 4*qCurrent[3]*zUp[4] - 8*qCurrent[4]*zUp[4] + 4*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};
        	if(boundaryCase == 42)
        		{
        		spatialTerm[0] = (4*(xUp[1]*xUp[1]) + 4*(xUp[2]*xUp[2]) + 4*(xUp[3]*xUp[3]) + 4*(xUp[4]*xUp[4]) - 2*(yUp[3]*yUp[3]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 2*(zUp[3]*zUp[3]) - 4*(zUp[4]*zUp[4]) + 8*qCurrent[2]*qCurrent[3] + 8*qCurrent[3]*qCurrent[4] - 8*qCurrent[2]*xUp[0] - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] - 4*qCurrent[2]*xUp[3] - 8*qCurrent[3]*xUp[3] + 2*xUp[0]*xUp[3] - 8*qCurrent[4]*xUp[4] - 2*xUp[1]*xUpDerivative[5] - xUp[1]*xUpDerivative[8] - 2*xUp[2]*xUpDerivative[10] - xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yUp[0] - 8*qCurrent[4]*yUp[0] - 2*qCurrent[3]*yUp[3] - 4*qCurrent[4]*yUp[3] - 4*yUp[0]*yUp[3] - 2*yUp[1]*yUpDerivative[0] - yUp[1]*yUpDerivative[3] - 2*yUp[4]*yUpDerivative[10] - yUp[4]*yUpDerivative[13] - 8*qCurrent[2]*zUp[0] + 10*qCurrent[3]*zUp[0] - 8*qCurrent[4]*zUp[0] + qCurrent[1]*(8*qCurrent[3] - 4*(2*xUp[0] + 2*xUp[1] + xUp[3] + 2*yUp[0] + yUp[3] - 2*zUp[1])) + 8*qCurrent[2]*zUp[2] - 4*qCurrent[2]*zUp[3] + 10*qCurrent[3]*zUp[3] - 4*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 4*qCurrent[0]*(4*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[4] - 3*xUp[0] - 2*xUp[3] + yUp[3] + 3*zUp[0] + zUp[3]) + 8*qCurrent[4]*zUp[4] - 2*zUp[2]*zUpDerivative[0] - zUp[2]*zUpDerivative[3] - 2*zUp[4]*zUpDerivative[5] - zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[1] = 4*(qCurrent[0]*qCurrent[0]) + 12*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) - 2*qCurrent[3]*xUp[0] - 4*qCurrent[2]*xUp[1] - 2*xUp[0]*xUp[1] - 4*qCurrent[2]*xUp[2] - 4*qCurrent[3]*xUp[3] - 4*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[6] - xUp[2]*xUpDerivative[11] - 2*qCurrent[3]*yUp[0] + 4*xUp[0]*yUp[0] + 2*xUp[3]*yUp[0] - 4*qCurrent[3]*yUp[1] - 4*qCurrent[4]*yUp[1] + 4*xUp[1]*yUp[1] - 4*qCurrent[2]*yUp[2] + 4*xUp[2]*yUp[2] - 4*qCurrent[3]*yUp[3] + 2*xUp[0]*yUp[3] + 4*xUp[3]*yUp[3] - 2*yUp[1]*yUp[3] - 4*qCurrent[4]*yUp[4] + 4*xUp[4]*yUp[4] - yUp[1]*yUpDerivative[1] - yUp[4]*yUpDerivative[11] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xUp[0] + 2*xUp[1] + xUp[3] + 2*yUp[0] + yUp[3] - 2*zUp[1])) - 4*qCurrent[2]*zUp[1] + 4*qCurrent[3]*zUp[1] - 4*qCurrent[4]*zUp[1] + 2*zUp[0]*zUp[1] + 2*qCurrent[1]*(4*qCurrent[2] + 4*qCurrent[4] + xUp[0] - 4*xUp[1] - 4*yUp[1] + yUp[3] - zUp[0] - zUp[3]) + 2*zUp[1]*zUp[3] - zUp[2]*zUpDerivative[1] - zUp[4]*zUpDerivative[6];
        		spatialTerm[2] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 12*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 4*(qCurrent[4]*qCurrent[4]) + 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[2]*xUp[0] - 2*qCurrent[3]*xUp[0] - 8*qCurrent[2]*xUp[2] - 2*xUp[0]*xUp[2] - 4*qCurrent[3]*xUp[3] - 4*qCurrent[4]*xUp[4] - xUp[1]*xUpDerivative[7] - xUp[2]*xUpDerivative[12] - 4*qCurrent[3]*yUp[2] - 4*qCurrent[4]*yUp[2] + 2*qCurrent[2]*yUp[3] - 2*yUp[2]*yUp[3] - yUp[1]*yUpDerivative[2] - yUp[4]*yUpDerivative[12] - 2*qCurrent[2]*zUp[0] - 2*qCurrent[3]*zUp[0] + 4*xUp[0]*zUp[0] + 2*xUp[3]*zUp[0] + 4*xUp[1]*zUp[1] + qCurrent[1]*(8*qCurrent[2] - 4*(xUp[1] + xUp[2] + yUp[2] + zUp[1])) - 8*qCurrent[2]*zUp[2] + 4*qCurrent[3]*zUp[2] - 4*qCurrent[4]*zUp[2] + 4*xUp[2]*zUp[2] + 2*zUp[0]*zUp[2] - 2*qCurrent[2]*zUp[3] - 4*qCurrent[3]*zUp[3] + 2*xUp[0]*zUp[3] + 4*xUp[3]*zUp[3] + 2*zUp[2]*zUp[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xUp[0] + 2*xUp[2] + xUp[3] + 2*zUp[0] - 2*zUp[2] + zUp[3])) - 4*qCurrent[4]*zUp[4] + 4*xUp[4]*zUp[4] - zUp[2]*zUpDerivative[2] - zUp[4]*zUpDerivative[7];
        		spatialTerm[3] = qCurrent[0]*(4*qCurrent[1] + 4*qCurrent[2] + 4*qCurrent[4] - xUp[0] - 4*xUp[3] - 4*yUp[0] - yUp[3] + 5*zUp[0] + 5*zUp[3]) + (-2*(xUp[0]*xUp[0]) + 4*(yUp[0]*yUp[0]) + 4*(yUp[1]*yUp[1]) + 4*(yUp[2]*yUp[2]) + 4*(yUp[4]*yUp[4]) - 2*(zUp[0]*zUp[0]) - 4*(zUp[1]*zUp[1]) - 4*(zUp[2]*zUp[2]) - 4*(zUp[4]*zUp[4]) + 16*qCurrent[3]*qCurrent[4] + 4*qCurrent[3]*xUp[0] - 4*xUp[0]*xUp[3] - xUp[1]*xUpDerivative[5] - 2*xUp[1]*xUpDerivative[8] - xUp[2]*xUpDerivative[10] - 2*xUp[2]*xUpDerivative[13] - 8*qCurrent[3]*yUp[0] - 4*qCurrent[4]*yUp[0] - 12*qCurrent[3]*yUp[3] - 8*qCurrent[4]*yUp[3] + 2*yUp[0]*yUp[3] - 8*qCurrent[4]*yUp[4] - yUp[1]*yUpDerivative[0] - 2*yUp[1]*yUpDerivative[3] - yUp[4]*yUpDerivative[10] - 2*yUp[4]*yUpDerivative[13] + 4*qCurrent[3]*zUp[0] - 4*qCurrent[4]*zUp[0] + 4*qCurrent[1]*(4*qCurrent[3] - xUp[0] - 2*xUp[3] - yUp[0] - 2*yUp[1] - 2*yUp[3] + 2*zUp[1]) + 4*qCurrent[2]*(4*qCurrent[3] - xUp[0] - 2*xUp[3] - 2*yUp[2] - zUp[0] + 2*zUp[2] - 2*zUp[3]) + 12*qCurrent[3]*zUp[3] - 8*qCurrent[4]*zUp[3] + 2*zUp[0]*zUp[3] + 8*qCurrent[4]*zUp[4] - zUp[2]*zUpDerivative[0] - 2*zUp[2]*zUpDerivative[3] - zUp[4]*zUpDerivative[5] - 2*zUp[4]*zUpDerivative[8])/2.;
        		spatialTerm[4] = 4*(qCurrent[0]*qCurrent[0]) + 4*(qCurrent[1]*qCurrent[1]) + 4*(qCurrent[2]*qCurrent[2]) + 4*(qCurrent[3]*qCurrent[3]) + 12*(qCurrent[4]*qCurrent[4]) + 8*qCurrent[2]*qCurrent[4] + 2*qCurrent[4]*xUp[0] - 4*qCurrent[2]*xUp[4] - 2*xUp[0]*xUp[4] - xUp[1]*xUpDerivative[9] - xUp[2]*xUpDerivative[14] - 2*qCurrent[3]*yUp[0] - 4*qCurrent[2]*yUp[2] - 4*qCurrent[3]*yUp[3] + 2*qCurrent[4]*yUp[3] - 4*qCurrent[3]*yUp[4] - 8*qCurrent[4]*yUp[4] - 2*yUp[3]*yUp[4] - yUp[1]*yUpDerivative[4] - yUp[4]*yUpDerivative[14] - 2*qCurrent[3]*zUp[0] - 2*qCurrent[4]*zUp[0] + 4*yUp[0]*zUp[0] + 2*yUp[3]*zUp[0] + 4*yUp[1]*zUp[1] + qCurrent[1]*(8*qCurrent[4] - 4*(xUp[4] + yUp[1] + yUp[4] + zUp[1])) - 4*qCurrent[2]*zUp[2] + 4*yUp[2]*zUp[2] - 4*qCurrent[3]*zUp[3] - 2*qCurrent[4]*zUp[3] + 2*yUp[0]*zUp[3] + 4*yUp[3]*zUp[3] + qCurrent[0]*(4*qCurrent[3] - 2*(2*xUp[4] + 2*yUp[0] + yUp[3] + 2*zUp[0] + zUp[3] - 2*zUp[4])) - 4*qCurrent[2]*zUp[4] + 4*qCurrent[3]*zUp[4] - 8*qCurrent[4]*zUp[4] + 4*yUp[4]*zUp[4] + 2*zUp[0]*zUp[4] + 2*zUp[3]*zUp[4] - zUp[2]*zUpDerivative[4] - zUp[4]*zUpDerivative[9];
        		};

        spatialTerm = (0.5*L6)*spatialTerm;
        };

    }//end namespace
#undef HOSTDEVICE
#endif
