********************************************************************
*
*Paper : Parental Disabilty and Teenagers' Time Allocation
*
*File name :  atus.do
*
*This file creates variables for the number of people who are 
*with you during an activity and merges onto the activity file
*
*Author - Sabrina Wulff Pabilonia
*
*Last Updated : May 24, 2021
********************************************************************

# delimit cr

version 17.0 

clear 
clear matrix 
cap log close 
set more off  

* Define the directory where original ATUS data is stored
global ATUSPATH "A:/ATUS"

*define the work file directory for the project
global PATH "X:/Charlene teens parentdisability"

*log file
log using "$PATH/Log/atus.log", replace

*loop over all years of data: 2003-2019
forvalues YYYY = 2003/2019{

*****************
*new year of data
*****************
display `YYYY'

/*
*atus who with during an activity - lists the ATUS respondent, each activity, and the line number from roster of who with (multiple observations per activity-person combo with a separate observation for different persons who were with them)
*value of -1 is that the activity didn't ask who they were with so there will be no line number for the person (example personal care, sleep, and work - work was asked in 2011 forward) and a 1 is they were alone
*/

use "$ATUSPATH/atuswho_`YYYY'.dta", clear
*create a variable that indicates a unique person and activity
egen idnum = group(tucaseid tuactivity_n)
*calculate the most people someone is with in each activity 
sort idnum
by idnum: gen byte people = _N

tab people
drop people

*create variables that list the line numbers on the roster of the persons you are with during an activity
for num 1/19: gen byte linenoX = 0

	forvalues i = 1/19 {
	bys idnum: replace lineno`i' = tulineno[`i'] if _n==1
	}
	
*only keep the first observation line for each activity
by idnum: drop if _n~=1
*drop variables used to create who with variables
drop idnum tulineno tuwho_code trwhona

destring tucaseid, replace
sort tucaseid tuactivity_n
save "$PATH/Data/whowithactivity_`YYYY'.dta", replace

*****************
*ATUS roster file
*****************
use "$ATUSPATH/atusrost_`YYYY'.dta", clear

*get the relationship, gender and age of each person in the household to the ATUS respondent
keep tucaseid tulineno teage terrp tesex

*how many people are in the household - up to 19
*Always check with new year of data
tab tulineno

sort tucaseid tulineno
*how many listed on household roster
by tucaseid: egen numHHroster = max(tulineno)

*exclude non-HH children under age 18 listed on roster from the number of HH members
gen byte HHmem = 1
replace HHmem = 0 if terrp==40
by tucaseid: egen byte numHHmem = total(HHmem)
drop HHmem

*create variables for the lineno, sex, relationship, and age of the person you are with
*lineno is needed here to identify the line number of mother and father
for num 1/19: gen byte linenoX = 0
for num 1/19: gen byte sexX = 0
for num 1/19: gen byte relationX = 0
for num 1/19: gen byte ageX = 0

	forvalues i = 1/19 {
	by tucaseid: replace lineno`i' = tulineno[`i'] if tulineno==1
	by tucaseid: replace sex`i' = tesex[`i'] if tulineno==1
	by tucaseid: replace relation`i' = terrp[`i'] if tulineno==1
	by tucaseid: replace age`i' = teage[`i'] if tulineno==1
	}

*only keep the first observation line for each HH
drop if tulineno~=1
destring tucaseid, replace
format tucaseid %15.0f

*determine the tulineno of the mother and father
gen byte mother = .
gen byte father = .
for num 1/19: replace mother = linenoX if relationX==24 & sexX==2
for num 1/19: replace father = linenoX if relationX==24 & sexX==1

*determine if parents of same sex
for num 2/19: gen motherX = 1 if relationX==24 & sexX==2
for num 2/19: gen fatherX = 1 if relationX==24 & sexX==1
egen byte totalmoms = rowtotal(mother2-mother19)
egen byte totaldads = rowtotal(father2-father19)
gen byte twomoms = (totalmoms==2)
gen byte twodads = (totaldads==2)

*determine if they have an own child (household and nonhousehold)
gen byte ownchild = 0
for num 2/19: replace ownchild = 1 if relationX==22 | relationX==40

*generate number of siblings by age group (under 15 or 15-18)
for num 2/19: gen byte sib014_X = (relationX==25 & (ageX<15 & ageX>=0))
egen byte sibu15 = rowtotal(sib014_2-sib014_19)

for num 2/19: gen byte sib1518_X = (relationX==25 & (ageX>14 & ageX<19))
egen byte sibs15_18 = rowtotal(sib1518_2-sib1518_19)

*drop variables used to create who with variables
keep tucaseid mother father ownchild sibs15_18 sibu15 numHHmem numHHroster sex* relation* age* twomoms twodads

sort tucaseid

save "$PATH/Data/rosactivity_`YYYY'.dta", replace

**************************************************
*use ACTIVITY file and merge on who with variables
**************************************************
use "$ATUSPATH/atusact_`YYYY'.dta", clear
keep tucaseid tuactivity_n tewhere tuactdur tuactdur24 tutier1code tutier2code tutier3code tustarttim tustoptime
destring tucaseid, replace
sort tucaseid tuactivity_n
merge 1:m tucaseid tuactivity_n using "$PATH/Data/whowithactivity_`YYYY'.dta"
keep if _merge==3
drop _merge

*merge on roster
sort tucaseid
merge m:1 tucaseid using "$PATH/Data/rosactivity_`YYYY'.dta"
keep if _merge==3
drop _merge

*variable for whether they were with a certain person on the roster line during the activity (1 is yes, 0 is no)
*this corresponds with age, relation and sex of person
for num 2/19: gen byte withrosX = 0
for num 2/19: replace withrosX = (lineno1==X | lineno2==X | lineno3==X | lineno4==X | lineno5==X | lineno6==X | lineno7==X | lineno8==X | lineno9==X | lineno10==X | lineno11==X | lineno12==X | lineno13==X | lineno14==X | lineno15==X | lineno16==X | lineno17==X | lineno18==X | lineno19==X)

# delimit ;
	forvalues i = 2/19 {;
label variable  withros`i'          "With Person Number `i' on Roster";
label variable  age`i'            	"Edited: Age`i'";
label variable  relation`i'             "Edited: How is `i' related to you?";
label variable  sex`i'            	 "Edited: Sex`i'";

label define labelwithros`i'
1	"Yes"
0	"No";

label define labelrelation`i'
-1    "Blank"
-2    "Don't Know"
-3    "Refused"
18    "Self"
19    "Self"
20    "Spouse (Husband/Wife)"
21    "Unmarried Partner"
22    "Own Household Child"
23    "Grandchild"
24    "Parent (Mother/Father)"
25    "Brother/Sister"
26    "Other Related Person (Aunt, Cousin, Nephew)"
27    "Foster Child"
28    "Housemate/Roommate"
29    "Roomer/Boarder"
30    "Other Nonrelative"
40    "Own Non-Household Child";

label define labelsex`i'
-1    "Blank"
-2    "Don't Know"
-3    "Refused"
1     "Male"
2     "Female";


label values    relation`i'       	labelrelation`i';
label values    sex`i'           	labelsex`i';
label values    withros`i'			labelwithros`i'	;

};

# delimit cr

*reformat tucaseid to see the entire number in STATA editor
format tucaseid %15.0f

*keep diaries for teenagers aged 15-17
keep if age1<18

compress

save "$PATH/Data/teen_activity_who_roster_`YYYY'.dta", replace

*number of teens in data'
display `YYYY'

count if tuactivity_n==1

rm "$PATH/Data/rosactivity_`YYYY'.dta"
rm "$PATH/Data/whowithactivity_`YYYY'.dta"
}