************************************************************
*
*Last Updated: March 12, 2022
*
*Written by Sabrina Wulff Pabilonia
*
* Download basic CPS data from Census
*here: https://www.census.gov/data/datasets/time-series/demo/cps/cps-basic.html
*rename the CPS files to those input below

*This program calculates:
*1) CPS work-limiting disability status for parents of teens living with parents 2003-2019 (Figure1)
*2) Creates a dataset with a measure of disability in MIS 4 for the parents of ATUS teenagers 
**********************************************************************************************

version 17.0

clear
clear matrix
cap log close
set more off

*CPS data stored here
global CPSPATH "c:/basicCPS"  

*output goes here
global PATH "X:/Charlene teens parentdisability"

log using "$PATH/Log/cps_parentdisability_20220313.log", replace

#delimit ;

*household identifers differ 2003-April 2004 in CPS;
*See ATUS user's guide for matching instructions;

*2001-2003;

forvalues YYYY = 2001/2003{;
	clear;
		
	forvalues m = 1/12{;
		clear;
		local MM : di %02.0f `m';
		infix str15 hrhhid		1-15
			byte hrmonth		16-17
			int hryear4			18-21
			byte hrmis			63-64
			str4 hrsample		71-74
			str2 hrsersuf		75-76
			str2 huhhnum		77-78
			byte perrp			118-119
			byte peparent		120-121
			byte prtage			122-123
			byte pespouse		127-128
			byte pulineno		147-148
			byte pudis			198-199
			byte pudis1			202-203
			byte pudis2			204-205
			byte pesex			129-130
			byte prpertyp		161-162
            pwsswgt             613-622
		using $CPSPATH/cps_`YYYY'`MM'.dat; 

	*keep only those 15+ (want to determine parents of those aged 15-17) ; 
	keep if prtage>=15;
	
	gen numhrsample=substr(hrsample,2,2) ;
	tokenize "`c(ALPHA)'";
	forval b = 1/26 {;
		label def alphabet `b' "``b''", add;
	};
	encode hrsersuf, gen(serstr) label(alphabet) ;
	drop hrsersuf;
	tostring serstr, gen(hrsersuf) format(%02.0f);
	replace hrsersuf="00" if hrsersuf=="27";
	egen hrhhid2=concat(numhrsample hrsersuf huhhnum);
	sort hrhhid hrhhid2 pulineno;
	drop numhrsample hrsersuf huhhnum hrsample serstr;
	
	*save the CPS sub-file with unique name;
	save "$PATH/Data/cps`YYYY'`MM'.dta", replace;
	clear;
	};
};

*Jan2004-April2004;
	forvalues YYYY = 2004/2004{;
	clear;
		
	forvalues m = 1/4{;
		clear;
		local MM : di %02.0f `m';
		infix str15 hrhhid		1-15
			byte hrmonth		16-17
			int hryear4			18-21
			byte hrmis			63-64
			str4 hrsample		71-74
			str2 hrsersuf		75-76
			str2 huhhnum		77-78
			byte perrp			118-119
			byte peparent		120-121
			byte prtage			122-123
			byte pespouse		127-128
			byte pulineno		147-148
			byte pudis			198-199
			byte pudis1			202-203
			byte pudis2			204-205
			byte pesex			129-130
			byte prpertyp		161-162
            pwsswgt             613-622
		using $CPSPATH/cps_`YYYY'`MM'.dat; 

	*keep only those 15+ (want to determine parents of those aged 15-17)  ;
	keep if prtage>=15;
	
	gen numhrsample=substr(hrsample,2,2) ;
	tokenize "`c(ALPHA)'";
	forval b = 1/26 {;
		label def alphabet `b' "``b''", add;
	};
	encode hrsersuf, gen(serstr) label(alphabet) ;
	drop hrsersuf;
	tostring serstr, gen(hrsersuf) format(%02.0f);
	replace hrsersuf="00" if hrsersuf=="27";
	egen hrhhid2=concat(numhrsample hrsersuf huhhnum);
	sort hrhhid hrhhid2 pulineno;
	drop numhrsample hrsersuf huhhnum hrsample serstr;
	
	*save the CPS sub-file with unique name;
	save "$PATH/Data/cps`YYYY'`MM'.dta", replace;
	clear;
	};
};

*May2004-Dec2004;
forvalues YYYY = 2004/2004{;
	clear;
		
	forvalues m = 5/12{;
		clear;
		local MM : di %02.0f `m';
		infix str15 hrhhid		1-15
			byte hrmonth		16-17
			int hryear4			18-21
			byte hrmis			63-64
			str5 hrhhid2		71-75
			byte perrp			118-119
			byte peparent		120-121
			byte prtage			122-123
			byte pespouse		127-128
			byte pulineno		147-148
			byte pudis			198-199
			byte pudis1			202-203
			byte pudis2			204-205
			byte pesex			129-130
			byte prpertyp		161-162
            pwsswgt             613-622
		using $CPSPATH/cps_`YYYY'`MM'.dat; 

	*keep only those 15+ (want to determine parents of those aged 15-17) ; 
	keep if prtage>=15;
	
	*save the CPS sub-file with unique name;
	save "$PATH/Data/cps`YYYY'`MM'.dta", replace;
	clear;
	};
};

*2005-2006;
*did not have pecohab pelndad pelnmom until 2007;
forvalues YYYY = 2005/2006{;
	clear;
		
	forvalues m = 1/12{;
		clear;
		local MM : di %02.0f `m';
		infix str15 hrhhid		1-15
			byte hrmonth		16-17
			int hryear4			18-21
			byte hrmis			63-64
			str5 hrhhid2		71-75
			byte perrp			118-119
			byte peparent		120-121
			byte prtage			122-123
			byte pespouse		127-128
			byte pulineno		147-148
			byte pudis			198-199
			byte pudis1			202-203
			byte pudis2			204-205
			byte pesex			129-130
			byte prpertyp		161-162
            pwsswgt             613-622
		using $CPSPATH/cps_`YYYY'`MM'.dat; 

	*keep only those 15+ (want to determine parents of those aged 15-17)  ;
	keep if prtage>=15;
	
	*save the CPS sub-file with unique name;
	save "$PATH/Data/cps`YYYY'`MM'.dta", replace;
	clear;
	};
};

forvalues YYYY = 2007/2019{;
	clear;
		
	forvalues m = 1/12{;
		clear;
		local MM : di %02.0f `m';
		infix str15 hrhhid		1-15
			byte hrmonth		16-17
			int hryear4			18-21
			byte hrmis			63-64
			str5 hrhhid2		71-75
			byte perrp			118-119
			byte peparent		120-121
			byte prtage			122-123
			byte pespouse		127-128
			byte pulineno		147-148
			byte pudis			198-199
			byte pudis1			202-203
			byte pudis2			204-205
			byte pesex			129-130
			byte prpertyp		161-162
            pwsswgt             613-622
			byte pelndad		886-887
			byte pelnmom		888-889
		using "$CPSPATH/cps_`YYYY'`MM'.dat"; 

	*keep only those 15+ (want to determine parents of those aged 15-17)  ;
	keep if prtage>=15;
	
	*save the CPS sub-file with unique name;
	save "$PATH/Data/cps`YYYY'`MM'.dta", replace;
	clear;
	};
};

******************;
*append all months;
******************;
forvalues YYYY = 2001/2019{;
	clear;
use "$PATH/Data/cps`YYYY'01.dta", clear;

	forvalues m = 2/12{;
		local MM : di %02.0f `m';
		append using "$PATH/Data/cps`YYYY'`MM'.dta";
	};
				
gen int year = `YYYY';

gen long recnum = _n ;

save "$PATH/Data/cps`YYYY'.dta", replace;

	*remove monthly files;
	
	forvalues m = 1/12{;
		local MM : di %02.0f `m';
		rm "$PATH/Data/cps`YYYY'`MM'.dta";
	};
};

# delimit cr

*append all CPS years
use "$PATH/Data/cps2001.dta", clear
foreach YYYY of num 2002/2019{
append using "$PATH/Data/cps`YYYY'.dta"
}
save "$PATH/Data/cpsdisabilitydata.dta", replace

/*
*remove yearly files
forvalues YYYY = 2001/2019{
	rm "$PATH/Data/cps`YYYY'.dta"
	}
*/

**********************************************************************************
*this section creates a dataset with a disability indicator in MIS 4 for adults**
*exactly one year prior to MIS8 (similar month - outgoing rotation groups)
**********************************************************************************
use "$PATH/Data/cpsdisabilitydata.dta", clear
*drop children
keep if prpertyp==2

*disability if MIS4
keep if hrmis==4
gen byte mis4_disability = (pudis==1 | pudis1==1 | pudis2==1)
replace mis4_disability = . if (pudis==-2 | pudis1==-2 | pudis2==-2 | pudis==-3 | pudis1==-3 | pudis2==-3) 

destring hrhhid hrhhid2, replace
keep hryear4 mis4_disability hrhhid hrhhid2 pulineno 
sort hrhhid hrhhid2 pulineno
save "$PATH/Data/cpsdisabilitydata_MIS4.dta", replace

*********************************************************************************
*This section calculates the % of teens living with at least one disabled parent 
*Data for Figure 1 - 2003-2019
**********************************************************************************

use "$PATH/Data/cpsdisabilitydata.dta", clear

keep if year>2002

*create unique household ids for each month in CPS between 2003-2019 so it is possible to identify teen's parents
egen HHid = concat(hrhhid hrhhid2 hrmonth hryear4) 

*does the person have a work-limiting disability of 6+ months
gen byte workdisability = (pudis==1 | pudis1==1 | pudis2==1)
*missing data of disability (don't know or refused)
replace workdisability = . if (pudis==-2 | pudis==-3 | pudis1==-2 | pudis1==-3 | pudis2==-2 | pudis2==-3)

drop hryear4 recnum hrhhid hrhhid2 pudis* prpertyp

*all teenagers 15-17 who have parent in household (identified 2007 forward)
gen byte teen1517 = (prtage>=15 & prtage<=17)

*sort to make the teens last
sort HHid teen1517 prtage

*how many teens in each household
by HHid: egen byte teenHH = total(teen1517)
tab teenHH

*drops
*drop household with 12 teenagers and a nonrelated person in 2018
drop if teenHH==12

*keep only households with teenagers
drop if teenHH==0

*can't use relative relationship to identify but need to use parent line number, pulineno and workdisability because in CPS not all households have complete pulineno in each month
*for multiple teens create mom and dad for relative positions
*these do not usually account for unmarried partners having a "mom" or "dad" relationship unless biological, step, or adopted (similar to how we specify in ATUS)
*these do capture parents in grandparent headed households
by HHid: gen byte obsID = _n 

forvalues x = 0/8{
	gen byte mom`x' = 0  
	gen byte dad`x' = 0  
}

******************************
*teen -loop through all in HH*
******************************
*assign a line number to the mother or father based on peparent and pesex prior to 2007
forvalues x = 0/8{
	by HHid: replace mom`x' = 1 if pulineno==peparent[_N-`x'] & pesex==2 & teen1517[_N -`x']==1 & year<2007
	by HHid: replace dad`x' = 1 if pulineno==peparent[_N-`x'] & pesex==1 & teen1517[_N -`x']==1 & year<2007
}
*then use pespouse to identify other parent
forvalues x = 0/8{
    forvalues y = 1/16{
		by HHid: replace mom`x' = 1 if pulineno==pespouse[`y'] & dad`x'[`y']==1 & pesex==2 & teen1517[_N -`x']==1 & year<2007
		by HHid: replace dad`x' = 1 if pulineno==pespouse[`y'] & mom`x'[`y']==1 & pesex==1 & teen1517[_N -`x']==1 & year<2007
	}
}

drop pespouse peparent 

forvalues x = 0/8{
	by HHid: replace mom`x' = 1 if pulineno==pelnmom[_N -`x'] & teen1517[_N -`x']==1 & year>=2007
	by HHid: replace dad`x' = 1 if pulineno==pelndad[_N -`x'] & teen1517[_N -`x']==1 & year>=2007
}

*cannot calculate disability status for some parents so they should not be considered in the counts
forvalues x = 0/8{
	by HHid: replace mom`x' = . if workdisability == . & teen1517[_N -`x']==1 
	by HHid: replace dad`x' = . if workdisability == . & teen1517[_N -`x']==1 	
}

forvalues x = 0/8{
	by HHid: egen byte momteen1517_`x' = total(mom`x'), missing
	by HHid: egen byte dadteen1517_`x' = total(dad`x'), missing
	by HHid: gen byte momdisabled`x' = 1 if mom`x'==1 & workdisability==1
	by HHid: gen byte daddisabled`x' = 1 if dad`x'==1 & workdisability==1
	by HHid: egen byte totmomdisabled`x' = total(momdisabled`x')
	by HHid: egen byte totdaddisabled`x' = total(daddisabled`x')
	*any parent disability
	gen byte anyparent_`x' = 1 if totmomdisabled`x'==1
	replace anyparent_`x' = 1 if totdaddisabled`x'==1 
	*only replace the teen's observation
	by HHid: gen byte parentdisabledteen1517_`x' = anyparent_`x'[_N -`x'] if teen1517==1 & obsID==_N -`x'
	*there may be a pulineno listed for a parent but no actual parent reported that month or the disability status may be missing (drop these observations)
	by HHid: replace teen1517 = 0 if momteen1517_`x'~=1 & dadteen1517_`x'~=1 & teen1517==1 & obsID==_N -`x'
}

*for each teen determine the parental disability status
gen byte parentdisabledteen1517 = 0 if teen1517==1
replace parentdisabledteen1517 = 1 if (parentdisabledteen1517_0==1 | parentdisabledteen1517_1==1 | parentdisabledteen1517_2==1 | parentdisabledteen1517_3==1 | parentdisabledteen1517_4==1| parentdisabledteen1517_5==1 | parentdisabledteen1517_6==1 | parentdisabledteen1517_7==1 | parentdisabledteen1517_8==1 ) & teen1517==1

*edit HHid parentdisabledteen1517 obsID prtage pulineno teen1517 mom0 dad0 momteen1517_0 dadteen1517_0 momdisabled0 totmomdisabled0 parentdisabledteen1517_0 if teenHH>0

*final weight (12 months of data, 4 implied decimal places)
gen weight = (pwsswgt/1000)/12

sort year

*number of teens in CPS living with their parent
by year: egen teens1517 = sum(weight*teen1517)
*number of teens in CPS living with their parent and have disability
by year: egen teens_disabledparent = sum(weight*teen1517*parentdisabledteen1517)

*parent disability rate for teenagers aged 15-17
gen teens_dparent_rate = (teens_disabledparent/teens1517)*100

*************************
*number of observations**
*weighted and unweighted*
*************************

count if teen1517==1
by year: count if teen1517==1
count if parentdisabledteen1517==1
by year: count if parentdisabledteen1517==1

by year: egen num_teen1517 = total(teen1517)
by year: egen num_teens_disabledparent = total(teen1517*parentdisabledteen1517)

tab teens_dparent_rate

by year: keep if _n==1

order year teens_dparent_rate num_teen1517 num_teens_disabledparent teens1517 teens_disabledparent, first
 
export excel year teens_dparent_rate num_teen1517 num_teens_disabledparent teens1517 teens_disabledparent using "$PATH/Tables/CPS parental disability rate.xlsx", cell(A2) keepcellfmt sheet("rate", modify) first(var)

log close