********************************************************************
*
*Paper : Parental Disabilty and Teenagers' Time Allocation
*
*File name : atus_teen_parentdisability.do
*
*This file creates the tables for the paper
*
*Author - Sabrina Wulff Pabilonia
*
*Last Updated : March 13, 2022
********************************************************************

# delimit cr

version 17.0 

clear 
clear matrix 
cap log close 
set more off  

*define the work file directory for the project
global PATH "X:/Charlene teens parentdisability"

log using "$PATH/Log/atus_teen_parentdisability_20220313.log", replace 

use "$PATH/Data/atus_allyears.dta" 

********************************************** 
*Sample restrictions 
*teenagers aged 15-17 who are unmarried 
*live with at least one parent 
*have no own household children 
*interviewed during the school year 
*have valid response on parents' disability 
********************************************** 
svyset [pw=tufinlwgt]

count 

*month dummies 
tabulate tumonth, gen(m) 

*control for time of year 
gen byte summer = (tumonth==6 | tumonth==7 | tumonth==8) 
tab summer 

gen byte schoolyear = (summer==0) 

*keep those interviewed during the school year 
keep if schoolyear==1 

count 

*teen does not live with a parent
gen notlivewparent = (dmother==1 & dfather==1) 

*how many do not live with parent - 3.7% (likely other adult) 
svy: mean notlivewparent

*drop those who do not live with at least one parent 
drop if (dmother==1 & dfather==1) 

count 

*drop if living with samesex parents 
drop if (twomoms==1 | twodads==1) 

count 

*drop if married or unmarried partner living in household 
gen byte married2 = (trsppres==1 | trsppres==2) 
drop if married2==1 

count 

*how many have own household children 
tab trohhchild 
drop if trohhchild==1 

count 

*how many have own children (in or out of HH) - teen parent 
tab ownchild 
drop if ownchild==1 

count

**drop if no information on mother or father from ATUSCPS file (considering whether they are a single parent)
*drop if mom missing in ATUS-CPS but in ATUS you had a mom
drop if mother_miss_atuscps == 1
*drop if dad missing in ATUS-CPS but in ATUS you had a dad
drop if father_miss_atuscps == 1  

count

*drop if mother or father is missing work disability status on ATUSCPS file 
drop if mother_missing_wdisability == 1  & dmother==0
drop if father_missing_wdisability == 1  & dfather==0

count

*how many have parent with work-related disability (can't accept any work in next six months) 
gen parentdisabled = (mother_wdisability==1 | father_wdisability==1) 
gen byte mdisabled = (mother_wdisability==1) 
gen byte fdisabled = (father_wdisability==1) 

*any disability for either parent 
gen byte parentanydisability = (mother_anydisability==1 | father_anydisability==1) 
replace parentanydisability = . if mother_anydisability==. | father_anydisability==. 

*types of disability - too few to use individually
foreach x of new ear eye rem phy drs out{
gen byte parent_dis`x' = (mother_dis`x'==1 | father_dis`x'==1) 
replace parent_dis`x' = . if mother_dis`x'==. | father_dis`x'==. 
}

gen MIS4disability = (mother_MIS4disability==1 | father_MIS4disability==1) 

*LR disability for parent in matchedsample (report disabled in both MIS8 and MIS4)
gen LRdisability4 = (parentdisabled==1 & MIS4disability==1) if matchedsampleMIS4==1

*teen any disability
gen byte teen_disability = (teen_anydisability==1)

tab tuyear 

*************************************
*count for sample of teenagers 15-17 
*************************************
count 

*how many live with single mother 
gen byte singlemother = (dfather==1) 
tab singlemother 
*how many live with single father 
gen byte singlefather = (dmother==1) 
tab singlefather 

*how many live with two parents 
gen byte twoparents = (singlemother~=1 & singlefather~=1) 

*create day of the week dummies 
gen byte sunday = tudiaryday==1 
gen byte monday = tudiaryday==2 
gen byte tuesday = tudiaryday==3 
gen byte wed = tudiaryday==4 
gen byte thurs = tudiaryday==5 
gen byte friday = tudiaryday==6 
gen byte saturday = tudiaryday==7 
gen byte holiday = trholiday==1 

*weekend or holidays 
gen byte weekend = (sunday==1 | saturday==1 | holiday==1) 
gen byte weekday = (weekend~=1) 

*gender 
gen byte female = (sex1==2) 
gen byte male = (female==0) 

count if female==1 
count if male==1 

*REWEIGHT PROGRAM FOR EQUAL DAY OF THE WEEK REPRESENTATION FOR TEENAGER SAMPLE BY GENDER WITHIN YEAR
gen rewgtyr = 0
capture program drop reweight
program define reweight
	foreach x of num 2003/2019{
		quietly: tabstat tufinlwgt if female==1 & tuyear==`x', stat(sum) by(tudiaryday) format(%20.0g) save
		replace rewgtyr= 0 if tuyear==`x' & female==1 
		forvalues i = 1/7{
			scalar dayfac`i' = det(r(Stat`i'))/det(r(StatTotal))
			replace rewgtyr = (tufinlwgt/dayfac`i')/7 if tudiaryday==`i' & female==1 & tuyear==`x'
		}
		quietly: tabstat tufinlwgt if female==0 & tuyear==`x', stat(sum) by(tudiaryday) format(%20.0g) save
		replace rewgtyr= 0 if tuyear==`x' & female==0 
		forvalues i = 1/7{
			scalar dayfac`i' = det(r(Stat`i'))/det(r(StatTotal))
			replace rewgtyr = (tufinlwgt/dayfac`i') if tudiaryday==`i' & female==0 & tuyear==`x'
		}
	}
end
scalar drop _all

*use the reweighting programs for equal day of the week representation for both the final weight and the replicate weights
quietly: reweight

*use SDR method with replicate weights and mse (with reweight for equal day of the week)
svyset [pw=rewgtyr]

*footnote X - how many have a parent with a disability
svy: mean parentdisabled 

*create race/ethnicity variables 
*hispanic is not mutually exclusive 
gen byte hispanic = (pehspnon==1) 
*primary race category, other includes mixed race categories 
gen byte white = (ptdtrace==1) 
gen byte black = (ptdtrace==2) 
gen byte asian = (ptdtrace==4) 
gen byte other = (ptdtrace>=3 & asian~=1) 
gen byte nonwhite = (black==1 | asian==1 | other==1) 

*Census region 
quietly tab gereg, gen(reg) 

*education variables 
gen byte mbachelor = (motheredu==43 | motheredu==44 | motheredu==45 | motheredu==46) 
replace mbachelor = 0 if dmother==1 
gen byte fbachelor = (fatheredu==43 | fatheredu==44 | fatheredu==45 | fatheredu==46) 
replace fbachelor = 0 if dfather==1 

*household income - change categories 
*use hufaminc 2003-2009, use hefaminc 2010-2019 
gen byte incmis = 0 
gen byte inc30 = 0 
gen byte inc3075 = 0 
gen byte inc75 = 0 

replace incmis = 1 if hufaminc < 0 & hryear4<2010 
replace inc30 = 1 if (hufaminc>=1 & hufaminc<=8) & hryear4<2010 
replace inc3075 = 1 if (hufaminc>8 & hufaminc<=13) & hryear4<2010  
replace inc75 = 1 if (hufaminc>=14) & hryear4<2010 

replace incmis = 1 if (hefaminc < 0) & hryear4>=2010 
replace inc30 = 1 if (hefaminc>=1 & hefaminc<=8) & hryear4>=2010 
replace inc3075 = 1 if (hefaminc>8 & hefaminc<=13) & hryear4>=2010 
replace inc75 = 1 if hefaminc>=14 & hryear4>=2010
 
*year dummies 
for num 2003/2019: gen byte yearX = (tuyear==X) 

*age dummies 
drop age2-age19 
for num 15/17: gen byte ageX = (age1==X) 

*create the highest degree of either parent is bachelor's degree 
gen byte parentbach = (fbachelor==1 | mbachelor==1) 
gen byte notparentbach = (parentbach==0) 

gen minhelpcareadult =  minadultcare + minhelpadult  

*School year results separately by gender 
gen byte fschoolyear = (schoolyear==1 & female==1) 
gen byte mschoolyear = (schoolyear==1 & male==1) 

gen byte youngersibling = (youngerbrothers==1 | youngersisters==1)
gen byte oldersibling = (oldersisters==1 | olderbrothers==1)

*labels 
label var minschool "Schooling-related activities" 
label var minclass "Class" 
label var minhwk "Homework" 
label var minsportsextra "Extracurricular activities & playing sports"  
label var minwk "Work and work-related activities(not commuting)" 
label var minsleep "Sleep" 
label var minhousehold "Household production" 
label var minhousework "Housework (cleaning, laundry, etc.)" 
label var minbuy "Buying goods and services" 
label var minfoodprep "Food preparation and clean-up"   
label var totalcare "Care activities" 
label var minchildcare "Primary childcare" 
label var minhelpcareadult "Caring/helping household adults" 
label var minpetcare "Pet care & veterinary services" 
label var minallleisure "Leisure" 
label var minparent "Time with parent" 
label var minmother "Time with mother" 
label var minfather "Time with father" 
label var age15 "Age 15" 
label var age16 "Age 16" 
label var age17 "Age 17" 
label var white "White" 
label var nonwhite "Nonwhite" 
label var hispanic "Hispanic" 
label var singlemother "Single mother" 
label var singlefather "Single father" 
label var sibu15 "Number of siblings under age 15"
label var sibs15_18 "Number of siblings age 15-18"
label var msa "Lives in metropolitan statistical area"
label var parentdisabled "Parent has work-related disability" 
label var extra_adult "Extra adult (age 19+) not parent in household" 
label var parentbach "At least one parent has bachelor's degree"
label var notparentbach "Parent does not have bachelor's degree"
label var fschoolyear "Female interviewed during school year"
label var mschoolyear "Male interviewed during school year"
label var mdisabled "Mother has work-related disabled"
label var fdisabled "Father has work-related disabled"
label var parentanydisability "A parent has any disability"
label var twoparents "Lives in two-parent household"
label var incmis "Household income missing"
label var inc30 "Household income <$30,000"
label var inc3075 "Household income $30,000-$74,999"
label var inc75 "Household income $75,000+"
label var dfather "No father in the household"
label var dmother "No mother in the household"
label var motheremp "Mother is employed"
label var fatheremp "Father is employed"
label var schoolyear "Interviewed September-May"
label var youngersibling "At least one younger sibling"
label var oldersibling "At least one older sibling"
label var rewgtyr "ATUS weight reweighted for equal day of the week representation by gender"
label define parentdisabled 0 "Parent not disabled" 1 "Parent disabled" 
label values parentdisabled parentdisabled 
label var mother_anydisability "Mother has any disability"
label var father_anydisability "Father has any disability"
label var numsibs "Number of household siblings any age"
label var teen_disability "Teen has any disability"
label var LRdisability4 "A parent has work-limiting disability in MIS4 and MIS8"

keep schoolyear fschoolyear mschoolyear minschool minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minhelpcareadult minpetcare minallleisure minsleep minparent minmother minfather parentdisabled mdisabled fdisabled parentanydisability singlemother singlefather twoparents sunday monda tuesday wed thurs friday saturday weekend weekday female male rewgtyr age15 age16 age17 white nonwhite hispanic singlemother singlefather twoparents parentbach extra_adult sibu15 sibs15_18 incmis inc30 inc3075 inc75 msa reg1 reg2 reg3 reg4 weekday m2-m5 m9-m12 year2003-year2018 tumonth tuyear motheremp fatheremp mother_anydisability father_anydisability dmother dfather teen_disability numsibs sameageteen youngersibling oldersibling LRdisability4 matchedsampleMIS4 

*24 controls for Table 3 
global summarystats "age15 age16 age17 white nonwhite hispanic singlemother singlefather twoparents parentbach extra_adult numsibs youngersibling oldersibling sameageteen incmis inc30 inc3075 inc75 msa reg2 reg3 reg4 weekday" 

global mycontrols "age15 age16 nonwhite hispanic singlemother singlefather parentbach numsibs i.youngersibling i.oldersibling i.sameageteen extra_adult incmis inc3075 inc75 msa reg2 reg3 reg4 weekday m2-m5 m9-m12 year2003-year2018" 

global restrictcontrols "age15 age16 nonwhite hispanic singlemother singlefather parentbach numsibs i.youngersibling i.oldersibling i.sameageteen extra_adult incmis inc3075 inc75 msa reg2 reg3 reg4 weekday m2-m5 m9-m12 year2008-year2018"

global controls2005 "age15 age16 nonwhite hispanic singlemother singlefather parentbach numsibs i.youngersibling i.oldersibling i.sameageteen extra_adult incmis inc3075 inc75 msa reg2 reg3 reg4 weekday m2-m5 m9-m12 year2005-year2018"

global weekdaycontrols "age15 age16 nonwhite hispanic singlemother singlefather parentbach numsibs i.youngersibling i.oldersibling i.sameageteen extra_adult incmis inc3075 inc75 msa reg2 reg3 reg4 m2-m5 m9-m12 year2003-year2018" 

global mycontrols2 "age15 age16 nonwhite hispanic singlemother singlefather parentbach numsibs extra_adult incmis inc3075 inc75 msa reg2 reg3 reg4 weekday m2-m5 m9-m12 year2003-year2018" 

global mycontrols_noincome "age15 age16 nonwhite hispanic singlemother singlefather parentbach numsibs i.youngersibling i.oldersibling i.sameageteen extra_adult msa reg2 reg3 reg4 weekday m2-m5 m9-m12 year2003-year2018" 

global mycontrols_twoparents "age15 age16 nonwhite hispanic parentbach numsibs i.youngersibling i.oldersibling i.sameageteen extra_adult incmis inc3075 inc75 msa reg2 reg3 reg4 weekday m2-m5 m9-m12 year2003-year2018" 

*17 daily minutes outcomes outcomes 
global timeuse "minschool minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minhelpcareadult minpetcare minallleisure minsleep minparent minmother minfather" 

*create indicator for participation in activity 
foreach x of var minschool minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minhelpcareadult minpetcare minallleisure minsleep minparent minmother minfather{ 
gen d`x' = (`x'>0) 
} 

*17 participation outcomes 
global dtimeuse "dminschool dminclass dminhwk dminsportsextra dminwk dminhousehold dminhousework dminbuy dminfoodprep dminchildcare dminhelpcareadult dminpetcare dminallleisure dminsleep dminparent dminmother dminfather" 

save "$PATH/Data/teensample.dta", replace

*text p.X - What % have a parent with a work-related disability 
svy: mean parentdisabled, over(female) 
svy: mean parentdisabled 

*time use means for teens by disability status 
foreach x of varlist $timeuse { 
svy, subpop(male): mean `x' 
} 

foreach x of varlist $timeuse { 
svy, subpop(female): mean `x' 
} 

*need this version to do the p-value tests for Table 1 
version 10.0 


****************************************************************************** 
*Table 1-3 - unadjusted means by parental disability status 
****************************************************************************** 
******** 
*Females 
******** 

*% engage in activity for female sample 
svy, subpop(fschoolyear): mean $dtimeuse 

*calculate 4 vectors and put in matrix then use frmttable 

*Get means for all females who live with parent with work-related disability 
svy, subpop(if fschoolyear==1 & parentdisabled==1): mean $timeuse $summarystats 

*Store results in matrix named mu_1 
mat define mu_1 = e(b)' 

*Get mean for all females who do not live with parent with work-related disability 
svy, subpop(if fschoolyear==1 & parentdisabled==0): mean $timeuse $summarystats 

*Store results in matrix named mu_2 
mat define mu_2 = e(b)' 

*calculate the difference between the two means 
matrix difference = mu_1 - mu_2 

*p-value from Wald test for equality of means
mat V =J(41,1,.) 

local i = 1 
foreach x of var $timeuse $summarystats{ 
	svy, subpop(fschoolyear): mean `x', over(parentdisabled) coeflegend 
	test _b[_subpop_1] =  _b[_subpop_2] 
	matrix V[`i',1] = r(p) 
	local ++i 
} 

******** 
*Males 
******** 

*% engage in activity for male sample 
svy, subpop(mschoolyear): mean $dtimeuse 

*calculate 4 vectors and put in matrix then use frmttable 

*Get means for all males who live with parent with work-related disability 
svy, subpop(if mschoolyear==1 & parentdisabled==1): mean $timeuse $summarystats 

*Store results in matrix named mu_1_M 
mat define mu_1_M = e(b)' 

*Get mean for all males who do not live with parent with work-related disability 
svy, subpop(if mschoolyear==1 & parentdisabled==0): mean $timeuse $summarystats 

*Store results in matrix named mu_2_M 
mat define mu_2_M = e(b)' 

*calculate the difference between the two means 
matrix difference_M = mu_1_M - mu_2_M 

*p-value from Wald test for equality of means 
mat V_M =J(41,1,.) 

local i = 1 
foreach x of var $timeuse $summarystats{ 
	svy, subpop(mschoolyear): mean `x', over(parentdisabled) coeflegend 
	test _b[_subpop_1] =  _b[_subpop_2] 
	matrix V_M[`i',1] = r(p) 
	local ++i 
} 

*create matrix from submatrices 
matrix D = (mu_1, mu_2, difference, V, mu_1_M, mu_2_M, difference_M, V_M) 

frmttable using "$PATH/Tables/Table 1.doc", statmat(D) sd(3) varlabels replace ///
                ctitle("", Dmean_F, NoDmean_F, Diff_F, p-value_F, Dmean_M, NoDmean_M, Diff_M, p-value_M) 

clear matrix 

*number of observations - Table 1 	
count if fschoolyear==1 & parentdisabled==1  
count if fschoolyear==1 & parentdisabled==0  
count if mschoolyear==1 & parentdisabled==1  
count if mschoolyear==1 & parentdisabled==0  

tab mdisabled if schoolyear==1  
tab fdisabled if schoolyear==1 
tab parentdisabled if schoolyear==1 

version 17.0 

*does anyone in schoolyear sample have both parents disabled? - 17 
count if schoolyear==1 & fdisabled==1 & mdisabled==1 & female==1 
count if schoolyear==1 & fdisabled==1 & mdisabled==1 & male==1 

*June 2008 CPS started asking about any disability
*restricted time frame for any disability questions 
gen byte restrictyears = ((tuyear==2008 & tumonth>5) | tuyear>2008)
*some are missing any disability status so exclude those from the subsample analysi
replace restrictyears = . if parentanydisability==.
count if restrictyears==1

*footnote 5
tab parentanydisability if schoolyear==1 & restrictyears==1
svy, subpop(if schoolyear==1 & restrictyears==1): mean parentanydisability  

*do these parents work? 
tab motheremp if mother_anydisability==1 & dmother==0 
tab fatheremp if father_anydisability==1 & dfather==0 

corr mother_anydisability mdisabled if schoolyear==1 & dmother==0 & restrictyears==1

corr father_anydisability fdisabled if schoolyear==1 & dfather==0 & restrictyears==1

*footnote 5 
*overlap for 37 percent of teenagers 
svy, subpop(if schoolyear==1 & restrictyears==1 & (parentanydisability==1 | parentdisabled==1)): tab parentanydisability parentdisabled 

************
*Table 4
*Full Sample
************
*for tobit estimate average marginal effect for observed outcome including zeros (note that at the time of writing STATA output notation is wrong)
*GIRLS
*Tobit
svy, subpop(fschoolyear): tobit minschool i.parentdisabled $mycontrols, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table4_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare  {
svy, subpop(fschoolyear): tobit `x' i.parentdisabled $mycontrols, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table4_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
*OLS
foreach x of varlist minallleisure minsleep minparent {
svy, subpop(fschoolyear): reg `x' i.parentdisabled $mycontrols
outreg2 using "$PATH/Tables/Table4_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap
}

*BOYS
*Tobit 
svy, subpop(mschoolyear): tobit minschool i.parentdisabled $mycontrols, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table4_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare  {
svy, subpop(mschoolyear): tobit `x' i.parentdisabled $mycontrols, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table4_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
*OLS
foreach x of varlist minallleisure minsleep minparent {
svy, subpop(mschoolyear): reg `x' i.parentdisabled $mycontrols
outreg2 using "$PATH/Tables/Table4_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap
} 

*********************************************************
*Run Table 4 without income (potential mediating factor)*
*Has no effects on the results
*********************************************************
*GIRLS
*Tobit
svy, subpop(fschoolyear): tobit minschool i.parentdisabled $mycontrols_noincome, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table4_noincomecontrols_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare {
svy, subpop(fschoolyear): tobit `x' i.parentdisabled $mycontrols_noincome, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2  using "$PATH/Tables/Table4_noincomecontrols_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
*OLS
foreach x of varlist minallleisure minsleep minparent {
svy, subpop(fschoolyear): reg `x' i.parentdisabled $mycontrols_noincome
outreg2  using "$PATH/Tables/Table4_noincomecontrols_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap
}

*BOYS
*Tobit 
svy, subpop(mschoolyear): tobit minschool i.parentdisabled $mycontrols_noincome, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)  
outreg2  using "$PATH/Tables/Table4_noincomecontrols_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace 
*Tobit 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare { 
svy, subpop(mschoolyear): tobit  `x' i.parentdisabled $mycontrols_noincome, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table4_noincomecontrols_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(mschoolyear): reg `x' i.parentdisabled $mycontrols_noincome 
outreg2 using "$PATH/Tables/Table4_noincomecontrols_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

***************************************
**ROBUSTNESS CHECK USING OLS*
*Add to Appendix Table A3 and footnote
**************************************
*GIRLS
svy, subpop(fschoolyear): reg minschool i.parentdisabled $mycontrols 
outreg2 using "$PATH/Tables/Appendix_OLS_TableA3_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(minschool) nocons nor noobs replace
*OLS 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare minallleisure minsleep minparent{ 
svy, subpop(fschoolyear): reg `x' i.parentdisabled $mycontrols
outreg2 using "$PATH/Tables/Appendix_OLS_TableA3_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 
*BOYS
svy, subpop(mschoolyear): reg minschool i.parentdisabled $mycontrols 
outreg2 using "$PATH/Tables/Appendix_OLS_TableA3_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(minschool) nocons nor noobs replace
*OLS 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare minallleisure minsleep minparent{ 
svy, subpop(mschoolyear): reg `x' i.parentdisabled $mycontrols 
outreg2 using "$PATH/Tables/Appendix_OLS_TableA3_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

**ROBUSTNESS CHECK using zinb (the overdispersion parameter was less than 1 so try zip instead - plus zinb is for count data)
***ROBUSTNESS CHECK using zip with probit option  (Zero-inflated Poisson Model)
*first coefficient corresponds to those who did the activity (not directly compared to marginal effects for observed outcome after Tobit model, the second one predicts who did the activity
*note - outreg2 doesn't capture the margins  
*************************************************
*Table A4
*************************************************
*GIRLS
svy, subpop(fschoolyear): zip minschool i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare {
svy, subpop(fschoolyear): zip `x' i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
}
foreach x of varlist minallleisure minsleep {
svy, subpop(fschoolyear): poisson `x' i.parentdisabled $mycontrols
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
}
svy, subpop(fschoolyear): zip minparent i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
*BOYS 
svy, subpop(mschoolyear): zip minschool i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare {
svy, subpop(mschoolyear): zip `x' i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
}
foreach x of varlist minallleisure minsleep {
svy, subpop(mschoolyear): poisson `x' i.parentdisabled $mycontrols
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
}
svy, subpop(mschoolyear): zip minparent i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional)
outreg2 using "$PATH/Tables/Appendix_ZIP_TableA4_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 

*redo and get the marginal effects for the probit model for predicted probability of NOT participating 
*GIRLS
svy, subpop(fschoolyear): zip minschool i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional) predict(pr) 
outreg2 using "$PATH/Tables/Appendix_probit_Table4_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare {
svy, subpop(fschoolyear): zip `x' i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional) predict(pr)
outreg2 using "$PATH/Tables/Appendix_probit_Table4_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
}
svy, subpop(fschoolyear): zip minparent i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional) predict(pr)
outreg2 using "$PATH/Tables/Appendix_probit_Table4_girls.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
*BOYS 
svy, subpop(mschoolyear): zip minschool i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit
margins, dydx(parentdisabled) vce(unconditional) predict(pr) 
outreg2 using "$PATH/Tables/Appendix_probit_Table4_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare {
svy, subpop(mschoolyear): zip `x' i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit 
margins, dydx(parentdisabled) vce(unconditional) predict(pr) 
outreg2 using "$PATH/Tables/Appendix_probit_Table4_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
svy, subpop(mschoolyear): zip minparent i.parentdisabled $mycontrols, inf(i.parentdisabled $mycontrols) probit  
margins, dydx(parentdisabled) vce(unconditional) predict(pr)
outreg2 using "$PATH/Tables/Appendix_probit_Table4_boys.doc",  keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap

***************************************
*SIBLING AGE STRUCTURE AND DISABILITY**
*Table 5
***************************************
*convergence not achieved minhousehold (female) so I dropped the interaction with sameageteen
*GIRLS
svy, subpop(fschoolyear): tobit minschool i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen $mycontrols2 , ll(0)
margins, dydx(parentdisabled) at(youngersibling==1) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table5_girls_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace
svy, subpop(fschoolyear): tobit minschool i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen $mycontrols2 , ll(0)
margins, dydx(parentdisabled) at(oldersibling==1) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table5_girls_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs ap
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare  {
svy, subpop(fschoolyear): tobit `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2, ll(0)
margins, dydx(parentdisabled) at(youngersibling==1) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table5_girls_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
svy, subpop(fschoolyear): tobit `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2, ll(0)
margins, dydx(parentdisabled) at(oldersibling==1) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table5_girls_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
*OLS
foreach x of varlist minallleisure minsleep minparent {
svy, subpop(fschoolyear): reg `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2
margins, dydx(parentdisabled) at(youngersibling==1) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table5_girls_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
svy, subpop(fschoolyear): reg `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2
margins, dydx(parentdisabled) at(oldersibling==1)  post vce(unconditional)
outreg2 using "$PATH/Tables/Table5_girls_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
*BOYS
svy, subpop(mschoolyear): tobit minschool i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen $mycontrols2 , ll(0)
margins, dydx(parentdisabled) at(youngersibling==1) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table5_boys_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace
svy, subpop(mschoolyear): tobit minschool i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen $mycontrols2 , ll(0)
margins, dydx(parentdisabled) at(oldersibling==1) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table5_boys_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs ap
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare  {
svy, subpop(mschoolyear): tobit `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2, ll(0)
margins, dydx(parentdisabled) at(youngersibling==1) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table5_boys_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
svy, subpop(mschoolyear): tobit `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2, ll(0)
margins, dydx(parentdisabled) at(oldersibling==1) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table5_boys_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
*OLS
foreach x of varlist minallleisure minsleep minparent {
svy, subpop(mschoolyear): reg `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2
margins, dydx(parentdisabled) at(youngersibling==1) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table5_boys_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap
svy, subpop(mschoolyear): reg `x' i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen  $mycontrols2
margins, dydx(parentdisabled) at(oldersibling==1)  post vce(unconditional)
outreg2 using "$PATH/Tables/Table5_boys_siblings.xls", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap
}

*test for equality of coefficients of interactions of parental disability with youngersiblings and oldersiblings for boys' sleep (Wald test - can't reject null)
svy, subpop(mschoolyear): reg minsleep i.parentdisabled##i.youngersibling i.parentdisabled##i.oldersibling i.sameageteen $mycontrols2
test _b[1.parentdisabled#1.youngersibling] = _b[1.parentdisabled#1.oldersibling]

**************************************** 
*Table 6 major categories on school days 
*include homework time too 
**************************************** 
*GIRLS
svy, subpop(if fschoolyear==1 & weekday==1): tobit minschool i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table6_schooldays_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minhwk minwk minhousehold{ 
svy, subpop(if fschoolyear==1 & weekday==1): tobit `x' i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional) 
outreg2  using "$PATH/Tables/Table6_schooldays_girls.doc", keep(1.parentdisabled) bdec(3)  ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(if fschoolyear==1 & weekday==1): reg `x' i.parentdisabled $weekdaycontrols 
outreg2 using "$PATH/Tables/Table6_schooldays_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

*BOYS
svy, subpop(if mschoolyear==1 & weekday==1): tobit minschool i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table6_schooldays_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minhwk minwk minhousehold{ 
svy, subpop(if mschoolyear==1 & weekday==1): tobit `x' i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table6_schooldays_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(if mschoolyear==1 & weekday==1): reg `x' i.parentdisabled $weekdaycontrols 
outreg2 using "$PATH/Tables/Table6_schooldays_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

*************** 
*Non-schooldays 
*************** 
*GIRLS
svy, subpop(if fschoolyear==1 & weekday==0): tobit minschool i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table6_nonschooldays_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minhwk minwk minhousehold { 
svy, subpop(if fschoolyear==1 & weekday==0): tobit `x' i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table6_nonschooldays_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(if fschoolyear==1 & weekday==0): reg `x' i.parentdisabled $weekdaycontrols 
outreg2 using "$PATH/Tables/Table6_nonschooldays_girls.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

*BOYS
svy, subpop(if mschoolyear==1 & weekday==0): tobit minschool i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2  using "$PATH/Tables/Table6_nonschooldays_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist  minhwk minwk minhousehold{ 
svy, subpop(if mschoolyear==1 & weekday==0): tobit `x' i.parentdisabled $weekdaycontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table6_nonschooldays_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(if mschoolyear==1 & weekday==0): reg `x' i.parentdisabled $weekdaycontrols 
outreg2 using "$PATH/Tables/Table6_nonschooldays_boys.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

***********Try with heterosexual 2-parent families******************** 
*Table 7 - GIRLS
********************************************************************** 
*School year - does it matter if the mother or the father is disabled 
********************************************************************** 
*exclude the following smaller categories as the tobit doesn't estimate: extracurriculars, childcare, and adult care when we move to smaller sample 
*Tobit 
svy, subpop(if fschoolyear==1 & twoparents==1): tobit minschool i.mdisabled i.fdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled fdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table7.doc", keep(1.mdisabled 1.fdisabled) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare { 
svy, subpop(if fschoolyear==1 & twoparents==1): tobit `x' i.mdisabled i.fdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled fdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2  using "$PATH/Tables/Table7.doc", keep(1.mdisabled 1.fdisabled) bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS
foreach x of varlist minallleisure minsleep minparent minmother minfather{ 
svy, subpop(if fschoolyear==1 & twoparents==1): reg  `x' i.mdisabled i.fdisabled $mycontrols_twoparents 
outreg2 using "$PATH/Tables/Table7.doc", keep(1.mdisabled 1.fdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

*robustness check
*try OLS for school, class, homework
svy, subpop(if fschoolyear==1 & twoparents==1): reg minschool i.mdisabled i.fdisabled $mycontrols_twoparents 
svy, subpop(if fschoolyear==1 & twoparents==1): reg minclass i.mdisabled i.fdisabled $mycontrols_twoparents 
svy, subpop(if fschoolyear==1 & twoparents==1): reg minhwk i.mdisabled i.fdisabled $mycontrols_twoparents 

svy, subpop(if fschoolyear==1 & twoparents==1): tobit minschool i.mdisabled i.fdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(1.mdisabled 1.fdisabled) predict(ystar(0,.)) post vce(unconditional)

svy, subpop(if fschoolyear==1 & twoparents==1): tobit minclass i.mdisabled i.fdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(1.mdisabled 1.fdisabled) predict(ystar(0,.)) post vce(unconditional)

svy, subpop(if fschoolyear==1 & twoparents==1): tobit minhwk i.mdisabled i.fdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(1.mdisabled 1.fdisabled) predict(ystar(0,.)) post vce(unconditional)


*************** 
*Table 8 - BOYS
*************** 
******************************************************************************* 
**School year  - does it matter if the mother or the father is disabled (males) 
******************************************************************************* 
*exclude the following smaller categories as the tobit doesn't estimate: extracurriculars, childcare, and adult care when we move to smaller sample 
*Tobit 
svy, subpop(if mschoolyear==1 & twoparents==1): tobit minschool i.mdisabled i.fdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled fdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table8.doc", keep(1.mdisabled 1.fdisabled) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare { 
svy, subpop(if mschoolyear==1 & twoparents==1): tobit `x' i.mdisabled i.fdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled fdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2  using "$PATH/Tables/Table8.doc", keep(1.mdisabled 1.fdisabled) bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS
foreach x of varlist minallleisure minsleep minparent minmother minfather{ 
svy, subpop(if mschoolyear==1 & twoparents==1): reg  `x' i.mdisabled i.fdisabled $mycontrols_twoparents 
outreg2 using "$PATH/Tables/Table8.doc", keep(1.mdisabled 1.fdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

******************************** 
*Table 9 - single mother 
******************************** 
*GIRLS
*Tobit 
svy, subpop(if fschoolyear==1 & singlemother==1): tobit minschool i.mdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table9_girls.doc", keep(1.mdisabled) bdec(3) ctitle(minschool)  nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare  { 
svy, subpop(if fschoolyear==1 & singlemother==1): tobit `x' i.mdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table9_girls.doc", keep(1.mdisabled) bdec(3) ctitle(`x')  nor noobs ap 
} 
foreach x of varlist minallleisure minsleep minparent  { 
svy, subpop(if fschoolyear==1 & singlemother==1): reg `x' i.mdisabled $mycontrols_twoparents 
outreg2 mdisabled using "$PATH/Tables/Table9_girls.doc", keep(1.mdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

*BOYS
*petcare model will not converge 
*Tobit 
svy, subpop(if mschoolyear==1 & singlemother==1): tobit minschool i.mdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled) predict(ystar(0,.)) post vce(unconditional) 
outreg2 using "$PATH/Tables/Table9_boys.doc", keep(1.mdisabled) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare  { 
svy, subpop(if mschoolyear==1 & singlemother==1): tobit `x' i.mdisabled $mycontrols_twoparents, ll(0) 
margins, dydx(mdisabled) predict(ystar(0,.)) post vce(unconditional)
outreg2 using "$PATH/Tables/Table9_boys.doc", keep(1.mdisabled) bdec(3) ctitle(`x')  nor noobs ap 
} 
foreach x of varlist minallleisure minsleep minparent { 
svy, subpop(if mschoolyear==1 & singlemother==1): reg `x' i.mdisabled $mycontrols_twoparents 
outreg2 using "$PATH/Tables/Table9_boys.doc", keep(1.mdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

************************************************************* 
*Table A5
*Alternative type of disability based on ADL (not work-limiting necessarily) 
*Smaller sample and different definition 
*restrict to June 2008 forward when they put this question in 
* ((tuyear==2008 & tumonth>5) | tuyear>2008) 
************************************************************* 
*encounters missing prediction if use vce(unconditional) on margins so run without
*GIRLS
*Tobit 
svy, subpop(if fschoolyear==1 & restrictyears==1): tobit minschool i.parentanydisability $restrictcontrols, ll(0) 
margins, dydx(parentanydisability) predict(ystar(0,.)) post 
outreg2 using "$PATH/Tables/TableA5_girls.doc", keep(1.parentanydisability) bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare{ 
svy, subpop(if fschoolyear==1 & restrictyears==1): tobit `x' i.parentanydisability $restrictcontrols, ll(0) 
margins, dydx(parentanydisability) predict(ystar(0,.)) post 
outreg2 using "$PATH/Tables/TableA5_girls.doc", keep(1.parentanydisability)  bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(if fschoolyear==1 & restrictyears==1): reg `x' i.parentanydisability $restrictcontrols 
outreg2  using "$PATH/Tables/TableA5_girls.doc", keep(1.parentanydisability) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

*BOYS
*Tobit 
svy, subpop(if mschoolyear==1 & restrictyears==1): tobit minschool i.parentanydisability  $restrictcontrols, ll(0) 
margins, dydx(parentanydisability) predict(ystar(0,.)) post 
outreg2 using "$PATH/Tables/TableA5_boys.doc", keep(1.parentanydisability)  bdec(3) ctitle(minschool) nor noobs replace 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare { 
svy, subpop(if mschoolyear==1 & restrictyears==1): tobit `x' i.parentanydisability $restrictcontrols, ll(0) 
margins, dydx(parentanydisability) predict(ystar(0,.)) post 
outreg2 using "$PATH/Tables/TableA5_boys.doc", keep(1.parentanydisability)  bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(if mschoolyear==1 & restrictyears==1): reg `x' i.parentanydisability $restrictcontrols 
outreg2  using "$PATH/Tables/TableA5_boys.doc", keep(1.parentanydisability) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

  
********************************************************
*how many have LR disability as measured a year earlier
*footnote 3
*********************************************************

*match rate MIS4 to ATUS (52%)
count if schoolyear==1
count if schoolyear==1 & matchedsampleMIS4==1

*how many disabled in both months?
*proportion of the matched sample who is disabled that has a long-run disability (77% (MIS4))
svy, subpop(if schoolyear==1 & matchedsampleMIS4==1 & parentdisabled==1): prop LRdisability4 

**************************
**Footnote 9**************
**************************
*run for June 2008 forward and then add teen has any disability
*GIRLS
svy, subpop(if fschoolyear==1 & restrictyears==1): tobit minschool i.parentdisabled $restrictcontrols, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) vce(unconditional) post
outreg2 using "$PATH/Tables/Table4_girls_2008forward.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool) nor noobs replace
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare {
svy, subpop(if fschoolyear==1 & restrictyears==1): tobit `x' i.parentdisabled $restrictcontrols, ll(0)
margins, dydx(parentdisabled) predict(ystar(0,.)) vce(unconditional) post
outreg2 using "$PATH/Tables/Table4_girls_2008forward.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nor noobs ap
}
*OLS
foreach x of varlist minallleisure minsleep minparent {
svy, subpop(if fschoolyear==1 & restrictyears==1): reg `x' i.parentdisabled $restrictcontrols
outreg2 using "$PATH/Tables/Table4_girls_2008forward.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap
}

*BOYS
*Tobit 
svy, subpop(if mschoolyear==1 & restrictyears==1): tobit minschool i.parentdisabled $restrictcontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) vce(unconditional) post 
outreg2 using "$PATH/Tables/Table4_boys_2008forward.doc", keep(1.parentdisabled) bdec(3) ctitle(minschool)  nor noobs replace 
*Tobit 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare  { 
svy, subpop(if mschoolyear==1 & restrictyears==1): tobit  `x' i.parentdisabled $restrictcontrols, ll(0) 
margins, dydx(parentdisabled) predict(ystar(0,.)) vce(unconditional) post 
outreg2 using "$PATH/Tables/Table4_boys_2008forward.doc", keep(1.parentdisabled) bdec(3) ctitle(`x')  nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent { 
svy, subpop(if mschoolyear==1 & restrictyears==1): reg `x' i.parentdisabled $restrictcontrols 
outreg2 using "$PATH/Tables/Table4_boys_2008forward.doc", keep(1.parentdisabled) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

*****************************
*Add teen disability control*
*****************************
*how many have an ADL?
svy, subpop(if fschoolyear & restrictyears==1): mean teen_disability
svy, subpop(if mschoolyear==1 & restrictyears==1): mean teen_disability
svy, subpop(if schoolyear==1 & restrictyears==1): mean teen_disability

*GIRLS
*Tobit 
svy, subpop(if fschoolyear & restrictyears==1): tobit minschool i.parentdisabled i.teen_disability $restrictcontrols, ll(0)
margins, dydx(parentdisabled teen_disability) predict(ystar(0,.)) vce(unconditional) post
outreg2 using "$PATH/Tables/Table4_girls_2008forward_teendisability.doc", keep(1.parentdisabled 1.teen_disability) bdec(3) ctitle(minschool) nor noobs replace
*Tobit 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare {
svy, subpop(if fschoolyear & restrictyears==1): tobit `x' i.parentdisabled i.teen_disability $restrictcontrols, ll(0)
margins, dydx(parentdisabled teen_disability) predict(ystar(0,.)) vce(unconditional) post
outreg2 using "$PATH/Tables/Table4_girls_2008forward_teendisability.doc", keep(1.parentdisabled 1.teen_disability) bdec(3) ctitle(`x') nor noobs ap
}
*OLS
foreach x of varlist minallleisure minsleep minparent {
svy, subpop(if fschoolyear & restrictyears==1): reg `x' i.parentdisabled i.teen_disability $restrictcontrols
outreg2 using "$PATH/Tables/Table4_girls_2008forward_teendisability.doc", keep(1.parentdisabled 1.teen_disability) bdec(3) ctitle(`x') nocons nor noobs ap
}

*BOYS
*Tobit 
svy, subpop(if mschoolyear==1 & restrictyears==1): tobit minschool i.parentdisabled i.teen_disability $restrictcontrols, ll(0) 
margins, dydx(parentdisabled teen_disability) predict(ystar(0,.)) vce(unconditional) post 
outreg2 using "$PATH/Tables/Table4_boys_2008forward_teendisability.doc", keep(1.parentdisabled 1.teen_disability) bdec(3) ctitle(minschool) nor noobs replace 
*Tobit 
foreach x of varlist minclass minhwk minsportsextra minwk minhousehold minhousework minbuy minfoodprep minchildcare minpetcare { 
svy, subpop(if mschoolyear==1 & restrictyears==1): tobit  `x' i.parentdisabled i.teen_disability $restrictcontrols, ll(0) 
margins, dydx(parentdisabled teen_disability) predict(ystar(0,.)) vce(unconditional) post 
outreg2 using "$PATH/Tables/Table4_boys_2008forward_teendisability.doc", keep(1.parentdisabled 1.teen_disability) bdec(3) ctitle(`x') nor noobs ap 
} 
*OLS 
foreach x of varlist minallleisure minsleep minparent{ 
svy, subpop(if mschoolyear==1 & restrictyears==1): reg `x' i.parentdisabled i.teen_disability $restrictcontrols 
outreg2 using "$PATH/Tables/Table4_boys_2008forward_teendisability.doc", keep(1.parentdisabled 1.teen_disability) bdec(3) ctitle(`x') nocons nor noobs ap 
} 

log close
