/*
*Author - Sabrina Wulff Pabilonia
*Paper : Parental Disabilty and Teenagers' Time Allocation
*Last updated 03-13-2022
*File name - atus_parents_of_teens.do
*This file creates variables for the number of people who are 
with you during an activity and merges with activity file
*/

*define directories (use forward slash)
global DATA "X:/Charlene teens parentdisability/Data"
global LOG "X:/Charlene teens parentdisability/Log"
*original ATUS data stored
global ATUSPATH "A:/ATUS"

cap log close
*log file
log using "$LOG/atus_parents_of_teens.log", replace

version 17.0
clear all
set more 1

*loop over all years of data: 2003-2019
forvalues YYYY = 2003/2019{

*****************
*new year of data
*****************
display `YYYY'

/*
*atus who with during an activity - lists the ATUS respondent, each activity, and the line number from roster of who with (multiple observations per activity-person combo with a separate observation for different persons who were with them)
*value of -1 is that the activity didn't ask who they were with so there will be no line number for the person (example personal care, sleep, and work - work was asked in 2010 forward) and a 1 is they were alone
*/

use "$ATUSPATH/atuswho_`YYYY'.dta", clear
*create a variable that indicates a unique person and activity
egen idnum = group(tucaseid tuactivity_n)
*calculate the most people someone is with in each activity 
sort idnum
by idnum: gen byte people = _N

tab people
drop people

*create variables that list the line numbers on the roster of the persons you are with during an activity
for num 1/19: gen byte linenoX = 0

	forvalues i = 1/19 {
	bys idnum: replace lineno`i' = tulineno[`i'] if _n==1
	}
	
*only keep the first observation line for each activity
by idnum: drop if _n~=1
*drop variables used to create who with variables
drop idnum tulineno tuwho_code trwhona

destring tucaseid, replace

sort tucaseid tuactivity_n
save "$DATA/whowithactivity_`YYYY'.dta", replace

******************
*ATUS roster file*
******************
use "$ATUSPATH/atusrost_`YYYY'.dta", clear

*get the relationship, gender and age of each person in the household to the ATUS respondent
keep tucaseid tulineno teage terrp tesex

*how many people are in the household - up to 19
*Always check with new year of data
tab tulineno

sort tucaseid tulineno
*how many listed on household roster
by tucaseid: egen byte numHHroster = max(tulineno)

*exclude non-HH children under age 18 listed on roster from the number of HH members
gen byte HHmem = 1
replace HHmem = 0 if terrp==40
by tucaseid: egen byte numHHmem = total(HHmem)
drop HHmem

*create variables for the sex, relationship, and age of the person you are with
for num 1/19: gen byte linenoX = 0
for num 1/19: gen byte sexX = 0
for num 1/19: gen byte relationX = 0
for num 1/19: gen byte ageX = 0

forvalues i = 1/19 {
	by tucaseid: replace lineno`i' = tulineno[`i'] if tulineno==1
	by tucaseid: replace sex`i' = tesex[`i'] if tulineno==1
	by tucaseid: replace relation`i' = terrp[`i'] if tulineno==1
	by tucaseid: replace age`i' = teage[`i'] if tulineno==1
	}

*only keep the first observation line for each HH
drop if tulineno~=1
destring tucaseid, replace
format tucaseid %15.0f

*give the spouse/partner a line number
gen byte partner =.
for num 2/19: replace partner = linenoX if relationX==20 | relationX==21 

*number of own household children < age19
for num 2/19: gen byte HHchildX = 1 if relationX==22 & ageX<19
egen byte num_HHchildren = rowtotal(HHchild2-HHchild14)

*identify households where reside teen girls and/or teen boys 
*identify if household roster member qualifies
for num 2/19: gen byte teengirl_13_17_X = (relationX==22 & ageX>12 & ageX<18 & sexX==2)
for num 2/19: gen byte teengirl_15_17_X = (relationX==22 & ageX>14 & ageX<18 & sexX==2)

for num 2/19: gen byte teenboy_13_17_X = (relationX==22 & ageX>12 & ageX<18 & sexX==1)
for num 2/19: gen byte teenboy_15_17_X = (relationX==22 & ageX>14 & ageX<18 & sexX==1)

*do they have a own child in the household
gen byte parent_HHchildren = 0

forvalues i=2/19{
	replace parent_HHchildren = 1 if relation`i'==22 & age`i'<18
	}

*do they have a own child in the household or non-HH children
gen byte parent_anychildren = 0

forvalues i=2/19{
	replace parent_anychildren = 1 if (relation`i'==22 | relation`i'==40) & age`i'<18
	}

*do they have a teenager in the household
gen byte parent_of_teen_13_17 = 0

forvalues i=2/19{
	replace parent_of_teen_13_17 = 1 if relation`i'==22 & age`i'>12 & age`i'<18
	}

*parent with teenagers aged 15-17
gen byte parent_of_teen_15_17 = 0

forvalues i=2/19{
	replace parent_of_teen_15_17 = 1 if relation`i'==22 & age`i'>14 & age`i'<18
	}
		
*if part of a couple - only keep heterosexual couples
gen byte heterosexual_couple = 0

forvalues i=2/19{
	replace heterosexual_couple = 1 if (relation`i'==20 | relation`i'==21) & sex`i'~=sex1
	}

*if a couple is married to opposite sex 	
gen byte married = 0
forvalues i=2/19{
	replace married = 1 if (relation`i'==20) & sex`i'~=sex1
	}

*if single parent then no spouse or unmarried partner in household
gen byte single = 1

forvalues i=2/19{
	replace single = 0 if relation`i'==20 | relation`i'==21
	}
	
*keep those who are parents of teens ages 13-17 (either part of a heterosexual couple or single parent)
keep if parent_of_teen_13_17==1 & (heterosexual_couple==1 | single==1)

*drop variables used to create who with variables
keep tucaseid partner numHHmem numHHroster num_HHchildren sex* relation* age* heterosexual_couple married single teenboy_15_17_* teengirl_15_17_* parent_of_teen_15_17 parent_of_teen_13_17

sort tucaseid

save "$DATA/parent_roster_`YYYY'.dta", replace

***************************************************
*use ACTIVITY file and merge on who with variables*
***************************************************
use "$ATUSPATH/atusact_`YYYY'.dta", clear
keep tucaseid tuactivity_n tewhere tuactdur tuactdur24 tutier1code tutier2code tutier3code tustarttim tustoptime
destring tucaseid, replace
sort tucaseid tuactivity_n
merge 1:m tucaseid tuactivity_n using "$DATA/whowithactivity_`YYYY'.dta"
keep if _merge==3
drop _merge

*merge on roster
sort tucaseid
merge m:1 tucaseid using "$DATA/parent_roster_`YYYY'.dta"
keep if _merge==3
drop _merge

*variable for whether they were with a certain person on the roster line during the activity (1 is yes, 0 is no)
*this corresponds with age, relation and sex of person
for num 2/19: gen byte withrosX = 0
for num 2/19: replace withrosX = (lineno1==X | lineno2==X | lineno3==X | lineno4==X | lineno5==X | lineno6==X | lineno7==X | lineno8==X | lineno9==X | lineno10==X | lineno11==X | lineno12==X | lineno13==X | lineno14==X | lineno15==X | lineno16==X | lineno17==X | lineno18==X | lineno19==X)

# delimit ;

	forvalues i = 2/19 {;
label variable  withros`i'          "With Person Number `i' on Roster";
label variable  age`i'            	"Edited: Age`i'";
label variable  relation`i'             "Edited: How is `i' related to you?";
label variable  sex`i'            	 "Edited: Sex`i'";

label define labelwithros`i'
1	"Yes"
0	"No";

label define labelrelation`i'
-1    "Blank"
-2    "Don't Know"
-3    "Refused"
18    "Self"
19    "Self"
20    "Spouse (Husband/Wife)"
21    "Unmarried Partner"
22    "Own Household Child"
23    "Grandchild"
24    "Parent (Mother/Father)"
25    "Brother/Sister"
26    "Other Related Person (Aunt, Cousin, Nephew)"
27    "Foster Child"
28    "Housemate/Roommate"
29    "Roomer/Boarder"
30    "Other Nonrelative"
40    "Own Non-Household Child";

label define labelsex`i'
-1    "Blank"
-2    "Don't Know"
-3    "Refused"
1     "Male"
2     "Female";

label values    relation`i'       	labelrelation`i';
label values    sex`i'           	labelsex`i';
label values    withros`i'			labelwithros`i';

};

# delimit cr

*reformat tucaseid to see the entire number in STATA editor
format tucaseid %15.0f

compress

save "$DATA/parent_activity_who_roster_`YYYY'.dta", replace

rm "$DATA/parent_roster_`YYYY'.dta"

rm "$DATA/whowithactivity_`YYYY'.dta"
}