##################################
#
# Utility functions for plotting
#
# Written by M. Fidino
# 
##################################


# gen_mvn: 
#  simulate covariate values over cells in a raster
#
# rast = raster object of plane
#
# mu = numeric vector of length 2. Each mu is proportionally where you want
#  the mean value to be. (0,0) is the bottom left, (1,1) is top right.
# 
# sigma = Variance of covariate on x and y axes
#
# rho = Correlation of covariate on x and y axes
gen_mvn <- function(rast = NULL, mu = NULL,
										sigma = NULL, rho = NULL){
	# error checking
	if(length(mu) != 2 | !is.numeric(mu) | any(mu > 1) | any(mu < 0)){
		stop("mu must be a numeric vector of length 2.")
	}
	if(length(sigma) != 2 | !is.numeric(sigma) | any(sigma < 0)){
		stop("Sigma must be a non-negative numeric vector of length 2.")
	}
	if(length(rho) != 1 | !is.numeric(rho)| rho > 1 |rho < -1){
		stop("rho must be a numeric scalar between -1 and 1.")
	}
	
	# get bounds of raster
	bounds <- extent(rast)
	
	# input a proportion of where you want mu to be on x and y
	mu_loc <- c(bounds@xmin + mu[1] * (bounds@xmax - bounds@xmin),
							bounds@ymin + mu[2] * (bounds@ymax - bounds@ymin))
	
	Sigma <- diag(c(sigma[1] * abs(bounds@xmax - bounds@xmin),
									sigma[2] * abs(bounds@ymax - bounds@ymin)))
	# fill the off diagonal
	Sigma[2:3] <- rep(rho * prod(diag(Sigma)))
	
	to_return <- dmvnorm(xyFromCell(rast, 1:ncell(rast)), 
											 mean=mu_loc, 
											 sigma=Sigma)
}


# plot_dist:
#  This plots out the distribution of a species via gen_process
#  One plot will be generated for each sampling session.
#
# Arguments:
#
# rast: The spatial covariate you want to plot out.
#
# cov_name: A character object of the name of the column in values(rast) you
#  want to plot out. Only required if rast has multiple layers.
# 
# pixel_id: Either a vector of pixels a species is located or a list of vectors
#  generated by the gen_process function.
#
# window: Do you want to plot this out in a window. Defaults to TRUE
#
# ... arguments to be passed to the plotting device or plotting function
#
plot_dist <- function(rast = NULL, cov_name = NULL,
											pixel_id = NULL, window = TRUE, ...){
	
	# get coordinates for plotting
	rast_coord <- xyFromCell(rast, 1:ncell(rast))
  

	
	if(nlayers(rast) > 1){
	# get the values to plot
	vals <- getValues(rast)
	rast <- dropLayer(rast, 2:nlayers(rast))
	values(rast) <- as.numeric(vals[,cov_name])
	}
	
	# open a window if needed
  if(window){
  	open_graph(...)
  }
	
	# number of plots to make
	if(is.list(pixel_id)){
		nplot <- length(pixel_id)
		# plotting by X rows by 2 columns
		plot_rows <- ceiling(nplot / 2)
		par(mfrow = c(plot_rows, 2))
	} else {
		nplot <- 1
		pixel_id <- list(pixel_id)
	}
	
	for(pt in 1:nplot){
	plot(rast, main = paste("T =", pt))
	points(rast_coord[pixel_id[[pt]],], pch = 16)
	}

}

# open_graph: A way to open up a plotting window for different operating 
#  systems.
#
# Arguments: 
#
# width: width of plot, defaults to 7 inches
#
# height: height of plot, defaults to 7 inches
#
# mag: magnification, defaults to 1
#
open_graph <- function( width=7 , height=7 , mag=1.0 , ... ) {
	if ( .Platform$OS.type != "windows" ) { # Mac OS, Linux
		tryInfo = try( X11( width=width*mag , height=height*mag , type="cairo" , 
												... ) )
		if ( class(tryInfo)=="try-error" ) {
			lineInput = readline("WARNING: Previous graphics windows will be closed because of too many open windows.\nTO CONTINUE, PRESS <ENTER> IN R CONSOLE.\n")
			graphics.off() 
			X11( width=width*mag , height=height*mag , type="cairo" , ... )
		}
	} else { # Windows OS
		tryInfo = try( windows( width=width*mag , height=height*mag , ... ) )
		if ( class(tryInfo)=="try-error" ) {
			lineInput = readline("WARNING: Previous graphics windows will be closed because of too many open windows.\nTO CONTINUE, PRESS <ENTER> IN R CONSOLE.\n")
			graphics.off() 
			windows( width=width*mag , height=height*mag , ... )    
		}
	}
}
