# This file is made available under a CC-BY-NC 4.0 International License.
# Details of the license can be found at
# <https://creativecommons.org/licenses/by-nc/4.0/legalcode>. 
# 
# Giving appropriate credit includes citation of the related publication and
# providing a link to the repository:
# 
# Citation: Pulliam, JRC, C van Schalkwyk, N Govender, A von Gottberg, C 
# Cohen, MJ Groome, J Dushoff, K Mlisana, and H Moultrie. (2022) Increased
# risk of SARS-CoV-2 reinfection associated with emergence of Omicron in
# South Africa. _Science_ <https://www.science.org/doi/10.1126/science.abn4947>
# 
# Repository: <https://github.com/jrcpulliam/reinfections>
#
# Note: This script was not used in generating results for the manuscript but is
# provided for comparison. This simulation approach incorporates dynamical noise
# and is therefore preferable to the approach used (`sim_null.R`) but is _much_
# slower. Comparisons of the output (based on limited numbers of runs, due to the
# computational burden) suggested that the projection intervals generated by the
# two methods are similar, so the more efficient method was used.

suppressPackageStartupMessages({
  library(data.table)
})

.debug <- ''
.args <- if (interactive()) sprintf(c(
  file.path('output', 'posterior_90_null.RData'), # input
  file.path('data', 'ts_data_for_analysis.RDS'), # input
  file.path('utils', 'fit_fxn_null.RData'),
  file.path('test.json'), # NOTE: change this to pub.json run for full posterior!
  file.path('output', 'sim_90_null_dyn.RDS') # output
), .debug[1]) else commandArgs(trailingOnly = TRUE)

if(grepl('test', .args[4]) & !grepl('test', .args[1])){
  use_posterior <- file.path('output', 'posterior_90_null_test.RData')
  warning(paste0('Changing posterior input file to ', use_posterior, '.'))
  load(use_posterior)
}else{
  load(.args[1])
}

ts <- readRDS(.args[2])[is.na(elig), elig := 0]
load(.args[3])

configpth <- .args[4]
attach(jsonlite::read_json(configpth))
if(n_sims_per_param > 1)
  warning('Ignoring n_sims_per_param; running 1 simulation per parameter set.')
if(grepl('test', .args[4]) & mcmc$n_posterior > 3){
  mcmc$n_posterior <- 3
  warning('Only producing simulations for the first 3 parameter sets from the posterior to reduce computational time for testing.')
}

if(grepl('test', .args[4]) & !grepl('test', tail(.args, 1))){
  target <- file.path('output', 'sim_90_null_dyn_test.RDS')
  warning(paste0('Changing output file to ', target, '.'))
}else{
  target <- tail(.args, 1)
}

set.seed(2021)

## Calculate expected reinfections using posterior
sim.reinf <- matrix(0, nrow=nrow(ts), ncol=mcmc$n_posterior)
nbin.reinf <- matrix(0, nrow=nrow(ts), ncol=mcmc$n_posterior)

for(kk in 1:mcmc$n_posterior){
  hz <- lambda.post[kk] * ts$ma_tot
  out <- data.frame(matrix(0, nrow=nrow(ts), ncol=(nrow(ts) - cutoff)))
  for(ii in 1:(nrow(ts) - cutoff)){
    
    ## calculate probability of being diagnosed with reinfection by for each day >90 after last diagnosis 
    
    cum.prob <- rep(0, nrow(ts))
    cum.prob[(ii + cutoff):nrow(ts)] <-  (1-exp(-cumsum(hz[(ii + cutoff):nrow(ts)])))  
    
    ## draw ts$cnt[ii] random numbers. This, with the cum.probs will determine when (if at all) reinfection will be diagnosed 
    
    rand.nbrs <- runif(ts$cnt[ii])
    
    ## find position of date where rand.nbrs is smaller than cum.prob for the last time
    
    poss <- numeric(0)
    if(length(rand.nbrs)>0){
      for(jj in 1:length(rand.nbrs)){ 
        poss[jj] <- 2 + nrow(ts) - min(which(rev(cum.prob)<rand.nbrs[jj]))
      }
    }
    
    for(jj in 1:nrow(out)) { out[jj,ii] <- out[jj,ii] + sum(poss==jj)}
    
  }
  
  sim.reinf[,kk] <- apply(out, 1, sum)
  nbin.reinf[,kk] <-  rnbinom(n=nrow(ts), size=1/kappa.post[kk], mu=sim.reinf[,kk])
  print(kk)
}

saveRDS(nbin.reinf, file = target)
