function [FastProsSolution, FastProsStatus, FastProsResult] = ...
        FastPros2(model, biomassRxn, targetRxn, oxygenRxn, options)
%FastPros  is a function of FastPros which attempts to identify sets of knockout candidates for bioproductions.
% Modified by Laurence Legon (30/09/21) to allow termination after a time
% limit has been exceeded
% 
% [FastProsSolution, FastProsStatus, FastProsResult] = ...
%         FastPros(model, biomassRxn, targetRxn, oxygenRxn, options)
% 
% INPUTS
%  model        COBRA model structure containing the following required fields to perform FastPros.
%               Some fields in the model are generated by the function "reduceModelForFP".
%   rxns                    Rxns in the model
%   mets                    Metabolites in the model
%   S                       Stoichiometric matrix (sparse)
%   b                       RHS of Sv = b (usually zeros)
%   c                       Objective coefficients
%   lb                      Lower bounds for fluxes
%   ub                      Upper bounds for fluxes
%   rev                     Reversibility of fluxes
%   rxnAssociations         Each row represents reactions of original model 
%                           used for the reaction of the reduced model
%   rxnAssocMat             Matrix of reaction associations between reduced and original model
%                           (row: reactions in reduced model, column: reactions in original model)
%   unqGeneSetsMat          Matrix of genes-geneSets associations
%   geneSets                List of gene sets created by reaction combination
%   geneSetRxnMat           Matrix of geneSets-rxns associations
%                           (row: geneSets, column, rxns)
%   geneSetAssocRxns        List of reactions associated with gene sets
%   essentialGeneSets       Essential gene sets for the cell growth or the target producton
%   oriRxns                 Reactions in the original model
% 
%  biomassRxn       Reaction representing biomass objective function
%  targetRxn        Reaction whose flux is to be maximized
%  oxygenRxn        Reaction representing oxygen uptake
% 
% 
% OPTIONAL INPUTS
%  options
%   rxnList             Reaction list as knockout candidates 
%                       (default: all reactions in the model)
%   maxKoNum            Maximum knockout number (default: 10)
%   selStrainNum        Knockout strain number to be sel as parent strain in each iteration 
%                       (default: number of possible knockouts)
%   selIncUtargetStrains
%                       Select only strains whose utarget increased by the current knokcout
%                       (default: true)
%   verbFlag            Verbose flag (default: false)
% 
% 
% OUTPUTS
%  FastProsSolution     Solution structure of FastPros
%   koNum                       Knockout number of reaction or gene sets
%   koGeneSetIDs                Each row represents a set of IDs of knocked out genes
%   koGeneSets                  Each row represents a set of knocked out genes
%   koRxnSets                   Each row represents a set of knocked out reactionss
%                               "reaction A/reaction B" means a set of merged reactions without branching.
%                               "reaction C,reaction D" means a set of reactions encoded by the same gene sets.
%   utarget                     Each row represents utarget change by knockout of reaction sets
%                               utarget in the last column is the utarget of each target prodution strain.
%                               if the utarget is zero, alternative solutions including zero target production exist.
%                               if the utarget is infinite, the strain cannot grow over the growth threshold 
%                               without a certain amount of target production
%   prodRates                   Each row represents production rates of the target metabolite
%                               in the corresponding knockout strains
%   flux                        Each column represents flux distributions in the corresponding knockout strains
% 
%  FastProsStatus       Status structure of FastPros
%   existEssentialKoStrains     Column i in represents whether essential knockout strains exist 
%                               in the i th generation
%   existProdKoStrains          Column i in represents whether knockout strains with target production exist 
%                               in the i th iteration
%   firstProdKoNum              Number of knockout when the first production strain was identified
%   firstUnqProdKoNum           Number of knockout when the first production strain 
%                               without zero target production was identified
%   allowableKoGeneSetIDs       Gene set IDs of allowable knockouts
%   allowableKoGeneSets         Gene sets of allowable knockouts
%   allowableKoRxnSets          Reaction sets of allowable knockouts
%   maxKoNum                    Maximum knockout number
%   maxProdStrain               Structure of the identified strain with the maximum target production rate
%    koNum                      Knockout number
%    koGeneSetIDs                IDs of knocked out genes
%    koGeneSets                  Knocked out genes
%    koRxnSets                   Knocked out reactions
%    prodRate                    Production rate of the target metabolite
%    utarget                     utarget change by knockout of gene sets
%    flux                        Flux distribution of the identified strain with maximum target production rate
%   model                       Cobra model
%   modelGetUtarget             Cobra model used for utarget calculation
%   selStrainNum                Number of strains selected as parent strains for the next iteration
%   targetInfo                  Information of target metabolite and reaction
%    met                         Target metabolite abbreviation
%    metID                       Target metabolite ID
%    metMW                       Molecular weight of the target metabolite
%    metName                     Name of target metabolite
%    rxn                         Target reaction whose flux is to be maximized
%    rxnID                       Target reaction ID
%   targetRxn                   Target reaction whose flux is to be maximized
%   targetRxnID                 Target reaction ID
%   theorFlux                   Theoretical flux profile for maximized target production
%   theorProdRate               Theoretical maximum rate of target production
%   time                        Column i represents cumulative CPU time until the end of the i th generation 
% 
%  FastProsResult       Result structure of FastPros
%   essentialGeneSetCmbs        Each row represents essential gene set combinations 
%                               to reach minimum cell growth threshold or target production
%   koNum                       Total knockout number in a strain
%   productionKoStrains         Same structure to solution
%   selKoGeneSetIDs             Gene set IDs knocked out in strains selected as parent strains 
%                               of the next knockout iteration
%   selKoStrainUtarget          utarget in strains selected as parent strains of the next knockout iteration
% 
% 
% Aug. 5th, 2013    Satoshi OHNO


%% Default values

tic
if nargin < 4
    error('Model, biomassRxn, targetRxn and oxygenRxn must be specified.')
end

if ~exist('options','var')
    options.rxnList = model.oriRxns;
    options.maxKoNum = 10;
    options.selStrainNum = [];
    options.selIncUtargetStrains = true;
    options.verbFlag = false;
end
if ~isfield(options,'rxnList')
    options.rxnList = model.oriRxns;
end
if ~isfield(options,'maxKoNum')
    options.maxKoNum = 25;
elseif options.maxKoNum <=0
    error('options.maxKoNum must be larger than zero.')
end
if ~isfield(options,'selStrainNum')
    options.selStrainNum = []; 
elseif options.selStrainNum <= 0
    error('options.selStrainNum must be larger than zero')
end
if ~isfield(options,'selIncUtargetStrains'), options.selIncUtargetStrains = true; end
if ~isfield(options,'verbFlag'), options.verbFlag = false; end

biomassRxnID = findRxnIDs(model,biomassRxn);
if biomassRxnID == 0, error([num2str(biomassRxn) ' not found.']);end
targetRxnID = findRxnIDs(model,targetRxn);
if targetRxnID == 0, error([num2str(targetRxn) ' not found.']);end
oxygenRxnID = findRxnIDs(model,oxygenRxn);
if oxygenRxnID == 0, error([num2str(oxygenRxn) ' not found.']);end

metMWs = computeMW(model,model.mets,0);
targetInfo.rxn = targetRxn;
targetInfo.rxnID = targetRxnID;
targetInfo.metID = find(model.S(:,targetInfo.rxnID));
targetInfo.met = model.mets{targetInfo.metID}(1:end-3);
targetInfo.metName = model.metNames{targetInfo.metID};
targetInfo.metMW = metMWs(targetInfo.metID);
targetInfo = orderfields(targetInfo);


%% FBA on non-deletion strain

model = changeObjective(model,{biomassRxn,targetRxn},[1,10^-5]);
solWTx = solveModel(model);
solWTx(isnan(solWTx))=0;
solWTx = round(solWTx*10^6)/10^6;
if solWTx(biomassRxnID) <= model.lb(biomassRxnID);
    error('The growth rate of wild type strain is less than the growth threshold')
else
    if solWTx(targetRxnID) > 0
        error('Wild type strain produces the target metabolite, so FastPros cannont be applied')
    end
end


%% Calculate theoretical maximum production rate of target metabolite

modelMaxTarget = changeObjective(model,{biomassRxn,targetRxn},[0,1]);
modelMaxTarget = changeRxnBounds(modelMaxTarget,{oxygenRxn},-1000,'l');
solMaxTargetx = solveModel(modelMaxTarget);
solMaxTargetx(isnan(solMaxTargetx))=0;
solMaxTargetx = round(solMaxTargetx*10^6)/10^6;
theorFlux = solMaxTargetx;
theorProdRate = solMaxTargetx(targetRxnID);
if theorProdRate == 0
    error('Target metabolite cannot be produced.')
end


%% Remove essential gene sets from gene sets allowed to be knocked out

allowableKoGeneSetIDs = find(model.essentialGeneSets==0);

inputRxnIDs = findCombinedRxnIDs(model,options.rxnList);
inputRxnIDs = inputRxnIDs(inputRxnIDs~=0);
inputGeneSets = findGeneSetsFromRxns(model,model.rxns(inputRxnIDs));
inputGeneSetIDs = findGeneSetIDs(model,inputGeneSets);
allowableKoGeneSetIDs = columnVector(intersect(allowableKoGeneSetIDs,inputGeneSetIDs));
allowableKoGeneSets = model.geneSets(allowableKoGeneSetIDs);

% Convert gene set IDs to reactions
allowableKoRxnSets = findRxnSetsFromGeneSetIDs(model,allowableKoGeneSetIDs);

if size(allowableKoGeneSetIDs,1) < size(allowableKoGeneSetIDs,2)
    allowableKoGeneSetIDs = allowableKoGeneSetIDs';
    allowableKoRxnSets = allowableKoRxnSets';
end
allowableKoGeneNum = length(allowableKoGeneSetIDs);


%% Initialize FastPros results

if options.verbFlag == true
    disp('FastPros start.')
    disp(['Total ' num2str(allowableKoGeneNum) ' reaction sets are allowed to be knocked out.'])
end
time = zeros(1,options.maxKoNum+1);
startTime = toc;
FastProsStatus.maxKoNum = 0;
FastProsStatus.firstProdKoNum = 0;
FastProsStatus.firstUnqProdKoNum = 0;
FastProsStatus.maxProdStrain.koNum = 0;
FastProsStatus.maxProdStrain.koGeneSetIDs = 0;
FastProsStatus.maxProdStrain.koGeneSets = [];
FastProsStatus.maxProdStrain.koRxnSets = [];
FastProsStatus.maxProdStrain.utarget = 0;
FastProsStatus.maxProdStrain.prodRate = 0;
FastProsStatus.existProdKoStrains = false(1,options.maxKoNum);
FastProsStatus.existEssentialKoStrains = false(1,options.maxKoNum);


%% utarget calculation on non-deletion strain

modelGetUtarget = model;
modelGetUtarget.S = [modelGetUtarget.S;zeros(1,length(model.rxns))];
modelGetUtarget.S(end,targetRxnID) = 1;
modelGetUtarget.b = [modelGetUtarget.b;10^-5];
modelGetUtarget.c(targetRxnID) = 0;
[sol0Utargetx, ~, ~, sol0Utargety] = solveModel(modelGetUtarget);
sol0Utargetx(isnan(sol0Utargetx))=0;
sol0Utargety(isnan(sol0Utargety))=0;
sol0Utargetx = round(sol0Utargetx*10^5)/10^5;
sol0Utargety = round(sol0Utargety*10^5)/10^5;
utarget0 = sol0Utargety(end);
flux0 = sol0Utargetx;
% if options.verbFlag == true
%     disp(' ')
%     disp(['Objective Value is ' num2str(sol0.x(biomassRxnID))])
%     disp(['utarget = ' num2str(utarget0(1)) ' (as shadow price)'])
% end

isKnockedOut=zeros(allowableKoGeneNum,1);
for i = 1 : allowableKoGeneNum
    tempRxnTFs = logical(model.geneSetRxnMat(allowableKoGeneSetIDs(i),:));
    isKnockedOut(i) = sum(flux0(tempRxnTFs) ~= 0,1);
end
zeroFluxGeneSetIDs0 = allowableKoGeneSetIDs(isKnockedOut==0);
if ~isempty(options.selStrainNum)
else
    options.selStrainNum = allowableKoGeneNum;
end

tempResult.koNum = 0;
tempResult.selKoGeneSetIDs = [];
tempResult.selKoStrainUtarget = utarget0;
tempResult.productionKoStrains = struct([]);
tempResult.essentialGeneSetCombinations = [];
FastProsResult(1) = tempResult;
time(1) = toc-startTime;

if toc(options.startTimeVar)>options.timeLimit % CODE HERE WAS ADDED TO ALLOW FOR TERMINATION ONCE TIMELIMIT IS EXCEEDED
    disp('FastPros was terminated prematurely due to time limit being exceeded')
else


flux = flux0;
zeroFluxGeneSetIDs = zeroFluxGeneSetIDs0;


%% FastPros simulation

for tempKoNum = 1 : options.maxKoNum
    if options.verbFlag == true
        disp(' ')
        switch rem(tempKoNum,10)
            case 1
                if tempKoNum == 11; dispData=[num2str(tempKoNum) 'th Knockout Simulation'];
                else dispData=[num2str(tempKoNum) 'st Knockout Simulation']; end
            case 2
                if tempKoNum == 12; dispData=[num2str(tempKoNum) 'th Knockout Simulation'];
                else dispData=[num2str(tempKoNum) 'nd Knockout Simulation'];end
            case 3
                if tempKoNum == 13; dispData=[num2str(tempKoNum) 'th Knockout Simulation'];
                else dispData=[num2str(tempKoNum) 'rd Knockout Simulation'];end
            otherwise; dispData=[num2str(tempKoNum) 'th Knockout Simulation'];
        end
        disp(dispData)
    end
    pre_Flux = flux;
    pre_zeroFluxGeneSetIDs = zeroFluxGeneSetIDs;
    
    [tempResult, FastProsStatus,flux,zeroFluxGeneSetIDs] ...
        ...
        = calcUtargetSelKoStrains(...
        ...
        model, modelGetUtarget, allowableKoGeneSetIDs, ...
        biomassRxnID, targetRxnID, options,...
        pre_Flux, pre_zeroFluxGeneSetIDs,...
        tempKoNum, FastProsResult, FastProsStatus);
    
    if isempty(tempResult)
        break
    end
    FastProsResult(tempKoNum+1) = tempResult;
    time(tempKoNum+1) = toc-startTime;
    
    if options.verbFlag == true
        toc
    end
    if isempty(tempResult.selKoGeneSetIDs)
        if options.verbFlag == true
            disp(' ')
            disp('FastPros ended since no knockout strains are selected for the next generation.')
        end
        break
    end
    
    if ~isempty(tempResult.productionKoStrains) && max(tempResult.productionKoStrains.prodRates)>options.endTol % CODE ADDED HERE TO ALLOW FOR TERMINATION ONCE A MINIMUM LEVEL OF PRODUCTION HAS BEEN ACHIEVED
        disp('FastPros was terminated due to production threshold being exceeded')
        break
    end
    
    if toc(options.startTimeVar)>options.timeLimit % CODE HERE WAS ADDED TO ALLOW FOR TERMINATION ONCE TIMELIMIT IS EXCEEDED
        disp('FastPros was terminated prematurely due to time limit being exceeded')
        break
    end
end

end
%% Arrange results of FastPros

FastProsStatus.time = time(1:FastProsStatus.maxKoNum+1);
FastProsStatus.model = model;
FastProsStatus.allowableKoGeneSetIDs = allowableKoGeneSetIDs;
FastProsStatus.allowableKoGeneSets = allowableKoGeneSets;
FastProsStatus.allowableKoRxnSets = allowableKoRxnSets;
FastProsStatus.modelGetUtarget = modelGetUtarget;
FastProsStatus.targetInfo = targetInfo;
FastProsStatus.targetRxn = targetRxn;
FastProsStatus.targetRxnID = targetRxnID;
FastProsStatus.selStrainNum = options.selStrainNum;
FastProsStatus.theorFlux = theorFlux;
FastProsStatus.theorProdRate = theorProdRate;
FastProsStatus.maxProdRate = FastProsStatus.maxProdStrain.prodRate;

FastProsResult = orderfields(FastProsResult);
FastProsStatus = orderfields(FastProsStatus);

if any(FastProsStatus.existProdKoStrains)
    FastProsSolution = [FastProsResult([false,FastProsStatus.existProdKoStrains]).productionKoStrains];
else
    FastProsSolution = struct([]);
end

