function [Y Y_quant] = determinism(varargin)
% DETERMINISM   Determinism and bootstrap confidence interval for windowed RQA.
%    [Y Z] = DET(X,M,T,R,W,WS) Calculates the windowed RQA measure DET
%    from time series X, using embedding dimension M and delay T, applying
%    an adaptive recurrence threshold to have a fixed recurrence rate R. 
%    The windowing is performed in moving windows of length W and moving
%    step of WS. The output is a vector Y with the DET values and the 
%    5% and 95% percentile confidence bounds.
%
%    Example:
%         N = 2000; % length of time series
%         x = [sin((1:N)*2*pi/70), .5*randn(1,200), sin((1:N)*2*pi/70)]; % exemplary time series
%         [D DQ] = determinism(x,2,17,.1,300,150); % calculate DET
%         plot(D)

gui = 0;

%% settings from arguments
x = varargin{1}(:);
m = 3; tau = 1; rr = .05; win = 200; step = 100;
norm = 'nor';

if nargin >= 2
    m = varargin{2};
end
if nargin >= 3
    tau = varargin{3};
end
if nargin >= 4
    rr = varargin{4};
end
if nargin >= 5
    win = varargin{5};
end
if nargin >= 6
    step = varargin{6};
end

% confidence interval
ci = [5 95];

DET = @(x) sum(x(x > 1)) / sum(x);

%% windowed RQA
if gui, h = waitbar(0,'Boostrapping'); end
pl2 = []; pv2 = []; pw2 = []; det = []; lam = []; v = []; t2 = []; 
plN= []; pvN = []; pwN = [];

cnt = 1;
for i = 1:step:length(x)-win-(m-1)*tau, 
    if gui, waitbar((i/(length(x)-win-(m-1)*tau))/2), end
    D = pdist2(x(i:i+win),x(i:i+win),'chebychev'); % distance matrix
    e = quantile(D(:),rr); % threshold for fixed RR
   
    % recurrence plot by thresholding
    X = D <= e;
    X = double(X) - eye(size(X)); % remove main diagonal
       
   [dummy pl] = dl_all(X); % get diagonal lines in sub-RP
   det(cnt) = DET(pl); % determinism value
   plN(cnt) = length(pl); % number of diagonal lines
   cnt = cnt + 1;
   
   pl2 = [pl2; pl];
end


% bootstrap
nboot = 1000; 
nD = round(median(plN(1:step:end))); % median number of diagonal lines per window

clear boot*
for i = 1:nboot
    if gui, waitbar(0.5+(i/nboot)/2), end
    
    onesample = randsample(pl2,nD,true);
    tmp = DET(onesample);
    bootstatDET(i,:) = (tmp(:))';
      
end


% DET
mb = mean(bootstatDET); sb = std(bootstatDET);

Y_quant = prctile(bootstatDET,ci);
Y = [det'];




function [a_out, b_out]=dl_all(x)
% DL   Mean of the diagonal line lengths and their distribution.
%    A=DL(X) computes the mean of the length of the diagonal 
%    line structures in a recurrence plot.
%
%    [A B]=DL(X) computes the mean A and the lengths of the
%    found diagonal lines, stored in B. In order to get the 
%    histogramme of the line lengths, simply call 
%    HIST(B,[1 MAX(B)]).
%
%    Examples: X = crp(rand(200,1),1,1,.3,'fan','silent');
%              [l l_dist] = dl(X);
%              hist(l_dist,200)
%
%    See also CRQA, DL_KELO, DL_CENSI, TT.

% Copyright (c) 2008-
% Norbert Marwan, Potsdam Institute for Climate Impact Research, Germany
% http://www.pik-potsdam.de
%
% Copyright (c) 2001-2008
% Norbert Marwan, Potsdam University, Germany
% http://www.agnld.uni-potsdam.de
%
%% $Date: 2021/11/23 12:05:30 $
% $Revision: 4.1 $

narginchk(1,1)
nargoutchk(0,2)

warning off
if any(x(:))

  if min(size(x))>100000       % this should speed up the routine; the value
                             % depends on the available memory
    x2=uint8(x);
    N=size(x2);
    x3=zeros(2*N(2)+N(1),N(2));
    x3(N(2)+1:N(2)+N(1),1:N(2))=x2;
    N3=size(x3);
    
    i2=repmat(((1:1+N(2))+N(1)+N(2))',1,N(2));
    i4=i2+repmat((2*N(2)+N(1)+1)*[0:N(2)-1],size(i2,1),1);
    i4(:,end)=[];
    i4=reshape(i4,size(i4,1)*size(i4,2),1);
    x3(i4)=[];
    x3(end)=[];
    x2=(reshape(x3,N(1)+N(2),N(2)))';
  
    x2(end+1,:)=0;
    x=reshape(x2,size(x2,1)*size(x2,2),1);
    x2=x(2:end);x(end)=[];
    z0=find(x==0&x2==1);
    z1=find(x2==0&x==1);
  
  else
  
    N=size(x);
  %   x3=zeros(2*N(2)+N(1),N(2));
  %   x3(N(2)+1:N(2)+N(1),1:N(2))=x;
  %   N3=size(x3);
  %   
  %   i2=repmat(((1:1+N(2))+N(1)+N(2))',1,N(2));
  %   i4=i2+repmat((2*N(2)+N(1)+1)*[0:N(2)-1],size(i2,1),1);
  %   i4(:,end)=[];
  %   i4=reshape(i4,size(i4,1)*size(i4,2),1);
  %   x3(i4)=[];
  %   x3(end)=[];
  %   x=(reshape(x3,N(1)+N(2),N(2)))';
  %  
  %   x(end+1,:)=0;
    
  %  for i1=-ceil(N(2)/2):ceil(N(2)/2); temp=diag(x,i1); X(1:length(temp),1+i1+ceil(N(2)/2))=temp;
  %  end, x=double(X);
    x1=spdiags(double(x));
    z=reshape(x1,size(x1,1)*size(x1,2),1);
    z2(2:length(z)+1)=z;z2(1)=0;z2(end+1)=0;
    z=diff(z2);
    z0=find(z==1);
    z1=find(z==-1);
  
  end
  if length(z0)>length(z1), z0(end)=[]; end
  if length(z1)>length(z0), z1(end)=[]; end
  
  if isempty(z0), z0=0; end
  if isempty(z1), z1=0; end
  
  if z0(1)>z1(1)
    z0(2:end+1)=z0(1:end);z0(1)=0; 
    if length(z0)>length(z1) 
       z0(end)=[];
    end
  end

  l=sort(z1-z0); %l(end)=[];
  l1=l(find(l-1));
  
  if nargout==2
     b_out=zeros(length(l),1);
     b_out=l';
  end
  
  if nargout>0
     a_out=mean(l1);
  else
     mean(l1)
  end
  
else

  if nargout==2
     b_out=NaN;
  end

  if nargout>0
     a_out=NaN;
  else
     NaN
  end

end

warning on
