/*
 * Copyright (C) 2010,2011,2012 Scott Clark. All rights reserved.
 *
 * Developed by:
 * Scott Clark
 * Cornell University Center for Applied Mathematics
 * http://cam.cornell.edu
 * AND
 * Rob Egan
 * Department of Energy Joint Genome Institute
 * http://jgi.doe.gov
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a 
 * copy of this software and associated documentation files (the "Software"), 
 * to deal with the Software without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, 
 * and/or sell copies of the Software, and to permit persons to whom the 
 * Software is furnished to do so, subject to the following conditions:
 *
 *   1. Redistributions of source code must retain the above copyright notice, 
 *      this list of conditions and the following disclaimers.
 *   2. Redistributions in binary form must reproduce the above copyright 
 *      notice, this list of conditions and the following disclaimers in the 
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the names of Cornell University, The Joint Genome Institute, 
 *      nor the names of its contributors may be used to endorse or promote 
 *      products derived from this Software without specific prior written 
 *      permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS WITH THE SOFTWARE.
 */

// For more information on the license please see 
// The University of Illinois/NCSA Open Source License
// http://www.opensource.org/licenses/UoI-NCSA.php

// ALEhelpers.h

#ifndef _ALE_HELPERS_H_
#define _ALE_HELPERS_H_

#include <zlib.h>
#include <assert.h>
#include <sam.h>
#include <bam.h>
#include <search.h>
#include "kseq.h"
#include <math.h>

// define some constants

const static double lnfactconst2 = 0.918938533204672741780329;
const static double SIGNIFICANT_LIBRARY_FRACTION = 0.05;
const static double MAXIMUM_STD_DEV_FRACTION = 0.6667; // stddev can be at most 2/3rds of avg for a 'valid' distribution
const static double minAvgDepth = 10.0;

// define some ALEhelpers static variables
static double _minLogLike;
static int _metagenome;
static double _leastIdentity; // 95% identity is floor, unless there are worse reads that map
static long _multiple_primary_alignments;
static int _minimumQuality;

void setMinLogLike(double min);
double getMinLogLike();

double validateLogLikelihood(const double logLikelihood);

void setMetagenome();
int isMetagenome();

void setLeastIdentity(double identity);
double getLeastIdentity();

void setMultiplePrimaryAlignments();
long getMultiplePrimaryAlignments();


KSEQ_INIT(gzFile, gzread);

/**********************************************
 **** CONSTANTS - MUST BE SET EVERY TIME!!! ****
 **********************************************/

// the lengths must be a multiple of 4 (1048576 = 2^20)
#define MAX_READ_LENGTH             256
#define MAX_LINE_LENGTH             2048

//#define NUM_PAIRED_READS_ON_NODE    4000
//#define NUM_ASSEMBLY_PARTS          1
#define NUM_NODES                   1
#define KMER_LENGTH                 50
#define MIN_DEPTH                   1
#define MAX_NUM_READS_APPENDS_INTO  10000
#define N_PLACEMENTS                1000
#define PLACEMENT_THRESHOLD         4
#define LIKELIHOOD_THRESHOLD        0.000000001
#define INS_OR_STD_THRESHOLD        10.0
#define TEMP_OFFSETS_BUFF           400
#define POISSON_COMB                1

#define mapLens_MAX 100000
#define GCmaps_MAX 400

//static unsigned char powup[8] = {1,2,4,8,16,32,64,128};

// http://en.wikipedia.org/wiki/FASTQ_format
// generated from scripts.py
// lookup of q -> p. p == probability of being correct ( redefine Q=0 as 1/100, Q=0 & Q=1 are redefined as 0.25 as there are only 4 choices )
static double QtoP[63] = {0.25,0.25,0.36904265552,0.498812766373,0.601892829447,0.683772233983,0.748811356849,0.800473768503,0.841510680754,0.874107458821,0.9,0.920567176528,0.936904265552,0.949881276637,0.960189282945,0.968377223398,0.974881135685,0.98004737685,0.984151068075,0.987410745882,0.99,0.992056717653,0.993690426555,0.994988127664,0.996018928294,0.99683772234,0.997488113568,0.998004737685,0.998415106808,0.998741074588,0.999,0.999205671765,0.999369042656,0.999498812766,0.999601892829,0.999683772234,0.999748811357,0.999800473769,0.999841510681,0.999874107459,0.9999,0.999920567177,0.999936904266,0.999949881277,0.999960189283,0.999968377223,0.999974881136,0.999980047377,0.999984151068,0.999987410746,0.99999,0.999992056718,0.999993690427,0.999994988128,0.999996018928,0.999996837722,0.999997488114,0.999998004738,0.999998415107,0.999998741075,0.999999,0.999999205672,0.999999369043};
// lookup of q -> log(p).  Q=0 and Q=1 are limited at 0.25
static double QtoLogP[63] = {-1.386294361,-1.386294361,-0.996843044007323,-0.695524471331768,-0.507675873695919,-0.380130408066409,-0.289268187201664,-0.222551515972973,-0.172556572913703,-0.134551960288297,-0.105360515657826,-0.0827653026686952,-0.0651741731994102,-0.0514182741582494,-0.0406248442212775,-0.0321335740234123,-0.0254397275341249,-0.0201543647615499,-0.0159758692470951,-0.0126691702086946,-0.0100503358535015,-0.00797499827826794,-0.00632956293131057,-0.00502447389071099,-0.00398901726687351,-0.0031672882259884,-0.00251504651167362,-0.0019972555025822,-0.00158615046381827,-0.00125971852431254,-0.00100050033358353,-0.000794643880834491,-0.000631156481354146,-0.000501312870301668,-0.000398186436697967,-0.000316277776543464,-0.000251220196151159,-0.000199546139006532,-0.000158501879759277,-0.000125900466131129,-0.000100005000333347,-7.94359779538046e-05,-6.30977246195429e-05,-5.01199789851444e-05,-3.9811509467615e-05,-3.16232770105271e-05,-2.51191794839762e-05,-1.99528220562603e-05,-1.58490575956675e-05,-1.25893332453737e-05,-1.00000500002878e-05,-7.94331354807621e-06,-6.30959290540238e-06,-5.01188455944388e-06,-3.98107992443596e-06,-3.16228300002451e-06,-2.51188915476935e-06,-1.99526399049202e-06,-1.58489425589492e-06,-1.25892579247174e-06,-1.00000050002909e-06,-7.94328315520852e-07,-6.30957199034177e-07};
// lookup of q -> log((1-p)/4)
// assume probability of next best is uniform random by sym
//static double QtoLogPMiss[63] = {-1.673976434,-1.673976434,-1.84681137972,-2.07706988902,-2.30732839832,-2.53758690762,-2.76784541692,-2.99810392622,-3.22836243552,-3.45862094482,-3.68887945411,-3.91913796342,-4.14939647271,-4.37965498201,-4.60991349132,-4.8401720006,-5.07043050991,-5.30068901919,-5.53094752848,-5.7612060378,-5.99146454711,-6.22172305644,-6.45198156568,-6.68224007506,-6.91249858419,-7.14275709366,-7.37301560271,-7.60327411219,-7.83353262179,-8.06379113064,-8.2940496401,-8.52430814905,-8.75456665946,-8.98482516726,-9.21508367618,-9.44534218665,-9.6756006965,-9.90585920769,-10.1361177161,-10.3663762252,-10.5966347331,-10.8268932483,-11.0571517588,-11.2874102682,-11.5176687717,-11.747927267,-11.9781858014,-12.2084443057,-12.4387028027,-12.6689613262,-12.8992198261,-13.1294783791,-13.3597369152,-13.5899954211,-13.8202537893,-14.0505122651,-14.2807710537,-14.5110295491,-14.741288022,-14.9715467369,-15.2018049191,-15.4320637238,-15.6623224837};

// lookup of q -> log((1.0 - p)*p)  Q=0 and Q=1 are limited at 0.25
// assume probability goes mostly to next most likely (and that we hit that base)
static double QtoLogPMiss[63] = {-1.673976434,-1.673976434,-1.45736006261,-1.38629999923,-1.42870991089,-1.53142295456,-1.670819243,-1.83436108107,-2.01462464731,-2.20687854399,-2.40794560865,-2.61560890497,-2.82827628479,-3.04477889505,-3.26424397442,-3.4860112135,-3.70957587633,-3.93454902284,-4.16062903661,-4.38758084689,-4.61522052184,-4.8434036936,-5.07201676749,-5.30097018783,-5.53019324034,-5.75963002076,-5.9892362881,-6.21897700657,-6.44882441114,-6.67875648804,-6.90875577932,-7.13880843182,-7.36890345482,-7.59903211901,-7.82918750149,-8.05936410331,-8.28955755558,-8.51976439271,-8.74998185681,-8.98020776457,-9.21044037698,-9.4406783232,-9.6709204954,-9.90116602709,-10.1314142221,-10.3616645292,-10.5919165595,-10.8221698974,-11.0524242907,-11.2826795544,-11.512935465,-11.7431919613,-11.9734488637,-12.2037060719,-12.4339634093,-12.6642210663,-12.8944792045,-13.1247371832,-13.3549952457,-13.5852536347,-13.8155115579,-14.045770157,-14.2760287535};

const static double lnfactconst = 0.918938533204672741780329;


// pre-computed for speed, see paper
const static double negBinomZ[2048] = {0.0,-1.0986122886681098,-1.6863989535702286,-1.9965538818740673,-2.1890278101320408,-2.3269094352444308,-2.4345282685475302,-2.5229166728411707,-2.5979562587043410,-2.6631688995269558,-2.7208383950669170,-2.7725327443392769,-2.8193759684597319,-2.8622016542028774,-2.9016452477425374,-2.9382023228473892,-2.9722668689212499,-3.0041573035083737,-3.0341346523918231,-3.0624155689721810,-3.0891818576844665,-3.1145875712074047,-3.1387643876222220,-3.1618257448621327,-3.1838700620077782,-3.2049832792977306,-3.2252408827987784,-3.2447095343478765,-3.2634483956720879,-3.2815102130690965,-3.2989422128039974,-3.3157868455304604,-3.3320824092936769,-3.3478635741332279,-3.3631618263672425,-3.3780058468787675,-3.3924218348309880,-3.4064337859944751,-3.4200637331156254,-3.4333319543744554,-3.4462571548843370,-3.4588566253122512,-3.4711463809958745,-3.4831412843655127,-3.4948551530204242,-3.5063008554292603,-3.5174903959187160,-3.5284349903560792,-3.5391451337240434,-3.5496306606074626,-3.5599007994655518,-3.5699642214412366,-3.5798290843539315,-3.5895030724358374,-3.5989934322944608,-3.6083070055271107,-3.6174502583511776,-3.6264293085731274,-3.6352499501791917,-3.6439176757932024,-3.6524376972212496,-3.6608149642741958,-3.6690541820393658,-3.6771598267527628,-3.6851361604036366,-3.6929872441932523,-3.7007169509524589,-3.7083289766149408,-3.7158268508286176,-3.7232139467845871,-3.7304934903286462,-3.7376685684209758,-3.7447421369959928,-3.7517170282741312,-3.7585959575708965,-3.7653815296434248,-3.7720762446127281,-3.7786825034940752,-3.7852026133665406,-3.7916387922108061,-3.7979931734383348,-3.8042678101374836,-3.8104646790552099,-3.8165856843360051,-3.8226326610345911,-3.8286073784178436,-3.8345115430728400,-3.8403468018336997,-3.8461147445374868,-3.8518169066263512,-3.8574547715989587,-3.8630297733294436,-3.8685432982547945,-3.8739966874449387,-3.8793912385618268,-3.8847282077112864,-3.8900088111992286,-3.8952342271965725,-3.9004055973150620,-3.9055240281059951,-3.9105905924805069,-3.9156063310600704,-3.9205722534589098,-3.9254893395038781,-3.9303585403948498,-3.9351807798095773,-3.9399569549542326,-3.9446879375668509,-3.9493745748707747,-3.9540176904868480,-3.9586180853002593,-3.9631765382915893,-3.9676938073269987,-3.9721706299151420,-3.9766077239295630,-3.9810057882999876,-3.9853655036723623,-3.9896875330421184,-3.9939725223578817,-3.9982211011020028,-4.0024338828455610,-4.0066114657783078,-4.0107544332221163,-4.0148633541152590,-4.0189387834852663,-4.0229812628963364,-4.0269913208825869,-4.0309694733617834,-4.0349162240349408,-4.0388320647674085,-4.0427174759584839,-4.0465729268937620,-4.0503988760855973,-4.0541957716016661,-4.0579640513771968,-4.0617041435207328,-4.0654164666049644,-4.0691014299474713,-4.0727594338819202,-4.0763908700193170,-4.0799961214993470,-4.0835755632323130,-4.0871295621350177,-4.0906584773539718,-4.0941626604850292,-4.0976424557832347,-4.1010982003651906,-4.1045302244055453,-4.1079388513256747,-4.1113243979781586,-4.1146871748214107,-4.1180274860923758,-4.1213456299699089,-4.1246418987366216,-4.1279165789312495,-4.1311699515001283,-4.1344022919394545,-4.1376138704381757,-4.1408049520110586,-4.1439757966306932,-4.1471266593560383,-4.1502577904526792,-4.1533694355145734,-4.1564618355787131,-4.1595352272368764,-4.1625898427448256,-4.1656259101274165,-4.1686436532809221,-4.1716432920711615,-4.1746250424313658,-4.1775891164539241,-4.1805357224814461,-4.1834650651946053,-4.1863773456977604,-4.1892727616014005,-4.1921515071027260,-4.1950137730640735,-4.1978597470877803,-4.2006896135924663,-4.2035035538809415,-4.2063017462137244,-4.2090843658741335,-4.2118515852360483,-4.2146035738261398,-4.2173404983878262,-4.2200625229405055,-4.2227698088406207,-4.2254625148360585,-4.2281407971247731,-4.2308048094073314,-4.2334547029413807,-4.2360906265912961,-4.2387127268796672,-4.2413211480361372,-4.2439160320453650,-4.2464975186916529,-4.2490657456063818,-4.2516208483093560,-4.2541629602553028,-4.2566922128721938,-4.2592087356042798,-4.2617126559505607,-4.2642040995047621,-4.2666831899913911,-4.2691500493057930,-4.2716047975449607,-4.2740475530477138,-4.2764784324265337,-4.2788975506012443,-4.2813050208305237,-4.2837009547474709,-4.2860854623852598,-4.2884586522117356,-4.2908206311573274,-4.2931715046437020,-4.2955113766135851,-4.2978403495557602,-4.3001585245346554,-4.3024660012147597,-4.3047628778883258,-4.3070492514961902,-4.3093252176580732,-4.3115908706950812,-4.3138463036465513,-4.3160916083047800,-4.3183268752265223,-4.3205521937610039,-4.3227676520699996,-4.3249733371474353,-4.3271693348425755,-4.3293557298772667,-4.3315326058667587,-4.3337000453398371,-4.3358581297559509,-4.3380069395240692,-4.3401465540217306,-4.3422770516112221,-4.3443985096571982,-4.3465110045439834,-4.3486146116917475,-4.3507094055733466,-4.3527954597287373,-4.3548728467804922,-4.3569416384517572,-4.3590019055769522,-4.3610537181201554,-4.3630971451873091,-4.3651322550393843,-4.3671591151084801,-4.3691777920089789,-4.3711883515529575,-4.3731908587600303,-4.3751853778726852,-4.3771719723685196,-4.3791507049699785,-4.3811216376584294,-4.3830848316844522,-4.3850403475815671,-4.3869882451743960,-4.3889285835920475,-4.3908614212762647,-4.3927868159956445,-4.3947048248525631,-4.3966155042960180,-4.3985189101292308,-4.4004150975201286,-4.4023041210135565,-4.4041860345360186,-4.4060608914095400,-4.4079287443557913,-4.4097896455102834,-4.4116436464285922,-4.4134907980921740,-4.4153311509229702,-4.4171647547861266,-4.4189916589999720,-4.4208119123462488,-4.4226255630738152,-4.4244326589086898,-4.4262332470614894,-4.4280273742358807,-4.4298150866332620,-4.4315964299609112,-4.4333714494401004,-4.4351401898122900,-4.4369026953453989,-4.4386590098400811,-4.4404091766392186,-4.4421532386293103,-4.4438912382523270,-4.4456232175076371,-4.4473492179589131,-4.4490692807442160,-4.4507834465742047,-4.4524917557452008,-4.4541942481412109,-4.4558909632388772,-4.4575819401166932,-4.4592672174556007,-4.4609468335491425,-4.4626208263044171,-4.4642892332486728,-4.4659520915365629,-4.4676094379527411,-4.4692613089167175,-4.4709077404874158,-4.4725487683700731,-4.4741844279208314,-4.4758147541460351,-4.4774397817148435,-4.4790595449558577,-4.4806740778679881,-4.4822834141209720,-4.4838875870592219,-4.4854866297092171,-4.4870805747799807,-4.4886694546701911,-4.4902533014690595,-4.4918321469625182,-4.4934060226366892,-4.4949749596813202,-4.4965389889927314,-4.4980981411784686,-4.4996524465611110,-4.5012019351805579,-4.5027466367994080,-4.5042865809041066,-4.5058217967095873,-4.5073523131637421,-4.5088781589481650,-4.5103993624826906,-4.5119159519296836,-4.5134279551951808,-4.5149353999328898,-4.5164383135467565,-4.5179367231939231,-4.5194306557904600,-4.5209201380099771,-4.5224051962897223,-4.5238858568290006,-4.5253621455985877,-4.5268340883381182,-4.5283017105612098,-4.5297650375576630,-4.5312240943942417,-4.5326789059218244,-4.5341294967711443,-4.5355758913635933,-4.5370181139049865,-4.5384561883952204,-4.5398901386265429,-4.5413199881878930,-4.5427457604652846,-4.5441674786450079,-4.5455851657182151,-4.5469988444788179,-4.5484085375292391,-4.5498142672810369,-4.5512160559569539,-4.5526139255921727,-4.5540078980408643,-4.5553979949714805,-4.5567842378757266,-4.5581666480621728,-4.5595452466681827,-4.5609200546529545,-4.5622910928066318,-4.5636583817450642,-4.5650219419195741,-4.5663817936112316,-4.5677379569375081,-4.5690904518524098,-4.5704392981477717,-4.5717845154567307,-4.5731261232553422,-4.5744641408602007,-4.5757985874357798,-4.5771294819930359,-4.5784568433913355,-4.5797806903394882,-4.5811010413990232,-4.5824179149849309,-4.5837313293666302,-4.5850413026693282,-4.5863478528761750,-4.5876509978327684,-4.5889507552408739,-4.5902471426695319,-4.5915401775461397,-4.5928298771654452,-4.5941162586919777,-4.5953993391535297,-4.5966791354489800,-4.5979556643472668,-4.5992289424909600,-4.6004989863928785,-4.6017658124437810,-4.6030294369059828,-4.6042898759236852,-4.6055471455156081,-4.6068012615800393,-4.6080522398985764,-4.6093000961315198,-4.6105448458257854,-4.6117865044090376,-4.6130250871965073,-4.6142606093878067,-4.6154930860719103,-4.6167225322268575,-4.6179489627186694,-4.6191723923049954,-4.6203928356372810,-4.6216103072558621,-4.6228248216000809,-4.6240363929996615,-4.6252450356829362,-4.6264507637746819,-4.6276535913008958,-4.6288535321786695,-4.6300506002340880,-4.6312448091871099,-4.6324361726654502,-4.6336247041955652,-4.6348104172090885,-4.6359933250409835,-4.6371734409350545,-4.6383507780385802,-4.6395253494070490,-4.6406971680054161,-4.6418662467050096,-4.6430325982887952,-4.6441962354515223,-4.6453571707964825,-4.6465154168417238,-4.6476709860185075,-4.6488238906728130,-4.6499741430610229,-4.6511217553606157,-4.6522667396613899,-4.6534091079736015,-4.6545488722231125,-4.6556860442543710,-4.6568206358332169,-4.6579526586421851,-4.6590821242892524,-4.6602090442973836,-4.6613334301203313,-4.6624552931274037,-4.6635746446136954,-4.6646914958020957,-4.6658058578353208,-4.6669177417834344,-4.6680271586451170,-4.6691341193435223,-4.6702386347295954,-4.6713407155846332,-4.6724403726157409,-4.6735376164615285,-4.6746324576898930,-4.6757249068014586,-4.6768149742233813,-4.6779026703216564,-4.6789880053873727,-4.6800709896506625,-4.6811516332732417,-4.6822299463485848,-4.6833059389099025,-4.6843796209222077,-4.6854510022875848,-4.6865200928437547,-4.6875869023677312,-4.6886514405690747,-4.6897137171020260,-4.6907737415572708,-4.6918315234598307,-4.6928870722803362,-4.6939403974288432,-4.6949915082528983,-4.6960404140418763,-4.6970871240302099,-4.6981316473913939,-4.6991739932418559,-4.7002141706420488,-4.7012521885955589,-4.7022880560500546,-4.7033217818972526,-4.7043533749737643,-4.7053828440637586,-4.7064101978935131,-4.7074354451395246,-4.7084585944231474,-4.7094796543116049,-4.7104986333218468,-4.7115155399171957,-4.7125303825123988,-4.7135431694668490,-4.7145539090926487,-4.7155626096500658,-4.7165692793513472,-4.7175739263538459,-4.7185765587734174,-4.7195771846715928,-4.7205758120640029,-4.7215724489181374,-4.7225671031532004,-4.7235597826411686,-4.7245504952073221,-4.7255392486328063,-4.7265260506494782,-4.7275109089432243,-4.7284938311594900,-4.7294748248928489,-4.7304538976978510,-4.7314310570803570,-4.7324063105064607,-4.7333796653956171,-4.7343511291277736,-4.7353207090337657,-4.7362884124088467,-4.7372542464986465,-4.7382182185164075,-4.7391803356235886,-4.7401406049473085,-4.7410990335710821,-4.7420556285379201,-4.7430103968513100,-4.7439633454730732,-4.7449144813255320,-4.7458638112951324,-4.7468113422249170,-4.7477570809212093,-4.7487010341491578,-4.7496432086389557,-4.7505836110817228,-4.7515222481316179,-4.7524591264022087,-4.7533942524735640,-4.7543276328882138,-4.7552592741496857,-4.7561891827280620,-4.7571173650581411,-4.7580438275337933,-4.7589685765176197,-4.7598916183368027,-4.7608129592834016,-4.7617326056145375,-4.7626505635505731,-4.7635668392796333,-4.7644814389592582,-4.7653943687061595,-4.7663056346095436,-4.7672152427225560,-4.7681231990654398,-4.7690295096268764,-4.7699341803634487,-4.7708372171963287,-4.7717386260188661,-4.7726384126877974,-4.7735365830326426,-4.7744331428513895,-4.7753280979072024,-4.7762214539341699,-4.7771132166403687,-4.7780033916936722,-4.7788919847403832,-4.7797790013919830,-4.7806644472347957,-4.7815483278177968,-4.7824306486687105,-4.7833114152811191,-4.7841906331216073,-4.7850683076279186,-4.7859444442073213,-4.7868190482411714,-4.7876921250808238,-4.7885636800525360,-4.7894337184499349,-4.7903022455427227,-4.7911692665731396,-4.7920347867539581,-4.7928988112736448,-4.7937613452918484,-4.7946223939409185,-4.7954819623295553,-4.7963400555380744,-4.7971966786212619,-4.7980518366080789,-4.7989055345010874,-4.7997577772764188,-4.8006085698883894,-4.8014579172633995,-4.8023058243023158,-4.8031522958830095,-4.8039973368557698,-4.8048409520492159,-4.8056831462671505,-4.8065239242876752,-4.8073632908647204,-4.8082012507309990,-4.8090378085933247,-4.8098729691351183,-4.8107067370152778,-4.8115391168722326,-4.8123701133195205,-4.8131997309472769,-4.8140279743248238,-4.8148548479967070,-4.8156803564840089,-4.8165045042903314,-4.8173272958911699,-4.8181487357462345,-4.8189688282856178,-4.8197875779225958,-4.8206049890513141,-4.8214210660373960,-4.8222358132300815,-4.8230492349565006,-4.8238613355229170,-4.8246721192116784,-4.8254815902886490,-4.8262897529957600,-4.8270966115574483,-4.8279021701755109,-4.8287064330303684,-4.8295094042853544,-4.8303110880816664,-4.8311114885408184,-4.8319106097656386,-4.8327084558399687,-4.8335050308241074,-4.8343003387636854,-4.8350943836818363,-4.8358871695847432,-4.8366787004585117,-4.8374689802694872,-4.8382580129667074,-4.8390458024817971,-4.8398323527215554,-4.8406176675846790,-4.8414017509406229,-4.8421846066495480,-4.8429662385474170,-4.8437466504549951,-4.8445258461754674,-4.8453038294923498,-4.8460806041732427,-4.8468561739682761,-4.8476305426088571,-4.8484037138109084,-4.8491756912709407,-4.8499464786688575,-4.8507160796699988,-4.8514844979186247,-4.8522517370462728,-4.8530178006659623,-4.8537826923737413,-4.8545464157497422,-4.8553089743567730,-4.8560703717445843,-4.8568306114414543,-4.8575896969666621,-4.8583476318133885,-4.8591044194696869,-4.8598600634026674,-4.8606145670631324,-4.8613679338855933,-4.8621201672950836,-4.8628712706924269,-4.8636212474671972,-4.8643701009988343,-4.8651178346439670,-4.8658644517467788,-4.8666099556376086,-4.8673543496294744,-4.8680976370274580,-4.8688398211090682,-4.8695809051498129,-4.8703208924074222,-4.8710597861157119,-4.8717975895103889,-4.8725343058014152,-4.8732699381864917,-4.8740044898522870,-4.8747379639676147,-4.8754703636897787,-4.8762016921638454,-4.8769319525161876,-4.8776611478619580,-4.8783892813068492,-4.8791163559360697,-4.8798423748227338,-4.8805673410340349,-4.8812912576139258,-4.8820141275977278,-4.8827359540085302,-4.8834567398532034,-4.8841764881308123,-4.8848952018226495,-4.8856128838963819,-4.8863295373118030,-4.8870451650135598,-4.8877597699326536,-4.8884733549876795,-4.8891859230875676,-4.8898974771267776,-4.8906080199857112,-4.8913175545325576,-4.8920260836304150,-4.8927336101222689,-4.8934401368393416,-4.8941456666037135,-4.8948502022288478,-4.8955537465071641,-4.8962563022281520,-4.8969578721654230,-4.8976584590816152,-4.8983580657275949,-4.8990566948395919,-4.8997543491506148,-4.9004510313724889,-4.9011467442141008,-4.9018414903665111,-4.9025352725152196,-4.9032280933284946,-4.9039199554708999,-4.9046108615875053,-4.9053008143205581,-4.9059898162932409,-4.9066778701258151,-4.9073649784225291,-4.9080511437818828,-4.9087363687848384,-4.9094206560046372,-4.9101040080072540,-4.9107864273443811,-4.9114679165589319,-4.9121484781829450,-4.9128281147335624,-4.9135068287305366,-4.9141846226698815,-4.9148614990427753,-4.9155374603305910,-4.9162125090051640,-4.9168866475270621,-4.9175598783471832,-4.9182322039064283,-4.9189036266369079,-4.9195741489575129,-4.9202437732838149,-4.9209125020159226,-4.9215803375443743,-4.9222472822557135,-4.9229133385197814,-4.9235785087022244,-4.9242427951585990,-4.9249062002319386,-4.9255687262592067,-4.9262303755666190,-4.9268911504702935,-4.9275510532802249,-4.9282100862950866,-4.9288682518022311,-4.9295255520848045,-4.9301819894159111,-4.9308375660562564,-4.9314922842606652,-4.9321461462735590,-4.9327991543332379,-4.9334513106633562,-4.9341026174877971,-4.9347530770144656,-4.9354026914443523,-4.9360514629713332,-4.9366993937818240,-4.9373464860468879,-4.9379927419415530,-4.9386381636166083,-4.9392827532284436,-4.9399265129196124,-4.9405694448246784,-4.9412115510649723,-4.9418528337632379,-4.9424932950289096,-4.9431329369615034,-4.9437717616574437,-4.9444097712003536,-4.9450469676657320,-4.9456833531291160,-4.9463189296467789,-4.9469536992745180,-4.9475876640607188,-4.9482208260411733,-4.9488531872464296,-4.9494847496999013,-4.9501155154212713,-4.9507454864115541,-4.9513746646725565,-4.9520030521989522,-4.9526306509755562,-4.9532574629763673,-4.9538834901779714,-4.9545087345364145,-4.9551331980112288,-4.9557568825511140,-4.9563797900940010,-4.9570019225724993,-4.9576232819190826,-4.9582438700457656,-4.9588636888708910,-4.9594827402930344,-4.9601010262156846,-4.9607185485281464,-4.9613353091117469,-4.9619513098459693,-4.9625665525997702,-4.9631810392374458,-4.9637947716133688,-4.9644077515775376,-4.9650199809755158,-4.9656314616373347,-4.9662421953971743,-4.9668521840747681,-4.9674614294856712,-4.9680699334424068,-4.9686776977427050,-4.9692847241849112,-4.9698910145596011,-4.9704965706483684,-4.9711013942260989,-4.9717054870668091,-4.9723088509310829,-4.9729114875752876,-4.9735133987551885,-4.9741145862094198,-4.9747150516792198,-4.9753147968989540,-4.9759138235883071,-4.9765121334717044,-4.9771097282593066,-4.9777066096625564,-4.9783027793804466,-4.9788982391075134,-4.9794929905344301,-4.9800870353412288,-4.9806803752085571,-4.9812730118052633,-4.9818649467993685,-4.9824561818436024,-4.9830467185971354,-4.9836365587071896,-4.9842257038160191,-4.9848141555568546,-4.9854019155588283,-4.9859889854482065,-4.9865753668457575,-4.9871610613605037,-4.9877460706032979,-4.9883303961730423,-4.9889140396682397,-4.9894970026766456,-4.9900792867847992,-4.9906608935728940,-4.9912418246150656,-4.9918220814776380,-4.9924016657246604,-4.9929805789123716,-4.9935588225944807,-4.9941363983182718,-4.9947133076242336,-4.9952895520473879,-4.9958651331222423,-4.9964400523711356,-4.9970143113144339,-4.9975879114678561,-4.9981608543396838,-4.9987331414370013,-4.9993047742572676,-4.9998757542961627,-5.0004460830408517,-5.0010157619793896,-5.0015847925865247,-5.0021531763366216,-5.0027209147019667,-5.0032880091455061,-5.0038544611244218,-5.0044202720954205,-5.0049854435036201,-5.0055499767948675,-5.0061138734099195,-5.0066771347839092,-5.0072397623439198,-5.0078017575173996,-5.0083631217222946,-5.0089238563734551,-5.0094839628834968,-5.0100434426545437,-5.0106022970897754,-5.0111605275886211,-5.0117181355381977,-5.0122751223283126,-5.0128314893409103,-5.0133872379503543,-5.0139423695345391,-5.0144968854586818,-5.0150507870898453,-5.0156040757823899,-5.0161567528960305,-5.0167088197829042,-5.0172602777836435,-5.0178111282433981,-5.0183613725004790,-5.0189110118840023,-5.0194600477242695,-5.0200084813429244,-5.0205563140631586,-5.0211035471997958,-5.0216501820601982,-5.0221962199560224,-5.0227416621876264,-5.0232865100501991,-5.0238307648453411,-5.0243744278578966,-5.0249175003691953,-5.0254599836665461,-5.0260018790253591,-5.0265431877185067,-5.0270839110164918,-5.0276240501821094,-5.0281636064777429,-5.0287025811562938,-5.0292409754755445,-5.0297787906798552,-5.0303160280162782,-5.0308526887233249,-5.0313887740396890,-5.0319242851928774,-5.0324592234205126,-5.0329935899388678,-5.0335273859730671,-5.0340606127364129,-5.0345932714459378,-5.0351253633070243,-5.0356568895288394,-5.0361878513106229,-5.0367182498460963,-5.0372480863332445,-5.0377773619640989,-5.0383060779179480,-5.0388342353837450,-5.0393618355387053,-5.0398888795559555,-5.0404153686066966,-5.0409413038568092,-5.0414666864743314,-5.0419915176186665,-5.0425157984425182,-5.0430395301045765,-5.0435627137476677,-5.0440853505223044,-5.0446074415701219,-5.0451289880254793,-5.0456499910278980,-5.0461704517071482,-5.0466903711901692,-5.0472097506023408,-5.0477285910651952,-5.0482468936939915,-5.0487646596061806,-5.0492818899079639,-5.0497985857080883,-5.0503147481097646,-5.0508303782170545,-5.0513454771194377,-5.0518600459144114,-5.0523740856937600,-5.0528875975406295,-5.0534005825392985,-5.0539130417703264,-5.0544249763111244,-5.0549363872328099,-5.0554472756097502,-5.0559576425036319,-5.0564674889782886,-5.0569768160982189,-5.0574856249142499,-5.0579939164837535,-5.0585016918616601,-5.0590089520896377,-5.0595156982106673,-5.0600219312690067,-5.0605276523047138,-5.0610328623479761,-5.0615375624317318,-5.0620417535841362,-5.0625454368337612,-5.0630486131983288,-5.0635512836989047,-5.0640534493521843,-5.0645551111700033,-5.0650562701661386,-5.0655569273392889,-5.0660570837015992,-5.0665567402525111,-5.0670558979853242,-5.0675545578991521,-5.0680527209853743,-5.0685503882322331,-5.0690475606275491,-5.0695442391541663,-5.0700404247870576,-5.0705361185123401,-5.0710313212970517,-5.0715260341152879,-5.0720202579361136,-5.0725139937259494,-5.0730072424454731,-5.0735000050556094,-5.0739922825124504,-5.0744840757731735,-5.0749753857909132,-5.0754662135078128,-5.0759565598753040,-5.0764464258343205,-5.0769358123245789,-5.0774247202846983,-5.0779131506523605,-5.0784011043551835,-5.0788885823253311,-5.0793755854867557,-5.0798621147678240,-5.0803481710866807,-5.0808337553637974,-5.0813188685127049,-5.0818035114488085,-5.0822876850837062,-5.0827713903229821,-5.0832546280731439,-5.0837373992357753,-5.0842197047125239,-5.0847015453991649,-5.0851829221923577,-5.0856638359854287,-5.0861442876659524,-5.0866242781223638,-5.0871038082378783,-5.0875828788987665,-5.0880614909791060,-5.0885396453581455,-5.0890173429103740,-5.0894945845100938,-5.0899713710232266,-5.0904477033209155,-5.0909235822621435,-5.0913990087163734,-5.0918739835370008,-5.0923485075838837,-5.0928225817112720,-5.0932962067739647,-5.0937693836169151,-5.0942421130922364,-5.0947143960432966,-5.0951862333107032,-5.0956576257400785,-5.0961285741660367,-5.0965990794234282,-5.0970691423459993,-5.0975387637667229,-5.0980079445123829,-5.0984766854096062,-5.0989449872810466,-5.0994128509514720,-5.0998802772365730,-5.1003472669560201,-5.1008138209243787,-5.1012799399529998,-5.1017456248498609,-5.1022108764271135,-5.1026756954898440,-5.1031400828415148,-5.1036040392782320,-5.1040675656071803,-5.1045306626186209,-5.1049933311061055,-5.1054555718676680,-5.1059173856928259,-5.1063787733646873,-5.1068397356749093,-5.1073002734006581,-5.1077603873289075,-5.1082200782343401,-5.1086793468995841,-5.1091381940934024,-5.1095966205928907,-5.1100546271680196,-5.1105122145866737,-5.1109693836174523,-5.1114261350216506,-5.1118824695618343,-5.1123383880010049,-5.1127938910964055,-5.1132489796023384,-5.1137036542712151,-5.1141579158615933,-5.1146117651156295,-5.1150652027892161,-5.1155182296198340,-5.1159708463564204,-5.1164230537405597,-5.1168748525106666,-5.1173262434045315,-5.1177772271601052,-5.1182278045089094,-5.1186779761819547,-5.1191277429131397,-5.1195771054254626,-5.1200260644468027,-5.1204746207032912,-5.1209227749152548,-5.1213705277980255,-5.1218178800793464,-5.1222648324661417,-5.1227113856797990,-5.1231575404295295,-5.1236032974259151,-5.1240486573752060,-5.1244936209883187,-5.1249381889683034,-5.1253823620180192,-5.1258261408404922,-5.1262695261314706,-5.1267125185887776,-5.1271551189065843,-5.1275973277859892,-5.1280391459108570,-5.1284805739741959,-5.1289216126612338,-5.1293622626645794,-5.1298025246635666,-5.1302423993419746,-5.1306818873826954,-5.1311209894614338,-5.1315597062555236,-5.1319980384444204,-5.1324359867005533,-5.1328735516940709,-5.1333107340960415,-5.1337475345750816,-5.1341839537974279,-5.1346199924298350,-5.1350556511335297,-5.1354909305687482,-5.1359258313987075,-5.1363603542797600,-5.1367944998701711,-5.1372282688205670,-5.1376616617862716,-5.1380946794176658,-5.1385273223650705,-5.1389595912785238,-5.1393914868030688,-5.1398230095769977,-5.1402541602507474,-5.1406849394609697,-5.1411153478539475,-5.1415453860601250,-5.1419750547168563,-5.1424043544626787,-5.1428332859249002,-5.1432618497433120,-5.1436900465389073,-5.1441178769446987,-5.1445453415837719,-5.1449724410854367,-5.1453991760678521,-5.1458255471573180,-5.1462515549681216,-5.1466772001259624,-5.1471024832442689,-5.1475274049388222,-5.1479519658225890,-5.1483761665056615,-5.1488000076059226,-5.1492234897259213,-5.1496466134754364,-5.1500693794585164,-5.1504917882826984,-5.1509138405508121,-5.1513355368619997,-5.1517568778177827,-5.1521778640149618,-5.1525984960497855,-5.1530187745260116,-5.1534387000235986,-5.1538582731467377,-5.1542774944793006,-5.1546963646155080,-5.1551148841392980,-5.1555330536400570,-5.1559508736990667,-5.1563683449059550,-5.1567854678371905,-5.1572022430747211,-5.1576186712019778,-5.1580347527878878,-5.1584504884180244,-5.1588658786600448,-5.1592809240918944,-5.1596956252859174,-5.1601099828066470,-5.1605239972333532,-5.1609376691196305,-5.1613509990463902,-5.1617639875694614,-5.1621766352553999,-5.1625889426659954,-5.1630009103611316,-5.1634125389015857,-5.1638238288456533,-5.1642347807485205,-5.1646453951663789,-5.1650556726476626,-5.1654656137544643,-5.1658752190323547,-5.1662844890290724,-5.1666934243016653,-5.1671020253887781,-5.1675102928400598,-5.1679182271945754,-5.1683258290018701,-5.1687330988005309,-5.1691400371326299,-5.1695466445357123,-5.1699529215508626,-5.1703588687104300,-5.1707644865498299,-5.1711697756060007,-5.1715747364111770,-5.1719793694916438,-5.1723836753829486,-5.1727876546136002,-5.1731913077030960,-5.1735946351919360,-5.1739976375928407,-5.1744003154317104,-5.1748026692346230,-5.1752046995149286,-5.1756064068068994,-5.1760077916167235,-5.1764088544671312,-5.1768095958734159,-5.1772100163411752,-5.1776101164001780,-5.1780098965513437,-5.1784093573150871,-5.1788084991927610,-5.1792073226932311,-5.1796058283305060,-5.1800040166067998,-5.1804018880246394,-5.1807994430952897,-5.1811966823108797,-5.1815936061855297,-5.1819902152092343,-5.1823865098889623,-5.1827824907134730,-5.1831781581834120,-5.1835735128015417,-5.1839685550519707,-5.1843632854339443,-5.1847577044337747,-5.1851518125513305,-5.1855456102683055,-5.1859390980746811,-5.1863322764624034,-5.1867251459118551,-5.1871177069145640,-5.1875099599450607,-5.1879019054981983,-5.1882935440463767,-5.1886848760731894,-5.1890759020561097,-5.1894666224782702,-5.1898570378111160,-5.1902471485334241,-5.1906369551249041,-5.1910264580523666,-5.1914156577887649,-5.1918045548103224,-5.1921931495827529,-5.1925814425801970,-5.1929694342677726,-5.1933571251168571,-5.1937445155839628,-5.1941316061441736,-5.1945183972591389,-5.1949048893885781,-5.1952910829946282,-5.1956769785424797,-5.1960625764900934,-5.1964478772960980,-5.1968328814131128,-5.1972175893065407,-5.1976020014260333,-5.1979861182278793,-5.1983699401610615,-5.1987534676822591,-5.1991367012455294,-5.1995196412955229,-5.1999022882861796,-5.2002846426593949,-5.2006667048673174,-5.2010484753546322,-5.2014299545699201,-5.2018111429515459,-5.2021920409499058,-5.2025726489987960,-5.2029529675411803,-5.2033329970187427,-5.2037127378722428,-5.2040921905418775,-5.2044713554603685,-5.2048502330642146,-5.2052288237867348,-5.2056071280656830,-5.2059851463326048,-5.2063628790204755,-5.2067403265566208,-5.2071174893775867,-5.2074943679075707,-5.2078709625768420,-5.2082472738147079,-5.2086233020424668,-5.2089990476917984,-5.2093745111795062,-5.2097496929331513,-5.2101245933803799,-5.2104992129344456,-5.2108735520162508,-5.2112476110508288,-5.2116213904565845,-5.2119948906453866,-5.2123681120408847,-5.2127410550494941,-5.2131137201005089,-5.2134861075879471,-5.2138582179470747,-5.2142300515779745,-5.2146016088965776,-5.2149728903055745,-5.2153438962199115,-5.2157146270485955,-5.2160850831983421,-5.2164552650771974,-5.2168251730872406,-5.2171948076391059,-5.2175641691300889,-5.2179332579659850,-5.2183020745500199,-5.2186706192864714,-5.2190388925660178,-5.2194068948020140,-5.2197746263831046,-5.2201420877131124,-5.2205092791818135,-5.2208762011866154,-5.2212428541287768,-5.2216092383987398,-5.2219753543904339,-5.2223412024975016,-5.2227067831094862,-5.2230720966188748,-5.2234371434172795,-5.2238019238879270,-5.2241664384272584,-5.2245306874155943,-5.2248946712423452,-5.2252583902911800,-5.2256218449543885,-5.2259850356106652,-5.2263479626394425,-5.2267106264278560,-5.2270730273558756,-5.2274351658069120,-5.2277970421638482,-5.2281586567941458,-5.2285200100858908,-5.2288811024146700,-5.2292419341541905,-5.2296025056867093,-5.2299628173809154,-5.2303228696096751,-5.2306826627532947,-5.2310421971772616,-5.2314014732609415,-5.2317604913680382,-5.2321192518788946,-5.2324777551463901,-5.2328360015573105,-5.2331939914664618,-5.2335517252433998,-5.2339092032610814,-5.2342664258771725,-5.2346233934626536,-5.2349801063758248,-5.2353365649836334,-5.2356927696429407,-5.2360487207254547,-5.2364044185844367,-5.2367598635796009,-5.2371150560729669,-5.2374699964209261,-5.2378246849809837,-5.2381791221111644,-5.2385333081730439,-5.2388872435129556,-5.2392409284868346,-5.2395943634559536,-5.2399475487717444,-5.2403004847751316,-5.2406531718359615,-5.2410056102904177,-5.2413578004949626,-5.2417097427965640,-5.2420614375501815,-5.2424128850962610,-5.2427640857891262,-5.2431150399635369,-5.2434657479776217,-5.2438162101721328,-5.2441664268878938,-5.2445163984796608,-5.2448661252752737,-5.2452156076258811,-5.2455648458687856,-5.2459138403521308,-5.2462625914050243,-5.2466110993803854,-5.2469593646032351,-5.2473073874208103,-5.2476551681690111,-5.2480027071788324,-5.2483500047938030,-5.2486970613399349,-5.2490438771576855,-5.2493904525792736,-5.2497367879423491,-5.2500828835709532,-5.2504287398001015,-5.2507743569639844,-5.2511197353914332,-5.2514648754086650,-5.2518097773493269,-5.2521544415353096,-5.2524988683013598,-5.2528430579722940,-5.2531870108693539,-5.2535307273238026,-5.2538742076614220,-5.2542174522015648,-5.2545604612670891,-5.2549032351833915,-5.2552457742774195,-5.2555880788585325,-5.2559301492610908,-5.2562719857998568,-5.2566135887907892,-5.2569549585536617,-5.2572960954183348,-5.2576369996836840,-5.2579776716784199,-5.2583181117200253,-5.2586583201136952,-5.2589982971876470,-5.2593380432448296,-5.2596775586053948,-5.2600168435838679,-5.2603558984916718,-5.2606947236324055,-5.2610333193225607,-5.2613716858795092,-5.2617098236094622,-5.2620477328172726,-5.2623854138089472,-5.2627228669013926,-5.2630600923995310,-5.2633970906059586,-5.2637338618343996,-5.2640704063830741,-5.2644067245553545,-5.2647428166622241,-5.2650786830101834,-5.2654143238927862,-5.2657497396185464,-5.2660849304879145,-5.2664198967993165,-5.2667546388589299,-5.2670891569611760,-5.2674234514130109,-5.2677575225035813,-5.2680913705357213,-5.2684249958101601,-5.2687583986206228,-5.2690915792671245,-5.2694245380401536,-5.2697572752375939,-5.2700897911561428,-5.2704220860864641,-5.2707541603227090,-5.2710860141576417,-5.2714176478849657,-5.2717490617980092,-5.2720802561884099,-5.2724112313411515,-5.2727419875514308,-5.2730725251051327,-5.2734028442956795,-5.2737329454044701,-5.2740628287287112,-5.2743924945472029,-5.2747219431518007,-5.2750511748278148,-5.2753801898562953,-5.2757089885235446,-5.2760375711197369,-5.2763659379229813,-5.2766940892233647,-5.2770220252888302,-5.2773497464182926,-5.2776772528857405,-5.2780045449677973,-5.2783316229532877,-5.2786584871126738,-5.2789851377439092,-5.2793115751018700,-5.2796377994842176,-5.2799638111535359,-5.2802896103977224,-5.2806151974884941,-5.2809405727053651,-5.2812657363196749,-5.2815906886112014,-5.2819154298485378,-5.2822399603130483,-5.2825642802718997,-5.2828883899983721,-5.2832122897703613,-5.2835359798542916,-5.2838594605217857,-5.2841827320481372,-5.2845057946998431,-5.2848286487476779,-5.2851512944577683,-5.2854737321010381,-5.2857959619479731,-5.2861179842634280,-5.2864397993196244,-5.2867614073763649,-5.2870828087012995,-5.2874040035610541,-5.2877249922269209,-5.2880457749550622,-5.2883663520104305,-5.2886867236541661,-5.2890068901602270,-5.2893268517810244,-5.2896466087785381,-5.2899661614226989,-5.2902855099715591,-5.2906046546769279,-5.2909235958118561,-5.2912423336289454,-5.2915608683805413,-5.2918792003425148,-5.2921973297508353,-5.2925152568871034,-5.2928329819932767,-5.2931505053245473,-5.2934678271476567,-5.2937849477064729,-5.2941018672639464,-5.2944185860713233,-5.2947351043882529,-5.2950514224592693,-5.2953675405484368,-5.2956834588976829,-5.2959991777686390,-5.2963146974045889,-5.2966300180651089,-5.2969451399952368,-5.2972600634419074,-5.2975747886631259,-5.2978893159043920,-5.2982036454187798,-5.2985177774469685,-5.2988317122409549,-5.2991454500514150,-5.2994589911185654,-5.2997723356995561,-5.3000854840302827,-5.3003984363563523,-5.3007111929312840,-5.3010237539928875,-5.3013361197905624,-5.3016482905616709,-5.3019602665551879,-5.3022720480116945,-5.3025836351724651,-5.3028950282813376,-5.3032062275827867,-5.3035172333128946,-5.3038280457156644,-5.3041386650266462,-5.3044490914929048,-5.3047593253457137,-5.3050693668277962,-5.3053792161805520,-5.3056888736349244,-5.3059983394386068,-5.3063076138142344,-5.3066166970119601,-5.3069255892603575,-5.3072342907972505,-5.3075428018642024,-5.3078511226864054,-5.3081592535035487,-5.3084671945460711,-5.3087749460514502,-5.3090825082513726,-5.3093898813807243,-5.3096970656684475,-5.3100040613428066,-5.3103108686477034,-5.3106174878043548,-5.3109239190420965,-5.3112301625964422,-5.3115362186962249,-5.3118420875689170,-5.3121477694459642,-5.3124532645535432,-5.3127585731215117,-5.3130636953741632,-5.3133686315428577,-5.3136733818530457,-5.3139779465259966,-5.3142823257986125,-5.3145865198887199,-5.3148905290221711,-5.3151943534235739,-5.3154979933252147,-5.3158014489395811,-5.3161047204937466,-5.3164078082164172,-5.3167107123250794,-5.3170134330384373,-5.3173159705853488,-5.3176183251846307,-5.3179204970575453,-5.3182224864294225,-5.3185242935130672,-5.3188259185321911,-5.3191273617047949,-5.3194286232484211,-5.3197297033926318,-5.3200306023369288,-5.3203313203178828,-5.3206318575414802,-5.3209322142284110,-5.3212323905938019,-5.3215323868560516,-5.3218322032347380,-5.3221318399412798,-5.3224312971913896,-5.3227305751920673,-5.3230296741686232,-5.3233285943334545,-5.3236273358985011,-5.3239258990768299,-5.3242242840804384,-5.3245224911254692,-5.3248205204191761,-5.3251183721759903,-5.3254160466081171,-5.3257135439270513,-5.3260108643414474,-5.3263080080586205,-5.3266049752969522,-5.3269017662582945,-5.3271983811513213,-5.3274948201893038,-5.3277910835807427,-5.3280871715344746,-5.3283830842538586,-5.3286788219455632,-5.3289743848209969,-5.3292697730868896,-5.3295649869467496,-5.3298600266059664,-5.3301548922696247,-5.3304495841439472,-5.3307441024341671,-5.3310384473495365,-5.3313326190829695,-5.3316266178452469,-5.3319204438409731,-5.3322140972663670,-5.3325075783361626,-5.3328008872377497,-5.3330940241798839,-5.3333869893686412,-5.3336797829979270,-5.3339724052690540,-5.3342648563922408,-5.3345571365515072,-5.3348492459607479,-5.3351411848141597,-5.3354329533058475,-5.3357245516421150,-5.3360159800174127,-5.3363072386338892,-5.3365983276809690,-5.3368892473658303,-5.3371799978747489,-5.3374705794181905,-5.3377609921775298,-5.3380512363537518,-5.3383413121542826,-5.3386312197579375,-5.3389209593672442,-5.3392105311733919,-5.3394999353721522,-5.3397891721615887,-5.3400782417324821,-5.3403671442769642,-5.3406558799861843,-5.3409444490590037,-5.3412328516777619,-5.3415210880424135,-5.3418091583463729,-5.3420970627742079,-5.3423848015174995,-5.3426723747725404,-5.3429597827234474,-5.3432470255616780,-5.3435341034836847,-5.3438210166667695,-5.3441077653067444,-5.3443943495901989,-5.3446807697102541,-5.3449670258497175,-5.3452531182005636,-5.3455390469433954,-5.3458248122705525,-5.3461104143655591,-5.3463958534189766,-5.3466811296151109,-5.3469662431369258,-5.3472511941709691,-5.3475359829049607,-5.3478206095255159,-5.3481050742047360,-5.3483893771488278,-5.3486735185156817,-5.3489574985070734,-5.3492413173044238,-5.3495249750809695,-5.3498084720248302,-5.3500918083257023,-5.3503749841506414,-5.3506579996925732,-5.3509408551278757,-5.3512235506394941,-5.3515060864107999,-5.3517884626159606,-5.3520706794429893,-5.3523527370622048,-5.3526346356638426,-5.3529163754156128,-5.3531979565075645,-5.3534793791096362,-5.3537606434019764,-5.3540417495652273,-5.3543226977780716,-5.3546034882112252,-5.3548841210539955,-5.3551645964682484,-5.3554449146441554,-5.3557250757468058,-5.3560050799574146,-5.3562849274526805,-5.3565646184023183,-5.3568441529920694,-5.3571235313848611,-5.3574027537619946,-5.3576818202915497,-5.3579607311587409,-5.3582394865269389,-5.3585180865716602,-5.3587965314707828,-5.3590748213875257,-5.3593529565006959,-5.3596309369876503,-5.3599087630109015,-5.3601864347447252,-5.3604639523544737,-5.3607413160231738,-5.3610185259199366,-5.3612955822084087,-5.3615724850550865,-5.3618492346450806,-5.3621258311395472,-5.3624022747023083,-5.3626785655082285,-5.3629547037261567,-5.3632306895236850,-5.3635065230682573,-5.3637822045282864,-5.3640577340706326,-5.3643331118689597,-5.3646083380794600,-5.3648834128745744,-5.3651583364223647,-5.3654331088912048,-5.3657077304363430,-5.3659822012337344,-5.3662565214411071,-5.3665306912318096,-5.3668047107667798,-5.3670785802101788,-5.3673522997268615,-5.3676258694833994,-5.3678992896347149,-5.3681725603541244,-5.3684456818017621,-5.3687186541365355,-5.3689914775281053,-5.3692641521397695,-5.3695366781222544,-5.3698090556474947,-5.3700812848769806,-5.3703533659627904,-5.3706252990814107,-5.3708970843801893,-5.3711687220256836,-5.3714402121756732,-5.3717115549982717,-5.3719827506366835,-5.3722537992684138,-5.3725247010408426,-5.3727954561231774,-5.3730660646638375,-5.3733365268242315,-5.3736068427652199,-5.3738770126481645,-5.3741470366213848,-5.3744169148466296,-5.3746866474856283,-5.3749562346900817,-5.3752256766136535,-5.3754949734275392,-5.3757641252705266,-5.3760331323082280,-5.3763019946940602,-5.3765707125848037,-5.3768392861343308,-5.3771077154909728,-5.3773760008276197,-5.3776441422808423,-5.3779121400101859,-5.3781799941772146,-5.3784477049247492,-5.3787152724142189,-5.3789826967902599,-5.3792499782144461,-5.3795171168335365,-5.3797841128095447,-5.3800509662807103,-5.3803176774061239,-5.3805842463448759,-5.3808506732351784,-5.3811169582354266,-5.3813831014996003,-5.3816491031710010,-5.3819149634014227,-5.3821806823448020,-5.3824462601545617,-5.3827116969736482,-5.3829769929573086,-5.3832421482449480,-5.3835071629938147,-5.3837720373581330,-5.3840367714718438,-5.3843013654931955,-5.3845658195730310,-5.3848301338519562,-5.3850943084795553,-5.3853583436044019,-5.3856222393749764,-5.3858859959406180,-5.3861496134415328,-5.3864130920242639,-5.3866764318390050,-5.3869396330332302,-5.3872026957489867,-5.3874656201383431,-5.3877284063316564,-5.3879910544929270,-5.3882535647589629,-5.3885159372693590,-5.3887781721747015,-5.3890402696192803,-5.3893022297456330,-5.3895640526990940,-5.3898257386177777,-5.3900872876516805,-5.3903486999417423,-5.3906099756243879,-5.3908711148542388,-5.3911321177686924,-5.3913929845068900,-5.3916537152143382,-5.3919143100328553,-5.3921747690978084,-5.3924350925572071,-5.3926952805580282,-5.3929553332252649,-5.3932152507064730,-5.3934750331508106,-5.3937346806917175,-5.3939941934587212,-5.3942535716147146,-5.3945128152767143,-5.3947719245968280,-5.3950308997121654,-5.3952897407602665,-5.3955484478799596,-5.3958070212085882,-5.3960654608901404,-5.3963237670542856,-5.3965819398463228,-5.3968399793989619,-5.3970978858549632,-5.3973556593425345,-5.3976133000053483,-5.3978708079814428,-5.3981281834041601,-5.3983854264138689,-5.3986425371369888,-5.3988995157195685,-5.3991563622953684,-5.3994130769976518,-5.3996696599636325,-5.3999261113280079,-5.4001824312202036,-5.4004386197869714,-5.4006946771496738,-5.4009506034522481,-5.4012063988257735,-5.4014620634001513,-5.4017175973158640,-5.4019730007029736,-5.4022282736928036,-5.4024834164224798,-5.4027384290211087,-5.4029933116251785,-5.4032480643648437,-5.4035026873719891,-5.4037571807773555,-5.4040115447187320,-5.4042657793198252,-5.4045198847238467,-5.4047738610433687,-5.4050277084266760,-5.4052814270002765,-5.4055350168878640,-5.4057884782258192,-5.4060418111492616,-5.4062950157784293,-5.4065480922448605,-5.4068010406813025,-5.4070538612188388,-5.4073065539840819,-5.4075591191071002,-5.4078115567188689,-5.4080638669407186,-5.4083160499072589,-5.4085681057488619,-5.4088200345851130,-5.4090718365511625,-5.4093235117693403,-5.4095750603753094,-5.4098264824899180,-5.4100777782415337,-5.4103289477600036,-5.4105799911657444,-5.4108309085858348,-5.4110817001544769,-5.4113323659953023,-5.4115829062333329,-5.4118333209924092,-5.4120836103949435,-5.4123337745748454,-5.4125838136482427,-5.4128337277510044,-5.4130835169958234,-5.4133331815199117,-5.4135827214355636,-5.4138321368774536,-5.4140814279635308,-5.4143305948216360,-5.4145796375716504,-5.4148285563370973,-5.4150773512462100,-5.4153260224178998,-5.4155745699734448,-5.4158229940408154,-5.4160712947410721,-5.4163194721961139,-5.4165675265258439,-5.4168154578560221,-5.4170632663088005,-5.4173109520095046,-5.4175585150629715,-5.4178059556045834,-5.4180532737617426,-5.4183004696383152,-5.4185475433688834,-5.4187944950657583,-5.4190413248511158,-5.4192880328524806,-5.4195346191792391,-5.4197810839618530,-5.4200274273068390,-5.4202736493434029,-5.4205197501894427,-5.4207657299647316,-5.4210115887896189,-5.4212573267813200,-5.4215029440550939,-5.4217484407315508,-5.4219938169307236,-5.4222390727724425,-5.4224842083734863,-5.4227292238445806,-5.4229741193165690,-5.4232188948940143,-5.4234635506990010,-5.4237080868544192,-5.4239525034693248};



static const char theFourConverter[256][4] = {"TTTT","GTTT","CTTT","ATTT","TGTT","GGTT","CGTT","AGTT","TCTT","GCTT","CCTT","ACTT","TATT","GATT","CATT","AATT","TTGT","GTGT","CTGT","ATGT","TGGT","GGGT","CGGT","AGGT","TCGT","GCGT","CCGT","ACGT","TAGT","GAGT","CAGT","AAGT","TTCT","GTCT","CTCT","ATCT","TGCT","GGCT","CGCT","AGCT","TCCT","GCCT","CCCT","ACCT","TACT","GACT","CACT","AACT","TTAT","GTAT","CTAT","ATAT","TGAT","GGAT","CGAT","AGAT","TCAT","GCAT","CCAT","ACAT","TAAT","GAAT","CAAT","AAAT","TTTG","GTTG","CTTG","ATTG","TGTG","GGTG","CGTG","AGTG","TCTG","GCTG","CCTG","ACTG","TATG","GATG","CATG","AATG","TTGG","GTGG","CTGG","ATGG","TGGG","GGGG","CGGG","AGGG","TCGG","GCGG","CCGG","ACGG","TAGG","GAGG","CAGG","AAGG","TTCG","GTCG","CTCG","ATCG","TGCG","GGCG","CGCG","AGCG","TCCG","GCCG","CCCG","ACCG","TACG","GACG","CACG","AACG","TTAG","GTAG","CTAG","ATAG","TGAG","GGAG","CGAG","AGAG","TCAG","GCAG","CCAG","ACAG","TAAG","GAAG","CAAG","AAAG","TTTC","GTTC","CTTC","ATTC","TGTC","GGTC","CGTC","AGTC","TCTC","GCTC","CCTC","ACTC","TATC","GATC","CATC","AATC","TTGC","GTGC","CTGC","ATGC","TGGC","GGGC","CGGC","AGGC","TCGC","GCGC","CCGC","ACGC","TAGC","GAGC","CAGC","AAGC","TTCC","GTCC","CTCC","ATCC","TGCC","GGCC","CGCC","AGCC","TCCC","GCCC","CCCC","ACCC","TACC","GACC","CACC","AACC","TTAC","GTAC","CTAC","ATAC","TGAC","GGAC","CGAC","AGAC","TCAC","GCAC","CCAC","ACAC","TAAC","GAAC","CAAC","AAAC","TTTA","GTTA","CTTA","ATTA","TGTA","GGTA","CGTA","AGTA","TCTA","GCTA","CCTA","ACTA","TATA","GATA","CATA","AATA","TTGA","GTGA","CTGA","ATGA","TGGA","GGGA","CGGA","AGGA","TCGA","GCGA","CCGA","ACGA","TAGA","GAGA","CAGA","AAGA","TTCA","GTCA","CTCA","ATCA","TGCA","GGCA","CGCA","AGCA","TCCA","GCCA","CCCA","ACCA","TACA","GACA","CACA","AACA","TTAA","GTAA","CTAA","ATAA","TGAA","GGAA","CGAA","AGAA","TCAA","GCAA","CCAA","ACAA","TAAA","GAAA","CAAA", "AAAA"};

/*****************************
 **** STRUCTS FOR THE TREE ****
 *****************************/

struct IOtriplet{
  float prob;
  float mu;
  float sigma;
};

struct libraryInsertOrientationProbs{
  struct IOtriplet IOs[2]; // 1 : Oriented <- -> ....... 0 : Oriented -> <-; 2
};

struct placement{
  int assemPart;
  int offset1; // index on the assembly that this placement corresponds to
  int offset2;
  double likelihood; // likelihood of this placement (can be recalculated)
  unsigned char placeInfo; // orientation of this placement
  // placeInfo :
  // 1248
  // 00XXXXXX => Oriented <- ->
  // 10XXXXXX => Oriented -> <-
  // 11XXXXXX => Oriented -> ->
  // 01XXXXXX => Oriented <- <-
  // XX1XXXXX => This is the sequence on the assembly
  // XX0XXXXX => The compliment is the sequence on the assmebly
  // XXX1XXXX => left seq first
};

// a paired read! takes (MAX_READ_LENGTH/2 + 5) bytes each.
struct pairedRead{
  float mu; // expected distance between sequences
  float sigma; // standard deviation of distances between sequences
  unsigned char readInfo; // info on stuff like orientation, qvals etc
  // readInfo :
  // 1248
  // 01XXXXXX => Oriented <- ->
  // 10XXXXXX => Oriented -> <-
  // 11XXXXXX => Oriented -> ->
  // 00XXXXXX => Oriented <- <-
  // XX1XXXXX => This is the sequence on the assembly
  // XX0XXXXX => The compliment is the sequence on the assmebly
  // XXX1XXXX => Qvals are present
  // XXX0XXXX => No Qvals
  // XXXX1XXX => Orientation is known
  // XXXX0XXX => Orientation is unknown
  // XXXXX1XX => compliment/non-compliment known/unknown
  unsigned char hasQval; // does the read have quality values?

  double placementNormalizer;

  struct placement placements[N_PLACEMENTS];
  unsigned char numPlacements;
};

struct assemblyPart{
  unsigned int seqLen;
  char *sequence;
  float *depthInward;
  float *depthOutward;
};

struct assembly{
  //struct assemblyPart assemblyParts[NUM_ASSEMBLY_PARTS];
  struct assemblyPart *assemblyParts;
};

struct contig_struct{
  char *name;
  int seqLen;
  unsigned char isCircular;
  char *seq;
  int8_t *seqNum; // used only if realigning
  int ambiguousBaseCount;
  int32_t *ambiguousBasePositions; // used when finding SNP phases
  float *depth;
  float *matchLogLikelihood;
  float *insertLogLikelihood;
  float *depthLogLikelihood;
  float *kmerLogLikelihood;
};

struct assembly_struct{
  int numContigs;
  long totalAssemLen;
  double totalScore;
  double kmerAvgSum;
  double kmerAvgNorm;
  double placeAvgSum;
  double placeAvgNorm;
  double insertAvgSum;
  double insertAvgNorm;
  double depthScoreAvgSum;
  double depthScoreAvgNorm;
  double depthAvgSum;
  double depthAvgNorm;
  double overlapAvgSum;
  double overlapAvgNorm;
  long totalReads;
  long totalUnmappedReads;
  long totalMappedReads;
  long totalPlacedReads;
  double avgReadSize;
  struct contig_struct **contigs;
};

struct realign_parameters_struct {
  int8_t *scoringMatrix;
  int8_t matchScore, mismatchPenalty, gapOpenPenalty, gapExtendPenalty, minSoftClip;
};
typedef struct realign_parameters_struct realign_parameters_t;

enum MATE_ORIENTATION {
  VALID_FR = 0,
  VALID_RF,
  VALID_FF,
  NOT_PROPER_FR,
  NOT_PROPER_RF,
  NOT_PROPER_FF,
  UNMAPPED_PAIR,   // paired, neither mapped, evaluated by read
  CHIMER,     // paired, but to different contigs, evaluated by read
  SINGLE_UNMAPPED_MATE, // paired but only one this read mapped (mate is unmapped)
  UNMAPPED_MATE, // paired but this read unmapped, mate is mapped
  HALF_VALID_MATE, // paired but only one read is observed, delayed evaluation
  SINGLE_READ,// not paired
  UNMAPPED_SINGLE, // not paired, not mapped
  NO_READS,
  MATE_ORIENTATION_MAX
};

const static enum MATE_ORIENTATION MAPPED_PAIRED_ORIENTATION = NOT_PROPER_FF;
const static enum MATE_ORIENTATION IS_PAIRED_ORIENTATION = HALF_VALID_MATE;
const static char *MATE_ORIENTATION_LABELS[MATE_ORIENTATION_MAX] = {
  "FR",
  "RF",
  "FF",
  "NOT_PROPER_FR",
  "NOT_PROPER_RF",
  "NOT_PROPER_FF",
  "UNMAPPED_PAIR",
  "CHIMER",
  "SINGLE_UNMAPPED_MATE",
  "UNMAPPED_MATE",
  "HALF_VALID_MATE",
  "SINGLE_READ",
  "UNMAPPED_SINGLE",
  "NO_READS"
};

struct setOfAlignments{
  double likelihood;
  double likelihoodInsert;
  int start1, start2;
  int end1, end2;
  int contigId1, contigId2;
  char *name;
  bam1_t *bamOfAlignment1;
  bam1_t *bamOfAlignment2;
  enum MATE_ORIENTATION orientation;
  struct setOfAlignments *nextAlignment;
};


struct libraryMateParameters {
  double insertLength; // insert length mean
  double insertStd; // insert length std dev
  double zNormalizationInsert; // z normalization for this orientation
  double libraryFraction; // fraction of mates that map in this orientation
  long count;  // count of reads (in this orientation)
  long mapped; // count of reads that mapped
  long placed; // count of mapped reads that placed
  int isValid;
};

typedef struct libraryMateParameters libraryMateParametersT;

struct libraryParameters {
  libraryMateParametersT mateParameters[MATE_ORIENTATION_MAX];
  long avgReadSize;
  long numReads;
  double totalValidSingleFraction;
  double totalValidMateFraction;
  double totalChimerMateFraction;
  double totalUnmappedFraction;
  int qOff;
  int isSortedByName;
  enum MATE_ORIENTATION primaryOrientation;
};

typedef struct setOfAlignments alignSet_t;
typedef struct contig_struct contig_t;
typedef struct assembly_struct assemblyT;
typedef struct libraryParameters libraryParametersT;

struct _contig_ll {
  contig_t *contig;
  void *next;
};
typedef struct _contig_ll contig_ll;


typedef struct pairedRead pairedRead_t;
typedef struct readSequence readSequence_t;
typedef struct assemblyPart assemblyPart_t;
typedef struct assembly assembly_t;
typedef struct placement placement_t;
typedef struct libraryInsertOrientationProbs libInsOrProbs_t;

/***************************************************************
 **** HELPER FUNCTIONS FOR WRAPPING/RE-WRAPPING THE SEQUENCE ****
 ***************************************************************/

unsigned char seqToChar(const char pos1, const char pos2, const char pos3, const char pos4);
void charToSeq(unsigned char num, char seq[], const int len);
pairedRead_t FindBestPlacements(pairedRead_t theRead, assemblyPart_t thePart, libInsOrProbs_t libProbs);
double GetInsertProbNormal(const double sigma, const double point);
double PlacementLikelihoodFTQ(const readSequence_t read, const assemblyPart_t part, const int offset);
double PlacementLikelihoodFCQ(const readSequence_t read, const assemblyPart_t part, const int offset);
double PlacementLikelihoodBTQ(const readSequence_t read, const assemblyPart_t part, const int offset);
double PlacementLikelihoodBCQ(const readSequence_t read, const assemblyPart_t part, const int offset);
double GetLikelihoodOfPlacementByErrors(const double expErrors);
double GetExpectedError(const char readRes, const char partRes, const double qval);
void combinePlacements(pairedRead_t theRead, placement_t tempPlacements[]);
int SuggestPlacement(placement_t oldPlacements[], placement_t newPlacement, unsigned char numPlacements);
void charToSeqFour(unsigned char num, char seq[]);
double lnfact(double input);
double getQtoP(char qualChar, int qOff);
double getQtoLogP(char qualChar, int qOff);
double getQtoLogPMiss(char qualChar, int qOff);
void IncreaseAssemblyPartsByOne(assembly_t *theAssembly, int numParts);
double poissonInt(int k, double lambda);

void initAlignment(alignSet_t *dst);
void destroyAlignment(alignSet_t *dst);

void copyAlignment(alignSet_t *dst, const alignSet_t *src);
void printAlignment(const alignSet_t *src);

void swap(void **x, void **y);

void printAssembly(assemblyT *theAssembly);

int isGC(char seq);

int getGCtotal(char seq1[], int seq1len);

int findNumAssemPieces(kseq_t *ins);

void readAssembly(kseq_t *ins, assemblyT *theAssembly);

unsigned char *calculateContigGCcont(contig_t *contig, int windowSize);

double getContigAvgDepth(contig_t *contig);

int getSeqMapLenBAM(bam1_t *read);

int getFragmentMapLenBAM(bam1_t *read1);

enum MATE_ORIENTATION getPairedMateOrientation(bam1_t *read1);

enum MATE_ORIENTATION readNextBAM(samfile_t *ins, bam1_t *read1);

int assemblySanityCheck(assemblyT *theAssembly);

assemblyT *loadAssembly(char *filename);

int validateAssemblyIsSameAsAlignment(bam_header_t *header, assemblyT *theAssembly);

void freeContig(contig_t *contig);

void freeAssembly(assemblyT *theAssembly);

samfile_t *openSamOrBam(const char *fileName);

libraryParametersT *computeLibraryParameters(samfile_t *ins, double outlierFraction, int qOff);

int importLibraryParameters(libraryParametersT *libParams, char paramFile[256]);

void saveLibraryParameters(libraryParametersT *libParams, char aleFile[256]);

void writeToOutput(assemblyT *theAssembly, int fullOut, FILE *out);

void setMinimumQuality(int min);

#endif
