%
% run_mech_stauder.m
%
% This script is not expected to run outside of the local UAF network.
% It is included in order to show the checks made in constructing the focal
% mechanism catalog file for Stauder papers.
%
% This script relies on functions in the public github repository at
% https://github.com/carltape/mtbeach
% as well as functions in the GEOTOOLS private repository (local to UAF).
%
% calls read_mech_stauder.m
%
% Carl Tape, 2021-10-18
%

clear, clc, close all
spdy = 86400;

bdir = '/home/carltape/dwrite/source/Stauder_data/';

%--------------------------------------------------------------------------
% checking Stauder and Udias (1963) [1957]

ifile = [bdir 'StauderUdias1963_Table3.txt']; nheader = 7;
[eid,sotime1,sotime2,lat,lon,dep,mag,Pt,Pp,Tt,Tp] = ...
        textread(ifile,'%s%s%s%f%f%f%f%f%f%f%f','headerlines',nheader);

[otime,lon,lat,dep,M,M0,mag,eid] = read_mech_stauder(1);
    
% compare with trend-plunge values from the published table
% COMPARING THE RECOVERED P-T AXIS ANGLES BELOW WITH THE PUBLISHED
% P-T AXIS ANGLES, WE SEE LARGE DISCREPANCIES FOR TWO EVENTS.
% POSSIBLY THIS IS BECAUSE SOME OF THE PUBLISHED P-T AXES ARE NOT PERPENDICULAR.
[gamma,delta,M0,kappa,theta,sigma,K,N,S,thetadc,lam,Useu] = CMT2TT(M,false);
% get P-T plunge and azimuth angles from the basis
Upa = U2pa(Useu,1);     % convert south-east-up to up-south-east
Upa(:,[6 5 2 1]);
disp('                      ------ RECOVERED ANGLES -----   ------ LISTED ANGLES --------');
disp(sprintf('%8s %10s %7s %7s %7s %7s %7s %7s %7s %7s %8s','eid','date','Paz','Ppl','Taz','Tpl','Paz','Ppl','Taz','Tpl','check'));
for ii = 1:length(Upa)
    Xlist    = Upa(ii,[6 5 2 1]);
    Xrecover = [Pt(ii) Pp(ii) Tt(ii) Tp(ii)];
    xcheck = norm(Xlist - Xrecover);
   disp(sprintf('%8s %s %7.1f %7.1f %7.1f %7.1f %7.1f %7.1f %7.1f %7.1f %8.2f',...
       eid{ii},datestr(otime(ii),29),Xlist,Xrecover,xcheck ));
end

% reading StauderUdias1963
% tp2xyz.m: uniform radial value
% tp2xyz.m: uniform radial value
% convert_getbasis.m: south-east-up to up-south-east (GCMT)
% CMT2TT.m: 24 moment tensors to convert into lune + strike/dip/rake
% convert_getbasis.m: up-south-east (GCMT) to south-east-up
% isort = 1: eigenvalues/eigenvectors sorted by lam1 >= lam2 >= lam3
% U2pa: 5/24 p1 vectors need to be flipped
% U2pa: 15/24 p2 vectors need to be flipped
% U2pa: 9/24 p3 vectors need to be flipped
%                       ------ RECOVERED ANGLES -----   ------ LISTED ANGLES --------
%      eid       date     Paz     Ppl     Taz     Tpl     Paz     Ppl     Taz     Tpl    check
%   SU63-1 1950-03-27   343.9     8.2    77.9    25.9   140.0     4.0   106.0    19.0   206.00
%   SU63-2 1956-10-19   135.7     5.4   229.1    32.1   135.0     5.0   230.0    32.0     1.22
%   SU63-3 1955-06-02   153.9     6.8   252.3    51.0   154.0     7.0   252.0    51.0     0.42
%   SU63-4 1955-06-20   166.6    51.2   266.4     7.8   165.0    51.0   267.0     7.0     1.88
%   SU63-5 1955-03-14   157.2    23.0    63.8     7.9   157.0    23.0    64.0     8.0     0.28
%   SU63-6 1957-03-13   212.7    37.0   304.2     1.9   213.0    37.0   304.0     2.0     0.32
%   SU63-7 1957-03-16   136.4    15.5   238.9    37.8   137.0    16.0   238.0    38.0     1.16
%   SU63-8 1957-03-11   155.7     2.6   248.9    51.0   156.0     3.0   248.0    51.0     1.01
%   SU63-9 1957-03-11   150.2    10.2   249.6    42.1   150.0    10.0   250.0    42.0     0.52
%  SU63-11 1957-03-14   329.5    24.3   223.6    31.3   330.0    24.0   223.0    31.0     0.87
%  SU63-12 1958-02-22   226.6    48.0   135.8     0.7   226.0    48.0   136.0     1.0     0.67
%  SU63-13 1957-03-19   125.6     5.2   219.5    37.3   124.0     4.0   222.0    37.0     3.17
%  SU63-14 1958-08-14   168.5     8.2   268.6    50.5   170.0    10.0   265.0    51.0     4.31
%  SU63-15 1957-06-13   169.1    10.1   275.7    58.1   169.0    10.0   276.0    58.0     0.32
%  SU63-16 1957-03-12   156.5    12.6   255.8    35.8   157.0    13.0   255.0    36.0     1.02
%  SU63-17 1957-03-10   159.4     5.7   252.7    29.9   160.0     6.0   252.0    30.0     0.97
%  SU63-18 1957-03-09   153.5    12.6   252.8    35.8   154.0    13.0   252.0    36.0     1.02
%  SU63-19 1957-01-02   188.2    10.2   284.7    32.1   188.0    10.0   285.0    32.0     0.43
%  SU63-20 1957-01-02   176.3    22.2   281.7    33.1   176.0    22.0   282.0    33.0     0.46
%  SU63-21 1955-01-13   185.3    10.9   277.7    11.9   186.0    11.0   277.0    12.0     0.95
%  SU63-22 1957-03-15   182.2    19.1   282.8    28.1   182.0    19.0   283.0    28.0     0.28
%  SU63-23 1957-04-19   252.1    75.0   135.0     7.0   252.0    75.0   135.0     7.0     0.11
%  SU63-24 1957-03-17   177.0     5.0   271.9    44.0   177.0     5.0   272.0    44.0     0.09
%  SU63-25 1957-03-22   330.9     7.4   230.3    54.9   156.0     0.0   216.0    52.0   175.71

%--------------------------------------------------------------------------

% load catalog
[otime,lon,lat,dep,M,M0,mag,eid] = read_mech_stauder;

% check if any events are repeated
[osort,isort] = sort(otime);
for ii = 1:length(osort)-1
   k1 = isort(ii);
   k2 = isort(ii+1);
   odiff = (otime(k2)-otime(k1))*spdy;
   if odiff < 100
   disp(sprintf('%s (%8s) to %s (%8s) --> %12.0f seconds',...
       datestr(otime(k1),31),eid{k1},datestr(otime(k2),31),eid{k2},odiff));
   end
end

display_eq_summary(otime,lon,lat,dep,mag);

% BOUNDS FOR SET OF 73 EVENTS (display_eq_summary.m):
%                               MIN        MAX
%          Origin time : 1950-03-27 1966-08-07
%            Longitude :    -179.70     180.00
%             Latitude :      50.10      62.70
%           Depth (km) :       5.00     142.00
%            Magnitude :       5.00       7.30

%--------------------------------------------------------------------------

% compare with ISCGEM catalog
oran = [min(otime)-1 max(otime)+1];
ax2 = [min(wrapTo360(lon))-10 max(wrapTo360(lon)) min(lat)-3 max(lat)+3];
[otimeA,lonA,latA,depA,magA,eidA] = read_eq_iscgem(oran,ax2,[]);

OTIME_DIFF_SEC_MAX = 200;
HDIFF_DIST_KM = 1000;
args = [OTIME_DIFF_SEC_MAX HDIFF_DIST_KM];
idisplay = true;
idisplay2 = true;

% compare events from two catalogs
[imatch1,imatch2,imatch_otime,otime_diff_sec_all,...
    hdist_diff_km_all,vdist_diff_km_all,mag_diff_all,i1_1notin2,i2_2notin1] ...
    = seismicity_match(args,otime,otimeA,...
    lon,lonA,lat,latA,dep,depA,mag,magA,eid,eidA,idisplay,idisplay2);
  
% --------------------- enter seismicity_match.m -------------------
% seismicity_match.m: 73 events in set A, 548 events in set B
% criteria for matching events: abs(otime1 - otime2) <= 200.0 sec
%   
% max otime difference = 113.28 s (max allowable = 200.00 s)
% 71/73 events in A match 71/548 events in B
% 2/73 events in A DO NOT match 477/548 events in B
% 477/548 events in B DO NOT match 2/73 events in A

% list of A events NOT in B:
%   1  25 otime 1964-09-27 (1964271) 15:50:54 lon -152.00 lat  56.60 dep  27.00 km Mw 5.40 SB66-24
%   2  47 otime 1955-03-14 (1955073) 13:12:04 lon  180.00 lat  52.50 dep  30.00 km Mw 6.87 SU63-5
%
% 1955-03-14: in the ISCGEM supp catalog
% 1964-09-27: NOT in supp catalog: http://www.isc.ac.uk/cgi-bin/web-db-v4?event_id=864711&out_format=IMS1.0&request=COMPREHENSIVE

%--------------------------------------------------------------------------

if 0==1
    % pick an event common to two papers
    % A: paper 1 [SU63], ievent 22
    % B: paper 4 [S68T], ievent 1
    ipickA = 22; ipickB = 1;
    [otimeA,lonA,latA,depA,MA,eidA] = read_mech_stauder(1);
    [otimeB,lonB,latB,depB,MB,eidB] = read_mech_stauder(4);
    datestr(otimeA(ipickA),31)
    datestr(otimeB(ipickB),31)
    
    % note: the auxilliary plane is the one for (-N,-S)
    MpickA = MA(:,ipickA);
    figure; plot_beachballs(MpickA);
    MpickB = MB(:,ipickB);
    figure; plot_beachballs(MpickB);
    
    [gamma,delta,M0,kappa,theta,sigma,K,N,S,thetadc,lam,UA] = CMT2TT(MpickA,true);
    kappa,theta,sigma
    [gamma,delta,M0,kappa,theta,sigma,K,N,S,thetadc,lam,UB] = CMT2TT(MpickB,true);
    kappa,theta,sigma
    
    Upa = U2pa(UA,1);
    % check with the order listed in the table: Paz Ppl Taz Tpl
    disp([Upa([6 5 2 1]) ; [252 75 135 7]]);
    
    % pick an event common to two papers
    % A: paper 3 [S68M], ievent 6
    % B: paper 4 [S68T], ievent 7
    ipickA = 6; ipickB = 7;
    [otimeA,lonA,latA,depA,MA,eidA] = read_mech_stauder(3);
    [otimeB,lonB,latB,depB,MB,eidB] = read_mech_stauder(4);
    datestr(otimeA(ipickA),31)
    datestr(otimeB(ipickB),31)

    % note: the auxilliary plane is the one for (-N,-S)
    MpickA = MA(:,ipickA);
    figure; plot_beachballs(MpickA);
    MpickB = MB(:,ipickB);
    figure; plot_beachballs(MpickB);

    [gamma,delta,M0,kappa,theta,sigma,K,N,S,thetadc,lam,UA] = CMT2TT(MpickA,true);
    kappa,theta,sigma
    [gamma,delta,M0,kappa,theta,sigma,K,N,S,thetadc,lam,UB] = CMT2TT(MpickB,true);
    kappa,theta,sigma

    Upa = U2pa(UA,1);
    % check with the order listed in the table: Paz Ppl Taz Tpl
    disp([Upa([6 5 2 1]) ; [317 67 214 3]]);
    % auxilliary plane
    %[kappa2,theta2,sigma2] = dcfaultpar2aux(kappa,theta,sigma)
end

%==========================================================================
