function [otime,lon,lat,dep,M,M0,mag,eid] = read_mech_stauder(iind)
%READ_MECH_STAUDER read published focal mechanisms from Stauder et al. papers
%
% 1: Stauder and Udias (1963) [1957]
% 2: Stauder and Bollinger (1966) [1964]
% 3: Stauder (1968) [1965]
% 4: Stauder (1968) Tensional character...
%
% This function relies on utilities with the public github repository at
% https://github.com/carltape/mtbeach
%
% called by run_mech_stauder.m
%
% Carl Tape, 2021-10-18

global bdir
NFILES = 4;

bwrite = false;
% directory containing input text files; destinaton for output files
% MUST BE CHANGED TO MATCH THE USER'S LOCAL PATH
bdir = '/home/carltape/dwrite/source/Stauder_data/';
odir = bdir;

% load one file
if nargin==1
    [otime,lon,lat,dep,M,eid] = readMcat(iind);
    [mag,M0] = CMT2mw(M);
    return
end

% load all four files
[otime1,lon1,lat1,dep1,M1,eid1] = readMcat(1);
[otime2,lon2,lat2,dep2,M2,eid2] = readMcat(2);
[otime3,lon3,lat3,dep3,M3,eid3] = readMcat(3);
[otime4,lon4,lat4,dep4,M4,eid4] = readMcat(4);

% subset from the Stauder (1968, Tensional...) that do NOT appear in other three papers
%ipick = 1:length(eid4);
ipick = [3 5 6 9 12 13 14];

% combine the first three sets
otime = [otime1 ; otime2 ; otime3 ; otime4(ipick)];
lon = [lon1 ; lon2 ; lon3 ; lon4(ipick)];
lat = [lat1 ; lat2 ; lat3 ; lat4(ipick)];
dep = [dep1 ; dep2 ; dep3 ; dep4(ipick)];
M = [M1 M2 M3 M4(:,ipick)];
[mag,M0] = CMT2mw(M);
eid = [eid1 ; eid2 ; eid3 ; eid4(ipick)];

if bwrite
    for kk=1:NFILES
        [otimeX,lonX,latX,depX,MX,eidX,ftag] = readMcat(kk);
        ofile = sprintf('%s/psmeca/%s',odir,ftag);
        write_psmeca(ofile,otimeX,latX,lonX,depX,MX,eidX);
        ofile = sprintf('%s/%s',odir,ftag);
        write_mech_table(ofile,otimeX,lonX,latX,depX,MX,eidX);
    end
    % combined catalog
    ofile = [odir '/psmeca/stauder'];
    write_psmeca(ofile,otime,lat,lon,dep,M,eid);
    ofile = [odir 'stauder'];
    write_mech_table(ofile,otime,lon,lat,dep,M,eid);
end

%==========================================================================

function [otime,lon,lat,dep,M,eid,ftag] = readMcat(ipick)

if ipick==4
    ifile = [bdir 'Stauder1968tensional_Table1.txt']; nheader = 7;
    ftag = 'Stauder1968tensional';
    etag = 'S68T';
    disp(sprintf('reading %s',ftag));
    
    [eid,sotime1,sotime2,lat,lon,dep,mag,a,b,c,d] = ...
        textread(ifile,'%s%s%s%f%f%f%f%f%f%f%f','headerlines',nheader);
    n = length(lon);
    for ii=1:n, eid{ii} = [etag '-' eid{ii}]; end
    otime = NaN(n,1);
    for ii=1:n, otime(ii) = datenum([sotime1{ii} ' ' sotime2{ii}],'yyyy-mm-dd HH:MM:SS'); end

    kapA = wrapTo360(a - 90);
    kapB = wrapTo360(c - 90);
    thetaA = b;
    thetaB = d;

    [NA,KA] = sd2NK(kapA,thetaA);
    [NB,KB] = sd2NK(kapB,thetaB);
    % key choice: The slip vector is taken to be MINUS the normal vector of
    %             the second fault plane listed.
    N =  NA;
    S = -NB;

    [kappa,theta,sigma] = NS2sdr(N,S,false);

    M0 = mw2m0(1,mag);
    gamma = zeros(1,n);
    delta = zeros(1,n);
    %whos gamma delta M0 kappa theta sigma
    M = TT2CMT(gamma,delta,M0,kappa,theta,sigma);
    
else
    switch ipick
        case 1, ifile = [bdir 'StauderUdias1963_Table3.txt']; nheader = 7; ftag = 'StauderUdias1963'; etag = 'SU63';
        case 2, ifile = [bdir 'StauderBollinger1966_1964_Table1.txt']; nheader = 5; ftag = 'StauderBollinger1966'; etag = 'SB66';
        case 3, ifile = [bdir 'Stauder1968rat_Table2.txt']; nheader = 5; ftag = 'Stauder1968rat'; etag = 'S68M';
    end
    disp(sprintf('reading %s',ftag));
    
    [eid,sotime1,sotime2,lat,lon,dep,mag,Pt,Pp,Tt,Tp] = ...
        textread(ifile,'%s%s%s%f%f%f%f%f%f%f%f','headerlines',nheader);

    n = length(eid);
    for ii=1:n, eid{ii} = [etag '-' eid{ii}]; end
    otime = NaN(n,1);
    for ii=1:n, otime(ii) = datenum([sotime1{ii} ' ' sotime2{ii}],'yyyy-mm-dd HH:MM:SS'); end

    if 0
    figure; hold on;
    plot(wrapTo360(lon),lat,'ro');
    text(wrapTo360(lon),lat,eid);
    end

    % get basis
    Upa = [Tp Tt Pp Pt];
    Useu = U2pa(Upa,0);

    % get moment tensor
    lam = repmat([1 0 -1]', 1, n);
    M0 = mw2m0(1,mag);
    lam = lam .* repmat(M0',3,1);
    Mseu = CMTrecom(lam,Useu);
    M = convert_MT(5,1,Mseu);
    % [Mw,M0] = CMT2mw(M)
end

end

end

%==========================================================================
