// Cache Model for Symbolic Cache analysis:
// 1. "Symbolic cache": each address generated by the trace is to be encoded as a cache
//    line and the total cache line is user defined.
//    No need to interpret physical address.
// 2. Direct-mapped: a cache block can only go in one spot in the cache.
//    |-tag-|-----Index-----|---Offset---|
// 3. Set associative: n-way set associative
//    |---tag---|---Index---|---Offset---|
// 4. Fully associative: no index needed, only tags.
//    |---------tag---------|---Offset---|
#ifndef CACHE_MODEL_H
#define CACHE_MODEL_H


#include "llvm/ADT/SmallSet.h"
#include "llvm/ADT/SmallVector.h"
#include "llvm/IR/InstIterator.h"
#include "llvm/IR/Instructions.h"
#include "llvm/IR/Value.h"
#include "llvm/IR/DerivedTypes.h"
#include "llvm/IR/ValueMap.h"
#include "llvm/Support/Debug.h"
#include "llvm/Support/raw_ostream.h"

#include <boost/container/flat_set.hpp>

#include <iostream>
#include <fstream>
#include <string>
#include <map>
#include <cmath>
#include <bitset>         // std::bitset
#include <utility>

#define ARCH_SIZE 64 // 64-bit architecture
// #define MAX_CACHE_LINE 512 // Max Cache Line Number

#define DEBUG_TYPE "cachemodel"

using namespace llvm;
using namespace boost;

typedef struct MemOp {
    long long int Addr;                     // Physical Address printed on trace
    bool Type;                              // 0 for Load, 1 for Store
    std::string InstID;                     // Which LLVM Inst is associative with this MemOp
    int InOrderPC;                          // In-order Program Counter
    std::string DebugLoc;                   // Src file debug info
//    unsigned SetIndex;                      // This can uniquely determine which set the MemOp goto
//    long long int Tag;                      // Used for age transfer function
    bool Symbolic;                          // Symbolic Type
    container::flat_set<long long int> SymbolicValues; // Hold all concrete values

    MemOp *Prev;
    MemOp *Succ;

// for later map use
    bool operator < (const MemOp& MemOp) const
    {
        return InstID < MemOp.InstID;
    }
}MemOp;

typedef struct Dep {
    bool Type;            // 1 for Data Dependence, 0 for Control Dependence
    std::string SrcID;
    std::string DestID;
}Dep;




class CacheModel {

public:
    unsigned CacheSize;
    unsigned CacheLineSize;
    unsigned CacheAssoc;
    unsigned CacheOffsetBits;
    unsigned CacheSetNum;
    // Aka Cahce Set bits
    unsigned CacheIndexBits;
    unsigned CacheTagBits;
    unsigned MaxAddr;

    std::vector<MemOp> MemoryOps;
    std::vector<Dep> DDeps;
    std::vector<Dep> CDeps;

    std::string LogFilePath;

    CacheModel(unsigned cacheSize, unsigned cacheLineSize, unsigned cacheAssoc, std::string logFilePath)
    : CacheSize(cacheSize), CacheLineSize(cacheLineSize), CacheAssoc(cacheAssoc), LogFilePath(std::move(logFilePath)){
        auto isPowerOfTwo = [](unsigned x) {
            return (x != 0) && ((x & (x - 1)) == 0);
        };

        if (!(isPowerOfTwo(CacheLineSize) && isPowerOfTwo(CacheSize) && (isPowerOfTwo(CacheAssoc) || CacheAssoc == 1))) {
            LLVM_DEBUG(dbgs() << "Fatal: cache configuration invalid! \n");
        }

        CacheSetNum = CacheSize/(CacheLineSize*CacheAssoc);
        CacheOffsetBits = log2(CacheLineSize);
        CacheIndexBits = log2(CacheSetNum);
        CacheTagBits = ARCH_SIZE - CacheOffsetBits - CacheIndexBits;
        LLVM_DEBUG(dbgs()  << "Cache Setting:\t\t\t" << "\n\tCache Size:\t\t\t" << CacheSize
        << "\n\tCache Assoc:\t\t" << CacheAssoc << "\n\tCache Line Size:\t" << CacheLineSize
        << "\n\tCache Set Number:\t" << CacheSetNum << "\n\tCache Offset Bits:\t" << CacheOffsetBits
        << "\n\tCache Index Bits:\t" << CacheIndexBits << "\n\tCache Tag Bits:\t\t" << CacheTagBits << "\n";);
    }

    void Scan() {
        assert(!LogFilePath.empty());
        errs() << LogFilePath << "\n";
        std::ifstream inFile;
        inFile.open(LogFilePath);
        std::string errMsg = "Cant find log file from the path provided: " + LogFilePath;
        if(!inFile) {
            errs() << errMsg << "\n";
        }

        std::string line;
        int counter = 0;

        while(getline(inFile, line)) {
            std::stringstream ss(line);
            std::istream_iterator<std::string> begin(ss);
            std::istream_iterator<std::string> end;
            std::vector<std::string> vstrings(begin, end);
            if (line.find("Store:") != std::string::npos) {

//                std::bitset<64> addrBi =std::bitset<64>(addr);
//                LLVM_DEBUG(dbgs()  << addrBi.to_string() << "\n");

//                unsigned setIndex = 0;
//                for (unsigned index = CacheOffsetBits; index < (CacheOffsetBits+CacheIndexBits); ++index) {
//                    LLVM_DEBUG(dbgs()  << addrBi[index];);
//                    if (addrBi[index])
//                        setIndex += pow(2, index-CacheOffsetBits);
//                }
//                LLVM_DEBUG(dbgs()  << "\nSetIndex: " << setIndex << "\n");
//                LLVM_DEBUG(dbgs()  << "Tag: " << (addr >> (CacheOffsetBits+CacheIndexBits)) << "\n\n");
                MemOp currOp;
                if (line.find("SymbolicStore:") != std::string::npos){
                    LLVM_DEBUG(dbgs()  << "SymbolicStore: " <<vstrings[2] << "\n");
                    long long int addr = std::strtoll(vstrings[2].c_str(), nullptr, 16);
                    LLVM_DEBUG(dbgs()  << addr << "\n");
                    auto symbolicValues = container::flat_set<long long int> {};

                    long long int addrB = std::strtoll(vstrings[5].c_str(), nullptr, 16);
                    long long int addrE = std::strtoll(vstrings[7].c_str(), nullptr, 16);
                    int stepSize = std::strtol(vstrings[9].c_str(), nullptr, 10);

                    for (; addrB <= addrE; addrB+=stepSize) {
                        symbolicValues.insert(addrB);
                    }
                    currOp = {.Addr =  addr, .Type = true, .InstID = vstrings[11], .InOrderPC = counter,
                              .DebugLoc = vstrings[13], .Symbolic = true, .SymbolicValues = symbolicValues};
                }
                else{
                    LLVM_DEBUG(dbgs()  << "Store: " <<vstrings[1] << "\n");
                    long long int addr = std::strtoll(vstrings[1].c_str(), nullptr, 16);
                    LLVM_DEBUG(dbgs()  << addr << "\n");
                    currOp = {.Addr =  addr, .Type = true, .InstID = vstrings[3], .InOrderPC = counter,
                              .DebugLoc = vstrings[5], .Symbolic = false, .SymbolicValues = {}};
                }
                MemoryOps.push_back(currOp);
                counter++;
            }
            else if (line.find("Load:") != std::string::npos) {

//                std::bitset<64> addrBi =std::bitset<64>(addr);
//                LLVM_DEBUG(dbgs()  << addrBi.to_string() << "\n");
//                unsigned setIndex = 0;
//                for (unsigned index = CacheOffsetBits; index < (CacheOffsetBits+CacheIndexBits); ++index) {
//                    LLVM_DEBUG(dbgs()  << addrBi[index];);
//                    if (addrBi[index])
//                        setIndex += pow(2, index-CacheOffsetBits);
//                }
//                LLVM_DEBUG(dbgs()  << "\nSetIndex: " << setIndex << "\n");
//                LLVM_DEBUG(dbgs()  << "Tag: " << (addr >> (CacheOffsetBits+CacheIndexBits)) << "\n\n");

                MemOp currOp;
                if (line.find("SymbolicLoad:") != std::string::npos){
                    LLVM_DEBUG(dbgs()  << "SymbolicLoad: " <<vstrings[2] << "\n");
                    long long int addr = std::strtoll(vstrings[2].c_str(), nullptr, 16);
                    LLVM_DEBUG(dbgs()  << addr << "\n");
                    auto symbolicValues = container::flat_set<long long int> {};

                    long long int addrB = std::strtoll(vstrings[5].c_str(), nullptr, 16);
                    long long int addrE = std::strtoll(vstrings[7].c_str(), nullptr, 16);
                    int stepSize = std::strtol(vstrings[9].c_str(), nullptr, 10);

                    for (; addrB <= addrE; addrB+=stepSize) {
                        symbolicValues.insert(addrB);
                    }
                    currOp = {.Addr =  addr, .Type = true, .InstID = vstrings[11], .InOrderPC = counter,
                              .DebugLoc = vstrings[13], .Symbolic = true, .SymbolicValues = symbolicValues};
                }
                else{
                    LLVM_DEBUG(dbgs()  << "Load: " << vstrings[1] << "\n");
                    long long int addr = std::strtoll(vstrings[1].c_str(), nullptr, 16);
                    LLVM_DEBUG(dbgs()  << addr << "\n");
                    currOp = {.Addr =  addr, .Type = false, .InstID = vstrings[3], .InOrderPC = counter,
                              .DebugLoc = vstrings[5], .Symbolic = false, .SymbolicValues = {}};
                }
//                                .SetIndex = setIndex, .Tag = addr >> (CacheOffsetBits+CacheIndexBits)};
                MemoryOps.push_back(currOp);
                counter++;
            }
            else if (line.find("DDEP: ") != std::string::npos) {
                Dep currDep = {.Type = true, .SrcID = vstrings[2], .DestID = vstrings[4]};
                LLVM_DEBUG(dbgs() << "DDEP: " << currDep.SrcID << " : " << currDep.DestID << "\n\n";);
                if(currDep.SrcID != currDep.DestID)
                    DDeps.push_back(currDep);
            }
            else if (line.find("CDEP: ") != std::string::npos) {
                Dep currDep = {.Type = false, .SrcID = vstrings[2], .DestID = vstrings[4]};
                LLVM_DEBUG(dbgs() << "CDEP: " << currDep.SrcID << " : " << currDep.DestID << "\n\n";);
                if(currDep.SrcID != currDep.DestID)
                    CDeps.push_back(currDep);
            }
        }
    }

    unsigned GetOffsetBits(){
        return CacheOffsetBits;
    }
    
    
/**
 * @brief GetTySize - calculate the size of a pointer
 * @param ty
 * @return pointer size
 */
    static unsigned GetTySize(Type *ty) {
        Type *eleTy;
        unsigned len;
        unsigned size;

        switch (ty->getTypeID()) {
            case Type::ArrayTyID: {
                ArrayType *arrayTy = dyn_cast<ArrayType>(ty);
                len = arrayTy->getArrayNumElements();
                eleTy = arrayTy->getElementType();
                return len * GetTySize(eleTy);
            }
            case Type::PointerTyID: {
                // PointerType *pointerTy = dyn_cast<PointerType>(ty);
                // len = 1;
                // eleTy = pointerTy->getElementType();
                // return len*GetTySize(eleTy);
                return ARCH_SIZE;
            }
            case Type::VectorTyID: {
                VectorType *vectorTy = dyn_cast<VectorType>(ty);
                len = vectorTy->getVectorNumElements();
                eleTy = vectorTy->getElementType();
                return len * GetTySize(eleTy);
            }
            case Type::StructTyID: {
                size = 0;
                StructType *stTy = dyn_cast<StructType>(ty);
                len = stTy->getStructNumElements();
                while (len--) {
                    size += GetTySize(stTy->getElementType(len));
                }
                return size;
            }
            case Type::HalfTyID: {
                return 2;
            }
            case Type::FloatTyID: {
                return 4;
            }
            case Type::DoubleTyID:
            case Type::X86_MMXTyID: {
                return 8;
            }
            case Type::X86_FP80TyID: {
                return 10;
            }
            case Type::FP128TyID:
            case Type::PPC_FP128TyID: {
                return 16;
            }
            case Type::IntegerTyID: {
                IntegerType *intTy = dyn_cast<IntegerType>(ty);
//                LLVM_DEBUG(dbgs() << "'Type Checking' Integer Type Size is : " << intTy->getBitWidth() / 8 << "\n");
                return intTy->getBitWidth() / 8;
            }
            case Type::VoidTyID: {
                LLVM_DEBUG(dbgs() << "'Type Checking' error! Should not be Void Type!\n");
                return 0;
            }
            default: {
                LLVM_DEBUG(dbgs() << "'Type Checking' error! Unexpected Type ID: " << ty->getTypeID() << "\n");
                return 0;
            }
        }
    }

};

#undef  DEBUG_TYPE

#endif //CACHE_MODEL_H
