;+
;========================================================================
;;;
;;; FILE NAME:    $Id: limit_precision.pro 5412 2019-04-16 19:45:54Z psb6 $
;;;
;;; DESCRIPTION:  This routine converts a real number to a string with the
;;;               specified number of significant digits.  
;;;
;;; AUTHOR:       Pat Broos (patrick.broos@icloud.com)
;;;               Scott Koch (tsk@astro.psu.edu)
;;;               Copyright (C) 1996, Pennsylvania State University
;;;
;;; NOTES:        
;;;
;-
;========================================================================
FUNCTION limit_precision, value, sig_digits

if (value EQ 0) then return, '0.0'

if finite(value, /NAN) then return, 'NaN'

if finite(value, /INFINITY, SIGN=-1) then return, '-Inf'
if finite(value, /INFINITY, SIGN= 1) then return, '+Inf'

;; Determine if the value should be rounded up or down.
;; Our method is to scale by powers of 10 to get the requested number of digits to the left
;; of the decimal, and then use round() to make the up or down rounding decision.
right_shift_of_decimal_point = sig_digits - ceil( alog10( abs(double(value)) ) )

shifted_value = value * 10.0D^right_shift_of_decimal_point

round_up = (shifted_value LE round(shifted_value))



;; The integer round(shifted_value) contains the digits we want in our result,
;; but we cannot simply insert a decimal place in the correct place because we want 
;; the result to be formatted by IDL's "G" format code (to handle large/small values).
;; Thus, we must iteratively nudge value up or down by small amounts, pass that 
;; through the G formatting, and test whether we've achieved the rounding we want.

fmt = string( sig_digits, f='("(G20.",I0,")")' )
increment = (10.0D^(-right_shift_of_decimal_point)) / 10

done  = 0
steps = 0
while (~done) do begin
  if (round_up) then begin
    ; Nudge upward
    nudged_value  = value + steps*increment
;   print, nudged_value
    return_string = string(nudged_value, f=fmt )
    done = (value LE double( return_string ))
  endif else begin
    ; Nudge downward
    nudged_value = value - steps*increment
;   print, nudged_value
    return_string = string(nudged_value, f=fmt )
    done = (value GE double( return_string ))
  endelse

  steps++
  if (steps GE 100) then message, 'Loop failed to converge.'
endwhile

; Clean up the return string.
return_string = strcompress( return_string, /REMOVE_ALL )
if strmatch(return_string, '*.') then return_string = strmid(return_string, 0, strlen(return_string)-1)
return, return_string
end





PRO test_limit_precision, BASE=base, SIG_DIGITS=sig_digits

if ~keyword_set(sig_digits) then sig_digits = 4
help,  sig_digits
print, limit_precision( -!PI, sig_digits)
print, limit_precision(  !PI, sig_digits)

base = (n_elements(base) EQ 1) ? base : random()
for i=-10,10 do begin
  val = base * 10.^i
  pos = limit_precision( val,sig_digits)
  neg = limit_precision(-val,sig_digits)
  print, val, pos, neg, abs(100*[double(pos)-val, double(neg)+val] / val), F='(%"%14.8g => %12s %12s   (%0.2g%%  %0.2g%%)")'
endfor
return
end
