;;; $Id: interval_to_phase_density.pro 4834 2015-04-22 15:46:28Z psb6 $


;;; The "time" interval over which we wish to integrate phase is defined by the 'tstart' and 'tstop' inputs, which must be in units of 'phase', i.e. a real-valued 'period number' with respect to some zero-phase time (t0).  This time interval can be smaller than or larger than one period.
;;; 
;;; The 'phase_bin_edges' input vector defines a phase binning scheme that divides the phase interval [0,1] into N equally-sized bins. 
;;; The N+1 bin edges are listed in 'phase_bin_edges'.
;;; 
;;; The N-element 'phase_density_in_bin' input/output vector accumulates the integral of the time intervals over each phase bin.
;;; 
;;; The scalar 'area_in_one_period' input specifies the integral desired for an interval length of one period.
;;;                                   

;;; Enabling 'debug' mode adds little extra CPU time.  Below are example reports from the csh 'time' command:                             
;;;   DEBUG==0: 184.694u 0.226s 3:13.56 95.5%	0+0k 1+21io 0pf+0w
;;;   DEBUG==1: 188.150u 0.246s 3:16.49 95.8%	0+0k 0+9io 0pf+0w

PRO interval_to_phase_density, tstart_p, tstop_p, area_in_one_period, phase_bin_edges, phase_density_in_bin, $
    DEBUG=debug                            
  
if ~keyword_set(debug) then debug = 0

tstart = tstart_p
tstop  = tstop_p 

num_intervals         = n_elements(tstart)
phase_bins_per_period = n_elements(phase_bin_edges) - 1
area_per_phasebin     = double(area_in_one_period) / phase_bins_per_period


;; ----------------------------------------------------------------------------
;; Check validity of parameters.
if ( size(/TYPE,tstart) NE 5 || size(/N_ELEMENTS,tstart) NE num_intervals ) then begin
  help, tstart, tstop
  message, 'Parameter tstart must be type DOUBLE.'
endif
if ( size(/TYPE,tstop) NE 5 || size(/N_ELEMENTS,tstop) NE num_intervals ) then begin
  help, tstart, tstop
  message, 'Parameter tstop must be type DOUBLE, with the same number of elements as tstart.'
endif
; Verify tstart < tstop
if ~array_equal( (tstart LE tstop), 1) then begin
  help, tstart, tstop
  message, 'Parameter tstart must be <= tstop.'
endif



if n_elements(area_in_one_period) NE 1 then begin
  help, area_in_one_period
  message, 'Parameter area_in_one_period must be a scalar.'
endif


if ( size(/N_ELEMENTS,phase_density_in_bin) NE phase_bins_per_period ) then begin
  help, phase_bin_edges, phase_density_in_bin
  message, 'The length of parameter phase_density_in_bin does not correspond to the number of bins represented by parameter phase_bin_edges.'
endif

if (phase_bin_edges[ 0] NE 0) then message, 'First element of phase_bin_edges should be zero.'
if (phase_bin_edges[-1] NE 1) then message,  'Last element of phase_bin_edges should be zero.'


;; ----------------------------------------------------------------------------
;; Integrate over as many full periods as possible.
; Count the full periods in all the intervals.
; Every phase bin in those full periods produces the same integral, which we can efficiently tally here. 
num_full_periods_in_interval = floor(tstop - tstart)

phase_density_in_bin += area_per_phasebin * total(/INTEGER, num_full_periods_in_interval)


;; ----------------------------------------------------------------------------
; Reduce tstop to account for the full periods we have just integrated.
; Also, shift the time intervals (now spanning <1 period) so that tstart is in [0,1).
; Note that tstop will be in [0,2), not [0,1)!
period_boundary_before_tstart = floor(tstart)

tstart -= period_boundary_before_tstart
tstop  -= period_boundary_before_tstart + num_full_periods_in_interval

; The relationship tstart <= tstop should still hold, so let's verify that.
if debug && ~array_equal( (tstart LE tstop), 1) then message, 'BUG!'



;; ----------------------------------------------------------------------------
;; Integrate over the remaining interval, which is less than a period.

; First, we indentify the phase bin that contains tstart and the bin in [0,2) that contains tstop.
phase_bin_edges_2period = [phase_bin_edges[0:-2], 1 + phase_bin_edges]

tstart_ind = value_locate( phase_bin_edges_2period, tstart )
tstop_ind  = value_locate( phase_bin_edges_2period, tstop  )

; Let's verify that these bin indexes are consistent with 0 <= tstart and tstop < 2.
if debug && ~array_equal( (tstart_ind GE 0)                                    , 1) then message, 'BUG!'
if debug && ~array_equal( (tstop_ind  LE n_elements(phase_bin_edges_2period)-1), 1) then message, 'BUG!'

 
; Because there are three possible relationships between the interval (tstart,tstop) and the phase bin boundaries, we must process the intervals passed by the caller one at a time.
two_period_phase_density = dblarr(2*phase_bins_per_period)
for ii=0L,num_intervals-1 do begin

  ; There are two top-level cases to handle.
  if tstop_ind[ii] EQ tstart_ind[ii] then begin
    ; The interval lies entirely inside one phase bin.
    ; Add the integral of the full interval to that bin.
    two_period_phase_density[ tstart_ind[ii] ] += area_in_one_period * (tstop[ii] - tstart[ii])

  endif else begin
    ; The interval spans at lease two phase bins (i.e. contains at least one bin boundary).
    ; We divide the interval into three parts.
  
    ; First, handle the integral over any phase bins fully covered by the interval.
    num_full_bins = tstop_ind[ii] - tstart_ind[ii] - 1
    
    if (num_full_bins GT 0) then $
      two_period_phase_density[tstart_ind[ii] + 1 : tstop_ind[ii] - 1] += area_per_phasebin
  
    
    ; Second, handle the integral between tstart and the first bin boundary.
    boundary_phase = phase_bin_edges_2period[ tstart_ind[ii] + 1 ]
    
    two_period_phase_density[ tstart_ind[ii] ] += area_in_one_period * (boundary_phase - tstart[ii])
    
    
    ; Third, handle the integral between the last bin boundary and tstop.
    boundary_phase = phase_bin_edges_2period[ tstop_ind[ii] ]
    
    two_period_phase_density[ tstop_ind [ii] ] += area_in_one_period * (tstop[ii] - boundary_phase)
    
  endelse
endfor ; ii

;forprint, phase_bin_edges_2period, two_period_phase_density
;stop

; Fold this two-period phase density and add to the one-period phase density.
phase_density_in_bin += two_period_phase_density[                    0:phase_bins_per_period-1] + $
                        two_period_phase_density[phase_bins_per_period:*                      ]

return
end



PRO test,  tstart, tstop
phase_bins_per_period  = 10
area_in_one_period     = 1D

phase_bin_edges  = [dindgen(phase_bins_per_period) / phase_bins_per_period, 1D]  ; N+1 bin edges
phase_bin_center = (phase_bin_edges[0:-2] + phase_bin_edges[1:-1]) / 2D          ; center of each phase bin

phase_density_in_bin = dblarr(phase_bins_per_period)
            
interval_to_phase_density,  double(tstart), double(tstop), area_in_one_period, phase_bin_edges, phase_density_in_bin
plot, phase_bin_center, phase_density_in_bin, PSYM=4, ystyle=2, yrange=[0,0.1]

truth =  area_in_one_period * total(/DOUBLE, tstop-tstart) 
calc  = total(phase_density_in_bin)
print, 100*abs(truth-calc)/truth, truth, calc, F='(%"Error in total area calculation is %f%%: %f (truth), %f (tool)")'

return
end


;;;  speedtest,1000L, [1,2,3,4,5,6,7,8,9,10,50,100,500,1000,5000,10000L], id
PRO speedtest, num_runs, num_intervals, id
area_in_one_period    = 1D
phase_bins_per_period = 100 
phase_bin_edges       = [dindgen(phase_bins_per_period) / phase_bins_per_period, 1D]  ; N+1 bin edges
phase_density_in_bin  = dblarr(phase_bins_per_period)

num_tests = n_elements(num_intervals)
call_rate     = fltarr(num_tests)
interval_rate = fltarr(num_tests)

for jj=0,num_tests-1 do begin

  tstart = dindgen(num_intervals[jj])/num_intervals[jj]
  tstop  = 2 + reverse(tstart)
  
  clock_total   = tic()
  for ii=0L,num_runs do $
    interval_to_phase_density,  tstart, tstop, area_in_one_period, phase_bin_edges, phase_density_in_bin
  compute_time = toc(clock_total)
  
      call_rate[jj] =                    num_runs /compute_time
  interval_rate[jj] = (num_intervals[jj]*num_runs)/compute_time
  
  print, num_intervals[jj], round(interval_rate[jj]), round(call_rate[jj]), F='(%"%d intervals per call:  %d intervals per second;  %d calls per second")'

endfor
function_1d, id, num_intervals, interval_rate, DATASET=strmid("$Rev:: 4834    $",7)
return
end


