; Interface for WVT_BINNING to work with two-dimensional images. 
; Version 1.2, updated 09/08/05
;
;######################################################################
;+
; NAME:
;     WVT_IMAGE_WITH_EMAP
;
; AUTHORS:
;       Steven Diehl & Thomas S. Statler, Ohio University, US
;       diehl@helios.phy.ohiou.edu, tss@coma.phy.ohiou.edu
;
; PURPOSE:
;       Adaptive spatial binning of any type of 2-dimensional data with
;       the aim of obtaining a uniform signal-to-noise per bin distribution.
;       WVT_IMAGE_WITH_EMAP provides an interface to directly use 2d images as opposed
;       to pixel lists which are compatible with WVT_BINNING.
;
; EXPLANATION:
;       For additional information about WVT binning,
;       please refer to Diehl, S. & Statler, T.S. (2006).
;       Further information on the original VORONOI_2D_BINNING algorithm can 
;       be found in Cappellari M., Copin Y., 2003, MNRAS, 342, 345.
;
; CALLING SEQUENCE:
;     WVT_IMAGE_WITH_EMAP, ctsimage, emap, targetSN, binnedimage, xnode, ynode, weight $
;         [, snbin=snbin, mask=mask, , 
;            binnumber=binnumber, binvalue=binvalue, center=center, 
;            plotit=plotit, resume=resume, save_all=save_all, 
;            max_area=max_area, gersho=gersho, keepfixed=keepfixed, 
;            quiet=quiet ]
;
; EXAMPLE
;       See wvt_image_with_emap.example
;
; INPUTS (REQUIRED):
;       CTSIMAGE: Two-dimensional image (n_x,n_y) containing the counts 
;               per pixel.

;     TARGETSN: The desired signal-to-noise ratio in the final
;               2D-binned data. A TARGETSN between ~4-10 is standard for 
;               X-ray images, a temperature map would require higher 
;               TARGETSN (~30-100), depending on the spectral model used. 
;               For integral field spectroscopy, a TARGETSN of ~50 per pixel 
;               may be a reasonable value to extract stellar kinematics
;               information from galaxy spectra.
;            
; OUTPUTS:
;  BINNEDIMAGE: Two-dimensional image (n_x,n_y) containing the binned signal.
;        XNODE: Vector (size Nbins) of the X coordinates of the bin 
;               generators, i.e the geometric centers of the bins.
;        YNODE: Vector (size Nbins) of Y coordinates of the bin generators.
;        SNBIN: Vector (size Nbins) with the final SN of each bin.
;    BINNUMBER: Two-dimensional image (n_x,n_y) containing the bin number 
;               assigned to each input pixel. The index goes from 1 to 
;               Nbins. Pixels that were excluded during the binning process 
;               are marked "0" (see also the keyword MASK).
;               This vector alone is enough for making *any* subsequent
;               computation on the binned data. Everything else is optional.
;         AREA: Vector (size Nbins) with the number of pixels for each bin.
;     BINVALUE: Vector (size Nbins) with the final signal of each bin
;
; KEYWORDS (OPTIONAL):
;       CENTER: [Input] Vector (size 2) containing x and y values for the 
;               center. If this keyword is NOT supplied, [0,0] will be 
;               assumed and the algorithm will start at the highest S/N 
;               pixel with the bin accretion step. If the center is given, 
;               bin accretion will start there. 
;     MAX_AREA: [Input] Scalar specifying a maximum bin size (in pixel). 
;               Useful in cases where there is essentially no signal in a 
;               certain region. ATTENTION: In area where the bin size hits
;               MAX_AREA, the algorithm does NOT enforce a uniform S/N 
;               anymore. Use with care. 
;         MASK: [Input] Two-dimensional image (n_x,n_y) specifying which 
;               pixels should be included in the WVT binning algorithm. 
;               Valid pixels are designated as "1", excluded pixels as "0" 
;               (integer or byte). 
;    KEEPFIXED: [Input] Vector (size 2 x n_fixed) containing x and y 
;               coordinates of bin generators that you want to keep fixed 
;               in their position. The binning algorithm will move all other 
;               bins around as usual. Example use: Keep one bin fixed on 
;               the center of a galaxy.
;       PLOTIT: Set this keyword to one to produce a plot of the 
;               two-dimensional bin distribution and of the corresponding 
;               S/N at the end of the computation. A PLOTIT value of 2 will 
;               produce a similar plot after the bin accretion step. Setting 
;               PLOTIT to 3 results in graphical output for each iteration.
;               One should keep in mind that plotting slows down the 
;               algorithm significantly.
;        QUIET: by default the program shows the progress while accreting
;               pixels and then while iterating the CVT. Set this keyword
;               to avoid printing progess results. 
;       GERSHO: Use Gersho's conjecture with normal Voronoi tesselations, 
;               instead of the WVT algorithm. Except for some changes in the 
;               bin accretion step, this procedures is identical to the one 
;               implemented in Cappellari & Copin's code VORONOI_2D_BINNING.
;               Be aware, that Gersho's conjecture is only valid for strictly
;               positive data, where the S/N adds in quadrature!
;       RESUME: Set this keyword, if you want to start from an existing WVT. 
;               which is uniquely defined in the structure SAVE_ALL
;     SAVE_ALL: [Output/Input] Structure that saves all of the necessary 
;               information to restart WVT_IMAGE_WITH_EMAP from any given point. If the
;               keyword RESUME is set, the WVT iteration scheme is applied 
;               to the supplied WVT.
;
; EXTERNAL INTERACTIVE CONTROLS
;       PLOTIT: If you decide to plot the iteration steps, create 
;               file in the current directory with the name 'plotit'. A simple
;               way to do this is to issue the command 'touch plotit' in a 
;               shell. Remove the file if you want to stop plotting.
;    STOPMENOW: If you want to terminate the iteration at the next iteration,
;               create a file named 'stopmenow' in the current directory (e.g 
;               'touch stopmenow').
;
; PROCEDURES PROVIDED:
;       The following procedures are contained in the main WVT_BINNING program.
;       Refer to the main code for further explanations.
;            WVT_IMAGE_WITH_EMAP                     -- main interface
;
; FUNCTIONS PROVIDED:
;       The following functions are contained in the main WVT_BINNING program.
;       Refer to the main code for further explanations.
;            ADD_SIGNAL                    -- adds the signal of a bin
;            ADD_NOISE                     -- adds the noise of a bin
;      
; NOTE:
;       WVT_BINNING uses the function MATCH from the IDL astro library. We
;       included this function in the main program release for completeness.
;       Feel free to remove the file match.pro if you already have the IDL 
;       astro library (see idlastro.gsfc.nasa.gov/homepage.html) installed.
;
; MODIFICATION HISTORY (ORIGINAL VORONOI_2D_BINNING):
;       V0.1: First interface (10/2004)
;       V0.1: Exported the functions ADDSIGNAL and ADD_NOISE to make 
;           WVT_BINNING more flexible (11/2004)
;       V1.0: Major changes to work with the new WVT_BINNING calling sequence 
;           (07/12/2005)
;       V1.1: Fixed bug for keepfixed: the variable is no longer 
;           overwritten and also saved in SAVE_ALL now. (09/06/2005) 
;       V1.2: Fixed bug when incompatible ADD_SIGNAL or ADD_NOISE functions
;           are precompiled. (09/08/2005)
;
;
;----------------------------------------------------------------------------
FUNCTION ADD_SIGNAL, pixel_list
  ; Adds the signal inside the bin, specified by the indeces of the bin members
  COMMON DATAVALUES, P
;; The flux in the bin the sum of the counts divided by 
;; the sum of the exposure.
counts   = total(P.ctsimage[pixel_list])  
exposure = total(P.emap[pixel_list])
flux_estimate = counts / exposure
return, flux_estimate                       
END

;----------------------------------------------------------------------------
FUNCTION ADD_NOISE, pixel_list 
 ; Adds the noise inside the bin, specified by the indeces of the bin members
  COMMON DATAVALUES, P
;; The flux in the bin the sum of the counts divided by 
;; the sum of the exposure.
counts   = total(P.ctsimage[pixel_list])  
exposure = total(P.emap[pixel_list])
flux_estimate = counts / exposure

poisson_means = flux_estimate * P.emap[pixel_list]                        
flux_error = sqrt(total(poisson_means)) / exposure           

;print, flux_estimate/flux_error
return, flux_error                                                
END

;----------------------------------------------------------------------------
PRO WVT_IMAGE_WITH_EMAP, ctsimage, emap, targetSN, binnedimage, xnode, ynode, weight, $
    snbin=snbin, mask=mask, binnumber=binnumber, $
    binvalue=binvalue, center=center, plotit=plotit, resume=resume, $
    save_all=save_all, max_area=max_area, gersho=gersho, $
    keepfixed=keepfixed, quiet=quiet
  ; Interface for using WVT_BINNING with images instead of pixel lists
  ; Takes the images, converts them into pixel lists, stores all of their
  ; properties (ctsimage, emap, ...) in the global variable P (structure) 
  ; and passes everything on to WVT_BINNING. This was designed particularly 
  ; with X-ray images in mind. The structure P will hold all information 
  ; about the individual pixels (ctsimage, noise, ...)
  COMMON DATAVALUES, P

  ; Check if the ADD_SIGNAL and ADD_NOISE functions originate from this file
  ; If not, stop the procedure before you do something stupid...
  faddsignal = routine_info('add_signal',/source,/functions)
  faddnoise  = routine_info('add_noise' ,/source,/functions)
  fthisfile = routine_info('wvt_image_with_emap',/source)
  IF strcmp(faddsignal.path, fthisfile.path) EQ 0 $
    OR strcmp(faddnoise.path, fthisfile.path) EQ 0 THEN $
    message, 'PRECOMPILED ADD_SIGNAL OR ADD_NOISE FUNCTION'+ $
             ' NOT COMPATIBLE WITH WVT_IMAGE_WITH_EMAP!' + $
             ' To avoid this, type ".compile wvt_image_with_emap" before starting.'

  ; Turn images into 1D pixel lists if you don't resume an old session
  IF NOT(keyword_set(resume)) THEN BEGIN
    si=size(ctsimage,/dimension)
    m=long(si(0))
    n=long(si(1))
    binnedimage=fltarr(m,n)
    IF NOT(keyword_set(mask)) THEN mask=fix(ctsimage-ctsimage+1)
    IF NOT(keyword_set(center)) THEN center=0

    temp=where(mask NE 0, ngood)
    x=fltarr(ngood)
    y=fltarr(ngood)
    
    p={ ctsimage:dblarr(ngood), $
        emap:dblarr(ngood) $
       } 

    ii=0L
    FOR j=0L,n-1L DO BEGIN
      FOR i=0L,m-1L DO BEGIN
        ; Check if the pixel should be included or not
        IF mask(i,j) NE 0 THEN BEGIN
          x[ii]=float(i)
          y[ii]=float(j)
          p.ctsimage[ii]=ctsimage[i,j]
          p.emap[ii]=emap[i,j]
          ii=ii+1L
        ENDIF
      ENDFOR
    ENDFOR
    pixelsize=1.0
  ENDIF ELSE BEGIN
    ; In case you are resuming an old session, read all the information out of 
    ; the save_all structure
    x=save_all.x
    y=save_all.y
    p={ ctsimage:save_all.ctsimage, $
        emap:save_all.emap $
       }
    class=save_all.class
    neighborlist=save_all.neighborlist
    pixelsize=save_all.pixelsize
    targetSN=save_all.targetSN 
    xnode=save_all.xnode
    ynode=save_all.ynode
    snbin=save_all.snbin
    center=save_all.center
    keepfixed=save_all.keepfixed
  ENDELSE

  nfixed=fix(n_elements(keepfixed)/2.)
  IF nfixed GT 0 THEN keepfixed_in=keepfixed ELSE keepfixed=0
  FOR i=0,nfixed-1 DO keepfixed_in[*,i]=round(keepfixed[*,i])  

  ; Start the main binning algorithm
  WVT_BINNING, x, y, pixelsize, targetSN, class, $
      xnode, ynode, weight, binvalue=binvalue, $
      plotit=plotit, quiet=quiet, snbin=snbin, resume=resume, $
      neighborlist=neighborlist, max_area=max_area, $
      gersho=gersho, keepfixed=keepfixed_in, center=center

  ; Reconvert the output of WVT_BINNING back to images
  wvt_list_to_image, binnedimage, binvalue[class], x, y, dim=si
  wvt_list_to_image, binnumber, class+1, x, y, dim=si

  ; Create the structure save_all, which can be used to resume the procedure 
  ; at the point it was stopped. The structure can also be used to start with 
  ; a given tesselation
  save_all={ x:x, $
             y:y, $
             class:class, $
             neighborlist:neighborlist, $
             ctsimage:p.ctsimage, $
             emap:p.emap, $
             xnode:xnode, $
             ynode:ynode, $
             weight:weight, $
             binvalue:binvalue, $
             pixelsize:pixelsize, $
             targetSN:targetSN, $
             snbin:snbin, $
             center:center, $
             max_area:max_area, $
             keepfixed:keepfixed }

END




