############################################################################
# Copyright (c) 2001, Smithsonian Astrophysical Observatory
# You may do anything you like with this file except remove this
# copyright.
#
# FILE NAME: ciao.csh
#
# DEVELOPMENT: CXCDS User Infrastructure Software
#
# DESCRIPTION:
#
#       This script configures the environment for Chandra Science Center
#       end-user (ie, external to the CXCDS) software.
#
#       When the script is sourced, it checks for the existence of an
#       ASCDS_INSTALL environment variable. If found, the script assumes
#       that the environment is already set up and exits with a warning
#       message. Using the '-o' option will cause the script to reconfigure
#       to run using its values. The previous environment variables will
#       generally be replaced and the old install path will be removed 
#       from the user's PATH and LD_LIBRARY_PATH. 
#
#       User customizations should be added to the 'USER CUSTOMIZATIONS'
#       section which appears slightly beyond the middle of the script. 
#       If the user desires to modify the value of the installation root
#       (ASCDS_INSTALL) they should do so by setting the _nextASCDS_INSTALL
#       value at the beginning of the script.
#
#       This version of the script is written in the csh language, and
#       is compatible tcsh, too.
#
############################################################################

################################################################
#               Setup for ASCDS Installation root              #
################################################################

set _silent = "No"
set _overrideNeeded = "No" 
set _partialConfig = "No"
if ( $?ASCDS_INSTALL ) then 
   set _prevASCDS_INSTALL = $ASCDS_INSTALL 
   if (`echo $PATH | grep $ASCDS_INSTALL` != "" ) then
      set _overrideNeeded = "Yes"
   else
      set _partialConfig = "Yes"
   endif
else
   set _prevASCDS_INSTALL =  ""
endif 
set _nextASCDS_INSTALL = /iraf/ciao_3.1 

################################################################
#                  Handle command line options                 #
################################################################
set arg_cnt=1
set nargs=$#argv
foreach value ($argv[*])
   switch ( $argv[$arg_cnt] )
      case -h
         echo "-- CIAO setup script (csh version)"
         echo  "Usage:  source ciao.csh [-hoOqv]"
         echo "   -h       =  This help message"
         echo "   -o       =  Overide a previous configuration"
         echo "   -q       =  quiet mode- do not echo output to screen"
         echo "   -O <DIR> =  Override default ASCDS_INSTALL location with DIR"
	 echo "               (for internal CXC use, in csh/tcsh only)"
         echo "   -v       =  Print CIAO version information"
         exit
      breaksw
      case -v
         if ( $_silent == "No" ) then 
            if (( $_overrideNeeded == "No") || \
                 ( $_nextASCDS_INSTALL != $_prevASCDS_INSTALL )) then
               echo "This script sets the user's CIAO environment to utilize:"
               if ( -e $_nextASCDS_INSTALL/VERSION ) then
                  echo " CIAO version     : " `cat $_nextASCDS_INSTALL/VERSION`
               endif
               if ( -e $_nextASCDS_INSTALL/VERSION.prop_tk ) then
                  echo " Proposal Toolkit version : " `cat $_nextASCDS_INSTALL/VERSION.prop_tk`
               endif
               echo    " bin dir          : " $_nextASCDS_INSTALL/bin
            endif
            if ( $_overrideNeeded == "Yes" ) then
               echo "The current environment is configured for:"
               if ( -e $_prevASCDS_INSTALL/VERSION ) then
                  echo " CIAO version     : " `cat $_prevASCDS_INSTALL/VERSION`
               endif
               if ( -e $_prevASCDS_INSTALL/VERSION.prop_tk ) then
                  echo " Proposal Toolkit version : " `cat $_prevASCDS_INSTALL/VERSION.prop_tk`
               endif
               echo    " bin dir          : " $_prevASCDS_INSTALL/bin
            endif
         endif
         exit
      breaksw
      case -o
         set _overrideNeeded = "No"
      breaksw
      case -q
         set _silent = "-q"
      breaksw
      case -O
	 @ arg_cnt++
	 set _nextASCDS_INSTALL = $argv[$arg_cnt]
      breaksw
   endsw
   if ($arg_cnt == $nargs) then
      break
   endif
   @ arg_cnt++ 
end

if ( $_overrideNeeded == "No" ) then 
   if ( $_prevASCDS_INSTALL != "" ) then
      unsetenv PDIRS
      # Remove previous system param file from $PFILES if 'override'
      if ( `echo $PFILES | grep $_prevASCDS_INSTALL/param` != "" ) then
         setenv PFILES `echo "{$PFILES}" | sed "s%$_prevASCDS_INSTALL/param%%g"`
         setenv PFILES `echo "{$PFILES}" | sed "s%::%:%g"`
      endif
   endif
else
   if ( $_silent == "No" ) then 
      echo "ERROR: The current environment is configured for:"
      if ( -e $_prevASCDS_INSTALL/VERSION ) then
         echo " CIAO version     : " `cat $_prevASCDS_INSTALL/VERSION`
      endif
      if ( -e $_prevASCDS_INSTALL/VERSION.prop_tk ) then
         echo " Proposal Toolkit version : " `cat $_prevASCDS_INSTALL/VERSION.prop_tk`
      endif
      echo    " bin dir          : " $_prevASCDS_INSTALL/bin
      echo "Please use the '-o' option to override the previous setup."
   endif
   exit
endif

################################################################
#       Remove previous configuration values from setup        #
################################################################

#  remove ASCDS_BIN from path if it exists 
if ( $?ASCDS_BIN ) then 
   if ( `echo $path | grep $ASCDS_BIN` != "" ) then
      set path = (`echo " $path " | sed "s% $ASCDS_BIN % %g"`)
   endif
endif
#  Remove ASCDS_CONTRIB from path if it exists
if ( $?ASCDS_CONTRIB ) then
   if ( `echo $path | grep $ASCDS_CONTRIB` != "" ) then
	set path = (`echo " $path " | sed "s% $ASCDS_CONTRIB % %g"`)
   endif
endif
#  Remove ASCDS_CONTRIB/bin from path if it exists
if ( $?ASCDS_CONTRIB ) then
   if ( `echo $path | grep $ASCDS_CONTRIB/bin` != "" ) then
	set path = (`echo " $path " | sed "s% $ASCDS_CONTRIB/bin % %g"`)
   endif
endif
#  Remove ASCDS_CONTRIB/lib from LD_LIBRARY_PATH if it exists
if ( $?ASCDS_CONTRIB ) then
   if ( `echo $LD_LIBRARY_PATH | grep $ASCDS_CONTRIB/lib` != "" ) then
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%$ASCDS_CONTRIB/lib%%g"`
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%::%:%g"`
   endif
endif
#  Remove ASCDS_INSTALL/bin/interpreted from SLANG_SCRIPT_PATH if it exists
if ( $?ASCDS_INSTALL ) then
   if ( `echo $SLANG_SCRIPT_PATH | grep $ASCDS_INSTALL/bin/interpreted` != "" ) then
      setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" | sed "s%$ASCDS_INSTALL/bin/interpreted%%g"`
      setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" | sed "s%::%:%g"`
   endif
endif
#  Remove ASCDS_CONTRIB/bin/interpreted from SLANG_SCRIPT_PATH if it exists
if ( $?ASCDS_CONTRIB ) then
   if ( `echo $SLANG_SCRIPT_PATH | grep $ASCDS_CONTRIB/bin/interpreted` != "" ) then
      setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" | sed "s%$ASCDS_CONTRIB/bin/interpreted%%g"`
      setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" | sed "s%::%:%g"`
   endif
endif
#  Remove ASCDS_CONTRIB/param from PFILES if it exists
if ( $?ASCDS_CONTRIB ) then
   if ( `echo $PFILES | grep $ASCDS_CONTRIB/param` != "" ) then
	setenv PFILES `echo $PFILES | sed "s%$ASCDS_CONTRIB/param%%g"`
	setenv PFILES `echo $PFILES | sed "s%::%:%g"`
   endif
endif
#  Remove isis from path
if ( $?ASCDS_INSTALL ) then
   if ( `echo $path | grep $ASCDS_INSTALL/ots/isis/bin` != "" ) then
	set path = (`echo " $path " | sed "s%$ASCDS_INSTALL/ots/isis/bin% %g"`)
   endif
endif
#  Remove isis from SLANG_MODULE_PATH
if ( $?SLANG_MODULE_PATH ) then
  if ( $?ASCDS_INSTALL ) then
    if ( `echo $SLANG_MODULE_PATH | grep $ASCDS_INSTALL/ots/isis/lib/modules` != "" ) then
	setenv SLANG_MODULE_PATH `echo "$SLANG_MODULE_PATH" | sed "s%$ASCDS_INSTALL/ots/isis/lib/modules%%g"`
	setenv SLANG_MODULE_PATH `echo "$SLANG_MODULE_PATH" | sed "s%::%:%g"`
    endif
  endif
endif

#  remove ASCDS_LIB from LD_LIBRARY_PATH if appropriate
if ( $?ASCDS_LIB && $?LD_LIBRARY_PATH ) then 
   if ( `echo $LD_LIBRARY_PATH | grep $ASCDS_LIB` != "" ) then
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%$ASCDS_LIB%%g"`
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%::%:%g"`
   endif
endif

#  remove $ASCDS_INSTALL/ots/saord from LD_LIBRARY_PATH if appropriate
#if ( $?ASCDS_INSTALL && $?LD_LIBRARY_PATH ) then 
#   if ( `echo $LD_LIBRARY_PATH | grep $ASCDS_INSTALL/ots/saord` != "" ) then
#      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%$ASCDS_INSTALL/ots/saord%%g"`
#      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%::%:%g"`
#   endif
#endif

#  remove $ASCDS_LIB from BWIDGET_LIBRARY if appropriate
#if ( $?ASCDS_LIB && $?BWIDGET_LIBRARY ) then 
#   if ( `echo $BWIDGET_LIBRARY | grep $ASCDS_LIB` != "" ) then
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%$ASCDS_LIB%%g"`
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%::%:%g"`
#   endif
#endif

#  remove $ASCDS_INSTALL/ots/saord from BWIDGET_LIBRARY if appropriate
#if ( $?ASCDS_INSTALL && $?BWIDGET_LIBRARY ) then 
#   if ( `echo $BWIDGET_LIBRARY | grep $ASCDS_INSTALL/ots/saord` != "" ) then
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%$ASCDS_INSTALL/ots/saord%%g"`
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%::%:%g"`
#   endif
#endif

#  if ASCDS_IMAGER_PATH is already in path, remove it
if ( $?ASCDS_IMAGER_PATH ) then 
   if ( `echo $path | grep $ASCDS_IMAGER_PATH` != "" ) then
      set path = (`echo " $path " | sed "s% $ASCDS_IMAGER_PATH % %g"`)
   endif
endif

################################################################
#                set CIAO environment variables                #
################################################################

setenv ASCDS_INSTALL $_nextASCDS_INSTALL
unset _nextASCDS_INSTALL
#  The path to the root of the ASCDS executables.
setenv ASCDS_BIN $ASCDS_INSTALL/bin
setenv ASCDS_CONTRIB $ASCDS_INSTALL/contrib
setenv ASCDS_LIB  $ASCDS_INSTALL/lib
#  The path to the system parameter files.
setenv ASCDS_SYS_PARAM "$ASCDS_INSTALL/param"
#  The path of the working directory of the user
setenv ASCDS_WORK_PATH /tmp
setenv ASCDS_TMP /tmp
#  The calibration directory path
setenv ASCDS_CALIB $ASCDS_INSTALL/data
#  The path of the documentation
setenv ASCDS_DOC $ASCDS_INSTALL/doc/txt
#  For XPA, use local sockets.  Avoids problems for users
#  running ssh, and those without network cards.
setenv XPA_METHOD local
#  Sherpa needs these environment variables
setenv XSPEC_HELP_FILE $ASCDS_INSTALL/doc/xspec.hlp
#  setup CALDB related variables
setenv CALDB $ASCDS_INSTALL/CALDB
setenv CALDBCONFIG $CALDB/software/tools/caldb.config
setenv CALDBALIAS $CALDB/software/tools/alias_config.fits
#  setup ATOMDB variables
setenv ATOMDB $ASCDS_INSTALL/ATOMDB
#  OBSVIS2 specific environment variables
setenv OBSVIS_WEB_SERVER_NAME obsvis.harvard.edu
setenv OBSVIS_WEB_SERVER_PORT 80
setenv OBSVIS_CATALOG_DIR ${ASCDS_INSTALL}/data
setenv OBSVIS_IMAGER_PLUGIN_DIR ${ASCDS_INSTALL}/data
#if (! $?DYLD_BIND_AT_LAUNCH) then
#  setenv DYLD_BIND_AT_LAUNCH 1
#endif
#  ahelp specific environment setup
alias about "${ASCDS_INSTALL}/bin/ahelp -k \!*"

if (-d /iraf/ciao_3.1/ots/saord ) then
   set SAORDDIR  = /iraf/ciao_3.1/ots/saord
   set SAORDBIN=$SAORDDIR
else
   #  inhouse - external configurations should be set above
   set SAORDDIR = /proj/ciaox/ots/saord
   set SAORDBIN=$SAORDDIR/bin
endif
setenv SAORD_ROOT $SAORDDIR
setenv ASCDS_IMAGER_PATH $SAORDBIN

################################################################
#                set NRA environment variables                 #
################################################################

if ( -e /iraf/ciao_3.1/VERSION.prop_tk ) then
  set DATA_ROOT="/iraf/ciao_3.1/config"
  setenv ASCDS_PROP_PMS_DATA    $DATA_ROOT/pimms/data
  setenv ASCDS_PROP_PMS_MODEL   $DATA_ROOT/pimms/models
  setenv JCMLIBDATA             $DATA_ROOT/jcm_data

# The auxiliary ASCDS_PROP_NHBASE is set to allow use of the prop_colden 
#  wrapper script 

  setenv ASCDS_PROP_NHBASE      $JCMLIBDATA
  setenv JCMPATH                $DATA_ROOT/jcm_data

# The auxiliary ASCDS_PROP_DATE_DATA is set to allow use of the prop_dates 
#  wrapper script 

  setenv ASCDS_PROP_DATE_DATA   $JCMPATH

# The auxiliary ASCDS_PROP_PREC_DATA is set to allow use of the prop_precess 
#  wrapper script 

  setenv ASCDS_PROP_PREC_DATA   $JCMPATH

# Finally ASCDS_PROP_AO needs to be set to 06 for cycle 6

  setenv ASCDS_PROP_AO		06
endif

################################################################
#                   USER CUSTOMIZATIONS                        #
################################################################
# By default CIAO will write local parameter files to $HOME/cxcds_param,
# but you may customize that by redefining LOCPFILES directly below
set LOCPFILES="$HOME/cxcds_param"

if ($?PFILES) then
   #  use the PFILES local dir value for the PDIRS local dir value
   set SYSPF = (`echo $PFILES | sed "s%.*;%%"`)
   set LOCPF = (`echo $PFILES | sed "s%;.*%%"`)
else
   set LOCPF = ""
   set SYSPF = ""
endif

if ( `echo $LOCPF | grep $LOCPFILES` != "" ) then
   set LOCPF = (`echo " $LOCPF " | sed "s%$LOCPFILES%%g"`)
endif
if ( $LOCPF != "" ) then
   set LOCPF = ${LOCPFILES}:${LOCPF}
else
   set LOCPF = ${LOCPFILES}
endif

if ( `echo $SYSPF | grep "$ASCDS_INSTALL/param"` != "" ) then
   set SYSPF = (`echo " $SYSPF " | sed "s%$ASCDS_INSTALL/param%%g"`)
endif
if ( $SYSPF != "" ) then
   set SYSPF = $ASCDS_INSTALL/param:${SYSPF}
else
   set SYSPF = $ASCDS_INSTALL/param
endif

if ( -e $ASCDS_CONTRIB/param ) then
   set SYSPF = $ASCDS_CONTRIB/param:${SYSPF}
endif

setenv PFILES "$LOCPF;$SYSPF"
setenv PFILES `echo "{$PFILES}" | sed "s%::%:%g"`

set _tmp=`echo "$LOCPFILES" | sed "s%:% %g"`
foreach _dir (`echo "$_tmp"`) 
   if ( ! -d "$_dir" ) then
      mkdir $_dir     # if local param dir does not exist, create it
   endif 
end
unset _tmp 

################################################################
#                         Adjust paths                         # 
################################################################

#  Add ASCDS_BIN and ASCDS_CONTRIB to path- if it exists move to it the front
if ( `echo $path | grep $ASCDS_BIN` != "" ) then
   set path = (`echo " $path " | sed "s% $ASCDS_BIN % %g"`)
endif
if ( `echo $path | grep $ASCDS_CONTRIB` != "" ) then
   set path = (`echo " $path " | sed "s% $ASCDS_CONTRIB % %g"`)
endif
if ( `echo $path | grep $ASCDS_CONTRIB/bin` != "" ) then
   set path = (`echo " $path " | sed "s% $ASCDS_CONTRIB/bin % %g"`)
endif
if ( ${?path} ) then
   set path = ( $ASCDS_CONTRIB $ASCDS_BIN $path )
else
   set path = ( $ASCDS_CONTRIB $ASCDS_BIN )
endif
if ( -e $ASCDS_CONTRIB/bin ) then
   set path = ( $ASCDS_CONTRIB/bin $path )
endif

#  Add ASCDS_LIB to LD_LIBRARY_PATH- if it exists move it to the front
if ($?LD_LIBRARY_PATH ) then 
   if ( `echo $LD_LIBRARY_PATH | grep $ASCDS_LIB` != "" ) then
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%$ASCDS_LIB%%g"`
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%::%:%g"`
   endif
   if ( $LD_LIBRARY_PATH != "{}" ) then
      setenv LD_LIBRARY_PATH ${ASCDS_LIB}:${LD_LIBRARY_PATH}
   else
      setenv LD_LIBRARY_PATH ${ASCDS_LIB}
   endif
else
   setenv LD_LIBRARY_PATH ${ASCDS_LIB}
endif

#  Add ASCDS_CONTRIB/lib to LD_LIBRARY_PATH
#  if it exists move it to the front
if ($?ASCDS_CONTRIB ) then
  if ( -e $ASCDS_CONTRIB/lib ) then 
    if ( `echo $LD_LIBRARY_PATH | grep $ASCDS_CONTRIB/lib` != "" ) then
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%$ASCDS_CONTRIB/lib%%g"`
      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%::%:%g"`
    endif
    if ( $LD_LIBRARY_PATH != "{}" ) then
      setenv LD_LIBRARY_PATH ${ASCDS_CONTRIB}/lib:${LD_LIBRARY_PATH}
    else
      setenv LD_LIBRARY_PATH ${ASCDS_CONTRIB}/lib
    endif
  endif
endif

#  Add ASCDS_INSTALL/ots/saord to LD_LIBRARY_PATH
#  if it exists move it to the front
#if ($?LD_LIBRARY_PATH ) then 
#   if ( `echo $LD_LIBRARY_PATH | grep $ASCDS_INSTALL/ots/saord` != "" ) then
#      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%$ASCDS_INSTALL/ots/saord%%g"`
#      setenv LD_LIBRARY_PATH `echo "{$LD_LIBRARY_PATH}" | sed "s%::%:%g"`
#   endif
#   if ( $LD_LIBRARY_PATH != "{}" ) then
#      setenv LD_LIBRARY_PATH ${ASCDS_INSTALL}/ots/saord:${LD_LIBRARY_PATH}
#   else
#      setenv LD_LIBRARY_PATH ${ASCDS_INSTALL}/ots/saord
#   endif
#else
#   setenv LD_LIBRARY_PATH ${ASCDS_INSTALL}/ots/saord
#endif

#  Add ASCDS_LIB to BWIDGET_LIBRARY
#  if it exists move it to the front
#if ($?BWIDGET_LIBRARY ) then 
#   if ( `echo $BWIDGET_LIBRARY | grep $ASCDS_LIB` != "" ) then
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%$ASCDS_LIB%%g"`
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%::%:%g"`
#   endif
#   if ( $BWIDGET_LIBRARY != "{}" ) then
#      setenv BWIDGET_LIBRARY ${ASCDS_LIB}:${BWIDGET_LIBRARY}
#   else
#      setenv BWIDGET_LIBRARY ${ASCDS_LIB}
#   endif
#else
#   setenv BWIDGET_LIBRARY ${ASCDS_LIB}
#endif

#  Add ASCDS_INSTALL/ots/saord to BWIDGET_LIBRARY
#  if it exists move it to the front
#if ($?BWIDGET_LIBRARY ) then 
#   if ( `echo $BWIDGET_LIBRARY | grep $ASCDS_INSTALL/ots/saord` != "" ) then
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%$ASCDS_INSTALL/ots/saord%%g"`
#      setenv BWIDGET_LIBRARY `echo "{$BWIDGET_LIBRARY}" | sed "s%::%:%g"`
#   endif
#   if ( $BWIDGET_LIBRARY != "{}" ) then
#      setenv BWIDGET_LIBRARY ${ASCDS_INSTALL}/ots/saord:${BWIDGET_LIBRARY}
#   else
#      setenv BWIDGET_LIBRARY ${ASCDS_INSTALL}/ots/saord
#   endif
#else
#   setenv BWIDGET_LIBRARY ${ASCDS_INSTALL}/ots/saord
#endif

#  Add ASCDS_BIN/interpreted to SLANG_SCRIPT_PATH, or move it to front
set scripts=$ASCDS_BIN/interpreted
if ($?SLANG_SCRIPT_PATH ) then 
   if ( `echo $SLANG_SCRIPT_PATH | grep $scripts` != "" ) then
     setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" |sed "s%$scripts%%g"`
     setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" |sed "s%::%:%g"`
   endif
   if ( $SLANG_SCRIPT_PATH != "{}" ) then
      setenv SLANG_SCRIPT_PATH ${scripts}:${SLANG_SCRIPT_PATH}
   else
      setenv SLANG_SCRIPT_PATH ${scripts}
   endif
else
   setenv SLANG_SCRIPT_PATH ${scripts}
endif

#  Add ASCDS_CONTRIB/bin/interpreted to SLANG_SCRIPT_PATH, or move it to front
#  if it exists 
if ($?ASCDS_CONTRIB ) then
  if  ( -e $ASCDS_CONTRIB/bin/interpreted ) then
    set scripts=$ASCDS_CONTRIB/bin/interpreted
    if ( `echo $SLANG_SCRIPT_PATH | grep $scripts` != "" ) then
      setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" |sed "s%$scripts%%g"`
      setenv SLANG_SCRIPT_PATH `echo "{$SLANG_SCRIPT_PATH}" |sed "s%::%:%g"`
    endif
    if ( $SLANG_SCRIPT_PATH != "{}" ) then
      setenv SLANG_SCRIPT_PATH ${scripts}:${SLANG_SCRIPT_PATH}
    else
      setenv SLANG_SCRIPT_PATH ${scripts}
    endif
  endif
endif

#  Do we need iraf setup?
if ( ! $?iraf ) then
   setenv iraf /iraf/iraf
   setenv host $iraf/unix
endif    

#  Add SAORDBIN to path.
if ( `echo $path | grep $SAORDBIN` == "" ) then
  set path = ( $SAORDBIN $path )
endif
unset SAORDDIR SAORDBIN OS_REV scripts

# Load CXCdefaults X resources file
$ASCDS_INSTALL/bin/cxcldres $_silent

# Do we need isis variables setup?
if ( -d $ASCDS_INSTALL/ots/isis ) then
    setenv ISIS_SRCDIR $ASCDS_INSTALL/ots/isis
    setenv PGPLOT_DIR $ASCDS_INSTALL/ots/pgplot
    set isispath=${ASCDS_INSTALL}/ots/isis/bin
    if ( `echo $path | grep $isispath` != "" ) then
       set path = (`echo " $path " | sed "s% $isispath % %g"`)
    endif
    if ( ${?path} ) then
       set path = ( $isispath $path )
    else
       set path = ( $isispath )
    endif

    set imodules=${ASCDS_INSTALL}/ots/isis/lib/modules
    if ($?SLANG_MODULE_PATH ) then 
       if ( `echo $SLANG_MODULE_PATH | grep $imodules` != "" ) then
	  setenv SLANG_MODULE_PATH `echo "${SLANG_MODULE_PATH}" | sed "s%$imodules%%g"`
	  setenv SLANG_MODULE_PATH `echo "${SLANG_MODULE_PATH}" | sed "s%::%:%g"`
       endif
       if ( $SLANG_MODULE_PATH == ":" ) then
	  setenv SLANG_MODULE_PATH ""
       endif
       if ( $SLANG_MODULE_PATH != "{}" ) then
	  setenv SLANG_MODULE_PATH ${imodules}:${SLANG_MODULE_PATH}
       else
	  setenv SLANG_MODULE_PATH ${imodules}
       endif
    else
       setenv SLANG_MODULE_PATH ${imodules}
    endif
endif

# Setup XANADU variable
if ( -d $ASCDS_INSTALL/ots/lheasoft ) then
  setenv XANADU $ASCDS_INSTALL/ots/lheasoft
else
  if ( -d /proj/cm/DS.ots/xanadu.v11.2.SunOS5.8 ) then
    setenv XANADU /proj/cm/DS.ots/xanadu.v11.2.SunOS5.8
  else
    if ( $_silent == "No" ) then
      echo "Warning could not setup XANADU. Xspec models unavailable"
    endif
  endif
endif

################################################################
#                    Finished configuration                    #  
################################################################

if ( $_silent == "No" ) then
   echo "CIAO configuration is complete... "
   if ( $_prevASCDS_INSTALL != ""  && $_partialConfig == "No" ) then
      echo "The CIAO setup for this window has changed from"
      if ( -e $_prevASCDS_INSTALL/VERSION ) then
         echo " CIAO version     : " `cat $_prevASCDS_INSTALL/VERSION`
      endif 
      if ( -e $_prevASCDS_INSTALL/VERSION.prop_tk ) then
         echo " Proposal Toolkit version : " `cat $_prevASCDS_INSTALL/VERSION.prop_tk`
      endif 
      echo    " bin dir          : " $_prevASCDS_INSTALL/bin
      echo "To:"
   endif

   if (-e $ASCDS_INSTALL/VERSION) then
      echo " CIAO version     : " `cat $ASCDS_INSTALL/VERSION`
   endif
   if (-e $ASCDS_INSTALL/VERSION.prop_tk) then
      echo " Proposal Toolkit version : " `cat $ASCDS_INSTALL/VERSION.prop_tk`
   endif
   echo    " bin dir          : " $ASCDS_BIN
endif

