;;; $Id: ae_recipe.pro 5659 2022-01-29 18:05:58Z psb6 $
;;; Patrick Broos, 2004

;;; Programs to automate the AE recipe found in the Getting Started section of 
;;; the AE manual and in recipe.txt.

;!! These scripts assume a very particular set of conventions for paths to 
;!! the various files that AE uses, as shown in the Getting Started section of 
;!! the AE manual and in recipe.txt.


;#############################################################################
PRO wait_while_files_exist, delay, pattern
  get_lun, unit
  last_reported_files_found = ''
  repeat begin
    file_list = file_search(pattern, COUNT=count)
    if (count EQ 0) then begin
      if keyword_set(last_reported_files_found) then $
        print, now(), strjoin(repstr(last_reported_files_found, '../', ''), ', '),$
               F='(%"\n(%s) Resuming after file disappeared: %s")'

      close, unit, EXIT_STATUS=error_code
      if (error_code NE 0) then begin
        print, !ERROR_STATE.MSG, error_code, F='(%"wait_while_files_exist: CLOSE failed: %s (error %d)")'
        stop
      endif

      free_lun, unit, EXIT_STATUS=error_code
      if (error_code NE 0) then begin
        print, !ERROR_STATE.MSG, error_code, F='(%"wait_while_files_exist: FREE_LUN failed: %s (error %d)")'
        stop
      endif

      return
    endif

    if ~array_equal(last_reported_files_found, file_list) then begin
      print, now(), count, strjoin(repstr(file_list, '../', ''), ', '), F='(%"\n(%s) Waiting while %d files exist: %s")'
      last_reported_files_found = file_list
    endif
    wait, delay ; seconds
    ; When we are monitoring files on an NFS volume, NFS directory caching can hide file creation/removal operations
    ; performed on the server.
    ; Thus, we explicitly refresh those NFS directory caches by creating and removing files from *this* machine.
    dirs_to_monitor = file_dirname(file_list,/MARK_DIRECTORY)
    dirs_to_monitor = dirs_to_monitor[uniq(dirs_to_monitor, sort(dirs_to_monitor))]

    foreach this_directory, dirs_to_monitor do begin
      if ~file_test(this_directory, /DIRECTORY) then continue

      this_poke_file = this_directory+'poke_from_wait_while_files_exist'

      fs = fstat(unit)
      if fs.OPEN then begin
        print, now(), unit, fs.NAME, F='(%"wait_while_files_exist %s: file unit %d (%s) is unexpectedly open.")'
      endif else begin
        openw, unit, ERROR=error_code, this_poke_file
        if (error_code NE 0) then begin
          print, now(), !ERROR_STATE.MSG, error_code, F='(%"wait_while_files_exist %s: OPEN failed: %s (error %d)")'
          continue
        endif
      endelse

      close, unit, EXIT_STATUS=error_code
      if (error_code NE 0) then begin
        print, now(), !ERROR_STATE.MSG, error_code, F='(%"wait_while_files_exist %s: CLOSE failed: %s (error %d)")'
        continue
      endif

      file_delete, /ALLOW_NONEXISTENT,/QUIET, this_poke_file 
     ;print, '.', F='(A1,$)'
    endforeach
  endrep until 0
return   
end ; wait_while_files_exist



;#############################################################################
;; If FILE_WAS_FOUND can return a value then only one of the files specified has to be found, and FILE_WAS_FOUND returns a boolean vector identifying which files were found.
;; If FILE_WAS_FOUND cannot return a value, then all of the files specified have to be found.
PRO wait_until_files_exist, delay, file_list, FILE_WAS_FOUND=file_was_found, DELETE=delete
  
  num_required = arg_present(file_was_found) ? 1 : n_elements(file_list)
  
  get_lun, unit
  last_reported_files_found = replicate(1B,n_elements(file_list))
  repeat begin
    file_was_found = file_test(file_list)
    if (total(file_was_found,/INT) GE num_required) then begin
      if keyword_set(delete) then file_delete, file_list, /ALLOW_NONEXISTENT
      print, now(), strjoin(repstr(file_list[where(file_was_found)], '../', ''), ', '), F='(%"\n(%s) Resuming after finding file: %s")'

      close, unit, EXIT_STATUS=error_code
      if (error_code NE 0) then begin
        print, !ERROR_STATE.MSG, error_code, F='(%"wait_until_files_exist: CLOSE failed: %s (error %d)")'
        stop
      endif

      free_lun, unit, EXIT_STATUS=error_code
      if (error_code NE 0) then begin
        print, !ERROR_STATE.MSG, error_code, F='(%"wait_until_files_exist: FREE_LUN failed: %s (error %d)")'
        stop
      endif

      return
    endif
    
    if ~array_equal(last_reported_files_found, file_was_found) then begin
      ind = where(~file_was_found, count)
      print, now(), count, strjoin(repstr(file_list[ind], '../', ''), ', '), F='(%"\n(%s) Waiting until %d files exist: %s")'
      last_reported_files_found = file_was_found
    endif
    wait, delay ; seconds
    ; When we are monitoring files on an NFS volume, NFS directory caching can hide file creation/removal operations
    ; performed on the server.
    ; Thus, we explicitly refresh those NFS directory caches by creating and removing files from *this* machine.
    dirs_to_monitor = file_dirname(file_list,/MARK_DIRECTORY)
    dirs_to_monitor = dirs_to_monitor[uniq(dirs_to_monitor, sort(dirs_to_monitor))]

    foreach this_directory, dirs_to_monitor do begin
      if ~file_test(this_directory, /DIRECTORY) then begin
        print, now(), this_directory, F='(%"wait_while_files_exist %s: Directory %s does not yet exist.")'
        continue
      endif

      this_poke_file = this_directory+'poke_from_wait_until_files_exist'

      fs = fstat(unit)
      if fs.OPEN then begin
        print, now(), unit, fs.NAME, F='(%"wait_until_files_exist %s: file unit %d (%s) is unexpectedly open.")'
      endif else begin
        openw, unit, ERROR=error_code, this_poke_file
        if (error_code NE 0) then begin
          print, now(), !ERROR_STATE.MSG, error_code, F='(%"wait_until_files_exist %s: OPEN failed: %s (error %d)")'
          continue
        endif
      endelse

      close, unit, EXIT_STATUS=error_code
      if (error_code NE 0) then begin
        print, now(), !ERROR_STATE.MSG, error_code, F='(%"wait_until_files_exist %s: CLOSE failed: %s (error %d)")'
        continue
      endif

      file_delete, /ALLOW_NONEXISTENT,/QUIET, this_poke_file 
     ;print, '.', F='(A1,$)'
    endforeach
  endrep until 0
end ; wait_until_files_exist



;#############################################################################
; When an IDL process holding a semaphore is killed, the semaphore may continue to exist in the operating system.
; Future calls to wait_for_cpu() may be unable to 'grab' that zombie semaphore.
; The release_AE_semaphores procedure or the test_ae_semaphore (below), run from another IDL process, will sometimes destroy the zombie semaphores.
;
; If not, then on OS-X at least, you can try to find and remove zombie semaphores with the unix commands ipcs and ipcrm.  
; For example:
;   ipcs -s | grep $USER | awk '{print $2}' | xargs -I {}  ipcrm -s {}

PRO release_AE_semaphores
  ; Our convention is that we have no more than one semaphore for each CPU core.
  num_semaphores = !CPU.HW_NCPU
  semaphore_name = string(1+indgen(num_semaphores), F='(%"IDL%d")')

  for ii=0,num_semaphores-1 do begin
    success = sem_create(semaphore_name[ii], /DESTROY_SEMAPHORE)
    sem_delete, semaphore_name[ii]
  endfor
  return
  end

PRO test_ae_semaphore, num_semaphores

  semaphore_name = string(1+indgen(num_semaphores), F='(%"IDL%d")')
  for ii=0,num_semaphores-1 do begin
    ; Create the semaphore, or a reference to an existing semephore.
    if ~sem_create(semaphore_name[ii]) then message, 'sem_create() failed on '+semaphore_name[ii]
    
    ; Try to lock the semaphore.
    if sem_lock(semaphore_name[ii]) then begin
      sem_release, semaphore_name[ii]
      print, 'Locked and released semaphore '+semaphore_name[ii]
    endif else begin
      print, 'Cannot lock semaphore '+semaphore_name[ii]
    endelse
    
    sem_delete, semaphore_name[ii]
  endfor
return
end


FUNCTION wait_for_cpu, num_cpus_reserved_p, LOW_PRIORITY=low_priority

COMMON wait_for_cpu, locked_semaphore_name, hostname, defeat_semaphores

  if ~keyword_set(hostname) then begin
    ; Configure the AE semaphore tools.
    run_command, 'hostname', hostname

   ;defeat_semaphores = 0B  ; wait_for_cpu & release_cpu manipulate OS semaphores.
    defeat_semaphores = 1B  ; wait_for_cpu & release_cpu do nothing; useful when investigating system "panics"
  endif

  ; Ask IDL how many CPU cores this machine has.  This is not always accurate.
  num_cpus       = !CPU.HW_NCPU

  ; Create the requested number of semaphores (or references to an existing semaphore).
  DEFSYSV, '!num_cpus_reserved', EXISTS=var_exists

  ; The parameter num_cpus_reserved_p takes priority if passed.
  if                          (n_elements( num_cpus_reserved_p) EQ 1) then begin
    num_cpus_reserved = num_cpus_reserved_p
  ; Otherwise, look for a system variable the user has defined.
  endif else if var_exists && (n_elements(!num_cpus_reserved  ) EQ 1) then begin
    num_cpus_reserved = !num_cpus_reserved
  ; Otherwise, use all the CPUs.
  endif else $
    num_cpus_reserved = 0


  ; Our convention is that we have no more than one semaphore for each CPU core.
  ; However, negative values of the system var !num_cpus_reserved or 
  ; the function paramter num_cpus_reserved_p will allow more semaphores than CPUs.
  num_semaphores = 1 > (num_cpus - num_cpus_reserved)
  semaphore_name = string(1+indgen(num_semaphores), F='(%"IDL%d")')
  
  waiting_reported = 0
  repeat begin
    for ii=0,num_semaphores-1 do begin
      if defeat_semaphores then begin
        print, hostname, now(),  F='(%"Pretending to lock semaphore on %s at %s.")'
        return,                 semaphore_name[ii]
      endif
    
      catch, error_code
      if (error_code NE 0) then begin
        print, !ERROR_STATE.MSG
        continue
      endif else begin
        ; Create a reference to a semaphore in the OS (pre-existing, or created by this call)
        if ~sem_create(semaphore_name[ii]) then message, 'sem_create() failed on '+semaphore_name[ii]
        
        ; Try to lock the semaphore.
        if sem_lock(semaphore_name[ii]) then begin
          ; Record the name of the semaphore we're holding.
          locked_semaphore_name = semaphore_name[ii]
          if waiting_reported then $
            print, semaphore_name[ii], hostname, now(),  F='(%"Locked semaphore %s on %s; processing resumed at %s.")' $
          else $
            print, semaphore_name[ii], hostname, now(),  F='(%"Locked semaphore %s on %s at %s.")' 
          return,                 semaphore_name[ii]
        endif else begin
          ; Reading the manual page for sem_delete() we see that holding a reference to a semaphore created by another process is dangerous.  When that process dies, the semaphore will be released and 'marked for deletion' by the OS, but will not be deleted (because this process has a reference to it).  This process can then lock that semaphore.  The problem is that in the mean time another IDL process that calls sem_create() with that semaphore name will get a reference to a new SECOND SEMAPHORE WITH THE SAME NAME, which it can immediately lock.  In the end, two IDL processes will think they have exclusive locks of the same name.
          ; We hope we can prevent that by destroying references to semaphores that we have not locked, which should allow the OS to actually destroy a semaphore when the IDL process that has it locked terminates.
          sem_delete, semaphore_name[ii]
        endelse
      endelse ; (error_code EQ 0)
      
      ; We arrive here when another process holds the semaphore.
      ;print, semaphore_name[ii], ' is busy.'
      continue
    endfor
    if ~waiting_reported then begin
      print, hostname, now(), F='(%"Waiting for a CPU on %s (%s) ... ")'
      waiting_reported = 1
    endif
    wait, 15 * (keyword_set(low_priority) ? 20 : 1)
  endrep until 0
end


PRO release_cpu

COMMON wait_for_cpu, locked_semaphore_name, hostname, defeat_semaphores

if keyword_set(locked_semaphore_name) then begin
  if defeat_semaphores then begin
    print, hostname, now(),  F='(%"Pretending to release semaphore on %s at %s.")'
    return
  endif
  
  sem_release, locked_semaphore_name

  catch, error_code
  if (error_code NE 0) then begin
    print, locked_semaphore_name, hostname, now(),  F='(%"Call to sem_delete failed on semaphore %s on %s at %s.")'
  endif else begin
    sem_delete , locked_semaphore_name
    print, locked_semaphore_name, hostname, now(),  F='(%"Released semaphore %s on %s at %s.")'
  endelse
  catch, /CANCEL
  
  locked_semaphore_name = ''
endif
return
end



;#############################################################################
;;; Source List Manager tool

;;; See AE manual for usage.

;;; The master source list is maintained in the file "all.srclist".

;;; Comment lines begin with ";".

;;; We retain entries for REMOVED sources rather than excising them from the list 
;;; to keep sequence numbers stable.  A comment character (;) is used to hide them 
;;; from AE, and a comment "REMOVED" is added to distinguish them from regular comments.

;;; Note that the definitive coordinates of a source and the source label are always stored 
;;; in the source.stats file at the top level of the source directory; 
;;; the ADD and MOVE threads in this tool write coordinates there.
;;; This tool accepts a MERGE_NAME input only for the UPDATE_POSITIONS_CORR and UPDATE_POSITIONS_DATA
;;; threads since AE's position estimates may be stored in a named source.stats file.

;;; Single-axis 1-sigma position uncertainties ERROR_RA,ERROR_DEC should be supplied when adding or moving sources.
;;; Units are arcseconds.  Zero position errors are not allowed.  NaN values are allowed.
;;; If not supplied a default of NaN is assumed.

PRO ae_source_manager,       RA=      ra,       DEC=      dec, POSITION_TYPE=posntype, $
                       ERROR_RA=error_ra, ERROR_DEC=error_dec, $ ; single-axis 1-sigma uncertainty [arcsec]
                       NAME=sourcename, LABEL=label, $
                       
                       ADD=add, PROVENANCE=provenance, $

                       MOVE=move, NO_RENAME=no_rename, $ 
                       
                       ASTROMETRY=sky2wcs_astr,  DELTA_X=delta_x, DELTA_Y=delta_y,$ 
                       ;sky2wcs_astr = get_astrometry_from_eventlist('../../tangentplane_reference.evt')
                       
                       UPDATE_POSITIONS_RECON=update_positions_recon, $
                       UPDATE_POSITIONS_CORR=update_positions_corr, $
                       UPDATE_POSITIONS_DATA=update_positions_data, $
                       MERGE_NAME=merge_name, $
                       
                       REMOVE=remove, TRASH_DIR=trash_dir, OBSID_LIST=obsid_list, $
                       
                       ORPHAN_SOURCES=orphan_sources, $
                       
                       SORT_RA=sort_ra, SORT_BULLSEYE=sort_bullseye, $
                       SET_LABEL_AS_SEQUENCE=set_label_as_sequence, $
                       SKIP_BACKUP=skip_backup

src_stats_basename       = 'source.stats'
obs_stats_basename       = 'obs.stats'
srclist_fn               = 'all.srclist'
precision  = 1
creator_string = "ae_source_manager, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()
print

;; Check for common environment errors.
quiet = !QUIET  &  !QUIET = 1
catch, error_code
if (error_code NE 0) then begin
  print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
  retall
endif else resolve_routine, 'astrolib', /NO_RECOMPILE
catch, /CANCEL
astrolib
; Make sure forprint calls do not block for user input.
!TEXTOUT=2
!QUIET = quiet

if keyword_set(sourcename) then sourcename = strtrim(sourcename,2)  
if keyword_set(label)      then label      = strtrim(label     ,2)  


position_type_supplied = 'RA/DEC'


;; ------------------------------------------------------------------------
;; When new source positions are expressed as offsets from the current position, the required DELTA_X & DELTA_Y inputs must be the same length
move_to_offset = keyword_set(move) && keyword_set(sky2wcs_astr) 
if move_to_offset then begin
  position_type_supplied = 'DELTA_X/DELTA_Y'
  
  num_sources = n_elements(delta_x)
  if (num_sources EQ 0) then begin
    print, 'ERROR: DELTA_X & DELTA_Y inputs must be supplied.'
    retall
  endif
  
  if (n_elements(delta_y) NE num_sources) then begin
    print, 'ERROR: DELTA_X & DELTA_Y inputs must be same length.'
    retall
  endif
  
  ; RA and DEC will be computed later; set to null DOUBLE vectors to satisfy input parameter validation below.
  ra   = dblarr(num_sources)
  dec  = dblarr(num_sources)
endif


;; ------------------------------------------------------------------------
;; For /ADD and /MOVE, the required RA & DEC inputs must be the same length as the
;; optional inputs NAME, LABEL, POSITION_TYPE, & PROVENANCE.
if keyword_set(add) OR keyword_set(move) then begin
  num_sources = n_elements(ra)
  if (num_sources EQ 0) then begin
    print, 'ERROR: RA and DEC inputs must be supplied.'
    retall
  endif
  
  if (n_elements(dec) NE num_sources) then begin
    print, 'ERROR: RA and DEC inputs must be same length.'
    retall
  endif
  
  if (size(ra,/TYPE) NE 5) || (size(dec,/TYPE) NE 5) then begin
    print, 'ERROR: RA and DEC inputs must be double precision floats.'
    retall
  endif
  
  if (n_elements(error_ra) NE n_elements(error_dec)) then begin
    print, 'ERROR: ERROR_RA,ERROR_DEC inputs must be same length.'
    retall
  endif

  case (n_elements(error_ra)) of
    num_sources: 
    1          : begin
                 error_ra  = replicate(error_ra, num_sources)
                 error_dec = replicate(error_dec, num_sources)
                 end
    0          : begin
                 error_ra  = replicate(!VALUES.F_NAN, num_sources)
                 error_dec = replicate(!VALUES.F_NAN, num_sources)
                 print, 'Assuming position uncertainty is NaN".'
                 end
    else       : begin
                 print, 'ERROR: Inputs ERROR_RA,ERROR_DEC must be scalars or the same length as '+position_type_supplied
                 retall
                 end
  endcase


  case (n_elements(provenance)) of
    num_sources: provenance =           strtrim(provenance,2)
    1          : provenance = replicate(strtrim(provenance,2), num_sources)
    0          : provenance = replicate(            'unknown', num_sources)
    else       : begin
                 print, 'ERROR: PROVENANCE input must be scalars or the same length as '+position_type_supplied
                 retall
                 end
  endcase

  case (n_elements(posntype)) of
    num_sources: posntype =           strtrim(posntype,2)
    1          : posntype = replicate(strtrim(posntype,2), num_sources)
    0          : begin
                 print, 'ERROR: POSITION_TYPE must describe the position estimation method, or be the empty string (to retain existing value).'
                 retall
                 end
    else       : begin
                 print, 'ERROR: POSITION_TYPE input must be scalars or the same length as '+position_type_supplied
                 retall
                 end
  endcase

  if keyword_set(sourcename) then begin
    if (n_elements(sourcename) NE num_sources) then begin
      print, 'ERROR: NAME input must be same length as '+position_type_supplied
      retall
    endif
  endif 
  
  if keyword_set(label) then begin
    if (n_elements(label) NE num_sources) then begin
      print, 'ERROR: LABEL input must be same length as '+position_type_supplied
      retall
    endif
    
    if (total(strmatch(label,'')) GT 0) then begin
      print, 'ERROR: LABEL input must not be the empty string.'
      retall
    endif
  endif 
  
endif ; /ADD or /MOVE


;; ------------------------------------------------------------------------
;; Read any existing source list.  
;; We cannot use readcol because it won't read comment lines with spaces.
if file_test(srclist_fn) then begin
  Nlines = file_lines(srclist_fn) 
  if (Nlines GT 0) then begin
    master_srclist = strarr(Nlines)
    openr, unit, srclist_fn, /GET_LUN
    readf, unit, master_srclist
    free_lun, unit
  
    if NOT strmatch(master_srclist[0], ";*") then $
      master_srclist = ['; Master Source List',master_srclist] 
  endif ;(Nlines GT 0)
endif else begin
  if keyword_set(move) OR keyword_set(remove) then print, 'WARNING: file '+srclist_fn+' not found.'
endelse

;; We need master_srclist to have at least one comment line to make the code easier later, so we'll add a comment line if necessary.
if (n_elements(master_srclist) EQ 0) then master_srclist =  '; Master Source List'


;; ------------------------------------------------------------------------
if arg_present(orphan_sources) then begin
  orphan_sources   = !NULL
  source_directory = file_dirname(file_search('*/source.stats', COUNT=count))
  print, count, F='(%"Scanning %d source directories for orphans ...")'
  for ii=0L,count-1 do begin
    ; Check whether this source is in the master list.
    ind = where(/NULL, source_directory[ii] EQ master_srclist)
    if ~isa(ind, /INTEGER) then begin
      print, source_directory[ii], F='(%"WARNING: the directory %s appears to be an orphaned source (i.e. no longer in all.srclist).")'
      orphan_sources = [orphan_sources,source_directory[ii]]
    endif
  endfor ;ii
  return
endif ; keyword_set(orphan_sources)


;; ------------------------------------------------------------------------
;; For /MOVE and /REMOVE, existing sources must be identified via NAME or LABEL.
if (keyword_set(move)                   || $
    keyword_set(remove)                 || $
    keyword_set(update_positions_recon) || $
    keyword_set(update_positions_corr)  || $
    keyword_set(update_positions_data)     ) && ~keyword_set(sourcename) then begin
  
  if ~keyword_set(label) then begin
    print, 'ERROR: you must identify sources via NAME or LABEL inputs.'
    retall
  endif
  
  ; Look up NAME values using supplied LABEL values.
  print, 'Looking up sources by their LABELs ...'
  sourcename  = strarr(n_elements(label))
  for ii = 0L, n_elements(master_srclist)-1 do begin
    this_name = master_srclist[ii]
    if strmatch(this_name, ";*") then continue
    
    sourcedir = this_name + '/' 
    
    unnamed_src_stats_fn  = sourcedir + src_stats_basename
    unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    
    if (NOT keyword_set(error)) then begin
      this_label  = strtrim(psb_xpar( unnamed_src_stats, 'LABEL', COUNT=count),2)
      if (count GT 0) then begin
        ind = (where(label EQ this_label, count))[0]
        if (count GT 0) then begin
          if (sourcename[ind] NE '') then begin
            print, 'ERROR: Two sources have the same label '+this_label
            retall
          endif
          
          sourcename[ind] = this_name
        endif ; match to this_label found
      endif ; source has LABEL keyword
    endif else begin
      print, error
      print, 'WARNING! Could not read '+unnamed_src_stats_fn
    endelse
  endfor ;ii

  ind = where(sourcename EQ '', count)
  if (count GT 0) then begin
    print, 'ERROR: could not find these LABELs: '
    forprint, label, SUBSET=ind
    retall
  endif

  forprint, label, sourcename, F='(%"LABEL %s == %s")'
endif ; LABEL supplied instead of NAME


;; ------------------------------------------------------------------------
;; For /MOVE and /REMOVE, we must consider what should happen to any existing extractions that have been performed.
if (keyword_set(move) OR keyword_set(remove)) then begin
  if (n_elements(obsid_list) EQ 0) then obsid_list = ''
  
  num_sources = n_elements(sourcename)
  
  for ii = 0L, num_sources-1 do begin
    if file_test(sourcename[ii], /DIRECTORY) then begin
      ; Locate obs.stats files from existing extractions.
      obs_stats_fn = file_search(sourcename[ii], obs_stats_basename, COUNT=num_obs)
      if (num_obs GT 0) then begin
        ; We do NOT remove any extraction files here just because the source has moved.
        ; The obs.stats files must be preserved because they carry the FRACSPEC value that ae_make_catalog needs.
        ; It is the responsibility of ae_make_catalog and the CONSTRUCT stage to determine if the source has moved off the field of view of an ObsID already extracted.
      
        ; Keep a list of the obsids that observed the discarded sources.
        ; We parse the pathname to obs.stats to determine the ObsID.
        these_obsids = strmid(obs_stats_fn,  1+reform(strpos(obs_stats_fn, '/'), 1,num_obs))
        these_obsids = strmid(these_obsids, 0, reform(strpos(these_obsids, '/'), 1,num_obs))

        obsid_list   = [obsid_list, these_obsids]
        obsid_list   = obsid_list[uniq(obsid_list,sort(obsid_list))]
      endif
    endif
  endfor ;ii
  if (n_elements(obsid_list) GT 1) then obsid_list = obsid_list[1:*]
endif


;; ------------------------------------------------------------------------
;; ADD sources.  RA/DEC must be supplied, not DELTA_X/DELTA_Y.
if keyword_set(add) && ~move_to_offset then begin
  if NOT keyword_set(sourcename) then begin
    sourcename = strcompress(/REMOVE_ALL, adstring(ra,dec,precision,/TRUNCATE))
  endif
  
  for ii=0L,num_sources-1 do begin
    sourcedir = sourcename[ii] + '/' 
    unnamed_src_stats_fn  = sourcedir + src_stats_basename
                       
    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    ; Use existing src_stats file if possible.
    if file_test(unnamed_src_stats_fn) then begin
      print, 'WARNING: modifying existing source '+sourcename[ii]
      unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
      if keyword_set(error) then begin
        print, error
        message, 'ERROR reading '+unnamed_src_stats_fn
      endif
    endif else begin
      print, 'Adding source '+sourcename[ii]
      file_mkdir, sourcedir
      fxhmake, unnamed_src_stats, /INITIALIZE, /EXTEND, /DATE
      psb_xaddpar, unnamed_src_stats, 'LABEL', 'unlabeled', 'source label'
    endelse
 
    ; Check whether this source is in the master list.
    dum = where(sourcename[ii] EQ master_srclist, count)
    if (count EQ 0) then begin
      ; Append the source to the master list and assign a default label as the 1-based 
      ; sequence number in the list, counting removed sources.
      master_srclist = [master_srclist, sourcename[ii]]
      comment_flag = strmatch(master_srclist, ";*")
      removed_flag = strmatch(master_srclist, "*(REMOVED*")
      seq_num = round(n_elements(master_srclist)-total(comment_flag)+total(removed_flag)) 
      psb_xaddpar, unnamed_src_stats, 'LABEL', strtrim(string(seq_num),2), 'sequence number'
    endif
    
    psb_xaddpar, unnamed_src_stats, 'CREATOR', creator_string
    psb_xaddpar, unnamed_src_stats, 'OBJECT',   sourcename[ii], 'source name'
    get_date, date_today, /TIMETAG
    psb_xaddpar, unnamed_src_stats, 'POSNDATE',     date_today, 'UTC date RA,DEC were changed'
    psb_xaddpar, unnamed_src_stats, 'RA',               ra[ii], '[deg] source position', F='(F10.6)'
    psb_xaddpar, unnamed_src_stats, 'DEC',             dec[ii], '[deg] source position', F='(F10.6)'
    psb_xaddpar, unnamed_src_stats, 'ERR_RA'  ,   error_ra[ii], '[arcsec] 1-sigma uncertainty around (RA,DEC)', F='(F10.3)'              
    psb_xaddpar, unnamed_src_stats, 'ERR_DEC'  , error_dec[ii], '[arcsec] 1-sigma uncertainty around (RA,DEC)', F='(F10.3)'              
    
    psb_xaddpar, unnamed_src_stats, 'PROVENAN', provenance[ii], 'source provenance'
    if keyword_set(posntype[ii]) then $
    psb_xaddpar, unnamed_src_stats, 'POSNTYPE',   posntype[ii], 'type of source position'
    
    ; The initial range and goal for BACKSCAL are set values typical for far off-axis sources.
    ; If these defaults were set to the larger values typical for on-axis sources, then the first few extractions of far off-axis sources would build bkg regions much larger than necessary and would run VERY slowly.
    psb_xaddpar, unnamed_src_stats, 'BKSCL_LO',            5.0, 'smallest BACKSCAL allowed'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_GL',            5.0, 'target   BACKSCAL'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_HI',           10.0, 'largest  BACKSCAL allowed'

    ; Record any observer-supplied LABEL, overwriting the default label above.
    if keyword_set(label) then psb_xaddpar, unnamed_src_stats, 'LABEL', label[ii], 'source label'
    
    print, 'Source LABEL is "'+strtrim(psb_xpar( unnamed_src_stats, 'LABEL'),2)+'"'
    
    writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
  endfor ;ii

  GOTO, WRITE_SRCLIST
endif ; /ADD


;; ------------------------------------------------------------------------
;; /MOVE 
if keyword_set(move) then begin
  if NOT keyword_set(sourcename) then begin
    print, 'ERROR: existing source name must be supplied via NAME.'
    retall
  endif

  was_moved_flag = bytarr(num_sources)
  for ii=0L,num_sources-1 do begin
    sourcedir = sourcename[ii] + '/' 
    unnamed_src_stats_fn  = sourcedir + src_stats_basename

    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    ; Read existing src_stats file.
    if file_test(unnamed_src_stats_fn) then begin
      unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
      if keyword_set(error) then begin
        print, error
        message, 'ERROR reading '+unnamed_src_stats_fn
      endif
    endif else begin
      print, 'ERROR: source directory '+sourcename[ii]+' not found'
      continue
    endelse

    
    if move_to_offset then begin
      if (delta_x[ii] EQ 0) && (delta_y[ii] EQ 0) then continue
      
      ; Compute new coordinates by shifting existing coordinates.
      ad2xy, psb_xpar( unnamed_src_stats,'RA'), psb_xpar( unnamed_src_stats,'DEC'), sky2wcs_astr, this_x , this_y 
      xy2ad,            this_x+delta_x[ii],             this_y+delta_y[ii], sky2wcs_astr, this_ra, this_dec 
      
      ra [ii] = this_ra
      dec[ii] = this_dec
    endif ; keyword_set(shift)
    

    if keyword_set(no_rename) then begin
      sourcename_new = sourcename[ii]
    endif else begin
      sourcename_new = strcompress(/REMOVE_ALL, adstring(ra[ii],dec[ii],precision,/TRUNCATE))
    endelse
    
    ; Before modifying any files, determine if the source will be renamed and if
    ; there are any obstacles to doing so.
    name_is_changing = (sourcename[ii] NE sourcename_new)
    if name_is_changing AND file_test(sourcename_new) then begin
      print, 'WARNING: '+sourcename[ii]+' cannot be moved because new name duplicates existing source '+sourcename_new
      continue
    endif
    
    
    ; Compute how far source moved.
    hours_per_degree = 24D/360
    gcirc, 1, psb_xpar( unnamed_src_stats,'RA')*hours_per_degree, psb_xpar( unnamed_src_stats,'DEC'), ra[ii]*hours_per_degree, dec[ii], distance

    ; Check whether this source is in the master list.
    ind = where(sourcename[ii] EQ master_srclist, count)
    if (count EQ 0) then begin
      print, 'WARNING: source '+sourcename[ii]+' not found in '+srclist_fn
      ; Append the source to the master list and assign a default label as the 1-based 
      ; sequence number in the list, counting removed sources.
      master_srclist = [master_srclist, sourcename[ii]]
      comment_flag = strmatch(master_srclist, ";*")
      removed_flag = strmatch(master_srclist, "*(REMOVED*")
      seq_num = round(n_elements(master_srclist)-total(comment_flag)+total(removed_flag)) 
      psb_xaddpar, unnamed_src_stats, 'LABEL', string(seq_num), 'sequence number'
    endif else begin
      ; Change the name of this source.
      master_srclist[ind] = sourcename_new
    endelse
    
    psb_xaddpar, unnamed_src_stats, 'CREATOR', creator_string
    psb_xaddpar, unnamed_src_stats, 'OBJECT', sourcename_new, 'source name'
    get_date, date_today, /TIMETAG
    psb_xaddpar, unnamed_src_stats, 'POSNDATE',     date_today, 'UTC date RA,DEC were changed'
    psb_xaddpar, unnamed_src_stats, 'RA',               ra[ii], '[deg] source position', F='(F10.6)'
    psb_xaddpar, unnamed_src_stats, 'DEC',             dec[ii], '[deg] source position', F='(F10.6)'
    psb_xaddpar, unnamed_src_stats, 'ERR_RA'  ,   error_ra[ii], '[arcsec] 1-sigma uncertainty around (RA,DEC)', F='(F10.3)'              
    psb_xaddpar, unnamed_src_stats, 'ERR_DEC'  , error_dec[ii], '[arcsec] 1-sigma uncertainty around (RA,DEC)', F='(F10.3)'              
    
    psb_xaddpar, unnamed_src_stats, 'PREVNAME', sourcename[ii]
    if keyword_set(posntype[ii]) then $
    psb_xaddpar, unnamed_src_stats, 'POSNTYPE',   posntype[ii], 'type of source position'
    
    ; Record any observer-supplied LABEL.
    if keyword_set(label) then psb_xaddpar, unnamed_src_stats, 'LABEL', label[ii], 'source label'
    
    msg = string( (distance GT 2) ? 'WARNING: ' : '         ' , sourcename[ii], psb_xpar( unnamed_src_stats, 'LABEL'), distance, F='(%"%s%s (%s) moved %0.1f arcsec")' )

    ; Remove obsolete information from source.stats and resave.
    writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
    was_moved_flag[ii] = 1
    
    ; If the move was significant, remove neighborhood.evt files.
    ; When neighborhood.evt files are re-used (as a speedup), this reduces the risk that the aperture could drift outside the neighborhood.
    if (distance GT 1) then begin  ; movement > 1 arcsec
      filelist = file_search(sourcedir, 'neighborhood.evt', COUNT=count)
      if (count GT 0) then file_delete, filelist
    endif
    
    ; Rename the source directory.
    if name_is_changing then begin
      print, msg + ' (renamed to '+sourcename_new+')'
      
      ; Remove files that use the old source name.
      filelist = file_search(sourcedir, sourcename[ii]+'*', COUNT=count)
      if (count GT 0) then file_delete, filelist
      
      file_move, sourcedir, sourcename_new
    endif else print, msg
  endfor ;ii
  
  print,          total(/INT,  was_moved_flag), F='(%"%d sources were moved.")'
  num_not_moved = total(/INT, ~was_moved_flag)
  if (num_not_moved GT 0) then print, num_not_moved, F='(%"%d sources were not moved.")'
  
  ind = where(~was_moved_flag, count)

  GOTO, WRITE_SRCLIST
endif ; /MOVE


;; ------------------------------------------------------------------------
;; REMOVE sources
if keyword_set(remove) then begin
  get_date, date_today
  num_sources = n_elements(sourcename)
  
  openw, region1_unit, 'removed_sources.reg', /GET_LUN, /APPEND

  if NOT keyword_set(trash_dir) then trash_dir = './trash'
  file_mkdir, trash_dir
  
  was_removed_flag = bytarr(num_sources)
  for ii = 0L, num_sources-1 do begin
    sourcedir            = sourcename[ii] + '/'
    unnamed_src_stats_fn = sourcedir + src_stats_basename

    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    ; If a directory claimed to be an AE source directory does not have a source.stats file, then refuse to remove it.
    ; We don't want a mistake to lead to the loss of random files (by the file_delete call below).
    if ~file_test(unnamed_src_stats_fn) then begin
      print, sourcename[ii], F='(%"\nWARNING! Directory %s is not an AE source; skipping ...")'
      continue
    endif

    ; Append to removed_sources.reg a region file for the removed source.
    unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    
    if ~keyword_set(error) then begin
      this_ra    = psb_xpar( unnamed_src_stats, 'RA')
      this_dec   = psb_xpar( unnamed_src_stats, 'DEC')
      this_label = psb_xpar( unnamed_src_stats, 'LABEL')
    endif else begin
      print, error
      print, 'WARNING! Could not read '+unnamed_src_stats_fn
    endelse
    printf, region1_unit, this_ra, this_dec, this_label, date_today, sourcename[ii], F='(%"J2000;diamond point %10.6f %10.6f # text={%s} tag={removed} tag={UTC %s} color=yellow name={%s}")'
    
    ; Move source directory to the trash.
    if file_test(sourcedir, /DIRECTORY) then begin
      print, 'Moving '+sourcedir+' to '+expand_path(trash_dir)+'/ ...'
      file_copy, sourcedir, trash_dir, /RECURSIVE, /COPY_SYMLINK

      list = reverse(file_search(sourcedir,'*',/MATCH_INITIAL_DOT,COUNT=count))
      if (count GT 2000) then begin
        print, sourcedir, count, F='(%"\nWARNING!  %s contains %d files.  Investigate before continuing.")'
        stop
      endif
      if (count GT 0) then file_delete, list
      file_delete, sourcedir

      was_removed_flag[ii] = 1
    endif else print, 'WARNING: source directory '+sourcedir+' not found.' 

    ; We retain the master sourcelist entry rather than removing it to keep sequence numbers stable.
    ; A comment character (;) is used to hide it from AE, and a tag "REMOVED" is added to distinguish it
    ; from regular comments.
    ind = where(sourcename[ii] EQ master_srclist, count)
    if (count EQ 0) then begin
      print, 'WARNING: source '+sourcename[ii]+' not found in '+srclist_fn
    endif else master_srclist[ind] = ';'+master_srclist[ind]+' (REMOVED on UTC '+date_today+')'
  endfor ; ii
  print, total(/INT, was_removed_flag), F='(%"%d sources were removed.")'
  free_lun, region1_unit
  
  GOTO, WRITE_SRCLIST
endif ; /REMOVE


;; ------------------------------------------------------------------------
;; SORT source list
if keyword_set(sort_bullseye) or keyword_set(sort_ra) then begin
  ; Split the source list into comments and sources.
  comment_flag = strmatch(master_srclist, ";*")
  comment_ind = where(comment_flag, COMPLEMENT=src_ind, NCOMPLEMENT=num_entries)
  comment_list = master_srclist[comment_ind]

  if (num_entries GT 0) then begin
    ; Remove the comment lines.
    master_srclist = master_srclist[src_ind]
    
    ; Read the RA/DEC values.
    print, 'Looking up source coordinates ...'
    ra  = replicate(!VALUES.D_NAN,num_entries)
    dec = replicate(!VALUES.D_NAN,num_entries)
    for ii=0L, num_entries-1 do begin
      sourcedir = master_srclist[ii] + '/' 
      
      unnamed_src_stats_fn  = sourcedir + src_stats_basename
      unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
      
      if (NOT keyword_set(error)) then begin
        ra[ii]  = psb_xpar( unnamed_src_stats, 'RA')
        dec[ii] = psb_xpar( unnamed_src_stats, 'DEC')
      endif else begin
        print, error
        print, 'WARNING! Could not read '+unnamed_src_stats_fn
      endelse
    endfor ;ii
    
    if keyword_set(sort_ra) then begin
      print, 'Sorting by RA ...'
      sort_ind = sort(ra)
    endif else begin
      openw,  region_unit, 'bullseye.reg', /GET_LUN
      printf, region_unit, "# Region file format: DS9 version 3.0"
      printf, region_unit, 'global width=1 font="helvetica 12 normal"'
 
      sort_ind = lonarr(num_entries)
      ; Compute distances and position angles from the median position.
      ra_ref  = median(ra)
      dec_ref = median(dec)
      hours_per_degree = 24D/360
      gcirc,  1, ra_ref*hours_per_degree, dec_ref, ra*hours_per_degree, dec, distance
      posang, 1, ra_ref*hours_per_degree, dec_ref, ra*hours_per_degree, dec, angle
      
      ; Sort by distances.
      distance_sort_ind = sort(distance)
      
      ; Define annuli containing 200 sources.  Within each, sort by angle.
      group_size  = 199
      group_start = 0L
      repeat begin
        group_end = (group_start + group_size-1) < (num_entries-1)
        group_size= 200
        
        printf, region_unit, ra_ref, dec_ref, distance[distance_sort_ind[group_end]], F='(%"J2000;circle %10.6f %10.6f %d\" # tag={bullseye}")' 
        
        group_ind = distance_sort_ind[group_start:group_end]
        
        sorted_group_ind = group_ind[sort(angle[group_ind])]
        
        sort_ind[group_start] = sorted_group_ind
        
        group_start = group_end+1
      endrep until (group_start GE num_entries)
      free_lun, region_unit
      
;      forprint, distance, angle, SUBSET=sort_ind
;      plot, ra, dec, /NODATA
;      for ii=0L,(num_entries/10)-1 do begin
;        s=ii*10
;        e=s+9
;        oplot, ra[s:e], dec[s:e], PSYM=1
;        wait,0.5 
;      endfor
    endelse
  endif ; (num_entries GT 0)
  
  master_srclist = [comment_list, master_srclist[sort_ind]]

  GOTO, WRITE_SRCLIST
endif ; /SORT


;; ------------------------------------------------------------------------
;; SET_LABEL_AS_SEQUENCE
if keyword_set(set_label_as_sequence) then begin
  ; Assign LABEL values using 1-based sequence numbers.
  print, 'Assigning LABELs using 1-based sequence numbers ...'
  seq_num = 1
  for ii = 0L, n_elements(master_srclist)-1 do begin
    this_name = master_srclist[ii]
    if strmatch(this_name, ";*") then continue
    
    sourcedir = this_name + '/' 
    
    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    unnamed_src_stats_fn  = sourcedir + src_stats_basename
    unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    
    if (NOT keyword_set(error)) then begin
      old_label  = strtrim(psb_xpar( unnamed_src_stats, 'LABEL', COUNT=count),2)
      new_label  = strtrim(string(seq_num),2)
      seq_num    = seq_num + 1
      psb_xaddpar, unnamed_src_stats, 'LABEL', new_label, 'sequence number'
      
      msg = this_name+' labeled '+new_label
      if (count GT 0) then begin
        msg = msg+'  (was '+old_label+')'
        psb_xaddpar, unnamed_src_stats, 'PREVLABL', old_label
      endif
      print, msg
    
      writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
    endif else begin
      print, error
      print, 'WARNING! Could not read '+unnamed_src_stats_fn
    endelse
  endfor ;ii
  return
endif ; /SET_LABEL_AS_SEQUENCE

 
;; ------------------------------------------------------------------------
;; MOVE TO SPECIFIED TYPE OF POSITION
update_option_count = keyword_set(update_positions_recon) + $
                      keyword_set(update_positions_corr)  + $
                      keyword_set(update_positions_data)
if (update_option_count GT 1) then begin
  print, 'ERROR: UPDATE_POSITIONS_* options are mutually exclusive.'
  retall
endif
if (update_option_count EQ 1) then begin
  if NOT keyword_set(sourcename) then begin
    print, 'ERROR: source name must be supplied via NAME.'
    retall
  endif
  
  ;; Replace RA & DEC keywords with RA_CORR,DEC_CORR or RA_DATA,DEC_DATA.
  if keyword_set(update_positions_recon) then begin
    ra_kywd    = 'RA_ML'
    dec_kywd   = 'DEC_ML'
    posntype_new = 'AE reconstruction' 
  endif else if keyword_set(update_positions_corr) then begin
    ra_kywd    = 'RA_CORR'
    dec_kywd   = 'DEC_CORR'
    posntype_new = 'AE correlation'
  endif else begin
    ra_kywd    = 'RA_DATA'
    dec_kywd   = 'DEC_DATA'
    posntype_new = 'AE mean data'
  endelse
  
  if keyword_set(posntype) then begin
    print, 'ERROR: do not supply POSITION_TYPE; ae_source_manager will assign the value ', posntype_new
    retall
  endif
  
  num_sources = n_elements(sourcename)
  
  update = replicate({ra:0D, dec:0D, error_ra:0., error_dec:0., source_can_move:0B}, num_sources)

  if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                  else merge_subdir = ''
  if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)
  
  ; Find new coordinates and uncertainties for each source.
  for ii = 0L, num_sources-1 do begin
    basedir   = sourcename[ii] + '/' 
    sourcedir = basedir + merge_subdir[ii]
    stats_fn  = sourcedir + src_stats_basename

   ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(basedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    stats = headfits(stats_fn, ERRMSG=error)
    
    if (~keyword_set(error)) then begin
      ; Find new position estimates.
      update[ii].ra  = psb_xpar( stats,  ra_kywd, COUNT=count1)
      update[ii].dec = psb_xpar( stats, dec_kywd, COUNT=count2)
      
      update[ii].source_can_move = (count1 EQ 1) && (count2 EQ 1) && finite(update[ii].ra) && finite(update[ii].dec) 
      if ~update[ii].source_can_move then begin
        print, 'WARNING!  Keywords '+ra_kywd+','+dec_kywd+' not defined for '+sourcename[ii]+'; source NOT moved'
        continue
      endif
      
      
      ; Find new position uncertainties.
      update[ii].error_ra  = psb_xpar( stats, 'ERX_DATA', COUNT=count1) ; arcsec
      update[ii].error_dec = psb_xpar( stats, 'ERY_DATA', COUNT=count2) ; arcsec
      
      if ~((count1 EQ 1) && (count2 EQ 1)) then begin
        print, 'WARNING!  Position uncertainties not available for '+sourcename[ii]
        update[ii].error_ra  = !VALUES.F_NAN
        update[ii].error_dec = !VALUES.F_NAN
      endif
      
      if keyword_set(update_positions_corr) then begin
        ; CORRelation positions are subject to quantization errors.
        ; The correct mathematical treatment of this quantization noise is not obvious to me.
        ; Using QUANTCOR as a floor on ERX_DATA,ERY_DATA seems reasonable.
        quantization_corr = psb_xpar( stats, 'QUANTCOR', COUNT=count1) ; arcsec
        if (count1 NE 1) then print, 'WARNING! QUANTCOR not found in '+stats_fn $
        else begin
          if ~finite(quantization_corr) then print, 'WARNING! QUANTCOR is not finite!'
          if (0  GT  quantization_corr) then print, 'WARNING! QUANTCOR is not positive!'
          
          if (quantization_corr GT update[ii].error_ra) || (quantization_corr GT update[ii].error_dec) then begin
            print, quantization_corr, stats_fn, F='(%"A floor of %0.3f (QUANTCOR) has been imposed on the position uncertainties in %s.")'
          
            update[ii].error_ra  = max(/NAN, [quantization_corr, update[ii].error_ra ] )
            update[ii].error_dec = max(/NAN, [quantization_corr, update[ii].error_dec] )
          endif
        endelse ; QUANTCOR found
      endif ; /UPDATE_POSITIONS_CORR
      
      ; RECON positions are also derived from a reconstructed image, with pixels that could be large.
      ; However, the position estimate is a centroid over several pixels, so the scale of the quantization noise is not obvious.
      
    endif else begin
      print, error
      print, 'ERROR! Could not read '+stats_fn
    endelse
  endfor
  
  ind = where(update.source_can_move, count)
  if (count GT 0) then $
    ae_source_manager, /MOVE, NO_RENAME=keyword_set(no_rename), NAME=sourcename[ind],$
                       RA      =update[ind].ra      , DEC      =update[ind].dec      ,$
                       ERROR_RA=update[ind].error_ra, ERROR_DEC=update[ind].error_dec, $ ; single-axis 1-sigma uncertainty [arcsec]
                       POSITION_TYPE=posntype_new
  return
endif ; /UPDATE_POSITIONS_*


;; ------------------------------------------------------------------------
WRITE_SRCLIST:
  if NOT keyword_set(skip_backup) then begin
    ;; Maintain several backups of the source list.
    for ii=9,1,-1 do begin
      backup_fn       = string(srclist_fn,ii,  F='(%"%s-%d")')
      older_backup_fn = string(srclist_fn,ii+1,F='(%"%s-%d")')
      if file_test(backup_fn) then file_move, backup_fn, older_backup_fn, /OVERWRITE
    endfor
    
    if file_test(srclist_fn) then file_move, srclist_fn, backup_fn, /OVERWRITE
  endif
  
  ;; Save the modified source list.
  forprint, TEXTOUT=srclist_fn, master_srclist, /NoCOMMENT, /SILENT
  comment_flag = strmatch(master_srclist, ";*")
  removed_flag = strmatch(master_srclist, "*(REMOVED*")
  print, srclist_fn, n_elements(master_srclist)-total(comment_flag), total(removed_flag), F='(%"Master source list %s contains %d active and %d removed sources.")'

  if (keyword_set(move) OR keyword_set(remove)) then begin
    print
    print, 'The specified sources contained extractions from these obsids:'
    print, '  "'+strjoin(obsid_list,' ')+'"'
  endif  
  
  return
end ; ae_source_manager



;#############################################################################
;;; For a SINGLE obsid construct a catalog that leads to non-overlapping regions.
;;;
;;; Several input files are located using standard naming conventions as shown 
;;; in the Getting Started section of the AE manual.
;;;
;;; The required parameter EVTFILE_BASENAME specifies the name of the observation event list file in
;;; the directory ../obsXXXX/.  In our recipe we use EVTFILE_BASENAME='spectral.evt' (lightly cleaned) for all passes.
;;; AE stages apply a STATUS=0 filter (recorded in the src property S_FILTER) for sources that have event rates low enough to avoid removing valid events.

;;; SOURCE_NOT_OBSERVED is an output vector flagging sources not observed in this obsid. 
;;; This can be passed along to ae_standard_extraction to help speed it up a little..

;;; If /IGNORE_IF_FRESH is specified then processing is aborted if either of these conditions are true:
;;; * No sources in the catalog were observed by this obsid. This can happen in recon_detect extractions
;;;
;;; * All sources that were observed are found to have fresh extractions, determined by comparing the
;;;   timestamps on the position (POSNDATE) and on the extraction (EXTRDATE).

;;; WAS_IGNORED is an output boolean reporting whether if the IGNORE_IF_FRESH logic decided to 
;;; skip the processing of this observation.


PRO ae_make_catalog, obsname, EVTFILE_BASENAME=evtfile_basename, $
    SRCLIST_FILENAME=srclist_fn, $
    NOMINAL_PSF_FRACTION=nominal_psf_frac_kywd, MINIMUM_PSF_FRACTION=minimum_psf_frac_kywd, REGION_ONLY=region_only, REUSE_NEIGHBORHOOD=reuse_neighborhood_p, REUSE_PSF=reuse_psf,$
    RESTART=restart, SHOW=show, $
    SOURCE_NOT_OBSERVED=source_not_observed, IGNORE_IF_FRESH=ignore_if_fresh, WAS_IGNORED=was_ignored, $
    SEED_COLLATED_FILENAME=seed_collated_filename, $
    _EXTRA=extra


exit_code = 0
creator_string = "ae_make_catalog, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

obs_stats_basename       = 'obs.stats'
env_events_basename      = 'neighborhood.evt'
src_region_basename      = 'extract.reg'

was_ignored = 0
if ~keyword_set(evtfile_basename) then begin
  print, 'ERROR: parameter EVTFILE_BASENAME must be supplied'
  GOTO, FAILURE
endif

if keyword_set(srclist_fn) then begin
  print, 'WARNING: Aperture adjustment to avoid overlap may be unreliable when only a subset of full catalog is processed.'
endif else srclist_fn = 'all.srclist'

if ~keyword_set(obsname) then begin
  print, 'ERROR: parameter "obsname" must be supplied'
  GOTO, FAILURE
endif

if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
  print, 'ERROR: parameter "obsname" must be a scalar string'
  GOTO, FAILURE
endif

obsdir      = '../obs' + obsname + '/'
if ~file_test(obsdir) then begin
  print, 'ae_make_catalog: ERROR: could not find observation directory ', obsdir
  GOTO, FAILURE
endif

lock_fn     = obsdir + 'ae_lock'
file_delete, /ALLOW_NONEXISTENT, lock_fn
file_copy  , '/dev/null'       , lock_fn, /OVERWRITE



model_savefile = obsdir + 'ae_make_catalog.sav'
if keyword_set(restart) then begin
  print, model_savefile, F='(%"\nRESTARTING LOOP using state saved in %s.\n")'
  restore, /V, model_savefile
endif

;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir
  
active_catfile     = tempdir + 'temp.cat'
active_collatefile = tempdir + 'temp.collated'
temp_events_fn     = tempdir + 'temp.evt'
temp_region_fn     = tempdir + 'temp.reg'
temp_pi_fn         = tempdir + 'temp.pi'

if keyword_set(restart) then begin
  print, model_savefile, F='(%"\nRESTARTING LOOP using state saved in %s.\n")'
  GOTO, RESTART
endif




print, F='(%"\nae_make_catalog: ============================================================")'  
print,        'ae_make_catalog: PROCESSING OBSERVATION ' + obsname
print,   F='(%"ae_make_catalog: ============================================================")'  

;; =====================================================================

psf_energ         = 1.4967

if (n_elements(show) EQ 0) then show = 0

;; In this program all variables, except the keyword NOMINAL_PSF_FRAC, represent
;; PSF fractions as integer percentages.
nominal_psf_frac  = 90
minimum_psf_frac  = 40
if keyword_set(nominal_psf_frac_kywd) then nominal_psf_frac = 99 < round(100*nominal_psf_frac_kywd) > 10
if keyword_set(minimum_psf_frac_kywd) then minimum_psf_frac = 99 < round(100*minimum_psf_frac_kywd) > 10
initial_down_step =-32
initial_up_step   =  4
minimum_step      =  2
step_ratio        = 2.0

catfile     = obsdir + 'obs.cat'

evtfile     = obsdir + evtfile_basename
emapfile    = obsdir + 'obs.emap'
aspect_fn   = obsdir + 'obs.asol'

regionfile  = obsdir + 'extract.reg'

;; =====================================================================
;; Decide if we can safely honor a request to REUSE_NEIGHBORHOOD.
;; We want to make sure that neighborhoods are not reused when the event list has been changed since the last extraction.
;; We define the time of last extraction as the timestamp on the file obs.cat, which this tool writes just before finishing.
catfile_info = file_info(catfile)
evtfile_info = file_info(evtfile)

if keyword_set(reuse_neighborhood_p) && catfile_info.EXISTS then begin
  if (catfile_info.MTIME GT evtfile_info.MTIME) then reuse_neighborhood = 1 $
  else begin
    print, systime(0,evtfile_info.MTIME), systime(0,catfile_info.MTIME), F='(%"\nWARNING!  Your request to REUSE_NEIGHBORHOOD has been ignored because the event list was modified on %s, which is subsequent to your last extraction (on %s).\n")'
  endelse
endif


;; =====================================================================
;; Defensively, verify that the event data are NEWER than the aspect file!
;; A procedural mistake that shifts an ObsID's aspect file but fails to reprocess the event data (update the event positions) can lead to LOTS of grief!
aspect_info = file_info(aspect_fn)

if (aspect_info.MTIME GT evtfile_info.MTIME) then begin
  print, systime(0,aspect_info.MTIME), systime(0,evtfile_info.MTIME), F='(%"\n\nERROR!  The ASPECT FILE is newer (%s) than the event list (%s).\nAre you SURE that your event data has the correct astrometry?\n\n")'

  stop
endif ;


;; =====================================================================
;; CONSTRUCT INITIAL CATALOG


; Read the source list supplied.
readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'

; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)

if (num_sources EQ 0) then begin
  print, 'ae_make_catalog: ERROR: no entries read from source list ', srclist_fn
  GOTO, FAILURE
endif

sourcename = sourcename[ind]
print, num_sources, F='(%"\nae_make_catalog: %d sources found in catalog.")'



; First, write a temporary catalog holding the FULL target source list.
cat_all = replicate({sourcename: '', ra:0D, dec:0D, psf_frac:nominal_psf_frac, psf_energ:psf_energ, $
                     overlap   :0.0, is_writable:0B, aperture_is_writable:0B}, num_sources)
cat_all.sourcename = sourcename

fmt = '(A,1x,F10.6,1x,F10.6,1x,F5.3,1x,F7.5)'
forprint, TEXTOUT=active_catfile, cat_all.sourcename, cat_all.ra, cat_all.dec, cat_all.psf_frac/100., cat_all.psf_energ, F=fmt, /NoCOMMENT, /SILENT



if keyword_set(seed_collated_filename) then begin
  bt_all = mrdfits(seed_collated_filename, 1, /SILENT)
endif else begin
  ;; ---------------------------------------------------------------------
  ;; Perform a collation on the FULL target catalog constructed above (cat_all, with num_sources entries) to: 
  ;; 1. obtain RA and DEC coordinates so we can determine which sources are in the FOV of this ObsID.
  ;; 2. obtain FRACSPEC values from any existing extractions so that we start our search at a reasonable PSF fraction
  ;; 3. determine which sources are write protected.
  print, F='(%"\nae_make_catalog: ============================================================")'  
  print,        'ae_make_catalog: Running COLLATE stage to find any existing PSF fractions ' 
  print,   F='(%"ae_make_catalog: ============================================================\n")'  
  
  acis_extract, active_catfile, obsname, /SINGLE_OBS, COLLATED_FILENAME=active_collatefile, VERBOSE=0, _STRICT_EXTRA=extra
  
  bt_all = mrdfits(active_collatefile, 1, /SILENT)
endelse

if (n_elements(bt_all) NE num_sources) then message, 'BUG in ae_make_catalog!'

; The COLLATE stage reports write-permission status of the source directory via the tag IS_WRITABLE.
cat_all.is_writable = bt_all.is_writable


;; ---------------------------------------------------------------------
;; Ignore sources that are not observed by this ObsID.  
;; It's much faster to lookup the emap at each source position here, rather than ask AE to do it in the CONSTRUCT stage, where individual dmcoords calls are necessary (since the CONSTRUCT stage does not have a vector of RA and DEC for the catalog).

; Read the exposure map & setup array index to sky coordinate conversion.
emap = readfits(emapfile, emap_header, /SILENT)
extast, emap_header, emap2wcs_astr
  
; Choose an emap threshold that defines whether a source is observed or not.  
; 10% of emap_global_median is too large---we have seen the emap at the center of the I-array drop to 7% of emap_global_median.
; A similar "off-field" decision is made in AE's CONSTRUCT stage.
ind = where(finite(emap) AND (emap GT 0))
emap_global_median = median(emap[ind])
emap_threshold     = emap_global_median * 0.05
emap_xdim = (size(emap, /DIM))[0]
emap_ydim = (size(emap, /DIM))[1]

; Compare emap value at the location of each source to emap_threshold.
ad2xy, bt_all.RA, bt_all.DEC, emap2wcs_astr, xindex, yindex 

xindex >= 0
yindex >= 0
xindex <= (emap_xdim-1)
yindex <= (emap_ydim-1)
    
source_not_observed = (emap[xindex,yindex] LT emap_threshold)

not_observed_ind = where(source_not_observed, num_not_observed, COMPLEMENT=observed_ind, NCOMPLEMENT=num_sources)

;For each unobserved source, remove any existing obs.stats file to communicate to other AE stages that the source is off the field.
; This situation can occur when an on-field source is extracted, and then moves off-field.
if (num_not_observed GT 0) then begin
  obs_stats_fn =  sourcename[not_observed_ind] + '/' + obsname + '/' + obs_stats_basename

  ind = where(file_test(obs_stats_fn), num_to_remove)
  
  if (num_to_remove GT 0) then begin
    print, F='(%"\n===================================================================")'
    print, num_to_remove, F='(%"WARNING: removing the following %d files because those sources are no longer on this ObsID!")'

    file_delete, obs_stats_fn[ind], /ALLOW_NONEXISTENT, /VERBOSE
    print, F='(%"===================================================================\n")'
  endif
endif

; Abort if no source was observed in this ObsID. 
if (num_sources EQ 0) then begin
  print, 'ae_make_catalog: No sources were observed in obsid ', obsname
  file_delete, /ALLOW_NONEXISTENT, [catfile,regionfile,model_savefile]

  was_ignored = 1
  GOTO, CLEANUP
endif


;; ---------------------------------------------------------------------
;; Trim down collation and full catalog to retain only the sources observed by this ObsID; write active catalog.
 bt_obs =  bt_all[observed_ind]
cat_obs = cat_all[observed_ind]

forprint, TEXTOUT=active_catfile, cat_obs.sourcename, cat_obs.ra, cat_obs.dec, cat_obs.psf_frac/100., cat_obs.psf_energ, F=fmt, /NoCOMMENT, /SILENT


;; ---------------------------------------------------------------------
; During the AE runs, the catalog fields RA and DEC are left as zero as a flag 
; telling CONSTRUCT_REGIONS to get the coordinates from the existing source.stats file 
; (to avoid rounding errors from repeated conversion between IDL and FITS).
;
; For sources we have not extracted before, start with the nominal PSF fraction and
; set the initial adjustment to be a large downward step (so we can reasonably rapidly
; find our way to the minimum fraction for sources that need it.)
frac_step = replicate(initial_down_step, num_sources)

; For sources that we have extracted before, start with a slightly higher PSF fraction than used last time,
; and set the initial adjustment to be an upward step of moderate size.  WHY???
if tag_exist(bt_obs, 'FRACSPEC') then begin
  FRACSPEC = bt_obs.FRACSPEC * 100
  ind = where(finite(FRACSPEC) AND (FRACSPEC GT 0), count)
  if (count GT 0) then begin
      cat_obs[ind].psf_frac = minimum_psf_frac > (FRACSPEC[ind] + minimum_step/2.0) < nominal_psf_frac
    frac_step[ind]          = initial_up_step
  endif
endif



;; =====================================================================
;; If desired, abort the processing if all the sources have fresh extractions.
if keyword_set(ignore_if_fresh) && tag_exist(bt_obs, 'POSNDATE') && tag_exist(bt_obs, 'EXTRDATE') then begin
  POSNDATE = strtrim(bt_obs.POSNDATE,2)
  EXTRDATE = strtrim(bt_obs.EXTRDATE,2)
  ; Any missing dates prevent us from concluding that extractions are fresh.
  if ((total(/INT, strmatch(POSNDATE,'')) + total(/INT, strmatch(EXTRDATE,''))) EQ 0) then begin
    was_ignored = 1
    for ii=0L,num_sources-1 do begin
      ; Extraction is stale if position has a later timestamp.
      if date_conv(POSNDATE[ii],'R') GT date_conv(EXTRDATE[ii],'R') then begin
        was_ignored = 0
        break
      endif
    endfor ;ii
    if was_ignored then begin
      print, 'ae_make_catalog: All sources have up-to-date extractions in obsid ', obsname
      GOTO, CLEANUP      
    endif
  endif
endif ; keyword_set(ignore_if_fresh)



;; =====================================================================
;; Iteratively adjust PSF fractions to eliminate overlapping regions.
;; The strategy for avoiding infinite iteration is as follows:
;; Apertures that are write-protected are not touched.
;; Sources in conflict get smaller.
;; Sources not in conflict get larger.
;; Each source's step size is reduced when it changes direction.
;; When the step size is smaller than minimum_step the source is not adjusted.
;; Thus, it should be the case that the set of sources in play is steadily reduced.

; The dist_to_frac_tried data structure (101xnum_sources) keeps track of the distance to
; the nearest PSF fraction that has been tried so far.
row = abs(indgen(101) - nominal_psf_frac)
dist_to_frac_tried = rebin(reform(row, n_elements(row),  1,/OVERWRITE), n_elements(row), num_sources, /SAMPLE)


; In the first iteration, we run AE on ALL writable sources, even those whose apertures are "edited by the observer".
; That gives the CONSTRUCT stage the opportunity to revise the PSF images and other objects as needed, independent of whether that stage constructs a new aperture.
sources_to_process  = where(cat_obs.is_writable, num_to_process, COMPLEMENT=sources_protected, NCOMPLEMENT=num_protected)
if (num_to_process EQ 0) then begin
  print, 'ae_make_catalog: WARNING: All sources are write-protected.'
  was_ignored = 1
  GOTO, CLEANUP      
endif else if (num_protected GT 0) then begin
  ; Report which observed source directories have been declared "write protected" by the COLLATE stage..
  print, num_protected, F='(%"\nae_make_catalog: WARNING; the following %d source directories are write-protected.\n")'
  forprint, SUBSET=(sources_protected)[0:19<(num_protected-1)], cat_obs.sourcename
endif



; Computing the OVERLAP property is expensive (a call to dmcopy) so we try to carefully cache them.
; We have to maintain a 2-D array of OVERLAP values for each possible pair of neighbors because a given source's "neighbor" can change during processing as the aperture sizes are adjusted!
overlap_cache       = replicate(!VALUES.F_NAN,num_sources,num_sources)

first_iteration_completed = 0

repeat begin
  if first_iteration_completed then begin
    ; Save state in case something crashes.
    save, /COMPRESS, FILENAME=model_savefile
;   print, 'Saved state to ', model_savefile
  endif
  
  
RESTART:  
  ;; ---------------------------------------------------------------------
  ; Write a temp catalog holding only those sources that need AE processing on this pass.
  forprint, TEXTOUT=active_catfile, cat_obs.sourcename, cat_obs.ra, cat_obs.dec, cat_obs.psf_frac/100., cat_obs.psf_energ, SUBSET=sources_to_process, F=fmt, /NoCOMMENT, /SILENT
  
  ; Invalidate cached OVERLAP values that involve the sources we're about to reprocess.
  overlap_cache[sources_to_process,*] = !VALUES.F_NAN
  overlap_cache[*,sources_to_process] = !VALUES.F_NAN
  
  

  ;; ---------------------------------------------------------------------
  print, F='(%"\nae_make_catalog: ============================================================")'  
  print,        'ae_make_catalog: Running /CONSTRUCT_REGIONS on ' + active_catfile
  print,   F='(%"ae_make_catalog: ============================================================\n")'  
  ; After the first iteration, we always /REUSE_PSF and /REUSE_NEIGHBORHOOD.

  ; The active catalog will not contain any unobserved sources, so there is no need to use the SOURCE_NOT_OBSERVED capability in the AE calls below..
  acis_extract, active_catfile, obsname, evtfile, /CONSTRUCT_REGIONS, EMAP_FILENAME=emapfile, ASPECT_FN=aspect_fn,$
    REUSE_PSF=(keyword_set(reuse_psf) || first_iteration_completed),$
    REGION_ONLY=keyword_set(region_only), MASK_FRACTION=0.98, MASK_MULTIPLIER=1.0, _STRICT_EXTRA=extra


  ; Extract Events (to get SRC_CNTS statistic).
  ; Pass /REGION_ONLY since we care about PSF fraction only at the primary energy.
  print, F='(%"\nae_make_catalog: ============================================================")'  
  print,        'ae_make_catalog: Running /EXTRACT_EVENTS on ' + active_catfile
  print,   F='(%"ae_make_catalog: ============================================================\n")'  
  acis_extract, active_catfile, obsname, evtfile, /EXTRACT_EVENTS, EMAP_FILENAME=emapfile, ASPECT_FN=aspect_fn,$
    REUSE_NEIGHBORHOOD=(keyword_set(reuse_neighborhood) || first_iteration_completed),$
    /REGION_ONLY, _STRICT_EXTRA=extra


  ;; ---------------------------------------------------------------------
  ; Collate the active sources that we've just changed above, and merge with a collation of the full
  ; catalog that we maintain in "bt_obs" (which must be the same length as the "cat_obs" structure).
  ; We need the /SINGLE_OBSID option so that SRC_CNTS comes from obs.stats rather than from source.photometry.  
  print, F='(%"\nae_make_catalog: ============================================================")'  
  print,        'ae_make_catalog: Running COLLATE stage on ' + active_catfile
  print,   F='(%"ae_make_catalog: ============================================================\n")'  
  acis_extract, active_catfile, obsname, /SINGLE_OBS, COLLATED_FILENAME=active_collatefile, MATCH_EXISTING=(first_iteration_completed ? bt_obs : 0), VERBOSE=0, _STRICT_EXTRA=extra

  bt_active = mrdfits(active_collatefile, 1, /SILENT)
  if (n_elements(bt_active) NE n_elements(sources_to_process)) then message, 'BUG in ae_make_catalog!'



  ; Since we collated (above) only an active subset of the catalog, we have to merge the result back into the full table.
  ; The problem is that the first collation will usually be lacking columns for various properties that are created by the extraction above.
  ; So, we first expand the columns of bt_obs to match the columns in bt_active.

  ; Create a structure matching the existing table but with all fields nulled.
  ;struct_assign, {foobar:0}, template_row 
  template_row = null_structure(bt_active[0])

  temp = bt_obs
  bt_obs = replicate(template_row, num_sources)
  struct_assign, temp, bt_obs
  
  ; Then, we insert the collations of the active sources into the proper rows of the full collation table, bt_obs.
  bt_obs[sources_to_process] = bt_active
 
  
  ; The collation run above reports source apertures "edited by the observer" via the tag REG_EDIT.
  ; Save that information in cat_obs.aperture_is_writable so we won't attempt to adjust those apertures (below).
  ; Warn the observer about such apertures.
  cat_obs.aperture_is_writable = tag_exist(bt_obs, 'REG_EDIT') ? ~bt_obs.REG_EDIT : 1
  
  ind = where(~cat_obs[sources_to_process].aperture_is_writable, count)
  if (count GT 0) then begin
    print, F='(%"\nae_make_catalog: ============================================================")'  
    print, count, F='(%"ae_make_catalog: WARNING: the following %d writeable source directories have write-protected apertures, presumably chosen by the observer:")'
    
    forprint, bt_obs.CATALOG_NAME, bt_obs.LABEL, $
              tag_exist(bt_obs, 'PSF_FRAC') ? round(bt_obs.PSF_FRAC * 100) : intarr(num_sources),$
              SUBSET=sources_to_process[ind], F='(%"%s (%s): %3d%% PSF fraction")'
    wait, 20
  endif



  ;; ---------------------------------------------------------------------
  ;; Note that the speedup trick of collating only the sources we've changed (above) leaves the 
  ;; "neighbor" properties in bt_obs with corrupted values.  Thus we must recompute these here.
  if array_equal(strtrim(bt_active.obsdir,2), '') then begin
    ; We arrive here in those rare cases where AE decides that ALL of the sources in bt_obs are "not observed", even though the setup section of ae_make_catalog thinks they are "observed".  When this occurs, bt_obs will be missing most of its columns and the code in the "else" block below must not execute because it would fail. 

    ; FYI, the situation when only SOME of the active sources are "not observed" is handled in code about 60 lines below.
    bt_obs.distance_src2src = replicate(!VALUES.F_NAN, num_sources)
    bt_obs.distance_reg2reg = replicate(!VALUES.F_NAN, num_sources)
    ; Adopting a "no neighbor exists" flag value (e.g. -1) would be logical, but would require trapping that exception everywhere the neighbor tag is referenced.  Instead, we declare every source to be its own neighbor.
    bt_obs.neighbor         =                  lindgen(num_sources)

  endif else begin
    ; We arrive here in the nominal case---bt_obs has its normal set of columns (because AE decided that at least one source was "observed").

    ; For speedup in the calculation of bt_obs.distance_reg2reg later, we pre-compute the 2-D array of distances between source positions.
    if (n_elements(distance_src2src) EQ 0) then begin
      make_2d, bt_obs.X_CAT, bt_obs.X_CAT, xpos_i, xpos_j
      make_2d, bt_obs.Y_CAT, bt_obs.Y_CAT, ypos_i, ypos_j
      
      distance_src2src = sqrt((xpos_i-xpos_j)^2. + (ypos_i-ypos_j)^2.)  
      xpos_i=0  & xpos_j=0  &  ypos_i=0  &  ypos_j=0
      
      src_num = lindgen(num_sources)
      distance_src2src[src_num,src_num] = 1E10
    endif
    
    
    ; For each source region, find which source has a region most overlapping.
    ; Note the the "neighbor" relationship is NOT symmetric, e.g. A's most overlapping neighbor may be B, and B's most overlapping neighbor may be C!    
    temp_src_radius       = bt_obs.SRC_RAD
    temp_distance_src2src = fltarr(num_sources)
    temp_distance_reg2reg = fltarr(num_sources)
    temp_neighbor         = lonarr(num_sources)
    for ii = 0L, num_sources-1 do begin
      temp_distance_src2src[ii] = min(/NAN, distance_src2src[*,ii])
      temp_distance_reg2reg[ii] = min(/NAN, distance_src2src[*,ii] - temp_src_radius - temp_src_radius[ii], ind)
      ; If the distance_reg2reg vector is all NaNs, then we'll declare the source to be its own neighbor.
      ; This will occur when AE found that a source was not observed and skipped its extraction.
      ; Adopting a "no neighbor exists" flag value (e.g. -1) would be more logical, but would require trapping that exception everywhere the neighbor tag is referenced.
      temp_neighbor        [ii] = finite(temp_distance_reg2reg[ii]) ? ind : ii
    endfor   
        
    bt_obs.distance_src2src = temp_distance_src2src    
    bt_obs.distance_reg2reg = temp_distance_reg2reg
    bt_obs.neighbor         = temp_neighbor
  endelse
  
  if (n_elements( bt_obs) NE num_sources) then message, 'BUG in ae_make_catalog!'
  if (n_elements(cat_obs) NE num_sources) then message, 'BUG in ae_make_catalog!'

  
  ;; ---------------------------------------------------------------------
  ; Calculate an "overlap" metric for sources that are close enough to potentially have overlapping apertures.
  ; Our metric is going to be the fraction of THIS source's aperture that overlaps with the neighbor's aperture.
  print, F='(%"\nae_make_catalog: ============================================================")'  
  print,        'ae_make_catalog: Calculating OVERLAP of crowded extraction regions.'
  print,   F='(%"ae_make_catalog: ============================================================\n")'  

  if (num_sources EQ 1) then overlap_cache[0] = 0
  for this_source = 0L, num_sources-1 do begin
    neighbor_source = bt_obs[this_source].neighbor
    
        this_obsdir = strtrim(bt_obs[    this_source].OBSDIR,2)
    neighbor_obsdir = strtrim(bt_obs[neighbor_source].OBSDIR,2)
    
    
    if ~this_obsdir then begin
      ; In rare cases, the setup section of ae_make_catalog will have decided that a source is "observed", but AE will decide that it is not observed.  The collation of that source will have lots of null (NaN) properties, and the "neighbor" algorithm above should declare the source to be its own neighbor.  We mark these sources, and remove them from the "active" catalog, by assigning them a negative overlap value.
      overlap_cache[this_source,neighbor_source] = -1
      continue
    endif
    
    ; Skip computation when overlap has already been computed.
    if finite(overlap_cache[this_source,neighbor_source]) then continue
      
    if bt_obs[this_source].distance_reg2reg GE bt_obs[this_source].SRC_RAD then begin
      ; The neighbor is too far to invest the time in looking for shared counts.
      overlap_cache[this_source,neighbor_source] = 0.
      continue
    endif
    
    ; If we get here, then no source should be its own neighbor!
    if (this_source EQ neighbor_source)  then message, 'BUG in ae_make_catalog!'
    
    this_env_events_fn =     this_obsdir + env_events_basename
    this_region_fn     =     this_obsdir + src_region_basename
    neighbor_region_fn = neighbor_obsdir + src_region_basename

    ; Find area (in arbitrary units) of this source's extraction region.
    openw,  region_unit, temp_region_fn, /GET_LUN
    printf, region_unit, "# Region file format: DS9 version 3.0"
    printf, region_unit, 'global width=1 font="helvetica 12 normal"'
    
    ae_ds9_to_ciao_regionfile, this_region_fn, '/dev/null', /IGNORE_BACKGROUND_TAG, POLYGON_X=polygon_x, POLYGON_Y=polygon_y
    polygon = fltarr(2,n_elements(polygon_x))
    polygon[0,*] = polygon_x
    polygon[1,*] = polygon_y

    printf,   region_unit, 'polygon(' + strcompress(strjoin(string(polygon,F='(F8.2)'),","), /REMOVE) + ')', F='($,A0)'
    flush,    region_unit
    
    cmd = string(this_env_events_fn, temp_region_fn, temp_pi_fn, F="(%'dmextract ""%s[sky=region(%s)][bin pi]"" %s clobber=yes')")
    run_command, cmd
 
    this_region_area = psb_xpar( headfits(temp_pi_fn, EXT=1), 'BACKSCAL')
    
    ; Find area (in arbitrary units) of the intersection of this and the neighbor's extraction regions.
    ; CIAO uses the syntax "&" to represent the intersection of two regions.
    ae_ds9_to_ciao_regionfile, neighbor_region_fn, '/dev/null', /IGNORE_BACKGROUND_TAG, POLYGON_X=polygon_x, POLYGON_Y=polygon_y
    polygon = fltarr(2,n_elements(polygon_x))
    polygon[0,*] = polygon_x
    polygon[1,*] = polygon_y

    printf,   region_unit, '&polygon(' + strcompress(strjoin(string(polygon,F='(F8.2)'),","), /REMOVE) + ')'
    free_lun, region_unit
     
    cmd = string(this_env_events_fn, temp_region_fn, temp_pi_fn, F="(%'dmextract ""%s[sky=region(%s)][bin pi]"" %s clobber=yes')")
    run_command, cmd
 
    intersection_area = psb_xpar( headfits(temp_pi_fn, EXT=1), 'BACKSCAL')
    
    ; CIAO 4.0 seems to have a bug.  If the intersection of the regions is null, it returns BACKSCAL=1.0 (instead of 0).
    if (intersection_area EQ 1.0) then intersection_area = 0.0
        
    overlap_cache[this_source,neighbor_source]  = intersection_area / this_region_area
  endfor

  cat_obs.overlap = overlap_cache[lindgen(num_sources),bt_obs.neighbor]
  
  
  ; Estimate the number of counts not extracted for each source.
  counts_lost = tag_exist(bt_obs, 'SRC_CNTS') ? (bt_obs.SRC_CNTS/bt_obs.FRACSPEC) - bt_obs.SRC_CNTS $
                                              : replicate(0,num_sources)
  


  ;; ---------------------------------------------------------------------
  ; The relationship of overlapping regions can be complex.  I think it's essential to process the 
  ; sources sequentially rather than via vector operations.
  ; Note that the "neighbor" relationship is NOT symmetric, e.g. A's most overlapping neighbor may be B, and B's most overlapping neighbor may be C!    

  ; The code below seeks to choose the "sign" of each source's next PSF fraction adjustment:
  ; step_sign =  0: Frac is not changing; source will be omitted from AE run.
  ; step_sign =  1: Frac is scheduled for increase.
  ; step_sign = -1: Frac is scheduled for reduction.
  step_sign = replicate(0,num_sources)
  
  for ii=0L, num_sources-1 do begin
    if (~finite(cat_obs[ii].overlap)) then message, 'BUG in ae_make_catalog!'
    
    if (cat_obs[ii].overlap GT 0.) then begin
      ; This source region (ii) is sharing counts with that of the neighbor jj.
      jj = bt_obs[ii].neighbor
      
      ; If either is already scheduled for reduction then this overlap is being addressed and there's no more work to be done.
      if ((step_sign[ii] EQ -1) OR (step_sign[jj] EQ -1)) then continue

      ; Since the neighbor and thus overlap properties of sources are NOT symmetric, it is possible that one of these sources (ii or jj) is already scheduled for an INCREASE.  We ignore such a proposal and allow it to be scheduled for decrease below.
      
      ; We need to try to schedule one of them to be reduced.
      ; Even if our step size is at the minimum, we will shrink a region to avoid overlap.
      ii_can_reduce = cat_obs[ii].aperture_is_writable && (cat_obs[ii].psf_frac GT minimum_psf_frac)
      jj_can_reduce = cat_obs[jj].aperture_is_writable && (cat_obs[jj].psf_frac GT minimum_psf_frac)
      
      if (ii_can_reduce AND jj_can_reduce) then begin
        ; Use counts_lost to choose which should reduce.
        if            (counts_lost[ii] GT counts_lost[jj]) then begin
          step_sign[ii] =  0 ; stay put
          step_sign[jj] = -1 ; reduce
        endif else if (counts_lost[jj] GT counts_lost[ii]) then begin
          step_sign[jj] =  0 ; stay put
          step_sign[ii] = -1 ; reduce
        endif else begin
          ; When counts_lost estimates are equal (e.g. both zero) then reduce the larger PSF fraction.
          if (cat_obs[ii].psf_frac LT cat_obs[jj].psf_frac) then begin
            step_sign[ii] =  0 ; stay put
            step_sign[jj] = -1 ; reduce
          endif else begin
            step_sign[jj] =  0 ; stay put
            step_sign[ii] = -1 ; reduce
          endelse
        endelse
      endif else if ((NOT ii_can_reduce) AND (NOT jj_can_reduce)) then begin
        ; Overlap cannot be fixed; both stay put.
        step_sign[ii] = 0 ; stay put
        step_sign[jj] = 0 ; stay put
      endif else if (     ii_can_reduce  AND (NOT jj_can_reduce)) then begin
        step_sign[jj] =  0 ; stay put
        step_sign[ii] = -1 ; reduce
      endif else if ((NOT ii_can_reduce) AND      jj_can_reduce ) then begin
        step_sign[ii] =  0 ; stay put
        step_sign[jj] = -1 ; reduce
      endif else message, 'Bug in logic.'
    endif else if (cat_obs[ii].overlap EQ 0.) then begin
      ; NOT overlapping with anything, so consider increasing.
      ; Since the neighbor and thus overlap properties of sources are NOT symmetric, it is possible that one of these sources (ii or jj) is already scheduled for a DECREASE.  We respect such a proposal and do NOT schedule it for an increase here.
      
      ; If we're at the minimum step size then stop trying to increase.  This criterion is what terminates the adjustment of a source.

      if        cat_obs[ii].aperture_is_writable          && $
         (    step_sign[ii]          EQ 0               ) && $
         (      cat_obs[ii].psf_frac LT nominal_psf_frac) && $
         (abs(frac_step[ii])         GT minimum_step    )    then step_sign[ii] = 1
    endif else begin
      ; cat_obs[ii].overlap is negative, which is a flag that the source was not observed.  We want to stop processing it (by setting step_sign to zero).
      step_sign[ii]                      = 0
        cat_obs[ii].aperture_is_writable = 0
      print, strtrim(bt_obs[ii].LABEL,2), F='(%"ae_make_catalog thought that %s was observed, but AE decided that it was not.")'
    endelse
  endfor ;ii
  
  
  ; The direction of the adjustment desired in this iteration is carried in step_sign.
  ; The actual PSF fraction step to take is carried in frac_step, which is a signed quantity.
  ; When the direction of adjustment reverses we reduce the step size.  
  reversing_ind = where( frac_step*step_sign LT 0, count )
  if (count GT 0) then frac_step[reversing_ind] = (-frac_step[reversing_ind]/step_ratio)
  
  ; Adjust the PSF fractions in the catalog.
  sources_to_process = where(step_sign NE 0, num_to_process)

  if (num_to_process EQ 0) then break
  previous_cat_psf_frac = cat_obs.psf_frac
  
  ; For each source we're adjusting, the interval between cat_obs.psf_frac and cat_obs.psf_frac + frac_step 
  ; defines a range of new PSF fractions that are reasonable to take.
  ; For efficiency, we add the further criterion that we will choose the value in that range that
  ; is farthest from any PSF fraction we have already tried for this source.
  for ii=0L, num_to_process-1 do begin
    ind = sources_to_process[ii]
    ; Define the range of PSF fractions (expressed here as integer percentages) that are under consideration.
    if (frac_step[ind] GT 0) then begin
      ; Interval extends to the right; clip at nominal_psf_frac.
      min_frac = nominal_psf_frac < (cat_obs[ind].psf_frac + 1             )
      max_frac = nominal_psf_frac < (cat_obs[ind].psf_frac + frac_step[ind])
    endif else begin
      ; Interval extends to the left; clip at minimum_psf_frac.
      max_frac =     minimum_psf_frac > (cat_obs[ind].psf_frac - 1             )
      min_frac =     minimum_psf_frac > (cat_obs[ind].psf_frac + frac_step[ind])    
    endelse                     
    
    ; Select the best candidate in that range.
    dum = max(dist_to_frac_tried[min_frac:max_frac, ind], imax)
    cat_obs[ind].psf_frac = min_frac + imax
    
    ; Update the dist_to_frac_tried data structure.
    dist_to_frac_tried[0,ind] = dist_to_frac_tried[*,ind] < abs(indgen(101) - cat_obs[ind].psf_frac) 
  endfor ;ii                                                                  
  
  ; Defensively range check the catalog and loop to call AE.
  cat_obs.psf_frac = minimum_psf_frac > cat_obs.psf_frac < nominal_psf_frac
  
  print, F='(%"\nae_make_catalog: ============================================================")'  
  print, num_to_process, F='(%"ae_make_catalog: %d sources are being reprocessed ...")'
  forprint, cat_obs.sourcename, bt_obs.label, (cat_obs.psf_frac-previous_cat_psf_frac)/100., cat_obs.psf_frac/100., SUBSET=sources_to_process, F='(%"%s (%s): frac stepped by %5.2f to %4.2f")'

  first_iteration_completed = 1
endrep until (0)


;; =====================================================================
print, F='(%"\nae_make_catalog: ============================================================")'  
print,        'ae_make_catalog: Saving OVERLAP values for each extraction region'
print,   F='(%"ae_make_catalog: ============================================================\n")'  
ind = where(cat_obs.is_writable)


ae_poke_source_property, obsname, SOURCENAME=cat_obs[ind].sourcename, KEYWORD='OVERLAP', VALUE=cat_obs[ind].overlap, COMMENT='fraction of aperture extracted by neighbor', _EXTRA=extra


; Report pairs of sources that significantly overlap.
report_flag = cat_obs.overlap GT 0.1
for ii=0L,num_sources-1 do $
  if report_flag[ii] then report_flag[bt_obs[ii].neighbor] = 0

ind = where(report_flag, num_crowded)
if (num_crowded GT 0) then begin
  neighbor_label = (bt_obs.LABEL)       [bt_obs.neighbor]
  neighbor_name  = (bt_obs.CATALOG_NAME)[bt_obs.neighbor]
  ind = ind[reverse(sort((cat_obs.overlap)[ind]))]
  print, num_crowded, F='(%"\nae_make_catalog: WARNING!  these %d pairs of sources remain severely crowded (OVERLAP > 0.1).")'
  forprint, bt_obs.LABEL, bt_obs.CATALOG_NAME, neighbor_name, neighbor_label, cat_obs.overlap, SUBSET=ind, F='(%"%12s  %s <> %s  %12s  %7.1f")' 
endif


;; =====================================================================
; Write out the catalog for all sources in this ObsID, with RA & DEC filled in so it serves as an archive.
cat_obs.RA  = bt_obs.RA
cat_obs.DEC = bt_obs.DEC
forprint, TEXTOUT=catfile, cat_obs.sourcename, cat_obs.ra, cat_obs.dec, cat_obs.psf_frac/100., cat_obs.psf_energ, F=fmt, /NoCOMMENT, /SILENT
print, 'ae_make_catalog: Wrote catalog ', catfile


;; =====================================================================
;; Generate a region file for all extractions in this ObsID.
print, F='(%"\nae_make_catalog: ============================================================")'  
print,        'ae_make_catalog: Building regionfile ' + regionfile
print,   F='(%"ae_make_catalog: ============================================================\n")'  
ind_excessive = where(/NULL, cat_obs.overlap GT 0.1, COMPLEMENT=ind_acceptable)
if isa(ind_acceptable, /INTEGER) then begin
  acis_extract, cat_obs[ind_acceptable].sourcename, obsname, /SINGLE_OBS, COLLATED_FILENAME='/dev/null', MATCH_EXISTING=bt_obs, REGION_TAG='acceptable overlap', REGION_FILE=regionfile, VERBOSE=0, _STRICT_EXTRA=extra
endif

if isa(ind_excessive, /INTEGER) then begin
  acis_extract, cat_obs[ind_excessive].sourcename, obsname, /SINGLE_OBS, COLLATED_FILENAME='/dev/null', MATCH_EXISTING=bt_obs, REGION_TAG='excessive overlap', REGION_FILE=temp_region_fn, VERBOSE=0, _STRICT_EXTRA=extra

  cmd = string(temp_region_fn, regionfile, F='(%"egrep ''point|polygon'' %s | sed -e ''s/DodgerBlue/red/'' >> %s")')
  run_command, /QUIET, cmd
endif

;acis_extract, catfile, obsname, /SINGLE_OBS, COLLATED_FILENAME='/dev/null', MATCH_EXISTING=bt_obs, REGION_FILE=regionfile, VERBOSE=0, _STRICT_EXTRA=extra


; Show the regions in ds9.
cmd = string('ae_make_catalog ('+obsname+')', evtfile, regionfile, F='(%"ds9 -title \"%s\" -log \"%s[STATUS=0]\" -region %s -frame center >& /dev/null &")')
if show then run_command, /QUIET, cmd

if keyword_set(tempdir) && file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif


CLEANUP:
if keyword_set(tempdir) && file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then begin
  print, F='(%"\nae_make_catalog finished")'  
  return 
endif else begin
  print, 'ae_make_catalog: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end ; ae_make_catalog



;#############################################################################
;;; Perform a standard extraction on a single obsid.
;;;
;;; Several input files are located using standard naming conventions as shown 
;;; in the Getting Started section of the AE manual.
;;;
;;; The extraction regions must have been constructed previously, e.g. with ae_make_catalog.
;;;
;;; The required parameter EVTFILE_BASENAME specifies the name of the observation event list file in
;;; the directory ../obsXXXX/.  In our recipe we use EVTFILE_BASENAME='spectral.evt' (lightly cleaned) for all passes.
;;; AE stages apply a STATUS=0 filter (recorded in the src property S_FILTER) for sources that have event rates low enough to avoid removing valid events.

;;;
;;; If /BETTER_BACKGROUNDS is specified, then the ae_better_backgrounds tool is used to build bkg spectra.

;;; If BETTER_BACKGROUNDS is omitted, then a masked background emap and a masked event list are constructed by applying 
;;; the mask region file ../obsXXXX/mask.reg, typically created by ae_make_catalog and containing circular masks from the 
;;; /CONSTRUCT_REGIONS stage, plus any mask regions supplied via EXTRA_MASKFILE.
;;; The AE EXTRACT_BACKGROUNDS stage is then used to build bkg spectra for each source.

PRO ae_standard_extraction, obsname, EVTFILE_BASENAME=evtfile_basename, BETTER_BACKGROUNDS=better_backgrounds, $
    SRCLIST_FILENAME=srclist_fn, $
    SOURCE_NOT_OBSERVED=source_not_observed, $
    EXTRA_MASKFILE=extra_maskfile, REUSE_MASKING=reuse_masking, $
    EXTRACT_SPECTRA=extract_spectra, TIMING=timing, EXTRACT_BACKGROUNDS=extract_backgrounds,  $
    _EXTRA=extra

exit_code = 0
creator_string = "ae_standard_extraction, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

if (n_elements(extract_spectra) EQ 0)     then extract_spectra     = 1
if (n_elements(timing) EQ 0)              then timing              = 1
if (n_elements(extract_backgrounds) EQ 0) then extract_backgrounds = 1

if ~keyword_set(evtfile_basename) then begin
  print, 'ERROR: parameter EVTFILE_BASENAME must be supplied'
  GOTO, FAILURE
endif

if ~keyword_set(source_not_observed) then source_not_observed = 0

; I can't recall all the reasons that the default sourcelist (below) is all.srclist instead of obs.cat.
; Note that our recipe uses the target-length vector SOURCE_NOT_OBSERVED to efficiently skip over source not observed in this ObsID.
if ~keyword_set(srclist_fn)      then srclist_fn = 'all.srclist'
if keyword_set(extract_backgrounds) && (srclist_fn NE 'all.srclist') && ~keyword_set(reuse_masking) then $
  print, 'WARNING: Background estimation may be unreliable when only a subset of full catalog is processed.'

if ~keyword_set(obsname) then begin
  print, 'ERROR: parameter "obsname" must be supplied'
  GOTO, FAILURE
endif

if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
  print, 'ERROR: parameter "obsname" must be a scalar string'
  GOTO, FAILURE
endif


obsdir          = '../obs' + obsname + '/'
if ~file_test(obsdir) then begin
  print, 'ae_standard_extraction: ERROR: could not find observation directory ', obsdir
  GOTO, FAILURE
endif

lock_fn     = obsdir + 'ae_lock'
file_delete, /ALLOW_NONEXISTENT, lock_fn
file_copy  , '/dev/null'       , lock_fn, /OVERWRITE

evtfile         = obsdir + evtfile_basename
bkg_evtfile     = obsdir + 'background.evt'
emapfile        = obsdir + 'obs.emap'
bkg_emapfile    = obsdir + 'background.emap'

ardlib_filename = obsdir + 'ardlib.par'
pbk_filename    = obsdir + 'obs.pbkfile'
msk_filename    = obsdir + 'obs.mskfile'
aspect_fn       = obsdir + 'obs.asol'
maskfile        = obsdir + 'mask.reg'
collatefile     = obsdir + 'all.collated'

if NOT file_test(pbk_filename) then pbk_filename = 'NONE'


;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)

temp_events1_fn  = tempdir + 'temp1.evt'
temp_events2_fn  = tempdir + 'temp2.evt'
temp_emap_fn     = tempdir + 'temp.emap'

run_command, PARAM_DIR=tempdir


if keyword_set(extract_backgrounds) && ~keyword_set(better_backgrounds) && ~keyword_set(reuse_masking) then begin
  ;; =====================================================================
  ;; Make masked background event list & emap using the masks.
  print, F='(%"\nae_standard_extraction: ============================================================")' 
  print, 'ae_standard_extraction: Constructing masked background emap and event list'
  print, F='(%"ae_standard_extraction: ============================================================")'
  
  
  ; Build region file containing all the masks.
  if keyword_set(extra_maskfile) && ~file_test(extra_maskfile) then begin
    print, extra_maskfile, F='(%"\nae_standard_extraction: ERROR: could not find EXTRA_MASKFILE %s\n")' 
    GOTO, FAILURE
  endif
  
  if keyword_set(extra_maskfile) && file_test(extra_maskfile) && (file_lines(extra_maskfile) GT 0) then begin
    print, 'Spawning ds9 to perform coordinate conversions on EXTRA_MASKFILE ...'
    ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='acis_extract_'+session_name, OPTION_STRING='-iconify'
    ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, emapfile
  
    ;; Load region file into ds9 and resave in PHYSICAL coordinates.
    cmd = strarr(4)
    cmd[0] = string(my_ds9,                 F='(%"xpaset -p ''%s'' regions delete all")')
    cmd[1] = string(my_ds9, extra_maskfile, F='(%"xpaset -p ''%s'' regions load %s")')
    cmd[2] = string(my_ds9,                 F='(%"xpaset -p ''%s'' regions system physical")')
    cmd[3] = string(my_ds9, maskfile,       F='(%"xpaset -p ''%s'' regions save %s")')
    run_command, cmd, /QUIET
    run_command, string(my_ds9, F='(%"xpaset -p %s exit")'), /QUIET
  endif else begin
    cmd = string(maskfile,       F="(%'echo ""# Region file format: DS9 version 3.0"" >! %s')")
  endelse
  run_command, cmd, /UNIX
    
  cmd = string(obsname,maskfile, F="(%'grep -h background */%s/extract.reg           >>! %s')")
  run_command, cmd, /UNIX
  
  ;; CIAO (Jan 2008) has a bug when update=no is used with exclude which results in cropping of the output image.
  ;; As a workaround we add small regions at the corners of the emap.
  emap      = readfits(emapfile, emap_header)
  extast, emap_header, emap2wcs_astr
  emap_col_dim = (size(emap, /DIM))[0]
  emap_row_dim = (size(emap, /DIM))[1]
  crvalP = [psb_xpar( emap_header, 'CRVAL1P'), psb_xpar( emap_header, 'CRVAL2P')]
  crpixP = [psb_xpar( emap_header, 'CRPIX1P'), psb_xpar( emap_header, 'CRPIX2P')]
  cdeltP = [psb_xpar( emap_header, 'CDELT1P'), psb_xpar( emap_header, 'CDELT2P')]
  
  x_sky = crvalP[0] + cdeltP[0] * (([0,emap_col_dim-1]+1) - crpixP[0])
  y_sky = crvalP[1] + cdeltP[1] * (([0,emap_row_dim-1]+1) - crpixP[1])

  openw, unit, maskfile, /GET_LUN, /APPEND
  !TEXTUNIT = unit
  forprint, TEXTOUT=5, x_sky, y_sky, F='(%"circle(%f,%f,1) # tag={bug workaround}")', /NoCOMMENT
  free_lun, unit
  
  
  ; Apply masks to emap.
  print
  print, 'The following dmcopy can run a while on large catalogs.'
 
  cmd = string(emapfile, maskfile, temp_emap_fn, F="(%'dmcopy ""%s[exclude sky=region(%s)][opt full,update=no]"" %s clobber=yes')")
  run_command, cmd
  file_copy, /OVERWRITE, temp_emap_fn, bkg_emapfile
    
  ; Discard events where emap is zero.
  ; The image passed to dmimgpick below should span the event list, so the bug in HelpDesk Ticket #020605 should not be triggered.
  cmd = string(evtfile, temp_emap_fn, temp_events1_fn, F="(%'dmimgpick ""%s[cols time,ccd_id,chip,det,sky,pi,energy,status]"" %s %s method=closest clobber=yes')")
  run_command, cmd
  
  cmd = string(temp_events1_fn, temp_events2_fn, F="(%'dmcopy ""%s[#9>1][cols time,ccd_id,chip,det,sky,pi,energy,status]"" %s clobber=yes')")
  run_command, cmd
  file_copy, /OVERWRITE, temp_events2_fn, bkg_evtfile 
    
  cmd = string('ae_standard_extraction ('+obsname+')', bkg_evtfile, maskfile, bkg_emapfile, maskfile, F='(%"ds9 -title \"%s\" -tile -lock frame wcs -log %s -region %s -linear %s -region %s -zoom to fit  >& /dev/null &")')
  run_command, cmd
endif ; constructing background emap and event list



if keyword_set(extract_spectra) then begin
  ;; =====================================================================
  ;; Extract Spectra.
  print, F='(%"\nae_standard_extraction: ============================================================")' 
  print, 'ae_standard_extraction: Running /EXTRACT_SPECTRA stage'
  print, F='(%"ae_standard_extraction: ============================================================\n")'
  acis_extract, srclist_fn, obsname, evtfile, /EXTRACT_SPECTRA, EMAP_FILENAME=emapfile, ASPECT_FN=aspect_fn, ASPHIST_DIR=obsdir+'asphist', ARDLIB_FILENAME=ardlib_filename, PBKFILE=pbk_filename, MSKFILE=msk_filename, SOURCE_NOT_OBSERVED=source_not_observed, _EXTRA=extra
endif

if keyword_set(timing) then begin
  ;; =====================================================================
  ;; Timing Analysis.
  print, F='(%"\nae_standard_extraction: ============================================================")'
  print, 'ae_standard_extraction: Running /TIMING stage'
  print, F='(%"ae_standard_extraction: ============================================================\n")'
  acis_extract, srclist_fn, obsname, /TIMING, SOURCE_NOT_OBSERVED=source_not_observed, _EXTRA=extra
endif
  
if keyword_set(extract_backgrounds) then begin
  ;; =====================================================================
  ;; Extract background for each source.
  if keyword_set(better_backgrounds) then begin
    print, F='(%"\nae_standard_extraction: ============================================================")' 
    print, 'ae_standard_extraction: Running ae_better_backgrounds tool'
    print, F='(%"ae_standard_extraction: ============================================================\n")'
    ae_better_backgrounds, obsname, EVTFILE_BASENAME=evtfile_basename, SRCLIST_FILENAME=srclist_fn,  SOURCE_NOT_OBSERVED=source_not_observed, _EXTRA=extra
  endif else begin
    print, F='(%"\nae_standard_extraction: ============================================================")' 
    print, 'ae_standard_extraction: Running /EXTRACT_BACKGROUNDS stage'
    print, F='(%"ae_standard_extraction: ============================================================\n")'
    acis_extract, srclist_fn, obsname, bkg_evtfile, /EXTRACT_BACKGROUNDS, EMAP_FILENAME=bkg_emapfile, SOURCE_NOT_OBSERVED=source_not_observed, _EXTRA=extra
  endelse
endif

;; =====================================================================
;; Collate results.
; Commented out to reduce run times.
;print, F='(%"\nae_standard_extraction: ============================================================")' 
;print, 'ae_standard_extraction: Running COLLATE stage'
;print, F='(%"ae_standard_extraction: ============================================================\n")'
;acis_extract, srclist_fn, obsname, /SINGLE_OBS, COLLATED_FILENAME=collatefile, _EXTRA=extra, VERBOSE=0

CLEANUP:
if keyword_set(tempdir) && file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then begin
  print, F='(%"\nae_standard_extraction finished")'  
  return 
endif else begin
  print, 'ae_standard_extraction: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP

end ; ae_standard_extraction






;#############################################################################
;;; Screen single-ObsID extractions for pile-up.
PRO ae_pileup_screening, obsname, NUM_POSSIBLY_PILED=num_possibly_piled, RADIAL_PROFILE_REPORT=radial_profile_report

creator_string = "ae_pileup_screening, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

    par = ae_get_target_parameters()

    num_possibly_piled = 0
    
    obsdir                    = '../obs' + obsname + '/'
    catfile                   = obsdir + 'obs.cat'
    collatefile               = obsdir + 'obs.collated'
    possibly_piled_srclist_fn = obsdir+'possibly_piled.srclist'
    
    file_delete, [collatefile, possibly_piled_srclist_fn], /ALLOW_NONEXISTENT
    file_copy, '/dev/null', possibly_piled_srclist_fn
  
    if ~file_test(catfile) then begin
      print, obsname, F='(%"\nae_pileup_screening: WARNING: no sources were observed by ObsID %s.")' 
      return
    endif
    ; Collate the single-ObsID extraction to recover the most recent estimate for RATE_3x3.
    ; Omit the MATCH_EXISTING speedup option because an existing obs.collated file written by another tool will not have the RATE_3x3 column.
    acis_extract, catfile, obsname, /SINGLE_OBS, COLLATED_FILENAME=collatefile
    
    bt=mrdfits(collatefile, 1, /SILENT)    

    if ~tag_exist(bt, 'RATE_3x3') then begin
      print, collatefile, F='(%"\nae_pileup_screening: WARNING: the collation %s was found, but it does not contain a RATE_3x3 column.")' 
      return
    endif

    rate_in_cell = bt.RATE_3x3
    
    ; Identify extractions with a high estimated event rate within the central 3x3 pixel cell (RATE_3x3).
    ind = where(rate_in_cell GT 0.05, num_possibly_piled)
    if (num_possibly_piled EQ 0) then begin
      print, obsname, F='(%"\n\nObsID: %s:")'      
      print, 'ae_pileup_screening: No extractions are at risk of significant pile-up.'
      return
    endif
    
    print, F='(%"\nae_pileup_screening: ============================================================")' 
    print, 'ae_pileup_screening: Running single-ObsID merges on piled sources.'
    ; Sort those by RATE_3x3.
    ind = ind[ reverse(sort(rate_in_cell[ind])) ] 
    
    ; Perform complete single-ObsID merges ('EPOCH_XXXX') to build data products needed for pile-up reconstruction.
    ; These merges must perform photometry in the same set of energy ranges used for source validation, so that model.photometry files produced by pile-up reconstruction will match source.photometry files produced by AE.
    forprint, TEXTOUT=possibly_piled_srclist_fn, SUBSET=ind, bt.CATALOG_NAME, bt.LABEL, rate_in_cell, F="(%'%s ; (%s) %6.3f (ct/frame)')", /NoComm

    ; MUST match energy ranges in ae_better_backgrounds.
    eLO   = [0.5, 0.5, 2.0, 4.0]
    eHI   = [7.0, 2.0, 7.0, 7.0]
    eRNG  = [[eLO],[eHI]]

    acis_extract, possibly_piled_srclist_fn, obsname, MERGE_NAME='EPOCH_'+obsname, /MERGE_OBSERVATIONS, SKIP_TIMING=par.validation_mode, ENERGY_RANGE=eRNG[0,*], EBAND_LO=eLO, EBAND_HI=eHI, GENERIC_RMF_FN='generic.rmf'
    
    if arg_present(radial_profile_report) then $
      ae_radial_profile, radial_profile_report, SRCLIST=possibly_piled_srclist_fn, MERGE_NAME='EPOCH_'+obsname

    ; Report the properties of those high-rate extractions in a table.
    print, F='(%"\n\n")'      
    print, F='(%"\nae_pileup_screening: ============================================================")' 
    print,'                                                                                   AE estimate of count rate'
    print,'                                                                                   in central 3x3 pixel cell'
    print, obsname, F='(%"ObsID: %s:")'      
    
    forprint, SUBSET=ind, bt.CATALOG_NAME, bt.LABEL, replicate(obsname,n_elements(bt)), bt.THETA, bt.PSF_FRAC, bt.SRC_CNTS[0], rate_in_cell, bt.S_FILTER, F='(%"%s (%s) in ObsID %8s (THETA=%0.1f, PSF_FRAC=%0.2f): %5d (ct), %6.3f (ct/frame)  %s")'
    print
    
    ; Write to the file screenrc.XXXX.txt the "screen" commands required to set up a shell for each piled extraction.
    cd, CURRENT=cwd
    SCREEN_ARCH = getenv('SCREEN_ARCH')
    openw, unit, 'screenrc.'+obsname+'.csh', /GET_LUN
    printf, unit, obsname, F='(%"\n screen -S piled_${TARGET} -X setenv OBS     %s")'
    for ii=0,num_possibly_piled-1 do begin
      sourcename = (bt.CATALOG_NAME)[ind[ii]]
      printf, unit, cwd, sourcename, obsname, F='(%" screen -S piled_${TARGET} -X chdir        %s/%s/EPOCH_%s      ")'
      printf, unit,      sourcename, obsname, F='(%" screen -S piled_${TARGET} -X setenv PROMPT \"%s/EPOCH_%s %% \"")'
      printf, unit,      sourcename, obsname, SCREEN_ARCH, F='(%" screen -S piled_${TARGET} -X screen -t       %s/EPOCH_%s %s")'
      ; Example of how to send a unix command to the newly-created shell.  The \015 is the magic to get a newline sent to shell.
      ; screen -S piled_${TARGET} -X stuff  `printf '%b' 'date\015'`

      printf, unit, 'sleep 1.5'  ; Pause between shell launches to avoid collisions executing .cshrc.
    endfor ; ii
    free_lun, unit
return
end ; pileup_screening




;==========================================================================
;;; Tool to build  models (FITS images) of the readout streaks generated (in a specific ObsID, of course) by bright sources.
;;;
;;; These images have a normalization header keyword (like PSF images)---ABB will later scale these to model the counts expected from the streak (as it does with PSFs).
;;;

;==========================================================================
PRO ae_streak_model, sourcename, obsname, EXTRACTION_NAME=extraction_name, TEMPDIR=tempdir


exit_code = 0
creator_string = "ae_streak_model, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
  print, 'ERROR: parameter "obsname" must be a scalar string'
  GOTO, FAILURE
endif


tempdir_supplied_by_caller = keyword_set(tempdir)

;; If not passed, create a unique scratch directory.
;; Initialize AE command spawning system.
if ~tempdir_supplied_by_caller then begin
  temproot = temporary_directory( 'ae_streak_model.', VERBOSE=1, SESSION_NAME=session_name)
  tempdir  = temproot
endif
run_command, PARAM_DIR=tempdir


emap_basename            = 'obs.emap'

src_stats_basename       = 'source.stats'
obs_stats_basename       = 'obs.stats'
psf_basename             = 'source.psf'
streak_basename          = 'source.streak'
env_events_basename      = 'neighborhood.evt'

; ObsID-level files we are using.
obsdir                      = '../obs' + obsname + '/'
aspect_fn                   = obsdir + 'obs.asol'
emapfile                    = obsdir + emap_basename


;; =====================================================================

;; Get rid of pre-existing configuration for the CIAO commands we'll use below.
run_command, /QUIET, ['punlearn get_sky_limits']


;; For simplicity (if not speed) we choose to build the streak models on the same image grid used by the emap.
emap = readfits(emapfile, emap_header)
run_command, string(emapfile, F="(%'get_sky_limits %s verbose=0 precision=3')")
run_command, /QUIET, 'pget get_sky_limits dmfilter', binspec

if (binspec EQ '') then message, 'ERROR running get_sky_limits'



; Trim whitespace from source names.
sourcename = strtrim(sourcename,2)
num_sources= n_elements(sourcename)

if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)

for ii = 0L, num_sources-1 do begin
  print, sourcename[ii], F='(%"\nBuilding streak model for %s  ...")'
  
  basedir      = sourcename[ii] + '/' 
  src_stats_fn = basedir + src_stats_basename
  streak_fn    = basedir + obsname + '/' + streak_basename
  psf_fn       = basedir + obsname + '/' + psf_basename

  obsdir        = basedir + obsname + '/' + extraction_subdir[ii]
  stats_fn      = obsdir + obs_stats_basename
  env_events_fn = obsdir + env_events_basename
  
  src_stats    = headfits(src_stats_fn, ERRMSG=error)
  obs_stats    = headfits(stats_fn, ERRMSG=error)
  psf_header   = headfits(psf_fn)

  ;; =====================================================================
  ;; Set up a simulation of the streak that corresponds to the existing PSF simulation.
  ra  = psb_xpar( src_stats,   'RA')
  dec = psb_xpar( src_stats,   'DEC')
  
  ; Use the observation's aspect file.
  ; The ACIS_Exposure_Time=0 option is used to set the ACIS frame time to zero, which simulates just the ACIS readout streak.

  ae_make_psf, TEMP_DIR=tempdir, EVENT_FILE=env_events_fn, OBS_ASPECT_FN=aspect_fn, $
    ASPECT_BLUR=psb_xpar( psf_header, 'ASP_BLUR'), PIX_ADJ=strtrim(psb_xpar( psf_header, 'PIX_ADJ'),2), $
    ACIS_Exposure_Time=0, /SIM_STREAK 

  
  ;; Build a streak image
  ;; Match pixel size and image dimensions of emap. 
  desired_streak_counts = 1024. * 1000  ; 1000 counts per CCD row
  footprint             = 0 ; not used when BINSPEC passed
  skypixel_per_psfpixel = 0 ; not used when BINSPEC passed
  
  ae_make_psf, TEMP_DIR=tempdir, ra, dec, streak_fn, skypixel_per_psfpixel, footprint, psb_xpar( psf_header,'ENERGY'),$
               desired_streak_counts, X_CAT=psb_xpar( obs_stats, 'X_CAT'), Y_CAT=psb_xpar( obs_stats, 'Y_CAT'), OFF_ANGLE=psb_xpar( obs_stats,'THETA'), CHIP_ID=psb_xpar( obs_stats,'CCD_PRIM'), EMAP_VAL=psb_xpar( obs_stats,'EMAP_MED'), BINSPEC=binspec
             
  print, streak_fn, F='(%"\nWrote streak model to %s")'
endfor ; ii

CLEANUP:
if ~tempdir_supplied_by_caller && file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif

if (exit_code EQ 0) then begin
  return 
endif else begin
  print, 'ae_streak_model: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end ; ae_streak_model







;#############################################################################
;;; Apply "better" source masking to construct a background event list and emap.
;;; Then (optionally) re-extract the background for every source.

;;; This tool tries to mask pixels on the sky where the estimated flux from point sources is significant compared to the 
;;; estimated non-point source flux ( point_source_flux / diffuse_flux  >  THRESHOLD ).
;;; The goal is to get bright stars more heavily masked than weak ones.

;;; This tool uses the first part of the ae_better_backgrounds tool to build models of the point sources for each ObsId 
;;; (in the files obsXXXX/ae_better_backgrounds.sav).
;;; If you already have those models from running ae_better_backgrounds then supply /REUSE_MODELS to re-use them.

;;; To build those models, ae_better_backgrounds must estimate photometry for each source.
;;; Thus, your entire catalog must already have been extracted, including backgrounds for each source.  
;;; The ae_better_backgrounds tool will then run the MERGE_OBSERVATIONS stage on a SINGLE obsid to estimate fluxes for each source.
;;; Note that a source might have no net counts in any single observation.

;;; With those point source models in hand, we have a model of the point-source flux in every pixel on the sky.
;;; We also need a way to estimate the background flux in every pixel (to decide if that pixel needs to be masked).
;;; We use code from our adaptive smoothing tool for this.

;;; Once a sky mask is defined, it is applied to the event list file in the directory ../obsXXXX/ specified by the 
;;; required parameter EVTFILE_BASENAME to produce obsXXXX/background.evt.

PRO ae_better_masking, obsname, EVTFILE_BASENAME=evtfile_basename, THRESHOLD=threshold, $

    VERBOSE=verbose, PLOT=plot, $

    BACKGROUND_MODEL_FILENAME=background_model_filename, $

    SOURCE_NOT_OBSERVED=source_not_observed, GENERIC_RMF_FN=generic_rmf_fn, $
  
    SRCLIST_FILENAME=srclist_fn,  EXTRACTION_NAME=extraction_name, EMAP_BASENAME=emap_basename, $

		MIN_NUM_CTS=min_counts, EXTRA_MASKFILE=extra_maskfile, $
		
    REUSE_MODELS=reuse_models, SKIP_EXTRACT_BACKGROUNDS=skip_extract_backgrounds, _EXTRA=extra

  
exit_code = 0
creator_string = "ae_better_masking, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()


if NOT keyword_set(threshold)        then threshold = 0.1
if NOT keyword_set(min_counts)       then min_counts=100
if ~keyword_set(evtfile_basename) then begin
  print, 'ERROR: parameter EVTFILE_BASENAME must be supplied'
  GOTO, FAILURE
endif
if (evtfile_basename NE "validation.evt") then begin
  print, evtfile_basename, F='(%"\nERROR: EVTFILE_BASENAME is expected to be the aggressively-cleaned validation.evt; are you SURE you want to mask %s instead?")'
  stop
endif


if NOT keyword_set(emap_basename)    then emap_basename    = 'obs.emap'

if ~keyword_set(srclist_fn) then srclist_fn = 'all.srclist'

if (n_elements(plot) EQ 0) then plot = 1

if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
  print, 'ERROR: parameter "obsname" must be a scalar string'
  GOTO, FAILURE
endif

obsdir       = '../obs' + obsname + '/'
if ~file_test(obsdir) then begin
  print, 'ae_better_masking: ERROR: could not find observation directory ', obsdir
  GOTO, FAILURE
endif

lock_fn     = obsdir + 'ae_lock'
file_delete, /ALLOW_NONEXISTENT, lock_fn
file_copy  , '/dev/null'       , lock_fn, /OVERWRITE

evtfile      = obsdir + evtfile_basename
emapfile     = obsdir + emap_basename
bkg_evtfile  = obsdir + 'background.evt'  ; MASKED observation event list.
bkg_emapfile = obsdir + 'background.emap' ; MASKED observation emap.

       stowed_evt_fn = obsdir + 'stowed.evt'      ; UNMASKED stowed event list.
masked_stowed_evt_fn = obsdir + 'diffuse.bkg.evt' ; MASKED stowed event list, for image smoothing NOT for source extraction.


regionfile      = obsdir + 'polygons.reg'
wing_flux_fn    = obsdir + 'wing_flux.img' 
wing_counts_fn  = obsdir + 'wing_counts.img'
bkg_counts_fn   = obsdir + 'bkg_counts.img'
bkg_radius_fn   = obsdir + 'bkg_radius.img'

collatefile     = obsdir + 'all.collated'

obs_stats_basename       = 'obs.stats'
src_region_basename      = 'extract.reg'
psf_basename             = 'source.psf'
bkg_spectrum_basename    = 'background.pi'
rmf_basename             = 'source.rmf'
arf_basename             = 'source.arf'

;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir

temp_events_fn   = tempdir + 'temp.evt'
temp_region_fn   = tempdir + 'temp.reg'
temp_image_fn    = tempdir + 'temp.img'


; Abort if this is a gratings observation!  Diffuse work cannot be done with those data!
emap_header = headfits(emapfile)
gratingtype = strtrim(psb_xpar( emap_header,'GRATING'),2)
if ~strmatch(gratingtype,'NONE')  then begin
  print, obsname, gratingtype, F='(%"ObsID %s is not useful for diffuse work because the %s grating was used.  Aborting.")'

  GOTO, CLEANUP
endif


;; =====================================================================
;; BUILD AN IMAGE THAT MODELS THE LIGHT EXPECTED FROM ALL THE POINT SOURCES
;; Since we specify /SMOOTH_MODELS below and such smoothing is NOT desirable for building background spectra, we use separate savefiles for those two tasks.  
savefile_basename = 'ae_better_masking.sav'
model_savefile    = obsdir + savefile_basename

if ~keyword_set(reuse_models) || (~file_test(model_savefile)) then begin
  ae_better_backgrounds, obsname, /SMOOTH_MODELS, /BUILD_MODELS_ONLY, EVTFILE_BASENAME=evtfile_basename, $
    
      BACKGROUND_MODEL_FILENAME=background_model_filename, $
    
      SOURCE_NOT_OBSERVED=source_not_observed, GENERIC_RMF_FN=generic_rmf_fn, $
      
      SRCLIST_FILENAME=srclist_fn, EXTRACTION_NAME=extraction_name, $
      EMAP_BASENAME=emap_basename, $
  
      SAVEFILE_BASENAME=savefile_basename, $
      VERBOSE=verbose,  $
      
      _EXTRA=extra
endif

; The key variables computed by ae_better_backgrounds that we are restoring below are:
;
;   emap, emap_header, emap2wcs_astr, emap_col_dim, emap_row_dim, skypix_per_emappix:
;     Exposure map and related information.

;   bt: 
;     Collation on the observed sources, including photometry (NET_CNTS)

;   observation_counts_model_img:
;     Image (on the same pixel grid as emap) modeling the point source counts expected to be observed.
;   
;   models:
;     Structure the contains the model of each star.


; Remove any bkg_evtfile or bkg_emapfile files from a previous run (which could be either FITS files or symlinks).
file_delete, /ALLOW_NONEXISTENT, [bkg_evtfile, bkg_emapfile, masked_stowed_evt_fn]

if ~file_test(model_savefile) then begin
  print, model_savefile, F='(%"WARNING: File %s is missing; assuming no sources were observed.")'
  ; No masking is required, so simply symlink the masked data products to the input data products.
  file_link,       evtfile, bkg_evtfile 
  file_link,      emapfile, bkg_emapfile
  file_link, stowed_evt_fn, masked_stowed_evt_fn

  GOTO, CLEANUP
endif

; Restore the variables  emap, emap_header, emap2wcs_astr, emap_col_dim, emap_row_dim, skypix_per_emappix, bt, models, observation_counts_model_img that were saved by ae_better_backgrounds.
restore, /V, model_savefile
emap_dims = [emap_col_dim, emap_row_dim]

; Read the exposure map ("emap" and "emap_header") from disk, rather than using what was restored above, because it may have changed!
emap = readfits(emapfile, emap_header)


; Fail if the dimensions of the current emap do not match those used to build the point-source models.
if ~ARRAY_EQUAL(size(emap, /DIMEN), emap_dims) then begin
  print, size(emap, /DIMEN), emap_dims, F='(%"\nERROR: the dimensions of the current emap (%dx%d) do not match the grid (%dx%d) on which ae_better_backgrounds built its point-source models. Try ae_better_masking again with REUSE_MODELS=0.\n")' 
  GOTO, FAILURE
endif

band_full = 0
if ~almost_equal(bt.ENERG_LO[band_full], 0.5, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
if ~almost_equal(bt.ENERG_HI[band_full], 7.0, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 7.0 keV.\n")'

CATALOG_NAME = strtrim(bt.CATALOG_NAME,2)
NET_CNTS     = bt.NET_CNTS[band_full]
EXPOSURE     = bt.EXPOSURE
ENERG_LO     = bt[0].ENERG_LO[band_full]
ENERG_HI     = bt[0].ENERG_HI[band_full]
X_CAT        = bt.X_CAT
Y_CAT        = bt.Y_CAT

num_sources = n_elements(bt)


if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)


;; =====================================================================
;; Convert our model of the point source counts expected to be observed into flux units.
;; This is necessary because our masking algorithm is comparing a background flux estimate to this point source flux model.
star_counts = float(temporary(observation_counts_model_img))
star_flux = star_counts / emap


 
;; =====================================================================
;; Any region supplied by the observer via EXTRA_MASKFILE defines our initial mask.
if keyword_set(extra_maskfile) && ~file_test(extra_maskfile) then begin
  print, extra_maskfile, F='(%"\nae_better_masking: ERROR: could not find EXTRA_MASKFILE %s\n")' 
  GOTO, FAILURE
endif
  
if keyword_set(extra_maskfile) && file_test(extra_maskfile) && (file_lines(extra_maskfile) GT 0) then begin
  print, 'Spawning ds9 to perform coordinate conversions on EXTRA_MASKFILE ...'
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='ae_better_masking_'+session_name, OPTION_STRING='-iconify'
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, emapfile

  ;; Load region file into ds9 and resave in PHYSICAL coordinates.
  cmd = strarr(4)
  cmd[0] = string(my_ds9,                 F='(%"xpaset -p ''%s'' regions delete all")')
  cmd[1] = string(my_ds9, extra_maskfile, F='(%"xpaset -p ''%s'' regions load %s")')
  cmd[2] = string(my_ds9,                 F='(%"xpaset -p ''%s'' regions system physical")')
  cmd[3] = string(my_ds9, temp_region_fn, F='(%"xpaset -p ''%s'' regions save %s")')
  run_command, cmd, /QUIET

  cmd1 = string(emapfile, temp_region_fn, temp_image_fn, F="(%'dmcopy ""%s[sky=region(%s)][opt full,update=no]"" %s clobber=yes')")
  
  run_command, cmd1

  star_mask = (readfits(temp_image_fn) GT 0)

  run_command, string(my_ds9, F='(%"xpaset -p %s exit")'), /QUIET
endif else begin
  star_mask = bytarr(emap_col_dim,emap_row_dim)
endelse


;; =====================================================================
;; We are going to explicitly mask one pixel at each source position, mostly as a visual reminder of where the sources lie.
;; We cannot use xy2ad.pro/ad2xy.pro for conversions between array index and PHYSICAL (sky) coordinate systems.

crvalP = [psb_xpar( emap_header, 'CRVAL1P'), psb_xpar( emap_header, 'CRVAL2P')]
crpixP = [psb_xpar( emap_header, 'CRPIX1P'), psb_xpar( emap_header, 'CRPIX2P')]
cdeltP = [psb_xpar( emap_header, 'CDELT1P'), psb_xpar( emap_header, 'CDELT2P')]

for ii = 0L, num_sources-1 do begin
  xind_catalog = round((crpixP[0] + (X_CAT[ii]-crvalP[0])/cdeltP[0]) - 1)
  yind_catalog = round((crpixP[1] + (Y_CAT[ii]-crvalP[1])/cdeltP[1]) - 1)
  
  star_mask[xind_catalog,yind_catalog] = 1
endfor ;ii


;; =====================================================================
;; Apply star_mask to an in-band image of the data.

print, 'energy band is ', ENERG_LO, ENERG_HI
run_command, string(emapfile, F="(%'get_sky_limits %s verbose=0 precision=3')")
run_command, /QUIET, 'pget get_sky_limits dmfilter', filterspec

if (filterspec EQ '') then message, 'ERROR running get_sky_limits'

cmd = string(evtfile, 1000*[ENERG_LO,ENERG_HI], filterspec, temp_image_fn, F="(%'dmcopy ""%s[energy=%6.1f:%7.1f][bin %s]"" %s clobber=yes')")
run_command, cmd  

masked_data = readfits(temp_image_fn)
masked_emap = emap

total_star_counts = total(star_counts, /DOUBLE)
print, 100*total_star_counts/total(/INTEGER,masked_data), total_star_counts, F='(%"Stellar models explain %5.1f%% (%d) of the detected events.")'

mask_ind = where(star_mask, mask_count)
masked_data[mask_ind] = 0
masked_emap[mask_ind] = 0


; Defensively, MAKE SURE ALL IMPORTANT IMAGES HAVE THE SAME DIMENSIONS
if ~ARRAY_EQUAL(size(emap       , /DIMEN), emap_dims) || $
   ~ARRAY_EQUAL(size(star_counts, /DIMEN), emap_dims) || $ 
   ~ARRAY_EQUAL(size(star_mask  , /DIMEN), emap_dims) || $ 
   ~ARRAY_EQUAL(size(star_flux  , /DIMEN), emap_dims) || $ 
   ~ARRAY_EQUAL(size(masked_data, /DIMEN), emap_dims) || $ 
   ~ARRAY_EQUAL(size(masked_emap, /DIMEN), emap_dims) then $
    message, 'BUG: inconsistent array dimensions ...'

; Create an image with the same dimensions as all the others, containing the 1-D index of each element.
index_image = lindgen(emap_dims[0],emap_dims[1])


  
;; =====================================================================
;; Iteratively apply additional masking of pixels where flux from stars
;; is large compared to flux from background.
;; Much of this code is taken from adaptive_density_2d.pro.

; As explained in the header comments of adaptive_density_2d.pro
; we crop the data image where exposure is <10% nominal to 
; avoid artifacts in the bkg_flux image. 
; 
; For the same reasons we'll also add those low-exposure field edges 
; to the mask we're making for the background emap & event list.
; If the off-field threhold is set too high, then you'll mask out pixels in the hole at the center of the emap.
off_field = where(emap LT (0.05 * max(emap)), off_field_count)
print, off_field_count, ' emap pixels masked for <10% nominal exposure'
if (off_field_count GT 0) then begin
  star_mask  [off_field] =  1
  masked_data[off_field] =  0
  masked_emap[off_field] =  0
endif

; We can save considerable time by telling adaptive_density_2d to only compute
; background estimates where we need them, which is 
; (a) pixels that are not already masked
; (b) pixels where star_flux is nonzero.  
; The other pixels in bkg_flux will come back with the value NaN.
field_mask = (star_mask EQ 0) AND (star_flux GT 0)

max_radius = 500
print, threshold, F='(%"Masking pixels where star_flux/bkg_flux > %f")'

;; It is vital to the operation of the search below that the radii
;; list starts with r=0 (a kernel consisting of only the central pixel).
;; 2017 July; I am not sure the comment above is accurate.  It comes from a similar comment (which I don't understand) in adaptive_density_2d.pro.


;; We have to limit the number of kernels to avoid excessive memory requirements.
skip = 0
repeat begin
  skip++
  radii = skip * indgen(1+(max_radius/skip))
  max_radius_index = n_elements(radii)-1
  num_kernels      = n_elements(radii)
endrep until (num_kernels LT 100)

if NOT keyword_set(silent) then print, 'Using kernel radii: ', radii

kernel_footprint, kf_id, /CREATE, IMAGE=masked_data, RADII=radii

print, F='(%"Finished building Tophat kernels.")'
size_spec = size(masked_data)
bright_count = 0L
initial_radius_index = 0 

bkg_flux   = make_array( SIZE=size(star_flux), VALUE=!VALUES.f_nan, /FLOAT )
radius_map = make_array( SIZE=size(star_flux), VALUE=-1           , /INTEGER )

repeat begin
  masking_done = 1
  
  ; Find indexes of pixels we're considering masking.
  field_ind = where(field_mask, num_pixels)
  
  ; Sort them by their star_flux.
  sorted_index1D = field_ind[ reverse( sort( star_flux[field_ind] ) ) ]
  print, num_pixels, F='(%"Examining star_flux/bkg_flux in %d pixels ...")'

  checkpoint_rows = round([.2,.4,.6,.8]*num_pixels) > 1
  start_time = systime(1)
  
  for ii = 0L,num_pixels-1 do begin
    if (ii EQ checkpoint_rows[0]) then begin
      elapsed_time = (systime(1)-start_time)
      estimated_total_time = num_pixels * (elapsed_time/checkpoint_rows[0])

      print, round(100.0*checkpoint_rows[0]/num_pixels), ceil((estimated_total_time-elapsed_time)/60.), now(),$
	     F='(%"%d%% of pixels processed; estimate %d more minutes to finish.  (%s)")'
      checkpoint_rows = shift(checkpoint_rows,-1)
    endif

    index1D = sorted_index1D[ii]
    index_to_point, index1D, xx, yy, size_spec
    
    ; Find the background level at this pixel's location by searching for a 
    ; circular region that contains min_counts counts.
    ; Arrange for the search to have an efficient starting point and direction.
    radius_index     = initial_radius_index
    search_direction = 0 ;just started
    search_done = 0
;help,ii
    repeat begin
;help,radius_index
      ;; Retrieve the 1-D indexes of pixels that fall under the kernel.
      ;; We're doing Top Hat kernels and simply looking for min_counts
      ;; counts so we ignore the "weight" values returned by kernel_footprint.
      kernel_footprint, kf_id, xx, yy, radius_index, pixel_list
 
      counts = total(/INTEGER,masked_data[pixel_list])
      significance_is_good = (counts GE min_counts)
      
      if significance_is_good then begin
        if (search_direction EQ 1) then begin
          ; We were searching UP from bad kernels and found a good one, so
          ; stop and keep this kernel.
          search_done = 1 
        endif else begin
          ; We just started (search_direction==0), or were searching down (search_direction==-1) and found a good one, so we need to keep going down, if possible.
          if (radius_index LE 0) then begin
            search_done = 1 
          endif else begin
            search_direction = -1 ; down
            radius_index     = radius_index - 1
          endelse
        endelse
        
      endif else begin
        if (search_direction EQ -1) then begin
          ; We were searching DOWN from good kernels and found a bad one, so
          ; stop and keep the next larger kernel.
          radius_index++
          search_direction = 1 ;up
        endif else begin
          ; We just started (search_direction==0), or were searching up (search_direction==1) and found a bad one,
          ; so we need to keep going up, if possible.
          if (radius_index GE max_radius_index) then begin
            print, 'WARNING: search truncated at max kernel radius'
            search_done = 1 
          endif else begin
            search_direction = 1 ;up
            radius_index++
          endelse
        endelse
      endelse ; significance is bad

    endrep until search_done
    
    ;; Save the next smaller kernel as the starting point for the next pixel.
    ;; The way the search above works, if the starting kernel turns out to
    ;; be the one we're looking for, then we must step down one kernel and 
    ;; then back up, wasting time.  If we start just below the goal, then
    ;; we make one step and we're done.
    initial_radius_index = (radius_index - 1) > 0

    ; However, if "search truncated at max kernel radius" then the next pixel is likely to do the same,
    ; so the starting kernel should be max kernel radius.
    if (radius_index GE max_radius_index) then initial_radius_index = max_radius_index

    exposure = total(masked_emap[pixel_list])
    bkg_flux  [index1D] = counts /exposure
    radius_map[index1D] = radii[radius_index]
    
    if ((star_flux[index1D]/bkg_flux[index1D]) GT threshold) then begin
;     print, masked_data[index1D], xx, yy, F='(%"masked %d counts in pixel (%4d,%4d)")' 
      masking_done = 0
      bright_count++
      star_mask  [index1D] = 1
      masked_data[index1D] = 0
      masked_emap[index1D] = 0
      field_mask [index1D] = 0
    endif
    
  endfor

endrep until masking_done
print, bright_count, ' pixels added to mask due to stellar contamination'
kernel_footprint, kf_id, /DESTROY


;; =====================================================================
; Consistency check.
if ~array_equal(masked_emap[ where( star_mask) ], 0) then message, 'ERROR: star_mask inconsistent with masked_emap'



;; =====================================================================
;; Explicitly mask a large region around any source expected to have significant counts outside the model (PSF footprint).

; Apply the mask we have so far to star_counts image, which we will use below.
star_counts[where(/NULL, star_mask )] = 0

for ii = 0L, num_sources-1 do begin
  this_model = models[ii]
  
  ; Estimate the number of source counts within the PSF footprint that remain unmasked.
  
  unmasked_counts_under_psf = total( star_counts[this_model.col_min:this_model.col_max,$
                                                 this_model.row_min:this_model.row_max] )
  
  ; If the number of source counts beyond the PSF footprint (cropped_counts) is small compared to unmasked_counts_under_psf, then don't bother with additional masking.
  
  if (this_model.cropped_counts LT unmasked_counts_under_psf) then continue
  
  
  ; Define a mask region with dimensions twice those of the PSF footprint.  
  col_min = floor(1.5*this_model.col_min - 0.5*this_model.col_max)
  col_max = ceil (1.5*this_model.col_max - 0.5*this_model.col_min)
  
  row_min = floor(1.5*this_model.row_min - 0.5*this_model.row_max)
  row_max = ceil (1.5*this_model.row_max - 0.5*this_model.row_min)
  
  ; Find 1-D indexes of the image pixels under that mask region.
  mask_index = index_image[ (0>col_min):(col_max<(emap_col_dim-1)),$
                            (0>row_min):(row_max<(emap_row_dim-1)) ]
  
  
  ; If the number of source counts beyond the PSF footprint (cropped_counts) is small compared to the unmasked observed counts under the proposed mask, then don't bother with additional masking.
  observed_counts_under_big_mask = total(masked_data[mask_index])
  
  if (this_model.cropped_counts LT observed_counts_under_big_mask/10.0) then continue
  
  
  ; The mask is applied to star_mask, masked_data, masked_emap (as always done above).
  star_mask  [mask_index] = 1
  masked_data[mask_index] = 0
  masked_emap[mask_index] = 0
  
  print, this_model.label, round(this_model.cropped_counts), F='(%"Source %s has been masked beyond the PSF footprint, where %d counts are expected.")'
  help, unmasked_counts_under_psf, observed_counts_under_big_mask
  
  ; Set cropped_counts to zero for this model, to reflect the additional masking.
  ; That will improve the accuracy of the ObsID-wide total for cropped_counts reported later.
  models[ii].cropped_counts = 0 
endfor


;; =====================================================================
;; Apply our masking (carried in star_mask) to other images that must be masked.
mask_ind = where( star_mask, mask_count )
star_counts[mask_ind] = 0
star_flux  [mask_ind] = 0
print, mask_count, ' total emap pixels masked'

;  Multiply by emap to get background counts model.
bkg_counts  = bkg_flux  * emap
                          
;; Save results.
writefits, bkg_emapfile  , masked_emap, emap_header
writefits, wing_flux_fn  , star_flux  , emap_header
writefits, wing_counts_fn, star_counts, emap_header
writefits, bkg_counts_fn , bkg_counts , emap_header
writefits, bkg_radius_fn , radius_map , emap_header


;; =====================================================================
;; Background event list is made by discarding events where background emap is zero.
;; The emap passed to dmimgpick below should span the event list, so the bug in HelpDesk Ticket #020605 should not be triggered.
                
cmd2 = string(evtfile, bkg_emapfile, temp_events_fn, F="(%'dmimgpick ""%s[cols time,ccd_id,chip,det,sky,pi,energy,status]"" %s %s method=closest clobber=yes')")

cmd3 = string(temp_events_fn, bkg_evtfile, F="(%'dmcopy ""%s[#9>1][cols time,ccd_id,chip,det,sky,pi,energy,status]"" %s clobber=yes')")
run_command, cmd2
run_command, cmd3



;; =====================================================================
;; Masked stowed event list is made by discarding events where background emap is zero.
;; The emap passed to dmimgpick below should span the event list, so the bug in HelpDesk Ticket #020605 should not be triggered.
; DO NOT REMOVE THE TIME COLUMN!  That would later cause reproject_events to drop data from some CCDs (Help Desk Ticket #15799).
                
cmd2 = string(stowed_evt_fn, bkg_emapfile, temp_events_fn, F="(%'dmimgpick ""%s[cols time,ccd_id,chip,det,sky,pi,energy]"" %s %s method=closest clobber=yes')")

cmd3 = string(temp_events_fn, masked_stowed_evt_fn, F="(%'dmcopy ""%s[#8>0][cols time,ccd_id,chip,det,sky,pi,energy]"" %s clobber=yes')")
run_command, cmd2
run_command, cmd3




;; =====================================================================
run_command, 'grep polygon ' + obsdir+'extract.reg' + ' > ' + regionfile


if keyword_set(plot) then begin

  cmd = string('ae_better_masking ('+obsname+')', $
          bkg_evtfile, 1000*[ENERG_LO,ENERG_HI],  $
          regionfile, keyword_set(extra_maskfile) && file_test(extra_maskfile) ? '-region '+extra_maskfile : '', $
          bkg_emapfile, $
          regionfile, keyword_set(extra_maskfile) && file_test(extra_maskfile) ? '-region '+extra_maskfile : '', $
          masked_stowed_evt_fn,$
  ;       wing_counts_fn, $
  ;       bkg_counts_fn, $
          F='(%"ds9 -title \"%s\" -tile -lock frame wcs -linear -bin factor 4  \"%s[1][energy>%d && energy<%d]\" -region %s %s  %s -log -region %s %s   %s -linear -zoom to fit  >& /dev/null &")')
  
  release_cpu
  flush_stdin  ; eat any characters waiting in STDIN, so that they won't be mistaken as commands below.
  print, F='(%"\r\n\r\nPRESS RETURN to visualize masking results.")'
  temp = ''
  read, ':', temp
  run_command, cmd, /QUIET
  semaphore = wait_for_cpu()
  
  print, bkg_emapfile,                                                        F='(%"\nThe right  ds9 panel (%s) shows the masked exposure map.")'
  
  print, bkg_evtfile, ENERG_LO, ENERG_HI,                                     F='(%"The left ds9 panel (%s ) shows the masked event list (%0.1f:%0.1f keV).")'
endif ; keyword_set(plot)
  
print, wing_counts_fn, total(star_counts), ENERG_LO, ENERG_HI, bkg_evtfile, F='(%"\nIf needed, %s maps the %0.1f counts (%0.1f:%0.1f keV) from known point sources expected to remain in %s.")'

; Tally the counts expected from the PSFs wings that lie beyond the footprints of the PSF models (images) used to construct "observation_counts_model_img".
print, total(models.cropped_counts), bkg_evtfile, F='(%"An additional %0.1f expected point source counts lie beyond the footprints of the PSF models; some or all of them may remain in %s.")'

print, bkg_counts_fn, ENERG_LO, ENERG_HI,                                  F='(%"\nIf needed, %s maps the estimated background (counts in %0.1f:%0.1f keV) where it was computed.")'


if NOT keyword_set(skip_extract_backgrounds) then begin
  ;; =====================================================================
  ;; Extract background for each source.
  acis_extract, srclist_fn, obsname, bkg_evtfile, /EXTRACT_BACKGROUNDS, EMAP_FILENAME=bkg_emapfile, _EXTRA=extra
  
  
  ;; =====================================================================
  ;; Collate results.
  ; Commented out to reduce run times.
  ;acis_extract, srclist_fn, obsname, EXTRACTION_NAME=extraction_name, /SINGLE_OBS, COLLATED_FILENAME=collatefile, VERBOSE=0
endif


CLEANUP:
if keyword_set(tempdir) && file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then begin
  print, F='(%"\nae_better_masking finished")'  
  return 
endif else begin
  print, 'ae_better_masking: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP

end ; ae_better_masking





;#############################################################################
;;; Tool to mask stowed data to match a masked ObsID.
;;;
;;; In our standard directory structure, this tool should be called from the directory <target>/data/extract/.

;;; INPUTS
;;; Observation emap: 
;;;   obs.emap, which is typically a symlink to an emap built by L1_2_L2_emaps.
;;; 
;;; Observation event list and corresponding emap with point sources masked:
;;;   diffuse.evt , which is typically a symlink to background.evt  (built by ae_better_masking).
;;;   diffuse.emap, which is typically a symlink to background.emap (built by ae_better_masking).
;;;
;;; Stowed data event list and corresponding emap:
;;;   stowed.evt , which is typically a symlink to iarray.stowed.evt2 (from L1->L2 procedure)
;;;   stowed.emap, which is typically a symlink to a stowed emap built by L1_2_L2_emaps.
;;;   NOTE that this pair of data products are used by AE for extraction of diffuse sources.

;;; OUTPUTS
;;; Stowed data event list with point sources masked.
;;;   diffuse.bkg.evt
;;;   This data product is used only for image smoothing, NOT for extraction of diffuse sources in AE!!!!
;;;
;;; Image recording the scaling of the stowed data (stowed.emap / obs.emap):
;;;   diffuse.bkg.scaling     
;;;   This image is NOT masked!!  
;;;   It is used in a call to build_scene (diffuse_procedure.txt, tara_smooth.txt) that builds a scaled stowed image for use by tara_smooth.


PRO ae_mask_stowed_data, OBS_DIR=obs_dir

print, 'ae_mask_stowed_data is obsolete.'
retall

end ; ae_mask_stowed_data
  
  
  

;#############################################################################
;;; Build background regions which account for contamination from neighboring sources.
;;;
;;; The entire catalog must already have been processed through the EXTRACT_SPECTRA stage.
;;; This tool will run the MERGE_OBSERVATIONS stage on a SINGLE obsid to estimate fluxes for all the sources.
;;; Note that a source might have zero or negative net counts in any single observation.

;;; NEIGHBOR_INVALID_THRESHOLD (e.g. 0.10) can be supplied to override the default of 0.10 (10%).


;;; The required parameter EVTFILE_BASENAME specifies the name of the observation event list file in
;;; the directory ../obsXXXX/.  In our recipe we use EVTFILE_BASENAME='spectral.evt' (lightly cleaned) for all passes.
;;; AE stages apply a STATUS=0 filter (recorded in the src property S_FILTER) for sources that have event rates low enough to avoid removing valid events.
;;; This tool also applies a STATUS=0 filter to the backgrounds of such sources.

;;; The optional input BACKGROUND_MODEL_FILENAME can be used to supply a model (FITS image) 
;;; of any background component not represented by the point sources in the catalog.
;;; For example such a model could be constructed for diffuse emission so it will
;;; participate in the algorithm used to contruct background regions.
;;; The BACKGROUND_MODEL_FILENAME should be in units of observed counts (i.e. exposure 
;;; variation is represented in the model).

;;; The algorithm for choosing a background region consists of the following "search phases". 
                                     
;;; 1. The bkg region will grow until two conditions are met:
;;;   
;;;   A. The region contains at least MIN_NUM_CTS in-band counts.
;;; AND
;;;   B. BKSCL_LO <= BACKSCAL   (BKSCL_LO is stored in source.stats)
;;;   
;;; The MIN_NUM_CTS requirement is to avoid background estimates that have huge statistical uncertainty, which inflates
;;; Pb and can cause a source to be pruned. The pathological case is a source with an uncrowded on-axis observation and a
;;; crowded far off-axis observation. The off-axis observation needs to keep BACKSCAL small in order to fairly sample his
;;; crowded neighbor, but that small BACKSCAL leaves the on-axis observation with very few (perhaps zero) counts. If the
;;; MERGE then chooses to ignore the off-axis observation, then we're left with a composite extraction with a very uncertain
;;; background estimate, and an inflated Pb value. Aaarg!


;;; After phase #1, the algorithm will keep track of which bkg region, dubbed our "reserve region", achieved the
;;; best (smallest) background imbalance metric.


;;; 2. After #1 is achieved, we add ONE more pixel to the bkg region before considering any stopping criteria.
;;;    This tweak is designed to address a special case---where BKSCL_LO=BKSCL_GL, and the current extraction is responsible for that value (via a VOTE_LO from the previous run of this tool).
;;;    In such a situation, we want to proceed past the last acceptable region to determine if our previous VOTE_LO still applies.
;;;    If we simply stop at BKSCL_LO=BKSCL_GL, and cast no VOTE_LO, then another ObsID can drag the scaling range upward; on the next pass this ObsID will again find that range to be unacceptable, and will recast a VOTE_LO.
;;;    This leads to a never-ending cycle: 
;;;      - this ObsID casts a VOTE_LO
;;;      - the range is adjusted to satisfy that vote
;;;      - this ObsID abstains from voting
;;;      - the range is adjusted upward, beyond what this ObsID can tolerate
;;;      - this ObsID casts a VOTE_LO

;;; 2.5 After #2 is achieved, perform a one-time test whether the photometry of this extraction is significantly negative.
;;;     If that's true, then this extraction is unlikely to be used in the merge or this source is unlikely to survive.  
;;;     In the interest of run-time, we decide to stop searching for a "better" bkg region for this unpromising extraction.

;;; 3. After #2.5 is passed, growth of the bkg region will continue until BKSCL_GL <= BACKSCAL (the region reaches the goal
;;; specified for all ObsIds, BKSCL_GL, which is stored in source.stats).

;;; 4. After #3 is achieved, growth of the bkg region will stop when 
;;; 
;;; 4A. If the background imbalance metric is less than BACKGROUND_IMBALANCE_THRESHOLD, then accept the bkg region.
;;; OR
;;; 4B. When the region reaches the maximum allowed size (BKSCL_HI <= BACKSCAL), move to phase #5.


;;; 5. Since condition 4B stopped the search (no acceptable region was found), we rebuild and then adopt the reserve region.


PRO ae_better_backgrounds, obsname, EVTFILE_BASENAME=evtfile_basename, $
    NUMBER_OF_PASSES=number_of_passes, $ 
  
		THETA_RANGE=theta_range, BACKGROUND_MODEL_FILENAME=background_model_filename, $
  
    SOURCE_NOT_OBSERVED=source_not_observed, GENERIC_RMF_FN=generic_rmf_fn, $
    
    SRCLIST_FILENAME=srclist_fn, EXTRACTION_NAME=extraction_name, $
    EMAP_BASENAME=emap_basename, $

		MIN_NUM_CTS=min_num_cts, BACKGROUND_IMBALANCE_THRESHOLD=background_imbalance_threshold, $
    COMPACTNESS_GAIN=compactness_gain, NEIGHBOR_INVALID_THRESHOLD=neighbor_invalid_threshold_p, $
    
    VERBOSE=verbose, SHOW=show, PAUSE_FOR_REVIEW=pause_for_review, SKIP_RESIDUALS=skip_residuals, $
      
    PHOTOMETRY_STAGE_ONLY=photometry_stage_only, SKIP_PHOTOMETRY_STAGE=skip_photometry_stage, $
    BUILD_MODELS_ONLY=build_models_only, REUSE_MODELS=reuse_models, SMOOTH_MODELS=smooth_models, $
  
    SAVEFILE_BASENAME=savefile_basename, OBS_DIR=obs_dir_p

exit_code = 0
backgrounds_extracted = 0
creator_string = "ae_better_backgrounds, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

target_name = getenv('TARGET')
if ~keyword_set(target_name) then target_name = 'Unknown Target'


if (n_elements(verbose)                   EQ 0) then verbose                   = 0
if (n_elements(show)                      EQ 0) then show                      = 1
if (n_elements(theta_range)               NE 2) then theta_range               = [0,100.]
if (n_elements(compactness_gain)          NE 1) then compactness_gain          = 0.01
compactness_gain = float(compactness_gain)

if ~keyword_set(background_imbalance_threshold) then background_imbalance_threshold = 0.10

if keyword_set(reuse_models) && keyword_set(neighbor_invalid_threshold_p) then begin
  print, 'ERROR: parameter NEIGHBOR_INVALID_THRESHOLD cannot be changed when /REUSE_MODELS specified.'
  GOTO, FAILURE
endif

neighbor_invalid_threshold   = keyword_set(neighbor_invalid_threshold_p) ? neighbor_invalid_threshold_p : 0.10 ; 10%
if (neighbor_invalid_threshold GT 0.2) then begin
  help, neighbor_invalid_threshold
  message, 'WARNING: INVALID_THRESHOLD is large.  Perhaps you made a mistake?'
endif


if ~keyword_set(evtfile_basename) then begin
  print, 'ERROR: parameter EVTFILE_BASENAME must be supplied'
  GOTO, FAILURE
endif
if ~keyword_set(savefile_basename)      then savefile_basename = 'ae_better_backgrounds.sav'
if ~keyword_set(emap_basename)          then emap_basename     = 'obs.emap'

if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
  print, 'ERROR: parameter "obsname" must be a scalar string'
  GOTO, FAILURE
endif


obsdir                      = '../obs' + obsname + '/'

if keyword_set(obs_dir_p) then  obsdir=obs_dir_p + '/'

if ~file_test(obsdir) then begin
  print, 'ae_better_backgrounds: ERROR: could not find observation directory ', obsdir
  GOTO, FAILURE
endif

lock_fn     = obsdir + 'ae_lock'
file_delete, /ALLOW_NONEXISTENT, lock_fn
file_copy  , '/dev/null'       , lock_fn, /OVERWRITE

evtfile_p                   = obsdir + evtfile_basename
emapfile_p                  = obsdir + emap_basename
model_savefile              = obsdir + savefile_basename
model_image_fn              = obsdir + 'observation_counts_model.img' ; I wish I had named this 'inband_pointsource_model.img'.
data_image_fn               = obsdir + 'inband_data.img'
residual_image_fn           = obsdir + 'inband_residual.img'
collatefile                 = obsdir + 'all.collated'
regionfile                  = obsdir + 'extract.reg'

if ~keyword_set(srclist_fn)      then srclist_fn = 'all.srclist'
if (srclist_fn NE 'all.srclist') && ~keyword_set(reuse_models) then $
  print, 'WARNING: Background estimation may be unreliable when only a subset of full catalog is processed.'

if keyword_set(reuse_models) && (~file_test(model_savefile)) then begin
  print, 'WARNING: MODEL_SAVEFILE '+model_savefile+' not found; ignoring /REUSE_MODELS.'
  reuse_models = 0
endif

arcsec_per_skypixel = 0.492 

; Make spectra with 1024 channels to match default in source.pi.
; (This was changed from 685 to 1024 in Sept 2011.  The change should have been made when acis_extract.pro started producing spectra with 1024 channels, on 07/20/2009 3495.  I don't see any harm done---this tool is used on only point sources, and we never examine their spectra above 8 keV.)
DETCHANS = 1024

src_stats_basename         = 'source.stats'
obs_stats_basename         = 'obs.stats'
obs_frac_basename          = 'obs.psffrac'
src_region_basename        = 'extract.reg'
src_emap_basename          = 'source.emap'
psf_basename               = 'source.psf'
streak_basename            = 'source.streak'
rmf_basename               = 'source.rmf'
arf_basename               = 'source.arf'
bkg_spectrum_basename      = 'background.pi'
bkg_emap_basename          = 'background.emap'
bkg_pixels_region_basename = 'background_pixels.reg'
bkg_events_basename        = 'background.evt'

;; Create a unique scratch directory.
cache_dir        = 'cache'
tempdir          = 'tmp'
temproot = temporary_directory( 'AE.', SUBDIR1=cache_dir, SUBDIR2=tempdir, VERBOSE=1, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir

green_region_fn  = cache_dir + 'green.reg'
 gray_region_fn  = cache_dir + 'gray.reg'
inband_events_fn = cache_dir + 'temp.inband.evt'
temp_image_fn    = tempdir   + 'temp.img'
temp_events_fn   = tempdir   + 'temp.evt'
temp_text_fn     = tempdir   + 'temp.txt'

;; We assume that access to /tmp/ will often be faster than access to the event and emap data passed,
;; so let's start by copying those files to a cache.
if ~file_test(evtfile_p) then begin
  message, 'Cannot read ' + evtfile_p
endif

if ~file_test(emapfile_p) then begin
  message, 'Cannot read ' + emapfile_p
endif

fdecomp, evtfile_p, disk, item_path, item_name, item_qual
evtfile = cache_dir+ item_name+'.'+item_qual
; For efficiency, discard the columns we do not wish to propagate to output files.
run_command, string(evtfile_p, evtfile, F="(%'dmcopy ""%s[cols time,ccd_id,sky,pi,energy,status]"" %s clob+')")

fdecomp, emapfile_p, disk, item_path, item_name, item_qual
emapfile = cache_dir+ item_name+'.'+item_qual
file_copy, emapfile_p, emapfile



if (verbose GE 1) then begin
  color_manager, NCOLORS=ncolors
  
  ; The xpaaccess tool cannot find the ds9 session if its name has any space characters!
  my_ds9 = "DS9:ae_better_backgrounds"
  ; Look for an existing ds9 session ...
  ; Starting in CIAO 4.0, xpaaccess uses the exit code to return its result.
  ; Thus we can no longer allow run_command to interpret a non-zero exit code as failure.
  run_command, string(my_ds9, F='(%"xpaaccess %s")'), result, /IGNORE_STATUS, /QUIET
  if (result[0] NE 'yes') then $
    run_command, 'ds9 -xpa local -tile -title ae_better_backgrounds -log >& /dev/null &'
endif
if (verbose GE 2) then begin
  window,0
  window,1
  window,2
  term1=replicate(!VALUES.F_NAN,10000)
  term2=term1
  term3=term1
  term4=term1
endif


; The map "pixel_status" takes on these flag values:
off_field_status = 0B  ; not allowed in background region because exposure is too small
masked_status    = 1B  ; not allowed in background region because too close to source
available_status = 2B  ; available for use in background, but not yet considered
candidate_status = 3B  ; currently competing to be accepted into the background region
background_status= 4B  ; accepted into the background region


;; =====================================================================
;; Collect photometry information from the existing single-observation extraction.
if keyword_set(reuse_models) then begin
  print, F='(%"\n===================================================================")'
  print, 'WARNING! Using source models saved from previous session in '+model_savefile
  restore, model_savefile, /VERBOSE
  print, F='(%"===================================================================\n")'
endif else begin
  if ~keyword_set(skip_photometry_stage) then begin
    print, F='(%"\nae_better_backgrounds: ============================================================")' 
    print, 'ae_better_backgrounds: Running MERGE_OBSERVATIONS stage on observation '+obsname
    print, F='(%"ae_better_backgrounds: ============================================================\n")'
    ; We need to run MERGE_OBSERVATIONS using this SINGLE observation for two reasons
    ; not related to time variability of the sources:           
    ; (1) We must ensure that all the properties collated below refer to one observation.
    ;     For example, SRC_CNTS is used later with single-obsid background information 
    ;     in Pb calculations.
    ; (2) Since a source may have a different PSF in different obsids the effects of 
    ;     crowding may be quite different.
    ; Do not set SKIP_APERTURE, because we want RA_DATA/DEC_DATA available for ae_interObsID_astrometry tool.
    ;
    ; We compute single-ObsID photometry over the four energy bands we use for source validation.
    ; The source models we build below will be based on the full-band photometry, which is band number "band_full".
    merge_name   = 'EPOCH_'+obsname
    
    ; MUST match energy ranges in ae_pileup_screening.
    eLO   = [0.5, 0.5, 2.0, 4.0]
    eHI   = [7.0, 2.0, 7.0, 7.0]
    eRNG  = [[eLO],[eHI]]
    eBAND = ['full_band', 'soft_band', 'hard_band', 'vhard_band']

    ; Note that the SOURCE_NOT_OBSERVED speed-up input is not appropriate for the MERGE stage, and would be 
    ; ignored if supplied (see comments in MERGE stage).
    ; Specifically in this situation, we need the call below to destroy an existing EPOCH_XXXX merge when 
    ; this ObsID is no longer extracting the source.
    acis_extract, srclist_fn, obsname, EXTRACTION_NAME=extraction_name, MERGE_NAME=merge_name, /MERGE_OBSERVATIONS, /SKIP_PSF, /SKIP_NEIGHBORHOOD, SKIP_APERTURE=0, /SKIP_TIMING, ENERGY_RANGE=eRNG[0,*], EBAND_LO=eLO, EBAND_HI=eHI,  GENERIC_RMF_FN=generic_rmf_fn, VERBOSE=0

    
    print, F='(%"\nae_better_backgrounds: ============================================================")' 
    print, 'ae_better_backgrounds: Running COLLATE stage on observation '+obsname
    print, F='(%"ae_better_backgrounds: ============================================================\n")'
    ;; Collect the flux & background information from the existing extraction.
    ;; Do NOT use /SINGLE_OBSID because we need the photometry information from EPOCH_XXXX/source.photometry.
    ;; It's tempting to use the /MATCH_EXISTING option to save time, but I'm nervous about not having any control over the
    ;; format of the existing collatefile. 
    acis_extract, srclist_fn, obsname, EXTRACTION_NAME=extraction_name, MERGE_NAME=merge_name, COLLATED_FILENAME=collatefile, VERBOSE=0, SOURCE_NOT_OBSERVED=source_not_observed  
    
    if keyword_set(photometry_stage_only) then GOTO, CLEANUP
  endif ;~keyword_set(skip_photometry_stage)

  bt=mrdfits(collatefile, 1, /SILENT)
  temp = n_elements(bt)

  if ~tag_exist(bt, 'NUM_OBS') then begin
    print, 'ae_better_backgrounds: WARNING: No sources were observed in obsid ', obsname
    file_delete, /ALLOW_NONEXISTENT, [model_image_fn, model_savefile, data_image_fn, residual_image_fn]
    
    ; Also delete any savefile built by ae_better_masking, to reflect that there are no source models.
    file_delete, /ALLOW_NONEXISTENT, 'ae_better_masking.sav'
    GOTO, CLEANUP
  endif

  ; Sources which were not observed in this obsid are discarded here so that we don't waste time/space
  ; contructing various data structures which have an entry for each source.
  ; We must be careful to get source names from bt.CATALOG_NAME rather than from reading srclist_fn ourselves!
  bt = bt[where(bt.NUM_OBS GT 0, num_sources)]
  
  if (temp-num_sources GT 0) then print, temp-num_sources, F='(%"\nae_better_backgrounds: WARNING: Ignoring %d sources not in this observation.")'
  
  ; Verify that the merge was performed for every source observed by this ObsID, i.e. none were skipped due to some unwanted ObsID pruning algorithm in the MERGE stage..
  if array_equal(/NOT_EQUAL, (bt.MERGFRAC LT 1.0), 0) then begin
    print, F='(%"ae_better_backgrounds: ERROR: The MERGE_OBSERVATIONS stage seems to have skipped some sources!!!  There must be a bug!")'
    stop
  endif
  
endelse ; ~reuse_models
                                                                                          
num_sources = n_elements(bt)

ind = where(bt.NUM_OBS NE 1, count)
if (count GT 0) then begin
  print, 'ERROR: these sources have NUM_OBS != 1'
  forprint, SUBSET=ind, bt.CATALOG_NAME, bt.NUM_OBS
  GOTO, FAILURE
endif         


if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)


; Below we extract the source properties we're going to need.  
; This design is a bit sloppy---we require that all these properties correspond to the single obsid we're working with.
; Thus it's tempting to collate with /SINGLE_OBSID so we grab these properties directly from <src>/<obs>/obs.stats.
; However, we also need quantities (SRC_CNTS, NET_CNTS, EXPOSURE) calculated in /MERGE, run in single-obsid mode.
; To collate these quantities we must omit the /SINGLE_OBSID option.
; Thus, several single-source properties (e.g. SRC_AREA) are actually being collated from <src>/source.stats, 
; rather than from obs.stats.
;
; We could of course eliminate the MERGE stage from this algorithm, instead grabbing the single-obsid quantities we
; need directly from files in <src>/<obs>/, but that would require duplicating some code from the MERGE stage.

CATALOG_NAME = strtrim(bt.CATALOG_NAME,2)
PHOT_CREATOR = tag_exist(bt, 'PHOT_CREATOR') ? strtrim(bt.PHOT_CREATOR, 2) : strarr(num_sources) 

band_full = 0
if ~almost_equal(bt.ENERG_LO[band_full], 0.5, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
if ~almost_equal(bt.ENERG_HI[band_full], 7.0, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 7.0 keV.\n")'

SRC_CNTS       = bt.SRC_CNTS[band_full] 
NET_CNTS       = bt.NET_CNTS[band_full]
BACKSCAL       = bt.BACKSCAL[band_full]
PROB_NO_SOURCE = bt.PROB_NO_SOURCE[band_full]  ; NaN when no background spectrum is available.
EXPOSURE       = bt.EXPOSURE
ENERG_LO       = bt[0].ENERG_LO[band_full]
ENERG_HI       = bt[0].ENERG_HI[band_full]
MSK_RAD        = bt.MSK_RAD
X_CAT          = bt.X_CAT
Y_CAT          = bt.Y_CAT
RA             = bt.RA
DEC            = bt.DEC
LABEL          = strtrim(bt.LABEL,2)
THETA          = bt.THETA


; The SRC_AREA property is a hassle.  IF the observer has run ae_make_catalog on a source but then
; (for efficiency) skipped EXTRACT_SPECTRA because the region did not change much, then SRC_AREA
; will be missing.  In such cases we use instead PGN_AREA.
SRC_AREA = bt.SRC_AREA
PGN_AREA = bt.PGN_AREA
ind = where((SRC_AREA EQ 0) OR (~finite(SRC_AREA)), count)
if (count GT 0) then begin
  print, count, F='(%"WARNING: PGN_AREA substituted for SRC_AREA for %d sources.")'
  SRC_AREA[ind] = PGN_AREA[ind]
endif
S_FILTER  = tag_exist(bt,'S_FILTER') ? strtrim(bt.S_FILTER,2) : replicate('...',num_sources) ; optional STATUS filter specification


;; Check for invalid values in critical properties.
temp = SRC_AREA+SRC_CNTS+NET_CNTS+EXPOSURE+ENERG_LO+ENERG_HI+MSK_RAD+X_CAT+Y_CAT+RA+DEC+THETA
ind = where(~finite(temp), count)
if (count GT 0) then begin
  print, 'ERROR: NaN values found in critical columns (SRC_AREA,SRC_CNTS,NET_CNTS,EXPOSURE,ENERG_LO,ENERG_HI,MSK_RAD,X_CAT,Y_CAT,RA,DEC,THETA) of '+collatefile
  forprint, CATALOG_NAME, SUBSET=ind
  GOTO, FAILURE
endif

temp = SRC_AREA*EXPOSURE*ENERG_LO*ENERG_HI*MSK_RAD*X_CAT*Y_CAT*RA*abs(DEC)*THETA
ind = where(temp LE 0, count)
if (count GT 0) then begin
  print, 'ERROR: zero or negative values found in critical columns (SRC_AREA,EXPOSURE,ENERG_LO,ENERG_HI,MSK_RAD,X_CAT,Y_CAT,RA,DEC,THETA) of '+collatefile
  forprint, CATALOG_NAME, SUBSET=ind
  GOTO, FAILURE
endif

if (~array_equal(strtrim(bt.OBSNAME,2), obsname)) then begin
  print, 'ERROR: the collated photometry in '+collatefile+' appears to come from the wrong observation!'
  GOTO, FAILURE
endif



if keyword_set(reuse_models) then begin
  GOTO, CONSTRUCT_BACKGROUNDS
endif

print, F='(%"\nae_better_backgrounds: ============================================================")' 
print, 'ae_better_backgrounds: Building source models ...'
print, F='(%"ae_better_backgrounds: ============================================================\n")'

;; =====================================================================
;; Read the exposure map which defines the pixelization of the background region.
emap         = readfits(emapfile, emap_header)
extast, emap_header, emap2wcs_astr
emap_col_dim = (size(emap, /DIM))[0]
emap_row_dim = (size(emap, /DIM))[1]

skypix_per_emappix     = psb_xpar( emap_header, 'CDELT1P') ;linear sky pixels per emap pixel




event_header = headfits(evtfile, EXT=1)
DTCOR = psb_xpar( event_header, 'DTCOR')


;; =====================================================================
;; IMPORTANT REMARKS ABOUT PIXELIZATION OF THE SOURCE EXTRACTION REGION

;; It is important to keep in mind that actual extraction regions in AE are
;; polygons with corresponding areas that are estimated carefully.
;; Lots of calculations in this algorithm mimick the background subtraction
;; done in AE which scales the background counts by the ratio of the 
;; extraction polygon area and the background region area,
;; represented by the variables src_exposurearea and bkg_exposurearea.
;; As in AE, these are expressed in units of (skypix**2 s cm**2 count /photon).
;;
;; However at several points this algorithm must integrate a pixelized source
;; model (scaled PSF image) over the extraction region polygon.  
;; The crude method used for this is to sum up the source model pixels that fall
;; inside the region and then scale up by the ratio of the polygon area to the
;; pixelized aperture area.

 
;; =====================================================================
;; Construct a model of each point source by resampling and scaling the PSF.
;; All these models are in units of COUNTS, i.e. the star's flux has been passed through the exposure map.
;; We store only the non-zero portion of these huge sparse arrays.
num_PSF_models    = num_sources
num_streak_models = 0


; Default for is_significant is TRUE, appropriate for streak and user-supplied models.
models = replicate({LABEL:'', is_significant:1B, col_min:0, row_min:0, col_max:-1, row_max:-1, cropped_counts:0.0, counts_in_model:0.0, data_ptr: ptr_new()}, 2*num_sources)

; Fill in labels for each PSF model, and '...' for the extra models we may need to define for streaks.
models.LABEL          = [LABEL                , replicate('...', num_sources)]

; The first time this tool is run on an ObsID background spectra are not available and PROB_NO_SOURCE should be NaN.
; We want to designate such sources as "significant", so that they have a say in the construction of bkg regions.
source_is_significant = ~finite(PROB_NO_SOURCE) OR (PROB_NO_SOURCE LT neighbor_invalid_threshold)
models.is_significant = [source_is_significant, replicate(1B   , num_sources)]

print, total(/INT, source_is_significant), neighbor_invalid_threshold, F='(%"ae_better_backgrounds: %d source models are deemed significant (PROB_NO_SOURCE < %0.3f).")'


; Create two data structures that will model the counts expected from the known point sources,
; including their readout streaks.
; The 2-D array "observation_counts_model_img" is an ObsID-level model of background components (on the emap pixel grid)
; formed by summing the all the models of source PSFs and readout streaks. Units are observed counts. 
; It does NOT include any user-supplied model.
; The ae_better_masking tool is the primary consumer.
observation_counts_model_img     = fltarr(emap_col_dim, emap_row_dim)


for ii = 0L, num_sources-1 do begin
  print, F='(%"\n===================================================================")'
  print, CATALOG_NAME[ii], LABEL[ii], $
        (PHOT_CREATOR[ii] && ~strmatch(PHOT_CREATOR[ii], '*acis_extract*') ? 'from '+PHOT_CREATOR[ii] : ''), F='(%"Source: %s (%s) %s")'
      
  obsdir          = CATALOG_NAME[ii] + '/' + obsname + '/' + extraction_subdir[ii]
  obs_stats_fn    = obsdir + obs_stats_basename

  ; To support faster processing on multiple computers, we allow the observer to run this
  ; model-building part of the code AFTER the MERGE stage of AE has been run for another
  ; obsid.
  ; Thus, we can NOT expect that any of the merged data products (PSF, ARF, or RMF) 
  ; continue to correspond to the single obsid we're working on here, so we directly
  ; access the single-obsid response files we need.
  psf_fn          = CATALOG_NAME[ii] + '/' + obsname + '/' + psf_basename
  streak_fn       = CATALOG_NAME[ii] + '/' + obsname + '/' + streak_basename
  psf_frac_fn     = obsdir + obs_frac_basename

  ; We discarded earlier sources not observed in this ObsId (NUM_OBS EQ 0), so any missing extraction found here is a bug!
  if (~file_test(obs_stats_fn) || ~file_test(psf_fn)) then begin
    message, 'ERROR: cannot find '+obs_stats_fn+' or '+psf_fn
  endif

  if (NET_CNTS[ii] LE 0) then begin
    ; We only allow positive NET_CNTS to avoid negative star models in the algorithm.
    print, 'Skipping source with non-positive NET_CNTS'
    continue
  endif
  
  
  ;; ------------------------------------------------------------------------
  ;; Construct a *model* (in the form of a real-valued counts image) for the light that should have been observed from this source.  
  ;; We are modeling only power from the point source itself (derived from NET_CNTS photometry computed above); not any background.
  ;;
  ;; THE ALGORITHM BELOW ASSUMES THAT THE PSF IMAGE IS ALREADY A MODEL FOR THE DISTRIBUTION OF COUNTS *DETECTED* 
  ;; ON A FINITE, DITHERED ACIS, as opposed to a model of the distribution of photons striking the HRMA focal plane.
  ;; Currently (2009), MARX produces such an observational PSF.  
  ;; Thus, our desired source model is simply the observational PSF image scaled appropriately by NET_CNTS.
  ;; IF in the future we change to using a PSF image that assumes an infinite detector (using the MARX 5 option DetExtendFlag), then we should look back at version 3323 (Jan 2009)
  ;; for more complex code that applies the shape of the emap to the PSF image in order to simulate the distribution of observed counts.
  this_model = models[ii]
  
  ; Read the PSF for this obsid and set all the NaN values to zero to keep total,max,contour, etc. routines happy.
  psf_img = readfits(psf_fn, psf_header, /SILENT)
  
  ind = where(~finite(psf_img), count)
  if (count GT 0) then psf_img[ind] = 0
  
  ; Regrid the PSF to the emap's pixel grid and renormalize to model the source's observed brightness.
  ; We have to be careful with this normalization.
  ; * Our measurement of the source brightness, NET_CNTS, is obtained from a finite aperture.
  ; * We can estimate how many counts should be detected on an infinite detector by applying our aperture correction (PSF fraction value in obs.psffrac) to NET_CNTS.
  ; * The PSF image, and thus the source model we're building here, are finite however.  The total power in our model should reflect the fraction of the infinite PSF which is covered by the foot print of our PSF image (PSF_footprint_fraction below).
  ; * Regridding with hastrom.pro does NOT preserve the power -- we must make sure the regridded image (not the original one) has the desired normalization.
  
  ; First, we compute the fraction of the infinite PSF which is covered by the foot print of our PSF image.
  ; The PSF generator has estimated the integral of the infinite PSF and saved in the keyword PSF_TOTL.
  psf_total = psb_xpar( psf_header, 'PSF_TOTL')
  if (psf_total EQ 0) then begin
    print, "WARNING: obsolete PSFs in "+psf_fn
    PSF_footprint_fraction = 1.0
  endif else begin
    PSF_footprint_fraction = total(psf_img, /DOUBLE) / psf_total
    if (PSF_footprint_fraction GT 1.0001) || (psf_total LE 0) then print, "WARNING: PSF_TOTL keyword in "+psf_fn+" does not make sense!"
  endelse
  
  
  ; AFTER PSF_footprint_fraction is calculated, if desired, smooth the PSF in its finely-binned original form (before interpolation below).
  if keyword_set(smooth_models) then begin
    min_significance = 10 ; 10% noise goal
    
    arcsec_per_psfpixel = psb_xpar( psf_header, 'CDELT2') * 3600
    
    max_radius_arcsec = arcsec_per_psfpixel * max(size( psf_img, /DIMEN ))/10.0 ; arcsec
    
    adaptive_density_2d, psf_img, /ACCEPT_FLOAT_DATA, min_significance, /GAUSSIAN, MAX_NUM_KERNELS=50, MAX_RADIUS=max_radius_arcsec/arcsec_per_psfpixel, /SILENT, psf_img,error_map,radius_map, STDDEV_RATIOS=stddev_ratios
  
    print, minmax(radius_map)*arcsec_per_psfpixel, F='(%"Source model adaptively smoothed with Gaussian kernel; sigma range is [%0.2f:%0.2f] arcsec.")'
    
    if (max(stddev_ratios) GT 1.06) || (min(stddev_ratios) LT 0.94) then $
      print, psf_fn, stddev_ratios, F='(%"WARNING: Smoothing %s changed the standard deviations of the count distributions along the X and Y axes by factors of [%0.3f,%0.3f].")'
    
    if ~keyword_set(build_models_only) then print, "WARNING: /SMOOTH_MODELS is NOT recommended when building background spectra because PSFs are broadened."
  endif
  
  
  ; Regrid the PSF image to the emap's pixel grid.  
  ;psb_xaddpar, psf_header, 'EQUINOX', 2000.0      
  ;psb_xaddpar, emap_header, 'EQUINOX', 2000.0      
  hastrom, psf_img, psf_header, emap_header, MISSING=0 
  
  ; Extract the footprint of the PSF within the array psf_img, which is now large (the same size as the emap).
  ; Eliminate negative pixels introduced by interpolations.
  index_to_point, where(psf_img GT 0), col, row, size(psf_img)
  this_model.col_min         = min(col)
  this_model.row_min         = min(row)
  this_model.col_max         = max(col)
  this_model.row_max         = max(row)
  counts_model = 0 > psf_img[this_model.col_min:this_model.col_max, this_model.row_min:this_model.row_max]
  
  ; Look up the mono-energy of the PSF, and find the corresponding PSF fraction in file obs.psffrac.
  psf_energy = psb_xpar( psf_header, 'ENERGY')
 ;print, psf_energy, F='(%"PSF monoenergy is %0.2f keV")'
  RADIUS50 = psb_xpar( psf_header, 'RADIUS50') ;arcsec
  
  table = mrdfits(psf_frac_fn, 1, /SILENT, STATUS=status)
  if (status NE 0) then message, 'ERROR reading ' + psf_frac_fn
  
  ind = where(abs(table.energy - psf_energy) LT 0.1, count)
  if (count EQ 0) then begin
    print, "ERROR: no entry for the PSF's mono-energy was found in "+psf_frac_fn
    GOTO, FAILURE
  endif
  
  psf_fraction = table[ind[0]].fraction
  
  ; Estimate the number of counts that would be detected on an infinite CCD.
  counts_on_infinite_ccd = NET_CNTS[ii]/psf_fraction
   
  
  ; Scale the model of the source's PSF.
  model_normalization       = counts_on_infinite_ccd *    PSF_footprint_fraction / total(counts_model, /DOUBLE)
  this_model.cropped_counts = counts_on_infinite_ccd * (1-PSF_footprint_fraction) 
 
  if finite(model_normalization) && (model_normalization GT 0) then begin        
    counts_model *= model_normalization
    
    ; Add to ObsID-level model of background components.
    observation_counts_model_img[this_model.col_min:this_model.col_max, this_model.row_min:this_model.row_max] += counts_model
    
    ; Save the cropped counts model.
    this_model.counts_in_model =         total( counts_model, /DOUBLE)
    this_model.data_ptr        = ptr_new(float( counts_model ))
    
    models[ii] = this_model
    
    ; I can't think of any powerful consistency checks to perform on counts_in_model.  It should not be significantly smaller than NET_CNTS.
    if ((this_model.counts_in_model / NET_CNTS[ii]) LT 0.95) then begin 
      print, this_model.counts_in_model, NET_CNTS[ii], F='(%"ERROR! Source model has fewer counts (%0.1f) than extraction (%0.1f).")'
      GOTO, FAILURE
    endif
  endif else begin
    print, 'ERROR: normalization not finite or not positive:', model_normalization
    GOTO, FAILURE
  endelse

  
  
  ;; ------------------------------------------------------------------------
  ;; Construct a model of the source's readout streak, if it's expected to be bright enough to matter.
  counts_in_streak = counts_on_infinite_ccd * (1 - DTCOR)
  
  ; In a typical ObsID (2550) the typical background is ~2E-7 ct/s/skypix.  Let's ignore streaks that are weaker than that.
  streak_density_threshold = (mean(EXPOSURE, /NAN) * 2E-7)  ; ct/skypix
  
  ; To be conservative, we'll assume that the total streak counts (counts_in_streak) are divided among two columns (1" streak), which is 2*1024 sky pixels.
  if ((counts_in_streak/(2*1024.)) GT streak_density_threshold) then begin

    print, counts_in_streak, mean(EXPOSURE, /NAN), F='(%"\nThe surface brightness expected in the readout streak (%0.3g events in %d s spread over ~2 CCD columns) is expected to be significant with respect to a typical FI background (~2E-7 ct/s/skypix).")'
  
    ; Look for an existing streak model file.
    streak_header = headfits(streak_fn, ERRMSG=error)
    streak_found  = ~keyword_set(error)
    
    if (streak_found) then begin
      ; We can re-use an existing source.streak file, ONLY IF THE SOURCE HAS NOT MOVED SIGNIFICANTLY SINCE THAT MODEL WAS BUILT!
      ; The CONSTRUCT stage of AE makes a similar judgement about whether an existing PSF is acceptable to reuse.
      ; We'll tolerate a mispositioning of the steak model up to 0.1 skypix, chosen arbitrarily.
      streak_found = (abs(X_CAT[ii] - psb_xpar( streak_header, 'X_CAT')) LE 0.1) AND $
                     (abs(Y_CAT[ii] - psb_xpar( streak_header, 'Y_CAT')) LE 0.1)
    endif ; existing stread_fn found
  
    ; Build a streak model (image) with arbitrary scaling.
    if streak_found then print, streak_fn, F='(%"\nRe-using streak model %s")'$
                    else ae_streak_model, CATALOG_NAME[ii], obsname, EXTRACTION_NAME=extraction_subdir[ii], TEMPDIR=tempdir
  
    ; Read the STREAK image for this obsid and set all the NaN values to zero to keep total,max,contour, etc. routines happy.
    streak_img = readfits(streak_fn, streak_header, /SILENT)
    
    ind = where(~finite(streak_img), count)
    if (count GT 0) then streak_img[ind] = 0
  
    
    ; If desired, smooth the streak image in its original binning (before hastrom interpolation below).
    if keyword_set(smooth_models) then begin
      min_significance = 10 ; 10% noise goal

      ; We supply a FIELD_MASK to avoid wasting lots of time smoothing the nearly-empty parts of streak_img.
      ; That mask is built to select 99% of the streak power in a (fast) fixed-kernel smooth of the noisy model.
      ; A smoothing kernel similar to the "size" of the PSF seems appropriate.
      arcsec_per_streakpixel = psb_xpar( streak_header, 'CDELT2') * 3600
      blurred_streak_img = smooth(streak_img, 1+ceil(2*RADIUS50/arcsec_per_streakpixel))
      
      blurred_total = total(blurred_streak_img, /DOUBLE)
      blurred_threshold = max(blurred_streak_img) * 1.99
      
      repeat begin
        blurred_threshold /= 2.0
        field_mask = (blurred_streak_img GE blurred_threshold)
      endrep until ( (total( blurred_streak_img[ where(field_mask) ], /DOUBLE) / blurred_total) GT 0.99)

      adaptive_density_2d, streak_img, /ACCEPT_FLOAT_DATA, min_significance, /GAUSSIAN, MAX_NUM_KERNELS=50, MAX_RADIUS=max_radius_arcsec/arcsec_per_streakpixel, FIELD_MASK=field_mask, /SILENT, streak_img,error_map,radius_map

      print, minmax(radius_map)*arcsec_per_streakpixel, F='(%"Streak model adaptively smoothed with Gaussian kernel; sigma =[%0.2f:%0.2f] arcsec.")'
      
      if ~keyword_set(build_models_only) then print, "WARNING: /SMOOTH_MODELS is NOT recommended when building background spectra because PSFs are broadened."
    endif
  
  
    ; Regrid the streak image to the emap's pixel grid.  Eliminate negative pixels introduced by interpolations.
    ; (Normally, the steak image is built on the emap's grid, so hastrom doesn't have to do anything below.)
    ;psb_xaddpar, streak_header, 'EQUINOX', 2000.0      
    ;psb_xaddpar, emap_header, 'EQUINOX', 2000.0      
    hastrom, streak_img, streak_header, emap_header, MISSING=0 
    streak_img >= 0

    ; After re-gridding (by hastrom), all images derived from streak_img are now stale.
    blurred_streak_img = !VALUES.F_NAN
    blurred_total      = !VALUES.F_NAN
    blurred_threshold  = !VALUES.F_NAN
    field_mask         = !VALUES.F_NAN
    error_map          = !VALUES.F_NAN
    radius_map         = !VALUES.F_NAN

    ; Scale streak model to correspond to the source's flux, and add to our streak model
    model_normalization = counts_in_streak / total(streak_img, /DOUBLE)
    
    streak_img *= model_normalization
    
    
    
    ; For efficiency the vast regions in the streak model containing very little power must be cropped, to control the number of sources that will have to query this model to construct their background regions.
    ; That cropping must use a fresh "field mask" (not the one defined above before re-gridding by hastrom).
    arcsec_per_streakpixel = psb_xpar( streak_header, 'CDELT2') * 3600
    blurred_streak_img = smooth(streak_img, 1+ceil(2*RADIUS50/arcsec_per_streakpixel))
    
    blurred_total = total(blurred_streak_img, /DOUBLE)
    blurred_threshold = max(blurred_streak_img) * 1.99
    
    repeat begin
      blurred_threshold /= 2.0
      field_mask = (blurred_streak_img GE blurred_threshold)
    endrep until ( (total( blurred_streak_img[ where(field_mask) ], /DOUBLE) / blurred_total) GT 0.99)

    ind_bright = where(field_mask, count)
  
    if (count GT 1) then begin
      ; Streak models are stored in a variable-length section of the models array, AFTER the PSF models.
      this_model = models[num_PSF_models+num_streak_models]
      
      this_model.LABEL          = models[ii].LABEL + ' streak'
      this_model.is_significant = 1B
      
      index_to_point, ind_bright, col, row, size(streak_img)
      this_model.col_min         = min(col)
      this_model.row_min         = min(row)
      this_model.col_max         = max(col)
      this_model.row_max         = max(row)
      counts_model = streak_img[this_model.col_min:this_model.col_max, this_model.row_min:this_model.row_max]  
      
      ; If the cropped image is only one-dimensional then don't bother saving this model.
      if (size(counts_model, /N_DIM) EQ 2) then begin
        ; Add to ObsID-level model of background components.
        observation_counts_model_img[this_model.col_min:this_model.col_max, this_model.row_min:this_model.row_max] += counts_model
        
        ; Save the cropped counts model.
        this_model.counts_in_model =         total( counts_model, /DOUBLE)
        this_model.data_ptr        = ptr_new(float( counts_model ))
        
        models[num_PSF_models+num_streak_models] = this_model
        num_streak_models++

        print, min(counts_model[where(counts_model GT 0)]), F='(%"  Streak model is cropped at a density of ~%0.6f ct per emap pixel.")'
      endif ; cropped image is 2D
    endif ; cropping by field_mask

  endif ; build streak model  
  
endfor ;ii

num_models = num_PSF_models+num_streak_models




;; Accept the optional model (image) supplied by the observer.
if keyword_set(background_model_filename) then begin
  this_model = models[num_models]
  
  this_model.LABEL           = 'observer-supplied'
  this_model.is_significant  = 1B

  bkg_img = readfits(background_model_filename, bkg_header, /SILENT)

  ; Determine the total power the final model should have.
  power = total(bkg_img, /DOUBLE) 
  
  ; Regrid supplied image onto the emap's pixel grid.
  ; As described above in code that regrids PSF images to build point source models, hastrom.pro does NOT preserve the
  ; total counts in the image.
  ; We must renormalize to the regridded image to match the total counts in the supplied image.

  ;psb_xaddpar, bkg_header, 'EQUINOX', 2000.0      
  hastrom, bkg_img, bkg_header, emap_header, MISSING=0    
  
  ; Extract the footprint of the model in the emap's pixel grid.
  ; Eliminate negative pixels introduced by interpolations.
  index_to_point, where(bkg_img GT 0), col, row, size(bkg_img)
  this_model.col_min         = min(col)
  this_model.row_min         = min(row)
  this_model.col_max         = max(col)
  this_model.row_max         = max(row)
  counts_model = 0 > bkg_img[this_model.col_min:this_model.col_max, this_model.row_min:this_model.row_max]  


  ; Normalize to achieve the proper final power in the model.
  model_normalization = float(power/total(counts_model, /DOUBLE))
help, model_normalization
  
  counts_model *= model_normalization

  ; Save the cropped counts model.
  this_model.counts_in_model =         total( counts_model, /DOUBLE)
  this_model.data_ptr        = ptr_new(float( counts_model ))
  models[num_models++] = this_model
  
  print, power, round(this_model.counts_in_model), F='(%"\r\nObserver-supplied background model contains %d counts; reprojected model contains %d counts.")'
endif

;; Trim unused elements of models array.
models = models[0:num_models-1]


; Save the point source model (in units of counts) of the entire observation.
; Make sure we're saving single-precision images.
observation_counts_model_img = float(observation_counts_model_img) 
header = emap_header
psb_xaddpar, header, 'CROPCNTS', total(models.cropped_counts), '[count] expected point source counts omitted from image'
writefits, model_image_fn, observation_counts_model_img, header

print, model_savefile, F="(%'\r\nSource models, collated table, data image, and emap have been saved to %s.\r\n')"
save, /COMPRESS, emap, emap_header, emap2wcs_astr, emap_col_dim, emap_row_dim, skypix_per_emappix, bt, models, observation_counts_model_img, neighbor_invalid_threshold,  FILE=model_savefile

; Since we chose to re-build source models in this context, we should assume that any savefile built by ae_better_masking is obsolete.
file_delete, /ALLOW_NONEXISTENT, 'ae_better_masking.sav'


;; =====================================================================
;; Construct an in-band image of the data.
print, 'energy band is ', ENERG_LO, ENERG_HI

run_command, string(emapfile, F="(%'get_sky_limits %s verbose=0 precision=3')")
run_command, /QUIET, 'pget get_sky_limits dmfilter', filterspec

if (filterspec EQ '') then message, 'ERROR running get_sky_limits'

; The background estimation must be most accurate for weak sources---the same sources for which AE will be applying a STATUS=0 filter to aggressively clean the event data.
; We will do the same filtering here, so that the "inband_data" array below will be correct for these (majority) weak sources. 
; For the CIAO commands below, which are processing large files, we are careful to put the output file on a local disk (temproot) to avoid a bug that makes CIAO very slow over NFS!
cmd = string(evtfile, 1000*[ENERG_LO,ENERG_HI], inband_events_fn, F="(%'dmcopy ""%s[energy=%6.1f:%7.1f,STATUS=0]"" %s clobber=yes')")
run_command, cmd  

cmd = string(inband_events_fn, filterspec, temp_image_fn, F="(%'dmcopy ""%s[bin %s]"" %s clobber=yes')")
run_command, cmd  
file_copy, /OVERWRITE, temp_image_fn, data_image_fn



if ~keyword_set(skip_residuals) then begin
  inband_data = readfits(data_image_fn)
  
  ; Display an image of the residuals remaining after the point source model image is subtracted from the data image.
  ; This may help the user identify sources missing from the catalog.
  writefits, residual_image_fn, inband_data - observation_counts_model_img, emap_header
  
  cmd = string(target_name, 'ae_better_backgrounds ('+obsname+'): residuals of background model', inband_events_fn, data_image_fn, model_image_fn, residual_image_fn, green_region_fn, F='(%"ds9 -geometry 1400x1600 -title \"%s:%s\" -iconify -colorbar yes  %s -bin factor 4 -log   %s  -log -scale limits 0 2  %s -log -scale limits 0 2  %s  -single -zoom to fit -cmap rainbow -linear -scale limits -0.5 0.5 -regions load all %s -regions select all -regions color grey -regions select none -smooth function gaussian -smooth radius 2 -smooth yes -lock frame wcs >& /dev/null &")')
                                                               
  if show then begin
    run_command, /UNIX, 'egrep "cat|polygon" '+regionfile+'| sed -e "s/DodgerBlue/green/" >! '+green_region_fn
   ;run_command, /UNIX, 'sed -e ''s/DodgerBlue/grey/'' '+green_region_fn+' >! '+ gray_region_fn
  
    run_command, cmd, /QUIET
  
    print, model_image_fn, residual_image_fn, F="(%'\r\nAn iconified ds9 session shows the observed data (upper frames), a model for all the point sources (lower-left frame, %s) and the SMOOTHED residuals (lower-right frame, %s) remaining after the model is subtracted from the data.')"
  endif
endif ;~skip_residuals
                                                 
if keyword_set(build_models_only) then begin
  ; Wait for ds9 session spawned above to load green_region_fn, which is about to be destroyed by the CLEANUP code.
  wait, 30
  GOTO, CLEANUP 
endif


; The null_structure() function used by COLLATE stage should insert '...' when S_FILTER keyword is missing from obs.stats.
ind = where(S_FILTER EQ '...', count)
if (count GT 0) then begin
  print, 'The S_FILTER property appears to be missing for these sources in '+collatefile
  forprint, CATALOG_NAME, SUBSET=ind
  GOTO, FAILURE
endif




CONSTRUCT_BACKGROUNDS:

inband_data = readfits(data_image_fn)

print, F='(%"\nae_better_backgrounds: ============================================================")' 
print, 'ae_better_backgrounds: Building background regions and extracting spectra ...'
print, F='(%"ae_better_backgrounds: ============================================================\n")'


; Construct an initial pixel status map to be used at the onset of processing for each source.
; In this initial map, all pixels are available for use in backgrounds except those with very low exposure.
initial_pixel_status = replicate(available_status, emap_col_dim, emap_row_dim)
ind = where(emap LT max(emap)/10., count)
if (count GT 0) then initial_pixel_status[ind] = off_field_status

if (verbose GE 1) then begin
  ; Wait for ds9 to register with XPA.  
  ; Starting in CIAO 4.0, xpaaccess uses the exit code to return its result.
  ; Thus we can no longer allow run_command to interpret a non-zero exit code as failure.
  repeat begin
    run_command, string(my_ds9, F='(%"xpaaccess %s")'), result, /IGNORE_STATUS, /QUIET
    if (result[0] EQ 'yes') then break
    print, 'waiting for ds9 to come up...'
    wait,3
  endrep until (0)

  ; Display the event data.  
  run_command, /QUIET, string(my_ds9, data_image_fn, F='(%"xpaset -p %s file  %s")')
endif



;; =====================================================================
; Calculate the positions of the emap pixel centers in physical (x,y) coordinates, 
; keeping in mind the 1-based FITS convention vs the 0-based IDL array index convention.
; We cannot use xy2ad.pro/ad2xy.pro for conversions between array index and PHYSICAL (sky) coordinate systems.
crvalP = [psb_xpar( emap_header, 'CRVAL1P'), psb_xpar( emap_header, 'CRVAL2P')]
crpixP = [psb_xpar( emap_header, 'CRPIX1P'), psb_xpar( emap_header, 'CRPIX2P')]
cdeltP = [psb_xpar( emap_header, 'CDELT1P'), psb_xpar( emap_header, 'CDELT2P')]

emap_pixel_x = (findgen(emap_col_dim)+1-crpixP[0])*cdeltP[0] + crvalP[0]
emap_pixel_y = (findgen(emap_row_dim)+1-crpixP[1])*cdeltP[1] + crvalP[1]
make_2d, emap_pixel_x, emap_pixel_y
  
;; =====================================================================
;; Pre-calculate a normalized grid of positions relative to the source where
;; we will create the initial set of background pixel candidates.

; First, a fine grid that spans [-1,1]
N = 6
fine_grid_ii = (indgen(2*N) - (N-0.5)) / (N-0.5)
fine_grid_jj = fine_grid_ii
make_2d, fine_grid_ii, fine_grid_jj
fine_grid_ii = reform(fine_grid_ii, n_elements(fine_grid_ii))
fine_grid_jj = reform(fine_grid_jj, n_elements(fine_grid_jj))

; Also, a coarse grid that goes out much further from the source.
; This is particulary helpful when a background region needs to "hop over" a readout streak.
coarse_grid_ii = [-8,-4,-2,2,4,8]
coarse_grid_jj = coarse_grid_ii
make_2d, coarse_grid_ii, coarse_grid_jj
coarse_grid_ii = reform(coarse_grid_ii, n_elements(coarse_grid_ii))
coarse_grid_jj = reform(coarse_grid_jj, n_elements(coarse_grid_jj))

seed_grid_ii = [fine_grid_ii , coarse_grid_ii ]
seed_grid_jj = [fine_grid_jj , coarse_grid_jj ]



;; =====================================================================
;; Define a background region for each source.


processing_rate = fltarr(num_sources)

;; When /REUSE_MODELS is specified we need to read the catalog supplied to 
;; see which sources are to have backgrounds computed; see comments below.
readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';', NLINES=num_lines

if (num_lines EQ 0) then begin
  print, 'WARNING: no sources found in source list ', srclist_fn
  GOTO, CLEANUP
endif

; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_in_catalog)


if (num_in_catalog EQ 0) then begin
  print, 'WARNING: no sources found in source list ', srclist_fn
  GOTO, CLEANUP
endif

sourcename = sourcename[ind]
print, num_in_catalog, F='(%"\nae_better_backgrounds: building background spectra for %d sources.")'


case (n_elements(min_num_cts)) of
  num_sources:
  1          : min_num_cts = replicate(min_num_cts,num_sources)
  0          : min_num_cts = replicate(          5,num_sources)
  else       : begin
               print, 'ERROR: MIN_NUM_CTS input must be scalar or must match the length of the SRCLIST: ', n_elements(sourcename)
               GOTO, FAILURE
               end
endcase


for ii = 0L, num_sources-1 do begin
  t0 = systime(1)
  ;; ------------------------------------------------------------------------
  ;; When /REUSE_MODELS is specified the observer may supply a sourcelist that
  ;; does not match the sources found in the previously computed collated table (bt),
  ;; for example if he/she wants to construct a background for only one source
  ;; but wants to consider contamination from the whole catalog.
  ;; We are looping (ii) over the collated table so we can clearly index vectors
  ;; like SRC_CNTS, NET_CNTS, etc.
  ;; As we loop, we will skip any source in bt that is not in the sourcelist
  ;; supplied by the observer.
  srclist_ind = where(CATALOG_NAME[ii] EQ sourcename, count)

  if (count EQ 0) then begin
   ;print, 'Skipping source not in list: ', CATALOG_NAME[ii]
    continue
  endif 
  

  if (THETA[ii] LT theta_range[0]) || (THETA[ii] GT theta_range[1]) then begin
    print, 'Skipping source not in THETA_RANGE: ', CATALOG_NAME[ii]
    continue
  endif 
  
  sourcedir            = CATALOG_NAME[ii] + '/' 
  unnamed_src_stats_fn = sourcedir + src_stats_basename
  obsdir               = sourcedir + obsname + '/' + extraction_subdir[ii]
  region_fn            = obsdir + src_region_basename
  bkg_emap_fn          = obsdir + bkg_emap_basename
  bkg_pixels_region_fn = obsdir + bkg_pixels_region_basename
  bkg_events_fn        = obsdir + bkg_events_basename
  bkg_spectrum_fn      = obsdir + bkg_spectrum_basename
  obs_stats_fn         = obsdir + obs_stats_basename
  
  ; We assume that an existing source directory that is a symbolic link should not be written to.
  temp = file_info(sourcedir)
  is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
  if ~is_writable then begin
    print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
    continue
  endif 

  ; Remove any temp files and CIAO parameter files used by the previous source. 
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
    
  run_command, /QUIET, ['pset dmcopy clobber=yes', 'pset dmimgpick clobber=yes', 'pset dmextract clobber=yes']

  print, F='(%"\n===================================================================")'
  print, CATALOG_NAME[ii], LABEL[ii], F='(%"Source: %s (%s)")'  
  
  if (~file_test(obs_stats_fn)) then begin
    print, 'BACKGROUND CONSTRUCTION SKIPPED: source not observed.'
    continue
  endif

  if (verbose GE 2) then begin
    wset, 0
    erase
    wset, 1
    erase
    wset, 2
    erase
  endif
  
  ;; ------------------------------------------------------------------------
  ; Look up allowed range of background normalizations directly from source.stats (not from the collated table) 
  ; since they may have changed since we built the source models and collated table.
  unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)

  if (~keyword_set(error)) then begin
    BKSCL_LO = psb_xpar( unnamed_src_stats, 'BKSCL_LO') ; Smallest allowed bkg scaling.
    BKSCL_GL = psb_xpar( unnamed_src_stats, 'BKSCL_GL') ; Target bkg scaling.
    BKSCL_HI = psb_xpar( unnamed_src_stats, 'BKSCL_HI') ; Largest allowed bkg scaling.
    if (BKSCL_LO LE 0) || (BKSCL_GL LE 0) || (BKSCL_HI LE 0) then begin
      print, 'ERROR: BKSCL_LO,BKSCL_GL,BKSCL_HI are either missing from source.stats or are non-positive!'
      GOTO, FAILURE
    endif
    if ((BKSCL_LO/BKSCL_GL) GT 1.01 || (BKSCL_GL/BKSCL_HI) GT 1.01) then begin
      print, 'ERROR: BKSCL_GL is not between BKSCL_LO and BKSCL_HI!'
      GOTO, FAILURE
    endif
  endif else begin
    print, error
    message, 'ERROR reading '+unnamed_src_stats_fn
  endelse
  
  
  
  
  
  
  ;; For speed we have to apply ContainsPoints() on a sub-region of the emap.
  ae_ds9_to_ciao_regionfile, region_fn, '/dev/null', /IGNORE_BACKGROUND_TAG, POLYGON_X=polygon_x, POLYGON_Y=polygon_y

  bbox_ind = where((emap_pixel_x GE (min(polygon_x)-1)) AND (emap_pixel_y GE (min(polygon_y)-1)) AND $
                   (emap_pixel_x LT (max(polygon_x)+1)) AND (emap_pixel_y LT (max(polygon_y)+1)) )
  
  bbox_pixel_x = emap_pixel_x[bbox_ind]
  bbox_pixel_y = emap_pixel_y[bbox_ind]
  
  ;; Initialize variables used to control the main loop that adds pixels to a background region.
  max_num_bkg_pixels = !VALUES.F_INFINITY 
  is_first_pass      = 1
  search_phase       = 1
  accept_region      = 0
  
  
  ;; Initialize several variables that record properties of the entire set of BKG regions that we evaluate during our search.
  ;; We want these to "span" both passes of the search, if a second pass is needed to rebuild the "reserve" region; thus we initialize them BEFORE the top of the second pass.
  largest_acceptable_region  = {bkg_normalization_nominal: -!VALUES.F_INFINITY} 
  lowest_imbalance_region    = {bkg_normalization_nominal: -!VALUES.F_INFINITY, background_imbalance: !VALUES.F_INFINITY} 
  reserve_region             = {bkg_normalization_nominal: -!VALUES.F_INFINITY, background_imbalance: !VALUES.F_INFINITY, num_bkg_pixels:0L} 
 
  
  ;; ------------------------------------------------------------------------
  ;; This is the re-entry point into the search code for cases where we want to repeat the search in order to
  ;; re-construct the bkg region identified by the "reserve_region" structure.
  ;; The re-entry point must be placed BEFORE any variables used in the search are initialized (e.g. pixel_status image below).
  ;; The stopping criterion in a second pass through the search loop is the max_num_bkg_pixels goal.
  
REPEAT_SEARCH:


  ;; ------------------------------------------------------------------------
  ;; Estimate the integral of the emap over the source aperture.
  o_poly=obj_new('IDLanROI', polygon_x, polygon_y)

  aperture_ind = where((o_poly->ContainsPoints(bbox_pixel_x,bbox_pixel_y) EQ 1), aperture_count)
  if (aperture_count EQ 0) then begin
    print, 'WARNING: extraction region is smaller than one emap pixel.'
    dum = min(sqrt((bbox_pixel_x-X_CAT[ii])^2 + (bbox_pixel_y-Y_CAT[ii])^2),aperture_ind)
    aperture_count = 1
  endif
  
  aperture_ind = bbox_ind[aperture_ind]
  
  ; The exposure-area of the extraction region is not well estimated by the pixelized aperture represented by aperture_ind,
  ; so as in AE we multiply an accurate geometric area of the polygon by the mean emap value; units are (s cm**2 skypixel**2).
  mean_exposure    = mean(emap[aperture_ind], /DOUBLE)
  src_exposurearea = float(SRC_AREA[ii]*mean_exposure) ; Units are skypix**2 s cm**2 count /photon.
  
  ; This correction from the pixelized aperture area to the real extraction region area 
  ; must be used later where we again are trying to integrate something over the extraction region.
  aperture_pixelization_correction = SRC_AREA[ii] / ((skypix_per_emappix)^2 * aperture_count)
;help, aperture_count, aperture_pixelization_correction


  ;; ------------------------------------------------------------------------
  ;; Decide which region (set of pixels) around the current source should be eliminated from 
  ;; consideration for the background region for that source ("masked" status).
  ;; At a minimum it seems obvious we want to eliminate the extraction region itself.
  ;; Since our aperture is pixelated we test whether any of several positions in each pixel
  ;; are interior to the polygon.
  
  ;; We don't really want to apply aggressive masking around the current source because in crowded 
  ;; situations we may eliminate regions that are valuable for the goal of minimizing background bias.
  ;; However we have the additional goal of not including too many counts from the current source 
  ;; to fall in its own background region.
  ;; It's not obvious how to balance these desires.  A simple ad hoc penalty we'll adopt is
  ;; to set the model contamination goal for the current source to zero; then any source counts
  ;; that fall in the background region act as a penalty.  
  ;; This is done in code found a bit later.
  masked_emap = emap
  
  pixel_status = initial_pixel_status ; Either off_field_status or available_status.
  pixel_status[aperture_ind] = masked_status
  masked_emap [aperture_ind] = 0
  
  hs = abs(cdeltP[0]) * 0.3   ; "half-size" of emap pixel, in skypix units
  
  is_inside = (o_poly->ContainsPoints(bbox_pixel_x+hs,bbox_pixel_y+hs) EQ 1) OR $
              (o_poly->ContainsPoints(bbox_pixel_x-hs,bbox_pixel_y+hs) EQ 1) OR $
              (o_poly->ContainsPoints(bbox_pixel_x-hs,bbox_pixel_y-hs) EQ 1) OR $
              (o_poly->ContainsPoints(bbox_pixel_x+hs,bbox_pixel_y-hs) EQ 1)
  
  mask_ind = where(is_inside, mask_count)
  if (mask_count GT 0) then begin
    pixel_status[bbox_ind[mask_ind]] = masked_status
    masked_emap [bbox_ind[mask_ind]] = 0
  endif

  obj_destroy,o_poly 

  
  
  ;; ------------------------------------------------------------------------
  ;; Nominate a set of initial background pixel candidates consisting of two subsets.
  ;; First, we want to nominate a small random subset of the pixels adjacent to masked pixels, since for severe
  ;; crowding those close-in pixels are often the best choices.
  ;; Second we want to nominate a grid of pixels covering a broad surrounding area to give
  ;; the search lots of choices, including some far from the source. 
  ;; By using multiple seeds we allow discontiguous regions, but there's nothing wrong with that.
  
  index_to_point, where(pixel_status EQ masked_status), temp_col, temp_row, size(pixel_status)
  temp_col = [temp_col+1,temp_col+0,temp_col-1,temp_col+0]
  temp_row = [temp_row+0,temp_row+1,temp_row+0,temp_row-1]
  
  ; Calculate the 0-based column/row position of the source.
  ad2xy, RA[ii], DEC[ii], emap2wcs_astr, src_column, src_row
  
  ; Define a region in which candidate pixel "seeds" will be generated.
  seed_radius = 10 > (2 * MSK_RAD[ii] / skypix_per_emappix)    ; units are array indexes      
  
  nominated_candidates     = replicate({col:0, row:0}, n_elements(temp_col)+n_elements(seed_grid_ii))
  nominated_candidates.col = [temp_col,round(src_column + seed_radius * seed_grid_ii)]
  nominated_candidates.row = [temp_row,round(src_row    + seed_radius * seed_grid_jj)]
  num_candidates = 0L
 
;plot, nominated_candidates.col, nominated_candidates.row, PSYM=3, /YNOZ
  
  ;; ------------------------------------------------------------------------
  ;; To speed up the algorithm, define a search domain that is a subset of the full observation.
  ;; We consider only those models that overlap this domain, and we restrict the 
  ;; background region to lie in this domain.
  ;;
  ;; We want this domain to encompass a certain minimum observed area on the sky (1.0x1.0 arcmin^2)
  half_dim = (0.5 * 60) / (arcsec_per_skypixel * skypix_per_emappix)  ; half-width of nominal domain (emap pixel units)
  min_num_active_pixels = (2*half_dim)^2                               ; area of nominal domain

  domain = {col_min:floor(src_column-half_dim) > 1, $
            row_min:floor(src_row   -half_dim) > 1,    $
            col_max: ceil(src_column+half_dim) < (emap_col_dim-2),    $
            row_max: ceil(src_row   +half_dim) < (emap_row_dim-2)}
  step_size = 0
  repeat begin
    domain.col_min -= step_size
    domain.row_min -= step_size
    domain.col_max += step_size
    domain.row_max += step_size
    ; For convenience later, this domain stays 1 pixel away from the emap edges
    domain.col_min >= 1
    domain.row_min >= 1
    domain.col_max <= (emap_col_dim-2)
    domain.row_max <= (emap_row_dim-2)
  
    num_active_pixels = total(/INTEGER, pixel_status[domain.col_min:domain.col_max, domain.row_min:domain.row_max] EQ available_status)
    
    ; Stop if the domain coveres the entire emap (minus the 1-pixel margin).
    if (domain.col_min EQ 1) AND (domain.row_min EQ 1) AND (domain.col_max EQ (emap_col_dim-2)) AND (domain.row_max EQ (emap_row_dim-2)) then break
    
    step_size = 1 
  endrep until (num_active_pixels GE min_num_active_pixels)

; print, num_active_pixels * (arcsec_per_skypixel * skypix_per_emappix / 60)^2, num_active_pixels,$
;         F='(%"Bkg region domain is %4.1f arcmin^2 (%d emap pixels)")'

  ; The set of models we work with consists of:
  ;   "significant" models whose footprints intersect the domain, AND
  ;   the source we're processing (models[ii])!
  model_in_domain = (domain.col_max GE models.col_min) AND $
                    (domain.col_min LE models.col_max) AND $
                    (domain.row_max GE models.row_min) AND $
                    (domain.row_min LE models.row_max) AND models.is_significant
  model_in_domain[ii] = 1
  
  domain_ind    = where(model_in_domain, num_models)
  domain_models = models[domain_ind]
  
  ; Identify the source we're processing in this sub-list of models, and defensively confirm.
  self_index = (where(domain_ind EQ ii))[0] 

  self_model = domain_models[self_index]
  if (self_model.LABEL NE LABEL[ii]) then message, 'Bug in logic!'

  ;; ------------------------------------------------------------------------
  ;; Estimate the contamination from each source model falling within the extraction aperture
  ;; by integrating each source model over the aperture pixels (col,row).
  index_to_point, aperture_ind, col, row, size(emap)
                 
  aperture_model_counts = fltarr(num_models)
  for jj=0L,num_models-1 do begin
    this_model = domain_models[jj]
    ind = where( (col GE this_model.col_min) AND (col LE this_model.col_max) AND $
                 (row GE this_model.row_min) AND (row LE this_model.row_max), count )

    if (count GT 0) then begin
      model_inside_aperture     = (*this_model.data_ptr)[col[ind]-this_model.col_min, row[ind]-this_model.row_min]
      aperture_model_counts[jj] = total(model_inside_aperture,/DOUBLE) * aperture_pixelization_correction
    endif
  endfor ;jj
  
  ; As described earlier, set the model contamination (goal) for the current source to zero; 
  ; then any "self counts" that fall in the background region act as a penalty.  
  aperture_model_counts[self_index] = 0

  aperture_model_counts_total = total(aperture_model_counts)
 
  ;; ================================================================================================
  ;; We need some mechanism for the metric to favor compactness of the background region
  ;; since, lacking any specific information about background components unrelated to neighboring sources,
  ;; it seems the most one can say is that the background in the source aperture has better
  ;; correlation with nearby background pixels than it does with distant ones.
  ;; In other words, background regions should be "local".
  
  ;; To measure the "extent" of a proposed background region, I arbitrarily choose a 
  ;; WEIGHTED MEAN DISTANCE from the source position to the pixels in the bkg region.  
  ;; The weights are the EXPOSURE MAP values at each pixel.
  ;; Let's call this "extent" statistic Davg()---weighted average pixel distance.
  ;; This calculation is very different from a "moment of intertia" calculation, which depends on r^2.
  
  ;; An essential property that Davg() must have is that a pixel near the source is preferred
  ;; over a pixel far away, REGARDLESS OF THE EXPOSURE MAP VALUES AT THOSE TWO LOCATIONS.
  ;; For example, a bkg region consisting of 10 nearby pixels with emap=1 and r=1 must be preferred
  ;; over a bkg region consisting of 1 pixel with emap=10 and r>1.
  ;; I *think* this emap-weighted average pixel distance statistic has this property, but I 
  ;; have not written down a proof.
      
  ;; We do not care about the extent of the background region, Davg(), in absolute terms---it will 
  ;; have to be large when the background density is low, and it can be small when the dominant 
  ;; background component is a neighboring source.
  ;; Instead, we judge the "compactness" of any proposed background region in comparison to the
  ;; circular region that includes the same exposure-area (defined as the integral of the emap
  ;; over the background region).
  ;; I arbitrarily define a positive unitless "compactness_bias" metric as 
  ;;
  ;;  compactness_bias = compactness_gain * (Davg(bkg region) / Davg(circular region)) - 1) > 0
  ;;
  ;; When the bkg region is more compact than the reference circular region, compactness_bias =0.
  ;; When the bkg region and the reference circular region are similarly compact, compactness_bias~=0.
  ;; When the bkg region is has twice the Davg as the reference circular region, compactness_bias~=1.

  ;; NOTE that the nomalization in this metric, Davg(circular region), is a function of the
  ;; exposure-area of the proposed bkg region.
  ;; The SHAPE of this function depends on the emap features (edges, bad columns, etc.) near the source.
  
  ;; The code below tabulates Davg(circular region) as a function of exposure-area.
  ;;

  ;; ------------------------------------------------------------------------
  ;; First, we identify a set of emap pixels that are available for use in the background region,
  ;; and we sort these pixels by their distance from the source position.
  ;; This sorted list of pixels is presented by their 1-D indexes into the emap, stored in
  ;; "sorted_available_index"
  
  ; Compute distance from source to each pixel in the full emap, in units of emap pixels.
  dist_circle, distance_map, [emap_col_dim, emap_row_dim], src_column, src_row
  
  ; Find 1-D index (into full emap array) of pixels in the current search domain (defined above).
  domain_index   = (lindgen(emap_col_dim, emap_row_dim))[domain.col_min:domain.col_max,$
                                                         domain.row_min:domain.row_max]
  
  ; Select the subset of those pixels that are AVAILABLE for use in the background region.
  ; Some pixels (in the source aperture) are marked as not available to be used in the bkg region.
  available_index = domain_index[where(pixel_status[domain_index] EQ available_status, num_available)]

  ; Sort those pixels by distance from the source.
  sorted_available_index = available_index[sort(distance_map[available_index])]

  ;; ------------------------------------------------------------------------
  ;; Each of the pixels identified by sorted_available_index has a distance from the source position,
  ;; and that distance defines a circular region around the source.
  ;; Each of those circular regions has a particular exposure-area enclosed, defined as the integral
  ;; of the emap within the region.
  ;; Each of those circular regions also has a Davg() value, defined earlier as a weighted mean distance.
  ;; Thus, our set of pixels identified by sorted_available_index allows us to tabulate the function
  ;; Davg() as a function of exposure-area enclosed.
  
  Davg_template = {region_exposurearea:0.0D,$ ; Integral of emap over region.
                                              ; Units are skypix**2 s cm**2 count /photon.
                   region_Davg        :0.0D}  ; Weighted average pixel distance.
                                              ; Units are skypix.
  
  Davg_circular = replicate(Davg_template, num_available)
  
  
  ; Units can be confusing, so let's be very careful.
  ; Our inputs are vectors holding two properties of the sorted list of "available" emap pixels.
  
  ; We need the distance between the source position and the pixel, in units of skypix.
  ; A distance_map value of 1 is a distance of skypix_per_emappix [skypix].
  pixel_distance     = distance_map[sorted_available_index] * skypix_per_emappix
  
  ; We need the integral of the emap over that pixel, in units of skypix**2 s cm**2 count /photon.
  ; The geometric area of each emap pixel is skypix_per_emappix^2 [skypix**2].
  pixel_exposurearea =  masked_emap[sorted_available_index] * skypix_per_emappix^2
  
  ; With those vectors, we can calculate properties of the concentric circular regions that correspond 
  ; to each of these sorted pixels.
  
  ; We need "exposure-area" of each region, i.e. the sum of the exposure-areas of the pixels in that region.
  ; Since our set of pixels is sorted by distance, our concentric circular regions are sorted by radii.
  ; The exposure-areas can be calculated by a "cumulative" sum operation.
  Davg_circular.region_exposurearea = total(/CUMULATIVE, /DOUBLE, pixel_exposurearea)
  
  ; We need Davg() for each circular region.  Each Davg() is a weighted average of pixel distances.
  Davg_circular.region_Davg = total(/CUMULATIVE, /DOUBLE, pixel_exposurearea * pixel_distance) /$
                              total(/CUMULATIVE, /DOUBLE, pixel_exposurearea)


  ; Defensively check for obvious problems in the calculations above.
  temp = Davg_circular.region_exposurearea
  if min(temp[1:*]-temp) LT 0 then begin
    message, 'ERROR: Davg_circular.region_exposurearea is not monotonically increasing!'
    GOTO, FAILURE
  endif
  
  if (verbose GE 1) then function_1d, id_circular, Davg_circular.region_Davg, Davg_circular.region_exposurearea/1E10, TIT='Circular Bkg Regions', XTIT='Davg [skypix]', YTIT='exposure-area [1E10 skypix**2 s cm**2 count /photon]'
    ;; Summary:
    ;; Davg_circular[9].exposurearea is the integral of the emap over the 10-nearest pixels that are "available".
    ;; Davg_circular[9].region_Davg is the emap-weighted average of those 10 pixels' distances to the source.
  

  ;; =====================================================================
  ;; Define a data structure to describe an emap pixel that is a candidate to be added to the background region.
  ;; Some of the data types in this structure are chosen for speed in the inner loop later.
  candidate_template = { $
    ;; Some tags record properties of this single pixel.
    ;;   * ('col','row') are array index coordinates of the pixel.
    ;;   * 'pixel_exposurearea' is the integral of the emap over the pixel [skypix**2 s cm**2 count /photon].
    ;;   * 'pixel_distance' is distance from the source position in units of skypix.
    ;;   * 'model_counts' is a vector holding the integral of each stellar model over this pixel 
    ;;     (i.e. how much light each point source dumps into this pixel).
    col:0, row:0, pixel_exposurearea:0D, pixel_distance:0D, model_counts:fltarr(num_models), $
      
    ;; It is convenient to also store here properties of a proposed bkg region consisting of the current region plus this pixel. 
    ;; These must be DOUBLE to avoid accumulating roundoff errors as small increments are made.
    ;;   * 'bkg_emapdistance_sum' (DOUBLE) is the sum of 'pixel_exposurearea*pixel_distance' for all pixels in this proposed region. This is the numerator of the Davg() calculation.  
    ;;     Units are skypix**3 s cm**2 count /photon 
    ;;
    ;;   * 'bkg_exposurearea' (DOUBLE) is the sum of 'pixel_exposurearea' for all pixels in this proposed region.
    ;;     Units are skypix**2 s cm**2 count /photon.
    ;;     This is analogous to Davg_circular.region_exposurearea 
    ;;     This is the denominator of the Davg() calculation.
    ;;
    ;;   * 'bkg_Davg' is the region's average distance from the source position, weighted by the emap.
    ;;     This is analogous to Davg_circular.region_Davg.
    bkg_emapdistance_sum:0D, bkg_exposurearea:0D, bkg_Davg:!VALUES.D_NAN, $
    
    ;;   * Nominal and 'corrected' scaling for the background region (see comments later).
    bkg_scaling_nominal:0.0, bkg_scaling_correction:0.0, bkg_scaling_corrected:0.0, $
    
      ;;   * Total counts expected in the bkg region from point source models.
    bkg_model_counts_total       :0.0, $
    ;;   * 'bkg_model_counts_total' scaled to the extraction aperture. 
    subtracted_model_counts_total:0.0, $
    
    ;;   * inferred flat bkg component counts in bkg region
    bkg_flat_counts       :0.0, $
    ;;   * inferred flat bkg component counts in aperture
    aperture_flat_counts  :0.0, $
    ;;   * 'aperture_flat_counts', with a 'correction' applied (see comments later).
    subtracted_flat_counts:0.0, $
    
    ;;  * metric for how well the proposed bkg region subtracts each of the bkg components
    background_imbalance:0.0, $
    ;;  * metric for the compactness of the proposed bkg region (how 'local' it is).
    compactness_bias:0.0   }


  ; Discard any existing candidate store (full_candidate_store) data structure.
  full_candidate_store = 0  &  dum = temporary(full_candidate_store)

  ;; ------------------------------------------------------------------------
  ;; Iteratively add pixels to the background region until we've found enough counts.
  ;; As a failsafe we stop the loop after n_elements(emap)/10 iterations.
  bkg_data_counts_total  = 0L
  current_region_metric = 0
  
  ; One pixel candiate structure, named R, is used to store several properties of the proposed bkg region:
  ;    bkg_emapdistance_sum, bkg_exposurearea,bkg_Davg. 
  ; Vector cumulative properties, bkg_model_counts and subtracted_model_counts, are stored in variables,
  ; R_bkg_model_counts and R_subtracted_model_counts, to save storage.
  R = candidate_template
  R_bkg_model_counts = fltarr(num_models)  ;It's important to be float, not double, for speed in the inner loop later.



  BKSCL_LO_vote =  !VALUES.F_INFINITY
  BKSCL_HI_vote =  !VALUES.F_INFINITY
  bkg_radius    =  0
  
  num_bkg_pixels = 0L
  while (1) do begin
    ;; ================================================================================================
    ;; Try to add nominated candidate pixels to the candidate store.
    for ll=0L,n_elements(nominated_candidates)-1 do begin
      col = nominated_candidates[ll].col
      row = nominated_candidates[ll].row

      ; Pixels outside the current search domain are excluded.
      if (col LT domain.col_min) || (col GT domain.col_max) ||  $
         (row LT domain.row_min) || (row GT domain.row_max) then continue
      
      ; A pixel that has been masked, accepted into the background region, or already accepted as a candidate is ignored.
      if (pixel_status[col,row] EQ available_status) then begin
;        ; For speed we will discard a nominated candidate pixel if it is adjacent
;        ; to an existing candidate pixel.
;        if (pixel_status[col+1,row  ] EQ candidate_status) || $
;           (pixel_status[col-1,row  ] EQ candidate_status) || $
;           (pixel_status[col  ,row+1] EQ candidate_status) || $
;           (pixel_status[col  ,row-1] EQ candidate_status) then continue
              
        pixel_status[col,row] = candidate_status
        
        ; Add the pixel to the candidate store.
        pixel = candidate_template
        pixel.col = col
        pixel.row = row
        pixel.pixel_exposurearea = skypix_per_emappix^2 * emap[col,row] ; (skypix**2 s cm**2 count /photon)
        pixel.pixel_distance     = sqrt((col - src_column )^2 + (row - src_row)^2) * skypix_per_emappix  ;skypix
        
        ; Look up the number of counts each source model predicts for this pixel.
        ind_models_applicable = where( (col GE domain_models.col_min) AND (col LE domain_models.col_max) AND $
                                       (row GE domain_models.row_min) AND (row LE domain_models.row_max), num_models_applicable )
        
        if (num_models_applicable GT 0) then begin
          model_counts = fltarr(num_models)
        
          for jj=0L,num_models_applicable-1 do begin
            ind        = ind_models_applicable[jj]
            this_model = domain_models[ind]
            
            model_counts[ind] = (*this_model.data_ptr)[col-this_model.col_min, row-this_model.row_min]
          endfor ;jj
          
          pixel.model_counts    = temporary(model_counts)
        endif
        
        
        ; When the candidate store is full, enlarge it by a modest fraction.
        if (num_candidates EQ n_elements(full_candidate_store)) then begin
          temp = replicate(candidate_template, num_candidates + (100 > 0.05*num_candidates))
          
          if (num_candidates GT 0) then temp[0] = full_candidate_store
          full_candidate_store = temporary(temp)
        endif 

        
        ; Add this candidate pixel to the end of the list of active candidates.
        full_candidate_store[num_candidates] = pixel
        num_candidates++
      endif ; creating a new candidate pixel
    endfor ;ll

;   print, num_candidates, (100.0*num_candidates)/n_elements(full_candidate_store), F='(%"candidate store holds %d candidates (%d%% full)")'                 

    if (num_candidates LE 0) then begin
      
      ; Report to the ae_adjust_backscal_range tool that this ObsID cannot make a larger region.
      bkg_radius = !VALUES.F_INFINITY
      
      if acceptable_imbalance then begin
        print, bkg_normalization_nominal, num_bkg_pixels * (arcsec_per_skypixel * skypix_per_emappix / 60)^2, num_bkg_pixels, F='(%"WARNING: search reached hard limits on the size of the background region (nominal normalization =%6.1f; area = %4.1f arcmin^2, i.e. %d emap pixels). Accepting this region ...")' 
      
        accept_region = 1
        break  ;; BREAK FROM THE WHILE-loop
      endif else begin
        print, bkg_normalization_nominal, num_bkg_pixels * (arcsec_per_skypixel * skypix_per_emappix / 60)^2, num_bkg_pixels, F='(%"WARNING: search reached hard limits on the size of the background region (nominal normalization =%6.1f; area = %4.1f arcmin^2, i.e. %d emap pixels). Reverting to the \"reserve region\" found earlier in the search ...")' 
      
        ; Revert to the "reserve region".
        search_phase = 5
        break  ;; BREAK FROM THE WHILE-loop
      endelse
    endif



    ;; ================================================================================================
    ;; Each candidate background pixel corresponds to a proposed background region that's one pixel larger 
    ;; than the current region.
    ;; Calculate the properties of each such proposed background region
    
    ;; ------------------------------------------------------------------------
    ;; For speed, we evaluate only a *random subset* of full_candidate_store on each iteration of this search.
    ;; Recall that the full candidate store is held in full_candidate_store[0:num_candidates-1].
    ;; The vector 'CS_sample_ind' holds the indexes into full_candidate_store that define the random subset.
    ;; The structure array 'CS' holds the random subset.
    ;; Choosing how *many* of the candidates to evaluate is difficult.  The larger the number, the more stable the bkg region becomes (across repeated runs of this tool), but the slower it runs. 
    ;; The speed penalty as max_candidates_to_evaluate is increased is propably non-linear, due to thresholds in the computer hardware, such as the size of the processor cache.
    max_candidates_to_evaluate = 100
    if (num_candidates LE max_candidates_to_evaluate) then begin
      CS_sample_ind = lindgen(num_candidates)
    endif else begin
      CS_sample_ind = (sort(random(num_candidates)))[0:max_candidates_to_evaluate-1]
     ;print, 'random candidates are:', string(CS_sample_ind, F='(%"%d")')
    endelse
    CS = full_candidate_store[CS_sample_ind]
   
    ;; ------------------------------------------------------------------------
    ;; There are a lot of confusing variable names required, and they are different
    ;; from the names used in the manual.
    ;;
    ; We have observed scalar quantities, before and after scaling:
    ; bkg_data_counts_total   = N   : counts observed in bkg region
    ;
    ; We have modeled vector quantities with num_models elements, before and after scaling.
    ; aperture_model_counts   : contaminate counts predicted by models in src aperture
    ;                           Sum of this vector is aperture_model_counts_total = Bp
    ; bkg_model_counts        : contaminate counts predicted by models in bkg region  
    ;                           Sum of this vector is this.bkg_model_counts_total = Np
    ; subtracted_model_counts : bkg_model_counts with bkg scaling applied 
    ;                           Sum of this vector is subtracted_model_counts_total = Np * S 
    ;    
    ; We have some derived quanties.
    ; bkg_flat_counts         = Nf     : inferred flat bkg component counts in bkg region 
    ; aperture_flat_counts    = Bf     : inferred flat bkg component counts in aperture
    ; photometric_bias        = DELTAp : Bp - Np*S
    ; bkg_scaling_nominal     = S    : background scaling, nominal
    ; bkg_scaling_correction  = c    : correction to scaling
    ; bkg_scaling_corrected   =      : background scaling, corrected
    ;                                                        
    ;                   
    ; Quantities associated with the current bkg region use no prefix in their name; temporary
    ; updates to these quantities associated with the candidate region at hand use the "this_" prefix.

    ;; ------------------------------------------------------------------------
    ;; For each pixel that is a candiate for acceptance into the background region, 
    ;; evaluate the region metric we would find if we accepted that pixel.
    ;; EFFICIENCY INSIDE THIS LOOP IS CRITICAL TO PERFORMANCE!!
    ;; WE'VE OMITTED THE /DOUBLE OPTION TO THE TOTAL() CALLS AND USED SOME CODE
    ;; CONSTRUCTS THAT GAIN SPEED BUT LOSE CLARITY.
    ;; When thinking about execution speed, keep in mind that the following variables
    ;; are vectors with num_models elements:
    ;;   aperture_model_counts
    ;;   bkg_model_counts, this.model_counts, this_bkg_model_counts
    ;;   this_subtracted_model_counts
    ;;   photometric_bias
    
;t1=systime(1)


; In debugger, verify there are no type conversions being done (e.g. involving constants) 
; and that there are no needless
; DOUBLE computations (although I do not know if that actually matters for speed).
; 
; Look for intermediate vars to eliminate.
; Look at parentheses.
; Look for constants to pre-compute.
; 
; Think about cache memory usage, and whether use of temporary() can free up memory.

    ; Several global cumulative quantites are incremented/updated here for the candidate pixel
    ; we're considering, and stored for later retrieval if we choose this candidate.
    ; IT IS VERY IMPORTANT, however, that the total number of counts OBSERVED in the background,
    ; bkg_data_counts_total, is NOT updated before this candidate is judged.
    ; We must evaluate the candidates based solely on our MODELS of the contaminating
    ; background components, not considering what DATA the candidate pixel happens to contain.
    ; There are cases where all enlargements of the region are bad, and if we consider what data
    ; lies in the candidates then the algorithm will tend to favor pixels with no data, which
    ; introduces a horrible bias to our search.
    CS.bkg_emapdistance_sum = (R.bkg_emapdistance_sum + CS.pixel_exposurearea*CS.pixel_distance)
      ; Units are skypix**3 s cm**2 count /photon.
    CS.bkg_exposurearea = (R.bkg_exposurearea + CS.pixel_exposurearea) 
      ; Units are skypix**2 s cm**2 count /photon.
    CS.bkg_Davg = CS.bkg_emapdistance_sum / CS.bkg_exposurearea
      ; Units are skypix.
      

    ; Compute the exposure-area ratio, i.e. the nominal background scaling.
    ; Be sure that both terms are in units of (s cm**2 skypixel**2).
    CS.bkg_scaling_nominal      = src_exposurearea / float(CS.bkg_exposurearea)  

    ; Estimate how many counts from each contaminating source are statistically expected 
    ; to be in the proposed background region, and how many are expected to be subtracted.
    ; Note that one of these array elements represents "self", i.e. how much light from 
    ; the current source will be (mistakenly) subtracted.
    nx = N_elements(R_bkg_model_counts)
    ny = N_elements(CS)
    bkg_model_counts_2d              = rebin(reform(R_bkg_model_counts,    nx, 1,/OVERWRITE), nx, ny, /SAMPLE) + CS.model_counts
    CS.bkg_model_counts_total        = total(bkg_model_counts_2d,1)
    
    subtracted_model_counts_2d       = rebin(reform(CS.bkg_scaling_nominal,        1, ny,/OVERWRITE), nx, ny, /SAMPLE) * temporary(bkg_model_counts_2d)
    
    aperture_model_counts_2d         = rebin(reform(aperture_model_counts, nx, 1,/OVERWRITE), nx, ny, /SAMPLE)
    

    ;; ------------------------------------------------------------------------
    ;; Evaluate the quality of every proposed bkg region by computing some metrics.
    
    ; Compute a background imbalance metric that judges how fairly all the background components are represented
    ; in the background regions we are considering.
    ; This is tricky, and vital to the success of the search.
    ; Many approaches are possible.  We've tried computing this metric after the background has been
    ; adjusted to remove photometric bias, but found very unsatisfactory results (e.g. the algorithm
    ; sometimes latches onto power from distant sources, which it then has to scale down.)
    ;
    ; Thus, we choose to compute the metric using the nominal scaling.
    
    ; The denominator of the metric (its normalization) turns out to be problematic.
    ; (We need a normalization in order to reasonably combine this metric with the compactness metric.)
    ; * It turns out that it must be a constant (i.e. not varying between candidates) in order to avoid pathological behavior in the search.
    ; * If we allow the normalization to become zero, then the metric becomes infinity for all candidates and we lose
    ;   our power to guide the search.
    ;   Thus, we impose a floor of 1.0 in the normalizations below.
    CS.background_imbalance = total(abs(temporary(  aperture_model_counts_2d) - $
                                        temporary(subtracted_model_counts_2d)),1) / $
                              (1 > 2.0 * aperture_model_counts_total)


    ; Compute a compactness metric that judges the compactness of each background regions we are considering.
    
    ; For EACH proposed background region, find the reference circular region that has a similar exposurearea.
    ; Defensively check for obvious problems.
    Davg_circular_index = value_locate ( Davg_circular.region_exposurearea, CS.bkg_exposurearea ) > 0
    
;    if ~almost_equal(Davg_circular[Davg_circular_index].region_exposurearea, TOLERANCE=CS.bkg_exposurearea/100.,$
;                                                        CS.bkg_exposurearea) then begin
;      message, 'ERROR: CS.bkg_exposurearea is outside the range of Davg_circular.region_exposurearea!'
;      GOTO, FAILURE
;    endif
    
    
    ; As desribed in earlier comments, this metric is the ratio between two Davg() statistics.
    CS.compactness_bias = (float(CS.bkg_Davg) / Davg_circular[Davg_circular_index].region_Davg) - 1.0
    ; We choose to apply the fudge factor compactness_gain here, so that reported compactness values reflect this adjustment.
    CS.compactness_bias *= compactness_gain
    ; Make sure the metric non-negative so that the region_metric_slope calculation later is ok.
    CS.compactness_bias >= 0.0
    
    
    
    ;; ================================================================================================
    ;; Accept one of the proposed background regions.
    
    ;; ------------------------------------------------------------------------
    ;; Choose which candidate pixel should be accepted.  We try to strike a compromise between the goals of
    ;; minimum bias metric and compactness of the background region by including a compactness penalty in
    ;; the metric.
    ;;
    ;; We tried and rejected a scheme where the set of "good enough" metrics was defined relative to the minimum
    ;; metric in hand; this allowed the metric to creep up with each iteration unchecked.
    ;; 
    ;; We tried and rejected a scheme where we define an absolute range of "good enough" metrics; if at some point in our search
    ;; the min metric is forced out of this range, then distance will suddenly stop playing a role in 
    ;; the search.  This seems undesirable.
    ;;
    ;; Of course if the observer knows something about the structure of the background not related to
    ;; point sources, then supplying a map of that will drive the search to correctly subtract it.

    ;; If we simply accept the candidate pixel with the smallest new metric, then in situations 
    ;; where every candidate is increasing the metric we will sometimes choose candiates with low
    ;; exposure values.  This is short sighted---we're taking small steps towards badness even if
    ;; a larger step might result in a smaller RATE towards badness.
    ;; Thus, when all candidates are increasing the metric we will prefer the candidate with the smallest *rate* of metric increase, with respect to exposure!
    
    ; Combine the spectral quality and compactness metrics we're trying to simultaneously minimize.
    region_metric = (CS.background_imbalance + CS.compactness_bias) 

    region_metric_change = region_metric - current_region_metric
    region_metric_slope  = region_metric_change / float(CS.pixel_exposurearea)
    
    if            (min(region_metric_change, accept_ind) LE 0) then begin
      ; We can improve the bkg region (achieve a smaller region_metric), so let's choose the candidate that achieves the smallest metric.
    
    endif else begin
      ; All candidates are increasing the metric; we prefer the candidate with the smallest *rate* of metric increase, with respect to exposure
      best_slope = min(region_metric_slope, accept_ind)
    endelse
    
    
    current_region_metric = region_metric[accept_ind]
    
    if (verbose GE 2) then begin
      ; Define a "postage stamp" region around the source, for some VERBOSE displays we show.
      candidate_col = CS.col
      candidate_row = CS.row

      ps_cmin = min( candidate_col, MAX=ps_cmax)
      ps_rmin = min( candidate_row, MAX=ps_rmax)

      ; We put this display code here because we want CS and region_metric_slope to be intact.
      
     ;metric = hist_equal(-CS.background_imbalance, TOP=ncolors-100) + 100
      metric = hist_equal(-region_metric_slope, TOP=ncolors-100) + 100
                    
      metric_map = bytarr(emap_col_dim, emap_row_dim)
      metric_map[candidate_col, candidate_row] = metric
      
      ps_to_show = metric_map[ps_cmin:ps_cmax,ps_rmin:ps_rmax] 
      wset, 0
      tv, rebin(ps_to_show, 4*(ps_cmax-ps_cmin+1), 4*(ps_rmax-ps_rmin+1), /SAMPLE)
    endif
    

    ;; Accept the chosen pixel into the background region.
    ;; The running sum R_bkg_model_counts is maintained outside the candidate structures.
    R = CS[accept_ind]
    R_bkg_model_counts       += R.model_counts

    if ((pixel_status[R.col,R.row]) NE candidate_status) then message, 'BUG: the same pixel was accepted twice!'
         pixel_status[R.col,R.row] = background_status

    ;; As described earlier, the total observed counts in the background, bkg_data_counts_total,
    ;; must NOT be updated in the inner loop prior to evaluation of the candidate.
    ;; Thus we do that here, when the candidate is accepted.
    bkg_data_counts_total    += inband_data[R.col, R.row]

    num_bkg_pixels++


    ;; Find the chosen pixel, R, in the full candiate store.
    ;; 'Remove' it, by filling its slot with the last active candidate in full_candidate_store.
    full_accept_ind = CS_sample_ind[accept_ind]

    full_candidate_store[full_accept_ind] = full_candidate_store[num_candidates-1]
    num_candidates-- 

    
    ;; Nominate new candiates which are adjacent to the background pixel just accepted.
    if (n_elements(nominated_candidates) NE 4) then $
      nominated_candidates = replicate(nominated_candidates[0], 4)
    nominated_candidates.col = (R.col + [1,0,-1,0])
    nominated_candidates.row = (R.row + [0,1,0,-1])

    
    ;; ------------------------------------------------------------------------
    ;; Correct the background scaling (see derivation in the AE manual) to eliminate photometric bias.
    
    ; We need below the subtracted total if NOMINAL scaling were used.
    R.subtracted_model_counts_total = R.bkg_model_counts_total * R.bkg_scaling_nominal

    ; Infer how many observed counts in the bkg region are not attributable to our models, i.e.
    ; are from a "flat" component.
    R.bkg_flat_counts      = 0.0 > (bkg_data_counts_total - R.bkg_model_counts_total)
    R.aperture_flat_counts = R.bkg_flat_counts * R.bkg_scaling_nominal

    if (bkg_data_counts_total LE 0) then begin
      ; When there is no observed background data then we cannot compute any "correction" to the background scaling
      R.bkg_scaling_correction = 1.0

    endif else begin
      ; Compute a "photometric bias" with respect to the modeled contaminating sources, 
      ; i.e. the total counts we think are in the aperture minus the total counts we 
      ; think will be subtracted by the proposed region, assuming nominal scaling.
      photometric_bias          = aperture_model_counts_total - R.subtracted_model_counts_total
      
      ; We have a complication when the models show zero counts expected in the aperture
      ; (aperture_model_counts_total EQ 0) and we estimate no flat component (bkg_flat_counts EQ 0).
      ; In this case a bkg region containing any model counts must be scaled to zero in order to 
      ; eliminate photometric bias.
      ; We avoid this silly result by placing an arbitrary lower limit on the scaling.
      ; We also place an arbitrary upper limit to be conservative.
      R.bkg_scaling_correction = 0.1 > (1.0 + (temporary(photometric_bias) $
                                                / $
                                                ((R.bkg_model_counts_total+R.bkg_flat_counts) * R.bkg_scaling_nominal))) $
                                     < 10.0
    endelse ; (bkg_data_counts_total GT 0)
 
    
    ;; ------------------------------------------------------------------------
    ;; Apply the correction to the scaling, and recompute quantities that depend on the scaling.
    R.bkg_scaling_corrected    = R.bkg_scaling_nominal * R.bkg_scaling_correction
    bkg_exposurearea_corrected = R.bkg_exposurearea    / R.bkg_scaling_correction
    
    ; The quantity R.aperture_flat_counts is NOT recomputed; it is a function of the NOMINAL scaling!

    ; The "flat" component would, by definition, be correctly subtracted if we used the nominal scaling.
    ; Thus, the actual flat component we will be subtracting is related to what we wanted to 
    ; subtract by the bkg_scaling correction we have adopted.    
    R.subtracted_flat_counts        = R.aperture_flat_counts * R.bkg_scaling_correction
    
    ;; Some of the vector quantities that were computed earlier for the winning
    ;; candiate were not saved in the candidate structure (for efficiency reasons).
    ;; These must be recomputed here.
    R_subtracted_model_counts       = R_bkg_model_counts       * R.bkg_scaling_corrected
    R.subtracted_model_counts_total = R.bkg_model_counts_total * R.bkg_scaling_corrected
    
    
    ;; ------------------------------------------------------------------------
    ;; Evaluate the expected (modeled) quality of the rescaled background region, 
    ;; i.e. how well it represents all the bkg components.
   
    ; Compute a background imbalance metric as simply the fraction of observed bkg data that we think
    ; comes from the "wrong" background component.
    ; In the expressions below we compute the biases (model - subtracted) for each bkg component, including
    ; the flat component, then sum the absolute values of those biases, and then divide by two to handle
    ; double counting, and then normalize by the total counts expected in the aperture.
    R.background_imbalance = (       (abs(R.aperture_flat_counts  - R.subtracted_flat_counts ) $
                              + total(abs(  aperture_model_counts - R_subtracted_model_counts),1))) $
                             / $
                             (1 > 2.0 * ( R.aperture_flat_counts + aperture_model_counts_total)) 
                   
    
    ;; Print some status.
    if (verbose GE 2) then begin
      ps_to_show = pixel_status[ps_cmin:ps_cmax,ps_rmin:ps_rmax] > masked_status
      wset, 1
      tvscl, rebin(ps_to_show, 4*(ps_cmax-ps_cmin+1), 4*(ps_rmax-ps_rmin+1), /SAMPLE)
                                       
      if (verbose GE 3) then begin
        ; Save and plot the three terms of the metric:
        term1[num_bkg_pixels] = R.bkg_scaling_corrected
        term2[num_bkg_pixels] = R.background_imbalance
        term3[num_bkg_pixels] = aperture_model_counts_total - R.subtracted_model_counts_total
        term4[num_bkg_pixels] = R.COMPACTNESS_BIAS
        wset, 2
        ind_min = 0 > (num_bkg_pixels-200)
        y1 = term1[ind_min:num_bkg_pixels]
        y2 = term2[ind_min:num_bkg_pixels]
        y3 = term3[ind_min:num_bkg_pixels]
        y4 = term4[ind_min:num_bkg_pixels]
        plot,  y1, PSYM=1, YRANGE=minmax([y1,y2,y3]) ; +        is scaling   
        oplot, y2, PSYM=2                            ; *        is background imbalance
        oplot, y3, PSYM=4                            ; diamond  is photometric bias
        oplot, y4, PSYM=5                            ; triangle is compactness bias

        print, bkg_data_counts_total, F='(%"\n%d counts in bkg region")'
        if (verbose GE 4) then begin
          flag = (abs(aperture_model_counts) GT 0.1) OR (abs(R_subtracted_model_counts) GT 0.1)
          ;flag[self_index] = 0
          ind = where(flag, count)
          if (count GT 0) then begin
            forprint, domain_models.LABEL, aperture_model_counts - R_subtracted_model_counts, SUBSET=ind, F='(%"%6s %7.2f")'
          endif
        endif
        print
        print, R.bkg_scaling_correction, F='(%"scaling correction  =%5.2f")'
        print, R.background_imbalance,   F='(%"background imbalance=%5.2f")'
        print, R.compactness_bias,       F='(%"compactness metric  =%5.2f")'
    ;   print, 'accepted metric:', (current_region_metric)
    ;   stop
      endif ;(verbose GE 3)
    endif ;(verbose GE 2)
 

    ;; ================================================================================================
    ;; Decide whether the new background region is acceptable, or the search should continue.
    
    ;; Evaluate the stopping criteria for defining the background region (shown in block comment at top of this tool).
    bkg_normalization_nominal   = 1.0 / R.bkg_scaling_nominal
    bkg_normalization_corrected = 1.0 / R.bkg_scaling_corrected
    
    ; To catch infinite looping we look for NaN values used in the stopping criteria.
    vital_values = [bkg_normalization_nominal, R.background_imbalance, bkg_data_counts_total]
    if (~array_equal(finite(vital_values),1)) then begin
      message, 'ERROR: NaN value found in stopping criteria'
      GOTO, FAILURE
    endif
  
    
    
    ;; ------------------------------------------------------------------------
    ;; Collect information to be used later to cast "votes" for adjustments to BKSCL_LO and BKSCL_HI, 
    ;; and keep track of the "reserve region".
    
    reached_BKSCL_LO     = (bkg_normalization_nominal      GE BKSCL_LO)
    reached_BKSCL_GL     = (bkg_normalization_nominal      GE BKSCL_GL)
    reached_BKSCL_HI     = (bkg_normalization_nominal      GE BKSCL_HI)
    reached_min_num_cts  = (bkg_data_counts_total  GE min_num_cts[ii])
    acceptable_imbalance = (R.background_imbalance LE background_imbalance_threshold)
    
    ; For the BKSCL_HI vote, we must record the smallest bkg_normalization_nominal value that achieved our MIN_NUM_CTS requirement.
    if reached_min_num_cts then BKSCL_HI_vote <= bkg_normalization_nominal
    
    
    ; Maintain our "reserve region"---the bkg region to adopt if the nominal search fails to achieve all its goals.
    if (reached_BKSCL_LO && reached_min_num_cts) then begin
      ; We have achived the MIN_NUM_CTS and BKSCL_LO goals, so start keeping track of the region that has the lowest background imbalance.
    
      ; It's important to realize that background imbalance is NOT a monotonic metric; it will normally tend to oscillate 
      ; around zero as the algorithm proceeds because enlargement of the bkg region is quantized.
      ; In cases where the region completely covers a neighboring source background imbalance will tend to start growing since
      ; we have "run out" of samples for that neighbor.
      ; At any point in the search you can't really know if a background imbalance violation is going to get better or worse if you proceed.
      ; Thus, we maintain a "reserve" marker in the sequence of bkg regions built by the search that indicates the bkg region 
      ; that we would prefer IF the search produces a region with failing background imbalance.
      if (R.background_imbalance LE reserve_region.background_imbalance) then begin
        ; The bkg region is in the allowed scaling range, and is thus a candidate for our "reserve region".
        ; Reserve it if it's as good as or better than what we've reserved so far.
        reserve_region.num_bkg_pixels       = num_bkg_pixels
        reserve_region.background_imbalance = R.background_imbalance
        reserve_region.bkg_normalization_nominal    = bkg_normalization_nominal
      endif 
    endif ; "reserve region"
    
    

    ;; Among bkg regions that achieve the MIN_NUM_CTS requirement, keep track of the behavior of the background imbalance metric as we
    ; proceeded through the search so that we can cast a "vote" for the global BACKSCAL range that constrains all the extractions of this source.
    if reached_min_num_cts then begin
      ; Among the regions with acceptable imbalance, keep track of the one with the largest normalization.
      if acceptable_imbalance then begin
        ; The current bkg region is acceptable, so update largest_acceptable_region.
        if (bkg_normalization_nominal GT largest_acceptable_region.bkg_normalization_nominal ) then begin
          largest_acceptable_region.bkg_normalization_nominal = bkg_normalization_nominal
        endif
      endif ; acceptable_imbalance
      
      
      ; Among the regions with unacceptable imbalance, keep track of the one with the smallest background imbalance.
      if ~acceptable_imbalance then begin
        ; The current bkg region is unacceptable, so update lowest_imbalance_region.
        if (R.background_imbalance LE lowest_imbalance_region.background_imbalance) then begin
          lowest_imbalance_region.background_imbalance = R.background_imbalance
          lowest_imbalance_region.bkg_normalization_nominal    = bkg_normalization_nominal
        endif
      endif ; ~acceptable_imbalance 
      
    endif ; reached_min_num_cts
    
    
    ;; ------------------------------------------------------------------------
    ;; Implement the logic of the search algorithm described in the header comment.

    if            (search_phase EQ 1) then begin
      ; Phase 1 is looking for MIN_NUM_CTS and BKSCL_LO
      if reached_BKSCL_LO && reached_min_num_cts then search_phase = 2
      
    endif else if (search_phase EQ 2) then begin
      ; The 'else if' construct here is designed to force one iteration of the loop while in Phase 2.  On the iteration when the Phase 1->2 transition occurs (above), this block is skipped.  The next iteration run as Phase 2, and then this block moves us to Phase 3.

      ; Phase 2.5 is a one-time test for statistically significant negative photometry.
      search_phase = 2.5
      
      src_cnts_error        = (1 + sqrt(SRC_CNTS[ii]          + 0.75))
      bkg_subtraction_error = (1 + sqrt(bkg_data_counts_total + 0.75)) * R.bkg_scaling_corrected

      
      SNR = (SRC_CNTS[ii]     - bkg_data_counts_total*R.bkg_scaling_corrected) /$
            (src_cnts_error^2 + bkg_subtraction_error^2)  

      if (SNR LT -3) then begin
        print, SNR, F='(%"Search aborted because the photometry of this extraction is significantly negative (SNR = %0.1f).")'  
        break  ;; BREAK FROM THE WHILE-loop
      endif
      
      ; The loop is not executed under phase 2.5; we move directly to phase 3.  
      search_phase = 3
    endif
    
    
    if (search_phase EQ 3) then begin
      ; Phase 3 is looking for BKSCL_GL
      if reached_BKSCL_GL then search_phase = 4
    endif
    
    if (search_phase EQ 4) then begin
      ; Phase 4 is looking for an acceptable BACKGROUND_IMBALANCE or for BKSCL_HI
      if acceptable_imbalance then begin
        ; condition 4A
        accept_region = 1
        break  ;; BREAK FROM THE WHILE-loop
      endif else if reached_BKSCL_HI then begin
        ; condition 4B
        print, R.background_imbalance, bkg_normalization_nominal, num_bkg_pixels * (arcsec_per_skypixel * skypix_per_emappix / 60)^2, num_bkg_pixels, F='(%"WARNING: search reached BKSCL_HI with an unacceptable background imbalance (%6.2f).  (nominal normalization =%6.1f; area = %4.1f arcmin^2, i.e. %d emap pixels). Reverting to the \"reserve region\" found earlier in the search ...")' 

        search_phase = 5
        break  ;; BREAK FROM THE WHILE-loop
      endif
    endif
    
    
    ; If the bkg region has grown to max_num_bkg_pixels, then break out of the loop.  This mechanism is currently used only in Phase 5, to stop a retry of the search when it has rebuilt the 'reserve region'
    if (num_bkg_pixels GE max_num_bkg_pixels) then break  ;; BREAK FROM THE WHILE-loop

  endwhile ; loop adding pixels to background region

  
  

REGION_COMPLETE:
  ;; ================================================================================================
  ;; Save the accepted background region, print a summary, etc.
  
  
  ;; There are several ways we can "break" or "GOTO" out of the main while loop above (in which each iteration adds a pixel to the bkg region).

  ;; One algorithm exception is implemented:
  ;;   1. We can run out of candidate pixels (num_candidates LE 0).
  ;; We accept the current region if acceptable_imbalance is true, or we move to Phase 5 otherwise
  
  ;; In Phase 2.5 we can abort the search after determining that this extraction is unlikely to participate in a merge.

  ;; We leave Phase 4 via the "A" option---the current region satisfies all goals (the nominal case).
  
  ;; We leave Phase 4 via the "B" option---the search reached BKSCL_HI and we have to revert to the "reserve region".
  
  
  if accept_region then print, 'A region meeting all goals was accepted.'
  
  if is_first_pass && (search_phase EQ 5) then begin
    ; Negate is_first_pass so that we don't get here again.
    is_first_pass = 0
    
    if (reserve_region.num_bkg_pixels GT 0) then begin
      ; Restart the search and arrange for it to stop when the 'reserve region' has been rebuilt.
      ; Change hard endpoint of loop to correspond to our reserved region, then restart loop.
      max_num_bkg_pixels = reserve_region.num_bkg_pixels
      GOTO, REPEAT_SEARCH
    endif else print, 'No "reserve region" is available; accepting the region in-hand.' 
  endif
  
  
  
  if ~reached_min_num_cts then print, CATALOG_NAME[ii], LABEL[ii], min_num_cts[ii], F='(%"WARNING! %s (%s): adopted region did not achieve the MIN_NUM_CNTS goal (%d).")'  
  
  if ~reached_BKSCL_LO then print, CATALOG_NAME[ii], LABEL[ii], bkg_normalization_nominal, BKSCL_LO, F='(%"WARNING! %s (%s): adopted region has nominal BACKSCAL (%0.1f) smaller than BKSCL_LO (%0.1f).")'  

  
  if ~acceptable_imbalance then print, CATALOG_NAME[ii], LABEL[ii], background_imbalance_threshold, F='(%"WARNING! %s (%s): adopted region did not achieve the BACKGROUND_IMBALANCE_THRESHOLD goal (%0.1f).")'  

  if ~finite(R.bkg_scaling_corrected) then begin
    print, 'ERROR: R.bkg_scaling_corrected is not finite.'
    GOTO, FAILURE
  endif

  
  
  ; Compute the significance of the observed SRC_CNTS as a disproof of the "null hypothesis" which is 
  ; that there is no source, i.e. that all the observed counts are background.  
  ; We use equation A7 from Weisskopf 2006 (astro-ph/0609585):
  ; Note that 
  ;     bkg_normalization_corrected = 1.0 / bkg_scaling_corrected
  ; so the term R.bkg_scaling_corrected / (R.bkg_scaling_corrected+1D) the binomial() call below is equivalent to the form
  ;     1D/(1D + BACKSCAL)
  ; that is found in computations of PROB_NO_SOURCE in other parts of AE.
  
  ; As of IDL v8.5.1, binomial() has significant flaws and frequently produces very wrong results.
  ; See email to Harris Inc. on Oct 22, 2016.
  ; It also behaves very badly if GAUSSIAN=0 is (innocently) supplied (IDL bug report 69655).
  ; Prior to v8.5.1, binomial() could not handle inputs larger than 32767 (IDL bug report 69442).
  ; We use instead a simple and reliable algorithm recommended by Numerical Recipes (Chapter 6.4).
  
  PROB_NO_SOURCE_binomial = binomial_nr(SRC_CNTS[ii], $
                                        SRC_CNTS[ii] + bkg_data_counts_total, $
                                        R.bkg_scaling_corrected / (R.bkg_scaling_corrected+1D) ) > 0

  if ~finite(PROB_NO_SOURCE_binomial) then message, 'ERROR: PROB_NO_SOURCE is not finite.'
  
; PROB_NO_SOURCE_poisson  = (1 - poisson_distribution(bkg_data_counts_total*R.bkg_scaling_corrected, SRC_CNTS[ii] - 1)) > 0
    
  dt = systime(1)-t0
  print, num_bkg_pixels, dt, F='(%"Accepted %6d background pixels in %5d seconds.")' 
  processing_rate[ii] = num_bkg_pixels/dt
  
  
  ;; Summarize the quality of the background region.
  src_cnts_error        = (1 + sqrt(SRC_CNTS[ii]          + 0.75))
  bkg_subtraction_error = (1 + sqrt(bkg_data_counts_total + 0.75)) * R.bkg_scaling_corrected

  ; Some of the vector quantities that were computed earlier for the winning
  ; candiate were not saved in the candidate structure (for efficiency reasons).
  ; These must be recomputed here.
  R_subtracted_model_counts = R_bkg_model_counts * R.bkg_scaling_corrected
  
    
  print, bkg_data_counts_total, bkg_normalization_nominal, bkg_normalization_corrected, F='(%"\nBackground region: %d in-band background counts; nominal normalization =%6.1f; adopted normalization =%6.1f")' 
  
  print, SRC_CNTS[ii], src_cnts_error,                               F='(%"SRC_CNTS                              =%7.1f (+-%5.1f)")' 

  print, bkg_data_counts_total*R.bkg_scaling_corrected, bkg_subtraction_error, F='(%"bkg counts in aperture                = %6.1f (+-%5.1f)")' 

  print, PROB_NO_SOURCE_binomial,                                    F='(%"PROB_NO_SOURCE                        = %8.2g")'
; print, PROB_NO_SOURCE_binomial, PROB_NO_SOURCE_poisson, F='(%"PROB_NO_SOURCE, actual and asymptotic: %8.2g %8.2g")'
  
  print, "PREDICTED BIAS FROM CONTAMINATING SOURCES:"
  print, "         contaminating source label"
  print, "         |     PSF normalization"
  print, "         |     |     predicted to be in extraction aperture"
  print, "         |     |     |     predicted to be subtracted by background region"
  print, "         |     |     |     |"
  print, round(domain_models[self_index].counts_in_model), R_subtracted_model_counts[self_index], F='(%"      self %5d   --- %5.1f  (counts)")'
  print, R.aperture_flat_counts, R.subtracted_flat_counts,                   F='(%"      flat       %5.1f %5.1f")'

  flag = (abs(aperture_model_counts) GT 0.1) OR (abs(R_subtracted_model_counts) GT 0.1)
  flag[self_index] = 0 ; because a print above already reported the "self" model.
  ind = where(flag, count)
  if (count GT 0) then begin
    forprint, domain_models.LABEL, round(domain_models.counts_in_model), (aperture_model_counts), (R_subtracted_model_counts), SUBSET=ind, F='(%"%10s %5d %5.1f %5.1f")'
  endif
  print, R.aperture_flat_counts+aperture_model_counts_total, R.subtracted_flat_counts+total(R_subtracted_model_counts), F='(%"-------------------------\ntotal:           %5.1f %5.1f")'
  print, R.background_imbalance,  F='(%"background imbalance=%6.2f")'
  print, R.compactness_bias,      F='(%"compactness metric  =%6.2f")'

  
  ;; Ok, a background region is now defined to be the pixels where pixel_status[col,row] EQ background_status.
  region_index = where(pixel_status EQ background_status)
  index_to_point, region_index, col, row, size(pixel_status)

  ; Make a ds9 region file which marks the masked and background region pixels.
  openw,  region1_unit, bkg_pixels_region_fn, /GET_LUN
  printf, region1_unit, F='(%"# Region file format: DS9 version 3.0 \nglobal color=DodgerBlue font=\"helvetica 12 normal\" \nJ2000")'
  !TEXTUNIT = region1_unit  
  xy2ad, col, row, emap2wcs_astr, ra_pt, dec_pt
  forprint, TEXTOUT=5, /NoCOM, ra_pt, dec_pt, F='(%"cross point %10.6f %10.6f # tag={bkg region}")'
  
  index_to_point, where(pixel_status EQ masked_status), temp_col, temp_row, size(pixel_status)
  xy2ad, temp_col, temp_row, emap2wcs_astr, ra_pt, dec_pt
  forprint, TEXTOUT=5, /NoCOM, ra_pt, dec_pt, F='(%"x point %10.6f %10.6f # tag={disallowed region} color=Chocolate")'
  
  free_lun, region1_unit

  if (verbose GE 1) then begin
    run_command, /UNIX, 'egrep "cat|polygon" '+regionfile+'| sed -e "s/DodgerBlue/green/" >! '+green_region_fn
   ;run_command, /UNIX, 'sed -e ''s/DodgerBlue/grey/'' '+green_region_fn+' >! '+ gray_region_fn

    cmd1 = string(my_ds9,                       F='(%"xpaset -p %s regions delete all")') 
    cmd2 = string(my_ds9, bkg_pixels_region_fn, F='(%"xpaset -p %s regions load %s")')
    cmd3 = string(my_ds9, green_region_fn,      F='(%"xpaset -p %s regions load %s")')
    cmd4 = string(my_ds9, RA[ii], DEC[ii],      F='(%"xpaset -p %s pan to %10.6f %10.6f wcs fk5 degrees")')
    run_command, /QUIET, [cmd1,cmd2,cmd3,cmd4]
  endif  


  ;; Write the corresponding cropped background exposure map to a file.
  ;; WE CANNOT USE HEXTRACT BECAUSE THE PHYSICAL COORDINATE SYSTEM (AND WHO KNOWS WHAT ELSE) ARE NOT UPDATED!!!
  bkg_emap               = fltarr(emap_col_dim,emap_row_dim)
  bkg_emap[region_index] = emap[region_index] 
  bkg_emap_header = emap_header

  writefits, temp_image_fn, bkg_emap, bkg_emap_header


  ; Add a 1-pixel MARGIN AROUND THE NON-ZERO PIXELS IN temp_image_fn, AS A WORKAROUND TO THE DMIMGPICK BUG AT THE IMAGE EDGES (HelpDesk Ticket #020605).
  cmd = string(temp_image_fn, $
               0 > (1+min(col) - 1)                   , $
                   (1+max(col) + 1) < (emap_col_dim-1), $
               0 > (1+min(row) - 1)                   , $
                   (1+max(row) + 1) < (emap_row_dim-1), $
               bkg_emap_fn, F="(%'dmcopy ""%s[#1=%d:%d,#2=%d:%d]"" %s ')")
  run_command, cmd
 
  
  ;; The set of emap pixels above define the actual background region to apply to the event list.

  ; Apply any STATUS filter declared for this source, to be consistent with AE's extraction.
  ; Testing shows that it is more efficient to perform STATUS filter in dmcopy, after the filter on the emap column, than in dmimgpick.
  cmd1 = string(evtfile, bkg_emap_fn, temp_events_fn, $
                F="(%'dmimgpick %s %s %s method=closest')")


  ; BELOW WE REQUIRE EMAP VALUE TO BE >1, INSTEAD OF >0, BECAUSE CIAO 3.0.1 HAS A BUG THAT CAUSES ZERO VALUES TO PASS THE >0 TEST!
  cmd2 = string(temp_events_fn, $
                keyword_set(S_FILTER[ii]) ? ','+S_FILTER[ii] : '', $
                bkg_events_fn, F="(%'dmcopy ""%s[#7>1%s][cols time,ccd_id,sky,pi,energy,status]"" %s')")
  run_command, [cmd1,cmd2]


  ;; ------------------------------------------------------------------------
  ;; Extract background spectrum.
  ;; NOTE: if we ever implement a time filter on the background data then we must 
  ;; reduce bkg_exposurearea below by the ratio 
  ;; (EXPOSURE from bkg_spectrum_fn)/(EXPOSURE from bkg_events_fn) to account for the lost exposure.
  ;; Such time filtering might mess up MIN_NUM_CTS requirement!
  
  cmd = string(bkg_events_fn, DETCHANS, bkg_spectrum_fn, $
               F="(%'dmextract ""%s[bin pi=1:%d:1]"" %s opt=pha1 error=gaussian')")
  run_command, cmd

  
  ;; ------------------------------------------------------------------------
  ;; Save some statistics about the background region.
  
  ;; ------------------------------------------------------------------------
  ;; The AE convention is that the BACKSCAL keywords in spectrum files, derived from
  ;; integrals of the exposure map, are used to represent geometric
  ;; area, effective area, and integration time differences between the
  ;; source and background regions.  
  ;; The EXPOSURE keywords are NOT used for background scaling.  We set EXPOSURE=0
  ;; in the background spectrum as a flag to signify the AE convention is being used.
  ;;
  ;; Background scaling is a bit confusing.  
  ;; * In this code the variable bkg_scaling_corrected (S) is multiplied by the background.
  ;; * The BACKSCAL keyword in obs.stats is 1/bkg_scaling_corrected.
  ;; * However the keyword BACKSCAL in the background spectrum file represents
  ;;   the "measure" (integral of the emap) of the background region.
  ;;   It is later combined with BACKSCAL in the source spectrum file to get the
  ;;   actual scaling applied to the background spectrum.
  comment  = 'EXPOSURE not used for bkg scaling'
  comment2 = string(ENERG_LO, ENERG_HI, F="(%'[photon /cm**2 /s /skypixel**2] total bkg SB,, %0.2f:%0.2f keV')")
  comment3 = string(ENERG_LO, ENERG_HI, F="(%'[photon /cm**2 /s /skypixel**2] flat bkg SB,, %0.2f:%0.2f keV')")

  
  ; Write the keywords that are unique to background.pi.
  openw, unit, temp_text_fn, /GET_LUN
  printf, unit, comment, comment, comment, $
          bkg_exposurearea_corrected, '[skypix**2 s cm**2 count /photon]; '+comment, $
          neighbor_invalid_threshold, 'Pb threshold for neighbors'     , $
          S_FILTER[ii], $
          F='(%"#add\nONTIME = 0.0 / %s\nLIVETIME = 0.0 / %s\nEXPOSURE = 0.0 / %s\nBACKSCAL = %e / %s\nTHRESHLD = %0.3f / %s\nS_FILTER=\"...\"\nS_FILTER=\"%s\"")'
          ; S_FILTER is assigned twice above to work around a dmhedit bug (in CIAO 4.3) that converts a whitespace value to the integer zero.
  free_lun, unit
  
  cmd = string(bkg_spectrum_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
  run_command, cmd, /QUIET


  ; Write the keywords that are common in background.pi and obs.stats.
  ; Since dmhedit will RETAIN the datatype of keywords already in obs.stats, we explicitly delete keywords before updating them.
  openw, unit, temp_text_fn, /GET_LUN
  printf, unit, F='(%"#delete\nBACKCORR\nBACKGRND\nFLATGRND\nSELFCNTS\nBKGMETR1\nBKGMETR2\nBKGMETR")'

  printf, unit, creator_string, $
          R.bkg_scaling_correction,                            'scaling correction (already in BACKSCAL)', $
          bkg_data_counts_total / bkg_exposurearea_corrected,  comment2, $
          R.bkg_flat_counts     / bkg_exposurearea_corrected,  comment3, $
          R_subtracted_model_counts[self_index],               'expected subtracted counts from self', $
          R.background_imbalance,                              'bkg fairness metric', $
          R.compactness_bias,                                  'bkg compactness metric', $
          current_region_metric,                               'bkg region metric', $
          F='(%"#add\nCREATOR = %s\nBACKCORR = %f / %s\nBACKGRND = %e / %s\nFLATGRND = %e / %s\nSELFCNTS = %e / %s\nBKGMETR1 = %f / %s\nBKGMETR2 = %f / %s\nBKGMETR = %f / %s")'
  free_lun, unit
  
  cmd = string(bkg_spectrum_fn, obs_stats_fn, temp_text_fn, F="(%'dmhedit infile=""%s %s"" filelist=%s')")
  run_command, cmd, /QUIET

  ; Write the keywords unique to obs.stats.
  obs_stats = headfits(obs_stats_fn, ERRMSG=error)
  if keyword_set(error) then begin
    print, error
    message, 'ERROR reading '+obs_stats_fn
  endif
  
  psb_xaddpar, obs_stats, 'BKG_RAD',  bkg_radius,            'region is from ae_better_backgrounds'
  psb_xaddpar, obs_stats, 'BKG_CNTS', bkg_data_counts_total, string(ENERG_LO, ENERG_HI, F="(%'[count] background, %0.2f:%0.2f keV')") 
  psb_xaddpar, obs_stats, 'BACKSCAL', bkg_normalization_corrected, 'normalization for BKG_CNTS' 
  psb_xaddpar, obs_stats, 'BKSCL_LO', BKSCL_LO,              'smallest BACKSCAL allowed'
  psb_xaddpar, obs_stats, 'BKSCL_GL', BKSCL_GL,              'target   BACKSCAL'
  psb_xaddpar, obs_stats, 'BKSCL_HI', BKSCL_HI,              'largest  BACKSCAL allowed'
  
  
  
  ;; ------------------------------------------------------------------------
  ; This extraction gets to cast votes, used by the adjustment algorithm (ae_adjust_backscal_range), for an upper limit on BKSCL_LO (VOTE_LO) and a lower limit on BKSCL_HI (VOTE_HI).
  
  if finite(largest_acceptable_region.bkg_normalization_nominal) then begin
    ; Acceptable regions (those with acceptable background imbalance) were found in the search.
     
    if finite(lowest_imbalance_region.background_imbalance) then begin
      ; The search proceeded far enough to encounter an UNacceptable background imbalance.
      ; Thus, we can be fairly sure that the largest acceptable region (largest_acceptable_region) we've encountered is a useful upper limit on BKSCL_LO.
      BKSCL_LO_vote = largest_acceptable_region.bkg_normalization_nominal
    endif else begin
      ; The largest region we evaluated still had acceptable background imbalance.
      ; Thus, we have no information on how large the region could grow before we're unhappy; no informed vote on BKSCL_LO is possible.
      BKSCL_LO_vote =  !VALUES.F_NAN
    endelse

  endif else begin
    ; Every single region we tested during the search failed the background imbalance criterion.
    ; The least bad one is our vote for an upper limit on BKSCL_LO.
    BKSCL_LO_vote = lowest_imbalance_region.bkg_normalization_nominal
  endelse
    
  ; If the search did NOT identify any scaling that meets the MIN_NUM_CTS goal, then cast a vote to raise BKSCL_HI by some arbitrary amount.
  ; Do NOT try to estimate the BKSCL_HI that should enclose MIN_NUM_CTS by scaling up from the number of counts found in the
  ; current region!!!  That sort of data-based adjustment here will surely harm convergence of the search since the number of
  ; counts actually found in any region has a random component.
  if ~finite(BKSCL_HI_vote) then BKSCL_HI_vote = 2 * BKSCL_HI
    
  psb_xaddpar, obs_stats, 'VOTE_LO', BKSCL_LO_vote, 'vote for upper limit on BKSCL_LO' 
  psb_xaddpar, obs_stats, 'VOTE_HI', BKSCL_HI_vote, 'vote for lower limit on BKSCL_HI' 

  writefits, obs_stats_fn, 0, obs_stats


  if keyword_set(pause_for_review) then begin
    flush_stdin  ; eat any characters waiting in STDIN, so that they won't be mistaken as commands in the loop below.
    print, F='(%"\nPress return to continue to the next source ...")'
    read, '? ', cmd1
  endif  
endfor ; ii loop over the catalog

ind = where(processing_rate GT 0, count)
if (count GT 0) then print, 'Median processing rate (accepted pixels/s): ', median(processing_rate[ind])

backgrounds_extracted = 1


CLEANUP:
;  if (verbose GE 1) then run_command, string(my_ds9, F='(%"xpaset -p %s exit")'), /QUIET

  if (n_elements(models) GT 0) then ptr_free, models.data_ptr

if keyword_set(temproot) && file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif

if (exit_code EQ 0) then begin
  if backgrounds_extracted then begin
  
    if (n_elements(number_of_passes) EQ 0) then begin
      ; The caller is relying on this tool to decide if the source models would likely benefit from another pass through the photometry and background calculations.
      num_missing_backgrounds = total(/INT, ~finite(BACKSCAL))

      if (num_missing_backgrounds GT 0) && keyword_set(build_models_only) then begin
        print, 'ERROR: ae_better_backgrounds has been asked to build source models, but some sources are missing backgrounds!'
        goto, FAILURE
      endif
      
      if (num_missing_backgrounds GT 0) && ~keyword_set(reuse_models) then begin
        print, F='(%"\n===================================================================")'
        print, num_missing_backgrounds, F="(%'WARNING!  %d sources are missing backgrounds; running an extra pass through ae_better_backgrounds to help the source models converge.')"
        print, F='(%"===================================================================\n")'
        number_of_passes = 2
      endif else number_of_passes = 1
    endif
  
    if (--number_of_passes GT 0) then begin
      ; Make a recursive call after decrementing number_of_passes.
      ; We omit the PHOTOMETRY_STAGE_ONLY, SKIP_PHOTOMETRY_STAGE, REUSE_MODELS, BUILD_MODELS_ONLY options since the point of the extra passes is to get the photometry (and models) to converge!
      ae_better_backgrounds, obsname, EVTFILE_BASENAME=evtfile_basename, $
        NUMBER_OF_PASSES=number_of_passes, $ 
    
        THETA_RANGE=theta_range, BACKGROUND_MODEL_FILENAME=background_model_filename, $
      
        SOURCE_NOT_OBSERVED=source_not_observed, GENERIC_RMF_FN=generic_rmf_fn, $
        
        SRCLIST_FILENAME=srclist_fn, EXTRACTION_NAME=extraction_name, $
        EMAP_BASENAME=emap_basename, $
    
        MIN_NUM_CTS=min_num_cts, BACKGROUND_IMBALANCE_THRESHOLD=background_imbalance_threshold, $
        COMPACTNESS_GAIN=compactness_gain, NEIGHBOR_INVALID_THRESHOLD=neighbor_invalid_threshold_p, $
        
        VERBOSE=verbose, PAUSE_FOR_REVIEW=pause_for_review, /SKIP_RESIDUALS, $
        
        SAVEFILE_BASENAME=savefile_basename, OBS_DIR=obs_dir_p
    endif
  
  endif ; backgrounds_extracted
  return 
endif else begin
  print, 'ae_better_backgrounds: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end ; ae_better_backgrounds



;#############################################################################
;;; Tool to analyze the quality of background extractions, and to adjust the 
;;; background scaling range (BKSCL_LO,BKSCL_HI) allowed for each source, and
;;; the target background scaling (BKSCL_GL).

;;; The OVERLAP_LIMIT parameter is the same one accepted by the MERGE stage of AE.
;;; It defines extractions that are too crowded to be merged, and thus should have no vote
;;; here on the BACKSCAL range.
;;;
;;; The MIN_NUM_CTS parameter is a goal for the *merged* background spectrum.
;;; When this goal is in conflict with the "background imbalance" goal in ABB, the MIN_NUM_CTS will defer (i.e. not be met).
;#############################################################################
PRO ae_adjust_backscal_range, MIN_NUM_CTS=min_num_cts, $
  OVERLAP_LIMIT=overlap_limit, MERGE_NOT_OPTIMIZED=merge_not_optimized, NUM_SOURCES_ADJUSTED=num_sources_adjusted, $
  SRCLIST_FILENAME=srclist_fn, RERUN_SRCLIST_FILENAME=rerun_srclist_fn, EXTRACTION_NAME=extraction_name,$
  SCALE_RANGE_BY_FACTOR=scale_range_by_factor, SET_LIMIT_RATIO=set_limit_ratio

exit_code = 0
creator_string = "ae_adjust_backscal_range, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

if n_elements(overlap_limit) EQ 0 then overlap_limit    = 0.10
if ~keyword_set(min_num_cts)      then min_num_cts      = 100
if ~keyword_set(srclist_fn)       then       srclist_fn = 'all.srclist'
if ~keyword_set(rerun_srclist_fn) then rerun_srclist_fn = 'rerun.srclist'

;; Check for common environment errors.
quiet = !QUIET  &  !QUIET = 1
catch, error_code
if (error_code NE 0) then begin
  print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
  retall
endif else resolve_routine, 'astrolib', /NO_RECOMPILE
catch, /CANCEL
astrolib
; Make sure forprint calls do not block for user input.
!TEXTOUT=2
!QUIET = quiet
  
readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'

; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)

if (num_sources EQ 0) then begin
  print, 'ae_adjust_backscal_range: ERROR: no entries read from source list ', srclist_fn
  GOTO, FAILURE
endif

sourcename = sourcename[ind]
print, num_sources, F='(%"\nae_adjust_backscal_range: %d sources found in catalog.")'


if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)

src_stats_basename       = 'source.stats'
obs_stats_basename         = 'obs.stats'

obsid_list     = ''
LABEL          = strarr(num_sources)
single_single_conflict_flag  = bytarr(num_sources)
merged_single_conflict_flag  = bytarr(num_sources)
rerun_flag     = bytarr(num_sources)
BKSCL_GL_report = strarr(num_sources)
new_BKSCL_LO   = fltarr(num_sources)
new_BKSCL_GL   = fltarr(num_sources)
new_BKSCL_HI   = fltarr(num_sources)

obs_data_template =  {OBSNAME :'', $
                      SRC_CNTS:0L, $
                      BKG_CNTS:0L, $
                      BACKSCAL:0.0,$
                      BKG_RAD :0.0,$
                      BKGMETR1:0.0,$ ;'bkg fairness metric'
                      VOTE_LO :0.0,$
                      VOTE_HI :0.0,$
                      REG_EDIT:0B ,$
                      OVERLAP :0.0 $
                     }


single_single_conflict_fn = 'single-single-conflict.srclist'
merged_single_conflict_fn = 'merged-single-conflict.srclist'
openw, single_single_conflict_unit, single_single_conflict_fn, /GET_LUN
openw, merged_single_conflict_unit, merged_single_conflict_fn, /GET_LUN

for ii = 0L, num_sources-1 do begin
  ;; Construct filenames.
  sourcedir            = sourcename[ii] + '/'
  unnamed_src_stats_fn = sourcedir + src_stats_basename

  ; We assume that an existing source directory that is a symbolic link should not be written to.
  temp = file_info(sourcedir)
  is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
  if ~is_writable then begin
    print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
    continue
  endif 

  ; Look up allowed range of background normalizations.
  unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
  
  if (~keyword_set(error)) then begin
    LABEL[ii]= strtrim(psb_xpar( unnamed_src_stats, 'LABEL'),2)
    BKSCL_LO =         psb_xpar( unnamed_src_stats, 'BKSCL_LO', COUNT=count1)
    BKSCL_GL =         psb_xpar( unnamed_src_stats, 'BKSCL_GL', COUNT=count2)
    BKSCL_HI =         psb_xpar( unnamed_src_stats, 'BKSCL_HI', COUNT=count3)
  endif else begin
    print, error
    message, 'ERROR reading '+unnamed_src_stats_fn
  endelse

  if (count1*count2*count3 EQ 0) then begin
    print, 'ERROR: '+unnamed_src_stats_fn+' is missing the keywords BKSCL_LO,BKSCL_GL,BKSCL_HI.'
    print, "See 2009-06-22 What's New entry in the AE manual for patch to update your AE sources."
    GOTO, FAILURE
  endif


  ; Find a list of obs.stats files and read information related to extractions..
  pattern = sourcedir + '*/' + extraction_subdir[ii] + obs_stats_basename
  obs_stats_fn = file_search( pattern, COUNT=num_obs )
  
  if (num_obs EQ 0) then begin
    print, sourcename[ii], LABEL[ii], F='(%"%s (%s) was not observed.")'
    continue
  endif
  
  these_obsids = strmid(obs_stats_fn,  1+reform(strpos(obs_stats_fn, '/'), 1,num_obs))
  these_obsids = strmid(these_obsids, 0, reform(strpos(these_obsids, '/'), 1,num_obs))

  ;; Handle the tool mode where the existing BKSCL range is scaled by a specified factor.
  if keyword_set(scale_range_by_factor) then begin
    rerun_flag[ii] = 1  
    BKSCL_LO *= scale_range_by_factor
    BKSCL_GL *= scale_range_by_factor
    BKSCL_HI *= scale_range_by_factor
    
    print, sourcename[ii], LABEL[ii], BKSCL_LO, BKSCL_GL, BKSCL_HI, F='(%"%s (%s): [%0.1f--%0.1f--%0.1f]")'

    psb_xaddpar, unnamed_src_stats, 'BKSCL_LO', BKSCL_LO, 'smallest BACKSCAL allowed'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_GL', BKSCL_GL, 'target   BACKSCAL'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_HI', BKSCL_HI, 'largest  BACKSCAL allowed'
    writefits, unnamed_src_stats_fn, 0, unnamed_src_stats

    new_BKSCL_LO[ii] = BKSCL_LO
    new_BKSCL_GL[ii] = BKSCL_GL
    new_BKSCL_HI[ii] = BKSCL_HI

    obsid_list   = [obsid_list, these_obsids]
    obsid_list   =  obsid_list[uniq(obsid_list,sort(obsid_list))]
    
    continue
  endif
  

  ;; Handle the tool mode where the existing BKSCL limits are adjusted to have a specified ratio.
  ;; We arbitrarily let the BKSCL_HI limit remain fixed and adjust the BKSCL_LO limit.
  if keyword_set(set_limit_ratio) then begin
    rerun_flag[ii] = 1  
    BKSCL_LO = BKSCL_HI / set_limit_ratio
    BKSCL_GL = BKSCL_LO > (BKSCL_GL < BKSCL_HI)
    
    print, sourcename[ii], LABEL[ii], BKSCL_LO, BKSCL_GL, BKSCL_HI, F='(%"%s (%s): [%0.1f--%0.1f--%0.1f]")'

    psb_xaddpar, unnamed_src_stats, 'BKSCL_LO', BKSCL_LO, 'smallest BACKSCAL allowed'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_GL', BKSCL_GL, 'target   BACKSCAL'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_HI', BKSCL_HI, 'largest  BACKSCAL allowed'
    writefits, unnamed_src_stats_fn, 0, unnamed_src_stats

    new_BKSCL_LO[ii] = BKSCL_LO
    new_BKSCL_GL[ii] = BKSCL_GL
    new_BKSCL_HI[ii] = BKSCL_HI

    obsid_list   = [obsid_list, these_obsids]
    obsid_list   =  obsid_list[uniq(obsid_list,sort(obsid_list))]
    
    continue
  endif
  



  ;; Read info from each single extraction.
  obs_data = replicate(obs_data_template,num_obs)
  for jj=0,num_obs-1 do begin
    header = headfits(obs_stats_fn[jj], ERRMSG=error )
    if (keyword_set(error)) then begin
      print, error
      message, 'ERROR reading ' + obs_stats_fn[jj]
    endif
    obs_data[jj].OBSNAME  = strtrim(psb_xpar( header, 'OBSNAME'),2)
    obs_data[jj].SRC_CNTS =         psb_xpar( header, 'SRC_CNTS')
    obs_data[jj].BKG_CNTS =         psb_xpar( header, 'BKG_CNTS')
    obs_data[jj].BACKSCAL =         psb_xpar( header, 'BACKSCAL')
    obs_data[jj].BKG_RAD  =         psb_xpar( header, 'BKG_RAD')
    obs_data[jj].BKGMETR1 =         psb_xpar( header, 'BKGMETR1') ;'bkg fairness metric'
    obs_data[jj].VOTE_LO  =         psb_xpar( header, 'VOTE_LO')
    obs_data[jj].VOTE_HI  =         psb_xpar( header, 'VOTE_HI')
    obs_data[jj].REG_EDIT =         psb_xpar( header, 'REG_EDIT')
    obs_data[jj].OVERLAP  =         psb_xpar( header, 'OVERLAP')
  endfor ;jj
               
  ;; ------------------------------------------------------------------------
  ;; Ignore the extractions that AE's MERGE stage will exclude due to excessive OVERLAP.
  ;; The code below is lifted from AE's MERGE stage.

  overlap_is_acceptable = (obs_data.overlap LT overlap_limit)
  
  ; When a source has been hand edited (reg_edit EQ 1) we assume that the observer wants that extraction to be merged, regardless of any overlap it has with a neighbor's aperture!
  ind = where(~overlap_is_acceptable AND obs_data.reg_edit, count)
  if (count GT 0) then begin
    overlap_is_acceptable[ind] = 1B
    print, sourcename[ii], count, F="(%'WARNING: %s: accepting %d hand-edited apertures with excessive OVERLAP:')"
    forprint, SUBSET=ind, obs_data.obsname, obs_data.overlap, F="(%'  %12s OVERLAP=%0.3f')"
  endif
  
  accepted_ind = where(overlap_is_acceptable, num_obs, COMPLEMENT=rejected_ind, NCOMPLEMENT=num_rejected)
  
  if (num_obs EQ 0) then begin
    ; Although all the extractions have excessive overlap, we still need to merge some subset of the data so that we'll
    ; have rough photometry of the source to help us decide later whether to prune this source or his neighbor.
    ; Merging only the _single_ ObsId with the minimum overlap proved to be a poor design, because there are sometimes
    ; cases where that one happens to have photometry very differerent from one or more other ObsIds with overlap values 
    ; comparable to the smallest one.
    ; Thus we arbitrarily stretch the overlap limit upward by 20% for this source.
    stretched_overlap_limit = 1.2 * min(obs_data.overlap)
    print, n_elements(obs_data), sourcename[ii], stretched_overlap_limit, F="(%'WARNING: all %d extractions of %s have excessive OVERLAP; accepting only those with OVERLAP comparable to the best extraction (i.e. OVERLAP<=%0.2f).')"      
    accepted_ind = where(obs_data.overlap LT stretched_overlap_limit, num_obs)
    
  endif else if (num_rejected GT 0) then begin
    print, sourcename[ii], LABEL[ii], strjoin(obs_data[rejected_ind].OBSNAME, ','), F='(%"%s (%s): ignoring ObsIDs with excessive OVERLAP (%s)")'
    
  endif

  obs_data = obs_data[accepted_ind]
  

  ;; ------------------------------------------------------------------------
  if ~keyword_set(merge_not_optimized) then begin
    ;; Ignore the extractions that AE's MERGE stage is *likely* to exclude.
    ;; This is not easy to predict--for now we will exclude extractions with zero or negative net counts.
    net_counts   = obs_data.SRC_CNTS - (obs_data.BKG_CNTS / obs_data.BACKSCAL)
    accepted_ind = where(net_counts GT 0, count, COMPLEMENT=ignored_ind, NCOMPLEMENT=num_rejected)
      
    if (count GT 0) && (num_rejected GT 0) then begin
  ;   print, sourcename[ii], LABEL[ii], strjoin(obs_data[ignored_ind].OBSNAME, ','), F='(%"%s (%s): ignoring ObsIDs with zero or negative net counts (%s)")'
  
      print, sourcename[ii], LABEL[ii], num_rejected, F='(%"%s (%s): ignoring %d ObsIDs with zero or negative net counts. ")'
      num_obs  = count
      obs_data = obs_data[accepted_ind]
    endif
  endif ; ~keyword_set(merge_not_optimized)
  
  
  ;; ------------------------------------------------------------------------
  ; Adjust the target bkg scaling (BKSCL_GL) to better achieve goals in the merged dataset (assuming that all extractions are merged).
  
  ; AE is going to use the merged background data to compute the PROB_NO_SOURCE statistic.
  ; We could attempt to compute a confidence interval on PROB_NO_SOURCE, establish some sort of requirement 
  ; on that confidence interval, and then compute how many BKG_CNTS are needed.
  ; Instead, we'll punt and force the observer to specify a goal for the total number of BKG_CNTS 
  ; (assuming that all extractions are merged).
  ; Estimate the BACKSCAL range resizing needed to achieve this MIN_NUM_CTS goal.
  resize1 = float(min_num_cts) / total(obs_data.BKG_CNTS)
  if ~finite(resize1) then resize1 = 10
  
  ; It seems reasonable to require that the statistical uncertainty in the background subtraction should be
  ; smaller than the statistical uncertainty in the counts extracted from the source aperture.
  ; In other words, of the two noise terms in AE's calculation of error on NET_CNTS (assuming that all extractions are merged), 
  ; the term from the extraction aperture should dominate.
  ;
  ; Given:
  ;   NET_CNTS = SRC_CNTS - (BKG_CNTS * bkg_scaling)
  ;     where bkg_scaling is the scaling for the merged data.
  ;
  ;   src_cnts_error = (1 + sqrt(SRC_CNTS + 0.75))
  ;   bkg_cnts_error = (1 + sqrt(BKG_CNTS + 0.75))                              (1)
  ;
  ; Then the error on NET_CNTS is:
  ;   sqrt( src_cnts_error^2 + (bkg_cnts_error*bkg_scaling)^2 )
  ;
  ; We adopt this arbitrary goal: 
  ;      src_cnts_error = 4.0*bkg_cnts_error*bkg_scaling                        (2)
  ; Then the error on NET_CNTS is:
  ;      sqrt(src_cnts_error^2 + (src_cnts_error/4)^2) = src_cnts_error * 1.03
  ; Thus, the photometry error is only 3% larger than it would be with an infinitely large background sample.
  ;
  ; Thus, we are trying to estimate what sized bkg region would achive the goal in equation (2).
  ; Changing the bkg region size should change both BKG_CNTS and bkg_scaling, so there is not an obvious algebraic solution 
  ; for the desired resizing of the background region(s).  We'll have to iteratively estimate...
 
  src_cnts_error      = (1 + sqrt(total(/INT, obs_data.SRC_CNTS)   + 0.75))
  current_bkg_scaling = total(obs_data.BKG_CNTS/obs_data.BACKSCAL) / total(obs_data.BKG_CNTS)
  
  if ~finite(current_bkg_scaling) then begin
    resize2 = 10
  endif else begin
    resize2 = 1.0
    for kk=1,100 do begin
      proposed_bkg_scaling = current_bkg_scaling / resize2
      proposed_BKG_CNTS    = total(obs_data.BKG_CNTS)     * resize2
      
      proposed_bkg_cnts_error = (1 + sqrt(proposed_BKG_CNTS + 0.75))
      
      if (src_cnts_error GT 4.0*proposed_bkg_cnts_error*proposed_bkg_scaling) then begin
        ; The background subtraction error meets the goal; we can shrink the bkg region.
        resize2 /= 1.05
      endif else begin
        ; The background subtraction error is too large; we must expand the bkg region.
        resize2 *= 1.10
      endelse
;     print, resize2
    endfor ; kk
  endelse
  
  
  ; The larger of the two rescaling requests governs our action.
  requested_BKSCL_GL_scaling =  resize1 > resize2
  
  
  ; Impose limits on the scaling we will implement in one cycle, on the theory that very large changes are likely to lead to oscillations in the range as we iteratively call this routine and re-extract the backgrounds that are not satisfied.
  
  scaling_ceiling = 5.0
  scaling_floor   = 1.0 / 4.1
  
  ; These limits should be chosen so that a few cycles scaled at these limits will NOT bring us back to the same range.
  ; For example, the worst choice is scaling_floor = 1/scaling_ceiling; this allows a two-step cycle that repeats forever.
  ; I chose the limits above by playing with the following code that simulates a series of adjustments made using these limits.
;  net_scale = fltarr(10,10)
;  for ii=0,9 do $
;    for jj=0,9 do net_scale[ii,jj] = scaling_ceiling^(ii+1) * scaling_floor^(jj+1)
;  dataset_1d, id, alog10(net_scale)
      
  if (requested_BKSCL_GL_scaling LT 0.50) then begin
    ; The bkg regions want to significantly shrink; let's avoid a huge adjustment.
    ; The width of this "dead band", requested_BKSCL_GL_scaling=[0.50,1.0], should control how fast these adjustments converge.
    BKSCL_GL *=       (scaling_floor > requested_BKSCL_GL_scaling)
  endif else if (requested_BKSCL_GL_scaling GT 1.0) then begin
    ; The bkg regions must grow; let's avoid both a tiny or a huge adjustment.
    BKSCL_GL *=                 1.2 > (requested_BKSCL_GL_scaling < scaling_ceiling)
  endif else begin
    ; The bkg regions want to shrink only a little (0.50 < requested_BKSCL_GL_scaling < 1.0).  Let's declare success and let this source be stable! 
    ; 
  endelse
  
  
  ;; ------------------------------------------------------------------------
  ;; Calculate the constraints on scaling implied by the single-obsid votes.
  
  ; A VOTE_* value of 0 means "no vote".
  ind = where(obs_data.VOTE_LO EQ 0, count)
  if (count GT 0) then obs_data[ind].VOTE_LO = !VALUES.F_NAN
  ind = where(obs_data.VOTE_HI EQ 0, count)
  if (count GT 0) then obs_data[ind].VOTE_HI = !VALUES.F_NAN
  
  ; The smallest LO vote and the largest HI vote are the relevant ones.
  VOTE_HI = max(obs_data.VOTE_HI, /NAN, index_vote_hi)
  VOTE_LO = min(obs_data.VOTE_LO, /NAN, index_vote_lo) 
  ; We arbitrarily prohibit very small LO votes.
  if finite(VOTE_LO) then VOTE_LO >= 0.5

  ; Each vote is a constraint on the rescaling we are allowed to apply to the BKSCL range. 
  rescale_min = (VOTE_HI/BKSCL_HI) 
  if ~finite(rescale_min) then rescale_min = -!VALUES.F_INFINITY
  
  rescale_max = (VOTE_LO/BKSCL_LO) 
  if ~finite(rescale_max) then rescale_max =  !VALUES.F_INFINITY
  
  ; If those constraints conflict, then relax the one produced by VOTE_HI (rescale_min), only as much as required.
  if (rescale_min GT rescale_max) then begin
    ; Both constraints exist (are not NaN), but conflict.
    
    ; In version 3368 (2009-02-25) we tried a compromise scaling range that produces equal shortfalls at both ends, i.e.
    ;   (BKSCL_LO-VOTE_LO) = (VOTE_HI-BKSCL_HI)
    ; while maintaining the original ratio (BKSCL_HI/BKSCL_LO).
    ;
    ; However, we subsequently decided to let VOTE_LO (from background imbalance goal) override VOTE_HI (from MIN_NUM_CTS goal).
    ; Thus, we simply discard the rescale_min constraint and carry on.
    single_single_conflict_flag[ii] = 1
    
    printf, single_single_conflict_unit, sourcename[ii], LABEL[ii], VOTE_LO, VOTE_HI, $
    F='(%"%s ; (%s): cannot satisfy BKSCL_LO < %0.1f AND %0.1f < BKSCL_HI")'

    rescale_min = rescale_max ; The smallest relaxation of rescale_max that eliminates the conflict.
  endif 

  
  ;; ------------------------------------------------------------------------
  ; Reconcile the new goal for the merged background (BKSCL_GL) with constraints arising from the votes to move the bkg scaling range endpoints cast by the individual extractions .
  
  ; Calculate the scaling of the allowed BACKSCAL range that would be required to encompass the new BKSCL_GL value.
  rescale_request                                  = (BKSCL_GL/BKSCL_HI) > 1.0
  if (rescale_request EQ 1.0) then rescale_request = (BKSCL_GL/BKSCL_LO) < 1.0
  
  
  ; Record whether we have a conflict between the rescale_request derived from the merged data and the rescale_max limit imposed by the single extractions.
  if (rescale_request GT 1) && (rescale_request GT rescale_max) then begin
    merged_single_conflict_flag[ii] = 1
  
    printf, merged_single_conflict_unit, sourcename[ii], LABEL[ii], obs_data[index_vote_lo].OBSNAME, requested_BKSCL_GL_scaling, $
    F='(%"%s ; (%s): VOTE_LO from ObsID %s prevents desired growth of BKG_CNTS (merged) by a factor of %0.1f")'
  endif
  
  
  ; The single-obsid votes get the first say in deciding the direction of the scaling.
  ; If they don't care, then the new goal (BKSCL_GL) can decide.
  if      (rescale_min     GT 1.0) then direction = 'up' $
  else if (rescale_max     LT 1.0) then direction = 'down' $
  else if (rescale_request GT 1.0) then direction = 'up' $
  else if (rescale_request LT 1.0) then direction = 'down' $
  else direction = ''

  if            (direction EQ 'up') then begin
    ; Try to scale upward to satisfy both votes and goal, but no farther than rescale_max.
    rescale = rescale_max < (rescale_min > rescale_request)
  
  endif else if (direction EQ 'down') then begin
    ; Try to scale downward to satisfy both votes and goal, but no farther than rescale_min.
    rescale = rescale_min > (rescale_max < rescale_request)
  
  endif else begin
    rescale = 1.0
  endelse
  
  
  ; Apply the rescaling to the endpoints of the range.
  BKSCL_LO *= rescale
  BKSCL_HI *= rescale
  
  ; Force the goal to lie in the new range.
  BKSCL_GL = BKSCL_LO > (BKSCL_GL < BKSCL_HI)
  
  
  if (BKSCL_GL LT BKSCL_LO) then message, 'BUG!'
  if (BKSCL_GL GT BKSCL_HI) then message, 'BUG!'
  if ~finite(BKSCL_LO) || ~finite(BKSCL_GL) || ~finite(BKSCL_HI) then message, 'BKSCL_* values are not finite!'
  if (BKSCL_LO LE 0)   || (BKSCL_GL LE 0)   || (BKSCL_HI LE 0)   then message, 'BKSCL_* values are not positive!'
  
  ;; ------------------------------------------------------------------------
  ;; Save the adjusted scaling range and goal parameters only if any have changed significantly.
  fractional_change = [ BKSCL_LO / psb_xpar( unnamed_src_stats, 'BKSCL_LO'), $
                        BKSCL_GL / psb_xpar( unnamed_src_stats, 'BKSCL_GL'), $
                        BKSCL_HI / psb_xpar( unnamed_src_stats, 'BKSCL_HI') ]
         
  ; When the regions need to grow and all ObsIDs are producing the largest possible region (obs_data.BKG_RAD = inf) then do nothing.
  if ((max(fractional_change) GT 1.10) && (min(obs_data.BKG_RAD) LT !VALUES.F_INFINITY)) || $
      (min(fractional_change) LT 0.90)         then begin
  
    rerun_flag    [ii] = 1  
    BKSCL_GL_report[ii] = string( fractional_change[1], F='(%"BKSCL_GL*= %0.1f")' )
    
    print, sourcename[ii], LABEL[ii], BKSCL_GL_report[ii], round(BKSCL_LO), round(BKSCL_GL), (fractional_change[1] GT 1) ? '+' : '-', round(BKSCL_HI), resize1, resize2, F='(%"%s (%s): %s [%4d  %4d%s  %4d] resize={%0.1f,%0.1f}")'

    psb_xaddpar, unnamed_src_stats, 'BKSCL_LO', BKSCL_LO, 'smallest BACKSCAL allowed'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_GL', BKSCL_GL, 'target   BACKSCAL'
    psb_xaddpar, unnamed_src_stats, 'BKSCL_HI', BKSCL_HI, 'largest  BACKSCAL allowed'
    writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
    
    new_BKSCL_LO[ii] = BKSCL_LO
    new_BKSCL_GL[ii] = BKSCL_GL
    new_BKSCL_HI[ii] = BKSCL_HI
    
    obsid_list   = [obsid_list, these_obsids]
    obsid_list   =  obsid_list[uniq(obsid_list,sort(obsid_list))]
  endif
endfor ;ii
free_lun, single_single_conflict_unit, merged_single_conflict_unit


ind = where(rerun_flag, num_sources_adjusted)
if (num_sources_adjusted EQ 0) then begin 
  print, 'No adjustments are required.'
  file_copy, '/dev/null', rerun_srclist_fn, /OVERWRITE
endif else begin
  print
  forprint, TEXTOUT=rerun_srclist_fn, SUBSET=ind, sourcename, LABEL, BKSCL_GL_report, new_BKSCL_LO, new_BKSCL_GL, new_BKSCL_HI, F='(%"%s ; (%s): %s [%0.1f--%0.1f--%0.1f]")', /NoCOMMENT, /SILENT
  print, num_sources_adjusted, rerun_srclist_fn, n_elements(obsid_list)-1, F='(%"%d sources listed in ''%s'' require new background extractions from these %d ObsIds:")'
  forprint, obsid_list[1:*]
endelse



ind = where(single_single_conflict_flag, count)
if (count EQ 0) then begin 
  file_delete, single_single_conflict_fn, /ALLOW_NONEXISTENT
endif else begin
  print, count, single_single_conflict_fn, F='(%"\nWARNING!  For the %d sources listed in ''%s'', the single-ObsID extractions have cast conflicting votes regarding the range of BACKSCAL allowed.  \nSome extractions wish to have larger regions (to satisfy the MIN_NUM_CTS goal imposed on single extractions) and some wish to have smaller regions (so that all neighboring sources are fairly represented in the background spectrum). \nSatisfying both camps would require enlarging the *span* of the BACKSCAL range, which is not desirable. \nThe votes for smaller regions have been respected; the votes for larger regions have been ignored. \nYou may wish to review the background regions for these sources using the /SHOW stage.")'
endelse


ind = where(~single_single_conflict_flag AND merged_single_conflict_flag, count)

if (count EQ 0) then begin 
  file_delete, merged_single_conflict_fn, /ALLOW_NONEXISTENT
endif else begin
  print, count, merged_single_conflict_fn, min_num_cts, F='(%"\nWARNING! For the %d sources listed in ''%s'', one or more single-ObsID extractions will not allow the BACKSCAL range to be enlarged enough to meet the goals of the merged background spectrum (%d counts and insignificant contribution from the background to photometric uncertainty). \nYou may wish to review the background regions for these sources in the /SHOW stage.")'
endelse


CLEANUP:
if (exit_code EQ 0) then begin
  print, F='(%"\nae_adjust_backscal_range finished")'  
  return 
endif else begin
  print, 'ae_adjust_backscal_range: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end   ; ae_adjust_backscal_range



;#############################################################################
;;; Tool to reset BKSCL_* keywords in source.stats using values in an obs.stats. 
;#############################################################################
PRO ae_reset_backscal_range, SRCLIST_FILENAME=srclist_fn, EXTRACTION_NAME=extraction_name 

if ~keyword_set(srclist_fn)       then       srclist_fn = 'all.srclist'
readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'

; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)

if (num_sources EQ 0) then begin
  print, 'ae_reset_backscal_range: ERROR: no entries read from source list ', srclist_fn
  retall
endif

sourcename = sourcename[ind]
print, num_sources, F='(%"\nae_reset_backscal_range: %d sources found in catalog.")'

if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)

src_stats_basename       = 'source.stats'
obs_stats_basename         = 'obs.stats'

for ii = 0L, num_sources-1 do begin
  ;; Construct filenames.
  sourcedir            = sourcename[ii] + '/'
  unnamed_src_stats_fn = sourcedir + src_stats_basename

  ; We assume that an existing source directory that is a symbolic link should not be written to.
  temp = file_info(sourcedir)
  is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
  if ~is_writable then begin
    print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
    continue
  endif 

  ; Look up allowed range of background normalizations.
  unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
  
  if keyword_set(error) then begin
    print, error
    message, 'ERROR reading '+unnamed_src_stats_fn
  endif 
  
  ; Find a list of obs.stats files and read information related to extractions..
  pattern = sourcename[ii] + '/*/' + extraction_subdir[ii] + obs_stats_basename
  obs_stats_fn = file_search( pattern, COUNT=num_obs )
  
  if (num_obs EQ 0) then begin
    print, sourcename[ii], LABEL[ii], F='(%"%s (%s) was not observed.")'
    continue
  endif
  
  header = headfits(obs_stats_fn[0], ERRMSG=error )
  if (keyword_set(error)) then begin
    print, error
    message, 'ERROR reading ' + obs_stats_fn[0]
  endif

  psb_xaddpar, unnamed_src_stats, 'BKSCL_LO', psb_xpar( header, 'BKSCL_LO'), 'smallest BACKSCAL allowed'
  psb_xaddpar, unnamed_src_stats, 'BKSCL_GL', psb_xpar( header, 'BKSCL_GL'), 'target   BACKSCAL'
  psb_xaddpar, unnamed_src_stats, 'BKSCL_HI', psb_xpar( header, 'BKSCL_HI'), 'largest  BACKSCAL allowed'
  writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
endfor ;ii    
return
end ; ae_reset_backscal_range




;#############################################################################
;;; Code for analyzing Pb values from multiple energy bands in multiple merges to produce Pbslice.reg and prune.srclist.
;;;
;;; INVALID_THRESHOLD can be supplied to override the default of 0.01 (1%).
;;; When /IGNORE_PB is set only overlap is used as a pruning criterion.


; If requested, report sources that are NOT VALIDATED by the merges provided.
; The caller can use this information to avoid wasting resources running additional merges on these.

;#############################################################################
PRO ae_analyze_pb, $
                   ANALYZE=analyze, OVERLAP_LIMIT=overlap_limit,$   ; Analyze Stage
                   ; Input: list of collation files to analyze.
                   collated_filename,$ 
                   ; Other input parameters.
                   INVALID_THRESHOLD=invalid_threshold_p, PB_IN_VHARD_BAND=pb_in_vhard_band, $ 
                   BOUNDARIES=boundaries_string, IGNORE_PB=ignore_pb, $ 
                   ; Output: names of sources not validated by collations provided are written to this file.
                   NOT_VALIDATED_SRCLIST_FILENAME=not_validated_srclist_filename,$
                   ; Output: a description of the Pb calculation that most-strongly validates each source.
                   BEST_PHOT=best_phot_return, $
                   
                   ; Make Regions Stage
                   MAKE_REGIONS=make_regions, DS9_TITLE=ds9_title, _EXTRA=extra

creator_string = "ae_analyze_pb, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

;; Check for common environment errors.
quiet = !QUIET  &  !QUIET = 1
catch, error_code
if (error_code NE 0) then begin
  print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
  retall
endif else resolve_routine, 'astrolib', /NO_RECOMPILE
catch, /CANCEL
astrolib
; Make sure forprint calls do not block for user input.
!TEXTOUT=2
!QUIET = quiet
  
;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir, /QUIET
  
temp_region_fn     = tempdir + 'temp.reg'


if keyword_set(analyze) then begin
  if (n_elements(overlap_limit) NE 1) then  begin
    ; We require the caller to pass OVERLAP_LIMIT, so that we can verify that the caller and the merges we're analyzing have made the same choice for this parameter. 
    print, F='(%"ae_analyze_pb: ERROR: The parameter OVERLAP_LIMIT must be supplied.")' 
    retall
  endif

  ; These boundaries are defined below in terms of the percent chance that 
  ; the observed number or more counts could come from the background.
  
  if ~keyword_set(boundaries_string) then $
  boundaries_string   = ['0',  '0.001','0.003','0.006', '0.010','0.015',      '1']
  
  ; Since slice names are used to build file names, be sure to avoid shell special characters (e.g. []*?).
  slice_name   = [ boundaries_string+','+boundaries_string[1:*], 'CROWDED', 'REVIVED']
  num_slices   = n_elements(slice_name)
  the_crowded_slice = num_slices-2
  the_revived_slice = num_slices-1

  color_list =  ['DodgerBlue','green','cyan','yellow','red','magenta',$
                 'Cornsilk','Goldenrod','Chocolate','DarkSalmon','Peru','Sienna','Salmon', 'SandyBrown','DarkGoldenrod','Brown','IndianRed']

  slice_color                    = color_list[0:num_slices-1]
  slice_color[the_crowded_slice] = 'orange'
  slice_color[the_revived_slice] = 'Tan'



  ; Choose a Pb threshold, typically at one of the boundaries.
  invalid_threshold   = keyword_set(invalid_threshold_p) ? invalid_threshold_p : 0.01   ; 1%
  
  boundaries = float(boundaries_string)
  

  ; Remove stale critial output files at the outset, to reduce chance they will be mistakenly used.
  prune_fn = 'prune.srclist'
  file_delete, prune_fn, /ALLOW_NONEXISTENT
 
  if keyword_set(not_validated_srclist_filename) then file_delete, not_validated_srclist_filename, /ALLOW_NONEXISTENT

  ; Define energy bands for which Pb will be examined.
  template_row = {$
                  index_in_bands:0,$
                  e_low     :0.,$
                  e_high    :0.,$
                  name      :'',$
                  srclist_fn:'',$
                  region_fn :'',$
                  index_in_collation:0B}
  
  num_bands = 4 
  bands = replicate(template_row, num_bands)
  bands.index_in_bands     =            [     0,     1,     2,         3]
  bands.e_low              =            [   0.5,   0.5,   2.0,       4.0]
  bands.e_high             =            [   7.0,   2.0,   7.0,       7.0]
  bands.name               =            ['Full','Soft','Hard','VeryHard']
  bands.srclist_fn         = 'Pb_min_'+ ['full','soft','hard','vhard']+'_band.srclist'
  bands.region_fn          = 'Pb_min_'+ ['full','soft','hard','vhard']+'_band.reg'
  bands.index_in_collation =            [     0,     1,     2,         3]
  
  band_full   = bands[0]
  band_soft   = bands[1]
  band_hard   = bands[2]
  band_vhard  = bands[3]
  
    
  ; Read the columns of interest from each collated table.
  num_collations = n_elements(collated_filename)
  for jj=0L, num_collations-1 do begin
    this_filename = collated_filename[jj]
    
    if ~file_test(this_filename) then begin
      ; The VeryHard band is optional.
      print, this_filename, F='(%"ae_analyze_pb: ERROR: file %s is missing.")' 
      retall
    endif
    
    print, this_filename, F='(%"\nProcessing collation %s ...")'
  
    bt = mrdfits(this_filename, 1, /SILENT)
  
    if (jj EQ 0) then begin
      bt_template = bt[0]
    
      num_sources  = n_elements(bt)
      CATALOG_NAME =    strtrim(bt.CATALOG_NAME,2)
      LABEL        =    strtrim(bt.LABEL       ,2)
      PROVENAN     =    strtrim(bt.PROVENAN    ,2)
      
      IMMORTAL = tag_exist(bt, 'IMMORTAL') ?  bt.IMMORTAL : bytarr(num_sources)
      
      ; Pb values from individual merges fall into four categories, recorded by Pb_status:
      ; 3. Not merged  :      (MERGNUM EQ 0)              Pb unavailable because no extractions were merged.
      ; 2. Excessive overlap: (OVRLP_LO GE OVRLP_LM)      Pb suspect because OVERLAP of merged data exceeds the limit supplied to MERGE stage.
      ; 1. Pb ignored:        (SRC_CNTS LT MIN_NUM_CTS)   We refuse to interpret a small Pb value as a detection when very few counts observed.
      ; 0. Pb relevant:        NOT 1,2, or 3
  
      phot_template = {  LABEL          : '',$
                         MERGE_NAME     : '',$
                         collation_index: 0L,$
                         band_index     : 0L,$
                         THETA          : !VALUES.F_NAN,$
                         MERGNUM        : !VALUES.F_NAN,$
                         MERGFRAC       : !VALUES.F_NAN,$
                         OVRLP_LM       : !VALUES.F_NAN,$
                         OVRLP_LO       : !VALUES.F_NAN,$
                         SRC_CNTS       : !VALUES.F_NAN,$
                         NET_CNTS       : !VALUES.F_NAN,$
                         Pb             : !VALUES.F_NAN,$
                         Pb_status      : 3B }            ; Category number [0..3] from above.
                            
      phot = replicate(phot_template, num_sources, num_collations, num_bands)
      
      for jjj=0L,num_collations-1 do phot[*,jjj,*  ].collation_index = jjj
      for kkk=0L,num_bands     -1 do phot[*,  *,kkk].band_index      = kkk
                            
    endif else begin
      if (num_sources NE n_elements(bt)) || ~array_equal(LABEL, strtrim(bt.LABEL,2)) then begin
        print, this_filename, collated_filename[0], F='(%"ae_analyze_pb: ERROR: %s must contain the same sources as %s.")' 
        retall
      endif
    endelse

    ; Some collations are empty---no corresponding merges were found.
    if ~tag_exist(bt,'MERGNUM') ||  (max(bt.MERGNUM) EQ 0) then begin
      print, this_filename, F='(%"ae_analyze_pb: WARNING: %s contains no merges.")'
      continue
    endif
    
    ; Check for unexpected values in some columns.
    if ~almost_equal(/NAN, bt.OVRLP_LM, overlap_limit, TOLERANCE=0.001, DATA_RANGE=range) then begin
      print, range, overlap_limit, F='(%"ae_analyze_pb: ERROR: Some merges report OVRLP_LM values (%g ... %g) not equal to the value passed (%0.2f) !!!  Your analysis procedure is not consistent.")'
      stop
    endif


    ; Process source properties for each energy band of interest.
    for kk=0L, num_bands-1 do begin
      ; Skip VHARD band if directed.
      if (bands[kk].name EQ band_vhard.name) && ~keyword_set(pb_in_vhard_band) then continue
    
      index_in_collation = bands[kk].index_in_collation
      
      ; Check whether this energy band appears in the collation.
      if (index_in_collation GT (n_elements(bt[0].ENERG_HI)-1)) then begin
        print, this_filename, bands[kk].name, F='(%"ae_analyze_pb: WARNING: %s does not have photometry for the %s energy band.")' 
        continue
      endif

      print, bands[kk].name, F='(%"  Processing %s Band ...")'

      if ~almost_equal(bt.ENERG_LO[index_in_collation], bands[kk].e_low , DATA_RANGE=range) then print, bands[kk].name, index_in_collation, range, bands[kk].e_low, F='(%"\nWARNING: for %s Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be %0.1f keV.\n")'
      if ~almost_equal(bt.ENERG_HI[index_in_collation], bands[kk].e_high, DATA_RANGE=range) then print, bands[kk].name, index_in_collation, range, bands[kk].e_high, F='(%"\nWARNING: for %s Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be %0.1f keV.\n")'
      
      ; Store source properties for this collation and band.
      phot[*,jj,kk].LABEL      = strtrim(bt.LABEL     ,2)
      phot[*,jj,kk].MERGE_NAME = strtrim(bt.MERGE_NAME,2)
      phot[*,jj,kk].THETA      =         bt.THETA
      phot[*,jj,kk].MERGNUM    =         bt.MERGNUM
      phot[*,jj,kk].MERGFRAC   =         bt.MERGFRAC
      phot[*,jj,kk].OVRLP_LM   =         bt.OVRLP_LM
      phot[*,jj,kk].OVRLP_LO   =         bt.OVRLP_LO
      phot[*,jj,kk].SRC_CNTS   =         bt.SRC_CNTS      [index_in_collation]
      phot[*,jj,kk].NET_CNTS   =         bt.NET_CNTS      [index_in_collation]
      phot[*,jj,kk].Pb         =         bt.PROB_NO_SOURCE[index_in_collation]
      phot[*,jj,kk].Pb_status  =         0B ; "Pb is relevant" until problems are identified.
    endfor ;kk loop over bands
  endfor ;jj loop over collations

  
  
  ; Range-check all Pb values.
  if ~array_equal(finite(phot.Pb,/INFINITY), 0) then begin
    print, 'ae_analyze_pb: ERROR: Some PROB_NO_SOURCE values are Inf or -Inf!!!  There must be a bug!'
    stop
  endif
  
  if (min(phot.Pb,/NAN) LT 0) then begin
    print, 'ae_analyze_pb: ERROR: Some PROB_NO_SOURCE values are negative!!!  There must be a bug!'
    stop
  endif
  
  if (max(phot.Pb,/NAN) GT 1) then begin
    print, 'ae_analyze_pb: ERROR: Some PROB_NO_SOURCE values are >1!!!  There must be a bug!'
    stop
  endif
  
  ; If requested, we ignore all Pb values.
  if keyword_set(ignore_pb) then begin
    phot.Pb        = !VALUES.F_NAN
    phot.Pb_status = 1
  endif
  
  ; We refuse to interpret a small Pb value as a detection when very few counts observed.
  MIN_NUM_CTS   = 3
  ind = where((phot.SRC_CNTS LT MIN_NUM_CTS), count)
  if (count GT 0) then begin
    ; We want to ignore these Pb values in computation of Pb_min below so set to NaN.
    phot[ind].Pb        = !VALUES.F_NAN
    phot[ind].Pb_status = 1
  endif
  
  ; We disregard Pb values (derived from unreliable backgrounds) in merges that fail the OVERLAP limit.
  ind = where(phot.OVRLP_LO GE phot.OVRLP_LM, count)
  if (count GT 0) then begin
    ; We want to ignore these Pb values in computation of Pb_min below so set to NaN.
    phot[ind].Pb        = !VALUES.F_NAN
    phot[ind].Pb_status = 2
  endif
  
  ; No Pb values are available from empty/non-existant merges.
  ; Their Pb values should already be NaN, but we defensively set them to NaN here.
  ind = where((phot.MERGNUM EQ 0), count)
  if (count GT 0) then begin
    ; We want to ignore these Pb values in computation of Pb_min below so set to NaN.
    phot[ind].Pb        = !VALUES.F_NAN
    phot[ind].Pb_status = 3
  endif
  
  ; Find the photometry set (a band in a collation) that is the strongest detection.
  ; Identify "occasional" sources---ones that have one or more multiple-ObsID merge, that failed to validate in all multiple-ObsID merges, but do validate in one or more single-ObsID merges.

  validation_report = replicate( {$
      best_phot        : phot_template, $
      Pb_min_full_band : !VALUES.F_NAN, $
      Pb_min_soft_band : !VALUES.F_NAN, $
      Pb_min_hard_band : !VALUES.F_NAN, $
      Pb_min_vhard_band: !VALUES.F_NAN, $
      
      is_full_band_detection : 0B,$
      is_soft_band_detection : 0B,$
      is_hard_band_detection : 0B,$
      is_vhard_band_detection: 0B,$
      band_list              : '',$
      
      is_occasional               :0B,$
      tag_occasional              :'',$
      
      tally_singleObsID           :0,$
      tally_singleObsID_validation:0,$
      list_singleObsID_validation :'' }, num_sources)

    validation_report.Pb_min_full_band  = $
      min(/NAN, reform(phot[*,*, band_full.index_in_bands].Pb, num_sources, num_collations), DIM=2)
    
    validation_report.Pb_min_soft_band  = $
      min(/NAN, reform(phot[*,*, band_soft.index_in_bands].Pb, num_sources, num_collations), DIM=2)
    
    validation_report.Pb_min_hard_band  = $
      min(/NAN, reform(phot[*,*, band_hard.index_in_bands].Pb, num_sources, num_collations), DIM=2)
    
    validation_report.Pb_min_vhard_band = $
      min(/NAN, reform(phot[*,*,band_vhard.index_in_bands].Pb, num_sources, num_collations), DIM=2)
 
 
 
  for ii=0L,num_sources-1 do begin
    this_phot = reform(phot[ii,*,*], num_collations,num_bands) ; 2-D array ([collations, bands]) of phot structures for this source
    
    ;--------------------------------------------------------------------------
    ; IDENTIFY STRONGEST DETECTION
    this_Pb_min = min(/NAN, this_phot.Pb, ind_best)
    
    if finite(/NAN, this_Pb_min) then begin
      ; A "best" Pb value of NaN should occur only when all Pb values are NaN, which should be equivalent to all Pb_status values are non-zero.  Let's verify that!
      if ~array_equal(finite(/NAN,this_phot.Pb)           , 1) then message, 'BUG!' 
      if ~array_equal(            this_phot.Pb_status GT 0, 1) then begin
        print, 'ae_analyze_pb: ERROR: Some merges produced Pb=NaN, perhaps because they were missing backgrounds!!!'
        stop
      endif
 
      
      ; Since ALL Pb values are invalid (NaN), we need best_phot to record an appropriate status for the entire set of Pb values, not just the status of the random phot structure selected by min()!
      ; The Pb_status categories are nicely ordered such that the minimum value is the best characterization of the entire set of merges.
      this_status_min = min(/NAN, this_phot.Pb_status)
      mask = replicate(!VALUES.F_NAN , n_elements(this_phot))
      case this_status_min of
        0: message, 'BUG!'
        1: begin ; Among phots with this status, most SRC_CNTS defines the best photometry.
           mask[where(/NULL, this_phot.Pb_status EQ this_status_min)] = 1
           dum = max(/NAN, this_phot.SRC_CNTS * mask, ind_best)
           end
        2: begin ; Among phots with this status, smallest overlap defines the best photometry. 
           mask[where(/NULL, this_phot.Pb_status EQ this_status_min)] = 1
           dum = min(/NAN, this_phot.OVRLP_LO * mask, ind_best)
           end
        3: ; All phot structures have status "not observed".
        else: message, 'BUG!'
      endcase 
      
;forprint, this_phot.Pb, this_phot.Pb_status, round(this_phot.SRC_CNTS), this_phot.OVRLP_LO
;help, /ST, this_phot[ind_best], this_status_min 
;stop    
    endif ; this_Pb_min is Nan
    

    ; Record the "strongest detection".
    validation_report[ii].best_phot = this_phot[ind_best]
    dum = temporary(ind_best)

    ; When more than one Pb value is zero, Pb cannot measure the concept of "best".
    ; We want to report the zero Pb calculation that is most helpful to the observer.
    ; On-axis is more helpful than off-axis, so the observer will see the smallest polygon.
    ; Full-band is generally preferred over narrow bands.
    if (this_Pb_min EQ 0) then begin
      Pb_zero_phot = this_phot[ where(this_phot.Pb EQ 0, num_Pb_is_zero) ]
      if (num_Pb_is_zero GT 1) then begin
        ; Select the most on-axis Pb calculations.
        this_best_phot = Pb_zero_phot[ where(Pb_zero_phot.THETA EQ min(Pb_zero_phot.THETA)) ]
        
        ; Select the full-band (band_index 0) calculation, if possible.
        ind = where(/NULL, this_best_phot.band_index EQ 0)
        if ~isa(ind, /INTEGER) then ind = 0L
        this_best_phot = this_best_phot[ind] 
        
        ; Record the "strongest detection".
        validation_report[ii].best_phot = this_best_phot[0]
        
        ; Verify that the best_phot above matches this_Pb_min.
        if (this_Pb_min NE this_best_phot[0].Pb) then message, 'LOGIC ERROR in ae_analyze_pb!'

        ; Report the tie-break we just did.
;       print, LABEL[ii], F='(%"\nSource %s had multiple Pb calculations equal to zero:")'
;       forprint, Pb_zero_phot
;       print, 'The following Pb calculation was chosen:'
;       print, validation_report[ii].best_phot
      endif; (num_Pb_is_zero GT 1)
    endif ; (this_Pb_min EQ 0)



    ;--------------------------------------------------------------------------
    ; Identify single-ObsID Pb calculations that validate the source.
    validation_report[ii].tally_singleObsID = total(/INT, (this_phot[*,0].MERGNUM EQ 1))
    
    is_singleObsID_validation = max((this_phot.MERGNUM EQ 1) AND (this_phot.Pb LE invalid_threshold), DIM=2)
    
    ind = where(/NULL, is_singleObsID_validation, tally_singleObsID_validation)
    
    validation_report[ii].tally_singleObsID_validation = tally_singleObsID_validation
    if isa(ind, /INTEGER) then $
      validation_report[ii].list_singleObsID_validation = strjoin(repstr(this_phot[ind,0].MERGE_NAME,'EPOCH_',''),' ')
    
    ;--------------------------------------------------------------------------
    ; Identify "occasional" sources---ones that have one or more multiple-ObsID merge, that failed to validate in all multiple-ObsID merges, but do validate in one or more single-ObsID merges.
    ind_multiObsID = where(/NULL, this_phot.MERGNUM GT 1)
    if isa(ind_multiObsID, /INTEGER) then begin
      ; Multi-ObsID merges were considered.
      
      valid_in_some_multiObsID = (min(/NAN, this_phot[ind_multiObsID].Pb) LE invalid_threshold)
      
      if ~valid_in_some_multiObsID then begin
        ; Source was not validated by any multi-ObsID merge.
  
        if (validation_report[ii].best_phot.Pb LE invalid_threshold) then begin
          ; Source is valid in some single-ObsID merge, and thus called "occasional".
          
          if (validation_report[ii].best_phot.MERGNUM GT 1) then message, 'BUG!'
          
          validation_report[ii].is_occasional  = 1B
          validation_report[ii].tag_occasional = 'occasional'
        endif ; Source is valid.
      endif ; ; Source was not validated by any multi-ObsID merge.
    endif ; Multi-ObsID merges were considered.
  endfor ;ii loop over sources

  validation_comment = string(validation_report.best_phot.Pb        , F="(%'Pb_min= %0.5f')") +$
            string(validation_report.best_phot.MERGE_NAME, F="(%' from %s')")

  best_phot_return = validation_report.best_phot
  
  
  ;; ------------------------------------------------------------------------
  ;; Summarize the validation status of the sources, and decide which should be pruned now.
  
  force_into_crowded_slice = bytarr(num_sources)

  
  ; Report any sources that had no merges.
  ind = where(validation_report.best_phot.Pb_status EQ 3, count)
  if (count GT 0) then begin
    forprint, TEXTOUT=2, SUBSET=ind, CATALOG_NAME, LABEL, F="(%'%s ; (%s)')"
    print, count, F="(%'ae_analyze_pb: The %d sources above were not extracted in any ObsID; they have been placed in the CROWDED Pb slice (orange) to ensure they will be discarded.')"
    
    force_into_crowded_slice[ind] = 1B
    validation_comment      [ind] = 'not extracted in any ObsID'
  endif
  
  ; Report sources failing the OVERLAP limit in all observations.
  ind = where(validation_report.best_phot.Pb_status EQ 2, count)
  if (count GT 0) then begin
    print
    forprint, TEXTOUT=2, SUBSET=ind, CATALOG_NAME, LABEL, F="(%'%s ; (%s)')"
    print, count, F="(%'ae_analyze_pb: The %d sources above have excessive OVERLAP in all observations; they have been placed in the CROWDED Pb slice (orange) to ensure they will be discarded.')"
    
    force_into_crowded_slice[ind] = 1B
    validation_comment      [ind] = 'excessive OVERLAP in all observations'    
    ; Display the OVERLAP_LIMIT values that would be required for these sources to have data to merge.
    ;dataset_1d, id4, OVRLP_LO[band_full,ind] , BINSIZE=0.1, XTIT='OVERLAP_LIMIT required to survive', DENSITY_TITLE='excessive overlap sources'
  endif
  
  ; Propose which sources should be pruned.
  ; Classify sources into one of the Pb slices or into the CROWDED category (force_into_crowded_slice EQ 1).
  if keyword_set(ignore_pb) then begin
    prune_it = bytarr(      num_sources)
    slice    = replicate(0, num_sources)
  endif else begin
    ; Report sources too weak to produce a usable Pb.
    ind = where(validation_report.best_phot.Pb_status EQ 1, count)
    if (count GT 0) then begin
      print
      forprint, TEXTOUT=2, SUBSET=ind, CATALOG_NAME, LABEL, F="(%'%s ; (%s)')"
      print, count, F="(%'ae_analyze_pb: The %d sources above were too weak in all bands to define Pb; they have been placed in the WORST Pb slice (magenta) to ensure they will be discarded.')"
      validation_report [ind].best_phot.Pb = 0.999
      validation_comment[ind] = 'too weak in all bands to define Pb'      
    endif
     
    
    ; Identify sources with finite Pb values that fail the detection requirement.
    ; This initial list of sources to prune is modified in several places below. 
    prune_it = (validation_report.best_phot.Pb GT invalid_threshold)    ; prune_it    is  0 when Pb_min is NaN
    slice    = value_locate(boundaries, validation_report.best_phot.Pb) ; slice       is -1 when Pb_min is NaN
     
    
    if keyword_set(not_validated_srclist_filename) then begin
      ; If requested, report sources that are NOT VALIDATED by the merges provided.
      ; The caller can use this information to avoid wasting resources running additional merges on these.
      ; The "not validated" condition is NOT the same as the "prune_it" computed above!
      ; BOTH vectors are zero (false) for sources with Pb = NaN !!!!!
      pb_is_valid = (validation_report.best_phot.Pb LE invalid_threshold)

      ind = where(~pb_is_valid, count) ; "not validated"
      if (count GT 0) then begin
        forprint, TEXTOUT=not_validated_srclist_filename, SUBSET=ind, CATALOG_NAME, LABEL, validation_comment, F="(%'%s ; (%s) %s')", /NoCOMMENT
      endif else file_copy, '/dev/null', not_validated_srclist_filename
      
;forprint, textout=1, SUBSET=where(pb_is_valid), validation_report.best_phot.label, validation_report.best_phot.pb, collated_filename[validation_report.best_phot.COLLATION_INDEX], validation_report.best_phot.SRC_CNTS, bands[validation_report.best_phot.BAND_INDEX].name
;
;info, (validation_report.best_phot.SRC_CNTS)[where(pb_is_valid)]

      ; We return to the caller now, so the caller can run additional merges on these sources.
      ; Continuing on would simply waste a lot of time "reviving" condemned sources below.
      save, /COMPRESS, FILE='Pbslice.sav', num_sources, num_collations, num_bands, num_slices, bands, band_full, band_soft, band_hard, band_vhard, boundaries, bt_template, catalog_name, collated_filename, validation_comment, creator_string, extra, force_into_crowded_slice, ignore_pb, immortal, invalid_threshold, label, min_num_cts, pb_in_vhard_band, phot, provenan, prune_it, slice, slice_color, slice_name, validation_report
      
      return
    endif
  endelse ; ~keyword_set(ignore_pb)

  ; The boolean keyword IMMORTAL (in source.stats) can be used to prevent a source from being pruned.
  ind = where(/NULL, prune_it AND IMMORTAL, count)
  if isa(ind, /INTEGER) then begin
    forprint, TEXTOUT=2, SUBSET=ind, CATALOG_NAME, LABEL, F="(%'%s ; (%s)')"
    print, count, F="(%'ae_analyze_pb: The %d sources above fail the Pb requirement, but will be retained because they are declared to be IMMORTAL in their source.stats files.')"
    prune_it[ind] = 0B
    slice   [ind] = the_revived_slice
  endif


  
  ; Deal with sources we have determined should be in the "crowded" slice and should be pruned.
  ; The boolean keyword IMMORTAL (in source.stats) can be used to prevent a source from being pruned.
  ind = where(/NULL, force_into_crowded_slice AND IMMORTAL, count)
  if isa(ind, /INTEGER) then begin
    forprint, TEXTOUT=2, SUBSET=ind, CATALOG_NAME, LABEL, F="(%'%s ; (%s)')"
    print, count, F="(%'ae_analyze_pb: The %d sources above suffer excessive OVERLAP, but will be retained because they are declared to be IMMORTAL in their source.stats files.')"
    force_into_crowded_slice[ind] = 0B
    slice                   [ind] = the_revived_slice
  endif
  
  ind = where(/NULL, force_into_crowded_slice, count)
  if isa(ind, /INTEGER) then begin
    prune_it[ind] = 1
    slice   [ind] = the_crowded_slice
  endif
  
  if (min(slice) LT 0)            then message, 'ae_analyze_pb: ERROR: Pb boundaries do not span values in best_phot.Pb.'
  if (max(slice) GT num_slices-1) then message, 'ae_analyze_pb: ERROR: Pb boundaries do not span values in best_phot.Pb.'

  
  ;; ------------------------------------------------------------------------
  ; When two neighboring sources are both proposed to be pruned above, it's often the case that either one 
  ; would survive if the other disappeared.  For example, if one neighbor is pruned and the other is
  ; re-extracted, then its aperture may grow sufficiently to produce a satisfactory Pb.
  ; Thus, we wish to "revive" some of the sources proposed for pruning and let them participate in another round of catalog pruning.
  ; There are two tricky parts to this idea:
  ;
  ; 1. In the multi-ObsId case, we must be careful to identify condemned sources that have an apertures affected by a condemned neighbor.
  ;    Examining the relationship of the sources in the merge on hand is NOT sufficient---due to OVERLAP pruning
  ;    in the current merge, pathological cases exist where the sources appear to be isolated, but in some of the 
  ;    unmerged obsids they are interacting.  Thus, eliminating one source would alter the photometry of the other.
  ;    
  ; 2. Once two condemned sources are known to be interacting with each other, we must choose one of them to revive.
  ;    Choosing the brighter one (as measured by NET_CNTS) seems sensible, but again in pathological cases the 
  ;    current merge does not always reveal which is truly brighter over all the ObsIds.
  ;
  ; Thus, we are going to re-merge the list of condemned sources, without optimization and using a more liberal overlap limit, and then compare their full-band photometry to choose sources to revive.
  file_delete, 'condemned.srclist', /ALLOW_NONEXISTENT
  condemned_ind = where(prune_it, count)
  if (count GT 1) then begin
    print, F='(%"\nae_analyze_pb: ============================================================")' 
    print, 'ae_analyze_pb: MERGING and COLLATING the condemned sources to perform photometry.' 
    print, F='(%"ae_analyze_pb: ============================================================\n")'
    
    forprint, TEXTOUT=  'condemned.srclist', SUBSET=condemned_ind, CATALOG_NAME, LABEL, validation_comment, F="(%'%s ; (%s) %s')", /NoCOMMENT

    energy_range = [band_full.e_low, band_full.e_high]
    acis_extract, 'condemned.srclist', /MERGE_OBSERVATIONS, MERGE_NAME='trash', OVERLAP_LIMIT=0.5, ENERGY_RANGE=energy_range, EBAND_LO=energy_range[0], EBAND_HI=energy_range[1], /SKIP_PSF, /SKIP_NEIGHBORHOOD, /SKIP_APERTURE, /SKIP_TIMING, VERBOSE=0, _STRICT_EXTRA=extra
    
    ; Do NOT use /SINGLE_OBSID option, because we need the photometry information and we want source properties for the merge, not for the primary ObsID!
    acis_extract, 'condemned.srclist', COLLATED_FILENAME='condemned.collated', MERGE_NAME='trash', VERBOSE=0

    bt_condemned =                               mrdfits('condemned.collated', 1, /SILENT)

    ; Verify that the collation columns we need are not null.
    mm = minmax(bt_condemned.PSF_FRAC)
    if array_equal(       mm,  0) then message, 'BUG!'
    if array_equal(       mm,-99) then message, 'BUG!'
    if array_equal(finite(mm), 0) then message, 'BUG!'
    
    mm = minmax(bt_condemned.NET_CNTS)
    if array_equal(       mm,  0) then message, 'BUG!'
    if array_equal(       mm,-99) then message, 'BUG!'
    if array_equal(finite(mm), 0) then message, 'BUG!'
    
    if ~array_equal(finite(bt_condemned.RA ), 1) then message, 'BUG!  Some RA values missing; NEIGHBOR column will be damaged.'
    if ~array_equal(finite(bt_condemned.DEC), 1) then message, 'BUG!  Some RA values missing; NEIGHBOR column will be damaged.'
    if ~array_equal((bt_condemned.NEIGHBOR GE 0), 1) then message, 'BUG! NEIGHBOR column has negative values.'
    
    ; Look for pairs of condemned sources that are close enough to suspect that they are interacting (influencing each other's aperture size), and decide if we want to give one of them another chance to survive on the next pass.
    ;
    ; To judge whether the PSFs are interacting, we do not want to use a metric (e.g. DISTANCE_REG2REG) that depends on the extraction aperture sizes, which may be reduced from nominal.
    ; Instead, we will devise a neighbor-crowding metric that uses a parameterization of the PSF "size".
    
    ; In Dec 2007 I used MARX simulations at 1.5 keV with the readout streak disabled 
    ; to measure PSF fractions at 1.5 keV as a function of off-axis angle.  
    ; These polynomial curves were fit to those measurements.
    ; The off-axis angle off_angle is in arcminutes.
        this_theta = bt_condemned.THETA_HI
    neighbor_theta = this_theta[bt_condemned.NEIGHBOR]
    
    this_radius90     = (1.54 -0.315*    this_theta + 0.217*    this_theta^2) ; skypixel  
    neighbor_radius90 = (1.54 -0.315*neighbor_theta + 0.217*neighbor_theta^2) ; skypixel  
    
    ; Our circular approximation of PSF contours is an underestimate when the two sources have an unfavorable geometry; we add the arbitrary factor of 1.5 below to make the metric more conservative (i.e. revive more condemned sources).
    crowded_with_neighbor = bt_condemned.DISTANCE_SRC2SRC LT 1.5*(this_radius90 + neighbor_radius90)
    
    for ii=0L,count-1 do begin
      if crowded_with_neighbor[ii] then begin
        ; Find indexes for the pair in the full source list.
            this_ind      = condemned_ind[             ii          ]
        neighbor_ind      = condemned_ind[bt_condemned[ii].NEIGHBOR]

        
        ; Select the properties of these two sources, as measured in the all-obsid merge we just did.
            this_source = bt_condemned[             ii          ]
        neighbor_source = bt_condemned[bt_condemned[ii].NEIGHBOR]
        
        ; We define that an aperture is "reduced" if its PSF frac (in the all-obsid merge we just did) is a little smaller than the nominal 0.90 that we expect.
            this_is_reduced =      this_source.PSF_FRAC LT 0.87 
        neighbor_is_reduced =  neighbor_source.PSF_FRAC LT 0.87 

        ; Lookup photometry (measured in the all-obsid merge we just did) from the condemned collation.
            this_net_cnts =      this_source.NET_CNTS
        neighbor_net_cnts =  neighbor_source.NET_CNTS

        if            (~this_is_reduced && ~neighbor_is_reduced) then begin
          ; Neither aperture can be enlarged, so neither source should be revived.
          continue
        
        endif else if ( this_is_reduced && ~neighbor_is_reduced) then begin
          ; Revive the one source whose aperture can be enlarged.
          slice   [this_ind]     = the_revived_slice
          prune_it[this_ind]     = 0
          
        endif else if (~this_is_reduced &&  neighbor_is_reduced) then begin
          ; Revive the one source whose aperture can be enlarged.
          slice   [neighbor_ind] = the_revived_slice
          prune_it[neighbor_ind] = 0
        
        endif else if ( this_is_reduced &&  neighbor_is_reduced) then begin
          ; Both apetures can be enlarged, so revive the brighter source.
          
          ; Note that one or both sources in this pair may have already been revived, because it was involved in a second pair already considered.
          ; Nevertheless, the brighter source should be revived here.
          ; We must deal with the case where both have exactly the same net counts, which is common before backgrounds are available.
          if (this_net_cnts GT neighbor_net_cnts) then begin
            ; This source is brighter; revive it.
            slice   [this_ind]     = the_revived_slice
            prune_it[this_ind]     = 0
          endif else if (neighbor_net_cnts GT this_net_cnts) then begin
            ; Neighbor is brighter; revive it.
            slice   [neighbor_ind] = the_revived_slice
            prune_it[neighbor_ind] = 0
          endif else if (prune_it[this_ind] && prune_it[neighbor_ind]) then begin
            ; This source and neighbor have exactly the same net counts, and both are still condemned; revive this source.
            slice   [this_ind]     = the_revived_slice
            prune_it[this_ind]     = 0
          endif
        
        endif else message, 'BUG in ae_analyze_pb'
        
      endif ; close pair of sources
    endfor ;ii loop through condemned sources
  endif ; prune_it flag is true for some sources
  

  ; Record which bands validated each source.
  validation_report.is_full_band_detection = ~prune_it AND (validation_report.Pb_min_full_band  LE invalid_threshold)
  validation_report.is_soft_band_detection = ~prune_it AND (validation_report.Pb_min_soft_band  LE invalid_threshold)
  validation_report.is_hard_band_detection = ~prune_it AND (validation_report.Pb_min_hard_band  LE invalid_threshold)
  validation_report.is_vhard_band_detection= ~prune_it AND (validation_report.Pb_min_vhard_band LE invalid_threshold)

  validation_report.band_list = 'valid in '

  ind = where(/NULL, validation_report.is_soft_band_detection )
  if isa(ind, /INTEGER) then $
  validation_report[ind].band_list += 'Soft,'

  ind = where(/NULL, validation_report.is_hard_band_detection )
  if isa(ind, /INTEGER) then $
  validation_report[ind].band_list += 'Hard,'

  ind = where(/NULL, validation_report.is_vhard_band_detection)
  if isa(ind, /INTEGER) then $
  validation_report[ind].band_list += 'Vhard,'

  ; A full-band detection replaces the narrow band labels.
  ind = where(/NULL, validation_report.is_full_band_detection )
  if isa(ind, /INTEGER) then $
  validation_report[ind].band_list = 'valid in Full'

  ; Mark sources recommended for pruning OR tagged as "REVIVED".
  ind = where(/NULL, prune_it OR (slice EQ the_revived_slice))
  if isa(ind, /INTEGER) then $
  validation_report[ind].band_list = 'not validated'

  ; Remove trailing ',' characters.
  ; Note that mg_streplace function is from https://raw.github.com/mgalloy/mglib/master/src/strings/mg_streplace.pro
  for ii=0L, num_sources-1 do validation_report[ii].band_list = mg_streplace(validation_report[ii].band_list, ',$', '')


  ; Tabulate how many sources are in each proposed Pb slice.
  band_list_categories = (validation_report.band_list)[uniq(   validation_report.band_list,$
                                                          sort(validation_report.band_list))]
  num_categories  = n_elements(band_list_categories)
  
  table = intarr(num_slices, num_categories+1)
  for ii=0L,num_slices-1 do begin
    ind = where(slice EQ ii, count)
    if (count GT 0) then begin
      for jj=0L,num_categories-1 do $
        table[ii,jj] = round(total((validation_report.band_list)[ind] EQ band_list_categories[jj]))
    endif
  endfor ;ii loop over slices
     
  category_total = total(table,1)
  print
; print, 'Pb slice boundaries    :'
; forprint, boundaries, F="(%'  %5.1f %%')"
  print, invalid_threshold, F='(%"Pb threshold: %0.4f ")'
  print
  print, '', 'Pb    :', '['+slice_name+']', F='(A30,A10,20(A14))'
  print, '', 'total |', slice_color, F='(A30,A10,20(A14))'
  for jj=0L,num_categories-1 do print, band_list_categories[jj], category_total[jj],'|',table[*,jj], F='(A30,I8,A2,20(I14))'
  print, '  ---------------------------------------------------------------------------------------', F='(30x,A)'
  print, total(table),total(table,2), F='(30x,I8,2x,20(I14))'
  print
  
  ; Record list of recommended prunes.
  prune_ind = where(prune_it, count)
  if (count GT 0) then begin
    ; Sort this source list by off-axis angle, so that the observer will not have to frequently change bin size during a visual review in ds9.
    forprint, TEXTOUT=prune_fn, SUBSET=prune_ind[sort((validation_report.best_phot.THETA)[prune_ind])], CATALOG_NAME, LABEL, validation_comment, F="(%'%s ; (%s) %s')", /NoCOMMENT
  endif else file_copy, '/dev/null', prune_fn
  
  ; Save important data structures, for the MAKE_REGIONS call and for future investigations.
  save, /COMPRESS, FILE='Pbslice.sav', num_sources, num_collations, num_bands, num_slices, bands, band_full, band_soft, band_hard, band_vhard, boundaries, bt_condemned, bt_template, catalog_name, collated_filename, validation_comment, creator_string, extra, force_into_crowded_slice, ignore_pb, immortal, invalid_threshold, label, min_num_cts, pb_in_vhard_band, phot, provenan, prune_it, slice, slice_color, slice_name, validation_report, band_list_categories, num_categories, table     


  print, 'Saved validation information to Pbslice.sav'
  

  ;; ------------------------------------------------------------------------
  ;; Create a symlink in each source directory identifying the merge that produced the strongest validation.
  most_valid_basename = 'most_valid'
  most_valid_symlink = CATALOG_NAME +'/'+most_valid_basename
  
  file_delete, most_valid_symlink, /ALLOW_NONEXISTENT
  ind = where(validation_report.best_phot.MERGNUM GT 0, count)
  if (count GT 0) then  file_link, (validation_report.best_phot.MERGE_NAME)[ind], most_valid_symlink[ind]
  
  return
endif ; ANALYZE Stage



if keyword_set(make_regions) then begin
  run_command
  print
  restore, 'Pbslice.sav'
  
  ; Make a region file that slices the data in Pb so we can visualize the sources that would be retained for various pruning thresholds we might choose.
  ; We separate this long-running code from the similar tabulation code above to allow the observer to adjust the Pb slices efficiently. 
  pb_region_fn = 'Pbslice.reg'
  file_delete, pb_region_fn, /ALLOW_NONEXISTENT
  ; Process the slices in the order that puts the prune regions in "front" in ds9.
  for ii=num_slices-1, 0, -1 do begin
    basename='Pbslice_'+slice_name[ii]
    slice_srclist_fn = strcompress(/REMOVE_ALL, basename+'.srclist')  
    file_delete, slice_srclist_fn, /NOEXPAND_PATH, /ALLOW_NONEXISTENT
  
    ind = where(slice EQ ii, count)
    
   ;print, slice_name[ii], count, F='(%"Slice %s has %d sources.")'
    
    if (count GT 0) then begin
      ; Sort this slice by off-axis angle, so that the observer will not have to frequently change bin size during a visual review in ds9.
      ind = ind[sort((validation_report.best_phot.THETA)[ind])]

      this_validation_report = validation_report[ind]
      
      ; Build a sourcelist and region file for this slice, tagged by the energy band that produced best_phot.
      ; Each polygon region will come from the merge that produced best_phot..
      region_tag      = strarr(count,5)
      region_tag[*,0] = basename
      region_tag[*,1] = 'most valid band ' +bands[ this_validation_report.best_phot.band_index ].name
      region_tag[*,2] = 'most valid merge '+       this_validation_report.best_phot.MERGE_NAME
      region_tag[*,3] =                            this_validation_report.tag_occasional      
      region_tag[*,4] =                            this_validation_report.band_list

      forprint, TEXTOUT=slice_srclist_fn, SUBSET=ind, CATALOG_NAME, LABEL, validation_comment, F="(%'%s ; (%s) %s')", /NoCOMMENT
 
      acis_extract, slice_srclist_fn, MERGE_NAME=this_validation_report.best_phot.MERGE_NAME, COLLATED_FILENAME='/dev/null', REGION_FILE=temp_region_fn, REGION_TAG=region_tag, MATCH_EXISTING=bt_template, VERBOSE=0
      
      ; Color-code these regions.
      cmd = string(temp_region_fn, slice_color[ii], pb_region_fn, F='(%"egrep ''label|cat|polygon'' %s | sed -e ''s/DodgerBlue/%s/'' >> %s")')
      run_command, /QUIET, cmd
    endif else begin
      file_copy, '/dev/null', slice_srclist_fn, /NOEXPAND_PATH
    endelse
  endfor ;ii loop over slices
  
  ; Display the Pb slices on the multi-obsid full-band data in ds9.
  cmd = string(keyword_set(ds9_title) ? ds9_title : 'ae_analyze_pb', pb_region_fn, pb_region_fn,  F='(%"ds9 -title \"%s: %s\"  -log -bin factor 4 ../target.evt -region %s -frame center >& /dev/null &")')
  ;print, cmd
  run_command, /QUIET, cmd 
  
  
  
  if ~keyword_set(ignore_pb) then begin
    print
    ; Create source lists that will allow the observer to review each source in the energy band in which it is most significant (Pb EQ best_phot.Pb).
    file_delete, bands.srclist_fn, /ALLOW_NONEXISTENT
    file_delete, bands.region_fn , /ALLOW_NONEXISTENT
    
    for kk=0L, num_bands-1 do begin                       
      ind = where( bands[ validation_report.best_phot.band_index ].name EQ bands[kk].name, count)
      
      print, count, bands[kk].name, F='(%"%d sources are most significant in %s Band.")'
      if (count GT 0) then begin
        forprint, TEXTOUT=bands[kk].srclist_fn, SUBSET=ind[reverse(sort((validation_report.best_phot.Pb)[ind]))], CATALOG_NAME, LABEL, validation_comment, F="(%'%s ; (%s) %s')", /NoCOMMENT
        
        ; Gather the regions that are tagged with this band name.
        cmd = string(bands[kk].name, pb_region_fn, bands[kk].region_fn, F='(%"grep -i ''most valid band %s'' %s > %s")')
        run_command, /QUIET, cmd
          
        ; Append point regions that mark the source positions.
        cmd = string( pb_region_fn, bands[kk].region_fn, F='(%"grep ''cat'' %s >> %s")')
        run_command, /QUIET, cmd
        
      endif else begin
        file_copy, /OVERWRITE, '/dev/null', bands[kk].srclist_fn
        file_copy, /OVERWRITE, '/dev/null', bands[kk].region_fn
      endelse
    endfor ; kk loop over bands
  
  
    ; Make sourcelist and region file for sources that survived only in a narrow band (i.e. not in full band).
    narrow_srclist_fn = 'Pb_valid_band.srclist'
    narrow_region_fn  = 'Pb_valid_band.reg'
    file_delete, [narrow_srclist_fn,narrow_region_fn], /ALLOW_NONEXISTENT
    file_copy, /OVERWRITE, '/dev/null', narrow_srclist_fn
    file_copy, /OVERWRITE, '/dev/null', narrow_region_fn
   
    num_narrow_band = total((validation_report.is_soft_band_detection OR $
                             validation_report.is_hard_band_detection OR $
                             validation_report.is_vhard_band_detection) AND $
                            ~validation_report.is_full_band_detection, /INT)
    print, num_narrow_band, ' sources detected only in a narrow band'
    
    
    forprint, TEXTOUT=narrow_srclist_fn, CATALOG_NAME, LABEL, validation_comment, validation_report.band_list, F="(%'%s ; (%s) %s %s')", /NoCOMMENT

    ; Here we create a collation of the specific merge that validated each source.
    ; This collation DOES NOT contain the best Pb in each energy band, since different merges may produce the best Pb in each band.
    
    acis_extract, narrow_srclist_fn, MERGE_NAME=validation_report.best_phot.MERGE_NAME, COLLATED_FILENAME='/dev/null', REGION_FILE=temp_region_fn, REGION_TAG=validation_report.band_list, MATCH_EXISTING=bt_template, VERBOSE=0

    run_command, /QUIET, "egrep 'label|cat|polygon' "+temp_region_fn+" | sed -e '/Full/s/DodgerBlue/grey/' -e '/{valid in Soft}/s/DodgerBlue/red/'  -e '/hard/s/DodgerBlue/DodgerBlue/'   -e '/not validated/s/DodgerBlue/yellow/'    >> "+narrow_region_fn

    if (num_narrow_band GT 0) then begin
      cmd = string(narrow_region_fn, narrow_region_fn, F="(%'ds9 -title ""ae_analyze_pb: %s"" -log -bin factor 4 ""../target.soft.evt""  ""../target.hard.evt"" -regions load all %s -frame center >& /dev/null &')" )
      ;print, cmd
      ;run_command, /QUIET, cmd 
    endif ; (num_narrow_band GT 0) 
    
    
    
    
    ; Build a region file that represents "occasional" sources---ones that have one or more multiple-ObsID merge, that failed to validate in all multiple-ObsID merges, but do validate in one or more single-ObsID merges. Use the polygon from the ObsID that produced the strongest validation. 
    ind = where(validation_report.is_occasional, count)
    
    print, count, F='(%"\n%d sources (in occasional.srclist) are invalid in all their multi-ObsID merges but are valid in one or more single-ObsID merges.")'
    
    basename='occasional'
    occasional_srclist_fn = strcompress(/REMOVE_ALL, basename+'.srclist')
    occasional_region_fn  = strcompress(/REMOVE_ALL, basename+'.reg'    )
    file_delete, [occasional_srclist_fn,occasional_region_fn], /ALLOW_NONEXISTENT
    
    if (count GT 0) then begin
      ; Show the single-ObsID validations in the comment string.
      validation_comment[ind] = validation_comment[ind] + '; occasional (' + $
                                validation_report[ind].list_singleObsID_validation + ')'
    
      ; Sort these sources by Pb, so that the observer can see the most significant sources first during a visual review.
      ind = ind[sort((validation_report.best_phot.Pb)[ind])]
    
      this_validation_report = validation_report[ind]

      ; Each polygon region will come from the single-ObsID merge that produced the smallest Pb.
      region_tag      = strarr(count,3)
      region_tag[*,0] = basename
      region_tag[*,1] = 'most valid band ' +bands[ this_validation_report.best_phot.band_index ].name
      region_tag[*,2] = 'most valid merge '+       this_validation_report.best_phot.MERGE_NAME
      
      forprint, TEXTOUT=occasional_srclist_fn, SUBSET=ind, CATALOG_NAME, LABEL, validation_comment, F="(%'%s ; (%s) %s')", /NoCOMMENT
 
      acis_extract, occasional_srclist_fn, MERGE_NAME=this_validation_report.best_phot.MERGE_NAME, COLLATED_FILENAME='/dev/null', REGION_FILE=temp_region_fn, REGION_TAG=region_tag, MATCH_EXISTING=bt_template, VERBOSE=0
      
      ; Color-code these regions.
      cmd = string(temp_region_fn, occasional_region_fn, F='(%"egrep ''label|cat|polygon'' %s | sed -e ''s/DodgerBlue/red/'' >> %s")')
      run_command, /QUIET, cmd
    endif else begin
      file_copy, '/dev/null', occasional_srclist_fn
      file_copy, '/dev/null', occasional_region_fn
    endelse
  endif ;  ~keyword_set(ignore_pb) 

  
  ; Report how many sources were valid in each merge.
  print
  for jj=0L, num_collations-1 do $
    print, total(/INT, (min(/NAN, reform(phot[*,jj,*].Pb, num_sources,num_bands), DIM=2) LE invalid_threshold)), collated_filename[jj],  F='(%"%5d sources were valid in the merge %s.")'

  ; Print the Pb table again.
  category_total = total(table,1)
  print
; print, 'Pb slice boundaries    :'
; forprint, boundaries, F="(%'  %5.1f %%')"
  print, invalid_threshold, F='(%"Pb threshold: %0.4f ")'
  print
  print, '', 'Pb    :', '['+slice_name+']', F='(A30,A10,20(A14))'
  print, '', 'total |', slice_color, F='(A30,A10,20(A14))'
  for jj=0L,num_categories-1 do print, band_list_categories[jj], category_total[jj],'|',table[*,jj], F='(A30,I8,A2,20(I14))'
  print, '  ---------------------------------------------------------------------------------------', F='(30x,A)'
  print, total(table),total(table,2), F='(30x,I8,2x,20(I14))'
  print
endif ; keyword_set(make_regions)
return
end  ; ae_analyze_pb




;#############################################################################
;;; When /CHOOSE_PREFERRED_METHOD is supplied, this tool will decide which type of position estimate
;;; is preferred for each source, using tables/position.collated built in reposition_block1.

;;; When /CHOOSE_POSITION is supplied, a position estimate will be chosen for each source, based on
;;; the a priori preference and on the availability of actual estimates.
;;; Source lists are also created to guide visual review of the position updates.  See recipe.txt.


;#############################################################################
PRO ae_improve_positions, CHOOSE_PREFERRED_METHOD=choose_preferred_method, RECALCULATE_ALL=recalculate_all,$
                          CHOOSE_POSITION=choose_position, _EXTRA=extra

exit_code = 0
creator_string = "ae_improve_positions, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)

temp_text_fn     = tempdir   + 'temp.txt'
temp_collate_fn  = tempdir   + 'most_valid_merge.collated'
 
run_command, PARAM_DIR=tempdir


position_collatefile = 'tables/position.collated'

if keyword_set(choose_preferred_method) then begin
    bt = mrdfits(position_collatefile, 1, /SILENT)
    num_sources  = n_elements(bt)
    LABEL        = strtrim(bt.LABEL       ,2)
    CATALOG_NAME = strtrim(bt.CATALOG_NAME,2)
    POSNTYPE     = strtrim(bt.POSNTYPE    ,2)
    
    ;; Decide which position type we prefer for each source.
    prefer_eye   = (POSNTYPE    EQ 'eye'      ) OR $ ; Observer assigned position in context of recipe.txt.
                   (POSNTYPE    EQ 'published')      ; Observer assigned a published position. 
                 
    prefer_recon = (bt.PSF_FRAC LT 0.87) AND ~prefer_eye ; Aperture is reduced, so recon is likely to be best.     
    prefer_corr  = (bt.THETA    GT 5   ) AND ~prefer_eye AND ~prefer_recon ; Aperture is full size, but off-axis.
    prefer_data  =                           ~prefer_eye AND ~prefer_recon AND ~prefer_corr ; Aperture is full size, and on-axis.
    

    ;; Decide which position estimates we need to satisfy the preferences above.
    
    ; As of 2011 Oct our policy is to reposition all sources with RECON positions, because we find that the details of the reconstructed image often change when a source is moved, because the bin phase of the image is changed.
    need_recon = prefer_recon
    
    ; Note that it's reasonable to think that CORR positions are stable from pass to pass:
    ;  - CORR positions are independent of the source's aperture
    ;  - Although the neighborhood image may change when a source moves (because the bin phase changes), the CORR position is computed on an image with really small bins.
    ;
    ; However, we've decided to recalculate all the CORR positions, since that is pretty fast.
    need_corr  = prefer_corr
    
    ; Centroid positions are easily affected by moving or resizing the aperture, so they are always recomputed.
    need_data  = prefer_data
    
    
    if keyword_set(recalculate_all) then begin
      need_recon = prefer_recon
      need_corr  = prefer_corr
      need_data  = prefer_data
    endif
    
    
    ;; Decide which position types will be calculated for each source in order to meet the needs above and to provide the observer with additional position estimates that might be helpful to see during visual review.
    ; We do NOT want to run the CHECK_POSITIONS AE stage twice on the same source.
    ; Our recipe will be performing three CHECK_POSITIONS runs:
    ;  1. where compute_recon is true we will generate recon, correlation, and data positions
    ;  2. where compute_corr  is true we will generate        correlation  and data positions
    ;  3. where compute_data  is true we will generate                         data positions
    
    
    ; Group #1 should include all "prefer_eye" sources so that the user can see all possible position estimates when they review their previous by-eye position.
    
    ; Group #1 will also include sources with a close neighbor (defined using the same crition that raises the suspicion metric in the "choose_position" section of this tool), since CORR and DATA position estimates might be corrupted.
    has_close_neighbor = (bt.DISTANCE_REG2REG LT bt.SRC_RAD)

    ; Group #1 will also include sources with any piled extraction.
    is_piled = bytarr(num_sources)
    fn = 'possibly_piled.srclist'
    if logical_true( file_lines(fn) ) then begin
      readcol, fn, piled_sourcename, FORMAT='A', COMMENT=';' 
      ; Trim whitespace and remove blank lines.
      piled_sourcename = strtrim(piled_sourcename,2)

      foreach this_name, piled_sourcename[ where(/NULL, piled_sourcename NE '') ] do $
        is_piled[ where(/NULL, CATALOG_NAME EQ this_name) ] = 1B
      
      ind = where(/NULL, is_piled)
      if isa(ind, /INTEGER) then begin
        print, F='(%"\nWill reconstruct neighborhoods for the following piled sources:")'
        forprint, SUBSET=ind, CATALOG_NAME, LABEL, F='(%"  %s (%s)")'
      endif
    endif ; possibly_piled.srclist processed

    compute_recon = (need_recon OR prefer_eye OR has_close_neighbor OR is_piled) ; group #1

    compute_corr  = (need_corr) AND ~compute_recon                   ; group #2

    compute_data  = (need_data) AND ~compute_recon AND ~compute_corr ; group #3
    
    
    ; Summarize results for the observer, and create lists of sources requiring calculation of each type of position estimate.
    ; The files need_*.srclist are disjoint; we don't want to run the CHECK_POSITIONS AE stage twice on the same source.
    
    preferred_method = strarr(num_sources)
    ind = where(prefer_eye, count_eye) 
    if (count_eye GT 0) then begin
      print, count_eye, F='(%"The current positions of the following %d sources were estimated ''by-eye.''")' 
      preferred_method[ind] = POSNTYPE[ind]
      forprint, CATALOG_NAME, LABEL,  F="(%'%s (%s)')", SUBSET=ind, /NoCOMMENT
    endif  
    
 
    fn = 'need_recon.srclist'
    ind = where(prefer_recon, count_recon) 
    if (count_recon GT 0) then preferred_method[ind] = 'ML'
      
    ind = where( compute_recon, count_recalc)
    print, count_recon, count_recalc, F='(%"RECON positions are preferred for %d sources.\nRECON positions must be calculated for %d sources.")'
    if (count_recalc EQ 0) then begin
      ; Create an empty file.
      openw   , unit, fn, /GET_LUN
      free_lun, unit
    endif else begin
      forprint, TEXTOUT=fn, CATALOG_NAME, LABEL,  F="(%'%s ; (%s)')", SUBSET=ind, /NoCOMMENT
    endelse
    
    
    fn = 'need_corr.srclist'
    ind = where(prefer_corr, count_corr) 
    if (count_corr GT 0) then preferred_method[ind] = 'CORR'
      
    ind = where( compute_corr, count_recalc)
    print, count_corr, count_recalc, F='(%"CORR positions are preferred for %d sources.\nCORR positions must be calculated for %d sources.")'
    if (count_recalc EQ 0) then begin
      ; Create an empty file.
      openw   , unit, fn, /GET_LUN
      free_lun, unit
    endif else begin
      forprint, TEXTOUT=fn, CATALOG_NAME, LABEL,  F="(%'%s ; (%s)')", SUBSET=ind, /NoCOMMENT
    endelse
                             
    
    fn = 'need_data.srclist'
    ind = where(prefer_data, count_data) 
    if (count_data GT 0) then preferred_method[ind] = 'MEAN DATA'
      
    ind = where( compute_data, count_recalc)
    print, count_data, count_recalc, F='(%"MEAN DATA positions are preferred for %d sources.\nMEAN DATA positions must be calculated for %d sources.")'
    if (count_recalc EQ 0) then begin
      ; Create an empty file.
      openw   , unit, fn, /GET_LUN
      free_lun, unit
    endif else begin
      forprint, TEXTOUT=fn, CATALOG_NAME, LABEL,  F="(%'%s ; (%s)')", SUBSET=ind, /NoCOMMENT
    endelse
    
    
    save, /COMPRESS, FILE='ae_improve_positions.sav', preferred_method, prefer_eye, prefer_recon, prefer_corr, prefer_data, need_recon, need_corr, need_data, compute_recon, compute_corr, compute_data, num_sources, CATALOG_NAME, LABEL
endif ; choose_preferred_method



if keyword_set(choose_position) then begin
    restore, 'ae_improve_positions.sav'
    
    bt = mrdfits(position_collatefile, 1, /SILENT)
    num_sources  = n_elements(bt)
    
    POSNTYPE = strtrim(bt.POSNTYPE    ,2)

    ; If collation includes photometry, then we need to use the correct energy band for some properties.
    if tag_exist(bt, 'ENERG_LO') then begin
      band_full = 0
      if ~almost_equal(bt.ENERG_LO[band_full], 0.5, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
      if ~almost_equal(bt.ENERG_HI[band_full], 8.0, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 8.0 keV.\n")'

      SRC_CNTS = bt.SRC_CNTS[band_full]
    endif else begin
      SRC_CNTS = bt.SRC_CNTS
    endelse
    
    RA       = bt.RA
    DEC      = bt.DEC
    
    RA_ML    = tag_exist(bt, 'RA_ML')    ? bt.RA_ML    : replicate(!VALUES.F_NAN, num_sources)
    DEC_ML   = tag_exist(bt, 'DEC_ML')   ? bt.DEC_ML   : replicate(!VALUES.F_NAN, num_sources)
    recon_available = finite(RA_ML)   AND finite(DEC_ML)   AND (RA_ML   NE 0) AND (DEC_ML   NE 0)
    
    RA_CORR  = tag_exist(bt, 'RA_CORR')  ? bt.RA_CORR  : replicate(!VALUES.F_NAN, num_sources)
    DEC_CORR = tag_exist(bt, 'DEC_CORR') ? bt.DEC_CORR : replicate(!VALUES.F_NAN, num_sources)
    corr_available  = finite(RA_CORR) AND finite(DEC_CORR) AND (RA_CORR NE 0) AND (DEC_CORR NE 0)
    
    RA_DATA  = tag_exist(bt, 'RA_DATA')  ? bt.RA_DATA  : replicate(!VALUES.F_NAN, num_sources)
    DEC_DATA = tag_exist(bt, 'DEC_DATA') ? bt.DEC_DATA : replicate(!VALUES.F_NAN, num_sources)
    data_available  = finite(RA_DATA) AND finite(DEC_DATA) AND (RA_DATA NE 0) AND (DEC_DATA NE 0)
    
    use_eye       = prefer_eye
 
    use_recon     = need_recon AND  recon_available
    no_estimate   = need_recon AND ~recon_available

    use_corr      = need_corr  AND  corr_available
    no_estimate OR= need_corr  AND ~corr_available
    
    use_data      = need_data  AND  data_available
    no_estimate OR= need_data  AND ~data_available
    
    use_catalog   = ~use_eye AND ~use_recon AND ~use_corr AND ~use_data AND ~no_estimate
    use_catalog_comment = strarr(num_sources)

    ; Do not move sources marked as IMMORTAL.
    ; The boolean keyword IMMORTAL (in source.stats) can be used to prevent a source from being pruned.
    IMMORTAL = tag_exist(bt, 'IMMORTAL') ?  bt.IMMORTAL : bytarr(num_sources)
    
    ind = where(/NULL, IMMORTAL, count)
    if isa(ind, /INTEGER) then begin
      ; Record which position estimate was recommended for each of these sources.
      foreach this_ind, ind do begin
        case 1 of
          use_eye    [this_ind]: use_catalog_comment[this_ind] = 'use eye   position'
          use_recon  [this_ind]: use_catalog_comment[this_ind] = 'use recon position'
          use_corr   [this_ind]: use_catalog_comment[this_ind] = 'use corr  position'
          use_data   [this_ind]: use_catalog_comment[this_ind] = 'use data  position'
          no_estimate[this_ind]: use_catalog_comment[this_ind] = 'no estimate available'
          else: 
        endcase
      endforeach ; this_ind
      
      ; Move these sources to the "use catalog" category.
      use_catalog[ind] = 1B
      use_eye    [ind] = 0B
      use_recon  [ind] = 0B
      use_corr   [ind] = 0B
      use_data   [ind] = 0B
      no_estimate[ind] = 0B
      
      print, count, F="(%'ae_improve_positions: %d sources will not be moved because they are declared to be IMMORTAL in their source.stats files.')"
    endif ; IMMORTAL = True
    
    ; Verify that use_eye,use_recon,use_corr,use_data,no_estimate,use_catalog are disjoint.
    if (total(/INT, use_eye+use_recon+use_corr+use_data+no_estimate+use_catalog) NE num_sources) then message, 'BUG in  ae_improve_positions!'
    
    

    ;Define a suspicion metric to prioritize our visual review (since we'll probably get tired of looking at sources).
    ;
    ; 1. The basic suspicion metric is distance the source will move, normalized by the aperture size.  
    ; 
    ; 2. Sources with a reasonably close neighbor are at some risk that their position CORR and DATA estimates are corrupted, and are thus assigned a higher suspicion than isolated ones.
    ; 
    ; 3. Among the recon sources, we may wish the observer to review the existence of sources with large Pb, so they are assigned a higher suspicion than more significant sources.
     
    deghour = 24D/360
    gcirc, 1, RA*deghour, DEC, RA_ML  *deghour, DEC_ML  , ml_metric
    gcirc, 1, RA*deghour, DEC, RA_CORR*deghour, DEC_CORR, corr_metric
    gcirc, 1, RA*deghour, DEC, RA_DATA*deghour, DEC_DATA, data_metric
    ml_metric   /= (bt.SRC_RAD * 0.495)
    corr_metric /= (bt.SRC_RAD * 0.495) ; gcirc returns arcsec, SRC_RADIUS is in skypix
    data_metric /= (bt.SRC_RAD * 0.495)
    ind = where( bt.DISTANCE_REG2REG LT bt.SRC_RAD, count )
    if (count GT 0) then begin
      corr_metric[ind] += 10
      data_metric[ind] += 10
    endif
    

    ; Determine the best Pb value for each source.
    ; Since previously-computed 'Pbslice.sav' files probably contain more sources than we have now (and are probably archived in a "pass" directory), we recompute the best Pb values now.
    print, F='(%"\nae_improve_positions: Collating most_valid merges to obtain Pb_min values.")'
    
    forprint, TEXTOUT=temp_text_fn, CATALOG_NAME, LABEL, F="(%'%s ; (%s)')", /NoCOMMENT
    
    acis_extract, temp_text_fn, MERGE_NAME='most_valid', COLLATED_FILENAME=temp_collate_fn, MATCH_EXISTING=1, VERBOSE=1
    
    par = ae_get_target_parameters()
    ; Supply NOT_VALIDATED_SRCLIST_FILENAME to tell ae_analyze_pb to not waste time "reviving" condemned sources.
    ae_analyze_pb, /ANALYZE, temp_collate_fn, OVERLAP_LIMIT=0.10,  PB_IN_VHARD_BAND=par.PB_IN_VHARD_BAND, NOT_VALIDATED_SRCLIST_FILENAME='not_validated.srclist', BEST_PHOT=best_phot
    
    PROB_NO_SOURCE = best_phot.Pb
      
    if (n_elements(PROB_NO_SOURCE) NE num_sources) then begin
      print, savefile, position_collatefile, F='(%"ERROR: The files %s and %s contain different numbers of sources.")'
      retall
    endif
    
    ind = where(PROB_NO_SOURCE GT 0.003, count )
    if (count GT 0) then begin
      ml_metric[ind] += 10
    endif

    ; Resave data structures to document recommendations.
    save, /COMPRESS, FILE='ae_improve_positions.sav'
    print
    
    ;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    ;Segment the catalog into five sources lists, each sorted by suspicion so that we can visually review them later.
    ;Make region files that color-code the five categories of sources.

    file_delete, 'use_eye.srclist', 'use_recon.srclist', 'use_data.srclist', 'use_corr.srclist', 'no_estimate.srclist', 'use_catalog.srclist', 'use_eye.reg', 'use_recon.reg', 'use_data.reg', 'use_corr.reg', 'no_estimate.reg', 'use_catalog.reg', 'use_eye.collated', 'use_recon.collated', 'use_data.collated', 'use_corr.collated', 'no_estimate.collated', 'use_catalog.collated', 'review.reg', /QUIET
     
    ind = where(use_eye, count) 
    if (count GT 0) then begin
      forprint, TEXTOUT='use_eye.srclist', CATALOG_NAME, LABEL, F="(%'%s ; (%s)')", SUBSET=ind, /NoCOMMENT
      
      acis_extract, 'use_eye.srclist', MERGE_NAME='position', COLLATED_FILENAME='use_eye.collated', REGION_TAG='use_eye', REGION_FILE='use_eye.reg', MATCH_EXISTING=bt, VERBOSE=0
   
      run_command, 'sed -e "s/DodgerBlue/yellow/"   use_eye.reg |grep -v NaN >> review.reg'

      print, count, F='(%">>>>> %d sources will retain positions defined by the observer.")' 
    endif else file_copy, '/dev/null', 'use_eye.srclist'

    ind = where(use_recon, count) 
    if (count GT 0) then begin
      ind = ind[ reverse(sort(ml_metric[ind])) ]
      forprint, TEXTOUT='use_recon.srclist', CATALOG_NAME, LABEL, ml_metric, F="(%'%s ; (%s) %4.1f')", SUBSET=ind, /NoCOMMENT
      
      acis_extract,    'use_recon.srclist', MERGE_NAME='position', COLLATED_FILENAME=   'use_recon.collated', REGION_TAG='use_recon'   , REGION_FILE=   'use_recon.reg', MATCH_EXISTING=bt, VERBOSE=0
      
      run_command, 'sed -e "s/DodgerBlue/green/"     use_recon.reg |grep -v NaN >> review.reg'
      
      print, count, F='(%">>>>> %d sources will move to RECON position.")' 
      
      dataset_1d, id_ra , (3600*(RA  -  RA_ML)*cos(DEC/!RADEG))[ind], DATASET='use_recon', BINSIZE=0.01
      dataset_1d, id_dec, (3600*(DEC - DEC_ML)                )[ind], DATASET='use_recon', BINSIZE=0.01
    endif else file_copy, '/dev/null', 'use_recon.srclist'

    ind = where(use_corr, count) 
    if (count GT 0) then begin
      ind = ind[ reverse(sort(corr_metric[ind])) ]
      forprint, TEXTOUT='use_corr.srclist', CATALOG_NAME, LABEL, corr_metric, F="(%'%s ; (%s) %4.1f')", SUBSET=ind, /NoCOMMENT
      
      acis_extract,    'use_corr.srclist', MERGE_NAME='position', COLLATED_FILENAME=   'use_corr.collated', REGION_TAG='use_corr'   , REGION_FILE=   'use_corr.reg', MATCH_EXISTING=bt, VERBOSE=0
  
      run_command, 'sed -e "s/DodgerBlue/DodgerBlue/"    use_corr.reg |grep -v NaN >> review.reg'

      print, count, F='(%">>>>> %d sources will move to CORRELATION position.")' 
      
      dataset_1d, id_ra , (3600*(RA  -  RA_CORR)*cos(DEC/!RADEG))[ind], DATASET='use_corr', BINSIZE=0.01
      dataset_1d, id_dec, (3600*(DEC - DEC_CORR)                )[ind], DATASET='use_corr', BINSIZE=0.01
    endif else file_copy, '/dev/null', 'use_corr.srclist'

    ind = where(use_data, count) 
    if (count GT 0) then begin
      ind = ind[ reverse(sort(data_metric[ind])) ]
      forprint, TEXTOUT='use_data.srclist', CATALOG_NAME, LABEL, data_metric, F="(%'%s ; (%s) %4.1f')", SUBSET=ind, /NoCOMMENT
      
      acis_extract,    'use_data.srclist', MERGE_NAME='position', COLLATED_FILENAME=   'use_data.collated', REGION_TAG='use_data'   , REGION_FILE=   'use_data.reg', MATCH_EXISTING=bt, VERBOSE=0
      
      run_command, 'sed -e "s/DodgerBlue/cyan/"     use_data.reg |grep -v NaN >> review.reg'

      print, count, F='(%">>>>> %d sources will move to MEAN DATA position.")' 
      
      dataset_1d, id_ra , (3600*(RA  -  RA_DATA)*cos(DEC/!RADEG))[ind], DATASET='use_data', BINSIZE=0.01, XTIT='proposed RA  change (arcsec)', DENSITY_TITLE='ae_improve_positions'
      dataset_1d, id_dec, (3600*(DEC - DEC_DATA)                )[ind], DATASET='use_data', BINSIZE=0.01, XTIT='proposed DEC change (arcsec)', DENSITY_TITLE='ae_improve_positions'
    endif else file_copy, '/dev/null', 'use_data.srclist'

    ind = where(no_estimate, count) 
    if (count GT 0) then begin
      print, count, F="(%'\n>>>>> WARNING!  The %d sources below are MISSING the preferred position estimate!')" 
      forprint,                                CATALOG_NAME, LABEL, preferred_method, F="(%'  %s (%s) is missing %s')"  , SUBSET=ind
      forprint, TEXTOUT='no_estimate.srclist', CATALOG_NAME, LABEL, preferred_method, F="(%'%s ; (%s) is missing %s')", SUBSET=ind, /NoCOMMENT
      
      acis_extract, 'no_estimate.srclist', MERGE_NAME='position', COLLATED_FILENAME='no_estimate.collated', REGION_TAG='no_estimate', REGION_FILE='no_estimate.reg', MATCH_EXISTING=bt, VERBOSE=0
  
      run_command, 'sed -e "s/DodgerBlue/red/"    no_estimate.reg |grep -v NaN >> review.reg'

    endif else file_copy, '/dev/null', 'no_estimate.srclist'

    ind = where(use_catalog, count) 
    if (count GT 0) then begin
      ind = ind[ reverse(sort(PROB_NO_SOURCE[ind])) ]
      forprint, TEXTOUT='use_catalog.srclist', CATALOG_NAME, LABEL, PROB_NO_SOURCE, use_catalog_comment, F="(%'%s ; (%s) Pb= %8.2g ; %s')", SUBSET=ind, /NoCOMMENT
      
      acis_extract, 'use_catalog.srclist', MERGE_NAME='position', COLLATED_FILENAME='use_catalog.collated', REGION_TAG='use_catalog', REGION_FILE='use_catalog.reg', MATCH_EXISTING=bt, VERBOSE=0
   
      run_command, 'sed -e "s/DodgerBlue/Tan/"   use_catalog.reg |grep -v NaN >> review.reg'

      print, count, F='(%">>>>> %d sources are prohibited from moving.")' 

    endif else file_copy, '/dev/null', 'use_catalog.srclist'
    
    dataset_1d, id_ra , PS_CONFIG={filename:'ae_improve_positions_ra.ps' }, /PRINT
    dataset_1d, id_dec, PS_CONFIG={filename:'ae_improve_positions_dec.ps'}, /PRINT
    
    
    ; Append to review.reg markers for all removed files.
    if file_test('removed_sources.reg') then begin
      run_command, 'cat removed_sources.reg >> review.reg'
    endif
    
    ; Append to review.reg the panda regions in psf_hook.reg.
    if file_test('psf_hook.reg') then begin
      run_command, 'cat psf_hook.reg >> review.reg'
    endif
    
    ; Append to review.reg the ObsID markers in tables/aimpoints.reg.
    if file_test('tables/aimpoints.reg') then begin
      run_command, 'cat tables/aimpoints.reg >> review.reg'
    endif
endif ; choose_position


CLEANUP:
if keyword_set(tempdir) && file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then begin
  print, F='(%"\nae_improve_positions finished")'  
  return 
endif else begin
  print, 'ae_improve_positions: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP

end  ; ae_improve_positions



;#############################################################################
;;; Algorithm for analyzing the 7 routine spectral fits produced by recipe.txt,
;;; suggesting which other models should be run, and suggesting the best spectral model
;;; (to be reviewed by the observer).
;#############################################################################
PRO ae_suggest_spectral_model, COLLATE=collate, ANALYZE=analyze, $
      NH_min=NH_min_p, NH_max=NH_max_p, kT_min=kT_min_p, kT_max=kT_max_p

;; Check for common environment errors.
quiet = !QUIET  &  !QUIET = 1
catch, error_code
if (error_code NE 0) then begin
  print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
  retall
endif else resolve_routine, 'astrolib', /NO_RECOMPILE
catch, /CANCEL
astrolib
; Make sure forprint calls do not block for user input.
!TEXTOUT=2
!QUIET = quiet
  
  
; ---------------------------------------------------------------------
; Collate the fitting results.
; The * in the HDUNAME pattern is there to pick up any models that required customization, e.g. via MODEL_CHANGES_FILENAME='xspec_scripts/noerr.xcm'

if keyword_set(collate) then begin
  acis_extract, 'xspec.srclist', MERGE_NAME='photometry', HDUNAME='nogrp_tbabs_vapec_A*'   , COLLATED_FILENAME='tables/tbabs_vapec_A.collated' 
  
  bt = mrdfits('tables/tbabs_vapec_A.collated',1, /SILENT)
  
  acis_extract, 'xspec.srclist', MERGE_NAME='photometry', HDUNAME='nogrp_tbabs_vapec_B*'   , COLLATED_FILENAME='tables/tbabs_vapec_B.collated'   , MATCH_EXISTING=bt
  acis_extract, 'xspec.srclist', MERGE_NAME='photometry', HDUNAME='nogrp_tbabs_vapec_C*'   , COLLATED_FILENAME='tables/tbabs_vapec_C.collated'   , MATCH_EXISTING=bt
  acis_extract, 'xspec.srclist', MERGE_NAME='photometry', HDUNAME='nogrp_tbabs_vapec_D*'   , COLLATED_FILENAME='tables/tbabs_vapec_D.collated'   , MATCH_EXISTING=bt
  acis_extract, 'xspec.srclist', MERGE_NAME='photometry', HDUNAME='nogrp_tbabs_vapec_E*'   , COLLATED_FILENAME='tables/tbabs_vapec_E.collated'   , MATCH_EXISTING=bt
  acis_extract, 'xspec.srclist', MERGE_NAME='photometry', HDUNAME='nogrp_tbabs_vapec_std1*', COLLATED_FILENAME='tables/tbabs_vapec_std1.collated', MATCH_EXISTING=bt
  acis_extract, 'xspec.srclist', MERGE_NAME='photometry', HDUNAME='nogrp_tbabs_vapec_std2*', COLLATED_FILENAME='tables/tbabs_vapec_std2.collated', MATCH_EXISTING=bt
  
  save, /COMPRESS, FILE='fit_spectra.sav'
endif else restore, 'fit_spectra.sav'


if keyword_set(analyze) then begin
  ; These parameters define what NH and kT values are considered "extreme".
  if keyword_set(NH_min_p) then NH_min = NH_min_p else NH_min =  1E20
  if keyword_set(NH_max_p) then NH_max = NH_max_p else NH_max =  1E23
  if keyword_set(kT_min_p) then kT_min = kT_min_p else kT_min =  0.5
  if keyword_set(kT_max_p) then kT_max = kT_max_p else kT_max = 15
  
  ; We must put the standard models at the start of this list for later code to work properly.
  num_standard_models = 2
  collated_files = ['tables/tbabs_vapec_std1.collated',$
                    'tables/tbabs_vapec_std2.collated',$
                    'tables/tbabs_vapec_A.collated',   $
                    'tables/tbabs_vapec_B.collated',   $
                    'tables/tbabs_vapec_C.collated',   $
                    'tables/tbabs_vapec_D.collated',   $
                    'tables/tbabs_vapec_E.collated']
                    
  num_models = n_elements(collated_files)
  
  is_frozen  = replicate(0B, num_models)
  is_frozen[0:num_standard_models-1] = 1
  
  bt = mrdfits(collated_files[0], 1, /SILENT)
  num_sources = n_elements(bt)
  
  review_reason = strarr(num_sources)
  NH            = fltarr(num_models, num_sources)
  KT            = fltarr(num_models, num_sources)
  FH8        = fltarr(num_models, num_sources)
  FCH8       = fltarr(num_models, num_sources)
  MODEL         = strarr(num_models, num_sources)
  CSTAT         = fltarr(num_models, num_sources)   

  band_full = 0
  if ~almost_equal(bt.ENERG_LO[band_full], 0.5, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
  if ~almost_equal(bt.ENERG_HI[band_full], 8.0, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 8.0 keV.\n")'
  
  NET_CNTS      = bt.NET_CNTS  [band_full]
  SRC_SIGNIF    = bt.SRC_SIGNIF[band_full]
  LABEL = strtrim(bt.LABEL,2)
  
  for model_ind=0,num_models-1 do begin
    print, collated_files[model_ind]
    bt = mrdfits(collated_files[model_ind], 1, /SILENT)
    
    NH     [model_ind,*] =         bt.NH1 * 1E22
    KT     [model_ind,*] =         bt.KT1
    FH8    [model_ind,*] =         bt.FH8
    FCH8   [model_ind,*] =         bt.FCH8
    MODEL  [model_ind,*] = strtrim(bt.MODEL,2)
    CSTAT  [model_ind,*] =         bt.CSTAT
  endfor
   
  free_NH      = NH     [num_standard_models:*,*]
  free_KT      = KT     [num_standard_models:*,*]
  free_FH8  = FH8 [num_standard_models:*,*]
  free_FCH8 = FCH8[num_standard_models:*,*]
  
  dsn = 'thawed models'
  dataset_1d,id1, DATASET=dsn, alog10(free_NH)     , XTIT='log NH'
  dataset_1d,id2, DATASET=dsn,        free_KT      , XTIT='kT'
  dataset_1d,id3, DATASET=dsn, alog10(free_FH8 ), XTIT='log F[0.5:8]'
  dataset_1d,id4, DATASET=dsn, alog10(free_FCH8), XTIT='log Fc[0.5:8]'
  dataset_1d,id5, DATASET=dsn,     FCH8/FH8,   XTIT='FCH8/FH8 [0.5:8]'
                                        
  dataset_2d,id15, DATASET=dsn, alog10(free_NH), free_KT,  XTIT='log NH', YTIT='kT', PSYM=1, NAN=[15,-1]
  
  dataset_2d,id16, DATASET=dsn, alog10(free_NH), alog10(free_FCH8), XTIT='log NH', YTIT='log Fc[0.5:8]', PSYM=1, NAN=[15,-10]
  dataset_2d,id17, DATASET=dsn,        free_KT , alog10(free_FCH8), XTIT='kT'    , YTIT='log Fc[0.5:8]', PSYM=1, NAN=[-5,-10]
  
  ; ---------------------------------------------------------------------
  ;Identify astrophysically extreme fits.
  
  is_extreme =  (alog10(FH8)   LT -16)    OR $     ; flux range
                (alog10(FH8)   GT -13)    OR $     
                (alog10(FCH8)  LT -16)    OR $     ; corrected flux range
                (alog10(FCH8)  GT -13)    OR $     
                (       NH        LT NH_min) OR $     ; NH range
                (       NH        GT NH_max) OR $     
                (       KT        LT kT_min) OR $     ; kT range
                (       KT        GT kT_max) OR $   
                ((FCH8/FH8) GT 10)              ; flux correction factor
  
  print, 100.0 * total(/INT, is_extreme) / n_elements(is_extreme), n_elements(is_extreme), F='(%"\n%0.1f%% of the %d models are ''extreme''.")'
  
  
  ; ---------------------------------------------------------------------
  ; Fit a two-temperature model for bright sources.
  ; We put this step here merely so that the review_reason "bright" will appear first in the list of reasons.
  
  ind = where((SRC_SIGNIF GT 6), count)
  if (count GT 0) then begin
    review_reason[ind] = review_reason[ind]+'bright '
    
    forprint, TEXTOUT='thermal_2T.srclist', bt.CATALOG_NAME, SUBSET=ind, /NoCOMMENT
    print, count, F='(%"\nFit the appropriate non-standard model to %d sources in thermal_2T.srclist.")'
  endif
  
  ; ---------------------------------------------------------------------
  ; Begin building a recommendation for the best model for each source.
  
  ; Initially, we simply select the model with the smallest Cstat value. However, we're willing to accept an alternative model with a modestly increased Cstat if the alternative eliminates an undesirable property of the minimum-Cstat model, such as an astrophysically unlikely set of parameter values or a frozen parameter.
  
  ; I don't know of a rigorous statistical argument addressing the question of "how far" from the best fit one should stray when choosing alternate models. Desperate for some delta-Cstat threshold to define "acceptable" alternative fits, I turn to the table in Section 2.2 of the XSPEC12 manual that associates a delta_stat value of 4.61 with a 90% confidence region for two parameters. I latch onto the two parameter case because the observer is (mostly) thinking about the physical plausibility of NH and kT. I latch onto the 90% confidence level arbitrarily.
  
  ; Similarly, I don't know a rigorous Cstat increase one should tolerate to avoid a best-fit model that has a frozen parameter. I arbitrarily choose a value of 1.0
  
  extreme_cstat_penalty = 4.61
  frozen_cstat_penalty  = 1.0
  
  bestfit_model_ind = lonarr(num_sources)
  chosen_model_ind  = lonarr(num_sources)
  
  for ii=0L,num_sources-1 do begin
    ; Find the best-fit model, i.e. the one with the smallest Cstat.
    min_cstat = min(CSTAT[*,ii], ind, /NAN)
    
    ; Record the best-fit model for use later.
    ; Our initial recommendation ("cmi"=="chosen model index") is the best-fit model.
    bestfit_model_ind[ii] = ind
    cmi                   = ind      
    
    ; Loop through the models looking for a preferred one.
    ; "tmi" is "this model index"
    for tmi=0,num_models-1 do begin
      ; Compare this model and currently chosen model with regard to extreme parameters.
      skip = 0
      case 1 of
        ( is_extreme[tmi,ii] &&  is_extreme[cmi,ii]):  ; no difference
        (~is_extreme[tmi,ii] && ~is_extreme[cmi,ii]):  ; no difference
        ( is_extreme[tmi,ii] && ~is_extreme[cmi,ii]):$ ; extreme never preferred
          skip = 1
        (~is_extreme[tmi,ii] &&  is_extreme[cmi,ii]):$ ; Reasonable model preferred if CSTAT not too high.
          begin
          if (CSTAT[tmi,ii] LT (CSTAT[cmi,ii] + extreme_cstat_penalty)) then begin
            print, MODEL[cmi,ii], MODEL[tmi,ii], LABEL[ii], F='(%"  Replacing extreme model %s with reasonable model %s for source %s")'
            cmi = tmi
            review_reason[ii] = review_reason[ii]+'not_best '
            skip = 1
          endif
          end
      endcase
      if skip then continue
    
      ; Compare this model and currently chosen model with regard to frozen parameters.
      case 1 of
        ( is_frozen[tmi] &&  is_frozen[cmi]):  ; no difference
        (~is_frozen[tmi] && ~is_frozen[cmi]):  ; no difference
        ( is_frozen[tmi] && ~is_frozen[cmi]):$ ; frozen never preferred
          skip = 1
        (~is_frozen[tmi] &&  is_frozen[cmi]):$ ; Thawed model preferred if CSTAT not too high.
          begin
          if (CSTAT[tmi,ii] LT (CSTAT[cmi,ii] + frozen_cstat_penalty)) then begin
            print, MODEL[cmi,ii], MODEL[tmi,ii], LABEL[ii], F='(%"  Replacing frozen model %s with thawed model %s for source %s")'
            cmi = tmi
            review_reason[ii] = review_reason[ii]+'not_best '
            skip = 1
          endif
          end
      endcase
      if skip then continue
    
      
      ; If we get here then this model and the chosen model have the same extreme and frozen properties.
      ; Now we simply accept a smaller Cstat
      if (CSTAT[tmi,ii] LT CSTAT[cmi,ii]) then begin
        print, MODEL[cmi,ii], MODEL[tmi,ii], LABEL[ii], F='(%"  Replacing model %s with better model %s for source %s")'
        cmi = tmi
        review_reason[ii] = review_reason[ii]+'not_best '
       continue
      endif
      
      ; Save the chosen model.
      chosen_model_ind[ii] = cmi
    endfor ; tmi
  endfor ; ii
  
  chosen_model = MODEL[chosen_model_ind,indgen(num_sources)]
  print, F='(%"\nThe distribution of recommended models is shown below:")'
  forprint, file_basename(collated_files,'.collated'), histogram(chosen_model_ind) 
  
  
  ; ---------------------------------------------------------------------
  ; Identify for review sources where the chosen model remains extreme.
  
  ind = where(is_extreme[chosen_model_ind,indgen(num_sources)], count)
  if (count GT 0) then review_reason[ind] = review_reason[ind]+'extreme '
  print, count, F='(%"\nThe recommended model for %d sources is extreme.")'
    
   
  
  ; ---------------------------------------------------------------------
  ; Identify sources where the chosen model has extremely low NH.
  ; For these we perform another unfrozen fit with initial parameter values at a different point than in the routine models.
  ; We do not recommend this model (since we don't know yet how this one will turn out.)
  chosen_has_low_NH = (NH[chosen_model_ind,indgen(num_sources)] LE NH_min)
  
  ind = where(chosen_has_low_NH, count)
  if (count GT 0) then begin
    review_reason[ind] = review_reason[ind]+'NH_low '
    forprint, TEXTOUT='NH_low.srclist', bt.CATALOG_NAME, SUBSET=ind, /NoCOMMENT
    print, count, F='(%"\nFit the appropriate non-standard model to %d sources in NH_low.srclist.")'
  endif
  
  
  ; ---------------------------------------------------------------------
  ; Identify sources where the chosen model has extremely high kT.
  ; For these we perform a fit with kT frozen at its high limit, and we recommend that as the best model.
  
  ; Identify sources where ALL the free models have extremely high kT.
  ; For these we want to show the observer a fit with kT frozen at its high limit, but we do not recommend it as the best model.
  
  chosen_is_hot = (KT[chosen_model_ind,indgen(num_sources)] GT kT_max)
  all_are_hot   = (min(free_KT, DIMENSION=1)                GT kT_max)
  
  ind = where(chosen_is_hot, count)
  if (count GT 0) then begin
    review_reason[ind] = review_reason[ind]+'kT_max '
    chosen_model [ind] = 'nogrp_tbabs_vapec_kT_max*'  ; The wildcard is needed to match _noerr models.
  endif
  
  ind = where(chosen_is_hot OR all_are_hot, count)
  if (count GT 0) then begin
    forprint, TEXTOUT='thermal_kT_max.srclist', bt.CATALOG_NAME, SUBSET=ind, /NoCOMMENT
    print, count, F='(%"\nFit the appropriate non-standard model to %d sources in thermal_kT_max.srclist.")'
  endif
  
  
  ; ---------------------------------------------------------------------
  ; When all the free models have extreme kT, or when the chosen thermal model requires a high temperature, then run a power law model.
  
  ; THIS RECIPE WAS WRITTEN FOR ANALYSIS OF RICH STAR-FORMATION REGIONS, WHERE A POWERLAW IS LIKELY TO BE THE CORRECT MODEL ONLY FOR A FEW BACKGROUND AGN. Thus, our algorithm for choosing the "best model" to suggest to the observer DOES NOT CONSIDER THE POWERLAW FIT. The observer is expect to carefully review the cases where a powerlaw model was tried and make the hard decision about when it should be adopted, perhaps using other information such as where the source lies in the cluster, counterpart properties, variability, etc.
  
  ind = where(all_are_hot OR ((KT[chosen_model_ind,indgen(num_sources)] GT 6)), count)
  if (count GT 0) then begin
    review_reason[ind] = review_reason[ind]+'powerlaw '
    
    forprint, TEXTOUT='pow.srclist', bt.CATALOG_NAME, SUBSET=ind, /NoCOMMENT
    print, count, F='(%"\nFit the appropriate non-standard model to %d sources in pow.srclist.")'
  endif
  
  
  ; ---------------------------------------------------------------------
  ; Identify for review sources where the best-fit model has frozen parameters. 
  
  ind = where(bestfit_model_ind LT num_standard_models, count)
  if (count GT 0) then review_reason[ind] = review_reason[ind]+'best-fit is frozen '
  print, count, F='(%"\nThe best-fit model for %d sources is one of the standard models with frozen kT.")'
    
      
  ; ---------------------------------------------------------------------
  ; Flag for visual review sources whose fits are sensitive to initial parameters.
  ; The sensitivity criterion must of course OMIT the frozen "standard" models!
  
  is_sensitive = (abs(alog10(min(free_NH,    DIM=1) / max(free_NH,    DIM=1))) GT 0.5) OR $ ; NH stability
                 (abs(alog10(min(free_FH8,DIM=1) / max(free_FH8,DIM=1))) GT 0.5) OR $ ; flux stability
                 (abs(       min(free_KT,    DIM=1) - max(free_KT,    DIM=1))  GT 1)        ; kT stability
                 
  ind = where(is_sensitive, count)
  if (count GT 0) then review_reason[ind] = review_reason[ind]+'sensitive '
  print, count, F='(%"\nThe free models for %d sources are sensitive to initial parameters.")'
  
  
  save, /COMPRESS, FILE='fit_spectra.sav'
endif
return
end ; ae_suggest_spectral_model



;#############################################################################
;;; Copied from match_xy.pro, so that I do not have to distribute match_xy.pro to AE community.
;#############################################################################
FUNCTION get_astrometry_from_eventlist, eventlist_fn

;; Read event file header to define ACIS sky coordinate system.
theader = headfits(eventlist_fn, EXT=1, ERRMSG=error )
if (keyword_set(error)) then begin
  print, error
  message, 'ERROR reading ' + eventlist_fn
endif

; Build astrometic structure from data header.
    fxbfind, theader, 'TTYPE', dum1, TTYPE, dum2, 'null'
    fxbfind, theader, 'TCTYP', dum1, TCTYP, dum2, 'null'
    fxbfind, theader, 'TCRVL', dum1, TCRVL, dum2, 0.0D
    fxbfind, theader, 'TCRPX', dum1, TCRPX, dum2, 0.0D
    fxbfind, theader, 'TCDLT', dum1, TCDLT, dum2, 0.0D
    colnames = strlowcase( strtrim(TTYPE,2) )
    x_ind    = where(strlowcase(colnames) EQ 'x')
    y_ind    = where(strlowcase(colnames) EQ 'y')

    ;; The xy2ad.pro and ad2xy.pro programs we are using in the match_xy package to convert between celestial and "X/Y" coordinates think of the X/Y coordinates as 0-based image array indexes.
    ;; However, when a Chandra event list is defining the tangent plane, we would like the X/Y coordinates in match_xy to correspond to the PHYSICAL "SKY" system defined by Chandra.
    ;; The requires adding one to the CRPIX astrometry keywords below.

    make_astr, event2wcs_astr, DELTA=  TCDLT[[x_ind,y_ind]], CTYPE=TCTYP[[x_ind,y_ind]], $
                               CRPIX=1+TCRPX[[x_ind,y_ind]], CRVAL=TCRVL[[x_ind,y_ind]]

return, event2wcs_astr
end


;#############################################################################
;;; This tool compares single-ObsID mean data source positions and estimates the astrometric alignment among the ObsIDs.
;;; If a reference catalog (in match_xy format) is supplied (via REF_CATALOG and REF_NAME) then each ObsID is compared to it as well.

;;; Single-ObsID source positions are found in the single-ObsID collations obsXXXX/all.collated, created by the tool ae_better_backgrounds.
;;;
;;; The list of ObsID names is supplied in the string vector "obsname_list_p".
;;;
;;; If you've defined the environment var OBS_LIST in the usual way, then the call would be:
;;;  ae_interObsID_astrometry, strtrim(strsplit(getenv('OBS_LIST'), /EXTRACT), 2), ...

;;; The estimated shift in ObsID[ii]'s event positions required to align it with ObsID[jj], computed by comparing X-ray source positions in the two ObsIDs, is reported in obs_obs_shift[jj,ii].  
;;;
;;; The shift in ObsID[ii]'s event positions required to align it with the reference, computed by directly matching X-ray and reference source positions, is reported in obs_ref_shift[ii]. 
;;;
;;; Our best estimate for the shift in ObsID[ii]'s event positions required to align it with the reference, based on all the information in obs_obs_shift and obs_ref_shift, is reported in weighted_shift[ii].
;;;
;;; Our recommendation for the ObsID shifts that should be implemented are reported in the "recommendation" structure.


;#############################################################################
PRO ae_interObsID_astrometry, obsname_list_p, ASTROMETRY=fits_astrometry, REF_CATALOG=ref_cat, REF_NAME=refname_p, $
                              TARGET_NAME=target_name, SIGNIFICANCE_THRESHOLD=significance_threshold,$
                              THETA_THRESHOLD=theta_threshold, DETECTION_THRESHOLD=detection_threshold,$
                              DEBUG=debug, PLOT_LEVEL=plot_level, $
                              
                              ; Outputs
                              recommendation, $ 
                                
                              weighted_xshift, weighted_yshift, weighted_xshift_error, weighted_yshift_error, $

                              obs_obs_xshift,  obs_obs_yshift,  obs_obs_xshift_error,  obs_obs_yshift_error, $

                              obs_ref_xshift,  obs_ref_yshift,  obs_ref_xshift_error,  obs_ref_yshift_error

                              

FORWARD_FUNCTION build_AE_cat  ; found in match_xy.pro

exit_code = 0
creator_string = "ae_interObsID_astrometry, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()
print

if (n_elements(obsname_list_p) EQ 0) then begin
  print, "ERROR: 'obsname' parameter is empty."
  GOTO, FAILURE
endif

if (n_elements(plot_level) EQ 0) then plot_level=1

;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)
run_command, PARAM_DIR=tempdir

cd, CURRENT=cwd

;; Create an array of structures to hold information about each observation.
template_row = {obsname     :'',$ ; Name of suffixed ObsID passed, with _FI, _BI suffix
                obsname_base:'',$ ; Actual ObsID name, without _FI, _BI suffix.
                obsdir      :'',$ ; Path to ObsID directory.
                collatefile :'',$ ; Path to collation of FI and BI extractions.
                evtfile     :'',$ ; Path to an event list with ObsID's header information.
                obsdir_repro:'' $ ; Path to directory where L2 data for ObsID will be shifted 
                }                 ;(distinct from AE's obsXXXX/ directories).

obsid = replicate(template_row, n_elements(obsname_list_p))

obsid.obsname      = obsname_list_p

for ii=0, n_elements(obsid)-1 do obsid[ii].obsname_base = mg_streplace(obsid[ii].obsname, '_FI|_BI', '')
; Note that mg_streplace function is from https://raw.github.com/mgalloy/mglib/master/src/strings/mg_streplace.pro

obsid.obsdir       = '../obs' + obsid.obsname + '/'
obsid.collatefile  =  tempdir + obsid.obsname_base+'.collated' 
obsid.evtfile      =  tempdir + obsid.obsname_base+'.evt'      

obsid.obsdir_repro = file_dirname(file_dirname(file_readlink(obsid.obsdir+'obs.emap', $
                                                             /ALLOW_NONSYMLINK, /ALLOW_NONEXISTENT))) 


; For this astrometry work, we want a moderate match criterion.
if ~keyword_set(significance_threshold) then significance_threshold= 0.9
; And we want our source positions (mean data algorithm) to not be too biased.
; And we want to avoid spurious source detections.
if ~keyword_set(theta_threshold       ) then theta_threshold       = 5     ; arcmin off-axis
if ~keyword_set(detection_threshold   ) then detection_threshold   = 0.003 ; threshold on PROB_NO_SOURCE


refname = keyword_set(refname_p) ? refname_p : ''

;=============================================================================
;; Skip ObsIDs that are missing their collation.
good_ind = where( file_test(obsid.obsdir + '/all.collated'), num_found, COMPLEMENT=bad_ind, NCOMPLEMENT=num_missing )

if (num_missing GT 0) then begin
  print, 'WARNING: the following catalogs are missing:'
  forprint, SUBSET=bad_ind, obsid.obsdir + '/all.collated'
endif

if (num_found EQ 0) then begin
  print, 'ERROR: no catalogs were found; aborting.'
  GOTO, FAILURE
endif

obsid = obsid[good_ind]


;=============================================================================
;; Skip ObsIDs that are missing RA_DATA/DEC_DATA columns, e.g. because no sources lie there.
is_good = bytarr(num_found)
for ii=0,num_found-1 do begin
  fn = obsid[ii].obsdir + '/all.collated'
  bt1 = mrdfits(fn, 1, /SILENT)
  if tag_exist(bt1,'RA_DATA') && tag_exist(bt1, 'DEC_DATA') then is_good[ii] = 1 $
  else print, fn, F="(%'\nWARNING: The catalog %s contains no RA_DATA/DEC_DATA columns, probably because no sources lie in that ObsID.\n')"
endfor ;ii

good_ind = where(is_good, num_found, COMPLEMENT=bad_ind, NCOMPLEMENT=num_missing )

if (num_found EQ 0) then begin
  print, 'ERROR: no catalogs have RA_DATA/DEC_DATA columns; aborting.'
  GOTO, FAILURE
endif

obsid = obsid[good_ind]


;=============================================================================
;; Combine collations from pairs of pseudo-ObsIDs with names of the form "_FI" and "_BI".
;; We use (THETA EQ NaN), rather than (RA_DATA EQ NaN) to identify sources that are "not observed" by an ObsID,
;; because RA_DATA could be unavailable for other reasons (e.g. zero counts in aperture).
row_index = indgen(n_elements(obsid))
for ind1 = 0,      n_elements(obsid)-1 do begin

  ; If a flag indicates that we've already considered this ObsID, then go to the next one.
  if (obsid[ind1].collatefile EQ '') then continue

  ; Look for a second part of this ObsID.
  ind2 = where( (obsid.obsname_base EQ obsid[ind1].obsname_base) AND $
                (         row_index NE       ind1              ), count)
  ; ind1 is the index of one part of this ObsID, e.g. xxxxx_FI.
  ; ind2 is the index of the other part of this ObsID (if it exists), e.g. xxxxx_BI.
  
  case count of
   0: begin
      ; This ObsID has only one part, so just read the collation file.
      obsdir1 = obsid[ind1].obsdir

      ; Convert AE collations to match_xy catalogs (with X/Y coordinates and NOT_FIDUCIAL column).
      bt1 = build_AE_cat(obsdir1 + 'all.collated', fits_astrometry, /USE_MEAN_DATA_POSITIONS, theader1)
      end

   1: begin
      print, 'Combining "FI" and "BI" extractions of ObsID ', obsid[ind1].obsname_base
      
      ; This ObsID has two parts, so combine the collations.
      obsdir1 = obsid[ind1].obsdir
      obsdir2 = obsid[ind2].obsdir

      ; Convert AE collations to match_xy catalogs (with X/Y coordinates and NOT_FIDUCIAL column).
      bt1 = build_AE_cat(obsdir1 + 'all.collated', fits_astrometry, /USE_MEAN_DATA_POSITIONS, theader1)
      bt2 = build_AE_cat(obsdir2 + 'all.collated', fits_astrometry, /USE_MEAN_DATA_POSITIONS          )

      if n_elements(bt1) NE n_elements(bt2) then begin
        print, obsdir1 + 'all.collated', obsdir2 + 'all.collated', F='(%"ERROR: %s and %s have different numbers of sources")' 
        GOTO, FAILURE
      endif

      for ii=0,n_elements(bt1)-1 do begin
        if ~finite(bt1[ii].THETA) && ~finite(bt2[ii].THETA) then continue  ; source not observed by this ObsID

        if  finite(bt1[ii].THETA) &&  finite(bt2[ii].THETA) then begin
          ; If a source appears in both FI and BI collations then it must span a CCD boundary.
          ; Omit it from the combined collation since both the DATA positions are probably biased.
          bt1[ii].RA_DATA  = !VALUES.F_NAN
          bt1[ii].DEC_DATA = !VALUES.F_NAN
          bt1[ii].X        = !VALUES.F_NAN
          bt1[ii].Y        = !VALUES.F_NAN
          print, 'WARNING: Ignoring source observed in both FI and BI: ', bt1[ii].LABEL
          continue
        endif

        if  finite(bt1[ii].THETA) && ~finite(bt2[ii].THETA) then continue  ; source already in bt1 collation


        if ~finite(bt1[ii].THETA) &&  finite(bt2[ii].THETA) then begin
          ; Copy source from bt2 collation to bt1 collation.
          destination  = bt1[ii]
          struct_assign, bt2[ii], destination
          bt1[ii]      =          destination
        endif
      endfor ; ii

      ; Flag that row ind2 has been processed, by clearing its collatefile tag.
      obsid[ind2].collatefile = ''
      
      end ; count EQ 1
   else: begin
         print, 'ERROR: your list of ObsIDs seems to have duplicates.'
         GOTO, FAILURE
         end
  endcase
  
  ; Decide which single-ObsID source positions are NOT useful ("NOT_FIDUCIAL") with respect to astrometry.
  ; We do not trust a single-ObsID source position when any of the following are true:
  ;   * Extraction has low significance (>detection_threshold), i.e. we're not sure a source is present.
  ;   * Mean data position is likely biased due to PSF asymmetry far off-axis (>theta_threshold').
  ;   * Mean data position is likely biased due to asymmetric background from neighboring point sources.
  ;     Estimating that bias is difficult.  
  ;     We are going to estimate the fraction of SRC_CNTS expected to come from nearby point sources:
  ;            ((BKG_CNTS/BACKSCAL) * (1 - (FLATGRND/BACKGRND)) - SELFCNTS) / SRC_CNTS
  ;     and impose a threshold on that fraction.
  
  band_full = 0
  if ~almost_equal(bt1.ENERG_LO[band_full], 0.5, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
  if ~almost_equal(bt1.ENERG_HI[band_full], 7.0, DATA_RANGE=range) then print, band_full, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 7.0 keV.\n")'
  
  ; Fraction of bkg counts expected from the wings of point sources (including SELFCNTS).
  wing_fraction_bkg         = 1 - (bt1.FLATGRND/bt1.BACKGRND)                             
  
  ; Number of bkg counts expected in aperture.
  bkg_cnts_in_aperture            = bt1.BKG_CNTS[band_full] / bt1.BACKSCAL[band_full]
  
  ; Number of aperture counts expected from the wings of neighbors.
  asymmetric_bkg_cnts_in_aperture = (bkg_cnts_in_aperture * wing_fraction_bkg) - bt1.SELFCNTS
  
  ; Fraction of aperture counts expected from the wings of neighbors.
  asymmetric_fraction_src         =  asymmetric_bkg_cnts_in_aperture  / bt1.SRC_CNTS[band_full]
  
  asymmetric_fraction_src[where(/NULL, ~finite(asymmetric_fraction_src))] = !VALUES.F_NAN
  
  bt1.NOT_FIDUCIAL = (bt1.PROB_NO_SOURCE[band_full] GT detection_threshold) OR $
                     (bt1.THETA                     GT     theta_threshold) OR $
                     (asymmetric_fraction_src       GT 0.10           )

  ; Position estimates obtained from custom apertures (marked by REG_EDIT=1=True) may or may not be biased,
  ; depending on the assymetry of the aperture.  We must ask the observer to decide.
  if tag_exist(bt1, 'REG_EDIT') then begin
    foreach query_ind, where(/NULL, ~bt1.NOT_FIDUCIAL AND bt1.REG_EDIT) do begin
      this_extraction = bt1[query_ind]
      
      msg = string(this_extraction.OBJECT, this_extraction.LABEL, this_extraction.OBSNAME, F='(%"Please decide if the single-ObsID position estimate for %s (%s) in ObsID %s is corrupted by aperture assymmetry, pile-up, or another problem.")' )
      print, msg
      title = (keyword_set(target_name) ? target_name+': ' : '')+'ae_interObsID_astrometry: Custom Aperture Decision Needed'
      TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='Ignore this position estimate.', BUTTON_LABEL='Use this position estimate in astrometry analysis.', BUTTON_ID=button_id, QUIT_ID=quit_id, PRESSED_ID=pressed_id
  
      if (pressed_id EQ quit_id) then bt1[query_ind].NOT_FIDUCIAL = 1B
      
      print, this_extraction.OBJECT, this_extraction.LABEL, this_extraction.OBSNAME, (bt1[query_ind].NOT_FIDUCIAL ? 'NOT ' : ''), F='(%"%s (%s) in ObsID %s is marked as %sfiducial.")'
    endforeach
  endif


    if keyword_set(debug)  then begin
      forprint, SUBSET=where( bt1.NOT_FIDUCIAL ), TEXTOUT=1,$
        indgen(n_elements(bt1)), bt1.LABEL,$
        bt1.PROB_NO_SOURCE[band_full],$
        bt1.THETA                    ,$
        asymmetric_fraction_src*100  ,$
        wing_fraction_bkg, bkg_cnts_in_aperture, asymmetric_bkg_cnts_in_aperture,$
        F='(%"%d: %s Pb=%6.4f  %4.1f arcmin  %0.1f%% contam (%0.2f %0.1f %0.1f) ")'
      print, 'Stopped for debugging ...'
      stop
    endif


  ; Defensively check for sources that claim to have X/Y coordinates but have any celestial coordinates that are not well defined.
  celestial_is_not_defined = (bt1.RA       EQ 0) OR ~finite(bt1.RA      ) OR $
                             (bt1.DEC      EQ 0) OR ~finite(bt1.DEC     ) OR $
                             (bt1.RA_DATA  EQ 0) OR ~finite(bt1.RA_DATA ) OR $
                             (bt1.DEC_DATA EQ 0) OR ~finite(bt1.DEC_DATA)
  
  error_ind = where(/NULL, finite(bt1.X) AND celestial_is_not_defined)
  if isa(error_ind, /INTEGER) then begin
         print, 'ERROR: BUG: source has finite X/Y coords but celestial coords =0.'
         help, /st, bt1[error_ind]
         GOTO, FAILURE
  endif
  
  ; Save this ObsIDs match_xy catalog to disk.
  mwrfits, bt1, obsid[ind1].collatefile, theader1, /CREATE
  
  ; Symlink to one part's event file.
  file_link, cwd+'/' + obsdir1 + 'spectral.evt', obsid[ind1].evtfile
 
;  help, bt1, obsid[ind1].collatefile
;  stop
endfor ;ind1

; Retain only structure rows that correspond to ObsID catalogs (FI+BI sources).
obsid = obsid[ where(obsid.collatefile) ]

num_obs = n_elements(obsid)   



print, refname, theta_threshold, detection_threshold, F="(%'\n\nFor all estimates of frame offsets between ObsIDs or between an ObsID and the %s catalog, we ignore matches involving an X-ray extraction that fails any of the following requirements:\n    THETA          < %0.1f\' \n    PROB_NO_SOURCE < %0.3g \n    less than 10%% of SRC_CNTS come from nearby point sources.')" 

;=============================================================================
;; Define matrices to hold astrometric offset estimates between pairs of ObsIDs.
obs_obs_xshift          = replicate(!VALUES.F_NAN, num_obs, num_obs)
obs_obs_yshift          = replicate(!VALUES.F_NAN, num_obs, num_obs)

obs_obs_xshift_variance = replicate(!VALUES.F_NAN, num_obs, num_obs)
obs_obs_yshift_variance = replicate(!VALUES.F_NAN, num_obs, num_obs)

print, F="(%'\n\n\n======================== ALIGNMENT BETWEEN PAIRS OF ObsIDs  ========================\n')"

print, F="(%'\nEach source that is extracted from a pair of ObsIDs provides a measurement of the astrometric offset between them.  The offset observed between the pair of extractions participates in the astrometric analysis only if BOTH    extractions meet the requirements stated earlier.')" 

print, F="(%'\nThe mean offsets of the remaining \'fiducial\' pairs, weighted by their uncertainties (Bevington, 2nd Edition, 4.17), are reported in the matrix below.')"

obsid_exposure = fltarr(num_obs)

for ii = 0,num_obs-1 do begin
  collatefile_ii  = obsid[ii].collatefile
  ; Read match_xy catalog.
  ; Note that the NOT_FIDUCIAL column was computed above, when FI & BI extractions were combined.
  cat_ii = mrdfits(collatefile_ii, 1, /SILENT)

  obsid_exposure[ii] = mean(cat_ii.exposure, /NAN)
  

  for jj = ii+1, num_obs-1 do begin
    collatefile_jj  = obsid[jj].collatefile
    
    ; Read match_xy catalog.
    ; Note that the NOT_FIDUCIAL column was computed above, when FI & BI extractions were combined.
    ; Note that cat.X and cat.Y are NaN when no "mean data" position estimate is available.
    cat_jj = mrdfits(collatefile_jj, 1, /SILENT)
    
    if n_elements(cat_ii) NE n_elements(cat_jj) then begin
      print, collatefile_ii, collatefile_jj, F='(%"ERROR: %s and %s have different numbers of sources")' 
      continue
    endif
    
    if total(/INT,strcmp(cat_ii.CATALOG_NAME, cat_jj.CATALOG_NAME)) NE n_elements(cat_ii) then begin
      print, collatefile_ii, collatefile_jj, F='(%"ERROR: %s and %s have different source names")' 
      continue
    endif
    
    ; Compute the offset FROM ObsJJ TO ObsII.
    deltaX          = cat_ii.X - cat_jj.X
    deltaY          = cat_ii.Y - cat_jj.Y

    ; We impose an arbitrary floor on the AE position errors because we are uncomfortable with a very bright (espeicially a piled) source exerting too much influence.
    deltaX_variance = (cat_jj.X_ERR>0.02)^2 + (cat_ii.X_ERR>0.02)^2 ;  See match_xy.pro.
    deltaY_variance = (cat_jj.Y_ERR>0.02)^2 + (cat_ii.Y_ERR>0.02)^2

    num_extractions = total(/INT, finite(deltaX) AND finite(deltaY))
    
    ; Compare only those extractions that are likely to produce reasonably accurate position estimates:
    is_fiducial =       finite(deltaX) AND       finite(deltaY) AND $
                  ~cat_ii.NOT_FIDUCIAL AND ~cat_jj.NOT_FIDUCIAL
      
    good_ind = where(is_fiducial, num_fiducial)
    
    
    ; If that produces no offset samples, then try again using all ACIS sources.????????
    if (num_fiducial LT 1) then begin
      print, obsid[ii].obsname_base, obsid[jj].obsname_base, F='(%"\nWARNING: with default parameters, no offset estimate between ObsID %s and ObsID %s could be estimated.  You can produce more matches (with lower quality) by relaxing the THETA_THRESHOLD parameter.")'

      continue
    endif

    
    deltaX          = deltaX         [good_ind]
    deltaY          = deltaY         [good_ind]
                               
    deltaX_variance = deltaX_variance[good_ind]
    deltaY_variance = deltaY_variance[good_ind]
    
    
    ; Estimate the astrometric shift by a weighted average of the shifts found in all the  pairs of extractions.
    ; Wikipedia (https://en.wikipedia.org/wiki/Weighted_arithmetic_mean) says that the ML estimator corresponds to weighting each data point (in our case, an offset calculated from one pair of extractions) by the inverse of its variance.
    ; Bevington (2nd Edition) Chapter 4 concurs.    
    weight                       = 1. / deltaX_variance                           ; Bevington (2nd Edition) 4.17
    weight_total                 = total(/DOUBLE, /NAN, weight) 
    this_obs_obs_xshift          = total(/DOUBLE, /NAN, weight*deltaX) / weight_total
    this_obs_obs_xshift_variance =                                  (1 / weight_total) ; Bevington (2nd Edition) 4.19
    num_rejectedX = 0
  ; info, weight
    
    weight                       = 1. / deltaY_variance                           ; Bevington (2nd Edition) 4.17
    weight_total                 = total(/DOUBLE, /NAN, weight) 
    this_obs_obs_yshift          = total(/DOUBLE, /NAN, weight*deltaY) / weight_total
    this_obs_obs_yshift_variance =                                  (1 / weight_total) ; Bevington (2nd Edition) 4.19
    num_rejectedY = 0
  ; info, weight

    print, obsid[ii].obsname_base, obsid[jj].obsname_base, this_obs_obs_xshift, sqrt(this_obs_obs_xshift_variance), this_obs_obs_yshift, sqrt(this_obs_obs_yshift_variance), '[skypix]',  num_fiducial-num_rejectedX, num_fiducial-num_rejectedY, num_extractions, F='(%"\n%5s / %5s: XSHIFT= %7.3f+-%5.3f, YSHIFT= %7.3f+-%5.3f %s  # %d, %d of %d extractions")'
    
    obs_obs_xshift         [ii,jj] =  this_obs_obs_xshift
    obs_obs_xshift         [jj,ii] = -this_obs_obs_xshift
    obs_obs_xshift_variance[ii,jj] =  this_obs_obs_xshift_variance
    obs_obs_xshift_variance[jj,ii] =  this_obs_obs_xshift_variance
    
    obs_obs_yshift         [ii,jj] =  this_obs_obs_yshift
    obs_obs_yshift         [jj,ii] = -this_obs_obs_yshift
    obs_obs_yshift_variance[ii,jj] =  this_obs_obs_yshift_variance
    obs_obs_yshift_variance[jj,ii] =  this_obs_obs_yshift_variance
  endfor ; jj
endfor ; ii



if ~keyword_set(refname) then begin
  ref_cat_supplied = 0B
endif else if ~keyword_set(ref_cat) then begin
  ;=============================================================================
  ;; When REF_NAME is passed and REF_CATALOG is not, assume REF_NAME refers to an ObsID.
  ind_ref =where(/NULL, strmatch(obsid.obsname_base, refname), COMPLEMENT=ind_remaining)
  if isa(ind_ref, /INTEGER) then begin
    ref_cat_supplied = 1B
    
    ; Read match_xy catalog.
    ; Note that cat.X and cat.Y are NaN when no "mean data" position estimate is available.
    ref_cat = mrdfits(obsid[ind_ref].collatefile, 1, /SILENT)
    ref_cat = ref_cat( where(finite(ref_cat.X)) )
    refname = 'ref'+refname
    
    ; Adopt this ObsID's entries for the reference shifts
    obs_ref_xshift          = obs_obs_xshift         [ind_ref,*]
    obs_ref_yshift          = obs_obs_yshift         [ind_ref,*]
    obs_ref_xshift_variance = obs_obs_xshift_variance[ind_ref,*]
    obs_ref_yshift_variance = obs_obs_yshift_variance[ind_ref,*]
    
    ; Remove this ObsID's entries from the obs_obs matrices. 
    obs_obs_xshift         [ind_ref,*] = !VALUES.F_NAN
    obs_obs_yshift         [ind_ref,*] = !VALUES.F_NAN
    obs_obs_xshift_variance[ind_ref,*] = !VALUES.F_NAN
    obs_obs_yshift_variance[ind_ref,*] = !VALUES.F_NAN

    obs_obs_xshift         [*,ind_ref] = !VALUES.F_NAN
    obs_obs_yshift         [*,ind_ref] = !VALUES.F_NAN
    obs_obs_xshift_variance[*,ind_ref] = !VALUES.F_NAN
    obs_obs_yshift_variance[*,ind_ref] = !VALUES.F_NAN
  endif else begin
    ref_cat_supplied = 0B
    print, 'ERROR: REF_CATALOG omitted and REF_NAME is not in the list of ObsIDs.'
    GOTO, FAILURE
  endelse
  
endif else if keyword_set(ref_cat) then begin
  ;=============================================================================
  ; A reference catalog was supplied, so match it to each of the ObsID catalogs.
  ref_cat_supplied = 1B
  obs_ref_xshift          = replicate(!VALUES.F_NAN, num_obs)
  obs_ref_yshift          = replicate(!VALUES.F_NAN, num_obs)
  obs_ref_xshift_variance = replicate(!VALUES.F_NAN, num_obs)
  obs_ref_yshift_variance = replicate(!VALUES.F_NAN, num_obs)

  
  for ii = 0,num_obs-1 do begin
    print, obsid[ii].obsname_base, refname, F="(%'\n\n\n======================== ALIGNMENT BETWEEN ObsID %s and %s  ========================\n')"
    
    collatefile_ii  = obsid[ii].collatefile
  
    ; Read match_xy catalog.
    ; Note that the NOT_FIDUCIAL column was computed above, when FI & BI extractions were combined.
    ; Note that cat.X and cat.Y are NaN when no "mean data" position estimate is available.
    acis_cat = mrdfits(collatefile_ii, 1, /SILENT)

    ; Ignore sources from this ObsID that lack a position estimate or position error.
    acis_cat = acis_cat[ where(finite(acis_cat.X) AND finite(acis_cat.X_ERR), count) ]
    
    if (count EQ 0) then begin
      print, 'ERROR: no ACIS position errors available from ', collatefile_ii
      GOTO, FAILURE
    endif

    ; Evaluate the alignment of the catalogs.
    ; Note that match_xy's offset estimates will ignore ACIS positions that we marked as "NOT_FIDUCIAL" above.
    region_filename = 'shift.'+strlowcase(refname)+'.to.'+obsid[ii].obsname_base+'.reg'
    
    match_xy_estimate_offset, acis_cat, OBS_NAME='obs'+obsid[ii].obsname_base, ref_cat, REF_NAME=refname, $
            SIGNIFICANCE_THRESHOLD=significance_threshold, ASTROMETRY=fits_astrometry, $
            REGION_FILENAME=region_filename, PLOT_LEVEL=plot_level, $
            OBS_XSHIFT=this_xshift      ,       OBS_YSHIFT=this_yshift, $
      ERROR_OBS_XSHIFT=this_xshift_error, ERROR_OBS_YSHIFT=this_yshift_error

    
    ; If that produces no offset estimate, then try again using all ACIS sources and a more liberal match threshold.
    if ~finite(this_xshift) || ~finite(this_yshift) then begin
      print, F='(%"\nWARNING: with default parameters, no offset estimate could be estimated.  You can produce more matches (with lower quality) by relaxing the SIGNIFICANCE_THRESHOLD and/or THETA_THRESHOLD parameters.")'
    endif

    obs_ref_xshift         [ii] = this_xshift
    obs_ref_yshift         [ii] = this_yshift
    
    obs_ref_xshift_variance[ii] = this_xshift_error^2
    obs_ref_yshift_variance[ii] = this_yshift_error^2
    
  endfor ; ii
endif ; reference catalog supplied



;=============================================================================
;; Combine the one-hop and two-hop shifts estimates available for each ObsID.
;; Remember that we represent missing offset estimates and uncertainties by NaNs! 
if ref_cat_supplied then begin
  ; Convert some variances to standard deviations, for clarity and to be returned to caller. 
  obs_ref_xshift_error = sqrt( obs_ref_xshift_variance)
  obs_ref_yshift_error = sqrt( obs_ref_yshift_variance)

  id = lonarr(num_obs)
  for ii = 0,num_obs-1 do begin
    this_id = 0L
    function_1d, this_id, WIDGET_TITLE=obsid[ii].obsname_base, TITLE=obsid[ii].obsname_base, XTIT='XSHIFT (skypix)', YTIT='YSHIFT (skypix)', PLOT_WINDOW_OPTIONS='/FORCE_UNITY_ASPECT,SHOW_DATE=0', DATASET='weighted average', [0],[0]
    id[ii] = this_id
  endfor

  obs_ref_inflation = replicate(1.0, num_obs)
  
  repeat begin
    weighted_xshift          = replicate(!VALUES.F_NAN, num_obs)
    weighted_yshift          = replicate(!VALUES.F_NAN, num_obs)
    weighted_xshift_variance = replicate(!VALUES.F_NAN, num_obs)
    weighted_yshift_variance = replicate(!VALUES.F_NAN, num_obs)
    inflate_flag             =    bytarr(               num_obs)
    
    for ii = 0,num_obs-1 do begin
      ; Collate one-hop and two-hop shift estimates for ObsID[ii].
      xshift          = replicate(!VALUES.F_NAN, num_obs)
      yshift          = replicate(!VALUES.F_NAN, num_obs)
      xshift_variance = replicate(!VALUES.F_NAN, num_obs)
      yshift_variance = replicate(!VALUES.F_NAN, num_obs)
      
      for jj = 0,num_obs-1 do begin
        if (jj EQ ii) then begin
          ; One-hop offsets
          xshift         [jj] = obs_ref_xshift         [jj]
          yshift         [jj] = obs_ref_yshift         [jj]
          xshift_variance[jj] = obs_ref_xshift_variance[jj] * obs_ref_inflation[jj]
          yshift_variance[jj] = obs_ref_yshift_variance[jj] * obs_ref_inflation[jj]

          title =  obsid[ii].obsname_base+'->'+refname
          psym  = 7
        endif else begin
          ; Two-hop offsets are added; uncertainties are added in quadrature.
          ; For example, a short ObsID may have few matches to the reference catalog, but lots of matches to a deep ObsID, which has good matches to the reference catalog.
          xshift         [jj] = obs_obs_xshift         [jj,ii] + obs_ref_xshift         [jj]
          yshift         [jj] = obs_obs_yshift         [jj,ii] + obs_ref_yshift         [jj]
          xshift_variance[jj] = obs_obs_xshift_variance[jj,ii] + obs_ref_xshift_variance[jj] * obs_ref_inflation[jj]
          yshift_variance[jj] = obs_obs_yshift_variance[jj,ii] + obs_ref_yshift_variance[jj] * obs_ref_inflation[jj]
          
          title = obsid[ii].obsname_base+'->'+obsid[jj].obsname_base+'->'+refname
          psym  = 6
        endelse
        
        ; Note that obs_ref offsets are plotted with error bars inflated as needed.
        if finite(xshift[jj]) && finite(yshift[jj]) then $
        function_1d, id[ii], PSYM=psym, DATASET=title, REDRAW=0, $
                                  xshift         [jj] , $
                                  yshift         [jj] , $
                     X_ERROR=sqrt(xshift_variance[jj]), $
                     Y_ERROR=sqrt(yshift_variance[jj]), $
                     SUBTIT=string(obsid[ii].obsname_base, refname, sqrt(obs_ref_inflation[ii]),  F='(%"Error bars for %s->%s are inflated by %0.1f.")')
      endfor ;jj
      
      if ~array_equal(finite(xshift),finite(yshift         )) then message, 'BUG in ae_interObsID_astrometry'
      if ~array_equal(finite(xshift),finite(xshift_variance)) then message, 'BUG in ae_interObsID_astrometry'
      if ~array_equal(finite(xshift),finite(yshift_variance)) then message, 'BUG in ae_interObsID_astrometry'
      num_estimates = total(finite(xshift))
      if (num_estimates EQ 0) then begin
        print, obsid[ii].obsname_base, F='(%"No shift estimates available for %s")'
        continue
      endif ; no estimates
      
      ; Calculate weighted mean of the shift estimates for ObsID[ii].
      ; As per https://en.wikipedia.org/wiki/Weighted_arithmetic_mean#Dealing_with_variance and Bevington (2nd Edition) Chapter 4
      weight                       = 1. / xshift_variance                                 ; Bevington (2nd Edition) 4.17
      weight_total                 = total(/DOUBLE, /NAN, weight) 
      weighted_xshift         [ii] = total(/DOUBLE, /NAN, weight*xshift) / weight_total
      weighted_xshift_variance[ii] =                                  (1 / weight_total) ; Bevington (2nd Edition) 4.19

      weight                       = 1. / yshift_variance                                 ; Bevington (2nd Edition) 4.17
      weight_total                 = total(/DOUBLE, /NAN, weight) 
      weighted_yshift         [ii] = total(/DOUBLE, /NAN, weight*yshift) / weight_total
      weighted_yshift_variance[ii] =                                  (1 / weight_total) ; Bevington (2nd Edition) 4.19

      if (num_estimates EQ 1) then continue   ; if only one estimate then no inflation required
      
      
      ; Calculate the "standard score" or "z-score" for the residuals between the "data" (shift estimates) and model (weighted mean).
      Zx = abs(xshift - weighted_xshift[ii]) / sqrt(xshift_variance)
      Zy = abs(yshift - weighted_yshift[ii]) / sqrt(yshift_variance)
      
      ; If reduced chi^2 is larger than 1.0, then flag the largest Z terms.
      if (total(/DOUBLE, /NAN, Zx^2) / (num_estimates - 1)) GT 1 then begin
        dum = max(Zx,imax)
        inflate_flag[imax] = 1B
      endif
      
      if (total(/DOUBLE, /NAN, Zy^2) / (num_estimates - 1)) GT 1 then begin
        dum = max(Zy,imax)
        inflate_flag[imax] = 1B
      endif
  
    endfor ;ii
    
    ; Break if no more inflation is necessary.
    if array_equal(inflate_flag, 0B) then break
    
    ; Inflate uncertainties for every obs_ref flagged as suspicious.
    ind = where(inflate_flag) 
    obs_ref_inflation[ind] *= 1.1                                       
    print, F='(%"..")'
    forprint, SUBSET=ind, obsid.obsname_base+'->'+refname, sqrt(obs_ref_inflation), F='(%"  %s uncertainties inflated by a factor of %0.2f")'

  endrep until 0

  ; Convert some variances to standard deviations, for clarity and to be returned to caller. 
  weighted_xshift_error = sqrt(weighted_xshift_variance)
  weighted_yshift_error = sqrt(weighted_yshift_variance)

  for ii = 0,num_obs-1 do function_1d, id[ii], REDRAW=1, PSYM=8, PLOTSYM='3,2,THICK=1', DATASET='weighted average',$
                                       weighted_xshift      [ii],$
                                       weighted_yshift      [ii],$
                               X_ERROR=weighted_xshift_error[ii],$
                               Y_ERROR=weighted_yshift_error[ii]


  obs_ref_xshift_report = string(     obs_ref_xshift                        , F='(%"%7.3f")'    ) + $
                          string(     obs_ref_xshift_error                  , F='(%" +-%-4.2f")')
                        
  weighted_xshift_report= string(    weighted_xshift                        , F='(%"  ==>%7.3f")'    ) + $
                          string(abs(weighted_xshift)/weighted_xshift_error , F='(%" S=%-4.1f")') 

  obs_ref_yshift_report = string(     obs_ref_yshift                        , F='(%"%7.3f")'    ) + $
                          string(     obs_ref_yshift_error                  , F='(%" +-%-4.2f")')
                        
  weighted_yshift_report= string(    weighted_yshift                        , F='(%"  ==>%7.3f")'    ) + $
                          string(abs(weighted_yshift)/weighted_yshift_error , F='(%" S=%-4.1f")') 

  inflation_report      = string( sqrt(obs_ref_inflation)                   , F='(%"*%4.1f")')
  
  inflation_report[where(/NULL, obs_ref_inflation EQ 1)] = '     '

  refname_formatted     = string(refname, F='(%"%14s")')
endif else begin
   obs_ref_xshift_report = ''
   obs_ref_yshift_report = ''
  weighted_xshift_report = ''
  weighted_yshift_report = ''
  inflation_report       = ''
  refname_formatted      = ''
endelse

; Convert some variances to standard deviations, for clarity and to be returned to caller. 
obs_obs_xshift_error = sqrt(obs_obs_xshift_variance)
obs_obs_yshift_error = sqrt(obs_obs_yshift_variance)

obs_obs_xshift_report = reform(string(obs_obs_xshift, F='(%"%7.3f")') + string(obs_obs_xshift_error, F='(%" +-%-4.1f")'), num_obs, num_obs)

obs_obs_yshift_report = reform(string(obs_obs_yshift, F='(%"%7.3f")') + string(obs_obs_yshift_error, F='(%" +-%-4.1f")'), num_obs, num_obs)

exposure_formatted = '      ('+string(obsid_exposure/1000, F='(%"%3d")')+' ks)'
obsname_formatted  =           string(obsid.obsname_base , F='(%"%14s")')
                                                                                    
; Remove the entries where no estimate was made.
ind = where( strmatch(obs_obs_xshift_report      , '*NaN +-NaN*'), count)
if (count GT 0) then  obs_obs_xshift_report[ind] = '        -     '
ind = where( strmatch(obs_obs_yshift_report      , '*NaN +-NaN*'), count)
if (count GT 0) then  obs_obs_yshift_report[ind] = '        -     '

; Hide the upper-right half of this symmetric matrix and mark the diagonals, for easier reading.
;for jj = 0,num_obs-2 do obs_obs_xshift_report[jj+1:*, jj] = '              '
;for jj = 0,num_obs-2 do obs_obs_yshift_report[jj+1:*, jj] = '              '
for jj = 0,num_obs-1 do obs_obs_xshift_report[jj    , jj] = '        X     '
for jj = 0,num_obs-1 do obs_obs_yshift_report[jj    , jj] = '        X     '

print, F="(%'\n\n\n======================== ALIGNMENT SUMMARY ========================')"
print, F='(%"\nXSHIFTS [skypix]")'
for jj = 0,num_obs-1 do print, 'shift ', obsname_formatted[jj], ' by ', obs_obs_xshift_report[*,jj], obs_ref_xshift_report[jj], inflation_report[jj], weighted_xshift_report[jj], F='(100A)'

print, 'to align with       ',obsname_formatted, refname_formatted, F='(100A)'
print, '                    ',exposure_formatted                  , F='(100A)'

print, F='(%"\nYSHIFTS [skypix]")'
for jj = 0,num_obs-1 do print, 'shift ', obsname_formatted[jj], ' by ', obs_obs_yshift_report[*,jj], obs_ref_yshift_report[jj], inflation_report[jj], weighted_yshift_report[jj], F='(100A)'

print, 'to align with       ',obsname_formatted , refname_formatted, '                recommendation'   , F='(100A)'
print, '                    ',exposure_formatted,  '              ', '               (weighted average)', F='(100A)'
print


;=============================================================================
; Make recommendations about shifting ObsIDs using the best-available shift estimates.
; Our policy is to recommend only those shifts that have more than 1-sigma significance.
; The "xshift" and "xshift" structure tags cary the best shift estimates, even if SNR is low..
; The "action_xshift" and "action_yshift" tags carry the shift we are recommending.

template_row = {obsid:'', obsdir:'', xshift:!VALUES.F_NAN      , yshift:!VALUES.F_NAN, $
                                     xshift_error:!VALUES.F_NAN, yshift_error:!VALUES.F_NAN, label:'weighted average',$
                     action:'HOLD', action_xshift:0.           , action_yshift:0.}

recommendation = replicate(template_row, num_obs)
recommendation.obsid        = obsid.obsname_base
recommendation.obsdir       = obsid.obsdir_repro
recommendation.xshift       = weighted_xshift      
recommendation.yshift       = weighted_yshift      
recommendation.xshift_error = weighted_xshift_error
recommendation.yshift_error = weighted_yshift_error
recommendation.action       = 'HOLD'

ind = where(/NULL, abs(weighted_xshift/weighted_xshift_error) GT 1 )
if isa(ind, /INTEGER) then begin
  recommendation[ind].action        = 'SHIFT'
  recommendation[ind].action_xshift = weighted_xshift [ind]
endif

ind = where(/NULL, abs(weighted_yshift/weighted_yshift_error) GT 1 )
if isa(ind, /INTEGER) then begin
  recommendation[ind].action        = 'SHIFT'
  recommendation[ind].action_yshift = weighted_yshift [ind]
endif

; If shift estimate is missing for either axis, then report this as the action "NA".
recommendation[where(/NULL, ~finite(weighted_xshift) OR ~finite(weighted_yshift))].action = 'NA'

save, /COMPRESS, FILE='ae_interObsID_astrometry.sav',$
  refname, significance_threshold, theta_threshold,  detection_threshold, obsid, recommendation,$
 weighted_xshift, weighted_yshift, weighted_xshift_error, weighted_yshift_error,$
  obs_obs_xshift,  obs_obs_yshift,  obs_obs_xshift_error,  obs_obs_yshift_error,$
  obs_ref_xshift,  obs_ref_yshift,  obs_ref_xshift_error,  obs_ref_yshift_error



if keyword_set(target_name) then begin
    print, F="(%'\n======================== RECOMMENDATIONS ========================')"
    ; Show recommended shifts and set up a screen session to implement them.
    SCREEN_ARCH = getenv('SCREEN_ARCH')
    cmd_prefix = "screen -S 'adjust_astrometry_"+target_name+"'"
    cd,  CURRENT=cwd

    run_command, /QUIET, string(cmd_prefix,         F='(%"%s -X quit ")'), /IGNORE_STATUS

    cmd = [string(cmd_prefix, target_name, SCREEN_ARCH, F='(%"%s -d -m -t ''%s'' %s")'),$
           ; Window #1 ("target-level") is cd'd to <target>/data/
           string(cmd_prefix,       '../..', F='(%"%s -X chdir ''%s'' ")'),$
                                                     "setenv PROMPT 'target-level %% '" ,$
           string(cmd_prefix, SCREEN_ARCH,   F='(%"%s -X screen -t ''target-level'' %s")'),$
           string(cmd_prefix,                F='(%"%s -X stuff ''pwd^M'' ")')]
    run_command, /QUIET, cmd, /OMIT_FAST_START

    ds9_cmd = "ds9 -lock frame wcs -lock bin yes -lock scale yes -lock colorbar yes -linear -title '"+target_name+" proposed shifts' "
    vector_fn  = "shift." + recommendation.obsid + ".reg"

    for ii=0, n_elements(recommendation)-1 do begin
      this_obs = recommendation[ii].obsid
      case       recommendation[ii].action of
      'NA'   : print, this_obs                         , F='(%"\nObsID %s:  WARNING: NO adjustment estimate could be made.")'
      'HOLD' : print, this_obs,recommendation[ii].label, F='(%"\nObsID %s (%s):  NO adjustment recommended.")'
      'SHIFT': $
         begin
         chdir = cwd+'/' + recommendation[ii].obsdir
         paste = string(file_which('adjust_astrometry.csh'), $
                      recommendation[ii].action_xshift, $
                      recommendation[ii].action_yshift, F='(%"ciao; %s  deltax=%0.3f  deltay=%0.3f")')

         print, this_obs, recommendation[ii].label, paste, F='(%"\nObsID %s (%s):\n  %s")'

         ; ObsID windows are cd'd to data/pointing*/obsid_XXXX/ with shift command pre-pasted.
         cmd = [string(cmd_prefix, this_obs,        F='(%"%s           -X setenv PROMPT ''%s \%\% '' ")'),$
                string(cmd_prefix,           chdir, F='(%"%s           -X chdir ''%s'' ")')          ,$
                string(cmd_prefix, this_obs, SCREEN_ARCH, F='(%"%s     -X screen -t ''%s'' %s")')          ,$
                string(cmd_prefix, this_obs, paste, F='(%"%s -p ''%s'' -X stuff     ''\f pwd^M %s'' ")')]
         run_command, /QUIET, cmd
         end
      endcase

      ; Draw vectors (for FIDUCIAL sources) showing proposed movement of this ObsID's catalog
      arcsec_per_skypixel = 0.492 ; (0.492 arcsec/skypix)
      dx = recommendation[ii].action_xshift
      dy = recommendation[ii].action_yshift
      
      if ( recommendation[ii].action EQ 'SHIFT' ) then begin
        length  = arcsec_per_skypixel*sqrt(dx^2 + dy^2)   ; arcsec
        azimuth = atan(dy, dx)*!RADEG  ; CC from +X axis
      endif else begin
        length = 0
        azimuth  = 0
      endelse

      ; Read (RA_DATA, DEC_DATA) coordinates and NOT_FIDUCIAL flag from ObsID collation.
      ; Note that cat.X and cat.Y are NaN when no "mean data" position estimate is available.
      cat_ii = mrdfits(obsid[ii].collatefile, 1, /SILENT)

      num_sources = n_elements(cat_ii)

      ind = where(/NULL, finite(cat_ii.X) AND ~cat_ii.NOT_FIDUCIAL)
      if isa(ind, /INTEGER) then begin
        forprint, TEXTOUT=vector_fn[ii], SUBSET=ind, /NoCOM, cat_ii.RA_DATA, cat_ii.DEC_DATA, replicate(length,num_sources), replicate(azimuth,num_sources), cat_ii.LABEL, replicate(this_obs,num_sources), F='(%"J2000;vector %10.6f %10.6f %0.2f\" %0.2f  # vector=1 text={%s} tag={proposed shift} tag={%s} color=red")'

      endif else $
        forprint, TEXTOUT=vector_fn[ii], /NoCOM, mean(cat_ii.RA_DATA), mean(cat_ii.DEC_DATA), this_obs, this_obs, F='(%"J2000;text %10.6f %10.6f # text={No FIDUCIAL sources in ObsID %s} tag={proposed shift} tag={%s} color=red")'
  
      ds9_cmd = ds9_cmd + string(recommendation[ii].obsdir+'/acis.validation.evt2', vector_fn[ii], F='(%"%s -region %s ")')

    endfor ; ii

    ; Remove the unneeded window 0 to avoid confusion.
    run_command, /QUIET, string(cmd_prefix, F='(%"%s -p 0 -X kill ")')
    
    ; Tell observer about the screen session built above.
    ind = where(recommendation.action EQ 'SHIFT', count)
    if count EQ 0 then begin
      run_command, /QUIET, string(cmd_prefix,  F='(%"%s -X quit ")')

    endif else begin
      print, strjoin(recommendation[ind].obsid,' '), cmd_prefix, F='(%"\nA detached screen session has been set up to apply the recommended astrometry adjustments to ObsIds\n  %s \nRun the following in a new shell to attach to that session:\n  %s -r ")' 

      print, F='(%"\n\nIF YOU DECIDE TO SHIFT ANY ObsID, then do NOT exit this IDL sesssion.\nCONTINUE READING THE PROCEDURE to apply the shifts and to update a variety of data products that are affected by an astrometry adjustment!")'

    endelse ; ObsID shifts were recommended
    
    ; Show ds9 command to display the shift vectors on top of the ObsID data.
   ;print, ds9_cmd, F='(%"\n  %s\n")'
    run_command, ds9_cmd+' -zoom to fit >& /dev/null &'     
endif ; keyword_set(target_name)

CLEANUP:
if keyword_set(tempdir) && file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then begin
  return 
endif else begin
  print, 'ae_interObsID_astrometry: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end ; ae_interObsID_astrometry



;############################################################################# 
; Tool to predicts new source positions based on ObsID shifts that have just been applied.
PRO ae_reposition_to_follow_obsid_shifts, event2wcs_astr, recommendation

; We must run a stripped-down 'position' merge now to predict which set of ObsIDs will be used by the position merge in  SECTION VI.  The 'most_valid' merge we have on-hand is a POOR predictor!
print, 'Running a "position" merge to figure out which ObsIDs affect each source position.'

par = ae_get_target_parameters()

target_name = getenv('TARGET')
if ~keyword_set(target_name) then target_name = 'Unknown Target'

acis_extract, 'all.srclist', MERGE_NAME='position', /MERGE_OBSERVATIONS, MERGE_FOR_POSITION=par.MERGE_FOR_POSITION, MIN_NUM_CTS=3, OVERLAP_LIMIT=0.10, ENERGY_RANGE=[0.5,7], /SKIP_PSF, /SKIP_NEIGHBORHOOD, /SKIP_APERTURE, /SKIP_SPECTRA, /SKIP_TIMING, GENERIC_RMF_FN='generic.rmf'

readcol, COUNT=num_sources, 'all.srclist', sourcename, FORMAT='A', COMMENT=';'
delta_x     = fltarr(num_sources) ; default is to not move the source
delta_y     = fltarr(num_sources)
for ii=0,num_sources-1 do begin
  ; Read FITS table listing ObsIDs that were retained by "position" merge.
  sourcedir = sourcename[ii] + '/position/' 
  merged_list_fn       = sourcedir + 'ObsIDs_merged.fits'
  if ~file_test(merged_list_fn) then begin
    print, sourcename[ii], merged_list_fn, F='(%"Skipping source %s because file %s not found.")'
    continue
  endif

  obs_data = mrdfits(merged_list_fn,1,/SILENT)
  obs_data.obsid = strtrim(obs_data.obsid,2) ; FITS string columns can end up with trailing blanks!

  ; Compute mean shift of those ObsIDs, weighted by median exposure map values.
  ; THIS WEIGHTED MEAN MUST BE CALCULATED ON ALL ObsIDs used in the merge, even those not passed to ae_interObsID_astrometry or those not moving!!!

  ; If any weight is undefined then skip this source.
  weight = obs_data.emap_med
  if ~array_equal(finite(weight), 1) then begin
    print, merged_list_fn, F='(%"WARNING!  Some EMAP_MED fields in %s are not finite!  No position updated computed for this source.")' 
    continue
  endif

  num_obs    = n_elements(weight)
  obs_xshift = fltarr(num_obs)
  obs_yshift = fltarr(num_obs)

  for jj=0,num_obs-1 do begin
     ; Find shift recommendation for this ObsID.
     this_recommendation = recommendation[ where(/NULL, recommendation.obsid EQ obs_data[jj].obsid) ]

     if (this_recommendation        EQ !NULL  ) then begin
       print, obs_data[jj].obsid, F='(%"Warning: ObsID %s was used in a MERGE, but ae_interObsID_astrometry did not check its astrometry.")'
       continue
     endif

     obs_xshift[jj] = this_recommendation.action_xshift
     obs_yshift[jj] = this_recommendation.action_yshift
  endfor ; jj

  ; Calculate weighted shift.
  delta_x[ii] = total(/DOUBLE, weight*obs_xshift) / total(/DOUBLE, weight)
  delta_y[ii] = total(/DOUBLE, weight*obs_yshift) / total(/DOUBLE, weight)
endfor ; ii

; Plot distribution of shifts.
dataset_1d, idx, delta_x, XTIT='delta_X (skypix)', WIDGET_TITLE='Source Shifts to Follow Event Data', BINSIZE=0.01
dataset_1d, idy, delta_y, XTIT='delta_Y (skypix)', WIDGET_TITLE='Source Shifts to Follow Event Data', BINSIZE=0.01

print, median(delta_x),median(delta_y), F='(%"\nMedian proposed source movements are (%0.3f,%0.3f) skypix.\n")'

title = string(target_name, F='(%"TARGET %s: Following The Light")')
msg = 'Do you want to accept the proposed source movements?'
print, msg
TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='No, abort.', BUTTON_LABEL='Yes, MOVE the sources', BUTTON_ID=button_id, QUIT_ID=quit_id, PRESSED_ID=pressed_id

if (pressed_id EQ button_id) then begin
  ae_source_manager, /MOVE, NAME=sourcename, ASTROMETRY=event2wcs_astr, DELTA_X=delta_x, DELTA_Y=delta_y, POSITION_TYPE=''
endif

return
end ; ae_reposition_to_follow_obsid_shifts




;############################################################################# 
;;; Driver program that executes an IDL procedure written by an external actor (e.g. another IDL session).
;;;
;;; Example:   ae_processing_server, '../obs1234/ae_code_block.pro'

PRO ae_processing_server, procedure_path

procedure_name = file_basename(procedure_path , '.pro')

; Put the preocedure directory at the head of !PATH.
!PATH = file_dirname(procedure_path) + ":" + !PATH
help, procedure_name, !PATH

; Loop forever.  External actor will include "exit" in procedure to terminate this IDL session.
while 1 do begin
  ; Wait until external actor writes IDL procedure
  wait_until_files_exist, 10, procedure_path
  
  ; Pause to ensure completion of write operation on procedure_path.
  wait, 2
  
  ; Compile procedure.
  catch, error_code
  if (error_code NE 0) then begin
    print, 'COMPILATION FAILURE in '+procedure_path
      print, 'Type .continue when you have resolved the problem.'
      stop
      continue
  endif else RESOLVE_ROUTINE, procedure_name
  catch, /CANCEL

  ; Verify the compiled routine came from the correct place.
  print, (routine_info(procedure_name, /SOURCE)).PATH 
  
  
  ; Execute procedure.
  print, procedure_path,  now(), F='(%"\n\n\n==========================================================================\nRUNNING PROCEDURE %s at %s\n==========================================================================\n ")'
  
  CALL_PROCEDURE, procedure_name
  
;; If an error handler is defined as below, we must ignore "expected" errors, like floating overflow.
;  catch, error_code
;  if (error_code NE 0) then begin
;    print, 'EXECUTION FAILURE in '+procedure_path
;    print, !ERROR_STATE.MSG
;    print, 'Type .continue when you have resolved the problem.'
;    stop
;    continue
;  endif else CALL_PROCEDURE, procedure_name
;  catch, /CANCEL
  
  print, 'Finished CALL_PROCEDURE'
  
  ; Remove procedure, which signals completion to external actor.
  file_delete, procedure_path
  ; Pause to ensure completion of remove operation before looping.
;  wait, 10
endwhile

return
end ; ae_processing_server




;############################################################################# 
; This block of code performs a full extraction of the catalog from a single ObsID.
; That task consists of the following steps:
; 1  construct non-overlapping apertures
; 2  extract apertures
; 3  calculate single-ObsID photometry
; 4  build point source models
; 5  build background regions/spectra
; 6  participate in BACKSCAL adjustment iterations
;
; That cycle requires iteration to converge (because photometry requires a background estimate).
; This tool will automatically repeat steps 3,4,5  (after step 6) when this is the first background extraction for >5% of sources, to help the photometry advance toward convergence.
; Specifying /REPEAT_EXTRACT_BACKGROUNDS will force that extra cycle to be run.

PRO extract_and_pileup_block1, par, EXTRACT_BACKGROUNDS       =extract_backgrounds,$
                                    ADJUST_BACKSCAL           =adjust_backscal,$
                                    REPEAT_EXTRACT_BACKGROUNDS=repeat_extract_backgrounds,$
                                    BACKGROUND_MODEL_FILENAME=background_model_filename, $
                                    RESTART=restart

    if n_elements(extract_backgrounds) EQ 0 then extract_backgrounds = 1
    if n_elements(adjust_backscal    ) EQ 0 then adjust_backscal     = 1
    
    run_command, 'hostname'

    obsdir = "../obs"+getenv("OBS")+"/" 
    file_delete, obsdir+"ae_finished", /ALLOW_NONEXISTENT 
    
    ; Claim a CPU before starting processing
    semaphore = wait_for_cpu(LOW_PRIORITY=strmatch(obsdir,'*_BI'))
    
    generic_rmf_fn = par.validation_mode ? 'generic.rmf' : '' 

    ae_make_catalog       , getenv("OBS"), SOURCE_NOT_OBSERVED=source_not_observed, EVTFILE="spectral.evt", REUSE_NEIGHBORHOOD=par.reuse_neighborhood, REGION_ONLY=par.validation_mode, REUSE_PSF=~par.DISCARD_PSF, RESTART=restart
    
    save, /COMPRESS, source_not_observed, FILE=obsdir+"/source_not_observed.sav" 

    ae_standard_extraction, getenv("OBS"), SOURCE_NOT_OBSERVED=source_not_observed, EVTFILE="spectral.evt", REUSE_NEIGHBORHOOD=par.reuse_neighborhood, REUSE_ARF=par.validation_mode, TIMING=~par.validation_mode, GENERIC_RMF_FN=generic_rmf_fn, EXTRACT_BACKGROUNDS=0 

    ; Release the CPU.
    release_cpu
    
    if par.check_for_pileup then begin 
      ae_pileup_screening, getenv("OBS"), NUM_POSSIBLY_PILED=npp

      if (npp GT 0) then begin
        screen_window = 5 + where( strtrim(strsplit(getenv('OBS_LIST'), /EXTRACT), 2) EQ strtrim(getenv("OBS"),2) )
        title = string(getenv("OBS"), screen_window, getenv("TARGET"), F='(%"ObsID %s (window %d in %s) has piled sources.")')
        msg = 'Close when you have completed whatever pile-up corrections you wish to run.'
        print, msg
        TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='Close', PRESSED_ID=trash
      endif
    endif ; par.check_for_pileup

    if extract_backgrounds then begin
      ; Claim a CPU before starting processing
      semaphore = wait_for_cpu(LOW_PRIORITY=strmatch(obsdir,'*_BI'))
    
      ae_standard_extraction, getenv("OBS"), SOURCE_NOT_OBSERVED=source_not_observed, EVTFILE="spectral.evt", EXTRACT_SPECTRA=0, TIMING=0, GENERIC_RMF_FN=generic_rmf_fn, /BETTER_BACKGROUNDS, NEIGHBOR_INVALID_THRESHOLD=par.neighbor_invalid_threshold, NUMBER_OF_PASSES=par.NUMBER_OF_PASSES, SKIP_RESIDUALS=par.skip_residuals, BACKGROUND_MODEL_FILENAME=background_model_filename
      
      ; Release the CPU.
      release_cpu

      if adjust_backscal then begin
        ; Signal to the ae_adjust_backscal_range session that this ObsID has built all backgrounds.
        file_delete,                 obsdir+["ae_abort_requested","ae_run_requested"], /ALLOW_NONEXISTENT 
        file_move, obsdir+"ae_lock", obsdir+"ae_finished", /OVERWRITE 
        
        while 1 do begin 
          wait_until_files_exist, 10, obsdir+["ae_abort_requested","ae_run_requested"], FILE_WAS_FOUND=flag, /DELETE 
          ; This IDL session waits here while the ae_adjust_backscal_range session is building rerun.srclist. 
          if flag[0] then break 
    
          ; Claim a CPU before starting processing
          semaphore = wait_for_cpu(LOW_PRIORITY=strmatch(obsdir,'*_BI'))
          
          ae_standard_extraction, SRCLIST_FILENAME="rerun.srclist", getenv("OBS"), EVTFILE="spectral.evt", EXTRACT_SPECTRA=0, TIMING=0, GENERIC_RMF_FN=generic_rmf_fn, /BETTER_BACKGROUNDS, /REUSE_MODELS
    
          ; Release the CPU.
          release_cpu
          
          ; Signal to the ae_adjust_backscal_range session that this ObsID has built these backgrounds.
          file_move, obsdir+"ae_lock", obsdir+"ae_finished", /OVERWRITE 
        endwhile
      endif ; ~adjust_backscal
  
      ; Claim a CPU before starting processing
      semaphore = wait_for_cpu(LOW_PRIORITY=strmatch(obsdir,'*_BI'))
                                                          
      ; If the collation run by BETTER_BACKGROUNDS above was missing background estimates, then the source models built  by BETTER_BACKGROUNDS are biased and the background extractions are damaged.  Repeat BETTER_BACKGROUNDS to help the photometry advance toward convergence.
      bt = mrdfits(obsdir + 'all.collated',1)

      if tag_exist(bt, 'NUM_OBS') then begin
        num_observed            = total(/INT, (bt.NUM_OBS GT 0))
        num_missing_backgrounds = total(/INT, (bt.NUM_OBS GT 0) AND ~finite(bt.BACKSCAL[0]))
        fraction_missing_backgrounds = float(num_missing_backgrounds)/num_observed
      endif else fraction_missing_backgrounds = 0

      if (fraction_missing_backgrounds GT 0.05) || keyword_set(repeat_extract_backgrounds) then begin
        print, 100*fraction_missing_backgrounds, F='(%"\nextract_and_pileup_block1: Since this was the FIRST cycle of background estimation for %d%% of sources, BETTER_BACKGROUNDS is being repeated now to help the photometry advance toward convergence.")'
        
        ae_standard_extraction, getenv("OBS"), SOURCE_NOT_OBSERVED=source_not_observed, EVTFILE="spectral.evt", EXTRACT_SPECTRA=0, TIMING=0, GENERIC_RMF_FN="generic.rmf", /BETTER_BACKGROUNDS, NEIGHBOR_INVALID_THRESHOLD=par.neighbor_invalid_threshold, NUMBER_OF_PASSES=1, /SKIP_RESIDUALS, BACKGROUND_MODEL_FILENAME=background_model_filename 
      endif ; 
      
      ; Rebuild single-ObsID merges and their collations using the backgrounds calculated above.
      ; We will soon examine those collations to make validation decisions, so we want them to be up-to-date!
      print,  F='(%"\nextract_and_pileup_block1: Rebuilding single-ObsID merges and their collations using the backgrounds calculated above.")'
      ae_standard_extraction, getenv("OBS"), SOURCE_NOT_OBSERVED=source_not_observed, EVTFILE="spectral.evt", EXTRACT_SPECTRA=0, TIMING=0, GENERIC_RMF_FN=generic_rmf_fn, /BETTER_BACKGROUNDS, /PHOTOMETRY_STAGE_ONLY, VERBOSE=0
  
      ; Release the CPU.
      release_cpu
    endif ; extract_backgrounds

    ; Do NOT rename obsdir+"ae_lock" to obsdir+"ae_finished" here, because the caller may need to do additional
    ; work before making that "finished" declaration to other IDL sessions.
    
return
end ; extract_and_pileup_block1



;#############################################################################
; This block of code interacts with extract_and_pileup_block1 to iteratively adjust BACKSCAL ranges.
PRO manage_extract_and_pileup_block1

par           = ae_get_target_parameters()
overlap_limit = 0.10

obsname = strtrim(strsplit(getenv('OBS_LIST'), /EXTRACT), 2)
if ~keyword_set(obsname) then begin
  print, 'manage_extract_and_pileup_block1: ERROR: environment variable OBS_LIST must be defined.'
  stop
endif
obsdir  = '../obs'+obsname+'/'

; Wait for full extractions to finish.
wait_until_files_exist, 60, obsdir+'ae_finished', /DELETE

if par.check_for_pileup then begin 
  run_command, "cut -f1 -d ')' ../obs*/possibly_piled.srclist | sort | uniq > possibly_piled.srclist", /UNIX
endif

if ~par.skip_residuals then begin
  msg = 'Iconified ds9 sessions for each ObsID show the observed data (upper frames), a model for all the point sources (lower-left frame) and the SMOOTHED residuals (lower-right frame) remaining after the model is subtracted from the data.'
  title = string(getenv("TARGET"), F='(%"TARGET %s: iconified ds9 sessions")')
  print, msg
  TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='This window will close when you quit IDL.', LIFETIME=3600.*24*5
endif

; =========================================================================================
; Manage adjustment of BACKSCAL range.
; In this section we reconsider the background scaling range allowed for each source, make adjustments where needed, and re-compute background spectra for those sources adjusted.  Ideally, that process would be repeated until no adjustments are needed or until adjustments become futile.  We will repeat this cycle 20 times, which is almost always adequate.

for ii=1,20 do begin

  ae_adjust_backscal_range, SRCLIST_FILENAME=(ii EQ 1) ? 'all.srclist' : 'rerun.srclist',$
                            OVERLAP_LIMIT=overlap_limit, MIN_NUM_CTS=100, $
                            NUM_SOURCES_ADJUSTED=num_sources_adjusted

  if (num_sources_adjusted EQ 0) then break

  ; Signal the ObsID sessions to extract rerun.srclist and wait for them to finish.
  foreach file, obsdir+'ae_run_requested' do begin
    run_command, 'touch ' + file
    wait, 0.2
  endforeach
  wait_until_files_exist, 30, obsdir+'ae_finished', /DELETE
endfor ;ii

if (num_sources_adjusted GT 0) then begin
  ; Count the total number of sources.
  readcol, 'all.srclist', sourcename, FORMAT='A', COMMENT=';'
  sourcename = strtrim(sourcename,2)
  ind = where(sourcename NE '', num_sources)

  fraction_sources_adjusted = num_sources_adjusted / float(num_sources)
  if (fraction_sources_adjusted GT 0.01) then $
    print, round(100*fraction_sources_adjusted), F='(%"\n\nWARNING: BACKSCAL range adjustments have not converged for more than %d%% of sources.\n\n")'
endif

; Tell each ObsID that BACKSCAL adjustment is complete, so it can move on.
print, F="(%'\nTelling ObsIDs that BACKSCAL adjustment is complete ...\n ')" 
foreach file, obsdir+'ae_abort_requested' do begin
  run_command, 'touch ' + file
  wait, 0.2
endforeach
wait_while_files_exist, 5, obsdir+'ae_abort_requested'

; Wait for completion of final single-ObsID merges and collations.
print, F="(%'\nWaiting while ObsIDs rebuild single-ObsID merges and collations ...\n ')" 
wait_until_files_exist, 30, obsdir+'ae_finished'

end ; manage_extract_and_pileup_block1



;############################################################################# 
; This is the block of code that validates source candidates in the section titled
; "PRUNE INSIGNIFICANT SOURCE CANDIDATES" in validate_procedure.txt
PRO validation_block1, pass_name, VERIFY_EXTRACTIONS=verify_extractions ,$
                                  CALCULATE_BEST_PB =calculate_best_pb, $
                       NUM_PRUNED=num_pruned


    par               = ae_get_target_parameters()
    overlap_limit     = 0.10

    obsname = strtrim(strsplit(getenv('OBS_LIST'), /EXTRACT), 2)
    obsdir  = '../obs'+obsname+'/'

    ; =========================================================================================
    ; Validate the extractions
    if keyword_set(verify_extractions) then verify_extractions, /IGNORE_APERTURE_CORRECTIONS, /IGNORE_ARF_DATES
    
    
    ; =========================================================================================
    ; Assess Source Validity
    single_ObsID_collation_fn = obsdir + 'all.collated'
    single_ObsID_model_fn     = obsdir + 'ae_better_backgrounds.sav'
    theta_range_srclist       = 'theta_range_merges.srclist'
    
    ; Determine if we require every possible merge (single-ObsID merges and all theta merges for every source).
    ; When the caller specifically asks for the best Pb value (/CALCULATE_BEST_PB), we must obviously calculate Pb in every possible merge.
    ; When we need to prune afterglow sourcess, we must judge the impact of afterglows in the Pb calculation for the best-possible merge for each source.
      
    analyze_every_possible_merge = keyword_set(calculate_best_pb) || par.CHECK_FOR_AFTERGLOWS


    if array_equal(/NOT_EQUAL, file_test(single_ObsID_collation_fn), 1) then message, 'Some all.collated files are missing!'

    ; Verify that the single-ObsID photometry was built using the most recent backgrounds.
    if ~array_equal((file_info(single_ObsID_collation_fn)).MTIME GT $
                    (file_info(single_ObsID_model_fn    )).MTIME, 1) then message, 'Some all.collated files are stale!'

    if (n_elements(obsname) GT 1) then begin
      ;------------------------------------------------------------------------
      ; Determine which sources require theta-range merges on this pass.
      
      if analyze_every_possible_merge then begin

        file_copy, 'all.srclist', theta_range_srclist, /OVERWRITE

      endif else begin
        ; When our only interest is validation, we can skip sources validated by the existing single-ObsID merges.
        
        ae_analyze_pb, /ANALYZE, single_ObsID_collation_fn, OVERLAP_LIMIT=overlap_limit, INVALID_THRESHOLD=par.INVALID_THRESHOLD, PB_IN_VHARD_BAND=par.PB_IN_VHARD_BAND, NOT_VALIDATED_SRCLIST_FILENAME='not_validated.srclist'

        file_copy, 'not_validated.srclist', theta_range_srclist, /OVERWRITE
      endelse
    
      ;------------------------------------------------------------------------
      ; Run merges defined by off-axis angle (theta) slices, for the set of sources determined above.
      theta_lo = [0,2,5,  8,13] ; arcmin
      theta_hi = [3,6,9, 14,99] ; arcmin
      
      merge_name  = 'theta_' + string(theta_lo, F='(%"%2.2d")') +'-'+ string(theta_hi, F='(%"%2.2d")')
      theta_collation_fn = 'tables/' + merge_name + '.collated'

      ; Remove any existing collations that we will read below.
      file_delete, theta_collation_fn, /ALLOW_NONEXISTENT

      if logical_true( file_lines(theta_range_srclist) ) then begin
        ; Common parameters for MERGE calls.
        common_params = {$
          MERGE_OBSERVATIONS: 1B                       ,$

          ; MUST match energy ranges in ae_better_backgrounds.
          EBAND_LO          : [0.5, 0.5, 2.0, 4.0]     ,$
          EBAND_HI          : [7.0, 2.0, 7.0, 7.0]     ,$
          ENERGY_RANGE      : [0.5, 7.0]               ,$
          OVERLAP_LIMIT     : overlap_limit            ,$
          SKIP_APERTURE     : ~par.CHECK_FOR_AFTERGLOWS,$
          SKIP_TIMING       : 1B                       ,$
          GENERIC_RMF_FN    : 'generic.rmf'            }
  
        for ii=0,n_elements(theta_lo)-1 do $                                             
          acis_extract, theta_range_srclist, MERGE_NAME=merge_name[ii], THETA_RANGE=[theta_lo[ii], theta_hi[ii]], /SKIP_SINGLE_OBSID_MERGES, /SKIP_PSF, /SKIP_NEIGHBORHOOD, _EXTRA=common_params
      endif ; theta_range_srclist not empty
      
      ; Collate these merges.  MATCH_EXISTING=0 because an existing merge may be empty.
      for ii=0,n_elements(theta_lo)-1 do $
        acis_extract, 'all.srclist', MERGE_NAME=merge_name[ii], COLLATED_FILENAME=theta_collation_fn[ii], MATCH_EXISTING=0, VERBOSE=0

    endif $ ;(num_obs GT 1)
    else theta_collation_fn = !NULL

    ;------------------------------------------------------------------------
    ; Validate candidates using all collations (single-ObsID and theta-ranges).
    if ~analyze_every_possible_merge then begin
      ; Because we do NOT know the best possible Pb for every source, our Pb table and color coding should not rank
      ; sources with passing Pb values.  We'll define just two slices (validated and not validated).
      boundaries   = ['0', string(par.INVALID_THRESHOLD, F='(%"%g")'), '1']
    endif
    
    ae_analyze_pb, /ANALYZE, [theta_collation_fn, single_ObsID_collation_fn], OVERLAP_LIMIT=overlap_limit, INVALID_THRESHOLD=par.INVALID_THRESHOLD, BOUNDARIES=boundaries, PB_IN_VHARD_BAND=par.PB_IN_VHARD_BAND, GENERIC_RMF_FN='generic.rmf'


    ;------------------------------------------------------------------------
    ; Collate the merge that most strongly validated each source (to tables/most_valid_merge.collated).
    ; Note that the Pb's in this "most valid" collation were calculated in a variety of energy bands---NOT exclusively in full band.
    restore, 'Pbslice.sav'  
    region_tag      = strarr(NUM_SOURCES,2)
    region_tag[*,0] = 'most_valid'
    region_tag[*,1] = validation_report.tag_occasional
    
    acis_extract, 'all.srclist', MERGE_NAME='most_valid', COLLATED_FILENAME='tables/most_valid_merge.collated', REGION_FILE='all.reg', REGION_TAG=region_tag, LABEL_FILENAME='label.txt', VERBOSE=0


    ;------------------------------------------------------------------------
    ; Look for afterglows, only when the most-valid of all possible merges is available.
    par = ae_get_target_parameters() ; Observer may have changed configuration.
    if par.CHECK_FOR_AFTERGLOWS then begin
      ; Identify merges suspected to contain afterglow events; sort by Pb calculated with suspect events removed.
      ae_afterglow_report, 'all.srclist', MERGE_NAME='most_valid', BAND_NUM=0,$
                           INVALID_THRESHOLD=par.INVALID_THRESHOLD, /SORT_BY_PB, VERBOSE=0

      if logical_true( file_lines('agr.srclist') ) then begin
        ; Re-run tool on just the suspect sources to produce a cleaner report.
        ae_afterglow_report, 'agr.srclist', MERGE_NAME='most_valid', BAND_NUM=0,$
                             INVALID_THRESHOLD=par.INVALID_THRESHOLD, suspect_name, Pb_revised, ag_count, ag_fraction

        save, /COMPRESS, suspect_name, Pb_revised, ag_count, ag_fraction, FILE='agr.sav'
      endif
    endif ; CHECK_FOR_AFTERGLOWS

    ;------------------------------------------------------------------------
    ; Look for sources with no position uncertainties.
    bt = mrdfits('tables/most_valid_merge.collated', 1, /SILENT)
    
    missing_errors_fn = 'missing_position_errors.srclist'
    file_delete, missing_errors_fn, /ALLOW_NONEXISTENT
    ind = where(/NULL, ~finite(bt.ERR_RA ) OR (bt.ERR_RA  LE 0) OR $
                       ~finite(bt.ERR_DEC) OR (bt.ERR_DEC LE 0)    )

    if isa(ind, /INTEGER) then begin
      forprint, TEXTOUT=missing_errors_fn, bt.CATALOG_NAME, bt.LABEL, bt.POSNTYPE, F="(%'%s ; (%s) %s')", SUBSET=ind, /NoCOMMENT
      print
      msg1 = string(getenv('TARGET'), n_elements(ind), missing_errors_fn, F='(%"%s: %d sources (in %s) are currently missing position uncertainties.")')
      print, msg1

      if ~logical_true( file_lines('prune.srclist') ) then begin
        ; Nothing to prune.  Ask observer to decide what to do about missing position errors.
        msg2 = 'Do you plan to reposition sources later, or do you want to assign position uncertainties to these sources now (by running "position" merges for them)?'
        print, msg2
        TimedMessage, tm_id, [msg1,msg2], TITLE=getenv('TARGET')+': Missing Position Uncertainties', QUIT_LABEL='I will reposition sources later ...', BUTTON_LABEL='Calculate these position uncertainties now.', BUTTON_ID=button_id, QUIT_ID=quit_id, PRESSED_ID=pressed_id

        if (pressed_id EQ button_id) then begin
          acis_extract, missing_errors_fn, MERGE_NAME='position', /MERGE_OBSERVATIONS, MERGE_FOR_POSITION=par.MERGE_FOR_POSITION, MIN_NUM_CTS=3, OVERLAP_LIMIT=overlap_limit, ENERGY_RANGE=[0.5,7], /SKIP_TIMING, GENERIC_RMF_FN='generic.rmf'

          acis_extract, missing_errors_fn, MERGE_NAME='position', /CHECK_POSITIONS

          acis_extract, missing_errors_fn, MERGE_NAME='position', COLLATED_FILENAME='tables/missing_position_errors.collated'
          
          bt = mrdfits('tables/missing_position_errors.collated',1, /SILENT)

          forprint, bt.CATALOG_NAME, bt.LABEL, bt.ERX_DATA, bt.ERY_DATA, F="(%'%s (%s) ERX_DATA=%0.3F-->ERR_RA  ERY_DATA=%0.3f-->ERR_DEC [arcsec]')"

          ae_poke_source_property, SOURCENAME=bt.CATALOG_NAME, $
                  KEYWORD='ERR_RA' , VALUE=bt.ERX_DATA, COMMENT='[arcsec] 1-sigma uncertainty around (RA,DEC)'

          ae_poke_source_property, SOURCENAME=bt.CATALOG_NAME, $
                  KEYWORD='ERR_DEC', VALUE=bt.ERY_DATA, COMMENT='[arcsec] 1-sigma uncertainty around (RA,DEC)'
        endif ; assign uncertainties
      endif ; nothing to prune
    endif ; missing position errors

    ;------------------------------------------------------------------------
    ; Build Pbslice.reg last, when data products are ready for visual review.
    ae_analyze_pb, /MAKE_REGIONS, DS9_TITLE=getenv('TARGET')+', '+pass_name
    
    
    
    
    ; =========================================================================================
    ;Prune Catalog
    num_pruned = 0L
    ; Spurious "afterglow" sources:
    fn = 'agr_prune.srclist'
    if par.CHECK_FOR_AFTERGLOWS && file_test(fn) && logical_true( file_lines(fn) ) then begin
    
;      title = string( getenv("TARGET"), F='(%"Target %s has AFTERGLOW candidates.")')
;      msg   = string( fn, F='(%"YOU MUST identify and remove false positives from %s.\nClose when it is convenient to begin that review.")')
;      print, msg
;      TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='Close', PRESSED_ID=trash
    
;;; As of 2020 Jan, we find no value in the visual review below.
;;;   acis_extract, COLLATED_FILENAME='tables/most_valid_merge.collated', SRCLIST_FILENAME=fn, /SHOW_REGIONS, /OMIT_BKG_REGIONS

      msg   = string( fn, F='(%"YOU MUST identify false positives in %s, and mark each with a semicolon prefix.  Press CLOSE to prune the un-marked afterlow sources.")')
      print, msg
      TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='Close', PRESSED_ID=trash

      if file_test(fn) && logical_true( file_lines(fn) ) then begin
        readcol, COUNT=num_to_prune, fn, sourcename, FORMAT='A', COMMENT=';'
        if (num_to_prune GT 0) then $
          ae_source_manager, /REMOVE, NAME=sourcename, TRASH_DIR='/tmp/$TARGET/'+pass_name+'/invalid_sources/'
        num_pruned +=num_to_prune
      endif
    endif 

    ; Sources with high Pb or other problems:
    fn = 'prune.srclist'
    if logical_true( file_lines(fn) ) then begin
      par = ae_get_target_parameters() ; Observer may have changed configuration.
      if keyword_set(par.REVIEW_PRUNING) then begin
        title = string( getenv("TARGET"), F='(%"Invalid sources in target %s are ready for your review.")')
        msg   = string( fn,fn, F='(%"After you have reviewed the proposed pruning in %s, and edited %s to reflect your decisions, press CLOSE to prune the catalog.")')
        print, msg
        TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='Close', PRESSED_ID=trash
      endif
    
      readcol, COUNT=num_to_prune, fn, sourcename, FORMAT='A', COMMENT=';'
      if (num_to_prune GT 0) then $
        ae_source_manager, /REMOVE, NAME=sourcename, TRASH_DIR='/tmp/$TARGET/'+pass_name+'/invalid_sources/'
        num_pruned +=num_to_prune
    endif

    ae_source_manager, ORPHAN_SOURCES=os
end ; validation_block1



;############################################################################# 
; This is the block of code that estimates source positions in the section titled
; "REPOSITION SOURCE CANDIDATES" in validate_procedure.txt
PRO reposition_block1, PAUSE_AFTER_MERGES=pause_after_merges

    par = ae_get_target_parameters()

    ; Identify sources that may be piled (so that observer will be guided to review those neighborhoods).
    obsname = strtrim(strsplit(getenv('OBS_LIST'), /EXTRACT), 2)
    if ~keyword_set(obsname) then begin
      print, 'reposition_block1: ERROR: environment variable OBS_LIST must be defined.'
      stop
    endif
    foreach this_obsid, obsname do ae_pileup_screening, this_obsid
    ; Store that list in ./possibly_piled.srclist
    run_command, "cut -f1 -d ')' ../obs*/possibly_piled.srclist | sort | uniq > possibly_piled.srclist", /UNIX


    ; Run position-optimized merges; collate results.
    acis_extract, 'all.srclist', MERGE_NAME='position', /MERGE_OBSERVATIONS, MERGE_FOR_POSITION=par.MERGE_FOR_POSITION, MIN_NUM_CTS=3, OVERLAP_LIMIT=0.10, ENERGY_RANGE=[0.5,7], /SKIP_TIMING, GENERIC_RMF_FN='generic.rmf'

    if keyword_set(pause_after_merges) then begin
      print, 'PAUSING AFTER PHOTOMETRY-OPTIMIZED MERGES (as requested)!'
      stop
    endif

    acis_extract, 'all.srclist', MERGE_NAME='position', COLLATED_FILENAME='tables/position.collated', LABEL_FILENAME='label.txt'

    ; Decide which position estimates are needed for each source
    ae_improve_positions, /CHOOSE_PREFERRED_METHOD
    file_delete, 'merge_position_lock', /ALLOW_NONEXISTENT

    ; ----------
    ; At this point other IDL sessions are computing position estimates.
    ; While that CPU-limited processing is running, let's do a special theta-limited merge and use that to
    ; identify PSF hooks that may be significantly bright.
    theta_lo = 0
    theta_hi = 5
    merge_name  = 'theta_' + string(theta_lo, F='(%"%2.2d")') +'-'+ string(theta_hi, F='(%"%2.2d")')
    theta_collation_fn = 'tables/' + merge_name + '.collated'

    acis_extract, 'all.srclist', MERGE_NAME=merge_name, THETA_RANGE=[theta_lo, theta_hi], /MERGE_OBSERVATIONS,  OVERLAP_LIMIT=0.10, EBAND_LO=0.5, EBAND_HI=7.0, ENERGY_RANGE=[0.5,7.0], /SKIP_PSF, /SKIP_NEIGHBORHOOD, /SKIP_APERTURE, /SKIP_TIMING, GENERIC_RMF_FN='generic.rmf'

    acis_extract, 'all.srclist', MERGE_NAME=merge_name, COLLATED_FILENAME=theta_collation_fn, MATCH_EXISTING='tables/position.collated', VERBOSE=0

    ; We guess that "hook sources" fainter than 4 counts would not survive P_B pruning. 
    ; We guess that hooks are not resolvable beyond 5' off-axis. 
    ae_make_psf_hook_regions, COLLATED_FILENAME=theta_collation_fn, HOOK_CNTS_THRESHOLD=4, THETA_THRESHOLD=theta_hi

    ; Gather PSF hook regions for each ObsID, for use in the SHOW stage.
    obsname = strtrim(strsplit(getenv('OBS_LIST'), /EXTRACT), 2)
    obsdir  = '../obs'+obsname+'/'
    for ii=0, n_elements(obsname)-1 do $
      run_command, string(obsname[ii], obsdir[ii], F='(%"grep %s psf_hook.reg > %s/psf_hook.reg")'), /IGNORE_STATUS

    ; Visual review of possible hook sources is easier with neighborhood image reconstructions.
    ; These are done in the 'theta_00-05' merge, NOT IN THE 'position' merge!
    review_fn = 'near_psf_hook.srclist'
    if logical_true( file_lines(review_fn) ) then begin
      acis_extract, review_fn, MERGE_NAME=merge_name, THETA_RANGE=[theta_lo, theta_hi], /MERGE_OBSERVATIONS,  OVERLAP_LIMIT=0.10, EBAND_LO=0.5, EBAND_HI=7.0, ENERGY_RANGE=[0.5,7.0], /SKIP_TIMING, GENERIC_RMF_FN='generic.rmf'

      acis_extract, review_fn, MERGE_NAME=merge_name, /CHECK_POSITIONS
    endif

    review_fn = 'bright_psf_hook.srclist'
    if logical_true( file_lines(review_fn) ) then begin
      acis_extract, review_fn, MERGE_NAME=merge_name, THETA_RANGE=[theta_lo, theta_hi], /MERGE_OBSERVATIONS,  OVERLAP_LIMIT=0.10, EBAND_LO=0.5, EBAND_HI=7.0, ENERGY_RANGE=[0.5,7.0], /SKIP_TIMING, GENERIC_RMF_FN='generic.rmf'

      acis_extract, review_fn, MERGE_NAME=merge_name, /CHECK_POSITIONS
    endif


    ; ----------
    ; When the position estimates are ready, recommend a new position for each source.
    wait_while_files_exist, 60, 'positions_*_lock'

    acis_extract, 'all.srclist', MERGE_NAME='position'  , COLLATED_FILENAME='tables/position.collated'

    ae_improve_positions, /CHOOSE_POSITION
end ; reposition_block1



;############################################################################# 
; This is a block of code that used to appear explicity in photometry_procedure.txt.
PRO verify_extractions, srclist_fn, DIFFUSE=diffuse,$
                        IGNORE_APERTURE_CORRECTIONS=ignore_aperture_corrections,$
                        IGNORE_ARF_DATES           =ignore_arf_dates

creator_string = "verify_extractions, version " +strmid("$Rev:: 5659  $",7,5) +strmid("$Date: 2022-01-29 11:05:58 -0700 (Sat, 29 Jan 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()
print

    if ~keyword_set(srclist_fn) then srclist_fn = 'all.srclist'

    ; Read source list and find all extraction directories, defined by <source>/<obsid>/obs.stats.
    readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'
    extraction_list = file_dirname(file_search(sourcename, 'obs.stats', COUNT=num_found), /MARK)

    trash = strsplit(extraction_list, '/', COUNT=path_depth)
    extraction_list = extraction_list[where(path_depth EQ 2, num_found)]
    print, num_found, ' extractions found'

    ; Verify that every extraction has a spectrum.
    file_list = extraction_list + 'source.pi'
    found_ind = where( file_test(file_list), num_found, COMPLEMENT=missing_ind, NCOMPLEMENT=num_missing)
    if (num_missing GT 0) then begin
      forprint, TEXTOUT=2, file_list, SUBSET=missing_ind
      print, 'WARNING! The spectrum files listed above are missing!'
      print, 'Type .continue when you are satisfied that this is expected.'
      stop
    endif 
    extraction_list = extraction_list[found_ind]
    print, num_found, ' spectra found'

  if ~keyword_set(diffuse) then begin
    if ~keyword_set(ignore_aperture_corrections) then begin
      ; Verify that every extraction has PSF fractions at 5 energies.
      file_list = extraction_list + 'obs.psffrac'
      aperture_correction_t = bytarr(num_found)
      for ii=0,num_found-1 do begin
        if ~file_test(file_list[ii]) then continue
        header = headfits(file_list[ii], EXT=1)
        aperture_correction_t[ii] = (sxpar(header,'NAXIS2') GE 5) 
      endfor ;ii
      found_ind = where( aperture_correction_t, num_found, COMPLEMENT=missing_ind, NCOMPLEMENT=num_missing)
      if (num_missing GT 0) then begin
        forprint, TEXTOUT=2, file_list, SUBSET=missing_ind
        print, 'WARNING! The aperture correction files listed above are missing or contain less than 5 energies!'
        print, 'Type .continue when you have RESOLVED THIS PROBLEM.'
        stop
      endif 
      extraction_list = extraction_list[found_ind]
      print, num_found, ' aperture corrections found'
    endif ; ~keyword_set(ignore_aperture_corrections)

    ; Verify that every extraction has a background spectrum.
    file_list = extraction_list + 'background.pi'
    found_ind = where( file_test(file_list), num_found, COMPLEMENT=missing_ind, NCOMPLEMENT=num_missing)
    if (num_missing GT 0) then begin
      forprint, TEXTOUT=2, file_list, SUBSET=missing_ind
      print, 'WARNING! The background files listed above are missing!'
      print, 'Type .continue when you have RESOLVED THIS PROBLEM.'
      stop
    endif 
    extraction_list = extraction_list[found_ind]
    print, num_found, ' background spectra found'
  endif ; Point Sources
  

  if ~keyword_set(ignore_arf_dates) then begin
    ; Verify that every ARF is contemporary with its spectrum.
    failure = 0
    foreach extraction, extraction_list do begin
      arf_file_list = file_search(extraction, 'source*.arf')
      arf_file_time = (file_info(arf_file_list                )).MTIME
      
      src_file_time = (file_info(extraction + 'source.pi')).MTIME
      obsolete_ind  = where(arf_file_time LT src_file_time, count)
    
      if (count GT 0) then begin
        failure = 1
        forprint, TEXTOUT=2, arf_file_list, SUBSET=obsolete_ind
      endif
    endforeach
    if failure then begin
      print, 'WARNING! The ARF files listed above are obsolete!'
      print, 'Type .continue when you have RESOLVED THIS PROBLEM.'
      stop
    endif
  endif ; ~keyword_set(ignore_arf_dates) 
    
    ; Verify that every background spectrum is contemporary with its source spectrum.
    bkg_file_list =            extraction_list + 'background.pi'
    bkg_file_time = (file_info(bkg_file_list                )).MTIME
    src_file_time = (file_info(extraction_list + 'source.pi')).MTIME
    obsolete_ind  = where(src_file_time GT bkg_file_time, count)
    if (count GT 0) then begin
      forprint, TEXTOUT=2, bkg_file_list, SUBSET=obsolete_ind
      print, 'WARNING! The background files listed above are obsolete!'
      print, 'Type .continue when you have RESOLVED THIS PROBLEM.'
      stop
    endif

    print, systime(0,min(bkg_file_time,imin)), bkg_file_list[imin], F="(%'The oldest background was made on %s: %s')" 
    print, systime(0,max(bkg_file_time,imax)), bkg_file_list[imax], F="(%'The newest background was made on %s: %s')"

    ; Look for pile-up corrections that use inconsistent source names (telephone number).
    ; This occurs when a source is repositioned after the pile-up correction.
    
    foreach this_extraction, extraction_list do begin
      directory_name = file_dirname(this_extraction)
      obsname        = file_basename(this_extraction)
      epoch          = directory_name+'/EPOCH_'+obsname+'/' 
    
      correction_list = file_search(epoch+'recon_*.pi', COUNT=num_corrections)
      if (num_corrections EQ 0) then continue
      
      if (num_corrections GT 1) then begin
        print, epoch, F='(%"\nERROR: multiple pile-up corrections found in %s")'
        run_command,/UNIX,/QUIET, string(epoch,epoch, F='(%"ls -ltrd %s/recon* %s/*.xcm | cut -c30-")'), result
        forprint, result
        continue
      endif
      
      ; One correction found.  Check for inconsistent naming.
      result = stregex(correction_list[0],"recon_([[:alnum:]\.\+\-]+)",/SUB,/EXT)
      srcname_in_corrected_spectrum = result[1] 

     if (directory_name                EQ '') then message, 'BUG: directory_name is the empty string.'
     if (srcname_in_corrected_spectrum EQ '') then message, 'BUG: srcname_in_corrected_spectrum is the empty string.'
      
      if (directory_name NE srcname_in_corrected_spectrum) then begin
        print, epoch, srcname_in_corrected_spectrum, F='(%"\nWARNING: The directory %s contains pile-up correction products with a different source name (%s).\nYou have probably moved this source after running the pile-up correction.\nWhen the source name is stable, you should repeat the pile-up correction.\n")'
      endif
      
      ; Make ARF/RMF symlinks if needed.
      epoch_arf_file = epoch+srcname_in_corrected_spectrum+'.arf'
      epoch_rmf_file = epoch+srcname_in_corrected_spectrum+'.rmf'

      obsid_arf_file = '../'+obsname+'/source.arf'
      obsid_rmf_file = '../'+obsname+'/source.rmf'
      
      if ~file_test(epoch_arf_file) && file_test(obsid_arf_file) then file_link, obsid_arf_file, epoch_arf_file, /VERBOSE
      if ~file_test(epoch_rmf_file) && file_test(obsid_rmf_file) then file_link, obsid_rmf_file, epoch_rmf_file, /VERBOSE
    endforeach
    
return
end ; verify_extractions




;############################################################################# 
; This block of code calculates the geometric areas of diffuse regions.
; Each region area (arcsec**2) is saved in SRC_AREA keyword in source.stats
; Run from diffuse_procedure.txt
PRO calculate_diffuse_geometric_areas, srclist_fn, scene_fn
    
    ; Read the 2-column sourcelist, which as both source names and paths to defining region files in celestial coordinates.
    readcol, srclist_fn, sourcename, catalog_region_fn, FORMAT='A,A', COMMENT=';'
    
    ; Trim whitespace and remove blank lines.
    sourcename = strtrim(sourcename,2)
    ind = where(sourcename NE '', num_sources)
    
    sourcename = sourcename[ind]
    print, num_sources, F='(%"\n %d sources found in catalog.")'
    if (num_sources EQ 0) then retall
    
    ;; Create a randomly named scratch directory that will not conflict with another instance of AE.
    tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)
    flat_map_fn          = tempdir + 'flat.img'
    temp_region_fn       = tempdir + 'temp.reg'
    temp_image_fn        = tempdir + 'temp.img'
    
    run_command, PARAM_DIR=tempdir
    
    ; Create a map of the scene filled with 1's.
    scene     = readfits(scene_fn, header)
    flat_map  = make_array(/INTEGER, DIM=size(scene, /DIM), VALUE=1)
    empty_map = make_array(/FLOAT  , DIM=size(scene, /DIM), VALUE=!VALUES.F_NAN)

    writefits, flat_map_fn, flat_map, header
    
    ;; Load observation data into ds9.
    print, 'Spawning ds9 to perform coordinate conversions ...'
    ae_send_to_ds9, my_ds9, NAME='geometric_areas_'+session_name
    ae_send_to_ds9, my_ds9, flat_map_fn

    ; Create a data structure to store the list of image indexes belonging to each tesselate.
    pixels_in_tesselate = ptrarr(num_sources, /ALLOC)

    for ii=0,num_sources-1 do begin
      sourcedir = sourcename[ii] + '/' 
      unnamed_src_stats_fn  = sourcedir + 'source.stats'

      ;; Fail if the region file uses CIAO's "field()" syntax, which ds9 cannot parse.
      ae_ds9_to_ciao_regionfile, catalog_region_fn[ii], '/dev/null', FIELD_SYNTAX_FOUND=field_syntax_found
      if field_syntax_found then begin
        print, 'ERROR: the syntax "field()" is not allowed in region files defining diffuse sources.'
        retall
      endif
      
       ;; Load region file into ds9 and resave in PHYSICAL coordinates.
      cmd = strarr(6)
      cmd[0] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions delete all")')
      cmd[1] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions format ds9")')
      cmd[2] = string(my_ds9, catalog_region_fn[ii],   F='(%"xpaset -p ''%s'' regions load %s")')
      cmd[3] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions system physical")')
      cmd[4] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions format ciao")')
      cmd[5] = string(my_ds9, temp_region_fn,          F='(%"xpaset -p ''%s'' regions save %s")')
      run_command, cmd, /QUIET
      
      ; Use CIAO to find the scene pixels that are inside the region.
      cmd = string(flat_map_fn, temp_region_fn, temp_image_fn, F="(%'dmcopy ""%s[sky=region(%s)][opt full]"" %s clobber+')")
      run_command, cmd, /QUIET
      
      mask = readfits(temp_image_fn)

      *pixels_in_tesselate[ii] = where(mask GT 0, num_pixels_in_tesselate)
      
      geometric_area = num_pixels_in_tesselate * (3600*sxpar(header,'CDELT2'))^2
      print, sourcename[ii], geometric_area, F='(%"%30s %12.5g arcsec^2")'
      
      ; Write area to source.stats.
      unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)

      if keyword_set(error) then message, 'ERROR: could not read '+unnamed_src_stats_fn
      
      fxaddpar, unnamed_src_stats, 'SRC_AREA', geometric_area, '[arcsec**2] geometric area of region'
      writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
    endfor
    run_command, string(my_ds9, F='(%"xpaset -p ''%s'' exit")'), /QUIET
   
    ; Save the map template and the data structure holding the list of image indexes belonging to each tesselate.
    save, /COMPRESS, num_sources, sourcename, empty_map, pixels_in_tesselate, header, FILE='pixels_in_tesselate.sav'

    ; Clean up temp files.
    if file_test(tempdir) then begin
      list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
      if (count GT 0) then file_delete, list
      file_delete, tempdir
    endif
    end ; 



;############################################################################# 
; This block of code runs the diffuse image smoothing processing needed in diffuse_procedure.txt

PRO diffuse_smoothing_block1, significance

    if ~keyword_set(significance) then significance = 15  ;Desired SNR for tara_smooth calls.
    
    sig_label = string(significance, F='(%"sig%3.3d")')

    tempdir = temporary_directory( 'diffuse_smoothing_block1.', VERBOSE=1, SESSION_NAME=session_name)

    file_mkdir, ['full_band/sum_of_6_bands/','soft_band/sum_of_4_bands/','hard_band/sum_of_2_bands/']
    restore, 'build_composite.sav'

    ; =================
    ; FULL-BAND KERNELS
    ; Define a set of kernels on Full Band image; smooth narrow bands with those kernels.
    this_band = 'full_band'
    wait_until_files_exist, 30, this_band+'/'+scene_name+['.diffuse.img','.bkg.img','.diffuse.emap']
    tara_smooth, scene_name, significance, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=this_band, RUN_NAME=''


    foreach this_band, narrow_band_names do begin
      wait_until_files_exist, 30, this_band+'/'+scene_name+['.diffuse.img','.bkg.img','.diffuse.emap']
      tara_smooth, scene_name, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=this_band, RUN_NAME='kernels_from_full_'+sig_label, FIXED_RADIUS_MAP_FN='full_band/'+sig_label+'/tophat/fullfield.diffuse_filled.radius'
    endforeach

    ; Sum those narrow-band flux images.
    ; The dmregrid2 tool could do this, but we have seen it mysteriously crop the output image (CIAO 4.11).}
    wide_band_region_fn    = 'full_band/'+sig_label+'/tophat/fullfield.diffuse_filled.reg'
    wide_band_fn           = 'full_band/'+sig_label+'/tophat/fullfield.diffuse_filled.flux'
    sum_of_narrow_bands_fn = 'full_band/sum_of_6_bands/fullfield.diffuse_filled.flux'
    ratio_fn               = tempdir+'full_band.ratio'

    print, 'Summing'
    img=0
    foreach file, narrow_band_names+'/kernels_from_full_'+sig_label+'/tophat/fullfield.diffuse_filled.flux' do begin
      img += readfits(file, header)
      print, '  '+file
    endforeach

    writefits, sum_of_narrow_bands_fn, img, header
    fdecomp  , sum_of_narrow_bands_fn, disk, item_path, item_name, item_qual
    psb_xaddpar, header, 'BUNIT'   , 'sqrt(photon /cm**2 /s /arcsec**2)', 'photon surface brightness'
    writefits, item_path+item_name+'.sqrt.'+item_qual, sqrt(img), header

    ; Calculate ratio between wide-band smoothing and sum of narrow-band smoothing.
    ; dmimgcalc does not propagate NaN values, so we use IDL code.
    ratio_img = readfits(wide_band_fn) / readfits(sum_of_narrow_bands_fn, header)
    psb_xaddpar, header, 'BUNIT'   , 'none'      , 'ratio of photon surface brightness'
    psb_xaddpar, header, 'HDUNAME' , 'flux_ratio', 'wide-band / sum-of-narrow-bands'
    writefits, ratio_fn, ratio_img, header

    run_command, string(wide_band_fn,wide_band_region_fn, sum_of_narrow_bands_fn, ratio_fn, F='(%"ds9 -linear -scale mode 99.5 -title ''Full Band Flux: wide-band, sum of narrow bands, ratio'' %s -region %s %s %s -scale mode minmax -zoom to fit &")')


    ; =================
    ; SOFT-BAND KERNELS
    ; Define a set of kernels on Soft Band image; smooth narrow bands with those kernels.
    this_band = 'soft_band'
    wait_until_files_exist, 30, this_band+'/'+scene_name+['.diffuse.img','.bkg.img','.diffuse.emap']
    tara_smooth, scene_name, significance, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=this_band, RUN_NAME=''

    tara_smooth, scene_name, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=narrow_band_names[0:3], RUN_NAME='kernels_from_soft_'+sig_label, FIXED_RADIUS_MAP_FN='soft_band/'+sig_label+'/tophat/fullfield.diffuse_filled.radius'

    ; Sum those narrow-band flux images.
    ; The dmregrid2 tool could do this, but we have seen it mysteriously crop the output image (CIAO 4.11).}
    wide_band_region_fn    = 'soft_band/'+sig_label+'/tophat/fullfield.diffuse_filled.reg'
    wide_band_fn           = 'soft_band/'+sig_label+'/tophat/fullfield.diffuse_filled.flux'
    sum_of_narrow_bands_fn = 'soft_band/sum_of_4_bands/fullfield.diffuse_filled.flux'
    ratio_fn               = tempdir+'soft_band.ratio'

    print, 'Summing'
    img=0
    foreach file, narrow_band_names[0:3]+'/kernels_from_soft_'+sig_label+'/tophat/fullfield.diffuse_filled.flux' do begin
      img += readfits(file, header)
      print, '  '+file
    endforeach

    writefits, sum_of_narrow_bands_fn, img, header
    fdecomp  , sum_of_narrow_bands_fn, disk, item_path, item_name, item_qual
    psb_xaddpar, header, 'BUNIT'   , 'sqrt(photon /cm**2 /s /arcsec**2)', 'photon surface brightness'
    writefits, item_path+item_name+'.sqrt.'+item_qual, sqrt(img), header

    ; Calculate ratio between wide-band smoothing and sum of narrow-band smoothing.
    ; dmimgcalc does not propagate NaN values, so we use IDL code.
    ratio_img = readfits(wide_band_fn) / readfits(sum_of_narrow_bands_fn, header)
    psb_xaddpar, header, 'BUNIT'   , 'none'      , 'ratio of photon surface brightness'
    psb_xaddpar, header, 'HDUNAME' , 'flux_ratio', 'wide-band / sum-of-narrow-bands'
    writefits, ratio_fn, ratio_img, header

    run_command, string(wide_band_fn,wide_band_region_fn, sum_of_narrow_bands_fn, ratio_fn, F='(%"ds9 -linear -scale mode 99.5 -title ''Soft Band Flux: wide-band, sum of narrow bands, ratio'' %s -region %s %s %s -scale mode minmax -zoom to fit &")')


    ; =================
    ; HARD-BAND KERNELS
    ; Define a set of kernels on Hard Band image; smooth narrow bands with those kernels.
    this_band = 'hard_band'
    wait_until_files_exist, 30, this_band+'/'+scene_name+['.diffuse.img','.bkg.img','.diffuse.emap']
    tara_smooth, scene_name, significance, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=this_band, RUN_NAME=''

    tara_smooth, scene_name, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=narrow_band_names[4:5], RUN_NAME='kernels_from_hard_'+sig_label, FIXED_RADIUS_MAP_FN='hard_band/'+sig_label+'/tophat/fullfield.diffuse_filled.radius'

    ; Sum those narrow-band flux images.
    ; The dmregrid2 tool could do this, but we have seen it mysteriously crop the output image (CIAO 4.11).}
    wide_band_region_fn    = 'hard_band/'+sig_label+'/tophat/fullfield.diffuse_filled.reg'
    wide_band_fn           = 'hard_band/'+sig_label+'/tophat/fullfield.diffuse_filled.flux'
    sum_of_narrow_bands_fn = 'hard_band/sum_of_2_bands/fullfield.diffuse_filled.flux'
    ratio_fn               = tempdir+'hard_band.ratio'

    print, 'Summing'
    img=0
    foreach file, narrow_band_names[4:5]+'/kernels_from_hard_'+sig_label+'/tophat/fullfield.diffuse_filled.flux' do begin
      img += readfits(file, header)
      print, '  '+file
    endforeach

    writefits, sum_of_narrow_bands_fn, img, header
    fdecomp  , sum_of_narrow_bands_fn, disk, item_path, item_name, item_qual
    psb_xaddpar, header, 'BUNIT'   , 'sqrt(photon /cm**2 /s /arcsec**2)', 'photon surface brightness'
    writefits, item_path+item_name+'.sqrt.'+item_qual, sqrt(img), header

    ; Calculate ratio between wide-band smoothing and sum of narrow-band smoothing.
    ; dmimgcalc does not propagate NaN values, so we use IDL code.
    ratio_img = readfits(wide_band_fn) / readfits(sum_of_narrow_bands_fn, header)
    psb_xaddpar, header, 'BUNIT'   , 'none'      , 'ratio of photon surface brightness'
    psb_xaddpar, header, 'HDUNAME' , 'flux_ratio', 'wide-band / sum-of-narrow-bands'
    writefits, ratio_fn, ratio_img, header

    run_command, string(wide_band_fn,wide_band_region_fn, sum_of_narrow_bands_fn, ratio_fn, F='(%"ds9 -linear -scale mode 99.5 -title ''Hard Band Flux: wide-band, sum of narrow bands, ratio'' %s -region %s %s %s -scale mode minmax -zoom to fit &")')


    ; =================
    ; INDEPENDENT KERNELS
    ; Smooth Scale Band to verify scaling of Stowed Data.
    this_band = 'scale_band'
    wait_until_files_exist, 30, this_band+'/'+scene_name+['.diffuse.img','.bkg.img','.diffuse.emap']
    tara_smooth, scene_name, significance, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=this_band, RUN_NAME=''


    ; Smooth T-ReX diffuse bands (if found), using independent kernels.
    suffix = '.diffuse'
    foreach optional_band_name, ['500:700_band','700:1100_band','1100:2300_band'] do begin
      if file_test(optional_band_name+'/'+scene_name+suffix+'.img') then $
        tara_smooth, scene_name, significance, /TOPHAT, /DISCARD_EXISTING, BAND_NAME=optional_band_name, RUN_NAME=''
    endforeach


    ; =================
    ; REPORTS
    ; Show net counts in each image.
    foreach band, image_spec.name do begin
      observed_fn = band+'/'+scene_name+suffix+'.img'
      stowed_fn   = band+'/'+scene_name+'.bkg'+'.img'
      if ~file_test(observed_fn) then continue
      observed_counts = total(/DOUBLE, readfits(observed_fn))
      stowed_counts   = total(/DOUBLE, readfits(stowed_fn  ))
      net_counts      = observed_counts-stowed_counts
      help, observed_counts, stowed_counts, net_counts
      print, band, round(net_counts), round(100*net_counts/observed_counts), F='(%"INFORMATION: %15s has %5d net counts (%d%% of observed).")'
    endforeach

    ; Report any flux image with a large fraction of negative values.
    threshold = 0.01 ; 1%
    foreach file, file_search('.', '*.flux') do begin
      img = readfits(file, /SILENT)
      frac_negative = total(/INT, img LT 0) / float(total(/INT,finite(img)))
      if (frac_negative GT threshold) then $
        print, 100*frac_negative, file, F='(%"WARNING!  %2d%% of %s is NEGATIVE.")'
    endforeach
    exit, /NO_CONFIRM
    end ; diffuse_smoothing_block1



;############################################################################# 
; This block of code builds target-level data products for several procedures
; The ObsIDs included in these mosaics are specified by the wildcard pattern passed in 'L2_pattern'.

PRO build_target_emap_and_eventlist, obsid_directory_pattern 

; Make target-level emaps and event lists using three sets of event data:
;   * all the data
;   * on-axis data
;   * off-axis data
; The tangent plane for these data products is chosen by the build_scene tool.
; 
; A decision about the default field-of-view of each ObsID was made when L1_2_L2_emaps.pro was run earlier.
; Simple spatial filters can be applied to every ObsID here using the SPATIAL_FILTER option.
; On-axis and off-axis mosaics are built below by specifying annular regions around each ObsID's aimpoint (4096,4096).
; 
; Any of these build_scene calls can make images from the target-level event list they produce.  You can omit the IMAGE_FILTER_SPEC and IMAGE_NAME parameters to get images in three default energy bands, or you can specify custom bands and image names using those parameters.


; WAIT until all the event lists and full-band exposure maps are available (because this tool may be called 
; while those data products are being constructed).
obsdir = file_search(obsid_directory_pattern, /TEST_DIRECTORY,/MARK_DIRECTORY, COUNT=count)

if (count EQ 0) then message, 'ERROR: could not find any directories matching pattern '+obsid_directory_pattern

obs_event_fn = obsdir+'AE/validation.AI.evt'
obs_emap_fn  = obsdir+'AE/obs.AI.full.emap'

wait_until_files_exist, 10, obs_emap_fn
wait_until_files_exist, 10, obs_event_fn

msg = 'Iconified ds9 sessions for each ObsID show the single-ObsID exposure maps.'
title = string(getenv("TARGET"), F='(%"TARGET %s: iconified ds9 sessions")')
print, msg
TimedMessage, tm_id1, msg, TITLE=title, QUIT_LABEL='This window will close when you quit IDL.', LIFETIME=3600.*24*5

build_scene, 'fullfield', SUFFIX='.target', $
                 OBS_EVENT_FN=obs_event_fn, EMAP_BASENAME='obs.AI.full.emap', $
                 MERGED_EVENTFILE='fullfield.target.evt', MERGED_COLUMNLIST='sky,energy', MERGED_FILTERSPEC="energy=500:7000", $
                 /CREATE_TEMPLATE, $
                 /CREATE_IMAGE, IMAGE_FILTER_SPEC='', IMAGE_NAME=''

run_command,/UNIX, "ciao;dmcopy 'fullfield.target.evt[energy<2000]' fullfield.target.soft.evt clob+"
run_command,/UNIX, "ciao;dmcopy 'fullfield.target.evt[energy>2000]' fullfield.target.hard.evt clob+"
run_command,/UNIX, "ciao;dmcopy 'fullfield.target.evt[energy>4000]' fullfield.target.vhard.evt clob+"


build_scene, 'theta_0-9', SUFFIX='.target', SPATIAL_FILTER="sky=annulus(4096,4096,0',9')", $
                 OBS_EVENT_FN=obs_event_fn, EMAP_BASENAME='obs.AI.full.emap', $
                 MERGED_EVENTFILE='theta_0-9.target.evt', MERGED_COLUMNLIST='sky,energy', MERGED_FILTERSPEC="energy=500:7000", $
                 TEMPLATE_FN='fullfield_template.img', $
                 /CREATE_IMAGE, IMAGE_FILTER_SPEC='', IMAGE_NAME=''

build_scene, 'theta_8-99', SUFFIX='.target', SPATIAL_FILTER="sky=annulus(4096,4096,8',99')", $
                 OBS_EVENT_FN=obs_event_fn, EMAP_BASENAME='obs.AI.full.emap', $
                 MERGED_EVENTFILE='theta_8-99.target.evt', MERGED_COLUMNLIST='sky,energy', MERGED_FILTERSPEC="energy=500:7000", $
                 TEMPLATE_FN='fullfield_template.img', $
                 /CREATE_IMAGE, IMAGE_FILTER_SPEC='', IMAGE_NAME=''
run_command,/UNIX, 'cat pointing_*/obsid*/AE/ccd_outlines_cel.reg > ccd_outlines_cel.reg'

; Report maximum value in target-level exposure map.
emap_max = max(/NAN,readfits('fullfield.target.emap', /SILENT))

print, emap_max,  F="(%'\nMaximum value of target-level exposure map (@1keV) is %8.2g  s cm**2 count /photon\n')"

;print, emap_max, emap_max/1E6, emap_max/1E4,  F="(%'\nMaximum value of target-level exposure map (@1keV) is\n%10.2g  s cm**2 count /photon\n%10.2g Ms cm**2 count /photon\n%10.2g  s  m**2 count /photon')"

;print, emap_max/446./1000., emap_max/363./1000, emap_max/328./1000, emap_max/211.5/1000, emap_max/103.5/1000.,  F="(%'\nEquivalent ACIS-I on-axis observations:\n%5d ks in Cycle 3\n%5d ks in Cycle 8\n%5d ks in Cycle 12\n%5d ks in Cycle 17\n%5d ks in Cycle 22.')"

end ; build_target_emap_and_eventlist



;############################################################################# 
;;; Block of code from astrometry_recipe.txt 
;;; that builds and reconstructs single-ObsID tile images and searches for sources.
PRO astrometry_block1

    obsname = getenv("OBS") 
    obsdir   = "../obs" + obsname + "/" 
    readcol, "tiles.reg", lines, F="(A)", DELIM="@" 
    result = stregex(lines,"box[^-0-9]*([0-9]+\.[0-9]+)[^-0-9]+(-*[0-9]+\.[0-9]+)",/SUB,/EXT) 
    ra  = double(reform(result[1,*])) 
    dec = double(reform(result[2,*])) 
    ind = where(ra NE 0, num_tiles) 
    tile_label = "tile"+string(indgen(num_tiles), F="(I0)") 
    ae_source_manager, /ADD, RA=ra[ind], DEC=dec[ind], NAME=tile_label, LABEL=tile_label, PROVENANCE="tiles_for_recon", POSITION_TYPE="tile_center" 

    ae_make_catalog, obsname, EVTFILE="validation.evt", SHOW=0, /REGION_ONLY, NEIGHBORHOOD_SIZE=90, NOMINAL_PSF_FRAC=0.90, MINIMUM_PSF_FRAC=0.90
    acis_extract, "all.srclist", /MERGE_OBSERVATIONS, OVERLAP_LIMIT=1E6, /SKIP_SPECTRA, /SKIP_TIMING 
    acis_extract, "all.srclist", /CHECK_POSITIONS, MAXL=[400], ENERGY_RANGE=[0.5,7.0], /SKIP_CORRELATION 
    acis_extract, "all.srclist", COLLATED_FILENAME="/dev/null", REGION_FILE="tile_centers.reg", LABEL_FILENAME="label.txt" 


    ; Source searching is run from  pointing_*/obsid_*/fix_astrometry/
    pushd, '..'

    file_delete, 'all_tiles.srclist', /ALLOW_NONEXISTENT

    run_command,/UNIX, 'sed -e "\|^[^;]|s|^|recon_tiles/|" recon_tiles/all.srclist >> all_tiles.srclist'

    ; Run recon_detect, which accesses a data product named 'totalbackground.full.density'
    ae_recon_detect, "all_tiles.srclist", "validation.evt", SCENE_TEMPLATE_FN="obs.emap", BAND_NAME='full', OUTPUT_DIR='energy_0.5-7', FLOOR_CNTS=8, INVALID_THRESHOLD=1e-6, /HIGHCOUNT_MODE

    merge_band_cats, file_search('energy_*/peaks.fits'), 'validation.evt', /HIGHCOUNT_MODE, OUTPUT_DIR='unpruned_candidates'

    run_command, 'rsync -a --info=NAME unpruned_candidates/proposed.target.fits unpruned_candidates/proposed.target.reg   .'

    cmd = string(obsname, F='(%"ciao -q; ds9 -title \"Bright Sources in ObsID %s\" -iconify validation.evt -region proposed.target.reg -region showtext yes >& /dev/null &")')
    run_command, cmd

    screen_window = 2 + where( strtrim(strsplit(getenv('OBS_LIST'), /EXTRACT), 2) EQ strtrim(obsname,2) )
    title = string(obsname, screen_window, getenv("TARGET"), F='(%"Bright sources in ObsID %s (window %d in %s)")')
    msg = ['Open the iconized ds9 session; REMOVE all uncertain X-ray detections, and those with ambiguous positions.',$
           '','Closely examine groups of crowded detections.',$
           'Remove all detections that do not accurately mark the position of a clear point source.',$
           '','In a field with flat background, isolated single detections are only rarely suspicious.',$
           '','Resave region file (proposed.target.reg), then close.',$
           '','Tips:','  Multiple detections can be selected by a SHIFT + CLICK-DRAG gesture.',$
           '  You may not MOVE or ADD detections!']

    print, title, F='(%"\n%s are ready for review.")'
    forprint, msg
    TimedMessage, tm_id, msg, TITLE=title, QUIT_LABEL='Close', PRESSED_ID=trash
    
    target_cat =               mrdfits('proposed.target.fits', 1, theader, /SILENT)
    catalog_ds9_interface, target_cat, 'proposed.target.reg' ,             /PRUNE_CATALOG
    mwrfits,               target_cat, 'proposed.target.fits',    theader, /CREATE
    print,  F='(%"\n\nDO NOT QUIT this IDL session yet.\n")'
end ; astrometry_block1




;############################################################################# 
PRO ae_recipe
return
end

