;;; $Id: ae_convolve.pro 5170 2017-07-26 19:39:45Z psb6 $

;;; Inspired by convolve.pro in the IDL AstroLib.



;; ADDITIONAL TESTING NEEDED

; Do max_likelihood.pro and max_entropy.pro and filter_image.pro produce "correct" results for various test cases?
; Two simple test cases are tried on max_likelihood.pro in test_convolve (below)?

;; Investiate what data type is used in computations and returned, for various data types of inputs.



;; UPDATED HEADER COMMENTS NEEDED

; /CORRELATE is now available with or without the FFT method.

; We need a short discussion of the fact that the SHIFT that *should* be applied to the output image depends on one's expectations and the conventions assumed.  There is not a *single* definition of the terms "convolution" and "correlation" that everyone agrees on.  The PSF *conventions* assumed here are:
;
; For PSF arrays with ODD dimensions, the default PSF_ORIGIN asserts that the origin or "point source" is at the center of the "central pixel".
;
; For PSF arrays with EVEN dimensions, the default PSF_ORIGIN asserts that the origin or "point source" is at the center of the pixel to the lower-left of the pixel *corner* that defines the center of the image.


;;; =======================================================================
FUNCTION ae_convolve, image, psf, FT_PSF=ft_psf, FT_IMAGE=ft_image, NO_FT=no_ft, $
                   PSF_ORIGIN=psf_origin, $  
                   CORRELATE=correlate, AUTO_CORRELATION=auto_correlation, $
			             NO_PAD=no_pad, VERBOSE=verbose

                 
;; The auto-correlation option can be most easily performed with a recursive call.
if keyword_set(auto_correlation) then $
  return, ae_convolve( image, image, /CORRELATE, FT_IMAGE=ft_image, $
                       NO_FT=no_ft, PSF_ORIGIN=psf_origin, NO_PAD=no_pad, VERBOSE=verbose )


;; Lookup types and dimensions of the input parameters.
   image_spec = size(   image, /STRUCTURE)
ft_image_spec = size(ft_image, /STRUCTURE)
     psf_spec = size(   psf  , /STRUCTURE)
  ft_psf_spec = size(ft_psf  , /STRUCTURE)

; If either image or PSF are DOUBLE, then build both FFTs in DOUBLE.  
use_double_precision = (image_spec.TYPE_NAME EQ 'DOUBLE') || (psf_spec.TYPE_NAME EQ 'DOUBLE')


;; Report common violations of the calling requirements.  

; PSF smaller than data image.
; Improper variable TYPEs ...
; Arrays that are not 2-D ...
; PSF_ORIGIN must be two integers ...
; ...


;; ----------------------------------------------------------------------------
;; We will later need to shift the result of the formal discrete convolution computation so that it corresponds to the caller's convention regarding *where* in the supplied PSF array the corresponding "point source" is located.
;; Since we can only shift the convolution by integer pixels, this "PSF origin" location must be expressed in integer (0-based) pixel coordinates.
;; If the caller did not supply these coordinates via PSF_ORIGIN, then make an assumption here that will please the most callers.

if ~keyword_set(psf_origin) then begin
  ; For PSF arrays with ODD dimensions, the default below asserts that the origin or "point source" is at the center of the "central pixel".
  psf_origin = floor( (psf_spec.DIMENSIONS[0:1] - 1) / 2.0  )
  
  ; For PSF arrays with EVEN dimensions, the default above asserts that the origin or "point source" is at the center of the pixel to the lower-left of the pixel corner that defines the center of the image.
  
  if keyword_set(verbose) then print, psf_origin, F='(%"Using the default PSF_ORIGIN = [%d, %d], in 0-based pixel coordinates.")'
endif


;; ----------------------------------------------------------------------------
;; Determine the dimensions of the image on which an FFT or direct convolution will actually be performed..
;; Unless prohibited by the caller, we will pad the data image to prevent our "circular convolution" from producing wrap-around effects within the footprint of the supplied data image.
padded_dim    = image_spec.DIMENSIONS[0:1]

if ~keyword_set(no_pad) then $
  padded_dim +=   psf_spec.DIMENSIONS[0:1]


;; ----------------------------------------------------------------------------
;; Pad the data image (at the "top" and "right" edges).
;; Compute the FFT of the padded data image if needed.
if ~keyword_set(no_ft) && $
   ( ft_image_spec.N_DIMENSIONS EQ 2) && $
   ((ft_image_spec.TYPE_NAME EQ 'COMPLEX') || (ft_image_spec.TYPE_NAME EQ 'DCOMPLEX')) && $
   ( ft_image_spec.DIMENSIONS[0] EQ padded_dim[0]) && $
   ( ft_image_spec.DIMENSIONS[1] EQ padded_dim[1]) then begin
 
  if keyword_set(verbose) then print, 'Using the cached FT of "image".'
  
endif else begin
  ; For both computation methods (FFT and direct convolution) we want to pad the data image (unless directed not to).
  if keyword_set(no_pad) then begin
    padded_image      = image
  endif else begin
    ; Place the data image at the origin of a larger array padded with zeros to prevent wraparound effects.
    padded_image      = make_array( TYPE=image_spec.TYPE, DIMENSION=padded_dim )
    padded_image[0,0] = image
  
    if keyword_set(verbose) then print, psf_spec.DIMENSIONS[0:1], F='(%"Padded image by %d columsn and %d rows of zeros.")'
  endelse
  
  if ~keyword_set(no_ft) then begin
    if keyword_set(verbose) then print, 'FFT of image ...'    
    ft_image      = FFT( padded_image, DOUBLE=use_double_precision )
    ft_image_spec = size(ft_image, /STRUCTURE)
  endif
endelse


;; ----------------------------------------------------------------------------
;; Compute FFT of the PSF if needed.
if ~keyword_set(no_ft) then begin
  if ( ft_psf_spec.N_DIMENSIONS EQ 2) && $
     ((ft_psf_spec.TYPE_NAME EQ 'COMPLEX') || (ft_psf_spec.TYPE_NAME EQ 'DCOMPLEX')) && $
     ( ft_psf_spec.DIMENSIONS[0] EQ padded_dim[0]) && $
     ( ft_psf_spec.DIMENSIONS[1] EQ padded_dim[1]) then begin
    
      if keyword_set(verbose) then print, 'Using the cached FT of "PSF".' 
      
  endif else begin 
    if ~keyword_set(no_ft) then begin
      if keyword_set(verbose) then print, 'FFT of PSF ...'
      
      ; Place the psf at the origin of a larger array padded with zeros to prevent wraparound effects.
      padded_psf      = make_array( TYPE=psf_spec.TYPE, DIMENSION=padded_dim )
      
      padded_psf[0,0] = psf
    
      ; Compute ft_psf.
      ft_psf      = FFT( padded_psf, DOUBLE=use_double_precision )
      ft_psf_spec = size(ft_psf  , /STRUCTURE)
    endif
  endelse
endif ; ~keyword_set(no_ft)



;; ----------------------------------------------------------------------------
if keyword_set(correlate) then begin
  ;; Perform a correlation
  ;; Shift the result to accomodate the caller's convention regarding the location of the "origin" in the PSF array that was supplied.
  if keyword_set(no_ft) then begin
    if keyword_set(verbose) then print, 'Direct correlation ...'
    
    ; We want to supply options to IDL's "convol" routine so that it will emulate what the FFT method is doing.
    ; Rotate the PSF by 180 degrees to get correlation out of the convol() function.
    result = convol( padded_image, rotate(psf, 2), CENTER=0, /EDGE_WRAP ) 

    ; The shift below requires an extra "-1" to produce a result that agrees with the FFT method.
    ; The REASON we need this "-1" is a mystery TO ME!
    result = shift(result, -psf_origin[0] - 1, -psf_origin[1] - 1)
    
  endif else begin
    if keyword_set(verbose) then begin
      print, 'Inverse FFT ...'
      help, ft_image, ft_psf
    endif
    
    ; Rotate the PSF by 180 degrees (== take conjugate of PSF transform) to get correlation out of FFT multiplication.
    result = ft_image_spec.N_ELEMENTS * real_part(FFT(/INVERSE, ft_image * conj(ft_psf) )) 

    ; The sign of the required shift here is opposite from the convolution cases (below), presumably because of the PSF rotation.
    result = shift(result,  psf_origin[0],  psf_origin[1])
  endelse 
      
;; ----------------------------------------------------------------------------
endif else begin
  ;; Perform a convolution 
  ;; Shift the result to accomodate the caller's convention regarding the location of the "origin" in the PSF array that was supplied.
  if keyword_set(no_ft) then begin
    if keyword_set(verbose) then print, 'Direct convolution ...'
    
    ; We want to supply options to IDL's "convol" routine so that it will emulate what the FFT method is doing.
    result = convol( padded_image,         psf   , CENTER=0, /EDGE_WRAP )
    
    result = shift(result, -psf_origin[0], -psf_origin[1])
    
  endif else begin
    if keyword_set(verbose) then begin
      print, 'Inverse FFT ...'
      help, ft_image, ft_psf
    endif

    result = ft_image_spec.N_ELEMENTS * real_part(FFT(/INVERSE, ft_image *      ft_psf  ))
    
    result = shift(result, -psf_origin[0], -psf_origin[1])
  endelse     
    
endelse ; convolution


;; Trim the result to the dimensions of the supplied data image.
if keyword_set(no_pad) then return, result $
                       else return, result[0:image_spec.DIMENSIONS[0]-1, 0:image_spec.DIMENSIONS[1]-1 ]
                     
end ; convolve
;;; =======================================================================





;;; =======================================================================
PRO test_convolve

     fft_result_fn = '/tmp/fft.fits'
  convol_result_fn = '/tmp/convol.fits'

  ; Build a scene with point sources (delta functions) at pixels [0,0] and [50,50].
  data = fltarr(200,200)
  data[0,0] = 1
  data[50,50] = 1
  
  ; Build an asymmetric PSF shaped like an "L" with the legs meeting on the "central pixel" of an array with ODD dimensions.
  psf = fltarr(41,41)
  psf[20   , 20:*] = 1
  psf[20:25,20   ] = 1
  erase & tvscl, psf     
  
  
  print, F='(%"\nCONVOLUTION (PSF with ODD dimensions)")'
  f_result = ae_convolve(/VERBOSE, data, psf )
  c_result = ae_convolve(/VERBOSE, data, psf, /NO_FT )
  writefits,    fft_result_fn, f_result
  writefits, convol_result_fn, c_result
  
  ; The origins of the PSF's in the convolution should be at the coordinates of the delta functions in the scene:
  ; [0,0] and [100,100] in 0-based indexing, [1,1] and [101,101] in ds9's 1-based indexing.
  spawn, string(fft_result_fn, convol_result_fn, F='(%"ds9 %s %s")')            

  
  print, F='(%"\nAUTOCORRELATION (ODD dimensions)")'
  f_result = ae_convolve(/VERBOSE, psf, 0, /AUTO_CORRELATION )
  c_result = ae_convolve(/VERBOSE, psf, 0, /AUTO_CORRELATION, /NO_FT )
  writefits,    fft_result_fn, f_result
  writefits, convol_result_fn, c_result
     
  ; The peak should be at the "central pixel": [20,20] in 0-based indexing, [21,21] in ds9.
  spawn, string(fft_result_fn, convol_result_fn, F='(%"ds9 %s %s")')            


  
  ; Build an asymmetric PSF shaped like an "L" stored in an array with EVEN dimensions.
  ; The legs of the "L" meet at a specified location (psf_origin) within that array 
  psf = fltarr(60,60)
  psf_origin=[25,15]
  psf[25   , 15:15+40] = 1
  psf[25:30, 15      ] = 1 
  erase & tvscl, psf     
  
  print, F='(%"\nCONVOLUTION (PSF with EVEN dimensions)")'
  f_result = ae_convolve(/VERBOSE, data, psf, PSF_ORIGIN=psf_origin )
  c_result = ae_convolve(/VERBOSE, data, psf, PSF_ORIGIN=psf_origin, /NO_FT )
  writefits,    fft_result_fn, f_result
  writefits, convol_result_fn, c_result
  
  ; The origins of the PSF's in the convolution should be at the coordinates of the delta functions in the scene:
  ; [0,0] and [50,50] in 0-based indexing, [1,1] and [51,51] in ds9's 1-based indexing.
  spawn, string(fft_result_fn, convol_result_fn, F='(%"ds9 %s %s")')            
  

  print, F='(%"\nAUTOCORRELATION (EVEN dimensions)")'
  f_result = ae_convolve(/VERBOSE, psf, 0, /AUTO_CORRELATION )
  c_result = ae_convolve(/VERBOSE, psf, 0, /AUTO_CORRELATION, /NO_FT )
  writefits,    fft_result_fn, f_result
  writefits, convol_result_fn, c_result
     
  ; The peak is at the pixel below and to the left of the pixel corner at the center of the image: [29,29] in 0-based indexing, [30,30] in ds9.
  ; Whether this is "correct" depends on your point of view, I think.
  spawn, string(fft_result_fn, convol_result_fn, F='(%"ds9 %s %s")')            

  
  ; COMPATIBILITY WITH max_likelihood.pro
  ; Perhaps the most important test is whether convolve.pro follows the conventions assumed by max_likelihood.pro!
  
  ; Generate a fake "data image" consisting of a delta function.
  test_image = dblarr(10,10)
  test_image[3,3] = 1

  for jj=0,1 do begin
    case jj of 
      0: dim=5
      1: dim=6
    endcase
    
    ; Generate a PSF (with odd dimensions in the first pass, even dimensions in the second pass) consisting of a delta function, located where convolve.pro expects it to be..
    psf        = fltarr(dim,dim)
    psf_spec   = size(   psf  , /STRUCTURE)
    psf_origin = floor( (psf_spec.DIMENSIONS[0:1] - 1) / 2.0  )
    
    psf[psf_origin[0], psf_origin[1]] = 1
  
    ; Reconstruct the test image.  (Only one iteration is required for this simple case.)
    psf_ft     = 0
    maxlik_img = 0
    Max_Likelihood, test_image, psf, maxlik_img, FT_PSF=psf_ft
    
    ; Verify that the peak appears in the same pixel as the input signal, and that the peak value is near 1.0.
    print, max(maxlik_img), maxlik_img[3,3]
  endfor
  
return
end ; test_convolve
;;; =======================================================================

