;;; $Id: acis_extract_tools.pro 5661 2022-02-13 14:08:42Z psb6 $
;;; Accessory tools for ACIS Extract. 
;;; Patrick Broos, Penn State University, 2004


;;; ==========================================================================
;;; This program will "poke" a FITS keyword into the source.stats file (obsname omitted)
;;; or obs.stats file (obsname supplied) for each source supplied via the string array SOURCENAME
;;; or read from the file SRCLIST_FILENAME.

;;; KEYWORD is a scalar string; VALUE and COMMENT are either scalar string or 
;;; string vectors whos length matches the catalog.

PRO ae_poke_source_property, obsname, SRCLIST_FILENAME=srclist_fn, SOURCENAME=sourcename, $
                  KEYWORD=keyword, VALUE=value_p, COMMENT=comment_p, $
                  EXTRACTION_NAME=extraction_name, MERGE_NAME=merge_name

src_stats_basename       = 'source.stats'
obs_stats_basename       = 'obs.stats'

if keyword_set(srclist_fn) then readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'


; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)


if (num_sources EQ 0) then begin
  print, 'ERROR: no entries read from source list ', srclist_fn
  retall
endif

sourcename = sourcename[ind]
print, num_sources, F='(%"\n%d sources to modify ...\n")'


if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)

if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                else merge_subdir = ''
if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)


case n_elements(value_p) of
  1: value = replicate(value_p,num_sources)
  num_sources: value = value_p
  else: message, 'Length of VALUE parameter must be either 1 or '+string(num_sources)
endcase

case n_elements(comment_p) of
  1: comment = replicate(comment_p,num_sources)
  num_sources: comment = comment_p
  else: message, 'Length of COMMENT parameter must be either 1 or '+string(num_sources)
endcase

source_not_observed = bytarr(num_sources)

for ii = 0L, num_sources-1 do begin
  basedir   = sourcename[ii] + '/' 

  if keyword_set(obsname) then begin
    if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
      print, 'ERROR: parameter "obsname" must be a scalar string'
      retall
    endif
    
    obsdir    = basedir + obsname + '/' + extraction_subdir[ii]
    stats_fn  = obsdir    + obs_stats_basename
  endif else begin
    sourcedir = basedir + merge_subdir[ii]
    stats_fn  = sourcedir + src_stats_basename
  endelse
  
  ; We assume that an existing source directory that is a symbolic link should not be written to.
  temp = file_info(basedir)
  is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
  if ~is_writable then begin
    print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
    continue
  endif 

  stats = headfits(stats_fn, ERRMSG=error)
  
  if keyword_set(error) then begin
    source_not_observed[ii] = 1
    continue
  endif
    
  psb_xaddpar, stats, keyword, value[ii], comment[ii]
  writefits, stats_fn, 0, stats
endfor ;ii

ind = where(source_not_observed, count)
if (count GT 0) then begin
  print, src_stats_basename, count, F="(%'WARNING! ae_poke_source_property could not read %s for the following %d sources:')"
  forprint, SUBSET=ind, sourcename, F="(%'  %s ')"
endif
return
end ; ae_poke_source_property


;;;; ==========================================================================
;;;; This tool fits two diffuse sources simultaneously, with one serving as an 
;;;; observation of the astrophysical (sky) background.
;;;; Each source is assume to have its own background spectrum representing the
;;;; local instrumental background (see diffuse recipe).
;;;;
;;;;
;;;; The sky spectrum is modeled with cplinear, for want of an astrophysical model.
;;;;
;;;; The calibration (ARFs) for the diffuse sources is assumed to have been put 
;;;; onto a "per arcsec^2" basis by AE.
;;;;
;;;; Only chi^2 fitting is supported.
;
;PRO ae_fit_diffuse, object_name, sky_name, CHANNEL_RANGE=channel_range, $
;                  MODEL_FILENAME=model_filename, MODEL_CHANGES_FILENAME=model_changes_filename, $
;                  SNR_RANGE=snr_range, NUM_GROUPS_RANGE=num_groups_range, $
;                  INTERACTIVE=interactive
;                  
;creator_string = "ae_fit_diffuse, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
;print, creator_string, F='(%"\n\n%s")'
;print, now()
;
;; Create a unique scratch directory.
;tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)
;
;
;result = routine_info( 'acis_extract', /SOURCE )
;fdecomp, result.PATH, disk, codedir
;
;if NOT keyword_set(model_filename) then begin
;  print, 'ERROR:Parameter MODEL_FILENAME not supplied!'
;  GOTO, FAILURE
;endif
;
;; SNR_RANGE[1] is the user's goal for defining groups; SNR_RANGE[0] is the lower limit allowed before we abort the grouping attempt
;if (n_elements(snr_range) EQ 0) then $
;  snr_range = [1,3]
;if (n_elements(snr_range) NE 2) then begin
;  print, 'ERROR: keyword SNR_RANGE should be a 2-element vector giving the range of SNR allowed for each spectral group, e.g. [2.5,5].'
;  GOTO, FAILURE      
;endif
;
;if (snr_range[1] LT 0) then begin
;  print, 'ERROR: minimum SNR value (SNR_RANGE[1]) must be positive'
;  GOTO, FAILURE
;endif
;
;if (n_elements(num_groups_range) EQ 0) then $
;  num_groups_range = [2+8,250]
;if (n_elements(num_groups_range) NE 2) then begin
;  print, 'ERROR: keyword NUM_GROUPS_RANGE should be a 2-element vector specifying how many spectral groups are desired, e.g. [2+8,250].'
;  GOTO, FAILURE      
;endif
;           
;  
;run_command, PARAM_DIR=tempdir
;
;
;;; Create directory for this pair of sources.
;sourcename = object_name+'_AND_'+sky_name
;sourcedir = sourcename+'/'
;file_mkdir, sourcedir
;
;;; ------------------------------------------------------------------------
;;; Create symlinks to needed object files; group object spectrum
;suffix = ['.pi', '_bkg.pi', '.arf', '.rmf']
;fn = object_name+suffix
;file_delete, sourcedir+fn, /ALLOW_NONEXISTENT
;file_link, '../'+object_name+'/'+fn, sourcedir
;
;obj_src_spectrum_fn     = sourcedir+fn[0]
;obj_bkg_spectrum_fn     = sourcedir+fn[1]
;obj_grouped_spectrum_fn = ''
;
;ae_group_spectrum, obj_src_spectrum_fn, obj_bkg_spectrum_fn, obj_grouped_spectrum_fn, $
;                   CHANNEL_RANGE=channel_range, $
;                   SNR_RANGE=snr_range, NUM_GROUPS_RANGE=num_groups_range, $
;                   CREATOR_STRING=creator_string, $
;                   this_snr_goal, grp_name, channel_starting_group, num_groups, inband_counts
;
;obj_ignore_spec = string(num_groups, F='(%"1,%d")')
;
;;; ------------------------------------------------------------------------
;;; Create symlinks to needed object files; group object spectrum
;fn = sky_name+suffix
;file_delete, sourcedir+fn, /ALLOW_NONEXISTENT
;file_link, '../'+sky_name+'/'+fn, sourcedir
;
;sky_src_spectrum_fn     = sourcedir+fn[0]
;sky_bkg_spectrum_fn     = sourcedir+fn[1]
;sky_grouped_spectrum_fn = ''
;
;ae_group_spectrum, sky_src_spectrum_fn, sky_bkg_spectrum_fn, sky_grouped_spectrum_fn, $
;                   CHANNEL_RANGE=channel_range, $
;                   SNR_RANGE=snr_range, NUM_GROUPS_RANGE=num_groups_range, $
;                   CREATOR_STRING=creator_string, $
;                   this_snr_goal, junk, channel_starting_group, num_groups
;
;  
;sky_ignore_spec = string(num_groups, F='(%"1,%d")')
;
;
;;; ------------------------------------------------------------------------
;;; Choose energy bins for cplinear model of sky spectrum.
;;  We ask AE's grouping algorithm to create exactly 11 groups in order to get the 10 vertices that cplinear needs.
;ae_group_spectrum, sky_src_spectrum_fn, sky_bkg_spectrum_fn, '/dev/null', $
;                       CHANNEL_RANGE=channel_range, $
;                       SNR_RANGE=[0,this_snr_goal], NUM_GROUPS_RANGE=[11,11], $
;                       CREATOR_STRING=creator_string, $
;                       this_snr_goal, junk, channel_starting_group, num_groups 
;
;if (num_groups NE 11) then begin
;  print, 'ERROR: grouping algorithm was not able to choose channels for the cplinear sky model.'
;  retall
;endif
;
;; Use RMF & ARF to figure out the energy for each spectral channel.
;ae_channel_energy_and_arf, sourcedir+fn[3], sourcedir+fn[2], $
;  channel_number, channel_lowenergy, channel_highenergy, channel_midenergy
;
;energy_starting_group  = interpol(channel_midenergy, channel_number, channel_starting_group)
;
;
;;; ------------------------------------------------------------------------
;;; Build a name for the model using the basename of MODEL_FILENAME and if supplied
;;; appending the basename of MODEL_CHANGES_FILENAME.
;modelsubdir              = 'spectral_models/'
;
;fdecomp, model_filename, disk, dir, base_model_name
;
;model_name = base_model_name
;
;if keyword_set(model_changes_filename) then begin
;  fdecomp, model_changes_filename, disk, dir, model_changes_name, model_changes_qual
;
;  ; Look for a local file in the source directory to override the specified MODEL_CHANGES_FILENAME.
;  custom_model_name = strjoin([model_name,model_changes_name], '_')
;  fit_custom_fn     = sourcedir + modelsubdir + custom_model_name + '.' + model_changes_qual[0]
;  
;  if NOT file_test(fit_custom_fn) then begin
;    ; No local override found, so prepare to use the specified files.
;    if (n_elements(model_changes_filename) EQ 1) then begin
;      fit_custom_fn = model_changes_filename 
;    endif else begin
;      ; Concatenate all the MODEL_CHANGES files specified.
;      fit_custom_fn = tempdir + 'model_changes.xcm'
;      cmd = string(strjoin(model_changes_filename, ' '), fit_custom_fn, F='(%"cat %s >! %s")')
;      run_command, /UNIX, cmd, /QUIET
;    endelse
;  endif
;  
;  if file_test(fit_custom_fn) then begin
;    model_name = custom_model_name
;    print, 'CUSTOMIZATIONS to xcm script:'
;    run_command, /UNIX, 'cat '+fit_custom_fn
;    print
;  endif else fit_custom_fn = ''
;endif else fit_custom_fn = ''
;
;  
;;; ------------------------------------------------------------------------
;;; Build the fitting script.
;file_mkdir, sourcedir + modelsubdir
;fit_result_root         = grp_name + '_' + model_name
;fit_xcm_fn             = modelsubdir +fit_result_root + '.xcm'
;
;openw,  xcm_unit, sourcedir + fit_xcm_fn, /GET_LUN
;printf, xcm_unit, file_basename(obj_grouped_spectrum_fn), F='(%"set obj_spectrum_filename       \"%s\"")'
;printf, xcm_unit, file_basename(sky_grouped_spectrum_fn), F='(%"set sky_spectrum_filename       \"%s\"")'
;printf, xcm_unit, obj_ignore_spec,                    F='(%"set obj_ignore_spec             \"%s\"")'
;printf, xcm_unit, sky_ignore_spec,                    F='(%"set sky_ignore_spec             \"%s\"")'
;printf, xcm_unit, fit_result_root,                    F='(%"set model_name              \"%s\"")'
;printf, xcm_unit, 0,                                  F='(%"set c_stat_flag               %d")'
;printf, xcm_unit, inband_counts,                      F='(%"set src_cnts                  %d")'
;printf, xcm_unit, strjoin(string(energy_starting_group[1:10],F='(%"%4.2f")'),' '), $
;                                                      F='(%"set cplinear_energies        {%s}")'
;printf, xcm_unit, codedir+'xspec_scripts',            F='(%"set model_directory         \"%s\"")'
;printf, xcm_unit, keyword_set(interactive),           F='(%"set interactive_flag          %d")'
;free_lun, xcm_unit
;
;; Append MODEL_FILENAME to XSPEC script prefix using "sed" to insert any user-supplied customizations to the model.
;if (fit_custom_fn EQ '') then begin
;  cmd = string(model_filename, sourcedir + fit_xcm_fn, F="(%'cat %s >>! %s')")
;endif else begin
;  cmd = string(fit_custom_fn, model_filename, sourcedir + fit_xcm_fn, F="(%'sed -e ""/AE CUSTOMIZATIONS/r %s"" %s >>! %s')")
;endelse
;run_command, /UNIX, cmd
;
;;; ------------------------------------------------------------------------
;;; Perform the fit.
;ae_perform_fit, sourcedir, fit_result_root, INTERACTIVE=keyword_set(interactive)
;
;FAILURE: 
;return
;end



;;; ==========================================================================
;;; Routine that decides when a parameter confidence interval reported by the AE fitting scripts should be used and when it should be ignored.
;;; 
;;; The param_value, param_min, param_max inputs are the best-fit parameter value and the soft limits given to XSPEC.

;;; The lower_confidence_limit and upper_confidence_limit inputs are the confidence interval returned by XSPEC.
;;; Supply the scalar 0 when not available.
;;; Values in these vectors will be set to NaN when this routine determines that they should be ignored.

;;; The errstatus input is the set of status flags returned by XSPEC's "\nERROR" command.
;;; Supply the scaler empty string ('') when not available.

;;; The structure returned by this function contains all the inputs, plus an xspec_anom_flags string vector that is a set of flags indicating various anomalies that were found:

;;;   f: the parameter was frozen
;;;   r: the parameter value falls outside the soft limits declared in the fitting script

;;;   s: error estimation was skipped (either by script or by XSPEC because reduced chi-squared was too high

;;;   n: non-monotonic warning from XSPEC
;;;   l: search for lower limit failed, or ran into hard limit
;;;   u: search for upper limit failed, or ran into hard limit
;;;   m: "minimization may have run into problem"

;;;   L: the lower confidence limit falls outside the soft limits declared in the fitting script
;;;   U: the upper confidence limit falls outside the soft limits declared in the fitting script
;;;   o: the parameter value falls outside its reprorted confidence interval

;;; The structure also contains a scalar named "bisector" that can be used to classify the confidence intervals into three categories:
;;;   1. Intervals that are smaller than bisector.
;;;   2. Intervals that are larger than bisector.
;;;   3. Intervals that are consistent with bisector, i.e. intervals that contain bisector.
;;; The bisector value is chosen to produce nearly equal numbers of sources in categories 1 and 2.


FUNCTION validate_xspec_confidence_interval, parameter_name, label, param_min, param_max, $
      param_value, lower_confidence_limit, upper_confidence_limit, errstatus, param_was_frozen, $
      HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=significant_digits, PLOT=plot, VERBOSE=verbose

if ~keyword_set(significant_digits) then significant_digits = 2

if ~keyword_set(lower_confidence_limit) then lower_confidence_limit = replicate(!VALUES.F_NAN, n_elements(param_value))
if ~keyword_set(upper_confidence_limit) then upper_confidence_limit = replicate(!VALUES.F_NAN, n_elements(param_value))
if ~keyword_set(errstatus             ) then errstatus              = replicate(''           , n_elements(param_value))

; Inputs that are supposed to be numbers but are of type STRING must be converted to numbers.
if (size(/TNAME, param_min)              EQ 'STRING') then param_min              = float(param_min)
if (size(/TNAME, param_max)              EQ 'STRING') then param_max              = float(param_max)
if (size(/TNAME, param_value)            EQ 'STRING') then param_value            = float(param_value)
if (size(/TNAME, lower_confidence_limit) EQ 'STRING') then lower_confidence_limit = float(lower_confidence_limit)
if (size(/TNAME, upper_confidence_limit) EQ 'STRING') then upper_confidence_limit = float(upper_confidence_limit)



; Defintions of flags returned by "tclout error" command in XSPEC, numbered left-to-right:
;  1      new minimum found
;  2      non-monotonicity detected
;  3      minimization may have run into problem
;  4      hit hard lower limit
;  5      hit hard upper limit
;  6      parameter was frozen
;  7      search failed in -ve direction
;  8      search failed in +ve direction
;  9      reduced chi-squared too high
;
; The meaning of these flags is not discussed in the manual.  
; Reading the code that generates them (/usr/astro/heasoft-6.9/Xspec/src/XSFit/Fit/FitErrorCalc.cxx) may be helpful to understand their implications.


seq = indgen(n_elements(param_value))

null_flag = replicate('.',n_elements(errstatus))
bad_ordering_flag  = null_flag
range_flag         = null_flag
skipped_flag       = null_flag
frozen_flag        = null_flag
lowerlimit_flag    = null_flag
upperlimit_flag    = null_flag
non_monotonic      = null_flag
lower_search       = null_flag
upper_search       = null_flag
bad_minimize       = null_flag

; We declare below the flag values that are valid reasons to ignore (set to NaN) the lower and upper confidence limits.
;
; We don't know how to interpret the "minimization may have run into problem" flag from XSPEC, and choose to ignore it.
;
; We choose to ignore the vague ("non-monotonicity detected" ('n') flags from XSPEC, since the error command seems to continue its search after warnings about non-monotonicity.  2009 April 21
;
; The "search failed" flags sound ominous, but confidence interval values are still returned by XSPEC. 
; Often, the log of XSPEC messages will provide additional information about the failure, which might be helpful in deciding whether the confidence interval returned is meaningful.

bad_lerror_flaglist = '[osfLlc]'
bad_uerror_flaglist = '[osfUuc]'

; List of flag values (from validate_xspec_confidence_interval) that should invalidate the fit itself.
;bad_fit_flaglist    = '[ro]'
bad_fit_flaglist    = '[o]'



; If the "\nERROR" command was run on a frozen parameter then a flag in errstatus will indicate that.
; If the "\nERROR" command was skipped, then we can still infer that the parameter was frozen by examining the "frozen flag" saved by the fitting script.
ind = where(strmatch(errstatus, '?????T*') OR param_was_frozen, count)
if (count GT 0) then begin
  frozen_flag     [ind] = 'f'
  param_was_frozen[ind] = 1
endif

; Report when the best-fit value of a free parameter violated its soft limits.
ind = where(~param_was_frozen AND ((param_value LT param_min) OR (param_value GT param_max)), count)
if (count GT 0) then begin
  range_flag[ind] = 'r'
  print, count, parameter_name, F='(%"\nThese %d sources have %s out of range:")'
  forprint, seq, label, param_value, SUBSET=ind, F='(%"%5d %s %f")', TEXTOUT=2
endif

; AE writes the string "skipped" when the error command is not attempted.
ind = where(strmatch(errstatus, '*skipped*'), count)
if (count GT 0) then skipped_flag[ind] = 's'

; XSPEC can skip an error estimation if reduced chi^2 is too large.
ind = where(strmatch(errstatus, '????????T*'), count)
if (count GT 0) then skipped_flag[ind] = 's'

ind = where(strmatch(errstatus, '?T*'), count)
if (count GT 0) then non_monotonic[ind] = 'n'


; Discard confidence limits that we have so far deemed to be unreliable, and THEN do some tests that involve those limits.
anom_flags = frozen_flag+range_flag+skipped_flag+non_monotonic

ind = where( stregex(/BOOL, anom_flags, bad_uerror_flaglist), count )
if (count GT 0) then upper_confidence_limit[ind] = !VALUES.F_NAN

ind = where( stregex(/BOOL, anom_flags, bad_lerror_flaglist), count )
if (count GT 0) then lower_confidence_limit[ind] = !VALUES.F_NAN


; Confidence limits of zero are taken to be failures of the error estimation procedure.
ind = where(strmatch(errstatus, '???T*' ) OR strmatch(errstatus, '??????T*' ) OR (lower_confidence_limit EQ 0), count)
if (count GT 0) then lower_search[ind] = 'l'

ind = where(strmatch(errstatus, '????T*') OR strmatch(errstatus, '???????T*') OR (upper_confidence_limit EQ 0), count)
if (count GT 0) then upper_search[ind] = 'u'

ind = where(strmatch(errstatus, '??T*'), count)
if (count GT 0) then bad_minimize[ind] = 'm'


; Discard confidence limits that we have so far deemed to be unreliable, and THEN do some tests that involve those limits.
anom_flags += lower_search+upper_search+bad_minimize

ind = where( stregex(/BOOL, anom_flags, bad_uerror_flaglist), count )
if (count GT 0) then upper_confidence_limit[ind] = !VALUES.F_NAN

ind = where( stregex(/BOOL, anom_flags, bad_lerror_flaglist), count )
if (count GT 0) then lower_confidence_limit[ind] = !VALUES.F_NAN


; Check for confidence intervals that violate the parameter ranges.
ind = where( lower_confidence_limit LT param_min, count)
if (count GT 0) then lowerlimit_flag[ind] = 'L'

ind = where( upper_confidence_limit GT param_max, count)
if (count GT 0) then upperlimit_flag[ind] = 'U'


; Discard confidence limits that we have so far deemed to be unreliable, and THEN do some tests that involve those limits.
anom_flags += lowerlimit_flag+upperlimit_flag

ind = where( stregex(/BOOL, anom_flags, bad_uerror_flaglist), count )
if (count GT 0) then upper_confidence_limit[ind] = !VALUES.F_NAN

ind = where( stregex(/BOOL, anom_flags, bad_lerror_flaglist), count )
if (count GT 0) then lower_confidence_limit[ind] = !VALUES.F_NAN


; Look for a best-fit parameter value that lies outside its confidence interval.
ind = where((upper_confidence_limit LT param_value) OR (lower_confidence_limit GT param_value), count)
if (count GT 0) then bad_ordering_flag[ind] = 'o'


; Perform one last cleaning of the confidence limits that we have deemed to be unreliable.
anom_flags += bad_ordering_flag

ind = where( stregex(/BOOL, anom_flags, bad_uerror_flaglist), count )
if (count GT 0) then upper_confidence_limit[ind] = !VALUES.F_NAN

ind = where( stregex(/BOOL, anom_flags, bad_lerror_flaglist), count )
if (count GT 0) then lower_confidence_limit[ind] = !VALUES.F_NAN



; Hide errors when requested by caller.
if keyword_set(hide_errors_flag) then begin
  ind = where(hide_errors_flag, count)
  if (count GT 0) then begin
   anom_flags[ind] = 'hidden  '
   upper_confidence_limit[ind] = !VALUES.F_NAN 
   lower_confidence_limit[ind] = !VALUES.F_NAN 
  endif 
endif

; Mark the frozen cases.
ind = where( param_was_frozen, count )
if (count GT 0) then anom_flags[ind] = 'frozen    '
  
  
; Make some plots.
if keyword_set(plot) then begin
; To make B&W single-curve histograms that annotate diffuse parameter maps, uncomment the following line
; and comment out the other plot calls.

; dataset_1d,  id2, param_value           , DATASET='best-fit'   , color='white'  , BINSIZE=0.1, DENSITY_TITLE=' ', XTIT='', WIDGET_TITLE=parameter_name, PLOT_WINDOW_OPTIONS="SET_XMARGIN=[4,1], SET_YMARGIN=[2,1], SHOW_DATE=0, CHANGE_DRAW_SIZE=[20,90]"
  
  dataset_1d,  id2, param_value           , DATASET='best-fit'   , LINE=0, color='red'  , BINSIZE=0.1, $
               XTIT=parameter_name, WIDGET_TITLE=parameter_name, $
               PLOT_WINDOW_OPTIONS=string(median(param_value), F='(%"SET_BIG_MARKER=[%0.2f,0], SHOW_DATE=0")'), PS_CONFIG={filename:parameter_name+'_hist.ps'}, /PRINT
    
  dataset_1d,  id2, upper_confidence_limit, DATASET='upper limit', LINE=1, color='green' , BINSIZE=0.1 
  dataset_1d,  id2, lower_confidence_limit, DATASET='lower-limit', LINE=1, color='yellow', BINSIZE=0.1

  erru = upper_confidence_limit > param_value 
  errl = param_value < lower_confidence_limit 
  mid         = 0.5*(erru+errl)
  half_length = 0.5*(erru-errl)
  
  function_1d, id1, seq, mid, NSKIP_ERRORS=1, Y_ERROR=half_length, PSYM=3, LINE=6, DATA='confidence interval', WIDGET_TITLE=parameter_name, XTIT='source #', YTIT=parameter_name
  function_1d, id1, seq, param_value,                              PSYM=1, LINE=6, $
               PLOT_WINDOW_OPTIONS=string(median(param_value), F='(%"SET_BIG_MARKER=[0,%0.2f], SHOW_DATE=0")'),$
               PS_CONFIG={filename:parameter_name+'_CI.ps'}
  
;  function_1d, id1, seq, param_value           , DATASET='best-fit'   , color='blue'  , LINE=6, PSYM=2, YTIT=parameter_name
;  function_1d, id1, seq, upper_confidence_limit, DATASET='upper-limit', color='red'   , LINE=6, PSYM=1, YTIT=parameter_name
;  function_1d, id1, seq, lower_confidence_limit, DATASET='lower-limit', color='yellow', LINE=6, PSYM=1
endif

; Show sources with anomalies.
ind = where( stregex(/BOOL, anom_flags, '[a-zA-Z]' ), count )
if keyword_set(verbose) && (count GT 0) then begin
  print, count, parameter_name, F='(%"\nThese %d sources have one of the following anomalies on parameter %s:\n")'
  print, '  Reasons uncertainty on fit parameter is not reported:'
  print, '    o: improper confidence interval, upperlim < best_val OR lowerlim > best_val' 
  print, '    s: error computation skipped' 
  print, '    f: parameter was frozen' 
  print, '    L: lowerlim outside specified range (soft limits in XSPEC)' 
  print, '    U: upperlim outside specified range (soft limits in XSPEC)' 
  print, '    l: tclout error "hit hard lower limit" OR "search failed in -ve direction"' 
  print, '    u: tclout error "hit hard upper limit" OR "search failed in +ve direction"' 
  print, '    c: tclout error "reduced chi-squared too high"' 
  print     
  print, '  Reasons fit parameter itself is not reported:'
  print, '    o: improper confidence interval, upperlim < best_val OR lowerlim > best_val' 
  print     
  print, '  Other anomalies:'
  print, '    r: parameter outside range specified in fitting script' 
  print, '    n: tclout error "non-monotonicity detected"' 
  print, '    m: tclout error "minimization may have run into problem"' 
  print

  forprint, seq, label, anom_flags, SUBSET=ind, F='(%"%5d %s:  %s")', TEXTOUT=2
endif

if (n_elements(param_value) GT 1) then print, parameter_name, median(param_value), F='(%"\nMedian %s = %0.4g\n")'


; Find a parameter value that produces equal numbers of confidence intervals that are above and below that value.
lind = where(finite(lower_confidence_limit), lcount)
uind = where(finite(upper_confidence_limit), ucount)
if (lcount EQ 0) || (ucount EQ 0) then begin
  bisector = !VALUES.F_NAN
endif else begin
  lower = lower_confidence_limit[lind]
  upper = upper_confidence_limit[uind]
  lower = lower[reverse(sort(lower))]
  upper = upper[        sort(upper)]
  
  lind = 0L ; index into vector "lower" defining upper limit on bisector
  uind = 0L ; index into vector "upper" defining lower limit on bisector
  ldone = 0B
  udone = 0B
  repeat begin
    ; Advance lind (reduce the bisector upper limit), but undo if we fall off the end of the array or the bisector range becomes improper.
    lind++
    if (lind GE lcount) || (lower[lind] LT upper[uind]) then begin
      lind--
      ldone = 1
    endif
    
    ; Advance uind (increase the bisector lower limit), but undo if we fall off the end of the array or the bisector range becomes improper.
    uind++
    if (uind GE ucount) || (lower[lind] LT upper[uind]) then begin
      uind--
      udone = 1
    endif
  endrep until (ldone && udone)
  
  bisector = mean([lower[lind],upper[uind]])
endelse ; search for bisector


;; ------------------------------------------------------------------------
; Build parameter strings.
par_str = strarr(n_elements(param_value))
for ii=0L,n_elements(param_value)-1 do par_str[ii] = limit_precision( param_value[ii], significant_digits )

errl = (param_value - lower_confidence_limit)
erru = (upper_confidence_limit - param_value)
par_errl_str = string(errl, F='(%"_{-%0.1f}")')
par_erru_str = string(erru, F='(%"^{+%0.1f}")')

; Pad as necessary to get the decimal places to line up, when displayed in a left-justified table column.
offset_of_decimal = strpos(par_str, '.')
ind = where(offset_of_decimal LT max(offset_of_decimal), count)
if (count GT 0) then begin
 par_str[ind] = '{\phn}' + par_str[ind]
  
endif

ind = where((errl < erru) LT 0.1, count)
if (count GT 0) then begin
 par_errl_str[ind] = string(errl[ind], F='(%"_{-%0.2f}")')
 par_erru_str[ind] = string(erru[ind], F='(%"^{+%0.2f}")')
endif

ind = where(~finite(errl) AND ~finite(erru), count)
if (count GT 0) then begin
 ; It might look nicer to show whitespace instead of "..." when BOTH errors are missing, however doing so somewhat complicates the journal's task of converting to an ASCII table.
 par_errl_str[ind] = '_{\cdots}'
 par_erru_str[ind] = '^{\cdots}'
endif

ind = where(~finite(errl) AND finite(erru), count)
if (count GT 0) then begin
 par_errl_str[ind] = '_{\cdots}'
endif

ind = where(~finite(erru) AND finite(errl), count)
if (count GT 0) then begin
 par_erru_str[ind] = '^{\cdots}'
endif

; Mark parameters that violate their soft limits with a special string in the errors.
ind = where(/NULL, range_flag EQ 'r')
 par_errl_str[ind] = '\ddagger'
 par_erru_str[ind] = ''

; Mark frozen parameters with an asterisk.
ind = where(param_was_frozen, count)
if (count GT 0) then begin
 par_errl_str[ind] = '\!*'
 par_erru_str[ind] = ''
endif

param_range = '$'+par_str+'\phd'+par_errl_str+par_erru_str+'$'

; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
; Thus, we must explicitly change any null cells to '\nodata'.
; Note that param_value of +Inf or -Inf is retained, e.g. when NH=0 and log NH = -Inf.
ind = where(finite(param_value,/NAN), count)
if (count GT 0) then begin
  param_range[ind] = '\nodata'
endif


; Invalidate the fit result itself if certain anomalies were found.
; Let's display a string that forces the observer to investigate why the parameter is invalid!
ind = where(~param_was_frozen AND stregex(/BOOL, anom_flags, bad_fit_flaglist), count)
if (count GT 0) then begin
 param_range[ind] = 'WTH?'
endif 


; Pad tails of strings with blanks to make them all the same length.
str_length = strlen(param_range)
pad_length = max(str_length) - str_length
pad = strmid('                          ', 0, pad_length)
param_range += pad

;; We need to return components of param_range so that we can build the entries for the emission measure from the NORM results.
return, { $
parameter_name        :parameter_name        ,$
label                 :label                 ,$
param_min             :param_min             ,$
param_max             :param_max             ,$
param_value           :param_value           ,$
lower_confidence_limit:lower_confidence_limit,$  
upper_confidence_limit:upper_confidence_limit,$  
errstatus             :errstatus             ,$
param_was_frozen      :param_was_frozen      ,$
anom_flags            :anom_flags            ,$
bisector              :bisector              ,$
par_str               :par_str               ,$
par_errl_str          :par_errl_str          ,$
par_erru_str          :par_erru_str          ,$
param_range           :param_range            $
}

end  ; validate_xspec_confidence_interval



;;; ==========================================================================
;;; Transformation of NH reported by XSPEC (units of 10^22 cm^-2) to the units desired in LaTeX tables.
FUNCTION nh_trans, NH_from_xspec

return, NH_from_xspec

;; The absorption parameter in XSPEC models are in units of 1E22 /cm**2.
;return, 1e22 * NH_from_xspec  

end



;;; ==========================================================================
;;; ae_flatten_collation
;;; Convert an AE collation (with vector photometry columns) into a flat table with better column names.

;;; The parameter SKY_OFFSET=[deltaX,deltaY] is the optional astrometric offset you wish to apply
;;; to all source positions, in units of CIAO sky pixels.  The RA & x axes have opposite signs
;;; so if you want to increase RA of the sources you supply a negative deltaX value.  
;;; The DEC and y axes have the same signs.
;;;

;;; The parameter "distance" must be in units of pc.

;;; Unit specifications follow the standard in "Specification of Physical Units within OGIP FITS files" at
;;; http://heasarc.gsfc.nasa.gov/docs/heasarc/ofwg/docs/general/ogip_93_001/ogip_93_001.html


;;; ==========================================================================
PRO ae_flatten_collation, COLLATED_FILENAME=collated_filename, FLAT_TABLE_FILENAME=flat_table_filename, $
      DISTANCE=distance, DIFFUSE=diffuse, ABUNDANCES_TO_REPORT=abundances_to_report, $
      SORT=sort, SKY_OFFSET=sky_offset, SRC_SIGNIF_MIN=src_signif_min, CONFIDENCE_LEVEL=confidence_level, $
      PLOT=plot, VERBOSE=verbose, FAST=fast

creator_string = "ae_flatten_collation, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

if (n_elements(abundances_to_report) EQ 0) then abundances_to_report=['O','Ne','Mg','Si','S','Fe']
num_abundance_columns = 6
if (n_elements(abundances_to_report) GT num_abundance_columns) then begin
  print, "ERROR: a maximum of 6 abundances can be reported."
  retall
endif

if (n_elements(plot          )      EQ 0) then plot          =1
if (n_elements(verbose       )      EQ 0) then verbose       =1
if (n_elements(src_signif_min)      EQ 0) then src_signif_min=0
if ~keyword_set(confidence_level)         then confidence_level = 0.683 ; [-1sigma, +1sigma]


if (n_elements(flat_table_filename) EQ 0) then begin
  print, "ERROR: you must supply FLAT_TABLE_FILENAME; set to '' for a plot-only run."
  retall
endif


;; Create a unique scratch directory.
temproot = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)
tempdir = temproot

if ~keyword_set(fast) then run_command, PARAM_DIR=tempdir
  
null_val = !VALUES.F_NAN

;;==========================================================================
;; Read the source properties of all the sources.
bt = mrdfits(collated_filename, 1, theader_in, /SILENT, STATUS=status)
if (status NE 0) then begin
  print, 'ERROR reading ', collated_filename
  retall      
endif

num_sources = n_elements(bt)
print, num_sources, F='(%"\n%d sources found in collated table.\n")'

null_val    = !VALUES.F_NAN
null_vector = replicate(null_val, num_sources)

if (num_sources LT 2) then plot = 0


;;==========================================================================
;; Modify the catalog as needed.

; Verify catalog is sorted by RA.
sort_ind = keyword_set(sort) ? bsort(bt.RA) : indgen(num_sources)  
dum = where(sort_ind NE indgen(num_sources), count)
if (count GT 0) then begin
  print, 'WARNING!  Sorting this catalog by RA before extracting data ...'
  bt = bt[sort_ind]
endif

; Shift the positions.
if keyword_set(sky_offset) then begin
  ; Create astrometry conversion between sky & celestial coordinates, then
  ; adjust all the RA & DEC positions as specified & Rename the sources.
  make_astr, astr, CRVAL=[median(bt.RA),median(bt.DEC)], CRPIX=[1,1], DELT=[-0.000136667, 0.000136667]
  ad2xy, bt.RA, bt.DEC, astr, x, y
  xy2ad, x+sky_offset[0], y+sky_offset[1], astr, RA_FINAL, DEC_FINAL
  
  precision  = 1
  OBJECT_FINAL = strcompress(/REMOVE_ALL, adstring(RA_FINAL,DEC_FINAL,precision,/TRUNCATE))
  print, 'SKY_OFFSET=', sky_offset

  forprint, seq, bt.Name, OBJECT_FINAL, F='(%"%4d %s -> %s")'
  
  bt.RA     = RA_FINAL    
  bt.DEC    = DEC_FINAL   
  bt.Name = OBJECT_FINAL
endif 


;;==========================================================================
;; Verify that the expected photometry energy bands are in the expected place.
;; The AE default energy bands have been set to use the bands desired for HMSFRs
;; 0.5:8;  0.5:2, 2-8;  0.5:1.7, 1.7:2.8  2.8:8

band_total = 0
e_lo = 0.5
e_hi = keyword_set(diffuse) ? 7.0 : 8.0
if ~almost_equal(bt.ENERG_LO[band_total], e_lo, DATA_RANGE=range) then print, band_total, range, e_lo, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be %0.1f keV.\n")'
if ~almost_equal(bt.ENERG_HI[band_total], e_hi, DATA_RANGE=range) then print, band_total, range, e_hi, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be %0.1f keV.\n")'

band_hard = 2 < (n_elements(bt[0].ENERG_LO) - 1)
e_lo = 2.0
e_hi = keyword_set(diffuse) ? 7.0 : 8.0
if ~almost_equal(bt.ENERG_LO[band_hard], e_lo, DATA_RANGE=range) then print, band_total, range, e_lo, F='(%"\nWARNING: for Hard Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be %0.1f keV.\n")'
if ~almost_equal(bt.ENERG_HI[band_hard], e_hi, DATA_RANGE=range) then print, band_total, range, e_hi, F='(%"\nWARNING: for Hard Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be %0.1f keV.\n")'

band_soft = 1 < (n_elements(bt[0].ENERG_LO) - 1)
e_lo = 0.5
e_hi = 2.0
if ~almost_equal(bt.ENERG_LO[band_soft], e_lo, DATA_RANGE=range) then print, band_total, range, e_lo, F='(%"\nWARNING: for Soft Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be %0.1f keV.\n")'
if ~almost_equal(bt.ENERG_HI[band_soft], e_hi, DATA_RANGE=range) then print, band_total, range, e_hi, F='(%"\nWARNING: for Soft Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_LO should be %0.1f keV.\n")'


;band_500_1700 = 3
;if max(abs([bt.ENERG_LO[band_500_1700] - 0.5, bt.ENERG_HI[band_500_1700] - 1.7])) GT 0.01 then begin
;  print, band_500_1700, F='(%"\nERROR: Band %d is not 0.5:1.7 keV")'
;  return
;endif
;
;band_1700_2800 = 4
;if max(abs([bt.ENERG_LO[band_1700_2800] - 1.7, bt.ENERG_HI[band_1700_2800] - 2.8])) GT 0.01 then begin
;  print, band_1700_2800, F='(%"\nERROR: Band %d is not 1.7:2.8 keV")'
;  return
;endif
;
;band_2800_8000 = 5
;if max(abs([bt.ENERG_LO[band_2800_8000] - 2.8, bt.ENERG_HI[band_2800_8000] - 8.0])) GT 0.01 then begin
;  print, band_2800_8000, F='(%"\nERROR: Band %d is not 2.8:8.0 keV")'
;  return
;endif



GOTO, SKIP_HRS
;; ------------------------------------------------------------------------
;; Here we use NET_CNTS entries in pairs of photometry table rows to make HR = (hard_cnts - soft_cnts)/(hard_cnts + soft_cnts)
;; Compute hardness upper and lower 1-sigma errors using equation 1.31 in 
; "A Practical Guide to Data Analysis for Physical Science Students", L. Lyons, 1991.
hard_band = [band_hard, band_1700_2800, band_2800_8000]
soft_band = [band_soft , band_500_1700 , band_1700_2800]

for ii = 0, n_elements(hard_band)-1 do begin
  print, bt[0].ENERG_LO[soft_band[ii]], bt[0].ENERG_HI[soft_band[ii]], $
         bt[0].ENERG_LO[hard_band[ii]], bt[0].ENERG_HI[hard_band[ii]], $
         F='(%"Hardness Ratio using bands %3.1f:%3.1f and %3.1f:%3.1f")'

  ; Recall that NET_CNTS can be negative.  
  ; To ensure that HRs are bounded by [-1,1] we choose to clip such NET_CNTS entries at zero
  ; and choose to set their lower errors to zero.  
  ; We will later do similar clipping at zero when we offset NET_CNTS downward by NET_CNTS_SIGMA_LOW
  ; during Lyon's error propagation.
  
  hard_cnts = float(bt.NET_CNTS[hard_band[ii]]) > 0
  soft_cnts = float(bt.NET_CNTS[soft_band[ii]]) > 0
  
  hard_sigma_up   = bt.NET_CNTS_SIGMA_UP[hard_band[ii]]
  soft_sigma_up   = bt.NET_CNTS_SIGMA_UP[soft_band[ii]]

  hard_sigma_low  = bt.NET_CNTS_SIGMA_LOW[hard_band[ii]]
  soft_sigma_low  = bt.NET_CNTS_SIGMA_LOW[soft_band[ii]]
  
  ; Compute some metrics & flags to identify various conditions where the HRs and/or their errors are not reliable.
  is_undefined       = ((hard_cnts - hard_sigma_low) LE 0) AND ((soft_cnts - soft_sigma_low) LE 0)
  is_very_hard       = ((hard_cnts - hard_sigma_low) GT 0) AND ((soft_cnts - soft_sigma_low) LE 0)
  is_very_soft       = ((hard_cnts - hard_sigma_low) LE 0) AND ((soft_cnts - soft_sigma_low) GT 0)

  ; The Nx5 arrays "hard" and "soft" below will simplify the 5 evaluations of our function ((h-s)/(h+s)) needed  
  ; to do the Lyons calculations for upper and lower sigmas.
  ; 0th entry is nominal HR 
  ; 1st entry is where hard fluxuates upward,   HR fluxuates upward
  ; 2nd entry is where soft fluxuates upward,   HR fluxuates downward
  ; 3rd entry is where hard fluxuates downward, HR fluxuates downward
  ; 4rd entry is where soft fluxuates downward, HR fluxuates upward
  
  hard_cnts_more = hard_cnts + hard_sigma_up
  soft_cnts_more = soft_cnts + soft_sigma_up

  ; To retain range of HR we clip downward fluxuations in NET_CNTS at zero.
  hard_cnts_less = (hard_cnts - hard_sigma_low) > 0
  soft_cnts_less = (soft_cnts - soft_sigma_low) > 0
  
  hard = [[hard_cnts], [hard_cnts_more], [hard_cnts     ], [hard_cnts_less], [hard_cnts     ]]
  soft = [[soft_cnts], [soft_cnts     ], [soft_cnts_more], [soft_cnts     ], [soft_cnts_less]]
  
  ratios = (hard - soft) / (hard + soft)
  
  hr           = ratios[*,0]

  ; There's a questions here!  Lyon's book seems to say that upper sigma for f is computed using terms (1&2) where the 
  ; input parameters both fluxuate up, and lower sigma for f uses terms (3&4) where parameters fluxuate down.
  ; However it seems more reasonable to estimate the upper limit on f using the terms where f fluxuates upward (1&4)
  ; and estimate the lower limit on f using terms where f fluxuates downward (2&3).
  ; Talk to Niel and Eric about this!!!
  
  hr_sigma_up  = sqrt( (ratios[*,1] - hr)^2 + $
                       (ratios[*,2] - hr)^2 )

  hr_sigma_low = sqrt( (ratios[*,3] - hr)^2 + $
                       (ratios[*,4] - hr)^2 )


  ;; Insert null_val any place we want \nodata in the table.
  
  ; Handle ratios whose confidence interval includes +-1 (because confidence interval of hard or soft includes 0).
  ind = where(is_very_hard,count)
  if (count GT 0) then begin
    print, count, ' sources are very hard.'
    hr_sigma_up [ind] = null_val
  endif
  
  ind = where(is_very_soft,count)
  if (count GT 0) then begin
    print, count, ' sources are very soft.'
    hr_sigma_low[ind] = null_val
  endif

  ; Handle cases where we don't want to report the HR at all.
  ind = where(is_undefined,count)
  if (count GT 0) then begin
    print, count, ' sources have unreliable HR values.'
    hr          [ind] = null_val
    hr_sigma_up [ind] = null_val
    hr_sigma_low[ind] = null_val
  endif

  
  ; Save the results in named variables.
  case ii of
   0: begin
      hr1           = hr
      hr1_sigma_up  = hr_sigma_up
      hr1_sigma_low = hr_sigma_low
  is_undefined1       = is_undefined
  is_very_hard1       = is_very_hard
  is_very_soft1       = is_very_soft 
      end
   1: begin
      hr2           = hr
      hr2_sigma_up  = hr_sigma_up
      hr2_sigma_low = hr_sigma_low
  is_undefined2       = is_undefined
  is_very_hard2       = is_very_hard
  is_very_soft2       = is_very_soft 
      end
   2: begin
      hr3           = hr
      hr3_sigma_up  = hr_sigma_up
      hr3_sigma_low = hr_sigma_low
  is_undefined3       = is_undefined
  is_very_hard3       = is_very_hard
  is_very_soft3       = is_very_soft 
      end
  endcase
endfor ; ii

;; COUP SPECIAL CODE:
save, /COMPRESS, FILE='hr.sav', seq, OBJECT, $
      is_undefined1, is_very_hard1, is_very_soft1, hr1, hr1_sigma_up, hr1_sigma_low, $
      is_undefined2, is_very_hard2, is_very_soft2, hr2, hr2_sigma_up, hr2_sigma_low, $
      is_undefined3, is_very_hard3, is_very_soft3, hr3, hr3_sigma_up, hr3_sigma_low

SKIP_HRS:




;;==========================================================================
;; Define the columns that will appear in the output table.

fxbhmake, theader, num_sources, 'X-ray Properties', 'derived from ACIS Extract', /DATE, /INIT

; While defining FITS columns, we use 1-based column numbers.
out_colnum = 1

; Create a sequence number column.
fxbaddcol, this_colnum, theader, 0, 'Seq', 'sequence number', TDISP='I5'    


;; ------------------------------------------------------------------------
;; Define all the direct associations between columns in the collated table and columns in the flattened table.
col_assignment =  {ca, old_name:'', multiband:0B, high_precision:0B, tdisp:'', new_name:''}
col_assignment = [$

; Name and position (see http://cds.u-strasbg.fr/doc/catstd-3.3.htx)                
  tag_exist(bt,'OBJECT') ? $
   {ca, 'OBJECT             ', 0,0,'A18'  , 'Name'} : $
   {ca, 'CATALOG_NAME       ', 0,0,'A18'  , 'Name'},$
   
  {ca, 'LABEL               ', 0,0,'A15'  , 'Label'},$
  {ca, 'RA                  ', 0,1,'F10.6', 'RAdeg'},$
  {ca, 'DEC                 ', 0,1,'F10.6', 'DEdeg'},$
  {ca, 'POSNTYPE            ', 0,0,'A20'  , 'PosType'},$
  
  tag_exist(bt,'ERR_RA') ? $
   ; Modern recipe stores position errors in these columns.
   [$ 
    {ca, 'ERR_RA              ', 0,0,'F5.2' , 'RAerr' },$
    {ca, 'ERR_DEC             ', 0,0,'F5.2' , 'DEerr' },$
    {ca, 'ERR_POS             ', 0,0,'F5.2' , 'PosErr'} $
    ] : [ $
    ; Old recipe stored position errors in these columns.
    {ca, 'ERX_DATA            ', 0,0,'F5.2' , 'XPosErr'},$
    {ca, 'ERY_DATA            ', 0,0,'F5.2' , 'YPosErr'},$
    {ca, 'ERR_DATA            ', 0,0,'F5.2' , 'PosErr' } $
    ],$

; Scalar properties
  
  {ca, 'PHOT_CREATOR          ', 0,0,'A18'  , 'PhotometryCreator'},$
  tag_exist(bt,'PB_MIN') ? $
   [$ 
    {ca, 'PB_MIN                      ', 0,0,'E9.1' , 'ProbNoSrc_MostValid'},$
    {ca, 'PB_MIN_MERGE_NAME           ', 0,0,'A20'  , 'Merge_MostValid'},$
    {ca, 'PB_MIN_BAND                 ', 0,0,'A20'  , 'Band_MostValid'},$
    {ca, 'PB_MIN_SRC_CNTS             ', 0,1,'I7'   , 'SrcCounts_MostValid'},$
    {ca, 'PB_MIN_FULL_BAND            ', 0,0,'E9.1' , 'ProbNoSrc_t'},$
    {ca, 'PB_MIN_SOFT_BAND            ', 0,0,'E9.1' , 'ProbNoSrc_s'},$
    {ca, 'PB_MIN_HARD_BAND            ', 0,0,'E9.1' , 'ProbNoSrc_h'},$
    {ca, 'PB_MIN_VHARD_BAND           ', 0,0,'E9.1' , 'ProbNoSrc_v'},$
    {ca, 'TALLY_SINGLEOBSID_VALIDATION', 0,0,'I3'   , 'NumValidObsIDs'},$
    {ca, 'LIST_SINGLEOBSID_VALIDATION ', 0,0,'A200' , 'ValidObsIDList'},$
    {ca, 'IS_OCCASIONAL               ', 0,0,'I1'   , 'IsOccasional'},$
    {ca, 'PB_FULL_BAND                ', 0,0,'E9.1' , 'ProbNoSrc_t'},$ ; obsolete
    {ca, 'PB_SOFT_BAND                ', 0,0,'E9.1' , 'ProbNoSrc_s'},$ ; obsolete
    {ca, 'PB_HARD_BAND                ', 0,0,'E9.1' , 'ProbNoSrc_h'} $ ; obsolete
    ] : [ $
    {ca, 'PROB_NO_SOURCE    ', 1,0,'E9.1' , 'ProbNoSrc_'} $
    ],$
  
  {ca, 'PROB_KS             ', 0,0,'E9.1' , 'ProbKS_single'},$
  {ca, 'N_KS                ', 0,0,'I3'   ,   'N_KS_single'  },$

  {ca, 'MERGE_KS            ', 0,0,'E9.1' , 'ProbKS_merge'},$
  {ca, 'MERG_CHI            ', 0,0,'E9.1' , 'ProbChisq_PhotonFlux'},$
  
  {ca, 'EXPOSURE            ', 0,0,'G10.5', 'ExposureTimeNominal'},$
  {ca, 'FRACEXPO            ', 0,0,'F4.2' , 'ExposureFraction'},$  ; (fraction of time source was on live portion of detector)
  {ca, 'RATE_3x3            ', 0,0,'F4.2' , 'RateIn3x3Cell'},$
; {ca, 'WARNFRAC            ', 0,0,'F4.2' , 'StreakFraction'},$
  {ca, 'NUM_OBS             ', 0,0,'I3'   , 'NumObsIDs'},$
  {ca, 'MERGNUM             ', 0,0,'I3'   , 'NumMerged'},$
  {ca, 'MERGBIAS            ', 0,0,'G8.2' , 'MergeBias'},$
  {ca, 'MERGQUAL            ', 0,0,'G8.2' , 'MergeQuality'},$
  {ca, 'MERGE_NAME          ', 0,0,'A20'  , 'MergeName'},$

  {ca, 'THETA_LO            ', 0,0,'F4.1' , 'Theta_Lo'},$
  {ca, 'THETA               ', 0,0,'F4.1' , 'Theta'},$
  {ca, 'THETA_HI            ', 0,0,'F4.1' , 'Theta_Hi'},$
  {ca, 'PSF_FRAC            ', 0,0,'F4.2' , 'PsfFraction'},$
  {ca, 'SRC_AREA            ', 0,0,'F6.1' , 'SrcArea'},$
  {ca, 'AG_FRAC             ', 0,0,'F4.2' , 'AfterglowFraction'},$

; Photometry in three energy bands.
  {ca, 'SRC_CNTS            ', 1,1,'I7'   , 'SrcCounts_'},$
  {ca, 'BACKSCAL            ', 0,0,'G8.3' , 'BkgScaling'},$
  {ca, 'BKG_CNTS            ', 1,1,'I7'   , 'BkgCounts_'},$
  {ca, 'NET_CNTS            ', 1,0,'F7.1' , 'NetCounts_'},$
  {ca, 'MEAN_ARF            ', 1,0,'G8.3' , 'MeanEffectiveArea_'},$
  {ca, 'FLUX2               ', 1,0,'G10.3', 'PhotonFlux_'},$

; Omit MeanEnergy since it is very noisy for low-count sources!
; {ca, 'ENERG_MEAN_OBSERVED ', 1,0,'F5.2' , 'MeanEnergy_'},$

  {ca, 'ENERG_PCT50_OBSERVED', 1,0,'F5.2' , 'MedianEnergy_'}, $

; Columns used for special projects ...    
  {ca, 'AV_TYPE', 0,0,'A5'   , 'AvType'}, $
  {ca, 'AV'     , 0,0,'F5.2' , 'Av'}, $
  {ca, 'NHISM'  , 0,0,'F5.2' , 'NHISM'}, $

; Some of the columns generated by XSPEC fitting.
  {ca, 'FH2'  ,0,0,'G0.3' , 'FH2'  }, $
  {ca, 'FH2U' ,0,0,'G0.3' , 'FH2U' }, $
  {ca, 'FH2L' ,0,0,'G0.3' , 'FH2L' }, $
  {ca, 'F28'  ,0,0,'G0.3' , 'F28'  }, $
  {ca, 'F28U' ,0,0,'G0.3' , 'F28U' }, $
  {ca, 'F28L' ,0,0,'G0.3' , 'F28L' }, $
  {ca, 'FH8'  ,0,0,'G0.3' , 'FH8'  }, $
  {ca, 'FH8U' ,0,0,'G0.3' , 'FH8U' }, $
  {ca, 'FH8L' ,0,0,'G0.3' , 'FH8L' }, $
  {ca, 'FSO'  ,0,0,'G0.3' , 'FSO'  }, $
  {ca, 'FSOU' ,0,0,'G0.3' , 'FSOU' }, $
  {ca, 'FSOL' ,0,0,'G0.3' , 'FSOL' }, $
  {ca, 'FME'  ,0,0,'G0.3' , 'FME'  }, $
  {ca, 'FMEU' ,0,0,'G0.3' , 'FMEU' }, $
  {ca, 'FMEL' ,0,0,'G0.3' , 'FMEL' }, $
  {ca, 'FHA'  ,0,0,'G0.3' , 'FHA'  }, $
  {ca, 'FHAU' ,0,0,'G0.3' , 'FHAU' }, $
  {ca, 'FHAL' ,0,0,'G0.3' , 'FHAL' }, $
  {ca, 'FTO'  ,0,0,'G0.3' , 'FTO'  }, $
  {ca, 'FTOU' ,0,0,'G0.3' , 'FTOU' }, $
  {ca, 'FTOL' ,0,0,'G0.3' , 'FTOL' }, $
  {ca, 'F1H2' ,0,0,'G0.3' , 'F1H2' }, $
  {ca, 'F128' ,0,0,'G0.3' , 'F128' }, $
  {ca, 'F1H8' ,0,0,'G0.3' , 'F1H8' }, $
  {ca, 'F1SO' ,0,0,'G0.3' , 'F1SO' }, $
  {ca, 'F1ME' ,0,0,'G0.3' , 'F1ME' }, $
  {ca, 'F1HA' ,0,0,'G0.3' , 'F1HA' }, $
  {ca, 'F1TO' ,0,0,'G0.3' , 'F1TO' }, $
  {ca, 'F1CH2',0,0,'G0.3' , 'F1CH2'}, $
  {ca, 'F1C28',0,0,'G0.3' , 'F1C28'}, $
  {ca, 'F1CH8',0,0,'G0.3' , 'F1CH8'}, $
  {ca, 'F1CSO',0,0,'G0.3' , 'F1CSO'}, $
  {ca, 'F1CME',0,0,'G0.3' , 'F1CME'}, $
  {ca, 'F1CHA',0,0,'G0.3' , 'F1CHA'}, $
  {ca, 'F1CTO',0,0,'G0.3' , 'F1CTO'}, $
  {ca, 'F2H2' ,0,0,'G0.3' , 'F2H2' }, $
  {ca, 'F228' ,0,0,'G0.3' , 'F228' }, $
  {ca, 'F2H8' ,0,0,'G0.3' , 'F2H8' }, $
  {ca, 'F2SO' ,0,0,'G0.3' , 'F2SO' }, $
  {ca, 'F2ME' ,0,0,'G0.3' , 'F2ME' }, $
  {ca, 'F2HA' ,0,0,'G0.3' , 'F2HA' }, $
  {ca, 'F2TO' ,0,0,'G0.3' , 'F2TO' }, $
  {ca, 'F2CH2',0,0,'G0.3' , 'F2CH2'}, $
  {ca, 'F2C28',0,0,'G0.3' , 'F2C28'}, $
  {ca, 'F2CH8',0,0,'G0.3' , 'F2CH8'}, $
  {ca, 'F2CSO',0,0,'G0.3' , 'F2CSO'}, $
  {ca, 'F2CME',0,0,'G0.3' , 'F2CME'}, $
  {ca, 'F2CHA',0,0,'G0.3' , 'F2CHA'}, $
  {ca, 'F2CTO',0,0,'G0.3' , 'F2CTO'}, $
  {ca, 'FCH2' ,0,0,'G0.3' , 'FCH2' }, $
  {ca, 'FC28' ,0,0,'G0.3' , 'FC28' }, $
  {ca, 'FCH8' ,0,0,'G0.3' , 'FCH8' }, $
  {ca, 'FCSO' ,0,0,'G0.3' , 'FCSO' }, $
  {ca, 'FCME' ,0,0,'G0.3' , 'FCME' }, $
  {ca, 'FCHA' ,0,0,'G0.3' , 'FCHA' }, $
  {ca, 'FCTO' ,0,0,'G0.3' , 'FCTO' } $
]

col_assignment.old_name = strtrim(col_assignment.old_name,2)  
col_assignment.new_name = strtrim(col_assignment.new_name,2)  
  
for ii=0,n_elements(col_assignment)-1 do begin
  this = col_assignment[ii]
  
  if ~tag_exist(bt, this.old_name, INDEX=in_colnum, /QUIET) then continue
  
; comment = 'Column '+this.old_name+' in AE'
  dum     = psb_xpar( theader_in, string(1+in_colnum,F='(%"TTYPE%d")'), COMMENT=comment)
  tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
  if (count EQ 0) then tunit = ''
  
  example_value = (bt[0].(in_colnum))[0]
  ; Repair AE columns that have excessive precision.
  case size(example_value, /TNAME) of
    'DOUBLE': if ~this.high_precision then example_value = float(example_value)
    'LONG'  : if ~this.high_precision then example_value = fix  (example_value)
    else:
  endcase
  
  if (this.multiband) then begin
    fxbaddcol, this_colnum, theader, example_value, this.new_name+'t', comment, TUNIT=tunit, TDISP=this.tdisp
    fxbaddcol, this_colnum, theader, example_value, this.new_name+'s', comment, TUNIT=tunit, TDISP=this.tdisp
    fxbaddcol, this_colnum, theader, example_value, this.new_name+'h', comment, TUNIT=tunit, TDISP=this.tdisp
  endif else begin
    fxbaddcol, this_colnum, theader, example_value, this.new_name    , comment, TUNIT=tunit, TDISP=this.tdisp
  endelse
endfor ;ii


;; ------------------------------------------------------------------------
;; Add columns for energy fluxes.
if tag_exist(bt, 'FLUX2') && tag_exist(bt, 'ENERG_PCT50_OBSERVED') then begin
  fxbaddcol, this_colnum, theader, 0.0, 'EnergyFlux_t', '(EnergyFlux_s>0)+(EnergyFlux_h>0)'        , TUNIT='erg /cm**2 /s', TDISP='G10.3'
  fxbaddcol, this_colnum, theader, 0.0, 'EnergyFlux_s', 'PhotonFlux_s * MedianEnergy_s * 1.602E-09', TUNIT='erg /cm**2 /s', TDISP='G10.3'
  fxbaddcol, this_colnum, theader, 0.0, 'EnergyFlux_h', 'PhotonFlux_h * MedianEnergy_h * 1.602E-09', TUNIT='erg /cm**2 /s', TDISP='G10.3'
endif

;; ------------------------------------------------------------------------
;; Add columns reporting photometry confidence intervals from aprates tool.
comment = keyword_set(fast) ? ' (AE)' : ' (aprates)'
fxbaddcol, this_colnum, theader, 0.0, 'NetCounts_Lo_t', '1-sigma lower bound'+comment, TUNIT='count', TDISP='G8.3'
fxbaddcol, this_colnum, theader, 0.0, 'NetCounts_Hi_t', '1-sigma upper bound'+comment, TUNIT='count', TDISP='G8.3'
                                                            
fxbaddcol, this_colnum, theader, 0.0, 'NetCounts_Lo_s', '1-sigma lower bound'+comment, TUNIT='count', TDISP='G8.3'
fxbaddcol, this_colnum, theader, 0.0, 'NetCounts_Hi_s', '1-sigma upper bound'+comment, TUNIT='count', TDISP='G8.3'
                                                            
fxbaddcol, this_colnum, theader, 0.0, 'NetCounts_Lo_h', '1-sigma lower bound'+comment, TUNIT='count', TDISP='G8.3'
fxbaddcol, this_colnum, theader, 0.0, 'NetCounts_Hi_h', '1-sigma upper bound'+comment, TUNIT='count', TDISP='G8.3'


;; ------------------------------------------------------------------------
;; Add columns derived from spectral fitting results
if keyword_set(distance) then begin
  psb_xaddpar, theader, 'DISTANCE', distance, '[pc] distance'

  ; The "dscale" term (4*pi*distance^2) arises is two contexts.
  
  ; 1. The luminosity calculation involves integrating an observed flux over a sphere whose center is the source.
  
  ; 2. The "norm" parameter of many XSPEC plasma models (see the XSPEC manual) is defined as the "emission measure" integral divided by dscale.  The emission measure integral has units of cm^-3 and dscale has units of cm^2; thus "norm" appears to have units of cm^-5.
  ; Note that "norm" in XSPEC is also scaled by 1E-14, requiring a unitless 1E14 term in the em_offset quantity below.
  
  cm_per_parsec = 3.086D18
  dscale        = 4D*!PI*(distance * cm_per_parsec)^2  ; [cm^2]
  
  if keyword_set(diffuse) then begin
    ;; For diffuse sources extracted by AE the area of the region (in arcsec^2) is put into the ARF calibration file,
    ;; which has units of cm**2 count /photon /arcsec^2.
    ;; Thus, XSPEC "flux" and "norm" quantities saved by the fitting scripts have a per arcsec^2 normalization.
    ;; That's a good thing (in my opinion) in that the size of the extraction region the observer chose does not affect the astrophysical "brightness" quantities that we report.
    ;; However, using arcseconds as the units of the area normalization is observational, not astrophysical.
    ;; Thus, here we apply to flux and norm quantities a conversion from arcsec^-2 to the physical units of pc^-2.
    
    arcsec_per_pc = 360. * 3600. / (2*!PI*distance)
    dscale       *= arcsec_per_pc^2   ; [cm^2 arcsec^2 /pc^2]
    ; An interesting phenomenon appears in the calculation of dscale for the diffuse case above!
    ; In the term "4D*!PI*(distance * cm_per_parsec)^2" the quantity distance^2 appears in the numerator, while
    ; in the term "arcsec_per_pc^2" the quantity distance^2 appears in the denominator.
    ; Thus, for diffuse sources, dscale is always the same constant: 
    ;   cm_per_parsec^2 * (360*3600.)^2 / !PI = 5.0915728e+48  [cm^2 arcsec^2 /pc^2]
    ;
    ; The surface flux produced by the fitting scripts has           units of erg /s /cm^2 /arcsec^2.
    ; After multiplication by dscale, we get a surface luminosity in units of erg /s /pc^2.
    
    ; The "norm" value produced by the fitting has units of 1E-14 cm^-5 /arcsec^2.
    ; After multiplication by dscale, we get an "surface emission measure" in units of cm^-3 /pc^2.
  endif
  
  em_offset     = alog10(1E14 * dscale)
  lx_offset     = alog10(dscale)
           
  ; Hide errors when ???
  hide_errors_flag = 0
  
  if tag_exist(bt,'MODEL', INDEX=in_colnum) then begin
    dum     = psb_xpar( theader_in, string(1+in_colnum,F='(%"TTYPE%d")'), COMMENT=comment)
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
  
    fxbaddcol, this_colnum, theader, '', 'SpModel'        , comment, TUNIT=tunit, TDISP='A30'
  endif

  if tag_exist(bt,'PROVISNL', INDEX=in_colnum) then begin
    dum     = psb_xpar( theader_in, string(1+in_colnum,F='(%"TTYPE%d")'), COMMENT=comment)
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
  
    fxbaddcol, this_colnum, theader, 0, 'SpModelIsProvisional', comment, TUNIT=tunit, TDISP='I2'
  endif

  if tag_exist(bt,'HANDFIT', INDEX=in_colnum) then begin
    dum     = psb_xpar( theader_in, string(1+in_colnum,F='(%"TTYPE%d")'), COMMENT=comment)
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
  
    fxbaddcol, this_colnum, theader, 0, 'SpModelIsHandFit', comment, TUNIT=tunit, TDISP='I2'
  endif

  if tag_exist(bt,'CHI_SQR', INDEX=in_colnum) then begin
    dum     = psb_xpar( theader_in, string(1+in_colnum,F='(%"TTYPE%d")'), COMMENT=comment)
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
  
    if keyword_set(plot) then dataset_1d, id_chi, LINE=0, COLOR='white', bt.CHI_SQR, XTIT='ReducedChiSq', BINSIZE=0.1, PS_CONFIG={filename:'chisq_hist.ps'}, /PRINT

    fxbaddcol, this_colnum, theader, 0.0, 'ReducedChiSq'  , comment, TUNIT=tunit, TDISP='G8.3'
  endif
    
  if tag_exist(bt,'DOF', INDEX=in_colnum) then begin
    dum     = psb_xpar( theader_in, string(1+in_colnum,F='(%"TTYPE%d")'), COMMENT=comment)
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
  
    fxbaddcol, this_colnum, theader, 0, 'DegreesOfFreedom', comment, TUNIT=tunit, TDISP='I4'    
  endif
  
  if tag_exist(bt,'CSTAT', INDEX=in_colnum) then begin
    dum     = psb_xpar( theader_in, string(1+in_colnum,F='(%"TTYPE%d")'), COMMENT=comment)
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
  
    fxbaddcol, this_colnum, theader, 0.0, 'CStatistic'    , comment, TUNIT=tunit, TDISP='G8.3'
  endif
  
  
  ;; Examine NH results.
  if tag_exist(bt,'LMC', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    LMC = validate_xspec_confidence_interval( 'LMC', bt.LABEL, nh_trans(bt.LMC_MIN), nh_trans(bt.LMC_MAX), nh_trans(bt.LMC), nh_trans(bt.LMC_ERRL), nh_trans(bt.LMC_ERRU), bt.LMC_ERRS, bt.LMC_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'LMC'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'LMC_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'LMC_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'LMC_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'LMC_Flags', comment
  endif                  
  
  if tag_exist(bt,'NH1', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NH1 = validate_xspec_confidence_interval( 'NH1', bt.LABEL, nh_trans(bt.NH1_MIN), nh_trans(bt.NH1_MAX), nh_trans(bt.NH1), nh_trans(bt.NH1_ERRL), nh_trans(bt.NH1_ERRU), bt.NH1_ERRS, bt.NH1_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'NH1'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH1_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH1_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'NH1_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'NH1_Flags', comment
  endif                  
  
  if tag_exist(bt,'NH2', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NH2 = validate_xspec_confidence_interval( 'NH2', bt.LABEL,$
                                 nh_trans(bt.NH2_MIN ),$
                                 nh_trans(bt.NH2_MAX ),$
           (bt.NORM2/bt.NORM2) * nh_trans(bt.NH2     ),$
      tag_exist(bt,'NH2_ERRL') ? nh_trans(bt.NH2_ERRL) : 0,$
      tag_exist(bt,'NH2_ERRU') ? nh_trans(bt.NH2_ERRU) : 0,$
      tag_exist(bt,'NH2_ERRS') ?         (bt.NH2_ERRS) : 0,$
      bt.NH2_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'NH2'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH2_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH2_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'NH2_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'NH2_Flags', comment
  endif                  
  
  if tag_exist(bt,'NH3', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NH3 = validate_xspec_confidence_interval( 'NH3', bt.LABEL,$
                                 nh_trans(bt.NH3_MIN ),$
                                 nh_trans(bt.NH3_MAX ),$
           (bt.NORM3/bt.NORM3) * nh_trans(bt.NH3     ),$
      tag_exist(bt,'NH3_ERRL') ? nh_trans(bt.NH3_ERRL) : 0,$
      tag_exist(bt,'NH3_ERRU') ? nh_trans(bt.NH3_ERRU) : 0,$
      tag_exist(bt,'NH3_ERRS') ?         (bt.NH3_ERRS) : 0,$
      bt.NH3_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'NH3'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH3_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH3_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'NH3_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'NH3_Flags', comment
  endif                  
  

  if tag_exist(bt,'NH4', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NH4 = validate_xspec_confidence_interval( 'NH4', bt.LABEL,$
                                 nh_trans(bt.NH4_MIN ),$
                                 nh_trans(bt.NH4_MAX ),$
           (bt.NORM4/bt.NORM4) * nh_trans(bt.NH4     ),$
      tag_exist(bt,'NH4_ERRL') ? nh_trans(bt.NH4_ERRL) : 0,$
      tag_exist(bt,'NH4_ERRU') ? nh_trans(bt.NH4_ERRU) : 0,$
      tag_exist(bt,'NH4_ERRS') ?         (bt.NH4_ERRS) : 0,$
      bt.NH4_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'NH4'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH4_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH4_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'NH4_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'NH4_Flags', comment
  endif                  
  

  if tag_exist(bt,'NH5', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NH5 = validate_xspec_confidence_interval( 'NH5', bt.LABEL,$
                                 nh_trans(bt.NH5_MIN ),$
                                 nh_trans(bt.NH5_MAX ),$
                                 nh_trans(bt.NH5     ),$
      tag_exist(bt,'NH5_ERRL') ? nh_trans(bt.NH5_ERRL) : 0,$
      tag_exist(bt,'NH5_ERRU') ? nh_trans(bt.NH5_ERRU) : 0,$
      tag_exist(bt,'NH5_ERRS') ?         (bt.NH5_ERRS) : 0,$
      bt.NH5_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'NH5'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH5_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH5_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'NH5_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'NH5_Flags', comment
  endif                  
  
  if tag_exist(bt,'NH6', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NH6 = validate_xspec_confidence_interval( 'NH6', bt.LABEL,$
                                 nh_trans(bt.NH6_MIN ),$
                                 nh_trans(bt.NH6_MAX ),$
           (bt.NORM6/bt.NORM6) * nh_trans(bt.NH6     ),$
      tag_exist(bt,'NH6_ERRL') ? nh_trans(bt.NH6_ERRL) : 0,$
      tag_exist(bt,'NH6_ERRU') ? nh_trans(bt.NH6_ERRU) : 0,$
      tag_exist(bt,'NH6_ERRS') ?         (bt.NH6_ERRS) : 0,$
      bt.NH6_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )

    comment = 'from XSPEC'
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'NH6'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH6_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH6_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'NH6_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'NH6_Flags', comment
  endif                  
  
  if tag_exist(bt,'NH7', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NH7 = validate_xspec_confidence_interval( 'NH7', bt.LABEL,$
                                 nh_trans(bt.NH7_MIN ),$
                                 nh_trans(bt.NH7_MAX ),$
           (bt.NORM7/bt.NORM7) * nh_trans(bt.NH7     ),$
      tag_exist(bt,'NH7_ERRL') ? nh_trans(bt.NH7_ERRL) : 0,$
      tag_exist(bt,'NH7_ERRU') ? nh_trans(bt.NH7_ERRU) : 0,$
      tag_exist(bt,'NH7_ERRS') ?         (bt.NH7_ERRS) : 0,$
      bt.NH7_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )

    comment = 'from XSPEC'
   ;tunit   = '1 /cm**2'    ; if nh_trans() multiplies XSPEC value by 1E22
    tunit   = '1E22 /cm**2' ; if nh_trans() returns XSPEC value
    fxbaddcol, this_colnum, theader, 0.0, 'NH7'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH7_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'NH7_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'NH7_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'NH7_Flags', comment
  endif                  
  


  ;; Examine KT results.
  if tag_exist(bt,'KT1', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    KT1 = validate_xspec_confidence_interval( 'KT1', bt.LABEL, bt.KT1_MIN, bt.KT1_MAX, (bt.NORM1/bt.NORM1) * bt.KT1, bt.KT1_ERRL, bt.KT1_ERRU, bt.KT1_ERRS, bt.KT1_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
    
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'kT1'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kT1_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kT1_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'kT1_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'kT1_Flags', comment
  endif
    
  if tag_exist(bt,'KT2', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    KT2 = validate_xspec_confidence_interval( 'KT2', bt.LABEL, bt.KT2_MIN, bt.KT2_MAX, (bt.NORM2/bt.NORM2) * bt.KT2, bt.KT2_ERRL, bt.KT2_ERRU, bt.KT2_ERRS, bt.KT2_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
    
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'kt2'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt2_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt2_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'kt2_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'kt2_Flags', comment
  endif

  if tag_exist(bt,'KT3', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    KT3 = validate_xspec_confidence_interval( 'KT3', bt.LABEL, bt.KT3_MIN, bt.KT3_MAX, (bt.NORM3/bt.NORM3) * bt.KT3, bt.KT3_ERRL, bt.KT3_ERRU, bt.KT3_ERRS, bt.KT3_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
    
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'kt3'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt3_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt3_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'kt3_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'kt3_Flags', comment
  endif

  if tag_exist(bt,'KT4', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    KT4 = validate_xspec_confidence_interval( 'KT4', bt.LABEL,$
                                 bt.KT4_MIN,$
                                 bt.KT4_MAX,$
           (bt.NORM4/bt.NORM4) * bt.KT4,$
      tag_exist(bt,'KT4_ERRL') ? bt.KT4_ERRL : 0,$
      tag_exist(bt,'KT4_ERRU') ? bt.KT4_ERRU : 0,$
      tag_exist(bt,'KT4_ERRS') ? bt.KT4_ERRS : 0,$
      bt.KT4_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
    
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'kt4'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt4_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt4_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'kt4_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'kt4_Flags', comment
  endif

  if tag_exist(bt,'KT5', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    KT5 = validate_xspec_confidence_interval( 'KT5', bt.LABEL,$
                                 bt.KT5_MIN,$
                                 bt.KT5_MAX,$
           (bt.NORM5/bt.NORM5) * bt.KT5,$
      tag_exist(bt,'KT5_ERRL') ? bt.KT5_ERRL : 0,$
      tag_exist(bt,'KT5_ERRU') ? bt.KT5_ERRU : 0,$
      tag_exist(bt,'KT5_ERRS') ? bt.KT5_ERRS : 0,$
      bt.KT5_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )

    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'kt5'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt5_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt5_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'kt5_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'kt5_Flags', comment
  endif

  if tag_exist(bt,'KT6', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    KT6 = validate_xspec_confidence_interval( 'KT6', bt.LABEL,$
                                 bt.KT6_MIN,$
                                 bt.KT6_MAX,$
           (bt.NORM6/bt.NORM6) * bt.KT6,$
      tag_exist(bt,'KT6_ERRL') ? bt.KT6_ERRL : 0,$
      tag_exist(bt,'KT6_ERRU') ? bt.KT6_ERRU : 0,$
      tag_exist(bt,'KT6_ERRS') ? bt.KT6_ERRS : 0,$
      bt.KT6_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
    
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'kt6'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt6_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'kt6_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'kt6_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'kt6_Flags', comment
  endif


  
  ;; Examine TAU results.
  if tag_exist(bt,'TAU1', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    TAU1 = validate_xspec_confidence_interval( 'TAU1', bt.LABEL, alog10(bt.TAU1_MIN), alog10(bt.TAU1_MAX), (bt.NORM1/bt.NORM1) * alog10(bt.TAU1), alog10(bt.TAU1ERRL), alog10(bt.TAU1ERRU), bt.TAU1ERRS, bt.TAU1_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Tau1'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'Tau1_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'Tau1_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'Tau1_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Tau1_Flags', comment
  endif                  
  
  if tag_exist(bt,'TAU2', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    TAU2 = validate_xspec_confidence_interval( 'TAU2', bt.LABEL, alog10(bt.TAU2_MIN), alog10(bt.TAU2_MAX), (bt.NORM2/bt.NORM2) * alog10(bt.TAU2), alog10(bt.TAU2ERRL), alog10(bt.TAU2ERRU), bt.TAU2ERRS, bt.TAU2_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Tau2'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'Tau2_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'Tau2_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'Tau2_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Tau2_Flags', comment
  endif                  
  
  if tag_exist(bt,'TAU3', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    TAU3 = validate_xspec_confidence_interval( 'TAU3', bt.LABEL, alog10(bt.TAU3_MIN), alog10(bt.TAU3_MAX), (bt.NORM3/bt.NORM3) * alog10(bt.TAU3), alog10(bt.TAU3ERRL), alog10(bt.TAU3ERRU), bt.TAU3ERRS, bt.TAU3_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'               
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Tau3'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'Tau3_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'Tau3_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'Tau3_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Tau3_Flags', comment
  endif                  
  

  
  ;; Examine Photon Index results.
  if tag_exist(bt,'PH1', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    PH1 = validate_xspec_confidence_interval( 'PH1', bt.LABEL, bt.PH1_MIN, bt.PH1_MAX, bt.PH1, bt.PH1_ERRL, bt.PH1_ERRU, bt.PH1_ERRS, bt.PH1_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
              
    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Ph1'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Ph1_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Ph1_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Ph1_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Ph1_Flags', comment
  endif
 
  if tag_exist(bt,'PH7', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    PH7 = validate_xspec_confidence_interval( 'PH7', bt.LABEL,$
                                 bt.PH7_MIN,$
                                 bt.PH7_MAX,$
           (bt.NORM7/bt.NORM7) * bt.PH7,$
      tag_exist(bt,'PH7_ERRL') ? bt.PH7_ERRL : 0,$
      tag_exist(bt,'PH7_ERRU') ? bt.PH7_ERRU : 0,$
      tag_exist(bt,'PH7_ERRS') ? bt.PH7_ERRS : 0,$
      bt.PH7_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )

    tunit   = psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count)
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Ph7'      , comment      , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Ph7_LoLim', 'lower bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Ph7_HiLim', 'upper bound', TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Ph7_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Ph7_Flags', comment
  endif
 
  
  ;; Examine NORM results.
  if tag_exist(bt,'NORM1', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NORM1 = validate_xspec_confidence_interval( 'NORM1', bt.LABEL,$
                                 alog10(1E-20),$
                                 alog10(1E10 ),$
                                 alog10(bt.NORM1),$
      tag_exist(bt,'NOR1ERRL') ? alog10(bt.NOR1ERRL) : 0, $
      tag_exist(bt,'NOR1ERRU') ? alog10(bt.NOR1ERRU) : 0, $
      tag_exist(bt,'NOR1ERRS') ?        bt.NOR1ERRS  : 0, $
      bt.NOR1_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
    
    ; Convert NORM to emission measure for thermal models.
    par_str      = string(  em_offset + NORM1.param_value,  F='(%" %0.1f")')
    em1_range = '$'+par_str+'\phd'+NORM1.par_errl_str+NORM1.par_erru_str+'$'
  
    ; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
    ; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
    ; Thus, we must explicitly change any null cells to '\nodata'.
    ind = where(~finite(NORM1.param_value), count)
    if (count GT 0) then begin
      em1_range[ind] = '\nodata'
    endif
    
    ; Pad tails of strings with blanks to make them all the same length.
    str_length = strlen(em1_range)
    pad_length = max(str_length) - str_length
    pad = strmid('                          ', 0, pad_length)
    em1_range += pad
  
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Norm1'      ,         comment, TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm1_LoLim', 'lower bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm1_HiLim', 'upper bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Norm1_CI'   ,         comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'EmissionMeasure1_CI', comment, TUNIT='log(cm**(-3))'
    fxbaddcol, this_colnum, theader,  '', 'Norm1_Flags',         comment
  endif
  
  if tag_exist(bt,'NORM2', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NORM2 = validate_xspec_confidence_interval( 'NORM2', bt.LABEL, alog10(1E-20), alog10(1E10), alog10(bt.NORM2), $
      tag_exist(bt,'NOR2ERRL') ? alog10(bt.NOR2ERRL) : 0, $
      tag_exist(bt,'NOR2ERRU') ? alog10(bt.NOR2ERRU) : 0, $
      tag_exist(bt,'NOR2ERRS') ?        bt.NOR2ERRS  : 0, $
      bt.NOR2_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
    
    ; Convert NORM to emission measure for thermal models.
    par_str      = string(  em_offset + NORM2.param_value,  F='(%" %0.1f")')
    em2_range = '$'+par_str+'\phd'+NORM2.par_errl_str+NORM2.par_erru_str+'$'
  
    ; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
    ; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
    ; Thus, we must explicitly change any null cells to '\nodata'.
    ind = where(~finite(NORM2.param_value), count)
    if (count GT 0) then begin
      em2_range[ind] = '\nodata'
    endif
    
    ; Pad tails of strings with blanks to make them all the same length.
    str_length = strlen(em2_range)
    pad_length = max(str_length) - str_length
    pad = strmid('                          ', 0, pad_length)
    em2_range += pad
  
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Norm2'      ,         comment, TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm2_LoLim', 'lower bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm2_HiLim', 'upper bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Norm2_CI'   ,         comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'EmissionMeasure2_CI', comment, TUNIT='log(cm**(-3))'
    fxbaddcol, this_colnum, theader,  '', 'Norm2_Flags',         comment
  endif
  
  if tag_exist(bt,'NORM3', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NORM3 = validate_xspec_confidence_interval( 'NORM3', bt.LABEL, alog10(1E-20), alog10(1E10), alog10(bt.NORM3), $
      tag_exist(bt,'NOR3ERRL') ? alog10(bt.NOR3ERRL) : 0, $
      tag_exist(bt,'NOR3ERRU') ? alog10(bt.NOR3ERRU) : 0, $
      tag_exist(bt,'NOR3ERRS') ?        bt.NOR3ERRS  : 0, $
      bt.NOR3_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
    
    ; Convert NORM to emission measure for thermal models.
    par_str      = string(  em_offset + NORM3.param_value,  F='(%" %0.1f")')
    em3_range = '$'+par_str+'\phd'+NORM3.par_errl_str+NORM3.par_erru_str+'$'
  
    ; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
    ; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
    ; Thus, we must explicitly change any null cells to '\nodata'.
    ind = where(~finite(NORM3.param_value), count)
    if (count GT 0) then begin
      em3_range[ind] = '\nodata'
    endif
    
    ; Pad tails of strings with blanks to make them all the same length.
    str_length = strlen(em3_range)
    pad_length = max(str_length) - str_length
    pad = strmid('                          ', 0, pad_length)
    em3_range += pad
  
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Norm3'      ,         comment, TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm3_LoLim', 'lower bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm3_HiLim', 'upper bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Norm3_CI'   ,         comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'EmissionMeasure3_CI', comment, TUNIT='log(cm**(-3))'
    fxbaddcol, this_colnum, theader,  '', 'Norm3_Flags',         comment
  endif
  
  if tag_exist(bt,'NORM4', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NORM4 = validate_xspec_confidence_interval( 'NORM4', bt.LABEL, alog10(1E-20), alog10(1E10), alog10(bt.NORM4), $
      tag_exist(bt,'NOR4ERRL') ? alog10(bt.NOR4ERRL) : 0, $
      tag_exist(bt,'NOR4ERRU') ? alog10(bt.NOR4ERRU) : 0, $
      tag_exist(bt,'NOR4ERRS') ?        bt.NOR4ERRS  : 0, $
      bt.NOR4_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
    
    ; Convert NORM to emission measure for thermal models.
    par_str      = string(  em_offset + NORM4.param_value,  F='(%" %0.1f")')
    em4_range = '$'+par_str+'\phd'+NORM4.par_errl_str+NORM4.par_erru_str+'$'
  
    ; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
    ; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
    ; Thus, we must explicitly change any null cells to '\nodata'.
    ind = where(~finite(NORM4.param_value), count)
    if (count GT 0) then begin
      em4_range[ind] = '\nodata'
    endif
    
    ; Pad tails of strings with blanks to make them all the same length.
    str_length = strlen(em4_range)
    pad_length = max(str_length) - str_length
    pad = strmid('                          ', 0, pad_length)
    em4_range += pad
  
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Norm4'      ,         comment, TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm4_LoLim', 'lower bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm4_HiLim', 'upper bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Norm4_CI'   ,         comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'EmissionMeasure4_CI', comment, TUNIT='log(cm**(-3))'
    fxbaddcol, this_colnum, theader,  '', 'Norm4_Flags',         comment
  endif

  if tag_exist(bt,'NORM5', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NORM5 = validate_xspec_confidence_interval( 'NORM5', bt.LABEL, alog10(1E-20), alog10(1E10), alog10(bt.NORM5), $
      tag_exist(bt,'NOR5ERRL') ? alog10(bt.NOR5ERRL) : 0, $
      tag_exist(bt,'NOR5ERRU') ? alog10(bt.NOR5ERRU) : 0, $
      tag_exist(bt,'NOR5ERRS') ?        bt.NOR5ERRS  : 0, $
      bt.NOR5_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
    
    ; Convert NORM to emission measure for thermal models.
    par_str      = string(  em_offset + NORM5.param_value,  F='(%" %0.1f")')
    em5_range = '$'+par_str+'\phd'+NORM5.par_errl_str+NORM5.par_erru_str+'$'
  
    ; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
    ; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
    ; Thus, we must explicitly change any null cells to '\nodata'.
    ind = where(~finite(NORM5.param_value), count)
    if (count GT 0) then begin
      em5_range[ind] = '\nodata'
    endif
    
    ; Pad tails of strings with blanks to make them all the same length.
    str_length = strlen(em5_range)
    pad_length = max(str_length) - str_length
    pad = strmid('                          ', 0, pad_length)
    em5_range += pad
  
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Norm5'      ,         comment, TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm5_LoLim', 'lower bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm5_HiLim', 'upper bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Norm5_CI'   ,         comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'EmissionMeasure5_CI', comment, TUNIT='log(cm**(-3))'
    fxbaddcol, this_colnum, theader,  '', 'Norm5_Flags',         comment
  endif

  if tag_exist(bt,'NORM6', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NORM6 = validate_xspec_confidence_interval( 'NORM6', bt.LABEL, alog10(1E-20), alog10(1E10), alog10(bt.NORM6), $
      tag_exist(bt,'NOR6ERRL') ? alog10(bt.NOR6ERRL) : 0, $
      tag_exist(bt,'NOR6ERRU') ? alog10(bt.NOR6ERRU) : 0, $
      tag_exist(bt,'NOR6ERRS') ?        bt.NOR6ERRS  : 0, $
      bt.NOR6_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
    
    ; Convert NORM to emission measure for thermal models.
    par_str      = string(  em_offset + NORM6.param_value,  F='(%" %0.1f")')
    em6_range = '$'+par_str+'\phd'+NORM6.par_errl_str+NORM6.par_erru_str+'$'
  
    ; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
    ; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
    ; Thus, we must explicitly change any null cells to '\nodata'.
    ind = where(~finite(NORM6.param_value), count)
    if (count GT 0) then begin
      em6_range[ind] = '\nodata'
    endif
    
    ; Pad tails of strings with blanks to make them all the same length.
    str_length = strlen(em6_range)
    pad_length = max(str_length) - str_length
    pad = strmid('                          ', 0, pad_length)
    em6_range += pad
  
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Norm6'      ,         comment, TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm6_LoLim', 'lower bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm6_HiLim', 'upper bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Norm6_CI'   ,         comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'EmissionMeasure6_CI', comment, TUNIT='log(cm**(-3))'
    fxbaddcol, this_colnum, theader,  '', 'Norm6_Flags',         comment
  endif
  
  if tag_exist(bt,'NORM7', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    NORM7 = validate_xspec_confidence_interval( 'NORM7', bt.LABEL, alog10(1E-20), alog10(1E10), alog10(bt.NORM7), $
      tag_exist(bt,'NOR7ERRL') ? alog10(bt.NOR7ERRL) : 0, $
      tag_exist(bt,'NOR7ERRU') ? alog10(bt.NOR7ERRU) : 0, $
      tag_exist(bt,'NOR7ERRS') ?        bt.NOR7ERRS  : 0, $
      bt.NOR7_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=3, PLOT=plot, VERBOSE=verbose )
    
    ; Convert NORM to emission measure for thermal models.
    par_str      = string(  em_offset + NORM7.param_value,  F='(%" %0.1f")')
    em7_range = '$'+par_str+'\phd'+NORM7.par_errl_str+NORM7.par_erru_str+'$'
  
    ; Although the table generator would replace any 'NaN' strings with '\nodata', that mechanism won't work for these table cells
    ; because they are in LaTeX's math mode, and $\nodata$ is not legal syntax.
    ; Thus, we must explicitly change any null cells to '\nodata'.
    ind = where(~finite(NORM7.param_value), count)
    if (count GT 0) then begin
      em7_range[ind] = '\nodata'
    endif
    
    ; Pad tails of strings with blanks to make them all the same length.
    str_length = strlen(em7_range)
    pad_length = max(str_length) - str_length
    pad = strmid('                          ', 0, pad_length)
    em7_range += pad
  
    tunit   = 'log(' +psb_xpar( theader_in, string(1+in_colnum,F='(%"TUNIT%d")'), COUNT=count) +')'
    if (count EQ 0) then tunit = ''
    
    fxbaddcol, this_colnum, theader, 0.0, 'Norm7'      ,         comment, TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm7_LoLim', 'lower bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader, 0.0, 'Norm7_HiLim', 'upper bound'  , TUNIT=tunit, TDISP='F6.2'
    fxbaddcol, this_colnum, theader,  '', 'Norm7_CI'   ,         comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'EmissionMeasure7_CI', comment, TUNIT='log(cm**(-3))'
    fxbaddcol, this_colnum, theader,  '', 'Norm7_Flags',         comment
  endif



  abundance_min      =  0
  abundance_max      = 10
  abundance_CI       = strarr(num_sources)
  abundance_elements = !NULL
  fxbaddcol, this_colnum, theader,  '', 'Abundance_CI'   , 'conf intervals for abundances'
  tunit   = 'abundance'
  
  ; Examine O results.
  if (total(/INT, strmatch(abundances_to_report, 'O')) GT 0) && tag_exist(bt,'O_ERRS', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
      O = validate_xspec_confidence_interval( 'O', bt.LABEL, abundance_min, abundance_max, bt.O, $
      tag_exist(bt,'O_ERRL') ? bt.O_ERRL : 0, $
      tag_exist(bt,'O_ERRU') ? bt.O_ERRU : 0, $
      tag_exist(bt,'O_ERRS') ? bt.O_ERRS : 0, $
      bt.O_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
      abundance_CI       += ' & '+O.param_range
      abundance_elements  = [abundance_elements, 'O']
  
    fxbaddcol, this_colnum, theader, 0.0, 'O'      ,       comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'O_LoLim', 'lower bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'O_HiLim', 'upper bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'O_CI'   ,       comment, TUNIT=tunit
  endif
    
    
  ;; Examine Ne results.
  if (total(/INT, strmatch(abundances_to_report, 'Ne')) GT 0) && tag_exist(bt,'_NE') then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
      ; Variables and structure tags cannot have the name "NE" in IDL, thus "_NE" is used instead.
      _NE = validate_xspec_confidence_interval( 'NE', bt.LABEL, abundance_min, abundance_max, bt._NE, $
      tag_exist(bt,'NE_ERRL') ? bt.NE_ERRL : 0, $
      tag_exist(bt,'NE_ERRU') ? bt.NE_ERRU : 0, $
      tag_exist(bt,'NE_ERRS') ? bt.NE_ERRS : 0, $
      bt.NE_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
      abundance_CI       += ' & '+_NE.param_range
      abundance_elements  = [abundance_elements, 'Ne']
  
    ; We cannot use the name 'Ne' because that is a reserved word in IDL.
    fxbaddcol, this_colnum, theader, 0.0, 'Neon'    ,       comment, TUNIT=tunit  
    fxbaddcol, this_colnum, theader, 0.0, 'Neon_LoLim', 'lower bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'Neon_HiLim', 'upper bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Neon_CI'   ,       comment, TUNIT=tunit
  endif
    
    
  ;; Examine Mg results.
  if (total(/INT, strmatch(abundances_to_report, 'Mg')) GT 0) && tag_exist(bt,'MG') then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
      MG = validate_xspec_confidence_interval( 'MG', bt.LABEL, abundance_min, abundance_max, bt.MG, $
      tag_exist(bt,'MG_ERRL') ? bt.MG_ERRL : 0, $
      tag_exist(bt,'MG_ERRU') ? bt.MG_ERRU : 0, $
      tag_exist(bt,'MG_ERRS') ? bt.MG_ERRS : 0, $ 
      bt.MG_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
      abundance_CI       += ' & '+MG.param_range
      abundance_elements  = [abundance_elements, 'Mg']
  
    fxbaddcol, this_colnum, theader, 0.0, 'Mg'      ,       comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'Mg_LoLim', 'lower bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'Mg_HiLim', 'upper bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Mg_CI'   ,       comment, TUNIT=tunit
  endif
    
    
  ;; Examine Si results.
  if (total(/INT, strmatch(abundances_to_report, 'Si')) GT 0) && tag_exist(bt,'SI') then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
      SI = validate_xspec_confidence_interval( 'SI', bt.LABEL, abundance_min, abundance_max, bt.SI, $
      tag_exist(bt,'SI_ERRL') ? bt.SI_ERRL : 0, $
      tag_exist(bt,'SI_ERRU') ? bt.SI_ERRU : 0, $
      tag_exist(bt,'SI_ERRS') ? bt.SI_ERRS : 0, $ 
      bt.SI_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
      abundance_CI       += ' & '+SI.param_range
      abundance_elements  = [abundance_elements, 'Si']
  
    fxbaddcol, this_colnum, theader, 0.0, 'Si'      ,       comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'Si_LoLim', 'lower bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'Si_HiLim', 'upper bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Si_CI'   ,       comment, TUNIT=tunit
  endif
    
    
  ;; Examine S results.
  if (total(/INT, strmatch(abundances_to_report, 'S')) GT 0) && tag_exist(bt,'S') then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
      S = validate_xspec_confidence_interval( 'S', bt.LABEL, abundance_min, abundance_max, bt.S, $
      tag_exist(bt,'S_ERRL') ? bt.S_ERRL : 0, $
      tag_exist(bt,'S_ERRU') ? bt.S_ERRU : 0, $
      tag_exist(bt,'S_ERRS') ? bt.S_ERRS : 0, $
      bt.S_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
      abundance_CI       += ' & '+S.param_range
      abundance_elements  = [abundance_elements, 'S']
  
    fxbaddcol, this_colnum, theader, 0.0, 'S'      ,       comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'S_LoLim', 'lower bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'S_HiLim', 'upper bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'S_CI'   ,       comment, TUNIT=tunit
  endif
    
    
  ;; Examine Fe results.
  if (total(/INT, strmatch(abundances_to_report, 'Fe')) GT 0) && tag_exist(bt,'FE') then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
      FE = validate_xspec_confidence_interval( 'FE', bt.LABEL, abundance_min, abundance_max, bt.FE, $
      tag_exist(bt,'FE_ERRL') ? bt.FE_ERRL : 0, $
      tag_exist(bt,'FE_ERRU') ? bt.FE_ERRU : 0, $
      tag_exist(bt,'FE_ERRS') ? bt.FE_ERRS : 0, $
      bt.FE_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, PLOT=plot, VERBOSE=verbose )
      abundance_CI       += ' & '+FE.param_range
      abundance_elements  = [abundance_elements, 'Fe']
  
    fxbaddcol, this_colnum, theader, 0.0, 'Fe'      ,       comment, TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'Fe_LoLim', 'lower bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader, 0.0, 'Fe_HiLim', 'upper bound', TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'Fe_CI'   ,       comment, TUNIT=tunit
  endif
    
    
  ;; Examine Redshift results.
  if tag_exist(bt,'RS', INDEX=in_colnum) then begin
    ; Validate the parameter confidence interval produced by the fitting script.
    ; Confidence limits that should be ignored are set to NaN by the routine validate_xspec_confidence_interval.
    RS = validate_xspec_confidence_interval( 'RS', bt.LABEL, bt.RS_MIN, bt.RS_MAX, bt.RS, bt.RS_ERRL, bt.RS_ERRU, bt.RS_ERRS, bt.RS_FZ, HIDE_ERRORS_FLAG=hide_errors_flag, SIGNIFICANT_DIGITS=2, PLOT=plot, VERBOSE=verbose )
  
    comment = 'from XSPEC'
    tunit   = '?'
    fxbaddcol, this_colnum, theader, 0.0, 'RS'      , comment      , TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'RS_LoLim', 'lower bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader, 0.0, 'RS_HiLim', 'upper bound', TUNIT=tunit, TDISP='G8.3'
    fxbaddcol, this_colnum, theader,  '', 'RS_CI'   , comment      , TUNIT=tunit
    fxbaddcol, this_colnum, theader,  '', 'RS_Flags', comment
  endif
  
    
  
  ;; Convert XSPEC flux to luminosity.
  ;; Various energy bands and naming conventions are in use.
  
  tunit   = keyword_set(diffuse) ? 'log(erg /s /pc**2)' : 'log(erg /s)'

  template_row = {lumin, FITS_name:'', xspec_name:strarr(3), comment:'', log_lumin:null_vector}
  luminosities = replicate(template_row, 100)
  ii=0
  ; Apparent
  ; complete model
  comment = string(distance/1000., F='(%"apparent;%0.3f kpc")')               
  luminosities[ii++] = {lumin, 'FitLuminosity_t', ['FH8','FH7','F0P5_8'], comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_s', ['FH2',''   ,'F0P5_2'], comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_h', ['F28','F27','F2_8'  ], comment,null_vector}
  ; components     
  luminosities[ii++] = {lumin, 'FitLuminosity_1t', ['F1H8','F1H7',''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2t', ['F2H8','F2H7',''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3t', ['F3H8','F3H7',''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4t', ['F4H8','F4H7',''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5t', ['F5H8','F5H7',''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6t', ['F6H8','F6H7',''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7t', ['F7H8','F7H7',''], '#7;'+comment,null_vector}
                   
  luminosities[ii++] = {lumin, 'FitLuminosity_1s', ['F1H2',''    ,''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2s', ['F2H2',''    ,''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3s', ['F3H2',''    ,''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4s', ['F4H2',''    ,''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5s', ['F5H2',''    ,''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6s', ['F6H2',''    ,''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7s', ['F7H2',''    ,''], '#7;'+comment,null_vector}
                   
  luminosities[ii++] = {lumin, 'FitLuminosity_1h', ['F128','F127',''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2h', ['F228','F227',''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3h', ['F328','F327',''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4h', ['F428','F427',''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5h', ['F528','F527',''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6h', ['F628','F627',''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7h', ['F728','F727',''], '#7;'+comment,null_vector}


  ; ISM-corrected
  ; complete model
  comment = string(distance/1000., F='(%"ISM-corrected;%0.3f kpc")')               
  luminosities[ii++] = {lumin, 'FitLuminosity_tl', ['FLH8','FLH7','FL0P5_8'], comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_sl', ['FLH2',''    ,'FL0P5_2'], comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_hl', ['FL28','FL27','FL2_8'  ], comment,null_vector}
  ; components     
  luminosities[ii++] = {lumin, 'FitLuminosity_1tl', ['F1LH8','F1LH7',''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2tl', ['F2LH8','F2LH7',''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3tl', ['F3LH8','F3LH7',''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4tl', ['F4LH8','F4LH7',''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5tl', ['F5LH8','F5LH7',''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6tl', ['F6LH8','F6LH7',''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7tl', ['F7LH8','F7LH7',''], '#7;'+comment,null_vector}
                   
  luminosities[ii++] = {lumin, 'FitLuminosity_1sl', ['F1LH2',''    ,''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2sl', ['F2LH2',''    ,''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3sl', ['F3LH2',''    ,''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4sl', ['F4LH2',''    ,''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5sl', ['F5LH2',''    ,''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6sl', ['F6LH2',''    ,''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7sl', ['F7LH2',''    ,''], '#7;'+comment,null_vector}
                   
  luminosities[ii++] = {lumin, 'FitLuminosity_1hl', ['F1L28','F1L27',''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2hl', ['F2L28','F2L27',''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3hl', ['F3L28','F3L27',''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4hl', ['F4L28','F4L27',''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5hl', ['F5L28','F5L27',''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6hl', ['F6L28','F6L27',''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7hl', ['F7L28','F7L27',''], '#7;'+comment,null_vector}

  ; Absorption-corrected
  ; complete model
  comment = string(distance/1000., F='(%"absorption-corrected;%0.3f kpc")')               
  luminosities[ii++] = {lumin, 'FitLuminosity_tc', ['FCH8','FCH7','FC0P5_8'], comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_sc', ['FCH2',''    ,'FC0P5_2'], comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_hc', ['FC28','FC27','FC2_8'  ], comment,null_vector}
  ; components     
  luminosities[ii++] = {lumin, 'FitLuminosity_1tc', ['F1CH8','F1CH7',''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2tc', ['F2CH8','F2CH7',''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3tc', ['F3CH8','F3CH7',''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4tc', ['F4CH8','F4CH7',''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5tc', ['F5CH8','F5CH7',''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6tc', ['F6CH8','F6CH7',''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7tc', ['F7CH8','F7CH7',''], '#7;'+comment,null_vector}
                   
  luminosities[ii++] = {lumin, 'FitLuminosity_1sc', ['F1CH2',''    ,''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2sc', ['F2CH2',''    ,''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3sc', ['F3CH2',''    ,''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4sc', ['F4CH2',''    ,''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5sc', ['F5CH2',''    ,''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6sc', ['F6CH2',''    ,''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7sc', ['F7CH2',''    ,''], '#7;'+comment,null_vector}
                   
  luminosities[ii++] = {lumin, 'FitLuminosity_1hc', ['F1C28','F1C27',''], '#1;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_2hc', ['F2C28','F2C27',''], '#2;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_3hc', ['F3C28','F3C27',''], '#3;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_4hc', ['F4C28','F4C27',''], '#4;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_5hc', ['F5C28','F5C27',''], '#5;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_6hc', ['F6C28','F6C27',''], '#6;'+comment,null_vector}
  luminosities[ii++] = {lumin, 'FitLuminosity_7hc', ['F7C28','F7C27',''], '#7;'+comment,null_vector}


  for ii=0, n_elements(luminosities)-1 do begin
    ; Fetch this luminosity structure.
    this_lumin = luminosities[ii]

    if (this_lumin.FITS_name EQ '') then continue

    foreach this_xspec_name, this_lumin.xspec_name do begin
      if (this_xspec_name EQ '') then continue

      if tag_exist(bt, this_xspec_name, INDEX=in_colnum, /QUIET) then begin
        this_lumin.log_lumin = lx_offset+alog10( bt.(in_colnum) )

        fxbaddcol, this_colnum, theader, 0.0, this_lumin.FITS_name , this_lumin.comment, TUNIT=tunit, TDISP='F6.2'

        if keyword_set(plot) then dataset_1d, id_lumin, LINE=0, this_lumin.log_lumin, DATASET=this_lumin.FITS_name, BINSIZE=0.2, XTIT=tunit, WIDGET_TITLE='Luminosity'
        
        break
      endif
    endforeach
    
    ; Save this modified luminosity structure.
    luminosities[ii] = this_lumin
  endfor ; ii


  help, distance, dscale, lx_offset  
endif ; keyword_set(distance)


    
;;==========================================================================
;; Create an empty output table.        
tag_types = strmid(strtrim(psb_xpar( theader, 'TFORM*'),2), 0, 1, /REVERSE_OFFSET)

create_struct, bt_out, '', strtrim(psb_xpar( theader, 'TTYPE*'),2), strjoin(tag_types, ','), DIMEN=num_sources


;;==========================================================================
;; Populate the columns in the output table.

; While populating IDL structure tags, we use 0-based column numbers.
out_colnum = 0

; Populate the Seq column.
bt_out.(out_colnum++) = 1+indgen(num_sources)

;; ------------------------------------------------------------------------
;; Assign all the direct associations between columns in the collated table and columns in the flattened table.
for ii=0,n_elements(col_assignment)-1 do begin
  this = col_assignment[ii]
  
  if ~tag_exist(bt, this.old_name, INDEX=in_colnum, /QUIET) then continue
  print, this.old_name, this.new_name, F='(%"%25s-->%25s")'

  if (this.multiband) then begin
    bt_out.(out_colnum++) = bt.(in_colnum)[band_total]
    bt_out.(out_colnum++) = bt.(in_colnum)[band_soft]
    bt_out.(out_colnum++) = bt.(in_colnum)[band_hard]
  endif else begin
    temp = bt.(in_colnum)[0]                            
    bt_out.(out_colnum++) = (size(temp, /TNAME) EQ 'STRING') ? strtrim(bt.(in_colnum)[0],2) : temp
  endelse
endfor ;ii

; Our afterglow screening algorithm sufferes from false positives for bright sources.
; This has not been calibrated, so we must guess at an appropriate way to suppress reporting these.
if tag_exist(bt_out,'AfterglowFraction') && tag_exist(bt_out,'SrcCounts_t') then begin
  number_of_afterglow_events = bt_out.SrcCounts_t * bt_out.AfterglowFraction
  ind = where(number_of_afterglow_events GE 20, count)
  if (count GT 0) then bt_out[ind].AfterglowFraction = null_val
endif


;; ------------------------------------------------------------------------
;; Assign columns reporting photometry confidence intervals from aprates tool.
temp_text_fn = tempdir + 'aprates_out.par'
band_number = [band_total, band_soft, band_hard]
suffix      = [     't'  ,      's'     ,      'h'      ]
for jj=0,2 do begin
  dum = tag_exist(bt_out, 'NetCounts_Lo_'+suffix[jj], INDEX=ind_Lo)
  dum = tag_exist(bt_out, 'NetCounts_Hi_'+suffix[jj], INDEX=ind_Hi)                          
    
  if keyword_set(fast) then begin
    ; Use the AE error estimates on NET_CNTS to build confidence interval. 
    NET_CNTS           =  0 > bt.NET_CNTS          [band_number[jj]]
    NET_CNTS_SIGMA_LOW =  0 > bt.NET_CNTS_SIGMA_LOW[band_number[jj]]
    NET_CNTS_SIGMA_UP  =  0 > bt.NET_CNTS_SIGMA_UP [band_number[jj]] 
    
    bt_out.(ind_Lo) =  NET_CNTS - NET_CNTS_SIGMA_LOW
    bt_out.(ind_Hi) =  NET_CNTS + NET_CNTS_SIGMA_UP 
  endif else begin
    for ii=0L,num_sources-1 do begin
      BACKSCAL =  bt[ii].BACKSCAL[band_number[jj]]
      SRC_CNTS =  bt[ii].SRC_CNTS[band_number[jj]]
      BKG_CNTS =  bt[ii].BKG_CNTS[band_number[jj]]
      NET_CNTS =  bt[ii].NET_CNTS[band_number[jj]]
                              
      if finite(BACKSCAL) && finite(SRC_CNTS) && finite(BKG_CNTS) then begin
        ; The value supplied for max_counts (50) is advice received from the CXC through Helpdesk ticket #13525, circa 2011 June.
        ; See extensive testing by Frank Primini in email 2011 June 29.
        run_command, string(temp_text_fn, $
                            BACKSCAL, SRC_CNTS, BKG_CNTS, confidence_level, $
                            F="(%'aprates T_s=1 E_s=1 eng_s=1 flux_s=1 T_b=1 E_b=1 eng_b=1 flux_b=1 outfile=%s alpha=1.0 beta=0.0 max_counts=50  A_s=1 A_b=%0.5f n=%d m=%d conf=%0.3f clob+')")
         
        run_command, /QUIET, string(temp_text_fn, F="(%'pget %s src_cnts_err_lo src_cnts_err_up src_cnts')"), result
      endif else begin
        print, 'WARNING: '+bt[ii].CATALOG_NAME+' is missing some photometry data for band "'+suffix[jj]+'"'
        result = replicate('NaN',3)
      endelse
      
      result = repstr(result, 'INDEF', 'NaN')
      bt_out[ii].(ind_Lo) = float(result[0])  
      bt_out[ii].(ind_Hi) = float(result[1])  
      
      ; Defensively verify that NET_CNTS computed by aprates matches the AE value.
      temp_diff  = NET_CNTS - float(result[2])
      temp_ratio = NET_CNTS / float(result[2])
      if finite(temp_ratio) && (temp_ratio LT 0.99 || temp_ratio GT 1.01) && finite(temp_diff) && (abs(temp_diff) GT 1) then begin
        print, NET_CNTS, float(result[2])
        message, 'ERROR: AE and aprates disagree on NET_CNTS!'
      endif
      endfor ; ii
  endelse  
endfor ; jj

;; ------------------------------------------------------------------------
;; Estimate energy fluxes.
if tag_exist(bt_out,'EnergyFlux_t') then begin
  ; Conversion from photon flux to energy flux can be done in several ways.  
  ; Using MEAN energy should have low bias, but high noise; using MEDIAN should have lower noise but more bias.
  ; The sum of soft and hard fluxes should be less biased than AE's wide-band FLUX2 value.
  erg_per_kev = 1.602E-09
  bt_out.EnergyFlux_s = erg_per_kev * (bt.FLUX2[band_soft ] * bt_out.MedianEnergy_s) 
  bt_out.EnergyFlux_h = erg_per_kev * (bt.FLUX2[band_hard] * bt_out.MedianEnergy_h) 
  
  ; We must handle bands where PhotonFlux=0 and MedianE is NaN.
  bt_out.EnergyFlux_t = fltarr(num_sources)
  ind =                                    where(finite(bt_out.EnergyFlux_s), count)
  if (count GT 0) then bt_out[ind].EnergyFlux_t += bt_out[ind].EnergyFlux_s
  
  ind =                                    where(finite(bt_out.EnergyFlux_h), count)
  if (count GT 0) then bt_out[ind].EnergyFlux_t += bt_out[ind].EnergyFlux_h
endif


;; ------------------------------------------------------------------------
;; Assign columns derived from spectral fitting results
if tag_exist(bt_out,'SpModel'             ) then bt_out.SpModel              = bt.MODEL
if tag_exist(bt_out,'SpModelIsProvisional') then bt_out.SpModelIsProvisional = bt.PROVISNL
if tag_exist(bt_out,'SpModelIsHandFit'    ) then bt_out.SpModelIsHandFit     = strtrim(bt.HANDFIT,2)
if tag_exist(bt_out,'ReducedChiSq'        ) then bt_out.ReducedChiSq         = bt.CHI_SQR
if tag_exist(bt_out,'DegreesOfFreedom'    ) then bt_out.DegreesOfFreedom     = bt.DOF
if tag_exist(bt_out,'CStatistic'          ) then bt_out.CStatistic           = bt.CSTAT
                                          
if tag_exist(bt_out,'Abundance_CI'        ) then begin
  ; Add empty columns to Abundance_CI, as needed, to get num_abundance_columns columns.
  num_empty_columns = num_abundance_columns - n_elements(abundance_elements)
  for kk=1,num_empty_columns do abundance_CI += '& '
  
                                                 bt_out.Abundance_CI         = abundance_CI
  
  cd, CURRENT=cwd
  if n_elements(abundance_elements) EQ 0 then $
    print, file_which('/hmsfr_tables.tex'),cwd, F='(%"\nCOPY %s to %s and EDIT to hide all abundance columns (using the \"h\" column type and \\nocolhead command).\n")' $
  else $
    print, file_which('/hmsfr_tables.tex'),cwd, strjoin(abundance_elements,' '), F='(%"\nCOPY %s to %s and EDIT to hide the abundance columns for elements other than %s (using the \"h\" column type and \\nocolhead command).\n")'
endif

if tag_exist(bt_out,'LMC') then begin
  bt_out.LMC       = LMC.param_value
  bt_out.LMC_LoLim = LMC.lower_confidence_limit
  bt_out.LMC_HiLim = LMC.upper_confidence_limit
  bt_out.LMC_CI    = LMC.param_range
  bt_out.LMC_Flags = LMC.anom_flags
endif

if tag_exist(bt_out,'NH1') then begin
  bt_out.NH1       = NH1.param_value
  bt_out.NH1_LoLim = NH1.lower_confidence_limit
  bt_out.NH1_HiLim = NH1.upper_confidence_limit
  bt_out.NH1_CI    = NH1.param_range
  bt_out.NH1_Flags = NH1.anom_flags
endif

if tag_exist(bt_out,'NH2') then begin
  bt_out.NH2       = NH2.param_value
  bt_out.NH2_LoLim = NH2.lower_confidence_limit
  bt_out.NH2_HiLim = NH2.upper_confidence_limit
  bt_out.NH2_CI    = NH2.param_range
  bt_out.NH2_Flags = NH2.anom_flags
endif

if tag_exist(bt_out,'NH3') then begin
  bt_out.NH3       = NH3.param_value
  bt_out.NH3_LoLim = NH3.lower_confidence_limit
  bt_out.NH3_HiLim = NH3.upper_confidence_limit
  bt_out.NH3_CI    = NH3.param_range
  bt_out.NH3_Flags = NH3.anom_flags
endif

if tag_exist(bt_out,'NH4') then begin
  bt_out.NH4       = NH4.param_value
  bt_out.NH4_LoLim = NH4.lower_confidence_limit
  bt_out.NH4_HiLim = NH4.upper_confidence_limit
  bt_out.NH4_CI    = NH4.param_range
  bt_out.NH4_Flags = NH4.anom_flags
endif

if tag_exist(bt_out,'NH5') then begin
  bt_out.NH5       = NH5.param_value
  bt_out.NH5_LoLim = NH5.lower_confidence_limit
  bt_out.NH5_HiLim = NH5.upper_confidence_limit
  bt_out.NH5_CI    = NH5.param_range
  bt_out.NH5_Flags = NH5.anom_flags
endif

if tag_exist(bt_out,'NH6') then begin
  bt_out.NH6       = NH6.param_value
  bt_out.NH6_LoLim = NH6.lower_confidence_limit
  bt_out.NH6_HiLim = NH6.upper_confidence_limit
  bt_out.NH6_CI    = NH6.param_range
  bt_out.NH6_Flags = NH6.anom_flags
endif

if tag_exist(bt_out,'NH7') then begin
  bt_out.NH7       = NH7.param_value
  bt_out.NH7_LoLim = NH7.lower_confidence_limit
  bt_out.NH7_HiLim = NH7.upper_confidence_limit
  bt_out.NH7_CI    = NH7.param_range
  bt_out.NH7_Flags = NH7.anom_flags
endif



if tag_exist(bt_out,'kT1') then begin
  bt_out.KT1       = KT1.param_value
  bt_out.KT1_LoLim = KT1.lower_confidence_limit
  bt_out.KT1_HiLim = KT1.upper_confidence_limit
  bt_out.KT1_CI    = KT1.param_range
  bt_out.KT1_Flags = KT1.anom_flags
endif

if tag_exist(bt_out,'kt2') then begin
  bt_out.KT2       = KT2.param_value
  bt_out.KT2_LoLim = KT2.lower_confidence_limit
  bt_out.KT2_HiLim = KT2.upper_confidence_limit
  bt_out.KT2_CI    = KT2.param_range
  bt_out.KT2_Flags = KT2.anom_flags
endif

if tag_exist(bt_out,'kt3') then begin
  bt_out.KT3       = KT3.param_value
  bt_out.KT3_LoLim = KT3.lower_confidence_limit
  bt_out.KT3_HiLim = KT3.upper_confidence_limit
  bt_out.KT3_CI    = KT3.param_range
  bt_out.KT3_Flags = KT3.anom_flags
endif

if tag_exist(bt_out,'kt4') then begin
  bt_out.KT4       = KT4.param_value
  bt_out.KT4_LoLim = KT4.lower_confidence_limit
  bt_out.KT4_HiLim = KT4.upper_confidence_limit
  bt_out.KT4_CI    = KT4.param_range
  bt_out.KT4_Flags = KT4.anom_flags
endif

if tag_exist(bt_out,'kt5') then begin
  bt_out.KT5       = KT5.param_value
  bt_out.KT5_LoLim = KT5.lower_confidence_limit
  bt_out.KT5_HiLim = KT5.upper_confidence_limit
  bt_out.KT5_CI    = KT5.param_range
  bt_out.KT5_Flags = KT5.anom_flags
endif

if tag_exist(bt_out,'kt6') then begin
  bt_out.KT6       = KT6.param_value
  bt_out.KT6_LoLim = KT6.lower_confidence_limit
  bt_out.KT6_HiLim = KT6.upper_confidence_limit
  bt_out.KT6_CI    = KT6.param_range
  bt_out.KT6_Flags = KT6.anom_flags
endif


if tag_exist(bt_out,'Tau1') then begin
  bt_out.Tau1       = TAU1.param_value
  bt_out.Tau1_LoLim = TAU1.lower_confidence_limit
  bt_out.Tau1_HiLim = TAU1.upper_confidence_limit
  bt_out.Tau1_CI    = TAU1.param_range
  bt_out.Tau1_Flags = TAU1.anom_flags
endif

if tag_exist(bt_out,'Tau2') then begin
  bt_out.Tau2       = TAU2.param_value
  bt_out.Tau2_LoLim = TAU2.lower_confidence_limit
  bt_out.Tau2_HiLim = TAU2.upper_confidence_limit
  bt_out.Tau2_CI    = TAU2.param_range
  bt_out.Tau2_Flags = TAU2.anom_flags
endif

if tag_exist(bt_out,'Tau3') then begin
  bt_out.Tau3       = TAU3.param_value
  bt_out.Tau3_LoLim = TAU3.lower_confidence_limit
  bt_out.Tau3_HiLim = TAU3.upper_confidence_limit
  bt_out.Tau3_CI    = TAU3.param_range
  bt_out.Tau3_Flags = TAU3.anom_flags
endif


if tag_exist(bt_out,'Ph1') then begin
  bt_out.Ph1       = Ph1.param_value
  bt_out.Ph1_LoLim = Ph1.lower_confidence_limit
  bt_out.Ph1_HiLim = Ph1.upper_confidence_limit
  bt_out.Ph1_CI    = Ph1.param_range
  bt_out.Ph1_Flags = Ph1.anom_flags
endif

if tag_exist(bt_out,'Ph7') then begin
  bt_out.Ph7       = Ph7.param_value
  bt_out.Ph7_LoLim = Ph7.lower_confidence_limit
  bt_out.Ph7_HiLim = Ph7.upper_confidence_limit
  bt_out.Ph7_CI    = Ph7.param_range
  bt_out.Ph7_Flags = Ph7.anom_flags
endif

if tag_exist(bt_out,'Norm1') then begin
  bt_out.Norm1               = Norm1.param_value
  bt_out.Norm1_LoLim         = Norm1.lower_confidence_limit
  bt_out.Norm1_HiLim         = Norm1.upper_confidence_limit
  bt_out.Norm1_CI            = Norm1.param_range
  bt_out.EmissionMeasure1_CI = em1_range
  bt_out.Norm1_Flags         = Norm1.anom_flags
endif

if tag_exist(bt_out,'Norm2') then begin
  bt_out.Norm2               = Norm2.param_value
  bt_out.Norm2_LoLim         = Norm2.lower_confidence_limit
  bt_out.Norm2_HiLim         = Norm2.upper_confidence_limit
  bt_out.Norm2_CI            = Norm2.param_range
  bt_out.EmissionMeasure2_CI = em2_range
  bt_out.Norm2_Flags         = Norm2.anom_flags
endif

if tag_exist(bt_out,'Norm3') then begin
  bt_out.Norm3               = Norm3.param_value
  bt_out.Norm3_LoLim         = Norm3.lower_confidence_limit
  bt_out.Norm3_HiLim         = Norm3.upper_confidence_limit
  bt_out.Norm3_CI            = Norm3.param_range
  bt_out.EmissionMeasure3_CI = em3_range
  bt_out.Norm3_Flags         = Norm3.anom_flags
endif

if tag_exist(bt_out,'Norm5') then begin
  bt_out.Norm5               = Norm5.param_value
  bt_out.Norm5_LoLim         = Norm5.lower_confidence_limit
  bt_out.Norm5_HiLim         = Norm5.upper_confidence_limit
  bt_out.Norm5_CI            = Norm5.param_range
  bt_out.EmissionMeasure5_CI = em5_range
  bt_out.Norm5_Flags         = Norm5.anom_flags
endif

if tag_exist(bt_out,'Norm6') then begin
  bt_out.Norm6               = Norm6.param_value
  bt_out.Norm6_LoLim         = Norm6.lower_confidence_limit
  bt_out.Norm6_HiLim         = Norm6.upper_confidence_limit
  bt_out.Norm6_CI            = Norm6.param_range
  bt_out.EmissionMeasure6_CI = em6_range
  bt_out.Norm6_Flags         = Norm6.anom_flags
endif

if tag_exist(bt_out,'Norm7') then begin
  bt_out.Norm7               = Norm7.param_value
  bt_out.Norm7_LoLim         = Norm7.lower_confidence_limit
  bt_out.Norm7_HiLim         = Norm7.upper_confidence_limit
  bt_out.Norm7_CI            = Norm7.param_range
  bt_out.EmissionMeasure7_CI = em7_range
  bt_out.Norm7_Flags         = Norm7.anom_flags
endif



if tag_exist(bt_out,'O') then begin
  bt_out.O       = O.param_value
  bt_out.O_LoLim = O.lower_confidence_limit
  bt_out.O_HiLim = O.upper_confidence_limit
  bt_out.O_CI    = O.param_range
endif


if tag_exist(bt_out,'Neon') then begin
  bt_out.Neon       = _Ne.param_value
  bt_out.Neon_LoLim = _Ne.lower_confidence_limit
  bt_out.Neon_HiLim = _Ne.upper_confidence_limit
  bt_out.Neon_CI    = _Ne.param_range
endif

if tag_exist(bt_out,'Mg') then begin
  bt_out.Mg       = Mg.param_value
  bt_out.Mg_LoLim = Mg.lower_confidence_limit
  bt_out.Mg_HiLim = Mg.upper_confidence_limit
  bt_out.Mg_CI    = Mg.param_range
endif

if tag_exist(bt_out,'S') then begin
  bt_out.S       = S.param_value
  bt_out.S_LoLim = S.lower_confidence_limit
  bt_out.S_HiLim = S.upper_confidence_limit
  bt_out.S_CI    = S.param_range
endif

if tag_exist(bt_out,'Si') then begin
  bt_out.Si       = Si.param_value
  bt_out.Si_LoLim = Si.lower_confidence_limit
  bt_out.Si_HiLim = Si.upper_confidence_limit
  bt_out.Si_CI    = Si.param_range
endif

if tag_exist(bt_out,'Fe') then begin
  bt_out.Fe       = Fe.param_value
  bt_out.Fe_LoLim = Fe.lower_confidence_limit
  bt_out.Fe_HiLim = Fe.upper_confidence_limit
  bt_out.Fe_CI    = Fe.param_range
endif


if tag_exist(bt_out,'RS') then begin
  bt_out.RS       = RS.param_value
  bt_out.RS_LoLim = RS.lower_confidence_limit
  bt_out.RS_HiLim = RS.upper_confidence_limit
  bt_out.RS_CI    = RS.param_range
  bt_out.RS_Flags = RS.anom_flags
endif



for ii=0, n_elements(luminosities)-1 do begin
  ; Fetch this luminosity structure.
  this_lumin = luminosities[ii]

  if (this_lumin.FITS_name EQ '') then continue

  if tag_exist(bt_out, this_lumin.FITS_name, INDEX=out_colnum, /QUIET) then $
    bt_out.(out_colnum) = this_lumin.log_lumin
endfor ; ii





if ~keyword_set(flat_table_filename) then GOTO, CLEANUP

writefits,   flat_table_filename, 0, headfits(collated_filename)

psb_xaddpar, theader, 'CREATOR', psb_xpar( theader_in, 'CREATOR') + '; ' + creator_string
; /NO_TYPES preserves mixed-case TTYPE keywords that I have put in header myself.
mwrfits, bt_out, flat_table_filename, theader, /NO_TYPES 


;; Create ASCII version.
fdecomp, flat_table_filename, disk, item_path, item_name, item_qual
ascii_filename = item_path + item_name + '.txt'

byte_table = readfits(flat_table_filename, theader, EXT=1)
tbprint, theader, byte_table, 1+indgen(psb_xpar( theader,'TFIELDS')), TEXTOUT=ascii_filename, NUM_HEADER_LINES=4

;ftab_print, flat_table_filename, 1+indgen(n_tags(bt_out)), TEXTOUT=ascii_filename

CLEANUP:
if file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif

print, 'ae_flatten_collation finished'
return
end ; ae_flatten_collation




;;; ==========================================================================
;;; hmsfr_tables
;;; ==========================================================================

;;; The parameter "DISTANCE" is used only for diffuse source; it must be in units of pc.
;;;
;;; SRC_SIGNIF_MIN and/or NET_COUNTS_MIN can be supplied as a threshold against the SRC_SIGNIF and/or NET_CNTS column to omit sources from the spectroscopy tables.

;;; VVVVVVVVVVVVVVVVVVVV OBSOLETE  OBSOLETE  OBSOLETE  OPTION VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;;; One of the table columns computed in this code is an "Effective Exposure" time for each source, 
;;; which is intended to convey the "depth" to which that source was observed in terms of some 
;;; reference configuration of the observatory (position on the detector and epoch of calibration).
;;; Authors are free to choose any reference configuration desired; a convenient choice is the PIMMS effective area
;;; (http://asc.harvard.edu/cgi-bin/build_viewer.cgi?ea) for a specific Chandra proposal cycle.

;;; Recall that for an exposure map computed at the monoenergy E
;;;   emap(x,y) = ARF(x,y,E) * EXPOSURE
;;;
;;; Thus, we choose to express the actual emap value for each source as a product of some reference effective area 
;;; and an effective exposure time:                                                         
;;;   emap(x,y) = ARF(PIMMS,E)           * effective_exposure
;;;             = nominal_effective_area * effective_exposure
;;;
;;; Thus, the "nominal_effective_area" parameter to this tool should be set to the Chandra-ACIS effective area, 
;;; at the energy for which your exposure map was built, for your desired reference configuration.
;;; For example, if your exposure map was built for 1.0 keV then you might set nominal_effective_area to the EA reported
;;; by PIMMS for a specific Chandra proposal cycle.
;;; ^^^^^^^^^^^^^^^^^^^^ OBSOLETE  OBSOLETE  OBSOLETE  OPTION ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

PRO hmsfr_tables, flat_table_filename, template_filename, DIFFUSE=diffuse,$   ; DISTANCE=distance,$
                   PLOT=plot, EXTRACT_DIR=extract_dir, $
                   THERMAL_MODEL_PATTERN= thermal_model_pattern, $
                  THERMAL2_MODEL_PATTERN=thermal2_model_pattern, $
                  POWERLAW_MODEL_PATTERN=powerlaw_model_pattern, $
                  SRC_SIGNIF_MIN=src_signif_min, NET_COUNTS_MIN=net_counts_min, Nh_threshold=Nh_threshold,$
                  AV_TO_NH_FACTOR=av_to_nh_factor, PARTIAL_ABSORPTION_CORRECTION=partial_absorption_correction

COMMON hmsfr_tables, id1,id2,id3,id4,id5,id6,id36,id37,id7,id8,id9,id10,id11,id12,id102,id105,id107,id111

creator_string = "hmsfr_tables, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

;; Check for common environment errors.
existing_quiet = !QUIET  &  !QUIET = 1
catch, error_code
if (error_code NE 0) then begin
  print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
  retall
endif else resolve_routine, 'astrolib'
catch, /CANCEL
astrolib
; Make sure forprint calls do not block for user input.
!TEXTOUT=2
!QUIET = existing_quiet
  
;; When Nh (from XSPEC) is > this threshold we omit reporting soft fluxes.
if ~keyword_set(Nh_threshold) then Nh_threshold = !VALUES.F_NAN

if (n_elements(plot          ) EQ 0) then plot=1
if (n_elements(src_signif_min) EQ 0) then src_signif_min=0

if (n_elements(net_counts_min) EQ 0) then net_counts_min=0

;; The model name patterns below are Regular Expressions, evaluated by stregex().
if ~keyword_set(powerlaw_model_pattern) then powerlaw_model_pattern = 'pow:'
if ~keyword_set(  bremss_model_pattern) then   bremss_model_pattern = 'bremss:'
if ~keyword_set( thermal_model_pattern) then  thermal_model_pattern = 'T:'
if ~keyword_set(thermal2_model_pattern) then thermal2_model_pattern = '2T:|T\+T:"'
;if ~keyword_set(diffuse_model_pattern)  then diffuse_model_pattern = '*vpshock*'

if keyword_set(av_to_nh_factor) then begin
  av_to_nh_comment = repstr(string(av_to_nh_factor, F='(%"; Av = $N_{H}$ * 1e22 / %0.2e")'),'+','')
endif else begin
  av_to_nh_factor = 1.6e21 ; Vuong03 value for Milky Way
  av_to_nh_comment= ''
endelse
print, av_to_nh_factor, F='(%"\nAV_TO_NH_FACTOR = %0.3g")'


;; ------------------------------------------------------------------------
;; Prepare various table columns that will be needed in multiple places below.
collation_available = keyword_set(flat_table_filename)

if collation_available then begin
  bt = mrdfits( flat_table_filename, 1, theader )
  num_sources = n_elements(bt)
  seq = 1+indgen(num_sources)
  
  if (num_sources LT 2) then plot = 0
  
  null_val    = !VALUES.F_NAN
  null_vector = replicate(null_val, num_sources)
  
  creator_string = psb_xpar( theader, 'CREATOR') + '; ' + creator_string
  
  distance = psb_xpar( theader, 'DISTANCE') ; pc
  
  ; Use Latex math mode in object name.  
  ; Escape any special characters in CXOU or LABEL.
  CXOU           =         bt.Name
  CATALOG_NAME   = strtrim(bt.Name ,2)
  label          = strtrim(bt.LABEL,2) 
  for ii=0L,num_sources-1 do begin
    name = CXOU[ii]
    name = strjoin( strsplit(name, /EXTRACT, '+'), '$+$')
    name = strjoin( strsplit(name, /EXTRACT, '-'), '$-$')
    name = strjoin( strsplit(name, /EXTRACT, '_'), '\_')
   ;name = mg_streplace(name, '^diffuse\\_', '')  ; Remove prefix "diffuse_" to save space in tables.
    CXOU[ii] = name
    
    name = label[ii]
    name = strjoin( strsplit(name, /EXTRACT, '_'), '\_')
    label[ii] = name
  endfor


  null_flag = replicate('.',num_sources)
  
  net_counts     = round(  bt.NetCounts_t )
  median_e       =         bt.MedianEnergy_t
  SRC_SIGNIF     =         bt.NetCounts_t / (bt.NetCounts_Hi_t - bt.NetCounts_t)
  
  PosErr               = tag_exist(bt,'PosErr'        )       ? bt.PosErr               : null_vector
  SpModel              = tag_exist(bt,'SpModel'       )       ? strtrim(bt.SpModel,2)   : strarr(num_sources)
  SpModelIsProvisional = tag_exist(bt,'SpModelIsProvisional') ? bt.SpModelIsProvisional : bytarr(num_sources)
  SpModelIsHandFit     = tag_exist(bt,'SpModelIsHandFit'    ) ? bt.SpModelIsHandFit     : bytarr(num_sources)
  
  PROB_NO_SOURCE       = tag_exist(bt,'ProbNoSrc_min' )       ? bt.ProbNoSrc_min        : bt.ProbNoSrc_t
  MergeName            = tag_exist(bt,'MergeName'     )       ? strtrim(bt.MergeName,2) : strarr(num_sources)
  MergeName            = repstr(MergeName,'EPOCH_','')
  MergeName            = repstr(MergeName,'_','\_')
  
  src_area_arcsec=         bt.SrcArea
  
  ; Start a 'notes' column.
  notes_all_models = strarr(num_sources)
  
  notes_all_models[where(/NULL, SpModelIsProvisional)]  = ' Prov'
  ind =            where(/NULL, SpModelIsHandFit    )
  if isa(ind) then               notes_all_models[ind] += ' HandFit'

  if tag_exist(bt,'PhotometryCreator') then begin
    ind = where(/NULL, strmatch(bt.PhotometryCreator,'recon_spectrum*'))
    if isa(ind) then             notes_all_models[ind] += ' Reconstructed'
  endif

  ;; Interpret SpModel column
  ;; Verify the spectral model is as expected.
  SpModelType = strarr(num_sources) ; blank means "no fit"
  
  SpModelType[where(/NULL, SpModel EQ 'withheld')] = 'W'
  SpModelType[where(/NULL, stregex(/BOOL, SpModel, thermal_model_pattern) OR stregex(/BOOL, SpModel, bremss_model_pattern))] = 'T'
  SpModelType[where(/NULL, stregex(/BOOL, SpModel, powerlaw_model_pattern))] = 'P'

   chi_sqr     = tag_exist(bt,'ReducedChiSq') ? bt.ReducedChiSq : null_vector
   cstat       = tag_exist(bt,'CStatistic'  ) ? bt.CStatistic   : null_vector
endif ;collation_available



;; ------------------------------------------------------------------------
;; Process the template.
tempfilename = 'temp.tex'

openw, demo_unit, 'xray_properties.tex', /GET_LUN
openr, in_unit,  template_filename, /GET_LUN

table_name = ''
line = ''
while NOT eof(in_unit) do begin
  readf, in_unit, line
  
  if (table_name EQ '') then begin
    ; Processing template outside of a table.
    if strmatch(line, 'TEMPLATE*') then begin
      ; Open a new table (name parsed from TEMPLATE line) 
      table_name = (strsplit(line,/EXTRACT))[1]

      print
      print, 'opening table ', table_name
      openw, table_unit, tempfilename, /GET_LUN
      
      ;get_date, date, /TIMETAG
      printf, table_unit, "%  "+now()
      printf, table_unit, "%  "+creator_string 
      
      ; If table_name has any "_"s they must be escaped for LaTeX.
      printf, demo_unit, table_name, F='(%"\\include{./%s}")'

    endif else printf, demo_unit, line

  endif else begin
    ; Processing a table
    if strmatch(line, 'END*') then begin
      ; End of the table found.
      print, 'closing table ', table_name
      free_lun, table_unit
      
      ; Convert "NaN" to \nodata command.
      run_command, string(tempfilename, table_name+'.tex', F='(%"sed ''s/    NaN/\\\\nodata/g'' %s >! %s")')

      table_name = ''
      continue
    endif

    ; Copy line to table file.
    printf, table_unit, line
    
    if strmatch(line, '\\startdata*') then begin
      ; Parse data format spec from template .
      fmt = ''
      while 1 do begin
        readf, in_unit, line
        if strmatch(line, '\\enddata*') then break
        fmt = fmt + line
      endwhile
      print, 'FORMAT= '+fmt

      ;; ------------------------------------------------------------------------
      ;; Prepare column data used in multiple diffuse tables.
      if collation_available && strmatch(table_name, '*diffuse*') then begin
      
        src_area_pc    = src_area_arcsec * (2*!PI*distance/360./3600.)^2 ; pc^2
      
       ; If the source model has several components, be sure you're reporting the one(s) you want to report!!!
       ; Be sure you're reporting the energy band you want to report.

       ; The tool ae_flatten_collation produces columns named "FitLuminosity", which are actually surface
       ; luminosities, in units of log(erg /s /pc**2), NOT /arcsec**2.
       full_surfbright_corr_component1 = tag_exist(bt,'FitLuminosity_1tc') ? bt.FitLuminosity_1tc : null_vector
       full_surfbright_corr_component2 = tag_exist(bt,'FitLuminosity_2tc') ? bt.FitLuminosity_2tc : null_vector
       full_surfbright_corr_component3 = tag_exist(bt,'FitLuminosity_3tc') ? bt.FitLuminosity_3tc : null_vector
       full_surfbright_corr_component4 = tag_exist(bt,'FitLuminosity_4tc') ? bt.FitLuminosity_4tc : null_vector
       full_surfbright_corr_component5 = tag_exist(bt,'FitLuminosity_5tc') ? bt.FitLuminosity_5tc : null_vector
       full_surfbright_corr_component6 = tag_exist(bt,'FitLuminosity_6tc') ? bt.FitLuminosity_6tc : null_vector
       full_surfbright_corr_component7 = tag_exist(bt,'FitLuminosity_7tc') ? bt.FitLuminosity_7tc : null_vector
       
       ; Combined surface brightness of components modeling "diffuse".
       full_surfbright_corr_diffuse = fltarr(num_sources)
       ind = where(/NULL,                                  finite(full_surfbright_corr_component1))
       if isa(ind) then full_surfbright_corr_diffuse[ind] += 10.^(full_surfbright_corr_component1[ind]) 
       ind = where(/NULL,                                  finite(full_surfbright_corr_component2))
       if isa(ind) then full_surfbright_corr_diffuse[ind] += 10.^(full_surfbright_corr_component2[ind]) 
       ind = where(/NULL,                                  finite(full_surfbright_corr_component3))
       if isa(ind) then full_surfbright_corr_diffuse[ind] += 10.^(full_surfbright_corr_component3[ind]) 
       
       full_surfbright_corr_diffuse = alog10( full_surfbright_corr_diffuse )
       
       ; Combined surface brightness of components modeling "stars".
       full_surfbright_corr_stars = fltarr(num_sources)
       ind = where(/NULL,                                finite(full_surfbright_corr_component4))
       if isa(ind) then full_surfbright_corr_stars[ind] += 10.^(full_surfbright_corr_component4[ind]) 
       ind = where(/NULL,                                finite(full_surfbright_corr_component5))
       if isa(ind) then full_surfbright_corr_stars[ind] += 10.^(full_surfbright_corr_component5[ind]) 
       
       full_surfbright_corr_stars = alog10( full_surfbright_corr_stars )
       
       
       
       ;; Suppress flux columns when any parameter has no fit result.
       if tag_exist(bt,'NH1') && tag_exist(bt,'KT1') && tag_exist(bt,'NORM1') then begin
         invalid_fit = in_this_table AND (finite(bt.NH1,/NAN) OR ~finite(bt.KT1) OR ~finite(bt.NORM1))
         ind = where(invalid_fit, count)
         if (count GT 0) then full_surfbright_corr_component1[ind] = null_val
       endif
       
       if tag_exist(bt,'NH2') && tag_exist(bt,'KT2') && tag_exist(bt,'NORM2') then begin
         invalid_fit = in_this_table AND (finite(bt.NH2,/NAN) OR ~finite(bt.KT2) OR ~finite(bt.NORM2))
         ind = where(invalid_fit, count)
         if (count GT 0) then full_surfbright_corr_component2[ind] = null_val
         
         NH2_CI              = bt.NH2_CI
         KT2_CI              = bt.KT2_CI
         EmissionMeasure2_CI = bt.EmissionMeasure2_CI
       endif else begin
         NH2_CI              = replicate('\nodata', num_sources)
         KT2_CI              = replicate('\nodata', num_sources)
         EmissionMeasure2_CI = replicate('\nodata', num_sources)
       endelse
       
       if tag_exist(bt,'NH3') && tag_exist(bt,'KT3') && tag_exist(bt,'NORM3') then begin
         invalid_fit = in_this_table AND (finite(bt.NH3,/NAN) OR ~finite(bt.KT3) OR ~finite(bt.NORM3))
         ind = where(invalid_fit, count)
         if (count GT 0) then full_surfbright_corr_component3[ind] = null_val
         
         NH3_CI              = bt.NH3_CI
         KT3_CI              = bt.KT3_CI
         EmissionMeasure3_CI = bt.EmissionMeasure3_CI
       endif else begin
         NH3_CI              = replicate('\nodata', num_sources)
         KT3_CI              = replicate('\nodata', num_sources)
         EmissionMeasure3_CI = replicate('\nodata', num_sources)
       endelse
     
       if tag_exist(bt,'NH4') && tag_exist(bt,'KT4') && tag_exist(bt,'NORM4') then begin
         invalid_fit = in_this_table AND (finite(bt.NH4,/NAN) OR ~finite(bt.KT4) OR ~finite(bt.NORM4))
         ind = where(invalid_fit, count)
         if (count GT 0) then full_surfbright_corr_component4[ind] = null_val
         
         NH4_CI              = bt.NH4_CI
         KT4_CI              = bt.KT4_CI
         EmissionMeasure4_CI = bt.EmissionMeasure4_CI
       endif else begin
         NH4_CI              = replicate('\nodata', num_sources)
         KT4_CI              = replicate('\nodata', num_sources)
         EmissionMeasure4_CI = replicate('\nodata', num_sources)
       endelse
     
       ; Components 4 and 5 are behind the same absorber (NH4).
       if tag_exist(bt,'NH4') && tag_exist(bt,'KT5') && tag_exist(bt,'NORM5') then begin
         invalid_fit = in_this_table AND (finite(bt.NH4,/NAN) OR ~finite(bt.KT5) OR ~finite(bt.NORM5))
         ind = where(invalid_fit, count)
         if (count GT 0) then full_surfbright_corr_component5[ind] = null_val
         
         ;NH5_CI              = bt.NH5_CI
         KT5_CI              = bt.KT5_CI
         EmissionMeasure5_CI = bt.EmissionMeasure5_CI
       endif else begin
         ;NH5_CI              = replicate('\nodata', num_sources)
         KT5_CI              = replicate('\nodata', num_sources)
         EmissionMeasure5_CI = replicate('\nodata', num_sources)
       endelse
       
       if tag_exist(bt,'NH6') && tag_exist(bt,'KT6') && tag_exist(bt,'NORM6') then begin
         invalid_fit = in_this_table AND (finite(bt.NH6,/NAN) OR ~finite(bt.KT6) OR ~finite(bt.NORM6))
         ind = where(invalid_fit, count)
         if (count GT 0) then full_surfbright_corr_component6[ind] = null_val
         
         NH6_CI              = bt.NH6_CI
         KT6_CI              = bt.KT6_CI
         EmissionMeasure6_CI = bt.EmissionMeasure6_CI
       endif else begin
         NH6_CI              = replicate('\nodata', num_sources)
         KT6_CI              = replicate('\nodata', num_sources)
         EmissionMeasure6_CI = replicate('\nodata', num_sources)
       endelse

       ; Powerlaw component.
       if tag_exist(bt,'NH7') && tag_exist(bt,'PH7') && tag_exist(bt,'NORM7') then begin
         invalid_fit = in_this_table AND (finite(bt.NH7,/NAN) OR ~finite(bt.PH7) OR ~finite(bt.NORM7))
         ind = where(invalid_fit, count)
         if (count GT 0) then full_surfbright_corr_component7[ind] = null_val
         
         NH7_CI              = bt.NH7_CI
         PH7_CI              = bt.PH7_CI
       endif else begin
         NH7_CI              = replicate('\nodata', num_sources)
         PH7_CI              = replicate('\nodata', num_sources)
       endelse


       
       Tau1_CI = tag_exist(bt,'Tau1_CI') ? bt.Tau1_CI : replicate('\nodata', num_sources)
       Tau2_CI = tag_exist(bt,'Tau2_CI') ? bt.Tau2_CI : replicate('\nodata', num_sources)
       Tau3_CI = tag_exist(bt,'Tau3_CI') ? bt.Tau3_CI : replicate('\nodata', num_sources)
      
       RS_CI = tag_exist(bt,'NH3') ? bt.RS_CI : replicate('\nodata', num_sources)
      endif ;some diffuse table


      ;; ------------------------------------------------------------------------
      ; Now write out the data.
      case table_name of
      
       ;--------------------------------------------------------------------------------------------
       'observing_log': begin
         if keyword_set(diffuse) then break
       
         template_row = {OBJECT:'', DS_IDENT:'', OBS_ID:'', DETNAM:'', DATE_OBS:'', TSTART:0D, GROSS_EXPOSURE:0L, EXPOSURE:0L, RA_TARG:0D, DEC_TARG:0D, RA_PNT:0D, DEC_PNT:0D, ROLL_PNT:0., MODE:'', GRATING:'', OBSERVER:'', TGAINFIL:'', CALDBVER_EMAP:'', AIMPOINT:'', DX:0D, DY:0D }
         
         if ~keyword_set(extract_dir) then extract_dir = '../..'
         obsdata_filename = file_search( extract_dir+'/obs*/validation.evt', COUNT=num_obs )
         if (num_obs EQ 0) then begin
           print, 'WARNING: Skipping observing log table; cannot find validation.evt files.'
           break
         endif
         
         observing_log = replicate(template_row, num_obs)
         for ii=0L,num_obs-1 do begin
           ; Read event list just before flare filtering, to obtain GROSS_EXPOSURE.
           ; THE EXPOSUR* KEYWORDS ARE NOT CORRECT IN THE bad_flare.evt FILES!
           obs_dir = file_dirname(obsdata_filename[ii], /MARK_DIRECTORY)
           print, file_basename(obs_dir), F='(%"Gathering information from %s ...")'
           
           ; Find path to directory where ObsID was actually reprocessed, distinct from AE's obsXXXX/ directory.
           for ccd_id=0,9 do begin
             run_command, string(obs_dir + 'ardlib.par', ccd_id, F='(%"pget -abort %s AXAF_ACIS%d_BADPIX_FILE")'), result, /QUIET
             if (strtrim(result,2) NE 'CALDB') then break
           endfor ; ccd_id
           obsdir_repro = file_dirname(result, /MARK_DIRECTORY) 
           
           obsdata_withflares_filename = obsdir_repro+'acis.astrometric.calibrated.subpix.clean.evt'
           
           if file_test(obsdata_withflares_filename) then $
             withflares_header = headfits(obsdata_withflares_filename, EXT=1) $
           else begin
            withflares_header = 0
            print, 'WARNING: Skipping flare discard calculation; cannot find acis.astrometric.calibrated.subpix.clean.evt file.'
           endelse
           
           ; Read header keywords from event list AE used.
           theader = headfits(obsdata_filename[ii], EXT=1)
          
           ; Escape LaTeX special characters (_+-) in OBJECT string.
           name = strtrim(psb_xpar( theader,'OBJECT'  ),2)
           name = strjoin( strsplit(name, /EXTRACT, '+'), '$+$')
           name = strjoin( strsplit(name, /EXTRACT, '-'), '$-$')
           name = strjoin( strsplit(name, /EXTRACT, '_'), '\_')
           observing_log[ii].OBJECT       = name
           
           observing_log[ii].DS_IDENT     = strtrim(psb_xpar( theader,'DS_IDENT'),2)
           observing_log[ii].OBS_ID       = strtrim(psb_xpar( theader,'OBS_ID'  ),2)
           observing_log[ii].DETNAM       = strtrim(psb_xpar( theader,'DETNAM' ),2)
           observing_log[ii].TSTART       =         psb_xpar( theader,'TSTART')
           ; Discard time portion of DATE-OBS.
           observing_log[ii].DATE_OBS     = strmid(strtrim(psb_xpar( theader,'DATE-OBS'),2),0,10)
           ; See http://cxc.harvard.edu/ciao/faq/nomtargpnt.html         
          ;observing_log[ii].RA_TARG      =         psb_xpar( theader,'RA_TARG' ); coordinates in proposal
          ;observing_log[ii].DEC_TARG     =         psb_xpar( theader,'DEC_TARG'); coordinates in proposal
           observing_log[ii].RA_PNT       =         psb_xpar( theader,'RA_PNT' ) ; coordinates of optical axis
           observing_log[ii].DEC_PNT      =         psb_xpar( theader,'DEC_PNT') ; coordinates of optical axis
           observing_log[ii].ROLL_PNT     =         psb_xpar( theader,'ROLL_PNT')

           ; Abbreviate the long TGAIN filenames. 
           name = strtrim(psb_xpar( theader,'TGAINFIL'),2)
           name = repstr(name, 'acisD' , '' )
           name = repstr(name, 't_gain', '' )
           name = repstr(name, '.fits' , '' )
           name = repstr(name, 'N000'  , 'N')
           observing_log[ii].TGAINFIL     = name
           
           observing_log[ii].MODE         = repstr(strtrim(psb_xpar( theader,'READMODE'),2), 'TIMED','TE') +'-'+ $
                                            repstr(strtrim(psb_xpar( theader,'DATAMODE'),2), 'FAINT','F')

           name                           = repstr(strtrim(psb_xpar( theader,'GRATING'),2), 'NONE','')
           observing_log[ii].GRATING      = strmid(name,0,1) ; Report first letter of GRATING ("H" or "L")

           ; Remove common titles used in OBSERVER.
           name = strtrim(psb_xpar( theader,'OBSERVER'),2)
           name = repstr(name, 'Dr. '       , '' )
           name = repstr(name, 'DR. '       , '' )
           name = repstr(name, 'Dr '        , '' )
           name = repstr(name, 'DR '        , '' )
           name = repstr(name, 'Prof. '     , '' )
           name = repstr(name, 'Professor ' , '' )
           observing_log[ii].OBSERVER     = name

           if (psb_xpar( theader,'SIM_Z') GT -210) then begin
             observing_log[ii].AIMPOINT       = 'S'
             observing_log[ii].EXPOSURE       = round( psb_xpar( theader          ,'EXPOSUR7') )
             if keyword_set(withflares_header) then $
             observing_log[ii].GROSS_EXPOSURE = round( psb_xpar( withflares_header,'EXPOSUR7') )
             
           endif else begin
             observing_log[ii].AIMPOINT       = 'I'
             observing_log[ii].EXPOSURE       = round( psb_xpar( theader          ,'EXPOSUR3') )
             if keyword_set(withflares_header) then $
             observing_log[ii].GROSS_EXPOSURE = round( psb_xpar( withflares_header,'EXPOSUR3') )
           endelse
           
           ; Look up CALDB version used for exposure map.
           instmap_fn = file_search(obsdir_repro+'instmap', '*.instmap', COUNT=count)
           ; For backward compatibility ...
           if (count EQ 0) then $
           instmap_fn = file_search(obsdir_repro+'asphist', '*.instmap', COUNT=count)
           
           if (count GT 0) then begin
             run_command, string(instmap_fn[0], F='(%"dmlist %s opt=comments | grep contamN")'), result, /IGNORE_STATUS, /QUIET  
   
             observing_log[ii].CALDBVER_EMAP = (stregex(result[0],'contam(N....)',/SUB,/EXT))[1]
           endif ; count GT 0
           
           
           ; Calculate the cumulative astrometric shifts we have applied to the aspect file.
           ; The individual shifts can be displayed by:
           ;     dmhistory acis.astrometric.asol1 wcs_update
           asp_old = mrdfits(obsdir_repro+'acis.asol1'            , 1, /SILENT)
           asp_new = mrdfits(obsdir_repro+'acis.astrometric.asol1', 1, /SILENT)
           
           if (n_elements(asp_old) NE n_elements(asp_new)) then begin
             print, 'WARNING: The aspect tables acis.asol1 and acis.astrometric.asol1 have different lengths.'
             print, 'Interpolating to the same TIME grid.'
             temp    = temporary(asp_new)
             asp_new = replicate({RA:0D, DEC:0D}, n_elements(asp_old))
             ind = 0 > value_locate( temp.TIME, asp_old.TIME )
             asp_new.RA  = temp[ind].RA
             asp_new.DEC = temp[ind].DEC
           endif
           
           arcsec_per_skypixel = 0.492 ; (0.492 arcsec/skypix)
           
           dx = -3600* (asp_new.RA  - asp_old.RA ) * cos(asp_new.DEC * !DTOR) ; arcsec
           dy =  3600* (asp_new.DEC - asp_old.DEC)                            ; arcsec
           
           observing_log[ii].DX  = median(dx) /arcsec_per_skypixel ; skypix
           observing_log[ii].DY  = median(dy) /arcsec_per_skypixel ; skypix
          
         endfor ;ii

         ; Eliminate duplicate entries that arise because AE worked with _FI and _BI pseudo-ObsIDs
         observing_log = observing_log[ sort(observing_log.OBS_ID) ]
         for ii=0L,num_obs-2 do begin
         
           if (observing_log[ii].OBS_ID NE observing_log[ii+1].OBS_ID) then continue
           
           ; Discard the row with zero exposure.
           ind_discard = (observing_log[ii].EXPOSURE EQ 0) ? ii : ii+1
           
           observing_log[ind_discard].OBS_ID = ''
         endfor ;ii
         observing_log = observing_log[ where(observing_log.OBS_ID, num_obs) ]
         
         ; Sort ObsIDs by Start Time.
         observing_log = observing_log[ sort(observing_log.TSTART) ]
         

         onaxis_sexagesimal = repstr( adstring(observing_log.RA_PNT,observing_log.DEC_PNT, PRECISION=1, /TRUNCATE), ' ', ':')

         configuration_string = observing_log.GRATING
         ind = where(/NULL, configuration_string EQ '')
         configuration_string[ind] = (observing_log.AIMPOINT)[ind]
         configuration_string += ' ('+repstr(observing_log.DETNAM,'ACIS-','')+')'

         !TEXTUNIT = table_unit
         forprint, TEXTOUT=5, /NoCOMMENT, observing_log.OBJECT, observing_log.OBS_ID, observing_log.DS_IDENT, observing_log.DATE_OBS, observing_log.EXPOSURE, strmid(onaxis_sexagesimal,1,11), strmid(onaxis_sexagesimal,14,11), observing_log.ROLL_PNT, configuration_string, observing_log.MODE, observing_log.OBSERVER, observing_log.TGAINFIL, observing_log.CALDBVER_EMAP,$
         '('+string(observing_log.DX,F='(F+0.3)')+', '+string(observing_log.DY,F='(F+0.3)')+')',$
         round(100*(observing_log.GROSS_EXPOSURE - observing_log.EXPOSURE)/observing_log.GROSS_EXPOSURE), F=fmt  
         
         !TEXTUNIT = 0
         print, "ObsID shifts, in time order:"
         forprint, observing_log.OBS_ID, observing_log.DX,observing_log.DY, F='(%" %10s:  (%7.3f, %7.3f) skypix")'
         
         if (num_obs GT 1) then begin
           dataset_2d, id_offset, observing_log.DX,observing_log.DY, PSYM=1, XTIT='dx (skypix)',YTIT='dy (skypix)',TIT='ObsID Shifts', PLOT_WINDOW_OPTIONS='/FORCE_UNITY_ASPECT', PS_CONFIG={filename:'ObsID_shifts.ps'}, /PRINT
           
           seconds_per_day = 3600.*24.
           time = (observing_log.TSTART - min(observing_log.TSTART)) / seconds_per_day
           
           function_1d, id_xoffset, time, observing_log.DX*arcsec_per_skypixel, PSYM=1, LINE=6, XTIT='Days since first ObsID',YTIT='dx (arcsec)',TIT='ObsID Shifts', COLOR='red', PS_CONFIG={filename:'ObsID_dx.ps'}, /PRINT

           function_1d, id_yoffset, time, observing_log.DY*arcsec_per_skypixel, PSYM=1, LINE=6, XTIT='Days since first ObsID',YTIT='dy (arcsec)',TIT='ObsID Shifts', COLOR='red', PS_CONFIG={filename:'ObsID_dy.ps'}, /PRINT
         endif ; (num_obs GT 1)
         
         ; Build region file marking on-axis positions (http://cxc.harvard.edu/ciao/faq/nomtargpnt.html)
         openw, region_unit, 'aimpoints.reg', /GET_LUN
         !TEXTUNIT = region_unit
         forprint, TEXTOUT=5, /NoCOM, observing_log.RA_PNT, observing_log.DEC_PNT, observing_log.OBS_ID, observing_log.AIMPOINT, F='(%"J2000;box %10.6f %10.6f 16\" 16\" # tag={optical axis dither} tag={ObsID %s} tag={%s} color={blue} width=2")'
                           
         forprint, TEXTOUT=5, /NoCOM, observing_log.RA_PNT, observing_log.DEC_PNT, observing_log.OBS_ID, observing_log.OBS_ID, observing_log.AIMPOINT, F='(%"J2000;text %10.6f %10.6f # text={%s} tag={ObsID label} tag={ObsID %s} tag={%s} color={blue} width=2 font=\"helvetica 20 normal\"")'
         free_lun, region_unit                                                                          
         !TEXTUNIT = 0
       end

         
       ;--------------------------------------------------------------------------------------------
       'src_properties': begin
         if ~collation_available then break

         prob_no_source_string = string(alog10(PROB_NO_SOURCE), F='(%"%5.1f")')
         ind = where(PROB_NO_SOURCE LT 1E-5, count)
         if (count GT 0) then begin
           prob_no_source_string[ind] = "$<$-5"
         endif
         
         
         off_chip_flag = null_flag
         ind = where(bt.ExposureFraction LT 0.9, count)
         if ~keyword_set(diffuse) && (count GT 0) then off_chip_flag[ind] = 'g'
         
         pileup_flag = null_flag
         
         if tag_exist(bt,'RateIn3x3Cell') then begin
           ind = where(bt.RateIn3x3Cell GT 0.05, count)  ; See corresponding threshold in ae_pileup_screening tool.
           if (count GT 0) then pileup_flag[ind] = 'p'
         endif else print, 'WARNING!  The "pile-up" flag cannot be computed.'

         streak_flag = null_flag
         
         if tag_exist(bt,'StreakFraction') then begin
           ind = where(bt.StreakFraction GT 0.1, count)
           if (count GT 0) then streak_flag[ind] = 's'
         endif else print, 'WARNING!  The "streak" flag cannot be computed.'

         afterglow_flag = null_flag
         if tag_exist(bt,'AfterglowFraction') then begin
           ; Our afterglow screening algorithm sufferes from false positives for bright sources.
           ; This has not been calibrated, so we must guess at an appropriate way to suppress reporting these.
           number_of_afterglow_events = bt.SrcCounts_t * bt.AfterglowFraction
           ind = where((bt.AfterglowFraction GT 0.1) AND (number_of_afterglow_events LT 20), count)
           if (count GT 0) then afterglow_flag[ind] = 'a'
         endif else print, 'WARNING!  The "afterglow" flag cannot be computed.'
         
         anom_flags = off_chip_flag+pileup_flag+streak_flag+afterglow_flag
         
         
         ; Convert variability indices to three-level flags.
         var1_flag    = null_flag
         var2_flag    = null_flag
         var3_flag    = null_flag
         min_varindex = null_vector

         ; Calculate p-value for reported PROB_KS = min( P1,P2,...Pn), where n = N_KS.
         if tag_exist(bt,'ProbKS_single') && $
            tag_exist(bt,'N_KS_single'  ) && $
            tag_exist(bt,'ProbKS_merge' ) && $
            tag_exist(bt,'ProbChisq_PhotonFlux') then begin
           pval_for_PROB_KS = 1D - (1D - bt.ProbKS_single)^bt.N_KS_single
           pval_for_PROB_KS[where(/NULL, ~finite(bt.ProbKS_single))] = !VALUES.F_NAN
  
           var1_flag[ where(/NULL, finite(pval_for_PROB_KS)               ) ] = 'a'
           var1_flag[ where(/NULL,        pval_for_PROB_KS        LT 0.05 ) ] = 'b'
           var1_flag[ where(/NULL,        pval_for_PROB_KS        LT 0.005) ] = 'c'
           
           if tag_exist(bt,'ProbKS_merge') then begin
           var2_flag[ where(/NULL, finite(bt.ProbKS_merge)                ) ] = 'a'
           var2_flag[ where(/NULL,        bt.ProbKS_merge         LT 0.05 ) ] = 'b'
           var2_flag[ where(/NULL,        bt.ProbKS_merge         LT 0.005) ] = 'c'
           endif
           if tag_exist(bt,'ProbChisq_PhotonFlux') then begin
           var3_flag[ where(/NULL, finite(bt.ProbChisq_PhotonFlux)        ) ] = 'a'
           var3_flag[ where(/NULL,        bt.ProbChisq_PhotonFlux LT 0.05 ) ] = 'b'
           var3_flag[ where(/NULL,        bt.ProbChisq_PhotonFlux LT 0.005) ] = 'c'
           endif
           
           ; Omit certain variability flags when off_chip_flag flag set.
           ind = where(/NULL, off_chip_flag NE null_flag)
           var1_flag[ind] = '.'
           var2_flag[ind] = '.'
         end

         var_flag = var1_flag+var2_flag+var3_flag
         
         

         THETA = tag_exist(bt,'THETA') ? bt.THETA : intarr(num_sources)
          
         !TEXTUNIT = table_unit
         forprint, TEXTOUT=5, /NoCOMMENT, seq, label, CXOU, bt.RADEG, bt.DEDEG, PosErr, MergeName, THETA, $
         
         bt.NetCounts_t, (bt.NetCounts_Hi_t - bt.NetCounts_Lo_t)/2, $
         
         (bt.BkgCounts_t / bt.BkgScaling), bt.PsfFraction, $
         
         SRC_SIGNIF, prob_no_source_string, anom_flags, var_flag, $
         ;  (bt.EMAP_TOT/nominal_effective_area)/1000., $
         
         median_e, SpModelType, F=fmt

         ;; Save some important columns for the observer's convenience.

;        save, /COMPRESS, seq, CATALOG_NAME, OBJECT, SpModel, RA_FINAL, DEC_FINAL, net_counts, SRC_SIGNIF, PROB_NO_SOURCE, median_e, anom_flags,   FILE='src_properties.sav'
         end
         

       ;--------------------------------------------------------------------------------------------
       'thermal_spectroscopy': begin
         if ~collation_available then break

         dsn = 'thermal model'

         ; Determine which sources belong in this table.  We process both tbabs_vapec and tbabs_2vapec models.
         in_this_table = (SRC_SIGNIF     GE src_signif_min) AND $
                         (bt.NetCounts_t GE net_counts_min) AND $
                         (SpModelType    EQ 'T')

         ind = where( in_this_table, thermal_count )
         if keyword_set(diffuse) || (thermal_count EQ 0) || ~tag_exist(bt,'KT1') then begin
           print, thermal_count, 'NO sources in thermal_spectroscopy table'
           goto, END_thermal_spectroscopy
         endif
         print, thermal_count, F='(%"%d sources in thermal_spectroscopy table.")'


         if tag_exist(bt,'LMC') then begin
           LMC_val             = bt.LMC
           LMC_CI              = bt.LMC_CI
         endif else begin
           LMC_val             = null_vector
           LMC_CI              = replicate('\nodata', num_sources)
         endelse

         NH1_val     = bt.NH1
         KT1_val     = bt.KT1

         if tag_exist(bt,'NH2') then begin
           NH2_val             = bt.NH2
           NH2_CI              = bt.NH2_CI
         endif else begin
           NH2_val             = null_vector
           NH2_CI              = replicate('\nodata', num_sources)
         endelse
         
         if tag_exist(bt,'KT2') then begin
           KT2_CI              = bt.KT2_CI
           EmissionMeasure2_CI = bt.EmissionMeasure2_CI
         endif else begin
           KT2_CI              = replicate('\nodata', num_sources)
           EmissionMeasure2_CI = replicate('\nodata', num_sources)
         endelse
         
         ; Work will full-length vectors so user can find index for specific source in plotting tools.  
         ; Sources not in this table with have NaN values.
         full_flux= alog10(bt.FH8)

         soft_lumin      = bt.FitLuminosity_s
         hard_lumin      = bt.FitLuminosity_h 
         full_lumin      = bt.FitLuminosity_t 
         full_lumin1     = tag_exist(bt,'FitLuminosity_1t' ) ? bt.FitLuminosity_1t  : null_vector
         full_lumin2     = tag_exist(bt,'FitLuminosity_2t' ) ? bt.FitLuminosity_2t  : null_vector
         
         if keyword_set(partial_absorption_correction) then begin
           ; These "l" corrected luminosities are produced by the LMC fitting scripts.
           correction_comment = "; luminosities are corrected for only MW and LMC (not circumstellar) absorption"
           soft_lumin_corr = bt.FitLuminosity_sl
           hard_lumin_corr = bt.FitLuminosity_hl
           full_lumin_corr = bt.FitLuminosity_tl
           full_lumin1_corr= tag_exist(bt,'FitLuminosity_1tl') ? bt.FitLuminosity_1tl : null_vector
           full_lumin2_corr= tag_exist(bt,'FitLuminosity_2tl') ? bt.FitLuminosity_2tl : null_vector
         endif else begin
           correction_comment = ''
           soft_lumin_corr = bt.FitLuminosity_sc
           hard_lumin_corr = bt.FitLuminosity_hc
           full_lumin_corr = bt.FitLuminosity_tc
           full_lumin1_corr= tag_exist(bt,'FitLuminosity_1tc') ? bt.FitLuminosity_1tc : null_vector
           full_lumin2_corr= tag_exist(bt,'FitLuminosity_2tc') ? bt.FitLuminosity_2tc : null_vector
         endelse

         ind = where(in_this_table EQ 0, count)
         if (count GT 0) then begin
           NH1_val         [ind] = null_val
           NH2_val         [ind] = null_val
           KT1_val         [ind] = null_val
           full_flux       [ind] = null_val
           soft_lumin      [ind] = null_val
           soft_lumin_corr [ind] = null_val
           hard_lumin      [ind] = null_val
           hard_lumin_corr [ind] = null_val
           full_lumin      [ind] = null_val
           full_lumin_corr [ind] = null_val
           full_lumin1     [ind] = null_val
           full_lumin1_corr[ind] = null_val
           full_lumin2     [ind] = null_val
           full_lumin2_corr[ind] = null_val
         endif
         
         Av1 = string(NH1_val*1e22 / av_to_nh_factor, F='(%"%7.1f")')
         Av2 = string(NH2_val*1e22 / av_to_nh_factor, F='(%"%7.1f")')
         
         Av1[where(/NULL, ~finite(NH1_val))] = '\nodata'
         Av2[where(/NULL, ~finite(NH2_val))] = '\nodata'

         ;; Suppress flux columns when any parameter has no fit result.
         invalid_fit = in_this_table AND (finite(bt.NH1,/NAN) OR ~finite(bt.KT1) OR ~finite(bt.NORM1))
         ind = where(invalid_fit, count)
         if (count GT 0) then begin
           print, count, thermal_count, F='(%"\nThese %d (out of %d total) thermal sources have no good estimate for at least one fit parameter:")'

           ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
           !TEXTUNIT = 0
           forprint, 1+ind, bt[ind].Name, F='(%"%4d %s")'
         endif
         
         
         ;; Suppress corrected flux columns when Nh is large.
         invalid_flux = in_this_table AND (bt.NH1 GT Nh_threshold)
         ind = where(invalid_flux, count)
         if finite(Nh_threshold) && (count GT 0) then begin
           print, count, thermal_count, Nh_threshold, F='(%"\nThese %d (out of %d total) thermal sources have Nh (from XSPEC) > %f:")'

           ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
           !TEXTUNIT = 0
           forprint, 1+ind, bt[ind].Name, F='(%"%4d %s")'
           
           soft_lumin     [ind] = null_val
           soft_lumin_corr[ind] = null_val
           full_lumin_corr[ind] = null_val
         endif
         

         ; Add any model-specific notes ...
         notes = notes_all_models
         
;        ind = where( stregex(/BOOL, SpModel, thermal2_model_pattern), count )
;        if (count GT 0) then notes[ind] += ' 2T'

         notes = strtrim(notes,2)
         notes[where(/NULL, notes EQ '')] = '\nodata'
         
         
         ;; Write out table.
         ind = where( in_this_table )

         if keyword_set(plot) then begin
           this_chi_sqr      = null_vector
           this_chi_sqr[ind] = chi_sqr[ind]
           dataset_2d,id12, net_counts, this_chi_sqr, XTIT='NET_CNTS', YTIT='Reduced chi^2', PSYM=1, DATASET=dsn, NAN=[0,0]
         endif

         !TEXTUNIT = table_unit
         printf, table_unit, distance/1000.0, av_to_nh_comment, correction_comment, F='(%"\\\\\n\\hline\n\\multicolumn{30}{l}{{\\bf TARGET} (D=%0.3f kpc%s%s)}\\\\")'

         forprint, TEXTOUT=5, /NoCOMMENT, SUBSET=ind, $%
         seq, label, CXOU, MergeName, round(bt.NetCounts_t), SRC_SIGNIF,$
         chi_sqr,$
         LMC_CI,$
         bt.NH1_CI, Av1, bt.KT1_CI, bt.EmissionMeasure1_CI, $
            NH2_CI, Av2,    KT2_CI,    EmissionMeasure2_CI, $
         bt.abundance_CI, $
         full_lumin1    ,full_lumin1_corr,$
         full_lumin2    ,full_lumin2_corr,$
         soft_lumin_corr,hard_lumin_corr ,$
         full_lumin     ,full_lumin_corr ,$
         repstr(SpModel,'_','\_'), notes, cstat, chi_sqr,  F=fmt

         ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
         !TEXTUNIT = 0
         print
         print, ' seq         name            NH          kT         Norm      NC  Emed      SpModel'
         forprint, SUBSET=ind, seq,                      CATALOG_NAME, bt.NH1_Flags, bt.KT1_Flags, bt.NORM1_Flags, net_counts, median_e, SpModel, F='(%"%4d %s  %s  %s  %s  %4d %3.1f %s")'
         
         if keyword_set(plot) then begin
           dataset_1d,id1, NH1_val,      XTIT='Nh', DATASET=dsn
           dataset_1d,id2, KT1_val,      XTIT='kT'
           dataset_1d,id3, full_lumin_corr, XTIT='log Lc[0.5:8]', DATASET=dsn
           dataset_1d,id4, full_flux      , XTIT='log F[0.5:8]' , DATASET=dsn
           dataset_2d,id5, NH1_val, KT1_val, XTIT='Nh', YTIT='kT', PSYM=1, NAN=[0,-1],$
                      PS_CONFIG={filename:'NH1_vs_KT1.ps'}

           erru = bt.NH1_HiLim > NH1_val 
           errl =                NH1_val < bt.NH1_LoLim 
           mid         = 0.5*(erru+errl)
           half_length = 0.5*(erru-errl)
           
          ;dataset_2d,id6, alog10(net_counts), NH1_val,      XTIT='log NET_CNTS', YTIT='Nh', PSYM=1, NAN=[1,0], DATASET=dsn
           function_1d, id36, alog10(net_counts),     mid, PSYM=3, LINE=6, NSKIP_ERRORS=1, Y_ERROR=half_length, $
                        DATA='confidence interval', WIDGET_TITLE=dsn, XTIT='log NET_CNTS', YTIT='NH1'
           function_1d, id36, alog10(net_counts), NH1_val, PSYM=1, LINE=6, $
                        PLOT_WINDOW_OPTIONS=string(median(NH1_val), F='(%"SET_BIG_MARKER=[0,%0.2f], SHOW_DATE=0")'),$
                        PS_CONFIG={filename:'NH1_CI_vs_NC.ps'}

           erru = bt.KT1_HiLim > KT1_val 
           errl =                KT1_val < bt.KT1_LoLim 
           mid         = 0.5*(erru+errl)
           half_length = 0.5*(erru-errl)
           
          ;dataset_2d,id7, alog10(net_counts), KT1_val,      XTIT='log NET_CNTS', YTIT='kT', PSYM=1, NAN=[1,-5]
           function_1d, id37, alog10(net_counts),     mid, PSYM=3, LINE=6, NSKIP_ERRORS=1, Y_ERROR=half_length, $
                        DATA='confidence interval', WIDGET_TITLE=dsn, XTIT='log NET_CNTS', YTIT='kT1'
           function_1d, id37, alog10(net_counts), KT1_val, PSYM=1, LINE=6, $
                        PLOT_WINDOW_OPTIONS=string(median(KT1_val), F='(%"SET_BIG_MARKER=[0,%0.2f], SHOW_DATE=0")'),$
                        PS_CONFIG={filename:'KT1_CI_vs_NC.ps'}

           
           dataset_2d,id8, alog10(net_counts), full_lumin_corr, XTIT='log NET_CNTS', YTIT='log Lc[0.5:8]', PSYM=1, NAN=[1,20], DATASET=dsn
           
           dataset_2d,id9, alog10(net_counts), full_flux,  XTIT='log NET_CNTS', YTIT='log F[0.5:8]', PSYM=1, NAN=[1,-20], DATASET=dsn                          

           dataset_2d,id10, full_lumin_corr, NH1_val, YTIT='Nh', XTIT='log Lc[0.5:8]', PSYM=1, NAN=[20, 0], DATASET=dsn
           dataset_2d,id11, full_lumin_corr, KT1_val, YTIT='kT', XTIT='log Lc[0.5:8]', PSYM=1, NAN=[20,-5]
         endif

;        save, /COMPRESS, seq, NH1_val, KT1_val, $
;           soft_lumin, soft_lumin_corr, hard_lumin, hard_lumin_corr, full_lumin, full_lumin_corr, chi_sqr, cstat, FILE='thermal_spectroscopy.sav'

END_thermal_spectroscopy:         
         end ;thermal_spectroscopy

         
       ;--------------------------------------------------------------------------------------------
       'powerlaw_spectroscopy': begin
         if ~collation_available then break

         dsn = 'powerlaw model'

         ; Determine which sources belong in this table.
         in_this_table = (SRC_SIGNIF     GE src_signif_min) AND $
                         (bt.NetCounts_t GE net_counts_min) AND $
                         (SpModelType    EQ 'P')
         
         ind = where( in_this_table, powerlaw_count )
         if keyword_set(diffuse) || (powerlaw_count EQ 0) || ~tag_exist(bt,'PH1') then begin
           print, 'NO sources in powerlaw_spectroscopy table'
           goto, END_powerlaw_spectroscopy
         endif
         print, powerlaw_count, F='(%"%d sources in powerlaw_spectroscopy table.")'


         ; Work will full-length vectors so user can find index for specific source in plotting tools.  
         ; Sources not in this table with have NaN values.
         NH1_val     = bt.NH1
         PH1_val     = bt.PH1
         
         full_flux= alog10(bt.FH8)
         soft_lumin      = bt.FitLuminosity_s 
         soft_lumin_corr = bt.FitLuminosity_sc
         hard_lumin      = bt.FitLuminosity_h 
         hard_lumin_corr = bt.FitLuminosity_hc
         full_lumin      = bt.FitLuminosity_t 
         full_lumin_corr = bt.FitLuminosity_tc
         
         ind = where(in_this_table EQ 0, count)
         if (count GT 0) then begin
           NH1_val        [ind] = null_val
           PH1_val        [ind] = null_val
           full_flux      [ind] = null_val
           soft_lumin     [ind] = null_val
           soft_lumin_corr[ind] = null_val
           hard_lumin     [ind] = null_val
           hard_lumin_corr[ind] = null_val
           full_lumin     [ind] = null_val
           full_lumin_corr[ind] = null_val
         endif
         
         ;; Suppress flux columns when any parameter has no fit result.
         invalid_fit = in_this_table AND (finite(bt.NH1,/NAN) OR ~finite(bt.PH1) OR ~finite(bt.NORM1))
         ind = where(invalid_fit, count)
         if (count GT 0) then begin
           print, count, powerlaw_count, F='(%"\nThese %d (out of %d total) powerlaw sources have no good estimate for at least one fit parameter:")'

           ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
           !TEXTUNIT = 0
           forprint, SUBSET=ind, bt.Name
         endif
         
         
         ;; Suppress corrected flux columns when Nh is large.
         invalid_flux = in_this_table AND (bt.NH1 GT Nh_threshold)
         ind = where(invalid_flux, count)
         if finite(Nh_threshold) && (count GT 0) then begin
           print, count, thermal_count, Nh_threshold, F='(%"\nThese %d (out of %d total) powerlaw sources have Nh (from XSPEC) > %f:")'

           ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
           !TEXTUNIT = 0
           forprint, 1+ind, bt[ind].Name, F='(%"%4d %s")'
           
           soft_lumin     [ind] = null_val
           full_lumin_corr[ind] = null_val
         endif
         
         

         ; Add any model-specific notes ...
         notes = notes_all_models
         
         notes = strtrim(notes,2)
         notes[where(/NULL, notes EQ '')] = '\nodata'
         
         
         ;; Write out table.
         ind = where( in_this_table )

         if keyword_set(plot) then begin
           this_chi_sqr      = null_vector
           this_chi_sqr[ind] = chi_sqr[ind]
           dataset_2d,id12, net_counts, this_chi_sqr, XTIT='NET_CNTS', YTIT='Reduced chi^2', PSYM=1, DATASET=dsn, NAN=[0,0]
         endif
         
         !TEXTUNIT = table_unit
         printf, table_unit, distance/1000.0, av_to_nh_comment, F='(%"\\\\\n\\hline\n\\multicolumn{30}{l}{{\\bf TARGET} (D=%0.3f kpc%s)}\\\\")'

         forprint, TEXTOUT=5, /NoCOMMENT, SUBSET=ind, seq, label, CXOU, MergeName, $
         bt.NetCounts_t, SRC_SIGNIF, chi_sqr,$
         bt.NH1_CI, 10^NH1_val / 1.6e21,  bt.PH1_CI, $
         full_lumin, full_lumin_corr, repstr(SpModel,'_','\_'), notes, cstat, chi_sqr, F=fmt

         ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
         !TEXTUNIT = 0
         print
         print, ' seq         name          NH        gamma        Norm      NC  Emed      SpModel'
         forprint, SUBSET=ind, seq,                      CATALOG_NAME, bt.NH1_Flags, bt.PH1_Flags, bt.NORM1_Flags, net_counts, median_e, SpModel, F='(%"%4d %s  %s  %s  %s  %4d %3.1f %s")'
         
          if keyword_set(plot) then begin
           ; Plot will full-length vectors so user can find index for specific source.
           dataset_1d,id1, NH1_val,      XTIT='Nh', DATASET=dsn
           dataset_1d,id102, PH1_val,    XTIT='photon index'
           dataset_1d,id3, full_lumin_corr, XTIT='log Lc[0.5:8]', DATASET=dsn
           dataset_1d,id4, full_flux,  XTIT='log F[0.5:8]', DATASET=dsn

           dataset_2d,id105, NH1_val, PH1_val, XTIT='Nh', YTIT='photon index', PSYM=1, NAN=[0,-1]

          ;dataset_2d,id6, alog10(net_counts), NH1_val,   XTIT='log NET_CNTS', YTIT='Nh', PSYM=1, NAN=[1,0], DATASET=dsn
          ;dataset_2d,id107, alog10(net_counts), PH1_val, XTIT='log NET_CNTS', YTIT='photon index', PSYM=1, NAN=[-1,0]
           
           dataset_2d,id8, alog10(net_counts), full_lumin_corr, XTIT='log NET_CNTS', YTIT='log Lc[0.5:8]', PSYM=1, NAN=[1,20], DATASET=dsn
           
           dataset_2d,id9, alog10(net_counts), full_flux,  XTIT='log NET_CNTS', YTIT='log F[0.5:8]', PSYM=1, NAN=[1,-20], DATASET=dsn                          

           dataset_2d,id10 , full_lumin_corr, NH1_val, YTIT='Nh'      , XTIT='log Lc[0.5:8]', PSYM=1, NAN=[20,0], DATASET=dsn
           dataset_2d,id111, full_lumin_corr, PH1_val, YTIT='photon index', XTIT='log Lc[0.5:8]', PSYM=1, NAN=[0,-1]
         endif

;        save, /COMPRESS, seq, NH1_val, PH1_val, $
;           soft_lumin, soft_lumin_corr, hard_lumin, hard_lumin_corr, full_lumin, full_lumin_corr, chi_sqr, cstat, FILE='powerlaw_spectroscopy.sav'

END_powerlaw_spectroscopy:         
         end ;powerlaw_spectroscopy
 
 
       ;--------------------------------------------------------------------------------------------
       'diffuse_spectroscopy_style1': begin
         if ~collation_available then break

         dsn = 'diffuse model'
         
         ; Determine which sources belong in this table.  
         in_this_table = (SRC_SIGNIF     GE src_signif_min) AND $
                         (bt.NetCounts_t GE net_counts_min) AND $
                         (SpModelType    EQ 'T')
         
         ind = where( in_this_table, thermal_count )
         if ~keyword_set(diffuse) || (thermal_count EQ 0) || ~tag_exist(bt,'KT1') then begin
           print, 'NO sources in diffuse_spectroscopy table'
           goto, END_diffuse_spectroscopy_style1
         endif
         print, thermal_count, F='(%"%d sources in diffuse_spectroscopy table.")'

         ; Add any model-specific notes ...
         notes = notes_all_models
         
         notes = strtrim(notes,2)
         notes[where(/NULL, notes EQ '')] = '\nodata'

         ;; Write out table.
         ind = where( in_this_table )

         if keyword_set(plot) then begin
           this_chi_sqr      = null_vector
           this_chi_sqr[ind] = chi_sqr[ind]
           dataset_2d,id12, net_counts, this_chi_sqr, XTIT='NET_CNTS', YTIT='Reduced chi^2', PSYM=1, DATASET=dsn, NAN=[0,0]
         endif

         !TEXTUNIT = table_unit
         printf, table_unit, distance/1000.0, av_to_nh_comment, F='(%"\\\\\n\\hline\n\\multicolumn{30}{l}{{\\bf TARGET} (D=%0.3f kpc%s)}\\\\")'

         forprint, TEXTOUT=5, /NoCOMMENT, SUBSET=ind,$
         CXOU, net_counts, src_area_pc, src_area_arcsec/(60^2),$
         bt.NH1_CI,    NH2_CI,    NH3_CI, NH4_CI, NH6_CI, $ 
         bt.KT1_CI,    KT2_CI,    KT3_CI,         KT6_CI, $
         Tau1_CI, Tau2_CI, Tau3_CI, $
         bt.EmissionMeasure1_CI, EmissionMeasure2_CI, EmissionMeasure3_CI, EmissionMeasure5_CI, EmissionMeasure6_CI,$
         bt.abundance_CI, $

         (tag_exist(bt,'ReducedChiSq') && tag_exist(bt,'DegreesOfFreedom')) ? $
           string(round(bt.ReducedChiSq*bt.DegreesOfFreedom), F='(%"%5d")') + '/' + string(bt.DegreesOfFreedom,F='(%"%3d")') : $
           strarr(num_sources), $

         full_surfbright_corr_component1, full_surfbright_corr_component2, full_surfbright_corr_component3,$ 
         ; Combined surface brightness of components modeling "stars".
         full_surfbright_corr_stars,$
         full_surfbright_corr_component6,$
         
         repstr(SpModel,'_','\_'), notes, round(SRC_SIGNIF), cstat, chi_sqr, label, F=fmt

         ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
         !TEXTUNIT = 0
         print
         print, ' seq         name            NH          kT         Norm      NC  Emed      SpModel'
         forprint, SUBSET=ind, seq,                      CATALOG_NAME, bt.NH1_Flags, bt.KT1_Flags, bt.NORM1_Flags, net_counts, median_e, SpModel, F='(%"%4d %s  %s  %s  %s  %4d %3.1f %s")'

END_diffuse_spectroscopy_style1:         
         end ;diffuse_spectroscopy_style1

         
         
       ;--------------------------------------------------------------------------------------------
       'diffuse_spectroscopy_style2': begin
         if ~collation_available then break

         dsn = 'diffuse model'
         
         ; Determine which sources belong in this table.  
         in_this_table = (SRC_SIGNIF     GE src_signif_min) AND $
                         (bt.NetCounts_t GE net_counts_min) AND $
                         (SpModelType    EQ 'T')
         
         ind = where( in_this_table, thermal_count )
         if ~keyword_set(diffuse) || (thermal_count EQ 0) || ~tag_exist(bt,'KT1') then begin
           print, 'NO sources in diffuse_spectroscopy table'
           goto, END_diffuse_spectroscopy_style2
         endif
         print, thermal_count, F='(%"%d sources in diffuse_spectroscopy table.")'

         ; Add any model-specific notes ...
         notes = notes_all_models
         
         notes = strtrim(notes,2)
         notes[where(/NULL, notes EQ '')] = '\nodata'

         ;; Write out table.
         ind = where( in_this_table )

         if keyword_set(plot) then begin
           this_chi_sqr      = null_vector
           this_chi_sqr[ind] = chi_sqr[ind]
           dataset_2d,id12, net_counts, this_chi_sqr, XTIT='NET_CNTS', YTIT='Reduced chi^2', PSYM=1, DATASET=dsn, NAN=[0,0]
         endif

         !TEXTUNIT = table_unit
         printf, table_unit, distance/1000.0, av_to_nh_comment, F='(%"\\\\\n\\hline\n\\multicolumn{30}{l}{{\\bf TARGET} (D=%0.3f kpc%s)}\\\\")'

         forprint, TEXTOUT=5, /NoCOMMENT, SUBSET=ind,$
         CXOU,$
         strtrim(      net_counts            ,2)+'/'+$
         strtrim(round(net_counts/SRC_SIGNIF),2),$
         src_area_pc, src_area_arcsec/(60^2),$

         (tag_exist(bt,'ReducedChiSq') && tag_exist(bt,'DegreesOfFreedom')) ? $
           string(round(bt.ReducedChiSq*bt.DegreesOfFreedom),F='(%"%5d")')+'/'+$
           string(                      bt.DegreesOfFreedom ,F='(%"%3d")') : $
           strarr(num_sources), $

         bt.abundance_CI,$
         bt.NH1_CI, bt.KT1_CI, Tau1_CI, bt.EmissionMeasure1_CI, full_surfbright_corr_component1,$
            NH2_CI,    KT2_CI, Tau2_CI,    EmissionMeasure2_CI, full_surfbright_corr_component2,$
            NH3_CI,    KT3_CI, Tau3_CI,    EmissionMeasure3_CI, full_surfbright_corr_component3,$
            NH4_CI,                        EmissionMeasure5_CI,$
            ; Combined surface brightness,luminosity of components modeling "stars".
            full_surfbright_corr_stars, full_surfbright_corr_stars+alog10(src_area_pc),$
            NH6_CI,    KT6_CI,             EmissionMeasure6_CI, full_surfbright_corr_component6,$
            ; Powerlaw component
            NH7_CI, PH7_CI, full_surfbright_corr_component7, full_surfbright_corr_component7+alog10(src_area_pc),$
            ; Combined surface brightness,luminosity of components modeling "diffuse".
            full_surfbright_corr_diffuse, full_surfbright_corr_diffuse+alog10(src_area_pc),$
            repstr(SpModel,'_','\_'), notes, round(SRC_SIGNIF), cstat, chi_sqr, RS_CI, label, F=fmt

         ; We have to "give up" !TEXTUNIT to prevent the forprint below from closing table_unit.
         !TEXTUNIT = 0
         print
         print, ' seq         name            NH          kT         Norm      NC  Emed      SpModel'
         forprint, SUBSET=ind, seq,                      CATALOG_NAME, bt.NH1_Flags, bt.KT1_Flags, bt.NORM1_Flags, net_counts, median_e, SpModel, F='(%"%4d %s  %s  %s  %s  %4d %3.1f %s")'

END_diffuse_spectroscopy_style2:         
         end ;diffuse_spectroscopy_style2

         
         
       ;--------------------------------------------------------------------------------------------
        else: print, 'No definition found for a table named '+table_name
      endcase
      printf, table_unit, line  ;This is the \enddata line.
    endif ; data section 
    
  endelse ; processing a table
endwhile

free_lun, demo_unit, in_unit
print
print, 'To test tables run:'
print, '    lualatex xray_properties;lualatex xray_properties;  lualatex -jobname=xray_properties "\includeonly{./src_properties,./thermal_spectroscopy,./powerlaw_spectroscopy,./diffuse_spectroscopy}\input{xray_properties}" ; open xray_properties.pdf'
return
end ; hmsfr_tables




;;; ==========================================================================
;;; ae_make_movie
;;; ==========================================================================
; Find Floating underflow; run with /VERBOSE & check flatness of flux & energy; get MPEG license and try MPEG output; energy legend; look at color of overlapping sources; 

; See if /MOTION_VEC reduces file size.

;;; .run ae
;;; ae_make_movie,'theta.cat','1874'
;;; ae_make_movie, SCENE_TEMPLATE='obs1874/data/central_1.emap', $
;;;                SCENE_PARAM_FILE='scene.txt, JPEG_BASENAME='theta'
;;; OR
;;; ae_make_movie,'theta.cat','1874'
;;; ae_make_movie, SCENE_TEMPLATE='obs1874/data/central_1.emap', NUM_FRAMES=100, JPEG_BASENAME='theta'

;;; SCENE_PARAM_FILE is 4 column ASCII describing the scene in each frame:
;;;   time_fraction: [0:1] time tag of frame as fraction in interval [TSTART:TSTOP]
;;;   deltaX, deltaY: panning offset in arcseconds
;;;   deltaScale:     multiplicative adjustment to CDELT (deg/pixel) (e.g. 1.01 means zoom out 1%)

PRO ae_make_movie, catalog_or_srclist, obsname, MIN_COVERAGE=min_coverage, $
                   EXTRACTION_NAME=extraction_name, MERGE_NAME=merge_name, $
  
                   SCENE_TEMPLATE=scene_template, SCENE_PARAM_FILE=scene_param_file, NUM_FRAMES=num_frames, $
                   MPEG_FILENAME=mpeg_filename, JPEG_BASENAME=jpeg_basename, $
                   FWHM=fwhm, SATURATION=saturation, INVERT=invert, $
                   SHOW_FRAMES=show_frames, VERBOSE=verbose

COMMON ae_make_movie, num_sources, sources

creator_string = "ae_make_movie, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()
print, 'http://personal.psu.edu/psb6/TARA/ae_users_guide.html'
print, 'patrick.broos@icloud.com'
print
print, 'Note the following limitations to this method:'
print, '* Light curves and median energies are NOT background subtracted.'
print, '* Sources on multiple CCDs are skipped.'
print, '* Multiple observations are not supported.'
print, '* Bad time intervals are interpolated over.'
print

if (n_elements(min_coverage) NE 1) then min_coverage=0.5
if NOT keyword_set(saturation)   then saturation=0.25

if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,3000)

if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                else merge_subdir = ''
if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)


type = size(obsname,/TNAME)
if (type NE 'UNDEFINED') AND (type NE 'STRING') then begin
  print, 'parameter "obsname" must be a string'
  return
endif

src_stats_basename       = 'source.stats'
lc_smooth_basename       = 'source.smooth_lc'


color_manager

;; =============================================================================
;; READ ALL THE SOURCE INFORMATION INTO MEMORY.

if keyword_set(catalog_or_srclist) then begin
  
  if keyword_set(sources) then begin
    ptr_free, sources.TIME, sources.COUNT_RATE, sources.MEDIAN_ENERGY, sources.psf_img
    dum = temporary(sources)
  endif
 
  ;; Input catalog should be an ascii file with source names, e.g. output of 
  ;; prior call with /CHOOSE_REGIONS.
  readcol, catalog_or_srclist, sourcename, FORMAT='A', COMMENT=';'

  ; Trim whitespace and remove blank lines.
  sourcename = strtrim(sourcename,2)
  ind = where(sourcename NE '', num_sources)
  
  if (num_sources EQ 0) then begin
    print, 'ERROR: no entries read from source list ', catalog_or_srclist
    retall
  endif
  
  sourcename = sourcename[ind]
  print, num_sources, F='(%"\n%d sources found in catalog.\n")'
  
  st = {NAME:'', ra:0D, dec:0D, $
        TIME:ptr_new(), COUNT_RATE:ptr_new(), MEDIAN_ENERGY:ptr_new(), $
        nominal_rate:0.0, nominal_energy:0.0, min_energy:0.0, max_energy:0.0, xl:0, yl:0, psf_img:ptr_new()}
  sources = replicate(st, num_sources)


  for ii = 0L, num_sources-1 do begin
    sources[ii].NAME = sourcename[ii]

    ;; Construct filenames.
    sourcedir = sourcename[ii] + '/'                 + merge_subdir[ii]
    obsdir    = sourcename[ii] + '/' + obsname + '/' + extraction_subdir[ii]
    fit_stats_fn        = sourcedir + src_stats_basename
    lc_smooth_fn    = obsdir + lc_smooth_basename

    if (NOT file_test(fit_stats_fn)) then begin
      print, 'Source ', sourcename[ii], ' not observed.'
      continue
    endif
    
    stats = headfits(fit_stats_fn, ERRMSG=error)
    if (NOT keyword_set(error)) then begin
      sources[ii].ra  = psb_xpar( stats, 'RA')
      sources[ii].dec = psb_xpar( stats, 'DEC')
    endif else print, 'WARNING! Could not read '+fit_stats_fn

    ; Read smoothed LC & median energy time series.
    if (NOT file_test(lc_smooth_fn)) then begin
      print, 'Source ', sourcename[ii], ' skipped; no ', lc_smooth_basename, ' found.'
      continue
    endif
    
    pheader = headfits(lc_smooth_fn)
    t = mrdfits(lc_smooth_fn, 1, theader, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + lc_smooth_fn
    
    tmin = min(t.time, MAX=tmax)
    coverage = (tmax-tmin) / (psb_xpar( pheader, 'TSTOP')-psb_xpar( pheader, 'TSTART'))
    if (coverage LT min_coverage) then begin
      print, sourcename[ii], coverage, F='(%"Source %s skipped; LC coverage is only %4.2f")'
      continue
    endif
    
    if (n_elements(t) EQ 1) then t = [t[0],t[0]]

    sources[ii].TIME          = ptr_new(t.TIME, /NO_COPY)
    sources[ii].MEDIAN_ENERGY = ptr_new(t.MEDIAN_ENERGY / 1000., /NO_COPY)
    ; Some backward compatibility logic here:
    sources[ii].COUNT_RATE    = ptr_new( tag_exist(t, 'COUNT_RATE') ? t.COUNT_RATE : t.RATE, /NO_COPY)
    
    sources[ii].nominal_rate   = median(*(sources[ii].COUNT_RATE))
    sources[ii].nominal_energy = median(*(sources[ii].MEDIAN_ENERGY))
    sources[ii].min_energy     = min   (*(sources[ii].MEDIAN_ENERGY), MAX=max_energy)
    sources[ii].max_energy     = max_energy
  endfor ;ii

  print, fix(total(ptr_valid(sources.TIME))), min_coverage, F='(%"%d sources with >=%4.2f coverage in lc_smooth_basename.")'

  if NOT keyword_set(scene_template) then return
endif ;keyword_set(catalog_or_srclist)




;; =============================================================================
;; BUILD FRAME IMAGES
!EXCEPT=1

ptr_free, sources.psf_img

;; -----------------------------------------------------------------------------
;; Determine which sources are visible in the scene.
refhd = headfits(scene_template)
extast, refhd, refastr
scene_xsize = psb_xpar( refhd, 'NAXIS1')
scene_ysize = psb_xpar( refhd, 'NAXIS2')

scene_xcenter = (scene_xsize-1)/2.0D
scene_ycenter = (scene_ysize-1)/2.0D


if keyword_set(show_frames) then begin
  window, /FREE, XSIZE=scene_xsize<1024, YSIZE=scene_ysize<1024
  show_frames_win = !D.WINDOW
endif


ad2xy, sources.ra, sources.dec, refastr, xindex, yindex 

vis_index = where( ((xindex<yindex) GT 0) AND (xindex LT (scene_xsize-1)) $
                                          AND (yindex LT (scene_ysize-1)) $
                                          AND ptr_valid(sources.TIME), num_vis )

print, num_vis, F='(%"%d sources used for brightness & energy scaling")'
if (num_vis EQ 0) then begin
  return
endif


vis_sources = sources[vis_index]
xindex      = xindex [vis_index]
yindex      = yindex [vis_index]



if NOT keyword_set(fwhm) then begin
  ; Choose a PSF FWHM that's a fixed # of pixels.
  fwhm = 2.0 
endif

psf_exponent = 1.0
help, fwhm, psf_exponent

;; -----------------------------------------------------------------------------
;; Determine time tags of frames.
if keyword_set(scene_param_file) then begin
  readcol, scene_param_file, time_fractions, deltaX_arcsec, deltaY_arcsec, deltaScale, F='F,F,F,F'
  num_frames = n_elements(time_fractions)
  
  if (min(deltaScale) LE 0) then message, 'deltaScale must be positive'
  
endif else if keyword_set(num_frames) then begin
  time_fractions = (0.5 + findgen(num_frames))/num_frames
  deltaX_arcsec  = replicate(0,num_frames)
  deltaY_arcsec  = replicate(0,num_frames)
  deltaScale     = replicate(1,num_frames)
endif else begin
  print, 'You must supply either TIME_TAGS or NUM_FRAMES to specify the desired frame time tags.'
  return
endelse

; Convert time fractions to time tags.
time_fractions = 0 > time_fractions < 1
tstart = psb_xpar( refhd, 'TSTART')
tstop  = psb_xpar( refhd, 'TSTOP')
time_tags = tstart + time_fractions * (tstop-tstart)


if keyword_set(jpeg_basename) then begin
  files_to_remove = findfile( jpeg_basename + '.*.jpg', COUNT=count)
  if (count GT 0) then begin
    print, 'removing: ',files_to_remove
    file_delete, files_to_remove, /ALLOW_NONEXISTENT
  endif
  
  frame_num = lindgen(num_frames)
  jpeg_filenames = jpeg_basename + string(frame_num, F='(%".%4.4d.jpg")')
  forprint, frame_num, time_tags, TEXTOUT=jpeg_basename+'.lis', COMMENT='frame#    time'
endif


;; -----------------------------------------------------------------------------
;; Scale the brightness so the smallest nominal_rate has a brightness of 20%.
;; Choose a brightness rescaling table so that PSF peak pixels of the visible sources
;; at their nominal_rate fluxes would form a flat distribution over [min_level..1].
;; We'll add extra points to this table to extend it's flux range because we may 
;; need to EXTRAPOLATE the scaling 
;; curve beyond the range of the sources' nominal fluxes.  For example when the
;; brightest source flares we need to extrapolate beyond the nominal range.
;; This is a kind of histogram equalization scaling.
;; Only sources with positive nominal_rate fluxes are considered.
min_level = 0.2
max_level = 1.1

ind = where( vis_sources.nominal_rate GT 0, num_samples )
if (num_samples EQ 0) then begin
  print, 'All sources have zero nominal_rate fluxes; cannot continue.'
  return
endif

temp                 = vis_sources[ind].nominal_rate
nominal_rate_samples = temp[sort(temp)]

if (1) then begin
scaled_rate_samples  = (min_level + (max_level-min_level) * findgen(num_samples) / float(num_samples-1)) 
endif else begin
endelse

lin_params = linfit( nominal_rate_samples, scaled_rate_samples )

large_val = 10*max(nominal_rate_samples)
nominal_rate_samples = [0,nominal_rate_samples,large_val]
scaled_rate_samples  = [0,scaled_rate_samples, lin_params[0] + lin_params[1]*large_val]

if keyword_set(verbose) then begin
  function_1d, id2, nominal_rate_samples, scaled_rate_samples,  LINE=6, PSYM=1, XTIT='nominal rate', YTIT='central pixel value'
endif


;; -----------------------------------------------------------------------------
;; Construct an energy scaling table that makes good use of the color spectrum.
;; We'll use a table which results in a nearly flat distribution of scaled nominal energies
;; in the range [0..1].
;; We'll add extra points to this table to extend it's energy range because we may 
;; need to EXTRAPOLATE the scaling 
;; curve beyond the range of the sources' nominal energies.  For example
;; if the source with the hardest nominal energy gets even harder during a flare then
;; we'll need to scale an energy value outside the range of energies used to construct
;; the scaling curve.
;; This is a kind of histogram equalization scaling.
nominal_energy_samples = (vis_sources.nominal_energy)[sort( vis_sources.nominal_energy )]

scaled_energy_samples  = findgen(num_vis) / float(num_vis-1)

lin_params = linfit( nominal_energy_samples, scaled_energy_samples )

large_val = 10
nominal_energy_samples = [0,nominal_energy_samples,large_val]
scaled_energy_samples  = [0,scaled_energy_samples, lin_params[0] + lin_params[1]*large_val]

;; Determine max & min scaled energies that are possible.
linterp, nominal_energy_samples, scaled_energy_samples, min(vis_sources.min_energy), low_scaled_energy
linterp, nominal_energy_samples, scaled_energy_samples, max(vis_sources.max_energy), high_scaled_energy

if keyword_set(verbose) then begin
  function_1d, id4, nominal_energy_samples, scaled_energy_samples,  LINE=6, PSYM=1, XTIT='nominal energy', YTIT='scaled energy'
  help, low_scaled_energy, high_scaled_energy
endif


;; -----------------------------------------------------------------------------
;; Figure out an appropriate way to normalize PSF to 1.
wing_scale = 1
npix     = [100,100]
centroid = [50,50]
psf_raw = (  psf_gaussian(NPIX=npix, FWHM=fwhm,            CENTROID=centroid) + $
           2*psf_gaussian(NPIX=npix, FWHM=fwhm*wing_scale, CENTROID=centroid))
psf_peak = max( psf_raw )
help, psf_peak

if keyword_set(mpeg_filename) then $
  mpegID = mpeg_open( [scene_xsize,scene_ysize], MOTION_VEC_LENGTH=1, QUALITY=100 )

for frame_num = 0, num_frames-1 do begin

  ;; -----------------------------------------------------------------------------
  ;; Move scene center by the specified offset (in arcseconds) and adjust zoom.
  ; Convert specified offset for this frame from arcsec to pixels.
  cdelt = abs(refastr.CDELT[0])
  deltaX_pix = (deltaX_arcsec[frame_num] / 3600.) / cdelt
  deltaY_pix = (deltaY_arcsec[frame_num] / 3600.) / cdelt
  
  ; Find RA,DEC at current scene center
  xy2ad, scene_xcenter, scene_ycenter, refastr, scene_ra_center, scene_dec_center
  
  ; Redefine astrometry so that RA,DEC is offset as specified from the scene center.
  ; One is added to new scene center because FITS CRPIX values are 1-based.
  refastr.CRPIX = 1 + [scene_xcenter+deltaX_pix, scene_ycenter+deltaY_pix]
  refastr.CRVAL =     [scene_ra_center, scene_dec_center]
  
  ; Adjust zoom as specified.
print, refastr.CRVAL
;print, deltaScale[frame_num]
  refastr.CDELT = refastr.CDELT * deltaScale[frame_num]


  ;; -----------------------------------------------------------------------------
  ;; Compute positions of sources & find ones visible.
  ad2xy, sources.ra, sources.dec, refastr, xindex, yindex 

  vis_index = where( ((xindex<yindex) GT 0) AND (xindex LT (scene_xsize-1)) $
                                            AND (yindex LT (scene_ysize-1)) $
                                            AND ptr_valid(sources.TIME), num_vis )

  if (num_vis NE 0) then begin
    vis_sources = sources[vis_index]
    xindex      = xindex [vis_index]
    yindex      = yindex [vis_index]
    rates_this_frame    = fltarr(num_vis)
    energies_this_frame = fltarr(num_vis)
  endif else begin
    rates_this_frame    = fltarr(2)
    energies_this_frame = fltarr(2)
  endelse

  scene_brightness   = fltarr(scene_xsize,scene_ysize)
  scene_energy_sum   = fltarr(scene_xsize,scene_ysize)
  
  print, frame_num, deltaX_pix, deltaY_pix, num_vis, F='(%"Frame %d; offset=(%5.2f,%5.2f); %d visible sources")' 

  ;; -----------------------------------------------------------------------------
  ;; Build frame of the scene at the specified time tag.
  for ii = 0, num_vis-1 do begin
    ; Sample the COUNT_RATE and MEDIAN_ENERGY time series at the requested time tags 
    ; and send through the scaling tables constructed earlier.
    ; It's vital to use linterp so that no linear extrapolation will be done. ???

    linterp, *(vis_sources[ii].TIME), *(vis_sources[ii].COUNT_RATE),    time_tags[frame_num], rate
    linterp, *(vis_sources[ii].TIME), *(vis_sources[ii].MEDIAN_ENERGY), time_tags[frame_num], median_energy
    
    linterp, nominal_rate_samples,   scaled_rate_samples,   rate,          scaled_rate
    linterp, nominal_energy_samples, scaled_energy_samples, median_energy, scaled_energy
    rates_this_frame[ii]    = scaled_rate
    energies_this_frame[ii] = scaled_energy

    
    ; Generate a PSF image and add to the scene.  Add weighted energy image to running sum.
    ; There are some floating underflow exceptions we have to mask.
    xl = floor(xindex[ii] - fwhm*wing_scale) > 0
    yl = floor(yindex[ii] - fwhm*wing_scale) > 0
  
    xh = ceil (xindex[ii] + fwhm*wing_scale) < (scene_xsize-1)
    yh = ceil (yindex[ii] + fwhm*wing_scale) < (scene_ysize-1)
    
    save_except = !EXCEPT & !EXCEPT = 0
;    psf_img = (scaled_rate) * $
;             ( psf_gaussian(NPIX=[1+xh-xl,1+yh-yl], FWHM=fwhm, $
;                            CENTROID=[xindex[ii]]-xl,yindex[ii]-yl]) / psf_peak )^psf_exponent

    npix     = [1+xh-xl,1+yh-yl]
    centroid = [xindex[ii]-xl,yindex[ii]-yl]
    psf_raw = (  psf_gaussian(NPIX=npix, FWHM=fwhm,            CENTROID=centroid) + $
               2*psf_gaussian(NPIX=npix, FWHM=fwhm*wing_scale, CENTROID=centroid)) 
    
    psf_img = (scaled_rate/psf_peak) * psf_raw
;print, max(psf_img)                        
    scene_brightness[xl,yl] = scene_brightness[xl:xh,yl:yh] + psf_img
    scene_energy_sum[xl,yl] = scene_energy_sum[xl:xh,yl:yh] + psf_img * scaled_energy

    error = check_math(MASK=32)    ;Clear floating underflow
    !EXCEPT = save_except

    
    if (ii GT 0) AND ((ii MOD 100) EQ 0) then print, ii, ' sources processed'
  endfor ;ii

  ; Apply normalization step in the computation of weighted energy image.
  scene_energy = scene_energy_sum / scene_brightness
  ind = where( finite( scene_energy, /NAN ), count ) 
  if (count GT 0) then scene_energy[ind] = 0

  if keyword_set(verbose) then begin
    dataset_1d, id1, rates_this_frame
    dataset_1d, id3, energies_this_frame
    s='' & read,s
  endif

  ;; -----------------------------------------------------------------------------
  ;; Construct color image using the HSV model.
  
  ; Scale scene_energy to a normalized hue image
  hue_norm = (scene_energy - low_scaled_energy) / ((high_scaled_energy - low_scaled_energy)>1e-8)

  tara_hsv2rgb,  0.0 > hue_norm < 1.0, saturation, scene_brightness > 0, 0, $
                red_data, grn_data, blu_data

  ; This is the place we would introduce a user-supplied background for the scene, e.g. 
  ; red, green, blue planes from a diffuse emission analysis.
  ; Add those into the point source planes, e.g. 
  ; red_data = red_data +  red_bkg * (0.3 / max(red_bkg))
    
  if keyword_set(show_frames) then begin
    wset, show_frames_win
    color_manager, /X_TRUE
  endif
  
  ; Its IMPORTANT to specify HIGH_VALUE below so that all frames are scaled the same!
  ; A HIGH_VALUE < 1 will "clip" the bright stars.
  rgb_scale, red_data, grn_data, blu_data, $
             LOW_VALUE =0, LOW_NORM=0, $
             HIGH_VALUE=0.8, HIGH_NORM=1, INVERT=keyword_set(invert), $
             red_channel, grn_channel, blu_channel, DO_PLOT=keyword_set(show_frames)

  ; Add graphic showing time_fractions.
  bar_height = ceil(0.05*scene_ysize)
  xx = round( (scene_xsize-1) * time_fractions[frame_num] ) 
  red_channel[xx, 0:bar_height] = 1
  grn_channel[xx, 0:bar_height] = 1
  blu_channel[xx, 0:bar_height] = 1
  
  
  ;; -----------------------------------------------------------------------------
  ;; Save color image.
  img = bytarr(3,scene_xsize,scene_ysize)
         
  num_levels = 256       
  img[0,*,*] = floor( num_levels * red_channel ) < (num_levels-1)
  img[1,*,*] = floor( num_levels * grn_channel ) < (num_levels-1)
  img[2,*,*] = floor( num_levels * blu_channel ) < (num_levels-1)

  if keyword_set(mpeg_filename) then $
    mpeg_put, mpegID, FRAME=frame_num, IMAGE=img
  
  if keyword_set(jpeg_basename) then $
    write_jpeg, jpeg_filenames[frame_num], img, TRUE=1, QUALITY=100
endfor; frame_num

if keyword_set(mpeg_filename) then begin
  print, 'Writing ', mpeg_filename, ' ...'
  mpeg_save,  mpegID,  FILENAME=file_expand_path(mpeg_filename)
  mpeg_close, mpegID
endif
        
return
end


;; =============================================================================
;; =============================================================================
PRO test_psfs

img=psf_gaussian(NPIX=200, FWHM=fwhm, CENTROID=100,NDIM=1)
x=indgen(200)
function_1d,id,x,img

; We'd like stars to get "larger" when they flare, but am having a hard time finding a PSF that gives the right look.
; Perhaps FWHM should vary with flux!  That would slow down code.

; This one seems disk-like in movie.
limg = alog(img) > (-exp(1))
function_1d,id,x,limg

; This one seems fuzzy is movie.
function_1d,id,x,img^0.25
return
end


;; =============================================================================
;; =============================================================================
PRO test_colors
show_frames=1

scene_xsize = 300
scene_ysize = 300
color_manager

if keyword_set(show_frames) then begin
  window, /FREE, XSIZE=scene_xsize<1024, YSIZE=scene_ysize<1024
  show_frames_win = !D.WINDOW
endif

ima=psf_gaussian(NPIX=[scene_xsize,scene_ysize],FWHM=90,CENTROI=[100,100])
imb=psf_gaussian(NPIX=[scene_xsize,scene_ysize],FWHM=90,CENTROI=[200,100])
imc=psf_gaussian(NPIX=[scene_xsize,scene_ysize],FWHM=90,CENTROI=[150,200])
scene_brightness=ima+imb+imc
scene_energy_sum=ima+2*imb+4*imc

scene_energy = scene_energy_sum / scene_brightness
ind = where( finite( scene_energy, /NAN ), count ) 
if (count GT 0) then scene_energy[ind] = 0

hue_norm = scene_energy / max(scene_energy)

;function_2d,id,scene_energy

  saturation = 0.5

  tara_hsv2rgb, hue_norm, saturation, scene_brightness > 0, 0, $
                red_data, grn_data, blu_data

  if keyword_set(show_frames) then begin
    wset, show_frames_win
    color_manager, /X_TRUE
  endif

  
  rgb_scale, red_data, grn_data, blu_data, $
             LOW_VALUE =0,  LOW_NORM=0, $
             HIGH_VALUE=0.8, HIGH_NORM=1, $
             red_channel, grn_channel, blu_channel, DO_PLOT=keyword_set(show_frames)

stop
function_2d,id,hue_norm,DATA='hue_norm'
function_2d,id,hue_img,DATA='hue_img'
function_2d,id,red_data,DATA='red_data'
function_2d,id,grn_data,DATA='grn_data'
function_2d,id,blu_data,DATA='blu_data'

return
end
  
;hue_norm=0.5 + 0.2*findgen(200)/199.
;value_data=replicate(1,200)
;make_2d,hue_norm,value_data

;; =============================================================================
;; =============================================================================
PRO make_scene_params, num_frames, filename

openw, unit, filename, /GET_LUN
!TEXTUNIT = unit

; One cycle on full field.
zero = replicate(0,num_frames)
one  = replicate(1,num_frames)
forprint, TEXTOUT=5, findgen(num_frames)/(num_frames-1), zero, zero, one, /NoCOMMENT

; One cycle while linear zooming.
; For f3.reg
xoffset =   8.86 ;arcsec
yoffset = -78.7 ; arcsec
zoom    =  4.69 

deltaX     = replicate(xoffset/float(num_frames)      , num_frames)
deltaY     = replicate(yoffset/float(num_frames)      , num_frames)
deltaScale = replicate(exp( -alog(zoom) / num_frames ), num_frames)
forprint, TEXTOUT=5, findgen(num_frames)/(num_frames-1), deltaX, deltaY, deltaScale, /NoCOMMENT

; One cycle on zoomed field.
forprint, TEXTOUT=5, findgen(num_frames)/(num_frames-1), zero, zero, one, /NoCOMMENT

free_lun, unit

return
end



;; =============================================================================
;; =============================================================================
PRO plot_spectrum, sourcename

base = sourcename +'/' + sourcename
src = mrdfits(base + '.pi',     1, src_header)
bkg = mrdfits(base + '_bkg.pi', 1, bkg_header)

energy = 14.45/1000.*(1+indgen(n_elements(src)))

backscal = psb_xpar( src_header, 'BACKSCAL') / bkg.BACKSCAL
spectrum = src.counts - backscal * bkg.counts

plot, energy, spectrum[0:547], PSYM=10

ind = where(src.counts NE 0)
function_1d, id, energy[ind], (src.counts)[ind],             LINE=6, PSYM=4, DATASET='source'
ind = where(bkg.counts NE 0)
function_1d, id, energy[ind], -backscal * (bkg.counts)[ind], LINE=6, PSYM=1, DATASET='background'

return
end



;; =============================================================================
;; =============================================================================
PRO ae_build_model_changes_file, parameter_name, sourcename,$
                                 parameter_min, parameter_init, parameter_max,$
                                 model_changes_filename, MERGE_NAME=merge_name, $
                                 COMMENT=comment, WROTE_FILE=wrote_file

; Verify input vectors are the same length.
num_sources = max([n_elements(sourcename), n_elements(parameter_min), n_elements(parameter_init), n_elements(parameter_max)], MIN=Nmin)

if (num_sources NE Nmin) then begin
  print, 'ERROR: vectors sourcename, parameter_min, parameter_init, parameter_max must be the same length.'
  retall
endif

if ~keyword_set(comment) then comment = strarr(num_sources)
if  (n_elements(comment) NE num_sources) then begin
  print, 'ERROR: vector COMMENT must be the same length as other inputs.'
  retall
endif


modelsubdir              = 'spectral_models/'

sourcename = strtrim(sourcename,2)

if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                else merge_subdir = ''
if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)

num_scripts_written = 0L
min_range           = !VALUES.F_INFINITY
wrote_file          = bytarr(num_sources)

for ii=0, num_sources-1 do begin
  if ~finite(parameter_min [ii]) || ~finite(parameter_init[ii]) || ~finite(parameter_max [ii]) then continue

  basedir   = sourcename[ii] + '/' 
  sourcedir = basedir + merge_subdir[ii]
  
  file_mkdir, sourcedir + modelsubdir
  
  openw, unit, sourcedir + modelsubdir + model_changes_filename, /GET_LUN
  if keyword_set(comment[ii]) then $
  printf, unit,  comment[ii],                       F='(%"# %s")'
  printf, unit, parameter_name, parameter_min [ii], F='(%"  set custom_value(%s_min)  %0.2f")'
  printf, unit, parameter_name, parameter_init[ii], F='(%"  set custom_value(%s_init) %0.2f")'
  printf, unit, parameter_name, parameter_max [ii], F='(%"  set custom_value(%s_max)  %0.2f")'
  printf, unit, ''
  printf, unit, '  foreach variable_name [array names custom_value] {'
  printf, unit, '    if [info exists $variable_name] {'
  printf, unit, '      set $variable_name $custom_value($variable_name)'
  printf, unit, '    } else {'
  printf, unit, '      puts "ERROR: ''$variable_name'' is not the name of a variable in the fitting script!"'
  printf, unit, '      tclexit 98'
  printf, unit, '    }'
  printf, unit, '  }'
  free_lun, unit
  num_scripts_written++
  wrote_file[ii] = 1
  min_range <= (parameter_max [ii] - parameter_min [ii])
endfor ;ii
print, model_changes_filename, num_scripts_written, num_sources, parameter_name, min_range, F='(%"\nWrote %s for %d out of %d sources; min range for %s was %0.2f.")'
return
end ; 



;;; ==========================================================================
;;; Set default spectral model preference.
;;; Use this tool to establish a default spectral model preference (stored in the 
;;; keyword BEST_MDL in the primary HDU of source.spectra) prior to running the
;;; interactive tool ae_spectra_viewer
;;; The parameter 'hduname' is a regular expression understood by strmatch().
;;; The parameter 'hduname' can be a scalar or a vector.
;;; ==========================================================================

PRO ae_default_model_preference, catalog_or_srclist, hduname, FORCE=force, $
                  MERGE_NAME=merge_name

modelsubdir              = 'spectral_models/'
fit_stats_basename       = 'source.spectra'
src_stats_basename       = 'source.stats'

if ~isa(/STRING, catalog_or_srclist) then begin
  print, 'ERROR: the "catalog_or_srclist" parameter must be a string or string array.' 
  retall
endif

;; Interpret the parameter catalog_or_srclist.
if array_equal(file_test(catalog_or_srclist,/DIRECTORY),1) then begin
  ; A list of source names (source directories) has been passed.
  sourcename  = catalog_or_srclist
  num_sources = n_elements(sourcename)
  if (verbose GT 0) then print, num_sources, F='(%"\n%d source names passed.\n")'
  
endif else if (n_elements(catalog_or_srclist) EQ 1) && file_test(catalog_or_srclist,/REGULAR) then begin
  ; A single filename should be an ASCII file containing a list of source names. 
  readcol, catalog_or_srclist, sourcename, FORMAT='A', COMMENT=';', COUNT=num_sources
  
  if (num_sources EQ 0) then begin
    print, 'WARNING: catalog contains zero sources; AE run aborted.'
    retall
  endif
  
  ; Trim whitespace and remove blank lines.
  sourcename = strtrim(sourcename,2)
  ind = where(sourcename NE '', num_sources)
  
  if (num_sources EQ 0) then begin
    print, 'ERROR: no entries read from source list ', catalog_or_srclist
    retall
  endif
  
  sourcename = sourcename[ind]
  if (verbose GT 0) then print, num_sources, F='(%"\n%d sources found in catalog.\n")'
  
endif else begin
  ; Cannot figure out what catalog_or_srclist means ...
  help, catalog_or_srclist
  print, 'ae_default_model_preference: ERROR, the "catalog_or_srclist" parameter is neither an existing ASCII file nor a list of source names.' 
  retall
endelse



if (n_elements(hduname) EQ 1) then hduname = replicate(hduname, num_sources)

if (n_elements(hduname) NE num_sources) then begin
  print, 'ERROR: # of sources in catalog does not match length of hduname vector.'
  retall
endif

if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                else merge_subdir = ''
if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)


;; Create a unique scratch directory.
temproot = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)
tempdir = temproot

run_command, PARAM_DIR=tempdir

for ii = 0L, num_sources-1 do begin
  basedir   = sourcename[ii] + '/' 
  sourcedir = basedir + merge_subdir[ii]

 ; We assume that an existing source directory that is a symbolic link should not be written to.
  temp = file_info(basedir)
  is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
  if ~is_writable then begin
    print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
    continue
  endif 

  print, sourcename[ii], F='(%"\nSource: %s")'
  
  ;; Find the HDUs in source.spectra.
  fit_stats_fn  = sourcedir + fit_stats_basename

  fit_stats_keywords = get_keywords_from_hdu_headers(fit_stats_fn, {BEST_MDL:'', HDUNAME:''}, ERRMSG=open_error)
    
  if keyword_set(open_error) then begin
    print, 'No spectral models found.'
    continue
  end
  
  ; If an existing BEST_MDL declaration found, respect it.
  existing_BEST_MDL = fit_stats_keywords[0].BEST_MDL
  if keyword_set(existing_BEST_MDL) then begin
    if keyword_set(force) then begin
      print, 'Overriding existing preference for ', existing_BEST_MDL
    endif else begin
      print, 'Respecting existing preference for ', existing_BEST_MDL
      continue
    endelse
  endif

  ; Find the last HDU matching the specified model name spec.
  ind = where( strmatch(fit_stats_keywords.HDUNAME, hduname[ii], /FOLD_CASE), count )
  
  if (count GT 1) then begin
    ind = ind[count-1]
    print, count, hduname[ii], fit_stats_keywords[ind].HDUNAME,  F='(%"  %d spectral models match %s; using the most recent: %s )")'
    count = 1
  endif
      
  if (count EQ 0) then begin
    print, hduname[ii], sourcename[ii], F='(%"WARNING! Cannot find any spectral model matching ''%s'' for %s")'
    continue
  endif
  
  ;; Write the preference.
  cmd = string(fit_stats_fn, fit_stats_keywords[ind].HDUNAME, $
                 F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=BEST_MDL value=""%s"" comment=""default preferred model""')")

  run_command, cmd
  
  ; Delete any "best model" symlink we find, and remake it.
  link_name = sourcedir + modelsubdir + 'best_model'
  if file_test(link_name, /DANGLING_SYMLINK) OR file_test(link_name, /SYMLINK) then file_delete, link_name
  if file_test(link_name) then begin
    print, 'WARNING: existing regular file '+link_name+' has not been changed.'
  endif else begin
    file_link, fit_stats_keywords[ind].HDUNAME, link_name
  endelse
   
  
endfor ;ii

CLEANUP:
if file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif
return
end ; ae_default_model_preference



;;; ==========================================================================
;;; Verify consistency between BEST_MDL declaration (in source.spectra) and
;;; symlink spectral_models/best_model.
;;; ==========================================================================
PRO ae_validate_best_model_declaration, catalog_or_srclist, MERGE_NAME=merge_name, FORCE=force

modelsubdir              = 'spectral_models/'
fit_stats_basename       = 'source.spectra'
src_stats_basename       = 'source.stats'

readcol, catalog_or_srclist, sourcename, FORMAT='A', COMMENT=';'

; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)

if (num_sources EQ 0) then begin
  print, 'ERROR: no entries read from source list ', catalog_or_srclist
  retall
endif

sourcename = sourcename[ind]
print, num_sources, F='(%"\n%d sources found in catalog.\n")'

if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                else merge_subdir = ''
if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)


;; Create a unique scratch directory.
temproot = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)
tempdir = temproot

run_command, PARAM_DIR=tempdir

for ii = 0L, num_sources-1 do begin
  basedir   = sourcename[ii] + '/' 
  sourcedir = basedir + merge_subdir[ii]
  link_name = sourcedir + modelsubdir + 'best_model'

 ; We assume that an existing source directory that is a symbolic link should not be written to.
  temp = file_info(basedir)
  is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
  if ~is_writable then begin
    print, sourcename[ii], F='(%"\nWARNING: skipping protected source %s.")'
    continue
  endif 

  
  ;; Find the HDUs in source.spectra.
  fit_stats_fn  = sourcedir + fit_stats_basename

  fit_stats_keywords = get_keywords_from_hdu_headers(fit_stats_fn, {BEST_MDL:'', HDUNAME:''}, ERRMSG=open_error)
    
  if keyword_set(open_error) then begin
    print, sourcedir, F='(%"\nINFORMATION: No spectral models found for source %s.")'
    continue
  end
  
  ; Find an existing BEST_MDL declaration.
  existing_BEST_MDL = fit_stats_keywords[0].BEST_MDL
  if ~keyword_set(existing_BEST_MDL) then begin
    if file_test(link_name) then begin
      print, sourcename[ii], link_name, F='(%"\nERROR: source %s has no BEST_MDL declaration, but %s exists.  Something is wrong!")'
    endif else begin
      print, sourcename[ii], F='(%"\nINFORMATION: source %s has no BEST_MDL declaration.")'
    endelse
    continue
  endif

  ; Verify that existing BEST_MDL declaration refers to an existing HDU.
  if array_equal(fit_stats_keywords.HDUNAME NE existing_BEST_MDL, 0B) then begin
    print,  fit_stats_fn, existing_BEST_MDL, F='(%"\nERROR: %s has no HDU corresponding to BEST_MDL declaration (%s).")'
  endif

  ; Verify that existing BEST_MDL declaration refers to an existing model directory.
  model_dir = sourcedir + modelsubdir + existing_BEST_MDL
  if ~file_test(model_dir) then begin
    print, model_dir, existing_BEST_MDL, F='(%"\nERROR: model directory %s corresponding to BEST_MDL declaration (%s) is missing.")'
  endif

  ; Verify that symlink "best_model" exists, correponds to BEST_MDL declaration, and is a symlink.
  case 1 of
    file_test(link_name, /DANGLING_SYMLINK): begin
      print, link_name, F='(%"\nERROR: %s is a dangling symlink.")'
      end

    ~file_test(link_name): begin
      print, link_name, F='(%"\nREPAIR: created missing symlink %s")'
      file_link, /VERBOSE, existing_BEST_MDL, link_name
      end

    file_test(link_name, /SYMLINK): begin
      if (file_readlink(link_name) NE existing_BEST_MDL) then $
        print, link_name, existing_BEST_MDL, F='(%"\nERROR: symlink %s does not point to declared best model (%s).")'
      end

    file_test(link_name): begin
      run_command, /UNIX, "ls -o "+link_name+"|egrep -v '^total'", result, /QUIET
      print, link_name, F='(%"\nERROR: %s is a directory (not symlink), containing:")'
      forprint, result

      msg = ['DO YOU WISH TO DISCARD THE FOLLOWING FILES in '+link_name+' ?', result]
      TimedMessage, tm_id, msg, TITLE='ae_validate_best_model_declaration: "best_model" is a directory!', QUIT_LABEL='No, do nothing.', BUTTON_LABEL='Yes, DISCARD those files.', BUTTON_ID=button_id, QUIT_ID=quit_id, PRESSED_ID=pressed_id
  
      if (pressed_id EQ button_id) then begin
        file_delete, link_name, /RECURSIVE, /VERBOSE
      endif
      end
      
    else: begin
      print, link_name, '  Unhandled branch in case statement!'
      stop
      end
  endcase

endfor ;ii

CLEANUP:
if file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif
return
end ; ae_validate_best_model_declaration



;; =============================================================================
;; spectra_viewer tool
;;
;; Example: ae_spectra_viewer, 'tables/xspec.collated', KEY=['NH','KT','PH','FH8']
;; =============================================================================

;; =============================================================================
;; Procedure to handle GUI updates when a source is selected.
PRO ae_spectra_viewer_select_source, st, source_index

modelsubdir              = 'spectral_models/'
fit_stats_basename       = 'source.spectra'
src_stats_basename       = 'source.stats'

; Range check the source number.
num_sources  = n_elements((*st).sourcename)
source_index = 0 > source_index < (num_sources-1)

(*st).current_source_index = source_index


; Now we can display information about the newly-selected source.
this_sourcename   = (*st).sourcename  [source_index]
this_source_label = (*st).source_label[source_index]
this_merge_subdir = (*st).merge_subdir[source_index]

sourcedir = this_sourcename + '/' + this_merge_subdir 

widget_control, (*st).source_name_id ,  SET_VALUE=this_sourcename
widget_control, (*st).merge_subdir_id,  SET_VALUE=this_merge_subdir
widget_control, (*st).source_label_id,  SET_VALUE=this_source_label
widget_control, (*st).source_number_id, SET_VALUE=1+source_index

; Look for the presentation_rank of the selected source.
current_rank = ((*st).presentation_rank)[source_index]
if (current_rank GE 0) then (*st).current_rank = current_rank

; Update the presentation rank index widget.
; The index current_rank is 0-based; the widget wants to show a 1-based index.
if ((*st).rank_id GT 0) then $
  widget_control, (*st).rank_id, SET_VALUE=1+(*st).current_rank

print, sourcedir, this_source_label, F='(%"\n----------------------------\n%s (%s)")'
this_note = (*st).notes[source_index]
if keyword_set(this_note) then print, this_note


;--------------------------------------------------------------------------
;; Display the source statistics.
src_stats_fn = sourcedir + src_stats_basename                                          
src_stats    = headfits(src_stats_fn, ERRMSG=error)
if (~keyword_set(error)) then begin
  sxdelpar, src_stats, ['SIMPLE','BITPIX','NAXIS','EXTEND','DATE','CREATOR','RA_ML','DEC_ML','QUANTML','QUANTCOR','RA_CORR','DEC_CORR','RA_DATA','DEC_DATA','ERR_DATA','ERX_DATA','ERY_DATA','RA_PSF','DEC_PSF','END']

  if (widget_info( (*st).stats_header_id, /VALID_ID )) then $
    widget_control, (*st).stats_header_id, SET_VALUE=src_stats
endif

;--------------------------------------------------------------------------
;; Prepare the spectral models table.
num_models  = (*st).num_models

num_columns = n_elements((*st).keylist)  

; Clear the table     
if (num_models GT 0) then begin
  ; Unfortunately we cannot get the table to tell us how many rows it has, not even by GET_VALUE., so we have to keep track ourselves.
;  widget_control, (*st).model_table, /DELETE_ROWS, USE_TABLE_SELECT=[0,0,num_columns-1,(*st).num_models-1]
  all_rows = replicate(-1,2,(*st).num_models)
  all_rows[1,*] = indgen((*st).num_models)
  widget_control, (*st).model_table, /DELETE_ROWS, USE_TABLE_SELECT=all_rows

  num_models = 0
endif


;--------------------------------------------------------------------------
;; Find the HDUs in source.spectra.
fit_stats_fn  = sourcedir + fit_stats_basename

; Note that get_keywords_from_hdu_headers() will return the value 0B for fields of type BYTE when the corresponding FITS keyword is missing.
fit_stats_keywords = get_keywords_from_hdu_headers(fit_stats_fn, {BEST_MDL:'', HIST_MDL:'', PROVISNL:0B, NOCOLLAT:0B, CATEGORY:'', HDUNAME:''}, ERRMSG=open_error)

model_names = ''
selected_model_index = -1
selected_model_name   = strtrim(fit_stats_keywords[0].BEST_MDL,2)
historical_model_name = strtrim(fit_stats_keywords[0].HIST_MDL,2)


if keyword_set(open_error) then begin
  print, 'ERROR: cannot open '+fit_stats_fn
  ; Clear the source statistics.
  widget_control, (*st).src_signif_id , SET_VALUE=''
  widget_control, (*st).median_E_id   , SET_VALUE=''
  widget_control, (*st).net_cnts_id   , SET_VALUE=''
  widget_control, (*st).exposure_id   , SET_VALUE=''
 ;widget_control, (*st).ag_frac_id    , SET_VALUE=''
  widget_control, (*st).psf_frac_id   , SET_VALUE=''
  widget_control, (*st).theta_id      , SET_VALUE=''
  widget_control, (*st).variability_id, SET_VALUE=''

  num_models = 0
endif else  num_models = n_elements(fit_stats_keywords) - 1


if (num_models GT 0) then begin
    model_names = strtrim(fit_stats_keywords[1:*].HDUNAME, 2)
;forprint, model_names

    ; Retain models specified by  HDUNAME search pattern interpreted by strmatch().
    ; Retain BEST_MDL.
    show_model = (model_names NE '...')
    show_model AND= strmatch(model_names, (*st).hduname, /FOLD_CASE)
    show_model[ where(/NULL, model_names EQ   selected_model_name) ] =1
    show_model[ where(/NULL, model_names EQ historical_model_name) ] =1

    ind = where( show_model, num_models )

    if (num_models GT 0) then begin
      model_names = model_names[ind]
      
;     ; SORT the model names to be convenient for our standard spectral modeling recipe (recipe.txt).
;     name_template = ['*_A*', '*_B*', '*_C*', '*_D*', '*_E*', '*_std1*', '*_std2*', '*kT_max*', '*_pow*']
;     tail_index = 0L
;     for jj = 0,n_elements(name_template)-1 do begin
;       ind = where(strmatch(model_names, name_template[jj], /FOLD_CASE), count)
;       
;       ; Process each match.
;       for kk=0L,count-1 do begin
;         this_index = ind[kk]
;         
;         ; Only process matches found in the unsorted tail.
;         if (this_index LT tail_index) then continue
;       
;         ; Swap the matched model with the first model in the unsorted tail.
;         temp                    = model_names[this_index]
;         model_names[this_index] = model_names[tail_index]
;         model_names[tail_index] = temp
;         tail_index++
;       endfor ;kk
;     endfor ;jj
      
      if (*st).sort_by_model_name then begin
        model_names = model_names[sort(model_names)]
      endif

      if (*st).reverse_sort_by_model_name then begin
        model_names = model_names[reverse(sort(model_names))]
      endif
      
  ;      case widget_info( (*st).plot_name_list, /DROPLIST_SELECT ) of
  ;        0: name = '/ldata.ps'
  ;        1: name = '/icounts.ps'
  ;      endcase
      
      plot_files = strarr(num_models,2)
      plot_files[*,0] = sourcedir + modelsubdir + model_names + '/ldata.ps'
      plot_files[*,1] = sourcedir + modelsubdir + model_names + '/icounts.ps'
;forprint, plot_files[*,1]

      for ii=0,n_elements(plot_files)-1 do begin
        if NOT file_test(plot_files[ii]) then plot_files[ii] = (*st).no_model_file
      endfor
    endif ; (num_models GT 0)
endif ; (num_models GT 0)

num_gv_pairs  = n_elements((*(*st).gv_pids)) / 2  

num_gv_pairs_needed = (num_models > num_gv_pairs)

;--------------------------------------------------------------------------
;; Refresh the existing gv displays, and create more if needed.

;; NOTE that the gv manual says that you can send it a signal
;;    kill -HUP
;; to reload, rather than using the --watch option.  But it doesn't work on our X11 environment.
;;
;; I also find that neither symlinks nor hard links will reliably induce gv to redraw.
;; Thus, we're forced to COPY the PostScript files we want to display.

if (num_gv_pairs_needed GT 0) then begin
  ;; Display the plots.

  gv_files= strarr(num_gv_pairs_needed,2)
  gv_files[*,0] = (*st).tempdir + string(indgen(num_gv_pairs_needed), F='(%"%d_g")')                
  gv_files[*,1] = (*st).tempdir + string(indgen(num_gv_pairs_needed), F='(%"%d_c")')                
  
  ; Link up the files we need to display to the gv processes and
  ; link the null file to any extra gv processes.
  for ii=0,num_gv_pairs_needed-1 do begin
    if (ii LT num_models) then begin
      file_copy, /OVERWRITE,/FORCE,   plot_files[ii,0],  gv_files[ii,0] 
      file_copy, /OVERWRITE,/FORCE,   plot_files[ii,1],  gv_files[ii,1] 
    endif else begin
      file_copy, /OVERWRITE,/FORCE, (*st).no_model_file, gv_files[ii,0]
      file_copy, /OVERWRITE,/FORCE, (*st).no_model_file, gv_files[ii,1]
    endelse
  endfor

  ; Do we need more gv sessions?
  if (num_gv_pairs_needed GT num_gv_pairs) then begin
    gv_logfile = (*st).tempdir + 'gv.log'
    gv_pids = strarr(num_gv_pairs_needed,2)
    if (num_gv_pairs GT 0) then gv_pids[0,0] = *(*st).gv_pids $
                           else file_delete, gv_logfile, /ALLOW_NONEXISTENT
    
    ; See these URLs for the X11 color chart:
    ; http://en.wikipedia.org/wiki/X11_color_names
    ; http://www.mcfedries.com/Books/cightml/x11color.htm
    ; http://www.febooti.com/products/iezoom/online-help/html-color-names-x11-color-chart.html
    ; See <X11root>/lib/X11/rgb.txt for the colors known to a given machine.
    
;      color = ['Cornsilk','Tan','SandyBrown','Goldenrod','Peru','DarkGoldenrod','Chocolate','Sienna','Brown','DarkSalmon','Salmon','IndianRed']
    color = ['Cornsilk','Goldenrod','Chocolate','DarkSalmon', 'Tan','Peru','Sienna','Salmon', 'SandyBrown','DarkGoldenrod','Brown','IndianRed']
    color = [color,color]
    
    seq = num_gv_pairs + indgen(num_gv_pairs_needed-num_gv_pairs)
    pairs_per_row = 3
    row = seq  /  pairs_per_row
    col = seq MOD pairs_per_row
    sort_ind = reverse(sort(-100*row+col))
    
    screen_width = 2550
    gv_width  = 630
    gv_height = 514
    
    dx = 2550/pairs_per_row  ; offset for each pair
    dy = gv_height

    ; I cannot find a gv option or resource to provide a title for the X11 window.
    fmt = "(%'gv --watch -xrm ""GV*background:%s"" -geometry %dx%d+%d+%d --ad=%s %s >>&! %s &')" 
      
    for jj=0,n_elements(seq)-1 do begin
      ind = sort_ind[jj]
      ii = seq[ind]
      xpos =       dx*col[ind]
      ypos = 550 + dy*row[ind]

      cmd = string(color[ii], gv_width, gv_height, xpos+70, ypos, (*st).gv_resource_file, gv_files[ii,1], gv_logfile, F=fmt)
      run_command, /QUIET, cmd, result, /UNIX, PARAM_DIR=(*st).tempdir
      gv_pids[ii,1] =  (strsplit(result[0], /EXTRACT))[1] 
      wait, 0.5

      cmd = string(color[ii], gv_width, gv_height, xpos   , ypos, (*st).gv_resource_file, gv_files[ii,0], gv_logfile, F=fmt)
      run_command, /QUIET, cmd, result, /UNIX, PARAM_DIR=(*st).tempdir
      gv_pids[ii,0] =  (strsplit(result[0], /EXTRACT))[1] 
      wait, 0.5
    endfor
    
    num_gv_pairs = num_gv_pairs_needed
    
    *(*st).gv_pids = gv_pids
  endif ; creating new gv processes
endif ; (num_gv_pairs_needed GT 0)


;--------------------------------------------------------------------------
;; Redraw the model table.
if (num_models EQ 0) then begin
  print, 'No fit results available for ', this_sourcename
  
endif else begin
  ;--------------------------------------------------------------------------
  ;; Populate the table of fit parameters.
  widget_control, (*st).model_table, INSERT_ROWS=num_models, ROW_HEIGHTS=(*st).row_heights

  table       = strarr(num_columns, num_models)
;    table_color = strarr(num_columns, num_models)
  CSTAT       = replicate(!VALUES.f_nan, num_models)
  
  ; For each model, find each FITS keyword and format into a cell.
  ; It is vital to reference the extention by name rather than by number because we have re-ordered model_names!
  for ii=0,num_models-1 do begin
    model_header = headfits(fit_stats_fn, EXTEN=(where(fit_stats_keywords.HDUNAME EQ model_names[ii]))[0], ERRMSG=error)
    
    if keyword_set(error) then message, 'BUG!'
    
    ; Hide common parameters for model components with zero normalization.
    foreach component, strtrim(1+indgen(10),2) do begin
      value =  psb_xpar( model_header, 'NORM'+component, COUNT=count)
      if (count EQ 1) && (value EQ 0) then begin
        psb_xaddpar, model_header, 'NH' +component, ' '
        psb_xaddpar, model_header, 'PH' +component, ' '
        psb_xaddpar, model_header, 'KT' +component, ' '
        psb_xaddpar, model_header, 'TAU'+component, ' '
      endif
    endforeach ; component


    ; Construct "virtual parameters".
    if keyword_set((*st).distance) then begin
      cm_per_parsec = 3.086D18
      dscale        = 4D*!PI*((*st).distance * cm_per_parsec)^2

      if psb_xpar( src_stats, 'DIFFUSE' ) then begin
        ;; For diffuse sources extracted by AE the area of the region (in arcsec^2) is put into the calibration (ARF file).
        ;; Thus, XSPEC "flux" and "norm" quantities saved by the fitting scripts have a per arcsec^2 normalization.
        ;; That's a good thing (in my opinion) in that the size of the extraction region the observer chose does not affect the astrophysical "brightness" quantities that we report.
        ;; However, using arcseconds as the units of the area normalization is observational, not astrophysical.
        ;; Thus, here we apply to flux and norm quantities a conversion from arcsec^-2 to the physical units of pc^-2.
        
        arcsec_per_pc = 360. * 3600. / (2 * !PI * (*st).distance)
        dscale       *= arcsec_per_pc^2
        ; An interesting phenomenon appears in the calculation of dscale for the diffuse case above!
        ; In the term "4D*!PI*(distance * cm_per_parsec)^2" the quantity distance^2 appears in the numerator, while
        ; in the term "arcsec_per_pc^2" the quantity distance^2 appears in the denominator.
        ; Thus, for diffuse sources, dscale is always the same constant: 
        ;   cm_per_parsec^2 * (360*3600.)^2 / !PI = 5.0915728e+48  [cm^2 arcsec^2 /pc^2]
        ;
        ; The surface flux produced by the fitting scripts has           units of erg /s /cm^2 /arcsec^2.
        ; After multiplication by dscale, we get a surface luminosity in units of erg /s /pc^2.
        
        ; The "norm" value produced by the fitting has units of 1E-14 cm^-5 /arcsec^2.
        ; After multiplication by dscale, we get an "surface emission measure" in units of cm^-3 /pc^2.
        print, 'L* values are "surface brightness" [erg /s /pc^2]'
      endif

      for jj=0,num_columns-1 do begin
       keyname = (*st).keylist[jj]
       case keyname of
        'LCH8'   : psb_xaddpar, model_header, 'LCH8'   , dscale*(psb_xpar(model_header,'FCH8' ))
        'LC28'   : psb_xaddpar, model_header, 'LC28'   , dscale*(psb_xpar(model_header,'FC28' ))
        'L1CH7'  : psb_xaddpar, model_header, 'L1CH7'  , dscale*(psb_xpar(model_header,'F1CH7'))
        'L2CH7'  : psb_xaddpar, model_header, 'L2CH7'  , dscale*(psb_xpar(model_header,'F2CH7'))
        'L3CH7'  : psb_xaddpar, model_header, 'L3CH7'  , dscale*(psb_xpar(model_header,'F3CH7'))
        'L4CH7'  : psb_xaddpar, model_header, 'L4CH7'  , dscale*(psb_xpar(model_header,'F4CH7'))
        'L5CH7'  : psb_xaddpar, model_header, 'L5CH7'  , dscale*(psb_xpar(model_header,'F5CH7'))
        'L4_5CH7': psb_xaddpar, model_header, 'L4_5CH7', dscale*(psb_xpar(model_header,'F4CH7')+$
                                                                 psb_xpar(model_header,'F5CH7'))
         else: 
       endcase
      endfor ; jj
    endif ; "virtual parameters"

    value = psb_xpar( model_header,'CSTAT', COUNT=count)
    if (count EQ 1) then CSTAT[ii] = value

    for jj=0,num_columns-1 do begin
     keyname = (*st).keylist[jj]
     
     value =  psb_xpar( model_header, keyname, COUNT=count) 
     ; FITS keywords in these headers that happen to be IDL reserved words (e.g. "NE") have '_' prepended by the collation stage.
     if (count EQ 0) then begin
       value =psb_xpar( model_header, (stregex(keyname, '^_(.+)', /SUB, /EXT))[1], COUNT=count) 
     endif
     
     
     if (count EQ 1) then begin
     
      case size(value, /TNAME) of
        'STRING': format = '(A)'
        'BYTE'  : format = '(I)'
        'INT'   : format = '(I)'
        'LONG'  : format = '(I)'
        else: format = '(g10.2)'
      endcase
       
;if (jj EQ 0) then format = '(A)'
       if (keyname EQ 'CSTAT'  ) then format = '(g10.4)'
       if (keyname EQ 'CHI_SQR') then format = '(g10.3)'
       
       formatted_value = string(value, F=format)
       table[jj,ii] = formatted_value
       
       if (strtrim(formatted_value,2) EQ '') then begin
         ; Parameter is null (e.g. because component has zero normalization.
         ; Do not apply any color highlights.
         print, keyname, model_names[ii], F='(%"Component containing %s is null in %s.")'
         continue
       endif
       
       if strmatch(keyname, 'FC*') then begin
         ; This looks like an absorbtion-corrected flux value.
         ; Look for the corresponding apparent flux value.
         ; Highlight (red) the FC entry if the correction is larger than flux_correction_limit.

         ; Note that mg_streplace function is from https://raw.github.com/mgalloy/mglib/master/src/strings/mg_streplace.pro
         keyname_apparent = mg_streplace(keyname, 'FC', 'F')
           value_apparent = psb_xpar( model_header, keyname_apparent, COUNT=count_apparent) 

         if (count_apparent EQ 1) && (alog10(value/value_apparent) GT (*st).flux_correction_limit) then $
           widget_control, (*st).model_table, BACKGROUND_COLOR=[255,100,100], USE_TABLE_SELECT=[[jj,ii],[jj,ii]]
       endif ; FC* keyword
       
       ; Look for range keywords for this parameter.
       par_min = psb_xpar( /NAN, model_header, keyname+'_MIN', COUNT=cmin) 
       par_max = psb_xpar( /NAN, model_header, keyname+'_MAX', COUNT=cmax) 
       if (cmin EQ 1) && finite(par_min) && (cmax EQ 1) && finite(par_max) then begin
         ; If parameter is out-of-range, then highlight (yellow) it in the display.
         if (value LT par_min) || (par_max LT value) then begin
           widget_control, (*st).model_table, BACKGROUND_COLOR=[255,255,0], USE_TABLE_SELECT=[[jj,ii],[jj,ii]]
           
           print, keyname, strtrim(formatted_value,2), par_min, par_max, F='(%"WARNING: parameter %s = %4s is outside soft limits (%0.1f--%0.1f)")'
         endif ; out-of-range
       endif ; parameter range is defined
       
     endif ; (count EQ 1)
    endfor
  endfor
  
  ind = where((*st).keylist EQ 'CSTAT', count)
  if (count GT 0) then table[ind,*] = string(CSTAT-min(/NAN, CSTAT), F='(g10.4)')
  
  widget_control, (*st).model_table, SET_VALUE=table
  
  
  ;; Identify any model selection previously made and stored in the primary HDU..
  selected_model_name = strtrim(fit_stats_keywords[0].BEST_MDL,2)
  selected_category   = strtrim(fit_stats_keywords[0].CATEGORY,2)
  is_provisional      =         fit_stats_keywords[0].PROVISNL
  do_not_collate      =         fit_stats_keywords[0].NOCOLLAT
  
  ; Set GUI to match value of BEST_MDL.
  selected_model_index = where(model_names EQ selected_model_name, count)
  if (count EQ 1) then begin
    table_select = [[0,selected_model_index],[num_columns-1,selected_model_index]]
  endif else begin
    table_select = [[-1,-1],[-1,-1]]
    if (selected_model_name NE '') && (selected_model_name NE '...') then $
      print, selected_model_name, F='(%"BEST_MDL \"%s\" not found.")'
  endelse

  widget_control,  (*st).model_table,    SET_TABLE_SELECT=table_select     
  widget_control,  (*st).provisional_id, SET_VALUE=is_provisional
  widget_control,  (*st).DoNotCollateID, SET_VALUE=do_not_collate

  ; Set GUI to match value of CATEGORY.
  selected_category_index = where((*st).category_list EQ selected_category, count)
  if (count EQ 1) then begin
    widget_control,  (*st).category_id, SET_DROPLIST_SELECT=selected_category_index
  endif else begin
    widget_control,  (*st).category_id, SET_DROPLIST_SELECT=0
    if (selected_category NE '') && (selected_category NE '...') then $
      print, selected_category, F='(%"CATEGORY \"%s\" not found.")'
  endelse
  

endelse ; num_models GT 0
  

; Save the models found so the event handler can see them next time.
*(*st).model_names = model_names
 (*st).num_models  = num_models    


if NOT keyword_set(open_error) then fits_close, fcb


;; Show the source statistics.
widget_control, (*st).src_signif_id, SET_VALUE=string(F='(%"%0.1f")', (*st).src_signif   [source_index])
widget_control, (*st).median_E_id  , SET_VALUE=string(F='(%"%0.1f")', (*st).median_energy[source_index])
widget_control, (*st).net_cnts_id  , SET_VALUE=string(F='(%"%8d")'  , (*st).net_cnts     [source_index])
widget_control, (*st).exposure_id  , SET_VALUE=string(F='(%"%4d")'  , (*st).exposure     [source_index])
;widget_control, (*st).ag_frac_id   , SET_VALUE=string(F='(%"%4.2f")', (*st).ag_frac      [source_index])

widget_control, (*st).psf_frac_id  , SET_VALUE=string(F='(%"%4.2f")', (*st).PSF_FRAC     [source_index])
widget_control, (*st).theta_id     , SET_VALUE=string(F='(%"%4.1f")', (*st).THETA        [source_index])

pval_for_PROB_KS = (*st).pval_for_PROB_KS[source_index]
merge_ks         = (*st).MERGE_KS        [source_index]
merg_chi         = (*st).MERG_CHI        [source_index]
case 1 of 
  ~finite(pval_for_PROB_KS)          : prob_ks_text = '...'
         (pval_for_PROB_KS LT 0.005) : prob_ks_text = 'definite'
         (pval_for_PROB_KS LT 0.05)  : prob_ks_text = 'possible'
                                else : prob_ks_text = 'no evidence'
endcase 

case 1 of 
  ~finite(merge_ks)          : merge_ks_text = '...'
         (merge_ks LT 0.005) : merge_ks_text = 'definite'
         (merge_ks LT 0.05)  : merge_ks_text = 'possible'
                        else : merge_ks_text = 'no evidence'
endcase 

case 1 of 
  ~finite(merg_chi)          : merg_chi_text = '...'
         (merg_chi LT 0.005) : merg_chi_text = 'definite'
         (merg_chi LT 0.05)  : merg_chi_text = 'possible'
                        else : merg_chi_text = 'no evidence'
endcase 

widget_control, (*st).variability_id   , SET_VALUE=string(prob_ks_text, merge_ks_text, merg_chi_text, F='(%"Variability: ''%s'' (single-ObsID), ''%s'' (multi-ObsID, KS), ''%s'' (multi-ObsID, chi^2)")') 


if keyword_set((*st).my_ds9) then $
  ae_send_to_ds9, (*st).my_ds9, TEMP_DIR=(*st).tempdir, PAN_TO_COORDS=[(*st).ra[source_index], (*st).dec[source_index]]

return ; ae_spectra_viewer_select_source
end                                                                                                                                    


;; =============================================================================
;;; Widget Event Handler Procedure
PRO ae_spectra_viewer_event, event

widget_control, /HOURGLASS
modelsubdir              = 'spectral_models/'
fit_stats_basename       = 'source.spectra'
src_stats_basename       = 'source.stats'


;; Get the state structure.
top_base = event.handler
widget_control, top_base, GET_UVALUE=st

num_sources  = n_elements((*st).sourcename)

source_index = 0 > (*st).current_source_index < (num_sources-1)

num_models  = (*st).num_models

this_sourcename = (*st).sourcename[source_index]
sourcedir       = this_sourcename + '/' + (*st).merge_subdir[source_index] 
fit_stats_fn    = sourcedir + fit_stats_basename

merged_sequenced_lc_basename = this_sourcename + '.sequenced_lc.ps'
    
;; Process the event.
new_src_flag = 0
select_model_flag = 0

DestroyFlag = 0
case event.ID of

;--------------------------------------------------------------------------
  top_base: $
   begin
;   help, /ST, event
   widget_control, (*st).model_table, SCR_XSIZE=event.X, SCR_YSIZE=event.Y - (*st).geometry_without_table.SCR_YSIZE
   end

;--------------------------------------------------------------------------
  (*st).model_table: $
   begin
  if (Event.TYPE EQ 4) then begin
    row_num = Event.SEL_TOP
    if (row_num EQ Event.SEL_BOTTOM) AND (row_num GE 0) AND (row_num LT num_models) then begin
      select_model_flag = 1
      
      cmd = string(fit_stats_fn, (*(*st).model_names)[row_num], $
                 F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=BEST_MDL value=""%s"" comment=""preferred model""')")

      if keyword_set((*st).read_only) then begin
        msg = 'WARNING: you CANNOT record a best-model decision in READ_ONLY mode!'
        print, msg
        TimedMessage, tm_id, msg, TITLE='READ_ONLY mode!', QUIT_LABEL='Close', PRESSED_ID=trash
      endif else begin
        print, 'Selected model ', (*(*st).model_names)[row_num]
        run_command, /QUIET, cmd, PARAM_DIR=(*st).tempdir
      endelse
      
      ; Delete any "best model" symlink we find, and remake it.
      link_name = sourcedir + modelsubdir + 'best_model'
      if file_test(link_name, /DANGLING_SYMLINK) OR file_test(link_name, /SYMLINK) then file_delete, link_name
      if file_test(link_name) then begin
        print, 'WARNING: existing file'+link_name+' has not been changed.'
      endif else begin
        file_link, (*(*st).model_names)[row_num], link_name
      endelse
      
      table_select = [[0,row_num], [n_elements((*st).keylist)-1,row_num]]
      widget_control,  (*st).model_table, SET_TABLE_SELECT=table_select     
    endif
  endif
  end

  
;--------------------------------------------------------------------------
  (*st).provisional_id: $
   begin
   is_provisional = Event.select
   cmd = string(fit_stats_fn, is_provisional ? 'T' : 'F', $
             F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=PROVISNL value=""%s"" datatype=boolean comment=""BEST_MDL is provisional""')")

   if keyword_set((*st).read_only) then begin
    msg = 'WARNING: your inputs are IGNORED in READ_ONLY mode!'
    print, msg
    TimedMessage, tm_id, msg, TITLE='READ_ONLY mode!', QUIT_LABEL='Close', PRESSED_ID=trash
   endif else run_command, /QUIET, cmd, PARAM_DIR=(*st).tempdir
   end
  

   
;--------------------------------------------------------------------------
  (*st).lightcurve_button: $
   begin
   possible_lightcurve_fn =  this_sourcename+'/'+ $
                                ['all_inclusive/', 'all-inclusive/', (*st).merge_subdir[source_index] ]+$
                                merged_sequenced_lc_basename
   lc_found = 0B
   foreach lightcurve_fn, possible_lightcurve_fn do begin
     if file_test(lightcurve_fn) then begin
        lc_found = 1B
        break
     endif
   endforeach
   
   if lc_found then begin
     cmd = string(lightcurve_fn, F="(%'open %s')")
     run_command, cmd, PARAM_DIR=(*st).tempdir
   endif else begin
     print, 'Cannot find any of the following files:'
     forprint, possible_lightcurve_fn
   endelse
   end
   
   
;--------------------------------------------------------------------------
  (*st).DoNotCollateID: $
   begin
   do_not_collate = Event.select
   cmd = string(fit_stats_fn, do_not_collate ? 'T' : 'F', $
             F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=NOCOLLAT value=""%s"" datatype=boolean comment=""do not collate any model""')")

   if keyword_set((*st).read_only) then begin
    msg = 'WARNING: your inputs are IGNORED in READ_ONLY mode!'
    print, msg
    TimedMessage, tm_id, msg, TITLE='READ_ONLY mode!', QUIT_LABEL='Close', PRESSED_ID=trash
   endif else run_command, /QUIET, cmd, PARAM_DIR=(*st).tempdir
   end
  
;--------------------------------------------------------------------------
  (*st).category_id: $
   begin
   cmd = string(fit_stats_fn, ((*st).category_list)[event.index], $
             F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=CATEGORY value=""%s"" datatype=string comment=""observer decision"" ')")

   if keyword_set((*st).read_only) then begin
    msg = 'WARNING: your inputs are IGNORED in READ_ONLY mode!'
    print, msg
    TimedMessage, tm_id, msg, TITLE='READ_ONLY mode!', QUIT_LABEL='Close', PRESSED_ID=trash
   endif else run_command, /QUIET, cmd, PARAM_DIR=(*st).tempdir
   end
   
;--------------------------------------------------------------------------
;  (*st).plot_name_list: $
;   begin
;   end

   
   
   
;--------------------------------------------------------------------------
  (*st).source_label_id: $
   begin
   widget_control, (*st).source_label_id, GET_VALUE=source_label                                       
   ind = where(strlowcase(strtrim(source_label[0],2)) EQ strlowcase((*st).source_label), count)
   if (count GE 1) then begin
     source_index = ind[0]
     ae_spectra_viewer_select_source, st, source_index
   endif else print, 'Cannot find that source label.'
   end
   
;--------------------------------------------------------------------------
  (*st).source_number_id: $
   begin
   widget_control, (*st).source_number_id, GET_VALUE=source_number                                       
   ae_spectra_viewer_select_source, st, source_number-1
   end

;--------------------------------------------------------------------------
  (*st).prev_button: $
   begin
   source_index-- 
   source_index >= 0 
   ae_spectra_viewer_select_source, st, source_index
   end

;--------------------------------------------------------------------------
  (*st).next_button: $
   begin
   source_index++ 
   source_index <= (num_sources-1)                                                                                                
   ae_spectra_viewer_select_source, st, source_index
   end

;--------------------------------------------------------------------------
  (*st).rank_id: $
   begin
   widget_control, (*st).rank_id, GET_VALUE=rank                                       
   ; The index in the widget is 1-based; we want a 0-based index.
   rank = 0 > --rank < max((*st).presentation_rank)
   ind = where((*st).presentation_rank EQ rank, count)
   if (count NE 1) then message, 'BUG!'
   (*st).current_rank = rank
   ae_spectra_viewer_select_source, st, ind[0]
   end

;--------------------------------------------------------------------------
  (*st).prev_rank_button: $
   begin
   if ((*st).current_rank LE 0) then begin
     print, 'You are at the first source in '+(*st).srclist_fn
   endif else begin
     (*st).current_rank--
     ind = where((*st).presentation_rank EQ (*st).current_rank, count)
     if (count NE 1) then message, 'BUG!'
     ae_spectra_viewer_select_source, st, ind[0]
   endelse
   end

;--------------------------------------------------------------------------
  (*st).next_rank_button: $
   begin
   if ((*st).current_rank GE max((*st).presentation_rank)) then begin
     print, 'You are at the last source in '+(*st).srclist_fn
   endif else begin
     (*st).current_rank++
     ind = where((*st).presentation_rank EQ (*st).current_rank, count)
     if (count NE 1) then message, 'BUG!'
     ae_spectra_viewer_select_source, st, ind[0]
   endelse
   end

;--------------------------------------------------------------------------
  (*st).done_button: DestroyFlag = 1

;--------------------------------------------------------------------------
 else: begin
       print, 'unknown event in ae_spectra_viewer'
       help,/ST,Event
       end
endcase
  
       
; DONE button
if (DestroyFlag) then begin
  ; Kill the gv processes.
  if (n_elements(*(*st).gv_pids) GT 0) then begin
    run_command, /QUIET, /UNIX, 'kill -TERM ' + strjoin('-' + *(*st).gv_pids,' '), STATUS=status, PARAM_DIR=(*st).tempdir
  endif
  
  
  ; Remove the temp_dir.
  files = file_search((*st).tempdir+'/*', /MATCH_INITIAL_DOT, COUNT=count)
  if (count GT 0) then file_delete, files, /ALLOW_NONEXISTENT
  file_delete, (*st).tempdir, /ALLOW_NONEXISTENT

  widget_control, top_base, /DESTROY
  if keyword_set((*st).my_ds9) then $
    run_command, string((*st).my_ds9, F='(%"xpaset -p ''%s'' exit")'), /QUIET

  print, (*st).title, F='(%"\nSpectra Viewer%s terminated")'
endif


return
end ; ae_spectra_viewer_event


;==========================================================================
;;; Widget Creation Routine

;;; The optional parameter 'HDUNAME' is a search pattern interpreted by strmatch() specifying a set of model names you wish to review.

;;; The parameter "distance" must be in units of pc.
;==========================================================================
PRO ae_spectra_viewer, collated_filename, SRCLIST_FILENAME=srclist_fn, DISTANCE=distance, $
                       DISPLAY_FILE=display_file, REGION_FILENAME=region_file, $
                       CATEGORY_LIST=category_list_p, KEYLIST=keylist_p, READ_ONLY=read_only,$
                       HDUNAME=hduname, MERGE_NAME=merge_name, NOTES=notes, BLOCK=block,$
                               SORT_BY_MODEL_NAME=        sort_by_model_name,$
                       REVERSE_SORT_BY_MODEL_NAME=reverse_sort_by_model_name,$
                       TITLE=title_p
                       

creator_string = "ae_spectra_viewer, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

;; Read the source properties of all the sources.
bt = mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status)
if (status NE 0) then begin
  print, 'ERROR reading ', collated_filename
  retall      
endif

num_sources = n_elements(bt)
print, num_sources, F='(%"\n%d sources found in collated table.\n")'

title = keyword_set(title_p) ? ' ('+title_p+')' : ''

if ~keyword_set(hduname) then hduname = '*'

if keyword_set(keylist_p) then keylist = keylist_p $
 else keylist = ['GAL','LMC','NH','NHISM',$
                 'NH1','KT','KT1','TAU1','L1CH7',$
                 'NH2','KT2','L2CH7',$
                 'L3CH7',$
                 'NH4','L4_5CH7',$
                 'PH1',$
                 'FCH8','LCH8',$
                 'FCH7','LCH7',$
                 'CSTAT','CHI_SQR','ReducedChiSq','HANDFIT']
keylist = strtrim(keylist,2)


; Decide which items in keylist are available or computable.
key_available = replicate(0B, n_elements(keylist))

for ii=0, n_elements(keylist)-1 do begin
  keyname = keylist[ii]
  case keyname of
    'DATE'   : key_available[ii] = 1B
    'LCH8'   : key_available[ii] = tag_exist(bt,'FCH8' )
    'LC28'   : key_available[ii] = tag_exist(bt,'FC28' )
    'L1CH7'  : key_available[ii] = tag_exist(bt,'F1CH7')
    'L2CH7'  : key_available[ii] = tag_exist(bt,'F2CH7')
    'L3CH7'  : key_available[ii] = tag_exist(bt,'F3CH7')
    'L4CH7'  : key_available[ii] = tag_exist(bt,'F4CH7')
    'L5CH7'  : key_available[ii] = tag_exist(bt,'F5CH7')
    'L4_5CH7': key_available[ii] = tag_exist(bt,'F4CH7') && tag_exist(bt,'F5CH7')
     else    : key_available[ii] = tag_exist(bt, keyname)
  endcase
endfor ; ii

keylist = keylist[where(key_available)]







column_labels = ['model',keylist]
flux_correction_limit = '0.5' ; dex


if keyword_set(category_list_p) then category_list = ['no category', category_list_p] $
 else category_list = ['no category', 'good fit', 'bad fit', 'fit by hand', 'poor grouping', 'cplinear problems']


;; Look up the source properties of all the sources.
sourcename      = strtrim(bt.CATALOG_NAME   ,2)
label           = strtrim(bt.LABEL          ,2)
ra              =         bt.RA
dec             =         bt.DEC

ag_frac  = tag_exist(bt, 'AG_FRAC') ? bt.AG_FRAC  : fltarr(num_sources) 

; Calculate p-value for reported PROB_KS = min( P1,P2,...Pn), where n = N_KS.
if tag_exist(bt, 'PROB_KS') && tag_exist(bt, 'N_KS') then begin
  pval_for_PROB_KS = 1D - (1D - bt.PROB_KS)^bt.N_KS
  pval_for_PROB_KS[where(/NULL, ~finite(bt.PROB_KS))] = !VALUES.F_NAN
endif else pval_for_PROB_KS = replicate(!VALUES.f_nan, num_sources)

MERGE_KS = tag_exist(bt,'MERGE_KS') ? bt.MERGE_KS : replicate(!VALUES.f_nan, num_sources)
MERG_CHI = tag_exist(bt,'MERG_CHI') ? bt.MERG_CHI : replicate(!VALUES.f_nan, num_sources)
PSF_FRAC =                            bt.PSF_FRAC                       
THETA    = tag_exist(bt,   'THETA') ? bt.THETA    : replicate(!VALUES.f_nan, num_sources) 

band_total = 0
if ~almost_equal(bt.ENERG_LO[band_total], 0.5, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
if ~almost_equal(bt.ENERG_HI[band_total], 8.0, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 8.0 keV.\n")'

src_signif     =       bt.SRC_SIGNIF          [band_total]
median_energy  =       bt.ENERG_PCT50_OBSERVED[band_total]
net_cnts       = round(bt.NET_CNTS            [band_total]        )
exposure       = round(bt.EXPOSURE            [band_total] / 1000.)

;; Accept any NOTES supplied by the caller.
case n_elements(notes) of
  0: notes = strarr(num_sources)
  1: notes = replicate(notes, num_sources)
  num_sources: 
  else: begin
        print, 'ERROR: length of NOTES input not equal to number of sources'
        retall
        end
endcase


;; Handle any explicit SRCLIST_FILENAME supplied.
if keyword_set(srclist_fn) then begin
  readcol, srclist_fn, subset_sourcename, FORMAT='A', COMMENT=';', DELIM='@'
  
  ; Trim whitespace and ignore blank lines.
  subset_sourcename = strtrim(subset_sourcename,2)

  presentation_rank = replicate(-1L, num_sources)
  rank = 0L
  for ii=0L, n_elements(subset_sourcename)-1 do begin
    if (subset_sourcename[ii] EQ '') then continue
    
    ; Parse lines with semicolons into source names and notes.
    ind = strpos(subset_sourcename[ii],';')
    if (ind NE -1) then begin
      this_note             = strtrim(strmid(subset_sourcename[ii],ind+1) ,2)
      subset_sourcename[ii] = strtrim(strmid(subset_sourcename[ii],0,ind) ,2)
    endif else this_note = ''
    
    ; Look for this sourcename in the full list.
    ind = where(sourcename EQ subset_sourcename[ii], count)
    if (count GT 0) then begin
      ; Mark this source as being in the subset, and record any note that was parsed.
      presentation_rank[ind[0]]  = rank++
      notes            [ind[0]] += this_note
    endif else print, subset_sourcename[ii], F='(%"\nSource ''%s'' is missing from collated table!")'
    
    if (count GT 1) then print, subset_sourcename[ii],  F='(%"WARNING: source %s appears multiple times in table %s")'
  endfor 
  show_rank_navigation = 1
endif else begin
  srclist_fn  = collated_filename
  presentation_rank = lindgen(num_sources)
  show_rank_navigation = 0
endelse

;; Configure to display the "first" source (rank EQ 0).
ind = where(presentation_rank EQ 0, count)
if (count NE 1) then message, 'BUG!'
initial_source_number = 1+ind[0]



; MERGE_NAME is taken from AE keyword parameter if available, or from collated table otherwise.
if keyword_set(merge_name) then begin
  merge_subdir = merge_name    + '/'
endif else if tag_exist(bt,'MERGE_NAME') then begin
  merge_subdir = strtrim(bt.MERGE_NAME,2) + '/'
endif else begin
  merge_subdir = ''
endelse

if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)


;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir

; Identify a gv resources file we can use.
result = routine_info( 'ae_spectra_viewer', /SOURCE )
fdecomp, result.PATH, disk, codedir

cd, CURRENT=cwd
gv_resource_file = cwd+"/gv_resources.txt"
if (NOT file_test(gv_resource_file)) then begin
  gv_resource_file = codedir + "/gv_resources.txt"

  if (NOT file_test(gv_resource_file)) then begin
    print, 'Cannot find '+gv_resource_file
    retall
  endif
endif

; Identify a PostScript file to display when a gv session is not needed.
; Copy it to the tempdir for speed.
no_model_file       = "no_model.eps"
local_no_model_file = tempdir + no_model_file

if (NOT file_test(no_model_file)) then begin
  no_model_file = codedir + no_model_file

  if (NOT file_test(no_model_file)) then begin
    print, 'Cannot find '+no_model_file
    retall
  endif
endif
file_copy, /OVERWRITE,/FORCE, no_model_file, local_no_model_file



top_base = widget_base(TITLE='ACIS Extract Spectra Viewer'+title, /BASE_ALIGN_CENTER, /COLUMN, $
                        /SPACE, /XPAD, /YPAD, /TLB_SIZE_EVENTS)

 nav_base = widget_base( top_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER)
 
   base = widget_base( nav_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER, FRAME=1)
     source_number_id = cw_field( base, /INTEGER, /RETURN_EVENTS, XSIZE=4, VALUE=initial_source_number,  TITLE='Seq # in '+file_basename(collated_filename))                
     prev_button = widget_button( base, VALUE='Previous' )
     next_button = widget_button( base, VALUE='  Next  ' )
   
   plot_name_list = 0L ;widget_droplist( nav_base, VALUE=['grouped spectra','cumulative spectra'])
   
   source_label_id  = cw_field( nav_base, /STRING , /RETURN_EVENTS, XSIZE=30, VALUE='', TITLE='Source Label/Name/Merge:')                
   source_name_id   = cw_field( nav_base, /STRING, /NOEDIT, XSIZE=30, TITLE='')
   merge_subdir_id  = cw_field( nav_base, /STRING, /NOEDIT, XSIZE=18, TITLE='')
   ;widget_label( nav_base, /DYNAMIC_RESIZE, VALUE=' ' )                                

   rank_id          = 0L
   prev_rank_button = 0L
   next_rank_button = 0L
 
   if show_rank_navigation then begin
     base = widget_base( nav_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER, FRAME=1)
       rank_id = cw_field( base, /INTEGER, /RETURN_EVENTS, XSIZE=4, VALUE=1,  TITLE='Seq # in '+file_basename(srclist_fn))                
       prev_rank_button = widget_button( base, VALUE='Previous' )
       next_rank_button = widget_button( base, VALUE='  Next  ' )
   endif

   done_button = widget_button( nav_base, VALUE='Quit' )

 mid_base = widget_base( top_base, /ROW, SPACE=150, /XPAD, /YPAD, /BASE_ALIGN_CENTER )
 lightcurve_button = widget_button( mid_base, VALUE='Open lightcurve' )
 flag_base = widget_base( mid_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER, FRAME=1)
   provisional_id   = cw_bgroup( flag_base, ['PROVISIONAL'   ], /NONEXCLUSIVE, SET_VALUE=0 )
   DoNotCollateID   = cw_bgroup( flag_base, ['DO NOT COLLATE'], /NONEXCLUSIVE, SET_VALUE=0 )
   
   category_id = widget_droplist( flag_base, VALUE=category_list )
   widget_control, category_id, SET_DROPLIST_SELECT=0
   
 status_field_id= widget_label(mid_base, /DYNAMIC_RESIZE, VALUE='Flux is RED when correction > '+flux_correction_limit+' dex; Parameter is YELLOW when outside soft range limits.' )

 source_base = widget_base( top_base, /ROW, SPACE=15, /XPAD, /YPAD, /BASE_ALIGN_CENTER )
 
   src_signif_id  =     cw_field(source_base, /STRING, /NOEDIT, XSIZE=4, TITLE='SRC_SIGNIF')
   median_E_id    =     cw_field(source_base, /STRING, /NOEDIT, XSIZE=4, TITLE='MedianEnergy')
   net_cnts_id    =     cw_field(source_base, /STRING, /NOEDIT, XSIZE=6, TITLE='NET_CNTS')
   exposure_id    =     cw_field(source_base, /STRING, /NOEDIT, XSIZE=4, TITLE='ks')
  ;ag_frac_id     =     cw_field(source_base, /STRING, /NOEDIT, XSIZE=4, TITLE='AG_FRAC')
   ag_frac_id     = 0L
   psf_frac_id    =     cw_field(source_base, /STRING, /NOEDIT, XSIZE=4, TITLE='PSF_FRAC')
   theta_id       =     cw_field(source_base, /STRING, /NOEDIT, XSIZE=4, TITLE='THETA')
   variability_id = widget_label(source_base, /DYNAMIC_RESIZE)

num_models = 10

column_widths = [300,replicate(100,n_elements(keylist))] ; units are pixels
row_heights   = 25
device, get_screen_size=screen_size
scr_xsize = (100 + total(/INT, column_widths)) < (screen_size[0] - 100)

; Record how large the tool is WITHOUT the widget_table we're about to create below.
geometry_without_table = Widget_Info(top_base, /Geometry)


model_table = widget_table( top_base, ALIGNMENT=2, /ALL_EVENTS, COLUMN_LABELS=column_labels, COLUMN_WIDTHS=column_widths, ROW_HEIGHTS=row_heights, /RESIZEABLE_ROWS, /RESIZEABLE_COLUMNS, SCROLL=0,   SCR_XSIZE=scr_xsize, XSIZE=n_elements(column_labels), YSIZE=num_models,   /DISJOINT_SELECTION )


; Leisa dislikes: FONT='-adobe-helvetica-bold-r-normal--14-140-75-75-p-77-iso8859-1')

; OK: '-adobe-helvetica-medium-r-normal--14-140-75-75-p-77-iso8859-1'
; Too large: '-adobe-helvetica-bold-r-normal--25-180-100-100-p-138-iso8859-1')
; ??         '-adobe-courier-bold-r-*-*-*-*-*-100-m-150-iso8859-1'
; OK:  '9x15bold'  


header_base = widget_base(TITLE='Fits Header', GROUP=top_base, XOFFSET=1500, YOFFSET=0, /COLUMN, /SPACE, /XPAD, /YPAD)

  stats_header_id = widget_text( header_base, /SCROLL, XSIZE=80, YSIZE=20 )


; If requested, start a ds9 session that will pan to each source.
if keyword_set(display_file) then begin
  option_string = ''
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='ae_spectra_viewer'+session_name, OPTION_STRING=option_string
  if keyword_set(region_file) then $
    ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, display_file, region_file $
  else $
    ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, display_file
  
endif else my_ds9=''


state = { read_only:keyword_set(read_only), title:title,$
  
          source_label_id:source_label_id, source_name_id:source_name_id, merge_subdir_id:merge_subdir_id,$
          source_number_id:source_number_id, prev_button:prev_button, next_button:next_button, $
          rank_id:rank_id, prev_rank_button:prev_rank_button, next_rank_button:next_rank_button, $
          status_field_id:status_field_id, $
          geometry_without_table:geometry_without_table,$
          model_table:model_table, row_heights:row_heights, stats_header_id:stats_header_id, $
          done_button:done_button, src_signif_id:src_signif_id, median_E_id:median_E_id, net_cnts_id:net_cnts_id, exposure_id:exposure_id, ag_frac_id:ag_frac_id, psf_frac_id:psf_frac_id, theta_id:theta_id, variability_id:variability_id, $ 
        
          plot_name_list:plot_name_list, lightcurve_button:lightcurve_button, provisional_id:provisional_id, DoNotCollateID:DoNotCollateID,$
          category_list:category_list, category_id:category_id,$
          
          flux_correction_limit:float(flux_correction_limit), $
          distance:keyword_set(distance) ? float(distance) : 0.0, $
          
          model_names:ptr_new(/ALLOC), num_models:num_models, $
                  sort_by_model_name:keyword_set(        sort_by_model_name), $
          reverse_sort_by_model_name:keyword_set(reverse_sort_by_model_name), $
          
          hduname:hduname, keylist:['HDUNAME',keylist], $
          current_rank:0L, current_source_index:-1L, $
          sourcename:sourcename, source_label:label, ra:ra, dec:dec, src_signif:src_signif, median_energy:median_energy, net_cnts:net_cnts, exposure:exposure, ag_frac:ag_frac, PSF_FRAC:PSF_FRAC, THETA:THETA, pval_for_PROB_KS:pval_for_PROB_KS, MERGE_KS:MERGE_KS, MERG_CHI:MERG_CHI, $
          merge_subdir:merge_subdir, $
          notes:notes, srclist_fn:srclist_fn, presentation_rank:presentation_rank, $
          
          gv_resource_file:gv_resource_file, no_model_file:local_no_model_file, tempdir:tempdir, gv_pids:ptr_new(/ALLOC),$
          
          my_ds9:my_ds9 }

; Save state structure.
widget_control, top_base, SET_UVALUE=ptr_new(state, /NO_COPY)

widget_control, header_base, /REALIZE
widget_control, top_base,    /REALIZE


event={ID:source_number_id, TOP:top_base, HANDLER:top_base}
ae_spectra_viewer_event, event

;dum = ae_spectra_viewer_event({ID:choose_button, TOP:top_base, HANDLER:top_base})

 
  
; The fancy combination of JUST_REG and NO_BLOCK is necessary to get the
; widget to behave when called by either widget applications or normal
; programs (which block on tty reads).
xmanager, 'ae_spectra_viewer', top_base, EVENT_HANDLER='ae_spectra_viewer_event', $
          JUST_REG=keyword_set(block), NO_BLOCK=(keyword_set(block) EQ 0)
return
end ; ae_spectra_viewer



;==========================================================================
; Source reviewer tool.
; Useful for reviewing counterparts.
;==========================================================================

;At startup:
;- User responsible for loading images and region files.
;- Region file in frame 1 must have tags for each source label, to be used to highlight the selected source.
;- Read persistent storage (e.g. ASCII list of source names or labels) to identify approved sources; mark each one:
;  "xpaset frame 1"
;  "xpaset regions group foo_bar property include no"
;
;
;When a source is selected by ds9 mouse click:
;- Match frames to the one receiving the click.
;  "xpaset match frame wcs"
;- Compute distance from click to all sources (gcirc.pro) to identify closest source.
;
;When a source is selected by widget navigation controls:
;- Pan first frame to source position and match all frames.
;  "xpaset frame 1"
;  "xpaset match frame wcs"
;
;When a source is selected by any method:
;- Un-highlight the previously selected source.
;- Highlight the polygon belonging to the selected source.
;  "xpaset frame 1"
;  "xpaset regions select none"
;  "xpaset regions select group foo"
;  "xpaset regions width 3"
;  We manage the the "selected source" ourselves, rather than relying on the ds9 ability to select a region file because:
;  1. user can select muliple regions, violating the idea of the "current" source.
;  2. ...
;  
; 
;When a source is "committed"/"approved":
;- Mark the polygon belonging to the selected source.
;  "xpaset frame 1"
;  "xpaset regions group foo_bar property include no"
;- Record in some persistent storage that source has been approved.
;
;
;NOTES:
;- Use WCS for all coordinates in code and for all region files.
;- Composite catalog can provide all X-ray and IR properties we need to display.
;- Population plots will have to come from elsewhere.
;
;
;
;Two Modes of Tool
;* ds9 mouse/key input mode
;
;Widget has button to enter this mode.  Widget event handler for this button includes a loop that uses "imexam any" access point to get both mouse clicks and keystrokes from ds9.  Clicks select sources.  Meaningful keystrokes include:
;
;'a' to approve/commit the selected source.
;'q' to break out and resume widget processing (e.g. to annotate a source)
;
;
;* widget processing mode
;
;Widget includes button to enter ds9 mode, navigation controls, and annotation fields/flags.  Navigation controls include "next un-approved source".
;
;

;; =============================================================================
FUNCTION ae_source_viewer_print_source, cat, source_index, SKIP_PLOTS=skip_plots, INIT=init

COMMON ae_source_viewer, id1, id1b, id2, id2b
      
dum = check_math()

band_total = 0
if ~almost_equal(cat.ENERG_LO[band_total], 0.5, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
if ~almost_equal(cat.ENERG_HI[band_total], 8.0, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 8.0 keV.\n")'

large_phot_error = 0.1  ; mag

match_xy_types, isolated_type, successful_primary_type, secondary_type, failed_primary_type, init_type

FLUX2     =             cat.ACIS.FLUX2             [band_total]
FLUX2_hi  = FLUX2*(1 + (cat.ACIS.NET_CNTS_SIGMA_UP [band_total] / cat.ACIS.NET_CNTS[band_total]))
;FLUX2_lo  = FLUX2*(1 - (cat.ACIS.NET_CNTS_SIGMA_LOW[band_total] / cat.ACIS.NET_CNTS[band_total]))

E_median =        cat.ACIS.ENERG_PCT50_OBSERVED[band_total]


if keyword_set(init) then begin
  id1=0 & id1b=0 & id2=0 & id2b=0
  
  ; Make population plots for Jiang data.
  is_match = (cat.IR.TYPE EQ successful_primary_type) AND (cat.IR.JIANG.TYPE EQ successful_primary_type)
  
  H_good = is_match AND (cat.IR.JIANG.EH        LT large_phot_error)
  K_good = is_match AND (cat.IR.JIANG.EK        LT large_phot_error)
  
  ind = where(K_good)
  function_1d, id1, (cat.IR.JIANG.K_CIT)  [ind], alog10(FLUX2[ind]),                PSYM=3, LINE=6, COLOR='green', DATASET='Jiang', XTIT='K',        YTIT='log NET_CNTS/MEAN_ARF/EXPOSURE', PLOT_WINDOW_OPTIONS='SET_XMARGIN=[6,0], SET_YMARGIN=[3,2], SHOW_DATE=0'
                                                                                                                                                     
; dataset_2d, id1b, (cat.IR.JIANG.K_CIT)  [ind], alog10(FLUX2[ind]),                PSYM=3,         COLOR='green', DATASET='Jiang', XTIT='K',        YTIT='log NET_CNTS/MEAN_ARF/EXPOSURE'

  ae_ds9_to_ciao_regionfile, 'Jiang_flux_K_96p7.reg', '/dev/null', POLYGON_X=polygon_x, POLYGON_Y=polygon_y
  
  function_1d, id1, polygon_x, polygon_y, LINE=0, COLOR='green', DATASET='96.7%'
; dataset_2d, id1b, polygon_x, polygon_y,         COLOR='green', DATASET='96.7%'
  
  
  ind = where(H_good AND K_good)
  function_1d, id2, E_median[ind], (cat.IR.JIANG.H_CIT - cat.IR.JIANG.K_CIT)[ind],  PSYM=3, LINE=6, COLOR='green', DATASET='Jiang', XTIT='E_median', YTIT='H-K',                            PLOT_WINDOW_OPTIONS='SET_XMARGIN=[6,0], SET_YMARGIN=[3,2], SHOW_DATE=0'
  
; dataset_2d, id2b, E_median[ind], (cat.IR.JIANG.H_CIT - cat.IR.JIANG.K_CIT)[ind],  PSYM=3,         COLOR='green', DATASET='Jiang', XTIT='E_median', YTIT='H-K'

  ae_ds9_to_ciao_regionfile, 'Jiang_color_absorption_96p7.reg', '/dev/null', POLYGON_X=polygon_x, POLYGON_Y=polygon_y
  
  function_1d, id2, polygon_x, polygon_y, LINE=0, COLOR='green', DATASET='96.7%'
; dataset_2d, id2b, polygon_x, polygon_y,         COLOR='green', DATASET='96.7%'
  
  
  ; Make population plots for 2MASS data.
  is_match = (cat.IR.TYPE EQ successful_primary_type) AND (cat.IR.TWOMASS.TYPE EQ successful_primary_type)

  H_good = is_match AND (cat.IR.TWOMASS.H_CMSIG LT large_phot_error) AND ~strmatch(cat.IR.TWOMASS.PH_QUAL, '?U?') AND ~strmatch(cat.IR.TWOMASS.PH_QUAL, '?E?')
  K_good = is_match AND (cat.IR.TWOMASS.K_CMSIG LT large_phot_error) AND ~strmatch(cat.IR.TWOMASS.PH_QUAL, '??U') AND ~strmatch(cat.IR.TWOMASS.PH_QUAL, '??E')

  ind = where(K_good)
  function_1d, id1, (cat.IR.TWOMASS.K_CIT)[ind], alog10(FLUX2[ind]), PSYM=3, LINE=6, COLOR='blue', DATASET='2MASS'

;  dataset_2d, id1b, (cat.IR.TWOMASS.K_CIT)[ind], alog10(FLUX2[ind]), PSYM=3,         COLOR='blue', DATASET='2MASS'
  
  ae_ds9_to_ciao_regionfile, '2MASS_flux_K_96p8.reg', '/dev/null', POLYGON_X=polygon_x, POLYGON_Y=polygon_y
  
  function_1d, id1, polygon_x, polygon_y, LINE=0, COLOR='blue', DATASET='96.8%'
; dataset_2d, id1b, polygon_x, polygon_y,         COLOR='blue', DATASET='96.8%'


  ind = where(H_good AND K_good)
  function_1d, id2, E_median[ind], (cat.IR.TWOMASS.H_CIT - cat.IR.TWOMASS.K_CIT)[ind],  PSYM=3, LINE=6, COLOR='blue', DATASET='2MASS'

;  dataset_2d,  id2b, E_median[ind], (cat.IR.TWOMASS.H_CIT - cat.IR.TWOMASS.K_CIT)[ind],  PSYM=3,         COLOR='blue', DATASET='2MASS'

  ae_ds9_to_ciao_regionfile, '2MASS_color_absorption_96p5.reg', '/dev/null', POLYGON_X=polygon_x, POLYGON_Y=polygon_y
  
  function_1d, id2, polygon_x, polygon_y, LINE=0, COLOR='blue', DATASET='96.5%'
; dataset_2d, id2b, polygon_x, polygon_y,         COLOR='blue', DATASET='96.5%'

  if keyword_set(skip_plots) then begin
    widget_control, id1 , /DESTROY, BAD_ID=bad_id
    widget_control, id1b, /DESTROY, BAD_ID=bad_id
    widget_control, id2 , /DESTROY, BAD_ID=bad_id
    widget_control, id2b, /DESTROY, BAD_ID=bad_id
  endif
  return, 0
endif

source = cat[source_index]

if max(abs([source.ACIS.ENERG_LO[band_total] - 0.5, source.ACIS.ENERG_HI[band_total] - 8.0])) GT 0.01 then begin
  return, string(band_total, F='(%"\nERROR: Band %d is not 0.5:8.0 keV")')
endif



; Show the selected sources on the scatter plots with no screening.
if (source.IR.TYPE EQ successful_primary_type) then begin
  ; Jiang data                            
  H_good = (source.IR.JIANG.TYPE EQ successful_primary_type)
  K_good = (source.IR.JIANG.TYPE EQ successful_primary_type)

  K_CIT  = K_good           ? source.IR.JIANG.K_CIT                         : !VALUES.F_NAN
  HK_CIT = H_good && K_good ? source.IR.JIANG.H_CIT - source.IR.JIANG.K_CIT : !VALUES.F_NAN
  
  H_error  = source.IR.JIANG.EH
  K_error  = source.IR.JIANG.EK

  ; 2MASS data.
  H_good = (source.IR.TWOMASS.TYPE EQ successful_primary_type) AND ~strmatch(source.IR.TWOMASS.PH_QUAL, '?U?') AND ~strmatch(source.IR.TWOMASS.PH_QUAL, '?E?')
  K_good = (source.IR.TWOMASS.TYPE EQ successful_primary_type) AND ~strmatch(source.IR.TWOMASS.PH_QUAL, '??U') AND ~strmatch(source.IR.TWOMASS.PH_QUAL, '??E')
  
  K_CIT  = [  K_CIT, K_good           ? source.IR.TWOMASS.K_CIT                           : !VALUES.F_NAN ]
  HK_CIT = [ HK_CIT, H_good && K_good ? source.IR.TWOMASS.H_CIT - source.IR.TWOMASS.K_CIT : !VALUES.F_NAN ]

  H_error  = [H_error, source.IR.TWOMASS.H_CMSIG]
  K_error  = [K_error, source.IR.TWOMASS.K_CMSIG]
  
  HK_error = sqrt(H_error^2 + K_error^2)
  
endif else begin
  K_CIT    = replicate(!VALUES.F_NAN,2)
  HK_CIT   = replicate(!VALUES.F_NAN,2)
  K_error  = replicate(!VALUES.F_NAN,2)
  HK_error = replicate(!VALUES.F_NAN,2)
endelse

if ~keyword_set(skip_plots) then begin
  s = [source_index,source_index]
  function_1d, id1, K_CIT,     PSYM=4, LINE=6, COLOR='red', DATASET='Selected', NSKIP_ERRORS=1, $
            X_ERROR=K_error, $
                                 alog10(FLUX2[s]), $
    ERROR=(alog10(FLUX2_hi[s]) - alog10(FLUX2[s]))
  
  function_1d, id2, E_median[s], X_ERROR=[0,0], HK_CIT, ERROR=HK_error, PSYM=4, LINE=6, COLOR='red', DATASET='Selected', NSKIP_ERRORS=1
endif



; Build a report summarizing source properties.
acis_report = strarr(80)
line = 0

if (source.ACIS07.TYPE EQ successful_primary_type) then begin
  acis_report[line++] = string(source.ACIS07.ID, F="(%'ACIS ID in Broos07 is %d')")
              line++
endif

N=1+source.IR.NUM_SM
if (N GT 1) then begin
  acis_report[line++] = string(N,F="(%'ACIS position is consistent with %d IR sources.')")
              line++
endif

;acis_report[line++] = string(source.ACIS.POSNTYPE,F="(%'ACIS position comes from   : %s.')")
;       line++

acis_report[line++] = string(source.ACIS.THETA,    F="(%'off-axis angle (arcmin)    : %0.1f')")

print, 'NEED TO CALCULATE AND REPORT p-values for ProbKS_single, rather than ProbKS_single itself.'
text = 'no evidence'
if (source.ACIS.PROB_KS LT 0.05)  then text = 'possible'
if (source.ACIS.PROB_KS LT 0.005) then text = 'definite'
if ~finite(source.ACIS.PROB_KS) then text = '...'

acis_report[line++] = 'variability                : '+text
            line++

PROB_NO_SOURCE = tag_exist(source.ACIS,'PB_MIN') ? source.ACIS.PB_MIN : source.ACIS.PROB_NO_SOURCE[band_total]

acis_report[line++] = string(source.ACIS.ENERG_LO[band_total],source.ACIS.ENERG_HI[band_total], $
                        (PROB_NO_SOURCE LT 1E-5) ? "<0.00001" : string(PROB_NO_SOURCE, F='(%"%0.5f")'), $
                        F="(%'P_B           (%0.1f:%d keV)  : %s')")
        
acis_report[line++] = string(source.ACIS.ENERG_LO[band_total],source.ACIS.ENERG_HI[band_total],source.ACIS.NET_CNTS[band_total], F="(%'net counts    (%0.1f:%d keV)  : %d')")

acis_report[line++] = string(source.ACIS.ENERG_LO[band_total],source.ACIS.ENERG_HI[band_total],source.ACIS.ENERG_PCT50_OBSERVED[band_total], F="(%'median energy (%0.1f:%d keV)  : %0.1f')")
            line++

acis_report = acis_report[0:line]



ir_report   = strarr(80)
line = 0

; Here, we summarize IR info for *either* a successful or failed Primary mMatch between ACIS and IR.  
; But below we decide how to present the info based on the type of match.
if (source.IR.TYPE EQ successful_primary_type) || (source.IR.TYPE EQ failed_primary_type)  then begin
  ; However, within the IR catalog, we are interested only in successful Primary Matches to published catalogs.
  if (source.IR.JIANG.TYPE EQ successful_primary_type) then begin
    J_CIT = source.IR.JIANG.J_CIT 
    H_CIT = source.IR.JIANG.H_CIT 
    K_CIT = source.IR.JIANG.K_CIT 
    JHval = string(J_CIT - H_CIT, F="(%'%4.1f')")
    HKval = string(H_CIT - K_CIT, F="(%'%4.1f')")
 
    Jval = string(' ', $
                           J_CIT, $
                         ((source.IR.JIANG.EJ GT large_phot_error) || ~finite(source.IR.JIANG.EJ)) ? ':' : ' ', $
                  F="(%'%s%4.1f%s')")
    if ~finite(J_CIT) then begin
      Jval  = '  ... '
      JHval = ' ...'
    endif
    
    Hval = string(' ', $
                           H_CIT, $
                         ((source.IR.JIANG.EH GT large_phot_error) || ~finite(source.IR.JIANG.EH)) ? ':' : ' ', $
                  F="(%'%s%4.1f%s')")
    if ~finite(H_CIT) then begin
      Hval  = '  ... '
      JHval = ' ...'
      HKval = ' ...'
    endif
  
    Kval = string(' ', $
                           K_CIT, $
                         ((source.IR.JIANG.EK GT large_phot_error) || ~finite(source.IR.JIANG.EK)) ? ':' : ' ', $
                  F="(%'%s%4.1f%s')")
    if ~finite(K_CIT) then begin
      Kval  = '  ... '
      HKval = ' ...'
    endif
 
    Jiang_id   = string(source.IR.JIANG.ID, '',         F="(%'%16s %3s')")
    Jiang_vals = string(Jval, Hval, Kval, JHval, HKval, F="(%'%s %s %s %s %s')")
    Jiang_vals = repstr(Jiang_vals,'NaN','...')
  endif else begin
    Jiang_id   = ''
    Jiang_vals = ''
  endelse
  
  
  ; However, within the IR catalog, we are interested only in successful Primary Matches to published catalogs.
  if (source.IR.TWOMASS.TYPE EQ successful_primary_type) then begin
    J_CIT = source.IR.TWOMASS.J_CIT 
    H_CIT = source.IR.TWOMASS.H_CIT 
    K_CIT = source.IR.TWOMASS.K_CIT 
    JHval = string(J_CIT - H_CIT, F="(%'%4.1f')")
    HKval = string(H_CIT - K_CIT, F="(%'%4.1f')")
     
    Jval = string(strmatch(source.IR.TWOMASS.PH_QUAL, 'U??')                                                     ? '>' : ' ', $
                           J_CIT, $
                         ((source.IR.TWOMASS.J_CMSIG GT large_phot_error) || ~finite(source.IR.TWOMASS.J_CMSIG)) ? ':' : ' ', $
                  F="(%'%s%4.1f%s')")
    if strmatch(source.IR.TWOMASS.PH_QUAL, 'E??') || ~finite(J_CIT) then begin
      Jval  = '  ... '
      JHval = ' ...'
    endif
    
    Hval = string(strmatch(source.IR.TWOMASS.PH_QUAL, '?U?')                                                     ? '>' : ' ', $
                           H_CIT, $
                         ((source.IR.TWOMASS.H_CMSIG GT large_phot_error) || ~finite(source.IR.TWOMASS.H_CMSIG)) ? ':' : ' ', $
                  F="(%'%s%4.1f%s')")
    if strmatch(source.IR.TWOMASS.PH_QUAL, '?E?') || ~finite(H_CIT) then begin
      Hval  = '  ... '
      JHval = ' ...'
      HKval = ' ...'
    endif

    Kval = string(strmatch(source.IR.TWOMASS.PH_QUAL, '??U')                                                     ? '>' : ' ', $
                           K_CIT, $
                         ((source.IR.TWOMASS.K_CMSIG GT large_phot_error) || ~finite(source.IR.TWOMASS.K_CMSIG)) ? ':' : ' ', $
                  F="(%'%s%4.1f%s')")
    if strmatch(source.IR.TWOMASS.PH_QUAL, '??E') || ~finite(K_CIT) then begin
      Kval  = '  ... '
      HKval = ' ...'
    endif
    
    Twomass_id = string(source.IR.TWOMASS.DESIGNATION, source.IR.TWOMASS.CC_FLG, F="(%'%16s %3s')")
    Twomass_vals = string(Jval, Hval, Kval, JHval, HKval,                        F="(%'%s %s %s %s %s')")
    Twomass_vals = repstr(Twomass_vals,'NaN','...')
  endif else begin
    Twomass_id   = ''
    Twomass_vals = ''
  endelse

  
  ; However, within the IR catalog, we are interested only in successful Primary Matches to published catalogs.
  if (source.IR.GLIMPSE.TYPE EQ successful_primary_type) then begin
    MAG3_6 = string(source.IR.GLIMPSE.MAG3_6, F="(%'%5.1f')")
    MAG4_5 = string(source.IR.GLIMPSE.MAG4_5, F="(%'%5.1f')")
    MAG5_8 = string(source.IR.GLIMPSE.MAG5_8, F="(%'%5.1f')")
    MAG8_0 = string(source.IR.GLIMPSE.MAG8_0, F="(%'%5.1f')")
                              
    Glimpse_id   = string(source.IR.GLIMPSE.DESIGNATION, F="(%'%s')")
    Glimpse_vals = string(MAG3_6,MAG4_5,MAG5_8,MAG8_0,   F="(%'%s %s %s %s')")
    Glimpse_vals = repstr(Glimpse_vals,'NaN','...')
  endif else begin
    Glimpse_id   = ''
    Glimpse_vals = ''
  endelse

  ir_report[line++] = '   J      H      K    J-H  H-K    3.6   4.5   5.8   8.0        NIR ID             MIR ID'
  ir_report[line++] = ' ------------------------------  ---------------------- -------------------- --------------------'

  
  ; Display Jiang first if available.
  if (source.IR.JIANG.TYPE EQ successful_primary_type) then begin
    ir_report[line++] = string(  Jiang_vals, Glimpse_vals,   Jiang_id, Glimpse_id, F="(%'%30s  %23s %20s %s')")
    ir_report[line++] = string(Twomass_vals, '',             Twomass_id,           F="(%'%30s  %23s %20s')")
  endif else begin
    ir_report[line++] = string(Twomass_vals, Glimpse_vals, Twomass_id, Glimpse_id, F="(%'%30s  %23s %20s %s')")
    ir_report[line++] = string(  Jiang_vals, '',             Jiang_id,             F="(%'%30s  %23s %20s')")
  endelse
  
endif else begin
 ir_report[line++] = 'NO IR MATCH.'
endelse

       line++
ir_report = ir_report[0:line]


legend   = strarr(80)
line = 0
legend[line++] = 'Region Legend:'   
legend[line++] = '  ACIS        :diamond  green (match), cyan (no match), red (failed match)' 
legend[line++] = '  IR          :+        green (match), cyan (no match), magenta (secondary match)'
legend[line++] = '    GLIMPSE   :box      blue' 
legend[line++] = '    Jiang     :circle   blue'  
legend[line++] = '    2MASS     :X        blue' 
       line++
legend[line++] = 'Around each ACIS source a graphic depicts the domain over which'
legend[line++] = 'an IR source with zero position error could match the ACIS source.'  
       line++
legend[line++] = '- For unmatched ACIS sources (cyan diamond) a large cyan circle'
legend[line++] = '  depicts this domain.'  
       line++
legend[line++] = '- For matched ACIS sources (green or red diamond) the line region'
legend[line++] = '  depicting the match extends from the ACIS source through the'
legend[line++] = '  IR source out to the distance where the match would have failed'
legend[line++] = '  if the IR position error was zero.' 
legend = legend[0:line]

dum = check_math()

case source.IR.TYPE of
  successful_primary_type: $
    begin
    return, [acis_report, ir_report, legend]
    end
  failed_primary_type    : $
    begin
    print
    print, 'IR Properties of Failed Match:'
    print, ir_report
    return, [acis_report, 'POTENTIAL IR MATCH ASSIGNED TO ANOTHER ACIS SOURCE.', '','', legend]
    end
  else                   : $
    begin
    return, [acis_report, ir_report, legend]
    end
endcase



end


;; =============================================================================
;; Procedure to handle GUI updates when a source is selected.
PRO ae_source_viewer_select_source, st, source_index, PAN_TO_SOURCE=pan_to_source

; Range check the source number.
num_sources = n_elements((*st).sourcename)
source_index = 0 > source_index < (num_sources-1)

; If the cat has been modified, now is a good time to save it to disk.
if ~keyword_set((*st).read_only) && ((*st).pending_save GT 0) then begin
  ; Write the modified catalog structure to disk.
  if (((*st).save_counter++ MOD 100) EQ 0) then begin
    print, 'Backing up ', (*st).catalog_fn
    ; Periodically save a backup copy of the catalog structure.
    for ii=9,1,-1 do begin
      backup_fn       = string((*st).catalog_fn,ii,  F='(%"%s-%d")')
      older_backup_fn = string((*st).catalog_fn,ii+1,F='(%"%s-%d")')
      if file_test(backup_fn) then file_move, backup_fn, older_backup_fn, /OVERWRITE
    endfor
    
    if file_test((*st).catalog_fn) then file_move, (*st).catalog_fn, backup_fn, /OVERWRITE
  endif

  cat = (*st).cat
  save, cat, FILE=(*st).catalog_fn
  print, 'Saved ', (*st).catalog_fn
  (*st).pending_save = 0
endif


run_command, /QUIET, string((*st).my_ds9,             F='(%"xpaset -p %s frame first")')

if ((*st).current_source_index GE 0) then begin
  ; Un-highlight the currently-selected source.
   this_source_label = (*st).source_label[(*st).current_source_index]
   run_command, /QUIET, string((*st).my_ds9,                    F='(%"xpaset -p %s regions select none")')
   run_command, /QUIET, string((*st).my_ds9, this_source_label, F='(%"xpaset -p %s regions select group %s")')
   run_command, /QUIET, string((*st).my_ds9,                    F='(%"xpaset -p %s regions width 1")')
endif

(*st).current_source_index = source_index

if keyword_set(pan_to_source) then begin
  run_command, /QUIET, string((*st).my_ds9, (*st).RA[source_index], (*st).DEC[source_index], $
                                            F='(%"xpaset -p %s pan to %10.6f %10.6f wcs fk5 degrees")')
  ;After panning, we may have falled off of the image ds9 built from the event data; we must force ds9 to re-compute that image.
  run_command, /QUIET, string((*st).my_ds9, F='(%"xpaset -p %s bin function sum")')
  run_command, /QUIET, string((*st).my_ds9, F='(%"xpaset -p %s match frame wcs")')
endif

; Now we can highlight the newly-selected source and update the GUI
this_sourcename   = (*st).sourcename  [source_index]
this_source_label = (*st).source_label[source_index]
this_merge_subdir = (*st).merge_subdir[source_index]

run_command, /QUIET, string((*st).my_ds9,                    F='(%"xpaset -p %s regions select none")')
run_command, /QUIET, string((*st).my_ds9, this_source_label, F='(%"xpaset -p %s regions select group %s")')
run_command, /QUIET, string((*st).my_ds9,                    F='(%"xpaset -p %s regions width 3")')
       
       
; If necessary, add the annotation to the list of options.
annotation_string = strtrim((*st).cat[source_index].UNRELIABLE_ANNOTATION,2)
if (annotation_string EQ '') then annotation_string = ' '

widget_control, (*st).unreliable_annotation_id, GET_VALUE=annotation_list
ind = where(annotation_list EQ annotation_string, count)
if (count EQ 0) then annotation_list = [annotation_list, annotation_string]

; Display the annotations. 
ind = where(annotation_list EQ annotation_string, count)
widget_control, (*st).unreliable_annotation_id, SET_VALUE=annotation_list, SET_COMBOBOX_SELECT=ind, SENSITIVE=((*st).cat[source_index].REVIEW_STATE EQ 'unreliable')


; If necessary, add the annotation to the list of options.
annotation_string = strtrim((*st).cat[source_index].PROVISIONAL_ANNOTATION,2)
if (annotation_string EQ '') then annotation_string = ' '

widget_control, (*st).provisional_annotation_id, GET_VALUE=annotation_list
ind = where(annotation_list EQ annotation_string, count)
if (count EQ 0) then annotation_list = [annotation_list, annotation_string]

; Display the annotations. 
ind = where(annotation_list EQ annotation_string, count)
widget_control, (*st).provisional_annotation_id, SET_VALUE=annotation_list, SET_COMBOBOX_SELECT=ind, SENSITIVE=((*st).cat[source_index].REVIEW_STATE EQ 'provisional')

ind = where(strmatch((*st).review_state_names, (*st).cat[source_index].REVIEW_STATE), count)
if (count EQ 0) then ind = 0
widget_control, (*st).review_state_id,  SET_VALUE=ind


widget_control, (*st).flag_group_id, SET_VALUE=[(*st).cat[source_index].uncatalogued_NIR,$
                                                (*st).cat[source_index].uncatalogued_MIR          ,$
                                                (*st).cat[source_index].nebular_contamination   ]

widget_control, (*st).source_name_id,   SET_VALUE=this_sourcename
widget_control, (*st).merge_subdir_id,  SET_VALUE=this_merge_subdir
widget_control, (*st).source_label_id,  SET_VALUE=this_source_label    
widget_control, (*st).source_number_id, SET_VALUE=1+source_index

;; Read any existing analysis notes file.  
;; We cannot use readcol because it won't read comment lines with spaces.
notes_fn = (*st).cat[source_index].ACIS.CATALOG_NAME + '/notes.txt'

if file_test(notes_fn) then begin
  Nlines = file_lines(notes_fn) 
endif else Nlines = 0

if (Nlines GT 0) then begin
  notes = strarr(Nlines)
  openr, unit, notes_fn, /GET_LUN
  readf, unit, notes
  free_lun, unit
endif else begin
  notes = strarr(1)
endelse
widget_control, (*st).analysis_notes_id, SET_VALUE=notes


;; Read any existing footnotes file.  
;; We cannot use readcol because it won't read comment lines with spaces.
notes_fn = (*st).cat[source_index].ACIS.CATALOG_NAME + '/footnotes.txt'

if file_test(notes_fn) then begin
  Nlines = file_lines(notes_fn) 
endif else Nlines = 0

if (Nlines GT 0) then begin
  notes = strarr(Nlines)
  openr, unit, notes_fn, /GET_LUN
  readf, unit, notes
  free_lun, unit
endif else begin
  notes = strarr(1)
endelse
widget_control, (*st).footnotes_id, SET_VALUE=notes



print, this_sourcename, this_source_label, F='(%"\n----------------------------\n%s (%s)")'
temp = (*st).notes[source_index]
if keyword_set(temp) then print, temp

;; Show the source report
if (widget_info( (*st).report_id, /VALID_ID )) then $
  widget_control, (*st).report_id, SET_VALUE=ae_source_viewer_print_source( (*st).cat, source_index, SKIP_PLOTS=(*st).skip_plots )
  
return
end                                                                                                                                    


;; =============================================================================
;;; Widget Event Handler Procedure
PRO ae_source_viewer_event, event

widget_control, /HOURGLASS


;; Get the state structure.
top_base = event.handler
widget_control, top_base, GET_UVALUE=st                                                                                                 


num_sources  = n_elements((*st).sourcename)

source_index = 0 > (*st).current_source_index < (num_sources-1)

    
;; Process the event.
DestroyFlag = 0
case event.ID of

;--------------------------------------------------------------------------
  (*st).review_state_id: $
   begin
   (*st).cat[source_index].REVIEW_STATE = Event.value
   (*st).pending_save++
   widget_control, (*st).unreliable_annotation_id,  SENSITIVE=(Event.value EQ 'unreliable')
   widget_control, (*st).provisional_annotation_id, SENSITIVE=(Event.value EQ 'provisional')
   
   run_command, /QUIET, string((*st).my_ds9, F='(%"xpaset -p %s frame first")')
   run_command, /QUIET, string((*st).my_ds9, (*st).source_label[source_index], (Event.value NE 'unreviewed') ? "no" : "yes", F='(%"xpaset -p %s regions group ''%s'' property include %s")')
   end
  
;--------------------------------------------------------------------------
  (*st).unreliable_annotation_id: $
   begin
   ; Save the observer's string.
   input_string = strtrim(Event.STR,2)
   (*st).cat[source_index].UNRELIABLE_ANNOTATION = input_string
   (*st).pending_save++
   
   ; And incorporate it into the list of options.
   widget_control, (*st).unreliable_annotation_id, GET_VALUE=annotation_list
   if (input_string NE '') && (total(/INT, annotation_list EQ input_string) EQ 0) then begin
     annotation_list = [annotation_list, input_string]
     widget_control, (*st).unreliable_annotation_id, SET_VALUE=annotation_list, SET_COMBOBOX_SELECT=n_elements(annotation_list)-1
   endif
   end
   
;--------------------------------------------------------------------------
  (*st).provisional_annotation_id: $
   begin
   ; Save the observer's string.
   input_string = strtrim(Event.STR,2)
   (*st).cat[source_index].PROVISIONAL_ANNOTATION = input_string
   (*st).pending_save++
   
   ; And incorporate it into the list of options.
   widget_control, (*st).provisional_annotation_id, GET_VALUE=annotation_list
   if (input_string NE '') && (total(/INT, annotation_list EQ input_string) EQ 0) then begin
     annotation_list = [annotation_list, input_string]
     widget_control, (*st).provisional_annotation_id, SET_VALUE=annotation_list
     widget_control, (*st).provisional_annotation_id, SET_COMBOBOX_SELECT=n_elements(annotation_list)-1
   endif
   end
   

;--------------------------------------------------------------------------
  (*st).flag_group_id: $
   begin
   widget_control, (*st).flag_group_id, GET_VALUE=value
   (*st).cat[source_index].uncatalogued_NIR        = value[0]
   (*st).cat[source_index].uncatalogued_MIR        = value[1]
   (*st).cat[source_index].nebular_contamination   = value[2]
   (*st).pending_save++
   end
   

;--------------------------------------------------------------------------
  (*st).mouse_select_button: $
   begin
   print, F="(%'\n\n  Left-click selects source.')" 

   ; Ask ds9 for a mouse click or keystroke and parse the result
   run_command, /QUIET, string((*st).my_ds9, F='(%"xpaget %s imexam any coordinate wcs fk5 degrees")'), result
   tokens = stregex(result,'(.+) +([0-9]+\.[0-9]+) +(-*[0-9]+\.[0-9]+)',/SUB,/EXT)
   keystroke = strtrim(tokens[1],2)
   ra_click  = double(tokens[2])
   dec_click = double(tokens[3])
   click_frame = 1
   if (strlen(keystroke) GT 1) then keystroke=''
   
   ;--------------------------------------------------------------------------
   ; Select the source nearest to the mouse click or keystroke.
   ; An event off the image returns zeros for coordinates.
   deghour = 24D/360
   gcirc, 1, (*st).RA*deghour, (*st).DEC,  ra_click*deghour, dec_click,  distance
   
   far_from_source = (ra_click EQ 0) || (dec_click EQ 0) || (min(distance, source_index) GT 10.0)
   
   ; Match frames to the one receiving the click.
   ;run_command, /QUIET, string((*st).my_ds9, click_frame, F='(%"xpaset -p %s frame %d")')
   ;run_command, /QUIET, string((*st).my_ds9,              F='(%"xpaset -p %s match frame wcs")')

   ; Display current status of selected source.
   if ~far_from_source then ae_source_viewer_select_source, st, source_index
   
   ; Flush any widget events that may have arrived while in ds9 mode.
   widget_control, top_base, /CLEAR_EVENTS
   end


;--------------------------------------------------------------------------
  (*st).approve_multiple_button: $
   begin
   print, F="(%'\nInteractively select and approve sources in ds9:\n  Left-click selects source.\n  Key ""a"" selects source and sets approved flag.\n  Key ""m"" matches all frames to the selected one.')"
   print, F="(%'  To interact with IDL widget press ""q"" or left-click more than 10 arcsec from any source.')"
   
   finished = 0
   repeat begin
     update_widget = 0
     ; Ask ds9 for a mouse click or keystroke and parse the result
     run_command, /QUIET, string((*st).my_ds9, F='(%"xpaget %s imexam any coordinate wcs fk5 degrees")'), result
     tokens = stregex(result,'(.+) +([0-9]+\.[0-9]+) +(-*[0-9]+\.[0-9]+)',/SUB,/EXT)
     keystroke = strtrim(tokens[1],2)
     ra_click  = double(tokens[2])
     dec_click = double(tokens[3])
     click_frame = 1
     if (strlen(keystroke) GT 1) then keystroke=''
     
 ;help,  keystroke, ra_click, dec_click    
      
     ; An event off the image returns zeros for coordinates.
     ; If event is mouse click then exit this mode.
     if (keystroke EQ '') && (ra_click*dec_click EQ 0) then break
      
     ;--------------------------------------------------------------------------
     ; Select the source nearest to the mouse click or keystroke.
     deghour = 24D/360
     gcirc, 1, (*st).RA*deghour, (*st).DEC,  ra_click*deghour, dec_click,  distance
     
     far_from_source = (min(distance, source_index) GT 10.0)
     
     ; Match frames to the one receiving the click.
     ;run_command, /QUIET, string((*st).my_ds9, click_frame, F='(%"xpaset -p %s frame %d")')
     ;run_command, /QUIET, string((*st).my_ds9,              F='(%"xpaset -p %s match frame wcs")')
  
     ;--------------------------------------------------------------------------
     ; Respond to Keystroke
     case keystroke of
      '':  begin
           if far_from_source then begin
             finished = 1
             break
           endif
           update_widget = 1
           end
           
      'a': begin
           if far_from_source then begin
             print, 'Place cursor closer to a source.'
             break
           endif
           ; Set REVIEW_STATE to "approved" for the currently-selected source.
           (*st).cat[source_index].REVIEW_STATE = 'approved'
           (*st).pending_save++
           run_command, /QUIET, string((*st).my_ds9, F='(%"xpaset -p %s frame first")')
           run_command, /QUIET, string((*st).my_ds9, (*st).source_label[source_index], "no", F='(%"xpaset -p %s regions group ''%s'' property include %s")')
           update_widget = 1
           end
           
      'm': begin
           run_command, /QUIET, string((*st).my_ds9, F='(%"xpaset -p %s match frame wcs")')
           end

      'q': begin
           ; Exit the ds9 Mode and return to widget event processing.
           finished = 1
           end
           
      else:
     endcase
     
     ; Display current status of selected source.
     if update_widget then ae_source_viewer_select_source, st, source_index

   endrep until finished
   
   ; Flush any widget events that may have arrived while in ds9 mode.
   widget_control, top_base, /CLEAR_EVENTS
    
   end

   
   
;--------------------------------------------------------------------------
  (*st).match_button: run_command, /QUIET, string((*st).my_ds9, F='(%"xpaset -p %s match frame wcs")')
   
   
;--------------------------------------------------------------------------
  (*st).source_number_id: $
   begin
   widget_control, (*st).source_number_id, GET_VALUE=source_number                                       
   ae_source_viewer_select_source, st, source_number-1, /PAN_TO_SOURCE
   end

;--------------------------------------------------------------------------
  (*st).source_label_id: $
   begin
   widget_control, (*st).source_label_id, GET_VALUE=source_label                                       
   ind = where(strlowcase(strtrim(source_label[0],2)) EQ strlowcase((*st).source_label), count)
   if (count GE 1) then begin
     source_index = ind[0]
     ae_source_viewer_select_source, st, source_index, /PAN_TO_SOURCE
   endif else print, 'Cannot find that source label.'
   end
   
;--------------------------------------------------------------------------
  (*st).prev_button: $
   begin
   source_index-- 
   source_index >= 0 
   ae_source_viewer_select_source, st, source_index, /PAN_TO_SOURCE
   end

;--------------------------------------------------------------------------
  (*st).next_button: $
   begin
   source_index++ 
   source_index <= (num_sources-1)                                                                                                
   ae_source_viewer_select_source, st, source_index, /PAN_TO_SOURCE
   end

;--------------------------------------------------------------------------
  (*st).prev_rank_button: $
   begin
   if ((*st).current_rank LE 0) then begin
     print, 'You are at the first source in '+(*st).srclist_fn
   endif else begin
     (*st).current_rank--
     ind = where((*st).presentation_rank EQ (*st).current_rank, count)
     if (count NE 1) then message, 'BUG!'
     ae_source_viewer_select_source, st, ind[0], /PAN_TO_SOURCE
   endelse
   end

;--------------------------------------------------------------------------
  (*st).next_rank_button: $
   begin
   if ((*st).current_rank GE max((*st).presentation_rank)) then begin
     print, 'You are at the last source in '+(*st).srclist_fn
   endif else begin
     (*st).current_rank++
     ind = where((*st).presentation_rank EQ (*st).current_rank, count)
     if (count NE 1) then message, 'BUG!'
     ae_source_viewer_select_source, st, ind[0], /PAN_TO_SOURCE
   endelse
   end

;--------------------------------------------------------------------------
  (*st).analysis_notes_id: $
   begin
   ; When the text widget is losing keyboard focus we must save its contents.
   if (tag_names(Event, /STRUCTURE_NAME) EQ 'WIDGET_KBRD_FOCUS') && (Event.ENTER EQ 0) then begin
     ;help, /st, event
     notes_fn = (*st).cat[source_index].ACIS.CATALOG_NAME + '/notes.txt'
     widget_control, (*st).analysis_notes_id, GET_VALUE=notes
     
     notes = strtrim(notes,0)
     
     ; If the file exists, then we must write it so the user is able to remove all the text.
     if file_test(notes_fn) ||(total(/INT, strlen(notes)) GT 0) then begin
       ;; Maintain several backups of the notes file.
       for ii=6,1,-1 do begin
         backup_fn       = string(notes_fn,ii,  F='(%"%s-%d")')
         older_backup_fn = string(notes_fn,ii+1,F='(%"%s-%d")')
         if file_test(backup_fn) then file_move, backup_fn, older_backup_fn, /OVERWRITE
       endfor
      
       if file_test(notes_fn) then file_move, notes_fn, backup_fn, /OVERWRITE
       
       ; Save the modified notes.
       print, 'Saving analysis notes to ', notes_fn
       forprint, TEXTOUT=notes_fn, notes, /NoCOMMENT, /SILENT
     endif
   endif
   end
 
   
;--------------------------------------------------------------------------
  (*st).footnotes_id: $
   begin
   ; When the text widget is losing keyboard focus we must save its contents.
   if (tag_names(Event, /STRUCTURE_NAME) EQ 'WIDGET_KBRD_FOCUS') && (Event.ENTER EQ 0) then begin
     ;help, /st, event
     notes_fn = (*st).cat[source_index].ACIS.CATALOG_NAME + '/footnotes.txt'
     widget_control, (*st).footnotes_id, GET_VALUE=notes
     
     notes = strtrim(notes,0)
     
     ; If the file exists, then we must write it so the user is able to remove all the text.
     if file_test(notes_fn) || (total(/INT, strlen(notes)) GT 0) then begin
       ;; Maintain several backups of the notes file.
       for ii=6,1,-1 do begin
         backup_fn       = string(notes_fn,ii,  F='(%"%s-%d")')
         older_backup_fn = string(notes_fn,ii+1,F='(%"%s-%d")')
         if file_test(backup_fn) then file_move, backup_fn, older_backup_fn, /OVERWRITE
       endfor
      
       if file_test(notes_fn) then file_move, notes_fn, backup_fn, /OVERWRITE
       
       ; Save the modified notes.
       print, 'Saving footnotes to ', notes_fn
       forprint, TEXTOUT=notes_fn, notes, /NoCOMMENT, /SILENT
     endif
   endif
   end
   
   
;--------------------------------------------------------------------------
  (*st).done_button: $
   begin
   DestroyFlag = 1
   ; Call ae_source_viewer_select_source one more time so it can save the modified catalog to disk.
   (*st).pending_save = 1000
   ae_source_viewer_select_source, st, source_index
   end
   
;--------------------------------------------------------------------------
 else: begin
       print, 'unknown event in ae_source_viewer'
       help,/ST,Event
       end
endcase
  




; DONE button
if (DestroyFlag) then begin
  ; Remove the temp_dir.
  files = file_search((*st).tempdir+'/*', /MATCH_INITIAL_DOT, COUNT=count)
  if (count GT 0) then file_delete, files, /ALLOW_NONEXISTENT
  file_delete, (*st).tempdir, /ALLOW_NONEXISTENT

  widget_control, top_base, /DESTROY

  print, F='(%"\nSource Viewer terminated")'
endif


return
end ; ae_source_viewer_event


;==========================================================================
;;; Widget Creation Routine

;;; Before calling this tool, create a ds9 session named 'ae_source_viewer' that shows the data and regions that you wish to see, e.g.
;;;   ds9 -title ae_source_viewer -tile iarray.source_search.evt -region polygons.reg -rgb -rgb lock colorbar yes  -red tj2.fits -green th2.fits -blue tk2.fits -region ir_cat.reg &
;;; The first frame must have regions that are tagged by the AE source label; this tool uses those tags to highlight the selected source.

;;; catalog_fn:
;;;   The name of an IDL savefile that holds an array of structures named "cat" that represents our catalog.
;;;   Annotations entered by the observer via this tool are stored in this data structure.
 ;==========================================================================
PRO ae_source_viewer, catalog_fn, SRCLIST_FILENAME=srclist_fn, DS9_TITLE=ds9_title, $
                      NOTES=notes, SKIP_PLOTS=skip_plots, BLOCK=block, READ_ONLY=read_only

resolve_routine, 'match_xy', /COMPILE_FULL_FILE

if ~keyword_set(ds9_title) then ds9_title = 'ae_source_viewer'
if ~keyword_set(catalog_fn) then catalog_fn = 'cat.sav'

;; Restore catalog.
if ~file_test(catalog_fn) then begin
  print, 'ERROR: cannot find file ', catalog_fn
  retall
endif

cat = ''
dum = temporary(cat)
restore, catalog_fn
if (n_elements(cat) EQ 0) then begin
  print, 'ERROR: savefile ', catalog_fn, ' must contain a structure array named "cat".'
  retall
endif

num_sources = n_elements(cat)

sourcename  = strtrim(cat.ACIS.CATALOG_NAME,2)
label       = strtrim(cat.ACIS.LABEL       ,2)
RA          = cat.ACIS.RA
DEC         = cat.ACIS.DEC


case n_elements(notes) of
  0: notes = strarr(num_sources)
  1: notes = replicate(notes, num_sources)
  num_sources: 
  else: begin
        print, 'ERROR: length of NOTES input not equal to number of sources'
        retall
        end
endcase

; Add annotation tags if necessary.
review_state_names  = ['unreviewed','approved','provisional'         ,'unreliable']
review_state_labels = ['unreviewed','approved','review again because','unreliable because']

if ~tag_exist(cat,'REVIEW_STATE') then begin
  temp  = replicate(create_struct('REVIEW_STATE','', 'UNRELIABLE_ANNOTATION','', 'PROVISIONAL_ANNOTATION','', $
                                  'uncatalogued_NIR',0B, 'uncatalogued_MIR',0B, 'nebular_contamination',0B,  $
                                  cat[0]), num_sources)
  struct_assign, cat, temp
  cat = temporary(temp)
  cat.REVIEW_STATE           = review_state_names[0]
  cat.UNRELIABLE_ANNOTATION  = ''
  cat.PROVISIONAL_ANNOTATION = ''
endif


;; Handle any explicit SRCLIST_FILENAME supplied.
if keyword_set(srclist_fn) then begin
  readcol, srclist_fn, subset_sourcename, FORMAT='A', COMMENT=';', DELIM='@'
  
  ; Trim whitespace and ignore blank lines.
  subset_sourcename = strtrim(subset_sourcename,2)

  presentation_rank = replicate(-1L, num_sources)
  rank = 0L
  for ii=0L, n_elements(subset_sourcename)-1 do begin
    if (subset_sourcename[ii] EQ '') then continue
    
    ; Parse lines with semicolons into source names and notes.
    ind = strpos(subset_sourcename[ii],';')
    if (ind NE -1) then begin
      this_note             = strtrim(strmid(subset_sourcename[ii],ind+1) ,2)
      subset_sourcename[ii] = strtrim(strmid(subset_sourcename[ii],0,ind) ,2)
    endif else this_note = ''
    
    ; Look for this sourcename in the full list.
    ind = where(sourcename EQ subset_sourcename[ii], count)
    if (count GT 0) then begin
      ; Mark this source as being in the subset, and record any note that was parsed.
      presentation_rank[ind[0]]  = rank++
      notes            [ind[0]] += this_note
    endif else print, subset_sourcename[ii], F='(%"Source %s is missing from collated table!")'
    
    if (count GT 1) then print, subset_sourcename[ii],  F='(%"WARNING: source %s appears multiple times in table %s")'
  endfor  
  show_rank_navigation = 1
endif else begin
  srclist_fn  = catalog_fn
  presentation_rank = lindgen(num_sources)
  show_rank_navigation = 0
endelse

widget_control, DEFAULT_FONT='micro'

dum = ae_source_viewer_print_source(cat, SKIP_PLOTS=keyword_set(skip_plots), /INIT)


;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir


; Wait for observer's ds9 session to register with XPA.
; Starting in CIAO 4.0, xpaaccess uses the exit code to return its result.
; Thus we can no longer allow run_command to interpret a non-zero exit code as failure.
my_ds9 = "DS9:"+repchr(ds9_title,' ','_')
repeat begin
  run_command, string(my_ds9, F='(%"xpaaccess ''%s''")'), result, /IGNORE_STATUS, /QUIET
  if (result[0] EQ 'yes') then break
  print, 'waiting for ds9 to come up...'
  wait,3
endrep until (0)

; Mark all the sources already reviewed.
print, 'Marking all the sources that have already been reviewed ...'
run_command, /QUIET, string(my_ds9, F='(%"xpaset -p %s frame first")')

for ii=0L,num_sources-1 do begin
  if (cat[ii].REVIEW_STATE NE 'unreviewed') then begin
    run_command, /QUIET, string(my_ds9, label[ii], F='(%"xpaset -p %s regions group ''%s'' property include no")')
    print, label[ii], cat[ii].REVIEW_STATE, F='(%"Source %s is %s.")'
  endif
endfor


  
;widget_control, DEFAULT_FONT='-adobe-helvetica-medium-r-normal--18-180-75-75-p-98-iso8859-1'                    
widget_control, DEFAULT_FONT='-*-fixed-bold-*-*-*-*-140-*-*-*-*-*-*'



top_base = widget_base(TITLE='ACIS Extract Source Viewer', /BASE_ALIGN_CENTER, /COLUMN, $
                        /SPACE, /XPAD, /YPAD)

; source_base = widget_base( top_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER )
 button_base = widget_base( top_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER)
 
   base = widget_base( button_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER, FRAME=1)
     source_number_id = cw_field( base, /INTEGER, /RETURN_EVENTS, XSIZE=4, VALUE=initial_source_number,  TITLE='Seq # in '+file_basename(catalog_fn))                
     prev_button = widget_button( base, VALUE='Previous' )
     next_button = widget_button( base, VALUE='  Next  ' )
   
   source_label_id  = cw_field( button_base, /STRING , /RETURN_EVENTS, XSIZE=15, VALUE='', TITLE='Source Label/Name:')                
   source_name_id   = cw_field( button_base, /STRING, /NOEDIT, XSIZE=18, TITLE='')

   rank_id          = 0L
   prev_rank_button = 0L
   next_rank_button = 0L
 
   if show_rank_navigation then begin
     base = widget_base( button_base, /ROW, /SPACE, /XPAD, /YPAD, /BASE_ALIGN_CENTER, FRAME=1)
       rank_id = cw_field( base, /INTEGER, /RETURN_EVENTS, XSIZE=4, VALUE=1,  TITLE='Seq # in '+file_basename(srclist_fn))                
       prev_rank_button = widget_button( base, VALUE='Previous' )
       next_rank_button = widget_button( base, VALUE='  Next  ' )
   endif
   
   mouse_select_button     = widget_button( button_base, VALUE='Mouse select' )
   done_button = widget_button( button_base, VALUE='  Quit  ' )

 review_base = widget_base( top_base, /ROW, SPACE=0, /XPAD, /YPAD, /BASE_ALIGN_BOTTOM, FRAME=2)
;   dum              = widget_label( annotation_base, VALUE='Proposed counterparts are:' )
   review_state_id  = cw_bgroup( review_base, review_state_labels, BUTTON_UVALUE=review_state_names, /EXCLUSIVE, /NO_RELEASE, /COLUMN, LABEL_TOP='Counterpart Match Status:' )
   
   
   annotation_base = widget_base( review_base, /COLUMN, SPACE=0, /XPAD, /YPAD, FRAME=2)
     provisional_annotation_id = widget_combobox(annotation_base, /EDITABLE, /DYNAMIC_RESIZE, VALUE=[' '] );, SCR_XSIZE=5, UNITS=1  )
     
     unreliable_annotation_id  = widget_combobox(annotation_base, /EDITABLE, /DYNAMIC_RESIZE, VALUE=[' ', '2 IR srcs may match 1 ACIS src', '2 ACIS srcs may match 1 IR src', 'Confusion -- multiple sources', 'Confusion -- nebular emission', 'Match has large offset'] );, SCR_XSIZE=5, UNITS=1  )

 flag_group_id  = cw_bgroup( top_base, ['Uncatalogued NIR counterpart', 'Uncatalogued MIR counterpart', 'Strong IR nebular emission'], /NONEXCLUSIVE, /COLUMN )
 
     
 button_base = widget_base( top_base, /ROW, SPACE=0, /XPAD, /YPAD, /BASE_ALIGN_BOTTOM )
   approve_multiple_button = widget_button( button_base, VALUE='Approve multiple sources' )
   match_button            = widget_button( button_base, VALUE='Match frames' )
 
 text_base = widget_base( top_base, /ROW, SPACE=0, /XPAD, /YPAD, /BASE_ALIGN_BOTTOM )
   base = widget_base( text_base, /COLUMN, SPACE=0, /XPAD, /YPAD )
   temp = widget_label( base, VALUE='Analysis notes:' )    
   analysis_notes_id = widget_text( base, /SCROLL, XSIZE=28, YSIZE=4, /EDITABLE, /WRAP, /KBRD_FOCUS_EVENTS, /IGNORE_ACCEL )
  
   base = widget_base( text_base, /COLUMN, SPACE=0, /XPAD, /YPAD )
   temp = widget_label( base, VALUE='Footnotes:' )    
   footnotes_id      = widget_text( base, /SCROLL, XSIZE=28, YSIZE=4, /EDITABLE, /WRAP, /KBRD_FOCUS_EVENTS, /IGNORE_ACCEL )
  
;misc_id   = cw_bgroup( top_base, [''], /NONEXCLUSIVE, SET_VALUE=0 )


report_base = widget_base(TITLE='Source Properties', GROUP=top_base, XOFFSET=0, YOFFSET=600, /COLUMN, /SPACE, /XPAD, /YPAD)

  report_id = widget_text( report_base, /SCROLL, XSIZE=60, YSIZE=15 )

state = { skip_plots:keyword_set(skip_plots), read_only:keyword_set(read_only), $
  
          review_state_names:review_state_names, review_state_id:review_state_id, $
          unreliable_annotation_id:unreliable_annotation_id,  provisional_annotation_id:provisional_annotation_id, $
          flag_group_id:flag_group_id, $
          analysis_notes_id:analysis_notes_id, footnotes_id:footnotes_id, $
          source_number_id:source_number_id, source_label_id:source_label_id, source_name_id:source_name_id, $
          prev_button:prev_button, next_button:next_button, $
          prev_rank_button:prev_rank_button, next_rank_button:next_rank_button, $
          mouse_select_button:mouse_select_button, $
          report_id:report_id, $
          approve_multiple_button:approve_multiple_button, match_button:match_button, done_button:done_button, $
          
          current_rank:-1L, current_source_index:-1L, sourcename:sourcename, source_label:label, RA:RA, DEC:DEC, $
          notes:notes, srclist_fn:srclist_fn, presentation_rank:presentation_rank, my_ds9:my_ds9, $
          
          catalog_fn:catalog_fn, pending_save:0, save_counter:0L, cat:cat, $
          
          tempdir:tempdir}

; Save state structure.
widget_control, top_base, SET_UVALUE=ptr_new(state, /NO_COPY)

widget_control, report_base, /REALIZE
widget_control, top_base,    /REALIZE

event={ID:source_number_id, TOP:top_base, HANDLER:top_base}
ae_source_viewer_event, event

  
; The fancy combination of JUST_REG and NO_BLOCK is necessary to get the
; widget to behave when called by either widget applications or normal
; programs (which block on tty reads).
xmanager, 'ae_source_viewer', top_base, EVENT_HANDLER='ae_source_viewer_event', $
          JUST_REG=keyword_set(block), NO_BLOCK=(keyword_set(block) EQ 0)
return
end



;==========================================================================
;;; Dim source simulator.
;==========================================================================
PRO ae_dim_source_sim, obsdir, num_events


obs_stats = headfits(obsdir+'/obs.stats')

src_radius = psb_xpar( obs_stats, 'SRC_RAD')

bt=mrdfits(obsdir+'/source.evt', 1, theader)

dim = floor(sqrt(n_elements(bt)/num_events))
region_file = string(obsdir, dim, dim, F='(%"%s/grid%dX%d.reg")') 
event_file  = string(obsdir, dim, dim, F='(%"%s/grid%dX%d.evt")')

x0=median(bt.X)
y0=median(bt.Y)
xoffset= findgen(dim)*3*src_radius
yoffset= findgen(dim)*3*src_radius
make_2d,xoffset,yoffset
help, xoffset,yoffset
xoffset = reform(xoffset,n_elements(xoffset))
yoffset = reform(yoffset,n_elements(xoffset))
;info,xoffset
;info,yoffset

openw,  region2_unit, region_file, /GET_LUN
printf, region2_unit, "# Region file format: DS9 version 3.0"
printf, region2_unit, 'global width=1 font="helvetica 12 normal"'
!TEXTUNIT = region2_unit
forprint, TEXTOUT=5, x0+xoffset, y0+yoffset, replicate(src_radius, n_elements(xoffset)), F='(%"circle(%f,%f,%f)")', /NoCOMMENT
free_lun, region2_unit

num_rows = num_events*dim^2

xoffset = rebin(xoffset, num_rows, /SAMPLE)
yoffset = rebin(yoffset, num_rows, /SAMPLE)

grid = bt[0:num_rows-1]
grid.X = grid.X + xoffset
grid.Y = grid.Y + yoffset

psb_xaddpar, theader, 'TLMAX18', max(grid.x)+100
psb_xaddpar, theader, 'TLMAX19', max(grid.y)+100
mwrfits, [bt,grid], event_file, theader, /CREATE

run_command, /QUIET, string(event_file, region_file, F='(%"ds9 %s -region %s >& /dev/null &")')
print, 'THETA=',psb_xpar( obs_stats,'THETA')
return
end



;==========================================================================
;;; An Interface to the CXC's ChaRT model, for use in building mono-energy PSFs for AE
;;; See http://asc.harvard.edu/chart/

;;; Example call:    ae_chart_interface, '104357.47-593251.3', '4495'


;!!!! 2013 Oct.  I HAVE LEARNED THAT ChaRT+MARXv5 cannot simulate PSFs for data processed with the EDSER sub-pixel positioning algorithm.
; See Chandra helpdesk ticket #15403 (Kenny Glotfelty) on Oct1,2.
;I have removing the ae_chart_interface tool for now.


; ;==========================================================================
; PRO ae_chart_interface, sourcename, obsname, ASPECT_FN=aspect_fn, SKIP_DOWNLOAD=skip_download
; 
; creator_string = "ae_chart_interface, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
; print, creator_string, F='(%"\n\n%s")'
; print, now()
; 
; if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
;   print, 'ERROR: parameter "obsname" must be a scalar string'
;   retall
; endif
; 
; 
; ; AE puts the 1.5 keV PSF first in the file.
; desired_psf_energy    = [1.49670, 0.2770, 4.510, 6.40, 8.60]
; 
; ; Choose ray densities that will generate a similar number of PSF events as AE uses ("desired_psf_counts" = [1E6,1E5,1E5,1E5,1E5] in acis_extract.pro).  
; ; ChaRT imposes a maximum density of 10 rays/mm**2.
; chart_density = ( [11.9,1.19085,2.36809,4.13947,15.7376] ) < 10  ; rays/mm**2
;                      
; src_stats_basename       = 'source.stats'
; psf_basename             = 'source.psf'
; obs_stats_basename       = 'obs.stats'
; env_events_basename      = 'neighborhood.evt'
; src_events_basename      = 'source.evt'
; 
; 
;; Create a unique scratch directory.
;tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)
; 
; temp_events_fn       = tempdir + 'temp.evt'
; temp_region_fn       = tempdir + 'temp.reg'
; temp_image_fn        = tempdir + 'temp.img'
; 
; 
; 
; 
; 
; ;; =====================================================================
; ;; Initialize stuff.
; run_command, PARAM_DIR=tempdir
; 
; ;; Get rid of pre-existing configuration for the CIAO commands we'll use below.
; run_command, /QUIET, ['punlearn get_sky_limits', 'punlearn dmcopy', 'pset dmcopy clobber=yes']
; 
; sourcedir       = sourcename + '/'  
; obsdir          = sourcename + '/' + obsname + '/'
; chartdir        = obsdir + 'chart'
; file_mkdir, chartdir
; 
; src_stats_fn    = sourcedir + src_stats_basename
; psf_fn          = obsdir +            psf_basename
; chart_psf_fn    = obsdir + 'chart_' + psf_basename
; stats_fn        = obsdir + obs_stats_basename
; env_events_fn   = obsdir + env_events_basename
; src_events_fn   = obsdir + src_events_basename
; 
; src_stats    = headfits(src_stats_fn, ERRMSG=error)
; obs_stats    = headfits(stats_fn, ERRMSG=error)
; event_header = headfits(src_events_fn, EXT=1)
; 
; ;; =====================================================================
; ;; Calculate information needed to create the PSF.
; ra  = psb_xpar( src_stats,   'RA')
; dec = psb_xpar( src_stats,   'DEC')
; 
; ;; ------------------------------------------------------------------------
; ;; Calculate HRMA coordinates from celestial coordinates.
; cmd = string(src_events_fn, aspect_fn, ra, dec, F="(%'dmcoords %s asolfile=%s opt=cel celfmt=deg ra=%10.6f dec=%10.6f')")
; run_command, cmd
; 
; run_command, /QUIET, 'pget dmcoords theta phi x y chip_id', result
; off_angle    = float(result[0])  ; arcmin
; phi          = float(result[1])
; xpos_catalog = float(result[2])
; ypos_catalog = float(result[3])
; chip_id      = fix  (result[4])
; 
; print, 'At the URL http://asc.harvard.edu/chart/runchart.html, submit a ChaRT run with the following configuration, for each of the FIVE mono-energies and ray densities shown below.'
; print, 'Use the BACK button in your browser to edit the energy and ray density fields.'
; print, off_angle, phi, desired_psf_energy, chart_density, F='(%"\n  Off-axis Angle = %0.2f\n  Azimuth Angle  = %0.1f\n  \"monochromatic energy and ray density\"\n  Monochromatic Energy = %6.3f %6.3f %6.3f %6.3f %6.3f\n  Ray Density          = %6.2f %6.2f %6.2f %6.2f %6.2f\n")'
; 
; 
; run_command, /UNIX, "open http://asc.harvard.edu/chart/runchart.html", STATUS=status, /QUIET
; 
; if ~keyword_set(skip_download) then begin
;   for ii=0,n_elements(desired_psf_energy)-1 do begin
;     print, 'Look for five emails from ChaRT; enter the URLs one at a time in any order'
;     url = ''
;     read, string(ii+1,F='(%"URL \#%d: ")'), url
;     
;     if (strtrim(url,2) EQ '') then break
;     run_command, /UNIX, string(chartdir, url, F='(%"wget --no-verbose --directory-prefix=%s %s")')
;     
;   endfor ;ii
; endif  
; 
; ray_dirname = chartdir
; ; read, 'Enter name of directory containing ChaRT ray files: ', ray_dirname
; 
; temp_ray_filename = file_search(ray_dirname, '*fits', COUNT=num_ray_files)
; 
; 
; ;; =====================================================================
; ;; Process each ray file in the same order as listed in desired_psf_energy.
; ray_filename = strarr(num_ray_files)
; psf_energy   = strarr(num_ray_files)
; 
; for ii=0,num_ray_files-1 do begin
;   ;; Look in the ChaRT header for the mono-energy used to generate the rays.
;   cmd = string(temp_ray_filename[ii], F="(%'dmkeypar %s SRC_E echo+')")
;   run_command, cmd, result, /QUIET
;   
;   dum = min( abs(desired_psf_energy - float(result[0])), imin )
;   ray_filename[ii] = temp_ray_filename [imin]
;   psf_energy  [ii] = desired_psf_energy[imin]
; endfor ;ii
; 
; forprint, psf_energy, ray_filename, F='(%"Rays for %6.3f keV PSF are in %s")'
; 
; psf_header = headfits(psf_fn)
; 
; ; The ChaRT thread says to use MARX's internal dither model, so we do not pass the observation's aspect file.
; ae_make_psf, TEMP_DIR=tempdir, EVENT_FILE=env_events_fn, ASPECT_BLUR=psb_xpar( psf_header, 'ASP_BLUR'), PIX_ADJ=strtrim(psb_xpar( psf_header, 'PIX_ADJ'),2), /SAOSACFile
; 
; ;; Convert ChaRT rays into an AE PSF file.
; ;; Match pixel size and image dimensions of existing PSF. 
; skypixel_per_psfpixel = psb_xpar( psf_header, 'CDELT1P')
; 
; ae_make_psf, TEMP_DIR=tempdir, ra, dec, chart_psf_fn, skypixel_per_psfpixel, psb_xpar( psf_header, 'NAXIS1')*skypixel_per_psfpixel, $
;   psf_energy, 0, X_CAT=xpos_catalog, Y_CAT=ypos_catalog, OFF_ANGLE=off_angle, CHIP_ID=chip_id, $
;   SAOSACFile=ray_filename
; 
; print, 'THIS IS CRAP!  The resulting MARX simulation covers a very short time interval, with little dither motion.  The resulting event list and PSF have severe spatial quantization effects!  Helpdesk ticket #15403.'
; stop  
; ;; =====================================================================
; ;; Plot a radial profile for the first AE PSF and first ChaRT PSF.
; ;; Both must be normalized by the appropriate PSF total.
; src_radius  = psb_xpar( obs_stats, 'SRC_RAD') * 2
; src_radius  = 20
; energy_range = [1.0,2.0]
; ae_radial_profile, report, SOURCENAME=sourcename, RA=ra, DEC=dec, SRC_RADIUS=src_radius,$
;   ENERGY_RANGE=energy_range, EVENTLIST_FN=env_events_fn, PSF_FN=chart_psf_fn, PSF_NAME='ChaRT PSF',$
;   TEMPDIR=tempdir, /PLOT
;  
; ae_radial_profile, report, SOURCENAME=sourcename, RA=ra, DEC=dec, SRC_RADIUS=src_radius,$
;   ENERGY_RANGE=energy_range, EVENTLIST_FN=env_events_fn, PSF_FN=psf_fn, PSF_NAME='Template PSF',$
;   TEMPDIR=tempdir, /PLOT
;  
; 
; ;for ii=0,1 do begin
; ;  psf_img = readfits((ii EQ 0) ? chart_psf_fn : psf_fn, psf_header, /SILENT)
; ;  print, 'Plotting radial profile at ', psb_xpar( psf_header,'ENERGY'), ' keV'
; ;  
; ;  ; Make an array that has the distances (in units of skypixel) from each PSF pixel to the source.
; ;  ; For speed, work with only pixels where the PSF is not zero.
; ;  skypixel_per_psfpixel = psb_xpar( psf_header, 'CDELT1P')
; ;  extast, psf_header, psf2wcs_astr
; ;  ad2xy, ra, dec, psf2wcs_astr, xind_catalog, yind_catalog
; ;  
; ;  dist_circle, psf_distance, size(psf_img, /DIM), xind_catalog, yind_catalog
; ;  psf_distance  = psf_distance * skypixel_per_psfpixel
; ;  
; ;  ; Sort by distance to be ready to form a 1-D model.
; ;  sort_ind      = sort(psf_distance)
; ;  psf_distance  = psf_distance [sort_ind]
; ;  psf_img       = psf_img      [sort_ind]
; ;
; ;  ; Form the cumulative distribution function: psf_distn(psf_distance).
; ;  ; This is the 1-D model of the composite PSF.
; ;  psf_distn = total(psf_img, /NAN, /DOUBLE, /CUMULATIVE) / psb_xpar( psf_header,'PSF_TOTL')
; ;  
; ;  if (ii EQ 0) then begin
; ;    chart_psf_distance = psf_distance
; ;    chart_psf_distn    = psf_distn
; ;  endif
; ;endfor
; ;
; ;function_1d, id2,       psf_distance,       psf_distn, DATASET='PSF Library', XTIT='distance [skypix]', YTIT='enclosed fraction'
; ;function_1d, id2, chart_psf_distance, chart_psf_distn, DATASET='ChaRT'
; 
; 
; CLEANUP:
; if file_test(tempdir) then begin
;   list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
;   if (count GT 0) then file_delete, list
;   file_delete, tempdir
; endif
; return
; end ; ae_chart_interface




;==========================================================================
;;; Convert source labels to source names.

;;; The parameter "labels" can either be a string array, or an integer.
;;; If an integer is supplied, then the tool prompts for you to enter the labels.
;==========================================================================
PRO ae_label2name, labels, collated_filename, catalog_names, indexes

collated_table = mrdfits(collated_filename,1)

all_labels = strlowcase(strtrim(collated_table.LABEL,2))

if (size(/TNAME, labels) EQ 'STRING') then begin
  num_labels = n_elements(labels)
endif else begin
  num_labels = labels
  labels = strarr(num_labels)
  flush_stdin  ; eat any characters waiting in STDIN, so that they won't be mistaken as commands in the loop below.
  print, num_labels, F='(%"\nEnter %d source labels, one per line ...")'
  read, labels
endelse

labels = strlowcase(strtrim(labels,2))

indexes = replicate(-1L,num_labels)
catalog_names  = replicate('XXXX', num_labels)

for ii=0,num_labels-1 do begin
  this_label = labels[ii]
  ind = where(all_labels EQ this_label, count)
  case count of
    0: print, 'WARNING: label '+labels[ii]+' not found'
    1: begin
       indexes      [ii] =                               ind
       catalog_names[ii] = (collated_table.CATALOG_NAME)[ind]
       end
    else: begin
          print, this_label, F="(%'ERROR: duplicate label %s found in catalog:')"
          forprint, collated_table.CATALOG_NAME, all_labels, SUBSET=ind, F='(%"  %s == %s")'
          end
  endcase
endfor

forprint, SUBSET=where(indexes NE -1), catalog_names, labels, F="(%'%s ; (%s)')"
return
end



;==========================================================================
;; Routine to create a set of fake sources that "tile" a field of view, to
;; be used with ae_recon_detect.

;; NEIGHBORHOOD_SIZE is the same keyword option used in AE to specify the
;; width and height of the source neighborhood, in arcseconds.
;;
;; THETA_RANGE (2-element vector, in arcmin) is useful only for single-ObsID emaps (e.g. in astrometry_recipe.txt).
;;
;; The result is a region file tiles.reg showing the tiles.  This can be 
;; hand edited in ds9 to reposition the tiles as desired.
;==========================================================================

PRO ae_neighborhood_tile, emap_filename, MASK_FILENAME=mask_filename, NEIGHBORHOOD_SIZE=neighborhood_size_arcsec, OVERLAP=overlap_arcsec, STYLE=style, THETA_RANGE=theta_range, COLOR=color_p
                                                                                     
if ~keyword_set(neighborhood_size_arcsec) then neighborhood_size_arcsec = 50 ; arcsec

if (n_elements(style) EQ 0) then style = 2

if (n_elements(theta_range) NE 2) then theta_range = [0,100.]

emap = readfits(emap_filename, emap_header)
extast, emap_header, emap2wcs_astr
emap_xdim = (size(emap, /DIM))[0]
emap_ydim = (size(emap, /DIM))[1]
arcsec_per_imgpixel = emap2wcs_astr.CDELT[1] * 3600

if keyword_set(mask_filename) then begin
  mask = readfits(mask_filename)
  
  ; Defensively, verify that emap and mask have the same dimensions.
  if ~ARRAY_EQUAL(size(emap, /DIMEN), size(mask, /DIMEN)) then $
    message, 'ERROR: mask and emap must have the same dimensions'
endif else mask = emap

mask = (mask GT 0)



tile_radius_arcsec = neighborhood_size_arcsec/2.0 
case style of
0: begin
  x_oddrow_offset_arcsec = 0
  x_pitch_arcsec         = neighborhood_size_arcsec - overlap_arcsec
  y_pitch_arcsec         = neighborhood_size_arcsec - overlap_arcsec
  core_radius_arcsec     =(neighborhood_size_arcsec   - 2*overlap_arcsec)/2.0
  end

1: begin
  x_oddrow_offset_arcsec = neighborhood_size_arcsec/2.0
  x_pitch_arcsec         = neighborhood_size_arcsec
  y_pitch_arcsec         = neighborhood_size_arcsec/2.0
  core_radius_arcsec     = 1
  end

2: begin
  x_oddrow_offset_arcsec = neighborhood_size_arcsec/2.0 - overlap_arcsec/2.0
  x_pitch_arcsec         = neighborhood_size_arcsec     - overlap_arcsec
  y_pitch_arcsec         = neighborhood_size_arcsec     - overlap_arcsec
  core_radius_arcsec     =(neighborhood_size_arcsec   - 2*overlap_arcsec)/2.0
  end
endcase

color_matrix = ['red','green','cyan','blue']
color_matrix = reform(color_matrix, 2,2)

x_pitch         = round(x_pitch_arcsec         / arcsec_per_imgpixel)
y_pitch         = round(y_pitch_arcsec         / arcsec_per_imgpixel)
x_oddrow_offset = round(x_oddrow_offset_arcsec / arcsec_per_imgpixel)
tile_radius     = round(tile_radius_arcsec     / arcsec_per_imgpixel)
core_radius     = ceil (core_radius_arcsec     / arcsec_per_imgpixel)

openw,  region_unit, 'tiles.reg', /GET_LUN
printf, region_unit, "# Region file format: DS9 version 3.0"
printf, region_unit, 'global width=1 font="helvetica 12 normal"'
printf, region_unit, "J2000"

num_tiles = 0L

x_grid     = tile_radius
is_odd_col = 0

y_grid     = tile_radius
is_odd_row = 0B

while 1 do begin
  x_center = x_grid
  y_center = y_grid
  color = keyword_set(color_p) ? color_p : color_matrix[is_odd_col,is_odd_row]

  tag   = string(is_odd_col,is_odd_row, F="(%'%d%d')")
  
  skip = 0
  ; The kk loop allows us to reposition the tile multiple times.
  for kk=0,1 do begin
    ; Skip tile that violates the THETA_RANGE.  This is useful only for single-ObsID emaps!!!
    xy2ad, x_center, y_center, emap2wcs_astr, RA, DEC
    gcirc, 2, RA, DEC, psb_xpar( emap_header,'RA_PNT'), psb_xpar( emap_header,'DEC_PNT'), theta_arcsec
    THETA = theta_arcsec/60.
    
    if (THETA LT theta_range[0]) || (THETA GT theta_range[1]) then begin
      skip = 1
      break
    endif

    
    ; Reposition tile that's falling off the edge of the emap.
    while ((x_center-tile_radius) LT            0 ) do x_center++
    while ((y_center-tile_radius) LT            0 ) do y_center++
    while ((x_center+tile_radius) GT (emap_xdim-1)) do x_center--
    while ((y_center+tile_radius) GT (emap_ydim-1)) do y_center--
    
    ; Boundaries of tile.
    x_tl    = (x_center-tile_radius) > 0
    y_tl    = (y_center-tile_radius) > 0
    x_th    = (x_center+tile_radius) < (emap_xdim-1)
    y_th    = (y_center+tile_radius) < (emap_ydim-1)
    
    ; Boundaries of core.
    x_cl    = (x_center-core_radius) > 0
    y_cl    = (y_center-core_radius) > 0
    x_ch    = (x_center+core_radius) < (emap_xdim-1)
    y_ch    = (y_center+core_radius) < (emap_ydim-1)
    
    ; Skip tile when its core does not intersect the mask.
    if total(/INT, mask[x_cl:x_ch, y_cl:y_ch]) EQ 0 then begin
      skip = 1
      break
    endif
    
    ; Determine fraction of tile that is outside the field-of-view of the (full) emap.
    tile_emap = emap[x_tl:x_th, y_tl:y_th]
    
    frac_off_field = total(/INT, tile_emap EQ 0) / float(n_elements(tile_emap))
    
    
    ; Reposition the tile when it's significantly off-field.
    if (frac_off_field GT 0.10) then begin
      ; Compute center-of-mass of the emap found underneath the tile's current position, and make that the new tile center.
      x_offset = indgen(1+x_th-x_tl)
      y_offset = indgen(1+y_th-y_tl)
      make_2d, x_offset, y_offset
    
      x_center = x_tl + round(total(tile_emap*x_offset) / total(tile_emap))
      y_center = y_tl + round(total(tile_emap*y_offset) / total(tile_emap))
      color = keyword_set(color_p) ? color_p : 'yellow'
      tag   = 'off-grid'
    endif else begin
      break
    endelse
  endfor ; kk
  
  if ~skip then begin
    ; Zero the mask under the (repositioned) tile core, indicating the region that needs no further coverage.
    x_cl    = (x_center-core_radius) > 0
    y_cl    = (y_center-core_radius) > 0
    x_ch    = (x_center+core_radius) < (emap_xdim-1)
    y_ch    = (y_center+core_radius) < (emap_ydim-1)
    mask[x_cl:x_ch, y_cl:y_ch] = 0B
  
    ; Convert tile coordinates from image to celestial system, and save region to file.
    xy2ad, x_center, y_center, emap2wcs_astr, ra, dec
    num_tiles++
    printf, region_unit, ra, dec, [2,2]*tile_radius_arcsec + 2*is_odd_col, color, tag, neighborhood_size_arcsec,  F="(%'box(%10.6f,%10.6f,%5.1f"",%5.1f"") # tag={tile fov} color={%s} tag={%s} tag={size=%d""}')"
  endif

  ; Advance to the next tile center.
  x_grid += x_pitch
  is_odd_col = ~is_odd_col
  
  if (x_grid GT (emap_xdim-1)) then begin
    ; Start a new row.
    x_grid     = tile_radius
    is_odd_col = 0

    y_grid    += y_pitch
    is_odd_row = ~is_odd_row

    if is_odd_row then x_grid += x_oddrow_offset
    if (y_grid GT (emap_ydim-1)) then break
  endif
endwhile



free_lun, region_unit

forprint, neighborhood_size_arcsec, TEXTOUT='neighborhood_size.txt', /NoCOMMENT, /SILENT

print, num_tiles, ' tiles written to tiles.reg.'

;run_command, /QUIET, string(keyword_set(mask_filename) ? mask_filename : emap_filename, F="(%'ds9 -log %s -region tiles.reg -zoom to fit >& /dev/null &')")
return
end ; ae_neighborhood_tile



PRO ae_rm_duplicate_tiles, region_file
      readcol, region_file, lines, F='(A)', DELIM='@'
      
      num_lines = n_elements(lines)
      good = replicate(1B,num_lines)

      result = stregex(lines,'box[^0-9]*([0-9]+\.[0-9]+)[^-0-9]*(-*[0-9]+\.[0-9]+)',/SUB,/EXT)
      x = double(reform(result[1,*]))
      y = double(reform(result[2,*]))
      
      ; Look for non-zero duplicates.
      for ii=0,num_lines-1 do begin
        if (x[ii] EQ 0) then continue
        
        sqr_dist = (x-x[ii])^2 + (y-y[ii])^2
        ind = where(sqr_dist LT 1^2, count)
        if (count EQ 1) then continue
        
        ind=ind[1:*]
        x[ind]    = 0
        y[ind]    = 0
        good[ind] = 0
      endfor

      forprint, TEXTOUT='/tmp/temp.reg', SUBSET=where(good), lines, /NOCOMMENT
      print, 'wrote cleaned regions to /tmp/temp.reg'
      run_command, 'wc /tmp/temp.reg '+region_file
      
      
      lines = lines[where(good, count)] + string(1+indgen(count), F='(%" text={%d}")')
      forprint, TEXTOUT='/tmp/temp.reg', lines, /NOCOMMENT
return
end






;==========================================================================
; Extract tile neighborhoods from a set of ObsIDs, perform various named merges.

; Several steps are involved:
;   1. AE builds PSF images for every ObsID at each tile center.
; 
;   2. AE extracts neighborhoods events from the heavily-cleaned data in every ObsID.
; 
;   3. For each tile, AE merges the extractions that satisfy the THETA_RANGE specified.
;
;   4. For each tile, AE builds all possible single-ObsID merges.
; 
; Inputs THETA_LO and THETA_HI define a set of off-axis ranges in which extractions are merged.
; On-axis tiling:
;    theta_lo = [0,2,5] ; arcmin
;    theta_hi = [3,6,9] ; arcmin
    
; Off-axis tiling:
;    theta_lo = [ 8,13] ; arcmin
;    theta_hi = [14,99] ; arcmin
    
; NEIGHBORHOOD_SIZE is width and height of tile, in arcseconds

; Optional inputs PSF_MODEL_ENERGY, PSF_MODEL_COUNTS are passed to acis_extract, so that the CHECK_POSITIONS stage will have access to a PSF suitable for every energy band you will be searching.

; Optional input SKIP_SINGLE_OBSID_MERGES omits the single-ObsID merges.

; We want the /MERGE stage to do its job without considering whether tile centers are "crowded", so we set a very high OVERLAP threshold.
; 
; Output MERGE_NAME_LIST is the set of MERGE_NAMES retained for any tile.


;==========================================================================

PRO ae_extract_tiles, srclist_fn, obsname, THETA_LO=theta_lo, THETA_HI=theta_hi, NEIGHBORHOOD_SIZE=neighborhood_size, PSF_MODEL_ENERGY=psf_model_energy, PSF_MODEL_COUNTS=psf_model_counts, SKIP_SINGLE_OBSID_MERGES=skip_single_obsid_merges, MERGE_NAME_LIST=merge_name_list

creator_string = "ae_extract_tiles, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()
exit_code = 0

if ~keyword_set(theta_lo) || ~keyword_set(theta_hi) then begin
  theta_lo = [0,2,5,  8,13] ; arcmin
  theta_hi = [3,6,9, 14,99] ; arcmin
endif


;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir
  
active_catfile     = tempdir + 'temp.cat'


; Read the source list supplied.
readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)

if (num_sources EQ 0) then begin
  print, 'ae_extract_tiles: ERROR: no entries read from source list ', srclist_fn
  GOTO, FAILURE
endif

sourcename = sourcename[ind]



; Write a temporary AE 5-column catalog holding the tile list.
cat_all = replicate({sourcename: '', ra:0D, dec:0D, psf_frac:0.90, psf_energ:1.4967}, num_sources)
cat_all.sourcename = sourcename

fmt = '(A,1x,F10.6,1x,F10.6,1x,F5.3,1x,F7.5)'
forprint, TEXTOUT=active_catfile, cat_all.sourcename, cat_all.ra, cat_all.dec, cat_all.psf_frac, cat_all.psf_energ, F=fmt, /NoCOMMENT, /SILENT


for ii=0,n_elements(obsname)-1 do begin
  print, F='(%"\nae_extract_tiles: ============================================================")'  
  print,        'ae_extract_tiles: PROCESSING OBSERVATION ' + obsname[ii]
  print,   F='(%"ae_extract_tiles: ============================================================")'  

  obsdir      = '../obs' + obsname[ii] + '/'
  if ~file_test(obsdir) then begin
    print, 'ae_extract_tiles: ERROR: could not find observation directory ', obsdir
    GOTO, FAILURE
  endif

  evtfile     = obsdir + "validation.evt"
  emapfile    = obsdir + 'obs.emap'
  aspect_fn   = obsdir + 'obs.asol'

  ; Single-ObsID extractions
  ;; ---------------------------------------------------------------------
  print, F='(%"\nae_extract_tiles: ============================================================")'  
  print,        'ae_extract_tiles: Running /CONSTRUCT_REGIONS on ' + active_catfile
  print,   F='(%"ae_extract_tiles: ============================================================\n")'  

  acis_extract, active_catfile, obsname[ii], evtfile, /CONSTRUCT_REGIONS, EMAP_FILENAME=emapfile, ASPECT_FN=aspect_fn, MASK_FRACTION=0.98, MASK_MULTIPLIER=1.0, PSF_MODEL_ENERGY=psf_model_energy, PSF_MODEL_COUNTS=psf_model_counts, SHOW=0

  
  ; Extract Events (to get SRC_CNTS statistic).
  ; Pass /REGION_ONLY since we care about PSF fraction only at the primary energy.
  print, F='(%"\nae_extract_tiles: ============================================================")'  
  print,        'ae_extract_tiles: Running /EXTRACT_EVENTS on ' + active_catfile
  print,   F='(%"ae_extract_tiles: ============================================================\n")'  
  acis_extract, active_catfile, obsname[ii], evtfile, /EXTRACT_EVENTS, EMAP_FILENAME=emapfile, ASPECT_FN=aspect_fn, NEIGHBORHOOD_SIZE=neighborhood_size[0], SHOW=0

endfor; ii

merge_name_list = []
; Off-axis angle (theta) slices.
for ii=0,n_elements(theta_lo)-1 do begin                                             
  merge_name = string(theta_lo[ii], theta_hi[ii], F='(%"theta_%2.2d-%2.2d")')

  acis_extract, srclist_fn, MERGE_NAME=merge_name, /MERGE_OBSERVATIONS, THETA_RANGE=[theta_lo[ii], theta_hi[ii]], OVERLAP_LIMIT=1E6, /SKIP_SPECTRA, /SKIP_TIMING 
  
  merge_name_list = [merge_name_list, merge_name]
endfor; ii


if ~keyword_set(skip_single_obsid_merges) then begin
  ; Single-ObsID merges, within the overall theta range in play.
  for ii=0,n_elements(obsname)-1 do begin
    merge_name = 'EPOCH_'+obsname[ii]
   
    acis_extract, srclist_fn, obsname[ii], MERGE_NAME=merge_name, /MERGE_OBSERVATIONS,  THETA_RANGE=[min(theta_lo), max(theta_hi)], OVERLAP_LIMIT=1E6, /SKIP_SPECTRA, /SKIP_TIMING
    
    merge_name_list = [merge_name_list, merge_name]
endfor; ii
endif



; Remove empty and duplicate MERGES, using ObsIDs_merged.fits files produced by each one.
print, F='(%"\nPruning duplicate merges ...")'
for ii=0, num_sources-1 do begin
  for jj=0     , n_elements(merge_name_list)-1 do begin
    reference_sourcedir = sourcename[ii] + '/' + merge_name_list[jj] + '/'
    reference_logfile   = reference_sourcedir+'ObsIDs_merged.fits'

    ; We may have removed this merge already.
    if ~file_test(reference_sourcedir) then continue

    ; Or, the merge may be empty (no extractions were found, but a merge directory was made by AE).
    if ~file_test(reference_logfile) then begin
        file_delete, reverse(file_search(reference_sourcedir,'*',/MATCH_INITIAL_DOT))
        file_delete,                     reference_sourcedir
        print, reference_sourcedir, F='(%"  Pruned %s (empty merge directory).")'
        continue
    endif

    reference_obs_data = mrdfits(reference_logfile,1,/SILENT)


    for kk=jj+1, n_elements(merge_name_list)-1 do begin
      test_sourcedir = sourcename[ii] + '/' + merge_name_list[kk] + '/'
      test_logfile   = test_sourcedir+'ObsIDs_merged.fits'

      ; We may have removed this merge already.
      if ~file_test(test_sourcedir) then continue
  
      ; Or, the merge may be empty (no extractions were found, but a merge directory was made by AE).
      if ~file_test(test_logfile) then begin
          file_delete, reverse(file_search(test_sourcedir,'*',/MATCH_INITIAL_DOT))
          file_delete,                     test_sourcedir
          print, test_sourcedir, F='(%"  Pruned %s (empty merge directory).")'
          continue
      endif
  
      test_obs_data = mrdfits(test_logfile,1,/SILENT)

      ; Finally, we have two not-empty merges to compare!
      ; We use array_equal() to compare lists of ObsIDs, assuming they are sorted the same way. 
      if array_equal( strtrim(reference_obs_data.obs_dir,2), strtrim(test_obs_data.obs_dir,2) ) then begin
        file_delete, reverse(file_search(test_sourcedir,'*',/MATCH_INITIAL_DOT))
        file_delete,                     test_sourcedir
        print, test_sourcedir, reference_sourcedir, F='(%"  Pruned %s (duplicate of %s).")'
      endif
    endfor; kk
  endfor; jj
endfor; ii


CLEANUP:
if (exit_code EQ 0) then return $
else begin
  print, 'ae_extract_tiles: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end; ae_extract_tiles




;==========================================================================
;;; Find source candidates in AE neighborhood image reconstructions.
;;; DE-DUPLICATION IS PERFORMED ELSEWHERE!

;;; This tool assumes a very specific directory tree, and must be run from a specific place in that tree,
;;; i.e. the directory containing AE's obsXXXX/ directories and containing the "tiles" directory.

;;; Example:
;;;   .r ae
;;;   ae_recon_detect, 'all_tiles.srclist', '../tangentplane_reference.evt', MERGE_NAME='*', MASK_REGION_FILENAME='',   SCENE_TEMPLATE_FN="../fullfield_template.img", BAND_NAME='full', OUTPUT_DIR='energy_0.5-2'
;;;
;;; BAND_NAME must be an energy band for which files totalbackground.AI.<band>.density are available.
;;; 
;;; The optional hdunumber parameter is a one-based HDU number (CIAO convention) that
;;; selects which image in neighborhood.img is searched.

;;;  INVALID_THRESHOLD for Pb can be supplied to override the default of 0.02%.
;;;
;;;  MASK_REGION_FILENAME (optional) is the name of a region file (in SEXAGESIMAL COORDIATES) defines the region on the sky that is searched for sources.  To exclude a region (e.g. a SNR), this file should use the syntax:
;;;    field()
;;;    -polygon(.......)


;==========================================================================
PRO ae_recon_detect, catalog_or_srclist, tangentplane_reference_fn, MERGE_NAME=merge_name, HDUNUMBER=hdunumber,$
      MASK_REGION_FILENAME=mask_region_filename, $
      FLOOR_CNTS=floor_cnts, INVALID_THRESHOLD=invalid_threshold_p, HIGHCOUNT_MODE=highcount_mode, $
      OUTPUT_DIR=output_dir, BAND_NAME=band_name, ENERGY_RANGE=energy_range, SCENE_TEMPLATE_FN=scene_template_fn,$
      DEBUG=debug

creator_string = "ae_recon_detect, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

FORWARD_FUNCTION get_astrometry_from_eventlist, build_generic_cat

; By default, we use a Pb threshold more liberal (larger than) that to be used for pruning the catalog (typically 0.01).
invalid_threshold = keyword_set(invalid_threshold_p) ? invalid_threshold_p : 0.02   ; 2%

if (n_elements(floor_cnts       ) EQ 0) then floor_cnts        = 2.5
if (n_elements(hdunumber        ) EQ 0) then hdunumber         = 2

; Define nominal (5x5) and low-count (7x7) photometry apertures in units of image pixels.
; Note that image pixel size varies with theta.
nominal_cell_size  = 5
lowcount_cell_size = 7

; We search in "high-count mode" as long as the central pixel can, by itself, pass the "floor_cnts" requirement. 
low_count_threshold = floor_cnts



if ~keyword_set(merge_name) then merge_name = './'
if ~keyword_set(band_name ) then band_name  = 'unknown_band'

if keyword_set(output_dir) then begin
   file_mkdir, output_dir
endif else     output_dir = '.'

debug = keyword_set(debug)

print, band_name,$
       keyword_set(energy_range) ? string(energy_range,F='(%"%0.1f:%0.1f keV")') : 'energy range not specified',$
       output_dir,$
       floor_cnts, invalid_threshold, keyword_set(highcount_mode) ? 'T' : 'F', nominal_cell_size,$
       lowcount_cell_size, low_count_threshold,$
       F='(%"\nae_recon_detect: Searching for sources in the %s band (%s); saving to %s/.\nFLOOR_CNTS=%0.1f, INVALID_THRESHOLD=%0.2f, HIGHCOUNT_MODE=%s, nominal_cell_size=%d, lowcount_cell_size=%d, low_count_threshold=%0.1f")'


;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

temp_events_fn       = tempdir + 'temp.evt'
temp_region_fn       = tempdir + 'temp.reg'
temp_image_fn        = tempdir + 'temp.img'





;; =====================================================================
;; Initialize stuff.
run_command, PARAM_DIR=tempdir

;; Get rid of pre-existing configuration for the CIAO commands we'll use below.
run_command, /QUIET, ['punlearn dmcopy', 'pset dmcopy clobber=yes']

env_bmap_basename        = 'totalbackground.density' ; merge of single-ObsID "totalbackground" maps for this band
env_image_basename       = 'neighborhood.img'        ; image of merged events in this band
psf_basename             = 'source.psf'
 
arcsec_per_skypixel = 0.492 ; (0.492 arcsec/skypix)

resolve_routine, 'match_xy', /COMPILE_FULL_FILE

;; Read event file header to define ACIS sky coordinate system.
event2wcs_astr = get_astrometry_from_eventlist(tangentplane_reference_fn)


if keyword_set(scene_template_fn) then begin
  scene = readfits(scene_template_fn, scene_hdr)
  bkg_ratio_map = make_array(/FLOAT, DIMENSION=size(scene, /DIM), VALUE=!VALUES.f_nan)
  bkg_ratio_map[*]=0
endif

readcol, catalog_or_srclist, tilename, FORMAT='A', COMMENT=';'

; Trim whitespace and remove blank lines.
tilename = strtrim(tilename,2)
ind = where(tilename NE '', num_tiles)

if (num_tiles EQ 0) then begin
  print, 'ERROR: no entries read from tile list ', catalog_or_srclist
  retall
endif

tilename = tilename[ind]
print, num_tiles, F='(%"\n%d tiles found in catalog.\n")'

tile_bkg_ratio_a       = fltarr(num_tiles)
src_area_a             = fltarr(num_tiles)
nominal_cell_area_a    = fltarr(num_tiles)
theta_a                = fltarr(num_tiles)

get_lun, region1_unit
get_lun, region2_unit
openw,  region1_unit, output_dir+'/recon_fov.reg'
printf, region1_unit, "# Region file format: DS9 version 3.0"
printf, region1_unit, 'global width=1 font="helvetica 12 normal"'

peaks_cat = []

for ii = 0L, num_tiles-1 do begin
  sourcedir = tilename[ii] + '/' 

  src_stats      = headfits(/SILENT, sourcedir+'source.stats')

  ; Find the set of named merges available.
  merge_dir_list = file_dirname(file_search(sourcedir, merge_name+'/'+env_image_basename, COUNT=num_merges), /MARK_DIRECTORY)

  if (num_merges EQ 0) then begin
    print, tilename[ii], F='(%"\n-------\nWARNING!  Could not find any merged image for %s")'

    printf, region1_unit, psb_xpar( src_stats,'RA'), psb_xpar( src_stats,'DEC'), strtrim(psb_xpar( src_stats,'LABEL'),2), F="(%'J2000;boxcircle point %10.6f %10.6f # tag={tile with no merges} text={%s} color={red}')"

    continue
  endif

  
  ; Process each merge to build a catalog for this tile.
  tile_cat = []
  
  for kk=0, num_merges-1 do begin
    merge_dir   = merge_dir_list[kk] 
    merge_stats = headfits(/SILENT, merge_dir+'source.stats')
    
    TILE_LABEL  = strtrim(psb_xpar( merge_stats,'LABEL'),2)
    TILE_NAME   = strmid(merge_dir,0,strlen(merge_dir)-1)
    IMAGE_NAME  = TILE_NAME +'; '+ band_name + ' band'
    
    print, IMAGE_NAME, TILE_LABEL, F='(%"\n----------------------------\nTile: %s (%s)")'

    ;; ------------------------------------------------------------------------
    ;; Sum the single-ObsID "totalbackground" maps for this energy band.
    ;; Those maps are in files ../obsXXXX/totalbackground.AI.<band>.density
    ;; Those maps have units of ct /skypix^2
    merged_list_fn     = merge_dir + 'ObsIDs_merged.fits'    
    obs_data           = mrdfits(merged_list_fn,1, /SILENT)
    
    totalbackground_fn = 'obs'+strtrim(obs_data.OBSNAME,2) + '/totalbackground.'+band_name+'.density'
    
    num_obs = n_elements(obs_data)
      
    ;; Reproject the total background map cutouts onto the grid of neighborhood.img.

    ; Note that these background maps are 2-D functions representing a calibrated density (ct /skypix^2), $
    ; not a 2-D HISTOGRAM that is counting something with "per pixel" units. Thus, when rebinning we choose method=average so that the scale of the map is unchanged.
    reproj_obs_bmap_fn  = tempdir + string(indgen(num_obs), F='(%"reproj%d.bmap")')
    temp_text_fn        = tempdir + 'temp.txt'
    resolution = 1 
    for hh=0, num_obs-1 do $
      run_command, string(totalbackground_fn[hh],$
                                    obs_data[hh].N_FILTER, merge_dir + env_image_basename,$
                          reproj_obs_bmap_fn[hh], resolution, $
                         F="(%'reproject_image  infile=""%s[%s]""  matchfile=%s  outfile=%s  method=average resolution=%d clob+')")

    ;; Sum the reprojected exposure maps.
    ;; The final map has units of ct /skypix^2
    terms   = string(1+indgen(num_obs),F="(%'img%d')")
    formula = strjoin(terms, '+')
  
    forprint, reproj_obs_bmap_fn, F='(%"%s")', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
    
    run_command, string(temp_text_fn, merge_dir + env_bmap_basename, formula, $
                        F="(%'dmimgcalc infile=""@-%s"" infile2=none outfile=%s operation=""imgout=((float)(%s))"" clob+')")          

    ;; ------------------------------------------------------------------------
    ;; Apply any spatial filter supplied in MASK_REGION_FILENAME.
    composite_psf_fn  = merge_dir + psf_basename
    maxlik_img_fn     = tempdir   + 'maxlik.img'

    if keyword_set(mask_region_filename) then begin
      ; Filter the background map, data image, and reconstructed image; store results in tmp directory.
      composite_bmap_fn = tempdir + env_bmap_basename
      composite_img_fn  = tempdir + env_image_basename

      run_command, /QUIET, string(merge_dir + env_bmap_basename ,            mask_region_filename, composite_bmap_fn,$
                                  F="(%'dmcopy     ""%s[sky=region(%s)][opt full]"" %s')")                 
      run_command, /QUIET, string(merge_dir + env_image_basename,            mask_region_filename, composite_img_fn,$
                                  F="(%'dmcopy  ""%s[1][sky=region(%s)][opt full,null=NaN]"" %s')")      
      run_command, /QUIET, string(merge_dir + env_image_basename, hdunumber, mask_region_filename, maxlik_img_fn,$
                                  F="(%'dmcopy ""%s[%d][pos=region(%s)][opt full,null=NaN]"" %s')")      
    endif else begin
      ; If no MASK_REGION_FILENAME, then just point to existing background map and data image.
      ; To simplify later code, pull the reconstructed image out as a primary HDU; store result in tmp directory.
      composite_bmap_fn = merge_dir + env_bmap_basename
      composite_img_fn  = merge_dir + env_image_basename

      run_command, /QUIET, string(merge_dir + env_image_basename, hdunumber,                       maxlik_img_fn,$
                                  F="(%'dmcopy ""%s[%d]"" %s')")      
    endelse

    composite_bmap    = readfits(/SILENT, composite_bmap_fn, bmap_hdr  ) ; ct /skypix^2
    composite_img     = readfits(/SILENT,  composite_img_fn, hood_hdr  ) ; ct /MLpix^2
    maxlik_img        = readfits(/SILENT,     maxlik_img_fn, maxlik_hdr) ; ct /MLpix^2
    composite_psf_hdr = headfits(/SILENT,  composite_psf_fn            )
      
    
  ; print, psb_xpar( maxlik_hdr, 'ML_ITERS'), ' iterations used in reconstruction'
    
    extast, maxlik_hdr, maxlik_astr
    arcsec_per_MLpixel = maxlik_astr.CDELT[1] * 3600
    xdim = (size(maxlik_img, /DIM))[0]
    ydim = (size(maxlik_img, /DIM))[1]

    ;; Create a graphic depicting the field of view of this reconstruction.
    if (kk EQ 0) then printf, region1_unit, psb_xpar( merge_stats,'RA'), psb_xpar( merge_stats,'DEC'), [xdim,ydim]*arcsec_per_MLpixel, TILE_LABEL, F="(%'J2000;box %10.6f %10.6f %5.1f"" %5.1f"" # tag={recon fov} text={%s}')"
    
    ;; ------------------------------------------------------------------------
    ;; Verify energy range used in composite image.
    if keyword_set(energy_range) then begin
      run_command, string(composite_img_fn, F="(%'dmlist ""%s[1]"" subspace | grep energy')"), result, /QUIET
    
      result = stregex(result[0],'Real4[[:space:]]*([0-9]+\.[0-9]+):[[:space:]]*([0-9]+\.[0-9]+)',/SUB,/EXT)
      
      if (result[1] NE energy_range[0]*1000) || (result[2] NE energy_range[1]*1000) then begin
        print, composite_img_fn, result[1:2]/1000., F='(%"ERROR: Energy range of %s (%0.1f:%0.1f keV) does not match ENERGY_RANGE supplied by caller!")'
        stop
      endif
    endif ;ENERGY_RANGE supplied
    
    ;; ------------------------------------------------------------------------
    ;; DISCUSSION
    ;;
    ;; The primary challenge of source detection is estimating the background near a putative source.
    ;; Background estimation is particularly challenging in this contexct because within a tile there
    ;; can be very large variations in exposure time, due to bad columns, partial overlap of ObsIDs, etc.
    ;; Methods that consider only the image of observed events are doomed to failure, because we 
    ;; generally have far too few events to characterize these exposure variations.
    ;; Thus, using a MODEL of exposure variations is extremely helpful.
    ;;
    ;; Constructing such a model is complicated by the fact that ACIS observations contain two types of
    ;; background:  events produced by particles interacting with the detectors, and "background" X-rays
    ;; that pass through the telescope.
    ;; The spatial distributions of those two background components have both similarities and differences.
    ;; The dithering of the CCDs on the sky has the same effect on particles and X-rays---each point on the sky
    ;; has a specific exposure TIME (cumulative number of seconds that part of the sky was able to produce events).
    ;; Thus, the small-scale variations in the density of particles and X-rays are identical.
    ;; However, at large scales particle and X-ray background events have different distributions, because
    ;; X-rays are focussed through the telescope and Optical Blocking Filter, and particles are not.
    ;;
    ;; The standard CIAO "exposure map" describes the spatial distribution of X-rays (at some energy).
    ;; It is not a good model for the spatial distribution of particle events.
    ;; The L1_2_L2_emaps tool attempts to infer the relative strengths of particle and X-ray backgrounds, 
    ;; and to build a map of the total background density (ct /skypix^2).
    ;; This is the spatial model we will use below to "take out" known spatial variations in total background
    ;; before we try to estimate "local" backgrounds.
    ;;
    ;; Note that we do not simply use the total background density (ct /skypix^2) map directly to 
    ;; calculate source significances, because that map is calibrated using particle and X-ray background
    ;; estimates AVERAGED OVER EACH CCD.  
    ;; In this context, we need to worry about variations of the X-ray background between tiles.
    ;;
    ;; In the code below, we're dividing a count image by a total background density, which produces
    ;; a map that represents the ratio between the observed and expected number of events.
    ;; Within this "exposure corrected" image, we can estimate a typical ratio value for the entire tile.
    ;; That typical value can then be multiplied by the density map to produce a (smooth, high-frequency) model
    ;; for the local background level.
    
    
    ;; ------------------------------------------------------------------------
    ;; Use an "exposure corrected" neighborhood image to estimate a tile-averaged "background ratio",
    ;; i.e. the ratio between the observed background in the tile to the background predicted by
    ;; our model (composite_bmap).
    ;;
    ;; This estimate makes a crude attempt to ignore stars, by coarsely rebinning the image to 
    ;; push most of the PSF into ~1 pixel, and then using a outlier rejection algorithm.
    ;; We ensure that the rebinned image retains a reasonable number (100) of pixels.
    radius50 = psb_xpar( composite_psf_hdr, 'RADIUS50')
    if NOT finite(radius50) then begin
      print, 'Keyword RADIUS50 not found; using on-axis value of 0.43 arcsec.'
      radius50 = 0.43
    endif
  
    num_across          = radius50 / arcsec_per_MLpixel
    
    
    ; Use a REPEAT instead of FOR loop to ensure consisent {MLpixel_per_bigpixel,proposed_xdim,proposed_ydim}
    ; for every loop termination possible.
    ; Rebin factor (MLpixel_per_bigpixel) is an INTEGER!
    
    MLpixel_per_bigpixel= 1 +     (ceil(num_across * 4.0)) ; Initialize to 1 plus desired first value.
    repeat begin
      MLpixel_per_bigpixel--    ; = ceil(num_across * 4.0) on first iteration.
      MLpixel_per_bigpixel >= 1 ; A termination criterion is "no rebinning". 
      proposed_xdim = floor((size(composite_img, /DIM))[0] / MLpixel_per_bigpixel)
      proposed_ydim = floor((size(composite_img, /DIM))[1] / MLpixel_per_bigpixel) 
    endrep until (MLpixel_per_bigpixel EQ 1) || (proposed_xdim*proposed_ydim GE 100)
    ; Initial rebin factor, ceil(num_across * 4.), is preferrred if it produces >= 100 pixels.
    ; Otherwise, try smaller integer rebin factors, down to a rebin factor of 1.
    
    skypix_per_bigpix = (MLpixel_per_bigpixel * arcsec_per_MLpixel / arcsec_per_skypixel)
    
    print, MLpixel_per_bigpixel, MLpixel_per_bigpixel*arcsec_per_MLpixel, F='(%"\nEstimating background in composite data image rebinned by %d (to %4.1f arcsec/pixel).")'
    
    ; Rebin the data image and the background model.
    ; We avoid "partial pixels" in the rebinned images by cropping the data image and the background model as needed.
    
    ; The units of the data image           (composite_img) are                                         ct /MLpix^2.
    ; The units of the rebinned data image (bigpixel_image) are 1/MLpixel_per_bigpixel^2 ct /MLpix^2 == ct /bigpix^2 
    bin_spec = string(proposed_xdim*MLpixel_per_bigpixel, MLpixel_per_bigpixel, $
                      proposed_ydim*MLpixel_per_bigpixel, MLpixel_per_bigpixel, F="(%'#1=1:%d:%d,#2=1:%d:%d')")

    cmd = string(composite_img_fn , bin_spec, temp_image_fn, F="(%'dmcopy ""%s[bin %s]"" %s')")
    run_command, /QUIET, cmd      
    bigpixel_image = readfits(temp_image_fn, bigpixel_hdr, /SILENT) ; ct /bigpix^2
 
    ; The units of the background model (composite_bmap) are                          ct /skypix^2.
    ; The units of the rebinned model    (bigpixel_bmap) are 1/MLpixel_per_bigpixel^2 ct /skypix^2.
    cmd = string(composite_bmap_fn, bin_spec, temp_image_fn, F="(%'dmcopy ""%s[bin %s]"" %s')")
    run_command, /QUIET, cmd      
    bigpixel_bmap  = readfits(temp_image_fn, bigpixel_hdr, /SILENT)
    extast, bigpixel_hdr, bigpixel_astr
 
    ; We WANT bigpixel_bmap to be in the same units as bigpixel_image so the two can be divided below.
    ; That requires a conversion from 1/skypix^2 to 1/MLpix^2 units.
    bigpixel_bmap *=  ( arcsec_per_MLpixel / arcsec_per_skypixel )^2
    
    ; Sometimes we get an edge where bigpixel_image has power but bigpixel_bmap is zero.
    bigpixel_image *= (bigpixel_bmap GT 0)

    ; Estimate a tile-wide (scalar) ratio between the observed background (bigpixel_image) and the 
    ; predicted background (bigpixel_bmap).
    estimate_poisson_background, bigpixel_image, EMAP=bigpixel_bmap, tile_bkg_ratio, SIGNIFICANCE=0.99, /VERBOSE

    ; Calculate a map of actual ratio between observed events and predicted background events..
    bigpixel_bkg_ratio = (bigpixel_image / bigpixel_bmap) ; unitless image

    if debug then writefits, merge_dir + 'bigpixel_bkg_ratio.img', bigpixel_bkg_ratio, bigpixel_hdr    
    
    print,  F='(%"\nRatio of observed counts to model background across the tile:")'
    print,            tile_bkg_ratio , F='(%"  robust average  = %0.3f")'
    print, median(bigpixel_bkg_ratio), F='(%"  median          = %0.3f")'
  
  ;help, composite_img, composite_bmap, bigpixel_image, bigpixel_bmap
  ;help, total(composite_img), total(bigpixel_image), tile_bkg_ratio
  ;print, mean(composite_bmap), mean(bigpixel_bmap)
  ;stop
  
    if keyword_set(scene_template_fn) then begin
      ; Resample the background tile onto the scene.
      hastrom, replicate(1, xdim, ydim), maxlik_hdr, bkg_tile, dum_hdr, scene_hdr, MISSING=0
      ind = where(bkg_tile GT 0, count)
      if (count GT 0) then bkg_ratio_map[ind] = tile_bkg_ratio
    endif
   
   
    ;; ------------------------------------------------------------------------
    ;; Estimate the size of AE's nominal extraction polygon in this tile.
    THETA     = psb_xpar( merge_stats,'THETA')
    src_area  = psb_xpar( merge_stats, 'SRC_AREA') ; skypix^2
    if (src_area EQ 0) then src_area  = !PI * psb_xpar( merge_stats, 'SRC_RAD')^2 
    if (src_area EQ 0) then begin
      print, 'ERROR:  SRC_AREA and SRC_RAD are zero.'
      retall
    endif
    print, src_area, F='(%"\nAssuming AE aperture area is %d skypixel**2.")'
    
     nominal_cell_area = ( nominal_cell_size*arcsec_per_MLpixel/arcsec_per_skypixel)^2  ; skypix^2
    lowcount_cell_area = (lowcount_cell_size*arcsec_per_MLpixel/arcsec_per_skypixel)^2  ; skypix^2

    theta_a               [ii] = THETA
    src_area_a            [ii] = src_area
    nominal_cell_area_a   [ii] = nominal_cell_area
    tile_bkg_ratio_a      [ii] = tile_bkg_ratio
    
  ;  print, 'NOMINAL_CELL_AREA  = ', nominal_cell_area, '        (skypixel**2)' 
  
  ;  save, /COMPRESS, tile_bkg_ratio, src_area, nominal_cell_area,FILE=merge_dir+'recon_detect.sav'
    
  
    ;; ------------------------------------------------------------------------
    ;; Search for sources.  Because we employ a mechanism to impose a minimum
    ;; separation between sources, we process pixels in maxlik_img, testing for local maxima,
    ;; in the order of decreasing pixel value.
    ;; In other words, we try to identify the brightest sources first, and progress to weaker sources.
    ;; This sorting is purposefully done on only individual pixels, rather than sliding cells, because
    ;; the latter approach can get in trouble in unusual circumstances.
    ;; For example, a very bright pixel in maxlik_img might not participate in a centroid
    ;; because it fell in the outer portions of a high-count CELL that was mistakenly considered too early.
    Nentries = 0
    ;; Build a catalog structure suitable for match_xy.
    emap_tot = psb_xpar( merge_stats,'EMAP_TOT') ; s cm**2 count /photon
    if ~finite(emap_tot) || ~(emap_tot GT 0) then $
      message, 'ERROR: EMAP_TOT is not positive.'

    cat_entry = { source_recon,$
                  ID:0L,$
                  X    :0.0,$
                  Y    :0.0,$
                  X_ERR:0.0,$
                  Y_ERR:0.0,$
                  RA   :0D,$
                  DEC  :0D,$
                  POSNTYPE :'recon_detect',$
                  PROVENAN :'',$
                  LABEL    :'',$
                  
                  TILE_NAME          :TILE_NAME,$
                  TILE_LABEL         :TILE_LABEL,$
                  BAND_NAME          :band_name,$
                  IMAGE_NAME         :IMAGE_NAME,$
                  IMAGE_EXPOSUREAREA :emap_tot,$ ; s cm**2 count /photon
                  CATALOG_NAME       :'',$
                  
                  MODE  :'',$
                  X_CELL:0,$
                  Y_CELL:0,$
                  CELL_COUNTS:0.0,$
                  SRC_CNTS   :0L,$
                  BKG_COUNTS_EXPECTED_IN_APERTURE :0.0,$
                  BKG_COUNTS_EXPECTED_OUTSIDE_CELL:0.0,$
                  NET_CNTS :0.0,$
                  Pb       :0.0,$
                  THRESHOLD:0.0,$
                  THETA    :0.0,$
                  NOT_FIDUCIAL:0B }
    
    cat = replicate(cat_entry, 20000)
    
    ; Defensively, verify that maxlik_img and composite_bmap have the same dimensions.
    if ~ARRAY_EQUAL(size(maxlik_img, /DIMEN), size(composite_bmap, /DIMEN)) then $
      message, 'ERROR: image and bmap must have the same dimensions'
  
    
    accepted_island_map = bytarr(xdim,ydim)
    
    ; We'll use the kernel_footprint utility to look up 1-D indexes of "photometry cells" in maxlik_img, taking array edges into account.
    if keyword_set(nominal_kf_in_maxlik_img) && ptr_valid(nominal_kf_in_maxlik_img) then $
    kernel_footprint, nominal_kf_in_maxlik_img, /DESTROY
    kernel_footprint, nominal_kf_in_maxlik_img, /CREATE, IMAGE=maxlik_img,$
                                                       KERNELS=replicate(1,nominal_cell_size,nominal_cell_size)
    
    if keyword_set(lowcount_kf_in_maxlik_img) && ptr_valid(lowcount_kf_in_maxlik_img) then $
    kernel_footprint, lowcount_kf_in_maxlik_img, /DESTROY
    kernel_footprint, lowcount_kf_in_maxlik_img, /CREATE, IMAGE=maxlik_img,$
                                                       KERNELS=replicate(1,lowcount_cell_size,lowcount_cell_size)
    
  
    ; We'll use the kernel_footprint utility to look up 1-D indexes of 5x5 "neighborhoods" in bigpixel_bkg_ratio, taking array edges into account.
    ; The size of this neighborhood defines the "local" in "local background", and has nothing to do with the size of the photometry cell ("nominal_cell_size" or "lowcount_cell_size").
    if keyword_set(kf_in_bigpixel_ratio) && ptr_valid(kf_in_bigpixel_ratio) then $
    kernel_footprint, kf_in_bigpixel_ratio, /DESTROY
    kernel_footprint, kf_in_bigpixel_ratio, /CREATE, IMAGE=bigpixel_bkg_ratio, KERNELS=replicate(1,5,5)
  

    ;; ------------------------------------------------------------------------ 
    ;; Even though our photometry cell ("nominal_cell_size" or "lowcount_cell_size") may be larger than 3x3, our 
    ;; definition of a local maximum always involves a 3x3 island in the recon 
    ;; image, NOT in any sort of smoothed photometry image.
    ll_neighbor_ind = [0,1,2,3]
    ur_neighbor_ind = [5,6,7,8]
    central_ind     = 4
    
    
  ;; ------------------------------------------------------------------------
  ;; Search for local maxima and decide if they are detections.
  ;; We do this first on the recon image as provided, looking for bright sources, which should have been made more compact by the Richardson-Lucy algorithm.
  ;; Then we smooth the remaining light in the recon image, and search that smoothed image for weak sources (which the Richardson-Lucy algorithm does not alter very much)
  
  localmax_img = maxlik_img
  
  foreach mode, keyword_set(highcount_mode) ? ['high-count'] : ['high-count','low-count'] do begin
                       
    ; We process pixels in localmax_img sorted by brightness (brightest first). 
    ; Ignore infinite pixels, and very small pixels (which cannot be useful local maxima).
    good_ind = where(finite(localmax_img) AND (localmax_img GT (floor_cnts/(lowcount_cell_size^2))), num_good)

    if (num_good EQ 0) then begin
      print, 'Skipping null recon image.'
      break
    endif

    sort_ind = good_ind[reverse(sort(localmax_img[good_ind]))]
    
    
    for jj = 0L, n_elements(sort_ind)-1 do begin
      ;Find the X,Y position of the cell, in the 0-based image coordinate system.
      temp = array_indices(localmax_img, sort_ind[jj])
      xindex = temp[0]
      yindex = temp[1]

      ; We will NOT test for a local maximum if the detection island falls off the edge of the image.
      if ((xindex LT 1) OR (xindex GT xdim-1-1) OR $
          (yindex LT 1) OR (yindex GT ydim-1-1)) then continue
    
      ; Extract 3x3 detection island and check for a local maximum
      localmax_island = localmax_img[xindex-1:xindex+1, yindex-1:yindex+1]

      if ~array_equal(finite(localmax_island), 1) then continue
  
      local_max_found = (localmax_island[central_ind] GE max(localmax_island[ll_neighbor_ind])) AND $
                        (localmax_island[central_ind] GT max(localmax_island[ur_neighbor_ind]))
  
      if (~local_max_found) then continue
    
      ; Do not accept a 3x3 island that overlaps a previously accepted island.
      if ~array_equal(accepted_island_map[xindex-1:xindex+1, yindex-1:yindex+1], 0B) then begin
        print, 'Rejected a local max too close to an accepted local max.'
        continue
      endif
      
      ; Change to "low-count mode" when we reach localmax_img pixels with low_count_threshold counts.
      ; This threshold is arbitrary; we're trying to guess when local maxima become poor estimators of source position.
      if (mode EQ 'high-count') && (localmax_island[central_ind] LE low_count_threshold) then break
  
      ;; ------------------------------------------------------------------------
      ;; Estimate the ratio between observed and model background at the peak's location.
      ;; We have two estimates available.
      
      ; The first is "tile_bkg_ratio", which assumes that the ratio is constant across the tile.
      ; This estimate should have little Poisson error, since a lot of data were used in the estimate.
      ; This should work well in tiles where the background is flat.

      ; The second ratio estimate we can make is a local one, using pixels in the image "bigpixel_bkg_ratio" near the position of the local maximum (xindex/yindex in the recon image).
      ; As of May 2017, we use the same algorithm (estimate_poisson_background) on these few local pixels as we did on the whole tile.
      ; This local estimate will have more Poisson noise, but in a region with bright diffuse emission (pulsar wind, SNR, etc.) this local estimate can eliminate thousands of spurious detections!
      xy2ad,  xindex,   yindex,   maxlik_astr,    ra_temp,   dec_temp
      ad2xy, ra_temp, dec_temp, bigpixel_astr, big_xindex, big_yindex
      big_xindex = round(big_xindex)
      big_yindex = round(big_yindex) ; Pixel coordinates given to kernel_footprint must be INTEGERS!

      kernel_footprint, kf_in_bigpixel_ratio, big_xindex, big_yindex, 0, bigpixel_ind

    ; Nearby pixels from the observed background (bigpixel_image) and predicted background (bigpixel_bmap) are selected by the index vector bigpixel_ind
    ; We're using the median statistic to estimate the 'background' level in those nearby pixels.
    ; Sigma-clipping algorithms (e.g. estimate_poisson_background) do not work when the background is sloped, because there is no large fraction of the pixels that are consistent with a single value.
    
    local_tile_bkg_ratio = median(bigpixel_bkg_ratio[bigpixel_ind], /EVEN)


     ;print, xindex, yindex, big_xindex, big_yindex, minmax(bigpixel_bkg_ratio[bigpixel_ind])
     if debug then begin
       forprint, indgen(n_elements(bigpixel_bkg_ratio)), bigpixel_bkg_ratio, bigpixel_image, bigpixel_bmap / (MLpixel_per_bigpixel * arcsec_per_MLpixel / arcsec_per_skypixel )^2, replicate(skypix_per_bigpix,n_elements(bigpixel_bkg_ratio)), SUBSET=bigpixel_ind[ sort(bigpixel_bkg_ratio[bigpixel_ind]) ], F='(%"  bigpixel_bkg_ratio[%3d] =  %6.1f = %4d (ct/bigpix^2)  /   %5.3f (ct/skypix^2)  /  (%4.1f skypix/bigpix)^2 ")'
;       stop
     endif
     help, local_tile_bkg_ratio


      ; If we use only the tile-wide background estimate, then strong variations in background flux across the tile will lead to spurious detections (where the background is under-estimated) and/or to reduced sensitivity (where the background is over-estimated).
      ; Solving BOTH of those problems would require ignoring the tile-wide background estimate, and using only local (more noisy) estimates.
      ; I hesitate to make that large change in our detection strategy, so I will use the local background estimate to address only the spurious detection problem, by adopting the **larger** of the two background estimates.
      
      ; Estimate the local background density (ct /skypix^2) by multiplying:
      ;   - a background ratio (unitless)
      ;   - our model of the total background (ct /skypix^2) at the location of the peak (xindex, yindex)
      bkg_counts_per_skypix = (tile_bkg_ratio > local_tile_bkg_ratio) * composite_bmap[xindex, yindex]

      
      ;; ------------------------------------------------------------------------  
      ;; Perform photometry on a cell in the maxlik image at this location.
      
      ; We use the kernel_footprint utility to return the 1-D indexes of the cell, 
      ; with the central pixel listed first, dealing with array edges.
      if (mode EQ 'high-count') then begin
        ; This branch is the "nominal" case, where the source produced enough counts for Richardson-Lucy to gather its light into the footprint of our nominal photometry aperture.  The criterion above is arbitrary.
        kernel_footprint, nominal_kf_in_maxlik_img, xindex, yindex, 0, pixel_ind
        
        maxlik_pixels = maxlik_img[pixel_ind]
          
        ; For photometry we add up the central pixel (first in list), plus others in cell that:
        ; (a) Are less than central pixel.
        ; (b) Are not in accepted_island_map.
        good_flag     = (maxlik_pixels LT 1*maxlik_pixels[0]) AND (accepted_island_map[pixel_ind] EQ 0)
        good_flag[0]  = 1

        this_cell_area = nominal_cell_area
      endif else begin
        ; Low-count sources on low backgrounds produce neighborhood images that are ones and zeros.
        ; Richardson-Lucy is generally unable to concentrate light from such a sparse image into the footprint of our nominal photometry aperture.
        ; In order to detect such low-count sources, we arbitrarily enlarge our photometry aperture.
        ; We also change the requirements on which pixels in that aperture may participate in the photometry, since in the photometry image the "central pixel" may be zero!
        kernel_footprint, lowcount_kf_in_maxlik_img, xindex, yindex, 0, pixel_ind
        
        maxlik_pixels = maxlik_img[pixel_ind]

        ; For photometry we add up the central pixel (first in list), plus others in cell that:
        ; (b) Are not in accepted_island_map.
        good_flag     = (accepted_island_map[pixel_ind] EQ 0)
        good_flag[0]  = 1

        this_cell_area = lowcount_cell_area
      endelse

      if (maxlik_pixels[0] NE maxlik_img[xindex, yindex]) then message, 'BUG in kernel_footprint utility!'

      if ~array_equal(finite(maxlik_pixels), 1) then continue

      recon_photometry = total(/NAN, maxlik_pixels * good_flag)
      
;if keyword_set(stop_flag) then stop

      ; We impose an arbitrary lower limit on our photometry estimate to avoid very weak sources that would
      ; probably be rejected in our recipe later anyway.
      if (recon_photometry LT floor_cnts) then begin
        if debug then print, xindex,yindex, localmax_img[xindex,yindex], recon_photometry, F='(%"centralpix(%3d,%3d)=%8.2g  phot=%6.1f < FLOOR_CNTS")'
        continue
      endif

      ;; ------------------------------------------------------------------------  
      ;; Estimate an integer valued SRC_CNTS (total counts in aperture) value, required by poisson_distribution(), that AE would find for a source at this location.
      
      ;; Presumably, reconstructions will push SOME of the background falling in the aperture into the source peak
      ;; that we're processing, and push SOME of the background in to random little peaks outside our cell.
      ;; Thus we expect that the recon photometry will fall between SRC_CNTS and NET_CNTS, but we don't know
      ;; in detail its relationship to SRC_CNTS
             
      ; A simple assumption is that the recon photometry above is a good estimate of SRC_CNTS.
      SRC_CNTS = round(recon_photometry)
      if (SRC_CNTS LT 0) then message, 'BUG!'
      
      
      ;; ------------------------------------------------------------------------  
      ; To compute Pb, we need a corresponding estimate for the background counts expected inside a typical AE aperture.
      ; We use our estimate for the area of such an AE aperture, "src_area", in units of skypix^2.
      bkg_counts_expected_in_aperture  = 0 > (bkg_counts_per_skypix *  src_area)
      bkg_counts_expected_outside_cell = 0 > (bkg_counts_per_skypix * (src_area - this_cell_area))


      ;; ------------------------------------------------------------------------  
      ;; Estimate AE's Pb statistic and compare to a threshold.

      ; To get good agreement with detection by eye and with wavdetect, we find it necessary to put
      ; in an explicit ad hoc adjustment that boosts SRC_CNTS off-axis.
      ; On-axis we add nothing; at 10' off-axis we add 50% of bkg_counts_expected_outside_cell.
      ;SRC_CNTS = round(counts_in_cell[xindex,yindex] + (bkg_counts_expected_outside_cell * THETA*(0.5/10)))  
  
      ; This is a flat law, boosting by bkg_counts_expected_outside_cell everywhere.
      ;SRC_CNTS = round(counts_in_cell[xindex,yindex] + bkg_counts_expected_outside_cell) 

      PROB_NO_SOURCE_poisson  = (1 - poisson_distribution(bkg_counts_expected_in_aperture, SRC_CNTS - 1)) > 0
  
      if (PROB_NO_SOURCE_poisson GT invalid_threshold) then begin
        if debug then print, xindex,yindex, localmax_img[xindex,yindex], recon_photometry, PROB_NO_SOURCE_poisson, F='(%"centralpix(%3d,%3d)=%0.2g  phot=%6.1f  Pb=%0.2f > invalid_threshold")'
        continue
      endif
      
      ;; ------------------------------------------------------------------------  
      ; Estimate position as the centroid of some set of recon pixels.
      ; We choose to centroid on the 3x3 island, even though our photometry cell is larger 
      ; ("nominal_cell_size" or "lowcount_cell_size"), to avoid
      ; severe perturbation of the centroid when the outer ring has a very bright pixel associated
      ; with another source.
      ; This is admittedly confusing: we let the outer ring help the source be significant but do
      ; not let it influence the centroid.
      offset = [-1,0,1]
      
      make_2d, offset, offset, x_offset, y_offset
      centroid_island       = localmax_island
      
      cat_entry.X           = xindex + total(centroid_island*x_offset) / total(centroid_island)
      cat_entry.Y           = yindex + total(centroid_island*y_offset) / total(centroid_island)
      cat_entry.MODE        = mode
      cat_entry.X_CELL      = xindex
      cat_entry.Y_CELL      = yindex
      cat_entry.CELL_COUNTS = recon_photometry
      cat_entry.SRC_CNTS    = SRC_CNTS
      cat_entry.bkg_counts_expected_in_aperture  = bkg_counts_expected_in_aperture
      cat_entry.bkg_counts_expected_outside_cell = bkg_counts_expected_outside_cell
      cat_entry.NET_CNTS    = SRC_CNTS - bkg_counts_expected_in_aperture
      cat_entry.Pb          = PROB_NO_SOURCE_poisson
      cat_entry.THRESHOLD   = invalid_threshold
      cat_entry.THETA       = THETA
      
      cat[Nentries] = cat_entry
      Nentries = Nentries + 1
      
      ; Record which pixels we wish to prohibit from being used to define another local maximum.
      accepted_island_map[xindex-1:xindex+1, yindex-1:yindex+1] = 1

      if debug then print, xindex,yindex, localmax_img[xindex,yindex], recon_photometry, PROB_NO_SOURCE_poisson, tile_bkg_ratio, local_tile_bkg_ratio, F='(%"centralpix(%3d,%3d)=%0.2g  phot=%5.1f  Pb=%0.3f;  DETECTION; background ratio = %0.3f, %0.3f (tile, local) )")'
      
    endfor ;jj loop over pixels

    if (mode EQ 'high-count') then begin
      ; Mask pixels judged to be associated with the bright sources we have found so far.
      ind = where(accepted_island_map,count)
      if (count GT 0) then maxlik_img[ind] = 0

      if debug then writefits, merge_dir + 'neighborhood_residual.img', maxlik_img, maxlik_hdr
      
;      ; Convolve with the PSF---a "matched filter" detection strategy.
;      localmax_img = ae_convolve( maxlik_img, composite_psf)

      ; Smooth the data to create peaks that represent centroids of event groupings.
      ; Use an odd-sized smoothing kernel to lower risk of inducing shifts in the result.
      sm_kernel_sigma = 1.0 * radius50 / arcsec_per_MLpixel
      sm_kernel_size  = 1 + 2*ceil(sm_kernel_sigma)
      sm_kernel       = psf_gaussian(ST_DEV=sm_kernel_sigma, NDIMENSION=2, NPIXEL=sm_kernel_size, /NORMALIZE)
      localmax_img    = ae_convolve( maxlik_img, sm_kernel )
                                                           
      
      if debug then writefits, merge_dir + 'smoothed_neighborhood_residual.img', localmax_img, maxlik_hdr

      print, Nentries, F='(%"Masked 3x3 regions around %d detected bright sources; smoothed residual image; searching for weak sources. ")'
      
      ; Re-enter (jj) loop over pixels.
    endif

  endforeach ; mode
    
    
    
    if (Nentries EQ 0) then begin
      print, 'No sources detected.'
      continue
    endif
        
    cat     = cat[0:Nentries-1]
  
    print, Nentries, F='(%"\n%d sources detected.")'
  
    
   ; Convert from image to celestial coordinates.
    xy2ad, cat.X, cat.Y, maxlik_astr, ra, dec
    cat.RA = ra
    cat.DEC=dec
    print
    print, '(celestial)             (centroid)   (cell,cell)  mode   counts_in_cell bkg_counts_expected_in_aperture'
    forprint, ra, dec, cat.X, cat.Y, cat.x_cell, cat.y_cell, cat.mode, cat.cell_counts, cat.bkg_counts_expected_in_aperture, F='(%"%10.6f %10.6f  %6.1f %6.1f  %3d %3d  %10s %7.1f %7.1f")'

  
    ; Convert celestial coordinates to the ACIS sky coordinate system found in tangentplane_reference_fn.
    temp_cat = build_generic_cat(ra, dec, 0, 0, event2wcs_astr, /QUIET)
    
    cat.X     = temp_cat.X
    cat.Y     = temp_cat.Y
    cat.ID    = 1+indgen(Nentries)
    cat.LABEL = IMAGE_NAME + string(cat.ID, F='(%" #%d")')

    ; Create a region file in celestial coordinates.
    openw,  region2_unit, merge_dir+'recon.reg'
    printf, region2_unit, "# Region file format: DS9 version 3.0"
    printf, region2_unit, 'global width=2 font="helvetica 12 normal"'
    !TEXTUNIT = region2_unit
    colors = replicate('cyan',Nentries)
    colors[where(/NULL, strmatch(cat.MODE, 'high-count'))] = 'red'
    forprint, TEXTOUT=5, /NoCOM, RA, DEC, cat.MODE, colors, F='(%"J2000;cross   point %10.6f %10.6f # tag={recon} tag={%s} color={%s}")'
    close, region2_unit
    
  ;  save, /COMPRESS, cat, tile_bkg_ratio, src_area, nominal_cell_area,FILE=merge_dir+'recon_detect.sav'
  
    ;; Combine the catalog we've built so far for this tile (tile_cat, from merges already processed) with the catalog we derived from this reconstructed tile image (cat).
    tile_cat = [tile_cat, cat]
  endfor ;kk loop over the tile merges
  
  
  ;; Combine the field catalog we have built so far (peaks_cat) with the tile catalog we just built (tile_cat).  
  if keyword_set(tile_cat) then begin
    peaks_cat = [peaks_cat, tile_cat]
  endif ;tile_cat has entries
  
endfor ;ii loop over the tiles
free_lun, region1_unit, region2_unit

num_peaks = n_elements(peaks_cat)

; Estimate realistic position errors (based on off-axis angle) so that we can identify and remove duplicate detections.
; We will estimate the position uncertainty as follows:
;  - Esimate R50 (radius of circle enclosing 50% PSF power) from THETA.
;  - Assume that the PSF is Gaussian, and that R50 ~= sigma.
;  - Estimate how many counts the source produced via NET_CNTS.
;  - Compute the position error as the standard sigma/sqrt(N).
;  - Inflate those statisitcal position errors, which can become very small for bright sources, to model systematic effects.

; In Dec 2007 I used MARX simulations at 1.5 keV with the readout streak disabled to measure PSF fractions at 1.5 keV as a function of off-axis angle.  
; These polynomial curves were fit to those measurements. The off-axis angle off_angle is in arcminutes.
off_angle = peaks_cat.THETA
radius50  = (0.85 -0.25*off_angle + 0.10*off_angle^2)    ; skypix

peaks_cat.X_ERR = radius50 / sqrt(peaks_cat.NET_CNTS) ;skypix
peaks_cat.Y_ERR = peaks_cat.X_ERR

systematic_error  = 0.08                 ; arcsec
print, systematic_error, F='(%"A \"systematic\" uncertainty of %0.2f arcsec has been added in quadrature to the statistical position uncertainties.")'
systematic_error /=  arcsec_per_skypixel ; skypix

peaks_cat.X_ERR = sqrt(peaks_cat.X_ERR^2 + systematic_error^2)   ; skypix
peaks_cat.Y_ERR = sqrt(peaks_cat.Y_ERR^2 + systematic_error^2)   ; skypix


peaks_cat.ID           = lindgen(num_peaks)
peaks_cat.CATALOG_NAME = band_name +$
                         string(round(peaks_cat.IMAGE_EXPOSUREAREA / 1E6), F='(%", %d Ms cm**2 count /photon")')
peaks_cat.PROVENAN     = strtrim(peaks_cat.TILE_NAME,2)+' '+strtrim(peaks_cat.CATALOG_NAME,2)


; Build region file of all the detections with boxes depicting the "99% error boxes".
significance_threshold = 0.99

match_xy        , match_state, peaks_cat, 'peaks', significance_threshold, /INIT, ASTROMETRY=event2wcs_astr, /QUIET

match_xy_analyze, match_state, peaks, OUTPUT_DIR=output_dir, /QUIET

; Build region file in celestial coordinates.
;catalog_ds9_interface, peaks_cat, /WRITE_REGFILE, ASTROMETRY=event2wcs_astr,output_dir+'/peaks.reg'


; Save catalog, which has both celestial and tangent plane coordinates.
get_date, date_today, /TIMETAG
psb_xaddpar, theader, 'DATE'   , date_today
psb_xaddpar, theader, 'CREATOR', creator_string
psb_xaddpar, theader, 'THRESHLD' , invalid_threshold, 'PROB_NO_SOURCE threshold'
mwrfits, peaks_cat, output_dir+'/peaks.fits', theader, /CREATE

save, /COMPRESS, peaks_cat, theta_a, src_area_a, nominal_cell_area_a, tile_bkg_ratio_a, floor_cnts, invalid_threshold, event2wcs_astr,$
      FILE=output_dir+'/peaks.sav'
    
print, num_peaks, output_dir, F='(%"\n%d significant peak detections have been saved to %s/. ")'


if keyword_set(scene_template_fn) then begin
  psb_xaddpar, scene_hdr, 'BUNIT', 'NONE', 'ratio between observed and model background'
  writefits, output_dir+'/bkg_ratio_map.img', bkg_ratio_map, scene_hdr  ; unitless image
  
  cmd = string(output_dir, band_name, output_dir, output_dir, $
        F='(%"ds9 -title \"%s (%s)\"  %s/bkg_ratio_map.img -region %s/recon_fov.reg -zoom to fit  >& /dev/null &")')
 ;run_command, /QUIET, cmd      
endif

if debug then begin
  print
  print, 'ds9 -linear -lock frame wcs  "neighborhood.img[0]" "neighborhood.img[1]" neighborhood_residual.img smoothed_neighborhood_residual.img totalbackground.density bigpixel_bkg_ratio.img -region load all recon.reg -zoom to fit &'
endif

print
print, 'WARNING: In this experimental version, local background is calculcated by median(), rather than by min().  We have not yet determined whether this change harms our sensitivity in tiles with flat background.'


return
end ; ae_recon_detect



;==========================================================================

;;; Perform extractions within specific time ranges.
;;; time_filter should be a string array of CIAO time specifications, e.g. "tstart:tstop".
;;; extraction_name should be a string array of names for the extractions.
;;;
;==========================================================================
PRO ae_timerange_extract, sourcename, obsname, time_filter, extraction_name, EVTFILE_BASENAME=evtfile_basename

resolve_routine, 'ae_recipe', /COMPILE_FULL_FILE, /NO_RECOMPILE

if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
  print, 'ERROR: parameter "obsname" must be a scalar string'
  retall
endif


;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

srclist_fn = tempdir + 'temp.cat'

src_stats_basename       = 'source.stats'
obs_stats_basename       = 'obs.stats'
num_sources = n_elements(time_filter)

;; Look up the existing extraction information.
    sourcedir = sourcename + '/' 
new_sourcedir = sourcename + '/' + extraction_name + '/'
    obsdir    = sourcename + '/' + obsname + '/' 
new_obsdir    = sourcename + '/' + obsname + '/' + extraction_name + '/'

src_stats_fn = sourcedir + src_stats_basename
obs_stats_fn = obsdir    + obs_stats_basename

obs_stats = headfits(obs_stats_fn)   

;; Make a srclist with multiple instances of the sourcename.
forprint, TEXTOUT=srclist_fn, replicate(sourcename,num_sources), /NoCOMMENT

;; Make sub-extraction directories; copy stats files; link extraction regions .
file_mkdir, new_sourcedir, new_obsdir

file_copy, /OVERWRITE, replicate(sourcedir + src_stats_basename,num_sources), new_sourcedir
file_copy, /OVERWRITE, replicate(obs_stats_fn,                  num_sources), new_obsdir

file_delete, /ALLOW_NONEXISTENT,                    new_obsdir + 'extract.reg'
file_link, replicate('../extract.reg',num_sources), new_obsdir + 'extract.reg'

  print, F='(%"\nae_timerange_extract: ============================================================")'  
  print,        'ae_timerange_extract: Extracting '+sourcename+' segments: ', extraction_name
  print,   F='(%"ae_timerange_extract: ============================================================\n")'  

ae_standard_extraction, obsname, EVTFILE_BASENAME=evtfile_basename, SRCLIST=srclist_fn, EXTRACTION_NAME=extraction_name, TIME_FILTER=time_filter, EXTRACT_BACKGROUNDS=0

; 
; In principle, we could run these for every obsid; the EXPOSURE would be zero for some obsids. 
; We'd have to check carefully whether that would be propagated correctely to the ARF/RMF weighting, and other things.
; For now we require the user to avoid null extractions.
; 
; Then we run a series of MERGE and FIT calls, one for each EXTRACTION_NAME value we have.
; Those results can be collated as is done in the pileup recipe.

acis_extract, srclist_fn, EXTRACTION_NAME=extraction_name, MERGE_NAME=extraction_name, /MERGE_OBSERVATIONS

;acis_extract, srclist_fn, MERGE_NAME=extraction_name, /FIT_SPECTRA, CHANNEL_RANGE=[35,548], /CSTAT, MODEL_FILENAME='xspec_scripts/thermal/tbabs_vapec.xcm'


return
end ; ae_timerange_extract


;==========================================================================
;;; Assign a supplied ds9 region expressed in celestial coordinates as the 
;;; extraction region for a source.
;;;
;;; If the optional parameter "obsname" is omitted, the supplied region becomes
;;; the extraction region for all observations of the source.
;;;
;;; ds9 is used to convert celestial coordinates in the supplied region to the
;;; sky coordinates required for extraction regions.
;==========================================================================

PRO ae_set_region, sourcename, celestial_region_fn, obsname, EXTRACTION_NAME=extraction_name

if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''

;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir

obs_stats_basename       = 'obs.stats'
src_region_basename      = 'extract.reg'
env_events_basename      = 'neighborhood.evt'

;; We can NOT search for "extract.reg" because /MERGE could make a file
;; sourcename/extraction_name/extract.reg which would be misinterpreted here as
;; an observation!  We must instead search for "obs.stats" which appears only in observation
;; directories.
pattern = sourcename + '/*/' + extraction_subdir + obs_stats_basename
obs_stats_fn = file_search( pattern, COUNT=num_obs )

if keyword_set(obsname) then begin
  if (size(obsname,/TNAME) NE 'STRING') || (size(obsname,/DIMEN) NE 0) then begin
    print, 'ERROR: parameter "obsname" must be a scalar string'
    retall
  endif

  num_obs_found = num_obs
  pattern = sourcename + '/' + obsname + '/' + extraction_subdir + obs_stats_basename
  obs_stats_fn = file_search( pattern, COUNT=num_obs )
  if (num_obs_found GT num_obs) then print, strjoin(obsname,' '), num_obs_found-num_obs, F='(%"WARNING!  Obsname %s was specified; ignoring %d other observations.\n")'
endif 

if (num_obs EQ 0) then begin
  print, 'No extractions found.'
  return
endif

obs_dir = strarr(num_obs)
for jj = 0, num_obs-1 do begin
  fdecomp, obs_stats_fn[jj], disk, dir
  obs_dir[jj] = dir
endfor

env_events_fn = obs_dir + env_events_basename
region_fn     = obs_dir + src_region_basename

; Give write-permission to the region files so that we can modify them.
file_chmod, region_fn, /U_WRITE

print, 'Spawning ds9 to perform coordinate conversions ...'
ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='ae_set_region_'+session_name
for ii=0,num_obs-1 do begin
  ;; Load observation data into ds9.
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, env_events_fn[ii], celestial_region_fn

  ;; Load region file into ds9 and resave in PHYSICAL coordinates.
  cmd1 = string(my_ds9,                          F='(%"xpaset -p %s regions system physical")')
  cmd2 = string(my_ds9, region_fn[ii],           F='(%"xpaset -p %s regions save %s")')
  run_command, [cmd1,cmd2], /QUIET
endfor

; Remove write-permission from the region files, so that ae_make_catalog will leave them alone.
file_chmod, region_fn, A_WRITE=0

return
end ;ae_set_region


;==========================================================================
;;; Make some useful plots from the collated AE data products.
;;; Build a regionfile that color codes weak sources, and stratifies strong ones on median energy (similar to CCCP Intro Paper, Figure 4.).
    
;; The input "collatefile" must be a FITS table produced by AE itself, not a "flattened" table produced by ae_flatten_collation! 
;==========================================================================

PRO ae_summarize_catalog, collatefile, bt, TITLE=title, MARKER_RADIUS=marker_radius_p, REGION_TAG=region_tag,$
                          WEAK_SLICE_THRESHOLD=weak_slice_threshold, BOUNDARIES=boundaries,$ 
                          OUTPUT_DIR=output_dir,$
                          SKIP_NEIGHBOR_ANALYSIS=skip_neighbor_analysis

COMMON ae_summarize_catalog, id0, id1, id2, id3, id4, id5, id6, id7, id8, id9, id10, id11, id12, id13, id14, id15, id16

;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir

temp_srclist_fn      = tempdir + 'temp.srclist'
temp_region_fn       = tempdir + 'temp.reg'
polygon_region_fn    = tempdir + 'polygon.reg'


if ~keyword_set(collatefile) then collatefile = 'all.collated'
if ~keyword_set(output_dir ) then output_dir  = '.'
if n_elements(weak_slice_threshold) EQ 0 then weak_slice_threshold = 5 ; net counts

  arcsec_per_skypixel  = 0.492 

  bt=mrdfits(collatefile, 1) 
  num_sources = n_elements(bt)
  bt_template = bt[0]
  
  ENERG_LO = bt.ENERG_LO
  ENERG_HI = bt.ENERG_HI
  band_total = 0
  if ~almost_equal(bt.ENERG_LO[band_total], 0.5, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
  if ~almost_equal(bt.ENERG_HI[band_total], 8.0, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 8.0 keV.\n")'

  print, 'Using the energy band ', ENERG_LO[0], ENERG_HI[0]
          
  CATALOG_NAME        = bt.CATALOG_NAME
  LABEL               = bt.LABEL
  MERGE_NAME          = bt.MERGE_NAME
  BACKGRND            = bt.BACKGRND/1E-9
  SRC_CNTS            = bt.SRC_CNTS            [band_total]
  NET_CNTS            = bt.NET_CNTS            [band_total]
  flux2               = bt.FLUX2               [band_total]
  ENERG_PCT50_OBSERVED= bt.ENERG_PCT50_OBSERVED[band_total]
  SRC_SIGNIF          = bt.SRC_SIGNIF          [band_total]
  PB_MIN              = bt.PB_MIN
  IS_OCCASIONAL       = bt.IS_OCCASIONAL
  PSF_FRAC            = bt.PSF_FRAC
  off_angle           = bt.THETA
  radius50            = (0.85 -0.25*off_angle + 0.10*off_angle^2)  * arcsec_per_skypixel  ; arcsec
  distance_src2src    = bt.DISTANCE_SRC2SRC                        * arcsec_per_skypixel  ; arcsec
  RA                  = bt.RA 
  DEC                 = bt.DEC
  crowding            = distance_src2src / radius50
  
  formatted_region_tag = replicate(keyword_set(region_tag) ? 'tag={'+region_tag+'}' : '', num_sources)

  ; Compute the offset vector to the nearest neighbor source.
  dx = replicate(!VALUES.F_NAN, num_sources)
  dy = dx
  if ~keyword_set(skip_neighbor_analysis) then begin
    for this_index=0L, num_sources-1 do begin
      neighbor_index = bt[this_index].NEIGHBOR
  
      ; For reasons I can't recall, a source is sometimes declared to be its own neighbor.  Ignore these cases.
      if (this_index EQ neighbor_index) then continue
      
      ; To avoid creating two offset vectors for "pairs" of sources, we plot only the vector from the brighter to the weaker extracction.
      ; Use FLUX2 to take out PSF fraction effects.
      if (flux2[this_index] LT flux2[neighbor_index]) then continue
      
      ; Plot only neighbors of bright sources, where the PSF hook might be detectable.
      if (NET_CNTS[this_index] LT 100) then continue
      
      dx[this_index] = -3600* ( RA[neighbor_index] -  RA[this_index]) * cos(DEC[this_index]/!RADEG) ; arcsec
      dy[this_index] =  3600* (DEC[neighbor_index] - DEC[this_index])                               ; arcsec
    endfor
  endif ; ~keyword_set(skip_neighbor_analysis)
  
  dt = keyword_set(title) ? title : ''
  dataset_1d, id0, SRC_CNTS            , DENSITY_TITLE=dt, BINSIZE=5   , XTIT='# of extracted counts, 0.5-8 keV'
  dataset_1d, id1, NET_CNTS            , DENSITY_TITLE=dt, BINSIZE=5   , XTIT='# of net counts, 0.5-8 keV'

  dataset_1d, id3, PSF_FRAC            , DENSITY_TITLE=dt, BINSIZE=0.05, XTIT='PSF Fraction @1.5keV', BIN_LOCATION=0.025
  
 ;dataset_1d, id4, distance_src2src    , DENSITY_TITLE=dt, BINSIZE=1   , XTIT='distance (arcsec) to nearest neighbor' 
  dataset_2d, id4, dx, dy, /UNITY_ASP         , PSYM=1,    XTIT='offset to nearest neighbor (arcsec)', YTIT='offset to nearest neighbor (arcsec)'
  dataset_2d, id4, [0], [0], COLOR='green',     PSYM=6, DATASET_NAME='brighter source', TIT='neighbors of bright sources'
  
  dataset_1d, id5, crowding            , DENSITY_TITLE=dt, BINSIZE=1   , XTIT='crowding metric (distance/R50) to nearest neighbor' 
  dataset_2d, id6, off_angle, distance_src2src, PSYM=1,    XTIT='off-axis angle (arcmin)', YTIT='distance (arcsec) to nearest neighbor'
  dataset_2d, id7, PSF_FRAC, BACKGRND, PSYM=1,             XTIT='PSF Fraction @1.5keV', YTIT='Background Surface Brightness (X10-9 photon /pixel**2 /cm**2 /s)'
  dataset_1d, id8, BACKGRND            , DENSITY_TITLE=dt,               XTIT='Background Surface Brightness (X10-9 photon /pixel**2 /cm**2 /s)'
  dataset_1d,id11, ENERG_PCT50_OBSERVED, DENSITY_TITLE=dt, BINSIZE=0.2 , XTIT='median energy (ENERG_PCT50_OBSERVED), 0.5-8.0 keV'
  dataset_1d,id12, SRC_SIGNIF          , DENSITY_TITLE=dt, BINSIZE=1   , XTIT='signal-to-noise ratio'

  
  dataset_1d,id15, alog10(1e-8 > PB_MIN[where(~IS_OCCASIONAL)]), BINSIZE=0.1 , DATASET='not Occasional', DENSITY_TITLE=dt, XTIT='log PB_MIN (most-valid merge,band)'
  dataset_1d,id16,               PB_MIN[where(~IS_OCCASIONAL)] , BINSIZE=1e-4, DATASET='not Occasional', DENSITY_TITLE=dt, XTIT='PB_MIN (most-valid merge,band)'
  
  ind = where(/NULL, IS_OCCASIONAL)
  if isa(ind) then begin
  dataset_1d,id15, alog10(1e-8 > PB_MIN[ind]), BINSIZE=0.1, DATASET='Occasional'
  dataset_1d,id16,               PB_MIN[ind], BINSIZE=1e-4, DATASET='Occasional'
  endif

  
  if tag_exist(bt,'MERGE_KS') then begin
    ; Calculate p-value for reported PROB_KS = min( P1,P2,...Pn), where n = N_KS.
    pval_for_PROB_KS = 1D - (1D - bt.PROB_KS)^bt.N_KS
    pval_for_PROB_KS[where(/NULL, ~finite(bt.PROB_KS))] = !VALUES.F_NAN

    print, total(/INT, finite(pval_for_PROB_KS)),  F='(%"The single-ObsID KS test for variability is available for %d sources.")'
    print, total(/INT, finite(bt.MERGE_KS)),  F='(%"The multi-ObsID KS test for variability is available for %d sources.")'
    print, total(/INT, finite(bt.MERG_CHI)),  F='(%"The chi^2 test for variability  is available for %d sources.")'

    dataset_1d, id10, min(/NAN, DIM=2, [[pval_for_PROB_KS],[bt.MERGE_KS],[bt.MERG_CHI]]), DENSITY_TITLE=dt, XTIT='p-value for variability metrics (pval_for_PROB_KS < MERGE_KS < MERG_CHI)'
  endif
  
  if tag_exist(bt,'MERGBIAS') then begin
    ind = where(bt.MERGBIAS GT 0.01, count)
    if (count GT 0) then dataset_1d, id13, (bt.MERGBIAS)[ind], DENSITY_TITLE=dt, XTIT='MERGBIAS'
  endif
  
  if tag_exist(bt,'MERGQUAL') then begin
    ind = where(bt.MERGQUAL LT 0.99, count)
    if (count GT 0) then dataset_1d, id14, (bt.MERGQUAL)[ind], DENSITY_TITLE=dt, XTIT='MERGQUAL'
  endif
  
  if tag_exist(bt,'ERR_DATA') then begin
    dataset_2d, id15, off_angle, bt.ERR_DATA, PSYM=1, XTIT='off-axis angle (arcmin)', YTIT='63% position uncertainty (arcsec)'
  endif
  
  
  theta_bins = [0,4,1E10] < max(off_angle, /NAN)
 ;colors     = ['blue', 'red', 'green', 'white']
  lines      = [0,1,2,3]
  for ii=0,n_elements(theta_bins)-2 do begin
    theta_min = theta_bins[ii]
    theta_max = theta_bins[ii+1]
    mask = (theta_min LE off_angle) AND (off_angle LE theta_max)
    if (total(mask) GT 0) then begin
      name = string(theta_min, theta_max, F='(%" %4.1f < theta < %4.1f (arcmin)")')
      print, name
      dataset_1d, id2, alog10(flux2*mask), COLOR='white', LINESTYLE=lines[ii], DATASET=name, DENSITY_TITLE=dt, BINSIZE=0.1, NORMALIZE_DENSITY=0, XTIT='log PhotonFlux_t (photon /cm**2 /s)', PS_CONFIG={filename:output_dir+'/photonflux_by_theta.ps'}, /PRINT
    endif
  endfor  
  
  ; Build a regionfile that color codes weak sources, and stratifies strong ones on median energy (similar to CCCP Intro Paper, Figure 4.).
  ; These colors, symbols, and widths were used in MOXC1 and MOCX2 papers.
  if ~keyword_set(boundaries) then $
  boundaries   = ['0',     '1',        '2',          '3',         '8'           ]
  slice_color  = [    'Gold','OrangeRed','YellowGreen','DodgerBlue','DarkOrchid']
  slice_symbol = [ 'diamond',   'circle',        'box',         'X',     'cross']
  slice_width  = [         3,          3,            3,           3,           1]
  
  slice_name   = [boundaries+'-'+boundaries[1:*]+' keV'               ,$
                  string(weak_slice_threshold,F='(%"NET_CNTS LT %0.1f")')]
  num_slices   = n_elements(boundaries)

  slice    = value_locate(float(boundaries), ENERG_PCT50_OBSERVED) 
  
  ; For "weak sources" we say that Emedian is "unknown".
  ind = where(NET_CNTS LT weak_slice_threshold, count)
  
  if (count GT 0) then begin
    slice[ind] = num_slices-1  ;WEAK slice
  endif
  
  if (min(slice) LT 0)            then message, 'ERROR: MedianEnergy boundaries do not span values in ENERG_PCT50_OBSERVED.'
  if (max(slice) GT num_slices-1) then message, 'ERROR: MedianEnergy boundaries do not span values in ENERG_PCT50_OBSERVED.'

  ; Choose a reasonable radius for the circular ds9 markers built below.
  if ~keyword_set(marker_radius_p) then begin
    gcirc, 2,  max(RA), mean(DEC),  min(RA),  mean(DEC), field_width_arcsec
    gcirc, 2, mean(RA),  max(DEC), mean(RA),   min(DEC), field_height_arcsec
    marker_radius_p  = (field_width_arcsec > field_height_arcsec) / 800
    marker_radius_p >= 2.0
  endif
  marker_radius = replicate( marker_radius_p,  num_sources)
  
  region_file = output_dir+'/MedianEnergy_slice.reg'
  file_delete, [region_file,polygon_region_fn], /ALLOW_NONEXISTENT
  openw,  region_unit, region_file, /GET_LUN
  printf, region_unit, "# Region file format: DS9 version 3.0"
  printf, region_unit, 'global width=1 font="helvetica 12 normal"'

  for jj=0,num_slices-1 do begin
    ; For MOXC papers, the layering of region slices we want in ds9 (front-to-back) is: 3:8, 0:1, 2:3, 1:2, "weak".
    ii = ([3,0,2,1,4])[jj]

    ind = where(slice EQ ii, count)
    if (count GT 0) then begin
      tag   = replicate('MedianEnergy '+slice_name[ii], num_sources)
      color = replicate(               slice_color[ii], num_sources)
      symbol= replicate(              slice_symbol[ii], num_sources)
      width = replicate(               slice_width[ii], num_sources)
      print, 'Building regions for slice '+tag[0]

;      forprint, TEXTOUT=5, SUBSET=ind, symbol, RA, DEC, LABEL, tag, color, width, formatted_region_tag, F='(%"J2000;%s point %12.8f %12.8f # text={%s} tag={%s} color={%s} width=%d %s")', /NoCOMMENT
      
      !TEXTUNIT = region_unit
      forprint, TEXTOUT=5, SUBSET=ind, RA, DEC, marker_radius, LABEL, tag, color, width, formatted_region_tag, F='(%"J2000;circle %12.8f %12.8f %0.1f\" # text={%s} tag={%s} tag={cat} color={%s} width=%d %s")', /NoCOMMENT
      
      printf, region_unit, 300, 560-ii*40, slice_name[ii], count, tag[0], color[0], F='(%"IMAGE;text %d %d {%s : %4d} # tag={%s} color={%s} tag={legend} font=\"helvetica 24 bold\"   ")'
      !TEXTUNIT = 0

      ; Build color-coded polygons.
      acis_extract, CATALOG_NAME[ind], MERGE_NAME=MERGE_NAME[ind], COLLATED_FILENAME='/dev/null', REGION_FILE=temp_region_fn, REGION_TAG=tag[0], MATCH_EXISTING=bt_template, VERBOSE=0

      cmd = string(temp_region_fn, color[0], polygon_region_fn, F='(%"egrep ''polygon'' %s | sed -e ''s/DodgerBlue/%s/'' >> %s")')
      run_command, /QUIET, cmd
    endif
  endfor
  free_lun, region_unit
  
  cmd = string(polygon_region_fn, region_file, F='(%"cat %s >> %s")')
  run_command, /QUIET, cmd

  run_command, string(keyword_set(dt) ? dt : 'target summary', region_file, F='(%"ds9 -title \"%s\" ../target.emap -region %s -zoom to fit &")')
  
;  fn = 'reduced_aperture.reg'
;  file_delete, fn, /ALLOW_NONEXISTENT
;  ind = where(PSF_FRAC LT 0.85, count)
;  if (count GT 0) then begin
;    forprint, TEXTOUT=temp_srclist_fn, CATALOG_NAME,  F="(%'%s')", SUBSET=ind, /NoCOMMENT
;    
;    acis_extract,     temp_srclist_fn, MERGE_NAME=(bt.MERGE_NAME)[ind], COLLATED_FILENAME='/dev/null', REGION_FILE=fn, REGION_TAG='reduced_aperture', VERBOSE=0, MATCH_EXISTING=bt
;  endif
  
return
end ; ae_summarize_catalog





  
  
  

;==========================================================================
;;; For each source position in a supplied AE collation, calculate validation sensitivity.
;;;
;;; Our approach is to imagine a hypothetical source at the location of each real source.
;;; The hypothetical source is extracted from the same apertures used for the real source,
;;; and has the same background spectra (including stray light from crowded neighbors).
;;; We calculate the smallest number of NetCounts required for the hypothetical source to
;;; meet the specified detection threshold.

;;; YOU SHOULD CAREFULLY CHOOSE THE COLLATION SUPPLIED, WITH YOUR GOALS IN MIND.
;;; NOTE THAT THE SET OF OBSERVATIONS used to validate each hypothetical source is assumed to
;;; be whatever set of observations were used in each AE merge chosen for the supplied AE collation.
;;; In older Townsley/Broos projects validity-optimized merges were run, producing
;;; collations Pb_full_band.collated, Pb_full_band.collated, Pb_soft_band.collated.

;;; In newer Townsley/Broos projects the merges used for validation ("theta-range" and "single-ObsID")
;;; are not optimized by the data observed.  The validation merge that is optimal for each source
;;; is recorded in most_valid_merge.collated.
;;; When your observations have similar aimpoints, you probably want sensitivity to be calculated
;;; under the assumption that validation merges use all observations.  
;;; The collation "all_inclusive.collated" does that.
;;;
;;; However, when you have misaligned observations, its hard to guess which of the 
;;; merges your procedues consider for validation is likely to be most sensitive.  
;;; For example, long observations in the 5' to 9' theta range could be more sensitive than 
;;; short observations on-axis.  Or a single observation with low local background could be
;;; more sensitive than any of the available multi-ObsID merges.
;;; In this case, you may want to have the hypothetical sources use whatever merge turned out to be 
;;; best for the real sources (even though that can be poor when a real sources is not constant).


;smallest NetCounts value (and FLUX2 value) 


;;; For each extraction described in the supplied collation, estimate the incident point source photon flux  
;;; required to achieve the specified Pb value on completeness_goal percent of random trials.

;;; The collation defines each hypothetical source's merged exposure time, ARF, average aperture size, local bkg, and energy band. 
;;;

;;; Example
;;;   .run ae
;;;   ae_sensitivity_simulation, 'Pb_full_band.collated', 0.003, COMPLETENESS_GOAL=0.5, EXPOS=8e4*[1./25, 1./5, 1.0, 5, 25, 50, 100], dlc, dlf
;;;   ae_sensitivity_simulation, 'Pb_hard_band.collated', 0.003, COMPLETENESS_GOAL=0.5, EXPOS=8e4*[1./25, 1./5, 1.0, 5, 25, 50, 100], dlc, dlf

;==========================================================================

PRO ae_sensitivity_simulation, collatefile, BAND_NUM=band_num_for_detection, INVALID_THRESHOLD=invalid_threshold,$
       MIN_NUM_CTS=min_num_cts, COMPLETENESS_GOAL=completeness_goal,$

       ; Optional inputs
       THETA_RANGE=theta_range, RUN_NAME=run_name, VERBOSE=verbose, $
       EXPOSURE=exposure,$  ; Force all extractions in collation to be scaled to a single exposure time.
 
       ; Outputs
       detection_probability_at_limit, detection_limit_counts, detection_limit_FLUX2

COMMON ae_sensitivity_simulation, idc, idf, id0, id1, id2, id3, id4, id5, id6, id7, id8, id9, id10, id11, id12, id13, id14

creator_string = "ae_sensitivity_simulation, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

if n_elements(verbose) EQ 0 then verbose = 0
if (n_elements(theta_range) NE 2) then theta_range = [0,100.]

bt=mrdfits(collatefile, 1, /SILENT) 


; Verify that every row uses the same energy band.
ENERG_LO     = bt[0].ENERG_LO[band_num_for_detection]
ENERG_HI     = bt[0].ENERG_HI[band_num_for_detection]
energy_range_for_detection = string(ENERG_LO,ENERG_HI, F="(%'%0.1f:%0.1f keV')")

if ~almost_equal(bt.ENERG_LO[band_num_for_detection], ENERG_LO, DATA_RANGE=lrange) then print, band_num_for_detection, lrange, F='(%"\nWARNING: Band #%d is not consistent:  %0.2f <= ENERG_LO <= %0.2f \n")'
if ~almost_equal(bt.ENERG_HI[band_num_for_detection], ENERG_HI, DATA_RANGE=hrange) then print, band_num_for_detection, hrange, F='(%"\nWARNING: Band #%d is not consistent:  %0.2f <= ENERG_HI <= %0.2f \n")'


; When detection band is 0.5:7, it is convenient for the photon flux upper limits we calculate to be defined
; for the 0.5:8 keV band, because that is one of the photon fluxes commonly reported.
; That requires finding the 0.5:8 keV band in the collation.
band_num_for_photon_flux = band_num_for_detection
energy_range_for_flux    = energy_range_for_detection

if (ENERG_LO EQ 0.5) && (ENERG_HI EQ 7) then begin
  ind = where(/NULL, (bt[0].ENERG_LO EQ 0.5) AND (bt[0].ENERG_HI EQ 8.0))

  if isa(ind) then begin
    band_num_for_photon_flux = ind[0]
    energy_range_for_flux = string(0.5,8.0, F="(%'%0.1f:%0.1f keV')")
    print, energy_range_for_flux, energy_range_for_detection,$
       F='(%"\nWARNING: to facilitate comparision of photon flux upper limits to standard AE catalogs, those limits will be calculated using MEAN_ARF from the %s band, not the %s band.")'
  endif
endif 

arf_title = 'MEAN_ARF,'+energy_range_for_flux+' (cm**2 count /photon)'




if ~keyword_set(run_name) then run_name = string(100*completeness_goal, energy_range_for_detection, F="(%'%d%% detection probability, %s')")

;if ~keyword_set(run_name) then run_name = string(100*completeness_goal, energy_range_for_detection, invalid_threshold, F="(%'%d%% detection probability, %s, Pb<%0.3f')")


; Show MEAN_ARF values for the FULL catalog, so that we can estimate the value corresponding to an on-axis 90% aperture (for comparison to an on-axis 100% ARF used by PIMMS).
dataset_1d, id14, bt.MEAN_ARF[band_num_for_photon_flux], BINSIZE=1, DENSITY_TITLE='full catalog', XTIT=arf_title

dataset_2d, id13, bt.MEAN_ARF[band_num_for_photon_flux], bt.THETA, PSYM=1, TITLE='full catalog', YTIT='THETA (arcmin)', XTIT=arf_title


; Then, select sources by off-axis angle.
bt = bt[ where((bt.THETA GE theta_range[0]) AND (bt.THETA LE theta_range[1])) ]

num_sources = n_elements(bt)

print, energy_range_for_detection, energy_range_for_flux,$
   F='(%"\nINFORMATION: NetCounts upper limits will be estimated in the %s band.\nINFORMATION: NetCounts upper limits will be converted to photon flux using the MEAN_ARF from the %s band.")'



CATALOG_NAME = strtrim(bt.CATALOG_NAME,2)
LABEL        =  strtrim(bt.LABEL,2)
THETA        =  bt.THETA
MEAN_ARF_for_photon_flux = $
                bt.MEAN_ARF[band_num_for_photon_flux]   
BKG_CNTS     =  bt.BKG_CNTS[band_num_for_detection]
BACKSCAL     =  bt.BACKSCAL[band_num_for_detection]
EMAP_TOT     =  bt.EMAP_TOT
RA           =  bt.RA
DEC          =  bt.DEC




;;; Handle the various cases for the optional EXPOSURE parameter.
num_exposures = n_elements(exposure)

case num_exposures of
  0: begin
     exposure = bt.EXPOSURE
     dataset_1d, id12, exposure, BINSIZE=1,  XTIT='EXPOSURE (s)'
     mean_bkg_in_aperture =  float(BKG_CNTS)/BACKSCAL

     print,  median(mean_bkg_in_aperture), median(BKG_CNTS), median(BACKSCAL), F='(%"\nSTARTING SIMULATION with  EXPOSURE values in collation\n  median bkg_in_aperture = %0.2f ct\n  median BKG_CNTS = %0.1f ct\n  median BACKSCAL = %0.1f")'
     end

  1: begin
     exposure = exposure[0] ; convert a 1-element vector into a scalar, to avoid failure below

     ;; Calculate the exposure scaling required for each source to achive the specified EXPOSURE time.
     
     exposure_scaling = exposure / bt.EXPOSURE
     if verbose GE 1 then info, exposure_scaling
     
     if (n_elements(exposure_scaling) NE num_sources) then message, 'exposure_scaling has the wrong dimensions!'
     
     ;; Scale the bkg surface brightness in each apertue to correspond to the exposure_scaling computed above.
     mean_bkg_in_aperture = exposure_scaling * float(BKG_CNTS)/BACKSCAL
     
     
     ;; Decide HOW we should implement these bkg_in_aperture levels---do we scale BKG_CNTS, or keep that fixed and scale BACKSCAL?
     ; An actual AE extraction is complicated ...
     ; Bkg regions of crowded sources have a limited number of possible counts, because they must fairly sample their neighbor.
     ; For uncrowded sources, AE has several rules that govern the total number of bkg counts.
     ; One of those rules, the goal for the number of bkg counts, is specified by the user (usually as 100 counts).
     ; All this really matters in this simultion, because the increasing the number of bkg counts (up to some limit) reduces the number of counts from source that are required to produce a given Pb.
     
     ind_longer = where(exposure_scaling GT 1, num_longer, COMPLEMENT=ind_shorter, NCOMPLEMENT=num_shorter)
     
     if (num_longer GT 0) then begin
       ; For simulation of longer observations, I'll scale BKG_CNTS by exposure (a good model for what will happen to the crowded sources).
       BKG_CNTS[ind_longer]  *= exposure_scaling[ind_longer]
       print, num_longer, num_sources, F='(%"For %d of %d sources BKG_CNTS has been scaled up.")' 
     endif 
     
     if (num_shorter GT 0) then begin
       ; For simulation of shorter observations, I'll keep BKG_CNTS constant (a good model for isolated sources).
       BKG_CNTS[ind_shorter] *= 1
     endif 
     
     ;; Calculate the revised BACKSCAL that achieve the desired mean_bkg_in_aperture.
     BACKSCAL = BKG_CNTS / mean_bkg_in_aperture
     if verbose GE 1 then info, BKG_CNTS
     if verbose GE 1 then info, BACKSCAL

     print, exposure/1000, median(mean_bkg_in_aperture), median(BKG_CNTS), median(BACKSCAL), F='(%"\nSTARTING SIMULATION with constant EXPOSURE ( %d ks)\n  median bkg_in_aperture = %0.2f ct\n  median BKG_CNTS = %0.1f ct\n  median BACKSCAL = %0.1f")'

     end

  else: begin
        ;; Recursive calls to run several simulations.
        detection_probability_at_limit = replicate(!VALUES.F_NAN, num_sources, num_exposures)
        detection_limit_counts         = replicate(!VALUES.F_NAN, num_sources, num_exposures)
        detection_limit_FLUX2          = replicate(!VALUES.F_NAN, num_sources, num_exposures)
        
        for ii=0,num_exposures-1 do begin
          ae_sensitivity_simulation, collatefile, invalid_threshold, EXPOSURE=exposure[ii], COMPLETENESS_GOAL=completeness_goal, THETA_RANGE=theta_range, RUN_NAME=run_name, limit_probability, limit_counts, limit_FLUX2, VERBOSE=verbose
          
          detection_probability_at_limit[*,ii] = limit_probability
          detection_limit_counts        [*,ii] = limit_counts
          detection_limit_FLUX2         [*,ii] = limit_FLUX2
        endfor
      
         
        function_1d, idc, exposure, median(detection_limit_counts, DIMENSION=1), PSYM=2, TITLE=run_name, XTIT='exposure (s)', YTIT='events from source (ct)'
        
        function_1d, idf, exposure, median(detection_limit_FLUX2 , DIMENSION=1), PSYM=2, TITLE=run_name, XTIT='exposure (s)', YTIT='FLUX2 (photon /cm**2 /s)'
        return
        end ; Recursive calls 
endcase


if (n_elements(BACKSCAL) NE num_sources) then message, 'BACKSCAL has the wrong dimensions!'


;; Search for a mean number of counts from the source that will produce a "detection" (Pb smaller than the threshold) probability of completeness_goal.
Nsim = 1000  
detection_probability_at_limit = replicate(!VALUES.F_NAN, num_sources)
detection_limit_counts         = replicate(!VALUES.F_NAN, num_sources)


for ii=0,num_sources-1 do begin
  if (ii MOD 100) EQ 0 then print, (100.*ii)/num_sources, F='(%"\n**** %d%% of sources processed.")'
  if verbose GE 2 then print, LABEL[ii], F='(%"%s:")'
  
  if (BACKSCAL[ii] EQ 0) || ~finite(BACKSCAL[ii]) then begin
    print, LABEL[ii], BACKSCAL[ii], F='(%"Ignoring source %s with BACKSCAL=%f")'
    continue
  endif
  
  ; The bkg region is a random realization of the BKG_CNTS mean.
    this_BKG_CNTS = random(POISSON=BKG_CNTS[ii]                                      , Nsim)
  
  ; I'll search using a set of NetCounts values that balance precision in our upper limit estimates
  ; against run time.  Integer values are NOT required.
  ; The smallest NetCounts value that should be tested is unclear; I'll start at 1 count.
; proposed_mean_signal_in_aperture = [1+0.5*findgen(38), 20+1*findgen(30), 50+2*findgen(25), 100+4*findgen(26)]
  proposed_mean_signal_in_aperture = 1+findgen(200)

; plot, proposed_mean_signal_in_aperture
; plot, (proposed_mean_signal_in_aperture[1:*] - proposed_mean_signal_in_aperture) / proposed_mean_signal_in_aperture[1:*]
  
  num_signals_considered = n_elements(proposed_mean_signal_in_aperture)
  
  detection_fraction = replicate(!VALUES.F_NAN, num_signals_considered)
  
  candidate_low_index  = -1
  candidate_high_index = num_signals_considered
  while 1 do begin
    ; Identify the contiguous set of signal values in proposed_mean_signal_in_aperture list that are still 
    ; candidates for the signal value we are seeking.
    ; That set consists of:
    ;  - the largest signal with a detection_fraction LT goal: candidate_low_index
    ;  - the smallest signal with a detection_fraction GE goal: candidate_high_index
    ;  - any signals between those.
    ind_low  = where(/NULL, detection_fraction LT completeness_goal) 
    ind_high = where(/NULL, detection_fraction GE completeness_goal) 

    if isa(ind_low ) then candidate_low_index  = ind_low [-1] ; Last element of ind_low
    if isa(ind_high) then candidate_high_index = ind_high[ 0] ; First element of ind_high
    
    if (candidate_low_index   GE candidate_high_index) then message, 'BUG in search_*_index calculation!'



    if (candidate_low_index+1 EQ candidate_high_index) then begin
      ; WE HAVE IDENTIFIED THE SIGNAL THAT BEST ACHIEVES THE DETECTION PROBABILITY GOAL.
      search_successful = 1B

      ; Check for exceptions.
      if (candidate_low_index LT 0) then begin
        print, LABEL[ii], proposed_mean_signal_in_aperture[candidate_high_index], detection_fraction[candidate_high_index], completeness_goal, F='(%"WARNING: Search failed for %s; at the lowest NetCounts considered (%d) the detection fraction (%0.2f) was above the completeness goal (%0.2f).")'

        search_successful = 0B
      endif
      
      if (candidate_high_index GT (num_signals_considered-1)) then begin
        ; Repair candidate_high_index
        candidate_high_index = num_signals_considered-1

        print, LABEL[ii], proposed_mean_signal_in_aperture[candidate_high_index], detection_fraction[candidate_high_index], completeness_goal, F='(%"WARNING: Search failed for %s; at the highest NetCounts considered (%d) the detection fraction (%0.2f) was below the completeness goal (%0.2f).")'

        search_successful = 0B
      endif

      ; Record the detection probability we have found.
      detection_probability_at_limit[ii] =               detection_fraction[candidate_high_index]

      if search_successful then begin
      ; Record the detection limit we have found.
              detection_limit_counts[ii] = proposed_mean_signal_in_aperture[candidate_high_index]

        if verbose GE 2 then print, detection_limit_counts[ii],$
                        100*detection_probability_at_limit[ii],$
                        F='(%"  detection limit is %3d net ct (%d%% completeness)")'
      endif else begin
        ; Record failed search.
        detection_limit_counts[ii] =  !VALUES.F_NAN
      endelse


      break ; No signals left to evaluate.
    endif ; BREAK OUT OF SEARCH
             
    ;;; CONTINUE WITH SEARCH ...

    ; Set set of signals worth evaluating consists of those between candidate_low_index and candidate_high_index.
    ; We chose one near the middle of that interval to evaluate next.
    evaluate_index = floor(mean([candidate_low_index+1,candidate_high_index-1]))
    evaluate_index >= 0
    evaluate_index <= (num_signals_considered-1)

    if (finite(detection_fraction[evaluate_index])) then message, 'BUG in search algorithm!'


    ;;; CALCULATE DETECTION FRACTION FOR THE PROPOSED SIGNAL SELECTED ABOVE.
    ;;;
    ; The number of counts in the aperture is a random realization of the combined mean bkg_in_aperture and mean signal from the star.
    this_mean_signal_in_aperture = proposed_mean_signal_in_aperture[evaluate_index]

    this_SRC_CNTS = random(POISSON=mean_bkg_in_aperture[ii] + this_mean_signal_in_aperture, Nsim)
  
    ; Compute the significance of the observed SRC_CNTS as a disproof of the "null hypothesis" which is 
    ; that there is no source, i.e. that all the observed counts are background.  
    ; We use equation A7 from Weisskopf 2006 (astro-ph/0609585):
    PROB_NO_SOURCE = replicate(!VALUES.F_NAN, Nsim)
    for jj=0L, Nsim-1 do begin
      V = this_SRC_CNTS[jj]
      N = this_SRC_CNTS[jj] + this_BKG_CNTS[jj]
      P = 1D/(1D + BACKSCAL[ii])
      
      if (V LT MIN_NUM_CTS) then begin
        PROB_NO_SOURCE[jj] = 2
        continue
      endif
      
      ; As of IDL v8.5.1, binomial() has significant flaws and frequently produces very wrong results.
      ; See email to Harris Inc. on Oct 22, 2016.
      ; It also behaves very badly if GAUSSIAN=0 is (innocently) supplied (IDL bug report 69655).
      ; Prior to v8.5.1, binomial() could not handle inputs larger than 32767 (IDL bug report 69442).
      ; We use instead a simple and reliable algorithm recommended by Numerical Recipes (Chapter 6.4).
  
      PROB_NO_SOURCE[jj] = binomial_nr(V, N, P )

      if ~finite( PROB_NO_SOURCE[jj] ) then message, 'ERROR: PROB_NO_SOURCE is not finite.'
    endfor ; jj

    num_detections         = total(/INT,        PROB_NO_SOURCE LE invalid_threshold)
    num_trials = total(/INT, finite(PROB_NO_SOURCE))

    detection_fraction[evaluate_index] = float(num_detections) / float(num_trials)

    if verbose GE 3 then print, this_mean_signal_in_aperture, evaluate_index, detection_fraction[evaluate_index], F='(%"  NET_CNTS=%0.1f  detection fraction[%d]=%0.2f")'

  endwhile ; binary search of signals in proposed_mean_signal_in_aperture

  ;if verbose GE 3 then wait, 5
endfor ; ii

; Convert limits from count units to photon flux units.
detection_limit_FLUX2 = detection_limit_counts / MEAN_ARF_for_photon_flux / EXPOSURE

; Summarize results.
print, total(/INT, finite(detection_limit_counts)), median(detection_limit_counts), 100*completeness_goal, F='(%"\n\n\n%d detection limits calculated; median is %3d net ct (%d%% completeness)")'


dataset_1d, id8, detection_probability_at_limit, DATASET_NAME=run_name, XTIT='detection probability at limit estimate'



nc_tit   = 'net events in aperture (ct)'
flux_tit = 'photon flux, '+energy_range_for_flux+' (ph /cm**2 /s)'
emap_tit = 'emap value (s cm**2 count /ph)'

dataset_1d, id0,        detection_limit_counts , DATASET_NAME=run_name, BINSIZE=1, XTIT=nc_tit

dataset_1d, id1, alog10(detection_limit_counts), DATASET_NAME=run_name, BINSIZE=0.1, XTIT='log '+nc_tit


dataset_1d, id2,        detection_limit_FLUX2 , DATASET_NAME=run_name, BINSIZE=1e-9, XTIT=flux_tit

dataset_1d, id3, alog10(detection_limit_FLUX2), DATASET_NAME=run_name, BINSIZE=0.05, XTIT='log '+flux_tit


dataset_2d, id4,        detection_limit_FLUX2 , THETA, DATASET=run_name, PSYM=1, XTIT=flux_tit, YTIT='THETA (arcmin)'

dataset_2d, id5, alog10(detection_limit_FLUX2), THETA, DATASET=run_name, PSYM=1, XTIT='log '+flux_tit, YTIT='THETA (arcmin)'

dataset_2d, id6,        detection_limit_FLUX2 , EMAP_TOT, DATASET=run_name, PSYM=1, XTIT=flux_tit, YTIT=emap_title

dataset_2d, id7, alog10(detection_limit_FLUX2), EMAP_TOT, DATASET=run_name, PSYM=1, XTIT='log '+flux_tit, YTIT=emap_title

; MEAN_ARF_for_photon_flux is very poorly correlated with detection limit flux.
;dataset_2d, id6,        detection_limit_FLUX2 , MEAN_ARF_for_photon_flux, DATASET=run_name, PSYM=1, XTIT=flux_tit, YTIT=arf_title
;dataset_2d, id7, alog10(detection_limit_FLUX2), MEAN_ARF_for_photon_flux, DATASET=run_name, PSYM=1, XTIT='log '+flux_tit, YTIT=arf_title


dataset_1d, id0, PS_CONFIG={filename:'netcnts_distr.ps'}, /PRINT
dataset_1d, id1, PS_CONFIG={filename:'LOG_netcnts_distr.ps'}, /PRINT
dataset_1d, id2, PS_CONFIG={filename:'photonflux_distr.ps'}, /PRINT
dataset_1d, id3, PS_CONFIG={filename:'LOG_photonflux_distr.ps'}, /PRINT

dataset_2d, id4, PS_CONFIG={filename:    'photonflux_vs_theta.ps'}, /PRINT
dataset_2d, id5, PS_CONFIG={filename:'LOG_photonflux_vs_theta.ps'}, /PRINT
dataset_2d, id6, PS_CONFIG={filename:    'photonflux_vs_emap.ps'}, /PRINT
dataset_2d, id7, PS_CONFIG={filename:'LOG_photonflux_vs_emap.ps'}, /PRINT

dataset_1d, id8, PS_CONFIG={filename:'detection_probability_distr.ps'}, /PRINT

get_date, date_today, /TIMETAG
save, /COMPRESS, /VERBOSE, FILE='ae_sensitivity_simulation_'+date_today+'.sav', $
date_today,$
collatefile, invalid_threshold,$
min_num_cts,  completeness_goal,$
theta_range, run_name, verbose,$
band_num_for_detection, band_num_for_photon_flux,$
energy_range_for_detection, energy_range_for_flux,$
CATALOG_NAME,$
LABEL   ,$
THETA   ,$
EMAP_TOT,$
RA ,$
DEC,$
MEAN_ARF_for_photon_flux,$
EXPOSURE,$
BKG_CNTS,$
BACKSCAL,$
detection_probability_at_limit, detection_limit_counts, detection_limit_FLUX2

return
end ; ae_sensitivity_simulation




;==========================================================================
;;; Perform a very simple scan of source lightcurves for cosmic ray afterglows.
                                      
;;; CIAO 4.4 introduced the tool acis_find_afterglow, which replaced the tool acis_run_hotpix.
;;;
;;; As of May 2008 the CXC's tool acis_run_hotpix misses short-duration afterglow incidents.
;;; The sort of scan on extracted data we're doing here is NOT as good as one done on raw data, but it's useful.

;;; Inputs are:
;;;   catalog filename, MERGE_NAME as in many other tools.
;;;   BAND_NUM: 0-based index of the energy band (in source.photometry) on which statistics should be computed.
;;;   MAX_FRAME_SEPARATION, MAX_OFFSET: parameters defining separation of events in time and space that suggests an afterglow origin.
;;;   INVALID_THRESHOLD for Pb_revised can be supplied to override the default of 0.01 (1%).

;;; Outputs are:
;;;   sourcename

;;;   Pb_revised (the PROB_NO_SOURCE in specified band recalculated if the suspected AG events were ignored)

;;;   fraction_inband_suspect_events (fraction of counts in specified band suspected to be AG events); also saved as AG_FRAC in source.stats


;;; See recipe.txt for example usage.

;==========================================================================
PRO ae_afterglow_report, catalog_or_srclist, MERGE_NAME=merge_name, BAND_NUM=band_num, $ ; input parameters
      MAX_FRAME_SEPARATION=max_frame_separation, MAX_OFFSET=max_offset, $                ; optional inputs
      INVALID_THRESHOLD=invalid_threshold_p, $
      SORT_BY_PB=sort_by_pb, SORT_BY_FRACTION=sort_by_fraction, VERBOSE=verbose, $
      sourcename, Pb_revised, num_inband_suspect_events, fraction_inband_suspect_events  ; output parameters
                    
creator_string = "ae_afterglow_report, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

if (n_elements(verbose)              EQ 0) then verbose             =1
if (n_elements(max_frame_separation) EQ 0) then max_frame_separation=3
if (n_elements(max_offset)           EQ 0) then max_offset          =1
if (n_elements(band_num)             EQ 0) then band_num            =0
invalid_threshold = keyword_set(invalid_threshold_p) ? invalid_threshold_p : 0.01   ; 1%

if (invalid_threshold GT 0.1) then begin
  help, invalid_threshold
  message, 'WARNING: INVALID_THRESHOLD is large.  Perhaps you made a mistake?'
endif

;; Check for common environment errors.
existing_quiet = !QUIET  &  !QUIET = 1
catch, error_code
if (error_code NE 0) then begin
  print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
  retall
endif else resolve_routine, 'astrolib'
catch, /CANCEL
astrolib
; Make sure forprint calls do not block for user input.
!TEXTOUT=2
!QUIET = existing_quiet

src_events_basename      = 'source.evt'
src_stats_basename       = 'source.stats'
src_photometry_basename  = 'source.photometry'

readcol, catalog_or_srclist, sourcename, FORMAT='A', COMMENT=';'

; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)

if (num_sources EQ 0) then begin
  print, 'WARNING: no entries read from source list ', catalog_or_srclist
  sourcename                     = ''
  Pb_revised                     = 0
  num_inband_suspect_events      = 0
  fraction_inband_suspect_events = 0
  return
endif

sourcename = sourcename[ind]
print, num_sources, F='(%"\n%d sources found in catalog.\n")'

if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                else merge_subdir = ''
if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)
                              
label                          = strarr(num_sources)
Pb_revised                     = fltarr(num_sources)
fraction_inband_suspect_events = fltarr(num_sources)
num_inband_suspect_events      = lonarr(num_sources)

ag_frame       = lonarr(1E4)
ag_frame_index = 0L

print, max_frame_separation, max_offset, F='(%"Scanning for pairs of events separated by up to %d exposures and offset by up to %d CCD pixels  ...")'

energ_lo = -1 
energ_hi = -1
for ii = 0L, num_sources-1 do begin
  basedir   = sourcename[ii] + '/' 
  sourcedir = basedir + merge_subdir[ii]

 ; We assume that an existing source directory that is a symbolic link should not be written to.
  temp = file_info(basedir)
  is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
  if ~is_writable then begin
    print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
    continue
  endif 

  merged_src_events_fn = sourcedir + src_events_basename
  photometry_fn        = sourcedir + src_photometry_basename
                           
  ; Read the composite extracted event list.
  if ~file_test(merged_src_events_fn) then begin
    print, sourcename[ii], src_events_basename, F='(%"\nSource %s has no %s ; skipping ...")'
    continue
  endif
  
  bt = mrdfits(merged_src_events_fn, 1, src_events_hdr, /SILENT, STATUS=status)
  if (status NE 0) then message, 'ERROR reading ' + merged_src_events_fn                         
                 
  if (psb_xpar( src_events_hdr, 'NAXIS2') LT 2) then begin
    print, 'WARNING: source '+sourcename[ii]+' has too few events to analyze.'
    continue
  endif

  frame_length =       psb_xpar( src_events_hdr, 'TIMEDEL') 

  ; Sort events by time.
  bt = bt[sort(bt.TIME)]
  
  chipx = bt.CHIPX    
  chipy = bt.CHIPY
  energy= bt.ENERGY
  time  = bt.TIME
  time -= time[0] 
  
  ; Between consecutive events compute time intervals in units of CCD frames, CHIPX/CHIPY offsets, and differences in event energies.
  frame  = round(time / frame_length)
  dframe = frame [1:*] - frame
  dx     = chipx [1:*] - chipx
  dy     = chipy [1:*] - chipy
  de     = energy[1:*] - energy

  ; Identify *pairs* of events that are suspect, i.e. "close" in time and space.
  bad_ind = where((dframe LE max_frame_separation) AND (abs(dx) LE max_offset) AND (abs(dy) LE max_offset), num_bad_pairs)

  ; Count how many individual events are suspect.
  num_events            = n_elements(bt)
  flagged_by_AE         = bytarr(num_events)
  if (num_bad_pairs GT 0) then begin
    flagged_by_AE[bad_ind  ] = 1
    flagged_by_AE[bad_ind+1] = 1
  endif
  
  ; Identify events flagged as afterglow by the CXC's aggressive algorithm acis_detect_afterglow (stored in STATUS bits 16-19) which was run on the data by our L1->L2 recipe.
  ; Code adapted from that in compare_event_lists.pro.
  status_word = swap_endian(ulong(bt.STATUS,0,num_events), /SWAP_IF_LITTLE_ENDIAN)
  mask = ishft('1'XUL,16) OR $
         ishft('1'XUL,17) OR $
         ishft('1'XUL,18) OR $
         ishft('1'XUL,19) 

  ag_bits  = long(ishft(mask AND status_word, -16))
  
  flagged_by_acis_detect_afterglow = (ag_bits NE 0)

  ; Build strings reporting the event energy, annotated with an * for events flagged by acis_detect_afterglow tool.
  energy_label = string(energy, F='(%"%5d")')
  annotation   = replicate(' ', num_events)
  ind = where(flagged_by_acis_detect_afterglow, count)
  if (count GT 0) then annotation[ind] = '*'
  energy_label += annotation
  
  ; Look up photometry in the specified energy band.
  bt = mrdfits(photometry_fn, 1, /SILENT, STATUS=status)
  if (status NE 0) then message, 'ERROR reading ' + photometry_fn
  
  if (energ_lo EQ -1) then begin
    energ_lo = bt[band_num].ENERG_LO
    energ_hi = bt[band_num].ENERG_HI
  endif else begin
    if (energ_lo NE bt[band_num].ENERG_LO) then message, 'ERROR: energy band for '+sourcedir+' does not match the previous sources! '
    if (energ_hi NE bt[band_num].ENERG_HI) then message, 'ERROR: energy band for '+sourcedir+' does not match the previous sources! '
  endelse
  
  SRC_CNTS = bt[band_num].SRC_CNTS
  BKG_CNTS = bt[band_num].BKG_CNTS 
  BACKSCAL = bt[band_num].BACKSCAL
  
  is_inband           = (1000*energ_lo LE energy) AND (energy LE 1000*energ_hi)
      
  ; Sanity check the photometry against the event data.
  ; SRC_CNTS_local is a tally of events with ENERGY values in band0.
  ; SRC_CNTS       is a tally of events with PI     values in band0.
  ; Since PI is a binned quantity, we expect small differences.
  ; Thus, we produce a warning only for large differences (>10%).
  SRC_CNTS_local = total(/INT, is_inband)
  diff = abs(SRC_CNTS - SRC_CNTS_local) 
  src_cnts_inconsistency = diff / float(SRC_CNTS > SRC_CNTS_local)
  if (diff GT 1) && (src_cnts_inconsistency GT 0.10) then $
    print, photometry_fn, merged_src_events_fn, 100*src_cnts_inconsistency, F="(%'WARNING! SRC_CNTS in %s and number of events in %s (with energy filter applied) differ by > %0.1f%%.')"
  
  num_inband_suspect_events[ii] = total(/INT, is_inband AND (flagged_by_AE OR flagged_by_acis_detect_afterglow))
   
  fraction_inband_suspect_events[ii] = num_inband_suspect_events[ii]/(SRC_CNTS > 1.0)



  ; Write afterglow statistics to source.stats file.
  stats_fn  = sourcedir + src_stats_basename
  stats = headfits(stats_fn, ERRMSG=error)
  
  if keyword_set(error) then begin
    print, 'WARNING! Could not read '+stats_fn
    continue
  endif
  
  label[ii] = psb_xpar( stats, 'LABEL')
  
  psb_xaddpar, stats, 'AG_FRAC', fraction_inband_suspect_events[ii], string(energ_lo, energ_hi, F="(%'suspected afterglow fraction, %0.2f:%0.2f keV')")

  writefits, stats_fn, 0, stats
  
  
  if (num_inband_suspect_events[ii] EQ 0) then continue
  
  
  ;; Compute the expected number of time intervals betweeen events that are less than the threshold, assuming random arrival.
  
  ; If the number of arrivals in a given time interval [0,t] follows the Poisson distribution, with mean = lambda*t, then the lengths of the inter-arrival times follow the Exponential distribution, with mean 1 / lambda.
  
  ; Total time on the detector.
  exposure               = float(psb_xpar( stats,'FRACEXPO')*psb_xpar( stats,'EXPOSURE')) 
  ; Mean rate of event production over the observation, count /s
  lambda                 = SRC_CNTS / exposure 
  ; Max separation (seconds) for which event pairs will be flagged as afterglows here.
  max_separation         = (max_frame_separation + 1) * frame_length              
  ; Fraction of event separations expected to be SMALLER than max_separation.
  ; This is the integral of an exponential distribution on the range [0:max_separation] == CDF(max_separation).
  ; THE FACTOR OF TWO BELOW ARISES because the exponential cdf is predicting how many INTERVALS between events meet the criteria,
  ; and each such occurance will (usually) generate TWO events (falsely) flagged as afterglows.
  false_positive_fraction = 2 * (1 - exp(-lambda * max_separation))                  
  
  
  ; Compute a revised Pb if all the suspected AG events were removed.
  ; We use equation A7 from Weisskopf 2006 (astro-ph/0609585):
  scaled_bkg = BKG_CNTS / float(BACKSCAL)
  SRC_CNTS_revised = (SRC_CNTS - num_inband_suspect_events[ii]) > 0

  ; As of IDL v8.5.1, binomial() has significant flaws and frequently produces very wrong results.
  ; See email to Harris Inc. on Oct 22, 2016.
  ; It also behaves very badly if GAUSSIAN=0 is (innocently) supplied (IDL bug report 69655).
  ; Prior to v8.5.1, binomial() could not handle inputs larger than 32767 (IDL bug report 69442).
  ; We use instead a simple and reliable algorithm recommended by Numerical Recipes (Chapter 6.4).
  
  Pb_revised[ii] = binomial_nr(SRC_CNTS_revised, $
                               SRC_CNTS_revised + BKG_CNTS, $
                               1D/(1D + BACKSCAL) ) > 0

  if ~finite(Pb_revised[ii]) then message, 'ERROR: PROB_NO_SOURCE is not finite.'
  
  
  if verbose GE 1 then print, sourcedir, strtrim(psb_xpar( stats,'LABEL'),2), num_inband_suspect_events[ii], false_positive_fraction*SRC_CNTS, frame_length * lambda, SRC_CNTS, Pb_revised[ii], F='(%"\n%s (%10s): %d in-band events meet the afterglow criteria.\n  %0.2f are expected from the Poisson arrival of photons (%0.4f event/frame, %d total events).\n  P_b rises to %0.5f if all the suspect events are removed.")'
  
;  print, sourcedir, strtrim(psb_xpar( stats,'LABEL'),2), num_inband_suspect_events[ii], (100.0*fraction_inband_suspect_events[ii]), SRC_CNTS, (100.0*num_inband_suspect_events[ii])/scaled_bkg, Pb_revised[ii], F='(%"\n%s (%10s): %d suspected in-band afterglow events (%0.1f%% of %d SRC_CNTS, %0.1f%% of background); P_b if suspects were removed: %0.5f")'
  
  if (num_inband_suspect_events[ii] GT 20) then begin
    if verbose GE 1 then print, 'More than 20 suspect events; skipped printing!'      
    continue
  endif
  
  if verbose GE 1 then print, 'dFRAME dCHIPX dCHIPY      energy    FRAME MOD 10000'
  if (num_bad_pairs GT 0) && (verbose GE 1) then begin
    forprint, SUBSET=bad_ind, dframe, dx,dy, energy_label[0:num_events-2],energy_label[1:*], frame[0:num_events-2] MOD 10000L, F='(%"  %4d %5d %5d    %6s->%6s        %4d")'
  endif
  
  ind = where(flagged_by_acis_detect_afterglow AND ~flagged_by_AE, count)
  if (count GT 0) then begin
    if verbose GE 1 then print, F='(%"These additional afterglow events were identified only by acis_detect_afterglow:")'
    if verbose GE 1 then forprint, SUBSET=ind, energy_label, frame MOD 1000, F='(%"                            %6s        %4d")'
  endif
  
  if (num_bad_pairs GT 0) then begin
    ag_frame[ag_frame_index] = frame[bad_ind]
    ag_frame_index          += num_bad_pairs
  endif
endfor ;ii


print, 'Energy Range used for reported statistics was ', energ_lo, energ_hi


if keyword_set(sort_by_fraction) then begin
  ; List the sources with suspect events, sorted by fraction_inband_suspect_events.
  fn = 'agr.srclist'
  file_delete, fn, /ALLOW_NONEXISTENT

  sort_key = fraction_inband_suspect_events
  ind = where(sort_key GT 0, count)
  if (count GT 0) then begin
    forprint, /NOCOM, TEXTOUT=fn, SUBSET=ind[reverse(sort(sort_key[ind]))], sourcename, label, F="(%'%s ; (%s)')"
  endif else begin
    print, 'NO AFTERGLOW EVENTS SUSPECTED!'
    file_copy, '/dev/null', fn
  endelse
endif 

if keyword_set(sort_by_pb) then begin
 ; List the sources with suspect events, sorted by Pb_revised.
  fn = 'agr.srclist'
  file_delete, fn, /ALLOW_NONEXISTENT

  ind = where(fraction_inband_suspect_events GT 0, count)
  if (count GT 0) then begin
    forprint, /NOCOM, TEXTOUT=fn, SUBSET=ind[reverse(sort(Pb_revised[ind]))], sourcename, label, F="(%'%s ; (%s)')"
  endif else begin
    print, 'NO AFTERGLOW EVENTS SUSPECTED!'
    file_copy, '/dev/null', fn
  endelse
      
  ; List the sources that would fail Pb if the AG events were removed.
  fn = 'agr_prune.srclist'
  file_delete, fn, /ALLOW_NONEXISTENT
  
  ind = where((Pb_revised GT invalid_threshold), count)
  if (count GT 0) then begin
    forprint, /NOCOM, TEXTOUT=fn, SUBSET=ind[reverse(sort(Pb_revised[ind]))], sourcename, label, F="(%'%s ; (%s)')"
  endif else begin
    print, 'NO SOURCES APPEAR TO BECOME NOT SIGNIFICANT WHEN AFTERGLOW EVENTS ARE REMOVED!'
    file_copy, '/dev/null', fn
  endelse
endif 


;dataset_1d, id, ag_frame[0:ag_frame_index-1], XTIT='Frame #'
return
end  ; ae_afterglow_report



;==========================================================================
;;; Tool that builds "panda" regions that mark the position of the "HRMA anomaly".
;;;   http://cxc.harvard.edu/ciao/caveats/psf_artifact.html

;;; Algorithm is stolen from the tool make_psf_asymmetry_region.

;;; The collation passed should have NET_CNTS in 0.5:7 keV band (band 0) and THETA_LO.
;;; This NET_CNTS value is used to estimate how many hook counts could be produced by ALL ObsIDs.
;;; This THETA_LO value is used to guess if ANY ObsID would have a resolved hook.
;;; A theta-limited merge is probably the best way to estimate NET_CNTS and THETA_LO in this context.

;;; Example
;;;   ae_make_psf_hook_regions, COLLATED_FILENAME='tables/theta_00-05.collated', HOOK_CNTS_THRESHOLD=4, THETA_THRESHOLD=5
;;;
;;; acis_extract, COLLATED_FILENAME='tables/theta_00-05.collated', /SHOW_REGIONS, /OMIT_BKG_REGIONS, OBSID_REGION_GLOB_PATTERN='../obs*/psf_hook.reg', SRCLIST_FILENAME='near_psf_hook.srclist'

;==========================================================================
PRO ae_make_psf_hook_regions, COLLATED_FILENAME=collated_filename,  HOOK_CNTS_THRESHOLD=hook_cnts_threshold, THETA_THRESHOLD=theta_threshold

creator_string = "ae_make_psf_hook_regions, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

        region_fn = 'psf_hook.reg'
bright_srclist_fn = 'bright_psf_hook.srclist'
  near_srclist_fn = 'near_psf_hook.srclist'
  
file_delete, [region_fn, bright_srclist_fn, near_srclist_fn], /ALLOW_NONEXISTENT

hook_power_fraction = 0.06  ; estimated fraction of total PSF counts lying in the hook (from CXC documents)

obs_stats_basename   = 'obs.stats'
src_region_basename  = 'extract.reg'
src_events_basename  = 'source.evt'

collated_table = mrdfits(collated_filename, 1, /SILENT, STATUS=status)
if (status NE 0) then begin
  print, 'ERROR: could not read '+collated_filename
  retall
endif

num_sources = n_elements(collated_table) 

if ~tag_exist(collated_table, 'NET_CNTS') then begin
  print, 'ERROR: your collation does not include a NET_CNTS column.'
  retall
endif

;; Estimate the number of counts that would be detected from each source on an infinite CCD.
band_total = 0
if ~almost_equal(collated_table.ENERG_LO[band_total], 0.5, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_LO <= %0.2f; ENERG_LO should be 0.5 keV.\n")'
ENERG_LO = mean(range)

if ~almost_equal(collated_table.ENERG_HI[band_total], 7.0, DATA_RANGE=range) then print, band_total, range, F='(%"\nWARNING: for Full Band (#%d),  %0.2f <= ENERG_HI <= %0.2f; ENERG_HI should be 7.0 keV.\n")'
ENERG_HI = mean(range)

multiObsID_NET_CNTS       = collated_table.NET_CNTS[band_total]

multiObsID_counts_in_PSF  = multiObsID_NET_CNTS / collated_table.PSF_FRAC

;; Estimate the total hook counts expected from ALL the merged ObsIds, which might have different roll angles.
multiObsID_counts_in_hook = hook_power_fraction * multiObsID_counts_in_PSF


RA  = collated_table.ra
DEC = collated_table.dec
   


openw,  region1_unit, region_fn, /GET_LUN
printf, region1_unit, "# Region file format: DS9 version 3.0"
printf, region1_unit, 'global width=1 font="helvetica 14 normal" '

src_generates_bright_hook = bytarr(num_sources)
src_is_near_bright_hook   = bytarr(num_sources)

for ii=0,num_sources-1 do begin
  ; Off-axis, we assume the hook could not be resolved into a spurious source.
  if (collated_table[ii].THETA_LO GT theta_threshold) then continue
  
  ; Skip hooks that would be weak, even if all ObsIDs had the same roll angle.
  if (multiObsID_counts_in_hook[ii] LT hook_cnts_threshold) then continue

  sourcename = collated_table[ii].CATALOG_NAME

  ;; Determine which ObsIDs were extracted.
  ;; We must look for "obs.stats", which appears only in observation directories; other files also appear in merge directories. 
  pattern_base = sourcename + '/*/'
  obs_stats_fn   = file_search( pattern_base + obs_stats_basename, COUNT=num_obs )
  
  if (num_obs EQ 0) then continue
  
  obsdir = strarr(num_obs)
  for jj = 0, num_obs-1 do begin
    fdecomp, obs_stats_fn[jj], disk, dir
    obsdir[jj] = dir
  endfor
  
  
  ; Color code the regions based on multiObsID_counts_in_hook.
  ; (Color coding by singleObsID_counts_in_hook would lead to visual confusion.)
  case 1 of
   (multiObsID_counts_in_hook[ii] LT  5): color = 'green'
   (multiObsID_counts_in_hook[ii] LT 10): color = 'cyan'
   (multiObsID_counts_in_hook[ii] LT 20): color = 'yellow'
   (multiObsID_counts_in_hook[ii] LT 40): color = 'red'
   else                                 : color = 'magenta'
  endcase
 
  ;; Build a region in celestial coordinates marking the hook in each ObsID.
  info_string = ''
  for jj=0,num_obs-1 do begin
    info_string += (jj GT 0) ? ', ' : ''
    
    src_events_fn  = obsdir[jj] + src_events_basename
    hook_region_fn = obsdir[jj] + 'psf_hook.reg'

    file_delete, hook_region_fn, /ALLOW_NONEXISTENT
    
    ; Read headers from obs.stats and source.evt files.
    obs_stats = headfits(obs_stats_fn[jj], ERRMSG=error)
    if keyword_set(error) then begin
      info_string += '-'
      print, 'ERROR:  Could not read '+obs_stats_fn[jj]
      continue
    endif

    ; Off-axis, we assume the hook could not be resolved into a spurious source.
    if (psb_xpar( obs_stats, 'THETA') GT theta_threshold) then begin
      info_string += '-'
      continue
    endif
    
    event_header = headfits(src_events_fn, EXT=1, ERRMSG=error)
    if keyword_set(error) then begin
      info_string += '-'
      print, 'ERROR:  Could not read '+src_events_fn
      continue
    endif

    
    ; Compute parameters of a ds9 "panda" region depicting the hook.
    ; The feature is at a Chandra position angle of 195 +- 25 degrees.
    ; The angle parameter (theta) of ds9 "panda" regions is defined as counter-clockwise from the +X axis.
    ; The ds9 angle "theta" is related to the position angle (PA) by : theta = (90 + PA - roll) MOD 360
    roll = psb_xpar( event_header, 'ROLL_NOM', COUNT=count)
    if (count EQ 0) then begin
      info_string += '-'
      print, 'ERROR: keyword ROLL_NOM not found in '+src_events_fn
      continue
    endif
    
    theta1 = (90.0 + 195.0 - 25.0 - roll) MOD 360
    theta2 = (90.0 + 195.0 + 25.0 - roll) MOD 360
  
    rmin = 0.6 ; arcseconds
    rmax = 1.0 ; arcseconds

    ; Estimate the number of hook counts this ObsID produces.
    singleObsID_NET_CNTS = psb_xpar( obs_stats, 'SRC_CNTS') - (psb_xpar( obs_stats, 'BKG_CNTS') / psb_xpar( obs_stats, 'BACKSCAL'))
    
    singleObsID_counts_in_PSF  = singleObsID_NET_CNTS / psb_xpar( obs_stats, 'PSF_FRAC')

    singleObsID_counts_in_hook = hook_power_fraction * singleObsID_counts_in_PSF

    info_string += string(round(singleObsID_counts_in_hook), F='(%"%d")')
   
    ; Omit region for very weak hooks.
    if (round(singleObsID_counts_in_hook) LT 1) then continue
    
    
    hook_region = string(RA[ii], DEC[ii], theta1, theta2, rmin, rmax, strtrim(collated_table[ii].LABEL,2), $
                         psb_xpar( obs_stats, 'OBSNAME'), color, round(singleObsID_counts_in_hook), $
                         F='(%"J2000;panda  %10.6f %10.6f  %0.1f %0.1f 1 %0.1f\" %0.1f\" 1 # tag={hook panda} tag={%s} tag={%s} color={%s} text={%d ct} font=\"helvetica 10 normal\" ")' )

    printf, region1_unit, hook_region
    
   ;; Also save the hook region to a file in the obsdir. 
   ;; It's tempting to save to extract.reg, but that file is sometimes write-protected by the observer in order to protect a hand-drawn aperture.
   openw,  region2_unit, hook_region_fn, /GET_LUN
   printf, region2_unit, hook_region
   free_lun, region2_unit
    
    src_generates_bright_hook[ii] = 1B
  endfor ;; jj

  if ~src_generates_bright_hook[ii] then continue
  
  info_string += ' ct'

  ;printf, region1_unit, RA[ii], DEC[ii] + (1.5/3600), 
  ;printf, region1_unit, RA[ii], DEC[ii] - (0.25/3600), info_string, strtrim(collated_table[ii].LABEL,2), color,
  
  printf, region1_unit, RA[ii], DEC[ii] - (0.25/3600), round(multiObsID_counts_in_hook[ii]), strtrim(collated_table[ii].LABEL,2), color, F='(%"J2000;text   %10.6f %10.6f # text={%dct} tag={multi-ObsID hook counts} tag={%s} color={%s} ")' 
  
  
  ; Mark nearby (separation LT 1x panda outer diameter) sources as susceptible to the hook.
  gcirc, 2, RA, DEC, RA[ii], DEC[ii], distance
  distance[ii] = !VALUES.F_NAN
  
  ind = where(distance LT (rmax * 1.0), count)
  if (count GT 0) then src_is_near_bright_hook[ind] = 1
endfor ; ii 
free_lun, region1_unit

;; Write a list of the sources for which we've generated a region, sorted by multiObsID_counts_in_hook
ind = where(src_generates_bright_hook, count)
if (count GT 0) then begin
  ind = ind[reverse(sort(multiObsID_counts_in_hook[ind]))]
  
  forprint, /NoCom, TEXTOUT=bright_srclist_fn, SUBSET=ind, collated_table.CATALOG_NAME, multiObsID_counts_in_hook, F='(%"%s ; %d ct in hook")'
  
  print, count, theta_threshold, hook_cnts_threshold, ENERG_LO, ENERG_HI, bright_srclist_fn, region_fn, F='(%"%d sources with THETA_LO < %0.1f arcmin expected to produce >%d hook counts (%0.1f--%0.1f keV) are listed in %s; their hook regions are in %s")'
endif else file_copy, '/dev/null', bright_srclist_fn

;; Write a list of the sources that are near bright hooks.
ind = where(src_is_near_bright_hook, count)
if (count GT 0) then begin
  forprint, /NoCom, TEXTOUT=near_srclist_fn, SUBSET=ind, collated_table.CATALOG_NAME, collated_table.LABEL, multiObsID_NET_CNTS, collated_table.MERGE_NAME, F='(%"%s ; (%s) %d NET counts in ''%s'' merge")'

  print, count, near_srclist_fn, F='(%"%d sources lying near those hook regions are listed in %s")'
endif else file_copy, '/dev/null', near_srclist_fn

end ; ae_make_psf_hook_regions



;==========================================================================
;;; Script to build exposure-type maps for specified sets of CCDs, binned by 1 skypixel

;;; CCD_LIST is a scalar string specifying the CCDs that should be 
;;; included in the emap, e.g. '012367'.
;;; If not supplied then EXPOSUR? keywords are used to define the CCD list.

;;; SCALING_PARAMS is a single or array of structures with the following fields:
;;;   map_filename            : path to store map (string)
;;;   units                   : physical units of map (string)
;;;   normalize_before_scaling: boolean; If T, then single-ObsID maps are normalized to 1 before scaling.
;;;   ccd_scaling             : float array of 10 single-ObsID scaling values

;;; If /ONLYTIME specified, then map has no effective area information.
;;; If  ONLYTIME omitted or ONLYTIME=0, then effective area at MONOENERGY (keV) is included.

;;; If the path to an existing image is passed in MATCHFILE, then the pixel grid of the maps will match that file.
;;; Otherwise, the pixel grid will span the CCDs specified, with a map pixel size of 1 skypix.

;;; Aspect histograms will be written to "asphist/".
;;; Instrument maps will be written to instmap/. 

;==========================================================================
PRO ae_make_emap, obsdata_filename,$
                  ARDLIB_FILENAME=ardlib_fn, ASPECT_FN=aspect_fn, PBKFILE=pbk_fn, MASKFILE=mask_fn,$
                  CCD_LIST=ccd_list, SPATIAL_FILTER=spatial_filter,$ 
                  MONOENERGY=monoenergy_p, SPECTRUM_FN=spectrum_fn_p, BAND_NAME=band_name, $
                  ONLYTIME=onlytime, SCALING_PARAMS=scaling_params,$
                  MATCHFILE=matchfile, PROMPT_FOR_MATCHFILE=prompt_for_matchfile, $
                  REUSE_ASPHIST=reuse_asphist,$
                  REUSE_INSTMAP=reuse_instmap,$
                  DISCARD_ASPHIST=discard_asphist,$
                  DISCARD_INSTMAP=discard_instmap,$
                  OUTPUT_DIR=output_dir_p

creator_string = "ae_make_emap, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, F='(%"\n\n===================================================================")'
print, creator_string, F='(%"%s")'
print, now()
print, F='(%"===================================================================")'
exit_code = 0

if ~keyword_set(aspect_fn)       then aspect_fn  = 'acis.astrometric.asol1'
if ~keyword_set(pbk_fn)          then pbk_fn     = 'acis.pbk0'
if ~keyword_set(mask_fn)         then mask_fn    = 'acis.msk1'

if keyword_set(onlytime) then begin
  if keyword_set( monoenergy_p) && finite(monoenergy_p)      then print,  monoenergy_p, F='(%"WARNING: /ONLYTIME makes MONOENERGY (%0.2f) irrelevant.")'
  if keyword_set(spectrum_fn_p) && (spectrum_fn_p NE 'NONE') then print, spectrum_fn_p, F='(%"WARNING: /ONLYTIME makes SPECTRUM_FN (%s) irrelevant.")'

  monoenergy = 10 ; keV  (We need *something* to pass in "monoenergy" parameter to mkinstmap.)
  spectrum_fn= 'NONE'
endif else begin
  monoenergy = keyword_set( monoenergy_p) ?  monoenergy_p : 1.0 ; keV
  spectrum_fn= keyword_set(spectrum_fn_p) ? spectrum_fn_p : 'NONE'
endelse

if keyword_set(output_dir_p) then begin
   output_dir = output_dir_p + '/'
   file_mkdir,  output_dir
endif else      output_dir = './'

if ~keyword_set(band_name) then band_name=''

;; ------------------------------------------------------------------------
;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

run_command, PARAM_DIR=tempdir, CIAO_VERSION=ciao_version

if version_string_compare(ciao_version, '4.6', /LESSTHAN) then begin
  pbk_parameter = 'pbkfile='+pbk_fn
endif else begin
  ; CIAO 4.6+ no longer wants a "pbkfile" input to mkinstmap.
  ; As of CIAO 4.10, the pbkfile input is still required, so we set it to NONE.
  pbk_parameter = 'pbkfile=NONE'
endelse

template_fn = tempdir + 'template.img'
fov_fn      = tempdir + 'fov.fits'
temp_events_fn = tempdir + 'temp.evt'

;; ------------------------------------------------------------------------
;; Manage directories for aspect histograms, instrument/time maps, and single-ObsID emaps.
asphist_dir  = 'asphist'
instmap_dir  = 'instmap'
ccd_emap_dir = 'emap'
file_mkdir, asphist_dir
file_mkdir, instmap_dir
file_mkdir, ccd_emap_dir

ccd_label   = string(indgen(10), F='(%"/ccd%d.")')
asphist_fn  =  asphist_dir + ccd_label + 'asphist'

; Time Maps (with no effective area information) are independent of energy.
timemap_fn  =  instmap_dir + ccd_label + 'timemap'
; Instrument Maps (with effective area information) are named by the monoenergy used, not by band_name, so that
; one Instrument Map can serve multiple bands that specify the same monoenergy.
areamap_fn  =  instmap_dir + ccd_label +string(monoenergy*1000,F='(%"%deV.instmap")')
; The generic "instmap_fn" is either timemap names or areamap names (for coding convenience).
instmap_fn = keyword_set(onlytime) ? timemap_fn : areamap_fn

; Single-CCD emaps are similarly named by the monoenergy used.
       ccd_emap_fn = ccd_emap_dir + ccd_label +string(monoenergy*1000,F='(%"%deV.emap")')
reproj_ccd_emap_fn =      tempdir + ccd_label +string(monoenergy*1000,F='(%"%deV.reproj.emap")')
; Reprojected single-CCD emaps cannot be re-used (and are thus not saved) because they depend on the scene, and scenes are defined empirically rather than via a specification.


;; Remove ALL existing aspect histograms and/or instrument maps, as specified by the caller.
;; In both cases, discard single-CCD emaps (which depend on both aspect histograms and instrument maps).
;; We do NOT remove the directories (asphist_dir,instmap_dir,ccd_emap_dir) because the user may have files there.
if  keyword_set(discard_asphist) then begin
  list = file_search(asphist_dir+'/*.asphist',COUNT=count)
  if (count GT 0) then file_delete, list, /VERBOSE
  
  list = file_search(ccd_emap_dir+'/*.emap',COUNT=count)
  if (count GT 0) then file_delete, list, /VERBOSE
endif

if  keyword_set(discard_instmap) then begin
  ; Remove instrument map and time map files, but not instmap_dir (which could contain other files the user wants).
  list = file_search(instmap_dir +'/*.instmap',COUNT=count)
  if (count GT 0) then file_delete, list, /VERBOSE

  list = file_search(instmap_dir +'/*.timemap',COUNT=count)
  if (count GT 0) then file_delete, list, /VERBOSE
  
  list = file_search(ccd_emap_dir+'/*.emap',COUNT=count)
  if (count GT 0) then file_delete, list, /VERBOSE
endif

if keyword_set(discard_asphist) || keyword_set(discard_instmap) then return


;; ------------------------------------------------------------------------
;; The ARDLIB_FN parameter must be passed.
if ~keyword_set(ardlib_fn) then begin
  print, 'ERROR: you must supply the path to your observation-specific ardlib.par file via ARDLIB_FILENAME'
  GOTO, FAILURE
endif

if ~file_test(ardlib_fn) then begin
  print, ardlib_fn, F='(%"\nERROR: ARDLIB_FILENAME %s not found.")'
  GOTO, FAILURE
endif



if keyword_set(reuse_asphist) then print, F='(%"\nWARNING! Using previously computed aspect histograms if available.")'
if keyword_set(reuse_instmap) then print, F='(%"\nWARNING! Using previously computed instrument maps if available.\n")'



;; ------------------------------------------------------------------------
; Determine which CCDs need to be processed.
theader = headfits(obsdata_filename, EXT=1, ERRMSG=error )
if (keyword_set(error)) then begin
  print, error
  print, 'ERROR reading ' + obsdata_filename
  GOTO, FAILURE
endif

; Determine which CCDs are present in the event list provided.
; We do not trust the DETNAM keyword because the observer may have applied a CCD_ID filter.
ccd_is_in_observation = bytarr(10)
for ccd_id = 0,9 do begin
  keyname    = string(ccd_id, F='(%"EXPOSUR%d")')
  exposure   = psb_xpar( theader, keyname, COUNT=count)
                 
  if (count EQ 0) || (exposure LE 0)  then continue
  ccd_is_in_observation[ccd_id] = 1B
endfor ;ccd_id

; Allow the caller to specify a subset of those CCDs to retain.
case n_elements(ccd_list) of
  0: ccd_list = strjoin(strtrim(string(where(ccd_is_in_observation)),2))
  1:  begin
      ; Retain only those CCDs that are in this observation.
      requested_list = ccd_list
      validated_list = ''
      for ii=0,strlen(requested_list)-1 do begin
        ccd_id = strmid(requested_list,ii,1)
        if ccd_is_in_observation[ccd_id] then validated_list += ccd_id
      endfor ; ii
      ccd_list = validated_list

      if (ccd_list EQ '') then begin
        print, requested_list, F='(%"None of the requested CCDs (%s) were found in the observation!")'
        GOTO, FAILURE
      endif
      end
  else: begin
        print, 'CCD_LIST input must be a scalar string!'
        help, ccd_list
        GOTO, FAILURE
        end
endcase

; Convert the list of CCDs in play from a string of digits (ccd_list) to an array of integers (active_ccds).
flag = bytarr(10)
for ii=0,strlen(ccd_list)-1 do flag[fix(strmid(ccd_list,ii,1))] = 1
active_ccds = where(flag, num_active_ccds)
if (num_active_ccds EQ 0) then begin
  print, F='(%"ERROR: No active CCDs identified; seems like a code bug ...")'
  help , ccd_list
  print, ccd_list
  stop
endif

print, active_ccds, F='(%"Your emaps involve the following CCDs: %d %d %d %d %d %d %d %d ")'



;; ------------------------------------------------------------------------
;; Create aspect histograms, instmaps, and single-CCD exposure maps for all active CCDs.

run_command, /QUIET, ['punlearn asphist','punlearn mkinstmap','punlearn mkexpmap','punlearn dmimgcalc','punlearn dmhedit', 'punlearn skyfov', 'pset skyfov clobber=yes']

run_command, string(obsdata_filename, aspect_fn, mask_fn, fov_fn, F="(%'skyfov %s kernel=fits aspect=%s mskfile=%s outfile=%s ')")

bt_fov = mrdfits(fov_fn,1,/SILENT)

f_infinity = !VALUES.F_INFINITY
scene_xymin =  [f_infinity,f_infinity]
scene_xymax = -[f_infinity,f_infinity]

emap_total = fltarr(10)

for ii=0,num_active_ccds-1 do begin
  ccd_id = active_ccds[ii]

  ;; ------------------------------------------------------------------------
  ;; Verify that BADPIX_FILE entry in ardlib.par is a readable file.
  keyname    = string(   ccd_id, F='(%"AXAF_ACIS%d_BADPIX_FILE")')
  while 1 do begin
    ; Copy observer-supplied ardlib.par to tempdir, where the pget command can find it (when spawned by run_commmand).
    file_copy, /OVERWRITE, /FORCE, ardlib_fn, tempdir + 'ardlib.par'
    run_command, string(  keyname, F='(%"pget -abort ardlib %s")'), badpix_fn    , /QUIET
    if (strtrim(badpix_fn,2) EQ 'CALDB') then break

    run_command, string(badpix_fn, F="(%'dmlist ""%s"" block')"  ), STATUS=status, /QUIET
    if (status EQ 0) then break
    
    print, ccd_id, badpix_fn, keyname, ardlib_fn, F='(%"ERROR: AE cannot find the observation-specific bad-pixel table for CCD%d: ''%s''\nwhich is obtained from parameter %s in %s.\n")'
    print, 'Investigate the failure.  If you can fix the problem and want AE to re-try then type ".continue"; if you want to abort the AE run type "retall".'
    stop
  endwhile

  ;; ------------------------------------------------------------------------
  ;; Remove existing files that are not approved for re-use.
  ;; Single-CCD emaps depend on both aspect histograms and instrument maps.
  if ~keyword_set(reuse_asphist) then file_delete, /ALLOW_NONEXISTENT, [asphist_fn[ccd_id], ccd_emap_fn[ccd_id]]
  if ~keyword_set(reuse_instmap) then file_delete, /ALLOW_NONEXISTENT, [instmap_fn[ccd_id], ccd_emap_fn[ccd_id]]
  
  ;; ------------------------------------------------------------------------
  ;; Create an aspect histogram file.
  if ~file_test(asphist_fn[ccd_id]) then begin
    run_command, string(aspect_fn, obsdata_filename, ccd_id, asphist_fn[ccd_id], $
                        F="(%'asphist infile=%s evtfile=""%s[ccd_id=%d]"" outfile=%s dtffile="""" clob+')")  
  endif

  bt = mrdfits(asphist_fn[ccd_id], 1, /SILENT, STATUS=status)
  if (status NE 0) then message, 'ERROR reading ' + asphist_fn[ccd_id]

  duration = total(bt.duration)
  
  keyname  = string(ccd_id, F='(%"EXPOSUR%d")')
  exposure = psb_xpar( theader, keyname)
  grating  = psb_xpar( theader, 'GRATING')

  if (abs(duration-exposure)/duration GT 0.01) then begin
    print, duration, asphist_fn[ccd_id], keyname, exposure, $
           F='(%"WARNING: Sum of DURATION column (%d) in %s does not match value of keyword %s (%d)!")'
    print, 'It is likely that your aspect histogram was not computed correctly!!!!!'
    wait, 1
  endif

  ;; ------------------------------------------------------------------------
  ;; Create instrument map.
  if ~file_test(instmap_fn[ccd_id]) then begin
;NOTE THAT THE PROPER VALUE FOR THE BPMASK QUALIFIER IS NOT YET SETTLED.
;We need Bit17 set for framestore shadow rows to be declared bad, i.e.
;BPMASK=FAINT        (=0x3f9ff, 260607)
;BPMASK=VFAINT       (=0x3ffff, 262143)
;Binary value: 11   11111001 11111111
;Binary value: 11   11111111 11111111
;Bit number  :  16 15      8 7      0
;
;The question of FAINT vs VFAINT hinges on the utility of Bit9, Bit10 (counting from zero), defined in https://space.mit.edu/CXC/docs/memo_bpix_status_bits_1.12.pdf.  That document describes both bits as "obsolete", and defines them as:
;
;Bit9: For VFAINT mode observations, this bit is used to identify the rows and columns that are immediately adjacent to the outer edge of a CCD. (i.e. CHIPX = 2 and 1023 and CHIPY = 2 and 1023). No events can be reported for these rows. See Table 2.
;
;Bit10: This bit is used only for VFAINT mode observations. It is used in a similar fashion as bit 8. For a bad pixel (or a pixel in a bad column), this bit is set to one for the sixteen pixels that surround the eight pixels for which bit 8 is set to one. See Table 2.
;
; Empirically (for one ObsID) I find that both these values of BPMASK produce identical instmaps.
; HelpDesk Ticket #023119 (2020 Dec 15) discusses these issues, and confirms that Bit9 and Bit10 mentioned above became obsolete in 2005, and that BPMASK=FAINT and BPMASK=VFAINT are equivalent.
;
; As of 2021 Nov 30 the CXC recommends BPMASK=0x03ffff (https://cxc.cfa.harvard.edu/ciao4.14/caveats/acis_shadow_badpix.html)

    if keyword_set(onlytime) then begin 
      run_command, string(obsdata_filename, pbk_parameter, instmap_fn[ccd_id], ccd_id, monoenergy, mask_fn, grating, $
                          F="(%'mkinstmap obsfile=""%s[EVENTS]"" %s outfile=%s detsubsys=""ACIS-%d;IDEAL;BPMASK=0X03FFFF"" mirror=""HRMA;AREA=1"" dafile=NONE pixelgrid=""1:1024:#1024,1:1024:#1024"" spectrumfile=NONE monoenergy=%f maskfile=%s grating=%s verbose=0 clob+')")  
    endif else begin
      run_command, string(obsdata_filename, pbk_parameter, instmap_fn[ccd_id], ccd_id, spectrum_fn, monoenergy, mask_fn, grating, $
                          F="(%'mkinstmap obsfile=""%s[EVENTS]"" %s outfile=%s detsubsys=""ACIS-%d;BPMASK=0X03FFFF"" pixelgrid=""1:1024:#1024,1:1024:#1024"" spectrumfile=%s monoenergy=%f maskfile=%s grating=%s verbose=0 clob+')")  

      if (monoenergy EQ 1.0) then begin
        ; Check (defensively) whether any observation or stowed events came from CCD pixels that are zero in the instmap.
        regfile1 =  string(ccd_id, F='(%"ccd%d_inconsistent.reg")')
        openw,  region1_unit, regfile1, /GET_LUN
        printf, region1_unit, F='(%"# Region file format: DS9 version 3.0 \nglobal color=DodgerBlue font=\"helvetica 12 normal\" \nimage")'

        config = replicate({event_fn:'', tag:'', color:''},2)
        config.event_fn = ['acis.spectral.evt2','acis.stowed.evt2']
        config.tag      = ['observation','stowed']
        config.color    = ['red','green']
        
        foreach this, config do begin
          fmt = '(%"'+string(this.tag, this.color, F='(%"cross point \%d \%d # tag={%s} color={%s}")')+'")'

          run_command, string(this.event_fn, ccd_id, instmap_fn[ccd_id], temp_events_fn, $
                        F="(%'dmimgpick ""%s[ccd_id=%d][cols CHIP]"" %s %s method=closest clobber+')")
          this_bt = mrdfits(temp_events_fn, 1, /SILENT)

          ind = where(this_bt.(2) EQ 0, num_inconsistent)
          if (num_inconsistent GT 0) then begin
            !TEXTUNIT = region1_unit
            forprint, TEXTOUT=5, /NoCOM, SUBSET=ind, this_bt.CHIPX, this_bt.CHIPY, F=fmt
            !TEXTUNIT = 0
          endif
          
          inconsistent_percentage = 100 * num_inconsistent / float(n_elements(this_bt))
          prefix = (inconsistent_percentage GT 0.2) ? 'WARNING: ' : 'INFORMATION: '
          print, prefix, inconsistent_percentage, this.event_fn, ccd_id, instmap_fn[ccd_id], $
              F='(%"%s %0.2f%% of events in %s come from CCD%d pixels that are zero in %s.")'
        endforeach
        free_lun, region1_unit
        file_gzip, /DELETE, regfile1
      end ; 1keV
    endelse
  endif ; ~file_test(instmap_fn[ccd_id])


  ;; ------------------------------------------------------------------------
  ;; Create single-ObsID emap.
  
  ;; Determine span of this CCD on the sky.
  row = bt_fov[where(/NULL, (bt_fov.CCD_ID EQ ccd_id) AND (bt_fov.SHAPE).Contains('polygon', /FOLD_CASE ))]
  xymin = floor([min(row.X,/NAN),min(row.Y,/NAN)])
  xymax = ceil ([max(row.X,/NAN),max(row.Y,/NAN)])
  
  num_x_pixels = xymax[0] - xymin[0]
  num_y_pixels = xymax[1] - xymin[1]
  
  xygrid = string(xymin[0], xymax[0], num_x_pixels, xymin[1], xymax[1], num_y_pixels, F="(%'%d:%d:#%d,%d:%d:#%d')")
    
  if ~file_test(ccd_emap_fn[ccd_id]) then begin
    ; The normalize=no option in mkexpmap multiplies the instmap by EXPOSUR? value, e.g. so that a normal emap
    ; map has units of s cm**2 count /photon, which is what AE requires.
    ; The normalize=yes option would ignore the CCD's EXPOSUR? value, and the emaps units would be cm**2 count /photon.
    run_command, string(instmap_fn[ccd_id], ccd_emap_fn[ccd_id], asphist_fn[ccd_id], xygrid, $
                      F="(%'mkexpmap instmapfile=%s outfile=%s asphistfile=%s xygrid=""%s""  normalize=no useavgaspect=no verbose=0 clob+')")
  endif
  
  ; Record the sum of all the pixels in each map, in case "SCALINE_PARAMS.normalize_before_scaling" is specified.
  emap_total[ccd_id] = total(/DOUBLE, /NAN, readfits(ccd_emap_fn[ccd_id]))
  
  scene_xymin = scene_xymin < xymin
  scene_xymax = scene_xymax > xymax
endfor ;ii



;; ------------------------------------------------------------------------
;; Create multi-CCD exposure maps.

  ;; ------------------------------------------------------------------------
  ;; Define the scene (field of view of the map) and build a "template image" on that scene.
  if keyword_set(matchfile) then begin
    ; The caller may have passed such a template image.
    template_fn = matchfile
    print, matchfile, F='(%"\nae_make_emap: Pixel grid of maps will match %s")'
    
  endif else if keyword_set(prompt_for_matchfile) then begin
    ; Prompt observer for the path to a template image.
    template_fn = dialog_pickfile( PATH='AE/', FILTER=['*.full.emap','*.img'], TITLE='Select a FITS image to define X/Y grid for maps.' ) 

  endif else begin
    ; We define the scene to cover the single-CCD emaps built above.
    ; LL corner is scene_xymin
    ; UR corner is scene_xymax
    
    num_x_pixels = scene_xymax[0] - scene_xymin[0]
    num_y_pixels = scene_xymax[1] - scene_xymin[1]
    
    binspec = string(scene_xymin[0], scene_xymax[0], num_x_pixels, scene_xymin[1], scene_xymax[1], num_y_pixels, F="(%'X=%d:%d:#%d,Y=%d:%d:#%d')")

    run_command, string(obsdata_filename, binspec, template_fn, F="(%'dmcopy ""%s[#row=1:10][bin %s]"" %s')")
  endelse

  ;; ------------------------------------------------------------------------
  ;; Reproject the single-CCD emaps to the scene.
  if ~keyword_set(resolution) then resolution = 1
  for ii=0,num_active_ccds-1 do begin
    ccd_id = active_ccds[ii]
  
    run_command, string(ccd_emap_fn[ccd_id], template_fn, reproj_ccd_emap_fn[ccd_id], resolution, $
                        F="(%'reproject_image  infile=""%s""  matchfile=%s  outfile=%s  method=average resolution=%d')")
  endfor ; ii



  ;; ------------------------------------------------------------------------
  ;; Now combine the (reprojected) single-obsid exposure maps, scaling each as specified.
  for jj=0,n_elements(scaling_params)-1 do begin
    this_param = scaling_params[jj]
    
    ; An empty filename is a flag to ignore this set of parameters.
    if ~this_param.map_filename then continue

    if this_param.normalize_before_scaling then this_param.ccd_scaling /= emap_total

    scaled_formula = strjoin(string(this_param.ccd_scaling[active_ccds],F="(%'(%0.4g*')")+string(1+indgen(num_active_ccds),F="(%'img%d)')"), '+')
    
    infile_stack   = strjoin(reproj_ccd_emap_fn[active_ccds],",")
    
    ; We hide the final map file in a scratch directory until we're finished editing it!
    temp_outfile =    tempdir+this_param.map_filename
    outfile      = output_dir+this_param.map_filename
    
    
    run_command, string(infile_stack, temp_outfile, scaled_formula, $
                        F="(%'dmimgcalc infile=""%s"" infile2=none outfile=%s operation=""imgout=(float)(%s)"" lookupTab=none verbose=1 ')")          
                      
    ; Record the HDUNAME, UNITS, and energy band in the map..                    
    cmd1 = string(temp_outfile, this_param.map_type, $
               F="(%'dmhedit infile=""%s"" filelist=none operation=add key=HDUNAME value=""%s"" comment=""map type""')")
    cmd2 = string(temp_outfile, this_param.units, $
               F="(%'dmhedit infile=""%s"" filelist=none operation=add key=BUNIT   value=""\'%s\'"" comment=""map units""')")
    
    if ~keyword_set(onlytime) then begin 
      cmd3 = string(temp_outfile, monoenergy, $
                   F="(%'dmhedit infile=""%s"" filelist=none operation=add key=ENERGY  value=""%0.2f"" comment=""[keV] calibration mono-energy""')")
    endif else cmd3=''
    
    run_command, [cmd1,cmd2,cmd3], /QUIET
    
    ; Report the pixel size of the emap.
    emap_header = headfits(temp_outfile)
    
    print, this_param.map_filename, psb_xpar( emap_header,'CDELT1P'), psb_xpar( emap_header,'CDELT2P'), F="(%'\nThe pixels in exposure map %s are %0.5f X %0.5f skypixel.')"
    
    ; Delete the output file because it could be a symlink.
    file_delete, outfile, /ALLOW_NONEXISTENT 
    ; Apply any spatial filter specified by caller.
    ; Reveal the completed map file by moving to final location.
    if keyword_set(spatial_filter) then $
      run_command, string(temp_outfile, spatial_filter, outfile, F="(%'dmcopy ""%s[%s][opt full]"" %s ')") $
    else $
      file_move,          temp_outfile,                 outfile, OVERWRITE=0 
      ; OVERWRITE=0 to check consistency of file_delete above.
  endfor ;jj

CLEANUP:
if file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then return $
else begin
  print, 'ae_make_emap: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end ; ae_make_emap



;==========================================================================
;;; Tool to create lightcurves suitable for identifying background flares.

;;; Images are made for each CCD in the specified set of CCDs (e.g. CCD_LIST='01236').

;;; Bright pixels (corresponding to sources, which might flare themselves) are masked.

;;; Single-CCD lightcurves are made for the unmasked regions.

;;; Single-CCD lightcurves, and an average lightcurve are plotted.

;;; Lightcurves use heavily-cleaned data (STATUS=0), limited to 0.5:7 keV range.

;==========================================================================
PRO ae_show_flares, obsdata_filename,  CCD_LIST=ccd_list, CCDS_TO_SUM=ccds_to_sum, BINSIZE=binsize, $
    OUTPUT_DIR=output_dir, TEMPLATE_FN=template_fn, SKIP_DS9=skip_ds9

creator_string = "ae_show_flares, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()
exit_code = 0

if ~keyword_set(binsize)    then binsize  = 200
if  keyword_set(output_dir) then begin
    file_mkdir, output_dir
endif else      output_dir = '.'

;; ------------------------------------------------------------------------
;; Create a unique scratch directory.
tempdir = temporary_directory( 'AE.', VERBOSE=1, SESSION_NAME=session_name)

temp_events1_fn   = tempdir + 'temp.evt'
temp_events2_fn   = tempdir + 'temp.evt'

run_command, PARAM_DIR=tempdir

base          = tempdir + string(indgen(10), F='(%"ccd%d.")')
obs_event_fn  = base + 'evt'
mask_fn       = base + 'mask'
bkg_event_fn  = base + 'bkg'
lightcurve_fn = output_dir+string(indgen(10), F='(%"/ccd%d.lc")')


;; ------------------------------------------------------------------------
; Convert the strings CCD_LIST and CCDS_TO_SUM supplied by caller into vectors of CCD indexes.
; Ignore CCDs not observed.
theader = headfits(obsdata_filename, EXT=1, ERRMSG=error )
if (keyword_set(error)) then begin
  print, error
  print, 'ERROR reading ' + obsdata_filename
  GOTO, FAILURE
endif

title = strtrim(psb_xpar( theader,'OBJECT'),2) +' ('+ strtrim(psb_xpar( theader,'OBS_ID'),2) +')'


; Determine which CCDs are present in the event list provided.
; We do not trust the DETNAM keyword because the observer may have applied a CCD_ID filter.
ccd_is_in_observation = bytarr(10)
for ccd_id = 0,9 do begin
  keyname    = string(ccd_id, F='(%"EXPOSUR%d")')
  exposure   = psb_xpar( theader, keyname, COUNT=count)
                 
  if (count EQ 0) || (exposure LE 0)  then continue
  ccd_is_in_observation[ccd_id] = 1B
endfor ;ccd_id

active_ccds = where(ccd_is_in_observation, num_active_ccds)


if n_elements(ccd_list) EQ 0 then $
  ccd_list = strjoin(strtrim(string(active_ccds),2))

if n_elements(ccds_to_sum) EQ 0 then ccds_to_sum = '01234689'

flag = bytarr(10)
for ii=0,strlen(ccds_to_sum)-1 do begin
  ccd_id = strmid(ccds_to_sum,ii,1)
  if ccd_is_in_observation[ccd_id] then flag[fix(ccd_id)] = 1
endfor
averaged_ccds = where(flag, num_averaged_ccds)


;; ------------------------------------------------------------------------
;; Get rid of pre-existing configuration for the CIAO commands we'll use below.
run_command, /QUIET, ['punlearn dmcopy', 'pset dmcopy clobber=yes', 'punlearn dmimgpick', 'pset dmimgpick clobber=yes', 'punlearn dmextract', 'pset dmextract clobber=yes', 'punlearn dmmerge', 'pset dmmerge clobber=yes']


;; Process each CCD.
for ii=0,num_active_ccds-1 do begin
  ccd_id = active_ccds[ii]
    
  run_command, string(obsdata_filename, ccd_id, obs_event_fn[ccd_id], F="(%'dmcopy ""%s[ccd_id=%d,STATUS=0,energy=500:7000]"" %s')")  
 
  ;; ------------------------------------------------------------------------
  ;; Determine span of the CCD on the sky.
  run_command, string(obs_event_fn[ccd_id], F="(%'dmstat ""%s[cols x,y]"" median=no sigma=no verbose=0')")  
 
  run_command, /QUIET, 'pget dmstat out_min out_max', result
  xymin = floor(float(strsplit(result[0],',', /EXTRACT))) 
  xymax = ceil (float(strsplit(result[1],',', /EXTRACT))) 
  
  ;; ------------------------------------------------------------------------
  ;; Create a data image for this CCD, binned 10x10 to try to get an off-axis star to fall in a few pixels.
  run_command, string(obs_event_fn[ccd_id], xymin[0], xymax[0], 10, xymin[1], xymax[1], 10, mask_fn[ccd_id], F="(%'dmcopy ""%s[bin x=%d:%d:%d,y=%d:%d:%d]"" %s')")  
  
  mask = readfits(mask_fn[ccd_id], header)
  
  ;; ------------------------------------------------------------------------
  ;; Estimate the mean of the image, ignore the off-field pixels.  .
  ind_onfield = where( smooth(float(mask),3) GT 0, count_onfield )
  estimate_poisson_background, mask[ind_onfield], pix_mean, SIGNIFICANCE=0.99, /VERBOSE

  pix_sigma = sqrt(pix_mean)
  
  ;; Mask the bright pixels (corresponding to sources, which might flare). 
  ind = where(mask GT (pix_mean + 3*pix_sigma), count)
  if (count GT 0) then mask[ind] = -1
  writefits, mask_fn[ccd_id], mask, header
  print, (100.0*count)/count_onfield, ccd_id, F='(%"The brightest %4.1f%% of the pixels on CCD %d have been masked.")'
  
  ;; ------------------------------------------------------------------------
  ;; Filter the event list with the mask  
  ;; The image passed to dmimgpick calls below spans the event list, so the bug in HelpDesk Ticket #020605 should not be triggered.
  cmd1 = string(obs_event_fn[ccd_id], mask_fn[ccd_id], temp_events1_fn, $
                F="(%'dmimgpick ""%s[cols time,sky,ccd_id]"" %s %s method=closest')")

  cmd2 = string(temp_events1_fn, bkg_event_fn[ccd_id], F="(%'dmcopy ""%s[#4>-1]"" %s')")
  run_command, [cmd1,cmd2]

  ;; ------------------------------------------------------------------------
  ;; Create light curve using either the supplied template, or using the first CCD's lightcurve as a template.
  if keyword_set(template_fn) then begin 
    binspec = string(template_fn, F='(%"[bin time=grid(%s[cols time_min,time_max])]")')
  endif else begin
    run_command, string(bkg_event_fn[ccd_id], F="(%'dmstat ""%s[cols time]"" median=no sigma=no verbose=0')")  
    run_command, /QUIET, 'pget dmstat out_min out_max', result
    tmin = floor(float(result[0])) 
    tmax = ceil (float(result[1])) 
  
    binspec = string(tmin, tmax, binsize, F="(%'[bin time=%d:%d:%d]')")
    template_fn = lightcurve_fn[ccd_id]
  endelse
  
  run_command, string(bkg_event_fn[ccd_id], binspec, lightcurve_fn[ccd_id], F="(%'dmextract infile=""%s%s"" bkg=none outfile=%s opt=ltc1')")
endfor ;ii


;; ------------------------------------------------------------------------
;; Create an average light curve across the specified CCDs.
if (num_averaged_ccds GT 0) then begin
  average_lightcurve_fn = output_dir+'/ccd'+ccds_to_sum+'.lc'
  
  run_command, string(strjoin(bkg_event_fn[averaged_ccds],','), temp_events2_fn, F="(%'dmmerge ""%s"" columnList="""" outfile=%s')")
  
  run_command, string(temp_events2_fn, binspec, average_lightcurve_fn, F="(%'dmextract infile=""%s%s"" bkg=none outfile=%s opt=ltc1')")
endif

;; ------------------------------------------------------------------------
flush_stdin  ; eat any characters waiting in STDIN, so that they won't be mistaken as commands in the loop below.
print, 'Press RETURN to show cleaned/masked events in ds9 and to show lightcurves  ...'
cmd = ''
read, '? ', cmd

;; ------------------------------------------------------------------------
;; Display the masked data.
if ~keyword_set(skip_ds9) then begin
  run_command, string(strjoin(bkg_event_fn[active_ccds],','), temp_events1_fn, F="(%'dmmerge ""%s"" columnList="""" outfile=%s')")
  
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME=title+'', OPTION_STRING='-linear -bin factor 8'
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, temp_events1_fn
endif

;; Plot the light curves.
print  
for ii=0,num_active_ccds-1 do begin
  ccd_id = active_ccds[ii]
 
  bt = mrdfits(lightcurve_fn[ccd_id], 1, theader, /SILENT)
  
  print, lightcurve_fn[ccd_id], median(100*bt.COUNT_RATE_ERR/bt.COUNT_RATE), F='(%"LC %s has ~%d%% errors.")'
  function_1d, id1, PSYM=1, LINESTYLE=6, DATASET=lightcurve_fn[ccd_id], (binsize/1000.)*bt.TIME_BIN, bt.COUNT_RATE, ERROR=bt.COUNT_RATE_ERR, YTIT='count /s', XTIT='time (ks)', TITLE=title
  
  dataset_1d,  id2,         LINESTYLE=0, DATASET=lightcurve_fn[ccd_id], BINSIZE=0.01,                bt.COUNT_RATE,                          XTIT='count /s', DENSITY_TITLE=title
endfor ;ii

if (num_averaged_ccds GT 0) then begin
  bt = mrdfits(average_lightcurve_fn, 1, /SILENT)
  print, average_lightcurve_fn, median(100*bt.COUNT_RATE_ERR/bt.COUNT_RATE), F='(%"LC %s has ~%d%% errors.")'
                                                                                 
  function_1d, id1, COLOR='yellow', PSYM=1, LINESTYLE=6, DATASET=average_lightcurve_fn, (binsize/1000.)*bt.TIME_BIN, bt.COUNT_RATE, ERROR=bt.COUNT_RATE_ERR    
  
  dataset_1d,  id2, COLOR='yellow',         LINESTYLE=0, DATASET=average_lightcurve_fn, BINSIZE=0.01,                bt.COUNT_RATE
endif

function_1d, id1, DATASET=lightcurve_fn[ccd_id], PS_CONFIG={filename:output_dir+'/lightcurves.ps'     }, /PRINT, LEGEND_STYLE=0

dataset_1d,  id2, DATASET=lightcurve_fn[ccd_id], PS_CONFIG={filename:output_dir+'/count_rate_distr.ps'}, /PRINT


wait, 5
CLEANUP:
if file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then return $
else begin
  print, 'ae_make_emap: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end  ; ae_show_flares


;==========================================================================
;;; Split a source list into randomized segments to support processing on multiple CPUs.
;==========================================================================
PRO ae_split_srclist, num_segments, segment_prefix, SRCLIST_FILENAME=srclist_fn, SCREEN_NAME=screen_name_p,$
                      QUIET=quiet,$
                      ; OUTPUT PARAMETERS
                      SEGMENT_LIST=segment_list

if ~keyword_set(srclist_fn ) then srclist_fn   = 'all.srclist'

screen_name = keyword_set(screen_name_p) ? repchr(screen_name_p,' ','_') : 'ACIS_Extract'

;; Check for common environment errors.
existing_quiet = !QUIET  &  !QUIET = 1
catch, error_code
if (error_code NE 0) then begin
  print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
  retall
endif else resolve_routine, 'astrolib'
catch, /CANCEL
astrolib
; Make sure forprint calls do not block for user input.
!TEXTOUT=2
!QUIET = existing_quiet

readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'

; Trim whitespace and remove blank lines.
sourcename = strtrim(sourcename,2)
ind = where(sourcename NE '', num_sources)

if (num_sources EQ 0) then begin
  print, 'ERROR: no entries read from source list ', srclist_fn
  retall
endif

sourcename = sourcename[ind]
print, num_sources, F='(%"\n%d sources found in catalog.\n")'

; Randomly permute the source list, so that off-axis angle distributions are similar among segments.
sourcename = sourcename[sort(random(num_sources))]


sources_per_segment = ceil(num_sources/float(num_segments)) > 1
num_segments        = ceil(num_sources/float(sources_per_segment))


segment_fn = segment_prefix + string(indgen(num_segments),F='(%".%2.2d.srclist")')

list = file_search(segment_prefix+'.[0-9][0-9].srclist', COUNT=count)
if (count GT 0) then file_delete, /NOEXPAND_PATH, /VERBOSE, list


segment_start = 0L
for ii=0L, num_segments-1 do begin
  segment_end = (segment_start+sources_per_segment-1) < (num_sources-1)
  forprint, TEXTOUT=segment_fn[ii], /NoComm, sourcename[segment_start:segment_end]
  print, 'Wrote '+segment_fn[ii]
  segment_start = segment_end + 1
  if (segment_start GE num_sources) then break
endfor

segment_list = string(indgen(num_segments), F="(%'.%2.2d')")

if ~keyword_set(quiet) then begin
  print, F='(%"\n Use the csh commands below to create a set of \"screen\" windows with $SEGMENT defined in each.\n")'
  
  print, strjoin(segment_list, ' '), F="(%'  setenv SEGMENT_LIST ""%s""')"
  print,                screen_name, F="(%'  setenv SCREEN_NAME  ""%s""')"
  
  SCREEN_ARCH = getenv('SCREEN_ARCH')
  print,             F="(%'  screen -S $SCREEN_NAME -d -m -t $SCREEN_NAME ')"
  print,                  '  foreach segment ($SEGMENT_LIST)'
  print,             F="(%'    screen -S $SCREEN_NAME -X setenv SEGMENT         ${segment} ')"          
  print,             F="(%'    screen -S $SCREEN_NAME -X setenv PROMPT ""segment ${segment} %% "" ')"
  print,SCREEN_ARCH, F="(%'    screen -S $SCREEN_NAME -X screen -t             ""${segment}"" %s')"
  print,                  '    sleep 0.5 '
  print,                  '  end '
  print,             F="(%'  screen -S $SCREEN_NAME -r \n')"
endif

return
end ; ae_split_srclist





;==========================================================================
;;; Define a field of view that covers the specified event list(s):

;;; ae_define_field_of_view, 'pointing_*/obsid_*/archive/primary/acis*evt2.fits.gz',FOV_SCALING=2,/SAVE


;;; Define a zero-rotation pixel grid on the field of view:

;;; ae_define_field_of_view, MHDR_COMMAND=mHdr_command, ARCSEC_PER=3.5

;==========================================================================
PRO ae_define_field_of_view, $
    ; To define a field of view, and optionally save it:
    event_list_pattern, PROMPT_FOR_BOUNDING_BOX=prompt_for_bb, $
    FOV_SCALING=fov_scaling, SAVE=save_it, PROJECT_FOV_FN=project_fov_fn, $
      
    ; To define a zero-rotation pixel grid on the field of view:
    ARCSEC_PER_PIXEL=arcsec_per_pixel, $
    MHDR_COMMAND    =mHdr_command      ; OUTPUT parameter
                             
creator_string = "ae_define_field_of_view, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)

if ~keyword_set(project_fov_fn) then project_fov_fn = 'project_fov.sav'

;; Define a field of view, if requested.
if keyword_set(event_list_pattern) then begin
  if keyword_set(fov_scaling) && keyword_set(prompt_for_bb) then $
    message, 'You can not supply FOV_SCALING and choose /PROMPT_FOR_BOUNDING_BOX.'

  if ~keyword_set(fov_scaling) then fov_scaling = 1.02
  
  ; Create a unique scratch directory.
  temproot = temporary_directory( 'AE.', VERBOSE=0, SESSION_NAME=session_name)
  tempdir = temproot
  run_command, PARAM_DIR=tempdir
  
  filelist = file_search( event_list_pattern, COUNT=num_files )
  
  if (num_files EQ 0) then begin
    print, 'ae_define_field_of_view: ERROR: no event list found'
    retall
  endif
  
  observing_log = replicate({OBS_ID:'', EXPOSURE:!VALUES.F_NAN, decimal_year:!VALUES.F_NAN}, num_files) 
  
  cel_min  =  !values.F_INFINITY
  cel_max  = -!values.F_INFINITY
  for ii=0, num_files-1 do begin
    ; Calculate the ObsID's date, in decimal years.
    theader = headfits(filelist[ii], EXT=1)
    observing_log[ii].OBS_ID = strtrim(psb_xpar( theader,'OBS_ID' ),2)
    observing_log[ii].EXPOSURE      =  psb_xpar( theader,'EXPOSURE')
    
    ; Very old event lists have no keyword recording the observation date in MJD.
    ; Older event lists prior to ~Dec 2019 use the MJD_OBS keyword.
    ; Newer event lists use only the MJD-OBS keyword.
    MJD_of_observation = psb_xpar( theader,'MJD-OBS', COUNT=count )
    if (count EQ 0) then $
    MJD_of_observation = psb_xpar( theader,'MJD_OBS', COUNT=count )
    if (count EQ 1) then begin
      observing_log[ii].decimal_year  = (MJD_of_observation - 50814.0D)/365. + 1998.0
      ; Decimal year 1998.0 corresponds to MJDREF=50814.0D; see https://cxc.harvard.edu/ciao4.2/ahelp/times.html
    endif else begin
      print, filelist[ii], F='(%"\nERROR: the FITS header of %s is missing both MJD_OBS and MJD-OBS keywords! \nCalculate the decimal year of the observation (e.g. 2010.7) by hand, assign to observing_log[ii].decimal_year, and then .continue. ")'
      stop
    endelse

    print, observing_log[ii].OBS_ID, observing_log[ii].decimal_year, round(observing_log[ii].EXPOSURE/1000),$
           F="(%'Finding bounding box for ObsID %s (Y%0.1f, %d ks) ...')"
  
    run_command, string(filelist[ii], F="(%'dmstat ""%s[cols RA,DEC]"" median=no sigma=no verbose=0')")  
  
    ; Use double-precision.
    run_command, /QUIET, 'pget dmstat out_min out_max', result
    cel_min <= double(strsplit(result[0],',', /EXTRACT)) ; degrees
    cel_max >= double(strsplit(result[1],',', /EXTRACT)) ; degrees
  endfor ; ii
  
 ; Since the weighted_epoch value calculated below is so important, we must carefully look for bad "decimal_year" values!
 chandra_mission_range = [1999,2030]
 if array_equal( (observing_log.decimal_year GE chandra_mission_range[0]) AND $
                 (observing_log.decimal_year LE chandra_mission_range[1])     , 1) then begin
   ; Calculate the mean observation date, weighted by exposure time.
   weighted_epoch = total(/DOUBLE, observing_log.EXPOSURE * observing_log.decimal_year) / total(/DOUBLE, observing_log.EXPOSURE)

   if ~finite(weighted_epoch) || $
       (weighted_epoch LT chandra_mission_range[0]) || $
       (weighted_epoch GT chandra_mission_range[1]) then begin
     print, weighted_epoch, chandra_mission_range, F='(%"ERROR: weighted_epoch (%f) is before or after the Chandra mission, [Y%d--Y%d].")'
     stop
   endif
 endif else begin
   print, chandra_mission_range, F='(%"ERROR: dates in MJD_OBS/MJD-OBS keywords are before or after the Chandra mission, [Y%d--Y%d].")'
   forprint, observing_log
   stop
 endelse



  ;; Define the bounding box.
  acis_center_coords    = [ mean([cel_max[0],cel_min[0]]),$
                            mean([cel_max[1],cel_min[1]]) ] ; degrees
  
  acis_bbox_size_arcmin = [ (cel_max[0]-cel_min[0])*cos(acis_center_coords[1]/!RADEG),$
                            (cel_max[1]-cel_min[1]) ] * 60 ; arcmin

  if keyword_set(prompt_for_bb) then begin ; /PROMPT_FOR_BOUNDING_BOX
    ; Ask user to define bounding rectangle.
    fov_scaling = !VALUES.F_NAN
    print, F='(%"\n\nENTER the center coordinates of your field-of-view rectangle, in decimal RA,DEC:")'
    center_coords = dblarr(2)
    read, center_coords
    
    print, F='(%"\n\nENTER the RA and DEC dimensions of your field-of-view rectangle, in arcseconds:")'
    bbox_size_arcsec = fltarr(2)
    read, bbox_size_arcsec
    bbox_size_arcmin = bbox_size_arcsec / 60.
  endif else begin
    center_coords    = acis_center_coords
    bbox_size_arcmin = acis_bbox_size_arcmin * fov_scaling
  endelse


  radius_arcmin     = sqrt( total((bbox_size_arcmin/2.0)^2) ) ; arcmin

  if keyword_set(save_it) then $
    save, /COMPRESS, /VERBOSE, FILE=project_fov_fn, observing_log, weighted_epoch, acis_center_coords, acis_bbox_size_arcmin, center_coords, fov_scaling, bbox_size_arcmin, radius_arcmin 

  ; Build a corresponding region file.
  openw,  region_unit,  mg_streplace(project_fov_fn, '.sav', '.reg'), /GET_LUN
  printf, region_unit, CENTER_COORDS, BBOX_SIZE_ARCMIN, F="(%'fk5;box(%10.6f,%10.6f,%0.2f\',%0.2f\') # tag={project fov} color={DodgerBlue}')"
  free_lun, region_unit

endif else begin
  ;; event_list_pattern NOT passed, so we must try to read from a field of view file.
  if keyword_set(fov_scaling) then begin
    print, 'ae_define_field_of_view: ERROR: cannot change FOV_SCALING without passing event_list_pattern.' 
    retall
  endif
  
  
  if ~file_test(project_fov_fn) then begin
    print, 'ae_define_field_of_view: ERROR: cannot find '+project_fov_fn
    retall
  endif
  
  restore, project_fov_fn
endelse

; Since the weighted_epoch value restored above is so important, we must carefully look for bad values!
chandra_mission_range = [1999,2030]
if ~finite(weighted_epoch) || $
   (weighted_epoch LT chandra_mission_range[0]) || $
   (weighted_epoch GT chandra_mission_range[1]) then begin
 print, weighted_epoch, chandra_mission_range, F='(%"ERROR: weighted_epoch (%f) is before or after the Chandra mission, [Y%d--Y%d].")'
 stop
endif

print, weighted_epoch, F='(%"\nWeighted Epoch: Y%0.1f")'
print, center_coords, $
      bbox_size_arcmin *60,     bbox_size_arcmin ,     bbox_size_arcmin /60.,$
  max(bbox_size_arcmin)*60, max(bbox_size_arcmin), max(bbox_size_arcmin)/60.,$
         radius_arcmin *60, radius_arcmin        ,        radius_arcmin /60,$
  F='(%"\nField Center: %0.6f %0.6f (RA,DEC) \n\nBounding Rectangle Size: %d X %d arcsec == %0.2f X %0.2f arcmin == %0.3f X %0.3f degrees \nBounding Square    Size:        %d arcsec ==         %0.2f arcmin ==         %0.3f degrees \nBounding Circle Radius :        %d arcsec ==         %0.2f arcmin ==         %0.3f degrees\n")'


if arg_present(mHdr_command) then begin
  if ~keyword_set(arcsec_per_pixel) then arcsec_per_pixel = 1.0

  ; Note that the coordinate system built by mHdr has zero rotation by default.
  mHdr_command = string( arcsec_per_pixel, bbox_size_arcmin[1]/60., center_coords, bbox_size_arcmin[0]/60.,$
                         F='(%"mHdr -p %0.1f -h %0.3f \"%0.6f %0.6f\" %0.3f region.hdr")' )
  print, mHdr_command
endif ; MHDR_COMMAND returning to caller



end ; ae_define_field_of_view


; Define the field of view of the mosaic as a square region that covers the X-ray data, with a specified margin.

; MJD = JD - 2400000.5



;==========================================================================
;;; Interface to STILTS tool "tapquery" 

;;; TAPVizieR documentation:
;    http://tapvizier.u-strasbg.fr/adql/about.html

;;; ADQL documentation and tips:
;     http://tapvizier.u-strasbg.fr/adql/help.html
;     http://tapvizier.u-strasbg.fr/TAPVizieR/tap/examples

; To discover the columns of a table, modify this example:
;   tapurl='http://wfaudata.roe.ac.uk/ukidssDR8-dsa/TAP'
;   adql  ='SELECT TOP 10 * FROM "reliableGpsPointSource"'
;   run_tapquery, /META, tapurl, adql
; NOTE THAT QUERIES CAN NOT REFERENCE COLUMNS THAT ARE GENERATED ON-THE-FLY BY THE SERVER!!!!

; SELECTION is an optional ADQL boolean expression, appended to the query's WHERE clause.
; /META will print the catalog column descriptions instead of downloading rows.
;==========================================================================
PRO run_tapquery, tapurl, adql_p, SELECTION=selection, outfile, META=meta,$
                  TEMP_DIR=tempdir
                 

run_command, PARAM_DIR=tempdir, /QUIET

adql = adql_p
if keyword_set(selection) then adql += ' AND ('+selection+')'

; Single and double quotes do quite different things in ADQL:
; * single quotes are used for string literals ('ICRS').
; * double quotes are used for what's known as "delimited identifiers", e.g. 
;   quoting table or column names that do not match the normal table/column name syntactical requirements,
;   such as "II/246/out".
;
; Suppose the ADQL query requires both double and single quotes, such as:
;   SELECT *  FROM "II/246/out" as t WHERE 1=CONTAINS(POINT('ICRS', t.raj2000, t.dej2000), BOX('ICRS', 83., 22., 0.01, 0.01))
; 
; Assigning that to an IDL string variable requires using repeated quotes,  
; either for the single quotes in the query:
;   adql='SELECT *  FROM "II/246/out" as t WHERE 1=CONTAINS(POINT(''ICRS'', t.raj2000, t.dej2000), BOX(''ICRS'', 83., 22., 0.01, 0.01))'
;
; OR for the double quotes in the query::
;   adql="SELECT *  FROM ""II/246/out"" as t WHERE 1=CONTAINS(POINT('ICRS', t.raj2000, t.dej2000), BOX('ICRS', 83., 22., 0.01, 0.01))"
;
; The string that must be passed to the shell in the call to tapquery must use fancy quoting to deal with the 
; single and double quotes in the query.
; One option is to have two quoted strings abutted, of the form 'ab"cd"ef'<--next to-->"gh'ij'kl"
;
; Another option is to abut single-quoted strings and escaped single-quote characters, 
; of the form 'ab"cd"efgh'\''ij'\''kl'
; This is achieved by replacing each ' character in the query with four characters: '\''
; For example:
; adql='SELECT *  FROM "II/246/out" as t WHERE 1=CONTAINS(POINT('\''ICRS'\'', t.raj2000, t.dej2000 ) ,BOX ('\''ICRS'\'', 83., 22., 0.01, 0.01))'
; The function quote_for_shell() does this for us here.

print, now(), tapurl, adql, F='(%"\nrun_tapquery: %s Running TAP Query with\nServer: %s\nQuery: %s")'

case 1 of 
 keyword_set(meta):$
   cmd = string( tapurl, quote_for_shell(adql),$
                 F='(%"stilts tapquery omode=meta parse=true sync=false maxrec=1000000 compress=true  tapurl=''%s'' adql=%s ")') 

 else:$
   cmd = string( outfile, tapurl, quote_for_shell(adql),$
                 F='(%"stilts tapquery omode=out out=''%s'' ofmt=''(auto)'' parse=true sync=false maxrec=1000000 compress=true  tapurl=''%s'' adql=%s ")')
endcase

run_command, cmd, result, /QUIET, /NO_RETRY, STATUS=status 

if keyword_set(status) then begin
  print, tapurl, adql, F='(%"\nThe TAP Query failed.  Try running it manually in TopCat:\nServer: %s\nQuery: %s")'
  print, outfile, F='(%"\nIf that is successful, then save the catalog in FITS format to %s and type \".continue\".")'
  stop
endif

print
forprint, result

;OTHER INTERESTING PARAMETERS of tapquery:
;http://www.star.bris.ac.uk/~mbt/stilts/sun256/tapquery-usage.html
;
;    stilts <stilts-flags> tapquery 
;                                  ocmd=<cmds>
;                                  omode=out|meta|stats|count
;                                  destruction=<iso8601>
;                                  executionduration=<seconds>
end ; run_tapquery




;WORKS:
;stilts tapquery \
;       tapurl='http://dc.zah.uni-heidelberg.de/__system__/tap/run/tap' \
;       adql="SELECT * \
;                FROM twomass.data AS t \
;                WHERE 1=CONTAINS(POINT('ICRS', t.RAJ2000, t.DEJ2000), \
;                              CIRCLE('ICRS', 83.633083, 22.0145, 5./3600.))" \
; parse=true \
; sync=false maxrec=100 \
; compress=true \
; progress=true \
; omode=out \
; out=test.fits ofmt='(auto)' 
;
;stilts tapquery \
; omode=out \
; out=test.fits ofmt='(auto)' \
; parse=true \
; sync=false maxrec=100 \
; compress=true \
; progress=true \
; tapurl='http://tapvizier.u-strasbg.fr/TAPVizieR/tap' \
; adql='SELECT * \
; FROM "II/246/out" ' 
;
;stilts tapquery \
; omode=out \
; out=test.fits ofmt='(auto)' \
; parse=true \
; sync=false maxrec=100 \
; compress=true \
; progress=true \
; tapurl='http://tapvizier.u-strasbg.fr/TAPVizieR/tap' \
; adql='SELECT *  FROM "II/246/out" ' 
; 
;stilts tapquery \
; omode=out \
; out=test.fits ofmt='(auto)' \
; parse=true \
; sync=false maxrec=100 \
; compress=true \
; progress=true \
; tapurl='http://tapvizier.u-strasbg.fr/TAPVizieR/tap' \
; adql='SELECT *  FROM "II/246/out" AS t  WHERE t.raj2000>0' 
;
;
;stilts tapquery \
; omode=out \
; out=test.fits ofmt='(auto)' \
; parse=true \
; sync=false maxrec=100 \
; compress=true \
; progress=true \
; tapurl='http://tapvizier.u-strasbg.fr/TAPVizieR/tap' \
;  adql='SELECT *  FROM "II/246/out" WHERE'" 1=CONTAINS(POINT('ICRS', 83.,22. ) , CIRCLE ('ICRS', 83., 22., 0.01))"
; 
;
;EXAMPLE:
;stilts tapquery \
; omode=out \
; out=test.fits ofmt='(auto)' \
; parse=true \
; sync=false maxrec=100 \
; compress=true \
; progress=true \
; tapurl='http://tapvizier.u-strasbg.fr/TAPVizieR/tap' \
; adql='SELECT *  FROM "II/246/out" as t \
;       '"WHERE 1=CONTAINS(POINT('ICRS', t.raj2000, t.dej2000 ) , BOX ('ICRS', 83., 22., 0.01, 0.01))"
;




;==========================================================================
;;; Execute a VO "cone query".

;;; RA_DEC is 2-vector (double) containing RA,DEC (degrees) of cone center.
;==========================================================================
PRO run_cone_query, serviceurl, RA_DEC_deg, radius_arcmin, outfile, TEMP_DIR=tempdir

run_command, PARAM_DIR=tempdir, /QUIET

cone_region_spec = string(RA_DEC_deg, radius_arcmin/60., F='(%"&RA=%0.5f&DEC=%0.5f&SR=%0.4f&VERB=2")')

print, now(), serviceurl, cone_region_spec, F='(%"\nrun_cone_query: %s Running Cone Query with\nServer: %s\nCone Spec: %s")'

cmd = string(serviceurl+cone_region_spec, outfile, F='(%"stilts tpipe in=''%s''  out=%s ")')

run_command, cmd, result, /QUIET, /NO_RETRY, STATUS=status 

if keyword_set(status) then begin
  print, serviceurl, RA_DEC_deg[0],RA_DEC_deg[1], radius_arcmin/60., $
    F='(%"\nThe query failed.  Try running Cone Search manually in TopCat:\nServer: %s\nRA=%f (deg)\nDEC=%f (deg)\nRadius=%f (deg)")'
  print, outfile, F='(%"\nIf that is successful, then save the catalog in FITS format to %s and type \".continue\".")'
  stop
endif

print
forprint, result

num_rows = psb_xpar(headfits(outfile, EXT=1, ERRMSG=error), 'NAXIS2')
print, num_rows, serviceurl, outfile, F='(%"\nae_download_project_catalog: Downloaded %d sources from service %s to %s.\n")'
end ; run_cone_query




;==========================================================================
;;; Download a catalog that covers a field of view saved by ae_define_field_of_view
;;; from a VO service that supports the Table Access Protocol (TAP).
;;; TAP services use the ADQL language (an astronomy variant of SQL).
;;;
;;; Example:
;;; Vizier = 'http://tapvizier.u-strasbg.fr/TAPVizieR/tap'
;;; ae_download_project_catalog, TEMP_DIR=tempdir,$
;;;  Vizier, 'II/246/out', 'RAJ2000','DEJ2000', '2mass_highqual.fits'

;==========================================================================
PRO ae_download_project_catalog, tapurl, catalog_name, RA_name, DEC_name, catalog_filename,$
                                 PROJECT_FOV_FN=project_fov_fn, SAMPLE=sample, _EXTRA=extra,$
                                 TEMP_DIR=tempdir, $
                                 NUM_ROWS=num_rows
                                 
creator_string = "ae_download_project_catalog, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)

; Restore the project-level field of view definition, establisher earlier by ae_define_field_of_view.
if ~keyword_set(project_fov_fn) then project_fov_fn = 'project_fov.sav'

restore, project_fov_fn


;; Check for an existing catalog file.
if file_test(catalog_filename) && User_Is_Present() then begin
  num_rows = psb_xpar(headfits(catalog_filename, EXT=1, ERRMSG=error), 'NAXIS2')
  if (keyword_set(error)) then begin
    print, error
  endif else begin
    msg = string(catalog_filename,num_rows, F='(%"The file %s already contains %d sources.\nDo you want to overwrite it?")' )
    TimedMessage, tm_id, msg, TITLE='ae_download_project_catalog: Existing Catalog Found', QUIT_LABEL='No, abort.', BUTTON_LABEL='Yes, overwrite the existing catalog.', BUTTON_ID=button_id, QUIT_ID=quit_id, PRESSED_ID=pressed_id

    if (pressed_id EQ quit_id) then return
  endelse
endif



;; Build an ADQL query to get sources with RA/DEC inside the project field-of-view.
;
; "SELECT *" means we want all the columns.
; The "as t" names the catalog "t" for easier reference in POINT().
; The CONTAINS function call implements a rectangular search region.
; ADQL queries often require both double and single quotation characters.
; To store such a string in an IDL variable requires repeating one type of quotes in the query.
; Here, we repeat the single quotes in the query.

if (tapurl EQ 'https://gea.esac.esa.int/tap-server/tap') then begin
  ; The ESA_gaia_archive server (https://gea.esac.esa.int/tap-server/tap) will not accept the double quotes that other servers require around the table name. 
  fmt='(%"SELECT %s * FROM %s as t WHERE 1=CONTAINS(POINT(''ICRS'', t.%s, t.%s), BOX(''ICRS'', %0.5f, %0.5f, %0.3f, %0.3f))")'
endif else begin
  ; Other servers (e.g. http://tapvizier.u-strasbg.fr/TAPVizieR/tap) require double quotes ("delimited identifiers") for complex table names (e.g. II/246/out ).
  fmt='(%"SELECT %s * FROM \"%s\" as t WHERE 1=CONTAINS(POINT(''ICRS'', t.%s, t.%s), BOX(''ICRS'', %0.5f, %0.5f, %0.3f, %0.3f))")'
endelse


adql = string( keyword_set(sample) ? 'TOP 10' : '' ,$
               catalog_name, RA_name,DEC_name, center_coords, bbox_size_arcmin/60., F=fmt)

run_tapquery, tapurl, adql, catalog_filename, TEMP_DIR=tempdir, _STRICT_EXTRA=extra

num_rows = psb_xpar(headfits(catalog_filename, EXT=1, ERRMSG=error), 'NAXIS2')
print, num_rows, catalog_name, catalog_filename, F='(%"\nae_download_project_catalog: Downloaded %d sources from catalog %s to %s.\n")'

end ; ae_download_project_catalog




;existing_quiet = !QUIET  &  !QUIET = 0
;cat = QueryVizier(catalog_name, center_coords, bbox_size_arcmin, /ALLCOLUMNS, /VERBOSE, /CFA)  
;!QUIET = existing_quiet
;
;if ~isa(cat, /ARRAY) then begin
;  print, F='(%"\n\nERROR: Vizier at the CfA did not return a catalog; trying Strasbourg ...\n")'
;  existing_quiet = !QUIET  &  !QUIET = 0
;  cat = QueryVizier(catalog_name, center_coords, bbox_size_arcmin, /ALLCOLUMNS, /VERBOSE)  
;  !QUIET = existing_quiet
;
;  if ~isa(cat, /ARRAY) then message, 'ERROR: Vizier in Strasbourg also did not return a catalog!'
;endif ; CfA failed
;
;get_date, date_today, /TIMETAG
;psb_xaddpar, theader, 'DATE', date_today
;psb_xaddpar, theader, 'CREATOR', creator_string  
;psb_xaddpar, theader, 'CATALOG', catalog_name  
;psb_xaddpar, theader,  "FNFITS", catalog_filename
;
;mwrfits, cat, catalog_filename, theader, /CREATE
;print, n_elements(cat), catalog_filename, F='(%"\nWrote %d sources to %s.\n")'






;==========================================================================
;;; Tool to clean artifacts from downloaded OIR tile images prior to building
;;; mosaics with Montage.
;==========================================================================
PRO ae_clean_oir_image, tile_img, REMOVE_NONPOSITIVE=remove_nonpositive,$
                                  SKY_LOWER_CUTOFF_RATIO=sky_lower_cutoff_ratio,$
                                  WIDGET_TITLE=widget_title, ID_DATASET_1D=id_dataset_1d, DATASET_NAME=dataset_name

; Desired ratio between pixel histogram at lower cutoff and pixel histogram peak.
if ~keyword_set(sky_lower_cutoff_ratio) then sky_lower_cutoff_ratio = 1e-2  

; Remove pixels that are identically zero
tile_img[ where(/NULL, (tile_img EQ 0), num_set_to_nan) ] = !VALUES.F_NAN
print, num_set_to_nan,  F='(%"ae_clean_oir_image: %d pixels with value zero set to NaN.")'


if keyword_set(remove_nonpositive) then begin
  ; We assume that non-positive pixels are guaranteed to be bad data.
  tile_img[ where(/NULL, (tile_img LE 0), num_set_to_nan) ] = !VALUES.F_NAN
  print, num_set_to_nan,  F='(%"ae_clean_oir_image: %d non-positive pixels set to NaN.")'
endif

message, /RESET
;mmm, tile_img, sky_mode, sky_sigma, DEBUG=0
sky, tile_img, sky_mode, sky_sigma, /NAN
if (sky_sigma LE 0) || (!ERROR_STATE.CODE NE 0) then begin
  print, '  ', !ERROR_STATE.MSG
  sky_mode  = median(tile_img)
  sky_sigma = stddev(/NAN, tile_img)
  print, sky_mode, sky_sigma, F='(%"ae_clean_oir_image: sky_mode set to median = %0.2f; sky_sigma set to stddev(/NAN, tile_img) = %0.2f.")'
endif


; Build a histogram of the sky pixels around the mode returned by "sky".
nbins = 1000
sky_min = (sky_mode - 5*sky_sigma)
sky_max = sky_mode + 2*sky_sigma
hist = histogram( tile_img, NBINS=nbins, MIN=sky_min, MAX=sky_max)
binsize    = float(sky_max -sky_min) / (nbins - 1)
bin_center = sky_min+binsize/2 + binsize*findgen(nbins)

if arg_present(id_dataset_1d) then $
  dataset_1d, id_dataset_1d, tile_img[where(tile_img LT sky_max)], BINSIZE=binsize, $
              WIDGET_TITLE=widget_title, XTIT='tile pixel (excluding bright tail; before cleaning)', DATASET_NAME=dataset_name

hist_max = max(hist, ind_hist_max)
; Search for a lower cutoff to the left of the histogram peak.
; If loop fails to "break", index "hh" will be zero.
for hh = ind_hist_max,1,-1 do begin
  ; Ignore empty bins!
  if (hist[hh] EQ 0) then continue
  if (hist[hh] / float(hist_max)) LT sky_lower_cutoff_ratio then break
endfor
sky_lower_cutoff = bin_center[hh]

tile_img[ where(/NULL, tile_img LE sky_lower_cutoff, num_set_to_nan) ] = !VALUES.F_NAN

print, num_set_to_nan, sky_lower_cutoff, F='(%"ae_clean_oir_image: %d pixels below %0.2f set to NaN.")'
return
end ; ae_clean_oir_image






;==========================================================================
;;; Download some standard infrared counterpart data products.

;;; A "project field-of-view" must already be defined, by the ae_define_field_of_view tool.
;==========================================================================
PRO ae_build_counterpart_products,$
      SKIP_CATALOGS=SKIP_CATALOGS,$
      SKIP_MOSAICS =skip_mosaics,$
      TEMP_DIR=tempdir


FORWARD_FUNCTION build_AE_cat   ; found in match_xy.pro
FORWARD_FUNCTION build_FITS_cat ; found in match_xy.pro

creator_string = "ae_build_counterpart_products, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()


tempdir_supplied_by_caller = keyword_set(tempdir)

;; If not passed, create a unique scratch directory.
;; Initialize AE command spawning system.
if ~tempdir_supplied_by_caller then begin
  temproot = temporary_directory( 'build_counterpart_products.', VERBOSE=1, SESSION_NAME=session_name)
  tempdir  = temproot
endif
run_command, PARAM_DIR=tempdir


download_catalog_fn         = tempdir +      'download_cat.fits' ; OIR cat downloaded.
ukidss_download_catalog_fn  = tempdir+'ukidss_download_cat.fits' ; UKIDSS cat downloaded.
   vvv_download_catalog_fn  = tempdir+   'vvv_download_cat.fits' ;    VVV cat downloaded.
   vmc_download_catalog_fn  = tempdir+   'vmc_download_cat.fits' ;    VMC cat downloaded.
       repaired_catalog_fn  = tempdir + 'repaired_cat.fits' ; Downloaded cat with structural improvements.
       Z_subset_catalog_fn  = tempdir + 'Z_subset_cat.fits' ; repaired_catalog, split into the band that produced the position
       Y_subset_catalog_fn  = tempdir + 'Y_subset_cat.fits'  
       J_subset_catalog_fn  = tempdir + 'J_subset_cat.fits'  
       H_subset_catalog_fn  = tempdir + 'H_subset_cat.fits'  
       K_subset_catalog_fn  = tempdir + 'K_subset_cat.fits'  

astrometric_reference_fn      = 'astrometric_reference_catalog.fits.gz' ; Gaia/NIR sources with fiducial positions.
astrometric_reference_regfile = 'astrometric_reference_catalog.reg.gz'

temp_catalog_fn   = tempdir + 'temp_cat.fits'
temp_catalog2_fn  = tempdir + 'temp_cat2.fits'
temp_image_fn     = tempdir + 'temp_img.fits'
temp_text_fn      = tempdir + 'temp.txt'
temp_tile_list_fn = tempdir + 'tile_list.fits'
montage_dir       = tempdir + 'montage'
montage_logfile   = './montage.log'

savefile = 'ae_build_counterpart_products.sav'



;; -------------------------------------------------------------
; Read the project field-of-view definition (which may have been hand edited) and build a region file.
project_fov_fn = 'project_fov.sav'
restore, project_fov_fn
openw,  region_unit,  mg_streplace(project_fov_fn, '.sav', '.reg'), /GET_LUN
printf, region_unit, CENTER_COORDS, BBOX_SIZE_ARCMIN, F="(%'fk5;box(%10.6f,%10.6f,%0.2f\',%0.2f\') # tag={project fov} color={DodgerBlue}')"
free_lun, region_unit

; If an archive does not accept ADQL queries and we have to do a cone search, we'll want to trim the result to a 
; rectangular field of view.
; Define a zero-rotation tangent plane coordinate system, with 1' pixels for convenience.
make_astr, astr, CRVAL=CENTER_COORDS, CRPIX=[1,1], DELT=[-1, 1]/60D
; Calculate positions of the rectangle's corners, in the tangent plane system
dx = BBOX_SIZE_ARCMIN[0]/2.0
dy = BBOX_SIZE_ARCMIN[1]/2.0
xy2ad, [dx,dx,-dx,-dx], [dy,-dy,-dy,dy], astr, RA_corners, DEC_corners
project_isInside_parameter = strjoin(string(RA_corners,F='(F0.5)')+','+string(DEC_corners,F='(F0.5)'),', ')
  ; The RA/DEC polygon specification required by the STILTS function isInside().



;; ==========================================================================
; Building a call to STILTS requires some fancy quoting tricks. 
; Certain elements (like literal strings" require quotation with double-quotes, e.g.
;   equals(trim(Bandpass),"K")
; STILTS expressions that include double-quotes and whitespace may require quotation with single-quotes, e.g. 
;   'equals(trim(Bandpass),"K") || Class==1'
; STILTS Processing Filters often have multiple elements separated by whitespace, e.g.
;   select 'equals(trim(Bandpass),"K") || Class==1'
;   addcol -desc "K band, Class 1" K_C1 'equals(trim(Bandpass),"K") || Class==1'
; These multiword strings must themselves be quoted to protect them from the shell.
; We use the function quote_for_shell() for that. 
;; ==========================================================================

file_mkdir, ['Gaia','Twomass','UKIDSS','VVV','VMC','Spitzer','GLIMPSE','SAGE','WISE','Herschel']

Vizier           = 'http://tapvizier.u-strasbg.fr/TAPVizieR/tap'
ESA_gaia_archive = 'https://gea.esac.esa.int/tap-server/tap'

Gaia_repaired_catalog_fn = 'Gaia/EDR3.fits.gz'

;; ==========================================================================
;; OUR TOP PRIORITY IS TO BUILD AN ASTROMETRIC REFERENCE CATALOG
;; so that reduction of the ACIS ObsIDs can proceed.
;; We combine Gaia with an NIR catalog (UKIDSS, VVV, or VMC).
;; ==========================================================================
if keyword_set(SKIP_CATALOGS) then begin
 ;restore, savefile
  GOTO, SKIP_CATALOGS
endif

;; ==========================================================================
;; Download Gaia Catalog

;; From 2019 March to 2020 December (v5560) we obtained DR2 ('I/345/gaia2') from Vizier.

;; From 2020 December to 2022 January we obtained EDR3 ('gaiaedr3') from Vizier.
;; The Gaia column names that had to change in our software are:
;;  radial_velocity        -> dr2_radial_velocity
;;  radial_velocity_error  -> dr2_radial_velocity_error
;; (Note that table 'I/350/gaiaedr3' uses CDS column names.)

;; On 2022 Jan 20 I discovered that gaiaedr3 is no longer on the Vizier TAP server, so I'm 
;; now downloading the table "gaiaedr3.gaia_source" from https://gea.esac.esa.int/tap-server/tap


file_delete, /ALLOW_NONEXISTENT, download_catalog_fn
ae_download_project_catalog, TEMP_DIR=tempdir,$
   ESA_gaia_archive, 'gaiaedr3.gaia_source', 'RA','DEC', download_catalog_fn
;  Vizier, 'gaiaedr3', 'RA','DEC', download_catalog_fn
;  Vizier, 'I/345/gaia2', 'RA','DEC', download_catalog_fn

; Our strategy is postpone cleaning of the Gaia catalog until after the Gaia/NIR union has been constructed.
; Our expectation is that NIR counterparts to "bad" Gaia sources are likely to also have poor position estimates,
; and we want those NIR counterparts to be discarded by the unioning step.
; The "bad" Gaia sources will then be discarded after the unioning step.

; For later convenience, convert position uncertainties from mas to arcsec units, and add a 1-D position uncertainty.
stilts_processing_filter = [$
    ; Name the table.
    'tablename "Gaia EDR3"',$
    ; Propagate Gaia positions to an epoch appropriate for the ACIS data.
    ; Convert position uncertainties from mas to arcsec units.
    string(weighted_epoch, F='(%"addcol -desc \"astrometry parameters\" astrom22 \"epochPropErr((%0.2f - ref_epoch), array(ra,dec,parallax,pmra,pmdec,dr2_radial_velocity,ra_error,dec_error,parallax_error,pmra_error,pmdec_error,dr2_radial_velocity_error, ra_dec_corr,ra_parallax_corr,ra_pmra_corr,ra_pmdec_corr,dec_parallax_corr,dec_pmra_corr,dec_pmdec_corr,parallax_pmra_corr,parallax_pmdec_corr,pmra_pmdec_corr))\"")'),$
      
    string(weighted_epoch, F='(%"addcol -desc \"Reference Epoch for *epochACIS*\" -units yr ACISepoch \"%0.2f\"")'),$
    'addcol -desc "" -units deg     ra_epochACIS       "astrom22[0]"' ,$
    'addcol -desc "" -units deg    dec_epochACIS       "astrom22[1]"' ,$
    'addcol -desc "" -units arcsec  ra_epochACIS_error "astrom22[6]/1000."' ,$
    'addcol -desc "" -units arcsec dec_epochACIS_error "astrom22[7]/1000."' ,$

    ; Discard epoch2000 columns (added by Vizier).
    ;'delcols "astrom22 ra_epoch2000 ra_epoch2000_error dec_epoch2000 dec_epoch2000_error"' ,$
    
    ; add 1-D position uncertainty and proper motion columns.
    'addcol -desc "1-D position error" -units arcsec    epochACIS_error "sqrt(square(ra_epochACIS_error)+square(dec_epochACIS_error))"',$
    'addcol -desc "1-D proper motion"  -units "mas /yr" pm_speed        "sqrt(square(pmra              )+square(pmdec              ))"',$
    
    ; Calculate the largest angular distance moved between ACISepoch and ObsID dates.
    string(max(abs(weighted_epoch - observing_log.DECIMAL_YEAR)), F='(%"addcol -desc \"Movement from ACISepoch to most outlying ObsID\" -units mas InterObsidDrift \"pm_speed * %0.3f\"")'),$

    ; Trim to the rectangular project field of view.
    'select "isInside(ra_epochACIS,dec_epochACIS, '+project_isInside_parameter+')"'$
    ] 

run_command, string(download_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), repaired_catalog_fn,$
                    F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
file_gzip, repaired_catalog_fn, Gaia_repaired_catalog_fn,  /DELETE, /VERBOSE



; Later, Gaia will be our astrometric reference for estimating shifts to near-IR and mid-IR catalogs, and for 
; estimating position uncertainties for those shifted catalogs.
; That work is most easily done with (slow) match_xy tools instead of (fast) STILTS tools.
; That processing will be faster (and hopefully more accurate too) if we ignore Gaia sources that are weak in
; the reddest Gaia band ("RP", ~600 to ~1000 nm ).  https://www.cosmos.esa.int/web/gaia/iow_20180316
;
; In this processing we RETAIN Gaia sources with large proper motion over the time interval between the ACISepoch 
; and the ObsIDs, but we mark them as "not fiducial" so they will not participate in the shift estimates.
Gaia_angular_shift_limit = 0.1 ; arcsec (small compared to on-axis ACIS PSF, similar to NIR position uncertainties)
fiducial_expression = string(Gaia_angular_shift_limit*1000, F='(%"tb.InterObsidDrift LE %0.3f")')

bt = mrdfits(Gaia_repaired_catalog_fn, 1, /SILENT)
magnitude       = bt.phot_rp_mean_mag  
magnitude       = magnitude[where(finite(magnitude), num_mag)]
magnitude_limit = magnitude[ (sort(magnitude))[(num_mag-1) < 1e5] ]
filter_expression = string(magnitude_limit,  F='(%"tb.phot_rp_mean_mag LT %0.3f")')
bt = 0
magnitude=0

;; We need a SKY (PHYSICAL) cartesian coordinate system in order to use match_xy tools.
;; Eventually, our procedures will declare such a system to be used everywhere in the target,
;; defined by the symlink ../tangentplane_reference.evt
;; But that may have not yet happened, so we need to choose one now, to be used only here.
event_file_pattern = '../pointing*/obsid*/primary/acis*evt2.fits*'
obs_event_fn = file_search(event_file_pattern, COUNT=num_obs)
if (num_obs EQ 0) then begin
  print, 'NO event data found!'
  stop
endif

exposure = fltarr(num_obs)
for ii=0, num_obs-1 do begin
  header = headfits(obs_event_fn[ii], EXT=1)
  exposure[ii] = psb_xpar( header,'EXPOSURE')
endfor ; ii
; Use the longest observation as the root tangent plane.
dum = max(exposure, imax)
event2wcs_astr = get_astrometry_from_eventlist(obs_event_fn[imax])


;; Read the Gaia sources that are brightest in the red filter.
gaia_red_cat = build_FITS_cat(Gaia_repaired_catalog_fn, event2wcs_astr,$
                             /ONLY_COORDINATES, FILTER_EXPRESSION=filter_expression,$
                                   RA_EXPRESSION='tb.ra_epochACIS'      ,       DEC_EXPRESSION='tb.dec_epochACIS',$
                             RA_ERROR_EXPRESSION='tb.ra_epochACIS_error', DEC_ERROR_EXPRESSION='tb.dec_epochACIS_error',$
                             FIDUCIAL_EXPRESSION=fiducial_expression, NAME='Gaia')



;; ==========================================================================
;; Download 2MASS Catalog (which will help us clean our deeper NIR catalog)
; targets/NGC6231/literature/Damiani_2016_aa_596_82.pdf Section 4 gives flags for 2MASS sources that should be discarded before matching.
twomass_repaired_catalog_fn = 'Twomass/2mass_highqual.fits.gz'

file_delete, /ALLOW_NONEXISTENT, download_catalog_fn
ae_download_project_catalog, TEMP_DIR=tempdir,$
  Vizier, 'II/246/out', 'RAJ2000','DEJ2000', download_catalog_fn

; For later convenience, convert error ellipse to a 1-D position uncertainty.
stilts_processing_filter = [$
    ; Name the table.
    'tablename "2MASS"',$
    ; add a 1-D position uncertainty.
    'addcol -desc "1-D position error" -after DEJ2000 -units arcsec position_error "sqrt(square(errmaj)+square(errmin))"',$
    ; Trim to the rectangular project field of view.
    'select "isInside(RAJ2000,DEJ2000, '+project_isInside_parameter+')"'$
    ]

run_command, string(download_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), repaired_catalog_fn,$
                    F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
file_gzip, repaired_catalog_fn, twomass_repaired_catalog_fn, /DELETE, /VERBOSE

; Make a tab-separated version too, which ds9 can load.
;run_command, 'dmcopy "Twomass/2mass_highqual.fits[2][cols RAJ2000,DEJ2000]" "Twomass/2mass_highqual.tsv[opt kernel=text/tsv]" clob+'




;; ==========================================================================
;; UKIDSS Survey Notes

;; The root catalog (GpsPointSource) MUST BE CLEANED IN SEVERAL WAYS.
;; For example, the priOrSec and frameSetID columns are extremely important.

; Quality flags are described at http://wsa.roe.ac.uk/ppErrBits.html and 
; at http://wsa.roe.ac.uk/www/gloss_j.html#gpspointsource_jpperrbits where the following is written:
;   "the higher the error quality bit flag value, the more likely it is that the detection is spurious".
;
; Vizier describes the "?ErrBits" columns as "count of the number of zero confidence pixels in the default (2arcsec diameter) aperture".
; I cannot find a definition of these columns at the WSA.
; I do not know if these columns carry information about astrometric reliability of a source.
;
; Examples of quality screening are given in Appendix 3 of Lucas 2008; beware that Lucas describes 2-byte ppErrBits 
; words but in recent releases they are 4-bytes.

; Lucas 2008 estimates saturation limits for the GPS survey:
;   "Conservatively, it is safer to use 2MASS at mK < 12.0, mH < 12.75, mJ < 13.25, though the typical saturation limits are ~0.5 mag brighter than this. 


; The  *Xi columns are "Offset of * detection from master position (+east/-west)"
; The *Eta columns are "Offset of * detection from master position (+north/-south)"
; The pair that are zeros tell you which band determined the master position.
; The master position comes for the shortest band detection.  Thus:
;   * Jxi and Jeta are always zero (J-position was used) or null (no J detection).
;   * Hxi and Heta are Gaussian when J position was used, and are null when K-position was used.
;   * Kxi and Keta are Gaussian when J or H position was used
;
; The source morphology columns (mergedClass, jClass, hClass, k_1Classs, k_2Class) are useful for 
; rejecting crowded sources, noise artifacts, galaxies, etc.

; Note that the WSA provides an aggressively cleaned "view" of the catalog named 'reliableGpsPointSource',
; defined at http://wsa.roe.ac.uk/www/WSA_VIEW_reliableGpsPointSourceSchema.html#reliableGpsPointSource
; THIS LEVEL OF CLEANING IS TOO AGGRESSIVE FOR ASTROMETRY TASKS, BECAUSE ALL BANDS ARE SCREENED (not just the band that produced the position).
;   WHERE  (priOrSec <= 0 OR priOrSec = frameSetID)
;   /*     High quality (but deblended allowed), highly likely point detection in J, H and K1: */
;   AND    jClass   = -1 AND jppErrBits   BETWEEN 0 AND 31
;   AND    hClass   = -1 AND hppErrBits   BETWEEN 0 AND 31
;   AND    k_1Class = -1 AND k_1ppErrBits BETWEEN 0 AND 31
;   /*     High quality, highly likely point detection, OR none, in K2: */
;   AND    (k_2Class=-1 OR k_2Class = -9999) AND k_2ppErrBits <= 31
;   /*     J, H and K (1st epoch) positional coincidence within 0.5 arcsec: */
;   AND    jXi   BETWEEN -0.5 AND +0.5 AND jEta   BETWEEN -0.5 AND +0.5
;   AND    hXi   BETWEEN -0.5 AND +0.5 AND hEta   BETWEEN -0.5 AND +0.5
;   AND    k_1Xi BETWEEN -0.5 AND +0.5 AND k_1Eta BETWEEN -0.5 AND +0.5

; The root catalog (GpsPointSource) MUST BE CLEANED IN SEVERAL WAYS.
; Columns of GpsPointSource defined at: http://wsa.roe.ac.uk/www/wsa_browser.html, WSA UKIDSS->Tables
;
;
; FAQ's are at http://wsa.roe.ac.uk/qa.html
; Known Problems are discussed at http://wsa.roe.ac.uk/knownIssues.html

; CITE SOME PAPERS HERE???



;; ==========================================================================
;; VISTA Survey Notes

; There are multiple VISTA surveys.
; We will have some targets in the VVV Survey, and some in the VMC Survey.
; Those two catalogs share column naming conventions.
; But they have different sets of filter bands, so separate code is needed for cleaning.

; As with UKIDSS, we want "source" tables, not "detection" tables.
;
;; The science center for VISTA is the VISTA Science Archive http://horus.roe.ac.uk/vsa/index.html

;; The root catalog (vvvSource) MUST BE CLEANED IN SEVERAL WAYS.
;; For example, there are priOrSec and frameSetID columns like UKIDSS has.
;; Columns of "vvvSource" defined at http://horus.roe.ac.uk/vsa/www/vsa_browser.html
;; It appears that VVV columns are similar to UKIDSS GPS columns, and hopefully the same cleaning
;; algorithms will be appropriate.

; Quality flags are described at http://horus.roe.ac.uk/vsa/ppErrBits.html 
; On the Nessie field Leisa and Pat reached the following conclusions (2019July):
; * In Z-band (which produced 46% of detections) none of the ppErrBit flags can discard obviously spurious detections
;   without also discarding many more obviously good detections.
;
; * The "near_bright_source" flag generated here has the same problem in Z-band, presumably because we have to 
;   use 2MASS J mags for guessing which sources are bright in Z-band.
;   
; * For astrometry work, avoiding spurious detections is more important than completeness and we filter with
;   the ppErrBit, near_bright_source, and Class flags.
; * For photometry or matching to X-ray we choose to ignore all those flags.


; Recommendations for choosing magnitude columns are given at http://horus.roe.ac.uk/vsa/dboverview.html.

; targets/NGC6231/literature/Damiani18_preprint_1804.01905.pdf Section 2.1 talks about VVV filtering
; their Appendix A -- gives photometric conversions from VVV to 2MASS.

; targets/NGC6231/literature/Kuhn_2017_AJ_154_87.pdf Section 3 talks about VVV flags, matching X-ray sources to NIR sources.




;; ==========================================================================
;; Download UKIDSS Catalog
  
; There are multiple UKIDSS surveys, but the Galactic Plane Survey (GPS) is most likely to cover our targets.
; We want "source" tables, not "detection" tables.

;; UKIDSS DR8 is available via a TAP service.
;; We want the Source Table "gpsSource".
; During the download, we remove only "duplicate sources".
;file_delete, /ALLOW_NONEXISTENT, ukidss_download_catalog_fn
;ae_download_project_catalog, TEMP_DIR=tempdir,$
;  'http://wfaudata.roe.ac.uk/ukidssDR8-dsa/TAP', 'gpsPointSource', 'ra','dec', ukidss_download_catalog_fn ,$
;                                     SELECTION='(priOrSec<=0 OR priOrSec=frameSetID)'

;; BUT, as of 2019 March this TAP service does not implement the ADQL queries used by ae_download_project_catalog
;;, SO, we use a Cone Search instead.
serviceurl = 'http://wfaudata.roe.ac.uk/ukidssDR8-dsa/DirectCone?DSACAT=UKIDSS_DR8&DSATAB=gpsSource'
file_delete, /ALLOW_NONEXISTENT, ukidss_download_catalog_fn
run_cone_query, serviceurl, CENTER_COORDS, radius_arcmin, ukidss_download_catalog_fn, TEMP_DIR=tempdir






;; ==========================================================================
;; Download VVV CATALOG
;; As of 2019 March VVV DR4 is available, but not though any TAP or Cone Search services.
;; So, I am using DR2.

;; The Vizier website has VVV DR2, but with some columns renamed, so I'll stay away from that.

;; As of 2019 Feb. I do not see any TAP service that offers VVV-DR2:
;; * TopCat shows no TAP services at the VSA.
;; * Something called http://wfaudata.roe.ac.uk/vvvDR1-dsa/TAP has DR1 data, but not DR2.
;; Nick Cross said in email:
;; We should have VVVDR4 on TAP, but have been going through the process of updating our TAP services. I will raise this issue again and hope to get that sorted soon.

;; So, (as with UKIDSS) I will build a URL to run a cone search for the VVV catalog.
;; TopCat's Cone Search tool shows that the DR2 table "vvvSource" is available at something called the DSA:
;; http://wfaudata.roe.ac.uk/vvvDR2-dsa/DirectCone?DSACAT=VVV_DR2&DSATAB=vvvSource&

;; For VVV we want the Source Table "vvvSource".
serviceurl = 'http://wfaudata.roe.ac.uk/vvvDR2-dsa/DirectCone?DSACAT=VVV_DR2&DSATAB=vvvSource'
file_delete, /ALLOW_NONEXISTENT, vvv_download_catalog_fn
run_cone_query, serviceurl, CENTER_COORDS, radius_arcmin, vvv_download_catalog_fn, TEMP_DIR=tempdir




;; ==========================================================================
;; Download VMC CATALOG
;; VMC DR4 is available via a TAP service.
;; We want the Source Table "vmcSource".
; During the download, we remove only "duplicate sources".
;file_delete, /ALLOW_NONEXISTENT, vmc_download_catalog_fn
;ae_download_project_catalog, TEMP_DIR=tempdir,$
;  'http://wfaudata.roe.ac.uk/vmcDR4-dsa/TAP', 'vmcSource', 'ra','dec', vmc_download_catalog_fn ,$
;                                     SELECTION='(priOrSec<=0 OR priOrSec=frameSetID)'

;; BUT, as of 2019 Oct this TAP service does not implement the ADQL queries used by ae_download_project_catalog
;; SO, we use a Cone Search instead.
serviceurl = 'http://wfaudata.roe.ac.uk/vmcDR4-dsa/DirectCone?DSACAT=VMC_DR4&DSATAB=vmcSource'
file_delete, /ALLOW_NONEXISTENT, vmc_download_catalog_fn
run_cone_query, serviceurl, CENTER_COORDS, radius_arcmin, vmc_download_catalog_fn, TEMP_DIR=tempdir






;; ==========================================================================
;; Determine which NIR survey has the best coverage.
num_sources_ukidss = psb_xpar( headfits(ukidss_download_catalog_fn, EXT=1), 'NAXIS2')
num_sources_vvv    = psb_xpar( headfits(   vvv_download_catalog_fn, EXT=1), 'NAXIS2')
num_sources_vmc    = psb_xpar( headfits(   vmc_download_catalog_fn, EXT=1), 'NAXIS2')

num_sources = max( [num_sources_ukidss,num_sources_vvv,num_sources_vmc], imax)

if (num_sources EQ 0) then begin
  print, F='(%"\nINFORMATION: no sources found in UKIDSS, VVV, or VMC catalogs!")'
  NIR_name = 'Twomass'
  band_list = ['j','h','k']

endif else begin
  ; We have one or more catalogs better than 2MASS; choose the one with most sources.
  if (num_sources LT 1000) then begin
    help, num_sources_ukidss,num_sources_vvv,num_sources_vmc
    print, 'ERROR: Less than 1000 sources found in UKIDSS, VVV, or VMC queries!'
    help, num_sources_ukidss,num_sources_vvv,num_sources_vmc
    message, 'Stopping for you to investigate ...'
  endif
  
  case imax of
   0: begin
      NIR_name = 'UKIDSS'
      band_list = ['j','h','k_1','k_2','h2']
      aperture_name = 'AperMag1'
      NIR_download_catalog_fn = ukidss_download_catalog_fn
      end
   1: begin
      NIR_name = 'VVV'
      band_list = ['z','y','j','h','ks']
      aperture_name = 'AperMag1'
      NIR_download_catalog_fn = vvv_download_catalog_fn
      end
   2: begin
      NIR_name = 'VMC'
      band_list = ['y','j','ks']
      aperture_name = 'AperMag3'
      NIR_download_catalog_fn = vmc_download_catalog_fn
      end
  endcase

endelse ; Deep NIR catalog available

; Record this decision in a symlink.
file_delete, /ALLOW_NONEXISTENT, 'NIR'
file_link, NIR_name, 'NIR'

print, NIR_NAME, F='(%"\nINFORMATION: Adopting %s as the NIR catalog.\n")' 




;; ==========================================================================
;; Repair problems in the NIR catalog we have chosen.
if (NIR_name EQ 'Twomass') then begin
  ;; If 2MASS is going to play the role of our "deep NIR catalog", we need it to have certain 
  ;; position-related columsn found in UKIDSS/VVV/VMC.
  FILE_GUNZIP, /VERBOSE, twomass_repaired_catalog_fn, download_catalog_fn 

  stilts_processing_filter = [$
    ; Add columns related to position to be compatible with UKIDSS/VVV/VMC catalogs.
    'addcol -desc "ID"             -before RAJ2000               sourceID 2MASS',$ 
    'addcol -desc "astrometric"    -before RAJ2000               reliable_star "(1==1)"',$
    'addcol -desc "position"       -before RAJ2000 -units arcsec RA     RAJ2000',$
    'addcol -desc "position"       -before RAJ2000 -units arcsec DEC    DEJ2000',$
    'addcol -desc "position error" -before RAJ2000 -units arcsec SigRA  0.0',$
    'addcol -desc "position error" -before RAJ2000 -units arcsec SigDEC 0.0' ]  

  run_command, string(download_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)),$
                      repaired_catalog_fn,$
                      F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
;  file_gzip, temp_catalog_fn, twomass_repaired_catalog_fn, /DELETE, /VERBOSE

endif else begin
  ;; Repair problems in the UKIDSS/VVV/VMC catalog we have chosen.
  ;; Identify sources likely to have well-determined position estimates.
  ;
  ; These three catalogs are quite similar, but use different filter sets.
  
  ; Here are STILTS processing steps common to all three catalogs.
  
  comment = 'star class; ppErrBits 0:524287; '+aperture_name+'Err 0:0.1'
  
  stilts_processing_filter = [$
    ; Name the table.
    string(NIR_NAME, F='(%"tablename \"%s\" ")'),$
  
    ; Remove duplicate detections.
    ; http://horus.roe.ac.uk/vsa/vvvGuide.html#Stars
    ; http://horus.roe.ac.uk/vsa/dboverview.html:
    'select "(priOrSec <= 0 || priOrSec == frameSetID)"',$
  
    ; Trim to the rectangular project field of view.
    'select "isInside(RA,DEC, '+project_isInside_parameter+')"',$
      
    ; These catalogs use special values instead of NaN to represent null values. This is extremely inconvenient, e.g. when making plots.
    ; Let's repair at least some important columns.
    'badval -9.999995E+8  "*Mag* *Gausig *Ell *PA *ClassStat *Xi *Eta"'  ]
  
  foreach this_band,band_list do $
    ; Determine which observation (e.g. J,H,K1,K2) contributed the source position estimate.
    stilts_processing_filter = [stilts_processing_filter, string(replicate(this_band,4),$
       F='(%"addcol -desc \"position is from %s band\"  %sPositionReported  \"%sXI==0 && %sETA==0\" ")')]
  
  ; In Nessie:
  ;  zPositionReported: 40%
  ;  yPositionReported: 10%
  ;  jPositionReported: 19%
  ;  hPositionReported: 17%
  ; ksPositionReported: 13%
  
  foreach this_band,band_list do $
    ; Identify sources likely to have well-determined position estimates.
    ; We don't have position uncertainties, but we do have shape information, quality flags, and SNR information (via magnitude errors).
    ; See http://wsa.roe.ac.uk/ppErrBits.html
    ; See http://horus.roe.ac.uk/vsa/ppErrBits.html
    ; The value 524287 is designed to reject the condition "possible cross-talk artefact", and worse.
    ; We find that bogus detections in the wings of bright stars are sometimes flagged as "close to saturated", but
    ; that flag is set for many more legitimate sources.
    ;
    ; Java does not have unsigned data types, so the ?pperrBits values will be negative when bit 31 is set.
    stilts_processing_filter = [stilts_processing_filter, string(comment,this_band,this_band,this_band,this_band, this_band,aperture_name,  this_band,aperture_name,  $
       F='(%"addcol -desc \"%s\" %sHighQuality \"%sClass==-1 && %sppErrBits>=0 && %sppErrBits<=524287 && %s%sErr>=0 && %s%sErr<=0.1\" ")')]
  
  case NIR_name of
    'UKIDSS': begin
      ; If downloaded from UKIDSS website, you may need to convert RA/DEC from radians to degrees.
        ; 'replacecol -units degrees RA  radiansToDegrees(RA)' ,$
        ; 'replacecol -units degrees DEC radiansToDegrees(DEC)' ,$
  
      stilts_processing_filter = [stilts_processing_filter,$
        'badval -9.999995E+8 "muRa muDec sigMuRa sigMuDec chi2 jmhPnt jmhPntErr hmk_1Pnt hmk_1PntErr h2mk_1Pnt h2mk_1PntErr"']
      end
  
    'VVV': begin
      stilts_processing_filter = [stilts_processing_filter,$
        ; VVV is missing the position error columns that UKIDSS has.
        'addcol -desc "position error" -after DEC -units arcsec SigDEC 0.0',$
        'addcol -desc "position error" -after DEC -units arcsec SigRA  0.0',$
  
        'badval -9.999995E+8 "zmyPnt zmyPntErr ymjPnt ymjPntErr jmhPnt jmhPntErr hmksPnt hmksPntErr"']
      end
  
    'VMC': begin
      stilts_processing_filter = [stilts_processing_filter,$
        ; VMC is missing the position error columns that UKIDSS has.
        'addcol -desc "position error" -after DEC -units arcsec SigDEC 0.0',$
        'addcol -desc "position error" -after DEC -units arcsec SigRA  0.0',$
  
        'badval -9.999995E+8 "ymjPnt ymjPntErr jmksPnt jmksPntErr ymjExt ymjExtErr jmksExt jmksExtErr"']
      end
  endcase
  forprint, stilts_processing_filter
  
  ; Run STILTS to perform the repairs.
  run_command, string(NIR_download_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), repaired_catalog_fn,$
                      F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY



  ;; ==========================================================================
  ;; Flag artefacts around bright stars.
  
  ;; Artifacts around bright stars are largely identified by selections on Class and ppEErrBits columns, but 
  ;; to identify surviving artifacts we're going to split the catalog in J,H,K positions and run matches
  ;; to identify all UKIDSS/VVV/VMC detections near bright 2MASS sources.
  ;; The defintion of "bright 2MASS source" (in J,H,K) was derived empirically for each catalog:
  ;;   UKIDSS: on a small subfield in M17SWeX.
  ;;   VVV   : on the full Nessie field
  ;;   VMC   : briefly reviewed the VVV limits on 30dor field; not many artifact seen.
  ;; by visually reviewing detections not removed by other means on top of UKIDSS/VISTA mosaics.
  ;; In that review, display sources selected by
  ;;    ?HighQuality && near_bright_source
  ;; and sort them by the corresponding 2MASS magnitude.
  
  ;; -------------------------------------------
  ; Groups sources by the band that provided their position estimate.
  
  exclusion_radius = 2.0 ; arcsec
  template_row = {select1          :'',$ ; Row selector for deep NIR catalog.
                  select2          :'',$ ; "Bright source" selector for 2MASS catalog.
                  keepcols2        :'',$ ; The 2MASS columns to retain.
                  subset_catalog_fn:''}  ; Temporary file to hold match output table.
  
  case NIR_name of
    'UKIDSS': begin
      band = replicate(template_row,3)
      band.select1   = 'select '+['jPositionReported','"hPositionReported || h2PositionReported"','"k_1PositionReported || k_2PositionReported"']
      band.select2   = 'select '                   +['"Jmag<11.0"','"Hmag<11.4"','"Kmag<11.0"']
      band.keepcols2 = 'keepcols "RAJ2000 DEJ2000 '+[ 'Jmag'      , 'Hmag'      , 'Kmag'      ]+'"'
      band.subset_catalog_fn = [J_subset_catalog_fn,H_subset_catalog_fn,K_subset_catalog_fn]
      end
  
    'VVV': begin
      band = replicate(template_row,5)
      band.select1   = 'select '+['zPositionReported','yPositionReported','jPositionReported','hPositionReported','ksPositionReported']
      band.select2   = 'select '+[       '"Jmag<9.5"',      '"Jmag<10.2"',      '"Jmag<10.0"',      '"Hmag<11.0"',         '"Kmag<9.0"']
      band.keepcols2 = 'keepcols "RAJ2000 DEJ2000 '+[ 'Jmag','Jmag','Jmag', 'Hmag', 'Kmag']+'"'
      band.subset_catalog_fn = [Z_subset_catalog_fn,Y_subset_catalog_fn,J_subset_catalog_fn,H_subset_catalog_fn,K_subset_catalog_fn]
      end
  
    'VMC': begin
      band = replicate(template_row,3)
      band.select1   = 'select '+['yPositionReported','jPositionReported','ksPositionReported']
      band.select2   = 'select '+[      '"Jmag<10.2"',      '"Jmag<10.0"',         '"Kmag<9.0"']
      band.keepcols2 = 'keepcols "RAJ2000 DEJ2000 '+[ 'Jmag','Jmag', 'Kmag']+'"'
      band.subset_catalog_fn = [Y_subset_catalog_fn,J_subset_catalog_fn,K_subset_catalog_fn]
      end
  endcase
                      
  foreach this_band, band do begin                                                                                                              
    cmd = string(exclusion_radius,$
                 ; NIR
                 repaired_catalog_fn,$
                 quote_for_shell(this_band.select1),$
                 quote_for_shell('RA DEC'),$
  
                 ; 2MASS
                 twomass_repaired_catalog_fn,$
                 quote_for_shell(this_band.select2),$
                 quote_for_shell(this_band.keepcols2),$
                 quote_for_shell('RAJ2000 DEJ2000'),$
  
                 ; Flag matches
                 quote_for_shell('addcol -desc "near bright 2MASS source" near_bright_source "(! null_Separation)"'),$
  
                 this_band.subset_catalog_fn,$
    F='(%"stilts tmatch2 matcher=sky params=%0.1f join=all1 find=best1 in1=%s icmd1=%s values1=%s     in2=%s icmd2=%s icmd2=%s values2=%s  fixcols=none ocmd=%s omode=out out=%s ")')
  
    run_command, cmd, /NO_RETRY
  endforeach
    
    
    
    
  ;; -------------------------------------------
  ; Concatenate the annotated sub-catalogs from above.
  expression = strjoin( '('+band_list+'PositionReported && '+band_list+'HighQuality && ! near_bright_source)', '||' )
  
  comment += '; ! near_bright_source'
  stilts_processing_filter = [$
      ; Declare whether source **detection** and **position** are reliable.
      'addcol -desc "'+comment+'" reliable_star "'+expression+'"' ,$
  
      ; Remove 2MASS columns.
      'delcols "RAJ2000 DEJ2000 Jmag"' $
      ] 
  
  cmd = string($
        strjoin("in='"+band.subset_catalog_fn+"' "),$
        ; Remove columns added by tmatch2 (which may sometimes be missing).
                "icmd="+quote_for_shell('delcols "GroupID* GroupSize* Separation"'),$
        strjoin("ocmd="+quote_for_shell(stilts_processing_filter),' '),$
        repaired_catalog_fn,$
        F='(%"stilts tcat %s %s %s omode=out out=%s ")')
  
  run_command, cmd, /NO_RETRY
  
endelse ; (NIR_name NE 'Twomass')




;; ==========================================================================
;; Estimate shift between the NIR catalog (repaired_catalog_fn) and our "red" Gaia sub-catalog.
;; Estimate NIR position uncertainties on this field.
 

;; Note that on a tiny field of view, the Gaia/UKIDSS offset distributions are nicely Gaussian, inferred UKIDSS uncertainties are ~0.045", and significant shifts are suggested, e.g.
;;  DELTAX=  0.043 (+-0.003), DELTAY= -0.102 (+-0.003) SKY pixels.
;; However, on a wide field systematic offsets ("field distortions") are evident, larger uncertainties are needed, and the suggested shifts are much smaller.

;; Plenty of bright K-band sources (even saturated ones) are NOT in Gaia.
;; The most-saturated K-band star IS in Gaia.



; Our "GLIMPSE+" catalog is GLIMPSE rows with all columns from match NIR sources and with the identifier for matching Gaia sources.

case NIR_name of
  'UKIDSS': begin
    filter_expression='tb.reliable_star EQ "T"'
     ra_expression = 'tb.RA' 
    dec_expression = 'tb.DEC'
    ; As a starting point, we previously estimated the following UKIDSS position uncertainties on the M17SWeX field.
    ra_error_expression  = '0.070' ; arcsec
    dec_error_expression = '0.055' ; arcsec
    ; Systematic position errors are apparent.
    nir_aligned_catalog_fn       = 'UKIDSS/DR8.gpsPointSource.fits.gz'
    gaia_nir_astrometric_union_fn= 'Gaia_UKIDSS_astrometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    gaia_nir_photometric_union_fn= 'Gaia_UKIDSS_2MASS_photometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    glimpse_plus_catalog_fn      = 'GLIMPSE/GLIMPSEc.with_Gaia-UKIDSS.fits.gz'
    end

  'VVV': begin
    filter_expression='tb.reliable_star EQ "T"'
     ra_expression = 'tb.RA' 
    dec_expression = 'tb.DEC'
    ; As a starting point, we previously estimated the following VVV position uncertainties on the Nessie field.
    ra_error_expression  = '0.0380' ; arcsec
    dec_error_expression = '0.0337' ; arcsec
    ; Systematic position errors are apparent.  
    nir_aligned_catalog_fn       = 'VVV/DR2.vvvSource.fits.gz'
    gaia_nir_astrometric_union_fn= 'Gaia_VVV_astrometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    gaia_nir_photometric_union_fn= 'Gaia_VVV_2MASS_photometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    glimpse_plus_catalog_fn      = 'GLIMPSE/GLIMPSEc.with_Gaia-VVV.fits.gz'
    end

  'VMC': begin
    filter_expression='tb.reliable_star EQ "T"'
     ra_expression = 'tb.RA' 
    dec_expression = 'tb.DEC'
    ; As a starting point, we previously estimated the following VMC position uncertainties on the T-ReX field.
    ra_error_expression  = '0.041' ; arcsec
    dec_error_expression = '0.044' ; arcsec
    nir_aligned_catalog_fn       = 'VMC/DR3.vmcSource.fits.gz'
    gaia_nir_astrometric_union_fn= 'Gaia_VMC_astrometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    gaia_nir_photometric_union_fn= 'Gaia_VMC_2MASS_photometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    glimpse_plus_catalog_fn      = 'GLIMPSE/GLIMPSEc.with_Gaia-VMC.fits.gz'
    end

  'Twomass': begin
    filter_expression=''
     ra_expression = 'tb.RA' 
    dec_expression = 'tb.DEC'
    ; As a starting point, ...
    ra_error_expression  = '0.2' ; arcsec
    dec_error_expression = '0.2' ; arcsec
    ; Systematic position errors are apparent (TBC).
    nir_aligned_catalog_fn       = 'Twomass/2mass_highqual_aligned.fits'
    gaia_nir_astrometric_union_fn= 'Gaia_2MASS_astrometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    gaia_nir_photometric_union_fn= 'Gaia_2MASS_photometric.fits.gz' ; Gaia/NIR sources with reliable positions.
    glimpse_plus_catalog_fn      = 'GLIMPSE/GLIMPSEc.with_Gaia.fits.gz'
    end

endcase

; Save some important configuration information, in case the long-running processing below is interrupted.
save, /COMPRESS, FILE=savefile,$
  tempdir                      ,$
  Gaia_repaired_catalog_fn     ,$
  twomass_repaired_catalog_fn  ,$
  astrometric_reference_fn     ,$
  astrometric_reference_regfile,$
  num_sources_ukidss           ,$
  num_sources_vvv              ,$
  num_sources_vmc              ,$
  NIR_name                     ,$
  nir_aligned_catalog_fn       ,$
  gaia_nir_astrometric_union_fn,$
  gaia_nir_photometric_union_fn,$
  glimpse_plus_catalog_fn


; For estimating position errors and alignment, we DISCARD the NIR sources that are not deemed "reliable" 
; detections with reliable positions, rather than matching with them and then ignoring their shift votes.
this_cat = build_FITS_cat(repaired_catalog_fn, event2wcs_astr, /ONLY_COORDINATES, NAME='NIR',$
                          FILTER_EXPRESSION=filter_expression,$
                              RA_EXPRESSION= ra_expression,$
                             DEC_EXPRESSION=dec_expression,$
                        RA_ERROR_EXPRESSION=ra_error_expression,$
                       DEC_ERROR_EXPRESSION=dec_error_expression)
  
match_xy_tune_uncertainties, gaia_red_cat, gaia_red_cat[0].CATALOG_NAME,$
                                 this_cat,     this_cat[0].CATALOG_NAME,$
                              0.99, ASTROMETRY=event2wcs_astr,$
                              OUTPUT_DIR='NIR',$
                              /SKIP_REGIONS,$
                             ;REGION_FILENAME=mg_streplace(gaia_nir_astrometric_union_fn, 'fits.gz', 'reg'),$
                              OBS_XSHIFT=obs_xshift,$
                              OBS_YSHIFT=obs_yshift,$
                        ERROR_OBS_XSHIFT=obs_xshift_error,$
                        ERROR_OBS_YSHIFT=obs_yshift_error,$
                                  X_ERROR_RECOMMENDED_SCALING=x_error_recommended_scaling,$
                                  Y_ERROR_RECOMMENDED_SCALING=y_error_recommended_scaling

 ra_error_estimated = float( ra_error_expression)*x_error_recommended_scaling
dec_error_estimated = float(dec_error_expression)*y_error_recommended_scaling
  
; The roundoff error of the single-precision X/Y positions calculated for catalog UKIDSS is approximately 0.64 mas.

; On the large M17SWeX field: DELTAX=  0.029 (+-0.000), DELTAY= -0.083 (+-0.000) pixels.
; UKIDSS RA/DEC coordinates shifted by -14.4, -41.0 mas to align with Gaia.

; Zx and Zy distributions have slightly heavy tails compared to the Gaussian model.
;The X/dx and Y/dy scatter plots show structure, indicating field distortion.
; Maximum match distance is ~0.25".

; The roundoff error of the single-precision X/Y positions calculated for catalog VVV is approximately 0.36 mas.
; On the Nessie field:     DELTAX=  0.177 (+-0.000), DELTAY= -0.242 (+-0.000) pixels.


; On the small Nessie field Zx and Zy distributions are remakabley similar to the Gaussian model.
;The X/dx and Y/dy scatter plots show structure, indicating field distortion.
; Maximum match distance is ~0.15".




;; ==========================================================================
;; Shift full NIR catalog, assign position uncertainties, and save.

;; OBS_XSHIFT,OBS_YSHIFT estimates above are in Chandra SKY system.
;; We want to alter RA/DEC coordinates; the STILTS function epochProp() is handy.
                               
; Parameters of STILTS function epochProp()
;   0:   ra                deg     right ascension
;   1:   dec               deg     declination
;   2:   parallax          mas     parallax
;   3:   pmra              mas/yr  proper motion in ra * cos(dec)
;   4:   pmdec             mas/yr  proper motion in dec
; Below, a negative value in event2wcs_astr.cdelt[1] implements the sign change between Xshift and RAshift.
ra_dec_shift_in_mas = string([obs_xshift,obs_yshift] * (event2wcs_astr.cdelt * 3600D) * 1000D, F='(%"%0.1f")')

astrom6 = string( ra_dec_shift_in_mas, F='(%"array(RA,DEC, 0.0, %s,%s)")')
  
stilts_processing_filter = [$
    ; Name the table.
    string(NIR_NAME, F='(%"tablename \"%s, shifted positions\" ")'),$

    ; Record the shifts we're applying.  
    ; These go in the Table Parameters data structure invented by TopCat/STILTS (stored as an image in the first HDU).
    ; These are not FITS keywords, and cannot be accessed by standard FITS software.
    'setparam -desc "shift in RA direction"  -unit mas RA_shift '+ra_dec_shift_in_mas[0],$
    'setparam -desc "shift in DEC direction" -unit mas DE_shift '+ra_dec_shift_in_mas[1],$

    ; Shift positions; assign position uncertainties.
    'addcol -desc "astrometry parameters" astrom6 "epochProp(1,'+astrom6+')" ' ,$
    'replacecol -desc "position (corrected)" -units degrees RA  "astrom6[0]" ' ,$
    'replacecol -desc "position (corrected)" -units degrees DEC "astrom6[1]" ' ,$
    'replacecol -desc "position error"       -units arcsec SigRA  '+string( ra_error_estimated,F='(F0.4)') ,$
    'replacecol -desc "position error"       -units arcsec SigDEC '+string(dec_error_estimated,F='(F0.4)') ,$
    'delcols "astrom6"' ,$
    ; add a 1-D position uncertainty.
    'addcol -desc "1-D position error" -after SigDEC -units arcsec SigPosition "sqrt(square(SigRA)+square(SigDEC))"'$
    ] 

run_command, string(repaired_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), temp_catalog_fn,$
                    F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY

; Record the shifts we applied as FITS header keywords.  (STILTS cannot do this.)
run_command, string(temp_catalog_fn, ra_dec_shift_in_mas[0], $
                   F="(%'dmhedit infile=""%s[2]"" filelist=none operation=add key=RA_shift  value=%s comment=""[mas] shift in RA direction""')")
run_command, string(temp_catalog_fn, ra_dec_shift_in_mas[1], $
                   F="(%'dmhedit infile=""%s[2]"" filelist=none operation=add key=DE_shift  value=%s comment=""[mas] shift in DEC direction""')")

file_gzip, temp_catalog_fn, nir_aligned_catalog_fn, /DELETE, /VERBOSE

print, NIR_NAME, ra_dec_shift_in_mas, F='(%"\nINFORMATION: %s RA/DEC coordinates shifted by %s, %s mas to align with Gaia.")'
print, NIR_NAME,1000* ra_error_estimated,$
                1000*dec_error_estimated,F='(%"INFORMATION: Estimated remaining %s position uncertainties on this field are %0.1f, %0.1f mas.\n")'




;; ==========================================================================
;; Build NIR/Gaia Union as an Astrometric Reference Catalog
;; ==========================================================================
;;
;; Finally, we can form the union between the (full) Gaia catalog and the (cleaned and shifted) NIR catalog. 
;;
;; We assume all Gaia sources are reliable detections of stars.
;; We may declare some Gaia sources to be "not fiducial", and later ignore their votes on frame alignment.
;; But we do not *remove* such sources here (either before or after the union with NIR) because they may be the correct match to ACIS sources.

;; When forming the union of Gaia and NIR (in STILTS) we want an overly aggressive match criterion to weed out NIR detections of Gaia sources that suffer systematic (not normal Poisson) position errors (e.g. from blending or from detector artifact).
;; 
;; The STILTS matcher does not have a "significance" threshold parameter---a match is asserted if the 1-D error circles of two sources overlap.  Thus, to get more aggressive matching we must inflate position errors in one or both catalogs.

position_error_inflation = '5.0'

cmd = string($
     ; Catalog #1: Gaia
     Gaia_repaired_catalog_fn,$
     strjoin(' icmd1='+quote_for_shell('')),$
     quote_for_shell(string(position_error_inflation, F='(%"ra_epochACIS dec_epochACIS \"epochACIS_error*%0.2f\" ")')),$

     ; Catalog #2: NIR (UKIDSS, VVV, VMC, or Twomass)
     nir_aligned_catalog_fn,$
     strjoin(' icmd2='+quote_for_shell([$
       ; For astrometric catalog,, we DISCARD the NIR sources that are not deemed "reliable" detections 
       ; with reliable positions, rather than retaining them and then ignoring their frame shift votes.
       'select "reliable_star"' $
       ])),$
     quote_for_shell(string(position_error_inflation, F='(%"RA DEC \"SigPosition*%0.2f\" ")')),$

     strjoin(' ocmd='+quote_for_shell([$
       ; Calculate separation of matches.
       'replacecol -desc "Gaia/NIR separation" -units arcsec Separation "skyDistanceDegrees(ra_epochACIS,dec_epochACIS,RA_2,DEC_2)*3600"',$
       ; Describe aggressiveness of matching.
       'addcol -desc "position error inflation in tmatch call" STILTS_match_Zlimit '+position_error_inflation, $
       ; Identify matches
       'addcol -desc "Gaia/NIR match" is_matched "(! null_Separation)"' ,$
       ; We prefer Gaia positions over NIR positions.
       'addcol -desc "Gaia position in ra_epochACIS dec_epochACIS" is_Gaia_position "! null_source_id"',$
       'replacecol  ra_epochACIS       "is_Gaia_position ?  ra_epochACIS       :  RA_2      "'         ,$
       'replacecol dec_epochACIS       "is_Gaia_position ? dec_epochACIS       : DEC_2      "'         ,$
       'replacecol  ra_epochACIS_error "is_Gaia_position ?  ra_epochACIS_error : SigRA      "'         ,$
       'replacecol dec_epochACIS_error "is_Gaia_position ? dec_epochACIS_error : SigDEC     "'         ,$
       'replacecol     epochACIS_error "is_Gaia_position ?     epochACIS_error : SigPosition"'         ,$
      ; Add a LABEL column for the benefit of match_xy tools.
      'addcol -desc "source label" LABEL ''concat("#", toString($index), is_Gaia_position ? " (Gaia)" : " (NIR)")'' ' $
       ])),$

     temp_catalog_fn,$
     F='(%"stilts tmatch2 matcher=skyerr params=1.0 join=1or2 find=best  in1=%s %s values1=%s   in2=%s %s values2=%s  fixcols=dups %s omode=out out=%s ")')

  run_command, cmd, /NO_RETRY
  file_gzip, temp_catalog_fn, gaia_nir_astrometric_union_fn, /DELETE, /VERBOSE


;; -------------------------------------------
;; Build the Astrometric Reference Catalog we will use for frame alignment work.
;; * We add the match_xy NOT_FIDUCIAL column (byte, not 'logical' type), to indicate frame alignment votes to ignore.
;; * We drop most columns, for faster access.
stilts_processing_filter = [$
    ; Identify "not fiducial" Gaia sources, via a limit on the "InterObsidDrift" column.
    string(replicate(Gaia_angular_shift_limit*1000,2),$
      F='(%"addcol -desc \"Gaia InterObsidDrift > %0.3f\"  NOT_FIDUCIAL \"(InterObsidDrift > %0.3f) ? toByte(1) : toByte(0)\"")'),$

    ; Rename some columns.
    'colmeta -name Gaia_ID source_ID',$
    'colmeta -name  NIR_ID sourceID',$

    ; For speed, retain only essential columns.
    'keepcols "ra_epochACIS dec_epochACIS ra_epochACIS_error dec_epochACIS_error epochACIS_error is_Gaia_position Gaia_ID NIR_ID NOT_FIDUCIAL"'$
    ]
  
run_command, string(gaia_nir_astrometric_union_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), temp_catalog_fn,$
                    F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
file_gzip, temp_catalog_fn, astrometric_reference_fn, /DELETE, /VERBOSE

; Depict this Reference Catalog in a region file.
bt = mrdfits(astrometric_reference_fn,1)

openw , regunit, temp_text_fn, /GET_LUN
printf, regunit, "# Region file format: DS9 version 3.0"
printf, regunit, 'global color=green width=2 font="helvetica 12 normal"'
printf, regunit, 'ICRS'
!TEXTUNIT = regunit
forprint, SUBSET=where(bt.is_Gaia_position NE 'T'), TEXTOUT=5, /NoComm, bt.ra_epochACIS, bt.dec_epochACIS, F='(%"point %10.6f %10.6f # point=diamond tag={NIR}")'

forprint, SUBSET=where(bt.is_Gaia_position EQ 'T'), TEXTOUT=5, /NoComm, bt.ra_epochACIS, bt.dec_epochACIS, F='(%"box %10.6f %10.6f 1\" 1\" 0 # tag={Gaia} color=magenta")'

forprint, SUBSET=where(bt.NOT_FIDUCIAL), TEXTOUT=5, /NoComm, bt.ra_epochACIS, bt.dec_epochACIS, F='(%"point %10.6f %10.6f # point=x tag={Gaia, not fiducial} color=red")'

free_lun, regunit
!TEXTUNIT = 0
bt = 0

file_gzip, temp_text_fn, astrometric_reference_regfile, /DELETE, /VERBOSE




;; ==========================================================================
;; Build Gaia/NIR Union as an OIR Photometry Catalog
;; See /bulk/cochise1/psb6/TARA/code/ae/patches/patch_to_ae_build_counterpart_products.pro
;; ==========================================================================
; Here, completeness of the OIR catalog is important, because knowing that an
; X-ray source has a counterpart is scientifically important, even if its photometry is damaged.

;; We assume all Gaia sources are legitimate detections with accurate positions.
;; Even if Gaia photometry is not needed, Gaia can improve the positions of matching NIR sources.

if (NIR_name EQ 'Twomass') then begin
  ; The NIR catalog is 2MASS, and we have already formed the union of Gaia and all of 2MASS.
  file_delete, /ALLOW_NONEXISTENT, gaia_nir_photometric_union_fn
  file_link, gaia_nir_astrometric_union_fn, gaia_nir_photometric_union_fn, /VERBOSE

endif else begin

  ;; The deep NIR catalog (UKIDSS,VVV,VMC) is incomplete at the bright end, and contains annoying
  ;; spurious detections in the wings of very bright stars.
  ;; Photometry of good detections at the bright end may be damaged by saturation.
  ;; We would love to discard those spurious rows, but (as mentioned earlier) we find empirically that 
  ;; none of the flags in *ppErrBits columns can cleanly identify spurious rows.
  ;; We find the same for our home-grown 2MASS-based "near_bright_source" flag, at least for Z-band.
  ;; Thus, for now, we are not removing any rows for the deep NIR catalog.
  ;; We DO NOT DISCARD deep NIR sources with bad photometry; their accurate positions are valuable
  ;; for discovering that an X-ray source has an NIR counterpart.
  
  stilts_NIR_filter = [$
    ; Record that we are not filtering the NIR catalog. 
    ; This goes in the Table Parameters data structure invented by TopCat/STILTS (stored as an image in the first HDU).
    ; These are not FITS keywords, and cannot be accessed by standard FITS software.
    string(NIR_name,NIR_name, 'None', F='(%"setparam -desc \"selection on %s catalog\" %s_selection \"%s\" ")') ]
  
  
  
  ;; The only 2MASS sources that are valuable are those that may have unique photometry (sources not in deep NIR and deep NIR sources with damaged photometry).
  stilts_2MASS_filter = 'select "Kmag<12"'
  
  
  
  ;; The appropriate field of view for this OIR catalog is the bounding box for the ACIS data.
  ; Define a zero-rotation tangent plane coordinate system, with 1' pixels for convenience.
  make_astr, astr, CRVAL=acis_center_coords, CRPIX=[1,1], DELT=[-1, 1]/60D
  ; Calculate positions of the rectangle's corners, in the tangent plane system
  dx = acis_bbox_size_arcmin[0]/2.0
  dy = acis_bbox_size_arcmin[1]/2.0
  xy2ad, [dx,dx,-dx,-dx], [dy,-dy,-dy,dy], astr, RA_corners, DEC_corners
  ACIS_isInside_parameter = strjoin(string(RA_corners,F='(F0.5)')+','+string(DEC_corners,F='(F0.5)'),', ')
    ; The RA/DEC polygon specification required by the STILTS function isInside().
  
  
  ;; 
  ;; The STILTS matcher does not have a "significance" threshold parameter---a match is asserted if the 1-D error circles of two sources overlap.  Thus, we control match significance by inflating position errors in one or both catalogs.
  
  ;; We want moderate match criteria to find true associations, rather than aggressive matches designed to hide NIR detetions with crap positions.
  position_error_inflation = '3.0'
  
  ;; Form the union of Gaia and NIR catalogs.
  cmd = string($
       ; Catalog #1: Gaia
       Gaia_repaired_catalog_fn,$
       strjoin(' icmd1='+quote_for_shell([$
         ; As with the GLIMPSE+ catalog, we retain only a few Gaia columns.  
         'keepcols "designation ra_epochACIS dec_epochACIS ra_epochACIS_error dec_epochACIS_error epochACIS_error"',$
         ; Trim to the rectangular ACIS field of view.
         'select "isInside(ra_epochACIS,dec_epochACIS, '+ACIS_isInside_parameter+')"',$
         ; Start building column "origin_epoch_ACIS"
         'addcol -desc "origin of ra_epochACIS,dec_epochACIS" -before ra_epochACIS  origin_epoch_ACIS ''concat("Gaia")''' $
       ])),$
       quote_for_shell(string(position_error_inflation, F='(%"ra_epochACIS dec_epochACIS \"epochACIS_error*%0.2f\" ")')),$
  
       ; Catalog #2: NIR (UKIDSS, VVV, VMC)
       nir_aligned_catalog_fn,$
       strjoin(' icmd2='+quote_for_shell([$
         ; Trim to the rectangular ACIS field of view.
         'select "isInside(RA,DEC, '+ACIS_isInside_parameter+')"',$
         ; Select desired sources.
         stilts_NIR_filter,$
         ; Describe aggressiveness of matching.
         'addcol -desc "position error inflation in tmatch call" STILTS_match1_Zlimit '+position_error_inflation $
         ])),$
       quote_for_shell(string(position_error_inflation, F='(%"RA DEC \"SigPosition*%0.2f\" ")')),$
  
       strjoin(' ocmd='+quote_for_shell([$
         ; Calculate separation of matches.
         'delcols "Separation"',$
         'addcol -desc "Gaia/'+NIR_name+' separation" -units arcsec Separation1 "skyDistanceDegrees(ra_epochACIS,dec_epochACIS,RA,DEC)*3600"',$
         ; Adopt NIR position for NIR-only rows.
         'addcol -desc "TEMPORARY" adopt_slave_position "null_ra_epochACIS"',$
         'replacecol origin_epoch_ACIS     ''adopt_slave_position ? concat("'+NIR_name+'"): origin_epoch_ACIS''     ',$
         'replacecol     ra_epochACIS       "adopt_slave_position ?  RA                   :     ra_epochACIS       "',$
         'replacecol    dec_epochACIS       "adopt_slave_position ? DEC                   :    dec_epochACIS       "',$
         'replacecol     ra_epochACIS_error "adopt_slave_position ? SigRA                 :     ra_epochACIS_error "',$
         'replacecol    dec_epochACIS_error "adopt_slave_position ? SigDEC                :    dec_epochACIS_error "',$
         'replacecol        epochACIS_error "adopt_slave_position ? SigPosition           :        epochACIS_error "' $
         ])),$
  
       temp_catalog_fn,$
       F='(%"stilts tmatch2 matcher=skyerr params=1.0 join=1or2 find=best  in1=%s %s values1=%s   in2=%s %s values2=%s  fixcols=dups %s omode=out out=%s ")')
  
    run_command, cmd, /NO_RETRY
  
  
  
  ;; Form the union of Gaia/NIR (from above) and 2MASS catalogs.
  ;; We use a more aggressive match here, to avoid creating 2MASS-only rows that are simply missed matches to Gaia.
  position_error_inflation       = '3.0'
  position_error_inflation_2mass = '7.0'
  cmd = string($
       ; Catalog #1: Gaia/NIR union (from above)
       temp_catalog_fn,$
       ; No additional filtering is needed.
       strjoin(' icmd1='+quote_for_shell('')),$
       quote_for_shell(string(position_error_inflation, F='(%"ra_epochACIS dec_epochACIS \"epochACIS_error*%0.2f\" ")')),$
  
       ; Catalog #2: 2MASS (bright sources only)
       twomass_repaired_catalog_fn,$
       strjoin(' icmd2='+quote_for_shell([$
         ; Trim to the rectangular ACIS field of view.
         'select "isInside(RAJ2000,DEJ2000, '+ACIS_isInside_parameter+')"',$
         ; Select desired sources.
         stilts_2MASS_filter,$
         ; Describe aggressiveness of matching.
         'addcol -desc "position error inflation in tmatch call" STILTS_match2_Zlimit '+position_error_inflation_2mass $       
         ])),$
       quote_for_shell(string(position_error_inflation_2mass, F='(%"RAJ2000 DEJ2000 \"position_error*%0.2f\" ")')),$
  
       strjoin(' ocmd='+quote_for_shell([$
         ; Adopt 2MASS position for 2MASS-only rows.
         'replacecol  adopt_slave_position  "null_ra_epochACIS"',$
         'replacecol origin_epoch_ACIS     ''adopt_slave_position ? concat("2MASS"): origin_epoch_ACIS''     ',$
         'replacecol     ra_epochACIS       "adopt_slave_position ? RAJ2000        :     ra_epochACIS       "',$
         'replacecol    dec_epochACIS       "adopt_slave_position ? DEJ2000        :    dec_epochACIS       "',$
         'replacecol     ra_epochACIS_error "adopt_slave_position ? position_error/sqrt(2): ra_epochACIS_error "',$
         'replacecol    dec_epochACIS_error "adopt_slave_position ? position_error/sqrt(2):dec_epochACIS_error "',$
         'replacecol        epochACIS_error "adopt_slave_position ? position_error :        epochACIS_error "',$
         ; Calculate separation of matches.
         'delcols "Separation adopt_slave_position"',$
         'addcol -desc "Gaia-'+NIR_name+'/2MASS separation" -units arcsec Separation2 "skyDistanceDegrees(ra_epochACIS,dec_epochACIS,RAJ2000,DEJ2000)*3600"',$
  
        ; Add a LABEL column for the benefit of match_xy tools.
        'addcol -desc "OIR label" Label ''concat("#", toString($index), " (", origin_epoch_ACIS, ")")'' ' $
         ])),$
  
       temp_catalog2_fn,$
       F='(%"stilts tmatch2 matcher=skyerr params=1.0 join=1or2 find=best  in1=%s %s values1=%s   in2=%s %s values2=%s  fixcols=dups %s omode=out out=%s ")')
  
  run_command, cmd, /NO_RETRY

  file_gzip, /DELETE, temp_catalog2_fn, gaia_nir_photometric_union_fn, /VERBOSE

endelse ; NIR catalog is UKIDSS/VVV/VMC



; Make a region file for this Gaia/NIR union catalog.
temp_cat = build_FITS_cat(gaia_nir_photometric_union_fn, event2wcs_astr, /ONLY_COORDINATES, $
                                 RA_EXPRESSION='tb.ra_epochACIS'      ,       DEC_EXPRESSION='tb.dec_epochACIS',$
                           RA_ERROR_EXPRESSION='tb.ra_epochACIS_error', DEC_ERROR_EXPRESSION='tb.dec_epochACIS_error',$
                           LABEL_EXPRESSION='tb.label',$
                           NAME='Gaia_NIR')
match_xy        , match_state, temp_cat, temp_cat[0].CATALOG_NAME, 0.99, /INIT, ASTROMETRY=event2wcs_astr
match_xy_analyze, match_state, Gaia_NIR, OUTPUT_DIR='',$
                  REGION_FILENAME=mg_streplace(gaia_nir_photometric_union_fn, 'fits.gz', 'reg')





;; ==========================================================================
;; Download the official Gaia/2MASS associations.
;
; The gaiadr2.tmass_best_neighbour table Gaia produced is a list of all Gaia/2MASS matches.
; But, it has NO COORDINATES so a region query is not possible.  We must download the entire catalog (450 million rows).
;ae_download_project_catalog, TEMP_DIR=tempdir,$
;  'http://gea.esac.esa.int/tap-server/tap', 'gaiadr2.tmass_best_neighbour', 'ra','dec', download_catalog_fn ,$

; What are those good for?



;; ==========================================================================
; GLIMPSE Catalog (Spitzer)
;; ==========================================================================
;; Coverage for a specific region can be checked at https://irsa.ipac.caltech.edu/applications/Radar/

;;; See https://irsa.ipac.caltech.edu/applications/Gator/GatorAid/GLIMPSE/GLIMPSE_I_S07.html for column descriptions.
;
; "The position is given in both Galactic (l,b) and equatorial (RA, Declination) J2000 coordinates, along with estimated uncertainties. The pointing accuracy is 1 arcsec (Werner et al. 2004). The SSC pipeline does pointing refinement of the images based on comparison with the 2MASS Point Source Catalog, whose absolute accuracy is typically <0.2 arcsec (Cutri et al. 2005). After applying the SSC geometric distortion corrections and updating to the 2MASS positions, the GLIMPSE point source accuracy is typically 0.3 arcsec absolute accuracy, limited by undersampling of the point-spread function. The position uncertainties are calculated by the bandmerger based on the uncertainties of individual detections, propagated through the calculation of the weighted mean position."

; We get GLIMPSE sources from Vizier's "GLIMPSE" catalog (II/293/glimpse), which is described as a combination of three IRSA catalogs: GLIMPSE I, GLIMPSE II, GLIMPSE 3D.  As far as I can tell, the IRSA does not offer this combo catalog.

; Note that the GLIMPSE team has produced catalogs for many Spitzer data sets.
; See https://irsa.ipac.caltech.edu/data/SPITZER/GLIMPSE/overview.html
; I do not know if Vizier's II/293/glimpse catalog contains more than the GLIMPSE I, GLIMPSE II, GLIMPSE 3D surveys.


glimpse_repaired_catalog_fn='GLIMPSE/Catalog.fits.gz'

file_delete, /ALLOW_NONEXISTENT, download_catalog_fn
ae_download_project_catalog, TEMP_DIR=tempdir, Vizier, 'II/293/glimpse', 'RAJ2000','DEJ2000', download_catalog_fn, NUM_ROWS=num_rows_GLIMPSE

if (num_rows_GLIMPSE EQ 0) then begin
  print, F='(%"\n\nINFORMATION: GLIMPSE catalog has no sources on this field.\n")'

endif else begin
  ;; -------------------------------------------
  ; Repair problems in the downloaded catalog.
  ; The GLIMPSE download from Vizier contains both "Catalog" (SSTGLMC...) and "Archive" (SSTGLMA...) sources.
  ; We want the Catalog (highly reliable) sources, identified by "C" in the "C" column.
  stilts_processing_filter = [$
      ; Name the table.
      'tablename "GLIMPSE"',$
      ; Select "Catalog" sources.
      'select ''equals(toString(C),"C")''',$
      ; Trim to the rectangular project field of view.
      'select "isInside(RAJ2000,DEJ2000, '+project_isInside_parameter+')"'$
      ] 
  
  run_command, string(download_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), repaired_catalog_fn,$
                      F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
  
  ;; -------------------------------------------
  ;; Estimate shift between the GLIMPSE catalog (repaired_catalog_fn) and our "red" Gaia sub-catalog.
  ;; Estimate GLIMPSE position uncertainties on this field.
  
  ; As a starting point, we previously estimated the following GLIMPSE position uncertainties on the M17SWeX field.
  ra_error_expression  = '0.117' ; arcsec
  dec_error_expression = '0.100' ; arcsec
  ;
  ; We currently do not have any indicator of unreliable detections in this GLIMPSE catalog, so the
  ; call below has no FILTER_EXPRESSION option.
  this_cat = build_FITS_cat(repaired_catalog_fn, event2wcs_astr, /ONLY_COORDINATES, NAME='GLIMPSE',$
                                  RA_EXPRESSION='tb.RAJ2000',$
                                 DEC_EXPRESSION='tb.DEJ2000',$
                            RA_ERROR_EXPRESSION=ra_error_expression,$
                           DEC_ERROR_EXPRESSION=dec_error_expression)
  
  match_xy_tune_uncertainties, gaia_red_cat, gaia_red_cat[0].CATALOG_NAME,$
                                this_cat, this_cat[0].CATALOG_NAME,$
                                0.99, ASTROMETRY=event2wcs_astr,$
                                OUTPUT_DIR='GLIMPSE',$
                                /SKIP_REGIONS,$
                               ;REGION_FILENAME='Gaia_GLIMPSE.reg',$
                                OBS_XSHIFT=obs_xshift,$
                                OBS_YSHIFT=obs_yshift,$
                          ERROR_OBS_XSHIFT=obs_xshift_error,$
                          ERROR_OBS_YSHIFT=obs_yshift_error,$
                                    X_ERROR_RECOMMENDED_SCALING=x_error_recommended_scaling,$
                                    Y_ERROR_RECOMMENDED_SCALING=y_error_recommended_scaling
  
  ; PosErr=0.16    DELTAX=  0.364 (+-0.001), DELTAY=  0.116 (+-0.001) pixels  (M17SWeX)
  
  
  ;; -------------------------------------------
  ;; Shift full GLIMPSE catalog, assign position uncertainties, and save.
  ;; OBS_XSHIFT,OBS_YSHIFT estimates above are in Chandra SKY system.
  ;; We want to alter RA/DEC coordinates; the STILTS function epochProp() is handy.
  ;; We DO NOT ALTER Galactic cooordinates!
  
  ; Parameters of STILTS function epochProp()
  ;   0:   ra                deg     right ascension
  ;   1:   dec               deg     declination
  ;   2:   parallax          mas     parallax
  ;   3:   pmra              mas/yr  proper motion in ra * cos(dec)
  ;   4:   pmdec             mas/yr  proper motion in dec
  ; Below, a negative value in event2wcs_astr.cdelt[1] implements the sign change between Xshift and RAshift.
  ra_dec_shift_in_mas = string([obs_xshift,obs_yshift] * (event2wcs_astr.cdelt * 3600D) * 1000D, F='(%"%0.1f")')
  
  astrom6 = string(ra_dec_shift_in_mas, F='(%"array(RAJ2000,DEJ2000, 0.0, %s,%s)")')
  
  stilts_processing_filter = [$
      ; Name the table.
      'tablename "GLIMPSE, shifted positions"',$
      ; Record the shifts we're applying.  
      ; These go in the Table Parameters data structure invented by TopCat/STILTS (stored as an image in the first HDU).
      ; These are not FITS keywords, and cannot be accessed by standard FITS software.
      'setparam -desc "shift in RA direction"  -unit mas RA_shift '+ra_dec_shift_in_mas[0],$
      'setparam -desc "shift in DEC direction" -unit mas DE_shift '+ra_dec_shift_in_mas[1],$
  
      ; Shift positions; assign position uncertainties.
      'addcol -desc "astrometry parameters" astrom6 "epochProp(1,'+astrom6+')" ' ,$
      'addcol -desc "position (corrected)" -units degrees -before RAJ2000 RA  "astrom6[0]" ' ,$
      'addcol -desc "position (corrected)" -units degrees -before RAJ2000 DEC "astrom6[1]" ' ,$
      'addcol -desc "position error"       -units arcsec  -before RAJ2000 SigRA  '+ ra_error_expression ,$
      'addcol -desc "position error"       -units arcsec  -before RAJ2000 SigDEC '+dec_error_expression ,$
      'delcols "astrom6 RAJ2000 DEJ2000"' ,$
      ; add a 1-D position uncertainty.
      'addcol -desc "1-D position error" -after SigDEC -units arcsec SigPosition "sqrt(square(SigRA)+square(SigDEC))"'$
      ] 
  
  run_command, string(repaired_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), temp_catalog_fn,$
                      F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
  
  ; Record the shifts we applied as FITS header keywords.  (STILTS cannot do this.)
  run_command, string(temp_catalog_fn, ra_dec_shift_in_mas[0], $
                     F="(%'dmhedit infile=""%s[2]"" filelist=none operation=add key=RA_shift  value=%s comment=""[mas] shift in RA direction""')")
  run_command, string(temp_catalog_fn, ra_dec_shift_in_mas[1], $
                     F="(%'dmhedit infile=""%s[2]"" filelist=none operation=add key=DE_shift  value=%s comment=""[mas] shift in DEC direction""')")
  
  file_gzip, temp_catalog_fn, glimpse_repaired_catalog_fn, /DELETE, /VERBOSE
  
  print, ra_dec_shift_in_mas, F='(%"\nINFORMATION: GLIMPSE RA/DEC coordinates shifted by %s, %s mas to align with Gaia.")'
  print, 1000*float( ra_error_expression)*x_error_recommended_scaling,$
         1000*float(dec_error_expression)*y_error_recommended_scaling,F='(%"INFORMATION: Estimated remaining GLIMPSE position uncertainties on this field are %0.1f, %0.1f mas.\n")'
  
  ; Nessie:
  ; GLIMPSE RA/DEC coordinates shifted by -128.7, -37.4 mas to align with Gaia.
  ; Estimated remaining GLIMPSE position uncertainties on this field are 117.8, 143.4 mas.
  

    
  ;; ==========================================================================
  ;; Add to GLIMPSE catalog all NIR (UKIDSS/VISTA) columns and Gaia identifier.
    
  ;; The STILTS matcher does not have a "significance" threshold parameter---a match is asserted if the 1-D error circles of two sources overlap.  Thus, to get more aggressive matching we must inflate position errors in one or both catalogs.
  
  position_error_inflation = '5.0'
    
  if (NIR_name EQ 'Twomass') then begin
    ;; Since GLIMPSE already has 2MASS information, we run a two-way match between GLIMPSE & Gaia
  
    cmd = string($
         ; Master Catalog #1: GLIMPSE
         glimpse_repaired_catalog_fn,$
         strjoin(' icmd1='+quote_for_shell([$
          ; Rename position-related columns for consistency with the three-way match case below.
          'colmeta -name  RA_1          RA         ',$
          'colmeta -name  DEC_1         DEC        ',$
          'colmeta -name  SigRA_1       SigRA      ',$
          'colmeta -name  SigDEC_1      SigDEC     ',$
          'colmeta -name  SigPosition_1 SigPosition' $
           ])),$
         quote_for_shell(string(position_error_inflation, F='(%"RA_1 DEC_1 \"SigPosition_1*%0.2f\" ")')),$
                                                                      
         ; Catalog #2: Gaia
         Gaia_repaired_catalog_fn,$
         strjoin(' icmd2='+quote_for_shell([$
           'keepcols "designation ra_epochACIS dec_epochACIS epochACIS_error"'$
           ])),$
         quote_for_shell(string(position_error_inflation, F='(%"ra_epochACIS dec_epochACIS \"epochACIS_error*%0.2f\" ")')),$
    
         ; Calculate separation of matches.
         strjoin(' ocmd='+quote_for_shell([$
          'addcol -desc "GLIMPSE/Gaia separation" -units arcsec Separation1 "skyDistanceDegrees(RA_1,DEC_1,ra_epochACIS,dec_epochACIS)*3600"'$
           ])),$
    
         strjoin(' ocmd='+quote_for_shell([$
           ; Describe aggressiveness of matching.
           'addcol -desc "position error inflation in tmatch call" STILTS_match_Zlimit '+position_error_inflation, $
           ; Identify matches
           'addcol -desc "GLIMPSE/Gaia match" is_matched1 "(! null_Separation1)"'$
           ])),$
    
         temp_catalog_fn,$
         F='(%"stilts tmatch2 matcher=skyerr params=1.0 join=all1 find=best  in1=%s %s values1=%s   in2=%s %s values2=%s   fixcols=dups %s %s omode=out out=%s ")')


  endif else begin
  
    ;; We run a three-way match between GLIMPSE, NIR, Gaia
  
    case NIR_name of
      'UKIDSS': begin
        photometry_offset_definitions = [$
           'addcol -desc "2MASS - UKIDSS" -units mag  jDelta     "is_matched2 ? (Jmag -   jAperMag1) : NULL"',$
           'addcol -desc "2MASS - UKIDSS" -units mag  hDelta     "is_matched2 ? (Hmag -   hAperMag1) : NULL"',$
           'addcol -desc "2MASS - UKIDSS" -units mag  kDelta     "is_matched2 ? (Kmag - k_1AperMag1) : NULL"' $
           ]
        end
    
      'VVV': begin
        photometry_offset_definitions = [$
           'addcol -desc "2MASS - UKIDSS" -units mag  jDelta     "is_matched2 ? (Jmag -  jAperMag1) : NULL"',$
           'addcol -desc "2MASS - UKIDSS" -units mag  hDelta     "is_matched2 ? (Hmag -  hAperMag1) : NULL"',$
           'addcol -desc "2MASS - UKIDSS" -units mag  kDelta     "is_matched2 ? (Kmag - ksAperMag1) : NULL"' $
           ]
        end
    
      'VMC': begin
        photometry_offset_definitions = [$
           'addcol -desc "2MASS - VMC" -units mag  jDelta     "is_matched2 ? (Jmag -  jAperMag3) : NULL"',$
           'addcol -desc "2MASS - VMC" -units mag  kDelta     "is_matched2 ? (Kmag - ksAperMag3) : NULL"' $
           ]
        end
    endcase
    
    cmd = string($
         ; Master Catalog #1: GLIMPSE
         glimpse_repaired_catalog_fn,$
         strjoin(' icmd1='+quote_for_shell('')),$
         quote_for_shell(string(position_error_inflation, F='(%"RA DEC \"SigPosition*%0.2f\" ")')),$
    
         ; Catalog #2: Gaia
         Gaia_repaired_catalog_fn,$
         strjoin(' icmd2='+quote_for_shell([$
           'keepcols "designation ra_epochACIS dec_epochACIS epochACIS_error"'$
           ])),$
         quote_for_shell(string(position_error_inflation, F='(%"ra_epochACIS dec_epochACIS \"epochACIS_error*%0.2f\" ")')),$
    
         ; Catalog #3: NIR
         nir_aligned_catalog_fn,$
         strjoin(' icmd3='+quote_for_shell([$
           ; We DISCARD the UKIDSS/VISTA sources that are "near bright source", where spurious detections are likely
           ; and where GLIMPSE is likely to already have 2MASS photometry.
           ; We do NOT require our ?HighQuality criteria, which selects sources suitable for astrometric reference.
           'select "! near_bright_source"' $
           ])),$
         quote_for_shell(string(position_error_inflation, F='(%"RA DEC \"SigPosition*%0.2f\" ")')),$
    
         ; Calculate separation of matches.
         strjoin(' ocmd='+quote_for_shell([$
          'addcol -desc "GLIMPSE/Gaia separation" -units arcsec Separation1 "skyDistanceDegrees(RA_1,DEC_1,ra_epochACIS,dec_epochACIS)*3600"',$
          'addcol -desc "GLIMPSE/NIR separation"  -units arcsec Separation2 "skyDistanceDegrees(RA_1,DEC_1,RA_3,DEC_3)*3600"'$
           ])),$
    
         strjoin(' ocmd='+quote_for_shell([$
           ; Describe aggressiveness of matching.
           'addcol -desc "position error inflation in tmatch call" STILTS_match_Zlimit '+position_error_inflation, $
           ; Identify matches
           'addcol -desc "GLIMPSE/Gaia match" is_matched1 "(! null_Separation1)"',$
           'addcol -desc "GLIMPSE/NIR match"  is_matched2 "(! null_Separation2)"',$
           ; ; Calculate NIR photometry offsets
           photometry_offset_definitions $
           ])),$
    
         temp_catalog_fn,$
         F='(%"stilts tmatchn multimode=pairs nin=3 matcher=skyerr params=1.0 join1=always  in1=%s %s values1=%s   in2=%s %s values2=%s   in3=%s %s values3=%s  fixcols=dups %s %s omode=out out=%s ")')
  
  
  ;;; The matching above IS NOT EQUIVALENT TO MATCHING GLIMPSE TO THE Gaia/NIR union WE BUILT EARLIER!!!!!!!!
  ;;; The gaia_nir_astrometric_union_fn file was built with a SUBET OF NIR SOURCES (about half in M17SWex) that are supposed to have accurate positions.
  ;;; Here, we're interested in photometry and do not want to discard half our NIR sources!!!!
  endelse ; NIR catalog is UKIDSS/VVV/VMC

  run_command, cmd, /NO_RETRY
  file_gzip, temp_catalog_fn, glimpse_plus_catalog_fn, /DELETE, /VERBOSE

endelse ; GLIMPSE has rows



;; ==========================================================================
; SAGE Catalog (Spitzer)
;; ==========================================================================
; SAGE is described at https://irsa.ipac.caltech.edu/data/SPITZER/SAGE/
; We think the GLIMPSE team has not processed the SAGE observations.
; We get SAGE sources from Vizier's "SAGE LMC and SMC IRAC Source Catalog" catalog (II/305/archive), which is probably a combination of two IRSA catalogs.



sage_repaired_catalog_fn='SAGE/Catalog.fits.gz'

file_delete, /ALLOW_NONEXISTENT, download_catalog_fn
ae_download_project_catalog, TEMP_DIR=tempdir, Vizier, 'II/305/archive', 'RAJ2000','DEJ2000', download_catalog_fn, NUM_ROWS=num_rows_SAGE

if (num_rows_SAGE EQ 0) then begin
  print, F='(%"\n\nINFORMATION: SAGE catalog has no sources on this field.\n")'

endif else begin
  ;; -------------------------------------------
  ; Repair problems in the downloaded catalog.
  ; The SAGE download from Vizier contains both "Catalog" and "Archive" sources.
  ; We want the Catalog (highly reliable) sources, identified by "1" in the "SSTISAGEMA" column.
  stilts_processing_filter = [$
      ; Name the table.
      'tablename "SAGE"',$
      ; Select "Catalog" sources.
      'select ''equals(toString(C),"1")''',$
      ; Trim to the rectangular project field of view.
      'select "isInside(RAJ2000,DEJ2000, '+project_isInside_parameter+')"'$
      ] 
  
  run_command, string(download_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), repaired_catalog_fn,$
                      F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
  
  ;; -------------------------------------------
  ;; Estimate shift between the SAGE catalog (repaired_catalog_fn) and our "red" Gaia sub-catalog.
  ;; Estimate SAGE position uncertainties on this field.
  
  ; As a starting point, we previously estimated the following SAGE position uncertainties on the T-ReX field.
  ra_error_expression  = '0.130' ; arcsec
  dec_error_expression = '0.109' ; arcsec
  ;
  ; We currently do not have any indicator of unreliable detections in this SAGE catalog, so the
  ; call below has no FILTER_EXPRESSION option.
  this_cat = build_FITS_cat(repaired_catalog_fn, event2wcs_astr, /ONLY_COORDINATES, NAME='SAGE',$
                                  RA_EXPRESSION='tb.RAJ2000',$
                                 DEC_EXPRESSION='tb.DEJ2000',$
                            RA_ERROR_EXPRESSION=ra_error_expression,$
                           DEC_ERROR_EXPRESSION=dec_error_expression)
  
  match_xy_tune_uncertainties, gaia_red_cat, gaia_red_cat[0].CATALOG_NAME,$
                                this_cat, this_cat[0].CATALOG_NAME,$
                                0.99, ASTROMETRY=event2wcs_astr,$
                                OUTPUT_DIR='SAGE',$
                                /SKIP_REGIONS,$
                               ;REGION_FILENAME='Gaia_SAGE.reg',$
                                OBS_XSHIFT=obs_xshift,$
                                OBS_YSHIFT=obs_yshift,$
                          ERROR_OBS_XSHIFT=obs_xshift_error,$
                          ERROR_OBS_YSHIFT=obs_yshift_error,$
                                    X_ERROR_RECOMMENDED_SCALING=x_error_recommended_scaling,$
                                    Y_ERROR_RECOMMENDED_SCALING=y_error_recommended_scaling
  
  ;; -------------------------------------------
  ;; Shift full SAGE catalog, assign position uncertainties, and save.
  ;; OBS_XSHIFT,OBS_YSHIFT estimates above are in Chandra SKY system.
  ;; We want to alter RA/DEC coordinates; the STILTS function epochProp() is handy.
  ;; We DO NOT ALTER Galactic cooordinates!
  
  ; Parameters of STILTS function epochProp()
  ;   0:   ra                deg     right ascension
  ;   1:   dec               deg     declination
  ;   2:   parallax          mas     parallax
  ;   3:   pmra              mas/yr  proper motion in ra * cos(dec)
  ;   4:   pmdec             mas/yr  proper motion in dec
  ; Below, a negative value in event2wcs_astr.cdelt[1] implements the sign change between Xshift and RAshift.
  ra_dec_shift_in_mas = string([obs_xshift,obs_yshift] * (event2wcs_astr.cdelt * 3600D) * 1000D, F='(%"%0.1f")')
  
  astrom6 = string(ra_dec_shift_in_mas, F='(%"array(RAJ2000,DEJ2000, 0.0, %s,%s)")')
  
  stilts_processing_filter = [$
      ; Name the table.
      'tablename "SAGE, shifted positions"',$
      ; Record the shifts we're applying.  
      ; These go in the Table Parameters data structure invented by TopCat/STILTS (stored as an image in the first HDU).
      ; These are not FITS keywords, and cannot be accessed by standard FITS software.
      'setparam -desc "shift in RA direction"  -unit mas RA_shift '+ra_dec_shift_in_mas[0],$
      'setparam -desc "shift in DEC direction" -unit mas DE_shift '+ra_dec_shift_in_mas[1],$
  
      ; Shift positions; assign position uncertainties.
      'addcol -desc "astrometry parameters" astrom6 "epochProp(1,'+astrom6+')" ' ,$
      'addcol -desc "position (corrected)" -units degrees -before RAJ2000 RA  "astrom6[0]" ' ,$
      'addcol -desc "position (corrected)" -units degrees -before RAJ2000 DEC "astrom6[1]" ' ,$
      'addcol -desc "position error"       -units arcsec  -before RAJ2000 SigRA  '+ ra_error_expression ,$
      'addcol -desc "position error"       -units arcsec  -before RAJ2000 SigDEC '+dec_error_expression ,$
      'delcols "astrom6 RAJ2000 DEJ2000"' ,$
      ; add a 1-D position uncertainty.
      'addcol -desc "1-D position error" -after SigDEC -units arcsec SigPosition "sqrt(square(SigRA)+square(SigDEC))"'$
      ] 
  
  run_command, string(repaired_catalog_fn, strjoin(' cmd='+quote_for_shell(stilts_processing_filter)), temp_catalog_fn,$
                      F='(%"stilts tpipe in=''%s'' %s omode=out out=%s ")'), /NO_RETRY
  
  ; Record the shifts we applied as FITS header keywords.  (STILTS cannot do this.)
  run_command, string(temp_catalog_fn, ra_dec_shift_in_mas[0], $
                     F="(%'dmhedit infile=""%s[2]"" filelist=none operation=add key=RA_shift  value=%s comment=""[mas] shift in RA direction""')")
  run_command, string(temp_catalog_fn, ra_dec_shift_in_mas[1], $
                     F="(%'dmhedit infile=""%s[2]"" filelist=none operation=add key=DE_shift  value=%s comment=""[mas] shift in DEC direction""')")
  
  file_gzip, temp_catalog_fn, sage_repaired_catalog_fn, /DELETE, /VERBOSE
  
  print, ra_dec_shift_in_mas, F='(%"\nINFORMATION: SAGE RA/DEC coordinates shifted by %s, %s mas to align with Gaia.")'
  print, 1000*float( ra_error_expression)*x_error_recommended_scaling,$
         1000*float(dec_error_expression)*y_error_recommended_scaling,F='(%"INFORMATION: Estimated remaining SAGE position uncertainties on this field are %0.1f, %0.1f mas.\n")'
endelse ; SAGE has rows
  



;; ==========================================================================
;; Other Catalogs

; IPHAS and VPHAS+ H-alpha surveys, along with Sloan, might be useful for adding visual data to SED fitting.

;; Coverage for a specific region can be checked at https://irsa.ipac.caltech.edu/applications/Radar/


SKIP_CATALOGS:

;; ==========================================================================
;; BUILD OIR IMAGE MOSAICS
;; ==========================================================================
if keyword_set(skip_mosaics) then GOTO, SKIP_MOSAICS

print, F='(%"\n\n========================================\nBUILDING IMAGE MOSAICS\n========================================\n")'

; Parameter for ae_clean_oir_image tool.
sky_lower_cutoff_ratio = 1e-2

;; Look up our previous decision on which NIR survey has the best coverage.
NIR_name = file_readlink('NIR') 


wget_options = (!VERSION.OS EQ 'darwin') ? '--compression=auto' : ''


; We will convert all mosaics to single-precision to save space on disk and to speed up ds9.  
r4_fmt = '(%"dmcopy \"%s[opt type=r4]\" %s clob+")'


; The EASY way to run SIA queries involves simply including the search parameters in the service URL
; as part of any STILTS command that accepts a service URL.
; For example we can do the query and filter the result in one "tpipe" call, e.g.
;
;  stilts tpipe in='http://wfaudata.roe.ac.uk/ukidssdr8-siap/?&POS=274.6513,-16.8468&SIZE=0.667' \
;               cmd='select '\''equals(trim(Bandpass),"K")'\'''  out=UKIDSS/tile_list.fits
; which resolves to
;  stilts tpipe in=http://wfaudata.roe.ac.uk/ukidssdr8-siap/?&POS=274.6513,-16.8468&SIZE=0.667  cmd=select 'equals(trim(Bandpass),"K")' out=UKIDSS/tile_list.fits


;; ==========================================================================
;; Build NIR mosaics
template_row = {name:'', mosaic_fn:'', tilelist_fn:'', stilts_filter_expression:''}


case NIR_name of
  'UKIDSS': begin
    basedir    = 'UKIDSS/'
    serviceurl = 'http://wfaudata.roe.ac.uk/ukidssdr8-siap/?'
    ; Generic coneskymatch Simple Image Access query template.
    ; Parameter "SIZE" is in units of degrees.
    sia_region_spec = string(CENTER_COORDS, bbox_size_arcmin/60., F='(%"&POS=%0.4f,%0.4f&SIZE=%0.3f,%0.3f")')
    band = replicate(template_row,3)
    band.name                     = reverse(['J','H','K'])
    band.stilts_filter_expression = 'equals(trim(toString(Bandpass)),"'+band.name+'")'
    ; UKIDSS/VVV/VMC tiles seem to have 0.2" pixels; we down-sample to save disk space and speed ds9.
    arcsec_per_pixel=0.4
    end

  'VVV': begin
    basedir    = 'VVV/'
    serviceurl = 'http://wfaudata.roe.ac.uk/vista-siap/?'
    ; Generic coneskymatch Simple Image Access query template.
    ; Parameter "SIZE" is in units of degrees.
    sia_region_spec = string(CENTER_COORDS, bbox_size_arcmin/60., F='(%"&POS=%0.4f,%0.4f&SIZE=%0.3f,%0.3f")')
    band = replicate(template_row,5)
    band.name                     = reverse(['Z','Y','J','H','Ks'])
    band.stilts_filter_expression = 'equals(trim(toString(Bandpass)),"'+band.name+'")'
    ; UKIDSS/VVV/VMC tiles seem to have 0.2" pixels; we down-sample to save disk space and speed ds9.
    arcsec_per_pixel=0.4
    end

  'VMC': begin
    basedir    = 'VMC/'
    serviceurl = 'http://wfaudata.roe.ac.uk/vista-siap/?'
    ; Generic coneskymatch Simple Image Access query template.
    ; Parameter "SIZE" is in units of degrees.
    sia_region_spec = string(CENTER_COORDS, bbox_size_arcmin/60., F='(%"&POS=%0.4f,%0.4f&SIZE=%0.3f,%0.3f")')
    band = replicate(template_row,3)
    band.name                     = reverse(['Y','J','Ks']) 
    band.stilts_filter_expression = 'equals(trim(toString(Bandpass)),"'+band.name+'")'
    ; UKIDSS/VVV/VMC tiles seem to have 0.2" pixels; we down-sample to save disk space and speed ds9.
    arcsec_per_pixel=0.4
    end

  'Twomass': begin
    basedir    = 'Twomass/'
    serviceurl = 'https://irsa.ipac.caltech.edu/cgi-bin/2MASS/IM/nph-im_sia?type=at&ds=asky&'
    ; Generic coneskymatch Simple Image Access query template.
    ; Parameter "SIZE" is in units of degrees.
    ; This service seems to accept only cone regions (not rectangles)
    sia_region_spec = string(CENTER_COORDS, radius_arcmin/60., F='(%"&POS=%0.4f,%0.4f&SIZE=%0.3f")')
    band = replicate(template_row,3)
    band.name                     = reverse(['J','H','K'])
    band.stilts_filter_expression = 'equals(trim(toString(Band)),"'+band.name+'") && contains(format,"fits")'
    ; 2MASS images have 1" pixels.
    arcsec_per_pixel=1.0
    end

 else: begin
       print, F='(%"\nERROR: symlink NIR does not contain \"UKIDSS\", \"VVV\", \"VMC\", or \"Twomass\".")'
       message, 'Stopping for you to investigate ...'
       end
endcase



band.mosaic_fn                = basedir+band.name+'.fits.gz'        ; Filename of mosaic image.
band.tilelist_fn              = basedir+band.name+'.tile_list.fits' ; Filename of FITS table returned by SIA query.


; Run SIA query to get a list of tiles on our scene.
file_delete, temp_tile_list_fn, /ALLOW_NONEXISTENT
cmd = string(serviceurl+sia_region_spec, temp_tile_list_fn, F='(%"stilts tpipe in=''%s'' out=%s ")')
run_command, cmd, /NO_RETRY, STATUS=status

if keyword_set(status) then begin
  print, replicate(basedir,3), F='(%"\nERROR: SIA query for %s image tiles failed (above).\nIf the error message is ''No TABLE element found'' then perhaps there is no %s coverage.\nEither investigate coverage or type \".c\" to skip %s mosaics.\n")'
  stop
endif else if (psb_xpar( headfits(temp_tile_list_fn, EXT=1), 'NAXIS2') EQ 0) then begin
  print, basedir, F='(%"\nINFORMATION: No %s image tiles were found.\n")'
endif else begin
  ; Process tiles in each band.
  foreach this_band, band do begin
    file_delete, this_band.tilelist_fn, /ALLOW_NONEXISTENT
    stilts_processing_filter = string(this_band.stilts_filter_expression, F='(%"select ''%s''")')
    run_command, string(temp_tile_list_fn,$
                        'cmd='+quote_for_shell(stilts_processing_filter),$
                        this_band.tilelist_fn, F='(%"stilts tpipe in=''%s'' %s  out=%s ")'), /NO_RETRY
    
    ; Mosaic the images produced by the query.
    bt = mrdfits(this_band.tilelist_fn, 1, theader, /SILENT)
    num_tiles = psb_xpar( theader, 'NAXIS2')
    if (num_tiles EQ 0) then begin
      print, this_band.name, F='(%"\nINFORMATION: No %s image tiles were found.\n")'
    endif else begin
      ; Download the images to scratch space.
      ; We can NOT easily derive image filenamed from URLs, because a single FITS file on the server can have multiple images.  So, we build filenames from the OBSID column and sequence number.
      ; Montage wants tile filenames to end in '.fits'.
      if (NIR_name EQ 'Twomass') then begin
        URL_column = strtrim(bt.DOWNLOAD,2)
        tile_fn    = string(indgen(num_tiles), F='(%"tile%d.fits")')
      endif else begin
        URL_column = strtrim(bt.REFERENCE,2)
        tile_fn    = string(indgen(num_tiles), F='(%"tile%d_")')+strtrim(bt.OBSID,2)+'.fits' 
      endelse
      
      
      raw_tile_dir = tempdir + basedir + this_band.name + '/raw_tiles/'
          tile_dir = tempdir + basedir + this_band.name + '/tiles/'
      file_delete, /RECURSIVE,  tile_dir, /ALLOW_NONEXISTENT
      file_mkdir, raw_tile_dir, tile_dir

      print, num_tiles, basedir, this_band.name, F='(%"\nINFORMATION: Fetching %d image tiles for %s (%s band).\n")'

      ; wget's progress bar is not useful, because run_command buffers output until command finishes.
      for ii=0,num_tiles-1 do begin
        fn = raw_tile_dir + tile_fn[ii]
        if ~file_test(fn) then $
         run_command, string(wget_options, fn, URL_column[ii], F='(%"wget --no-verbose %s -O %s ''%s''")'), /NO_RETRY
      endfor ; ii
    
      ; Try to invalidate (set to NaN) small pixel values that are inconsistent with the sky background.
      id_dataset_1d = 0L
      for ii=0,num_tiles-1 do begin
        ; Read raw tile image.
        ; Single-precision tiles are good enough for our purposes.
        tile_img = float(readfits(/SILENT, raw_tile_dir+tile_fn[ii], tile_header))
        print, tile_fn[ii], F='(%"Cleaning tile %s")'

        ae_clean_oir_image, tile_img, REMOVE_NONPOSITIVE=0,$
                                  SKY_LOWER_CUTOFF_RATIO=sky_lower_cutoff_ratio,$
                                  WIDGET_TITLE=this_band.mosaic_fn, ID_DATASET_1D=id_dataset_1d, DATASET_NAME=tile_fn[ii]

        ; Save cleaned tile image.
        writefits, tile_dir+tile_fn[ii], tile_img, tile_header
      endfor ;ii

      ; Build the mosaic (http://montage.ipac.caltech.edu/docs/mExec.html).
      ; Write to scratch space for performance and to hide from backups. 
      ae_define_field_of_view, MHDR_COMMAND=mHdr_command, ARCSEC_PER_PIXEL=arcsec_per_pixel
      run_command, mHdr_command
             
      file_delete, /RECURSIVE, montage_dir, /ALLOW_NONEXISTENT
      ; The -l ("level-only") option to mExec (passed through to mBgModel) specifies the offset-only model for adjusting tile backgrounds (rather than the default tilted plane model).
      ; As of 2020 May, we have no idea which model is best for any particular telescope or data product.
      run_command,string(temp_image_fn, tile_dir, montage_dir, montage_logfile, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s -r %s %s |& tee %s")'),/NO_RETRY
    
      ; Convert to single-precision (to save space) and store in final location
      run_command,string(temp_image_fn, F=r4_fmt, this_band.mosaic_fn)
    endelse ; NAXIS2 GT 0
  endforeach ; band
endelse ; SIA query returned results



;; ==========================================================================
;; Build Spitzer mosaics
;; See https://irsa.ipac.caltech.edu/ibe/sia.html
basedir     = 'Spitzer/'

; Low-level images are available from the IRSA Image Server
;serviceurl = 'https://irsa.ipac.caltech.edu/SIA?'
;sia_region_spec = string(CENTER_COORDS, RADIUS_ARCMIN/60., F='(%"POS=circle+%0.4f+%0.4f+%0.3f&DPTYPE=image&FORMAT=image/fits&CALIB=3")')


; High-level images are available from the Spitzer Enhanced Imaging Products server.
; https://irsa.ipac.caltech.edu/data/SPITZER/Enhanced/SEIP/overview.html
; Every Super Mosaic comes in a "mean" version (IRAC.1.mosaic.fits) and a "median" version (IRAC.1.median_mosaic.fits).

; Query:       https://irsa.ipac.caltech.edu/cgi-bin/Atlas/nph-atlas?mission=SEIP&hdr_location=%5CSEIPDataPath%5C&SIAP_ACTIVE=1&collection_desc=SEIP&POS=250.3581%2c-47.0951&SIZE=0.375&FORMAT=image%2ffits
serviceurl = 'https://irsa.ipac.caltech.edu/cgi-bin/Atlas/nph-atlas?mission=SEIP&hdr_location=%5CSEIPDataPath%5C&SIAP_ACTIVE=1&collection_desc=SEIP&'
sia_region_spec = string(CENTER_COORDS, RADIUS_ARCMIN/60., F='(%"POS=%0.4f,%0.4f&SIZE=%0.3f&FORMAT=image/fits")')
; Tiles seem to have 0.6" pixels.
arcsec_per_pixel=0.6

band = replicate(template_row,2)
band.name                     = ['IRAC4','MIPS24'] ; 'IRAC1','IRAC2','IRAC3',,'MIPS70','MIPS160'
band.mosaic_fn                = basedir+band.name+'.fits.gz'        ; Filename of mosaic image.
band.tilelist_fn              = basedir+band.name+'.tile_list.fits' ; Filename of FITS table returned by SIA query.
; Every Super Mosaic comes in a "mean" version (e.g. IRAC.1.mosaic.fits) and a "median" version (e.g. IRAC.1.median_mosaic.fits). 
; I'll use the median images, since median is a more robust statistic than mean, and we don't care about photometry in these images.
band.stilts_filter_expression = string(band.name,F='(%"equals(trim(band_name),\"%s\") && equals(trim(file_type),\"science\") && contains(fname,\"median\") ")')


; Run SIA query to get a list of tiles on our scene.
file_delete, temp_tile_list_fn, /ALLOW_NONEXISTENT
cmd = string(serviceurl+sia_region_spec, temp_tile_list_fn, F='(%"stilts tpipe in=''%s'' out=%s ")')
run_command, cmd, /NO_RETRY, STATUS=status

if keyword_set(status) then begin
  print, replicate(basedir,3), F='(%"\nERROR: SIA query for %s image tiles failed (above).\nIf the error message is ''No TABLE element found'' then perhaps there is no %s coverage.\nEither investigate coverage or type \".c\" to skip %s mosaics.\n")'
  stop
endif else if (psb_xpar( headfits(temp_tile_list_fn, EXT=1), 'NAXIS2') EQ 0) then begin
  print, basedir, F='(%"\nINFORMATION: No %s image tiles were found.\n")'
endif else begin
  ; Process tiles in each band.
  foreach this_band, band do begin
    file_delete, this_band.tilelist_fn, /ALLOW_NONEXISTENT
    stilts_processing_filter = string(this_band.stilts_filter_expression, F='(%"select ''%s''")')
    run_command, string(temp_tile_list_fn,$
                        'cmd='+quote_for_shell(stilts_processing_filter),$
                        this_band.tilelist_fn, F='(%"stilts tpipe in=''%s'' %s  out=%s ")'), /NO_RETRY
    
    ; Mosaic the images produced by the query.
    bt = mrdfits(this_band.tilelist_fn, 1, theader, /SILENT)
    num_tiles = psb_xpar( theader, 'NAXIS2')
    if (num_tiles EQ 0) then begin
      print, this_band.name, F='(%"\nINFORMATION: No %s image tiles were found.\n")'
    endif else begin
      ; Download the images to scratch space.
      ; We can NOT easily derive image filenamed from URLs, because a single FITS file on the server can have multiple images.  So, we build filenames from the OBSID column and sequence number.
      ; Montage wants tile filenames to end in '.fits'.
      URL_column = strtrim(bt.SIA_URL,2)
      tile_fn    = string(indgen(num_tiles), F='(%"tile%d_")')+file_basename(strtrim(bt.FNAME,2)) 
      
      raw_tile_dir = tempdir + basedir + this_band.name + '/raw_tiles/'
          tile_dir = tempdir + basedir + this_band.name + '/tiles/'
      file_delete, /RECURSIVE,  tile_dir, /ALLOW_NONEXISTENT
      file_mkdir, raw_tile_dir, tile_dir

      print, num_tiles, basedir, this_band.name, F='(%"\nINFORMATION: Fetching %d image tiles for %s (%s band).\n")'

      ; wget's progress bar is not useful, because run_command buffers output until command finishes.
      for ii=0,num_tiles-1 do begin
        fn = raw_tile_dir + tile_fn[ii]
        if ~file_test(fn) then $
         run_command, string(wget_options, fn, URL_column[ii], F='(%"wget --no-verbose %s -O %s ''%s''")'), /NO_RETRY
      endfor ; ii

      ; Try to invalidate (set to NaN) small pixel values that are inconsistent with the sky background.
      id_dataset_1d = 0L
      for ii=0,num_tiles-1 do begin
        ; Read raw tile image.
        ; Single-precision tiles are good enough for our purposes.
        tile_img = float(readfits(/SILENT, raw_tile_dir+tile_fn[ii], tile_header))
        print, tile_fn[ii], F='(%"Cleaning tile %s")'

        ae_clean_oir_image, tile_img, REMOVE_NONPOSITIVE=0,$
                                  SKY_LOWER_CUTOFF_RATIO=sky_lower_cutoff_ratio,$
                                  WIDGET_TITLE=this_band.mosaic_fn, ID_DATASET_1D=id_dataset_1d, DATASET_NAME=tile_fn[ii]

        ; Save cleaned tile image.
        writefits, tile_dir+tile_fn[ii], tile_img, tile_header
      endfor ;ii
      
    
      ; Build the mosaic (http://montage.ipac.caltech.edu/docs/mExec.html).
      ; Write to scratch space for performance and to hide from backups. 
      ae_define_field_of_view, MHDR_COMMAND=mHdr_command, ARCSEC_PER_PIXEL=arcsec_per_pixel
      run_command, mHdr_command
    
      file_delete, /RECURSIVE, montage_dir, /ALLOW_NONEXISTENT
      ; The -l ("level-only") option to mExec (passed through to mBgModel) specifies the offset-only model for adjusting tile backgrounds (rather than the default tilted plane model).
      ; As of 2020 May, we have no idea which model is best for any particular telescope or data product.
      run_command,string(temp_image_fn, tile_dir, montage_dir, montage_logfile, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s -r %s %s |& tee %s")'),/NO_RETRY
    
      ; Convert to single-precision (to save space) and store in final location
      run_command,string(temp_image_fn, F=r4_fmt, this_band.mosaic_fn)
    endelse ; NAXIS2 GT 0
  endforeach ; band
endelse ; SIA query returned results



;; ==========================================================================
;; Build Herschel mosaics
basedir     = 'Herschel/'

; The Herschel Science Archive at ESA lists "SimpleImage" products, but 
; the SIA table does not have an energy band column, and seems useless.
; serviceurl = 'http://archives.esac.esa.int/hsa/aio/jsp/siap.jsp?' 

; IPAC's Herschel High Level Images service provides an SIA table with an energy band column.
; https://irsa.ipac.caltech.edu/data/Herschel/HHLI/overview.html
serviceurl  = 'https://irsa.ipac.caltech.edu/cgi-bin/Atlas/nph-atlas?mission=HHLI&hdr_location=%5CHHLIDataPath%5C&SIAP_ACTIVE=1&collection_desc=HHLI&'
; Tiles seem to have 6.12" pixels.
arcsec_per_pixel=6.12

sia_region_spec = string(CENTER_COORDS, RADIUS_ARCMIN/60., F='(%"POS=%0.4f,%0.4f&SIZE=%0.3f&FORMAT=image/fits")')

band = replicate(template_row,2)
band.name                     = ['PACS70','SPIRE250']
band.mosaic_fn                = basedir+band.name+'.fits.gz'        ; Filename of mosaic image.
band.tilelist_fn              = basedir+band.name+'.tile_list.fits' ; Filename of FITS table returned by SIA query.
band.stilts_filter_expression = string(band.name,F='(%"equals(trim(band_name),\"%s\") && equals(trim(file_type),\"science\")    ")')


; Run SIA query to get a list of tiles on our scene.
file_delete, temp_tile_list_fn, /ALLOW_NONEXISTENT
cmd = string(serviceurl+sia_region_spec, temp_tile_list_fn, F='(%"stilts tpipe in=''%s'' out=%s ")')
run_command, cmd, /NO_RETRY, STATUS=status

if keyword_set(status) then begin
  print, replicate(basedir,3), F='(%"\nERROR: SIA query for %s image tiles failed (above).\nIf the error message is ''No TABLE element found'' then perhaps there is no %s coverage.\nEither investigate coverage or type \".c\" to skip %s mosaics.\n")'
  stop
endif else if (psb_xpar( headfits(temp_tile_list_fn, EXT=1), 'NAXIS2') EQ 0) then begin
  print, basedir, F='(%"\nINFORMATION: No %s image tiles were found.\n")'
endif else begin
  ; Process tiles in each band.
  foreach this_band, band do begin
    file_delete, this_band.tilelist_fn, /ALLOW_NONEXISTENT
    stilts_processing_filter = string(this_band.stilts_filter_expression, F='(%"select ''%s''")')
    run_command, string(temp_tile_list_fn,$
                        'cmd='+quote_for_shell(stilts_processing_filter),$
                        this_band.tilelist_fn, F='(%"stilts tpipe in=''%s'' %s  out=%s ")'), /NO_RETRY
    
    ; Mosaic the images produced by the query.
    bt = mrdfits(this_band.tilelist_fn, 1, theader, /SILENT)
    num_tiles = psb_xpar( theader, 'NAXIS2')
    if (num_tiles EQ 0) then begin
      print, this_band.name, F='(%"\nINFORMATION: No %s image tiles were found.\n")'
    endif else begin
      ; Download the images to scratch space.
      ; We can NOT easily derive image filenamed from URLs, because a single FITS file on the server can have multiple images.  So, we build filenames from the OBSID column and sequence number.
      ; Montage wants tile filenames to end in '.fits'.
      URL_column = strtrim(bt.SIA_URL,2)
      tile_fn    = string(indgen(num_tiles), F='(%"tile%d_")')+file_basename(strtrim(bt.FNAME,2)) 
      
      raw_tile_dir = tempdir + basedir + this_band.name + '/raw_tiles/'
          tile_dir = tempdir + basedir + this_band.name + '/tiles/'
      file_delete, /RECURSIVE,  tile_dir, /ALLOW_NONEXISTENT
      file_mkdir, raw_tile_dir, tile_dir

      print, num_tiles, basedir, this_band.name, F='(%"\nINFORMATION: Fetching %d image tiles for %s (%s band).\n")'

      ; wget's progress bar is not useful, because run_command buffers output until command finishes.
      for ii=0,num_tiles-1 do begin
        fn = raw_tile_dir + tile_fn[ii]
        if ~file_test(fn) then begin
         run_command, string(wget_options, fn, URL_column[ii], F='(%"wget --no-verbose %s -O %s ''%s''")'), /NO_RETRY
  
         ; These Herschel FITS files downloaded contain multiple image HDU's.
         ; Montage cannot handle that---we need to pull out the HDU's named "image".
         file_move, /OVERWRITE, fn, temp_image_fn
         run_command, string(temp_image_fn, fn,  F='(%"dmcopy \"%s[image]\" %s clob+")')
        endif
      endfor ; ii
  
      ; Try to invalidate (set to NaN) small pixel values that are inconsistent with the sky background.
      id_dataset_1d = 0L
      for ii=0,num_tiles-1 do begin
        ; Read raw tile image.
        ; Single-precision tiles are good enough for our purposes.
        tile_img = float(readfits(/SILENT, raw_tile_dir+tile_fn[ii], tile_header))
        print, tile_fn[ii], F='(%"Cleaning tile %s")'

        ae_clean_oir_image, tile_img, REMOVE_NONPOSITIVE=0,$
                                  SKY_LOWER_CUTOFF_RATIO=sky_lower_cutoff_ratio,$
                                  WIDGET_TITLE=this_band.mosaic_fn, ID_DATASET_1D=id_dataset_1d, DATASET_NAME=tile_fn[ii]

        ; Save cleaned tile image.
        writefits, tile_dir+tile_fn[ii], tile_img, tile_header
      endfor ;ii

      ; Build the mosaic (http://montage.ipac.caltech.edu/docs/mExec.html).
      ; Write to scratch space for performance and to hide from backups. 
      ae_define_field_of_view, MHDR_COMMAND=mHdr_command, ARCSEC_PER_PIXEL=arcsec_per_pixel
      run_command, mHdr_command
    
      file_delete, /RECURSIVE, montage_dir, /ALLOW_NONEXISTENT
      ; The -l ("level-only") option to mExec (passed through to mBgModel) specifies the offset-only model for adjusting tile backgrounds (rather than the default tilted plane model).
      ; As of 2020 May, we have no idea which model is best for any particular telescope or data product.
      run_command,string(temp_image_fn, tile_dir, montage_dir, montage_logfile, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s -r %s %s |& tee %s")'),/NO_RETRY
    
      ; Convert to single-precision (to save space) and store in final location
      run_command,string(temp_image_fn, F=r4_fmt, this_band.mosaic_fn)
    endelse ; NAXIS2 GT 0
  endforeach ; band
endelse ; SIA query returned results
 



;; ==========================================================================
;; ADDITIONAL NOTES on SIA Queries

; If we have do do SIA queries in TopCat ...
;   The SIA tool requires a bounding box (not rectangle); get from ae_define_field_of_view output.
;   Define subset categories on band/filter column; select K-band.
;   If field is far from square, plot image centers in TopCat and draw region to define "download" subset.
;   Build a "filename" column as concat("-O K_", ObsID, "_", $00, ".fits"); put at head of column list (for wget)
;   Save "filename" and "url" columns in "ascii" format.
;   Use editor to massage into wget calls; URLS have to be in single-quotes.
;
; The HARD way to run Simple Image Access (SIA) queries from STITS is like this: 
; stilts coneskymatch servicetype=sia \
;        serviceurl='http://wfaudata.roe.ac.uk/ukidssdr8-siap/?' \
;        ifmt=ascii in='project_fov.txt' ra='$1' dec='$2' sr='1.333/2' \
;        dataformat='image/fits' \
;        out=fitsimages.fits
;
; Write center coordinates to temp ASCII file.
;forprint, /NoCom, TEXTOUT=temp_text_fn, CENTER_COORDS
;
;fmt = '(%"stilts coneskymatch servicetype=sia serviceurl=''%s'' ifmt=ascii in=%s ra=''$1'' dec=''$2'' sr=%0.3f dataformat=''image/fits'' out=%s ")'
;cmd = string(serviceurl, temp_text_fn, max(bbox_size_arcmin)/60./2,$
;             basedir+tilelist_fn, F=fmt)

; To avoid putting the target coordinates in a file this more complex form could be used:
; stilts tloop start=0 end=1 ofmt=votable | \
;   stilts coneskymatch servicetype=sia ifmt=votable in=- \
;          ra=274.651343 dec=-16.846819 \
;          sr='1.333/2' dataformat='image/fits'
;          serviceurl='http://wfaudata.roe.ac.uk/ukidssdr8-siap/?'



; ;; ==========================================================================
; ;; Build 2MASS and WISE mosaics that cover the project field-of-view, using Montage.
; ;
; ; Note that the pixel grid defined by ae_define_field_of_view has zero rotation by default.
; ; We have found that the IPAC mosaic service at http://hachi.ipac.caltech.edu:8080/montage/index.html does not work for large fields of view.  (NASA's SkyView mosaic service may handle large fields.)  Instead, we use command-line tools in the Montage package (http://montage.ipac.caltech.edu/).
; 
; ; https://irsa.ipac.caltech.edu/data/2MASS/docs/sixdeg/
; ae_define_field_of_view, MHDR_COMMAND=mHdr_command, ARCSEC_PER_PIXEL=1.0
; run_command, mHdr_command
; file_delete, /RECURSIVE, montage_dir, /ALLOW_NONEXISTENT
; run_command,string(temp_image_fn,montage_dir, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s 2MASS J  %s")'),/NO_RETRY
; run_command,string(temp_image_fn, F=r4_fmt, 'Twomass/2MASS-J.fits.gz')
; 
; run_command,string(temp_image_fn,montage_dir, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s 2MASS H  %s")'),/NO_RETRY
; run_command,string(temp_image_fn, F=r4_fmt, 'Twomass/2MASS-H.fits.gz')
; 
; run_command,string(temp_image_fn,montage_dir, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s 2MASS K  %s")'),/NO_RETRY
; run_command,string(temp_image_fn, F=r4_fmt, 'Twomass/2MASS-K.fits.gz')
;  
; 
; ; http://wise2.ipac.caltech.edu/docs/release/allsky/expsup/sec1_1.html
; ae_define_field_of_view, MHDR_COMMAND=mHdr_command, ARCSEC_PER_PIXEL=2.75
; run_command, mHdr_command
; run_command,string(temp_image_fn,montage_dir, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s WISE 3.4 %s")'),/NO_RETRY
; run_command,string(temp_image_fn, 'WISE/w1.fits.gz', F=r4_fmt)
; 
; run_command,string(temp_image_fn,montage_dir, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s WISE 4.6 %s")'),/NO_RETRY
; run_command,string(temp_image_fn, 'WISE/w2.fits.gz', F=r4_fmt)
; 
      ; The -l ("level-only") option to mExec (passed through to mBgModel) specifies the offset-only model for adjusting tile backgrounds (rather than the default tilted plane model).
      ; As of 2020 May, we have no idea which model is best for any particular telescope or data product.
run_command,string(temp_image_fn,montage_dir, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s WISE 12  %s")'),/NO_RETRY
run_command,string(temp_image_fn, 'WISE/w3.fits.gz', F=r4_fmt)

run_command,string(temp_image_fn,montage_dir, F='(%"mExec -d2 -q -l -c -f region.hdr -o %s WISE 22  %s")'),/NO_RETRY
run_command,string(temp_image_fn, 'WISE/w4.fits.gz', F=r4_fmt)





;; ==========================================================================
;; Show the mosaics we have available.
mosaic_list = file_search( ['NIR/[ZYJHK]*.fits.gz','Spitzer/{IRAC,MIPS}*.fits.gz','Herschel/{PACS,SPIRE}*.fits.gz','WISE/{w3,w4}.fits.gz'] )


;Twomass/2MASS-?.fits.gz WISE/w?.fits.gz VVV/?.fits.gz UKIDSS/?.fits.gz
run_command,string(getenv('TARGET'), strjoin(mosaic_list,' '),$
  F='(%"ds9 -title \"%s\" -lock frame wcs -log -scale mode 99.0 %s -zoom to fit -region load all project_fov.reg -print resolution 150 -print level 3 -print color rgb -print destination file -print filename IR_coverage.ps -print &")')



SKIP_MOSAICS:

; Show the field-of-view definition.
ae_define_field_of_view


CLEANUP:
; Remove empty directories.
foreach dir, ['Gaia','Twomass','UKIDSS','VVV','VMC','Spitzer','GLIMPSE','WISE','Herschel'] do begin
  dum = file_search(dir, '*', COUNT=count)
  if (count EQ 0) then begin
    print, 'Removing the empty directory '+dir
    file_delete, dir
  endif
endforeach


;; We retain scratch space (tempdir) for observer to investigate problems and/or to re-run mosaic section without re-downloading tiles.

print, 'ae_build_counterpart_products finished'
end ; ae_build_counterpart_products






;==========================================================================
;;; Read an AE "target parameter file" (target_parameters.par)

;;; This file must be in "DTF" format, as defined by the CIAO "ascii kernel", for example:

; # ACIS Extract Target Parameter File
;
; INVALID_THRESHOLD    = 0.01       ; source is valid if Pb < 0.01 (1%)
; MERGE_FOR_POSITION   = 0          ; no optimized merge for positions
; MERGE_FOR_PHOTOMETRY = 0          ; no optimized merge for photometry
; PB_IN_VHARD_BAND     = 1          ; include "very hard" band in Pb calculations
; NEIGHBOR_INVALID_THRESHOLD = 0.10 ;single-ObsID Pb threshold for ignoring neighbors in ae_better_backgrounds
; 
;==========================================================================
FUNCTION ae_get_target_parameters

; Default parameter values are defined here.
par = { INVALID_THRESHOLD         : 0.01,$ ; source is valid if Pb < 0.01 (1%)
        PB_IN_VHARD_BAND          : 0B, $
        MERGE_FOR_PB              : 0B, $  ; Default is "disabled", to avoid selection bias in Pb calculations.
        MERGE_FOR_POSITION        : 1B, $
        MERGE_FOR_PHOTOMETRY      : 1B, $
        CHECK_FOR_PILEUP          : 0B, $
        CHECK_FOR_AFTERGLOWS      : 0B, $
        SKIP_RESIDUALS            : 0B, $
        DISCARD_PSF               : 0B, $
        REUSE_NEIGHBORHOOD        : 0B, $
        VALIDATION_MODE           : 0B, $
        REVIEW_PRUNING            : 0B, $
        NEIGHBOR_INVALID_THRESHOLD: 0.10, $ ; neighbor participates in ABB when its single-ObsID Pb < 0.10 (10%)
        NUMBER_OF_PASSES          :  1  , $ ; full executions of ae_better_backgrounds desired
        BACKGROUND_MODEL_FILENAME : ''    $ ; BACKGROUND_MODEL_FILENAME parameter to ABB
      }
      
tags = tag_names(par)


; Declarations in target_parameters.par over-ride the defaults.
parfile = 'target_parameters.par'

if file_test(parfile) && (file_lines(parfile) GT 0) then begin
  print, 'Reading target parameters from ', parfile
  readcol, parfile, lines, F='(A)', COMMENT='#', DELIM='@'

  num_lines = n_elements(lines)
  parnames  = stregex(/EXTRACT, lines, '^[[:alnum:]_]+')

  for ii=0,num_lines-1 do begin
  
    parname = parnames[ii]
    
    if array_equal( (tags EQ parname), 0B) then begin
      print, parname, F='(%"ae_get_target_parameters: WARNING parameter %s is not recognized.")'
      print, '  ', lines[ii]
      continue
    endif

    print, '  ', lines[ii]
    if ~execute( 'par.'+lines[ii] ) then message, 'Problem in ae_get_target_parameters!'
  endfor ;ii
  
endif else print, parfile, F='(%"This target has no parameter file (%s).")'

  
; Certain boolean parameters are then declared to be TRUE by the presence of files by the same name.

if file_test('PB_IN_VHARD_BAND'    ) then begin 
  par.PB_IN_VHARD_BAND     = 1B
  print, '  PB_IN_VHARD_BAND file found.'
endif

if file_test('CHECK_FOR_PILEUP'    ) then begin 
  par.CHECK_FOR_PILEUP     = 1B
  print, '  CHECK_FOR_PILEUP file found.'
endif

if file_test('CHECK_FOR_AFTERGLOWS') then begin 
  par.CHECK_FOR_AFTERGLOWS = 1B
  print, '  CHECK_FOR_AFTERGLOWS file found.'
endif

if file_test('SKIP_RESIDUALS'      ) then begin 
  par.SKIP_RESIDUALS       = 1B
  print, '  SKIP_RESIDUALS file found.'
endif

if file_test('DISCARD_PSF') then begin 
  par.DISCARD_PSF = 1B
  print, '  DISCARD_PSF file found.'
endif

if file_test('REUSE_NEIGHBORHOOD') then begin 
  par.REUSE_NEIGHBORHOOD = 1B
  print, '  REUSE_NEIGHBORHOOD file found.'
endif

if file_test('VALIDATION_MODE') then begin 
  par.VALIDATION_MODE = 1B
  print, '  VALIDATION_MODE file found.'
endif

if file_test('REVIEW_PRUNING') then begin 
  par.REVIEW_PRUNING = 1B
  print, '  REVIEW_PRUNING file found.'
endif

return, par
end ; ae_get_target_parameters



;==========================================================================
;;; ae_compare_spectra

;; Compare the shapes of the two spectra over CHANNEL_RANGE using a 2-sample Kolmogorov-Smirnov statistic.
;==========================================================================
PRO ae_compare_spectra, spectrum1_fn, spectrum2_fn, CHANNEL_RANGE=channel_range

creator_string = "ae_compare_spectra, version " +strmid("$Rev:: 5661  $",7,5) +strmid("$Date: 2022-02-13 07:08:42 -0700 (Sun, 13 Feb 2022) $", 6, 11)

print, creator_string, F='(%"\n\n%s")'
print, now()

  if (n_elements(channel_range) NE 2) then channel_range=[35,548]
  min_channel = fix(channel_range[0])
  max_channel = fix(channel_range[1])
  
  spectrum = replicate({filename:'', total_observed_counts:0L, cumulative_distn_ptr:ptr_new()}, 2)
  
  spectrum[0].filename=spectrum1_fn
  spectrum[1].filename=spectrum2_fn
  
  ; Calculate cumulative distribution for each spectra in the energy band specified.
  for ii=0,1 do begin
    sp = spectrum[ii]
    
    bt = mrdfits(sp.filename,1, /SILENT)
    channels        = bt.CHANNEL 
    observed_counts = bt.COUNTS 
    
    band_index = where((channels GE min_channel) AND (channels LE max_channel))
    inband_channels =        channels[band_index]
    inband_spectrum = observed_counts[band_index]
    
    sp.total_observed_counts =         total(/INT, inband_spectrum)
    sp.cumulative_distn_ptr  = ptr_new(total(inband_spectrum, /CUMULATIVE) / sp.total_observed_counts)
    
    function_1d, id0, inband_channels, *sp.cumulative_distn_ptr, DATASET=sp.filename, XTIT='channel', YTIT='cumulative distribution', LEGEND_STYLE=1
    
    spectrum[ii] = sp
  endfor ; ii

  ; Calculate KS statistic, and its p-value.
  n_eff =     (spectrum[0].total_observed_counts * spectrum[1].total_observed_counts) / $
         float(spectrum[0].total_observed_counts + spectrum[1].total_observed_counts)
  
  ks = max( abs( *spectrum[0].cumulative_distn_ptr - *spectrum[1].cumulative_distn_ptr ), imax )
  prob_ks, ks, n_eff, ks_spect

  print, spectrum.filename, channel_range, spectrum.total_observed_counts, ks_spect, F='(%"\nEvaluating the null hypothesis that %s and %s are observations of the same spectrum.\nOver the channel range [%d:%d], %d, %d counts were observed.\nKS statistic has p-value %0.3g")'
  
  print, F='(%"\nTHIS CALCULATION ASSUMES THAT THE ARFs FOR THESE TWO SPECTRA ARE IDENTICAL.")'
  
  function_1d, id0, PLOT_WINDOW_OPTIONS=string(inband_channels[imax], F='(%"SET_BIG_MARKER=[%d,0.5]")'), TIT=string(ks_spect, F='(%"p-value = %0.3g")')

return
end ; ae_compare_spectra




;==========================================================================
PRO plot_Pb_under_null_hypothesis

; Assume AE adjusted BACKSCAL range to get 100 counts in bkg region.
BKG_CNTS = 100

; Assume null hypothesis: no counts from star.

; Simulate a wide range of background levels, expressed as expected bkg counts in aperture.
num_backgrounds = 51
expected_bkg_counts_in_aperture = 3*10^(-findgen(num_backgrounds)/10D)

BACKSCAL = BKG_CNTS / expected_bkg_counts_in_aperture


; Simulate Pb for small SRC_CNTS values.
for SRC_CNTS=1,5 do begin
  Pb = dblarr(num_backgrounds)
  for jj=0,num_backgrounds-1 do Pb[jj] = binomial_nr(SRC_CNTS, SRC_CNTS + BKG_CNTS, 1D/(1D + BACKSCAL[jj]))

  function_1d, id1, Pb, expected_bkg_counts_in_aperture, DATASET=string(SRC_CNTS,F='(%"SRC_CNTS=%d")')
endfor ; SRC_CNTS
end ; plot_Pb_quantization






;==========================================================================
PRO acis_extract_tools
return
end

