;;; $Id: acis_extract.pro 5658 2022-01-25 14:06:00Z psb6 $
;;; A tool for extraction of ACIS point sources.
;;; Patrick Broos & Leisa Townsley, Penn State University, 2002
;;; patrick.broos@icloud.com   townsley@astro.psu.edu

;;; See extensive manual at http://personal.psu.edu/psb6/TARA/ae_users_guide.html

;;; This software is offered freely to the ACIS observer community under these 
;;; conditions:
;;; 1. You will not remove any of the header comments in the original file.
;;; 2. If you alter this file in minor ways to accomodate your local computing
;;;    environment, you will note that fact and your name in additional header
;;;    comments.
;;; 3. If you alter this file in ways that change its functionality you will note 
;;;    that fact and your name in additional header comments, AND YOU WILL CHANGE
;;;    THE NAME OF THE FILE/ROUTINE.  You are of course welcome to send us 
;;;    changes that you've found helpful and we will consider rolling them 
;;;    into acis_extract itself.

;;; Stage names to search with:
;;;    construct_regions show_regions extract_events check_positions new_catalog extract_spectra arf_correction_filename extract_backgrounds merge_observations fit_spectra cartoon_template                              
                  

; =============================================================================
;;; The run_command tool is our interface for launching other processes (e.g. CIAO commands).
;;; The STATUS output parameter is the exit-status from spawn(); 0 == success.

PRO run_command, DIRECTORY=directory, command, result, STATUS=status, IGNORE_STATUS=ignore_status, $
                 HEASOFT=heasoft, UNIX=unix, SAS=sas, $
                 QUIET=quiet, INTERACTIVE=interactive, OMIT_FAST_START=omit_fast_start, NO_RETRY=no_retry, $
                 PARAM_DIR=param_dir, $
                 CIAO_VERSION=ciao_version_p, MARX_VERSION=marx_version_p

COMMON run_command, spawn_prefix, ciao_env, heasoft_env, sas_env, ciao_version, marx_version

on_error, 2 

status = 0


;; ------------------------------------------------------------------------
;; Each caller maintains its own scratch directory for storing CIAO/HEASOFT parameter files, so there is no parameter contamination among callers.
;; The path to this scratch directory is stored in the variable "scratch_pfiles_directory" in the caller's scope.
;; The caller can pass that path via PARAM_DIR, or explicitly create the variable "scratch_pfiles_directory" before calling run_command.
param_dir_passed = keyword_set(param_dir)
if param_dir_passed then begin
  ; The caller has passed scratch directory paths, via PARAM_DIR.  
  ; Verify they are writable, and store in the variable "scratch_pfiles_directory" in the scope of the caller.
  for ii=0,n_elements(param_dir)-1 do if ~file_test(param_dir[ii], /DIRECTORY, /WRITE) then message, 'Writable parameter directory '+param_dir[ii]+' not found.'

  ; Store PARAM_DIR  the variable "scratch_pfiles_directory" in the scope of the caller.
  (SCOPE_VARFETCH('scratch_pfiles_directory', LEVEL=-1, /ENTER)) = param_dir
  
  if ~keyword_set(quiet) then begin
;    print, F='(%"\nHEASOFT & CIAO will store parameter files in:")'
;    print, param_dir, F='(2x,A)'
;    print
  end  

endif else if (n_elements( SCOPE_VARFETCH('scratch_pfiles_directory', LEVEL=-1, /ENTER)) GT 0) then begin
  ; The caller already has a scratch_pfiles_directory variable.
  ; For speed, assume that the paths it contains are writable.

  param_dir = SCOPE_VARFETCH('scratch_pfiles_directory', LEVEL=-1)

endif else begin
  ; The caller has no scratch directory.   Create one now and store its path in caller's scratch_pfiles_directory variable.
  param_dir = temporary_directory( 'run_command.')

  if ~keyword_set(quiet) then print, param_dir, F='(%"\nrun_command: The caller did not provide a directory for PFILES; creating new directory %s.\n")'

  ; Store PARAM_DIR the variable "scratch_pfiles_directory" in the scope of the caller.
  (SCOPE_VARFETCH('scratch_pfiles_directory', LEVEL=-1, /ENTER)) = param_dir
endelse



;; ------------------------------------------------------------------------
;; Initialize command spawning machinery, once per IDL session.
if ~keyword_set(ciao_env) then begin  

  ;; Verify that csh or tcsh is used in spawns.
  shell_path = getenv('SHELL')

  if ~strmatch(shell_path, "*csh") then begin
    print, shell_path, F='(%"\nERROR: The shell used in your spawned commands is %s; only csh and tcsh are supported.")' 
    retall
  endif

  ;; Set the variable FAST_START within the IDL environment.  
  ;; It will be inherited by all shells spawned; if the user has configured 
  ;; .cshrc correctly, the setup of the HEASOFT & CIAO packages will be skipped,
  ;; saving considerable time.
  ;; Shells spawned by run_command explicitly set up the package required.
  setenv, 'FAST_START=1'


  ; We support two architectures of macOS (previously OS X) machines: 
  ;  "x86_64" (Intel processors)
  ;  "arm"    (Apple M1 processors)
  ; Unfortunately (as of 2021) the arm machines have a bug that leads to random crashes of tcsh (and probably csh)
  ; when those shells are spawned by IDL.  The problem is that (as of 2021) there is no ARM version of IDL.
  ; M1 owners are forced to run IDL binaries compled for Intel processors.
  ; When IDL spawns a shell, the Intel compilation of that shell is run.  For unknown reasons, that Intel version
  ; is not reliable on M1 machines.  Randomly, such shells with produce the following failure:
  ;    tcsh(30796,0x2046f0e00) malloc: can't allocate region
  ;    :*** mach_vm_map(size=1048576, flags: 100) failed (error code=268435465)
  ;    tcsh(30796,0x2046f0e00) malloc: *** set a breakpoint in malloc_error_break to debug
  ;    Out of memory
  ; See https://openradar.appspot.com/FB9666962
  ;
  ; Our work-around for this situation is to make sure that the ARM version of the shell is run.
  spawn_prefix = !NULL
  if (!VERSION.OS EQ 'darwin') then begin
    foreach architecture, ['-arm64','-arm64e'] do begin
  
      spawn_prefix = ['arch', architecture]
      
      spawn, /NOSHELL, [spawn_prefix, 'date'], result, /STDERR, EXIT_STATUS=exit_status
      
      if (exit_status EQ 0) then begin
        ; We're on an M1 machine.
        print, 'Apple architecture is ', architecture
        break
      endif else begin
        spawn_prefix = !NULL
      endelse
    endforeach ; architectures
  endif ; macOS

  spawn_prefix = [spawn_prefix, shell_path,'-c']
;forprint, spawn_prefix
 

  spawn, /NOSHELL, [spawn_prefix, 'hostname'], hostname
  print, hostname, F='(%"hostname is %s\n")'
;  print, getenv('DISPLAY'), F='(%"DISPLAY = %s\n")'

  ;; Check for common environment errors.
  quiet_sysvar = !QUIET  &  !QUIET = 1
  ; Look for AstroLib ...
  catch, error_code
  if (error_code NE 0) then begin
    print, 'ERROR: the IDL Astronomy Users Library is not in your IDL path.'
    retall
  endif else resolve_routine, 'astrolib', /NO_RECOMPILE
  catch, /CANCEL
  astrolib
  ; Make sure forprint calls do not block for user input.
  !TEXTOUT=2

;  ;; On systems where the AstroLib is on an unreliable NFS mount, IDL can fail to find a routine needed for the first time in the middle of a long AE run.  It's better to load all the routines you will need now, and deal with any failures at the outset.
;  skip_list = ['SETLOG','TRNLOG','DELLOG','CATALOG_DS9_INTERFACE','get_astrometry_from_eventlist',  'APPLY_BI_CTI','APPLY_FI_CTI','EB_CUSTOM_PROPERTY1','EB_CUSTOM_PROPERTY2','EB_CUSTOM_PROPERTY3','EB_CUSTOM_PROPERTY4','EB_CUSTOM_PROPERTY5']
;
;  resolve_all, /CONTINUE_ON_ERROR, /QUIET, SKIP=skip_list, UNRESOLVED=unresolved_list
;
;  if (n_elements(unresolved_list) GT n_elements(skip_list)) then begin
;    print, F='(%"\nWARNING: IDL cannot find the following routines/functions.  IF they are needed later IDL will generate an error and halt:")'
;  
;    forprint, unresolved_list[n_elements(skip_list):*]
;    print
;    wait,5
;  endif

  
  ; Look for version of forprint.pro that has SUBSET parameter.
  catch, error_code
  if (error_code NE 0) then begin
    print, !ERROR_STATE.MSG
    print, 'ERROR: perhaps your IDL path contains an obsolete version of forprint.pro (IDL Astronomy Users Library).'
    retall
  endif else begin
    temp_fn = param_dir[0] + '/forprint_test.txt'
    forprint, TEXTOUT=temp_fn, /NoCOM, indgen(10), SUBSET=[0,1]
    if (file_lines(temp_fn) GT 2) then begin
      print, 'ERROR: the forprint.pro routine (IDL Astronomy Users Library) in your IDL path is producing incorrect results.'
      retall
    endif
    file_delete, temp_fn
  endelse
  catch, /CANCEL
  
  
  ; Verify that our FITS header interface can store and retrieve IEEE 754 special values, and BYTE values.
  ; These data values have caused problems in AstroLib FITS routines in the past.
  mkhdr, temp_header, ''
  psb_xaddpar, temp_header,    'NAN',  !VALUES.F_NAN     , 'reserved string'
  psb_xaddpar, temp_header,    'Inf',  !VALUES.F_INFINITY, 'reserved string'
  psb_xaddpar, temp_header, 'NegInf', -!VALUES.F_INFINITY, 'reserved string'
  psb_xaddpar, temp_header,   'Byte',  1B                , 'byte type'

  if ~finite(psb_xpar(temp_header,    'NAN'), /NAN              ) || $
     ~finite(psb_xpar(temp_header,    'Inf'), /INFINITY, SIGN=+1) || $
     ~finite(psb_xpar(temp_header, 'NegInf'), /INFINITY, SIGN=-1) || $
             psb_xpar(temp_header,   'Byte')   NE  1              then begin
    print, 'ERROR: the psb_xaddpar/psb_xpar routines are unable to store IEEE 754 special values in a FITS header.'
    print, 'Contact Patrick Broos, patrick.broos@icloud.com'
    retall
  endif

  ; Verify that our FITS header interface can retrieve IEEE 754 special values written by older versions of AstroLib that used a syntax that was not compliant with the FITS standard.
  temp_header[5] = string(F='(A-80)', 'NAN     =                  NAN /')
  temp_header[6] = string(F='(A-80)', 'INF     =                  Inf /')
  temp_header[7] = string(F='(A-80)', 'NEGINF  =                 -Inf /')
  
  if ~finite(psb_xpar(temp_header,    'NAN'), /NAN              ) || $
     ~finite(psb_xpar(temp_header,    'Inf'), /INFINITY, SIGN=+1) || $
     ~finite(psb_xpar(temp_header, 'NegInf'), /INFINITY, SIGN=-1)  then begin
    print, 'ERROR: the psb_xpar routine is unable to read IEEE 754 special values written by older versions of AstroLib.'
    print, 'Contact Patrick Broos, patrick.broos@icloud.com'
    retall
  endif
  
  

  ; Look for TARA ...
  catch, error_code
  if (error_code NE 0) then begin
    print, 'ERROR: the TARA package is not in your IDL path.'
    retall
  endif else resolve_routine, 'function_1d', /NO_RECOMPILE
  catch, /CANCEL
  !QUIET = quiet_sysvar

  
  library_path = (!VERSION.OS EQ 'darwin') ? 'DYLD_LIBRARY_PATH' : 'LD_LIBRARY_PATH'

  ;; Save the initial values of $PATH, $LD_LIBRARY_PATH, and $DYLD_LIBRARY_PATH
  baseline_path              = getenv('PATH')
  baseline_ld_library_path   = getenv('LD_LIBRARY_PATH')
  baseline_dyld_library_path = getenv('DYLD_LIBRARY_PATH')
  
  ;; ------------------------------------------------------------------------
  ;; Configure CIAO, save environment.
;  print, 'Configuring CIAO ...'
  spawn, /NOSHELL, [spawn_prefix, 'which ciao'], env, COUNT=count, EXIT_STATUS=exit_status
  if (exit_status NE 0) then begin
    print, 'ERROR: alias "ciao" is not defined.'
    retall
  endif
  ; Remove $LD_LIBRARY_PATH, and $DYLD_LIBRARY_PATH so spawned shell will start fresh.
  setenv, 'LD_LIBRARY_PATH='  
  setenv, 'DYLD_LIBRARY_PATH='
  ; Work-around a mysterious problem with the "module" alias on Linux systems (see email conversation with Satej Khedekar in April 2013). 
  setenv, 'module='

  ; We unsetenv DISPLAY to speed up 'ciao' and to prevent failure when $DISPLAY is
  ; not valid (e.g. as happens when 'screen' is used to manage processes).
  spawn, /NOSHELL, [spawn_prefix, 'unsetenv DISPLAY; ciao; printenv'], env
  env_ind = where(strmatch(env, '*=*') AND ~strmatch(env, '*BASH_FUNC*'), env_count, COMPLEMENT=print_ind, NCOMPLEMENT=print_count)
  if (env_count EQ 0) then begin
    print, 'ERROR: shell command "printenv" seems to have failed:'
    forprint, env
    retall
  endif
  for ii=0, print_count-1 do begin
    line = env[print_ind[ii]]
    if strmatch(line, 'Warning: X DISPLAY variable is not set.*') then continue
    if strmatch(line, '*GUI applications*') then continue
    if strmatch(line, 'CIAO configuration is complete*') then continue
    print, line
  endfor ;ii
  
  ; Rather than hard-coding the environment variables that CIAO and HEASOFT use, we simply 
  ; import the entire environment into the IDL process.
  ; Long strings in the environment are truncated, since some users have .cshrc files that
  ; cause some variables to grow without bound.
  for ii=0, env_count-1 do begin
    line = env[env_ind[ii]]
    if (strlen(line) GT 1000) then begin
      line = strmid(line, 0, 1000)
      print, 'WARNING: shell environment variable truncated to 1000 characters: '+line
    endif
    setenv, line
  endfor
  ciao_env = string("PATH", getenv("PATH"), library_path, getenv(library_path), F="(%'setenv %s \'%s\' ')")
                                 

  ; Restore baseline values of $PATH, $LD_LIBRARY_PATH, and $DYLD_LIBRARY_PATH.
  setenv, 'PATH='             +baseline_path
  setenv, 'LD_LIBRARY_PATH='  +baseline_ld_library_path
  setenv, 'DYLD_LIBRARY_PATH='+baseline_dyld_library_path
  
  ; Verify the CIAO version is adequate.
  cmd = 'cat '+getenv('ASCDS_INSTALL')+'/VERSION'
  spawn, /NOSHELL, [spawn_prefix, cmd], result
  ciao_version = (stregex(result[0],'CIAO ([0-9\.]+)',/SUB,/EXT))[1]
  if version_string_compare(ciao_version, '4.1', /LESSTHAN) then begin
    print
    print, 'ERROR: AE requires CIAO version 4.1 or later.'
    retall
  endif

  ; Verify the MARX version is adequate.  AE is NOT compatible with version 5.
  cmd = 'marx --version'
  spawn, /NOSHELL, [spawn_prefix, cmd], result, /STDERR
  marx_version = (stregex(result[0],'MARX version ([0-9\.]+)',/SUB,/EXT))[1]
  if version_string_compare(marx_version, '5.3'  , /LESSTHAN) || $
     version_string_compare(marx_version, '5.5.1', /GREATERTHAN) then begin
    print
    print, 'ERROR: AE is compatible with MARX versions 5.3 through 5.5.1.'
    ; See email titled "marx-users: MARX 5.5 released".
    retall
  endif
  
  
  ;; ------------------------------------------------------------------------
  ;;Configure HEASOFT, save environment.
;  print, 'Configuring HEASOFT ...'
  spawn, /NOSHELL, [spawn_prefix, 'which heasoft'], env, COUNT=count, EXIT_STATUS=exit_status
  if (exit_status NE 0) then begin
    print, 'ERROR: alias "heasoft" is not defined.'
    retall
  endif
  ; Remove $LD_LIBRARY_PATH, and $DYLD_LIBRARY_PATH so spawned shell will start fresh.
  setenv, 'LD_LIBRARY_PATH='  
  setenv, 'DYLD_LIBRARY_PATH='
  ; Work-around a mysterious problem with the "module" alias on Linux systems (see email conversation with Satej Khedekar in April 2013). 
  setenv, 'module='

  spawn, /NOSHELL, [spawn_prefix, 'unsetenv DISPLAY; heasoft; printenv'], env
  env_ind = where(strmatch(env, '*=*') AND ~strmatch(env, '*BASH_FUNC*'), env_count, COMPLEMENT=print_ind, NCOMPLEMENT=print_count)
  if (env_count EQ 0) then begin
    print, 'ERROR: shell command "printenv" seems to have failed:'
    forprint, env
    retall
  endif
  if (print_count GT 0) then forprint, env, SUBSET=print_ind

  ; Rather than hard-coding the environment variables that CIAO and HEASOFT use, we simply 
  ; import the entire environment into the IDL process.
  ; Long values in the environment are skipped, since some users have .cshrc files that
  ; cause some variables to grow without bound.
  for ii=0, env_count-1 do begin
    line = env[env_ind[ii]]
    if (strlen(line) LT 1000) then setenv, line
  endfor
  heasoft_env = string("PATH", getenv("PATH"), library_path, getenv(library_path), F="(%'setenv %s \'%s\' ')")

  ;; Restore baseline values of $PATH, $LD_LIBRARY_PATH, and $DYLD_LIBRARY_PATH.
  setenv, 'PATH='             +baseline_path
  setenv, 'LD_LIBRARY_PATH='  +baseline_ld_library_path
  setenv, 'DYLD_LIBRARY_PATH='+baseline_dyld_library_path


  ;; ------------------------------------------------------------------------
  if keyword_set(sas) then begin  
    ;;Configure SAS, save environment.
  ;  print, 'Configuring SAS ...'
    spawn, /NOSHELL, [spawn_prefix, 'which loadsas'], env, COUNT=count, EXIT_STATUS=exit_status
    if (exit_status NE 0) then begin
      print, 'ERROR: alias "loadsas" is not defined.'
      retall
    endif
    ; Remove $LD_LIBRARY_PATH, and $DYLD_LIBRARY_PATH so spawned shell will start fresh.
    setenv, 'LD_LIBRARY_PATH='  
    setenv, 'DYLD_LIBRARY_PATH='
    ; Work-around a mysterious problem with the "module" alias on Linux systems (see email conversation with Satej Khedekar in April 2013). 
    setenv, 'module='

  spawn, /NOSHELL, [spawn_prefix, 'unsetenv DISPLAY; loadsas; printenv'], env
  env_ind = where(strmatch(env, '*=*') AND ~strmatch(env, '*BASH_FUNC*'), env_count, COMPLEMENT=print_ind, NCOMPLEMENT=print_count)
    if (env_count EQ 0) then begin
      print, 'ERROR: shell command "printenv" seems to have failed:'
      forprint, env
      retall
    endif
    if (print_count GT 0) then forprint, env, SUBSET=print_ind
  
    ; Rather than hard-coding the environment variables that CIAO and HEASOFT use, we simply 
    ; import the entire environment into the IDL process.
    ; Long values in the environment are skipped, since some users have .cshrc files that
    ; cause some variables to grow without bound.
    for ii=0, env_count-1 do begin
      line = env[env_ind[ii]]
      if (strlen(line) LT 1000) then setenv, line
    endfor
    sas_env = string("PATH", getenv("PATH"), library_path, getenv(library_path), F="(%'setenv %s \'%s\' ')")
  
    ;; Restore baseline values of $PATH, $LD_LIBRARY_PATH, and $DYLD_LIBRARY_PATH.
    setenv, 'PATH='             +baseline_path
    setenv, 'LD_LIBRARY_PATH='  +baseline_ld_library_path
    setenv, 'DYLD_LIBRARY_PATH='+baseline_dyld_library_path
  endif

  

  ; Take steps to prevent $SHLVL from growing without bound.
  setenv, 'SHLVL=1'
  
  ; Measure the spawning overhead by calling run_comand recursively.
  ; We store the scratch directory path in scratch_pfiles_directory, so that the recursive call will not create a new scratch directory.
  scratch_pfiles_directory = param_dir

  t0=systime(1) & run_command, 'exit', /QUIET 
  run_time = systime(1)-t0
  print, run_time, F='(%"Spawning a shell is taking %0.2f seconds.")'
  if (run_time GT 0.2) then print, run_time, F='(%"\n\n++++++++++++++++++++++++++++++\nWARNING! Spawning a shell is taking %0.1f seconds.  \nUse the FAST_START environment variable to skip sections of .cshrc that are slow to execute.\n++++++++++++++++++++++++++++++\n\n")'

  
  ; Record the CALDB version.
  run_command, 'check_ciao_caldb', result, /QUIET, /IGNORE_STATUS
  forprint, result[0:-2]
  
  print, 'MARX version '+marx_version
endif ; Initialize command spawning machinery, once per IDL session.


if arg_present(ciao_version_p) then ciao_version_p=ciao_version
if arg_present(marx_version_p) then marx_version_p=marx_version

if ~keyword_set(command) then begin
  ; Warn user if call neither declared a PARAM_DIR or passed a command to run.
  ;if ~param_dir_passed then print, 'run_command: No command was passed.'
  return
endif

;; ------------------------------------------------------------------------
;; Spawn the specified commands.

; Retrieve the appropriate shell commands that will set up the correct environment to run commands from the requested package.
                             cmd_prefix = ciao_env
if keyword_set(heasoft) then cmd_prefix = heasoft_env
if keyword_set(sas)     then cmd_prefix = sas_env
if keyword_set(unix)    then cmd_prefix = ''

; Configure the spawned shell (via $PFILES) to use the desired parameter file scratch directory, and the appropriate parameter files directories for the requested package.
pfiles_env_var = string(strjoin(param_dir,':'), getenv("ASCDS_INSTALL"), getenv("ASCDS_INSTALL"), getenv("LHEASOFT"), F='(%"%s;%s/contrib/param:%s/param:%s/syspfiles")')

cmd_prefix = [ cmd_prefix, string(pfiles_env_var, F="(%'setenv PFILES \'%s\' ')") ]

; We also set PFILES in the IDL process to the same string, in case some tool explicitly spawns a CIAO/HEASOFT command, without using run_command.
setenv, 'PFILES='+pfiles_env_var  


if ~keyword_set(quiet) then begin
  if keyword_set(directory) then print, directory, F='(%"\nIn %s ...")' $
                            else print ;,            F='(%"\nSpawning:")'
  forprint, command, F="(%'  %s')"
  print
endif

cmd = strjoin(command,"; ")
result_count = 0

if keyword_set(directory)       then pushd, directory
if keyword_set(omit_fast_start) then setenv, 'FAST_START=0'
                                                       
; When the caller wants the user to interact with the spawned process, then we can NOT ask spawn to trap stdout/stderr.
cmd = strjoin([cmd_prefix,''], ';') + cmd

if keyword_set(interactive) then $
  spawn, /NOSHELL, [spawn_prefix, cmd], EXIT_STATUS=exit_status $
else begin
  spawn, /NOSHELL, [spawn_prefix, cmd], EXIT_STATUS=exit_status, result, COUNT=result_count, /STDERR
  
  ; Retry failed command if failure was not anticipated by caller.
  ; This allows us to automatically recover from various unreproducible failures we commonly see.
  if (exit_status NE 0) && ~keyword_set(ignore_status) && ~keyword_set(no_retry) then begin
    if (result_count GT 0) && strmatch(result[0],'*memory*') then begin
      ; Show "Out of memory" message.
      forprint, result, F="(%'  > %s')"
      print
    endif
    print, cmd, exit_status, F='(%"Retrying failed command %s\nEXIT_STATUS was %d...")'
    wait, 0.5
    spawn, /NOSHELL, [spawn_prefix, cmd], EXIT_STATUS=exit_status, result, COUNT=result_count, /STDERR
  endif
endelse

; Restore current directory and FAST_START environment variable.
if keyword_set(directory)       then popd
if keyword_set(omit_fast_start) then setenv, 'FAST_START=1'

if (exit_status EQ 0) || keyword_set(ignore_status) then begin
  ; Nominal spawn
  status = exit_status
  if (result_count GT 0) && ~keyword_set(quiet) then begin
    ; Show output text.
    forprint, result, F="(%'  > %s')"
    print
  endif

endif else begin
  ; Failed spawn. 
  status = exit_status
  
  ; If we didn't print the commands before, then do it now.
  if keyword_set(quiet) then begin
    if keyword_set(directory) then print, directory, F='(%"\nIn %s ...")' $
                              else print ;,            F='(%"\nSpawning:")'
    forprint, command, F="(%'  %s')"
    print
  endif
  
  ; Show output text.
  if (result_count GT 0) then forprint, result, F="(%'  > %s')"

  print, now(), exit_status, F='(%"\nERROR: process spawned above failed at %s with exit status %d.\n")'
  
;  if keyword_set(cmd_prefix) then begin
;    print, F="(%'Environment used for spawn was: ')"
;    for ii=0,n_elements(cmd_prefix)-1 do print, cmd_prefix[ii], F="(%':  %s\n')"
;  endif

  if arg_present(status) then return
  
  message, /NONAME, 'Investigate the failure, e.g. by trying the failing command manually in a shell.  If you can fix the problem and want AE to re-try the command then type ".continue"; if you have run the command manually and want AE to skip over it then type .skip and then .continue; if you want to abort the AE run type "retall".'
endelse

return
end  ; run_command



; =============================================================================
;; Extract the specified keywords from every HDU in a FITS file.

FUNCTION get_keywords_from_hdu_headers, filename, template, ERRMSG=errmsg, SILENT=silent

template_row = null_structure(template[0])

keywords     = tag_names(template_row)

; Open the file at the first HDU.
unit = fxposit(filename, /READONLY, 0, ERRMSG=errmsg, SILENT=silent)

if keyword_set(errmsg) then begin
  if ARG_PRESENT(errmsg) then return, template_row $
                         else message, string(filename, F='(%"ERROR: error opening %s in get_from_hdu_headers()")')
endif

table = []
while ~eof(unit) do begin
  MRD_HREAD, unit, header, status, SILENT=silent
  
  if (status NE 0) then message, string(filename, F='(%"ERROR: error reading header from %s in get_from_hdu_headers()")')
  
  this_row = template_row
  for ii=0,n_elements(keywords)-1 do begin
    temp = psb_xpar( header, keywords[ii], COUNT=count)
    
    ; FITS string keywords often have trailing blank characters.
    if size(temp, /TNAME) EQ 'STRING' then temp = strtrim(temp,2)
    
    if (count GT 0) then this_row.(ii) = temp[0]
  endfor ;ii
  
  table = [table, this_row]
endwhile
free_lun, unit

return, table
end



; =============================================================================
;; Convert a region file from ds9 to CIAO format, and reduce size of polygon
;; entries as required to get file size under the CIAO 32000 character limit.
;;
;; The coordinates of the first polygon are returned in POLYGON_X, POLYGON_Y.
;; Return region_edited=1 if the region file is write-protected, suggesting that it was edited by the observer.

;; The standard AE extraction region file contains both a polygon for the source aperture
;;  and a circle for the mask region, plus header lines related to ds9.


;; We want only the polygon line, and we must strip out any leading phrase
;; "physical;" and any trailing phrase "#..." since they seem to sometimes
;; (but not always) confuse CIAO.

PRO ae_ds9_to_ciao_regionfile, ds9_filename, ciao_filename, APPEND=append, $
        IGNORE_BACKGROUND_TAG=ignore_background_tag, MAX_VERTEXES=max_vertexes, PRECISION=precision, $
        POLYGON_X=first_polygon_x, POLYGON_Y=first_polygon_y, REGION_EDITED=region_edited, $
        FIELD_SYNTAX_FOUND=field_syntax_found

max_file_length = 32000
num_lines = 0
lines     = strarr(100)
first_polygon_x = 0
first_polygon_y = 0
field_syntax_found = 0

if ~keyword_set(max_vertexes) then max_vertexes = 100
if ~keyword_set(precision   ) then precision    = 2

;; We interpret a write-protected aperture region file to be "edited by the observer".
temp = file_info(ds9_filename)
region_edited = temp.EXISTS && ~temp.WRITE

;; Parse ds9 regionfile.
openr, region_unit, ds9_filename, /GET_LUN

    line = ''
    while not eof(region_unit) do begin
      readf, region_unit, line
      
      ;; Unless /APPEND specified, 
      ;; we MUST keep the FIRST line of the file since it specifies the format via the ds9 version number.
      ;; CIAO filtering seems to use the ds9 version number when interpreting the file.
      ;; Let's just keep all the comment lines since they don't seem to bother CIAO.
      if keyword_set(append) AND (strmid(line,0,1) EQ '#') then continue
      
      
      ;; Try to determine if ds9 wrote the file.
      ;; Skip globals, and entries with "background" tag if desired.
      if                                        (strpos(line,'global')     NE -1) then continue
      
      if keyword_set(ignore_background_tag) AND (strpos(line,'background') NE -1) then continue
      
      if                                        (strpos(line,'field')      NE -1) then field_syntax_found = 1
      
      ;; Strip off trailing comments
      comment_start = strpos(line,'#')
      if (comment_start GT 0) then line = strmid(line,0,comment_start)
      
      ;; Strip off leading "physical;" tags present in ds9 version 3.
      if (strmid(line,0,9) EQ 'physical;') then line = strmid(line,9)
      
      ;; Skip lines that are in celestial coordinates (e.g. "panda" regions written by AE's PSF hook screening).
      if (strmid(line,0,5) EQ 'J2000') then continue
      
      ;; Save line.
      lines[num_lines] = line
      num_lines = num_lines + 1
    endwhile
free_lun, region_unit

if (num_lines EQ 0) then begin
  print, 'ERROR: '+ds9_filename+' contains no acceptable regions!'
  run_command, 'cat '+ds9_filename, /UNIX
  retall
endif

;; Write CIAO regionfile.
resample_all_polygons = 0
for pass=1,10 do begin
  openw, region_unit, ciao_filename, /GET_LUN, APPEND=keyword_set(append)


  for ii=0,num_lines-1 do begin
    line = lines[ii]
  
    first = strpos(line, 'polygon(')
    if (first EQ -1) then begin
      ; Non-polygons are written as-is.
      printf, region_unit, line
      
    endif else begin
      ; Parse the polygon.
      ; Check for leading '-' or '+'.
      sign_str = strmid(line, 0, first)
      case 1 of
        (strpos(sign_str,'+') NE -1): sign='+'
        (strpos(sign_str,'-') NE -1): sign='-'
        else:                         sign=''
      endcase

      first   = first + 8
      last    = strpos(line, ')', first)
      coords  = strmid(line, first, (last-first))
      numbers = float(strsplit(coords, ',', /EXTRACT))
      ind       = 2*indgen(n_elements(numbers)/2)
      polygon_x = numbers[ind]
      polygon_y = numbers[ind+1]
      
      ; Save the first polygon's coordinates for the caller.
      if NOT keyword_set(first_polygon_x) then begin
        first_polygon_x = polygon_x
        first_polygon_y = polygon_y
      endif

      ; If necessary reduce the number of vertices to reduce file size.
      num_current_points    = n_elements(polygon_x)
      resample_this_polygon = resample_all_polygons
      
      while (resample_this_polygon || (num_current_points GE 2*max_vertexes)) && (num_current_points GE 8) do begin
        if (num_current_points GE 2*max_vertexes) then $
          print, num_current_points, F='(%"Resampling polygon with too many vertices (%d) ...")'
        
        num_new_points     = ceil(num_current_points/2.0)
        
        first_pt  = 2*indgen(num_new_points)
        second_pt = (first_pt+1) < (num_current_points-1)
        
        polygon_x = (polygon_x[first_pt] + polygon_x[second_pt])/2.0
        polygon_y = (polygon_y[first_pt] + polygon_y[second_pt])/2.0
      
        num_current_points = n_elements(polygon_x)
        resample_this_polygon = 0
      endwhile
      
      ; Polygons are printed with restricted precision to limit file size.
      polygon = fltarr(2,n_elements(polygon_x))
      polygon[0,*] = polygon_x
      polygon[1,*] = polygon_y

      format = string(precision, F='(%"%%0.%df")')
      src_region = sign + 'polygon(' + strcompress(strjoin(string(polygon,F='(%"'+format+'")'),","), /REMOVE) + ')'
      printf, region_unit, src_region
      lines[ii] = src_region
    endelse ; is a polygon
  endfor ;ii
  
  filesize = (fstat(region_unit)).cur_ptr
  free_lun, region_unit
  
  if keyword_set(append) OR (filesize LT max_file_length) then break
  resample_all_polygons = 1  
  print, 'Output file is too long; resampling polygons'
endfor ;pass

if (NOT keyword_set(append)) AND (filesize GT max_file_length) then begin
  print
  print, 'ERROR: could not reduce size of region file sufficiently'
  forprint, lines[0:num_lines-1]
  stop
endif 

return
end


; =============================================================================
;; Display a tiled set of data files in ds9, with optional region files for each.

;; First call creates ds9 session using string NAME to name the session, e.g.
;;   ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='acis_extract_'+session_name
;; If the string OPTION_STRING is supplied, it is passed to ds9, e.g.
;;   OPTION_STRING='-log -bin factor 8'

;; Subsequent calls display one or more data files and optional region files.

;; 
;; If a 2-vector PAN_TO_COORDS is supplied, ds9 will pan to the specified celestial coordinates.

PRO ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, $
      NAME=name, OPTION_STRING=option_string, $
      data_fn, region_fn, DESIRED_ZOOM=desired_zoom, MATCH_PHYSICAL=match_physical, PAN_TO_COORDS=pan_to_coords

if ~keyword_set(option_string) then option_string = '-log '

if keyword_set(tempdir) then run_command, PARAM_DIR=tempdir, /QUIET

if keyword_set(name) then begin
  ; The xpaaccess tool cannot find the ds9 session if its name has any space characters!
  name = repchr(name,' ','_')
  run_command, string(name, option_string, F='(%"ds9 -xpa local -tile -title ''%s'' -lock bin yes -lock frame wcs %s >& /dev/null &")'), /QUIET
  my_ds9 = "DS9:"+name
  
  ; Wait for ds9 to register with XPA.
  ; Starting in CIAO 4.0, xpaaccess uses the exit code to return its result.
  ; Thus we can no longer allow run_command to interpret a non-zero exit code as failure.
  repeat begin
    run_command, string(my_ds9, F='(%"xpaaccess ''%s''")'), result, /IGNORE_STATUS, /QUIET
    if (result[0] EQ 'yes') then break
    print, 'waiting for ds9 to come up...'
    wait,1
  endrep until (0)
  ; I have found that XPA and ds9 sometimes fail to communicate properly if the first XPA command (such as loading an image) is sent immediately after ds9 "comes up" above.  In the case I saw, ds9 fails to execute the XPA command, and displays a dialog box saying:
  ;    An internal error has been detected
  ;    invalid command name ""
  ; My workaround is to add a small pause after xpaaccess says that ds9 is ready.
  wait, 2.0
  return
endif


if keyword_set(pan_to_coords) then begin
  cmd    = strarr(2)
  cmd[0] = string(my_ds9, pan_to_coords, F='(%"xpaset -p ''%s'' pan to %10.6f %10.6f wcs fk5 degrees")')
  cmd[1] = string(my_ds9,                F='(%"xpaset -p ''%s'' match frame wcs")')

  run_command, cmd, /QUIET
  return
endif


; Delete existing frames.
num_frames  = n_elements(data_fn)
cmd    = strarr(2+3*num_frames+15)
ii=0
cmd[ii++] = string(my_ds9,                              F='(%"xpaset -p -n ''%s'' frame delete all")')

if (num_frames GT 0) then begin
  ; We create all the frames first in case adding a frame causes lots of computations on
  ; the existing frames.
  for jj = 0, num_frames-1 do $
    cmd[ii++] = string(my_ds9, 1+jj,                 F='(%"xpaset -p -n ''%s'' frame %s")')
endif
          
; Load data into each frame.
for jj = num_frames-1, 0, -1 do begin
  cmd[ii++] = string(my_ds9, 1+jj,         F='(%"xpaset -p -n ''%s'' frame %s")')
  cmd[ii++] = string(my_ds9, data_fn[jj],  F='(%"xpaset -p -n ''%s'' fits ''%s'' ")')
endfor
    
; Center the LAST frame, and command the supplied zoom factor.
cmd[ii++] = string(my_ds9, num_frames,        F='(%"xpaset -p -n ''%s'' frame %s")')
cmd[ii++] = string(my_ds9,                    F='(%"xpaset -p -n ''%s'' frame center")')
if keyword_set(desired_zoom) then begin
  cmd[ii++]   = string(my_ds9, desired_zoom,  F='(%"xpaset -p -n ''%s'' zoom to %f")')
endif else begin
  cmd[ii++] = string(my_ds9,                  F='(%"xpaset -p -n ''%s'' zoom to fit")')
endelse

; Align frames if there's more than one.
if (num_frames GT 1) then begin
  coordsys = keyword_set(match_physical) ? 'physical' : 'wcs'
  cmd[ii++] = string(my_ds9, coordsys,        F='(%"xpaset -p -n ''%s'' match frame %s")')
endif

run_command, cmd, /QUIET

;; While the user starts looking at images, overlay regions.
num_regionfiles = n_elements(region_fn) < num_frames
if (num_regionfiles GT 0) then begin
  cmd   = strarr(2*num_regionfiles + 1)
  ii=0
  for jj = num_regionfiles-1, 0, -1 do begin
    if (region_fn[jj] NE '') then begin
      cmd[ii++] = string(my_ds9, 1+jj,             F='(%"xpaset -p -n ''%s'' frame %d")')
      
      ; Allow the user to pass either "command ...." or a region filename.
      cmd[ii++] = string(my_ds9, strmatch(region_fn[jj], 'command *') ? region_fn[jj] : 'load '+region_fn[jj], $
                                                   F='(%"xpaset -p -n ''%s'' regions %s")')
    endif
  endfor

  run_command, cmd, /QUIET
endif 

;; It is useful to leave ds9 with the last frame selected, so that an observer reviewing reconstructions can stay in single-frame mode and see the recon immediately after moving to a new source.
run_command, /QUIET, string(my_ds9, num_frames, F='(%"xpaset -p ''%s'' frame %d")')

return
end

                                                    
; =============================================================================
; Read a source's ARF and RMF files.
; Return the energies of the PI channel boundaries and mid-points.
; Resample several column in the ARF table at the mid-points of the channels:
;   channel_base     = CIAO_ARF     (channel_midenergy)
;   channel_psf_frac = PSF_fraction (channel_midenergy)
;   channel_specresp = effective_ARF(channel_midenergy)

PRO ae_channel_energy_and_arf, rmf_fn, arf_fn, $
        channel_number, channel_lowenergy, channel_highenergy, channel_midenergy, channel_specresp, channel_psf_frac, channel_base

  ; Read the EBOUNDS table in the RMF which defines energy range (keV) for each channel.
  ; Channel numbers start at 1, not zero!
  fits_open, rmf_fn, fcb, /NO_ABORT, MESSAGE=error
  if keyword_set(error) then message, 'ERROR reading ' + rmf_fn
  
  fits_read,  fcb, table, header, /NO_PDU, EXTNAME='EBOUNDS', /NO_ABORT, MESSAGE=error
  if keyword_set(error) then message, 'ERROR: cannot find table EBOUNDS in ' + rmf_fn
  fits_close, fcb
  
  tbinfo, header, tb_str
  channel_number    = tbget(tb_str, table, 'CHANNEL')
  channel_lowenergy = tbget(tb_str, table, 'E_MIN')
  channel_highenergy= tbget(tb_str, table, 'E_MAX')
  channel_midenergy = 0.5 * (channel_lowenergy + channel_highenergy)
  
  ; Free the pointers allocated by tbinfo.
  ptr_free, tb_str.TSCAL, tb_str.TZERO
    
  if keyword_set(arf_fn) then begin
    ; Read the SPECRESP table in the ARF and look up an ARF value corresponding to each channel.
    ; We use linterp to truncate to the end points rather than extrapolating; the range of
    ; the RMF's EBOUNDS table may exceed that of the ARF..
    arf_table = mrdfits(arf_fn, 1, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + arf_fn

    linterp, 0.5*(arf_table.ENERG_LO + arf_table.ENERG_HI), arf_table.SPECRESP, channel_midenergy, channel_specresp
    
    if (total(strmatch(tag_names(arf_table), 'PSF_FRAC')) GT 0) then $
    linterp, 0.5*(arf_table.ENERG_LO + arf_table.ENERG_HI), arf_table.PSF_FRAC, channel_midenergy, channel_psf_frac
    
    if (total(strmatch(tag_names(arf_table), 'BASE')) GT 0) then $
    linterp, 0.5*(arf_table.ENERG_LO + arf_table.ENERG_HI), arf_table.BASE, channel_midenergy, channel_base
  endif else begin
    channel_specresp=0
    channel_psf_frac=0
    channel_base    =0
  endelse
return
end


; =============================================================================
; Radial profile analysis tool.

; OBSOLETE HEADER COMMENTS:
    ;; ------------------------------------------------------------------------
    ;; Evaluate the consistency between the data and the PSF.
    ;; The composite PSF we have on hand is the full PSF rather than one truncated
    ;; by all the extraction regions.  (We made the decision not to burn up the
    ;; disk space required to save truncated PSFs back in the EXTRACT stage.)
    ;; This full PSF cannot be compared to the composite EXTRACTED event data 
    ;; (which is computed below).  Instead we must compare to the un-truncated 
    ;; composite event data (computed above).
    ;;
    ;; However, we can't analyze data too far in the wings of the PSF because we
    ;; may run into a neighboring source.
    ;; Thus, we choose to work in a simple circular aperture that approximates the
    ;; smallest extraction region used in all the observations.
 
; Nov 2007  Commented out the call below in AE, due to worries about the CPU resources required.    
;    ae_radial_profile, ASPECT_FN='none', ra, dec, merged_env_events_fn, src_radius, energy_range, composite_psf_fn, tempdir, $
;                       PLOT=(verbose GT 1), WIDGET_IDS=ae_radial_profile_ids, PSF_NAME='PSF', $
;                       ks_psf, R_MEDIAN, EE_AT_RM
;    



; 
;;; Other parameters have defaults that are set up to run this tool in the AE extraction dir for a specific obsid.
; =============================================================================
PRO ae_radial_profile, report,  $
;;; PLUS 
;;; Style #1:
  ;   Required parameters
                       SRCLIST_FILENAME=srclist_fn, $
;;; Style #2:                      
  ;   Required parameters
                       SOURCENAME=sourcename, $
                       RA=ra_src, DEC=dec_src,$
                       SRC_RADIUS=src_radius, $  ; skypix
                         
;;; Optional parameters
  ;   both styles
                      MERGE_NAME=merge_name    , $ 
                      PSF_NAME=psf_name        , $ 
                      ENERGY_RANGE=energy_range, $ ; [min,max] in keV
                      PLOT=make_plot           , $
                      DATASET_PREFIX=dataset_prefix, $
                      REGION_FILENAME=region_file  , $  ; e.g. a source catalog.  This is NOT the analysis aperture!
                      VERBOSE=verbose          , $
                      TEMPDIR=tempdir          , $
                      BLOCK=block              , $
  ;   Style #1
                     ;SRC_RADIUS=src_radius: radius of analysis region; defaults to SRC_RAD value (skypix) in source.stats header.
  ;   Style #2
                      LABEL=label              , $
                      EVENTLIST_FN=merged_env_events_fn, $
                      PSF_FN=composite_psf_fn             

COMMON ae_radial_profile, session_name, my_ds9, id0, id1, id2, id3, tm_id, button_id, quit_id

creator_string = "ae_radial_profile, version " +strmid("$Rev:: 5658  $",7,5) +strmid("$Date: 2022-01-25 07:06:00 -0700 (Tue, 25 Jan 2022) $", 6, 11)

exit_code = 0

;; If not supplied by the caller, construct a local scratch directory.
tempdir_supplied_by_caller = keyword_set(tempdir)
if ~tempdir_supplied_by_caller then begin
  temproot = temporary_directory( 'ae_radial_profile.', VERBOSE=1, SESSION_NAME=session_name)
  tempdir  = temproot
endif else session_name = string(random()*1E7, F='(I7.7)')
run_command, PARAM_DIR=tempdir

temp_events_fn   = tempdir + 'temp.evt'
temp_image_fn    = tempdir + 'temp.img'
inband_events_fn = tempdir + 'temp.inband.evt'
temp_text_fn     = tempdir + 'temp.txt'
temp_region_fn   = tempdir + 'temp.reg'


src_stats_basename       = 'source.stats'
src_region_basename      = 'extract.reg'
env_events_basename      = 'neighborhood.evt'
env_image_basename       = 'neighborhood.img'
psf_basename             = 'source.psf'

if keyword_set(merge_name)     then merge_subdir = merge_name + '/' $
                               else merge_subdir = ''
                             
if ~keyword_set(energy_range)  then energy_range  = [0.5,8]
energy_range_label = string(energy_range, F='(%"[%0.2f:%0.2f] keV")')

if ~keyword_set(psf_name)       then psf_name      = ' PSF'
if ~keyword_set(label)          then label         = ''
if ~keyword_set(verbose)        then verbose       = 0
if ~keyword_set(dataset_prefix) then dataset_prefix = ''
                              
; =============================================================================
;; When a source list filename is supplied, we loop over those sources and call this tool recursively.
if keyword_set(srclist_fn) then begin
  print, creator_string, F='(%"\n\n%s")'
  
  ; Read a list of sourcenames from the file specified.
  readcol, srclist_fn, sourcename, FORMAT='A', COMMENT=';'
  
  ; Trim whitespace and remove blank lines.
  sourcename = strtrim(sourcename,2)
  ind = where(sourcename NE '', num_sources)
  
  if (num_sources EQ 0) then begin
    print, 'ERROR: no entries read from source list ', catalog_or_srclist
    retall
  endif
  
  sourcename = sourcename[ind]
  print, num_sources, F='(%"\n%d sources found in catalog.\n")'
  
  if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)

  if keyword_set(make_plot) then $
    TimedMessage, tm_id, '' , TITLE='ae_radial_profile', QUIT_LABEL='quit', BUTTON_LABEL='next source', PRESSED_ID=trash

  for ii = 0L, num_sources-1 do begin
    basedir   = sourcename[ii] + '/' 

    sourcedir = basedir + merge_subdir[ii]
    stats_fn              = sourcedir + src_stats_basename
    merged_env_events_fn  = sourcedir + env_events_basename
    composite_psf_fn      = sourcedir + psf_basename
    
    src_stats = headfits(stats_fn, ERRMSG=error)
    
    if keyword_set(error) then begin
      print, error
      print, 'ERROR: count not read ', stats_fn
      continue
    endif
    
    label        = strtrim(psb_xpar( src_stats, 'LABEL'),2)
     ra_src      =         psb_xpar( src_stats, 'RA')
    dec_src      =         psb_xpar( src_stats, 'DEC')
    src_radius   = keyword_set(src_radius) ? src_radius : psb_xpar( src_stats, 'SRC_RAD') 
  
    if ~keyword_set(ra_src) || ~keyword_set(dec_src) || ~keyword_set(src_radius) then begin
        print, 'ERROR:  RA, DEC, or SRC_RAD could not be found in '+stats_fn
        retall
    endif
    
    ; Recursive call ...
    ae_radial_profile, this_report, PLOT=make_plot, $
                       ENERGY_RANGE=energy_range, DATASET_PREFIX=dataset_prefix, REGION_FILENAME=region_file,$
                       TEMPDIR=tempdir, BLOCK=(num_sources GT 1),$
                       MERGE_NAME=merge_name, $
                       SOURCENAME=sourcename[ii], RA=ra_src, DEC=dec_src, SRC_RADIUS=src_radius, $ 
                       EVENTLIST_FN=merged_env_events_fn, PSF_FN=composite_psf_fn, LABEL=label

    if (ii EQ 0) then report = replicate(null_structure(this_report), num_sources)
    
    this_report.RATE_3X3 = psb_xpar( src_stats, 'RATE_3X3')
    this_report.THETA    = psb_xpar( src_stats, 'THETA'   )
    
    report[ii] = this_report
    
    if keyword_set(make_plot) then begin
      ; Allow the user to interact with the plots; move to next source when they press the button.
      msg = string( this_report.sourcename, this_report.label, this_report.src_cnts, this_report.theta, this_report.prob_ks_r, this_report.prob_ks_dx, this_report.prob_ks_dy, F='(%"%s (%s): %d cts; theta=%0.1f''; Pks (r,dx,dy) = %0.4f, %0.4f, %0.4f")')
      TimedMessage, tm_id, msg, PRESSED_ID=trash 
      repeat begin
        event = widget_event([id0,id1,id2,id3,quit_id,button_id], NOWAIT=0)
        
        abort = (event.ID EQ quit_id)
      endrep until abort || (event.ID EQ button_id)
      
      if abort then break
    endif
  endfor
  widget_control, tm_id, /DESTROY
  return
endif



; =============================================================================
;;; A single sourcename was passed.
if ~keyword_set(sourcename) || ~keyword_set(ra_src) || ~keyword_set(dec_src) || ~keyword_set(src_radius) then begin
    print, 'ERROR: in calling style #2 you must supply SOURCENAME, RA, DEC, and SRC_RADIUS'
    retall
endif

print, sourcename, F='(%"\n------------------------------\nRadial Profile for %s\n")'

basedir   = sourcename + '/' 

sourcedir = basedir + merge_subdir


if ~keyword_set(merged_env_events_fn) then merged_env_events_fn  = sourcedir + env_events_basename
if ~keyword_set(composite_psf_fn)     then composite_psf_fn      = sourcedir + psf_basename
                                           merged_region_fn      = sourcedir + src_region_basename


if (n_elements(make_plot) EQ 0) then make_plot=0

null_val = !VALUES.F_NAN
report = {ae_radial_profile, $
           sourcename      : ''                 , $
           label           : ''                 , $
           rate_3x3        : null_val           , $
           theta           : null_val           , $
           src_cnts        : 0L                 , $
           src_radius      : null_val           , $
           energy_range    : [null_val,null_val], $
           prob_ks_r       : null_val           , $
           prob_ks_dx      : null_val           , $
           prob_ks_dy      : null_val      }
    

;; ------------------------------------------------------------------------
;; Convert the declared celestial position of the source to the (x,y) tangent plane system of the composite event list.
theader = headfits(merged_env_events_fn, EXT=1, ERRMSG=error )
if (keyword_set(error)) then begin
  print, error
  print, 'ERROR reading ' + merged_env_events_fn
  GOTO, FAILURE
endif

      
fxbfind, theader, 'TTYPE', dum1, TTYPE, dum2, 'null'
fxbfind, theader, 'TCTYP', dum1, TCTYP, dum2, 'null'
fxbfind, theader, 'TCRVL', dum1, TCRVL, dum2, 0.0D
fxbfind, theader, 'TCRPX', dum1, TCRPX, dum2, 0.0D
fxbfind, theader, 'TCDLT', dum1, TCDLT, dum2, 0.0D
colnames = strlowcase( strtrim(TTYPE,2) )
x_ind    = where(strlowcase(colnames) EQ 'x')
y_ind    = where(strlowcase(colnames) EQ 'y')
make_astr, event2wcs_astr, DELTA=TCDLT[[x_ind,y_ind]], CTYPE=TCTYP[[x_ind,y_ind]], $
                           CRPIX=TCRPX[[x_ind,y_ind]], CRVAL=TCRVL[[x_ind,y_ind]]

ad2xy, ra_src, dec_src, event2wcs_astr, xpos_catalog, ypos_catalog
; The (x,y) coords must be incremented by 1 because ad2xy is thinking of 0-based IDL array indexes.
xpos_catalog++
ypos_catalog++

xpos_catalog = float(xpos_catalog)
ypos_catalog = float(ypos_catalog)


;; ------------------------------------------------------------------------
;; Mask (set to NaN) the PSF pixels outside a circular analysis region that is centered on the source position.

analysis_region = string(xpos_catalog, ypos_catalog, src_radius, F='(%"circle(%f,%f,%f)")')
analysis_name   = ''

;; We use the "opt full" below, so that the emap passed to dmimgpick later will have NaNs at all the edge pixels.
;; That should avoid the dmimgpick bug in HelpDesk Ticket #020605.

cmd = string(composite_psf_fn, analysis_region, temp_image_fn, F="(%'dmcopy ""%s[sky=%s][opt full,null=NaN,update=no]"" %s clobber=yes')")
  
run_command, cmd

; Read the masked PSF image, with astrometry, 
psf_image = readfits(temp_image_fn, psf_header, /SILENT)

psf_spec   = size(psf_image, /STRUCTURE)

psf_energy = psb_xpar( psf_header,'ENERGY')
 ra_psf    = psb_xpar( psf_header, 'RA' , COUNT=count1)
dec_psf    = psb_xpar( psf_header, 'DEC', COUNT=count2)

if (count1 NE 1) || (count2 NE 1) then begin
  print, 'WARNING: the PSF is missing RA/DEC keywords; using the source position instead.' 
   ra_psf    =  ra_src
  dec_psf    = dec_src
endif

extast, psf_header, psf2wcs_astr

skypixel_per_psfpixel = abs(psb_xpar( psf_header, 'CDELT1P'))

; Locate the origin of the PSF model in 0-based column,row units.
ad2xy, ra_psf, dec_psf, psf2wcs_astr, psf_origin_column, psf_origin_row


; Build arrays that record the distance from the PSF origin to each pixel in psf_image, in units of PSF pixels.
; We will need both single-axis distances, and radial distance.
make_2d, findgen(psf_spec.DIMENSIONS[0]), $
         findgen(psf_spec.DIMENSIONS[1]), $
         psf_dx, $
         psf_dy
psf_dx      -= psf_origin_column
psf_dy      -= psf_origin_row
psf_distance = sqrt(psf_dx^2 + psf_dy^2)
psf_azimuth  = atan(psf_dy, psf_dx)*!RADEG  ; CC from +X axis

; Convert all three to units of skypixels.
psf_dx       *= skypixel_per_psfpixel
psf_dy       *= skypixel_per_psfpixel
psf_distance *= skypixel_per_psfpixel


; Retain only unmasked PSF pixels (pixels within the circular analysis region).
ind = where(finite(psf_image), count)

if (count LT 3) then begin
  print, 'Radial profile analysis skipped -- too few PSF pixels in aperture.'
  return
endif

psf_dx       = psf_dx      [ind]
psf_dy       = psf_dy      [ind]
psf_distance = psf_distance[ind]
psf_azimuth  = psf_azimuth [ind]
psf_pixels   = psf_image   [ind]


;; ------------------------------------------------------------------------
;; Extract inband event data that lie in the footprint of the PSF pixels selected above.
;; We DO NOT filter with analysis_region here because it can easily extend beyond the PSF footprint!
;;
;; IT IS THE OBSERVER'S RESPONSIBILITY TO ENSURE THAT THE SRC POSITION ESTIMATE IS ACCURATE, 
;; i.e. that the data and PSF model are aligned !!!

cmd1 = string(merged_env_events_fn, temp_image_fn, temp_events_fn, $
              F="(%'dmimgpick ""%s[cols sky,energy]"" %s %s method=closest clobber+')")

cmd2 = string(temp_events_fn, 1000*energy_range, inband_events_fn, $
             F="(%'dmcopy ""%s[energy=%6.1f:%7.1f,#3>0]"" %s clobber+')")
run_command, [cmd1,cmd2]

inband_events = mrdfits(inband_events_fn, 1, theader, /SILENT, STATUS=status)
if (status NE 0) then begin
  print, 'ERROR reading ' + inband_events_fn
  GOTO, FAILURE
endif

inband_src_counts = psb_xpar( theader, 'NAXIS2')

if (inband_src_counts LT 4) then begin
  ; We need at least 4 counts for KS probability to be meaningful.
  print, 'Radial profile analysis skipped -- too few in-band counts.'
  return
endif 
    
; Compute the distances (in units of skypix) from the events to the source position.
event_dx       = (float(inband_events.X) - xpos_catalog)
event_dy       = (float(inband_events.Y) - ypos_catalog)
event_distance = sqrt(event_dx^2 + event_dy^2)
event_azimuth  = atan(event_dy, event_dx)*!RADEG  ; counter-clockwise from +X axis


if keyword_set(make_plot) then begin
  if keyword_set(my_ds9) then begin
    run_command, string(my_ds9, F='(%"xpaaccess ''%s''")'), result, /IGNORE_STATUS, /QUIET
    if (result[0] NE 'yes') then my_ds9=''
  endif

  if ~keyword_set(my_ds9) then ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='ae_radial_profile_'+session_name, OPTION='-linear -bin factor 0.125'

  ; We have to set TDMIN & TDMAX to get ds9 to produce a nice default binning/zoom.
  ; HEASARC standards say their datatype should be the same as that of the table column to which they refer. 
  openw, unit, temp_text_fn, /GET_LUN
  printf, unit, (xpos_catalog-1.5*src_radius), (xpos_catalog+1.5*src_radius), $
                (ypos_catalog-1.5*src_radius), (ypos_catalog+1.5*src_radius), $
                F='(%"#add\nTDMIN1=%0.2f\nTDMAX1=%0.2f\nTDMIN2=%0.2f\nTDMAX2=%0.2f")'
  free_lun, unit
  
  cmd = string(inband_events_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
  run_command, cmd, /QUIET
  cmd = string(  temp_events_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
  run_command, cmd, /QUIET
  
  
  run_command, string(merged_env_events_fn, temp_region_fn, xpos_catalog, ypos_catalog, F="(%'make_psf_asymmetry_region %s %s %0.2f %0.2f format=ds9 clob+')")
  
  openw   , region1_unit, temp_region_fn, /GET_LUN, /APPEND
  printf  , region1_unit, analysis_region, F='(%"%s # color={red}")'
  free_lun, region1_unit
  
  
  ; Display data, PSF, and image reconstruction in ds9.
  analysis_region_cmd = 'command "{' + analysis_region + '# color=red}"'

  data_list   = [temp_image_fn      , merged_env_events_fn, inband_events_fn]
  region_list = [analysis_region_cmd, temp_region_fn      , keyword_set(region_file) ? region_file : analysis_region_cmd]
  
  composite_img_fn = file_dirname(merged_env_events_fn,/MARK_DIRECTORY) + env_image_basename
  dum = headfits(composite_img_fn, ERRMSG=error, EXT=1)
  recon_available = ~keyword_set(error)
  
  if recon_available then begin
   data_list   = [data_list  , string(composite_img_fn, F="(%'""%s[1]""')")]
   region_list = [region_list, '']
  endif
  
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, data_list, region_list

  run_command, string(my_ds9,my_ds9,my_ds9,my_ds9,my_ds9,my_ds9, F='(%"xpaset -p ''%s'' frame 3; xpaset -p ''%s'' zoom to fit; xpaset -p ''%s'' match frame wcs")'), /IGNORE_STATUS, /QUIET

  if recon_available then begin
    run_command, string(my_ds9,my_ds9,my_ds9,my_ds9,my_ds9,my_ds9, F='(%"xpaset -p ''%s'' frame 2; xpaset -p ''%s'' regions select all; xpaset -p ''%s'' regions copy; xpaset -p ''%s'' regions select none; xpaset -p ''%s'' frame 4; xpaset -p ''%s'' regions paste")'), /IGNORE_STATUS, /QUIET
  endif
  
  print, energy_range_label, F='(%"\nThe two LEFT FRAMES in ds9 show the PSF pixels and the events (selected by the RED CIRCLE and the %s band) used in the radial profile analysis.\nThe right frame shows all neighborhood events and the location of the \"PSF hook\".")'
endif


;; ------------------------------------------------------------------------
;; Construct the cumulative radial distribution function for the PSF within the analysis aperture.
;; This is an "enclosed fraction" model for the PSF within the analysis aperture.
;; The radial distances at which this function is sampled are stored in the sorted vector psf_distance.
;; The values of the function at the sample point are stored in the corresponding vector psf_distance_distn. 

sort_ind          = sort(psf_distance)
psf_distance      = psf_distance [sort_ind]
sorted_psf_pixels = psf_pixels   [sort_ind]
    
psf_distance_distn = total(sorted_psf_pixels, /NAN, /DOUBLE, /CUMULATIVE) / total(sorted_psf_pixels, /NAN, /DOUBLE)


;; Similarly, construct cumulative azimuthal distributions for the PSF within the analysis aperture.
;; The azimuth angles at which this function is sampled are stored in the sorted vector psf_azimuth.
;; The values of the function at the sample point are stored in the corresponding vector psf_azimuth_distn. 

sort_ind          = sort(psf_azimuth)
psf_azimuth       = psf_azimuth [sort_ind]
sorted_psf_pixels = psf_pixels   [sort_ind]
    
psf_azimuth_distn = total(sorted_psf_pixels, /NAN, /DOUBLE, /CUMULATIVE) / total(sorted_psf_pixels, /NAN, /DOUBLE)


;; Similarly, construct cumulative single-axis offset distributions for the PSF within the analysis aperture.
;; The single-axis offsets at which this function is sampled are stored in the sorted vectors psf_dx, psf_dy.
;; The values of the function at the sample point are stored in the corresponding vector psf_dx_distn, psf_dy_distn. 

sort_ind          = sort(psf_dx)
psf_dx            = psf_dx       [sort_ind]
sorted_psf_pixels = psf_pixels   [sort_ind]
    
psf_dx_distn = total(sorted_psf_pixels, /NAN, /DOUBLE, /CUMULATIVE) / total(sorted_psf_pixels, /NAN, /DOUBLE)


sort_ind          = sort(psf_dy)
psf_dy            = psf_dy       [sort_ind]
sorted_psf_pixels = psf_pixels   [sort_ind]
    
psf_dy_distn = total(sorted_psf_pixels, /NAN, /DOUBLE, /CUMULATIVE) / total(sorted_psf_pixels, /NAN, /DOUBLE)


;; ------------------------------------------------------------------------
;; Sort the event offsets and distances.
;; This produces vectors of offset and distance values at which the cumulative distributions for the observed events take a step upward of size (1/inband_src_counts).
event_dx       = event_dx      [sort(event_dx      )]
event_dy       = event_dy      [sort(event_dy      )]
event_distance = event_distance[sort(event_distance)]
event_azimuth  = event_azimuth [sort(event_azimuth )]

;; ------------------------------------------------------------------------
; Evaluate the PSF's cumulative distribution functions for dx, dy, and distance at the sorted dx, dy, and distance values where we detected events.

; We MUST make sure not to extrapolate in a way that produces values outside of [0,1]
; in the common case where the range of the event distances exceeds the range of the
; PSF pixel distances.
; We could avoid extrapolation completely by using linterp.pro, however extrapolation
; from psf_distance[0] to zero gives a better PSF model than clipping to psf_distance_distn[0].

psf_distance_distn_samples = 0 > interpol([0,psf_distance_distn], [0,psf_distance], event_distance) < 1
      psf_dx_distn_samples = 0 > interpol([0,      psf_dx_distn], [0,psf_dx      ], event_dx      ) < 1
      psf_dy_distn_samples = 0 > interpol([0,      psf_dy_distn], [0,psf_dy      ], event_dy      ) < 1


;; ------------------------------------------------------------------------
; We now have pairs of cumulative distributions for three quantities: dx, dy, distance.
;
; For each pair, we have a cumulative distribution for the event data, which is a series of stairsteps, each of height (1/inband_src_counts), located at the coordinates in event_dx/event_dy/event_distance.
;
; For each pair, we also have the cumulative distribution for the PSF, evaluated at those same sample points and stored in psf_distance_distn_samples/psf_dx_distn_samples/psf_dy_distn_samples.

; For each pair, we will now evaluate the KS distance between the two cumulative distributions, and compute the associated null hypothesis probablity.
; These lines are derived from ksone.pro in AstroLib.
cum_distn_before_step = (  findgen(inband_src_counts)) / inband_src_counts
cum_distn_after_step  = (1+findgen(inband_src_counts)) / inband_src_counts

ks_distance = max( abs( psf_distance_distn_samples - cum_distn_before_step ) ) > $
              max( abs( psf_distance_distn_samples - cum_distn_after_step  ) )

prob_ks, ks_distance, inband_src_counts, prob_ks_r
;help, ks_distance
        
        
ks_distance = max( abs(       psf_dx_distn_samples - cum_distn_before_step ) ) > $
              max( abs(       psf_dx_distn_samples - cum_distn_after_step  ) )

prob_ks, ks_distance, inband_src_counts, prob_ks_dx
;help, ks_distance
        
        
ks_distance = max( abs(       psf_dy_distn_samples - cum_distn_before_step ) ) > $
              max( abs(       psf_dy_distn_samples - cum_distn_after_step  ) )

prob_ks, ks_distance, inband_src_counts, prob_ks_dy
;help, ks_distance



; ; Calculate the 50% encircled energy radius for the extracted data.
; R_MEDIAN = median(event_distance, /EVEN)
; 
; ; Find the PSF fraction enclosed by this radius.
; EE_AT_RM = 0 > interpol([0,psf_distance_distn], [0,psf_distance], R_MEDIAN) < 1

;; ------------------------------------------------------------------------
;; Visually examine the spatial distributions of these event data and of the PSF.
if keyword_set(make_plot) then begin
  if (n_elements(plot_ids) EQ 4) then begin
    id0 = plot_ids[0]
    id1 = plot_ids[1]
    id2 = plot_ids[2]
    id3 = plot_ids[3]
  endif
  
  stair_x = fltarr(2*inband_src_counts)
  stair_y = fltarr(2*inband_src_counts)
  ind     = 2*lindgen(inband_src_counts)
  stair_y[ind  ] = cum_distn_before_step
  stair_y[ind+1] = cum_distn_after_step

  stair_x[ind  ] = event_distance
  stair_x[ind+1] = event_distance
  function_1d,id0,BLOCK=block, stair_x, stair_y, DATASET=dataset_prefix+'data'+analysis_name, LINE=0, COLOR='green', TIT=sourcename, XTIT='r [skypix]', YTIT='cumulative distribution'

  stair_x[ind  ] = event_azimuth
  stair_x[ind+1] = event_azimuth
  function_1d,id1,BLOCK=block, stair_x, stair_y, DATASET=dataset_prefix+'data'+analysis_name, LINE=0, COLOR='green', TIT=sourcename, XTIT='azimuth, counter-clockwise from +X axis [degree]', YTIT='cumulative distribution'

  stair_x[ind  ] = event_dx
  stair_x[ind+1] = event_dx
  function_1d,id2,BLOCK=block, stair_x, stair_y, DATASET=dataset_prefix+'data'+analysis_name, LINE=0, COLOR='green', TIT=sourcename, XTIT='dx [skypix]', YTIT='cumulative distribution'

  stair_x[ind  ] = event_dy
  stair_x[ind+1] = event_dy
  function_1d,id3,BLOCK=block, stair_x, stair_y, DATASET=dataset_prefix+'data'+analysis_name, LINE=0, COLOR='green', TIT=sourcename, XTIT='dy [skypix]', YTIT='cumulative distribution'
  
  function_1d,id0, psf_distance, psf_distance_distn, DATASET=dataset_prefix+psf_name+''+analysis_name ,LEGEND_STYLE=1,  SUBT=string(prob_ks_r , F='(%"Pks = %0.4f")')
  function_1d,id1, psf_azimuth ,  psf_azimuth_distn, DATASET=dataset_prefix+psf_name+''+analysis_name ,LEGEND_STYLE=1;,  SUBT=string(prob_ks_r , F='(%"Pks = %0.4f")')
  function_1d,id2, psf_dx      ,       psf_dx_distn, DATASET=dataset_prefix+psf_name+''+analysis_name ,LEGEND_STYLE=1,  SUBT=string(prob_ks_dx, F='(%"Pks = %0.4f")')
  function_1d,id3, psf_dy      ,       psf_dy_distn, DATASET=dataset_prefix+psf_name+''+analysis_name ,LEGEND_STYLE=1,  SUBT=string(prob_ks_dy, F='(%"Pks = %0.4f")')

  
  ; Compute azimuth of the Chandra "PSF hook" feature; see similar calculation in make_psf_asymmetry_region.
  ; The feature is at a Chandra position angle of 195 +- 25 degrees.
  ; Our azimuth angle is defined as counter-clockwise from the +X axis.
  ; Azimuth is related to the position angle (PA) by : theta = (90 + PA - roll) MOD 360
  ; However, our azimuth plot runs from -180 to +180, so we want to put the hook azimuth in that range.
  roll = psb_xpar( theader, 'ROLL_NOM')
  hook_azimuth  = (90.0 + 195.0 - roll) MOD 360
  hook_azimuth += (hook_azimuth LT -180) ? 360 : 0
  hook_azimuth -= (hook_azimuth GT  180) ? 360 : 0
  
  function_1d,id1, [hook_azimuth,hook_azimuth],  [0.25,0.75], DATASET='PSF hook',COLOR='blue',LEGEND_STYLE=1
  
  plot_ids = [id0,id1,id2,id3]

  print, psf_energy, inband_src_counts, energy_range_label, src_radius, F='(%"\nPSF monoenergy is %0.2f keV; analyzed %d events in %s and in circular aperture (r = %0.1f skypix).")'
endif ; keyword_set(make_plot)

if (psf_energy LT energy_range[0]) || (psf_energy GT energy_range[1]) then print, 'WARNING!  PSF monoenergy is outside the event energy range!!'

report = {ae_radial_profile, $
           sourcename       ,$ ;: sourcename
           label            ,$ ;; label
           null_val         ,$ ;; rate_3x3
           null_val            ,$ ;; theta
           inband_src_counts,$ ;; src_cnts
           src_radius       ,$ ;: src_radius
           energy_range     ,$ ;: energy_range
           prob_ks_r        ,$ ;: prob_ks_r
           prob_ks_dx       ,$ ;: prob_ks_dx 
           prob_ks_dy       }  ;: prob_ks_dy
    
    

CLEANUP:
if ~tempdir_supplied_by_caller && file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif

if (exit_code EQ 0) then return $
else begin
  print, 'ae_radial_profile: Returning to top level due to fatal error.'
  retall
endelse


FAILURE:
exit_code = 1
GOTO, CLEANUP
end  ;  ae_radial_profile


    
; =============================================================================
;;; Routine to apply AE's grouping algorithm (see manual).
;;;
;;; To have the algorithm ignore background set bkg_spectrum_fn=''.
;;; If grouped_spectrum_fn is null a grouped filename will be constructed and returned.
;;; See manual for meaning of CHANNEL_RANGE, SNR_RANGE, NUM_GROUPS_RANGE.
;;; The parameters this_snr_goal, grp_name, group_codes, num_groups, inband_src_counts
;;; return information about the grouping that AE's /FIT_SPECTRA stage needs.
PRO ae_group_spectrum, src_spectrum_fn, bkg_spectrum_fn, grouped_spectrum_fn, $
                       CHANNEL_RANGE=channel_range, $
                       SNR_RANGE=snr_range, NUM_GROUPS_RANGE=num_groups_range, $
                       CREATOR_STRING=creator_string, $
                       XCM_FN=xcm_fn, $
                       this_snr_goal, grp_name, channel_starting_group, num_groups, inband_src_counts, inband_scaled_bkg_counts 

if NOT keyword_set(creator_string) then creator_string = "ae_group_spectrum, version " +strmid("$Rev:: 5658  $",7,5) +strmid("$Date: 2022-01-25 07:06:00 -0700 (Tue, 25 Jan 2022) $", 6, 11)

if (n_elements(channel_range) NE 2) then channel_range=[35,548]
min_channel = fix(channel_range[0])
max_channel = fix(channel_range[1])
 
; SNR_RANGE[1] is the user's goal for defining groups; SNR_RANGE[0] is the lower limit allowed before we abort the grouping attempt
if (n_elements(snr_range) EQ 0) then $
  snr_range = [1,3]
if (n_elements(snr_range) NE 2) then begin
  print, 'ERROR: keyword SNR_RANGE should be a 2-element vector giving the range of SNR allowed for each spectral group, e.g. [2.5,5].'
  return      
endif

if (snr_range[1] LT 0) then begin
  print, 'ERROR: minimum SNR value (SNR_RANGE[1]) must be positive'
  return
endif

if (n_elements(num_groups_range) EQ 0) then $
  num_groups_range = [2+8,250]
if (n_elements(num_groups_range) NE 2) then begin
  print, 'ERROR: keyword NUM_GROUPS_RANGE should be a 2-element vector specifying how many spectral groups are desired, e.g. [2+8,250].'
  return    
endif

;print, channel_range
;print, snr_range
;print, num_groups_range
;; ------------------------------------------------------------------------
;; Read the source & background spectra.
pheader   = headfits(src_spectrum_fn)
bin_table = mrdfits( src_spectrum_fn, 1, src_theader, /SILENT, STATUS=status)
if (status NE 0) then message, 'ERROR reading ' + src_spectrum_fn

src_channels        = bin_table.CHANNEL 
src_observed_counts = bin_table.COUNTS 
num_channels        = n_elements(src_channels)

if keyword_set(bkg_spectrum_fn) then begin
  ; This code does not handle every valid OGIP configuration.
  src_areascal        = float(psb_xpar( src_theader, 'AREASCAL'))
  if (src_areascal NE 1) then message, 'ERROR: expected AREASCAL=1 in src spectrum.'
  
  src_backscal        = float(psb_xpar( src_theader, 'BACKSCAL'))
  if (src_backscal EQ 0) then message, 'ERROR: BACKSCAL keyword missing from src spectrum.'
  
  bin_table = mrdfits( bkg_spectrum_fn, 1, bkg_theader, /SILENT, STATUS=status)
  if (status NE 0) then message, 'ERROR reading ' + bkg_spectrum_fn

  bkg_observed_counts = bin_table.COUNTS
  
  bkg_areascal        = float(psb_xpar( bkg_theader, 'AREASCAL', COUNT=count))
  if (count EQ 0) then $
    bkg_areascal      = bin_table.AREASCAL
  
  bkg_backscal        = float(psb_xpar( bkg_theader, 'BACKSCAL', COUNT=count))
  if (count EQ 0) then $
    bkg_backscal      = bin_table.BACKSCAL
  
  if (min(bkg_areascal) GT 0) && (min(bkg_backscal) GT 0) then begin  
    bkg_counts_in_src_region = (src_backscal/bkg_backscal) * (bkg_observed_counts / bkg_areascal)
  endif else begin
    if (total(bkg_observed_counts) EQ 0) then begin
      ; This looks like an empty background spectrum.
      bkg_counts_in_src_region = replicate(0.0, n_elements(bkg_observed_counts))
    endif else message, 'ERROR: '+bkg_spectrum_fn+'has data but either BACKSCAL or AREASCAL are missing!'
  endelse
endif else begin
  bkg_observed_counts      = 0
  bkg_counts_in_src_region = 0
endelse

group_bins_to_snr, src_observed_counts, bkg_observed_counts, bkg_counts_in_src_region, $
                   GROUP_WITHOUT_BACKGROUND=(keyword_set(bkg_spectrum_fn) EQ 0), $
                   START_INDEX=value_locate(src_channels, min_channel), STOP_INDEX=value_locate(src_channels, max_channel), $
                   SNR_RANGE=snr_range, NUM_GROUPS_RANGE=num_groups_range, /VERBOSE, $
                   this_snr_goal, group_codes

grp_name = strcompress(string(this_snr_goal, F='(%"grp%8.1f")'),/REMOVE_ALL)

channel_starting_group = src_channels[where(group_codes EQ 1, num_groups)]

fdecomp, src_spectrum_fn, disk, item_path, item_name, item_qual

if (NOT keyword_set(grouped_spectrum_fn)) then $
  grouped_spectrum_fn = item_path + item_name + '_' + grp_name + (('' EQ item_qual) ? '' : ('.' +item_qual))


;; ------------------------------------------------------------------------
;; Write the grouped source spectrum.
if (grouped_spectrum_fn NE '/dev/null') then begin
  psb_xaddpar, pheader, 'CREATOR', creator_string
  psb_xaddpar, pheader, "FNFITS", file_basename(grouped_spectrum_fn)
  ; Clean structural keywords from header, since the input file may have had data in this HDU.
  naxis = psb_xpar( pheader,'NAXIS')
  psb_xaddpar, pheader, 'NAXIS', 0
  if naxis gt 0 then for ii=1,naxis do sxdelpar,pheader,'NAXIS'+strtrim(ii,2)  
              
  writefits, grouped_spectrum_fn, 0, pheader
  
  row = { CHANNEL: 0, COUNTS: 0L, GROUPING:0 }
  bin_table = replicate(row, num_channels)
  bin_table.CHANNEL = src_channels 
  bin_table.COUNTS  = src_observed_counts
  bin_table.GROUPING= group_codes
  psb_xaddpar, src_theader, 'CREATOR' , creator_string
  psb_xaddpar, src_theader, 'SNR_GOAL', this_snr_goal
  psb_xaddpar, src_theader, 'NUMGRPS' , num_groups
  psb_xaddpar, src_theader, 'CHAN_LO' , min_channel, 'channel range' 
  psb_xaddpar, src_theader, 'CHAN_HI' , max_channel, 'channel range' 
  sxdelpar, src_theader, 'GROUPING'
  
  ; Clean structural keywords from header, since the input file may have had columns other than CHANNEL and COUNTS.
  fxhclean, src_theader
  
  mwrfits, bin_table, grouped_spectrum_fn, src_theader
  print, num_groups, this_snr_goal, file_basename(grouped_spectrum_fn), F='(%"Spectrum with %d groups (SNR=%0.1f) written to %s")'
  
  
  if keyword_set(xcm_fn) then begin
    openw,  unit1, xcm_fn, /GET_LUN
    printf, unit1, grouped_spectrum_fn,    F='(%"data   1:1 %s ")'
    printf, unit1, num_groups,             F='(%"ignore 1:1,%d ")'
    
    ; Display the backgrounds.
    printf, unit1, 'setplot background'
    
    printf, unit1, 'setplot energy'                
    printf, unit1,                          F="(%'setplot command rescale x 0.5 8.0')"
    printf, unit1, grouped_spectrum_fn,     F="(%'setplot command LAbel Top ""%s"" ')"
    printf, unit1, 'cpd /xs'
    printf, unit1, 'plot ldata'
    printf, unit1, 'exit'
    free_lun, unit1
  endif ; xcm_fn
endif

ind = where((src_channels GE min_channel) AND (src_channels LE max_channel))
inband_src_counts        = total(/INT,      src_observed_counts[ind])
inband_scaled_bkg_counts = total(      bkg_counts_in_src_region[ind])
return
end ; ae_group_spectrum



; =============================================================================
;;; Using an XSPEC script constructed by AE, call XSPEC to perform a fit and manage the data products produced.
;;;
;;; * sourcedir is the source extraction directory, e.g. '181948.97-161633.2/photometry/'
;;;
;;; * fit_result_root is a name for this fitting session---a combination of the 
;;;   grouping scheme and fitting script name(s), e.g. 'nogrp_tbabs_vapec_A'.
;;;   The fitting script is expected to be at the path 
;;;     sourcedir + 'spectral_models/' + fit_result_root + '.xcm'
;;;
;;; * By default, the call to XSPEC arranges for the XSPEC screen output to appear in the IDL window, in case 
;;;   your script uses the "interactive" option.  If you are sure it does not, then you can specify
;;;   INTERACTIVE=0 to hide the XSPEC screen output.
;;;
;;; * FIT_TIMEOUT sets a CPU time limit (in seconds) on XSPEC.

PRO ae_perform_fit, sourcedir, fit_result_root, INTERACTIVE=interactive, TEMPDIR=tempdir, FIT_TIMEOUT=fit_timeout


tempdir_supplied_by_caller = keyword_set(tempdir)
if ~tempdir_supplied_by_caller then begin
  temproot = temporary_directory( 'ae_perform_fit', VERBOSE=1, SESSION_NAME=session_name)
  tempdir = temproot
endif
run_command, PARAM_DIR=tempdir


if (strmid(sourcedir,0,/REVERSE) NE '/') then sourcedir += '/'

if (n_elements(interactive) EQ 0) then interactive = 1
if ~keyword_set(fit_timeout)      then fit_timeout = 600

;; These two directory names must be identically defined in the AE code.
modelsubdir             = 'spectral_models/'
fit_stats_basename      = 'source.spectra'

;; These file names must be identically defined in the fitting scripts.
latex_result_fn         = 'summary.ps'
latex_figure_file       = 'ldata.ps'

; Locate the LaTex template file.
latex_template_basename = 'xspec_template'
result = routine_info( 'acis_extract', /SOURCE )
fdecomp, result.PATH, disk, codedir
latex_template_fn       = codedir + latex_template_basename + '.tex'

fit_xcm_fn     =             modelsubdir + fit_result_root + '.xcm'
fit_result_dir = sourcedir + modelsubdir + fit_result_root + '/'
file_mkdir, fit_result_dir

;; Remove all the files that XSPEC will be writing from the directory where XSPEC is run.
output_file_list = ['icounts.ps', 'ldata.ps', 'xspec_run.log', 'model.xcm','model_before_errors.xcm','model.xspecsav', 'model.txt', 'xspec.log', 'summary.ps', 'KT1_vs_fitstat.ps']
file_delete, sourcedir     +output_file_list, /ALLOW_NONEXISTENT

;; Remove the HDU in source.spectra that XSPEC will be writing to.
fit_stats_fn  = sourcedir + fit_stats_basename 
if file_test(fit_stats_fn) then begin
  fit_stats_keywords = get_keywords_from_hdu_headers(fit_stats_fn, {HDUNAME:''}, ERRMSG=error)

  if keyword_set(error) then message, 'ERROR opening '+fit_stats_fn
  
  ; Find the HDU matching the specified model name.
  extension_number = where( strmatch(fit_stats_keywords.HDUNAME, fit_result_root, /FOLD_CASE), count )
  
  if (count GT 1) then message, 'ERROR: '+fit_stats_fn+' has multiple HDUs named '+fit_result_root
  if (count EQ 1) then begin
    cmd = string(fit_stats_fn, extension_number, F="(%'fdelhdu ""%s[%d]"" N Y')")
    run_command, /HEASOFT, cmd
  endif
endif ;file_test(fit_stats_fn)


;; Run xspec.  We pass "tclexit" to stdin to get xspec to die if it has an
;; error that brings up the xspec prompt.
if keyword_set(interactive) then begin
  run_command, /HEASOFT, DIRECTORY=sourcedir, STATUS=status, /INTERACTIVE, string(fit_xcm_fn, F='(%"xspec - %s |& tee ./xspec_run.log")')
endif else begin
  cmd = []
  if strmatch(getenv("SHELL"), "*csh") then cmd = [cmd, 'set time=(0 "CPU seconds: %U + %S  elapsed time: %E")']

  cmd = [cmd,string(fit_timeout, F='(%"limit cputime %d")')]
  cmd = [cmd,string(fit_xcm_fn , F='(%"echo tclexit 99 | xspec - %s >&! ./xspec_run.log")')]

  run_command, /HEASOFT, DIRECTORY=sourcedir, STATUS=status, cmd, /NO_RETRY
endelse

; "Exit codes in the range 129-255 represent jobs terminated by Unix
; "signals". Each type of signal has a number, and what's reported as the job
; exit code is the signal number plus 128. Signals can arise from within the
; process itself (as for SEGV, see below) or be sent to the process by some
; external agent (such as the batch control system, or your using the "bkill"
; command). 
run_log = fit_result_dir+'xspec_run.log'
if keyword_set(status) then begin      
  case status of
    ; This is an exit code from the AE fitting scripts.
    97 : print,                     'ERROR:!  XSPEC script was unable to recognize your computing platform (operating system plus processor hardware) and thus did not know where to find the shared library containing the "cplinear" model compiled for your platform .  You may have to add an entry to the "switch" statment in the fitting script, and build the cplinear library yourself (a simple process shown in the AE manual).'
    ; This is an exit code from the AE fitting scripts.
    98 : begin
         print,                       F='("WARNING! XSPEC script detected an error:  ",$)'
         run_command, "egrep '^ERROR:' " + sourcedir+'xspec_run.log', result, /QUIET,/IGNORE_STATUS,/NO_RETRY
         forprint, result
         print, run_log,              F='(%"Examine XSPEC session log %s")'
         end
    ; This is from the tclexit command in the pipe above.
    99 : print, run_log,              F='(%"WARNING!  Examine XSPEC session log %s; XSPEC script execution aborted.")'
    ; signal SIGXCPU (BSD)
    152: print, run_log, fit_timeout, F='(%"WARNING!  Examine XSPEC session log %s; XSPEC process killed after consuming %d CPU seconds.")'
    ; ???
    158: print, run_log, fit_timeout, F='(%"WARNING!  Examine XSPEC session log %s; XSPEC process killed after consuming %d CPU seconds.")'
    
    else:print, run_log, status,      F='(%"WARNING!  Examine XSPEC session log %s; XSPEC process failed for unknown reason (exit code %d).")'
  endcase
endif else begin
  run_command, "egrep '^WARNING:' " + sourcedir+'xspec_run.log', result, /QUIET,/IGNORE_STATUS,/NO_RETRY
  forprint, '  >'+result
  
  ; Save the model.
  if (NOT file_test(sourcedir+'model.txt')) then begin
    print, run_log, F='(%"WARNING!  Examine XSPEC session log %s; no model saved by XSPEC.")'
  endif else if file_test(fit_stats_fn) then begin
    cmd = string(sourcedir+'model.txt', fit_stats_fn, F="(%'dmappend %s %s')")
    run_command, cmd
  endif else begin
    cmd = string(sourcedir+'model.txt', fit_stats_fn, F="(%'dmcopy %s %s')")
    run_command, cmd
  endelse

  ; Use LaTeX to make a summary document.
  if file_test(sourcedir+latex_figure_file) then begin
    ;; Run latex, passing ^D (ASCII character #4) in case latex tries to prompt for input.
    cmd = [string(latex_template_fn, F='(%"printf ''\4'' | latex %s >&! ./latex_run.log")'), $
           string(latex_template_basename, latex_result_fn, F='(%"dvips %s -q -o %s")')]
    run_command, /UNIX, DIRECTORY=sourcedir, cmd
    
    file_delete, sourcedir + ['latex_run.log','xspec_template.aux', 'xspec_template.dvi', 'xspec_template.log'], /ALLOW_NONEXISTENT
  endif else begin
    print, run_log, F='(%"WARNING!  Examine XSPEC session log %s; no plot produced by XSPEC.")'
  endelse
endelse ; XSPEC status ok

;; Remove any existing files from the subdirectory designated to hold the fitting results, and then move the XSPEC output files there.
file_delete, fit_result_dir+output_file_list, /ALLOW_NONEXISTENT

ind = where(file_test(sourcedir+output_file_list), count)
if (count GT 0) then $
  file_move, /OVERWRITE, sourcedir+output_file_list[ind], fit_result_dir 

if ~tempdir_supplied_by_caller && file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif

return
end  ; ae_perform_fit






; =============================================================================
; Tool to make PSF images using MARX
; PSF models are reasonably calibrated ONLY when using MARX versions 5.3 through 5.5!!
; See release notes for AE Version 2016may25 and email titled "marx-users: MARX 5.5 released".
;
; Prior to MARX 5, the only PSF blur component we allowed MARX to model was the aspect blur.
; This tool modeled the pixelization of event position by ACIS, and (if needed) the position randomization done by acis_process_events (APE).
; In MARX 5, the marx tool models aspect blur, and the marx2fits tool models what happens in APE
; =============================================================================
;Are these PSF headers handled properly in /MERGE?

;;; This code is not very elegant!   We're trying to drive MARX simulations required by three different callers.

;;; 1. AE's CONSTRUCT_REGIONS stage calls this tool to build PSF images from MARX.  The observation's aspect file is provided by the caller. 

;;; 2. The ae_chart_interface tool calls this tool to "detect" rays produced by CXC's ChaRT model of the HRMA.  MARX's internal dither model is used, and the corresponding aspect file is returned to the caller.

;;; 3. The recon_spectrum tool calls this tool to produce *photons* from MARX, which are then fed to our CCD simulator.  MARX's internal dither model is used, and the corresponding aspect file is returned to the caller.


;;; Calls to this tool take two different forms.  A single "initialize" call provides parameters that are common for all sources in an observation.  Then, one or more "simulate" calls provide source-specific parameters.


;;; "INITIALIZE" CALL
;;;
;;;  1. ae_make_psf, EVENT_FILE='source.evt', OBS_ASPECT_FN='obs.asol', EXPOSURE=?, ...
;;;  2. ae_make_psf, EVENT_FILE='source.evt', /SAOSACFile, ...
;;;  3. ae_make_psf, EVENT_FILE='source.evt', /SpectrumFile, ...
;;;
;;; Required keywords:
;;; EVENT_FILE is any event file that can provide MARX with the geometry of the observation.
;;;
;;; OBS_ASPECT_FN (style #1) is an INPUT aspect file for the observation; it will be provided to MARX for dither and is used by dmcoords.
;;;
;;; Pass /SAOSACFile   if you will be supplying an incident photon list     in the subsequent "simulate" calls..  
;;; Pass /SpectrumFile if you will be supplying an incident photon spectrum in the subsequent "simulate" calls..  
;;;
;;; EXPOSURE is the duration of the MARX simulation.
;;;
;;;
;;; Optional keywords (may differ for the 3 callers):
;;;
;;;
;;; ASPECT_BLUR (arcsec) is the standard deviation for a Gaussian model of the expected PSF blur due to aspect reconstruction errors.
;;;  XXXXXX   If omitted, a default value is read from marx.par.
;;;
;;; PIX_ADJ = {{EDSER, CENTROID, RANDOMIZE, NONE} is the pix_adj parameter of marx2fits.
;;;  XXXXXX   If omitted, the value is read from PIX_ADJ in the event file.
;;; 
;2016 May 20: PIX_ADJ and ASPECT_BLUR are currently hard-coded to reflect Leisa's calibration of MARX to match some observed sources near the I and S aimpoints!!!!!!!!!!!!
;;;
;;;
;;; ACIS_Exposure_Time is used by the ae_streak_model tool to set the ACIS frame time to zero, which simulates just the ACIS readout streak.
;;;
;;;
;;;
;;;
;;; SIM_Y_OFFSET, SIM_Z_OFFSET specify how much the virtual Science Instrument Module (SIM) should be moved along the SIM_Y/SIM_Z axes (in units of mm).  Such movement allows one to control whether the virtual dither pattern "falls off the edge" of a CCD. 


;;; "SIMULATE" CALLS
;;;
;;; 1. ae_make_psf, ra, dec, psf_fn, skypixel_per_psfpixel, footprint, psf_energy, desired_psf_counts
;;;
;;; 2. ae_make_psf, ra, dec, psf_fn, skypixel_per_psfpixel, footprint, psf_energy, 0, EMAP_VAL=emap_val, SAOSACFile=ray_filename
;;;
;;; 3. ae_make_psf, ra, dec, SpectrumFile=incident_model_fn, MARX_ASPECT_FN=marx_aspect_fn, EXPOSURE=sim_duration, /MARX_ONLY
;;;
;;; ra, dec is celestial position of source.
;;;
;;; skypixel_per_psfpixel should be carefully chosen so that an ODD NUMBER of PSF pixels equals one ACIS pixel (0.492") 
;;; in order to get accurate boxcar smoothing below.
;;;
;;; footprint (skypixels) is desired dimension of square PSF image..
;;;
;;; psf_energy (keV) and desired_psf_counts can be vectors.
;;;
;;; MARX_ASPECT_FN (style #3) is the name of an OUTPUT file into which MARX will record the dither motion it generated for the simulation.
;;;
;;; X_CAT,Y_CAT (skypixels) is position of source in SKY system.  Will be computed if not passed.
;;;
;;; OFF_ANGLE is off-axis angle in arcmin.   Will be computed if not passed.
;;;
;;; CHIP_ID is CCD_ID mostly under the source.  Will be computed if not passed.

;;; EMAP_VAL is the level of the emap at the source position, assumed to be in units of s cm**2 count /photon at 1.0 keV.  
;;; Typical on-axis value will be used if not passed.

;;; SpectrumFile (style #3) supplies MARX with an incident photon spectrum.  

;;; In these "run" calls, TEMP_DIR should be a  path (including trailing slash) the directory where you want MARX output to go.


PRO ae_make_psf, TEMP_DIR=tempdir, $
  
                 EVENT_FILE=obsdata_filename_p, EXPOSURE=exposure_p, $
                 OBS_ASPECT_FN=obs_aspect_fn_p, ASPECT_BLUR=aspect_blur_p, PIX_ADJ=pix_adj_p, $
                 SIM_Y_OFFSET=sim_y_offset_passed, SIM_Z_OFFSET=sim_z_offset_passed, $
                 ACIS_Exposure_Time=acis_exposure_time, SIM_STREAK=sim_streak, $
                 
                 ra, dec, psf_fn, skypixel_per_psfpixel, footprint, psf_energy, desired_psf_counts, $
                 X_CAT=x_cat, Y_CAT=y_cat, OFF_ANGLE=off_angle, CHIP_ID=chip_id, EMAP_VAL=emap_val, $
                 
                 SAOSACFile=SAOSACFile, SpectrumFile=SpectrumFile, SpatialModelFile=SpatialModelFile, MARX_ASPECT_FN=marx_aspect_fn_p, MARX_ONLY=marx_only,$
                 RETAIN_EVENTLIST=retain_eventlist, BINSPEC=binspec
                  

; Store local variables in a COMMON block to retain information from initialization call.
COMMON ae_make_psf, marx_parameter_file, marxasp_parameter_file, obsdata_filename, event2wcs_astr, exposure, ccd_filter_spec, gratingtype, EFFICIENCY_TABLE, obs_aspect_fn, marx_aspect_fn, aspect_blur, pix_adj, SIM_X,SIM_Y,SIM_Z, s_aimpoint, DY_AVG,DZ_AVG,  marx_configuration_Iarray, marx_configuration_Sarray 

creator_string = "ae_make_psf, version " +strmid("$Rev:: 5658  $",7,5) +strmid("$Date: 2022-01-25 07:06:00 -0700 (Tue, 25 Jan 2022) $", 6, 11)


if keyword_set( obs_aspect_fn_p) then  obs_aspect_fn =  obs_aspect_fn_p
if keyword_set(marx_aspect_fn_p) then marx_aspect_fn = marx_aspect_fn_p

arcsec_per_ACISpixel = 0.492 
arcsec_per_skypixel  = 0.492 


; If not passed, create a scratch directory.
tempdir_supplied_by_caller = keyword_set(tempdir)
if ~tempdir_supplied_by_caller then begin
  temproot = temporary_directory( 'ae_make_psf', VERBOSE=1, SESSION_NAME=session_name)
  tempdir = temproot
endif
run_command, PARAM_DIR=tempdir, /QUIET, MARX_VERSION=marx_version

if keyword_set(obsdata_filename_p) then begin

    run_command, /QUIET, 'which marx', STATUS=status
    if keyword_set(status) then begin      
      print, 'ERROR: The MARX tools do not seem to be in your unix path!'
      retall
    endif
    
    print, creator_string, F='(%"\n%s, configuring MARX ...")'

    obsdata_filename = obsdata_filename_p

    obsdata_header = headfits(obsdata_filename, EXT=1, ERRMSG=error )
    if (keyword_set(error)) then begin
      print, error
      message, 'ERROR reading ' + obsdata_filename_p
    endif

    ; Save an astrometry structure for use later.
    fxbfind, obsdata_header, 'TTYPE', dum1, TTYPE, dum2, 'null'
    fxbfind, obsdata_header, 'TCTYP', dum1, TCTYP, dum2, 'null'
    fxbfind, obsdata_header, 'TCRVL', dum1, TCRVL, dum2, 0.0D
    fxbfind, obsdata_header, 'TCRPX', dum1, TCRPX, dum2, 0.0D
    fxbfind, obsdata_header, 'TCDLT', dum1, TCDLT, dum2, 0.0D
    colnames = strlowcase( strtrim(TTYPE,2) )
    x_ind    = where(strlowcase(colnames) EQ 'x')
    y_ind    = where(strlowcase(colnames) EQ 'y')
    make_astr, event2wcs_astr, DELTA=TCDLT[[x_ind,y_ind]], CTYPE=TCTYP[[x_ind,y_ind]], $
                               CRPIX=TCRPX[[x_ind,y_ind]], CRVAL=TCRVL[[x_ind,y_ind]]

                               
    exposure               = keyword_set(exposure_p)          ? exposure_p    : psb_xpar( obsdata_header,'EXPOSURE')
    
    
    ;; ------------------------------------------------------------------------
    ; Determine which CCDs are present in the event list provided, so that we can filter MARX simulations accordingly.
    ; We do not trust the DETNAM keyword because the observer may have applied a CCD_ID filter.
    ccd_is_in_observation = bytarr(10)
    for ccd_id = 0,9 do begin
      keyname    = string(ccd_id, F='(%"EXPOSUR%d")')
      this_exposure   = psb_xpar( obsdata_header, keyname, COUNT=count)
                     
      if (count EQ 0) || (this_exposure LE 0)  then continue
      ccd_is_in_observation[ccd_id] = 1B
    endfor ;ccd_id
    ccd_filter_spec = string( strjoin(strtrim(string(where(ccd_is_in_observation)),2),','), F='(%"ccd_id=%s")' )



    ;; ------------------------------------------------------------------------
    ;; Look up information required to configure the position of the virtual Science Instrument Module (SIM) in MARX.
    
    ;; The SIM_? keywords record the mechancial position of the SIM, commanded by the Chandra operators.
    SIM_X = float(psb_xpar( obsdata_header, 'SIM_X'))
    SIM_Y = float(psb_xpar( obsdata_header, 'SIM_Y'))
    SIM_Z = float(psb_xpar( obsdata_header, 'SIM_Z'))

    ;; The optical system also has uncommanded time-variable motion between the HRMA telescope and the SIM, due to thermal effects.  These motions are measured by the "FID lights" on Chandra, and are recorded in the time series columns (DY,DZ) in the observation's aspect file.
    ;; When MARX is given that aspect file it uses the (DY,DZ) values to adjust the position of its virtual SIM (I believe).
    ;; When MARX is NOT given the aspect file, we provide it with the D?_AVG keywords below, which record the average values of the (DY,DZ) columns.
    DY_AVG = float(psb_xpar( obsdata_header, 'DY_AVG')) ; Mean DY in observation's aspect file
    DZ_AVG = float(psb_xpar( obsdata_header, 'DZ_AVG')) ; Mean DZ in observation's aspect file

    ; Note that in early 2020 the CXC concluded that the motions measured by the FID lights have become large, and are
    ; now probably dominated by thermal motion between the HRMA and the aspect camera.
    ; They have thus decided to incorporate FID light offsets into the RA/DEC/ROLL columns of the aspect solution, and to
    ; assume zero motion between the HRMA and SIM (DY, DZ, DTHETA columns are now set to zero).
    ; In HelpDesk Ticket #022669, Kenny Glotfelty said the following:
    
      ; The DY, DZ, DTHETA in the asol files are computed by measuring the FID lights attached to the instruments and imaged on the aspect camera.  The assumption had been that changes in DY,DZ,DTHETA are changes in SIM with respect to HRMA.  As the values have become large, we have now concluded that the values are actually due to offsets between the HRMA and the aspect camera.  

      ; So, in DS10.8.3 -- the DY, DZ, DTHETA values are now folded back into the actual RA, DEC, ROLL values  (and the quaternions, q_att). The DY,  DZ, DTHETA values are now identically 0, as are the average values, DY|DZ|DTH_AVG. The change also changes the mean pointing, ie the  RA|DEC|ROLL_PNT keywords, and by convention, we set the tanget point to the same location as the mean pointing, therefore the RA|DEC|ROLL_NOM are also changed (along w/ the WCS keywords (CRVAL's & table equivalents).

    ; Note that in DS10.8.3 and later data products, the header keywords DY_AVG/DZ_AVG are zero.



   ; Determine aimpoint
    s_aimpoint  = (SIM_Z GT -210) 
    gratingtype = strtrim(psb_xpar( obsdata_header,'GRATING'),2)
    
    if ((strmatch(gratingtype,'HETG') || strmatch(gratingtype,'LETG')) && ~s_aimpoint) then message, 'ERROR: BUG in determining aimpoint!'
    
    ; Move the SIM if requested by the caller (e.g. when simulating pile-up).
    if keyword_set(sim_y_offset_passed) then SIM_Y += sim_y_offset_passed
    if keyword_set(sim_z_offset_passed) then SIM_Z += sim_z_offset_passed
    
    
    ; Look up the nominal SIM position MARX uses for each aimpoint.
    mc_type = {mc_type, DetectorType:'', nominal_SIM_X:0.0, nominal_SIM_Y:0.0, nominal_SIM_Z:0.0}
    
    DetectorType='ACIS-I'
    run_command, 'detinfo '+DetectorType+' | grep STF-STT', line, /QUIET
    reads, (stregex(line,'.*\((.*)\)',/sub,/ext))[1]  , nominal_SIM_X, nominal_SIM_Y, nominal_SIM_Z
    marx_configuration_Iarray = {mc_type, DetectorType, nominal_SIM_X, nominal_SIM_Y, nominal_SIM_Z}

    DetectorType='ACIS-S'
    run_command, 'detinfo '+DetectorType+' | grep STF-STT', line, /QUIET
    reads, (stregex(line,'.*\((.*)\)',/sub,/ext))[1] , nominal_SIM_X, nominal_SIM_Y, nominal_SIM_Z
    marx_configuration_Sarray ={mc_type, DetectorType, nominal_SIM_X, nominal_SIM_Y, nominal_SIM_Z}
    
    print, SIM_X,SIM_Y,SIM_Z, F='(%"SIM position during observation was (%0.4f, %0.4f, %0.4f) mm.")'


    ;; We can NOT use DitherModel=NONE because that puts the source at the wrong sky coordinates (bug in MARX).
    ;;
    ;; To get correct falloff of the PSF at the chip edges we must have MARX dither with the observation's aspect file.
    ;; The important effect seems to be non-zero values of dy and dz in the aspect file (fid light motion?).
    ;;
    ;;
    ;; For the purposes of image reconstruction, it would be good to let MARX include the ACIS readout streak, since
    ;; a real point source has that feature.  However, for the purpose of aperture correction we must NOT simulate the
    ;; streak because the mission has already accounted for the streak via the DTCOR mechanism.
    ;; NOTE that the choice of simulating the streak or not has an effect on the energy dependence of the 
    ;; aperture correction, via the CROPFRAC calculation!  It is not obvious what assumptions about the streak are
    ;; built into the HMRA and ACIS QE calibrations!

    ; Look in several places for marx.par, which has moved between MARX versions
    marx_data_dir = getenv('MARX_DATA_DIR')
    if (marx_data_dir EQ '') then begin
      print, 'ERROR: MARX is not configured ($MARX_DATA_DIR is not defined).'
      retall                          
    endif
    
       marx_parameter_file = tempdir + 'marx.par'
    marxasp_parameter_file = tempdir + 'marxasp.par'

    parfile_template       = marx_data_dir + ['/../par/marx.par','/../../share/marx/pfiles/marx.par','/../pfiles/marx.par','/../../marx.par']
    for ii = 0, n_elements(parfile_template)-1 do begin
      if file_test(parfile_template[ii]) then begin
        file_copy, /OVERWRITE,              parfile_template[ii]                ,    marx_parameter_file
        file_copy, /OVERWRITE, file_dirname(parfile_template[ii])+'/marxasp.par', marxasp_parameter_file
        break
      endif
    endfor                                                      
    if (ii EQ n_elements(parfile_template)) then begin
      print, 'ERROR: cannot find marx.par in '+marx_data_dir
      retall
    endif
  
    marx_cmd   = strarr(20)
    kk=0
    marx_cmd[kk++] = 'pset marx Verbose=yes'
    if keyword_set(SAOSACFile) then begin
      ;; ------------------------------------------------------------------------
      ; This is the case used by the ae_chart_interface tool.
      ;; ------------------------------------------------------------------------
      ; MARX will use its internal dither model and we will return the aspect timeseries to the caller in the file specified by MARX_ASPECT_FN.
      marx_cmd[kk++] = 'pset marx DitherModel=INTERNAL'
      
      marx_cmd[kk++] = 'pset marx SourceType=SAOSAC'
      
      
    endif else if keyword_set(SpectrumFile) then begin
      ;; ------------------------------------------------------------------------
      ; This is the case used by recon_spectrum.pro.
      ;; ------------------------------------------------------------------------
      ; MARX will use its internal dither model and we will return the aspect timeseries to the caller in the file specified by MARX_ASPECT_FN.
      marx_cmd[kk++] = 'pset marx DitherModel=INTERNAL'
      
      if keyword_set(SpatialModelFile) then begin
        marx_cmd[kk++] = 'pset marx SourceType=IMAGE'
        marx_cmd[kk++] = 'pset marx S-ImageFile='+SpatialModelFile
      endif else begin
        marx_cmd[kk++] = 'pset marx SourceType=POINT'
        
      endelse
      marx_cmd[kk++] = 'pset marx SpectrumType=FILE'
    endif else begin
      ;; ------------------------------------------------------------------------
      ; This is case used by AE to construct PSF images.
      ;; ------------------------------------------------------------------------
      ; Dither is supplied to MARX by the observation's aspect file.
      if ~keyword_set(obs_aspect_fn) || ~file_test(obs_aspect_fn) then message, 'ERROR: must supply an aspect file via OBS_ASPECT_FN.'

      ; Check for inconsistent header keywords in aspect file; see CXC Helpdesk ticket from 2015 April 30 and email to Moritz Guenther on 13 January 2016.
      aspect_header = headfits(obs_aspect_fn, EXT=1, ERRMSG=error )
      if (keyword_set(error)) then begin
        print, error
        message, 'ERROR reading ' + obs_aspect_fn
      endif

      gcirc, 2, psb_xpar( obsdata_header,'RA_NOM'),psb_xpar( obsdata_header,'DEC_NOM'), $
                psb_xpar(  aspect_header,'RA_NOM'),psb_xpar(  aspect_header,'DEC_NOM'), offset_arcsec
      if (offset_arcsec GT 0.01) then begin
        print, obs_aspect_fn, obsdata_filename, F='(%"\nae_make_psf: ERROR: You must change the RA_NOM/DEC_NOM/ROLL_NOM keywords in the aspect file (%s) to match those in the event file (%s), so that MARX will build PSF images using the same SKY coordinate system as the event list.")'
        retall
      endif
      
      
      marx_cmd[kk++] = 'pset marx DitherModel=FILE'
      marx_cmd[kk++] = 'pset marx DitherFile='+obs_aspect_fn
      if keyword_set(SpatialModelFile) then begin
        marx_cmd[kk++] = 'pset marx SourceType=IMAGE'
        marx_cmd[kk++] = 'pset marx S-ImageFile='+SpatialModelFile
      endif else begin
        marx_cmd[kk++] = 'pset marx SourceType=POINT'
        
      endelse
      marx_cmd[kk++] = 'pset marx SpectrumType=FLAT'
    endelse
    
    if (psb_xpar(obsdata_header,  'RA_NOM') EQ 0) || $
       (psb_xpar(obsdata_header, 'DEC_NOM') EQ 0) || $
       (psb_xpar(obsdata_header,'ROLL_NOM') EQ 0) then begin
     
      print, obsdata_filename, F='(%"ae_make_psf: ERROR: in %s RA_NOM/DEC_NOM/ROLL_NOM are either missing or are zero.")'
      retall
    endif
    
    marx_cmd[kk++] = 'pset marx TStart='   +string(psb_xpar( obsdata_header,  'TSTART') , F='(%"%0.1f")')
    marx_cmd[kk++] = 'pset marx RA_Nom='   +string(psb_xpar( obsdata_header,  'RA_NOM') , F='(%"%0.6f")')
    marx_cmd[kk++] = 'pset marx Dec_Nom='  +string(psb_xpar( obsdata_header, 'DEC_NOM') , F='(%"%0.6f")')
    marx_cmd[kk++] = 'pset marx Roll_Nom=' +string(psb_xpar( obsdata_header,'ROLL_NOM') , F='(%"%0.6f")')
    marx_cmd[kk++] = 'pset marx GratingType=' +gratingtype
    
    ; It seems that setting ExposureTime=0 for generating PSFs (with DitherFile supplied) does no harm.
    ; With small values of SourceFlux, MARX runs as long as the aspect file lasts.
    ; However, with large values of  SourceFlux, the MARX simulation is SHORTER than the aspect file, and
    ; you don't get the number of events expected!  Very odd.
    ; So, we set ExposureTime to the ObsID's EXPOSURE value.
    marx_cmd[kk++] = 'pset marx ExposureTime=' +string(exposure , F='(%"%d")')
    marx_cmd[kk++] = 'pset marx DetIdeal=yes'
    marx_cmd[kk++] = 'pset marx DetExtendFlag=no'
    
    if (n_elements(acis_exposure_time) NE 0) then $
      ; MARX will not allow ACIS_Exposure_Time=0, so put a floor on what was requested.
      marx_cmd[kk++] = 'pset marx ACIS_Exposure_Time='+string(1E-10 > acis_exposure_time, F='(%"%0.4g")')
    
    ; Are we simulating the streak?
    if ~keyword_set(sim_streak) then $
      marx_cmd[kk++] = 'pset marx ACIS_Frame_Transfer_Time=0'
    
    run_command, marx_cmd[ where(logical_true(marx_cmd)) ]
                      
    ; These emap_calibration values assume the map was made at 1 keV, and the source is on-axis.
    ; These values are used to calculate a MARX input 'flux' that will produce the requested number of counts in the PSF image.
    EFFICIENCY_TABLE = {EFFICIENCY_TABLE, energy:0.0, emap_calibration:0.0}
    EFFICIENCY_TABLE = [{EFFICIENCY_TABLE,0.277  , 2.02},$
                        {EFFICIENCY_TABLE,0.5    , 1.96},$ 
                        {EFFICIENCY_TABLE,1.0    , 2.01},$ 
                        {EFFICIENCY_TABLE,1.4967 , 2.02},$ 
                        {EFFICIENCY_TABLE,1.9    , 1.92},$ 
                        {EFFICIENCY_TABLE,2.1    , 1.06},$ 
                        {EFFICIENCY_TABLE,2.5    , 1.12},$ 
                        {EFFICIENCY_TABLE,4.51   , 1.10},$ 
                        {EFFICIENCY_TABLE,6.40   , 0.63},$ 
                        {EFFICIENCY_TABLE,8.60   , 0.17}]
;   forprint, EFFICIENCY_TABLE.energy, EFFICIENCY_TABLE.emap_calibration    




    ; Look up the PIX_ADJ keyword, which acis_process_events wrote to the event file; fail if not found.
    pix_adj = strtrim(psb_xpar( obsdata_header, 'PIX_ADJ', COUNT=count),2)
    if (count EQ 0) then begin
      print, strtrim(psb_xpar( obsdata_header,'OBS_ID'),2), F='(%"\nERROR! Keyword PIX_ADJ is missing from ObsID %s.\nYou **should** suspend your AE work until you have generated a modern event file using acis_process_events!\nType exit or .continue if you must ...\n")'
      stop
    endif
    
    if (pix_adj NE 'EDSER') then begin
      print, strtrim(psb_xpar( obsdata_header,'OBS_ID'),2), pix_adj, F='(%"\nERROR! Event positions in ObsID %s were  computed with the %s algorithm.  AE requires event positions from the EDSER algorithm to produce well-calibrated PSF images for your data.\n")'
      retall
    endif

;    case pix_adj of
;      'EDSER'    : pix_adj = pix_adj
;      'CENTROID' : begin
;                   print, F='(%"\nWARNING! acis_process_events used a position algorithm (CENTROID) not supported by marx2fits; using EDSER instead.\n")'
;                   pix_adj = 'EDSER'
;                   end
;      'RANDOMIZE': pix_adj = pix_adj
;      'NONE'     : pix_adj = pix_adj
;      else: begin
;            print, obsdata_filename_p, pix_adj, F='(%"\nWARNING! acis_process_events keyword PIX_ADJ in %s has unexpected value (%s); using EDSER in call to marx2fits tool!\n")'
;            pix_adj = 'EDSER'
;            end
;    endcase

    ; Look up default value for AspectBlur.
    run_command, 'pget -abort marx AspectBlur', result, /QUIET
    aspect_blur = float(result[0])
    
    
;2016 May 20: PIX_ADJ and ASPECT_BLUR are currently hard-coded to reflect Leisa's calibration of MARX to match some observed sources near the I and S aimpoints!!!!!!!!!!!!
    if s_aimpoint then begin
      aspect_blur = 0.07 ; arcsec
      pix_adj = 'EDSER'
    endif else begin
      aspect_blur = 0.07 ; arcsec
      pix_adj = 'NONE'
    endelse
    print, s_aimpoint ? 'ACIS-S' : 'ACIS-I', pix_adj, aspect_blur, F='(%"\nFor aimpoint %s AE has selected PIX_ADJ=%s and ASPECT_BLUR=%0.3f.\n")'

    
    
    if keyword_set(pix_adj_p) && (pix_adj_p NE pix_adj) then begin
      print, pix_adj_p, pix_adj, F='(%"\nWARNING! The marx2fits tool will be called with the requested PIX_ADJ value (%s), which differs from the value (%s) that AE recommends!\n")'
      pix_adj = pix_adj_p
    endif

    if (n_elements(aspect_blur_p) GT 0) && ~almost_equal(aspect_blur_p, aspect_blur) then begin
      print, aspect_blur_p, aspect_blur, F='(%"\nWARNING! The PSF will be built with the requested ASPECT_BLUR value (%0.3f), which differs from the value (%0.3f) that AE recommends!\n")'
      aspect_blur = aspect_blur_p 
    endif 


    return
endif
                                                                     

;; ==================================================================================
;; Perform MARX simulations to build PSF images.
temp_events_fn       = tempdir + 'temp.evt'
temp_image_fn        = tempdir + 'temp.img'
marxdir              = tempdir 
marx_events_filename = tempdir + 'marx.evt' 

if ((n_elements(x_cat) EQ 0) || (n_elements(y_cat) EQ 0) || (n_elements(off_angle) EQ 0) || (n_elements(chip_id) EQ 0)) then begin
  ;; Convert RA,DEC to the (x,y) system of this obsid.

  ;; Calculate off-axis angle (off_angle), used to calculate RADIUS50 header keyword in PSF.

  ;; Determine which CCD we are on/near (chip_id).
  ; chip_id's only use here is to set the MARX parameter DetectorType.
  ; The effect of DetectorType inside MARX is not clear.  It probably determines the default SIM position,
  ; however ae_make_psf explicitly controls the SIM position.  Although MARX may have CCD-specific parameters
  ; for some of its optional calculations, here MARX is only generating photon rays.
  ; Thus, I would wager that the CHIP_ID input does not alter the PSF generated.

  dmcoords_cmd = string(obsdata_filename, file_test(obs_aspect_fn) ? obs_aspect_fn : 'none', ra, dec, $
               F="(%'dmcoords %s asolfile=%s opt=cel celfmt=deg ra=%10.6f dec=%10.6f')")
  run_command, dmcoords_cmd
  
  run_command, /QUIET, 'pget dmcoords x y theta chip_id', dmcoords_result
  ; Parse the string returned by pget with the ON_IOERROR mechanism enabled in order to find type conversion errors.
  ON_IOERROR, TYPE_CONVERSION_ERROR1
  x_cat        = float(dmcoords_result[0])
  y_cat        = float(dmcoords_result[1])
  off_angle    = float(dmcoords_result[2])  ; arcmin
  chip_id      = fix  (dmcoords_result[3])
  if (0) then begin
    TYPE_CONVERSION_ERROR1:
    print, !ERROR_STATE.MSG
    print, 'ERROR: dmcoords results could not be parsed.'
    forprint, ['  The dmcoords call was : ', '    '+dmcoords_cmd   ]
    forprint, ['  The output of pget was: ', '    '+dmcoords_result]
    GOTO, CLEANUP
  endif 
  ON_IOERROR, NULL
endif

if keyword_set(exposure_p) then exposure = exposure_p 

if ~keyword_set(emap_val) then emap_val = exposure * 361.0

;; In Dec 2007 I used MARX simulations at 1.5 keV with the readout streak disabled 
;; to measure PSF fractions at 1.5 keV as a function of off-axis angle.  
;; These polynomial curves were fit to those measurements.
;; The off-axis angle off_angle is in arcminutes.
radius50 = (0.85 -0.25*off_angle + 0.10*off_angle^2) * arcsec_per_skypixel  ; arcsec


num_energies  = 1 > n_elements(psf_energy)
crop_fraction = fltarr(num_energies)
num_counts    = fltarr(num_energies)

; MARX has to be configured DetectorType=ACIS-I for CCDs 0,1,2,3 or DetectorType=ACIS-S for CCDs 4...9.
; The effect of DetectorType inside MARX is not clear.  It probably determines the default SIM position,
; however our MARX calls here explicitly control the SIM position.  Although MARX may have CCD-specific parameters
; for some of its optional calculations, here MARX is only generating photon rays.
; Thus, I would wager that the CHIP_ID input does not alter the PSF generated.

marx_configuration = (chip_id LE 3) ? marx_configuration_Iarray : marx_configuration_Sarray

; The observation's Science Instrument Module position (SIM_X,SIM_Y,SIM_Z) is specified to MARX as an offset from its nominal position (DetOffsetX,DetOffsetY,DetOffsetZ).
  
for jj=0, num_energies-1 do begin     
  if keyword_set(SpectrumFile) then begin
   ;; ------------------------------------------------------------------------
   ;; This is the case used by recon_spectrum.pro.
   ;; ------------------------------------------------------------------------
    ;
    ; Experiments demonstrate that the AspectBlur parameter adds noise to the MARX photon rays (e.g. to xpixel.dat) and NO noise is added to the aspect file produced by marxasp.  
    ; For pile-up reconstruction, we want rays WITHOUT blur (AspectBlur=0) to model just the HRMA PSF.
    cmd1 = string(marx_parameter_file, marxdir, ra, dec, $
                          marx_configuration.DetectorType, $
                  SIM_X - marx_configuration.nominal_SIM_X, $
                  SIM_Y - marx_configuration.nominal_SIM_Y + DY_AVG, $
                  SIM_Z - marx_configuration.nominal_SIM_Z + DZ_AVG, $
                  SpectrumFile[jj], exposure, marxdir, $
            F='(%"marx @@%s OutputDir=%s SourceRA=%0.6f SourceDEC=%0.6f DetectorType=%s DetOffsetX=%0.5f DetOffsetY=%0.5f DetOffsetZ=%0.5f  SpectrumFile=%s SourceFlux=-1 AspectBlur=0 ExposureTime=%d >! %s/marx.log")' )
    
    ; MARX used its internal dither model, so we return the aspect timeseries to the caller in the file specified by MARX_ASPECT_FN.
    cmd2 = string( marxasp_parameter_file, marxdir, marx_aspect_fn, F='(%"marxasp @@%s MarxDir=%s OutputFile=%s ")')
    
    run_command, [cmd1,cmd2]
    
    
    ; In MARX 5.0, marxasp can no longer add noise to simulate aspect reconstruction errors, so we do that ourselves in the SKY system.
    ; REMEMBER THAT THE xy2ad and ad2xy programs assume that (x,y) are  ZERO-BASED pixel indexes.  
    ; We would normally subtract 1 from the sky (x,y) positions before calling xy2ad, and add 1 to the sky positions after calling ad2xy.
    ; In the ad2xy/xy2ad pair of calls those decrements and increments cancel out, and are not shown.
    pheader   = headfits(marx_aspect_fn)
    bt        =  mrdfits(marx_aspect_fn, 1, theader, /SILENT, STATUS=status)
    if (keyword_set(error)) then message, 'ERROR reading ' + marx_aspect_fn
    
    ad2xy, bt.ra, bt.dec, event2wcs_astr, aspect_x, aspect_y 
    aspect_x += (aspect_blur / arcsec_per_skypixel) * random(n_elements(bt), /NORMAL)
    aspect_y += (aspect_blur / arcsec_per_skypixel) * random(n_elements(bt), /NORMAL)
    xy2ad, aspect_x, aspect_y, event2wcs_astr, aspect_ra, aspect_dec
    bt.ra  = aspect_ra
    bt.dec = aspect_dec
    bt.DY  = DY_AVG ; We must record in the simulated aspect file the (DY_AVG,DZ_AVG) focal plane offset that was given
    bt.DZ  = DZ_AVG ; to MARX above, so that the reproject_events/acis_process_events tools can compute accurate event coordinates.
    
    psb_xaddpar, pheader, 'CREATOR', creator_string
    writefits,   marx_aspect_fn, 0, pheader
    mwrfits, bt, marx_aspect_fn,    theader
    print, aspect_blur, DY_AVG, DZ_AVG, marx_aspect_fn, F='(%"Added to aspect timeseries: %0.3f arcsec noise, DY=%0.2f, DZ=%0.2f (%s)")'
    
    
  endif else if keyword_set(SAOSACFile) then begin
   ;; ------------------------------------------------------------------------
   ;; This is the case used by the ae_chart_interface tool to construct PSF images
   ;; MARX applies aspect reconstruction blur to the rays.
   ;; ------------------------------------------------------------------------
    cmd1 = string(marx_parameter_file, marxdir, ra, dec, $
                          marx_configuration.DetectorType, $
                  SIM_X - marx_configuration.nominal_SIM_X, $
                  SIM_Y - marx_configuration.nominal_SIM_Y + DY_AVG, $
                  SIM_Z - marx_configuration.nominal_SIM_Z + DZ_AVG, $
                  SAOSACFile[jj], aspect_blur, marxdir, $
            F='(%"marx @@%s OutputDir=%s SourceRA=%0.6f SourceDEC=%0.6f DetectorType=%s DetOffsetX=%0.5f DetOffsetY=%0.5f DetOffsetZ=%0.5f SAOSAC_Color_Rays=no SAOSACFile=%s  AspectBlur=%0.3f >! %s/marx.log")' )
          
    run_command, cmd1
; As of Oct 2013, neither ChaRT nor MARX dithers the rays, and the resulting PSF is full of spatial quantization artifacts if the EDSER position algorithm is used.
; See Chandra helpdesk ticket #15403.
    if arg_present(marx_aspect_fn) then begin
      ; MARX used its internal dither model, so we return the aspect timeseries to the caller in the file specified by MARX_ASPECT_FN.
      cmd2 = string( marxasp_parameter_file, marxdir, marx_aspect_fn, F='(%"marxasp @@%s MarxDir=%s OutputFile=%s ")')
      
      run_command, cmd2
      
      ; In MARX 5.0, marxasp can no longer add noise to simulate aspect reconstruction errors, so we do that ourselves in the SKY system.
      ; REMEMBER THAT THE xy2ad and ad2xy programs assume that (x,y) are  ZERO-BASED pixel indexes.  
      ; We would normally subtract 1 from the sky (x,y) positions before calling xy2ad, and add 1 to the sky positions after calling ad2xy.
      ; In the ad2xy/xy2ad pair of calls those decrements and increments cancel out, and are not shown.
      pheader   = headfits(marx_aspect_fn)
      bt        =  mrdfits(marx_aspect_fn, 1, theader, /SILENT, STATUS=status)
      if (keyword_set(error)) then message, 'ERROR reading ' + marx_aspect_fn
      
      ad2xy, bt.ra, bt.dec, event2wcs_astr, aspect_x, aspect_y 
      aspect_x += (aspect_blur / arcsec_per_skypixel) * random(n_elements(bt), /NORMAL)
      aspect_y += (aspect_blur / arcsec_per_skypixel) * random(n_elements(bt), /NORMAL)
      xy2ad, aspect_x, aspect_y, event2wcs_astr, aspect_ra, aspect_dec
      bt.ra  = aspect_ra
      bt.dec = aspect_dec
      bt.DY  = DY_AVG ;We must record in the simulated aspect file the (DY_AVG,DZ_AVG) focal plane offset we put into MARX.
      bt.DZ  = DZ_AVG ; 
      
      psb_xaddpar, pheader, 'CREATOR', creator_string
      writefits,   marx_aspect_fn, 0, pheader
      mwrfits, bt, marx_aspect_fn,    theader
      print, aspect_blur, DY_AVG, DZ_AVG, marx_aspect_fn, F='(%"Added to aspect timeseries: %0.3f arcsec noise, DY=%0.2f, DZ=%0.2f (%s)")'

    endif ; arg_present(marx_aspect_fn)
    
    
  endif else begin
   ;; ------------------------------------------------------------------------
   ;; This is the case used by AE's CONSTRUCT_REGIONS stage to construct PSF images; the observation's aspect file is supplied to MARX.
   ;; ------------------------------------------------------------------------
    ; MARX applies aspect reconstruction blur to the rays.     
    cmd1 = string(marx_parameter_file, marxdir, ra, dec, $
                          marx_configuration.DetectorType, $
                  SIM_X - marx_configuration.nominal_SIM_X, $
                  SIM_Y - marx_configuration.nominal_SIM_Y, $
                  SIM_Z - marx_configuration.nominal_SIM_Z, $
                  psf_energy[jj], psf_energy[jj], aspect_blur, marxdir, $
            F='(%"marx @@%s OutputDir=%s SourceRA=%0.6f SourceDEC=%0.6f DetectorType=%s DetOffsetX=%0.5f DetOffsetY=%0.5f DetOffsetZ=%0.5f MinEnergy=%0.4f MaxEnergy=%0.4f SourceFlux=%%0.4g NumRays=%%d AspectBlur=%0.3f >! %s/marx.log")' )
    
    ; We must employ our own little calibration table to estimate the emap value at the requested energy, because MARX requires an input flux, not a desired number of counts detected.
    linterp, EFFICIENCY_TABLE.energy, EFFICIENCY_TABLE.emap_calibration, psf_energy[jj], emap_calibration
    
    this_emap = emap_val * emap_calibration 
  
    
    ; Unfortunately, this attempt to predict the MARX 'flux' that will produce the counts we want does not actually work very well, for reasons unknown.  Thus, we resort to source-by-source calibration of the 'flux' computation.  We make a preliminary call to MARX simply to measure the flux/counts ratio, and we apply that 'tweak' to all the marx runs for this source.
    if (jj EQ 0) then begin
      flux    = 1E3 / this_emap ; try for 1E3 counts
      NumRays = 1E6             ; Let MARX allocate space for 1E6 counts
      run_command, string(flux,NumRays, F='(%"'+cmd1+'")'), /QUIET

      if (strmatch(gratingtype,'HETG') || strmatch(gratingtype,'LETG')) then begin
        run_command, string(marxdir,  F='(%"marx --dump %s/order.dat    | grep -c 0")'), result, /QUIET
      endif else begin
        run_command, string(marxdir,  F='(%"marx --dump %s/detector.dat | wc -l")')    , result, /QUIET
      endelse

      flux_tweak = 1E3 / long(result[0])
      print, this_emap/1e6, flux_tweak, F='(%"emap=%0.1f  tweak=%0.1f")'
    endif
                                                                         
    flux    = flux_tweak * (desired_psf_counts[jj] / this_emap)
    NumRays =       10L  *  desired_psf_counts[jj] ; MARX data structures must be large to hold desired_psf_counts
    
    run_command, string(flux,NumRays, F='(%"'+cmd1+'")')
  endelse

  ; For the calling style used by the recon_spectrum tool, we do not need the code below, which generates an image of the PSF. 
  ; Also, we are simulating the PSF of one spectrum, not a series of num_energies mono-energy PSFs.
  ; Thus, we can simply return to the caller here.
  if keyword_set(marx_only) then return
  
  
  
  ; Convert MARX rays into events.
  cmd2 = string(pix_adj, marxdir, temp_events_fn, marxdir, F='(%"marx2fits --pixadj=%s %s %s >>! %s/marx.log")' )
  run_command, cmd2
  
  ; Retain simulated events only for the set of CCDs present in the observation.
  
  
  cmd = string(temp_events_fn, ccd_filter_spec, marx_events_filename, F="(%'dmcopy ""%s[%s]"" %s clobber+')")
  run_command, cmd

  ; Devise a normalization for the cropped PSF image by counting the total events detected (in 0th order when gratings are used).
  if (strmatch(gratingtype,'HETG') || strmatch(gratingtype,'LETG')) then begin
    bt = mrdfits(marx_events_filename, 1, sim_header)
    num_counts[jj] = total(/INT, bt.order EQ 0)
  endif else begin
    sim_header = headfits(marx_events_filename, EXT=1, ERRMSG=error )
    if (keyword_set(error)) then begin
      print, error
      message, 'ERROR reading ' + marx_events_filename
    endif
    
    num_counts[jj] = psb_xpar( sim_header, 'NAXIS2')
  endelse

  
  ;; ------------------------------------------------------------------------
  ;; Bin up the simulated events into an image.
  
  if ~keyword_set(binspec) then begin
    ; Use the syntax to "xmin:xmax:delx" syntax to ensure that the image has square pixels to keep CIAO happy.
    ; Place the source position at the center of the central pixel in the PSF image.
    psf_half_dim = ceil(0.5*footprint/skypixel_per_psfpixel)
    
    xmin = x_cat - skypixel_per_psfpixel * (  psf_half_dim + 0.5)
    ymin = y_cat - skypixel_per_psfpixel * (  psf_half_dim + 0.5)
    xmax = xmin  + skypixel_per_psfpixel * (2*psf_half_dim + 0.999)
    ymax = ymin  + skypixel_per_psfpixel * (2*psf_half_dim + 0.999)
    
    binspec = string(xmin,xmax,skypixel_per_psfpixel,ymin,ymax,skypixel_per_psfpixel, F='(%"x=%0.4f:%0.4f:%0.6f,y=%0.4f:%0.4f:%0.6f")')
  endif ; ~keyword_set(binspec)
  
  cmd = string(marx_events_filename, binspec, temp_image_fn, F="(%'dmcopy ""%s[bin %s][opt type=i4,mem=200]"" %s clobber+')")
  run_command, cmd

  ;; Read the PSF image and the keywords specifying the transformation between pixel indexes and physical coordinates (x,y).
  ;; I do not recall why we change the PSF from type I4 to type FLOAT .... 
  psf_img = float(readfits(temp_image_fn, psf_header, /SILENT))

  extast, psf_header, psf2wcs_astr 
  arcsec_per_psfpixel = psf2wcs_astr.CDELT[1] * 3600
  
  ; We cannot use xy2ad.pro/ad2xy.pro for conversions between array index and PHYSICAL (sky) coordinate systems.
  crvalP = [psb_xpar( psf_header, 'CRVAL1P'), psb_xpar( psf_header, 'CRVAL2P')]
  crpixP = [psb_xpar( psf_header, 'CRPIX1P'), psb_xpar( psf_header, 'CRPIX2P')]
  cdeltP = [psb_xpar( psf_header, 'CDELT1P'), psb_xpar( psf_header, 'CDELT2P')]
  
  
  ; If MARX has defined a SKY coordinate system that differs from that used in the event list, then all sorts of bad things will happen!
  ; So, let's defensively verify that an arbitrary coordinate in the SKY system (4096.5,4096.5) is converted to the same RA/DEC coordinate by both the event list's header and by the PSF image's header.
  ;; REMEMBER THAT THE xy2ad and ad2xy programs assume that (x,y) are 
  ;; ZERO-BASED pixel indexes.  Thus we must subtract 1 from the sky (x,y) 
  ;; positions when converting to RA,DEC.
  xy2ad,               4096.5d - 1                        ,               4096.5d - 1                        , event2wcs_astr, ra1,dec1
  xy2ad, (crpixP[0] + (4096.5d - crvalP[0])/cdeltP[0]) - 1, (crpixP[1] + (4096.5d - crvalP[1])/cdeltP[1]) - 1, psf2wcs_astr  , ra2,dec2

  gcirc, 2, ra1,dec1, ra2,dec2, offset_arcsec
  if (offset_arcsec GT 0.01) then begin
    print, temp_image_fn, obsdata_filename, offset_arcsec, F='(%"\nae_make_psf: ERROR: The SKY coordinate systems defined in the PSF image (%s) and in the event list (%s) are offset by %0.2f arcsec!  Something is very wrong!")' 
    retall
  endif

  
  ;; ------------------------------------------------------------------------
  ; Compute the crop fraction on the raw data, before smoothing.
  crop_fraction[jj] = (num_counts[jj] - total(/INTEGER, psf_img)) / num_counts[jj]


  ;; ------------------------------------------------------------------------
  ;; Sum up all the light in the PSF within the aperture used for the HRMA calibration.
  ;; If possible, correct for light lost by cropping.
  psf_total = total(psf_img, /DOUBLE) / (1-crop_fraction[jj])

  
  get_date, date_today, /TIMETAG
  psb_xaddpar, psf_header, 'POSNDATE', date_today, 'UTC date RA,DEC were changed'
  psb_xaddpar, psf_header, 'RA',       ra,  '[deg] PSF position on the sky', F='(F10.6)'
  psb_xaddpar, psf_header, 'DEC',      dec, '[deg] PSF position on the sky', F='(F10.6)'
  psb_xaddpar, psf_header, 'X_CAT',    x_cat, '[skypixel] source position, from catalog'
  psb_xaddpar, psf_header, 'Y_CAT',    y_cat, '[skypixel] source position, from catalog'
  psb_xaddpar, psf_header, 'ENERGY',   psf_energy   [jj], '[keV] mono-energy of the PSF'
  psb_xaddpar, psf_header, 'SUMRCTS',  num_counts   [jj], 'number of counts simulated'
  psb_xaddpar, psf_header, 'CROPFRAC', crop_fraction[jj], 'fraction of the PSF cropped'
  psb_xaddpar, psf_header, 'PSF_TOTL', psf_total, 'normalization of this image'
  psb_xaddpar, psf_header, 'RADIUS50', radius50, '[arcsec] radius enclosing ~50% PSF, estimated'
  psb_xaddpar, psf_header, 'AIMPOINT', s_aimpoint ? 'ACIS-S' : 'ACIS-I', 'aimpoint of observation'
  psb_xaddpar, psf_header, 'GRATING',  gratingtype, 'Grating'
  psb_xaddpar, psf_header, 'ASP_BLUR', aspect_blur, '[arcsec] AspectBlur used by marx'
  psb_xaddpar, psf_header, 'PIX_ADJ' , pix_adj    , 'position algorithm used by marx2fits'
  psb_xaddpar, psf_header, 'MARXVER' , marx_version, 'MARX version'
  
  ;; ------------------------------------------------------------------------
  ;; Save to file.
  hduname = string(1+jj, F="(%'PSF%d')")
  psb_xaddpar, psf_header, 'HDUNAME', hduname
  psb_xaddpar, psf_header, 'CREATOR', creator_string
  
  if (jj EQ 0) then begin
    ; Remove any existing psf_fn before trying to write to it, since it might be a symlink!
    if keyword_set(psf_fn) then file_delete, psf_fn, /ALLOW_NONEXISTENT
    writefits, psf_fn, psf_img, psf_header 
  endif else begin
    sxdelpar, psf_header, ['SIMPLE','EXTEND']
    psb_xaddpar, psf_header, 'XTENSION', 'IMAGE   ', BEFORE='BITPIX'
    psb_xaddpar, psf_header, 'EXTNAME', hduname
    mwrfits, psf_img, psf_fn, psf_header
  endelse
endfor ;jj

print, F='(%"\n  energy  requested,simulated (ct x1000)  crop_fraction")'
forprint, psf_energy, round(desired_psf_counts/1000.), round(num_counts/1000.), crop_fraction, F='(%"  %6.3f  %13d,%4d       %0.3f")'

; Save a copy of final marx.par (written by MARX) and marx.log to the same directory holding the PSF images.
file_copy, /OVERWRITE, marxdir+'/marx.par', file_dirname(psf_fn)
file_copy, /OVERWRITE, marxdir+'/marx.log', file_dirname(psf_fn)

; Retain MARX events, if requested.
if keyword_set(retain_eventlist) then begin
  file_copy, /OVERWRITE, marx_events_filename, file_dirname(psf_fn)
endif

CLEANUP:
; Clean up scratch directory if not managed by caller.
if ~tempdir_supplied_by_caller && file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif
return
end  ;  ae_make_psf



; =============================================================================
;; Photometry Computation

;; This routine hides the confusing 2-D data structure that we used in the MERGE stage to store extraction information
;; so that we can easily sum SRC_CNTS, BKG_CNTS, NET_CNTS across an energy band or across obsids.
;; This code is awkwardly structured, making it difficult and perhaps dangerous to re-use outside the context of AE's MERGE stage!

;; The table of extraction quantities should be loaded FIRST, e.g.
;;   ae_photometry, OBS_DATA=obs_data, SRC_CHANNELS=src_channels
;; followed by loading the response files, e.g.
;;   ae_photometry, RMF_FN=merged_rmf_fn, ARF_FN=merged_arf_fn 
;; Then, you can compute photometry values, e.g.
;;         ae_photometry, eband_lo[jj], eband_hi[jj], photometry
;; And retrieve a corresponding FITS header.
;;         ae_photometry, TABLE_HEADER=table_header

;; NOTE THAT THIS PHOTOMETRY CODE IS NOT VERY SUITABLE FOR USE OUTSIDE OF AE'S "MERGE" STAGE.
;; CAUTION IS ADVISED WHEN USING THIS ROUTINE IN OTHER CONTEXTS!
;; For example, 
;;   * This code knows nothing about the OGIP keywords AREASCAL, whereas in XSPEC those quantities can scale src and/or bkg spectra.
;;   * This code ignores the EXPOSURE value in the background spectrum, whereas in XSPEC that quantity affect bkg scaling.
; =============================================================================
PRO ae_photometry, OBS_DATA=obs_data, RMF_FN=rmf_fn, ARF_FN=arf_fn, $
                   SRC_CHANNELS=src_channels_param, SRC_CNTS_spectrum=SRC_CNTS_spectrum, BKG_CNTS_spectrum=BKG_CNTS_spectrum, $
                   energ_lo, energ_hi, photometry, x_distribution_variance, y_distribution_variance, $
                   TABLE_HEADER=photometry_header, DIFFUSE=is_diffuse

;; The routine stores and uses several static data structures.         
COMMON ae_photometry, num_obs, $
  psf_x_var, bkg_x_var, psf_y_var, bkg_y_var, $
  src_observed_counts, bkg_observed_counts, bkg_counts_in_src_region, net_counts, net_flux, $  ; 2-D arrays
  channel_number, channel_lowenergy, channel_midenergy, channel_highenergy, channel_specresp,$ ; 1-D vectors from ARF
  src_channels, $                                                                              ; 1-D vectors from PI spectra
  total_bkg_exposurearea, total_src_exposurearea, total_exposure,$                             ; float scalars
  bkg_spectra_available                                                                        ; boolean scalars

;; ------------------------------------------------------------------------
; Start using a new set of extractions.
if keyword_set(obs_data) then begin
    src_channels = src_channels_param

    num_obs = n_elements(obs_data)
    
    bkg_spectra_available = array_equal( obs_data.bkg_spectrum_fn NE '', 1B )
    
    psf_x_var = obs_data.psf_x_var
    bkg_x_var = obs_data.bkg_x_var
    psf_y_var = obs_data.psf_y_var
    bkg_y_var = obs_data.bkg_y_var
    
    ;; We're going to need to do photometry for each obsid later (to get position errors), 
    ;; so we'll construct 2-D source and background spectra in vars src_observed_counts, bkg_observed_counts.
    
    ; These photometry arrays must be 2-D, even if we have only one obsid, because subsequent code uses the construct total(xxxx,2) to sum quantities over all obsids.  The extra row must be zero-filled.
    src_observed_counts   = obs_data.src_observed_counts
    bkg_observed_counts   = obs_data.bkg_observed_counts
    if (size(/N_DIM,src_observed_counts) EQ 1) then begin
      src_observed_counts = [[src_observed_counts],[fltarr(n_elements(src_observed_counts))]]
      bkg_observed_counts = [[bkg_observed_counts],[fltarr(n_elements(bkg_observed_counts))]]
    endif
    
    bkg_counts_in_src_region = make_array(/FLOAT, DIMENSION=size(bkg_observed_counts, /DIM))
    
    total_bkg_exposurearea = total(obs_data.bkg_backscal)
    total_src_exposurearea = total(obs_data.src_backscal)
;help, total_bkg_exposurearea,total_src_exposurearea
    for jj = 0, num_obs-1 do begin
      ; -----------------------------------------------------------------------
      ; Compute the real-valued spectrum for the background that should
      ; be within the source aperture.  NOTE that the AE convention is
      ; that the background scaling information is carried solely in the
      ; BACKSCAL keywords (not in EXPOSURE keywords), which are integrals
      ; of the emaps over the extraction regions.  (See also comments in 
      ; source extraction code.) 
      ; That integration is how the geometric area of the region comes into play.
      ; The emaps are expected to represent both effective area and integration 
      ; time variations across the field -- the EXPOSURE keywords in the 
      ; spectra are not relevant.  
      
      ; Thus, AT THE ENERGY FOR WHICH THE EMAP WAS COMPUTED we can estimate 
      ; the real-valued background that should be within the source aperture
      ; by scaling the observed counts in the background spectrum by the ratio
      ; of these "sensitivity" (emap) integrals:
      
      area_exposure_ratio = float(obs_data[jj].src_backscal)/obs_data[jj].bkg_backscal
      if ~finite(area_exposure_ratio) then area_exposure_ratio = 0.0
      
      ; If the source and background regions have ARFs with the same SHAPE,
      ; i.e. they are near each other, then src_backscal & bkg_backscal have the same
      ; dependence on energy, and area_exposure_ratio is independent of energy.
      ; In other words if we repeated the exercise with emaps made at a different  
      ; energy we'd get the same area_exposure_ratio value.
      ; 
      ; However, if the source and background regions have differently shaped ARFs 
      ; then src_backscal & bkg_backscal would vary differently with energy and
      ; thus area_exposure_ratio is a function of energy.
      ; We can NOT simply use the ratio of the src & bkg ARFs as the area_exposure_ratio
      ; because that accounts only for sensitivity differences between the regions; the
      ; geometric area information is found only in (src_backscal/bkg_backscal).
      ; Recalling that (src_backscal/bkg_backscal) IS the correct scaling for the energy
      ; at which the emap was computed, we see that the appropriate energy-dependence to
      ; put on area_exposure_ratio is the ratio of the two ARFs, scaled to 1.0 at the
      ; emap's energy.  The user must supply that in the keyword EMAP_ENERGY.
      
      ; OBVIOUSLY USING A BACKGROUND ARF IN THIS WAY CAN CORRECTLY SCALE ONLY THE X-RAY
      ; BACKGROUND, NOT THE INSTRUMENTAL BACKGROUND, SO THE BACKGROUND SUBTRACTION MAY
      ; STILL HAVE PROBLEMS.
      ;
      ; A better approach is probably to use stowed event data to produce an instrumental 
      ; background spectrum for each extraction (object and sky regions), and then in the
      ; spectral analysis you can explicitly model the astrophysical (sky) X-ray background
      ; and simultaneously fit both extractions.  See the AE manual and the diffuse recipe.
      
      
      if file_test(obs_data[jj].bkg_arf_fn) then begin
        ; Nov 2009                                                    
        ; Since AE no longer supports an energy-dependent (vector) AREASCAL, we have removed the code
        ; that used to appear below which warps that scaling vector using the ratio of the two ARFs
        ; The details of that code were probably obsolete anyway, now that diffuse ARFs now carry 
        ; area on the sky information, not just effective area and exposure information.
        ; We now throw an error message if someone supplies a background.arf file
        
        print, obs_data[jj].bkg_arf_fn, F='(%"ERROR: AE no longer allows you supply a separate ARF (background.arf) for the background spectrum: %s\nSee the recipe for diffuse sources in the AE manual.")'
        retall

      endif      
      
      ; Finally, apply our background scaling to the observed background spectrum. 
      bkg_counts_in_src_region[0,jj] = bkg_observed_counts[*,jj] * area_exposure_ratio

    endfor ; jj, looping over obsids

    ; Compute 2-D net counts array.  Since each bin is only 1 PI channel wide
    ; many bins will be negative (i.e. there will be background data there but no
    ; source event).
    net_counts               = src_observed_counts - bkg_counts_in_src_region
    
    total_exposure    = total(obs_data.src_exposure)
    
    
    ;; The calling protocol is that obs_data must be loaded first, then the responses.
    ;; So, as defensive programming, set the ARF values and  quantites derived from the ARF to zero.
    channel_specresp = replicate(0., 1 > n_elements(channel_number))    
    net_flux         = replicate(0., 1 > n_elements(channel_number),2)
endif ; keyword_set(obs_data)


;; ------------------------------------------------------------------------
;; Start using a new set of response files.
if keyword_set(rmf_fn) then begin
    ae_channel_energy_and_arf, rmf_fn, arf_fn, $
        channel_number, channel_lowenergy, channel_highenergy, channel_midenergy, channel_specresp

    ; If an ARF filename was passed, then make sure there are no zeros because we are dividing by the ARF later to compute "net_flux" array.
    if keyword_set(arf_fn) then begin
      ind = where(channel_specresp EQ 0, count)
      if (count GT 0) then begin
        print, 'WARNING: found ARF values equal to zero in '+arf_fn
        channel_specresp[ind] = 1E-10
      endif
      
      ; Compute 2-D net flux array; see algorithms section of manual.  Since each bin is only 1 PI channel wide
      ; many bins will be negative (e.g. there will be background data but zero source events).
      make_2d, channel_specresp, intarr(num_obs>2), channel_specresp_2D, dummy
      
      net_flux = net_counts / channel_specresp_2D / total_exposure
      
      ; WARNING!  The net_flux vector is very noisy at the ends, where the ARF is tiny.
      ; Thus, all the statistics derived from it (FLUX1, ENERG_MEAN_INCIDENT, ENERG_PCT25_INCIDENT,
      ; ENERG_PCT50_INCIDENT, ENERG_PCT75_INCIDENT) have low SYSTEMATIC error (because we
      ; "follow" the shape of the ARF), but have high RANDOM error (because a single src or bkg
      ; count where the ARF is tiny has a large effect on the statistic). 
      
    endif else begin
      ; No ARF filename was passed, so make sure we discard any ARF values and derived quantites from some previous call!!
      channel_specresp = replicate(0., n_elements(channel_number))    
      net_flux         = replicate(0., n_elements(channel_number),2)
    endelse
endif ; keyword_set(rmf_fn)
    


;; ------------------------------------------------------------------------
if arg_present (SRC_CNTS_spectrum) then begin
  SRC_CNTS_spectrum = total(src_observed_counts,2)
  BKG_CNTS_spectrum = total(bkg_observed_counts,2)
  src_channels_param= src_channels
endif


;; ------------------------------------------------------------------------
if arg_present(photometry) then begin
  x_distribution_variance = 0
  y_distribution_variance = 0

  f_nan      = !VALUES.F_NAN
  photometry = {ENERG_LO  :f_nan, ENERG_HI         :f_nan, $
                CHAN_LO   : 0,     CHAN_HI         : 0, $
                MEAN_ARF  :f_nan, SRC_CNTS         :0L, $
                BKG_CNTS  :0L,    BACKSCAL         :f_nan, $
                NET_CNTS  :f_nan, NET_CNTS_SIGMA_UP:f_nan, NET_CNTS_SIGMA_LOW:f_nan, $
                SRC_SIGNIF:f_nan, PROB_NO_SOURCE   :f_nan, $
                FLUX1     :f_nan, FLUX2            :f_nan, $
                ENERG_MEAN_OBSERVED: f_nan, ENERG_MEAN_INCIDENT: f_nan, $
                ENERG_PCT25_OBSERVED:f_nan, ENERG_PCT25_INCIDENT:f_nan, $
                ENERG_PCT50_OBSERVED:f_nan, ENERG_PCT50_INCIDENT:f_nan, $
                ENERG_PCT75_OBSERVED:f_nan, ENERG_PCT75_INCIDENT:f_nan }

  photometry.ENERG_LO = energ_lo
  photometry.ENERG_HI = energ_hi
  
  ; When the caller supplies identical energ_lo and energ_hi values, then it is simply trying to look up an ARF values at a specific energy.
  ; This has to be coded specially:
  if (energ_lo EQ energ_hi) then begin
    photometry.MEAN_ARF = interpol(channel_specresp, channel_midenergy, energ_lo)
    return
  endif
  
    
  ; Look up PI channel numbers that bracket [eband_lo,eband_hi] ranges.
  ; A channel is included only if its mid-energy is in the range, i.e. if >=50% of it is in the energy range.
  ind =         (where(channel_midenergy GE energ_lo))[0]
  photometry.CHAN_LO = channel_number[ind]
  if (ind EQ 0) then $
    print, energ_lo, energ_hi, F='(%"ERROR! Your spectra do not span the photometry band %0.1f:%0.1f keV!")'
  
  ind = (reverse(where(channel_midenergy LE energ_hi)))[0]
  photometry.CHAN_HI = channel_number[ind] > photometry.CHAN_LO
  if (ind EQ (n_elements(channel_midenergy)-1) ) then $
    print, energ_lo, energ_hi, F='(%"ERROR! Your spectra do not span the photometry band %0.1f:%0.1f keV!")'
  
  
  ; Find the first subscript indexes of the arrays src_observed_counts, bkg_observed_counts,
  ; net_counts, net_flux that correspond to this band's channel range.
  ; The variable "src_channels" (from the CHANNEL column in spectrum file) usually starts at 1
  band_index = where((photometry.CHAN_LO  LE src_channels)   AND $
                     (src_channels        LE photometry.CHAN_HI), band_num_channels)
  
  ; Compute the mean ARF value in the energy band.
  ; Since the ARF is not evenly sampled we must integrate the ARF & divide by Erange.
  photometry.MEAN_ARF = total(channel_specresp[band_index] * (channel_highenergy[band_index] - channel_lowenergy[band_index]), /D)$
                                                     / total((channel_highenergy[band_index] - channel_lowenergy[band_index]), /D)
  
  ; Tally the SRC counts, BKG counts, and NET counts for this band in each obsid, producing vectors with num_obs elements.
  ; When there's only one obsid, src_observed_counts and net_counts are still 2-D---the extra row is all zeros.
  ; To prevent SRC_CNTS_single_obs and BKG_CNTS_single_obs from having extra elements in this case, we trim them below.
  SRC_CNTS_single_obs = (total(src_observed_counts[band_index,*],1,/D))[0:num_obs-1]
  BKG_CNTS_single_obs = (total(bkg_observed_counts[band_index,*],1,/D))[0:num_obs-1]
  NET_CNTS_single_obs = (total(         net_counts[band_index,*],1,/D))[0:num_obs-1]

  ; Sum those single-obsid photometry values to get merged scalar SRC_CNTS and BKG_CNTS.
  photometry.SRC_CNTS = round(total(SRC_CNTS_single_obs,/D))
  photometry.BKG_CNTS = round(total(BKG_CNTS_single_obs,/D))
                                              
  ; Estimate the (real-valued, scalar) number of background counts falling in the source aperture for this band
  ; by summing over the energy band and summing over the obsids.
  band_scaled_bkg = total(bkg_counts_in_src_region[band_index,*],/D)
  
  ; Compute the BACKSCAL value which, when used to normalize BKG_CNTS 
  ; in the NET_CNTS equation below, will produce the "band_scaled_bkg" 
  ; estimated above.
  ; For bands with zero background counts, BACKSCAL would be NaN.  Not knowing what else
  ; to do in such cases, we use the scaling one would get by combining all the src regions
  ; and combining all the background regions.
  photometry.BACKSCAL = (band_scaled_bkg GT 0) ? photometry.BKG_CNTS / band_scaled_bkg $
                                               : total_bkg_exposurearea/total_src_exposurearea

  if bkg_spectra_available then begin  
    ; Compute the significance of the observed SRC_CNTS as a disproof of the "null hypothesis" which is 
    ; that there is no source, i.e. that all the observed counts are background.  
    ; We use equation A7 from Weisskopf 2006 (astro-ph/0609585):
    
    ; As of IDL v8.5.1, binomial() has significant flaws and frequently produces very wrong results.
    ; See email to Harris Inc. on Oct 22, 2016.
    ; It also behaves very badly if GAUSSIAN=0 is (innocently) supplied (IDL bug report 69655).
    ; Prior to v8.5.1, binomial() could not handle inputs larger than 32767 (IDL bug report 69442).
    ; We use instead a simple and reliable algorithm recommended by Numerical Recipes (Chapter 6.4).
    
    photometry.PROB_NO_SOURCE = binomial_nr(photometry.SRC_CNTS, $
                                            photometry.SRC_CNTS + photometry.BKG_CNTS, $
                                            1D/(1D + photometry.BACKSCAL) ) > 0
  
    if ~finite(photometry.PROB_NO_SOURCE) then message, 'ERROR: PROB_NO_SOURCE is not finite.'
  endif
  
  ; Extract the channel numbers and channel energies that are in this energy band.
  ; Integrate the net counts and flux over the obsids (subscript #2), and extract the elements that are in this energy band.
  band_channels          = src_channels     [band_index]
  band_channel_midenergy = channel_midenergy[band_index]
  
  band_net_counts        =  total(net_counts[band_index,*],2)
  band_net_flux          =  total(net_flux  [band_index,*],2)
  
  ; Sum across the energy band to get scalar NET_CNTS and FLUX1 photometry.
  photometry.NET_CNTS = total(band_net_counts,/D)
  photometry.FLUX1    = total(band_net_flux,/D)
  
  ; Compute the more biased, and less noisy, FLUX2 estimate.
  photometry.FLUX2    = photometry.NET_CNTS / photometry.MEAN_ARF / total_exposure
  
  ; ------------------------------------------------------------------------
  ; Characterize the shape of the merged spectrum, within this energy band, in various ways.
  
  ; These are simple "center of masses" for net_counts or net_flux.
  ; Due to background subtraction, these values can be very noisy and can fall outside the energy band!
  if (photometry.NET_CNTS GT 0) then begin
    mean_channel = total(band_channels * band_net_counts, /D) / photometry.NET_CNTS
    photometry.ENERG_MEAN_OBSERVED = interpol(channel_midenergy, channel_number, mean_channel )
  endif
  
  if (photometry.FLUX1 GT 0) then begin
    mean_channel = total(band_channels * band_net_flux, /D) / photometry.FLUX1
    photometry.ENERG_MEAN_INCIDENT = interpol(channel_midenergy, channel_number, mean_channel )
  endif
  
  ; These are percentiles of net_counts or net_flux, i.e. the energy below which a
  ; specified fraction of the net_counts or net_flux falls.
  ; The 50% percentiles might be called "median" net_counts or net_flux.
  ;
  ; NOTE that the cumulative distributions of net_counts or net_flux as you move across
  ; the band are NOT necessarily monotonic (a count in src region makes
  ; it jump up, while a count in bkg region makes it jump down).
  ; The function histogram_percentile() is responsible for making a sensible estimate
  ; when there are multiple "crossings" of the specified percentile.  It returns a
  ; real-valued 0-based index of the histogram passed to it which we convert to an energy.
  if (band_num_channels GT 4) then begin
    ramp = indgen(band_num_channels)
    ind = histogram_percentile(band_net_counts, 0.25, ERROR=error)
    if (error EQ 0) then photometry.ENERG_PCT25_OBSERVED = $
                              interpol(band_channel_midenergy, ramp, ind )
    
    ind = histogram_percentile(band_net_flux,   0.25, ERROR=error)
    if (error EQ 0) then photometry.ENERG_PCT25_INCIDENT = $
                              interpol(band_channel_midenergy, ramp, ind )
    
    ind = histogram_percentile(band_net_counts, 0.50, ERROR=error)
    if (error EQ 0) then photometry.ENERG_PCT50_OBSERVED = $
                              interpol(band_channel_midenergy, ramp, ind )
    
    ind = histogram_percentile(band_net_flux,   0.50, ERROR=error)
    if (error EQ 0) then photometry.ENERG_PCT50_INCIDENT = $
                              interpol(band_channel_midenergy, ramp, ind )
    
    ind = histogram_percentile(band_net_counts, 0.75, ERROR=error)
    if (error EQ 0) then photometry.ENERG_PCT75_OBSERVED = $
                              interpol(band_channel_midenergy, ramp, ind )
    
    ind = histogram_percentile(band_net_flux,   0.75, ERROR=error)
    if (error EQ 0) then photometry.ENERG_PCT75_INCIDENT = $
                              interpol(band_channel_midenergy, ramp, ind )
  endif

  ; ------------------------------------------------------------------------
  ; Compute uncertainies on some photometry estimates.
  
  ; We compute Gehrels upper (equation 7) & lower (equation 12) limits that define
  ; one-sided 84% confidence intervals (equivalent to Gaussian 1-sigma confidence intervals) 
  ; on SRC_CNTS and BKG_CNTS.    
  ; Then we propagate those 1-sigma upper & lowerlimits through the equation 
  ;   NET_CNTS = SRC_CNTS - BKG_CNTS/BACKSCAL
  ; using equation 1.31 in "A Practical Guide to Data Analysis for Physical Science Students",
  ; L. Lyons, 1991 to get upper and lower 1-sigma errors on NET_CNTS.
  
  src_cnts = photometry.SRC_CNTS
  bkg_cnts = photometry.BKG_CNTS
  
  src_cnts_limit_up = src_cnts + 1 + sqrt(src_cnts + 0.75)
  bkg_cnts_limit_up = bkg_cnts + 1 + sqrt(bkg_cnts + 0.75)
  
  ; These lower limits become NaN when SRC_CNTS or BKG_CNTS is 0.  
  ; Post-processing programs (e.g. hardness ratio calculators) should use this flag to 
  ; decide what to do, e.g. to set lower limit on NET_CNTS to zero, or skip calculating
  ; things derived from NET_CNTS.
  src_cnts_limit_low = src_cnts * ( 1 - 1/(9.0*src_cnts) - 1/(3.0*sqrt(src_cnts)) )^3.0 
  bkg_cnts_limit_low = bkg_cnts * ( 1 - 1/(9.0*bkg_cnts) - 1/(3.0*sqrt(bkg_cnts)) )^3.0 

  photometry.NET_CNTS_SIGMA_UP  = sqrt( ( src_cnts_limit_up  - src_cnts)^2 + $
                                        ((bkg_cnts_limit_up  - bkg_cnts)/photometry.BACKSCAL)^2 )

  photometry.NET_CNTS_SIGMA_LOW = sqrt( ( src_cnts_limit_low - src_cnts)^2 + $
                                        ((bkg_cnts_limit_low - bkg_cnts)/photometry.BACKSCAL)^2 )
             
  photometry.SRC_SIGNIF = photometry.NET_CNTS / photometry.NET_CNTS_SIGMA_UP

  
  
  ;; ------------------------------------------------------------------------
  ;; Estimate the variance of the parent distribution for the merged extraced counts.
  ;; This is a weighted average of the variances of the PSFs and flat backgrounds, both clipped by the apertures.
  ;; See comments in MERGE stage.
  
  ; Assign PSF and flat background weights such that they sum to SRC_CNTS_single_obs.
  ; Then normalize the weights to sum to 1.0.
  psf_weight =  NET_CNTS_single_obs               > 0
  bkg_weight = (SRC_CNTS_single_obs - psf_weight) > 0

  weight  = [psf_weight, bkg_weight]
  
  weight /= total(weight)

  if (n_elements(weight)    NE 2*num_obs) then message, 'BUG IN AE!!!'
  if (n_elements(psf_x_var) NE   num_obs) then message, 'BUG IN AE!!!'
  
  ; Sum the weighted variances and get a standard deviation for the event distribution.
  x_distribution_variance =  total(weight * [psf_x_var, bkg_x_var]) 
  y_distribution_variance =  total(weight * [psf_y_var, bkg_y_var]) 
endif ; arg_present(photometry)



;; ------------------------------------------------------------------------
if arg_present(photometry_header) then begin
    ; Write TTYPE keywords to the header with a description of each column in the comments part of the keyword.
    ; Although mwrfits will overwrite TTYPE keyword values, it will retain the comments we establish here.
    
    ; Unit specifications follow the standard in "Specification of Physical Units within OGIP FITS files" at
    ; http://heasarc.gsfc.nasa.gov/docs/heasarc/ofwg/docs/general/ogip_93_001/ogip_93_001.html
    photometry_header = 0  &  dum=temporary(photometry_header)
    psb_xaddpar, photometry_header, 'EXTNAME', 'WIDE_BAND_PHOTOMETRY'
    psb_xaddpar, photometry_header, 'EXPOSURE', total_exposure, 'total exposure in merged ObsIDs'
    psb_xaddpar, photometry_header, 'TTYPE1',  'ENERG_LO',  'energy range lower endpoint'
    psb_xaddpar, photometry_header, 'TUNIT1',  'keV'
    psb_xaddpar, photometry_header, 'TTYPE2',  'ENERG_HI',  'energy range upper endpoint'
    psb_xaddpar, photometry_header, 'TUNIT2',  'keV'
    psb_xaddpar, photometry_header, 'TTYPE3',  'CHAN_LO',  'channel range lower endpoint'
    psb_xaddpar, photometry_header, 'TUNIT3',  ''
    psb_xaddpar, photometry_header, 'TTYPE4',  'CHAN_HI',  'channel range upper endpoint'
    psb_xaddpar, photometry_header, 'TUNIT4',  ''
    psb_xaddpar, photometry_header, 'TTYPE5',  'MEAN_ARF',  'mean ARF value'
    psb_xaddpar, photometry_header, 'TUNIT5',  keyword_set(is_diffuse) ? 'arcsec**2 cm**2 count /photon' : 'cm**2 count /photon'
    psb_xaddpar, photometry_header, 'TTYPE6',  'SRC_CNTS',  'extracted counts in merged apertures'
    psb_xaddpar, photometry_header, 'TUNIT6',  'count'
    psb_xaddpar, photometry_header, 'TTYPE7',  'BKG_CNTS',  'extracted counts in merged background regions'
    psb_xaddpar, photometry_header, 'TUNIT7',  'count'
    psb_xaddpar, photometry_header, 'TTYPE8',  'BACKSCAL',  'background scaling'
    psb_xaddpar, photometry_header, 'TUNIT8',  ''
    psb_xaddpar, photometry_header, 'TTYPE9',  'NET_CNTS',  'net counts in merged apertures'
    psb_xaddpar, photometry_header, 'TUNIT9',  'count'
    psb_xaddpar, photometry_header, 'TTYPE10', 'NET_CNTS_SIGMA_UP', '1-sigma Gehrels upper uncertainty'
    psb_xaddpar, photometry_header, 'TUNIT10', 'count'
    psb_xaddpar, photometry_header, 'TTYPE11', 'NET_CNTS_SIGMA_LOW', '1-sigma Gehrels lower uncertainty'
    psb_xaddpar, photometry_header, 'TUNIT11', 'count'
    psb_xaddpar, photometry_header, 'TTYPE12', 'SRC_SIGNIF', 'NET_CNTS / NET_CNTS_SIGMA_UP'
    psb_xaddpar, photometry_header, 'TUNIT12', ''
    psb_xaddpar, photometry_header, 'TTYPE13', 'PROB_NO_SOURCE', 'p-value for no-source hypothesis'
    psb_xaddpar, photometry_header, 'TUNIT13', ''
    psb_xaddpar, photometry_header, 'TTYPE14', 'FLUX1', 'incident photon flux: SUM(ARF(Ei)) / EXPOSURE'
    psb_xaddpar, photometry_header, 'TUNIT14',  keyword_set(is_diffuse) ? 'photon /cm**2 /s /arcsec**2' : 'photon /cm**2 /s'
    psb_xaddpar, photometry_header, 'TTYPE15', 'FLUX2', 'incident photon flux: NET_CNTS / MEAN_ARF / EXPOSURE'
    psb_xaddpar, photometry_header, 'TUNIT15',  keyword_set(is_diffuse) ? 'photon /cm**2 /s /arcsec**2' : 'photon /cm**2 /s'
    psb_xaddpar, photometry_header, 'TTYPE16', 'ENERG_MEAN_OBSERVED', 'mean energy, observed spectrum'
    psb_xaddpar, photometry_header, 'TUNIT16', 'keV'
    psb_xaddpar, photometry_header, 'TTYPE17', 'ENERG_MEAN_INCIDENT', 'mean energy, incident spectrum'
    psb_xaddpar, photometry_header, 'TUNIT17', 'keV'
    psb_xaddpar, photometry_header, 'TTYPE18', 'ENERG_PCT25_OBSERVED', '25th energy percentile, observed spectrum'
    psb_xaddpar, photometry_header, 'TUNIT18', 'keV'
    psb_xaddpar, photometry_header, 'TTYPE18', 'ENERG_PCT25_INCIDENT', '25th energy percentile, incident spectrum'
    psb_xaddpar, photometry_header, 'TUNIT19', 'keV'
    psb_xaddpar, photometry_header, 'TTYPE20', 'ENERG_PCT50_OBSERVED', 'median energy, observed spectrum'
    psb_xaddpar, photometry_header, 'TUNIT20', 'keV'
    psb_xaddpar, photometry_header, 'TTYPE21', 'ENERG_PCT50_INCIDENT', 'median energy, incident spectrum'
    psb_xaddpar, photometry_header, 'TUNIT21', 'keV'
    psb_xaddpar, photometry_header, 'TTYPE22', 'ENERG_PCT75_OBSERVED', '75th energy percentile, observed spectrum'
    psb_xaddpar, photometry_header, 'TUNIT22', 'keV'
    psb_xaddpar, photometry_header, 'TTYPE23', 'ENERG_PCT75_INCIDENT', '75th energy percentile, incident spectrum'
    psb_xaddpar, photometry_header, 'TUNIT23', 'keV'
endif ; arg_present(table_header)


return
end ; ae_photometry




; =============================================================================
; Maximum-likelihood image reconstruction.

; This is a copy of Max_Likelihood.pro (Sept 2011) from the AstroLib, modified to use ae_convolve instead of convolve.pro.
;
; METHOD:
;       Maximum Likelihood solution is a fixed point of an iterative eq.
;       (derived by setting partial derivatives of Log(Likelihood) to zero).
;       Poisson noise case was derived by Richardson(1972) & Lucy(1974).
;       Gaussian noise case is similar with subtraction instead of division.
; =============================================================================
PRO ae_Max_Likelihood, data, psf, deconv, Re_conv, FT_PSF=psf_ft, NO_FT=noft, NO_PAD=no_pad, VERBOSE=verbose, $
						GAUSSIAN=gaussian, $
						POSITIVITY_EPS=epsilon, $
						UNDERFLOW_ZERO=under
					
  compile_opt idl2
	if N_elements( deconv ) NE N_elements( data ) then begin
		deconv = data
		deconv[*] = total( data )/N_elements( data )
		Re_conv = 0
	endif

	if N_elements( under   ) NE 1 then under   = 1.e-22
	if N_elements( epsilon ) NE 1 then epsilon = -1
	if N_elements( Re_conv ) NE N_elements( deconv ) then $
		Re_conv = ae_convolve( positivity( deconv, EPS=epsilon ), psf, FT_PSF=psf_ft, NO_FT=noft, NO_PAD=no_pad, VERBOSE=verbose )
						
	if keyword_set( gaussian ) then begin
		deconv = deconv + ae_convolve( data - Re_conv, psf, /CORREL, FT_PSF=psf_ft, NO_FT=noft, NO_PAD=no_pad, VERBOSE=verbose )
	endif else begin
		wp = where( Re_conv GT under, npos, ncomplement=nneg,complement=wz)
              
		if (npos GT 0) then Re_conv[wp] = ( data[wp]/Re_conv[wp] ) > 0
		if (nneg GT 0) then Re_conv[wz] = 1.
		deconv = deconv * ae_convolve( Re_conv, psf, FT_PSF=psf_ft, /CORREL, NO_FT=noft, NO_PAD=no_pad, VERBOSE=verbose )
		; The /CORREL option above is the 'flipping' of the PSF (rotation by 180 degrees) 
		; called for in the Richardson-Lucy algorithm.
	endelse
	   
	if N_params() GE 4 then $
		Re_conv = ae_convolve( positivity( deconv, EPS=epsilon ), psf, FT_PSF=psf_ft, NO_FT=noft, NO_PAD=no_pad, VERBOSE=verbose )
						
end ; ae_Max_Likelihood







; =============================================================================
; ACIS EXTRACT MAIN PROGRAM
; =============================================================================
PRO acis_extract, catalog_or_srclist, obsname, obsdata_filename_p, $
                  SOURCE_NOT_OBSERVED=source_not_observed, $
                  EXTRACTION_NAME=extraction_name, MERGE_NAME=merge_name, $
                  ALLOW_LINKED_SOURCE_DIRECTORIES=allow_linked_source_directories, $
                                  
                  CONSTRUCT_REGIONS=construct_regions, ASPECT_FN=aspect_fn, $
                  MASK_FRACTION=mask_fraction, MASK_MULTIPLIER=mask_multiplier, $
                  PSF_MODEL_ENERGY=psf_model_energy, PSF_MODEL_COUNTS=psf_model_counts_p, $
                  PSF_FOOTPRINT_MULTIPLIER=psf_footprint_multiplier, $
                  REGION_ONLY=region_only, REUSE_PSF=reuse_psf, $
                  ASPECT_BLUR=aspect_blur, PIX_ADJ=pix_adj, $
                  DIFFUSE=diffuse, $
                  QUERY_ONLY=query_only, $
                  PSF_ONLY=psf_only, $
                  
                  SHOW_REGIONS=show_regions, SRCLIST_FILENAME=srclist_filename, INDEX_FILE=index_file, $
                  REGION_FILENAME=region_file, DISPLAY_FILE=display_file, $
                  OBSID_REGION_GLOB_PATTERN=obsid_region_glob_pattern, $
                  MERGE_REGION_FILENAME=merge_region_filename, $
                  OMIT_BKG_REGIONS=omit_bkg_regions, OMIT_SINGLE_OBSID=omit_single_obsid, $
                  INCLUDE_PRUNED_OBSIDS=include_pruned_obsids, $
                  ADDITIONAL_DISPLAY_FILE=additional_display_file, $
                  DS9_OPTION_STRING=ds9_option_string, $
                                  
                  EXTRACT_EVENTS=extract_events, ONLY_EDITED=only_edited, $
                  WARNING_REGION_FILENAME=warning_region_filename, $ 
                  BUILD_NEIGHBORHOOD_EMAP=build_neighborhood_emap, NEIGHBORHOOD_SIZE=neighborhood_size,$
                  TIME_FILTER=time_filter, $
                  
                  CHECK_POSITIONS=check_positions, MAXLIKELIHOOD_ITERATIONS=maxlikelihood_iterations, $
                  SKIP_EVENT_REGIONS=skip_event_regions, $
                  SKIP_RECONSTRUCTION=skip_reconstruction, SKIP_CORRELATION=skip_correlation, $
                  
                  NEW_CATALOG=new_catalog, $

                  EXTRACT_SPECTRA=extract_spectra, $
                  EMAP_FILENAME=emap_filename_p, ENERGY_RANGE=energy_range, DETCHANS=DETCHANS, $
                  ASPHIST_DIR=asphist_dir, ARDLIB_FILENAME=ardlib_filename, $
                  PBKFILE=pbkfile, MSKFILE=mskfile, $
                  REUSE_NEIGHBORHOOD=reuse_neighborhood, REUSE_ARF=reuse_arf, GENERIC_RMF_FN=generic_rmf_fn, $
                  USE_MKRMF=use_mkrmf, ENERGY_GRID_SPEC=energy_grid_spec, $
                  EMAP_ENERGY=emap_energy, WMAP_ENERGY_RANGE=wmap_energy_range, $
                  FORCE_S_FILTER_RECALC=force_s_filter_recalc, $
                  
                  ARF_CORRECTION_FILENAME=arf_correction_filename, $
                  
                  EXTRACT_BACKGROUNDS=extract_backgrounds, REUSE_BACKGROUND=reuse_background, $
                  MIN_NUM_CTS=min_num_cts, TWEAK_BACKSCAL=tweak_backscal, $
                  
                  TIMING=timing, $
                  
                  MERGE_OBSERVATIONS=merge_observations, EBAND_LO=eband_lo, EBAND_HI=eband_hi, $
                  NOMINAL_PSF_ENERGY=nominal_psf_energy, $ 
                  OVERLAP_LIMIT=overlap_limit, EMAP_UNIFORMITY_LIMIT=emap_uniformity_limit, $
                  SKIP_PSF=skip_psf_p, SKIP_NEIGHBORHOOD=skip_neighborhood_p, SKIP_APERTURE=skip_aperture_p, SKIP_SPECTRA=skip_spectra_p, SKIP_TIMING=skip_timing_p, $
                  MERGE_FOR_PB=merge_for_pb, MERGE_FOR_POSITION=merge_for_position, MERGE_FOR_PHOTOMETRY=merge_for_photometry, $
                  MIN_QUALITY=min_quality, $
                  THETA_RANGE=theta_range,$
                  SKIP_SINGLE_OBSID_MERGES=skip_single_obsid_merges,$
                  PAGE_LONG_DIM=page_long_dim, PAGE_SHORT_DIM=page_short_dim, $
                  
                  FIT_SPECTRA=fit_spectra, CHANNEL_RANGE=channel_range, CSTAT_EXPRESSION=cstat_expression, $
                  MODEL_FILENAME=model_filename, MODEL_CHANGES_FILENAME=model_changes_filename, $
                  SNR_RANGE=snr_range, NUM_GROUPS_RANGE=num_groups_range, $
                  GROUP_WITHOUT_BACKGROUND=group_without_background, INTERACTIVE=interactive, $
                  FIT_TIMEOUT=fit_timeout, GROUPED_SPECTRUM_FILENAME=grouped_spectrum_pattern,$
                  
                  COLLATED_FILENAME=collated_filename, HDUNAME=hduname_p, SINGLE_OBSID=single_obsid, $
                  REGION_TAG=region_tag, MATCH_EXISTING=match_existing, LABEL_FILENAME=label_filename, $
                                  
                  PLOT=plot, CARTOON_TEMPLATE=cartoon_template, CARTOON_FWHM=cartoon_fwhm, $
                  VERBOSE=verbose, OUTPUT_DIR=output_dir, $
                  
                  SIMULATE_SOURCES=simulate_sources
                

creator_string = "acis_extract, version " +strmid("$Rev:: 5658  $",7,5) +strmid("$Date: 2022-01-25 07:06:00 -0700 (Tue, 25 Jan 2022) $", 6, 11)

if (n_elements(verbose) EQ 0)       then verbose=1

if (verbose GT 0) then begin
  print, F='(%"\n============================================================")'  
  print, creator_string, F='(%"%s")'
  print, now()
  print, 'http://personal.psu.edu/psb6/TARA/ae_users_guide.html'
  print, 'Join the email list to receive announcements: http://lists.psu.edu/cgi-bin/wa?A0=L-ASTRO-ACIS-EXTRACT'
  print, 'Contact: patrick.broos@icloud.com'
  print, F='(%"============================================================\n")'  
endif

exit_code = 0

;; Create a unique scratch directory.
cache_dir        = 'cache'
tempdir          = 'tmp'
temproot = temporary_directory( 'AE.', SUBDIR1=cache_dir, SUBDIR2=tempdir, VERBOSE=(verbose GT 0), SESSION_NAME=session_name)

temp_bkgimg_fn   = cache_dir + 'temp.bkg.img'
inband_events_fn = tempdir + 'temp.inband.evt'
temp_events_fn   = tempdir + 'temp.evt'
temp_events2_fn  = tempdir + 'temp2.evt'
temp_region_fn   = tempdir + 'temp.reg'
temp_image_fn    = tempdir + 'temp.img'
temp_text_fn     = tempdir + 'temp.txt'
temp_par1_fn     = tempdir + 'temp1.par'
temp_par2_fn     = tempdir + 'temp2.par'
temp_lc_fn       = tempdir + 'temp.lc'
temp_wgt_fn      = tempdir + 'temp.wgt'
temp_rmf_fn      = tempdir + 'temp.rmf'
temp_collate_fn  = tempdir + 'temp.collated'

;; We assume that access to /tmp/ will often be faster than access to the event and emap data passed,
;; so let's start by copying those files to a cache.
dmmerge_header_lookup_fn = cache_dir + 'dmmerge_header_lookup.txt'

if  keyword_set(obsdata_filename_p) then begin
  if ~file_test(obsdata_filename_p) then begin
    message, 'Cannot read ' + obsdata_filename_p
  endif

  fdecomp, obsdata_filename_p, disk, item_path, item_name, item_qual
  obsdata_filename = cache_dir+ item_name+'.'+item_qual
  file_copy, obsdata_filename_p, obsdata_filename
  
  obsdata_header = headfits(obsdata_filename, EXT=1, ERRMSG=error )
  if (keyword_set(error)) then begin
    print, error
    message, 'ERROR reading ' + obsdata_filename
  endif

  
  ; Determine which CCDs are present in the event list provided.
  ; We do not trust the DETNAM keyword because the observer may have applied a CCD_ID filter.
  ccd_is_in_observation = bytarr(10)
  for ccd_id = 0,9 do begin
    keyname    = string(ccd_id, F='(%"EXPOSUR%d")')
    exposure   = psb_xpar( obsdata_header, keyname, COUNT=count)
                   
    if (count EQ 0) || (exposure LE 0)  then continue
    ccd_is_in_observation[ccd_id] = 1B
  endfor ;ccd_id
endif

if keyword_set(emap_filename_p) then begin
  fdecomp, emap_filename_p, disk, item_path, item_name, item_qual
  emap_filename = cache_dir+ item_name+'.'+item_qual
  file_copy, emap_filename_p, emap_filename
  
  emap_header = headfits(emap_filename, ERRMSG=error )
  if (keyword_set(error)) then begin
    print, error
    message, 'ERROR reading ' + emap_filename
  endif
  
  if strtrim(psb_xpar(obsdata_header, 'OBS_ID'),2) NE strtrim(psb_xpar(emap_header, 'OBS_ID'),2) then begin
    print, string(obsdata_filename_p, emap_filename_p, F='(%"\nERROR: OBS_ID keywords in %s and %s do not match.")')
    GOTO, FAILURE
  endif
endif






if NOT keyword_set(mask_fraction)            then mask_fraction  =0.99
if NOT keyword_set(mask_multiplier)          then mask_multiplier=1.1
if NOT keyword_set(psf_footprint_multiplier) then psf_footprint_multiplier=1.0
if NOT keyword_set(maxlikelihood_iterations) then maxlikelihood_iterations = 400

if NOT keyword_set(nominal_psf_energy) then nominal_psf_energy = 1.49670

; These 5 energy values are merely historical---the energies at which calibration data was available to build the CIAO PSF Library.
; It is not clear if MARX is any more accurate at these energies than at others.
if ~keyword_set(psf_model_energy) then psf_model_energy = [0.277, 1.49, 4.5, 6.4, 8.6]

if keyword_set(psf_model_counts_p) then begin
  if (n_elements(psf_model_counts_p) NE n_elements(psf_model_energy)) then begin
    print, 'ERROR: parameters PSF_MODEL_ENERGY and PSF_MODEL_COUNT must have the name number of elements.'
    GOTO, FAILURE
  endif
  
  psf_model_counts = psf_model_counts_p
endif else begin
  psf_model_counts = replicate(1E5, n_elements(psf_model_energy)) 
  ; By default, we generated a lower-noise PSF image at the energy closest to nominal_psf_energy, because that's the energy we use for estimating recon positions.
  ;dum = min(abs(psf_model_energy - nominal_psf_energy), imin)
  ;psf_model_counts[imin] *= 10
endelse 

if n_elements(reuse_psf) EQ 0 then reuse_psf=1

if NOT keyword_set(     energy_range) then      energy_range = [0.5,8.0]
if    ~keyword_set(wmap_energy_range) then wmap_energy_range = [0.5,2.0]

if NOT (keyword_set(eband_lo) AND keyword_set(eband_hi)) then begin
  eband_lo = [0.5,  0.5, 2.0,  0.5, 1.7, 2.8,  0.5, 1.5, 2.5,  0.5, 1.0, 2.0, 4.0, 6.0,  0.5, 2.0,  9.0]
  eband_hi = [8.0,  2.0, 8.0,  1.7, 2.8, 8.0,  1.5, 2.5, 8.0,  1.0, 2.0, 4.0, 6.0, 8.0,  7.0, 7.0, 12.0]
endif

if n_elements(overlap_limit         ) EQ 0 then overlap_limit          = 0.10

if n_elements(emap_uniformity_limit ) EQ 0 then emap_uniformity_limit  = 0.50

if n_elements(build_neighborhood_emap) EQ 0 then build_neighborhood_emap = 0

if ~keyword_set(neighborhood_size) then neighborhood_size = 10 ; arcsec

if NOT keyword_set(asphist_dir)   then asphist_dir = './asphist'

if (n_elements(theta_range) NE 2) then theta_range = [0,100.]

if ~keyword_set(output_dir      ) then output_dir  = './'

type = size(obsname,/TNAME)
dim  = size(obsname,/DIMEN)
case dim of
  0   : fail = 0
  1   : fail = (type NE 'STRING')
  else: fail = (type NE 'STRING') || ~keyword_set(merge_observations)
endcase
if fail then begin
  print, 'ERROR: parameter "obsname" must be a string or string array'
  GOTO, FAILURE
endif
if keyword_set(obsname) then begin
  obsname = strtrim(obsname,2)
  print, strjoin(obsname,','), F='(%"\nObsID name is ''%s''")'
endif
 
arcsec_per_ACISpixel = 0.492 
arcsec_per_skypixel  = 0.492 

max_polygon_elements = 100


; See these URLs for the X11 color chart:
; http://en.wikipedia.org/wiki/X11_color_names
; http://www.mcfedries.com/Books/cightml/x11color.htm
; http://www.febooti.com/products/iezoom/online-help/html-color-names-x11-color-chart.html
; See <X11root>/lib/X11/rgb.txt for the colors known to a given machine.
region_colors = ['red','green','cyan','magenta','yellow','DodgerBlue','SlateBlue','Cornsilk','Goldenrod','Chocolate','DarkSalmon', 'Tan','Peru','Sienna','Salmon', 'SandyBrown','DarkGoldenrod','Brown','IndianRed']

src_stats_basename       = 'source.stats'
src_photometry_basename  = 'source.photometry'
model_photometry_basename=  'model.photometry'
obs_parameters_basename  = 'obs.parameters'
obs_stats_basename       = 'obs.stats'
obs_frac_basename        = 'obs.psffrac'
env_emap_basename        = 'neighborhood.emap'
env_events_basename      = 'neighborhood.evt'
env_image_basename       = 'neighborhood.img'
src_region_basename      = 'extract.reg'
bkg_region_basename      = 'background.reg'
bkg_pixels_region_basename = 'background_pixels.reg'
bkg_emap_basename        = 'background.emap'
src_emap_basename        = 'source.emap'
evt_region_basename      = 'evt.reg'
src_events_basename      = 'source.evt'
bkg_events_basename      = 'background.evt'
src_spectrum_basename    = 'source.pi'
bkg_spectrum_basename    = 'background.pi'
bkg_arf_basename         = 'background.arf'
rmf_basename             = 'source.rmf'
arf_basename             = 'source.arf'
psf_basename             = 'source.psf'
lc_binned_basename       = 'source.binned_lc'
lc_smooth_basename       = 'source.lc'
fit_stats_basename       = 'source.spectra'
modelsubdir              = 'spectral_models/'
event_plot_basename      = 'source.evt.ps'


if ~keyword_set(catalog_or_srclist) then begin
  ; Parameter not passed; assign num_sources to prevent code failures.
  num_sources = 1
endif else begin
  if ~isa(/STRING, catalog_or_srclist) then begin
    print, 'ERROR: the "catalog_or_srclist" parameter must be a string or string array.' 
    GOTO, FAILURE
  endif

  ;; Interpret the parameter catalog_or_srclist.
  if array_equal(file_test(catalog_or_srclist,/DIRECTORY),1) then begin
    ; A list of source names (source directories) has been passed.
    sourcename  = catalog_or_srclist
    num_sources = n_elements(sourcename)
    if (verbose GT 0) then print, num_sources, F='(%"\n%d source names passed.\n")'
    
  endif else if (n_elements(catalog_or_srclist) EQ 1) && file_test(catalog_or_srclist,/REGULAR) then begin
    ; A single filename should be an ASCII file containing a list of source names. 
    readcol, catalog_or_srclist, sourcename, FORMAT='A', COMMENT=';', COUNT=num_sources
    
    if (num_sources EQ 0) then begin
      print, 'WARNING: catalog contains zero sources; AE run aborted.'
        GOTO, CLEANUP
    endif
    
    ; Trim whitespace and remove blank lines.
    sourcename = strtrim(sourcename,2)
    ind = where(sourcename NE '', num_sources)
    
    if (num_sources EQ 0) then begin
      print, 'ERROR: no entries read from source list ', catalog_or_srclist
      GOTO, FAILURE
    endif
    
    sourcename = sourcename[ind]
    if (verbose GT 0) then print, num_sources, F='(%"\n%d sources found in catalog.\n")'
    
    if (num_sources GT 1) then begin
      ;; Look for duplicate source names.
      sort_index   = sort(sourcename)
      sorted_names = sourcename[sort_index]
      
      uniq_index = uniq(sorted_names)
      num_uniq   = n_elements(uniq_index)
      if (num_uniq LT num_sources) then begin
        print, F='(%"\n=============================================================================")'
        print, 'WARNING: The catalog contains duplicate source names:'
      
        next = 0
        for ii=0L,num_uniq-1 do begin
          this = uniq_index[ii]
          if (this GT next) then begin
            print
            for jj=next,this do begin
              duplicate_index = sort_index[jj]
              print, sourcename[duplicate_index], F='(%"  %s")'
            endfor
          endif ; (this GT next)
          
          next = this+1
        endfor
        
        print, '============================================================================='
      endif ;duplicates
    endif ; (num_sources GT 1)
  endif else begin
    ; Cannot figure out what catalog_or_srclist means ...
    help, catalog_or_srclist
    print, 'acis_extract: ERROR, the "catalog_or_srclist" parameter is neither an existing ASCII file nor a list of source names.' 
    GOTO, FAILURE
  endelse
endelse ; The parameter catalog_or_srclist was passed.



;; If the SOURCE_NOT_OBSERVED supplied is a vector then we consider it an input parameter;
;; otherwise we initialize SOURCE_NOT_OBSERVED
if (size(/N_DIM, source_not_observed) NE 1) then source_not_observed = replicate(0B,num_sources)

if (n_elements(source_not_observed) NE num_sources) then begin
  print, 'WARNING: the vector SOURCE_NOT_OBSERVED and the catalog had a different number of elements; ignoring SOURCE_NOT_OBSERVED.'
  source_not_observed = replicate(0B,num_sources)
endif

if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)

if keyword_set(merge_name)      then merge_subdir = merge_name + '/' $
                                else merge_subdir = ''
if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)

  
;; =============================================================================
if keyword_set(new_catalog) then begin
;; =============================================================================
  ;; Re-create 5-column catalog for the sources in the supplied source list.
  ;; For regions that were edited use the actual PSF fraction; for unedited regions
  ;; use the target PSF fraction.  For unobserved sources used the median of the
  ;; target PSF fractions.
  print, 'Reading source information ...'

  ra                   = replicate(!VALUES.D_NAN,num_sources)
  dec                  = replicate(!VALUES.D_NAN,num_sources)
  target_psf_fraction  = replicate(!VALUES.F_INFINITY,num_sources)
  region_edited        = bytarr(num_sources)
  psf_fraction         = replicate(!VALUES.F_INFINITY,num_sources)
  psf_energy           = fltarr(num_sources)

  ;; Read summary information from unnamed stats file for each source.
  for ii = 0L, num_sources-1 do begin
    ; Remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    unnamed_src_stats_fn = sourcename[ii] + '/' + src_stats_basename
    stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    
    if (NOT keyword_set(error)) then begin
      ra[ii]  = psb_xpar( stats, 'RA')
      dec[ii] = psb_xpar( stats, 'DEC')
      
      ; If existing object name and directory name don't match, warn the observer.
      if  (sourcename[ii] NE strtrim(psb_xpar( stats, 'OBJECT'),2)) then $
        print, 'WARNING!  Source catalog/directory name does not match OBJECT keyword.'
    endif else begin
      print, error
      print, 'WARNING! Could not read ', unnamed_src_stats_fn
    endelse


    if keyword_set(obsname) then begin
      ;; Look for PSF fraction in the single observation specified.
      obsdir   = sourcename[ii] + '/' + obsname + '/' + extraction_subdir[ii]
      stats_fn = obsdir + obs_stats_basename
      obsid_count_i = 1
    
    endif else begin
      ;; Look for PSF fractions in all observations.
      stats_fn = file_search( sourcename[ii] + '/*/' + extraction_subdir[ii] + obs_stats_basename, COUNT=obsid_count_i )
      if (obsid_count_i EQ 0) then begin
        print, '  Not present in any observation'
        continue
      endif
    endelse

    for jj = 0, obsid_count_i-1 do begin
      stats = headfits(stats_fn[jj], ERRMSG=error)
      
      if (NOT keyword_set(error)) then begin
        target_psf_fraction [ii] = target_psf_fraction [ii]   < psb_xpar( stats, 'FRACSPEC')
        
        if (psf_energy[ii] EQ 0) then psf_energy[ii] = psb_xpar( stats, 'PSF_ENGY')
        
        if (psb_xpar( stats, 'REG_EDIT')) then begin
          region_edited     [ii] = 1
          
          this_psf_fraction = psb_xpar( stats, 'PSF_FRAC')
          if (this_psf_fraction LT psf_fraction[ii]) then begin
            psf_fraction      [ii] = this_psf_fraction
            psf_energy        [ii] = psb_xpar( stats, 'PSF_ENGY')
          endif
        endif ;REG_EDIT true
      endif ;no error
    endfor ;jj
  endfor ;ii
  
  ; Start with the smallest FRACSPEC found in all observations.
  new_psf_fraction = target_psf_fraction
  
  ; For regions hand edited, use the smallest actual PSF fraction.
  ind = where(region_edited, count)
  if (count GT 0) then begin
    new_psf_fraction[ind] = psf_fraction[ind]
    print, count, ' extraction regions edited; using actual PSF fractions'
    forprint, sourcename[ind], psf_fraction[ind]
  endif
  
  ; For unobserved sources, use the median PSF fraction of the observed ones.  
  ind = where(new_psf_fraction EQ 0, count)
  if (count GT 0) then begin
    non_null_frac = target_psf_fraction[where(target_psf_fraction NE 0)]
    new_psf_fraction[ind] = median(non_null_frac)
  endif
  
  forprint, TEXTOUT=new_catalog, sourcename, ra, dec, new_psf_fraction, psf_energy, $
            F='(A,1x,F10.6,1x,F10.6,1x,F5.3,1x,F7.5)', /NoCOMMENT
  print, '============================================================================='
  print, 'Wrote catalog ', new_catalog
  print, '============================================================================='
  GOTO, CLEANUP
endif


;; =============================================================================
if keyword_set(collated_filename) && ~keyword_set(show_regions) && ~keyword_set(plot) then begin
;; =============================================================================
  
  fdecomp, collated_filename, disk, item_path, item_name, item_qual
  if ('' NE item_qual)  then item_name = item_name+ '.' +item_qual

  case n_elements(hduname_p) of
   0: hduname = strarr(num_sources)
   1: hduname = replicate(hduname_p, num_sources)
   else: begin
         hduname    = strarr(num_sources)
         hduname[0] = hduname_p
         end
  endcase
  
  dum = where(hduname EQ '', count)
  if (count GT 0) && (verbose GT 0) then print, count, F='(%"WARNING!  No HDUNAME specified for %d sources; will use result from the last fit performed.")'

  ;; If directed, use an existing table as a template.
  if keyword_set(match_existing) then begin
    status = -1
    case size(/TNAME, match_existing) of
      'STRUCT' : begin
                 template_row = match_existing[0]
                 status = 0
                 end
      'STRING' : if file_test(match_existing   , /READ) then template_row = (mrdfits(match_existing   , 1, theader, /SILENT, STATUS=status))[0]
      else     : if file_test(collated_filename, /READ) then template_row = (mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status))[0]
    endcase                         
    
    if (status EQ 0) then begin
      col_names = tag_names(template_row)
      num_cols  = n_elements(col_names)
      
      ; Create a structure matching the existing table but with all fields nulled.
     ;struct_assign, {foobar:0}, template_row 
      template_row = null_structure(template_row)

      bin_table = replicate(template_row, num_sources)
    endif
  endif
    
  if keyword_set(bin_table) then begin
    start_pass = 2
  endif else begin
    start_pass = 1

    col_names    =      strarr(1000)
    col_comments =      strarr(1000)
    col_types    = replicate(4,1000)  ; Default is FLOAT (IDL type 4)
    col_elements = replicate(1,1000)  ; Default is scalar
    num_cols     = 0 
    
    ;; Define a column for the source name in the catalog.
    col_names   [num_cols  ] = 'CATALOG_NAME'
    col_comments[num_cols  ] = 'source name in catalog'
    col_types   [num_cols++] = 7  ; STRING type
  
    ;; Define a column LABEL, the source label that will appear in region files.
    col_names   [num_cols  ] = 'LABEL'
    col_comments[num_cols  ] = 'source lable'
    col_types   [num_cols++] = 7  ; STRING type
  
    ;; Define a column IS_WRITABLE, set when the source directory is writable and not a symlink.
    COL_NAMES   [num_cols  ] = 'IS_WRITABLE'
    COL_COMMENTS[num_cols  ] = 'source directory is writable and not a symlink'
    COL_TYPES   [num_cols++] = 1  ; BYTE TYPE
  
    ;; Define a column OBSDIR, the pathname of the single-observation extraction collated.
    col_names   [num_cols  ] = 'OBSDIR'
    col_comments[num_cols  ] = 'extraction directory'
    col_types   [num_cols++] = 7  ; STRING type
  
    ;; Define a column MERGE_NAME, the pathname of the merge collated.
    col_names   [num_cols  ] = 'MERGE_NAME'
    col_comments[num_cols  ] = 'merge directory'
    col_types   [num_cols++] = 7  ; STRING type
  
    ;; Define a column PHOT_CREATOR, the origin of the reported photometry.
    col_names   [num_cols  ] = 'PHOT_CREATOR'
    col_comments[num_cols  ] = 'origin of the reported photometry'
    col_types   [num_cols++] = 7  ; STRING type
    
    ;; Define a column BEST_MDL, the name of the XSPEC model preferred by the observer (a keyword in source.spectra).
    col_names   [num_cols  ] = 'BEST_MDL'
    col_comments[num_cols  ] = 'XSPEC model name preferred'
    col_types   [num_cols++] = 7  ; STRING type

    ;; Define a column MODEL, the name of the XSPEC model collated (an HDUNAME in source.spectra).
    col_names   [num_cols  ] = 'MODEL'
    col_comments[num_cols  ] = 'XSPEC model name collated'
    col_types   [num_cols++] = 7  ; STRING type
  
    ;; Define a column NMODELS, the number of XSPEC models available (HDUs in source.spectra).
    col_names   [num_cols  ] = 'NMODELS'
    col_comments[num_cols  ] = '# of spectral models'
    col_types   [num_cols++] = 2  ; INTEGER type
  
    ;; Define a column PROVISNL, a boolean flag indicating whether MODEL is provisional.
    col_names   [num_cols  ] = 'PROVISNL'
    col_comments[num_cols  ] = 'Spectral fit is provisional'
    COL_TYPES   [num_cols++] = 1  ; BYTE TYPE
  
    ;; Define a column NOCOLLAT, a boolean flag indicating whether collation of a spectral model has been explicitly forbidden.
    col_names   [num_cols  ] = 'NOCOLLAT'
    col_comments[num_cols  ] = 'Spectral fit withheld'
    COL_TYPES   [num_cols++] = 1  ; BYTE TYPE

    ;; Define a column CATEGORY, the spectral fitting "category" assigned by the observer in ae_spectra_viewer.
    col_names   [num_cols  ] = 'CATEGORY'
    col_comments[num_cols  ] = 'ae_spectra_viewer category'
    col_types   [num_cols++] = 7  ; STRING type
  endelse
   
    
  ;; In "pass 1" below we figure out what column names, types, and dimensions are 
  ;; needed in the output binary table, then we make a blank structure to hold that.
  ;; In "pass 2" we actually populate that structure with data from the source files.
 for pass=start_pass,2 do begin
  case pass of
   1: if (verbose GT 0) then print, 'Scanning results to define column names in output file...'
   2: begin
      if (verbose GT 0) then print, 'Reading AE results...'
  
      ;; Initialize FLOAT and DOUBLE columns to NaN so missing data is flagged.
      for jj = 0, num_cols-1 do begin
        case size(bin_table[0].(jj),/TYPE) of
          4: bin_table.(jj) = !VALUES.F_NAN
          5: bin_table.(jj) = !VALUES.D_NAN
          else:
        endcase
      endfor
  
      ;; Initialize columns we explicitly created.
      bin_table.CATALOG_NAME = sourcename
      bin_table.LABEL        = 'src #'+strtrim(1+indgen(num_sources),2)
      bin_table.MODEL        = "no_fit"
      bin_table.NMODELS      = 0
      bin_table.PROVISNL     = 0
      bin_table.NOCOLLAT     = 0
      bin_table.CATEGORY     = "no category"
      bin_table.MERGE_NAME   = keyword_set(merge_name) ? merge_name : ''
      end
  endcase
  
  primary_obs_dir = strarr(num_sources)
  display_obs_dir = strarr(num_sources)
  primary_obsname = strarr(num_sources)
  display_obsname = strarr(num_sources)
  
  
  
  ; LOOP OVER SOURCES
  for ii = 0L, num_sources-1 do begin
    basedir              = sourcename[ii] + '/' 
    sourcedir            = basedir + merge_subdir[ii]
    
    unnamed_src_stats_fn = basedir   + src_stats_basename
    src_stats_fn         = sourcedir + src_stats_basename

    temp = file_info(basedir)
    if (pass EQ 2) then bin_table[ii].IS_WRITABLE = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)

    if ~temp.EXISTS then begin
      print, sourcename[ii], F='(%"\nWARNING! Source %s does not exist.")'  
      continue
    endif 
   
    ;; ------------------------------------------------------------------------
    ; Determine which data structures should be read for this source.
    read_fundamental_properties = 1B
    read_merge_stats            = 1B
    read_obsid_stats            = 1B
    read_photometry             = 1B
    read_spectral_fit           = 1B
    
    ; For efficiency, skip some data structures for sources already determined to be not extracted.
    if source_not_observed[ii] then begin
      read_merge_stats            = 0B
      read_obsid_stats            = 0B
      read_photometry             = 0B
      read_spectral_fit           = 0B
    endif
    
    ; If the named merge directory is missing we cannot read files there.
    if ~file_test(sourcedir) then begin
      if (pass EQ 2) AND (verbose GT 0) then print, merge_subdir[ii], sourcename[ii], F='(%"\nWARNING! Merge %s not found for source %s.")'  
      read_merge_stats            = 0B
      read_obsid_stats            = 0B
      read_photometry             = 0B
      read_spectral_fit           = 0B
    endif 

    ; If the unnamed merge is being collated, then doing read_fundamental_properties would be redundant.
    if ~keyword_set(merge_subdir[ii]) then read_fundamental_properties = 0B

    ;; ------------------------------------------------------------------------
    ;; Ingest keywords from the unnamed and named source.stats files.
    
    ;; Care is required here because the source.stats file presents an opportunity for collisions between multiple AE sessions running simultaneously.  For example, one session may be in the process of writing source.stats while this session is attempting to read it.
    
    ; Start with an empty FITS header that will hold keywords read from source.stats and obs.stats files.
    mkhdr, stats, ''

    ;; ------------------------------------------------------------------------
    if read_merge_stats then begin
      ; First, try to read the source.stats file we are supposed to collate (which may be "named" by MERGE_NAME input).
      ; We try a few times because file access can collide with another AE process that's writing the file.
      for kk=1,2 do begin
        stats = headfits(src_stats_fn, ERRMSG=error)
        if ~keyword_set(error) then break
        print, error
        wait, kk*0.5
        print, 'Trying again to read ', src_stats_fn
      endfor ;kk

      if keyword_set(error) then begin
        print, src_stats_fn, F='(%"\nWARNING! File %s could not be read.\n")'  
        read_fundamental_properties = 1B ; Try to at least read the fundamental properties.
        read_obsid_stats            = 0B
        read_photometry             = 0B
        read_spectral_fit           = 0B
      
        ; Remake an empty FITS header "stats", which will have been destroyed by the failed headfits call.
        mkhdr, stats, ''
      endif
    endif ; read_merge_stats 

    ;; ------------------------------------------------------------------------
    if read_fundamental_properties then begin
      ; Read the unnamed source.stats file to get fundamental source properties.
      ; Values of these properties in the named stats file will be overwritten.
      for kk=1,2 do begin
        unnamed_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
        if ~keyword_set(error) then break
        print, error
        wait, kk*0.5
        print, 'Trying again to read ', unnamed_src_stats_fn
      endfor ;kk

      if keyword_set(error) then begin
        print, sourcename[ii], unnamed_src_stats_fn, F='(%"\nWARNING! Skipping source %s because %s could not be read.\n")'  
        continue
      endif else begin
        comment = ''
        psb_xaddpar, stats, 'LABEL'   , psb_xpar( unnamed_stats, 'LABEL'   , COMMENT=comment), comment
        psb_xaddpar, stats, 'OBJECT'  , psb_xpar( unnamed_stats, 'OBJECT'  , COMMENT=comment), comment
        psb_xaddpar, stats, 'PROVENAN', psb_xpar( unnamed_stats, 'PROVENAN', COMMENT=comment), comment
        psb_xaddpar, stats, 'POSNDATE', psb_xpar( unnamed_stats, 'POSNDATE', COMMENT=comment), comment
        psb_xaddpar, stats, 'POSNTYPE', psb_xpar( unnamed_stats, 'POSNTYPE', COMMENT=comment), comment
        psb_xaddpar, stats, 'RA'      , psb_xpar( unnamed_stats, 'RA'      , COMMENT=comment), comment, F='(F10.6)'
        psb_xaddpar, stats, 'DEC'     , psb_xpar( unnamed_stats, 'DEC'     , COMMENT=comment), comment, F='(F10.6)'
        psb_xaddpar, stats, 'ERR_RA'  , psb_xpar( unnamed_stats, 'ERR_RA'  , COMMENT=comment), comment, F='(F10.3)'
        psb_xaddpar, stats, 'ERR_DEC' , psb_xpar( unnamed_stats, 'ERR_DEC' , COMMENT=comment), comment, F='(F10.3)'
        psb_xaddpar, stats, 'IMMORTAL', psb_xpar( unnamed_stats, 'IMMORTAL', COMMENT=comment), comment
        psb_xaddpar, stats, 'PREVNAME', psb_xpar( unnamed_stats, 'PREVNAME', COMMENT=comment), comment
        psb_xaddpar, stats, 'BKSCL_LO', psb_xpar( unnamed_stats, 'BKSCL_LO', COMMENT=comment), comment
        psb_xaddpar, stats, 'BKSCL_GL', psb_xpar( unnamed_stats, 'BKSCL_GL', COMMENT=comment), comment
        psb_xaddpar, stats, 'BKSCL_HI', psb_xpar( unnamed_stats, 'BKSCL_HI', COMMENT=comment), comment
      endelse   
    endif ; read_fundamental_properties
    
    
    src_label = string(sourcename[ii], strtrim(psb_xpar( stats,'LABEL'),2), F='(%"Source %s (%s)")')
    
    fxhclean, stats
    sxdelpar, stats, ['DATE','COMMENT','HISTORY','EXTVER','CREATOR']

    
    ;; ------------------------------------------------------------------------
    if read_obsid_stats then begin
      ;; Figure out which ObsID should report extraction properties and aperture.
      
      ; If the observer specified an obsname then use it, otherwise try to use the PRIM_OBS keyword (deepest observation) to make that decision.
      if keyword_set(obsname) then begin
        primary_obsname[*] = obsname
        display_obsname[*] = obsname
        primary_count = 1
      endif else begin
        primary_obsname[ii] = strcompress(psb_xpar( stats, 'PRIM_OBS', COUNT=primary_count), /REMOVE_ALL)
      endelse
  
      if (primary_count EQ 1) then begin
        ; We've identified an ObsID of interest (from PRIM_OBS or from caller).  
        if keyword_set(obsname) then begin
          ; Display the specified observation.
          display_obsname[ii] = obsname
        endif else begin
          ; Otherwise, show the polygon from the Primary Observation.
          display_obsname[ii] = primary_obsname[ii]
        endelse
        
        ; Build the path to the primary observation, using any extraction name supplied
        primary_obs_dir[ii]   = sourcename[ii] + '/' + primary_obsname[ii] + '/' + extraction_subdir[ii]
        
        obs_stats_fn = primary_obs_dir[ii] + obs_stats_basename
        obs_stats    = headfits(obs_stats_fn, ERRMSG=error)
      
        if (NOT keyword_set(error)) then begin
          fxhclean, obs_stats
          sxdelpar, obs_stats, ['DATE','COMMENT','HISTORY','EXTVER','CREATOR']
          
          ; If the caller did NOT specify an obsname, then remove all keywords that use single-ObsID coordinate systems so that the caller cannot make the mistake of assuming that AE is reporting positions in some project-level SKY system.
          if ~keyword_set(obsname) then sxdelpar, obs_stats, ['X_CAT','X_PSF','X_DATA','EX_DATA','CAT2DATA', $
                                                              'Y_CAT','Y_PSF','Y_DATA','EY_DATA']
  
          
          ; There can be duplicate keywords in the source header (stats) and the observation header (obs_stats).
          ; Precedence is governed by the /SINGLE_OBSID option.
          kywds = strcompress(strmid(obs_stats,0,8), /REMOVE_ALL)
          
          for jj=0,n_elements(kywds)-1 do begin
            kywd = kywds[jj]
            
            if (kywd EQ '')    then continue
            if (kywd EQ 'END') then break
            
            dum = psb_xpar( stats, kywd, COUNT=count)
            if ((count EQ 0) OR keyword_set(single_obsid)) then begin
              ; There's no keyword name conflict between obs.stats and source.stats, 
              ; or the observer specified that obs.stats takes precedence, so
              ; use the data from obs.stats.
              val = psb_xpar( obs_stats, kywd, COMMENT=comment)
              psb_xaddpar, stats, kywd, val, comment
            endif
          endfor ;jj
        endif else begin
          if (pass EQ 2) AND (verbose GT 0) then print,  'INFORMATION: '+src_label+' not extracted in observation '+primary_obsname[ii]
          ; Mark this source as unobserved so we can skip some code later.
          primary_obs_dir[ii] = ''
        endelse
        
      endif else begin
        ; ~(primary_count EQ 1)
        ; Called did not specify an ObsID, and PRIM_OBS not found.
        if (pass EQ 2) AND (verbose GT 0) then print, 'WARNING! '+src_label+': no "obsname" parameter supplied, and cannot find kywd PRIM_OBS in '+src_stats_basename
        
        display_obsname[ii] = ''
        
      endelse
      
      ; Build the path to the display observation, using any extraction name supplied
      display_obs_dir[ii]   = sourcename[ii] + '/' + display_obsname[ii] + '/' + extraction_subdir[ii]
  
    endif ;read_obsid_stats
    
    
    
    ;; ------------------------------------------------------------------------
    ; Extract all the keywords we have gathered above from source.stats and obs.stats files.
    kywds = strcompress(strmid(stats,0,8), /REMOVE_ALL)
    
    for jj=0,n_elements(kywds)-1 do begin
      kywd = kywds[jj]
      
      if (kywd EQ '')    then continue
      if (kywd EQ 'END') then break
      
      val               = psb_xpar( stats, kywd, COMMENT=comment)
      this_type         = size(val,/TYPE)
      this_num_elements = size(val,/N_ELEMENTS)

      name     = repchr(kywd, '-', '_')
      this_col = (where(name EQ col_names, count))[0]        

      case pass of 
       1: begin
          ; We're defining the output file columns.
          ; Have we seen this column name before?
          if (count EQ 0) then begin
            ; Add a new column.
            col_names   [num_cols] = name
            col_types   [num_cols] = this_type
            col_elements[num_cols] = this_num_elements
            col_comments[num_cols] = comment
            num_cols = num_cols + 1
          endif else begin
            ; Make sure column type can represent this source's value.
            if (col_types[this_col] EQ 7) OR (this_type EQ 7) then begin 
              ;STRING (type 7) can represent anything
              col_types[this_col] = 7
            endif else begin
              ; DOUBLE (type 5) is "highest" numerical value we allow.
              col_types[this_col] = (col_types[this_col] > this_type) < 5
            endelse
            
            col_elements[this_col] = col_elements[this_col] > this_num_elements
            col_comments[this_col] = comment
          endelse
          end
          
       2: begin
          ; If MATCH_EXISTING was used to define the table columns, then there may not be one for the keyword in hand.
          if (count GT 0) then begin
            ; Extract the data.
            ; Without the [0] construct on the left side IDL will replicate a scalar on the right
            ; to fill all the elements of a vector on the left.
            temp    = bin_table[ii].(this_col)
            temp[0] = val
            bin_table[ii].(this_col) = temp
          endif
          end
      endcase
    endfor ;jj


    ;; When we're interested in single-obsid data, skip the photometry and fitting data products.
    if keyword_set(single_obsid) || (item_path EQ '/dev/') then GOTO, COLLATE_LOOPEND
    

    
    ;; ------------------------------------------------------------------------
    if read_photometry then begin    
      ;; Scan the source.photometry file for binary table column names.
      ;; "Model" photometry (e.g. from pile-up recon tool) takes precidence over AE photometry.
      
      photometry_fn    = sourcedir + model_photometry_basename
      if ~file_test(photometry_fn) then $
        photometry_fn  = sourcedir + src_photometry_basename
      
      if file_test(photometry_fn) then begin
        photometry_table  = mrdfits(photometry_fn, 1, photometry_hdr, /SILENT, STATUS=status)
      endif else status = 1
      
      if (status EQ 0) then begin
        flux_row          = photometry_table[0]
        this_num_elements = n_elements(photometry_table)
        tag_names = tag_names(photometry_table)
        for jj=0,n_elements(tag_names)-1 do begin
          val       = flux_row.(jj)
          
          ; Enclose unit specifications in square brackets (the convention that dmhedit uses) to match the format of FITS keywords in source.stats, obs.stats, and source.spectra files.
          comment   = '[' + strtrim(psb_xpar( photometry_hdr, string(1+jj,F='(%"TUNIT%d")')),2) + ']'
          dum       =               psb_xpar( photometry_hdr, string(1+jj,F='(%"TTYPE%d")'), COMMENT=extra_comment)
          if (extra_comment NE '') then comment += ' ' + extra_comment
          
          this_type = size(val,/TYPE)
  
          name     = tag_names[jj]
          this_col = where(name EQ col_names, count)
          
          case pass of 
           1: begin
              ; We're defining the output file columns.
              ; Have we seen this column name before?
              if (count EQ 0) then begin
                ; Add a new column.
                col_names   [num_cols] = name
                col_types   [num_cols] = this_type
                col_elements[num_cols] = this_num_elements
                col_comments[num_cols] = comment
                num_cols = num_cols + 1
              endif else begin
                ; Make sure column type can represent this source's value.
                if (col_types[this_col] EQ 7) OR (this_type EQ 7) then begin 
                  ;STRING (type 7) can represent anything
                  col_types[this_col] = 7
                endif else begin
                  ; DOUBLE (type 5) is "highest" numerical value we allow.
                  col_types[this_col] = (col_types[this_col] > this_type) < 5
                endelse
                
                col_elements[this_col] = col_elements[this_col] > this_num_elements
                col_comments[this_col] = comment
              endelse
  
              end
              
           2: begin
              ; If MATCH_EXISTING was used to define the table columns, then there may not be one for the keyword in hand.
              if (count GT 0) then begin
                ; Extract the data.
                ; Without the [0] construct on the left side IDL will replicate a scalar on the right
                ; to fill all the elements of a vector on the left.
                temp    = bin_table[ii].(this_col)
                temp[0] = photometry_table.(jj)
                bin_table[ii].(this_col)   = temp
                bin_table[ii].PHOT_CREATOR = psb_xpar( photometry_hdr, 'CREATOR')
              endif
              end
          endcase
        endfor ;jj
      endif ;status
    endif; read_photometry
    
    

    ;; ------------------------------------------------------------------------
    ; Find the desired HDU in spectral fit results file.
    if read_spectral_fit then begin
      fit_stats_fn = sourcedir + fit_stats_basename    
  
      ; Note that get_keywords_from_hdu_headers() will return the value 0B for fields of type BYTE when the corresponding FITS keyword is missing.
      fit_stats_keywords = get_keywords_from_hdu_headers(fit_stats_fn, {BEST_MDL:'', PROVISNL:0B, NOCOLLAT:0B, CATEGORY:'', HDUNAME:''}, ERRMSG=error)
  
      if ~keyword_set(error) then begin
        
        if (pass EQ 2) then begin
          bin_table[ii].NMODELS  = n_elements(fit_stats_keywords) - 1
          bin_table[ii].BEST_MDL = fit_stats_keywords[0].BEST_MDL
          bin_table[ii].PROVISNL = fit_stats_keywords[0].PROVISNL
          bin_table[ii].NOCOLLAT = fit_stats_keywords[0].NOCOLLAT
          bin_table[ii].CATEGORY = fit_stats_keywords[0].CATEGORY
        endif
        
        if fit_stats_keywords[0].NOCOLLAT then begin
          if (pass EQ 2) then begin
            bin_table[ii].MODEL    = 'withheld'
            if (verbose GT 0) then print, 'WARNING! '+src_label+': Fit results explicitly withheld.'
          endif
          continue
        endif
    
        this_hduname = hduname[ii]
        
        if (this_hduname EQ 'BEST_MDL') then begin
          ; Look for preferences stored in Primary HDU.
          this_hduname = fit_stats_keywords[0].BEST_MDL
          
          if keyword_set(this_hduname) then begin
          endif else begin
            this_hduname = ''
            if (pass EQ 2) then print, 'WARNING! '+src_label+': FITS keyword BEST_MDL was not found. '
          endelse
        endif
    
        if (this_hduname EQ '') || (this_hduname EQ '...') then begin
          ; Use the last HDU in the file.
          ind   = n_elements(fit_stats_keywords) - 1
          count = 1
          if (pass EQ 2) AND (verbose GT 0) then print, src_label, fit_stats_keywords[ind].HDUNAME, F='(%"%s: using the most recent spectral model: %s")'
        endif else begin
          ; Find the last HDU matching the specified model name spec.
          ind = where( strmatch(fit_stats_keywords.HDUNAME, this_hduname, /FOLD_CASE), count )
          
          if (count GT 1) then begin
            ind = ind[count-1]
            if (pass EQ 2) AND (verbose GT 0) then print, src_label, count, this_hduname, fit_stats_keywords[ind].HDUNAME,  F='(%"%s: %d spectral models match %s; using the most recent: %s )")'
            count = 1
          endif
        endelse
        
        if (count EQ 0) then begin
          if (pass EQ 2) AND (verbose GT 0) then print, src_label, this_hduname, F='(%"WARNING! %s: Cannot find any spectral model matching ''%s''")'
        endif else begin
          stats = headfits(fit_stats_fn, EXTEN=ind[0], ERRMSG=error)
    
          if (keyword_set(error)) then begin 
            if (pass EQ 2) then print, 'WARNING! Error reading '+fit_stats_fn
          endif else begin
            if (pass EQ 2) then begin
              bin_table[ii].MODEL   = fit_stats_keywords[ind].HDUNAME
            endif
            
            fxhclean, stats
            sxdelpar, stats, ['DATE','COMMENT','HISTORY','EXTVER','CREATOR','EXTNAME']
              
            kywds = strcompress(strmid(stats,0,8), /REMOVE_ALL)
              
            for jj=0,n_elements(kywds)-1 do begin
              kywd = kywds[jj]
              
              if (kywd EQ '')    then continue
              if (kywd EQ 'END') then break
              
              val               = psb_xpar( stats, kywd, COMMENT=comment)
              this_type         = size(val,/TYPE)
              this_num_elements = size(val,/N_ELEMENTS)
      
              name     = repchr(kywd, '-', '_')
              this_col = where(name EQ col_names, count)        
      
              case pass of 
               1: begin
                  ; We're defining the output file columns.
                  ; Have we seen this column name before?
                  if (count EQ 0) then begin
                    ; Add a new column.
                    col_names   [num_cols] = name
                    col_types   [num_cols] = this_type
                    col_elements[num_cols] = this_num_elements
                    col_comments[num_cols] = comment
                    num_cols = num_cols + 1
                  endif else begin
                    ; Make sure column type can represent this source's value.
                    if (col_types[this_col] EQ 7) OR (this_type EQ 7) then begin 
                      ;STRING (type 7) can represent anything
                      col_types[this_col] = 7
                    endif else begin
                      ; DOUBLE (type 5) is "highest" numerical value we allow.
                      col_types[this_col] = (col_types[this_col] > this_type) < 5
                    endelse
                    
                    col_elements[this_col] = col_elements[this_col] > this_num_elements
                    col_comments[this_col] = comment
                  endelse
                  end
                  
               2: begin
                ; If MATCH_EXISTING was used to define the table columns, then there may not be one for the keyword in hand.
                if (count GT 0) then begin
                    ; Extract the data.
                    ; Without the [0] construct on the left side IDL will replicate a scalar on the right
                    ; to fill all the elements of a vector on the left.
                    temp    = bin_table[ii].(this_col)
                    temp[0] = val
                    bin_table[ii].(this_col) = temp
                  endif
                  end
              endcase
            endfor ;jj
          endelse ;(no error)
        endelse ;(count GT 0)
      endif ; fit data file read
    endif ;read_spectral_fit

COLLATE_LOOPEND:    
  endfor ;ii (loop over sources)


  ;; ------------------------------------------------------------------------
  ;; Compute distances between source locations.
  if (item_path NE '/dev/') then begin
    case pass of
     1: begin
        ;; Add the distance_src2src column computed here.
        col_names   [num_cols] = 'distance_src2src'
        col_comments[num_cols] = '[skypixel] distance between sources'
        num_cols = num_cols + 1

        ;; Add the distance_reg2reg column computed here.
        col_names   [num_cols] = 'distance_reg2reg'
        col_comments[num_cols] = '[skypixel] ~distance between source apertures'
        num_cols = num_cols + 1

        ;; Add the neighbor column computed here.
        col_names   [num_cols] = 'neighbor'
        col_comments[num_cols] = '0-based index of nearest neighbor'
        col_types   [num_cols] = 3 ; LONG type
        num_cols = num_cols + 1
        end
      
     2: begin
        dum = where(col_names EQ 'SRC_RAD', count)
        if (count GT 0) then begin
          if (verbose GT 0) then print, 'computing distances between sources'
          ;; For each source aperture, find which source has a region most overlapping.
          ;; Note the the "neighbor" relationship is NOT symmetric, e.g. A's most overlapping neighbor may be B, and B's most overlapping neighbor may be C!
          src_radius       = bin_table.SRC_RAD
          distance_src2src = fltarr(num_sources)
          distance_reg2reg = fltarr(num_sources)
          neighbor         = lonarr(num_sources) ;0-based index of nearest neighbor
          deghour          = 24D/360
          ra_hrs           = bin_table.RA*deghour
          dec_degrees      = bin_table.DEC
          for ii = 0L, num_sources-1 do begin
            ; Compute distances between this source and neighbors in units of skypix.
            ; Earlier AE versions avoided a loop in the distance computation, but for large catalogs the IDL process exceeded its 4GB memory limit.
            gcirc, 1, ra_hrs, dec_degrees,  ra_hrs[ii], dec_degrees[ii],  distance_to_neighbors ; distance in arcsec
            distance_to_neighbors    /= arcsec_per_skypixel                                     ; convert to skypix
            distance_to_neighbors[ii] = !VALUES.F_NAN                                           ; ignore yourself
          
            distance_src2src[ii] = min(/NAN, distance_to_neighbors)
            distance_reg2reg[ii] = min(/NAN, distance_to_neighbors - src_radius - src_radius[ii], ind)
            ; If the distance_reg2reg vector is all NaNs, then we'll declare the source to be its own neighbor.
            ; Adopting a "no neighbor exists" flag value (e.g. -1) would be more logical, but would require trapping that exception everywhere the neighbor tag is referenced.
            neighbor        [ii] = finite(distance_reg2reg[ii]) ? ind : ii
          endfor   
          
          bin_table.distance_src2src = distance_src2src
          bin_table.distance_reg2reg = distance_reg2reg
          bin_table.neighbor         = neighbor
        endif ;'SRC_RAD' available
        end
    endcase
  endif ; (item_path NE '/dev/')
    
      
  ;; ------------------------------------------------------------------------
  ;; Construct structure array.
  if (pass EQ 1) then begin      
    ;; Construct the empty output binary table.
    format_codes = ['B','B','I','J','F','D','C','A']
    repeat_count = string(col_elements[0:num_cols-1], F='(I0)')
    ind = where(repeat_count EQ '1', count)
    if (count GT 0) then repeat_count[ind] = ''
    
    tag_descript = strjoin(repeat_count + format_codes[col_types], ',')
    
    ; The create_struct routine writes a temp file; to avoid conflict we run it from our temp dir.
    pushd, tempdir
    create_struct, bin_table, '', col_names[0:num_cols-1], tag_descript, DIMEN=num_sources
    popd
    
    if (num_cols NE n_tags(bin_table)) then begin
      print, 'ERROR: the AstroLib routine create_struct.pro did not return the requested structure.'
      GOTO, FAILURE
    endif
  endif ;(pass EQ 1)
 endfor ;pass loop

 bin_table.OBSDIR       = primary_obs_dir


  ;; ------------------------------------------------------------------------
  ;; Write an empty primary HDU.
  fxhmake, pheader, /EXTEND, /DATE, /INITIALIZE
  psb_xaddpar, pheader, 'CREATOR', creator_string  

  psb_xaddpar, pheader, "FNFITS", item_name
  
  if (item_path NE '/dev/') then writefits, collated_filename, 0, pheader

  
  ;; ------------------------------------------------------------------------
  ;; Now finally we can write out this giant structure array.
  if (start_pass EQ 1) then begin
    ; Pass 1 was executed, and we have TTYPE and TUNIT information to write to the FITS header.
    fxbhmake, theader, num_sources, 'EXTRACTION RESULTS', /DATE
    
    ; Store the description for each column (col_comments) in the comment part of its TTYPE keyword.
    ; Parse those descriptions for unit specifications, using the bracket convention that dmhedit uses, e.g. "[arcsec]".
    for jj = 0, num_cols-1 do begin
      this = col_comments[jj]
      token_start = stregex(this, '\[([^]]*)\]', /SUB, LENGTH=token_length)
      this_unit_spec   =         strmid(this, token_start[1], token_length[1])
      this_col_comment = strtrim(strmid(this, token_start[0]+token_length[0]),2)
      psb_xaddpar, theader, string(1+jj,F='(%"TTYPE%d")'), col_names[jj], this_col_comment
      psb_xaddpar, theader, string(1+jj,F='(%"TUNIT%d")'), this_unit_spec
    endfor
  endif

  get_date, date_today, /TIMETAG
  psb_xaddpar, theader, 'DATE', date_today
  psb_xaddpar, theader, 'CREATOR', creator_string  
  
  if (item_path NE '/dev/') then mwrfits, bin_table, collated_filename, theader
  
  if tag_exist(bin_table, 'OBJECT') then begin
    object = strtrim(bin_table.OBJECT,2)
    ind = where((object NE '') AND (object NE '0') AND (sourcename NE object), count)
    if (count GT 0) then begin
      print, 'WARNING!  The following catalog source names (left) do not match the OBJECT keyword (right).'
      forprint, sourcename[ind], bin_table[ind].OBJECT, F='A,4x,A'
    endif
  endif

  
  ;; ------------------------------------------------------------------------
  ;; Write a table of source names and labels.
  LABEL = strtrim(bin_table.LABEL,2)
  if keyword_set(label_filename) then begin
    forprint, TEXTOUT=label_filename, /NoCOMMENT, sourcename, LABEL, SUBSET=sort(LABEL), F='(%"%s ; ''%s''")'
  endif

  if (num_sources GT 1) then begin
    ;; Look for duplicate source labels.
    sort_index    = sort(LABEL)
    sorted_labels = LABEL[sort_index]
    
    uniq_index = uniq(sorted_labels)
    num_uniq   = n_elements(uniq_index)
    if (num_uniq LT num_sources) then begin
      print, F='(%"\n=============================================================================")'
      print, 'WARNING: The catalog contains duplicate source labels:'
    
      next = 0
      for ii=0L,num_uniq-1 do begin
        this = uniq_index[ii]
        if (this GT next) then begin
          print
          for jj=next,this do begin
            duplicate_index = sort_index[jj]
            print, sourcename[duplicate_index], LABEL[duplicate_index], F='(%"%s (%s)")'
          endfor
        endif ; (this GT next)
        
        next = this+1
      endfor
      
      print, '============================================================================='
    endif ;duplicates
  endif
  
  
  ;; ------------------------------------------------------------------------
  ;; Collate ds9 regions across the catalog.
  if keyword_set(region_file) then begin
  
    ; Handle the optional input REGION_TAG.
    if (n_elements(region_tag) GT 0) then begin
      case size(region_tag, /N_DIM) of
        0:    formatted_region_tag = 'tag={'+region_tag+'}'
        
        1:    formatted_region_tag = 'tag={'+region_tag+'}'
              
        2:    begin
              ; Concatenate elements along the second dimension.
              num_rows = (size(region_tag, /DIM))[0]
              
              formatted_region_tag = strarr(num_rows)
              for ii = 0,num_rows-1 do formatted_region_tag[ii] = strjoin( 'tag={'+region_tag[ii,*]+'}', ' ', /SINGLE)
              end
        
        else: begin
              print, 'ERROR: parameter REGION_TAG cannot have more than two dimensions.'
              GOTO, FAILURE
              end
      endcase
    endif ; REGION_TAG was passed
    
    case n_elements(formatted_region_tag) of
      0:           user_tag_string = strarr(num_sources)
      1:           user_tag_string = replicate(formatted_region_tag,num_sources)
      num_sources: user_tag_string =           formatted_region_tag
      else: begin
            print, 'ERROR: parameter REGION_TAG must be a scaler or a vector as long as the catalog'
            GOTO, FAILURE
            end
    endcase
;help, user_tag_string
;forprint, user_tag_string
    
    
;   if tag_exist(bin_table, 'PROVENAN') then begin
;     user_tag_string = 'tag={PROVENAN:'+strtrim(bin_table.PROVENAN,2)+ '} ' + user_tag_string
;   endif
    
    piled_tag   = strarr(num_sources)
    piled_label = strarr(num_sources)
    if tag_exist(bin_table, 'RATE_3x3') then begin
      ind = where(bin_table.RATE_3x3 GT 0.05, count)  ; See corresponding threshold in ae_pileup_screening tool.
      if (count GT 0) then begin
        piled_tag  [ind] = 'tag={piled}' ; width=4 (abandoned)
        piled_label[ind] = ' (PILED)'
      endif
    endif
    
    if (verbose GT 0) then print, 'Building region file ...'
    color = replicate('DodgerBlue', num_sources)
    openw,  region1_unit, region_file, /GET_LUN
    printf, region1_unit, "# Region file format: DS9 version 3.0"
    printf, region1_unit, 'global width=1 font="helvetica 12 normal" dashlist=2 4'
    printf, region1_unit, F='(%"# Catalog (crosses), Data Mean (diamonds), Correlation Peak (circles), and Reconstruction Peak (boxes) positions")'
    
    !TEXTUNIT = region1_unit
    
   
    ; Catalog positions.
    ind = where(finite(bin_table.RA) AND finite(bin_table.DEC), count)
    if (count GT 0) then $
      forprint, TEXTOUT=5, /NoCOM, SUBSET=ind, bin_table.RA, bin_table.DEC, LABEL+piled_label, user_tag_string, color, F='(%"J2000;cross   point %10.6f %10.6f # text={%s} tag={cat} %s color=%s")'
    
    dec_offset = 1 / 3600.   ; 1 arcsec
    ra_offset  = dec_offset / cos(bin_table.DEC*!DTOR)
    
    ; Labels.
    ;forprint, TEXTOUT=5, /NoCOM, bin_table.RA+ra_offset, bin_table.DEC-dec_offset, LABEL, user_tag_string, color, F='(%"J2000;text %10.6f %10.6f # text={%s} tag={label} %s color=%s")'
      
    ; PSF fractions, using a boxcircle symbol of size zero
    if tag_exist(bin_table, 'PSF_FRAC') then begin
      ind = where(finite(bin_table.PSF_FRAC), count)
      if (count GT 0) then $
        forprint, TEXTOUT=5, /NoCOM, SUBSET=ind, bin_table.RA+ra_offset, bin_table.DEC, 100*bin_table.PSF_FRAC, user_tag_string, color, F='(%"J2000;point %10.6f %10.6f # point=boxcircle 0 text={%d%%} tag={fraction} %s color=%s")' 
    endif
 
    ; Data positions.
    if tag_exist(bin_table, 'RA_DATA') && tag_exist(bin_table, 'DEC_DATA') then begin
      ind = where(finite(bin_table.RA_DATA) AND finite(bin_table.DEC_DATA), count)
      if (count GT 0) then $
        forprint, TEXTOUT=5, /NoCOM, SUBSET=ind, bin_table.RA_DATA, bin_table.DEC_DATA, user_tag_string, color, F='(%"J2000;diamond point %10.6f %10.6f # tag={data} %s color=%s")'
    endif
    
    ; Correlation positions.
    if tag_exist(bin_table, 'RA_CORR') && tag_exist(bin_table, 'DEC_CORR') then begin
      ind = where(finite(bin_table.RA_CORR) AND finite(bin_table.DEC_CORR), count)
      if (count GT 0) then $
        forprint, TEXTOUT=5, /NoCOM, SUBSET=ind, bin_table.RA_CORR, bin_table.DEC_CORR, user_tag_string, color, F='(%"J2000;circle  point %10.6f %10.6f # tag={corr} %s color=%s")'
    endif
    
    ; ML positions.
    if tag_exist(bin_table, 'RA_ML') && tag_exist(bin_table, 'DEC_ML') then begin
      ind = where(finite(bin_table.RA_ML) AND finite(bin_table.DEC_ML), count)
      if (count GT 0) then $
        forprint, TEXTOUT=5, /NoCOM, SUBSET=ind, bin_table.RA_ML, bin_table.DEC_ML, user_tag_string, color,     F='(%"J2000;box     point %10.6f %10.6f # tag={ml} %s color=%s")'
    endif
    
    
    ; Polygons
    for ii = 0L, num_sources-1 do begin
      
      src_events_fn = display_obs_dir[ii] + src_events_basename
      region_fn     = display_obs_dir[ii] + src_region_basename
      
      if ~file_test(src_events_fn) || ~file_test(region_fn) then continue
      
      ; Build astrometic structure from event data header.
      theader = headfits( src_events_fn, EXT=1, ERRMSG=error )
      if (keyword_set(error)) then begin
        print, error
        message, 'ERROR reading ' + src_events_fn
      endif
      
      fxbfind, theader, 'TTYPE', dum1, TTYPE, dum2, 'null'
      fxbfind, theader, 'TCTYP', dum1, TCTYP, dum2, 'null'
      fxbfind, theader, 'TCRVL', dum1, TCRVL, dum2, 0.0D
      fxbfind, theader, 'TCRPX', dum1, TCRPX, dum2, 0.0D
      fxbfind, theader, 'TCDLT', dum1, TCDLT, dum2, 0.0D
      colnames = strlowcase( strtrim(TTYPE,2) )
      x_ind    = where(strlowcase(colnames) EQ 'x')
      y_ind    = where(strlowcase(colnames) EQ 'y')
      make_astr, event2wcs_astr, DELTA=TCDLT[[x_ind,y_ind]], CTYPE=TCTYP[[x_ind,y_ind]], $
                                 CRPIX=TCRPX[[x_ind,y_ind]], CRVAL=TCRVL[[x_ind,y_ind]]
  
      ;; Convert polygon to WCS & write to the region file.      
      ae_ds9_to_ciao_regionfile, region_fn, '/dev/null', /IGNORE_BACKGROUND_TAG, POLYGON_X=polygon_x, POLYGON_Y=polygon_y
      
      if (n_elements(polygon_x) GT 1) then begin
        ;; REMEMBER THAT THE xy2ad and ad2xy programs assume that (x,y) are 
        ;; ZERO-BASED pixel indexes.  Thus we must subtract 1 from the sky (x,y) 
        ;; positions when converting to RA,DEC.
        xy2ad, polygon_x-1, polygon_y-1, event2wcs_astr, polygon_ra, polygon_dec
        
        polygon = dblarr(2,n_elements(polygon_ra))
        polygon[0,*] = polygon_ra
        polygon[1,*] = polygon_dec
    
        src_region = 'polygon(' + strcompress(strjoin(string(polygon,F='(F10.6)'),","), /REMOVE) + ')'
        
        formatted_label = (tag_exist(bin_table, 'DIFFUSE') && bin_table[ii].DIFFUSE) ? string(LABEL[ii],F='(%"text={%s}")') : ''

        printf, region1_unit, src_region, display_obsname[ii], user_tag_string[ii], piled_tag[ii], color[ii], formatted_label, F='(%"J2000;%s # tag={obs%s} %s %s color=%s %s")' 
      endif
    endfor ; ii loop
    
    free_lun, region1_unit
  endif

  if (verbose GT 0) then begin
    print, '============================================================================='
    if (item_path NE '/dev/') then begin
      print, 'Wrote source properties to ', collated_filename
      print, 'Column names are: ', tag_names(bin_table)
    endif
    if keyword_set(region_file) then print, 'Wrote source apertures to ', region_file
    print, '============================================================================='
  endif


  GOTO, CLEANUP
endif ;keyword_set(collated_filename)



;; =============================================================================
if keyword_set(plot) AND keyword_set(construct_regions) then begin
;; =============================================================================

  if ~keyword_set(obsname) then begin
    print, F="(%'\nERROR!  An ""obsname"" parameter must be supplied.')"
    GOTO, FAILURE
  endif
  
  ;; Read the source properties of all the sources.
  if ~keyword_set(collated_filename) || ~file_test(collated_filename) then begin
    print, 'Collating source properties ...'
    if ~keyword_set(catalog_or_srclist) then begin
      print, collated_filename, collated_filename, F='(%"ERROR: The collation %s does not exist.  Supply the source list file, or build %s yourself.")'
      GOTO, FAILURE      
    endif
    collated_filename = temp_collate_fn
    acis_extract, catalog_or_srclist, obsname, /SINGLE_OBSID, COLLATED_FILENAME=collated_filename, VERBOSE=0
  endif
  
  ;; Read the source properties of all the sources.
  bt = mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status)
  if (status NE 0) then begin
    print, 'ERROR reading ', collated_filename
    GOTO, FAILURE      
  endif
  
  if tag_exist(bt, 'DIFFUSE') then begin
    print, F="(%'\nWARNING!  No CONSTRUCT_REGIONS plots are available for diffuse extractions.')"
    GOTO, FAILURE
  endif

  obsname_in_collation = strtrim(bt.obsname,2)
  if (total((obsname_in_collation NE '') AND (obsname_in_collation NE obsname)) GT 0) then begin
    print, collated_filename, obsname, F="(%'\nERROR!  Not all collations in %s are for the ObsId specified (%s).')"
    GOTO, FAILURE
  endif

  num_sources = n_elements(bt)
  
  sourcename           = bt.CATALOG_NAME
  label                = bt.LABEL
  xpos_catalog         = bt.X_CAT
  ypos_catalog         = bt.Y_CAT
  off_angle            = bt.THETA
  psf2cat_offset       = bt.PSF2CAT
  target_psf_fraction  = bt.FRACSPEC
  psf_fraction         = bt.PSF_FRAC
  src_radius           = bt.SRC_RAD
  src_area             = bt.PGN_AREA
  src_area_type        = 'via polyfillv (IDL)'
  
  distance_src2src     = bt.distance_src2src
  neighbor             = bt.neighbor
  distance_reg2reg     = bt.distance_reg2reg

  f_nan = replicate(!VALUES.F_NAN,num_sources)
  cropfrac277           = f_nan
  cropfrac15            = f_nan
  skypixel_per_psfpixel = f_nan
  
  ;; Read summary information for each source.
  print, 'reading source information'
  for ii = 0L, num_sources-1 do begin
    psf_fn    = sourcename[ii] + '/' + obsname + '/' + psf_basename
    
    header = headfits(psf_fn, ERRMSG=error, EXT=0)
    if (NOT keyword_set(error)) then begin
      skypixel_per_psfpixel[ii] = psb_xpar( header, 'CDELT1P')
    endif

    header = headfits(psf_fn, ERRMSG=error, EXT=1)
    if (NOT keyword_set(error)) then cropfrac277[ii] = psb_xpar( header, 'CROPFRAC')

    header = headfits(psf_fn, ERRMSG=error, EXT=2)
    if (NOT keyword_set(error)) then cropfrac15 [ii] = psb_xpar( header, 'CROPFRAC')
 endfor ;ii
      

  ;; Show summary information.
  tit = 'Extraction Regions for dataset "' + obsname + '"'
  dataset_2d, id1, NAN=[-0.5,0], PSY=1, off_angle, alog10(src_area),   TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='log Aperture Area (skypixel**2) via polyfillv (IDL)', PS_CONFIG={filename:output_dir+'aperturearea_vs_theta.ps'}, /PRINT

  dataset_2d, id6, NAN=[-0.5,0], PSY=1, off_angle, psf2cat_offset, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='Offset between catalog position and PSF centroid (skypixel)', PS_CONFIG={filename:output_dir+'psfoffset_vs_theta.ps'}, /PRINT
  
  
  sn = indgen(num_sources)
;  function_1d, id5, LI=6,PSY=4, sn, psf_fraction, DATASET='Actual', TITLE=tit, XTIT='Source # (0-based)', YTIT='PSF Fraction' 
;  function_1d, id5, LI=6,PSY=1, sn, target_psf_fraction, DATASET='Requested'

  dataset_2d, id8, NAN=[-0.5,0], PSY=1, off_angle, psf_fraction, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='PSF Fraction', PS_CONFIG={filename:output_dir+'psffrac_vs_theta.ps'}, /PRINT
  
;  function_1d, id7, LI=6,PSY=1, off_angle, distance_reg2reg, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='~Distance Between Source Apertures (skypixel)'

;  x=f_nan
;  y=f_nan
;  ind = where(distance_reg2reg LT 2, count)
;  if (count GT 0) then begin
;    x[ind] = sn[ind]
;    y[ind] = neighbor[ind]
;    dataset_2d, id8, NAN=[-0.5,0], PSY=1, x, y, TITLE='Pairs of Crowded Sources', XTIT='Source # (0-based)', YTIT='Source # (0-based)'
;  endif

  tit = 'Summary of PSF Images'
  dataset_2d, id2, NAN=[-0.5,0], PSY=1, off_angle, cropfrac277, DATASET='277 eV', TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='CROPFRAC'
      
  dataset_2d, id2, NAN=[-0.5,0], PSY=6, off_angle, cropfrac15  , DATASET='1.5 keV', PS_CONFIG={filename:output_dir+'cropfrac_vs_theta.ps'}, /PRINT

  dataset_2d, id3, NAN=[-0.5,0], PSY=1, off_angle, skypixel_per_psfpixel, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='PSF pixel size (skypixel)', PS_CONFIG={filename:output_dir+'psf_pixsize_vs_theta.ps'}, /PRINT

  
;  savefile = 'construct_regions_'+obsname+'.sav'
;  save, /COMPRESS, sourcename, xpos_catalog, ypos_catalog, off_angle, psf2cat_offset, target_psf_fraction, psf_fraction, src_radius, src_area, distance_src2src, distance_reg2reg, neighbor, FILE=savefile
;
;  print, '============================================================================='
;  print, 'IDL vectors saved in ', savefile
;  print, '============================================================================='
  GOTO, CLEANUP
endif ; /CONSTRUCT_REGIONS,   /PLOT


;; =============================================================================
if keyword_set(plot) AND keyword_set(check_positions) then begin
;; =============================================================================

  ;; Read the source properties of all the sources.
  if ~keyword_set(collated_filename) || ~file_test(collated_filename) then begin
    print, 'Collating source properties ...'
    if ~keyword_set(catalog_or_srclist) then begin
      print, collated_filename, collated_filename, F='(%"ERROR: The collation %s does not exist.  Supply the source list file, or build %s yourself.")'
      GOTO, FAILURE      
    endif
    collated_filename = temp_collate_fn
    acis_extract, catalog_or_srclist, (keyword_set(obsname) ? obsname : ''),  MERGE_NAME=(keyword_set(merge_name) ? merge_name : ''), COLLATED_FILENAME=collated_filename
  endif
  
  ;; Read the source properties of all the sources.
  bt = mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status)
  if (status NE 0) then begin
    print, 'ERROR reading ', collated_filename
    GOTO, FAILURE      
  endif
  
  num_sources = n_elements(bt)
  
  sourcename       =                            bt.CATALOG_NAME
  merge_subdir     =                            bt.MERGE_NAME
  ra               =                            bt.RA
  dec              =                            bt.DEC
  off_angle        =                            bt.THETA
  ra_ml            = tag_exist(bt,'RA_ML')    ? bt.RA_ML    : intarr(num_sources)
  dec_ml           = tag_exist(bt,'DEC_ML')   ? bt.DEC_ML   : intarr(num_sources)
  ra_corr          = tag_exist(bt,'RA_CORR')  ? bt.RA_CORR  : intarr(num_sources)
  dec_corr         = tag_exist(bt,'DEC_CORR') ? bt.DEC_CORR : intarr(num_sources)
  quantization_corr= tag_exist(bt,'QUANTCOR') ? bt.QUANTCOR : intarr(num_sources)
  ra_data          =                            bt.RA_DATA
  dec_data         =                            bt.DEC_DATA
  src_radius       =                            bt.SRC_RAD
  src_counts       =                            bt.SRC_CNTS[0]
      
  ;; Compute distance between catalog, correlation, & data positions.
  deghour = 24D/360
  gcirc, 1, ra*deghour,      dec,      ra_corr*deghour, dec_corr, cat2corr_offset
  gcirc, 1, ra*deghour,      dec,      ra_data*deghour, dec_data, cat2data_offset
  gcirc, 1, ra_data*deghour, dec_data, ra_corr*deghour, dec_corr, corr2data_offset
  
  ; The gcirc routine doesn't propagate NaN's, so we have to deal with them explicitly.
  ; We also have to detect missing (zero) coordinates.
  ind = where((finite(ra) AND (ra GT 0)) EQ 0, count)
  if (count GT 0) then begin
    cat2corr_offset [ind] = !VALUES.F_NAN
    cat2data_offset [ind] = !VALUES.F_NAN
  endif
  ind = where((finite(ra_data) AND (ra_data GT 0)) EQ 0, count)
  if (count GT 0) then begin
    corr2data_offset[ind] = !VALUES.F_NAN
    cat2data_offset [ind] = !VALUES.F_NAN
  endif
  ind = where((finite(ra_corr) AND (ra_corr GT 0)) EQ 0, count)
  if (count GT 0) then begin
    cat2corr_offset [ind] = !VALUES.F_NAN
    corr2data_offset[ind] = !VALUES.F_NAN
  endif
  
  ; Mark sources where no DATA position available (SRC_CNTS==0).
  ind = where(src_counts EQ 0, count)
  if (count GT 0) then begin
    cat2corr_offset [ind]=-1
    cat2data_offset [ind]=-1
    corr2data_offset[ind]=-1
  endif
      
  ;; Compute sky positions for an arbitrary sky (x,y) system with 
  ;; 0.5" pixels centered on the mean source position.
  make_astr, composite_astr, DELTA=[-0.000136667D,0.000136667D],  $
                             CRPIX=[0D,0D], CRVAL=[mean(ra,  /DOUBLE),mean(dec, /DOUBLE)]

  ad2xy, ra,      dec,      composite_astr, xpos_catalog, ypos_catalog
  ad2xy, ra_corr, dec_corr, composite_astr, xpos_corr,    ypos_corr
  
  ;; Show summary information.
  color_manager
  ;; ------------------------------------------------------------------------
  ; It's important to make the partvelvec plot have unity aspect.
  ; Estimate the plot region size in device units.
    xrange = minmax(xpos_catalog)
    yrange = minmax(ypos_catalog)
    wset,0
    xlen_est = !D.X_SIZE - !D.X_CH_SIZE * total( !X.margin )
    ylen_est = !D.Y_SIZE - !D.Y_CH_SIZE * total( !Y.margin )
  
    ; Enlarge the axis ranges to center desired region and have 1-1 aspect.
    pixel_size = max( [(xrange[1] - xrange[0]) / xlen_est, $
                       (yrange[1] - yrange[0]) / ylen_est] )
                
    xrange = ((xrange[0]+xrange[1]) / 2.) + $
                        pixel_size * xlen_est * [-0.5,0.5]
    
    yrange = ((yrange[0]+yrange[1]) / 2.) + $
                        pixel_size * ylen_est * [-0.5,0.5]

    window,xsize=1000,ysize=800
    partvelvec, xpos_corr-xpos_catalog, ypos_corr-ypos_catalog, xpos_catalog, ypos_catalog, TITLE='Vector (length magnified) from Catalog to Correlation Position', XRANGE=xrange, YRANGE=yrange, XSTYLE=1, YSTYLE=1, LENGTH=0.12


  ;; ------------------------------------------------------------------------
  tit = 'Comparison of Position Estimates'                
  ;dataset_1d, id6, ( ra_data -  ra_ml)*3600, DATASET='RA' , DENSITY_TITLE=tit, XTIT='Data Mean - Recon Peak (seconds)'
  ;dataset_1d, id6, (dec_data - dec_ml)*3600, DATASET='DEC'
  
;  dataset_2d, id1, NAN=[-0.5,0], off_angle, cat2corr_offset,  PSYM=1, DATASET_NAME='Catalog to Correlation Peak', TITLE=tit, XTIT='Average Off-axis Angle (arcmin)', YTIT='Offset (arcsec)'
;  dataset_2d, id1, NAN=[-0.5,0], off_angle, cat2data_offset,  PSYM=6, DATASET_NAME='Catalog to Data Mean'
;  dataset_2d, id1, NAN=[-0.5,0], off_angle, corr2data_offset, PSYM=4, DATASET_NAME='Data Mean to Correlation Peak'

  count_bins = [0,20,200,1E10] < max(src_counts)
  colors     = ['blue', 'red', 'green', 'white']
  for ii=0,n_elements(count_bins)-2 do begin
    mask = (count_bins[ii]+1 LE src_counts) AND (src_counts LE count_bins[ii+1])
    
    if (total(mask) GT 0) then begin
      name = string(count_bins[ii]+1, count_bins[ii+1], F='(%"%d <= source_counts <= %d")')
      
      dataset_2d, id2, NAN=[-0.5,0], PSY=1, COLOR=colors[ii],   off_angle*mask, cat2corr_offset* mask, DATASET=name+', Catalog to Correlation Peak', TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='Offset (arcsec)'
      
      dataset_2d, id2, NAN=[-0.5,0], PSY=6, COLOR=colors[ii], off_angle*mask, cat2data_offset* mask, DATASET=name+', Catalog to Data Mean'
      
      dataset_2d, id2, NAN=[-0.5,0], PSY=4, COLOR=colors[ii], off_angle*mask, corr2data_offset*mask, DATASET=name+', Data Mean to Correlation Peak';, PS_CONFIG={filename:output_dir+'foobar.ps'}, /PRINT
      
      
      
;      ; Compare AE's 1-D position error ERR_DATA (63%) with the 68% "position uncertainty" by Kim et al. (ChaMP source catalog paper)
;      ; Scale up ERR_DATA by 1.07 to approximate a 68% error circle radius---see Tim Naylor's appendix.
;      mask AND= (bt.MERGNUM EQ 1) AND (bt.PSF_FRAC GT 0.80)
;      
;      if (total(mask) GT 0) then begin
;          kim_uncertainty      = 0.1137 * off_angle - 0.4600 * alog10(src_counts) - 0.2398        ; (eq. 14)
;        
;        ind = where(alog10(src_counts) GT 2.1336, count)
;        if (count GT 0) then begin
;          kim_uncertainty[ind] = (0.1031 * off_angle - 0.1945 * alog10(src_counts) - 0.8034)[ind] ; (eq. 14)
;        endif
;        kim_uncertainty = 10.^kim_uncertainty
;       ;info, kim_uncertainty
;        
;        
;        dataset_2d, id3, NAN=[-0.5,0], PSY=3, COLOR=colors[ii],   off_angle*mask, ((1.07*bt.ERR_DATA) / kim_uncertainty)*mask, DATASET=name+', one ObsID, >80% aperture', TITLE='Position Uncertainty Estimates', XTIT='Off-axis Angle (arcmin)', YTIT='ERR_DATA (63%) * 1.07 / Kim uncertainty (68%)'      
;      endif
    endif ;(total(mask) GT 0)
  endfor
  
  dataset_1d, id9, BINSIZE=0.1, cat2corr_offset,  DATASET='Catalog to Correlation Peak', DENSITY_TITLE=tit, XTIT='Offset (arcsec)'
  dataset_1d, id9, BINSIZE=0.1, cat2data_offset,  DATASET='Catalog to Data Mean'
  dataset_1d, id9, BINSIZE=0.1, corr2data_offset, DATASET='Data Mean to Correlation Peak';, PS_CONFIG={filename:output_dir+'foobar.ps'}, /PRINT
  
  sn = indgen(num_sources)
  function_1d, id7, LI=6,PSY=1, COLOR='white', sn, cat2corr_offset,  DATASET='Catalog to Correlation Peak', TITLE=tit, XTIT='Source # (0-based)', YTIT='Offset (arcsec)'
  function_1d, id7, LI=6,PSY=6, COLOR='white', sn, cat2data_offset,  DATASET='Catalog to Data Mean'
  function_1d, id7, LI=6,PSY=4, COLOR='white', sn, corr2data_offset, DATASET='Data Mean to Correlation Peak'
    
  ;; ------------------------------------------------------------------------
;  save, /COMPRESS, sourcename, ra, dec, ra_ml, dec_ml, ra_corr, dec_corr, ra_data, dec_data, cat2corr_offset, cat2data_offset, corr2data_offset,  quantization_corr, off_angle, src_counts, src_radius, FILE='check_positions.sav'
;  
;  print, '============================================================================='
;  print, 'IDL vectors saved in "check_positions.sav".'
;  print, '============================================================================='
  GOTO, CLEANUP
endif ; /CHECK_POSITIONS, /PLOT


;; =============================================================================
if keyword_set(plot) && (keyword_set(extract_events) OR keyword_set(extract_spectra)) then begin
;; =============================================================================

  if ~keyword_set(obsname) then begin
    print, F="(%'\nERROR!  An ""obsname"" parameter must be supplied.')"
    GOTO, FAILURE
  endif
  
  ;; Read the source properties of all the sources.
  if ~keyword_set(collated_filename) || ~file_test(collated_filename) then begin
    print, 'Collating source properties ...'
    if ~keyword_set(catalog_or_srclist) then begin
      print, collated_filename, collated_filename, F='(%"ERROR: The collation %s does not exist.  Supply the source list file, or build %s yourself.")'
      GOTO, FAILURE      
    endif
    collated_filename = temp_collate_fn
    acis_extract, catalog_or_srclist, obsname, /SINGLE_OBSID, COLLATED_FILENAME=collated_filename, VERBOSE=0
  endif
  
  ;; Read the source properties of all the sources.
  bt = mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status)
  if (status NE 0) then begin
    print, 'ERROR reading ', collated_filename
    GOTO, FAILURE      
  endif

  obsname_in_collation = strtrim(bt.obsname,2)
  if (total((obsname_in_collation NE '') AND (obsname_in_collation NE obsname)) GT 0) then begin
    print, collated_filename, obsname, F="(%'\nERROR!  Not all collations in %s are for the ObsId specified (%s).')"
    GOTO, FAILURE
  endif
   
  num_sources = n_elements(bt)
  
  sourcename           = bt.CATALOG_NAME
  obsdir       = strtrim(bt.OBSDIR,2)
                       
  xpos_catalog         = bt.X_CAT
  ypos_catalog         = bt.Y_CAT
  src_counts     = float(bt.SRC_CNTS[0])
  bkg_counts           = bt.BKG_CNTS[0]
  backscal             = bt.BACKSCAL[0]
  arf_mean             = bt.MEAN_ARF[0]
  background           = bt.BACKGRND
  off_angle            =  tag_exist(bt,'THETA')    ? bt.THETA    : intarr(num_sources)

  xpos_data            = bt.X_DATA
  ypos_data            = bt.Y_DATA
  er_xpos_data         = bt.EX_DATA
  er_ypos_data         = bt.EY_DATA
  cat2data_offset      = bt.CAT2DATA
                         
  region_edited        = bt.REG_EDIT
  psf_fraction         = bt.PSF_FRAC
  primary_ccd_fraction = bt.CCD_FRAC
  num_emap_pixels      = bt.EMAP_NUM
  mean_exposure        = bt.EMAP_AVG
  median_exposure      = bt.EMAP_MED
  min_exposure         = bt.EMAP_MIN
  max_exposure         = bt.EMAP_MAX
  
  warnfrac             = tag_exist(bt, 'WARNFRAC') ? bt.WARNFRAC : 0
  
  if tag_exist(bt, 'PROB_KS') then begin
    log_probks           = alog10(bt.PROB_KS)
    
    ind = where(bt.PROB_KS EQ 0, count)
    if (count GT 0) then log_probks[ind] = (-10) < min(/NAN, log_probks)
  endif else log_probks = 0

  ; Set integer-valued properties to NaN for unobserved sources.
  ind = where(~obsdir, count)
  if (count GT 0) then begin
    src_counts[ind] = !VALUES.F_NAN
  endif
  
  if keyword_set(extract_spectra) then begin
    src_area             = bt.SRC_AREA
    src_area_type        = 'via dmextract'
  endif else begin
    src_area             = bt.PGN_AREA
    src_area_type        = 'via polyfillv (IDL)'
  endelse

  f_nan = replicate(!VALUES.F_NAN,num_sources)
  psf_fraction_min     = f_nan
  psf_fraction_max     = f_nan
  
  ;; Read summary information from stats file for each source.
  print, 'reading source information'
  for ii = 0L, num_sources-1 do begin
      if ~obsdir[ii] then continue
      psf_frac_fn = obsdir[ii] + obs_frac_basename
      t = mrdfits(psf_frac_fn, 1, /SILENT, STATUS=status)
      
      if (status EQ 0) then begin
        psf_fraction_min    [ii] = min(t.fraction)
        psf_fraction_max    [ii] = max(t.fraction)
      endif 
  endfor
  

  ;; Show summary information.
  tit = 'Source Extraction Regions for dataset "' + obsname + '"'
  sn = indgen(num_sources)
; function_1d, id2, LI=6,PSY=1, sn, region_edited,     TITLE=tit, XTIT='Source # (0-based)', YTIT='Source Aperture Edited in ds9 (T/F)'
  
; dataset_2d, id3, NAN=[-0.5,0], PSY=6, off_angle, src_area, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='Area of Source Aperture (skypixel**2)', DATASET=src_area_type

;  dataset_1d, id3b,           src_area  , DENSITY_TITLE=tit,      XTIT='Area of Source Aperture (skypixel**2)', DATASET=src_area_type
;  dataset_1d, id4, BINSIZE=1, src_counts, DENSITY_TITLE=tit,      XTIT='In-band Counts in Source Aperture', PS_CONFIG={filename:output_dir+'counts_hist.ps'}, /PRINT

; dataset_2d, id5, NAN=[-0.5,0], PSY=1, src_counts, cat2data_offset, TITLE=tit, XTIT='In-band Counts in Source Aperture', YTIT='Catalog/Data Offset (skypixel)'

  count_bins = [0,20,200,1E10] < max(src_counts, /NAN)
  colors     = ['blue', 'red', 'green', 'white']
  for ii=0,n_elements(count_bins)-2 do begin
    mask = (count_bins[ii]+1 LE src_counts) AND (src_counts LE count_bins[ii+1])
    if (total(mask) GT 0) then begin
      name = string(count_bins[ii]+1, count_bins[ii+1], F='(%"%d <= source_counts <= %d")')
      
      dataset_2d, id6, NAN=[-0.5,0], PSY=6, COLOR=colors[ii], off_angle*mask, cat2data_offset*mask, DATASET=name, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='Catalog/Data Offset (skypixel)', PS_CONFIG={filename:output_dir+'dataoffset_vs_theta.ps'}, /PRINT
    endif
  endfor

  function_1d, id9, LI=6,PSY=6, off_angle, psf_fraction_max, COLOR='red',   DATASET='maximum',  TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='PSF Fraction'
  function_1d, id9, LI=6,PSY=1, off_angle, psf_fraction,     COLOR='white', DATASET='@ fiducial energy'
  function_1d, id9, LI=6,PSY=5, off_angle, psf_fraction_min, COLOR='blue',  DATASET='minimum', PS_CONFIG={filename:output_dir+'psffrac.ps'}, /PRINT

;  savefile = 'extract_events_'+obsname+'.sav'
;  save, /COMPRESS, sourcename, xpos_catalog, ypos_catalog, xpos_data, ypos_data, er_xpos_data, er_ypos_data, cat2data_offset, region_edited, psf_fraction, src_counts, off_angle, warnfrac, FILE=savefile
;
;  print, '============================================================================='
;  print, 'IDL vectors saved in ', savefile
;  print, '============================================================================='

  ;; If /EXTRACT_SPECTRA not specified, then skip everything else.
  if ~keyword_set(extract_spectra) then GOTO, CLEANUP


  ;; Show more summary information.
; function_1d, id8 , LI=6,PSY=1, sn, arf_mean,             TITLE=tit, XTIT='Source # (0-based)', YTIT='Mean ARF value (cm**2 count /photon)'
  function_1d, id10, LI=6,PSY=1, sn, primary_ccd_fraction, TITLE=tit, XTIT='Source # (0-based)', YTIT='Fraction of Source Counts on Primary CCD', PS_CONFIG={filename:output_dir+'multiple_ccds.ps'}, /PRINT
  
;   if tag_exist(bt, 'PROB_KS') then function_1d, id12, LI=6,PSY=1, sn, log_probks,           TITLE=tit, XTIT='Source # (0-based)', YTIT='log  p-value, wrt uniform LC model'
  if tag_exist(bt, 'WARNFRAC') then $
  function_1d, id11, LI=6,PSY=1, sn, warnfrac,             TITLE=tit, XTIT='Source # (0-based)', YTIT='Fraction of Src Events in WARNING_REGION'

; dataset_3d, id7, xpos_catalog, ypos_catalog, arf_mean,   TITLE='Map of mean ARF value', XTIT='X (sky coordinates)', YTIT='Y (sky coordinates)', ZTIT='mean ARF value', STAT_CODE=0

;   if tag_exist(bt, 'PROB_KS') then dataset_2d, id13, NAN=[-0.5,0], PSY=1, src_counts, log_probks,  TITLE=tit, XTIT='In-band Counts in Source Aperture', YTIT='log p-value wrt uniform LC model', PS_CONFIG={filename:output_dir+'ks_vs_counts.ps'}, /PRINT

  dataset_2d, id14, NAN=[-0.5,0], PSY=1, off_angle, arf_mean, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='Mean ARF value (cm**2 count /photon)', PS_CONFIG={filename:output_dir+'meanarf_vs_theta.ps'}, /PRINT
  

;  savefile = 'extract_spectra_'+obsname+'.sav'
;  save, /COMPRESS, sourcename, xpos_catalog, ypos_catalog, region_edited, psf_fraction, primary_ccd_fraction, arf_mean, num_emap_pixels, mean_exposure, median_exposure, min_exposure, max_exposure, src_area, src_counts, log_probks, cat2data_offset, off_angle, warnfrac, FILE=savefile
;
;  print, '============================================================================='
;  print, 'IDL vectors saved in ', savefile
;  print, '============================================================================='
  GOTO, CLEANUP
endif ; /EXTRACT_SPECTRA, /PLOT


;; =============================================================================
if keyword_set(plot) && keyword_set(extract_backgrounds) then begin
;; =============================================================================

  if ~keyword_set(obsname) then begin
    print, F="(%'\nERROR!  An ""obsname"" parameter must be supplied.')"
    GOTO, FAILURE
  endif
  
  ;; Read the source properties of all the sources.
  if ~keyword_set(collated_filename) || ~file_test(collated_filename) then begin
    print, 'Collating source properties ...'
    if ~keyword_set(catalog_or_srclist) then begin
      print, collated_filename, collated_filename, F='(%"ERROR: The collation %s does not exist.  Supply the source list file, or build %s yourself.")'
      GOTO, FAILURE      
    endif
    collated_filename = temp_collate_fn
    acis_extract, catalog_or_srclist, obsname, /SINGLE_OBSID, COLLATED_FILENAME=collated_filename, VERBOSE=0
  endif
  
  ;; Read the source properties of all the sources.
  bt = mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status)
  if (status NE 0) then begin
    print, 'ERROR reading ', collated_filename
    GOTO, FAILURE      
  endif

  obsname_in_collation = strtrim(bt.obsname,2)
  if (total((obsname_in_collation NE '') AND (obsname_in_collation NE obsname)) GT 0) then begin
    print, collated_filename, obsname, F="(%'\nERROR!  Not all collations in %s are for the ObsId specified (%s).')"
    GOTO, FAILURE
  endif
  
  num_sources = n_elements(bt)
  
  sourcename       = bt.CATALOG_NAME
  obsdir   = strtrim(bt.OBSDIR,2)
  xpos_catalog     = bt.X_CAT
  ypos_catalog     = bt.Y_CAT
  src_counts = float(bt.SRC_CNTS[0])
  bkg_radius       = bt.BKG_RAD
  bkg_counts = float(bt.BKG_CNTS[0])
  backscal         = bt.BACKSCAL[0]
  background       = bt.BACKGRND
  mean_exposure    = bt.EMAP_AVG
  off_angle        =  tag_exist(bt,'THETA')    ? bt.THETA    : intarr(num_sources)
   
  ; Set integer-valued properties to NaN for unobserved sources.
  ind = where(~obsdir, count)
  if (count GT 0) then begin
    src_counts[ind] = !VALUES.F_NAN
    bkg_counts[ind] = !VALUES.F_NAN
  endif

  ind = where(finite(bkg_radius), count)
  if keyword_set(region_file) && (count GT 0) then begin
    ; Make a region file showing the background regions.
    openw,  region2_unit, region_file, /GET_LUN
    printf, region2_unit, "# Region file format: DS9 version 3.0"
    printf, region2_unit, 'global width=1 font="helvetica 12 normal"'
    !TEXTUNIT = region2_unit
    forprint, TEXTOUT=5, SUBSET=ind, xpos_catalog, ypos_catalog, bkg_radius, F='(%"circle(%0.1f,%0.1f,%0.1f) # tag={background}")', /NoCOMMENT
    free_lun, region2_unit
  endif
  
  ;; Show summary information.
  tit = 'Background Extraction Regions for dataset "' + obsname + '"'
  sn = indgen(num_sources)
 ;function_1d, id3, LI=6,PSY=1, sn, bkg_counts, TITLE=tit, XTIT='Source # (0-based)', YTIT='In-band Counts in Background Region'
 ;function_1d, id7, LI=6,PSY=1, sn, alog10(background), TITLE=tit, XTIT='Source # (0-based)', YTIT='log Background Surface Brightness [photon /cm**2 /s /skypixel**2]'
 ;function_1d, id2, LI=6,PSY=1, sn, backscal,TITLE=tit, XTIT='Source # (0-based)', YTIT='scaling of bkg extraction (BACKSCAL)'

  dataset_1d, id20, BINSIZE=2 , bkg_counts, DENSITY_TITLE=tit, XTIT='Counts in Background Region', PS_CONFIG={filename:output_dir+'bkdg_counts_hist.ps'}, /PRINT
  dataset_1d, id21, BINSIZE=20, backscal  , DENSITY_TITLE=tit, XTIT='Scaling of Bkg Extraction (BACKSCAL)', PS_CONFIG={filename:output_dir+'backscal_hist.ps'}, /PRINT

  abscissa = alog10(src_counts/mean_exposure)
  ordinate = alog10(background)
  dataset_2d, id4, NAN=[min(/NAN,abscissa), min(/NAN,ordinate)-1], PSY=1, abscissa, ordinate, TITLE=tit, XTIT='log In-band Flux in Source Aperture [photon /cm**2 /s]', YTIT='log Background Surface Brightness [photon /cm**2 /s /skypixel**2]', PS_CONFIG={filename:output_dir+'bkgd_sb_vs_flux.ps'}, /PRINT

if tag_exist(bt,'THETA') then begin
  abscissa = off_angle
  ordinate = alog10(background)
  dataset_2d, id6, NAN=[min(/NAN,abscissa), min(/NAN,ordinate)-1], PSY=1, abscissa, ordinate, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='log Background Surface Brightness [photon /cm**2 /s /skypixel**2]', PS_CONFIG={filename:output_dir+'bkgd_sb_vs_theta.ps'}, /PRINT
endif

;  savefile = 'extract_backgrounds_'+obsname+'.sav'
;  save, /COMPRESS, sourcename, bkg_counts, bkg_radius, background, src_counts, off_angle, FILE=savefile
;
;  print, '============================================================================='
;  print, 'IDL vectors saved in ', savefile
;  print, '============================================================================='
  GOTO, CLEANUP
endif ; /EXTRACT_BACKGROUNDS, /PLOT



;; =============================================================================
; We initialize run_command here so that the stages above (which don't use run_command) will run faster.
; The tempdir must appear before the cache_dir in PFILES so that we can whack parameter files in tempdir and re-create fresh ones for each sources.
run_command, PARAM_DIR=[tempdir,cache_dir], CIAO_VERSION=ciao_version, MARX_VERSION=marx_version
;; =============================================================================



;; =============================================================================
if keyword_set(plot) && keyword_set(merge_observations) then begin
;; =============================================================================

  ;; Read the source properties of all the sources.
  if ~keyword_set(collated_filename) || ~file_test(collated_filename) then begin
    print, 'Collating source properties ...'
    if ~keyword_set(catalog_or_srclist) then begin
      print, collated_filename, collated_filename, F='(%"ERROR: The collation %s does not exist.  Supply the source list file, or build %s yourself.")'
      GOTO, FAILURE      
    endif
    collated_filename = temp_collate_fn
    acis_extract, catalog_or_srclist, (keyword_set(obsname) ? obsname : ''),  MERGE_NAME=(keyword_set(merge_name) ? merge_name : ''), COLLATED_FILENAME=collated_filename
  endif
  
  bt = mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status)
  if (status NE 0) then begin
    print, 'ERROR reading ', collated_filename
    GOTO, FAILURE      
  endif
  
  num_sources = n_elements(bt)
  
  band_full           = 0  
  print, 'Using the energy band ', bt[0].ENERG_LO[band_full], bt[0].ENERG_HI[band_full]

  sourcename     = bt.CATALOG_NAME
  merge_subdir   = bt.MERGE_NAME
  ra             = bt.RA
  dec            = bt.DEC
  off_angle      =  tag_exist(bt,'THETA')    ? bt.THETA    : intarr(num_sources)
  obsid_count    = bt.NUM_OBS
  background     = bt.BACKGRND
  src_counts     = bt.SRC_CNTS  [band_full]
  bkg_counts     = bt.BKG_CNTS  [band_full]
  net_counts     = bt.NET_CNTS  [band_full]
  backscal       = bt.BACKSCAL  [band_full]
  src_signif     = bt.SRC_SIGNIF[band_full]
  arf_mean       = bt.MEAN_ARF  [band_full]
  flux1          = bt.FLUX1     [band_full]
  flux2          = bt.FLUX2     [band_full]
  scal_max       = bt.SCAL_MAX
  scal_min       = bt.SCAL_MIN
  SELFCNTS       = tag_exist(bt,'SELFCNTS') ? bt.SELFCNTS : intarr(num_sources)
  BKGMETR1       = tag_exist(bt,'BKGMETR1') ? bt.BKGMETR1 : intarr(num_sources) ;'bkg fairness metric'
  MERGE_KS       = tag_exist(bt,'MERGE_KS') ? bt.MERGE_KS : fltarr(num_sources)

  ;; Create a list of the sources with the most extreme background regions, for use in visual review.
  review_score = fltarr(num_sources)
  review_note  = strarr(num_sources)
  
  num_per_category = 5 < num_sources
  sub_score        = reverse(findgen(num_per_category) / num_per_category)
  
  metric = BKGMETR1
  ind = (reverse(sort(metric)))[0:num_per_category-1]
  review_score[ind] = 1 + sub_score
  review_note [ind] = string(metric[ind], F="(%'background fairness metric = %0.2f')")
  
  metric = SELFCNTS/NET_COUNTS
  ind = (reverse(sort(metric)))[0:num_per_category-1]
  review_score[ind] = 2 + sub_score
  review_note [ind] = string(metric[ind], F="(%'SELFCNTS/NET_CNTS = %0.2f')")
  
  ind =         (sort(BACKSCAL)) [0:num_per_category-1]
  review_score[ind] = 3 + sub_score
  review_note [ind] = 'Low-tail of BACKSCAL'
  
  ind = (reverse(sort(BACKSCAL)))[0:num_per_category-1]
  review_score[ind] = 4 + sub_score
  review_note [ind] = 'High-tail of BACKSCAL'
  
  ind =         (sort(BKG_COUNTS)) [0:num_per_category-1]
  review_score[ind] = 5 + sub_score
  review_note [ind] = 'Low-tail of BKG_CNTS'
  
  ind = (reverse(sort(BKG_COUNTS)))[0:num_per_category-1]
  review_score[ind] = 6 + sub_score
  review_note [ind] = 'High-tail of BKG_CNTS'
  
  ind = where(review_score GT 0)
  forprint, TEXTOUT='bkg_review.srclist', bt.CATALOG_NAME, review_note, F='(%"%s ; %s")', SUBSET=ind[ reverse(sort(review_score[ind])) ], /NoCOMMENT

print, 'ADD A REVIEW OF SOURCE WITH LARGE OR SMALL BACKSCAL CORRECTION (BACKCORR)!'  
  
  
  
  
  
  ;; Make some plots.
  log_ks_spect   = alog10(bt.KS_SPECT)
  ind = where(bt.KS_SPECT EQ 0, count)
  if (count GT 0) then log_ks_spect[ind] = (-10) < min(/NAN, log_ks_spect)
  
  cartoon_acis   = 0
  cartoon_sky    = 0
  psf_missing    = replicate(1B,num_sources)
  if keyword_set(cartoon_template) then begin
    cartoon_acis_fn = 'cartoon_acis.img'
    cartoon_sky_fn  = 'cartoon_sky.img'

    if ~keyword_set(cartoon_fwhm) then cartoon_fwhm = 3
    refhd = headfits(cartoon_template)
    extast, refhd, refastr
    cartoon_xsize = psb_xpar( refhd, 'NAXIS1')
    cartoon_ysize = psb_xpar( refhd, 'NAXIS2')
  endif

  for ii = 0L, num_sources-1 do begin
    sourcedir        = sourcename[ii] + '/' + merge_subdir[ii] + '/'
    composite_psf_fn = sourcedir + psf_basename
 
    if keyword_set(cartoon_template) then begin
      if (flux2[ii] LE 0) then begin
        print, 'FLUX2[0] not positive -- source not added to cartoon'
        continue
      endif
      
      if NOT finite(flux2[ii]) then continue
      
      ;; Use ACIS PSF to build a cartoon that exhibits the Chandra PSF.
      if file_test(composite_psf_fn) then begin
       ;print, 'Reading ', composite_psf_fn
        psf_missing[ii] = 0
        psf_img = readfits(composite_psf_fn, psf_header, /SILENT)
        
        ; Set all the NaN values to zero to keep future computations happy.
        ind = where(finite(psf_img) EQ 0, count)
        if (count GT 0) then psf_img[ind] = 0

        ; We have to be careful with the normalization of the PSF image.
        ; The PSF on disk has is implicitly scaled using PSF_TOTL to sum to <=1 
        ; due to cropping, i.e. total(psf_img)/PSF_TOTL <=1.
        ; Regridding with hastrom does NOT preserve the power -- we must renormalize so
        ; the power is back to the value in the original PSF.
        ; We have to normalize PSF _before_ resampling to scene grid because
        ; the PSF may fall partially off the scene.
          
        psf_total = psb_xpar( psf_header, 'PSF_TOTL')
        if (psf_total EQ 0) then begin
          print, "WARNING: obsolete PSFs in "+psf_fn
          power = 1.0
        endif else begin
          power = total(psf_img, /DOUBLE) / psf_total
        endelse
        
        ;psb_xaddpar, psf_header, 'EQUINOX', 2000.0      
        hastrom, psf_img, psf_header, refhd, MISSING=0  

        normalization = flux2[ii] * (power/total(psf_img, /DOUBLE))
        
        if finite(normalization) then begin        
          cartoon_acis = cartoon_acis + psf_img * normalization
        endif else begin
          print, sourcename[ii], cartoon_acis_fn, F='(%"WARNING: it appears that the ACIS PSF for %s is smaller than your scene pixels; that source will be missing from %s.")'
        endelse
      endif
      
      ;; Use constant PSF to build a cartoon that is independent of the Chandra PSF.
      ; We can not use /NORMALIZE in psf_gaussian() because the PSF may fall partially off the scene.
      ad2xy, ra[ii], dec[ii], refastr, xindex, yindex 
      psf_img = psf_gaussian(NPIX=[cartoon_xsize,cartoon_ysize], FWHM=cartoon_fwhm, CENTROID=[xindex,yindex])
        
      cartoon_sky = cartoon_sky + psf_img * flux2[ii]
    endif ;keyword_set(cartoon_template)
  endfor
  
  ;; Construct cartoon image.
  if keyword_set(cartoon_template) then begin
    print, '============================================================================='
    num_psf_missing = total(psf_missing)
    if (num_psf_missing EQ num_sources) then begin
      print, 'WARNING: construction of ',cartoon_acis_fn,' skipped because no composite PSFs found.'
    endif else begin
      if (num_psf_missing GT 0) then begin
        print, num_psf_missing, cartoon_acis_fn, F='(%"WARNING: %d sources omitted from %s because no composite PSF found.")'
      endif else print, 'Wrote ' + cartoon_acis_fn

      psb_xaddpar, psf_header, 'CREATOR', creator_string
      writefits, cartoon_acis_fn, cartoon_acis/max(cartoon_acis), psf_header
      run_command, 'ds9 -log '+cartoon_acis_fn+' -frame center >& /dev/null &'
    endelse

    psb_xaddpar, refhd, 'CREATOR', creator_string
    writefits, cartoon_sky_fn, cartoon_sky/max(cartoon_sky), refhd
    run_command, 'ds9 -log '+cartoon_sky_fn+' -frame center >& /dev/null &'
    print, 'Wrote ' + cartoon_sky_fn
    print, '============================================================================='
  endif

  
  ;; Compute sky positions for an arbitrary sky (x,y) system with 
  ;; 0.5" pixels centered on the mean source position.
  make_astr, composite_astr, DELTA=[-0.000136667D,0.000136667D],  $
                             CRPIX=[0D,0D], CRVAL=[mean(ra,  /DOUBLE),mean(dec, /DOUBLE)]

  ad2xy, ra, dec, composite_astr, xpos_catalog, ypos_catalog
  
  
  ;; Show summary information.
  tit = 'Composite Extractions'
  sn = indgen(num_sources)
 ;function_1d, id5 , LI=6,PSY=1, sn, net_counts,  TITLE=tit, XTIT='Source # (0-based)', YTIT='Net Counts' 
 ;function_1d, id3 , LI=6,PSY=1, sn, src_signif,  TITLE=tit, XTIT='Source # (0-based)', YTIT='Source Significance' 
 ;function_1d, id6 , LI=6,PSY=1, sn, obsid_count, TITLE=tit, XTIT='Source # (0-based)', YTIT='# Observations of Source' , PS_CONFIG={filename:output_dir+'times_src_observed.ps'}, /PRINT
  
 ;function_1d, id7 , LI=6,PSY=1, sn, arf_mean,    TITLE=tit, XTIT='Source # (0-based)', YTIT='Mean ARF value (cm**2 count /photon)'

 ;function_1d, id2 , LI=6,PSY=1, sn, backscal,    TITLE=tit, XTIT='Source # (0-based)', YTIT='scaling of bkg extraction (BACKSCAL)'
 ;function_1d, id4 , LI=6,PSY=4, sn, scal_max/scal_min, DATASET='SCAL_MAX/SCAL_MIN among merged observations'
 ;function_1d, id10, LI=6,PSY=1, sn, log_ks_spect, TITLE=tit, XTIT='Source # (0-based)',  YTIT='log  p-value, src/bkg spectra'

  dataset_1d, id20, BINSIZE=2 , bkg_counts, DENSITY_TITLE=tit, XTIT='Counts in Background Region', PS_CONFIG={filename:output_dir+'bkgd_counts_composite_hist.ps'}, /PRINT
  
  dataset_1d, id21, BINSIZE=20, backscal  , DENSITY_TITLE=tit, XTIT='Scaling of Bkg Extraction (BACKSCAL)', PS_CONFIG={filename:output_dir+'backscal_composite_hist.ps'}, /PRINT
  
  dataset_1d,  id8, arf_mean, BINSIZE=1, DENSITY_TITLE=tit, XTIT='Mean ARF value (cm**2 count /photon)', PS_CONFIG={filename:output_dir+'meanarf.ps'}, /PRINT
  
; dataset_3d, id1, xpos_catalog, ypos_catalog, arf_mean, STAT_CODE=0, TITLE='Map of mean ARF value', XTIT='composite X (0.5" pixels)', YTIT='composite Y (0.5" pixels)', ZTIT='mean ARF value'

 ;dataset_2d, id11, NAN=[0,0], PSY=1, net_counts, log_ks_spect, TITLE=tit, XTIT='Net Counts', YTIT='log p-value, src/bkg spectra', PS_CONFIG={filename:output_dir+'ks_src_bkgd_vs_netcts.ps'}, /PRINT
  
                                                      



  abscissa = alog10(flux2)
  ordinate = alog10(background)
  dataset_2d, id12, NAN=[min(/NAN,abscissa), min(/NAN,ordinate)-1], PSY=1, abscissa, ordinate, TITLE=tit, XTIT='log In-band Source Flux [photon /cm**2 /s]', YTIT='log Background Surface Brightness [photon /cm**2 /s /skypixel**2]', PS_CONFIG={filename:output_dir+'bkgd_sb_vs_flux.ps'}, /PRINT

if tag_exist(bt,'THETA') then begin
  abscissa = off_angle
  ordinate = alog10(background)
  dataset_2d, id6, NAN=[min(/NAN,abscissa), min(/NAN,ordinate)-1], PSY=1, abscissa, ordinate, TITLE=tit, XTIT='Off-axis Angle (arcmin)', YTIT='log Background Surface Brightness [photon /cm**2 /s /skypixel**2]', PS_CONFIG={filename:output_dir+'bkgd_sb_vs_theta.ps'}, /PRINT
endif

if tag_exist(bt,'MERGE_KS') then begin
  abscissa = alog10(src_counts)
  ordinate = alog10(MERGE_KS)
  dataset_2d, id30, NAN=[min(/NAN,abscissa), min(/NAN,ordinate)-1], PSY=1, abscissa, ordinate, TITLE=tit, XTIT='log SRC_CNTS', YTIT='log p-value, No-variability Hypothesis', PS_CONFIG={filename:output_dir+'merge_ks_vs_flux.ps'}, /PRINT
endif

  
 ;dataset_2d, id17, NAN=[0,0], PSY=1, net_counts, src_signif,  TITLE=tit, XTIT='Net Counts', YTIT='Source Significance', PS_CONFIG={filename:output_dir+'srcsig_vs_netcts.ps'}, /PRINT

  ;dataset_2d, id13, NAN=[-0.5,0], PSY=1, src_signif, alog10(background), TITLE=tit, XTIT='Source Significance', YTIT='log Background Surface Brightness [photon /cm**2 /s /skypixel**2]', PS_CONFIG={filename:output_dir+'bkgd_sb_vs_srcsig.ps'}, /PRINT
  
if tag_exist(bt,'THETA') then begin
 ;dataset_2d, id15, NAN=[-0.5,0], PSY=1, off_angle, src_signif,        TITLE=tit, XTIT='Average Off-axis Angle (arcmin)', YTIT='Source Significance', PS_CONFIG={filename:output_dir+'srcsig_vs_theta.ps'}, /PRINT

  theta_cut = median(off_angle)
  ind_inner = where(off_angle LT theta_cut, COMPLEMENT=ind_outer)
  dataset_1d, id14, alog10(net_counts[ind_inner]), DATASET=string(theta_cut, F='(%"Average Off-axis Angle < %0.1f (arcmin)")'), BINSIZE=0.1, DENSITY_TITLE=tit, XTIT='log Net Counts'
  dataset_1d, id14, alog10(net_counts[ind_outer]), DATASET=string(theta_cut, F='(%"Average Off-axis Angle > %0.1f (arcmin)")'), BINSIZE=0.1,  PS_CONFIG={filename:output_dir+'netcts_vs_theta.ps'}, /PRINT

  
  dataset_2d, id18, NAN=[-0.5,0], PSY=1, off_angle, scal_max/scal_min, TITLE=tit, XTIT='Average Off-axis Angle (arcmin)', YTIT='SCAL_MAX/SCAL_MIN among merged observations', PS_CONFIG={filename:output_dir+'backscal_vs_theta.ps'}, /PRINT
endif  

  abscissa = alog10(flux1)
  ordinate = alog10(flux2)
  dataset_2d, id16, NAN=[min(/NAN,abscissa), min(/NAN,ordinate)-1], PSY=1, abscissa, ordinate, TITLE=tit, XTIT='log Flux1 (photon /cm**2 /s) -- see manual', YTIT='log Flux2 (photon /cm**2 /s) -- see manual', DATASET='merged obserations'
  dataset_2d, id16, PSY=1, abscissa, abscissa, DATASET='1-1 line', PS_CONFIG={filename:output_dir+'flux2_vs_flux1.ps'}, /PRINT

;  save, /COMPRESS, sourcename, xpos_catalog, ypos_catalog, off_angle, arf_mean, obsid_count, background, src_counts, net_counts, src_signif, flux1, flux2, scal_max, scal_min, log_ks_spect, FILE='merge_observations.sav'
;
;  print, '============================================================================='
;  print, 'IDL vectors saved in merge_observations.sav'
;  print, '============================================================================='
  GOTO, CLEANUP
endif ; /PLOT /MERGE



;; =============================================================================
if keyword_set(construct_regions) AND keyword_set(diffuse) then begin
;; =============================================================================

  ;; Read a 2-column catalog.
  readcol, catalog_or_srclist, sourcename, catalog_region_fn, FORMAT='A,A', COMMENT=';'
  
  ; Trim whitespace and remove blank lines.
  sourcename = strtrim(sourcename,2)
  ind = where(sourcename NE '', num_sources)
  
  if (num_sources EQ 0) then begin
    print, 'ERROR: no entries read from source list ', catalog_or_srclist
    GOTO, FAILURE
  endif
  
  sourcename          = sourcename         [ind]
  catalog_region_fn   = catalog_region_fn  [ind]
  print, num_sources, F='(%"\n%d sources found in catalog.\n")'
  

  ;; Load observation data into ds9.
  print, 'Spawning ds9 to perform coordinate conversions ...'
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='AE_obs'+obsname+'_'+session_name, OPTION_STRING='-bin factor 8'
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, obsdata_filename
  
;  flush_stdin  ; eat any characters waiting in STDIN, so that they won't be mistaken as commands in the loop below.
;  print, 'Configure binning & zoom as desired, then press RETURN ...'
;  cmd = ''
;  read, '? ', cmd

  ;; Process each source.
  for ii = 0L, num_sources-1 do begin
    ;; Construct directory & filenames for source.
    sourcedir = sourcename[ii] + '/'
    
    unnamed_src_stats_fn = sourcedir + src_stats_basename
   
    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 
    
    ; Remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    run_command, /QUIET, 'pset dmcopy clobber=yes'
    
    print, sourcename[ii], F='(%"\nSource: %s")'
    file_mkdir, sourcedir

    fxhmake,  unnamed_src_stats, /INITIALIZE, /EXTEND, /DATE
    psb_xaddpar, unnamed_src_stats, 'CREATOR', creator_string
    psb_xaddpar, unnamed_src_stats, 'OBJECT', sourcename[ii], 'source name'
    psb_xaddpar, unnamed_src_stats, 'LABEL',  sourcename[ii], 'source name'
    psb_xaddpar, unnamed_src_stats, 'DIFFUSE', 'T', 'diffuse source'
    psb_xaddpar, unnamed_src_stats, 'RA',  !VALUES.F_NAN, 'diffuse source has no position'
    psb_xaddpar, unnamed_src_stats, 'DEC', !VALUES.F_NAN, 'diffuse source has no position'
    psb_xaddpar, unnamed_src_stats, 'REGIONFN', catalog_region_fn[ii], 'region file defining the source'
    writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
    
    ;; Construct filenames for specific obsid.
    obsdir    = sourcename[ii] + '/' + obsname + '/' + extraction_subdir[ii]
    stats_fn  = obsdir + obs_stats_basename
    region_fn = obsdir + src_region_basename
    file_mkdir, obsdir
    
    
    ;; Fail if the region file uses the "field()" construct, which will corrupt the algorithm we use in recipe.txt to compute the geometric area of the extraction region.
    ae_ds9_to_ciao_regionfile, catalog_region_fn[ii], '/dev/null', FIELD_SYNTAX_FOUND=field_syntax_found
    
    if field_syntax_found then begin
      print, 'ERROR: the syntax "field()" is not allowed in region files defining diffuse sources.'
      GOTO, FAILURE
    endif
    
     ;; Load region file into ds9 and resave in PHYSICAL coordinates.
    cmd = strarr(6)
    cmd[0] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions delete all")')
    cmd[1] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions format ds9")')
    cmd[2] = string(my_ds9, catalog_region_fn[ii],   F='(%"xpaset -p ''%s'' regions load %s")')
    cmd[3] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions system physical")')
    cmd[4] = string(my_ds9,                          F='(%"xpaset -p ''%s'' regions format ciao")')
    cmd[5] = string(my_ds9, region_fn,               F='(%"xpaset -p ''%s'' regions save %s")')
    run_command, cmd, /QUIET
    
    
    ;; For point sources, this stage determines once and for all whether the source was observed by this ObsID, by examining the emap. 
    ;; However, diffuse extractions cannot be done unless there are actual events in the aperture.
    ;; Indeed, there must be events within the energy range specified for the WMAP, which is input to mkwarf.
    ;; Also, there is some chance that the observer might radically alter extract.reg by hand, after it is built here.
    ;; Thus, we choose to make no off-field decision here (obs.stats is written), and we let the EXTRACT stage eliminate the extraction (remove obs.stats) if it finds that it cannot be done.
    
    fxhmake,  stats, /INITIALIZE, /EXTEND, /DATE
    psb_xaddpar, stats, 'CREATOR', creator_string
    psb_xaddpar, stats, 'OBSNAME',  obsname, 'observation identifier'
    psb_xaddpar, stats, 'DIFFUSE', 'T', 'diffuse source'
    psb_xaddpar, stats, 'MSK_RAD',  !VALUES.F_NAN, '[skypixel] mask radius'
    psb_xaddpar, stats, 'X_CAT',    !VALUES.F_NAN, '[skypixel] source position'
    psb_xaddpar, stats, 'Y_CAT',    !VALUES.F_NAN, '[skypixel] source position'
    writefits, stats_fn, 0, stats
  endfor ; ii
  run_command, string(my_ds9, F='(%"xpaset -p ''%s'' exit")'), /QUIET
endif ;keyword_set(construct_regions) AND keyword_set(diffuse)


  
;; =============================================================================
if keyword_set(construct_regions) AND NOT keyword_set(diffuse) then begin
;; =============================================================================

   
  ae_make_psf_not_configured = 1
  
  
  ;; Read the 5-column catalog.
  readcol, catalog_or_srclist, sourcename, ra, dec, target_psf_fraction, fiducial_psf_energy, FORMAT='A,D,D,F,D', COMMENT=';'
  
  ; Trim whitespace and remove blank lines.
  sourcename = strtrim(sourcename,2)
  ind = where(sourcename NE '', num_sources)
  
  if (num_sources EQ 0) then begin
    print, 'ERROR: no entries read from source list ', catalog_or_srclist
    GOTO, FAILURE
  endif
  
  sourcename          = sourcename         [ind]
  ra                  = ra                 [ind]
  dec                 = dec                [ind]
  target_psf_fraction = target_psf_fraction[ind]
  fiducial_psf_energy = fiducial_psf_energy[ind]
  print, num_sources, F='(%"\n%d sources found in catalog.\n")'
  
  
  ;; Read the exposure map & setup array index to sky coordinate conversion.
  emap = readfits(emap_filename, emap_header, /SILENT)
  extast, emap_header, emap2wcs_astr
  
  ; Verify that the exposure map is NOT normalized, an assumption appearing at several places in the 
  ; design of AE (e.g. ae_make_psf, scaling diffuse ARFs to surface brightness units).
  dum = psb_xpar( emap_header, 'EXPOSURE', COMMENT=comment)
  if ~strmatch(comment,'* NOT *') then begin
    print, 'ERROR:  AE requires that exposure maps have units of (s cm**2 count /photon), i.e. the normalize="no" option was supplied to mkexpmap.'
    GOTO, FAILURE
  endif
  
  ; Choose an emap threshold that defines whether a source is observed or not.
  ; 10% of emap_global_median is too large---we have seen the emap at the center of the I-array drop to 7% of emap_global_median.
  ; A similar "off-field" decision is made in ae_make_catalog.
  ind = where(finite(emap) AND (emap GT 0))
  emap_global_median = median(emap[ind])
  emap_threshold     = emap_global_median * 0.05
  emap_xdim = (size(emap, /DIM))[0]
  emap_ydim = (size(emap, /DIM))[1]

  ;; In order to efficiently skip obviously off-field sources, without having to spawn dmcoords below, 
  ;; we will try to parse the approximate source position from the source name, 
  ;; and flag the source if it falls outside the emap array.
  result = stregex(sourcename, '([0-9]{2})([0-9]{2})([0-9]{2}\.[0-9]+)([+-][0-9]{2})([0-9]{2})([0-9]{2}\.[0-9]+)',/SUB,/EXT)
  
  ra_approx  = 15D * tenv( result[1,*],  result[2,*],  result[3,*])
  dec_approx =       tenv( result[4,*],  result[5,*],  result[6,*])
                                                                          
  ad2xy, ra_approx, dec_approx, emap2wcs_astr, xindex, yindex 
    
  emap_is_small         = (reform(result[0,*]) NE '') AND ((xindex LT 0) OR (xindex GE emap_xdim) OR $
                                                           (yindex LT 0) OR (yindex GE emap_ydim)) 
  source_not_observed OR= emap_is_small

  if (n_elements(emap_is_small) NE num_sources) then message, 'BUG IN AE!!!'
 
  
;window,0,XSIZE=256,YSIZE=256  

  ;; ------------------------------------------------------------------------
  mask_radii = fltarr(num_sources)
  
  ;; Process each source.
  for ii = 0L, num_sources-1 do begin
    ; Skip sources already determined to be off-field.
    if source_not_observed[ii] then continue
    
    ;; Construct directory & filenames for source.
    sourcedir = sourcename[ii] + '/'
    
    unnamed_src_stats_fn = sourcedir + src_stats_basename
  
    ;; ------------------------------------------------------------------------
    ;; Construct OR update the UNNAMED source.stats FITS file where the coordinates are stored,.
    ;; We must be very careful here!
    ;;
    ;; * In the AE recipe, ae_source_manager writes RA/DEC to this file, then ae_make_catalog
    ;;   is called many times using a catalog with RA=0,DEC=0 as a flag to indicate that we
    ;;   should read the coordinates from source.stats.
    ;;   In this case we specifically DO NOT WRITE the RA/DEC values to the stats file 
    ;;   in order to avoid rounding errors from repeated conversion during FITS I/O.
    ;;
    ;; * In general, we want the observer to be able to populate source.stats with useful
    ;;   keywords (e.g. PROVENAN) and have them not be destroyed here.
    ;;
    ;; * However, it's good programming practice to whack all the keywords that AE wrote
    ;;   from a previous extraction since we're changing the source fundamentals here
    ;;   (position and PSF fraction).
    
    ; Look for an existing unnamed version of source.stats file, which is where coordinates are stored.
    if file_test(unnamed_src_stats_fn) then begin
        ; We try a few times because file access can collide with another AE process that's writing the file.
      for kk=1,10 do begin
        unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
        if ~keyword_set(error) then break
        print, error
        wait, kk*0.5
      endfor ;kk
    endif else begin
      error = 1
    endelse
    
    
    if keyword_set(error) then begin
      if (ra[ii] EQ 0) AND (dec[ii] EQ 0) then begin
        print, 'ERROR: coordinates in catalog are (0,0) but cannot find an existing file '+unnamed_src_stats_fn
        GOTO, FAILURE
      endif

      ; Create a new FITS header to hold the fundamental source property keywords.
      file_mkdir, sourcedir  
      fxhmake, unnamed_src_stats, /INITIALIZE, /EXTEND, /DATE
      get_date, date_today, /TIMETAG
      psb_xaddpar, unnamed_src_stats, 'CREATOR' , creator_string
      psb_xaddpar, unnamed_src_stats, 'OBJECT'  , sourcename[ii], 'source name'
      psb_xaddpar, unnamed_src_stats, 'POSNDATE', date_today, 'UTC date RA,DEC were changed'
      psb_xaddpar, unnamed_src_stats, 'RA'      ,     ra[ii], '[deg] source position', F='(F10.6)'
      psb_xaddpar, unnamed_src_stats, 'DEC'     ,    dec[ii], '[deg] source position', F='(F10.6)'
      psb_xaddpar, unnamed_src_stats, 'POSNTYPE',  'unknown', 'type of source position'
      psb_xaddpar, unnamed_src_stats, 'PROVENAN',  'unknown', 'source provenance'
      psb_xaddpar, unnamed_src_stats, 'BKSCL_LO',        20.0, 'smallest BACKSCAL allowed'
      psb_xaddpar, unnamed_src_stats, 'BKSCL_GL',        30.0, 'target   BACKSCAL'
      psb_xaddpar, unnamed_src_stats, 'BKSCL_HI',        40.0, 'largest  BACKSCAL allowed'

      ; Create the unnamed version of source.stats to store these fundamental source property keywords.
      writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
;     print, 'Creating new file source.stats.'
    endif else begin
      ; Then we either read OR write the RA/DEC keywords, depending on how AE is being called.
      if (ra[ii] EQ 0) AND (dec[ii] EQ 0) then begin
        ; This branch is taken when ae_make_catalog is the caller.
        ra [ii] = psb_xpar( unnamed_src_stats, 'RA' , COUNT= ra_available)
        dec[ii] = psb_xpar( unnamed_src_stats, 'DEC', COUNT=dec_available)
        if ~ra_available || ~dec_available then message, 'ERROR: RA & DEC are missing from '+unnamed_src_stats_fn
;       print, 'Coordinates taken from the existing file source.stats.'                                     
      endif else begin
        ; This branch is taken when there is an existing unnamed_src_stats_fn and 
        ; the observer has supplied a normal catalog with coordinates.
        get_date, date_today, /TIMETAG
        psb_xaddpar, unnamed_src_stats, 'POSNDATE', date_today, 'UTC date RA,DEC were changed'
        psb_xaddpar, unnamed_src_stats, 'RA'      ,     ra[ii], '[deg] source position', F='(F10.6)'
        psb_xaddpar, unnamed_src_stats, 'DEC'     ,    dec[ii], '[deg] source position', F='(F10.6)'
;       print, 'Coordinates saved to existing file source.stats.'

        ; We should not have to change the OBJECT property, but we go ahead and do so to ensure
        ; that it matches the sourcename.
        psb_xaddpar, unnamed_src_stats, 'CREATOR', creator_string
        psb_xaddpar, unnamed_src_stats,  'OBJECT', sourcename[ii], 'source name'
        
        ; Overwrite the fundamental source property keywords to the unnamed version of source.stats.
        writefits, unnamed_src_stats_fn, 0, unnamed_src_stats
      endelse
    endelse ;Existing source.stats file found.
    
    
    
    ;; ------------------------------------------------------------------------
    ;; Construct filenames for specific obsid.
    obsdir     = sourcedir + obsname + '/' + extraction_subdir[ii]
    psf_fn     = sourcedir + obsname + '/' + psf_basename
    obs_parameters_fn = obsdir + obs_parameters_basename
    obs_stats_fn      = obsdir + obs_stats_basename
    region_fn         = obsdir + src_region_basename
    
    ;; ------------------------------------------------------------------------
    ;; Convert RA,DEC to the (x,y) system of this obsid.
    ;; Skip sources that fall off the field of view.
    dmcoords_cmd = string(obsdata_filename, aspect_fn, ra[ii], dec[ii],  F="(%'dmcoords %s asolfile=%s opt=cel celfmt=deg ra=%10.6f dec=%10.6f')")
    run_command, dmcoords_cmd, /QUIET

    run_command, /QUIET, 'pget dmcoords x y theta chip_id', dmcoords_result
    ; Parse the string returned by pget with the ON_IOERROR mechanism enabled in order to find type conversion errors.
    ON_IOERROR, TYPE_CONVERSION_ERROR1
    xpos_catalog = float(dmcoords_result[0])
    ypos_catalog = float(dmcoords_result[1])
    off_angle    = float(dmcoords_result[2])  ; arcmin
    chip_id      = fix  (dmcoords_result[3])
    if (0) then begin
      TYPE_CONVERSION_ERROR1:
      print, !ERROR_STATE.MSG
      print, 'ERROR: dmcoords results could not be parsed.'
      forprint, ['  The dmcoords call was : ', '    '+dmcoords_cmd   ]
      forprint, ['  The output of pget was: ', '    '+dmcoords_result]
      GOTO, FAILURE
    endif 
    ON_IOERROR, NULL


    ;; Find the source position in the exposure map and determine if that
    ;; position was observed.
    ad2xy, ra[ii], dec[ii], emap2wcs_astr, xindex, yindex 
    xindex = round(xindex)
    yindex = round(yindex)
    
    if ((xindex GE 0) AND (xindex LT emap_xdim) AND $
        (yindex GE 0) AND (yindex LT emap_ydim)) then begin
      ; Source is on the emap array, so test the emap value at the source position.
      emap_val_i        = emap[xindex,yindex]
      emap_is_small[ii] = emap_val_i LT emap_threshold
      
    endif else emap_is_small[ii] = 1 ; Source is outside the emap array. 
        
    source_not_observed[ii] OR= emap_is_small[ii]
      

    
    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    ; If we made it here then the source is not write protected.
    ; If its emap value is small (off-field) then remove any existing obs.stats file to communicate to other AE stages that the source is off the field.
    ; This situation can occur when an on-field source is extracted, and then moves off-field.
    if emap_is_small[ii] then begin
      if file_test(obs_stats_fn) then begin
        print, F='(%"\n===================================================================")'
        print, obs_stats_fn, sourcename[ii], strtrim(psb_xpar( unnamed_src_stats,'LABEL'),2), F='(%"WARNING: removing the existing file %s because source %s (%s) is no longer on this ObsID!")'
        print, F='(%"===================================================================")'
        
        file_delete, obs_stats_fn, /ALLOW_NONEXISTENT
      endif
      continue
    endif
    
    if keyword_set(query_only) then continue

    
    
    ; Remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    
    ;; ------------------------------------------------------------------------
    ;; We are going to process this source, so announce the "next source".
    print, F='(%"\n===================================================================")'
    print, sourcename[ii], strtrim(psb_xpar( unnamed_src_stats,'LABEL'),2), xpos_catalog, ypos_catalog, F='(%"\nSource %s (%s) at sky=(%d,%d):")'

    if (off_angle GT 50) then begin
      print, 'ERROR: dmcoords computed crazy theta value for source: ', off_angle
      forprint, ['  The dmcoords call was : ', '    '+dmcoords_cmd   ]
      forprint, ['  The output of pget was: ', '    '+dmcoords_result]
      GOTO, FAILURE
    endif
    
    file_mkdir, obsdir

    ;; In Dec 2007 I used MARX simulations at 1.5 keV with the readout streak disabled 
    ;; to measure PSF fractions at 1.5 keV as a function of off-axis angle.  
    ;; These polynomial curves were fit to those measurements.
    ;; The off-axis angle off_angle is in arcminutes.
    radius50 = (0.85 -0.25 *off_angle + 0.10 *off_angle^2) * arcsec_per_skypixel  ; arcsec
    radius90 = (1.54 -0.315*off_angle + 0.217*off_angle^2) * arcsec_per_skypixel  ; arcsec
    radius95 = (2.45 -0.32 *off_angle + 0.24 *off_angle^2) * arcsec_per_skypixel  ; arcsec
    radius98 = (5.51 -0.87 *off_angle + 0.31 *off_angle^2) * arcsec_per_skypixel  ; arcsec
 

    ; At this point we think this source was observed by this ObsID, and we are about to revise any existing extraction of the source in this ObsID.  Formally, that revision makes stale any existing MERGE for this source that involves this ObsID.  A defensive design would, at this point, destroy all such merges.  Doing that is complicated,  because we may have multiple named merges, and not all of them may have used this ObsID.  Doing that may also be expensive in computer time (since there would be a lot of I/O involved).  Doing that may also annoy the oberserver, since for some purposes a stale merge in the hand is better than a fresh merge not yet computed.
    
    ; So, if we're not going to search and destroy existing merges, what about the existing keywords in unnamed_src_stats that we have in our hands?  Should we remove merge-related keywords we find in there?  This FITS header (from file <sourcename>/source.stats) may or may not be part of an existing merge.  (Expert AE users tend to use named merges, and never run the un-named merge that would put information in unnamed_src_stats.)
    
    ; Through version 4864, we did indeed remove merge-related keywords from unnamed_src_stats, and resave to unnamed_src_stats_fn.  However, that write to unnamed_src_stats_fn would sometimes create a race condition for extractions of other ObsIDs that are running in parallel.  Reads to source.stats would fail, and the poor user would have to intervene to restart the failed IDL session.  So, after v4864 we no longer "clean" source.stats here!
    
    
    ;; ------------------------------------------------------------------------
    ;; Use a set of PSFs computed earlier if possible.
    
    ; Look for PSF with desired fiducial_psf_energy value & current format standard.
    psf_found = 0
    fits_open, psf_fn, fcb, /NO_ABORT, MESSAGE=error
    if NOT keyword_set(error) then begin
      for extension =0, fcb.NEXTEND do begin
        fits_read, fcb, dum, header, /HEADER_ONLY, /NO_PDU, EXTEN_NO=extension

        psf_found = (abs(fiducial_psf_energy[ii] - psb_xpar( header, 'ENERGY')) LT 0.1) AND $
                    (psb_xpar( header, 'PSF_TOTL') NE 0) AND (psb_xpar( header, 'RADIUS50') NE 0)

        if psf_found then break
      endfor
      fits_close, fcb
    
      ; Unless /REGION_ONLY we require PSFs at multiple energies.
      if (~keyword_set(region_only)) && ((1+fcb.NEXTEND) LT n_elements(psf_model_energy)) then psf_found = 0
    endif
    
    ; Discard any existing PSF that is intolerably far from the source position (0.1 skypix, chosen arbitrarily). 
    ; We *think* that a mispositioning of the PSF image in the SKY system will have the following effects:
    ;   - Extraction aperture will be shifted from ideal position.  
    ;   - A systematic error will be introduced into the PSF fraction, just as happens when the source position is wrong. 
    ;   - The shifted aperture will induce a tiny bias in the mean data position estimate because the wings of the observed source are asymmetricly cropped.
    ;   - The astrometry of image recons should NOT be damaged, because that code depends on the position of the PSF model on the sky (recorded in the PSF header), not the current source position.
    ;   - The source models used for "better backgrounds" and "better masking" will be shifted on the sky, slightly damaging the accuracy of those tools.
    
    if psf_found then begin
      psf_found = 0
      
      ; Look for PSF position recorded by ae_make_psf.
      xpos_old = psb_xpar( header, 'X_CAT', COUNT=x_count)
      ypos_old = psb_xpar( header, 'Y_CAT', COUNT=y_count)
      
      ; Or look for mkpsf parameters in the HISTORY kywds.
      if (x_count EQ 0) || (y_count EQ 0) then begin
        x_result = stregex(header,':x=([0-9]+\.[0-9]*)',/SUB,/EXT)
        y_result = stregex(header,':y=([0-9]+\.[0-9]*)',/SUB,/EXT)
        x_ind = where(x_result[1,*],x_count)
        y_ind = where(y_result[1,*],y_count)
        if (x_count EQ 1) && (y_count EQ 1) then begin
          xpos_old = float(x_result[1,x_ind[0]])
          ypos_old = float(y_result[1,y_ind[0]])  
        endif
      endif
      
      if (x_count EQ 1) && (y_count EQ 1) then begin
        psf_found = (abs(xpos_catalog - xpos_old) LE 0.1) AND $
                    (abs(ypos_catalog - ypos_old) LE 0.1)
      endif
    endif

    ; If caller specifies REUSE_PSF=0 then ignore existing PSFs.
    if ~keyword_set(reuse_psf) then psf_found = 0
    
    if psf_found then begin
      psf_img = readfits(psf_fn, psf_header, EXT=extension, /SILENT)
      print, 'Using existing PSF images ...'
      GOTO, PSF_IN_HAND
    endif

    
    ;; We arbitrarily choose PSF pixel sizes to meet the goal of having at least  
    ;; min_num_across pixels across the 50% PSF fraction radius.
    min_num_across    = 10
    
     
    ;; ------------------------------------------------------------------------
    ;; Generate PSFs via MARX.
    if ae_make_psf_not_configured then begin
      ; In this setup call, pass cache_dir as TEMP_DIR so that marx.par will be preserved when tempdir is whacked. 
      ae_make_psf, TEMP_DIR=cache_dir, EVENT_FILE=obsdata_filename, OBS_ASPECT_FN=aspect_fn, ASPECT_BLUR=aspect_blur, PIX_ADJ=pix_adj
      ae_make_psf_not_configured = 0
    endif
    
    ; Calls to ae_make_psf below require a CHIP_ID input.
    ; Its only use there is to set the MARX parameter DetectorType.
    ; The effect of DetectorType inside MARX is not clear.  It probably determines the default SIM position,
    ; however ae_make_psf explicitly controls the SIM position.  Although MARX may have CCD-specific parameters
    ; for some of its optional calculations, here MARX is only generating photon rays.
    ; Thus, I would wager that the CHIP_ID input to ae_make_psf does not alter the PSF generated.
    ; Thus, I assume that checking the accuracy of CHIP_ID (calculated by dmcoords) is not very important.
    ; But, I will defensively print a warning is CHIP_ID is not one of the CCDs active in this observation.
    if  ~ccd_is_in_observation[chip_id] then begin
      print, strjoin(strtrim(where(ccd_is_in_observation),2),','), chip_id, F='(%"WARNING! Event list contains data from CCDs %s but dmcoords says this source is on CCD %d.")'
    endif

    ; The list of requested PSF energies is carried in psf_model_energy.
    ; The 'catalog' read earlier has a column 'fiducial_psf_energy' specifying, for each source independently, which energy should be used to make the extraction region.  That flexibility was probably a poor design decision, but we're stuck with it.
    ; Since we will later contour the *first* PSF to build the extraction region, here we want to re-order psf_model_energy so that the first element is closest to fiducial_psf_energy.
    dum = min(abs(psf_model_energy - fiducial_psf_energy[ii]), imin)
    
    psf_energy               = psf_model_energy
    psf_energy        [0]    = psf_model_energy[imin]
    psf_energy        [imin] = psf_model_energy[0]
    
    desired_psf_counts       = psf_model_counts
    desired_psf_counts[0]    = psf_model_counts[imin]
    desired_psf_counts[imin] = psf_model_counts[0]
    
    if keyword_set(region_only) then begin
      ; If /REGION_ONLY then only build the first PSF image.
      psf_energy         = psf_energy[0]
      desired_psf_counts = desired_psf_counts[0]
    endif 
    
    ; Choose a pixel size that's convenient for boxcar smoothing and that samples the core well.
    binspa_candidates = 1.0 / [  3,5,7,9,11,13]  
   
    for jj=0,n_elements(binspa_candidates)-1 do begin
      skypixel_per_psfpixel = binspa_candidates[jj]
      
      num_across = radius50 / (arcsec_per_skypixel * skypixel_per_psfpixel)
      
      if (num_across GE min_num_across) then break
    endfor; jj
    
    ; In this call MARX will be run, so pass tempdir as TEMP_DIR.
    footprint = 16 > (2*radius98/arcsec_per_skypixel)  ; sky pixels
    footprint *= psf_footprint_multiplier
    retain_eventlist = keyword_set(simulate_sources)

    ae_make_psf, TEMP_DIR=tempdir, $
                 ra[ii], dec[ii], psf_fn, skypixel_per_psfpixel, footprint, psf_energy, desired_psf_counts, $
                 X_CAT=xpos_catalog, Y_CAT=ypos_catalog, OFF_ANGLE=off_angle, CHIP_ID=chip_id, EMAP_VAL=emap_val_i, $
                 RETAIN_EVENTLIST=retain_eventlist

    psf_img = readfits(psf_fn, psf_header, EXT=0, /SILENT)


PSF_IN_HAND:
    if keyword_set(psf_only) then continue

    psf_total = psb_xpar( psf_header, 'PSF_TOTL')
    
    ;; ------------------------------------------------------------------------
    ;; Set all the NaN values to zero to keep total,max,contour, etc. routines happy.
    ind = where(finite(psf_img) EQ 0, count)
    if (count GT 0) then psf_img[ind] = 0
    
    ;; ------------------------------------------------------------------------
    ;; Get some astrometry information about PSF image.
    xdim = (size(psf_img, /DIM))[0]
    ydim = (size(psf_img, /DIM))[1]
        
    ; We cannot use xy2ad.pro/ad2xy.pro for conversions between array index and PHYSICAL (sky) coordinate systems.
    crvalP = [psb_xpar( psf_header, 'CRVAL1P'), psb_xpar( psf_header, 'CRVAL2P')]
    crpixP = [psb_xpar( psf_header, 'CRPIX1P'), psb_xpar( psf_header, 'CRPIX2P')]
    cdeltP = [psb_xpar( psf_header, 'CDELT1P'), psb_xpar( psf_header, 'CDELT2P')]

    ;; Convert the source position to a real-valued quantity in the PSF image array index coordinate system.
    xind_catalog = (crpixP[0] + (xpos_catalog-crvalP[0])/cdeltP[0]) - 1
    yind_catalog = (crpixP[1] + (ypos_catalog-crvalP[1])/cdeltP[1]) - 1

    ;; Make an array that has the distances (in PSF pixel units) from each pixel to the source.
    dist_circle, distance, [xdim,ydim], xind_catalog, yind_catalog
      
    ;; ------------------------------------------------------------------------
    ;; To later assess pileup, estimate the PSF fraction contained in a 3x3 sky pixel cell.
    cell_radius_skypix = 1.5 ;skypix (1/2 of a 3x3 skypix cell)
    cell_radius_psfpix = cell_radius_skypix/cdeltP[0] ; PSF pixels
    
    ind = where(distance LE cell_radius_psfpix, count)
    
    square_to_circle_area_ratio = (2*cell_radius_psfpix)^2 / count ; Numerator and denominator are in units of psfpix^2.
    
    cell_frac = 1.0 < (square_to_circle_area_ratio * (total(psf_img[ind], /DOUBLE) / psf_total))
    ; Above, the right term is the fraction of PSF power within the circle that has a 3 skypix diameter.
    ; The left term is the ratio between the area of a square 3x3 cell and the area of our circle; this crude extrapolation is equivalent to assuming that the PSF outside the circle is equal the the mean PSF within the circle (which is clearly not true).
    ; With more code, we could of course find the coordinates of each PSF pixel and then actually sum the PSF image over the square cell. 
      


    ;; ------------------------------------------------------------------------
    ;; Find offset between PSF centroid and requested position.
    ;; This is a CENTROID calculation, NOT a PSF FRACTION -- the denominator must
    ;; be the sum of the (cropped) psf_img array.
    denominator = total( psf_img, /DOUBLE )
    ind = lindgen(xdim,ydim)
    xpos_psf = total( psf_img * (ind mod xdim), /DOUB ) / denominator
    ypos_psf = total( psf_img * (ind  /  xdim), /DOUB ) / denominator

    xpos_psf = (xpos_psf+1-crpixP[0])*cdeltP[0] + crvalP[0]
    ypos_psf = (ypos_psf+1-crpixP[1])*cdeltP[1] + crvalP[1]

    psf2cat_offset_i = sqrt((xpos_psf-xpos_catalog)^2 + (ypos_psf-ypos_catalog)^2)
    

    ;; ------------------------------------------------------------------------
    ;; Find a contour of the psf_img which encloses the desired PSF fraction.
    
    ; Smooth the PSF (convolve with a Gaussian kernal) a little to produce smoother extraction regions (contours) below.
    ; As I recall, we require the kernel to have odd dimensions so that smooth() will not shift the astrometry of the smoothed PSF.
    arcsec_per_sigma    = 0.1 * (radius50 / 0.85)                ; 0.1 arcsecond at theta=0
    arcsec_per_psfpixel = psb_xpar( psf_header, 'CDELT2') * 3600
    PSFpixel_per_sigma  = arcsec_per_sigma / arcsec_per_PSFpixel ; PSF pixel
  
    kernel  = psf_gaussian( NPIXEL=(      1 + 2*(ceil(3*PSFpixel_per_sigma))), $
                            FWHM  =(2*sqrt(2* aLog(2)))*PSFpixel_per_sigma, /NORMALIZE )  ; FWHM=2.355*sigma
  
    smooth_psf_img = convol( psf_img, kernel, /CENTER, /EDGE_TRUNCATE )

    
    ;; Try to make a good initial guess for the contour by assuming the PSF is a cone.
    ;; Initial value for step corresponds to a change in fraction of 0.05
    peak = max(smooth_psf_img)
    contour_level = peak *          (1 - (target_psf_fraction[ii])^0.33)
    initial_step  = peak * 0.05 * 0.33 * (target_psf_fraction[ii])^(-0.66)
    step          = 0
    
    indent = ''
    done = 0
    loop_count = 0
    repeat begin
      loop_count = loop_count + 1
      
      ; Compute a single contour polygon.
      contour, smooth_psf_img, /CLOSED, LEVELS=[contour_level], $
               PATH_XY=xy, PATH_INFO=info, /PATH_DATA_COORDS
      ind = info[0].offset + indgen(info[0].N)
      polygon_x = float(reform(xy[0, ind]))
      polygon_y = float(reform(xy[1, ind]))
      
    
      ; Find the enclosed pixels & compute the PSF fraction.
      ind = polyfillv( polygon_x, polygon_y, xdim, ydim )
      
      psf_fraction_i = total(psf_img[ind], /DOUBLE) / psf_total

;tvscl, psf_img < contour_level
;plots,polygon_x,polygon_y,/dev,color=0
      
      ; Choose next search step.
      pf_error = target_psf_fraction[ii] - psf_fraction_i
      if (abs(pf_error) LT 0.01) then done = 1

      ; Initialize step based on sign of pf_error.
      if (step EQ 0) then step = initial_step * ((pf_error GT 0) ? -1 : 1)
      
      ; If the PSF fraction is too small and we're stepping up, then reverse.
      if ((pf_error GT 0) AND (step GT 0)) then step = -step/2.

      ; If the PSF fraction is too large and we're stepping down, then reverse.
      if ((pf_error LT 0) AND (step LT 0)) then step = -step/2.
      
      ; Stop if we're down to taking tiny steps.
      if (abs(step/contour_level) LT 0.01) then done = 1

      ; Stop if we've taken an excessive number of steps.
      if (loop_count GT 20) then begin
        print, 'WARNING, contour search aborted!!!'
        done = 1
      endif

      ; Take a step, but avoid a non-positive level.
;print, psf_fraction_i, contour_level, step
      contour_level = (contour_level + step) > (contour_level/2.)
      contour_level<= peak*0.95
      indent = indent + ' '
    endrep until (done)
    
    print, psf_fraction_i, loop_count, F='(%"Found PSF fraction %4.2f after trying %d contours.")' 
    
    ;; Compute the area of the enclosed pixels.
    src_area_i = n_elements(ind) * (cdeltP[0])^2

    
    ;; ------------------------------------------------------------------------
    ;; Construct a source extraction region corresponding to the contour.

;function_1d, id0, polygon_x,polygon_y, LINE=6, PSYM=1, COLOR='green', DATAS='contour'

    max_vertexes = 100
  
    ; The polygon obtained from CONTOUR can have a large number of closely-spaced vertices.
    ; We want to reduce the number of vertices to <max_vertexes (for performance) while avoiding the construction of 
    ; polygon segments that cross, which generates CIAO warnings.
    ; Several home-grown algorithms for simplying polygons have failed; now we are trying the Douglas-Peucker (DP) algorithm.
    ; The "tolerance" parameter specifies (I think) the max distance we allow between the original vertices and the DP polygon.
    Douglas_Peucker_tolerance = 0.5; PSF pixels
    show_report = 0
    
    repeat begin
      vertices = poly_simplify(transpose([[polygon_x],[polygon_y]]), TOL=Douglas_Peucker_tolerance)
    
      num_vertices = (size(vertices, /DIM))[1]
      
      if (num_vertices LE max_vertexes) then break
      
      ; Increase the tolerance parameter to reduce the number of vertices.
      Douglas_Peucker_tolerance *= 1.5
      show_report = 1
    endrep until 0

    if show_report then print, Douglas_Peucker_tolerance, num_vertices, F='(%"A Douglas-Peucker tolerance of %0.1f PSF pixels was required to achieve %d polygon vertices.")'
    
;function_1d, id0, vertices[0,*],vertices[1,*], LINE=6,PSYM=6, COLOR='red', DATAS='polygon'
    
    ;; Convert the polygon from PSF image indexes to physical (x,y) coordinates, 
    ;; keeping in mind the 1-based FITS convention vs the 0-based IDL array index convention.
    polygon_x = (vertices[0,*]+1-crpixP[0])*cdeltP[0] + crvalP[0]
    polygon_y = (vertices[1,*]+1-crpixP[1])*cdeltP[1] + crvalP[1]
    
    ;; Format x/y pairs into a list of polygon vertices.
    vertices = string(polygon_x, F='(F0.2)') + ',' + string(polygon_y, F='(F0.2)')
    
    ;; Eliminate duplicate vertices, which can arise from the finite precision of our formatted coordinates.
    ;; Starting in version 4.3, CIAO complains if polygon line segments have zero length.
    discard = bytarr(num_vertices)
    for this_ind=0L, num_vertices-1 do begin
      next_ind = (this_ind+1) mod num_vertices
 
      discard[this_ind] = (vertices[this_ind] EQ vertices[next_ind])
    endfor
        
    ind = where(discard EQ 0, num_vertices)
    vertices = vertices[ind]
    
    ;; Construct a ds9 polygon region specification, using whitespace to visually separate vertices.
    src_region = 'polygon(' + strjoin(vertices,",  ") + ')'

    ;; Estimate a "radius" for this polygon.
    polygon_radii  = sqrt((polygon_x-xpos_catalog)^2 + (polygon_y-ypos_catalog)^2)
    src_radius_max = max(polygon_radii, MIN=src_radius_min)

    ;; ------------------------------------------------------------------------
    ;; Find a circular mask region which excludes almost all of the PSF.
    
    ;; Start with the radius that encloses the source region.
    ;; Search for a radius (in PSF pixel units) that encloses mask_fraction of the light.
    for mask_radius = src_radius_max/cdeltP[0],xdim/2 do begin
      ind = where(distance LE mask_radius)
      this_mask_fraction = (total(psf_img[ind], /DOUBLE) / psf_total)
      
      if (verbose GT 1) then $
        print, 100*this_mask_fraction, mask_radius*cdeltP[0], F='(%"%5.1f%% @ mask_radius=%5.1f")'
      if (this_mask_fraction GT mask_fraction) then break
    endfor
   
    ;; Convert the mask radius from units of PSF pixels to sky pixels.
    mask_radius = mask_multiplier * (mask_radius*cdeltP[0])
    if (verbose GT 1) then print, mask_radius, F='(%"mask_radius=%5.1f skypix")'

    mask_radii[ii] = mask_radius
    mask_region = string(xpos_catalog, ypos_catalog, mask_radius, F='(%"circle(%f,%f,%f)")')

      
    ; Determine if observer has "frozen" the aperture by write-protection the aperture region file.
    if file_test(region_fn) && ~file_test(/WRITE, region_fn) then begin
      ; RETAIN EXISTING APERTURE
      region_edited = 1  ; Observer has edited/protected the aperture region file.
      
      ; Read the polygon provided and calculate its area, so that other tools (e.g. ae_make_catalog) have access to accurate PGN_AREA and SRC_RAD keywords in obs.stats.
      ae_ds9_to_ciao_regionfile, region_fn, '/dev/null', /IGNORE_BACKGROUND_TAG, POLYGON_X=polygon_x, POLYGON_Y=polygon_y
      
      src_area_i = poly_area(polygon_x, polygon_y)  ; [skypixel**2]
      
      
      ; For an aperture provided by observer, two properties we are about to write to obs.stats are not well defined.
      psf_fraction_i          = !VALUES.F_NAN
      target_psf_fraction[ii] = !VALUES.F_NAN
      
      ; The "mask radius" property (MSK_RAD in obs.stats) is poorly-designed.  Originally, it simply recorded the radius of a circular mask constructed above.  But later other tools (e.g. ae_better_backgrounds) used MSK_RAD to estimate the general size of the PSF.
      ; Thus, here we must ignore any mask region that may (or may not) be in the aperture file provided by the observer, and record (as MSK_RAD in obs.stats) the mask_radius quantity calculated above.
      
      ; Even when the observer supplies the aperture, the "fiducial_psf_energy" property (PSF_ENGY in obs.stats) remains a valid property of the source extraction, used by code elsewhere.  Thus we do NOT set this quantity to NULL.
      
      print, region_fn,  sqrt(src_area_i/!PI), F="(%'\nWARNING!  The existing (write-protected) aperture %s (radius~=%0.1f skypix) has been adopted.')"
      
    endif else begin
      ; CONSTRUCT APERTURE
      region_edited = 0  ; Observer has not edited/protected the aperture region file.
      
      ;; ------------------------------------------------------------------------
      ;; Write the source region to a file.
      openw,  region_unit, region_fn, /GET_LUN
      printf, region_unit, "# Region file format: DS9 version 3.0"
     ;printf, region_unit, 'global width=1 font="helvetica 12 normal"'
      printf, region_unit, src_region, obsname, F='(%"%s # move=0 tag={%s}")' 
      
      ;; ------------------------------------------------------------------------
      ;; Add mask region to the file.
      printf, region_unit, mask_region + " # color=red background"
      free_lun, region_unit
    endelse ; CONSTRUCT APERTURE
    

    ;; ------------------------------------------------------------------------
    ;; Save information for summary plots.
    ;; WE DISCARD ANY EXISTING obs.stats FILE ... when the aperture changes, ALL source properties become stale.
    fxhmake,  obs_stats, /INITIALIZE, /EXTEND, /DATE
    psb_xaddpar, obs_stats, 'CREATOR', creator_string
    psb_xaddpar, obs_stats, 'OBSNAME',  obsname, 'observation identifier'
    psb_xaddpar, obs_stats, 'EMAP_AVG', emap_val_i, '[s cm**2 count /photon] exposure map value nearest source'
    psb_xaddpar, obs_stats, 'THETA',    off_angle, '[arcmin] off-axis angle'
    psb_xaddpar, obs_stats, 'PSF2CAT',  psf2cat_offset_i,  '[skypixel] PSF to catalog offset'
    psb_xaddpar, obs_stats, 'FRACSPEC', target_psf_fraction[ii], 'target PSF fraction'
    psb_xaddpar, obs_stats, 'PSF_FRAC', psf_fraction_i, string(fiducial_psf_energy[ii], F='(%"PSF fraction via polyfillv @%6.4f keV")')
    psb_xaddpar, obs_stats, 'PSF_ENGY', fiducial_psf_energy[ii], '[keV] fiducial PSF energy'
    psb_xaddpar, obs_stats, 'MSK_RAD',  mask_radius,  '[skypixel] mask radius'
    psb_xaddpar, obs_stats, 'PGN_AREA',      src_area_i,      '[skypixel**2] src polygon area, via polyfillv/poly_area'
    psb_xaddpar, obs_stats, 'SRC_RAD',  sqrt(src_area_i/!PI), '[skypixel] sqrt(PGN_AREA/!PI)'
    psb_xaddpar, obs_stats, 'REG_EDIT', region_edited ? 'T' : 'F', 'T=aperture edited by observer'
    psb_xaddpar, obs_stats, 'CELLFRAC', cell_frac   , 'PSF fraction in 3x3 skypix cell'
    psb_xaddpar, obs_stats, 'X_CAT',    xpos_catalog, '[skypixel] source position, from catalog'
    psb_xaddpar, obs_stats, 'Y_CAT',    ypos_catalog, '[skypixel] source position, from catalog'
    psb_xaddpar, obs_stats, 'X_PSF',    xpos_psf    , string(fiducial_psf_energy[ii], F='(%"[skypixel] centroid of PSF image @%6.4f keV")')
    psb_xaddpar, obs_stats, 'Y_PSF',    ypos_psf    , string(fiducial_psf_energy[ii], F='(%"[skypixel] centroid of PSF image @%6.4f keV")')
    writefits, obs_stats_fn, 0, obs_stats

  endfor ;ii
  
  ;; Check for mask radii (units of skypixel) that are small with respect to the emap pixel size.
  emap_pixel_size = psb_xpar( emap_header,'CDELT1P')  ; units are skypixel

  if (emap_pixel_size GT 1.1) then begin 
    print, F='(%"\nWARNING!  We strongly recommend an exposure map pixel size of 1 SKY pixel or smaller, particularly if the tools ae_better_masking or ae_better_backgrounds will be used.")'
  endif
  
  span = mask_radii / emap_pixel_size
  ind = where( (span NE 0) AND (span LT 4), count )
  if (count GT 0) then begin
    print
    print, 'WARNING!  The following sources have mask radii that are small with respect to the emap pixel size.'
    print, 'Once the mask is pixelized, the source masking may be inadequate.'
    forprint, sourcename[ind], mask_radii[ind]
  endif ;ii
  
  count = total(source_not_observed, /INT)
  if (count GT 0) then print, count, F="(%'\nWARNING!  CONSTRUCT_REGIONS skipped for %d sources not observed.')"

endif ;keyword_set(construct_regions)



;; =============================================================================
if keyword_set(show_regions) && keyword_set(collated_filename) then begin
;; =============================================================================
  ;; Get rid of pre-existing configuration for the CIAO commands we'll use below.
  run_command, /QUIET, 'pset dmcopy clobber=yes'

  deghour = 24D/360
  
  pan_only = keyword_set(display_file) AND keyword_set(region_file)  
  
  option_string = (pan_only ? '-lock colorbar yes ' : '-bin buffersize 512 ')
  
  if keyword_set(ds9_option_string) then option_string += ds9_option_string
  ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, NAME='acis_extract_'+session_name, OPTION_STRING=option_string

      
  ;; Read the source properties of all the sources.
  bt = mrdfits(collated_filename, 1, theader, /SILENT, STATUS=status)
  if (status NE 0) then begin
    print, 'ERROR reading ', collated_filename
    GOTO, FAILURE      
  endif
  
  num_sources = n_elements(bt)

  ;; Look up the source properties of all the sources.
  sourcename      = strtrim(bt.CATALOG_NAME   ,2)
  ra              =         bt.RA                 
  dec             =         bt.DEC                
  label           = strtrim(bt.LABEL          ,2) 
  posntype        = tag_exist(bt, 'POSNTYPE') ? strtrim(bt.POSNTYPE,2) : '' 
    
  source_notes      =         strarr(num_sources)


  if pan_only then begin
    ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, display_file, region_file
    point_ra = 0
    merge_available = 0
    recon_available = 0
    num_additional_display_file = 0
  endif else begin
    if keyword_set(region_file) && file_test(region_file) && (file_lines(region_file) GT 0) then begin
      ; Count any observer-supplied "point" or "panda" or "text" regions.
      print, 'Reading ', region_file
      readcol, region_file, lines, F='(A)', DELIM='@'
      ind = where( (strmid(lines,0,1) NE '#') AND stregex(/BOOLEAN, /FOLD_CASE, lines, 'point|panda|text'), num_external_points )
      print, region_file, num_external_points, F='(%"%s contains %d point or panda or text regions")'
    endif else num_external_points = 0
    
    ; Make arrays to hold region strings & coordinates for observer-supplied and catalog positions.
    point_region = strarr(num_external_points+num_sources)
    point_ra     = dblarr(num_external_points+num_sources)
    point_dec    = dblarr(num_external_points+num_sources)
  
    if (num_external_points GT 0) then begin
      ; Parse external "point" regions for RA & DEC values.
      ; This regular expression should work for negative DEC, and space or comma separation, and with or without ().
      lines  = lines[ind]
      result = stregex(lines,'(point|panda|text)[^0-9]*([0-9]+\.[0-9]+)[^-0-9]*(-*[0-9]+\.[0-9]+)',/SUB,/EXT)
      point_ra    [0] = double(reform(result[2,*]))
      point_dec   [0] = double(reform(result[3,*]))
      point_region[0] = lines
    endif 
      
    ; Append "point" regions that mark all the sources in the catalog.
    point_ra    [num_external_points] = ra
    point_dec   [num_external_points] = dec
    for ii = 0L, num_sources-1 do begin
      point_region[num_external_points+ii] = string(ra[ii], dec[ii], label[ii], F='("J2000;point(",F10.6,",",F10.6,") # point=box color=red width=3 text={",A,"}")')
    endfor
  endelse
 
  
  ; Use a subset of the sources if requested.
  if keyword_set(srclist_filename) then begin
    readcol, srclist_filename, sourcenames_to_show, FORMAT='A', COMMENT=';', DELIM='@'
    
    ; Trim whitespace and remove blank lines.
    sourcenames_to_show = strtrim(sourcenames_to_show,2)
    ind = where(sourcenames_to_show NE '', num_to_show)
    
    if (num_to_show EQ 0) then begin
      print, 'ERROR: no entries read from source list ', srclist_filename
      GOTO, FAILURE
    endif
    
    sourcenames_to_show = sourcenames_to_show[ind]
    print, num_to_show, srclist_filename, F='(%"\n%d sources found in %s\n")'
    
    ; Create an array specifying the indexes into bt corresponding to the sources listed in sourcenames_to_show.
    presentation_list = replicate(-1L, num_to_show)
    
    for ii = 0,num_to_show-1 do begin
      ; Parse lines with semicolons into source names and notes.
      ind = strpos(sourcenames_to_show[ii],';')
      if (ind NE -1) then begin
        this_note               = strtrim(strmid(sourcenames_to_show[ii],ind+1) ,2)
        sourcenames_to_show[ii] = strtrim(strmid(sourcenames_to_show[ii],0,ind) ,2)
      endif else this_note = ''
    
      ind = where(sourcename EQ sourcenames_to_show[ii], count)
      
      if (count GT 0) then begin
        presentation_list[ii] = ind[0]
        source_notes           [ind[0]] += this_note
      endif else print, sourcenames_to_show[ii], collated_filename, F='(%"WARNING: source %s is missing from table %s")'
      
      if (count GT 1) then print, sourcenames_to_show[ii], collated_filename, F='(%"WARNING: source %s appears multiple times in table %s")'
    endfor ;ii
    
    ind = where(presentation_list NE -1, count)
    if (count EQ 0) then begin
      print, 'Zero sources listed in ', srclist_filename, ' were found in ', collated_filename
      GOTO, FAILURE      
    endif
    presentation_list = presentation_list[ind]
    
    
  endif else if keyword_set(index_file) then begin
    ; Defined a subset and/or ordering of the catalog using supplied ASCII 1-based index file.
    readcol, index_file, source_num, FORMAT='L', COMMENT=';'
    num_to_show = n_elements(source_num)
    
    ; source_num is 1-based but indexing is 0-based.
    presentation_list  = 0 > (source_num-1) < (num_sources-1)
  endif else presentation_list = lindgen(num_sources)
    
  
  ; Because our "source list" was read here (via FITS table) rather than at the top of AE (via catalog_or_srclist)
  ; we need to deal with some loose ends now.
  ; MERGE_NAME is taken from AE keyword parameter if available, or from collated table otherwise.
  if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                  else extraction_subdir = ''
  if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)
  
  dum = where(tag_names(bt) EQ 'MERGE_NAME', count)
  if keyword_set(merge_name) then begin
    merge_subdir = merge_name    + '/'
  endif else if (count EQ 1) then begin
    merge_subdir = strtrim(bt.MERGE_NAME,2) + '/'
  endif else begin
    merge_subdir = ''
  endelse
  
  if (n_elements(merge_subdir) EQ 1) then merge_subdir = replicate(merge_subdir,num_sources>1)

  
  ; Convert RAs to hours for gcirc below.
        ra_hrs =       ra * deghour
  point_ra_hrs = point_ra * deghour
  
  energy_range_label = string(energy_range, F='(%"[%0.2f:%0.2f] keV ")')
  desired_zoom = 8
  
  presentation_index = 0L
  ii = presentation_list[presentation_index]
  
  flush_stdin  ; eat any characters waiting in STDIN, so that they won't be mistaken as commands in the loop below.
  while 1 do begin
    ;; Build names for the files we'll need to access.
    sourcedir            = sourcename[ii] + '/' + merge_subdir[ii] 
    src_stats_fn         = sourcedir            + src_stats_basename
    merged_env_events_fn = sourcedir + env_events_basename
    merged_src_events_fn = sourcedir + src_events_basename
    composite_img_fn     = sourcedir + env_image_basename
    merged_region_fn     = sourcedir + src_region_basename

    if pan_only then ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, PAN_TO_COORDS=[ra[ii], dec[ii]]
      
    ; Report position of current source in presentation_list.
    sequence_number = (where(presentation_list EQ ii, count))[0]
    sequence_msg    = (count EQ 0) ? '' : string(1+sequence_number, n_elements(presentation_list), F="(%'#%d of %d')")
  
    print, F='(%"\n===================================================================")'
    print, sourcename[ii], label[ii], posntype[ii], sequence_msg, source_notes[ii], F='(%"\nSource %s (%s) (%s) %s: %s")'
    
    ;; We can NOT search for "neighborhood.evt" because /MERGE could make a file
    ;; sourcename/extraction_name/neighborhood.evt which would be misinterpreted here as
    ;; an observation!  We must instead search for "obs.stats" which appears only in observation
    ;; directories, and then see which of those observations has a "neighborhood.evt".
    
    ; If the observer has specified a list of observations then look for them, otherwise look for any.
    pattern_base = sourcename[ii] + '/' + (keyword_set(obsname) ? obsname : '*') + '/'
    obs_stats_fn   = file_search( pattern_base + extraction_subdir[ii] + obs_stats_basename, COUNT=num_obs )
    
    if (num_obs EQ 0) then begin
      print, '  Not present in any observation'
      goto, COMMAND
    endif
    
    obs_dir_all = strarr(num_obs)
    for jj = 0, num_obs-1 do begin
      fdecomp, obs_stats_fn[jj], disk, dir
      obs_dir_all[jj] = dir
    endfor
    

    good_obs  = where(file_test(obs_dir_all + env_events_basename), num_obs)
    if (num_obs EQ 0) then begin
      print, 'ERROR: none of these neighborhoods could be found:'
      forprint, obs_dir_all + env_events_basename
      GOTO, FAILURE
    endif
    
    ;; ------------------------------------------------------------------------
    ;; Build a structure holding the properties of each ds9 frame we might display.
    f_nan = !VALUES.F_NAN
    frame_template = {ds9_frame_id         :'',$
                      obs_dir              :'',$
                      env_events_fn        :'',$
                      obs_stats_fn         :'',$
                      region_fn            :'',$
                      bkg_pixels_region_fn :'',$
                      hook_region_fn       :'',$
                      obsid_region_fn      :'',$
                      display_data_fn      :'',$
                      edited_region_fn     :'',$
                      obsname              :'',$
                      obsname_status       :'',$
                      emap_val             :f_nan,$
                      off_angle            :f_nan,$ ; [arcmin]
                      msk_rad              :f_nan,$ ; [skypixel]
                      src_counts           :f_nan,$
                      bkg_counts           :f_nan,$
                      backscal             :f_nan,$
                      backcorr             :f_nan,$
                      psf_frac             :f_nan,$
                      overlap              :f_nan,$
                      VOTE_LO              :'',$
                      VOTE_HI              :'',$
                      S_FILTER             :'',$
                      TSTART               :f_nan,$
                      show_t               :1B }
                    
    frame = replicate(frame_template, num_obs)
    
    frame.obs_dir              = obs_dir_all[good_obs]
    frame.env_events_fn        = frame.obs_dir + env_events_basename
    frame.obs_stats_fn         = frame.obs_dir + obs_stats_basename
    frame.region_fn            = frame.obs_dir + src_region_basename
    frame.bkg_pixels_region_fn = frame.obs_dir + bkg_pixels_region_basename
    frame.hook_region_fn       = frame.obs_dir + 'psf_hook.reg'
                    
    ; Look up in source.stats which extractions were merged.
    src_stats = headfits(src_stats_fn, ERRMSG=error)
    
    if keyword_set(error) then begin
      print, error
      print, 'ERROR: File '+src_stats_fn+' is missing!'
      goto, COMMAND    
    endif
    
    obsnames_merged = psb_xpar( src_stats,'OBSNAME', COUNT=count)
    pruning_known = (count GT 0)
    if ~pruning_known then begin
      obsnames_merged=''
      print, 'WARNING: no merge information available.'
    endif else obsnames_merged = strsplit(/EXTRACT, strtrim(obsnames_merged, 2), ',') 

    ; Look up various single-ObsID properties in obs.stats files.
    for jj = 0, num_obs-1 do begin
      ; Read this observation's statistics.
      stats = headfits(frame[jj].obs_stats_fn)
      this_obsname               = strtrim( psb_xpar( stats, 'OBSNAME'),2 )
      frame[jj].obsname          = this_obsname
      if pruning_known && array_equal(strmatch(obsnames_merged, this_obsname), 0) then begin
        frame[jj].obsname_status = 'pruned'
        frame[jj].show_t         = keyword_set(include_pruned_obsids)
      endif

      if keyword_set(obsid_region_glob_pattern) then $
        frame[jj].obsid_region_fn = repstr(obsid_region_glob_pattern, '*', frame[jj].obsname)
      
      frame[jj].TSTART           =          psb_xpar( headfits(frame[jj].env_events_fn, EXT=1), 'TSTART')
      
      frame[jj].emap_val         =          psb_xpar( stats, 'EMAP_AVG')
      frame[jj].off_angle        =          psb_xpar( stats, 'THETA')
      frame[jj].msk_rad          =          psb_xpar( stats, 'MSK_RAD') ; [skypixel]
      frame[jj].src_counts       =          psb_xpar( stats, 'SRC_CNTS')
      frame[jj].bkg_counts       =          psb_xpar( stats, 'BKG_CNTS')
      frame[jj].backscal         =          psb_xpar( stats, 'BACKSCAL')
      frame[jj].backcorr         =          psb_xpar( stats, 'BACKCORR')
      frame[jj].psf_frac         =          psb_xpar( stats, 'PSF_FRAC')
      frame[jj].overlap          =          psb_xpar( stats, 'OVERLAP')
      temp                       =          psb_xpar( stats, 'VOTE_LO')
      frame[jj].VOTE_LO          = (temp LT psb_xpar( stats, 'BKSCL_HI')) ? string(round(temp),F='(%"%d")') : ''
      temp                       =          psb_xpar( stats, 'VOTE_HI')
      frame[jj].VOTE_HI          = (temp GT psb_xpar( stats, 'BKSCL_LO')) ? string(round(temp),F='(%"%d")') : ''
      
      ; Determine what STATUS filter was chosen for this ObsID.
      ; If the S_FILTER keyword is missing (the filtering decision has not yet been made) then we apply a STATUS=0 filter, which is best for weak sources
      this_S_FILTER              = strtrim( psb_xpar( stats, 'S_FILTER', COUNT=count),2)
      if (count EQ 0) then this_S_FILTER = 'STATUS=0'
      frame[jj].S_FILTER         = this_S_FILTER
    endfor
    
    ; Sort ObsIDs by TSTART
    frame = frame[sort(frame.TSTART)]

    frame.display_data_fn      = tempdir + string(frame.obsname, F='(%"%s_")') + env_events_basename
    frame.edited_region_fn     = tempdir + string(frame.obsname, F='(%"%s_")') + src_region_basename

    if keyword_set(omit_single_obsid) then frame.show_t = 0

    ;; ------------------------------------------------------------------------
    if pan_only then goto, SHOW_STATS
    
    
    ;; ------------------------------------------------------------------------
    ;; Display all the neighborhoods, & reconstruction if available, as quickly as possible.

    ; If available the merged neighborhood (or merged extracted events) will be shown in a ds9 frame.
    merge_display_fn = merged_env_events_fn
    merge_available  = file_test(merge_display_fn)
    if ~merge_available then begin
      merge_display_fn = merged_src_events_fn
      merge_available  = file_test(merge_display_fn)
    endif
    
    
    ; If available the composite reconstructed image will be shown in the last ds9 frame.
    dum = headfits(composite_img_fn, ERRMSG=error, EXT=1)
    recon_available = (NOT keyword_set(error))
    
    
    
    ; Identify annotation point regions that are nearby.
    gcirc, 1, ra_hrs[ii], dec[ii], point_ra_hrs, point_dec, distance  ; [arcsec]
    threshold = 2*max(frame.msk_rad)*arcsec_per_skypixel > 30         ; [arcsec]
    ind = where( distance LT threshold, num_nearby_points )
    if (num_nearby_points GT 0) then nearby_point_region = point_region[ind]

    ; We construct a temporary region file for each observation which is passed to
    ; and optionally modified by ds9.  The temporary region file contains:
    ; 1. source extraction region file for the observation
    ; 2. an optional user-supplied REGION_FILENAME 
    ; 3. point regions marking nearby sources in the whole catalog 
    ; 4. the background region built by  ae_better_backgrounds, if available
    ; 5. regions marking "PSF hooks", if available
    ; Parts 2&3 are in the "nearby_point_region" string array constructed above.
    print, n_elements(obsnames_merged), num_obs, F='(I0," of ",I0," ObsIDs were merged:",$)'

    ; Combine the desired region files for each ObsID.
    ; Control the ordering of regions so that background regions are behind other symbols.
    cmd = 'cat '+frame.region_fn
    
    if (num_nearby_points GT 0) then begin
      ; Append "point" regions that are nearby.
      openw, unit, temp_text_fn, /GET_LUN 
      printf, unit, 'global width=1 font="helvetica 12 normal"'
      printf, unit, 'fk5'
      printf, unit, nearby_point_region
      free_lun, unit 
      cmd += ' '+temp_text_fn
    endif
    
    if ~keyword_set(omit_bkg_regions) then begin
      ind = where(file_test(frame.bkg_pixels_region_fn), count)
      if (count GT 0) then cmd[ind] += ' '+(frame.bkg_pixels_region_fn)[ind]
    endif

    ind = where(file_test(frame.hook_region_fn), count)
    if (count GT 0) then cmd[ind] += ' '+(frame.hook_region_fn)[ind]

    ind = where(file_test(frame.obsid_region_fn), count)
    if (count GT 0) then cmd[ind] += ' '+(frame.obsid_region_fn)[ind]

    cmd +=' >! '+frame.edited_region_fn
    run_command, cmd, /QUIET
    
    
    ; Extract in-band neighborhood event list, applying each ObsID's choice of STATUS filter.
    ; Recall that AE does NOT apply a STATUS filter to single-ObsID neighborhood event lists.
    ind = where(frame.S_FILTER NE '', count)
    if (count GT 0) then frame[ind].S_FILTER = ','+frame[ind].S_FILTER
     
    ; We'd like to use the option [cols sky] in dmcopy below to discard columns that ds9 does not need.
    ; That would speed up ds9.  
    ; HOWEVER,we must retain the TDMIN/TDMAX keywords for the X/Y columns, which we have carefully crafted.
    ; Alas, dmcopy does not properly handle TDMIN/TDMAX keywords (e.g. rename TDMIN11 to TDMIN1)!
    ; So, we must retain all columns below.
    infile = frame.env_events_fn + string(1000*energy_range, F="(%'[energy=%6.1f:%7.1f')") + frame.S_FILTER + ']'
    
    ind_to_show = where(frame.show_t, num_to_show)
    if (num_to_show GT 0) then begin
      forprint, SUBSET=ind_to_show, TEXTOUT=temp_par1_fn, /NoCOM, infile 
      forprint, SUBSET=ind_to_show, TEXTOUT=temp_par2_fn, /NoCOM, frame.display_data_fn
      
      run_command, /QUIET, string(temp_par1_fn, temp_par2_fn, F="(%'dmcopy ""@-%s"" ""@-%s""')")
    endif
    print
    
    ; Combine the region files that we want to display on the composite data frames.
    ; Control the ordering of regions so that background regions are behind other symbols.
    cmd = strarr(4)

    ; At the front should be the position estimates
    file_delete, /ALLOW_NONEXISTENT, temp_region_fn
    if file_test(merged_region_fn) then cmd[0] = 'grep -v polygon '+merged_region_fn+' >! '+temp_region_fn
    
    ; Then we want the "point" regions that are nearby.
    if (num_nearby_points GT 0)    then cmd[1] = 'cat '+temp_text_fn+               ' >>! '+temp_region_fn
    
    ; Then we want the extraction polygons.
    if file_test(merged_region_fn) then cmd[2] = 'grep  polygon '+merged_region_fn+ ' >>! '+temp_region_fn
    
    ; Then we want any user-supplied region file for the merged frames.
    
    if keyword_set(merge_region_filename) && file_test(merge_region_filename) then $
      cmd[3] = 'cat '+merge_region_filename+ ' >>! '+temp_region_fn
    
    run_command, cmd, /QUIET
    
        
    if merge_available then begin
      ; Extract in-band merged neighborhood event list.
      ; Note that each ObsID had its own choice of STATUS filter applied when this was constructed in the MERGE stage.
      cmd = string(merge_display_fn, 1000*energy_range, temp_events_fn, $
                   F="(%'dmcopy ""%s[energy=%6.1f:%7.1f]"" %s')")
      run_command, cmd, /QUIET

      merge_frame_index = n_elements(frame)
      frame = [frame, frame_template]
      frame[-1].display_data_fn  = temp_events_fn
      frame[-1].edited_region_fn = temp_region_fn
    endif
    
    if recon_available then begin
      recon_frame_index = n_elements(frame)
      frame = [frame, frame_template]
      frame[-1].display_data_fn  = string(composite_img_fn, F="(%'""%s[1]""')")
      frame[-1].edited_region_fn = temp_region_fn
    endif
    
    num_additional_display_file = n_elements(additional_display_file)
    if (num_additional_display_file GT 0) then begin
      additional_frame_index = n_elements(frame) + indgen(num_additional_display_file)
      for kk=0,num_additional_display_file-1 do begin
        frame = [frame, frame_template]
        ; Paths in additional_display_file are either absolute or relative to the source directory root.
        frame[-1].display_data_fn  = strmatch(additional_display_file[kk], '/*') ? '' : sourcename[ii] + '/'
        frame[-1].display_data_fn +=          additional_display_file[kk]
        frame[-1].edited_region_fn = temp_region_fn
      endfor
    endif
    
    ; Assign ds9 frame numbers
    ind_to_show = where(frame.show_t, num_to_show)
    if (num_to_show GT 0) then frame[ind_to_show].ds9_frame_id = strtrim(1+indgen(num_to_show),2)
    

    ;; ------------------------------------------------------------------------
    ;; Print a table showing various single-ObsID properties.
SHOW_STATS:    
    
    frame.emap_val /= (total(frame.emap_val, /NAN) > 1)
    net_counts      = frame.src_counts - frame.bkg_counts/frame.backscal
    print
    print, energy_range_label, F="(%'     |        |            %18s                |')"
    print, '     |exposure|counts in aperture|      background           |theta|PSF |phot |'
    print, 'frame|   %    |gross   bkg    net|counts SCAL CORR   vote    | ('') |frac| (ct)| ObsId'
    
    forprint, SUBSET=indgen(num_obs), frame.ds9_frame_id, round(100*frame.emap_val), frame.src_counts, frame.bkg_counts/frame.backscal, round(net_counts), frame.bkg_counts, round(frame.backscal), frame.backcorr, frame.VOTE_LO, frame.VOTE_HI, frame.off_angle, 100*frame.psf_frac, round(net_counts/frame.psf_frac), frame.obsname, frame.obsname_status, F='(%"%4s |%7d |%5d %5.1f %6d|%5d %5d %4.2f [%3s:%3s] |%4.1f |%3d |%5d| %s %s")'

    if merge_available then begin
      print, frame[merge_frame_index].ds9_frame_id, merge_subdir[ii], merge_subdir[ii]+file_basename(merge_display_fn), F='(%"%4d | merge %s (%s)")'
    endif else begin
      print, merge_subdir[ii], F='(%"     | merge %s (event file not available)")'
    endelse
    
    if recon_available then print, frame[recon_frame_index].ds9_frame_id, merge_subdir[ii]+file_basename(composite_img_fn) , F='(%"%4d | reconstruction : %s")'
    
    if (num_additional_display_file GT 0) then forprint, frame[additional_frame_index].ds9_frame_id, additional_display_file, F='(%"%4d | additional data: %s")' 
    
    
    ;; ------------------------------------------------------------------------
    if ~pan_only then begin
      ;; Pass all the image files and region files to ds9.
      if (num_to_show GT 0) then begin
        ae_send_to_ds9, my_ds9, TEMP_DIR=tempdir, (frame.display_data_fn)[ind_to_show], (frame.edited_region_fn)[ind_to_show], DESIRED_ZOOM=desired_zoom
      endif
      
      run_command, string(my_ds9, sourcename[ii] + '/extractions.ps', F='(%"xpaset -p ''%s'' print filename %s")'), /QUIET
      
      
      ; Make sure the region coordinate system stays PHYSICAL.
      ; The reconstructed image has no PHYSICAL system and if it's loaded in ds9 the
      ; region system is unfortunately changed to IMAGE for all the frames.
      ; run_command, string(my_ds9, F='(%"xpaset -p ''%s'' regions system physical")'), /QUIET
    endif
    
    
    ;; ------------------------------------------------------------------------
    ;; Wait for command.
COMMAND:
;    print, F='(%"\nTHE SOURCE MASK REGION is marked with the ds9 property ''background'' \n\rto distinguish it from the SOURCE EXTRACTION REGION.")'
    print, F='(%"\n\rCommands\n\r  n,+,'''' : next source\n\r   p,b,- : previous source\n\r       f : First source\n\r       l : Last source\n\r     xxx : source label (case-insensitive; unique substring is accepted)")'
    if (sequence_number GE 0) then print, F='(%"       r : rejoin sequence after the current source")'
    print, F='(%"       s : save regions (extraction & mask) in seleted frame\n\r       g : click in ds9 to show coordinates\n\r       q : quit")'
    print, 'DO NOT save edited regions via ds9 menu!  Use "s" command instead!'
    
    cmd = ''
    read, '? ', cmd
    
    cmd = strlowcase(cmd)
    
    ; Note whether the command contains leading and/or trailing whitespace.
     leading_whitespace = (cmd NE strtrim(cmd,1))
    trailing_whitespace = (cmd NE strtrim(cmd,0))
    
    cmd = strtrim(cmd,2)
    
    ;; Execute specified command.
    label_entered = 0
    case 1 of
      (cmd EQ 's'): $
        ; Save the regions in the selected frame.
        ; Make sure the region coordinate system remains PHYSICAL.
        ; The reconstructed image has no PHYSICAL system and if it's loaded in ds9 the
        ; region system is unfortunately changed to IMAGE for all the frames.
        begin
        run_command, string(my_ds9, F='(%"xpaget ''%s'' frame")'), current_frame, /QUIET
        current_frame = fix(current_frame)

        if (current_frame GT num_obs) then begin
          print, 'No observation in current frame.'
        endif else begin
          jj = current_frame-1
          cmd = strarr(3)
          cmd[0] = string(my_ds9,                       F='(%"xpaset -p ''%s'' regions system physical")')
          cmd[1] = string(my_ds9, (frame.edited_region_fn)[jj], F='(%"xpaset -p ''%s'' regions save %s")')
          cmd[2] = string(        (frame.edited_region_fn)[jj], (frame.region_fn)[jj], F="(%'grep -v point %s >! %s')")
          run_command, cmd, /QUIET
          file_chmod, (frame.region_fn)[jj], A_WRITE=0
          print, 'Saved and write-protected region file ', (frame.frame.region_fn)[jj]
        endelse
        goto, COMMAND
        end

      (cmd EQ 'g'): $
        begin
         print, F="(%'\n\n  Left-click in ds9 to print coordinates.')" 
      
         ; Ask ds9 for a mouse click or keystroke and parse the result
         run_command, /QUIET, string(my_ds9, F='(%"xpaget %s imexam any coordinate wcs fk5 degrees")'), result
         tokens = stregex(result,'(.+) +([0-9]+\.[0-9]+) +(-*[0-9]+\.[0-9]+)',/SUB,/EXT)
         keystroke = strtrim(tokens[1],2)
         ra_click  = double(tokens[2])
         dec_click = double(tokens[3])
         print, ra_click, dec_click, F='(%"%10.6fD  %10.6fD")'
        goto, COMMAND
        end
        
      (cmd EQ 'n') OR (cmd EQ '+') OR (cmd EQ '')  : presentation_index++
      (cmd EQ 'p') OR (cmd EQ 'b') OR (cmd EQ '-') : presentation_index = 0 > --presentation_index
      (cmd EQ 'f')                                 : presentation_index = 0
      (cmd EQ 'l')                                 : presentation_index = n_elements(presentation_list) - 1
      (cmd EQ 'q')                                 : ii = num_sources

      (cmd EQ 'r'): $
        ; Jump the presentation_index to the current source.
        begin
        if (sequence_number GE 0) then presentation_index = 1+sequence_number
        end
      
      else: $ ; User typed a source label.
        begin
        label_entered = 1
        ; If an exact match is found, then assume that's the label they intended.
        ind = where(strmatch(label, cmd, /FOLD_CASE), count)
        if (count GT 0) then begin
          ; Exact label entered.
          ii = ind[0]
        endif else begin
          ; Perhaps the user is abbreviating a label.
          pattern = cmd
          if  ~leading_whitespace then pattern = '*'+pattern
          if ~trailing_whitespace then pattern =     pattern+'*'
          
          ind = where(strmatch(label, pattern, /FOLD_CASE), count)
        
          if            (count EQ 0) then begin
            ; User's string not found in any label; remain at the current source.
            print, F='(%"\nWARNING: the specified string matches no source labels.")'
            goto, COMMAND
          endif else if (count EQ 1) then begin
            ; Unambiguous abbreviation entered.  
            ii = ind[0]
          endif else if (count LT 10) then begin
            ; Abbreviation matches a small number of labels; remain at the current source.
            ; Show a reasonable number of matches to the user.
            print, F='(%"\nWARNING: the specified string matches these source labels.")'
            forprint, SUBSET=ind, label, F='(%"  %s")'
            goto, COMMAND
          endif else begin
            ; Abbreviation matches a large number of labels; remain at the current source.
            print, count, F='(%"\nWARNING: the specified string matches %d source labels.")'
            goto, COMMAND
          endelse
        endelse
        end
    endcase
    
    if (presentation_index GE n_elements(presentation_list)) || (ii GE num_sources) then break
    if ~label_entered then ii = presentation_list[presentation_index]


    ; The code below attempts to reproduce for the next source the absolute zooming the user had in the selected frame when the user commanded a move to the next source.    
    ; Alas, this code does not work reliably because  "zoom" is defined in terms of the image binsize, which can differe between the frame selected when we leave a source and the frame to which a "zoom" command is sent in the next source.
    ; We should try to keep track of which _type_ of frame (event list or image) the user had selected when leaving a source, and then apply the "zoom" command to that same "type" of frame in the new source, but this is a pain to code.
    ; Even if we did that, the recon image in two adjacent sources may not have the same binsize.
    ; Thus, there is no feasible way to ask ds9 what the current FOV of a frame is (in arcseconds) and to then command that same zoom in the frames of a new source.
    if NOT pan_only then begin
      ; Record the zoom setting the user left in the currently-selected frame.
      run_command, string(my_ds9, F='(%"xpaget ''%s'' zoom")'), /QUIET, result
      desired_zoom = float(result[0])
    endif
  endwhile
  
  run_command, string(my_ds9, F='(%"xpaset -p ''%s'' exit")'), /QUIET
endif ;keyword_set(show_regions)



;; =============================================================================
if keyword_set(extract_events) OR keyword_set(extract_spectra) then begin
;; =============================================================================
  
  nominal_exposure = psb_xpar( obsdata_header, 'EXPOSURE')

  ; Look up what CCDs are present in observation.
  active_ccds = where(ccd_is_in_observation, num_active_ccds)

  if ~keyword_set(DETCHANS) then DETCHANS = 1024

  rmf_dir = ''
  if keyword_set(extract_spectra) then begin
    dafile  = 'CALDB'

    if version_string_compare(ciao_version, '4.6', /LESSTHAN) then begin
      if keyword_set(pbkfile) then begin
        if (pbkfile NE 'NONE') then begin
          if NOT file_test(pbkfile) then begin
            print, 'ERROR: the supplied PBKFILE file ', pbkfile, ' is not found!'
            GOTO, FAILURE
          endif
        endif
      endif else begin
        print, 'WARNING: PBKFILE parameter omitted; Dead Area correction will be skipped.'
        pbkfile = 'NONE'
      endelse
      pbk_parameter = 'pbkfile='+pbkfile

    endif else begin
      ; As of CIAO 4.10, the pbkfile input is still required, so we set it to NONE.
      pbk_parameter = 'pbkfile=NONE'
    endelse



    
    if keyword_set(mskfile) then begin
      if (mskfile NE 'NONE') then begin
        if NOT file_test(mskfile) then begin
          print, 'ERROR: the supplied MSKFILE file ', mskfile, ' is not found!'
          GOTO, FAILURE
        endif
      endif
    endif else begin
      print, 'WARNING: MSKFILE parameter omitted; .'
      mskfile = 'NONE'
    endelse
    
    ;; Copy observer-supplied ardlib.par to cache_dir, where the pget command can find it (when spawned by run_commmand).
    if NOT keyword_set(ardlib_filename) then begin
      print, 'ERROR: you must supply ARDLIB_FILENAME in /EXTRACT_SPECTRA stage.'
      GOTO, FAILURE
    endif

    if NOT file_test(ardlib_filename) then begin
      print, ardlib_filename, F='(%"ERROR: ARDLIB_FILENAME %s not found.")'
      GOTO, FAILURE
    endif
    
    file_copy, ardlib_filename, cache_dir + 'ardlib.par'
 
    
    ;; Figure out if this is a gratings observation.
    grating_spec = strtrim(psb_xpar( obsdata_header, 'GRATING'),2)
    case grating_spec of
      'NONE':
      'HETG':
      'LETG': print, 'WARNING: the PSF library is not correct for LETG data; various AE calculations (especially PSF fraction) will be incorrect!'
      else  : begin
              print, 'ERROR: the GRATING keyword in '+obsdata_filename_p+'has an invalid value.'
              GOTO, FAILURE
              end
    endcase
    
    ;; Sanity-check the aspect histograms supplied.
    
    for ccd_id = 0,9 do begin
      if ~ccd_is_in_observation[ccd_id] then continue
    
      asphist_fn = asphist_dir + string(ccd_id, F='(%"/ccd%d.asphist")')
      keyname    = string(ccd_id, F='(%"EXPOSUR%d")')
      exposure   = psb_xpar( obsdata_header, keyname)
      
      if (exposure GT 0) && file_test(asphist_fn) then begin
        print, 'Verifying total exposure in ', asphist_fn
        bt = mrdfits(asphist_fn, 1, /SILENT, STATUS=status)
        if (status NE 0) then message, 'ERROR reading ' + asphist_fn
  
        if (n_elements(bt) EQ 1) then begin
          print, 'WARNING!  The aspect histogram ', asphist_fn, ' was empty.'
          wait, 1
          continue
        endif
        
        duration = total(bt.duration)
        
        if (abs(duration-exposure)/duration GT 0.01) then begin
          print, duration, asphist_fn, keyname, exposure, $
                 F='(%"WARNING!  Sum of DURATION column (%d) in %s does not match value of keyword %s (%d)!")'
          print, 'It is likely that your aspect histogram was not computed correctly!!!!!'
          wait, 1
        endif
      endif
    endfor

    ;; ------------------------------------------------------------------------
    ;; Remind the user about what's in ardlib.
    print, ardlib_filename, F='(%"\n \nExamining important parameters in %s")'
    
    ;; Verify that BADPIX_FILE entries in ardlib.par are readable files.
    for ii=0,num_active_ccds-1 do begin
      ccd_id = active_ccds[ii]
  
      keyname    = string(   ccd_id, F='(%"AXAF_ACIS%d_BADPIX_FILE")')
      while 1 do begin
        run_command, string(  keyname, F='(%"pget -abort ardlib %s")'), badpix_fn    , /QUIET
        run_command, string(badpix_fn, F="(%'dmlist ""%s"" block')"  ), STATUS=status, /QUIET
        if (status EQ 0) then break
        
        print, ccd_id, badpix_fn, keyname, ardlib_filename, F='(%"ERROR: AE cannot find the observation-specific bad-pixel table for CCD%d: ''%s''\nwhich is obtained from parameter %s in %s.\n")'
        print, 'Investigate the failure.  If you can fix the problem and want AE to re-try then type ".continue"; if you want to abort the AE run type "retall".'
        stop
      endwhile
    endfor

    
    ; Determine whether CALDB or Townsley CTI correction is specified.
    run_command, 'pdump ardlib', ardlib_contents, /QUIET
    print
    forprint, ardlib_contents[where(strmatch(ardlib_contents,'AXAF_RMF_FILE*'))]
    run_command, 'pget -abort ardlib AXAF_RMF_FILE', rmf_dir, /QUIET
    rmf_dir = strtrim(rmf_dir,2)

    if (n_elements(rmf_dir) GT 1) then begin
      print, 'ERROR: pget command unexpectedly returned multiple lines:'
      print, rmf_dir
      GOTO, FAILURE
    endif
    
    if (rmf_dir EQ 'CALDB') then begin
      print, F='(%"\nThe AXAF_RMF_FILE parameter claims your data are CTI corrected by CALDB.\nThe QEU files below should also refer to the CALDB.")'
      forprint, ardlib_contents[where(strmatch(ardlib_contents,'AXAF_ACIS?_QEU*'))]
 
      
    endif else if file_test(rmf_dir, /DIRECTORY) then begin
      print, F='(%"\nThe AXAF_RMF_FILE parameter claims your data are CTI corrected by Townsley et al.\nThe QEU files below should also refer to the Townsley et al. calibration library.")'
      forprint, ardlib_contents[where(strmatch(ardlib_contents,'AXAF_ACIS?_QEU*'))]
      
       ; The source.rmf for many sources will turn out to be a sym link to the Townsley RMF library.  To make the extraction directories portable we want these sym links to reference a sym link to the library which can be easily changed later.  To make life easy on the user we will manage this library sym link in the extraction directory.
       ; Save the path supplied by the user, and strip off any trailing slash and any double slash since file_readlink() does not return them.
       rmf_dir_supplied = repstr(rmf_dir,'//','/')
       if (strmid(rmf_dir_supplied,0,/REVERSE) EQ '/') then $
               rmf_dir_supplied = strmid(rmf_dir_supplied,0,strlen(rmf_dir_supplied)-1)
       
       for ii = 1,10 do begin
         rmf_dir = string(ii, F='(%"RMF_LIBRARY%d")')
         
         ; If there's an existing sym link by this name then read it and see if it points to rmf_dir_supplied.
         if file_test(rmf_dir, /SYMLINK) then begin
           if (file_readlink(rmf_dir) EQ rmf_dir_supplied) then break
           ; Not the link we need; try another name.
           continue
         endif
        
         ; If there's some other type of object with this name then try another name.
         if file_test(rmf_dir) OR file_test(rmf_dir,/DANGLING_SYMLINK) then continue
         
         ; The name is free, so make the symlink we desire.
         file_link, rmf_dir_supplied, rmf_dir, /VERBOSE
         break
       endfor
       
       if (file_readlink(rmf_dir) NE rmf_dir_supplied) then begin
         print, 'ERROR: could not create symbolic link ', rmf_dir, ' to ', rmf_dir_supplied
         GOTO, FAILURE
       endif
       
       ; For convenience later make sure rmf_dir ends in slash.
       rmf_dir = rmf_dir + '/'

    endif else begin
      print, 'ERROR: the AXAF_RMF_FILE parameter value "',rmf_dir,'" must be CALDB or the path to the Townsley et al. calibration library.'
      GOTO, FAILURE
    endelse    
;    wait, 10
    

    ; CIAO 4.1 requires the CTI_APP keyword in the event file.
    CTI_APP  = psb_xpar( obsdata_header, 'CTI_APP' , COUNT=CTI_APP_available)
    if ~CTI_APP_available then begin
      print, 'ERROR: CIAO (v4.1 and above) requires the CTI_APP keyword in the event file!  See http://cxc.harvard.edu/ciao4.1/why/cti.html'
      GOTO, FAILURE
    endif
    

    ; Figure out whether to use mkrmf or mkacisrmf or the Townsley RMF library.
    if keyword_set(use_mkrmf) then begin
      print, 'WARNING!  You have specified that RMFs are generated by the tool mkrmf!  '
      rmf_tool = 'mkrmf'

    endif else if (rmf_dir EQ 'CALDB') then begin
      ; With modern data, we let CALDB match up the GAINFILE and phase 2 response files as shown on the ahelp page for mkacisrmf.
      gainfile             = 'CALDB'
      phase2_response_file = 'CALDB'
      rmf_tool = 'mkacisrmf'
      
     endif else begin
      rmf_tool = 'Townsley'
      ; Make spectra with 685 channels to match RMFs from CTI corrector.
      DETCHANS = 685
    endelse

    
;    ; Determine if CIAO is configured to deal with OBF contamination 
;    ; (ardlib says CALDB and the Cal Database actually contains the magic file).
;    run_command, 'pget -abort ardlib AXAF_ACIS0_CONTAM_FILE', contam_param, /QUIET
;    contam_file = "$CALDB/data/chandra/acis/contam/acisD1999-08-13contamN0004.fits"
;    contam_enabled = ((contam_param EQ 'CALDB') AND file_test(contam_file)) OR file_test(contam_param)
;    if contam_enabled then begin
;      print, F='(%"\nWARNING! Correction for ACIS contamination appears to be enabled in CIAO/CALDB.")'
;      forprint, ardlib_contents[where(strmatch(ardlib_contents,'*AXAF_ACIS?_CONTAM_FILE*'))]
;      print, 'DO NOT run the ARF_CORRECTION_FILENAME stage to correct for contamination.'
;    endif else begin
;      print, F='(%"\nCorrection for ACIS contamination appears to be disabled in CIAO/CALDB.")'
;      print, 'You should update CIAO/CALDB or run the ARF_CORRECTION_FILENAME stage with your own contamination correction.'
;    endelse
    
    print
    print, 'PAUSING FOR 10 SECONDS SO YOU CAN REVIEW ARDLIB INFORMATION ABOVE:'
    wait, 10
  endif  ;keyword_set(extract_spectra)
  
  
  energy0 = fltarr(num_sources)
  energy1 = fltarr(num_sources)
  
  case n_elements(time_filter) of
    0:           time_filter_spec = replicate(                      '',num_sources)
    1:           time_filter_spec = replicate('[time='+time_filter+']',num_sources)
    num_sources: time_filter_spec =           '[time='+time_filter+']'
    else: begin                               
          print, 'ERROR: parameter TIME_FILTER must be a scaler or a vector as long as the catalog'
          GOTO, FAILURE
          end
  endcase

  
  ;; =============================================================================
  ;; TOP OF LOOP ITERATING OVER SOURCES IN THE CATALOG
  ;; =============================================================================
  for ii = 0L, num_sources-1 do begin
    repeat_this_source = 0

    ; Skip sources already determined to be off-field.
    if source_not_observed[ii] then continue

    ;; Construct filenames.
    sourcedir   = sourcename[ii]  + '/'
    obsdir      = sourcedir + obsname + '/' + extraction_subdir[ii]
    psf_fn      = sourcedir + obsname + '/' + psf_basename
    
    rmf_dir_from_source = '../../' + rmf_dir
    if keyword_set(merge_subdir[ii]) then rmf_dir_from_source = '../' + rmf_dir_from_source
     
    obs_parameters_fn = obsdir + obs_parameters_basename
    stats_fn          = obsdir + obs_stats_basename
    psf_frac_fn       = obsdir + obs_frac_basename
    env_emap_fn       = obsdir + env_emap_basename
    env_events_fn     = obsdir + env_events_basename
    region_fn         = obsdir + src_region_basename
    src_emap_fn       = obsdir + src_emap_basename
    src_events_fn     = obsdir + src_events_basename
    src_events_bary_fn= obsdir + 'source_barycentric.evt'
    src_spectrum_fn   = obsdir + src_spectrum_basename
    rmf_fn            = obsdir + rmf_basename
    mkrmf_log_fn      = obsdir + "mkrmf.log"
    mkrmf_wgt_fn      = obsdir + "mkrmf.wgt"
    arf_fn            = obsdir + arf_basename
    
    if (NOT file_test(stats_fn)) then begin
;     print, F='(%"\n===================================================================")'
;     print, 'Source: ', sourcename[ii]
;     print, 'EXTRACTION SKIPPED: source not observed.'
      source_not_observed[ii] = 1
      continue
    endif

    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 


    ; Remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    run_command, /QUIET, ['pset dmcopy clobber=yes', 'pset dmextract clobber=yes', 'pset dmimgpick clobber=yes', 'pset mkarf clobber=yes', 'pset mkwarf clobber=yes', 'pset mkrmf clobber=yes verbose=2', 'pset mkacisrmf clobber=yes verbose=2', 'pset addrmf clobber=yes']
    
    print, F='(%"\n===================================================================")'
    print, 'Source: ', sourcename[ii]
    stats = headfits(stats_fn)
    xpos_catalog = psb_xpar( stats, 'X_CAT')
    ypos_catalog = psb_xpar( stats, 'Y_CAT')
    is_diffuse   = psb_xpar( stats, 'DIFFUSE')

    ; Look for an obs.parameters file.
    obs_parameters    = headfits(obs_parameters_fn, ERRMSG=error)
    
    if (keyword_set(error)) then begin
      ; Parameter file is missing; create an empty one.
      fxhmake,  obs_parameters, /INITIALIZE, /EXTEND, /DATE
      psb_xaddpar, obs_parameters, 'CREATOR', creator_string
      psb_xaddpar, obs_parameters, 'OBSNAME',  obsname, 'observation identifier'
      choose_S_FILTER = 1B
    endif else begin
      ; When obs.parameters file exists, it's contents over-ride obs.stats.
      
      ; Copy keywords from parameter file into obs_stats header.
      
      ; Adopt an existing S_FILTER decision, or record that this stage should make that decision.
      kywd = 'S_FILTER' 
      val = psb_xpar( obs_parameters, kywd, COMMENT=comment, COUNT=count)
      choose_S_FILTER = (count EQ 0)
      if ~choose_S_FILTER then psb_xaddpar, stats, kywd, val, comment
    endelse 
    
    
    ; Look for any existing declaration that defines a filter on the STATUS column.
    ; If the keyword is missing (the filtering decision has not yet been made) then the default value is:
    ;   * no STATUS filter for diffuse sources (filtering is observer's responsibility)
    ;   * STATUS=0 filter for point sources, so that weak sources will not be pruned before we realize that they merit aggressive background cleaning.
    S_FILTER = strtrim(psb_xpar( stats, 'S_FILTER', COUNT=count),2)
    if (count EQ 0) then S_FILTER = is_diffuse ? '' : 'STATUS=0'
    
    ; Record in obs.stats the S_FILTER that is about to be used in this source's extraction.
    ; Recall that obs.parameters is the archive where the S_FILTER decision is preserved between AE passes, since the contents of obs.stats are destroyed by the CONSTRUCT stage and by ae_source_manager. 
    psb_xaddpar, stats, 'S_FILTER', S_FILTER, 'STATUS filter applied by AE'



    ;; ------------------------------------------------------------------------
    ;; Find & format the extraction region specifications in the region file.
    
    ;; Fail if the region file uses the "field()" construct, which will corrupt the algorithm we use in recipe.txt to compute the geometric area of the extraction region.
    ae_ds9_to_ciao_regionfile, region_fn, temp_region_fn, /IGNORE_BACKGROUND_TAG, REGION_EDITED=region_edited, FIELD_SYNTAX_FOUND=field_syntax_found
    
    if field_syntax_found then begin
      print, 'ERROR: the syntax "field()" is not allowed in region files defining diffuse sources.'
      GOTO, FAILURE
    endif
 
    if (keyword_set(only_edited) && ~region_edited) then continue


    ;; ------------------------------------------------------------------------
    ;; CONSTRUCT NEIGHBORHOOD EVENT LIST. 
    ;; ------------------------------------------------------------------------
    if is_diffuse then begin
      ;; DIFFUSE SOURCE
      build_neighborhood = 0
      
      ; Symlink to observation event list (the file passed, not the copy made in /tmp/).
      file_delete, env_events_fn, /ALLOW_NONEXISTENT
      file_link, '../../' +(keyword_set(extraction_subdir[ii]) ? '../' :'')+ obsdata_filename_p, env_events_fn
      
      env_header = headfits(env_events_fn, EXT=1)
      
      if keyword_set(reuse_neighborhood) then print, 'WARNING: REUSE_NEIGHBORHOOD option is ignored for diffuse extractions.'  
    endif else begin
      ;; POINT SOURCE
      build_neighborhood = keyword_set(reuse_neighborhood) ?  ~file_test(env_events_fn) : 1
      
      ; Enlarge the neighborhood footprint from the default, if necessary, to accomodate the PSF size and to avoid an empty neighborhood. 
      min_neighborhood_size = 2.0*psb_xpar( stats, 'MSK_RAD') ; skypix
      for attempt=0,4 do begin
        if build_neighborhood then begin
          image_radius = round((neighborhood_size/2.0/arcsec_per_skypixel) > min_neighborhood_size)  ; skypixel
          image_dim    = (2*image_radius) + 1                                   ; skypixel
          cx = xpos_catalog - image_dim/2.0   
          cy = ypos_catalog - image_dim/2.0   
          
          ; I cannot recall if we have a good reason for choosing integer SKY values for neighborhood boundaries.
          neighborhood_dm_filter = string(cx,cx+image_dim, cy,cy+image_dim, F="(%'x=%d:%d,y=%d:%d')")
          
          ; Normally, the neighborhood is extracted from the observation event list, 
          ; but if /SIMULATE_SOURCES is set then the MARX events used to build the PSF are used instead.
          infile = keyword_set(simulate_sources) ? file_dirname(psf_fn, /MARK_DIRECTORY) + 'marx.evt' $
                                                 : obsdata_filename
          ; Since the user can direct that neighborhoods are reused, implementing a STATUS filter here would be very dangerous. 
          ; If the STATUS filter spec is changed after the neighborhood was cached, then the src aperture extraction would use one filter spec and background extractions could use a different filter spec.

          if ~keyword_set(build_neighborhood_emap) then begin
            ; For speed, we directly filter the event list.
            cmd = string(infile, neighborhood_dm_filter, time_filter_spec[ii], env_events_fn, $
                         F="(%'dmcopy ""%s[%s]%s"" %s')")
            run_command, cmd

          endif else begin
            ; If a neighborhood emap must be built, then we define a consistent set of neighborhood events by selecting the events that lie on the pixels in that neighborhood emap (rather than selecting events that lie inside the region neighborhood_dm_filter.
            
            ; First we apply the region neighborhood_dm_filter to select pixels from the observation emap.
            if (ciao_version EQ '4.14') then begin
              print, 'ERROR: the BUILD_NEIGHBORHOOD_EMAP option does not work correctly in CIAO 4.14; see HelpDesk Ticket #023895' 
              retall
            endif
            cmd = string(emap_filename, neighborhood_dm_filter, temp_image_fn, $
                         F="(%'dmcopy ""%s[%s][opt null=0]"" %s')")
            run_command, cmd
            
            ; Second, we work-around the dmimgpick bug reported in HelpDesk Ticket #020605, by adding a border of zeros around the neighborhood emap image.
            temp_header = headfits(temp_image_fn)
            
            cmd = string( temp_image_fn, 1+psb_xpar( temp_header, 'NAXIS1'),$
                                         1+psb_xpar( temp_header, 'NAXIS2'), env_emap_fn,$
                         F="(%'dmcopy ""%s[bin #1=0:%d,#2=0:%d]"" %s')")
            run_command, cmd
            
            print, 'WARNING: When the dmimgpick bug reported in HelpDesk Ticket #020605 is resolved, code near line 7038 can be simplified.'
            
            ; Third, we use dmimgpick and dmcopy to select events that lie on non-zero neighborhood emap pixels.
            cmd1 = string(infile, env_emap_fn, temp_events_fn, $
                          F="(%'dmimgpick ""%s[cols time,ccd_id,chip,det,sky,pi,energy,status]"" %s %s method=closest')")
      
            cmd2 = string(temp_events_fn, env_events_fn, F="(%'dmcopy ""%s[#9>0][cols time,ccd_id,chip,det,sky,pi,energy,status]"" %s')")
            run_command, [cmd1,cmd2]

          endelse
        endif ; build_neighborhood
        
        neighborhood_events = mrdfits(env_events_fn, 1, env_header, /SILENT, STATUS=status)
        if (status NE 0) then message, 'ERROR reading ' + env_events_fn
        
        if (psb_xpar( env_header, 'NAXIS2') GT 0) then begin
          break 
        endif else begin
          min_neighborhood_size *= 2
          build_neighborhood     = 1
        endelse
      endfor ; attempt
      
      if (attempt GT 4) then begin
         help, xpos_catalog, ypos_catalog, image_radius
         print, 'ERROR: could not find a neighborhood that includes any events!!'
         GOTO, FAILURE
      endif
      
      ; Identify the X and Y columns in env_events_fn.
      fxbfind, env_header, 'TTYPE', dum1, TTYPE, dum2, 'null'
      colnames = strlowcase( strtrim(TTYPE,2) )
      x_colnum = 1+where(strlowcase(colnames) EQ 'x')
      y_colnum = 1+where(strlowcase(colnames) EQ 'y')
      
      if build_neighborhood then begin
        ; We have to set TDMIN & TDMAX to get ds9 to produce a nice default binning of the data.
        ; HEASARC standards say their datatype should be the same as that of the table column to which they refer. 
        openw, unit, temp_text_fn, /GET_LUN
        printf, unit, x_colnum, (cx), x_colnum, (cx+image_dim), $
                      y_colnum, (cy), y_colnum, (cy+image_dim), $
                      F='(%"#add\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f")'
        free_lun, unit
        
        cmd = string(env_events_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
        run_command, cmd, /QUIET
        
        ; Record the neighborhood boundaries in the parameter file.
        psb_xaddpar, obs_parameters, 'N_FILTER', neighborhood_dm_filter, '[sky] boundaries of neighborhood' 
        writefits, obs_parameters_fn, 0, obs_parameters
      endif
      
      ; Copy neighborhood boundaries from the parameter file to the stats file.
      kywd = 'N_FILTER' 
      val = psb_xpar( obs_parameters, kywd, COMMENT=comment, COUNT=count)
      if (count EQ 1) then psb_xaddpar, stats, kywd, val, comment
 
     
      ;; ------------------------------------------------------------------------
      ; We will later run dmextract to extract a spectrum from the aperture.  
      ; The BACKSCAL keyword written by dmextract will be the aperture area normalized by a "field area". 
      ; We need to calculate this field area in skypixel**2 so that we can later rescale this normalized BACKSCAL to get an aperture area in skypixel**2.
      
      readcol, '$ASCDS_INSTALL/data/cxo.mdb', temp1, temp2, F='A,A'

      ind = where(strmatch(temp1,'Chandra:ACIS:field'), count)
      if (count EQ 1) then begin
        ; Starting in CIAO 4.5, dmextract instead reads the field area it needs for computing BACKSCAL from $ASCDS_INSTALL/data/cxo.mdb.
        ; The CIAO 4.5 release notes say:
          ; The usual normalization uses the TLMAX/TLMIN keyword values, but for ACIS a fixed value of (8192)**2 is now used in order to correctly handle the new behaviour of the reprojection tools. This value is set in the $ASCDS_INSTALL/data/cxo.mdb file.
        backscal_normalization = (float(temp2[ind]))[0]^2

      endif else begin
        ; Prior to CIAO 4.5, the dimensions of this "field" were calculated by dmextract using the TLMIN/TLMAX values for the SKY coordinates (x/y).
        ; That field area normally had the value 8192.0 * 8192.0
        fxbfind, env_header, 'TLMIN', dum1, TLMIN, dum2, 0.0
        fxbfind, env_header, 'TLMAX', dum1, TLMAX, dum2, 0.0
        backscal_normalization = ( (TLMAX[x_colnum-1]-TLMIN[x_colnum-1])*(TLMAX[y_colnum-1]-TLMIN[y_colnum-1]) )[0] ; skypixel**2
      endelse

      ; In both epochs of CIAO, backscal_normalization should be 8192^2.  Any other value should be investigated!
      if (backscal_normalization NE 8192.0^2) then begin
        print, 'ERROR: Ask Patrick Broos to help you investigate why backscal_normalization is not 8192^2!!'
        help, backscal_normalization
        GOTO, FAILURE
      endif
    endelse ; point source
    
    
    ;; ------------------------------------------------------------------------
    ;; Compute PSF fractions using the PSF images made earlier by dividing the 
    ;; light falling inside the source region by the total light.  
    f_nan = !VALUES.F_NAN

    fits_open, psf_fn, fcb, /NO_ABORT, MESSAGE=error
    if keyword_set(error) then begin
      ; A PSF is not available for diffuse sources.  For point sources it might be
      ; the case that a reasonable PSF is not available. 
      psf_fraction = replicate({energy:0.0D, fraction:0.0,     x_sdev:f_nan,     y_sdev:f_nan, $
                                                           bkg_x_sdev:f_nan, bkg_y_sdev:f_nan}, 2)
      psf_fraction.energy   = [0,10]
      psf_fraction.fraction = [1,1]
      fiducial_psf_energy   = f_nan
      if (~is_diffuse) then begin
        print, psf_fn, error, F='(%"WARNING: attempt to read %s failed with message \"%s\"; using PSF Fraction of 1.0")'
      endif
      
    endif else begin
      ; PSF is available.
      ; If /REGION_ONLY then compute only 1 PSF fraction to save time.
      num_psf_images = keyword_set(region_only) ? 1 : 1 + fcb.NEXTEND
      psf_fraction = replicate({energy:0.0D, fraction:0.0,     x_sdev:f_nan,     y_sdev:f_nan, $
                                                           bkg_x_sdev:f_nan, bkg_y_sdev:f_nan}, num_psf_images)
    
      for jj =0, num_psf_images-1 do begin
        fits_read, fcb, dum, psf_header, /HEADER_ONLY, /NO_PDU, EXTEN_NO=jj
        psf_fraction[jj].energy = psb_xpar( psf_header, 'ENERGY')
        psf_total               = psb_xpar( psf_header, 'PSF_TOTL')
        if (psf_total EQ 0) then begin
          print, "WARNING: obsolete PSFs in "+psf_fn+" may have incorrect scaling.  AE versions 3.6 or higher can make better PSFs, but REMEMBER THAT THE /CONSTRUCT_REGIONS STAGE WILL OVERWRITE THE SOURCE'S EXTRACTION REGION FILE!"
          fits_read, fcb, psf_img, EXTEN_NO=jj
          psf_total = total(psf_img, /DOUBLE)
          help, psf_total
        endif
        
        cmd = string(psf_fn, 1+jj, temp_region_fn, temp_image_fn, $
                   F="(%'dmcopy ""%s[PSF%d][sky=region(%s)][opt update=no]"" %s')")
        run_command, cmd
        
        psf_img = readfits(temp_image_fn, psf_header, /SILENT)
        ind = where(finite(psf_img) EQ 0, num_nan)
        if (num_nan GT 0) then begin
          ind = where(psf_img GT 0, num_pos)
          print, 100*num_pos/float(num_nan+num_pos), $
            F='(%"WARNING: PSF model covers only %5.2f%% of extraction region; PSF fraction is underestimated")'
        endif
        
        ; PSF Fraction is power in region divided by total power to infinity.
        psf_total_in_region       = total(psf_img, /DOUBLE, /NAN)
        psf_fraction[jj].fraction = psf_total_in_region / psf_total
        
        
        ; It would be great to save the PSF truncated by the extraction region, 
        ; plus a flat background image truncated, so that later we could use them both 
        ; to build a 2-D parent distribution for the events extracted from multiple obsids  .
        ; However that would take a lot of space!  So, below we save 1-D descriptions of 
        ; these distributions.
        ;
        ; Compute the standard deviation of the PSF inside the extraction region for use later in position error estimates.  
        ; First let's compute the CENTROID of the psf_img array inside the extraction region, expressed in the array index coordinate system, by computing  marginal sums of psf_img weighted by the array indexes.  
        ; The denominator is NOT psf_total as it was above for the PSF fraction.
        xdim = (size(psf_img, /DIM))[0]
        ydim = (size(psf_img, /DIM))[1]
        x_ind = lindgen(xdim,ydim) mod xdim
        y_ind = lindgen(xdim,ydim)  /  xdim
        
        xpos_psf = total( psf_img * x_ind, /DOUB, /NAN ) / psf_total_in_region
        ypos_psf = total( psf_img * y_ind, /DOUB, /NAN ) / psf_total_in_region
        
        ; Then we compute the variances of this distribution along the two array index axes.
        x_variance = total( psf_img * (x_ind-xpos_psf)^2, /DOUB, /NAN ) / psf_total_in_region
        y_variance = total( psf_img * (y_ind-ypos_psf)^2, /DOUB, /NAN ) / psf_total_in_region

        ; Finally compute standard deviations and convert to units of skypixel
        psf_fraction[jj].x_sdev = sqrt(x_variance) * psb_xpar( psf_header, 'CDELT1P')
        psf_fraction[jj].y_sdev = sqrt(y_variance) * psb_xpar( psf_header, 'CDELT2P')

        ; Compute the single-axis variances of a flat background truncated by the extraction region.
        ; Compute standard deviations and convert to units of skypixel
        psf_footprint  = finite(psf_img) AND (psf_img GT 0)
        bkg_x_variance = total( psf_footprint * (x_ind-xpos_psf)^2, /DOUB, /NAN ) / total(psf_footprint)
        bkg_y_variance = total( psf_footprint * (y_ind-ypos_psf)^2, /DOUB, /NAN ) / total(psf_footprint)
  
        psf_fraction[jj].bkg_x_sdev = sqrt(bkg_x_variance) * psb_xpar( psf_header, 'CDELT1P')
        psf_fraction[jj].bkg_y_sdev = sqrt(bkg_y_variance) * psb_xpar( psf_header, 'CDELT2P')
      endfor ;jj
      fits_close, fcb
      
      ;; Sort psf_fraction by energy so its ready for interpolation later.
      psf_fraction = psf_fraction[ sort(psf_fraction.energy) ]
      
      fiducial_psf_energy = psb_xpar( stats, 'PSF_ENGY')
      
    endelse ;  PSF  available
    
    
    ;; Find the PSF that's closest to the fiducial energy.
    dum = min(abs(psf_fraction.energy - fiducial_psf_energy), imin)
    fiducial_psf_fraction = psf_fraction[imin]
    

    fxhmake,  pheader, /INITIALIZE, /EXTEND, /DATE
    psb_xaddpar, pheader, 'CREATOR', creator_string
    writefits, psf_frac_fn, 0, pheader

    theader = 0   &   dum=temporary(theader)
    psb_xaddpar, theader, 'EXTNAME', 'PSF_FRACTION'
    mwrfits, psf_fraction, psf_frac_fn, theader
    
    psb_xaddpar, stats, 'REG_EDIT', region_edited ? 'T' : 'F', 'T=aperture edited by observer'
    psb_xaddpar, stats, 'PSF_FRAC', fiducial_psf_fraction.fraction, string(fiducial_psf_fraction.energy, F='(%"PSF fraction via DM @%6.4f keV")')

    
    ;; ------------------------------------------------------------------------
    ;; Find the set of emap pixels that fall within the region, and compute various 
    ;; statistics on those emap values.
    ;; The CRITICAL STATISIC is "src_exposurearea", the integral of the emap over the aperture.
    ;; For point sources (where apertures and emap pixels can be of comparable size) src_exposurearea
    ;; is estimated from the "mean_exposure" statistic and CIAO's measurement of the aperture's geometric area.
    ;; For diffuse sources (where apertures are large) src_exposurearea is calculated by summing 
    ;; the emap pixels within the aperture.

    if is_diffuse then begin
      ; The extraction of emap pixels we're about to do will crash dmcopy if the aperture is fully outside the FOV of the emap.
      ; This can occur for diffuse apertures, so we must screen for that via a throw-away extraction of the event list.
      ; The CONSTRUCT_REGIONS stage for diffuse sources intentionally does not attempt to make the off-field decision.
      cmd = string(obsdata_filename, temp_region_fn, temp_events_fn, $
                   F="(%'dmcopy ""%s[sky=region(%s)][cols x,y]"" %s')")
      run_command, cmd
      
      temp_count = mrdfits(temp_events_fn, 1, /SILENT, STATUS=status)
      if (status NE 0) then message, 'ERROR reading ' + temp_events_fn
      ; THIS WAS A THROW-AWAY DIFFUSE EXTRACTION WITHIN THE POLYGON; the true extraction will use a pixelized region (later).
      file_delete, temp_events_fn 
  
      if ~keyword_set(temp_count) then begin
          print, stats_fn, F='(%"\nSource is not in field of view (no data found in aperture); removing %s.")'
          
          ; Remove obs.stats to communicate to other AE stages that the source is off the field.
          file_delete, stats_fn, /ALLOW_NONEXISTENT
          continue
      endif
    endif ; is_diffuse
    
    ; Now, we can carry on with the emap pixel extraction.
    cmd = string(emap_filename, temp_region_fn, src_emap_fn, $
                 F="(%'dmcopy ""%s[sky=region(%s)][opt null=NaN]"" %s')")
    run_command, cmd
    
    emap = readfits(src_emap_fn, emap_header, /SILENT)

    if ~is_diffuse then begin
      ; For POINT SOURCES, our emap statistics must include the emap=0 pixels, because that's appropriate
      ; below when we use mean_exposure and the aperture geometric area to estimate src_exposurearea.  
      ind  = where(finite(emap), num_emap_pixels)
  
      if (num_emap_pixels GT 0) then begin
        emap_pixels = emap[ind]
      endif else begin
        ; No pixel centers fall in extraction polygon, so use closest emap value (recorded by CONSTRUCT_REGIONS stage).
        emap_pixels = [psb_xpar( stats, 'EMAP_AVG')]
        print, 'No emap pixel centers are in source aperture; using closest emap value EMAP_AVG=', emap_pixels
      endelse
    endif else begin
      ; For DIFFUSE SOURCES, the aperture commonly extends way beyond the FOV of this ObsID, so our
      ; emap statistics would be meaningless if they include emap=0 pixels.
      ind  = where(finite(emap) AND (emap GT 0), num_emap_pixels)
  
      if (num_emap_pixels GT 0) then begin
        emap_pixels = emap[ind]
      endif else begin
        ; The aperture contains no positive emap pixels.
        emap_pixels = 0
        print, 'WARNING: The aperture contains no positive emap pixels.'
      endelse
    endelse

    
    ; If a time filter was supplied we must scale down the emap by the exposure fraction.
    if keyword_set(time_filter_spec[ii]) then begin
      exposure_fraction = psb_xpar( env_header, 'EXPOSURE') / nominal_exposure
      help, exposure_fraction
      emap_pixels = temporary(emap_pixels) * exposure_fraction
    endif
    
    if ~array_equal( finite(emap_pixels) AND (emap_pixels GE 0), 1) then begin
      print, 'ERROR: emap_pixels has NaN or negative values.  BUG!!!'
      retall
    endif
    
    mean_exposure     = mean  (emap_pixels, /DOUBLE)
    median_exposure   = median(emap_pixels, /EVEN)
    min_exposure      = min   (emap_pixels, MAX=max_exposure)
    
    ; Calculate a robust mean of the emap pixels that are greater than zero.
    ind  = where(emap GT 0, num_positive_emap_pixels)
    if (num_positive_emap_pixels GT 0) then begin
      resistant_mean, emap[ind], 1, robust_mean_exposure  ; 1-sigma clipping seems to work best near CCD edges.
    endif else robust_mean_exposure = 0

    ; Integrate the emap, ignoring any negative emap pixels.
    ; This is NOT ACCURATE for point sources, where the aperture is an unconstrained polygon.
    ; This IS accurate for diffuse sources, where the aperture is defined to be the set of emap
    ; pixels inside a polygon.
    integral_exposure = total (emap_pixels>0, /DOUBLE) ; Units are emappix^2 s cm**2 count /photon.

    
    ;; ------------------------------------------------------------------------
    ;; Extract the source event list without any energy filter.
    ;;
    ;; For point sources (small regions where edges are complex with respect to 
    ;; the pixelization of the emap) we use the exact region.
    ;;
    ;; For diffuse sources (large regions with unimportant edges) we use the set
    ;; of emap pixels above to define the actual source region to apply to the event list.
    ;; The region boundary is made to be "pixelized".
    ;; 
    ;; We extract from neighborhood events instead of full event list to save  significant time.
    ;; We make dmcopy write to local disk, to avoid the CIAO slowdown when destination is an NFS volume.
    if (NOT is_diffuse) then begin
      ; Point Source
      cmd = string(env_events_fn, temp_region_fn, $
                   keyword_set(S_FILTER) ? ','+S_FILTER : '', $
                   temp_events2_fn, $
                   F="(%'dmcopy    ""%s[sky=region(%s)%s]"" %s')")
      run_command, cmd
      
      file_move, /OVERWRITE, temp_events2_fn, src_events_fn

      ; Any existing barycentric event list (made by ae_fold_lightcurve) is now stale, so remove it.
      file_delete, [src_events_bary_fn,src_events_bary_fn+'.gz'], /ALLOW_NONEXISTENT

    endif else begin
      ;; Diffuse Source
      ;; The extraction region is defined as the emap pixels that are inside the supplied ds9 region.
      ;; The events lying on those pixels are identified by dmimgpick.
      ;; We do NOT apply any STATUS filters to diffuse sources.
      
      ; We work-around the dmimgpick bug reported in HelpDesk Ticket #020605, by adding a border of zeros around the source.emap image.
      ;; We make dmcopy write to local disk, to avoid the CIAO slowdown when destination is an NFS volume.
      temp_header = headfits(src_emap_fn)
      
      cmd = string( src_emap_fn, 1+psb_xpar( temp_header, 'NAXIS1'),$
                                 1+psb_xpar( temp_header, 'NAXIS2'), temp_image_fn,$
                   F="(%'dmcopy ""%s[bin #1=0:%d,#2=0:%d]"" %s')")
      run_command, cmd
      file_move, /OVERWRITE, temp_image_fn, src_emap_fn
      
      print, 'WARNING: When the dmimgpick bug reported in HelpDesk Ticket #020605 is resolved, code near line 7353 can be simplified.'

      cmd1 = string(env_events_fn, src_emap_fn, temp_events_fn, $
                    F="(%'dmimgpick ""%s[cols time,ccd_id,chip,det,sky,pi,energy]"" %s %s method=closest')")

      ;; BELOW WE REQUIRE EMAP VALUE TO BE >1, INSTEAD OF >0, BECAUSE CIAO 3.0.1 HAS A BUG THAT CAUSES ZERO VALUES TO PASS THE >0 TEST!
      ;; We make dmcopy write to local disk, to avoid the CIAO slowdown when destination is an NFS volume.
      cmd2 = string(temp_events_fn, temp_events2_fn, F="(%'dmcopy ""%s[#8>1][cols time,ccd_id,chip,det,sky,pi,energy]"" %s')")
      run_command, [cmd1,cmd2]
      
      file_move, /OVERWRITE, temp_events2_fn, src_events_fn
    endelse ; Diffuse Source
    
    
    wideband_events = mrdfits(src_events_fn, 1, src_events_hdr, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + src_events_fn

    wideband_src_counts = psb_xpar( src_events_hdr, 'NAXIS2')

    
    
    ;; ------------------------------------------------------------------------
    ;; Set TDMIN & TDMAX in source.evt to the footprint of src_emap_fn, so that ds9 will produce a nice default binning of the data.
    ; HEASARC standards say their datatype should be the same as that of the table column to which they refer. 
  
    run_command, /QUIET, string(src_emap_fn, F="(%'get_sky_limits %s verbose=0 precision=1')")
    run_command, /QUIET, 'pget get_sky_limits dmfilter', filterspec
  
    tokens = strsplit(filterspec, '=:#,', /EXTRACT)
    xmin  = float(tokens[1])
    xmax  = float(tokens[2])
    ymin  = float(tokens[5])
    ymax  = float(tokens[6])
      
    if is_diffuse then begin
      psb_xaddpar, stats, 'X_CAT', mean([xmin,xmax]), '[skypixel] middle of region bbox'
      psb_xaddpar, stats, 'Y_CAT', mean([ymin,ymax]), '[skypixel] middle of region bbox'
    endif
    
    
    ; Identify the X and Y columns in src_events_fn.
    fxbfind, src_events_hdr, 'TTYPE', dum1, TTYPE, dum2, 'null'
    colnames = strlowcase( strtrim(TTYPE,2) )
    x_colnum = 1+where(strlowcase(colnames) EQ 'x')
    y_colnum = 1+where(strlowcase(colnames) EQ 'y')
    
    openw, unit, temp_text_fn, /GET_LUN
    printf, unit, x_colnum, (xmin), x_colnum, (xmax), $
                  y_colnum, (ymin), y_colnum, (ymax), S_FILTER, $
                  F='(%"#add\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f\nS_FILTER=\"...\"\nS_FILTER=\"%s\"")'
                  ; S_FILTER is assigned twice above to work around a dmhedit bug (in CIAO 4.3) that converts a whitespace value to the integer zero.
    free_lun, unit
    
    cmd = string(src_events_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
    run_command, cmd, /QUIET
 
      
    ;; ------------------------------------------------------------------------
    ;; Determine if any source events fall in the "warning region" supplied.
    if keyword_set(warning_region_filename) then begin
      cmd = string(src_events_fn, warning_region_filename, temp_events_fn, $
                 F="(%'dmcopy ""%s[sky=region(%s)]"" %s')")
      run_command, cmd

      warning_events = mrdfits(temp_events_fn, 1, /SILENT, STATUS=status)
      if (status NE 0) then message, 'ERROR reading ' + temp_events_fn

      num_warning_events = psb_xpar( headfits(temp_events_fn, EXT=1), 'NAXIS2')
      psb_xaddpar, stats, 'WARNFRAC', float(num_warning_events)/(1>wideband_src_counts), 'fraction of events in warning region'
    endif else sxdelpar, stats, ['WARNFRAC']
    
   
    ;; ------------------------------------------------------------------------
    ;; Extract in-band source event list to compute statistics.
    ;; We do the energy filtering in IDL rather than CIAO for speed.
    if (wideband_src_counts EQ 0) then begin
      inband_src_counts = 0
    endif else begin
      inband_ind = where((1000*energy_range[0] LE wideband_events.energy) AND (wideband_events.energy LE 1000*energy_range[1]), inband_src_counts)
    endelse

    if (inband_src_counts EQ 0) then begin
      ;; There are no in-band data so we skip various statistics.
      print, 'WARNING: no in-band data found in source region.'
      
      sxdelpar, stats, ['X_DATA','Y_DATA','EX_DATA','EY_DATA','CAT2DATA','MEDIAN_E']
      psb_xaddpar, stats, 'SRC_CNTS', 0, string(energy_range, F="(%'[count] %0.2f:%0.2f keV, in extraction region')")

      ;; We'd like to create a fake event at the source position in the structure inband_events 
      ;; because it's used often in the code below (e.g. to figure out for which CCDs we need ARFs).
      dmcoords_cmd = string(obsdata_filename, keyword_set(aspect_fn) ? aspect_fn : '', xpos_catalog, ypos_catalog,  F="(%'dmcoords %s asolfile=%s opt=sky x=%6.1f y=%6.1f')")
      run_command, /QUIET, dmcoords_cmd

      run_command, /QUIET, 'pget dmcoords chip_id chipx chipy', dmcoords_result
      ; Parse the string returned by pget with the ON_IOERROR mechanism enabled in order to find type conversion errors.
      ON_IOERROR, TYPE_CONVERSION_ERROR2
      chip_id = fix(dmcoords_result[0])
      chipx   = fix(dmcoords_result[1])
      chipy   = fix(dmcoords_result[2])
      if (0) then begin
        TYPE_CONVERSION_ERROR2:
        print, !ERROR_STATE.MSG
        print, 'ERROR: dmcoords results could not be parsed.'
        forprint, ['  The dmcoords call was : ', '    '+dmcoords_cmd   ]
        forprint, ['  The output of pget was: ', '    '+dmcoords_result]
        GOTO, FAILURE
      endif 
      ON_IOERROR, NULL


      if (chipx LT 1) || (chipx GT 1024) || $
         (chipy LT 1) || (chipy GT 1024) || ~ccd_is_in_observation[chip_id] then begin
        ;; But ... the dmcoords tool can sometimes return silly results, such as
        ;; CHIP coordinates outside the range [1:1024] or a chip_id value not in the list of active detectors.
        ;; So, if that happens we'll use values from the nearest event.
        
        ; Read neighborhood event list here, because for diffuse sources we have not done so already.
        neighborhood_events = mrdfits(env_events_fn, 1, env_header, /SILENT, STATUS=status)
        if (status NE 0) then message, 'ERROR reading ' + env_events_fn
        
        dum = min( (neighborhood_events.x - xpos_catalog)^2 + (neighborhood_events.y - ypos_catalog)^2, ind )
        chip_id = neighborhood_events[ind].ccd_id
        chipx   = neighborhood_events[ind].chipx
        chipy   = neighborhood_events[ind].chipy  
        print, 'dmcoords produced silly result; using nearest event'
      endif

      inband_events = {x:xpos_catalog, y:ypos_catalog, energy:1.0, ccd_id:chip_id, chipx:chipx, chipy:chipy }

      ;; We should NOT skip the rest of the extraction!  We need the empty spectrum to
      ;; carry forward the information about this observation's exposure, even if no events
      ;; were detected.

    endif else begin
      ;; Normal case: some in-band events found.
      inband_events = wideband_events[inband_ind]

      ; Compute mean of data, and the "standard error of the mean" by dividing by sqrt(N).
      ; That formula is asymptotically correct as N gets large.  I don't know what to do for small N.
      ; We do NOT use the Student's T distribution for this calculation.  The Student's T is applied in the case where the variance of the parent distribution is (poorly) estimated from the N data points. In our case we accurately compute the variance from the PSF which is assumed to be the parent distribution.
      xpos_data_i = mean(inband_events.x, /DOUBLE)
      ypos_data_i = mean(inband_events.y, /DOUBLE)
      
      er_xpos_data_i = fiducial_psf_fraction.x_sdev / SQRT(inband_src_counts)
      er_ypos_data_i = fiducial_psf_fraction.y_sdev / SQRT(inband_src_counts)

      
      median_energy_i   = median([inband_events.energy], /EVEN)
      cat2data_offset_i = sqrt((xpos_catalog-xpos_data_i)^2     + (ypos_catalog-ypos_data_i)^2)
            
      psb_xaddpar, stats, 'X_DATA',   xpos_data_i      , '[skypixel] source position, mean of data'
      psb_xaddpar, stats, 'Y_DATA',   ypos_data_i      , '[skypixel] source position, mean of data'
      psb_xaddpar, stats, 'EX_DATA',  er_xpos_data_i   , '[skypixel] 1-sigma error on X_DATA'
      psb_xaddpar, stats, 'EY_DATA',  er_ypos_data_i   , '[skypixel] 1-sigma error on Y_DATA'
      psb_xaddpar, stats, 'CAT2DATA', cat2data_offset_i, '[skypixel] catalog to data offset'
     ;psb_xaddpar, stats, 'MEDIAN_E', median_energy_i,   string(energy_range, F="(%'[keV] median energy, %0.2f:%0.2f keV, in extraction region')")
      psb_xaddpar, stats, 'SRC_CNTS', inband_src_counts, string(energy_range, F="(%'[count] %0.2f:%0.2f keV, in extraction region')")
  
  
      ;; Save lowest two in-band energies for Garmire NH estimation.
      ind = sort(inband_events.energy)
      energy0[ii] = inband_events[ind[0]].energy
      if (inband_src_counts GT 1) then energy1[ii] = inband_events[ind[1]].energy

    endelse ; inband_src_counts > 0

    psb_xaddpar, stats, 'EMAP_NUM', num_emap_pixels, '# exposure map pixels in aperture'
    psb_xaddpar, stats, 'EMAP_AVG', mean_exposure,   '[s cm**2 count /photon] mean exposure map value in aperture'
    psb_xaddpar, stats, 'EMAP_MED', median_exposure, '[s cm**2 count /photon] median exposure map value in aperture'
    psb_xaddpar, stats, 'EMAP_MIN', min_exposure,    '[s cm**2 count /photon] min exposure map value in aperture'
    psb_xaddpar, stats, 'EMAP_MAX', max_exposure,    '[s cm**2 count /photon] max exposure map value in aperture'

    psb_xaddpar, stats, 'CREATOR', creator_string
    writefits, stats_fn, 0, stats
    
    ;; ------------------------------------------------------------------------
    ;; If /EXTRACT_SPECTRA not specified, then skip everything else in loop.
    if NOT keyword_set(extract_spectra) then continue
    
  
    ;; ------------------------------------------------------------------------
    ;; Create source spectrum and assign a BACKSCAL value that equal to the integral 
    ;; of the exposure map over the source extraction region (the AE convention).  
    ;; The exposure map carries information both on how much _time_ each section of the
    ;; sky was observed (due to dithering) and how _sensitive_ (effective area at the energy
    ;; used to create the emap) the observatory is at each location.  The integration is how 
    ;; the geometric area of the region comes into play.
    ;; Thus the observed background signal at each location is proportional to the emap there
    ;; (integration time & sensitivity), and the total background counts detected in the
    ;; extraction region is the sum (integral) of the observed signal across the region.

    if (NOT is_diffuse) then begin
      ; POINT SOURCE
      ; The natural method would be to simply dmextract from src_events_fn (where polygon is 
      ; already applied), but alas dmextract computes BACKSCAL as the polygon region area 
      ; divided by the "field area".
      ; To be compatible with BACKSCALs computed in background spectra, we want here the 
      ; actual area of the polygon in skypixel**2.
      ; When we extract from neighborhood.evt, dmextract computes BACKSCAL as the fractional
      ; area of the polygon, with respect to the field area, which we've computed earlier in backscal_normalization.
    
      ; Also, in CIAO 2.3 we had some instances where dmextracting from src_events_fn led to BACKSCAL=0!
      cmd = string(env_events_fn, temp_region_fn, $
                   keyword_set(S_FILTER) ? ','+S_FILTER : '', $
                   DETCHANS, src_spectrum_fn, $
                   F="(%'dmextract ""%s[sky=region(%s)%s][bin pi=1:%d:1]"" %s opt=pha1 error=gaussian')")
      run_command, cmd

      src_header = headfits(src_spectrum_fn, EXT=1)
      src_area   = backscal_normalization * psb_xpar( src_header, 'BACKSCAL') ; skypixel**2
;help, src_area, backscal_normalization, psb_xpar( src_header, 'BACKSCAL')

      if (src_area EQ 0) then message, 'ERROR: dmextract produced BACKSCAL=0!'
      src_area_comment = '[skypixel**2] src extraction area by dmextract'

      ; We do NOT simply sum up the emap pixels that fall inside the region 
      ; (and multiply by the pixel area) because the emap binning may be 
      ; quite coarse compared to the polygon, and the area of the polygon
      ; would be only crudely approximated by the number of emap pixels contained.
      ; Instead we multiply the geometric area of the region, as computed by CIAO, 
      ; times the mean exposure map value in the source region.
      src_exposurearea = src_area*mean_exposure ; Units are skypix**2 s cm**2 count /photon.
      
      ; Determine the set of CCD's that *might* have observed this aperture, so that we can later decide for which CCDs we should burn computing time to calculate the ARF.
      ; I do not know of any perfect and efficient way to do this.
      ; If the neighborhood has a reasonable number of events then we use the ccd_ids found in those events, else we use all the CCDs in the observation.
      ccd_is_in_neighborhood = (psb_xpar( env_header, 'NAXIS2') GT 10) ? (histogram([neighborhood_events.ccd_id], MIN=0, MAX=9) GT 0) : ccd_is_in_observation


    endif else begin
      ; DIFFUSE SOURCE
      
      ; We're going to need emap_energy later, so check for it now before we burn lots of time making a diffuse RMF!
      if NOT keyword_set(emap_energy) then begin
        print, F='(%"ERROR: for diffuse extraction you must supply an EMAP_ENERGY value (keV).")'
        GOTO, FAILURE
      endif
      
      ; If the aperture is empty, then mkwarf and mkacisrmf are going to fail, so print a warning and abort this extraction.
      if (wideband_src_counts EQ 0) then begin
        print, stats_fn, F='(%"\nSource is not in field of view (no events found in aperture); removing $s.")'
        
        ; Remove obs.stats to communicate to other AE stages that the source is off the field.
        file_delete, stats_fn, /ALLOW_NONEXISTENT
        continue
      endif

      ; In this case we've "pixelized" the region above when we extracted source.evt above.
      ; We don't care about DM's estimate of the geometric area so we can just make a 
      ; spectrum from source.evt. 
      ; We include the WMAP option to support mkwarf and mkacisrmf later.
      ; We do NOT follow the recommendation on ahelp pages to bin the WMAP as [bin det=8] for 3 reasons:
      ;   1. We think this coarse binning could increase the chance of error messages in mkwarf and mkacisrmf at the field edges.
      ;   2. Omitting ranges for detx/dety leads to a WMAP that has a huge margin of zeros around the detector.
      ;   3. Testing shows that mkacisrmf has the same run-time with x1 and x8 binning in the WMAP.
      min_detx = min (wideband_events.detx, MAX=max_detx)
      min_dety = min (wideband_events.dety, MAX=max_dety)
      cmd = string(src_events_fn, DETCHANS, src_spectrum_fn, 1000*wmap_energy_range, min_detx-1, max_detx+1, min_dety-1, max_dety+1, $
                   F="(%'dmextract ""%s[bin pi=1:%d:1]"" %s opt=pha1 error=gaussian wmap=""[energy=%6.1f:%7.1f][bin detx=%d:%d,dety=%d:%d]""')")
      run_command, cmd
      
      ; If the WMAP is empty, then mkwarf and mkacisrmf are going to fail, so print a warning and abort this extraction.
      wmap = readfits(src_spectrum_fn, /SILENT)
      if (total(/INT, wmap) EQ 0) then begin
        print, obsdir, stats_fn, F='(%"WARNING! Aborted extraction because WMAP in %s is zero! You should investigate, and then remove %s so that the MERGE will not see it.")'
        continue
      endif
      
      ; Here we can directly integrate the emap, being careful to convert from square emap pixels to square skypixel.  
      src_header = headfits(src_spectrum_fn, EXT=1)
      pixel_size = psb_xpar( emap_header, 'CDELT1P')
      src_exposurearea = (pixel_size^2) * integral_exposure ; Units are skypix**2 s cm**2 count /photon.

      ; Now, for diffuse sources the concept of "source extraction area" (src_area) is generally more poorly defined 
      ; than in the point source case because the diffuse region is likely to have very strong variations in the 
      ; emap, including portions that are zero due to point source masks and field edges.
      ; In order to get some kind of characteristic area for the region we're going to simply divide the emap integral
      ; by the median of the non-zero emap pixel values.
      ind  = where(emap GT 0, count)
  
      if (count GT 0) then begin
        src_area         = src_exposurearea / median(emap[ind])
        src_area_comment = '[skypixel**2] approximate FOV, emap integral / median emap'
      endif else begin
        src_area         = 0
        src_area_comment = '[skypixel**2] emap is zero in aperture'
      endelse
      
      ccd_is_in_neighborhood = replicate(0B, 10)
    endelse ; DIFFUSE SOURCE
    
    
    ; We start a dmhedit command file here, and finish & run it later.
    openw, unit, temp_text_fn, /GET_LUN
    comment = '[skypixel**2 s cm**2 count /photon] EXPOSURE not used for bkg scaling'
    printf, unit, src_exposurearea, comment, F='(%"#add\nBACKSCAL = %f / %s")'


    ;; ------------------------------------------------------------------------
    ;; We must decide which CCDs may have observed this source so we can compute
    ;; an ARF for them.   CIAO should do this, but it does not!
    ;; We could just compute an ARF for every CCD, but as a compromise to save
    ;; computations we'll process whatever CCDs appear in the neighborhood.
    src_ccd_fraction          = histogram([inband_events.ccd_id], MIN=0, MAX=9) / float(n_elements(inband_events))
    
    ccd_list                  = where((src_ccd_fraction GT 0) OR ccd_is_in_neighborhood, ccd_count)
    
    
    ;; ------------------------------------------------------------------------
    ;; We must work around CIAO's sometimes inappropriate assignment of the EXPOSURE
    ;; keyword, so that xspec will derive accurate flux values.  The problem is 
    ;; explained at http://asc.harvard.edu/ciao/ahelp/times.html:
    ;;
    ;; "As explained above the ONTIME/LIVETIME/EXPOSURE keywords are
    ;; for the aim chip which, by CXC convention, corresponds to the first
    ;; GTI. Each chip has different values of each of these keywords. IF
    ;; someone is doing analysis on an off-aim-chip CCD, AND the values
    ;; are considerably different for each chip, then the analysis will be
    ;; impacted. For most observations the values for the different chips
    ;; are about the same but this can be problem for observations with a
    ;; lot of telemetry saturation, crowded fields, or period of extended
    ;; background flares."
    
    ;; Our solution is choose ONTIME/LIVETIME/EXPOSURE values from the source's primary CCD,
    ;; and to then weight each CCD's ARF (below) by the ratio of it's EXPOSURn to
    ;; EXPOSURE.
    primary_ccd_fraction_i    = max(src_ccd_fraction, primary_ccd)
    
    primary_ccd_name = string(primary_ccd,F='(I0)')
    src_ontime   = psb_xpar( src_header, 'ONTIME'  + primary_ccd_name)
    src_livetime = psb_xpar( src_header, 'LIVTIME' + primary_ccd_name)
    src_exposure = psb_xpar( src_header, 'EXPOSUR' + primary_ccd_name)

    if (src_exposure LE 0) then begin
      print, 'EXPOSUR' + primary_ccd_name, src_spectrum_fn, F='(%"WARNING: keyword %s in %s is not positive; using EXPOSURE instead.")'
      src_exposure = psb_xpar( src_header, 'EXPOSURE')
    endif

    comment = "primary CCD is " + primary_ccd_name
    print, comment
    printf, unit, src_ontime,   comment, $
                  src_livetime, comment, $
                  src_exposure, comment, $
                  S_FILTER    , $
                  F='(%"#add\nONTIME = %f / [s] %s\nLIVETIME = %f / [s] %s\nEXPOSURE = %f / [s] %s\nS_FILTER=\"...\"\nS_FILTER=\"%s\" / STATUS filter applied by AE")'
                  ; S_FILTER is assigned twice above to work around a dmhedit bug (in CIAO 4.3) that converts a whitespace value to the integer zero.
    free_lun, unit
    
    cmd = string(src_spectrum_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
    run_command, cmd, /QUIET


    if is_diffuse then begin
      ; After 2010? CXC threads use the newish tool sky2det to make wmaps, instead of dmextract, to account for bad pixels and CCD edges.  We calibrate diffuse extractions with our own algorithm, and thus continue to use dmextract.
      ; However, dmextract does not propagate many of the header keywords (e.g. SUM_2X2, FEP_CCD, ORCMODE, OCLKPAIR) that are needed in the new version of mkwarf, so we do that below.  (See CXC Ticket #16170.)
      openw, unit, temp_par2_fn, /GET_LUN
        printf, unit, ['cycle,s,h,,,'   , $
                       'detnam,s,h,,,'  , $
                       'sum_2x2,s,h,,,' , $
                       'orc_mode,s,h,,,', $
                       'oclkpair,s,h,,,', $
                       'firstrow,s,h,,,', $
                       'exptime,s,h,,,' , $
                       'fep_ccd,s,h,,,' , $
                       'detnam,s,h,,,'  , $
                       'sim_x,s,h,,,'   , $
                       'sim_y,s,h,,,'   , $
                       'sim_z,s,h,,,'   , $
                       'dy_avg,s,h,,,'  , $
                       'dz_avg,s,h,,,'  , $
                       'dth_avg,s,h,,,' , $
                       'ontime,s,h,,,'  , $
                       'grating,s,h,,,' ], F='(%"%s")'
      free_lun, unit
      
      cmd = string(src_events_fn, temp_par1_fn, temp_par2_fn, temp_par1_fn, src_spectrum_fn, F="(%'dmmakepar ""%s[events]"" %s template=%s clob+ \ndmreadpar %s ""%s[WMAP]"" clob+')")
      run_command, cmd
    endif




    ;; ------------------------------------------------------------------------
    ;; Save spectral information for summary plots.
    psb_xaddpar, stats, 'CCD_CNT',  ccd_count,              'number of CCDs under source'
    psb_xaddpar, stats, 'CCD_PRIM', primary_ccd,            'primary CCD ID '
    psb_xaddpar, stats, 'CCD_FRAC', primary_ccd_fraction_i, 'primary CCD fraction'
    psb_xaddpar, stats, 'SRC_AREA',      src_area, src_area_comment
    psb_xaddpar, stats, 'SRC_RAD',  sqrt(src_area/!PI),           '[skypixel] sqrt(SRC_AREA/!PI)'
    psb_xaddpar, stats, 'EXPOSURE', src_exposure,                 '[s] EXPOSURE from src spectrum file'

    
    ;; ------------------------------------------------------------------------
    ;; ------------------------------------------------------------------------
    ;; BUILD ARF AND RMF FILES
    ;; There are six cases here because we have two ways to build an ARF:
    ;;   1. mkarf for point sources
    ;;   2. mkwarf for diffuse sources
    ;; and three ways to build an RMF:
    ;;   A. mkrmf
    ;;   B. mkacisrmf
    ;;   C. use RMF library for Townsley et al. CTI corrector
    ;;
    ;; On http://cxc.harvard.edu/ciao/bugs/mkacisrmf.html you'll find a warning that
    ;; XSPEC requires the energy grids in the ARF and RMF to match, and a warning that
    ;; mkacisrmf can modify the energy grid you request, and a recommendation that 
    ;; the mkacisrmf should be run first then mkarf/mkwarf told to match the binning.
    ;; 
    ;; When the Townsley et al. CTI corrector is used the same is true -- the 
    ;; RMF binning is predefined by the RMF library and one must construct the ARF
    ;; to match the binning.
    ;;
    ;; BUT, mkwarf uses WMAP, then passes a "weightfile" output to mkrmf.

    ;;POINT SOURCES
    ;; 1A: mkrmf     with explicit energy binning, then mkarf to match energy binning
    ;; 1B: mkacisrmf with explicit energy binning, then mkarf to match energy binning
    ;; 1C: weighted RMF from Townsley library,     then mkarf to match energy binning
    
    ;;DIFFUSE SOURCES
    ;; 2A: mkwarf    with WMAP & with explicit energy binning, 
    ;;                                         then mkrmf with energy binning passed through "weightfile" 
    ;; 2B: mkacisrmf with WMAP & with explicit energy binning, 
    ;;                                         then mkwarf with WMAP to match energy binning
    ;; 2C: weighted RMF from Townsley library, then mkwarf with WMAP to match energy binning

    ;;
    ;; Two large code blocks are shared among "orthogonal" subgroups of these six
    ;; cases so it is not convenient to block structure the code.
    ;; Also there are ordering requirements since some ARF constructors require the RMF
    ;; and some RMF constructors require the ARF.
    ;; Thus the code below has an odd structure (or lack of structure).
    ;; ------------------------------------------------------------------------
    ;; ------------------------------------------------------------------------
    file_delete, arf_fn, /ALLOW_NONEXISTENT
    
    ; Decide whether we will build an RMF file (which takes a long time).
    
    if keyword_set(generic_rmf_fn) then begin
      ; The observer is supplying an RMF via GENERIC_RMF_FN.
      rmf_fn    = generic_rmf_fn
      reuse_rmf = 1B
      print, 'Using generic RMF file.'
      
    endif else if is_diffuse then begin
      ; For diffuse source we never reuse the RMF since the observer might change the region.
      reuse_rmf = 0B
      file_delete, rmf_fn, /ALLOW_NONEXISTENT
      
    endif else if file_test(rmf_fn) then begin
      ; We found an existing point-source RMF (or a symlink to one), so reuse it to avoid the very long computation to build a new one.
        reuse_rmf = 1B
        print, 'Re-using existing RMF file.'
        
    endif else if file_test(rmf_fn,/DANGLING_SYMLINK) then begin
      ; We found a symlink, currently dangling (the referenced file does not exist).
      reuse_rmf = 0B
      
      if file_test(file_dirname(file_readlink(rmf_fn))) then begin
        ; If the DIRECTORY of the referenced file exists, then we assume the observer wants the RMF to be stored there,
        ; e.g. to save space on the volume currently holding the target.
        print, file_dirname(rmf_fn), F='(%"Storing RMF file in %s, as specified by symlink.")'
      
      endif else begin
        ; If the DIRECTORY of the referenced file does NOT exist, then the symlink in improper and we remove it.
        file_delete, rmf_fn
        print, rmf_fn, F='(%"Removed improper symlink %s.")'
      endelse
      
    endif else reuse_rmf = 0B  ; No RMF file or symlink found.
    


    ; NOTE that if you use an energy range too large then two things can happen:
    ;   (1) mkacisrmf may give a warning:
    ;       INFO: Effective user energy (keV) grids will be re-arranged in 0.25000 - 11.00000
    ;       See Bug Page for mkacisrmf (CIAO 4.1).
    ;   (2) The mkwarf may fail because the energy range is outside that covered in the FEF files:
    ;       ERROR: Min egridspec energy=0.25 below min FEF energy=0.277
    ;       ERROR: Max egridspec energy=11 above max FEF energy=9.886
    ;
    ; As of Jan 2011, the range of RMF information in CALDB is  0.24500 - 9.88600 keV.
    if ~keyword_set(energy_grid_spec) then energy_grid_spec = '0.3:9.886:0.005'
    
    ; The addrmf tool has the unfortunate habit of re-ordering the SPECRESP and EBOUNDS
    ; HDU's.  Ideally we'd be able to deal with that by simply specifying the HDU name
    ; in subsequent commands (e.g. mkwarf) that need the RMF.  
    ; But, the CXC and the LKT CTI corrector use different legal names for the
    ; matrix (SPECRESP in older CIAO versions, MATRIX in CIAO 4.0, and SPECRESP MATRIX in Townsley et al.).  
    ; Also CIAO 4.0 seems to have a bug that prevents it from accepting the HDU name [SPECRESP MATRIX] in the mkarf call.  
    ; Re-ordering the HDU's after addrmf (using dmcopy & dmappend) results in a file that has
    ; FITS errors (according to fverify) and leads to errors in XSPEC.
    ; So ... we're going to have to keep track of which HDU _number_ in the RMF file
    ; contains the SPECRESP matrix.
    if reuse_rmf then begin
      fits_open, rmf_fn, fcb, /NO_ABORT, MESSAGE=error
      if keyword_set(error) then message, 'ERROR reading ' + rmf_fn
      fits_close, fcb
      
      specresp_hdu_number = 1 + (where(strmatch(fcb.EXTNAME, '*SPECRESP*') OR strmatch(fcb.EXTNAME, '*MATRIX*')))[0]
    endif else specresp_hdu_number = 2


    ;; ------------------------------------------------------------------------
    ;; RMF: 1A, 1B
    if (rmf_tool NE 'Townsley') AND (NOT is_diffuse) then begin
      ;; Use CIAO to build an RMF appropriate for the peak on the primary CCD.
      on_this_ccd = where(inband_events.ccd_id EQ primary_ccd)
    
      chipx  = mean( inband_events[on_this_ccd].chipx, /DOUBLE )
      chipy  = mean( inband_events[on_this_ccd].chipy, /DOUBLE )
      
      if (rmf_tool EQ 'mkrmf') then begin
        ;; RMF: 1A
        ;; Look up correct FEF file and call mkrmf.
        cmd = string(src_spectrum_fn, primary_ccd, round(chipx), round(chipy), F="(%'acis_fef_lookup infile=%s chipid=%d chipx=%d chipy=%d')")
  
        run_command, cmd
        
        cmd = string(rmf_fn, energy_grid_spec, DETCHANS, mkrmf_log_fn, $
                     F="(%'mkrmf infile="")acis_fef_lookup.outfile"" weights=none outfile=%s axis1=""energy=%s"" axis2=""pi=1:%d:1"" logfile=%s')")
        if ~reuse_rmf then run_command, cmd
      endif else begin
        ;; RMF: 1B
        ;; Use the more recent tool mkacisrmf.  The detector location is specified by (ccd_id,chipx,chipy).
        ;; We must specify a matched pair "infile" = phase2_response_file and "gain" = gainfile or use CALDB for both;
        ;; see http://asc.harvard.edu/ciao/threads/mkacisrmf/index.html#location
        cmd = string(phase2_response_file, rmf_fn, energy_grid_spec, DETCHANS, primary_ccd, round(chipx), round(chipy), $
                     gainfile, obsdata_filename, mkrmf_log_fn,$
                     F="(%'mkacisrmf infile=%s outfile=%s wmap=none energy=""%s"" channel=1:%d:1 chantype=PI ccd_id=%d chipx=%d chipy=%d gain=%s obsfile=%s logfile=%s')")
        if ~reuse_rmf then run_command, cmd
      endelse
    endif ; (rmf_tool NE 'Townsley') AND (NOT is_diffuse)


    ;; ------------------------------------------------------------------------
    ;; RMF: 1C, 2C 
    if (rmf_tool EQ 'Townsley') then begin
      ;; Townsley et. al CTI corrector RMF library.
      ;; We'll weight the RMFs in the library by the number of counts in each RMF region.
      rmf_files  = 'junk'
      rmf_counts = 0L
      for jj = 0, ccd_count-1 do begin
        this_ccd = ccd_list[jj]
        ind = where(inband_events.ccd_id EQ this_ccd, count)
        if (count GT 0) then begin
          these_events = inband_events[ind]
          
          yrange_spec = ['_y1-128','_y129-256','_y257-384','_y385-512','_y513-640','_y641-768','_y769-896','_y897-1024']
          chipy_histogram = histogram([these_events.chipy], MIN=1, BINSIZE=128, NBINS=8)
          rmf_files  = [rmf_files, string(this_ccd,F='(%"ccd%d")') + yrange_spec + '.rmf']
          rmf_counts = [rmf_counts, chipy_histogram]
        endif ;(count GT 0)
      endfor
      
      ind = where(rmf_counts GT 0, num_rmf)
      rmf_files  = rmf_files [ind]
      rmf_counts = rmf_counts[ind]
      
;help, total(/INT, rmf_counts), n_elements(inband_events)

      ind = where(file_test(rmf_dir + rmf_files) EQ 0, num_not_found)
      if (num_not_found GT 0) then begin
        print, 'ERROR: cannot find RMF files: ', rmf_files[ind]
        print, 'WARNING: Skipping RMF & ARF generation; you must make your own or remove this source from the catalog before running /MERGE_OBSERVATIONS!'
        continue
      endif
      
      if (num_rmf EQ 1) then begin
        ; We use a symbolic link to avoid copying RMF's.
        if ~reuse_rmf then file_link, rmf_dir_from_source + rmf_files[0], rmf_fn, /VERBOSE
 
      endif else begin
        ; We run the addrmf command using an ASCII list file because the command line
        ; parameters seem to have limiting length restrictions
        rmf_weights = rmf_counts / total(rmf_counts)
        forprint, rmf_files, rmf_weights, F='(A0,1x,F5.3)', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
        cd, CURRENT=cwd  &  cwd=cwd+'/'
        cmd = string( temp_text_fn, cwd+rmf_fn, F="(%'addrmf ""@%s"" rmffile=%s')")

        if ~reuse_rmf then begin
          run_command, /HEASOFT, DIRECTORY=rmf_dir, cmd, STATUS=status 
        
          if NOT keyword_set(status) then begin
            specresp_hdu_number = 3
          endif else begin
            ; Sometimes the library RMFs don't have the same structure and addrmf fails.
            ; In this case we simply link to the library RMF with the largest weight.
            file_delete, rmf_fn, /ALLOW_NONEXISTENT
            dum = max(rmf_weights,imax)
            file_link, rmf_dir_from_source + rmf_files[imax], rmf_fn, /VERBOSE
          endelse
        endif
      endelse ;num_rmf GT 1
      
      if ~reuse_rmf then begin
        ; As of CIAO 3.2.2 the mkwarf program creates another problem for us.  
        ; Since it insists on looking up RMF information, if we use an energy range that's outside that 
        ; known by the CALDB then we will get this error message:
        ;         ERROR: Max egridspec energy=9.98 above max FEF energy=9.886
        ;       See Bug Page for mkacisrmf (CIAO 4.1).
        ; Thus we have to trim down the energy range used in the Townsley RMF.
        cmd = string(rmf_fn, specresp_hdu_number, temp_rmf_fn, $
                   F="(%'dmcopy ""%s[%d][ENERG_HI<9.88]"" %s option=all')")
        run_command, cmd
         
        file_delete, rmf_fn, /ALLOW_NONEXISTENT
        file_move, temp_rmf_fn, rmf_fn
      endif
    endif ;(rmf_tool EQ 'Townsley')


    ;; ------------------------------------------------------------------------
    ;; RMF: 2B
    if (rmf_tool EQ 'mkacisrmf') AND is_diffuse then begin
          ;; Use the more recent tool mkacisrmf.  It gets the weights directly from src_spectrum_fn[WMAP]
          ;; rather than from the "weightfile" output of mkwarf.
          ;; The ccd_id, chipx, chipy parameters are ignored when wmap supplied, but they must be present.
          ;;
          ;; The "infile" parameter must be CALDB so that mkacisrmf will use information in the header
          ;; of the WMAP file to select the appropriate "Phase 2 response calibration file".
          ;; The "gain" parameter must also be CALDB so that an appropraitely matched gain file is used.
          ;; Every Phase 2 response calibration file has a corresponding matched gain file in the CALDB.
          ;; See http://asc.harvard.edu/ciao/threads/mkacisrmf/index.html#extracted
          print
          print, 'WARNING: mkacisrmf can run for a very long time ...'
          cmd = string(rmf_fn, src_spectrum_fn, energy_grid_spec, DETCHANS, keyword_set(aspect_fn) ? aspect_fn : '', mkrmf_log_fn, $
                     F="(%'mkacisrmf infile=CALDB outfile=%s wmap=""%s[WMAP]"" energy=""%s"" channel=1:%d:1 chantype=PI ccd_id=3 chipx=0 chipy=0 gain=CALDB asolfile=%s logfile=%s')")

          if ~reuse_rmf then run_command, cmd
     endif ;(rmf_tool EQ 'mkacisrmf')


    ;; ------------------------------------------------------------------------
    ;; ARF: 1A, 1B, 1C
    if (NOT is_diffuse) then begin
      ; Remove existing ARF files, because the contents of "ccd_list" can change as the aperture is changed.
      if ~keyword_set(reuse_arf) then file_delete, /ALLOW_NONEXISTENT, obsdir + string(indgen(10), F='(%"source%d.arf")') 
      
      ; Construct a point source ARF, with consideration that a source can span multiple CCDs.
      ccd_arf_fn = obsdir      + string(ccd_list, F='(%"source%d.arf")')
      asphist_fn = asphist_dir + string(ccd_list, F='(%"/ccd%d.asphist")')
      ccd_name   = string(ccd_list, F='(I0)')
      detsubsys  = "ACIS-" + ccd_name
      
      primary_arf_fn = ccd_arf_fn[ where(ccd_list EQ primary_ccd) ]
      
      fracexpo = fltarr(ccd_count)
      specresp = 0
      for jj=0,ccd_count-1 do begin
        ; Make an ARF for a single CCD at the catalog position.
        ; Since the extraction aperture has a finite size, over which the ARF could
        ; vary (due to exposure variations) we really ought to compute an ARF for the
        ; extraction region, somehow weighted by the PSF.  At present there's no
        ; easy way to do that.
        
        ; In 2007 at the introduction of CIAO 3.4 the recommended value for the obsfile parameter changed from a redirection to
        ; the parameter asphistfile to an event file.  This is noted in the CIAO 3.4 release notes and in the revision history of
        ; the CIAO thread "Step-by-Step Guide to Creating ACIS Spectra for Pointlike Sources". 
        ; I only noticed this change and revised AE in Feb 2010!!
        
        cmd = string(ccd_arf_fn[jj], asphist_fn[jj], obsdata_filename, rmf_fn, specresp_hdu_number, $
                     xpos_catalog, ypos_catalog, grating_spec, detsubsys[jj], pbk_parameter, mskfile, dafile, $
                     F="(%'mkarf outfile=%s asphistfile=""%s[ASPHIST]"" obsfile=""%s[EVENTS]"" engrid=""grid(%s[%d][cols ENERG_LO,ENERG_HI])""  sourcepixelx=%f sourcepixely=%f grating=%s detsubsys=%s %s maskfile=%s dafile=%s verbose=0')")
        
        ; Do the mkarf run if we are not reusing ARFs or if the ARF we're after is not on the disk.
        ; We've made the judgement (2014 Jan) that source repositioning will not significantly alter the ARF.
        if ~keyword_set(reuse_arf) || ~file_test(ccd_arf_fn[jj]) then run_command, cmd
        
        arf_table = mrdfits(ccd_arf_fn[jj], 1, arf_header, /SILENT, STATUS=status)
        if (status NE 0) then message, 'ERROR reading ' + ccd_arf_fn[jj]
        
        ; Weight the single-CCD ARF by the ratio of that CCD's EXPOSURn to the
        ; EXPOSURE keyword in the spectrum (!) file, not the ARF file!!
        ; See notes above about "reassinging ONTIME/LIVETIME/EXPOSURE values" in the spectrum file.
  
        this_ccd_exposure = psb_xpar( src_header, 'EXPOSUR' + ccd_name[jj])

        if (this_ccd_exposure LE 0) then begin
          print, 'EXPOSUR' + ccd_name[jj], src_spectrum_fn, F='(%"WARNING: keyword %s in %s is not positive; using EXPOSURE instead.")'
          this_ccd_exposure = src_exposure
        endif

        specresp = specresp + (this_ccd_exposure/float(src_exposure)) * arf_table.SPECRESP
        
        fracexpo[jj] = psb_xpar( arf_header, 'FRACEXPO')
      endfor ;jj
    
      ;; Let's use the primary CCD's FITS headers for the final ARF file, 
      ;; but set ONTIME, LIVTIME, & EXPOSURE to match files in spectrum, and
      ;; update FRACEXPO to be the fraction of time spent on _any_ CCD.
      pheader    = headfits(primary_arf_fn)
      arf_header = headfits(primary_arf_fn, EXT=1)
           
      ; Omit comments (with units) in fxaddpar calls below to retain comments already in header.
      psb_xaddpar, arf_header, 'ONTIME',   src_ontime
      psb_xaddpar, arf_header, 'LIVETIME', src_livetime
      psb_xaddpar, arf_header, 'EXPOSURE', src_exposure
      ; Assuming that each single-CCD FRACEXPO value is computed (by mkarf) with respect to the same total exposure value, we sum them.
      psb_xaddpar, arf_header, 'FRACEXPO', total(fracexpo), strjoin(string(100*fracexpo,F='(%"%d%%")'),' + ')
    endif ; point source

                
    ;; ------------------------------------------------------------------------
    ;; ARF: 2B, 2C
    if ((rmf_tool EQ 'mkacisrmf') OR (rmf_tool EQ 'Townsley')) AND is_diffuse then begin
        if keyword_set(reuse_arf) then print, 'WARNING: REUSE_ARF option is ignored for diffuse extractions.'
        
        ; We've already built (above) a weighted RMF, either manually using 
        ; the Townsley RMF library or via mkacisrmf using CALDB.
        ; Now we want to build an ARF with the same energy binning that's
        ; weighted using the WMAP output of dmextract above.
        ;
        ; Even though we don't care about the "weightfile" output of mkwarf, we have   
        ; to supply a filename -- "none" is not accepted -- and the tool will still try
        ; to look up RMF information.  For certain early epochs of data CALDB won't be able
        ; to find CTI-corrected RMF information and will die.  As a workaround we supply
        ; an arbitrary but valid value for "feffile" to prevent CALDB from doing a search.

        dummy_feffile = (file_search( getenv('CALDB')+'/data/chandra/acis/fef_pha/*fef*', COUNT=fef_count ))[0]
        if (fef_count EQ 0) then message, 'ERROR: could not find a dummy feffile to pass to mkwarf'

        cmd = string(src_spectrum_fn, arf_fn, temp_wgt_fn, rmf_fn, specresp_hdu_number, dummy_feffile, pbk_parameter, mskfile, dafile, keyword_set(aspect_fn) ? aspect_fn : '', $
                     F="(%'mkwarf infile=""%s[WMAP]"" outfile=%s weightfile=%s spectrumfile=NONE egrid=""grid(%s[%d][cols ENERG_LO,ENERG_HI])"" feffile=%s %s mskfile=%s dafile=%s asolfile=%s')")
      
        run_command, cmd
        
        if ~file_test(arf_fn) then begin
          print, stats_fn, F='(%"WARNING! Aborted extraction because mkwarf failed! You should investigate, and then remove %s so that the MERGE will not see it.")'
          continue
        endif

        ; Need to read ARF to populate arf_table, specresp, & pheader in prep for later section 
        ; where PSF Fraction=1 should be applied.
        arf_table = mrdfits(arf_fn, 1, arf_header, /SILENT, STATUS=status)
        if (status NE 0) then message, 'ERROR reading ' + arf_fn
  
        specresp = arf_table.SPECRESP
        pheader = headfits(arf_fn)    
     endif ;(rmf_tool NE 'Townsley') AND is_diffuse



    ;; ------------------------------------------------------------------------
    ;; ARF, RMF: 2A
    if (rmf_tool EQ 'mkrmf') AND is_diffuse then begin
      ;; Build a weighted ARF using the WMAP output of dmextract above.
      ;; We specify the energy binning directly and it is then passed to mkrmf
      ;; through the "weightfile" output of mkwarf.
        cmd = string(src_spectrum_fn, arf_fn, mkrmf_wgt_fn, energy_grid_spec, pbk_parameter, mskfile, dafile, keyword_set(aspect_fn) ? aspect_fn : '', $
                     F="(%'mkwarf infile=""%s[WMAP]"" outfile=%s weightfile=%s spectrumfile=NONE egrid=""%s"" feffile=CALDB %s mskfile=%s dafile=%s asolfile=%s')")
      
        run_command, cmd

        if ~file_test(arf_fn) then begin
          print, stats_fn, F='(%"WARNING! Aborted extraction because mkwarf failed! You should investigate, and then remove %s so that the MERGE will not see it.")'
          continue
        endif

        ; Need to read ARF to populate arf_table, specresp, & pheader in prep for later section 
        ; where PSF Fraction=1 should be applied.
        arf_table = mrdfits(arf_fn, 1, arf_header, /SILENT, STATUS=status)
        if (status NE 0) then message, 'ERROR reading ' + arf_fn
  
        specresp = arf_table.SPECRESP
        pheader = headfits(arf_fn)    

        ;; The mkrmf tool wants the "weightfile" produced by mkwarf above.
        ;; The "energy=0:1" param to mkrmf is a dummy to keep the parser happy -- the energy grid comes
        ;; from the weightfile.
        cmd = string(mkrmf_wgt_fn, rmf_fn, DETCHANS, mkrmf_log_fn, $
                     F="(%'mkrmf infile=CALDB weights=%s outfile=%s axis1=""energy=0:1"" axis2=""pi=1:%d:1"" logfile=%s')")
        if ~reuse_rmf then run_command, cmd

    endif ; (rmf_tool EQ 'mkrmf') AND is_diffuse
    

    
    
    ;; ------------------------------------------------------------------------
    ;; Scale the ARF.
    ENERG_LO = arf_table.ENERG_LO
    ENERG_HI = arf_table.ENERG_HI
    channel_midenergy = 0.5 * (ENERG_LO + ENERG_HI)

    if (NOT is_diffuse) then begin
      ;; For point sources we are scaling down the ARF by the PSF fraction, so that the calibration 
      ;; is corrected for the light we discarded.
  
      ; We use linterp instead of interpol() to avoid wild extrapolation if the PSF samples
      ; we have don't cover the energy range of the ARF very well.
      if (n_elements(psf_fraction) GT 1) then begin
        linterp, psf_fraction.energy, psf_fraction.fraction, channel_midenergy, psf_frac_column
      endif else begin
        psf_frac_column = replicate(psf_fraction.fraction, n_elements(ENERG_LO))
      endelse
      
      if (max(psf_frac_column) GT 1) then begin
        print, 'ERROR: PSF fraction larger than 1 detected'
        GOTO, FAILURE 
      endif

      effective_fov      = 1
      effective_fov_unit = 'NA'
      specresp_unit      = 'cm**2 count /photon'
    endif else begin
      ;; For diffuse sources we choose to scale up the ARF (and change its units) so as to put 
      ;; the calibration on a "per arcsec**2" basis.
      ;; See Broos et al. (2010) and the "Diffuse Sources" section of the AE manual for an explanation of this idea; 
      ;; the comment block below is simply a different wording of the same explanation.
      ;; Our diffuse calibration strategy does NOT make use of the CIAO tool sky2tdet, introduced in CIAO 4.3 (Dec 2010).
            
      ; The observer will at some point in the reduction want to normalize fluxes and luminosities by
      ; a "field of view on the sky" in order to get a "surface brightness". I think that diffuse
      ; extractions from multiple obsids can be most clearly combined in the MERGE stage if they are
      ; FIRST expressed in terms of the physical quantity of surface brightness. The fundamental
      ; reason for this is that the concept of "field of view on the sky" is most clearly defined for
      ; a single obsid. An extraction region fixed on the sky may cover very differently sized
      ; portions of the detector in each obsid, due to field edges and point source masking.
      ; 
      ; Thus, I think that at this point we should express the physical calibration of this
      ; single-obsid diffuse extraction in terms of count /s /cm**2 /skypixel**2 (rather than the usual
      ; count /s /cm**2). It is at first tempting to use the OGIP keyword AREASCAL to store the skypixel**2
      ; quantity, since XSPEC will normalize the spectrum by AREASCAL when it is read in. However, to
      ; support the cplinear background model AE uses AREASCAL for the purpose of background spectrum
      ; scaling; thus it would be very confusing to use AREASCAL here to represent field of view on
      ; the sky. The other places we could effect a field of view normaliation would seem to be in the
      ; EXPOSURE keyword or in the ARF; I choose the latter. Thus, we will compute a field of view
      ; (skypixel**2) quantity below, and will scale up the ARF by that quantity; the ARF will then be in
      ; units of cm**2 skypixel**2. XSPEC will then be modeling surface brightness (not flux) in units of
      ; count /s /cm**2 /skypixel**2.
      ; 
      ; What "field of view" quantity is appropriate? IF the response of the instrument was constant
      ; over the extraction region, then the geometric size of the region would obviously be the
      ; appropriate quantity. However, the response of the detector is clearly not constant over the
      ; extraction region, due to field edges, source masking, multiple CCDs in the region,
      ; vignetting, etc. Our goal should be understood to be to find a field of view size that, when
      ; combined with the ARF given to XSPEC, will correctly calibrate the surface brightness of the
      ; extracted data. Now, IF we had an ARF which accounted for the exposure variation across the
      ; extraction region (including areas of zero response), then an appropriate field of view would
      ; seem to be the simple geometric area of the region. However, the ARF we get from mkwarf is
      ; very definately not such an ARF. Instead, mkwarf forms a weighted average of ARFs computed
      ; over the CCD tiles which were involved in the observation. All of these ARFs are "full
      ; strength", i.e. none are reduced to account for lower-than-normal exposure anywhere (e.g. due
      ; to chip gaps, source masking, or field edges); see Helpdesk Ticket #8154. In short, mkwarf
      ; (unlike mkarf) knows nothing about the bad pixels and chip edges in the exposure map.
      ; 
      ; Thus, we can think of the field of view quantity that we seek as the physical size of a
      ; virtual detector whose response is described by the ARF in hand (from mkwarf) that would
      ; produce the observed data in the given EXPOSURE time. This is analogous to the "effective
      ; exposure time" that we compute for point sources, which is the exposure time required to
      ; produce the observed data if the source was located on a section of the detector with a nominal
      ; response (e.g. a PIMMS ARF computed on-axis).
      ; 
      ; Now, we have on hand an exposure map within the extraction region, computed for some
      ; mono-energy E0. Note that the exposure map at any point (x,y) is supposed to be precisely
      ; ARF(x,y,E0) * EXPOSURE. The integral of this exposure map (cm**2 s skypixel**2) can be thought of
      ; as a complete "measure" of the depth of the observation (at energy E0) in which effective area
      ; (cm**2), integration time (s), and extraction region size (skypixel**2) are interchangeable. IF
      ; the exposure map is correctly calibrated, then our goal would seem to be to give XSPEC an
      ; EXPOSURE value and ARF such that the observed spectrum at energy E0 is normalized by the
      ; integral of the exposure map (cm**2 s skypixel**2). Thus we should simply scale the ARF in hand so
      ; that ARF(E0) = <integral of emap> / EXPOSURE.

      ; The units of effective_fov are (arcsec/skypixel)**2 (s cm**2 skypixel**2) / (s cm**2) = arcsec**2
      ARF_at_emap_energy = interpol(specresp, channel_midenergy, emap_energy)
      
      effective_fov   = (arcsec_per_skypixel^2) * src_exposurearea / (src_exposure * ARF_at_emap_energy)
      effective_fov_unit = 'arcsec**2'
      psf_frac_column = 1
      specresp_unit   = 'arcsec**2 cm**2 count /photon'
      
      ; However, in practice we often find that emap(x,y,E0) != ARF(x,y,E0) * EXPOSURE. It is not
      ; clear why this occurs. Certainly there is a danger that the observer may compute the exposure
      ; map months or years prior to the ARF; if the mission calibration is revised during that period
      ; then the two will be inconsistent. There may be other software-related reasons why the two do
      ; not agree perfectly. 
      
      ; To sanity check the weighted ARF we'd like to compare it to the emap.
      ; However, the emap is full of regions with lower-than-nominal exposure time, which mkwarf knows nothing about.
      ; We need an estimate of the emap average within the aperture, if bad columns and dithered CCD edges did not exist.
      ; The robust mean estimate we computed earlier (robust_mean_exposure) should serve nicely.
      
      EA_at_typical_exposure = robust_mean_exposure / src_exposure
      ; Above, the term "exposure" is ambiguous.  
      ; robust_mean_exposure is an emap value with units s cm**2 ct /photon.
      ; src_exposure is an EXPOSURE time with units of seconds.
      
      print, emap_energy, ARF_at_emap_energy, robust_mean_exposure, src_exposure, EA_at_typical_exposure, F='(%"\nmkwarf has calculated the observatory effective area (EA) at the monoenergy of your emap (E0=%0.1f keV), averaged over the extraction region:\n  ARF(E0,region) = %0.1f cm**2. \n\nSince emap(E0,x,y) = ARF(E0,x,y) * EXPOSURE, a typical emap value in the aperture divided by EXPOSURE (= %0.4g / %d = %0.1f cm**2) should be close to the mkwarf EA.")' 
      
      
      temp = minmax([ARF_at_emap_energy, EA_at_typical_exposure])
      if (temp[1]/temp[0] GT 1.2) then begin
        print, ARF_at_emap_energy/EA_at_typical_exposure, emap_energy, F='(%"\nWARNING!  The inconsistency between those two estimates of ARF(E0,region) seems large (ratio=%0.2f). \n  Verify that your emap was constructed at the monoenergy you declared (EMAP_ENERGY=%0.1f keV). \n  Check whether your emap and ARF were built with different epochs of CALDB. ")'
      endif
      
    endelse ; diffuse source    
    
    ;; ------------------------------------------------------------------------
    ;; Write the ARF file.
    ; We want to write a binary table which has ENERG_LO, ENERG_HI, & SPECRESP columns that
    ; carry the TTYPE, TFORM, TUNIT, TLMIN, TLMAX, etc. information from arf_header, and
    ; which has the new columns BASE & PSF_FRAC.
    ; There are no convenient tools in AstroLib to read all the keywords asssociated with
    ; a column and write them back out (possibly in a different column number).
    ; Thus I will be lazy and assume the CIAO ARF file has ENERG_LO, ENERG_HI, & SPECRESP 
    ; in columns 1, 2, & 3 with no other columns.  We'll use the CIAO header and the
    ; various column-specific keywords for columns 1-3, adding BASE & PSF_FRAC as columns
    ; 4 & 5.
    arf_row = {ENERG_LO:0.0, ENERG_HI:0.0, SPECRESP:0.0, BASE:0.0, PSF_FRAC:0.0, EFFECTIVE_FOV:0.0}
    arf_table = replicate( arf_row, n_elements(ENERG_LO) )
    
    arf_table.ENERG_LO      = ENERG_LO
    arf_table.ENERG_HI      = ENERG_HI
    arf_table.BASE          = specresp
    arf_table.PSF_FRAC      = psf_frac_column
    arf_table.EFFECTIVE_FOV = effective_fov
    arf_table.SPECRESP      = specresp * psf_frac_column * effective_fov

    psb_xaddpar, pheader, 'CREATOR', creator_string
    writefits, arf_fn, 0, pheader
    
    ; Unit specifications follow the standard in "Specification of Physical Units within OGIP FITS files" at
    ; http://heasarc.gsfc.nasa.gov/docs/heasarc/ofwg/docs/general/ogip_93_001/ogip_93_001.html
    psb_xaddpar, arf_header, 'CREATOR', creator_string
    psb_xaddpar, arf_header, 'TUNIT3' , specresp_unit
    psb_xaddpar, arf_header, 'TUNIT4' , 'cm**2 count /photon', 'from mkarf'
    psb_xaddpar, arf_header, 'TUNIT5' , ''                  , 'PSF fraction'
    psb_xaddpar, arf_header, 'TUNIT6' , effective_fov_unit  , 'effective FOV (for diffuse source)'
    
    if (ccd_count GT 1) then $
      psb_xaddpar, arf_header, 'DETNAM', 'ACIS-' + strjoin(string(ccd_list,F='(I0)')), 'source spans multiple CCDs'
      
    mwrfits, arf_table, arf_fn, arf_header

    
    ;; Record the mean ARF value over the set of ARF bins whose mid-energy falls in the range specified by user.
    ;; (Thus a bin is included only if >=50% of it is in the energy range.) 
    ;; Since the ARF is not evenly sampled we must integrate the ARF & divide by Erange.
    arf_energy = 0.5 * (arf_table.ENERG_LO + arf_table.ENERG_HI)
    ind = where((arf_energy GE energy_range[0]) AND (arf_energy LE energy_range[1]))
    band_arf_table = arf_table[ind]
    arf_mean_i = total(band_arf_table.SPECRESP * (band_arf_table.ENERG_HI - band_arf_table.ENERG_LO), /D)$
                                         / total((band_arf_table.ENERG_HI - band_arf_table.ENERG_LO), /D)

    ;; Save calibration information for summary plots.
    psb_xaddpar, stats, 'FRACEXPO', psb_xpar( arf_header,'FRACEXPO'), 'FRACEXPO from src ARF file'
    psb_xaddpar, stats, 'MEAN_ARF', arf_mean_i, '['+specresp_unit+'] mean ARF value'

    
    if (~is_diffuse) then begin
      ;; Estimate the detected count rate (including background) within the central 3x3 cell (for use in screening for pile-up).
      ;; This must be done here, rather than at the end of EXTRACT_EVENTS section, because we need FRACEXPO (which is obtained above).
      rate_in_cell = (psb_xpar( stats, 'CELLFRAC') < 1.0) * (psb_xpar( stats, 'SRC_CNTS') / psb_xpar( stats, 'PSF_FRAC')) / $
                                                        (psb_xpar( stats, 'FRACEXPO') * psb_xpar( stats, 'EXPOSURE'))   ; count/s

      rate_in_cell *= psb_xpar( env_header  ,'EXPTIME') ; count/frame
      
      psb_xaddpar, stats, 'RATE_3x3', rate_in_cell, string(energy_range, F="(%'[count /frame] %0.2f:%0.2f keV, in 3x3 cell')")
      
      
      if (choose_S_FILTER || keyword_set(FORCE_S_FILTER_RECALC)) then begin
        ; It is important that AE choose an appropriate STATUS filter only ONCE, to ensure that the src aperture extraction and background extraction are consistent.  
        ; If we revise our decision each time this stage is run, a source could end up with inconsistent extractions. 
        ; Thus, we make it here only when the source property S_FILTER does not already exist in obs.parameters, or when when the observer calls for a new decision.


        ; We'd like AE to aggressively clean (via STATUS=0) as many sources as possible (to increase S/N), i.e. we'd like the max_cell_rate_for_aggressive_cleaning parameter below to be as large as possible.
        ; Ideally, the choice of that parameter would be based on a deep understanding of the counts/frame rate required to generate false positives from the aggressive cleaning steps (5x5 cleaning, aggressive afterglow detection).
        ; Lacking that understanding, we choose a threshold empirically using the procedure below.
      
;    obsname = getenv('OBS')
;    dir     = '../obs' + obsname + '/'  
;    acis_extract, dir+'obs.cat', obsname, COLLATED_FILENAME=dir+'obs.collated'
;
;    bt = mrdfits(dir+'obs.collated', 1)
;    lightly_cleaned   = (strtrim(bt.S_FILTER,2) EQ '')
;    just_below_threshold = ~lightly_cleaned AND (bt.RATE_3x3 GT 0.001)
;
;    forprint, SUBSET=where(lightly_cleaned)                           , bt.CATALOG_NAME, /NoCom, TEXTOUT='/tmp/lightly_cleaned.srclist'
;    forprint, SUBSET=where(just_below_threshold)                      , bt.CATALOG_NAME, /NoCom, TEXTOUT='/tmp/just_below_threshold.srclist'
;    forprint, SUBSET=where(~lightly_cleaned AND ~just_below_threshold), bt.CATALOG_NAME, /NoCom, TEXTOUT='/tmp/other.srclist'
;    
;    
;    acis_extract, '/tmp/lightly_cleaned.srclist'     , obsname, REGION_FILE='/tmp/lightly_cleaned.reg'     , REGION_TAG='lightly cleaned', COLLATED_FILENAME='/dev/null'
;                                                                                                                                          
;    acis_extract, '/tmp/just_below_threshold.srclist', obsname, REGION_FILE='/tmp/just_below_threshold.reg', REGION_TAG='near threshold', COLLATED_FILENAME='/dev/null'
;                                                                                                                                          
;    acis_extract, '/tmp/other.srclist'               , obsname, REGION_FILE='/tmp/other.reg'               , REGION_TAG='other', COLLATED_FILENAME='/dev/null'
;    exit
;
;  egrep "{cat}|polygon" /tmp/lightly_cleaned.reg      | sed -e 's/DodgerBlue/red/'     > /tmp/review.reg
;  egrep "{cat}|polygon" /tmp/just_below_threshold.reg | sed -e 's/DodgerBlue/orange/' >> /tmp/review.reg
;  egrep       "polygon" /tmp/other.reg                | sed -e 's/DodgerBlue/gray/'   >> /tmp/review.reg
;
;
;  ds9 "../../pointing_1/obsid_7033/bad_validation.evt[energy>500,energy<8000]" -region /tmp/review.reg &
;
;  This ds9 session shows the events (from our L1->L2 recipe) that should be discarded when a STATUS=0 filter is applied to the event list now passed to AE (spectral.evt).  The green polygons mark the extractions where AE has decided this filter is appropriate.  You should confirm that none of the "sources" you can see in these data are marked by green regions.
;       
       
        ; If you CHANGE this parameter, then you need to force the recalculation of the S_FILTER source property (see below) for ObsIDs already extracted by suppling /FORCE_S_FILTER_RECALC, or by removing S_FILTER from the obs.parameters file, which archives the S_FILTER keyword between passes.
        
        max_cell_rate_for_aggressive_cleaning = 0.005 ; count/frame, chosen Jul 2011 using IC1805 data.
    
        ; Make the S_FILTER decision.
        if (rate_in_cell LT max_cell_rate_for_aggressive_cleaning)  then begin
          S_FILTER_decision = 'STATUS=0'
          S_FILTER_comment  = 'STATUS=0 filter applied by AE'
        endif else begin
          S_FILTER_decision = ''
          S_FILTER_comment  = 'no STATUS filter applied by AE'
        endelse
        
        ; Record the decision in the parameter file.
        psb_xaddpar, obs_parameters, 'S_FILTER', S_FILTER_decision, S_FILTER_comment 
        writefits, obs_parameters_fn, 0, obs_parameters
        
        ; Force this source to be immediately re-extracted if the S_FILTER decision made above is inconsistent with the extraction just performed.
        repeat_this_source = (S_FILTER NE S_FILTER_decision)
        if repeat_this_source then $
          print, S_FILTER, S_FILTER_decision, F="(%'\nSTATUS filter has changed from \'%s\' to \'%s\'.')"
      endif ; choose_S_FILTER
    endif ; point source
    
    
    psb_xaddpar, stats, 'CREATOR', creator_string
    get_date, date_today, /TIMETAG
    psb_xaddpar, stats, 'EXTRDATE', date_today, 'UTC date of extraction'
    writefits, stats_fn, 0, stats
    
    
    ;; =============================================================================
    ;; BOTTOM OF LOOP ITERATING OVER SOURCES IN THE CATALOG
    ;; =============================================================================
    ;; The S_FILTER mechanism entails an unusual complication to our workflow---a parameter of the extraction (the type of STATUS filter desired) is not known until after the extraction has been done (because the decision requires SRC_CNTS, FRACEXPO, & PSF_FRAC).
    ;; We do not want to leave the data products produced by this extraction in an inconsistent state, where the S_FILTER keyword in obs.stats is not consistent with the STATUS filter used to perform the extraction.
    
    ;; Below, we implement a general capability for the extraction code to force an immediate re-extraction of the source.
    ;; Obviously, an infinite loop would occur if this mechanism was abused.
    ;; The technique we use to re-extract is to decrement the loop counter and then proceed to the top of the loop.
    if repeat_this_source then begin
      ii--
      print, 'Repeating the extraction of this source ...'
      continue
    endif
  endfor ; ii

  
  
  count = total(source_not_observed, /INT)
  if (count GT 0) then print, count, F="(%'\nWARNING!  EXTRACT_SPECTRA skipped for %d sources not observed.')"

;  save, /COMPRESS, sourcename, energy0, energy1, FILE='lowest_energies.sav'
;  print, '============================================================================='
;  print, 'Lowest two in-band energies (useful for NH estimation) saved in lowest_energies.sav'
;  print, '============================================================================='
  
endif ;keyword_set(extract_events) OR keyword_set(extract_spectra)



;; =============================================================================
if keyword_set(timing) then begin
;; =============================================================================
    color_manager, /PS_PSEUDO, RED=red, GREEN=green, WHITE=white, BLACK=black, LANDSCAPE=0
    !P.FONT =  0   ; "hardware" font

    ; SNR_RANGE[1] is the user's goal for defining groups; SNR_RANGE[0] is the lower limit allowed before we abort the grouping attempt
    if (n_elements(snr_range) EQ 0) then $
      snr_range = [4,10]
    if (n_elements(snr_range) NE 2) then begin
      print, 'ERROR: keyword SNR_RANGE should be a 2-element vector giving the range of SNR allowed for each time bin, e.g. [2.5,5].'
      GOTO, FAILURE      
    endif
    
    if (snr_range[1] LT 0) then begin
      print, 'ERROR: minimum SNR value (SNR_RANGE[1]) must be positive'
      GOTO, FAILURE
    endif
    
    if (n_elements(num_groups_range) EQ 0) then $
      num_groups_range = [2,50]
    if (n_elements(num_groups_range) NE 2) then begin
      print, 'ERROR: keyword NUM_GROUPS_RANGE should be a 2-element vector specifying how many time bins are desired, e.g. [2+8,250].'
      GOTO, FAILURE      
    endif

  
  for ii = 0L, num_sources-1 do begin
    ; Skip sources already determined to be off-field.
    if source_not_observed[ii] then continue

    ;; Construct filenames.
    sourcedir   = sourcename[ii]  + '/'
    obsdir      = sourcedir + obsname + '/' + extraction_subdir[ii]

    unnamed_src_stats_fn = sourcedir + src_stats_basename
    obs_stats_fn         =    obsdir + obs_stats_basename
    src_events_fn        =    obsdir + src_events_basename
    lc_smooth_fn         =    obsdir + lc_smooth_basename
    event_plot_fn        =    obsdir + event_plot_basename
    
    if (NOT file_test(obs_stats_fn)) then begin
;     print, F='(%"\n===================================================================")'
;     print, 'Source: ', sourcename[ii]
;     print, 'EXTRACTION SKIPPED: source not observed.'
      source_not_observed[ii] = 1
      continue
    endif

    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    ; Remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    run_command, /QUIET, ['pset dmcopy clobber=yes', 'pset dmextract clobber=yes']
  
    print, F='(%"\n===================================================================")'
    print, 'Source: ', sourcename[ii]
    obs_stats = headfits(obs_stats_fn)
    arf_mean_i = psb_xpar( obs_stats,'MEAN_ARF')
    
    if (arf_mean_i EQ 0) then begin
      print, 'WARNING: the MEAN_ARF value in obs.stats is either missing or zero; the light curve normalization will be incorrect.'
      arf_mean_i = 1.0
    endif



    ;; WE DO NOT APPLY BARYCENTRIC CORRECTIONS TO EVENT DATA HERE!!!
    ;; 1. CHANGING the time system in event lists here (e.g. source.evt) seriously corrupts the de-dithering 
    ;; operation in ae_make_psf, which seriously damages pile-up correction (recon_spectrum.pro).
    ;; 2. Applying barycentric corrections to a COPY of source.evt is hugely wasteful of disk space and CPU time,
    ;; since most sources will not warrent any later timing analysis.



    ;; Build in-band source event list to compute statistics.
    ;; We do NOT go ahead and sort the events by time with dmsort (in preparation for the Kolmogorov-Smirnov statistic computation below) because as of Oct 2005 dmsort has bugs which sometimes trash GTI tables!
    cmd = string(src_events_fn, 1000*energy_range, inband_events_fn, $
                 F="(%'dmcopy ""%s[energy=%6.1f:%7.1f]"" %s')")
    run_command, cmd

    inband_events = mrdfits(inband_events_fn, 1, src_header, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + inband_events_fn
    
    
    ;; The code below is complex because we must deal with the case of zero inband events.
    ;; In such a case we cannot simply skip our timing analysis because our later multi-obsid 
    ;; variability analysis needs to know about the exposure found in this obsid, even if no
    ;; counts were observed.

    ; If the FITS table is empty, mrdfits will return a scalar zero.
    if NOT keyword_set(inband_events) then begin
      ;; There are no in-band data so we skip various statistics.
      print, 'WARNING: no in-band data found in source region.'
      inband_src_counts      = 0
      primary_ccd_fraction_i = 0
      primary_ccd            = psb_xpar( obs_stats,'CCD_PRIM')
      
    endif else begin
      inband_src_counts = n_elements(inband_events)
    
      ; Handle sources spanning CCDs and dim sources.
      ccd_histogram = histogram([inband_events.ccd_id], MIN=0, MAX=9)
      
      primary_ccd_fraction_i = max(ccd_histogram, primary_ccd) / total(ccd_histogram)
    endelse
        
    ;; ------------------------------------------------------------------------
    ;; Find the GTI table corresponding to the primary CCD.
    extno   = 0
    success = 0
    repeat begin
      extno = extno + 1
      gti_header = headfits(inband_events_fn, EXTEN=extno, ERRMSG=error)
      if keyword_set(error) then break

      hduname = strtrim(psb_xpar( gti_header, 'HDUNAME'),2)
      if strmatch(hduname, string(primary_ccd,F='(%"GTI%d")')) then success = 1
    endrep until success

    if (success EQ 0) then begin
      ; Try to find *any* GTI table.
      extno   = 0
      success = 0
      repeat begin
        extno = extno + 1
        gti_header = headfits(inband_events_fn, EXTEN=extno, ERRMSG=error)
        if keyword_set(error) then break
  
        hduname = strtrim(psb_xpar( gti_header, 'HDUNAME'),2)
        if strmatch(hduname, "GTI*") then begin
          success = 1
          print, primary_ccd, hduname, F='(%"\nWARNING! Could not find GTI table for CCD%d; using GTI table %s.")'
        endif
      endrep until success
  
      if (success EQ 0) then begin
        print, inband_events_fn, F='(%"\nERROR: no GTI table found in %s!  Timing analysis skipped.")'
        continue
      endif
    endif
    
    gti_table = mrdfits(src_events_fn, extno, gti_header, /SILENT)
    
    if ((gti_table.start)[0] EQ (gti_table.stop)[0]) then begin
      print, hduname, inband_events_fn, F='(%"\nERROR: empty GTI table %s found in %s!  Timing analysis skipped.")'
      continue
    endif
  
    
    num_gti = n_elements(gti_table)

    total_ontime    = total(gti_table.stop - gti_table.start, /DOUBLE)
      
      
    
    if (inband_src_counts EQ 0) then begin
      probks = !VALUES.F_NAN
      print, 'Timing analysis skipped -- no in-band counts.'
    endif else begin
      ;; ------------------------------------------------------------------------
      ;; Sort the event times to support two compuations later which require sorting:
      ;; 1. Cumulative distribution of observed events.
      ;; 2. Call to function uniq() in the light curve table code.
      event_time = (inband_events.time)[sort(inband_events.time)]
      
      event_time_bin_index = lonarr(inband_src_counts)
      
      ;; Compute the Kolmogorov-Smirnov statistic for a uniform model light curve.
      ;; The code below essentially concatenates all the good time intervals together
      ;; and computes the cumulative exposure time at each of the observed events.
      ;; BACKGROUND is NOT subtracted, and variation in the background is not accounted for!
      prior_ontime = 0
      for jj = 0, num_gti-1 do begin
        prior_ontime = prior_ontime + ((event_time - gti_table[jj].start) > 0)  $
                                    - ((event_time - gti_table[jj].stop ) > 0)
      endfor
      
      uniform_cum_distn = prior_ontime / total_ontime
      
      ; These lines derived from ksone.pro in AstroLib.
      cum_distn_before_step = (  findgen(inband_src_counts)) / inband_src_counts
      cum_distn_after_step  = (1+findgen(inband_src_counts)) / inband_src_counts

      ks_distance = max( abs( uniform_cum_distn - cum_distn_before_step ) ) > $
                    max( abs( uniform_cum_distn - cum_distn_after_step  ) )
      
      if (primary_ccd_fraction_i LT 1) then begin
        print, 'WARNING!  The source spans multiple CCDs; variability may be overestimated.'
      endif  
      
      if (inband_src_counts LT 4) then begin
        ; We need at least 4 counts for KS probability to be meaningful.
        probks = !VALUES.F_NAN
        print, 'KS variability analysis skipped -- too few in-band counts.'
      endif else prob_ks, ks_distance, inband_src_counts, probks


      ;; ------------------------------------------------------------------------
      ;; Plot photon arrival times vs energy, and overplot the cumulative distributions
      ;; used in KS above.
      device, FILENAME=event_plot_fn
      
      tstart = psb_xpar( gti_header, 'TSTART')
      tstop  = psb_xpar( gti_header, 'TSTOP')
      
      tit = string(sourcename[ii],probks, F='(%"%s, P!DKS!N=%0.2g")')
            
      plot, (event_time-tstart)/1000., [inband_events.energy/1000.], XRANGE=[0,(tstop-tstart)/1000.], YRANGE=energy_range, XSTYLE=1+2, XMARGIN=[8,6], YSTYLE=1+2+8, THICK=3, PSYM=(inband_src_counts LT 200) ? 1 : 3, TITLE=tit, XTIT='Time (ks)', YTIT='Energy (keV)'
      
      axis, YAXIS=1, /SAVE, YRANGE=[0,1], YSTYLE=1+2, YTICKS=2, YTIT='Cumulative Distribution, data & uniform model', COLOR=red
      
      xx = [tstart, rebin(event_time,2*inband_src_counts,/SAMPLE), tstop]
      yy = [rebin(cum_distn_before_step,2*inband_src_counts,/SAMPLE), 1, 1]

      oplot, (xx-tstart)/1000., yy, COLOR=red
      
      ; Calculate points that lie on the model's cumulative distribution, such that the cumulative plot is horizontal during the dead intervals

      ; We want a point at (TSTART,0), at (TSTOP,1), and a point for each X-ray event.
      xx = [tstart,event_time,tstop]
      yy = [0,uniform_cum_distn,1]
      
      ; We want a point where exposure time begins accumulating (the start of the first GTI). 
       xx = [xx,  gti_table[0].start]
       yy = [yy,  0                 ]

      ; We want a point where exposure time stops accumulating (the end of the last GTI). 
       xx = [xx,  gti_table[num_gti-1].stop]
       yy = [yy,  1                        ]

      ; And we want points defining the intervals **between** GTIs.
      gti_fractions = (gti_table.stop - gti_table.start)/total_ontime
      for jj = 0, num_gti-2 do begin
        xx = [xx, gti_table[jj].stop, gti_table[jj+1].start]
        cum_fraction = total(gti_fractions[0:jj], /DOUBLE)
        yy = [yy, cum_fraction, cum_fraction]
        
      endfor
      for jj = 0, num_gti-1 do begin
        ; Illustrate GTI boundaries.
        oplot, (gti_table[jj].start - [tstart,tstart])/1000, [0,1], LINE=1, COLOR=white
        oplot, (gti_table[jj].stop  - [tstart,tstart])/1000, [0,1], LINE=1, COLOR=white
      endfor
      
      sind = sort(xx)
      
      oplot, (xx[sind]-tstart)/1000., yy[sind], COLOR=green

      device, /close
    endelse  ; (inband_src_counts GT 0) 

    
      
    ;; ------------------------------------------------------------------------
    ;; Compute a finely-binned and unequally-binned light curve which will then be grouped.
    ;; Compute smoothed light curve, and median energy curve, using adaptive kernel smoothing.
    
    ;; We're going to place time bin boundaries at the times that events occurred so we don't 
    ;; lose any timing resolution.  We'll define other bin boundaries as necessary to get a 
    ;; reasonable sampling of the time interval.
    ;; We rely below on the fact that event_time was sorted earlier.
    ;;
    ;; The convention that the /MERGE stage expects is that an event is counted (COUNTS column) 
    ;; in the bin to the LEFT of the boundary.  
    t_min   = gti_table[0        ].start
    t_max   = gti_table[num_gti-1].stop
    
    if (t_max LE t_min) then begin
      print, 'ERROR: time span covered by GTI table is zero!'  
      continue
    endif

    min_num_bins = 400
    max_binsize  = (t_max - t_min) / min_num_bins
    
    dim = min_num_bins + 10 + inband_src_counts
    bin_edges = dblarr(dim)
 
    bin_edges[0] = t_min
    event_ind    = 0L
    for jj=1L,dim-1 do begin
      ; Start by assuming that we'll define an empty bin with the nominal width.
      nominal_bin_stop = bin_edges[jj-1] + max_binsize

      ; If there is an event remaining, and its timestamp is close enough to define a bin then do it.
      if (event_ind LT inband_src_counts) then begin
        event_bin_stop = event_time[event_ind]
        if (event_bin_stop LE nominal_bin_stop) then begin
          ; Place a bin edge at this event timestamp.
          bin_edges[jj] = event_bin_stop

          ; Assign this time bin to every event in this time bin..
          ; Note there can be multiple events with the same timestamp.
          ; Recall that we've sorted "event_time" previously.
          repeat begin
            event_time_bin_index[event_ind] = jj-1
            event_ind = event_ind + 1
            if (event_ind GE inband_src_counts) then break ; repeat loop
          endrep until (event_time[event_ind] GT event_bin_stop)
          
          continue ; jj loop
        endif ; Defining a time bin.
      endif ; Unprocessed events remaining.
      
      ; Otherwise try to construct an empty bin of nominal width.
      if (nominal_bin_stop GE t_max) then break
      
      bin_edges[jj] = nominal_bin_stop
    endfor ;jj
    if (jj GE (dim-1)) then message, 'ERROR: loop failed to terminate.'
    
    ; The final bin edge is at TSTOP.
    bin_edges[jj] = t_max
    
    bin_edges = bin_edges[0:jj]
    
   
    ; Convert a list of bin edges to a list (one shorter) of min, center, and max values.
    nbins      = n_elements(bin_edges) - 1
    bin_min    = bin_edges[0:nbins-1]
    bin_max    = bin_edges[1:nbins]
    bin_center = 0.5*(bin_min + bin_max)
    
     
    ;; Compute the exposure*EA in each bin by integrating the GTIs over the bins, multiplying by DTCOR to account for time lost during frame transfers, and multiplying by an ARF value averaged over some energy range.
    ;; The ARF value accounts for FRACEXPO (fraction of time source was on live portion of detector).
    ;; Of course, in the CIAO conventions exposure time lost to dithering over dead portions of the detector is accounted for by reducing the ARF, not by changing any quantities that are formally related to "time".
    ;; The ARF value also accounts for the finite extraction aperture.
    exposure = dblarr(nbins)
    DTCOR = psb_xpar( src_header, 'DTCOR')
    for jj = 0, num_gti-1 do begin
      this_gti_integral = ((gti_table[jj].stop < bin_max) - (gti_table[jj].start > bin_min)) > 0

      exposure += (DTCOR * this_gti_integral * arf_mean_i)
    endfor
;   print, 'total exposure = ', total(exposure)
      
    nan_mask  = replicate(!VALUES.F_NAN,nbins)

    if (inband_src_counts EQ 0) then begin
      ;; Since there are no events, make various null columns for the light curve file.
      flux                = nan_mask
      flux_error          = nan_mask
      error               = nan_mask
      median_energy       = nan_mask   
      median_energy_error = nan_mask
      radius              = nan_mask
      time_histogram      = replicate(0, nbins)
      group_codes         = replicate(-1,nbins)
      group_codes[0]      = 1
      this_snr_goal       = !VALUES.F_NAN
    endif else begin
      ;; Do some sanity checking.
      if (max(event_time_bin_index) GT (nbins-1)) then message, 'ERROR: bug in code.'
      
      ;; Make a high resolution histogram of the event times & save reverse indexes.
      ;; We use NBINS keyword to ensure that we know the number of elements in "time_histogram" so that we can
      ;; properly decode the rindex array returned.
      time_histogram = histogram( event_time_bin_index, MIN=0L, BINSIZE=1, NBINS=nbins, REVERSE_INDICES=rindex )
      
      ; Re-order the data vector so that data belonging to each bin are grouped
      ; together, and those groups appear in the order of the bins.
      ; Sorting the data up front will avoid applying REVERSE_INDEXES to the data
      ; vector multiple times later.
      ; Expand the first segment of REVERSE_INDICES into vectors member_index_start and member_index_stop
      ; that define segments of the ordered data vector (sorted_energy_data) which belong to each time bin.
      member_index_start    =rindex[0:nbins-1] - (nbins + 1)
      member_index_stop     =rindex[1:nbins]   - (nbins + 1) - 1
      sorting_indexes= rindex[nbins+1:*]
      sorted_energy_data    = (inband_events.energy)[sorting_indexes]

      ;; Group the high resolution, low significance histogram.  We don't have a background available here.
      group_bins_to_snr, time_histogram, 0, 0, /GROUP_WITHOUT_BACKGROUND, $
                         SNR_RANGE=snr_range, NUM_GROUPS_RANGE=num_groups_range, $
                         this_snr_goal, group_codes

      ; Compute median energy over each group.
      ind = [where(group_codes EQ 1, num_groups), nbins]

      median_energy       = replicate(!VALUES.F_NAN,nbins)
      median_energy_error = replicate(!VALUES.F_NAN,nbins)
      
      defensive_count = 0L
      defensive_data  = sorted_energy_data[0]
      for kk=0,num_groups-1 do begin
        ind_left  = ind[kk]
        ind_right = ind[kk+1] - 1
        
        ; Locate the data associated with each group.
        concat_array_segments, sorted_energy_data, member_index_start[ind_left : ind_right], $
                                                   member_index_stop [ind_left : ind_right], data, num_data

        defensive_count=defensive_count+n_elements(data)
        ; Compute the median energy statistic and estimate an error using the 
        if (num_data GT 0) then begin
          defensive_data =[defensive_data, data]

          ; Compute the median energy and 1-sigma (68%) confidence interval.
          median_with_ci, data, CONFIDENCE_LEVEL=0.6827, median_value, limit_lower, limit_upper

          median_energy      [ind_left : ind_right] = median_value
          
          median_energy_error[ind_left : ind_right] = (limit_upper - limit_lower) / 2.0
        endif
      endfor ;kk

      if (inband_src_counts NE defensive_count) then message, 'BUG in grouping code detected!'
      if (total(defensive_data[1:*] NE sorted_energy_data) GT 0) then message, 'BUG in grouping code detected!'
      

      ;; Adaptively smooth the light curve.
      ;; With a normal histogram, the first and last time values at which an 
      ;; estimate of the distribution are available (i.e. the centers of the 
      ;; first & last bins) are inset half a bin from the range of the data ([t_min, t_max]).  
      ;; Similarly, with the variable bin size used here, distribution estimates 
      ;; whose kernels fall outside the range of the data are not meaningful.   
      ;; (For a flat kernel there are groups of identical smoothed values at the 
      ;; beginning and end of the vector.)  
      ;; * Thus, the largest useful kernel would be positioned at the center of 
      ;; the time range and would have a radius of nbins/2 (= MAX_RADIUS).
      ;; * Later we will discard smoothed values whose kernel falls outside the range of the data.
      
      ;; For bright sources there are a large number of time bins; if we let adaptive_density_2d try every possible
      ;; set of radii, then it will be very slow and consume a huge amount of memory.
      ;; Thus, we supply a MAX_RADIUS parameter which anticipates the worst-case situation where we need 
      ;; our kernel to encompass this_snr_goal^2 bins that each have 1 count, plus up to min_num_bins that are empty.
      max_radius = ceil((min_num_bins + this_snr_goal^2)/2) 
      
      ;; We use the FIELD_MASK input to leave gaps in light curve corresponding to bad time intervals.
      
      ;; Because we've decided to construct light curves even when the source spans multiple CCDs, we
      ;; can sometimes end up with the entries where time_histogram GT 1 and EXPOSURE EQ 0, which adaptive_density_2d
      ;; would choke on.  We simply discard those events.
      ;; Numerical precision problems can also produce a mismatch between "time_histogram" and "exposure" vectors.
      time_histogram = time_histogram * (exposure GT 0)

      adaptive_density_2d, time_histogram, this_snr_goal, EMAP=exposure, $
                           FIELD_MASK=(exposure GT 0), MAX_RADIUS=max_radius, $
                           flux, flux_error, radius, SILENT=1

      ;; Set smoothed quantities to NaN at the ends of the time series where the kernel falls outside the time range of the data.
      bin_index = lindgen(nbins)
      first_index = (        where((radius LE (bin_index-0))       AND finite(flux)) )[0]
      last_index  = (reverse(where((radius LE (nbins-1-bin_index)) AND finite(flux))))[0]
      
      if (((last_index-first_index) LT 1) OR (first_index EQ -1) OR (last_index EQ -1)) then begin
        ; All of the bins used the full dataset, i.e. the smooth curves are flat.
        ; Retain just two non-masked bins in the middle.
        ind   = where(finite(flux), num_finite)
        mid   = (num_finite/2) > 1
        ind   = ind[[mid-1,mid]]
      endif else begin
        ; Trim down the time series.
        ind = bin_index[first_index:last_index]
      endelse
       
      nan_mask[ind] = 1.0
    endelse ; (inband_src_counts GT 0)
    
    
    ;; Write smoothed lightcurve and median energies to a FITS table.
    pheader = headfits(inband_events_fn)
    psb_xaddpar, pheader, 'CREATOR', creator_string

    ; Unit specifications follow the standard in "Specification of Physical Units within OGIP FITS files" at
    ; http://heasarc.gsfc.nasa.gov/docs/heasarc/ofwg/docs/general/ogip_93_001/ogip_93_001.html
    energy_range_label = string(energy_range, F='(%"[%0.2f:%0.2f] keV ")')
    fxbhmake, theader, nbins, 'LIGHTCURVE', 'nonstandard format', /DATE, /INIT
    psb_xaddpar, theader, 'CREATOR', creator_string
    psb_xaddpar, theader, 'MEAN_ARF', arf_mean_i, string(energy_range_label, F="(%'[cm**2 count /photon] mean ARF, %s')")
    psb_xaddpar, theader, 'PROB_KS',  probks    , string(energy_range_label, F="(%'p-value for no-variability hypothesis, %s, single ObsID')")
    psb_xaddpar, theader, 'TSTART', psb_xpar( pheader, 'TSTART'), 'Observation start time'
    psb_xaddpar, theader, 'TSTOP',  psb_xpar( pheader, 'TSTOP') , 'Observation stop time'
    psb_xaddpar, theader, 'SNR_GOAL', this_snr_goal
    psb_xaddpar, theader, 'NUMGRPS', fix(total(group_codes>0))
    psb_xaddpar, theader, 'TUNIT1', "s",       'left bin edge'
    psb_xaddpar, theader, 'TUNIT2', "s",       'bin center'
    psb_xaddpar, theader, 'TUNIT3', "s",       'right bin edge'
    psb_xaddpar, theader, 'TUNIT4', "count",  '# events in bin'
    psb_xaddpar, theader, 'TUNIT5', "s cm**2 count /photon",'exposure*EA in bin, DTCOR applied'
    psb_xaddpar, theader, 'TUNIT6', "bins",    'kernel radius'
    psb_xaddpar, theader, 'TUNIT7', "photon /s /cm**2 ", 'via adaptive kernel smoothing'
    psb_xaddpar, theader, 'TUNIT8', "photon /s /cm**2 ", 'error on COUNT_RATE'
    psb_xaddpar, theader, 'TUNIT9', "flag",    'grouping flags'
    psb_xaddpar, theader, 'TUNIT10', "eV",      'median event energy in group'
    psb_xaddpar, theader, 'TUNIT11', "eV",      'error on GRP_MEDIAN_ENERGY'

    ; We could compute exposure under each window using radius values -- would have to do that BEFORE trimming
    ; the ends of the vectors above.
    row = { TIME_MIN: 0D, TIME: 0D, TIME_MAX: 0D, COUNTS:0, EXPOSURE:0.0, RADIUS:0.0, COUNT_RATE: 0.0, RATE_ERR: 0.0, GROUPING:0, GRP_MEDIAN_ENERGY: 0.0, GRP_MEDIAN_ENERGY_ERR: 0.0}
    bin_table = replicate(row, nbins)
    bin_table.TIME_MIN          = bin_min 
    bin_table.TIME              = bin_center 
    bin_table.TIME_MAX          = bin_max 
    bin_table.COUNTS            = time_histogram
    bin_table.EXPOSURE          = exposure
    bin_table.RADIUS            = nan_mask * radius
    bin_table.COUNT_RATE        = nan_mask * flux  
    bin_table.RATE_ERR          = nan_mask * flux_error 
    bin_table.GROUPING          = group_codes
    bin_table.GRP_MEDIAN_ENERGY     = median_energy
    bin_table.GRP_MEDIAN_ENERGY_ERR = median_energy_error
    
    writefits, lc_smooth_fn, 0, pheader
    mwrfits, bin_table, lc_smooth_fn, theader

    ;; Add the primary GTI table to the lightcurve file, to help document how EXPOSURE column was built.
    ;; WARNING!  It is tempting to add a MEAN_ARF column to this GTI table, so that in later timing analysis
    ;; the sensitivity of this extraction is directly available.  However, this is a very BAD IDEA.
    ;; The problem is that "GTI" tables are special tables in the CIAO data model.  Extra columns are not
    ;; properly managed; for example when the data model sorts a GTI table (e.g. after a dmmerge call) extra
    ;; columns are not re-ordered---corrupting the table.

    mwrfits, gti_table, lc_smooth_fn, gti_header



    print, total(/INT, (group_codes EQ 1)), (this_snr_goal GT snr_range[0]) ? '>=' : '<', this_snr_goal, lc_smooth_fn, F='(%"Grouped and smoothed light curves (%d bins with SNR%s%0.1f) & median energy timeseries written to %s")'                    

    psb_xaddpar, obs_stats, 'PROB_KS',  probks, string(energy_range_label, F="(%'p-value for no-variability hypothesis, %s, single ObsID')")
    psb_xaddpar, obs_stats, 'CREATOR', creator_string
    writefits, obs_stats_fn, 0, obs_stats
    
    if (this_snr_goal GT snr_range[0]) then begin
      ; If we expected to achieve the SNR goal in adaptive_density_2d then defensively check that occured.
      snr = (nan_mask * flux) / (nan_mask * flux_error)
      if (total(/INT, (snr/this_snr_goal LT 0.95)) GT 0) then print, 'WARNING! Some smoothed light curve samples failed to meet the SNR goal.'
    endif
  endfor; ii
  
  count = total(source_not_observed, /INT)
  if (count GT 0) then print, count, F="(%'\nWARNING!  TIMING skipped for %d sources not observed.')"

  color_manager, /X_PSEUDO

endif ;keyword_set(timing)



;; =============================================================================
if keyword_set(arf_correction_filename) then begin
;; =============================================================================
  ;; Read ARF correction table.
  correction_table = mrdfits(arf_correction_filename, 1, /SILENT, STATUS=status)
  if (status NE 0) then message, 'ERROR reading ' + arf_correction_filename

  corr_energy = correction_table.ENERGY
  corr_factor = correction_table.(1)
  
  arf_correction_column = (tag_names(correction_table))[1]
  
  function_1d, id1, corr_energy, corr_factor, TITLE=arf_correction_filename, XTIT='Energy (keV)', YTIT=arf_correction_column

  for ii = 0L, num_sources-1 do begin
    print, F='(%"\n===================================================================")'
    print, 'Source: ', sourcename[ii]

    ;; Construct filenames.
    obsdir    = sourcename[ii] + '/' + obsname + '/' + extraction_subdir[ii]
    arf_fn    = obsdir + arf_basename
    stats_fn  = obsdir + obs_stats_basename

    if (NOT file_test(arf_fn)) then begin
      print, 'CORRECTION SKIPPED: no ARF found.'
      continue
    endif

    ;; ------------------------------------------------------------------------
    ;; Read the existing ARF.
    pheader   = headfits(arf_fn)
    arf_table = mrdfits(arf_fn, 1, arf_header, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + arf_fn


    ;; ------------------------------------------------------------------------
    ;; Add a new column if necessary.
    if (NOT tag_exist( arf_table, arf_correction_column, INDEX=colnum )) then begin
      colnum = n_tags(arf_table)
      
      old_arf_table = temporary(arf_table)
      
      arf_row = create_struct( old_arf_table[0], arf_correction_column, 0.0 )
      arf_table = replicate( arf_row, n_elements(old_arf_table) )
      
      copy_struct, old_arf_table, arf_table
    endif

    ;; ------------------------------------------------------------------------
    ;; Interpolate correction to ARF energy samples, and write to table.
    arf_energy = 0.5 * (arf_table.ENERG_LO + arf_table.ENERG_HI)
    arf_table.(colnum) = interpol(corr_factor, corr_energy, arf_energy)


    ;; ------------------------------------------------------------------------
    ;; Recompute SPECRESP column
    colnames = tag_names(arf_table)
    is_correction = (colnames NE 'ENERG_LO') AND (colnames NE 'ENERG_HI') AND (colnames NE 'SPECRESP') AND (colnames NE 'BASE')
    
    specresp = arf_table.BASE
    for colnum=0, n_tags(arf_table)-1 do begin
      if (is_correction[colnum]) then begin
        specresp = specresp * arf_table.(colnum)
        print, 'Applying ', colnames[colnum]
      endif
    endfor
    
    arf_table.SPECRESP = specresp
    
    
    ;; ------------------------------------------------------------------------
    ;; Rewrite the ARF file.    
    writefits, arf_fn, 0, pheader
    mwrfits, arf_table, arf_fn, arf_header

    ;; ------------------------------------------------------------------------
    ;; Record the mean ARF value over the set of ARF bins whose mid-energy falls in the range specified by user.
    ;; (Thus a bin is included only if >=50% of it is in the energy range.) 
    ;; Since the ARF is not evenly sampled we must integrate the ARF & divide by Erange.
    arf_energy = 0.5 * (arf_table.ENERG_LO + arf_table.ENERG_HI)
    ind = where((arf_energy GE energy_range[0]) AND (arf_energy LE energy_range[1]))
    band_arf_table = arf_table[ind]
    arf_mean_i = total(band_arf_table.SPECRESP * (band_arf_table.ENERG_HI - band_arf_table.ENERG_LO), /D)$
                                                                       / total((band_arf_table.ENERG_HI - band_arf_table.ENERG_LO), /D)

    stats = headfits(stats_fn)
    psb_xaddpar, stats, 'MEAN_ARF', arf_mean_i, '[cm**2 count /photon] mean ARF value'
    writefits, stats_fn, 0, stats
  endfor
endif ;keyword_set(arf_correction_filename)



;; =============================================================================
if keyword_set(extract_backgrounds) then begin
;; =============================================================================

  if NOT keyword_set(min_num_cts)        then min_num_cts=5
  if NOT keyword_set(tweak_backscal)     then tweak_backscal=1
  
  if (tweak_backscal LE 0) then begin
    print, 'ERROR: parameter "tweak_backscal" must be positive'
    GOTO, FAILURE
  endif
  
  ;; For speed we'll do our search for a background region using an image rather than an event list.
  ;; Apply the specified energy filter so we're counting useful background events.

  ;; The background estimation must be most accurate for weak sources---the same sources for which AE will be applying a STATUS=0 filter to aggressively clean the event data.
  ;; We will do the same filtering here, so that the image in "temp_bkgimg_fn" below will be correct for these (majority) weak sources. 
  
  run_command, string(emap_filename, F="(%'get_sky_limits %s verbose=0 precision=2')")
  run_command, /QUIET, 'pget get_sky_limits dmfilter', filterspec

  ; Change from "xmin:xmax:#bins" syntax to "xmin:xmax:delx" syntax to ensure that 
  ; temp_bkgimg_fn has square pixels to keep CIAO happy.
  tokens = strsplit(filterspec, '=:#,', /EXTRACT)
  xmin  = float(tokens[1])
  xmax  = float(tokens[2])
  xbins = float(tokens[3])
  ymin  = float(tokens[5])
  ymax  = float(tokens[6])
  binsize = (xmax-xmin)/xbins
  filterspec = string(xmin,xmax,binsize,ymin,ymax,binsize, F='(%"x=%7.2f:%7.2f:%7.4f,y=%7.2f:%7.2f:%7.4f")')

  cmd = string(obsdata_filename, 1000*energy_range, filterspec, temp_bkgimg_fn, F="(%'dmcopy ""%s[energy=%6.1f:%7.1f,STATUS=0][bin %s]"" %s')")
  run_command, cmd  

  
  num_emap_pixels = replicate(-1L,num_sources)

  bkg_geometric_area = !PI*100^2
  for ii = 0L, num_sources-1 do begin
    ; Skip sources already determined to be off-field.
    if source_not_observed[ii] then continue

    ;; Construct filenames.
    sourcedir            = sourcename[ii] + '/'
    unnamed_src_stats_fn = sourcedir + src_stats_basename
    obsdir               = sourcedir + obsname + '/' + extraction_subdir[ii]
    obs_parameters_fn    = obsdir + obs_parameters_basename
    obs_stats_fn         = obsdir + obs_stats_basename
    src_region_fn        = obsdir + src_region_basename
    src_spectrum_fn      = obsdir + src_spectrum_basename
    bkg_region_fn        = obsdir + bkg_region_basename
    bkg_pixels_region_fn = obsdir + bkg_pixels_region_basename
    bkg_emap_fn          = obsdir + bkg_emap_basename
    bkg_events_fn        = obsdir + bkg_events_basename
    bkg_spectrum_fn      = obsdir + bkg_spectrum_basename

    if (~file_test(obs_stats_fn)) then begin
;     print, F='(%"\n===================================================================")'
;     print, 'Source: ', sourcename[ii]
;     print, 'EXTRACTION SKIPPED: source not observed.'
      source_not_observed[ii] = 1
      continue
    endif
    
    ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(sourcedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    if keyword_set(reuse_background) AND file_test(bkg_spectrum_fn) then continue
    
    ; Remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    run_command, /QUIET, ['pset dmcopy clobber=yes', 'pset dmimgpick clobber=yes', 'pset dmextract clobber=yes']

    
    ; Look up allowed range of background normalizations.
    unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    
    if (~keyword_set(error)) then begin
      BKSCL_LO = psb_xpar( unnamed_src_stats, 'BKSCL_LO') ; Smallest allowed bkg scaling.
      BKSCL_GL = psb_xpar( unnamed_src_stats, 'BKSCL_GL') ; Target bkg scaling.
      BKSCL_HI = psb_xpar( unnamed_src_stats, 'BKSCL_HI') ; Largest allowed bkg scaling.
    endif else begin
      print, error
      message, 'ERROR reading '+unnamed_src_stats_fn
    endelse

    print, F='(%"\n===================================================================")'
    print, sourcename[ii], strtrim(psb_xpar( unnamed_src_stats,'LABEL'),2), F='(%"Source: %s (%s)")'      
    
    ; Look up exposure integral in source extraction region.
    header = headfits(src_spectrum_fn, ERRMSG=error, EXT=1)    
    if (~keyword_set(error)) then begin
      src_exposurearea = psb_xpar( header, 'BACKSCAL') ; Units are skypix**2 s cm**2 count /photon.
      DETCHANS         = psb_xpar( header, 'DETCHANS')
    endif else begin
      print, 'WARNING: source spectrum '+src_spectrum_fn+' not found.'
      src_exposurearea = 1
      DETCHANS = 1024
    endelse
    
    bkg_region_supplied = file_test(bkg_region_fn)
      
    ;; Initialize vars used in search below.
    obs_stats = headfits(obs_stats_fn, ERRMSG=error)
    if keyword_set(error) then begin
      print, error
      message, 'ERROR reading '+obs_stats_fn
    endif
    xpos_catalog = psb_xpar( obs_stats, 'X_CAT')
    ypos_catalog = psb_xpar( obs_stats, 'Y_CAT')
    SRC_CNTS     = psb_xpar( obs_stats, 'SRC_CNTS')
    is_diffuse   = psb_xpar( obs_stats, 'DIFFUSE')
    
    ; If obs.parameters file exists, then it's contents over-ride obs.stats.
    obs_parameters    = headfits(obs_parameters_fn, ERRMSG=error)
    
    if (~keyword_set(error)) then begin
      ; Copy keywords in parameter file into obs_stats header.
      kywd = 'S_FILTER' 
      val = psb_xpar( obs_parameters, kywd, COMMENT=comment, COUNT=count)
      if (count EQ 1) then psb_xaddpar, obs_stats, kywd, val, comment
    endif 
    
    
    ; Look for any existing declaration that defines a filter on the STATUS column.
    ; If the keyword is missing (the filtering decision has not yet been made) then the default value is:
    ;   * no STATUS filter for diffuse sources (filtering is observer's responsibility)
    ;   * STATUS=0 filter for point sources, so that weak sources will not be pruned before we realize that they merit aggressive background cleaning.
    S_FILTER = strtrim(psb_xpar( obs_stats, 'S_FILTER', COUNT=count),2)
    if (count EQ 0) then S_FILTER = is_diffuse ? '' : 'STATUS=0'
    
    if (~bkg_region_supplied) then begin
      ;; Make a starting guess for the (unmasked) geometric area of the background region.
      ;; If a previous extraction was done use that saved radius, otherwise use the bkg_geometric_area
      ;; value from the previous source.
      ;; Adjust the guess upward if the specified background area ratio suggests that.
      bkg_radius = psb_xpar( obs_stats, 'BKG_RAD', COUNT=count)
      if (count EQ 1) && (bkg_radius GT 0) then begin
        bkg_geometric_area = !PI *              (bkg_radius)^2
      endif else begin
        bkg_geometric_area = !PI * (psb_xpar( obs_stats, 'SRC_RAD'))^2 * BKSCL_GL
      endelse 
    endif ;NOT bkg_region_supplied
      
    BKSCL_HI_vote = !VALUES.F_INFINITY
    done        = 0
    loop_count  = 0
    upper_bound =  !VALUES.F_INFINITY
    lower_bound = -!VALUES.F_INFINITY
    repeat begin
      loop_count = loop_count + 1
        
      ;; ------------------------------------------------------------------------
      if bkg_region_supplied then begin
        ;; If we find bkg_region_fn already exists then we use it.
        ;; This gives the observer a hook for supplying non-circular background regions.
        ae_ds9_to_ciao_regionfile, bkg_region_fn, temp_region_fn
          
        bkg_radius = 0
        bkg_region = string(temp_region_fn, F="(%'region(%s)')")
        done       = 1
  
      endif else begin
        ;; Otherwise use a circular background region.
        if ~finite(bkg_geometric_area) || (bkg_geometric_area LE 0) then begin
          if is_diffuse then $
            print, bkg_region_fn, F='(%"ERROR: expected you to supply background region (%s) for this diffuse source.")' $
          else $
            print, 'ERROR: variable "bkg_geometric_area" in acis_extract is not positive!'
          GOTO, FAILURE
        endif
        bkg_radius = sqrt(bkg_geometric_area/!PI)
        bkg_region = strcompress( string(xpos_catalog, ypos_catalog, bkg_radius, $
                                  F='(%"circle(%0.1f,%0.1f,%0.1f)")'), /REMOVE_ALL )
      endelse ;construct circular region
    
      ;; ------------------------------------------------------------------------
      ;; Find out how many in-band events are in that region.
      cmd = string(temp_bkgimg_fn, bkg_region, $
                   temp_image_fn, F="(%'dmcopy ""%s[sky=%s][opt update=no]"" %s')")
      run_command, cmd
      
      bkg_image = readfits(temp_image_fn, /SILENT)
      bkg_counts = total(bkg_image)

      ;; ------------------------------------------------------------------------
      ;; Evaluate the stopping criteria for defining the background region.
      ; 1. The region's background exposure ratio (bkg_exposurearea/src_exposurearea) MUST end up in the range 
      ;    [BKSCL_LO,BKSCL_HI] (obtained from source.stats).  (For coding convenience, a value just above BKSCL_HI is acceptible.)
      ;
      ; 2. If the region's background exposure ratio reaches the target value of BKSCL_GL and the region 
      ;    contains at least MIN_NUM_CTS in-band counts then the search is stopped.
      
      ; Integrate the emap over the region.
      ; Note that the region may extend beyond the boundaries of the emap
      ; and the emap is full of "holes" made by source masks.
      cmd = string(emap_filename, bkg_region, bkg_emap_fn, F="(%'dmcopy ""%s[sky=%s]"" %s')")
      run_command, cmd
    
      emap = readfits(bkg_emap_fn, emap_header, /SILENT)
    
      pixel_size = psb_xpar( emap_header, 'CDELT1P')
    
      ; Ignore any negative emap pixels.
      bkg_exposurearea = (pixel_size^2) * total(emap>0, /DOUBLE) ; Units are skypix**2 s cm**2 count /photon.

      BACKSCAL         = bkg_exposurearea / src_exposurearea
      
      ; Record the smallest bkg_normalization value with acceptible bkg_counts
      if (bkg_counts GE min_num_cts) then BKSCL_HI_vote <= BACKSCAL
    
      if bkg_region_supplied then begin
        num_emap_pixels[ii] = total(emap GT 0)
        done = 1
      endif else begin
        if            (BACKSCAL LT BKSCL_LO) then begin
          ; Region is too small; raise lower bound.
          lower_bound = max([lower_bound,bkg_geometric_area])
        endif else if (BACKSCAL GT BKSCL_HI) then begin
          ; Region is too large; lower upper bound.
          upper_bound = min([upper_bound,bkg_geometric_area])
        endif else begin
        
          ; Region's size falls in acceptible range.  Now aim for target size and minimum number of counts.
          if (bkg_counts LT min_num_cts) || (BACKSCAL LT BKSCL_GL) then begin
            ; Region is too small; raise lower bound.
            if ((upper_bound-lower_bound)/upper_bound LT 0.05) then begin
              done = 1
              print, "Background region area is smaller than, but within 5% of, the asymptotic value."
            endif else begin            
              lower_bound = max([lower_bound,bkg_geometric_area])
            endelse
          endif else begin
            ; If we make it here, then BACKSCAL is in its acceptible range, and we have enough counts.
            ; We arbitrarily decide to stop the search for the optimal region if the search range has narrowed.
            if ((upper_bound-lower_bound)/upper_bound LT 0.05) then begin
              done = 1
              print, "Background region area is larger than, but within 5% of, the asymptotic value."
            endif else begin            
            ; Region is too large; lower upper bound.
              upper_bound = min([upper_bound,bkg_geometric_area])
            endelse
          endelse
        endelse
        
        
        ; Adjust the region size in preparation for the next iteration.
        if (NOT finite(upper_bound)) then begin
            ; If no upper bound is yet known, then take a big step up.
            bkg_geometric_area = bkg_geometric_area * 2.
        endif else if (NOT finite(lower_bound)) then begin
            ; If no lower bound is yet known, then take a big step down.
            bkg_geometric_area = bkg_geometric_area / 2.
        endif else begin
            ; Bisect the search interval.
            bkg_geometric_area = 0.5 * (upper_bound + lower_bound)
        endelse
        ;help, upper_bound, bkg_geometric_area, lower_bound
        
        ; Enforce a hard upper limit on the bkg region size (4' radius) to prevent an infinite search.
        if (bkg_geometric_area GT (!PI * (2*60.0*4)^2)) then begin
          print, 'WARNING, background region search aborted when region grew to 4 arcmin radius!!!'
          done = 1
        endif
        print, sqrt([lower_bound,upper_bound]/!PI), F='(%"Radius range = %0.1f:%0.1f")'
      endelse ; ~bkg_region_supplied


      ; Stop if we've taken an excessive number of steps.
      if (loop_count GT 50) then begin
        print, 'WARNING, background region search aborted after 50 iterations!!!'
        done = 1
      endif
    endrep until (done)
  
    ;; Summarize the quality of the background region.
    if ~finite(BACKSCAL) then begin
      print, 'ERROR: BACKSCAL is not finite.'
      GOTO, FAILURE
    endif

    
    ; Compute the significance of the observed SRC_CNTS as a disproof of the "null hypothesis" which is 
    ; that there is no source, i.e. that all the observed counts are background.  
    ; We use equation A7 from Weisskopf 2006 (astro-ph/0609585):

    ; As of IDL v8.5.1, binomial() has significant flaws and frequently produces very wrong results.
    ; See email to Harris Inc. on Oct 22, 2016.
    ; It also behaves very badly if GAUSSIAN=0 is (innocently) supplied (IDL bug report 69655).
    ; Prior to v8.5.1, binomial() could not handle inputs larger than 32767 (IDL bug report 69442).
    ; We use instead a simple and reliable algorithm recommended by Numerical Recipes (Chapter 6.4).

    PROB_NO_SOURCE_binomial = binomial_nr(SRC_CNTS, $
                                          SRC_CNTS + bkg_counts, $
                                          1D/(1D + BACKSCAL) ) > 0

    if ~finite(PROB_NO_SOURCE_binomial) then message, 'ERROR: PROB_NO_SOURCE is not finite.'
    
;   PROB_NO_SOURCE_poisson  = (1 - poisson_distribution(bkg_counts/BACKSCAL, SRC_CNTS - 1)) > 0
  
    src_cnts_error        = (1 + sqrt(SRC_CNTS   + 0.75))
    bkg_subtraction_error = (1 + sqrt(bkg_counts + 0.75)) / BACKSCAL

    print, bkg_counts, BACKSCAL, loop_count,               F='(%"\nBackground region: %d in-band background counts; normalization =%6.1f; %d iterations.")' 
    
    print, SRC_CNTS, src_cnts_error,                       F='(%"SRC_CNTS                              =%8.1f (+-%5.1f)")' 
  
    print, bkg_counts / BACKSCAL, bkg_subtraction_error,   F='(%"bkg counts in aperture                =%8.1f (+-%5.1f)")' 
  
    print, PROB_NO_SOURCE_binomial,                        F='(%"PROB_NO_SOURCE                        = %8.2g")'
;   print, PROB_NO_SOURCE_binomial, PROB_NO_SOURCE_poisson, F='(%"PROB_NO_SOURCE, actual and asymptotic:%8.2g %8.2g")'
    
      
    ;; ------------------------------------------------------------------------
    cmd = string(emap_filename, bkg_region, bkg_emap_fn, F="(%'dmcopy ""%s[sky=%s]"" %s')")
    run_command, cmd
    
    ; Remove a region file produced by the ae_better_backgrounds tool to avoid confusion about how
    ; the background spectrum was created..
    file_delete, bkg_pixels_region_fn, /ALLOW_NONEXISTENT
                
    ;; The set of emap pixels above define the actual background region to apply to the event list.

    ; We work-around the dmimgpick bug reported in HelpDesk Ticket #020605, by adding a border of zeros around the background.emap image.
    ;; We make dmcopy write to local disk, to avoid the CIAO slowdown when destination is an NFS volume.
    temp_header = headfits(bkg_emap_fn)
    
    cmd = string( bkg_emap_fn, 1+psb_xpar( temp_header, 'NAXIS1'),$
                               1+psb_xpar( temp_header, 'NAXIS2'), temp_image_fn,$
                 F="(%'dmcopy ""%s[bin #1=0:%d,#2=0:%d]"" %s')")
    run_command, cmd
    file_move, /OVERWRITE, temp_image_fn, bkg_emap_fn
    
    print, 'WARNING: When the dmimgpick bug reported in HelpDesk Ticket #020605 is resolved, code near line 9290 can be simplified.'
    
    
    ; Apply any STATUS filter declared for this source, to be consistent with AE's extraction.
    ; Testing shows that it is more efficient to perform STATUS filter in dmcopy, after the filter on the emap column, than in dmimgpick.
    cmd1 = string(obsdata_filename, bkg_emap_fn, temp_events_fn, $
                  F="(%'dmimgpick ""%s[cols time,ccd_id,sky,pi,energy,status]"" %s %s method=closest')")

    ; BELOW WE REQUIRE EMAP VALUE TO BE >1, INSTEAD OF >0, BECAUSE CIAO 3.0.1 HAS A BUG THAT CAUSES ZERO VALUES TO PASS THE >0 TEST!
    cmd2 = string(temp_events_fn, $
                  keyword_set(S_FILTER) ? ','+S_FILTER : '', $
                  bkg_events_fn, F="(%'dmcopy ""%s[#7>1%s][cols time,ccd_id,sky,pi,energy,status]"" %s')")
    run_command, [cmd1,cmd2]


    ;; ------------------------------------------------------------------------
    ;; Extract background spectrum.
    ;; NOTE: if we ever implement a time filter on the background data then we must 
    ;; reduce bkg_exposurearea below by the ratio 
    ;; (EXPOSURE from bkg_spectrum_fn)/(EXPOSURE from bkg_events_fn) to account for the lost exposure.
    ;; Such time filtering would mess up MIN_NUM_CTS requirement!
    
    cmd = string(bkg_events_fn, DETCHANS, bkg_spectrum_fn, $
                 F="(%'dmextract ""%s[bin pi=1:%d:1]"" %s opt=pha1 error=gaussian')")
    run_command, cmd


    ;; ------------------------------------------------------------------------
    ;; The AE convention is that the BACKSCAL keywords, derived from
    ;; integrals of the exposure map, are used to represent geometric
    ;; area, effective area, and integration time differences between the
    ;; source and background regions.  
    ;; The EXPOSURE keywords are NOT used for background scaling.  We set EXPOSURE=0
    ;; in the background spectrum as a flag to signify the AE convention is being used.
    
    comment = 'EXPOSURE not used for bkg scaling'
    
    openw, unit, temp_text_fn, /GET_LUN
    printf, unit, comment, comment, comment, bkg_exposurearea*tweak_backscal, '(skypixel**2 s cm**2 count /photon); '+comment, S_FILTER, $
                  F='(%"#add\nONTIME = 0.0 / [s] %s\nLIVETIME = 0.0 / [s] %s\nEXPOSURE = 0.0 / [s] %s\nBACKSCAL = %f / %s\nS_FILTER=\"...\"\nS_FILTER=\"%s\"")'
                  ; S_FILTER is assigned twice above to work around a dmhedit bug (in CIAO 4.3) that converts a whitespace value to the integer zero.
    free_lun, unit
    
    cmd = string(bkg_spectrum_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
    run_command, cmd, /QUIET


    ;; ------------------------------------------------------------------------
    ;; Save information for summary plots.
    if (bkg_radius EQ 0) then begin
      psb_xaddpar, obs_stats, 'BKG_RAD',  0,                            'observer supplied background'
    endif else begin
      psb_xaddpar, obs_stats, 'BKG_RAD',  bkg_radius,                   '[skypixel] background extraction radius'
    endelse

    psb_xaddpar, obs_stats, 'BKG_CNTS', bkg_counts,                     string(energy_range, F="(%'[count] %0.2f:%0.2f keV, in background region')") 
    psb_xaddpar, obs_stats, 'BACKSCAL', BACKSCAL,                      'scaling of BKG_CNTS to src aperture' 
    psb_xaddpar, obs_stats, 'BKSCL_LO', BKSCL_LO,                      'smallest BACKSCAL allowed'
    psb_xaddpar, obs_stats, 'BKSCL_GL', BKSCL_GL,                      'target   BACKSCAL'
    psb_xaddpar, obs_stats, 'BKSCL_HI', BKSCL_HI,                      'largest  BACKSCAL allowed'
  
    ; This extraction gets to cast votes, used by the adjustment algorithm, for an upper limit on BKSCL_LO and a lower limit on BKSCL_HI.
    
    ; The masked background algorithm we've used above does not have any requirements on BKSCL_LO, so don't cast a vote (0 is no-vote flag).
    BKSCL_LO_vote = 0
      
    ; If the search did NOT identify any scaling that meets the MIN_NUM_CTS goal, then cast a vote to raise BKSCL_HI.
    if ~finite(BKSCL_HI_vote)               then BKSCL_HI_vote = 2 * BKSCL_HI
      
    psb_xaddpar, obs_stats, 'VOTE_LO', BKSCL_LO_vote, 'vote for upper limit on BKSCL_LO' 
    psb_xaddpar, obs_stats, 'VOTE_HI', BKSCL_LO_vote, 'vote for lower limit on BKSCL_HI' 
    
    ; Recall that the value in bkg_exposurearea is the integral of the exposure map over the bkg region.
    ; Recall that this exposure map integral has units of (skypixel**2 s cm**2 count /photon).
    psb_xaddpar, obs_stats, 'BACKGRND', bkg_counts / bkg_exposurearea , string(energy_range, F="(%'[photon /cm**2 /s /skypixel**2] background SB, %0.2f:%0.2f keV')")
    psb_xaddpar, obs_stats, 'CREATOR', creator_string
    writefits, obs_stats_fn, 0, obs_stats

  endfor ; ii
  
  ;; Report the smallest background regions.
  ind = where(num_emap_pixels NE -1, count)
  if (count GT 0) then begin
    print, F='(%"\n\rThe following sources used observer-supplied background regions.  \n\rUse the ds9 calls below to review whether the pixelization of the regions is acceptable.\n\rSOURCE NAME           # UNMASKED PIXELS IN BKG REGION")'
    forprint, sourcename[ind], num_emap_pixels[ind], $
              'ds9 ' + sourcename[ind] + '/' + obsname + '/' + extraction_subdir[ind] + bkg_emap_basename, $
              F='(A20,2x,I7,4x,A)'
  endif
  
  count = total(source_not_observed, /INT)
  if (count GT 0) then print, count, F="(%'\nWARNING!  EXTRACT_BACKGROUNDS skipped for %d sources not observed.')"
  
endif ;keyword_set(extract_backgrounds)





;; =============================================================================
if keyword_set(merge_observations) then begin

  if keyword_set(merge_for_pb) then begin
    print, F='(%"\n\nWARNING! \nWARNING!  The MERGE_FOR_PB increases sensitivity to variable sources at the cost of an increased false detection rate arising from the additional number of data sets that are searched. \nPausing for 60 seconds ...")'
    wait, 60
  endif

;debug_pb = fltarr(num_sources)

  ;; =============================================================================
  ;; Computing composite source & background spectra is a little tricky.  See the theory section of the manual! 
  
  cd, CURRENT=cwd  &  cwd=cwd+'/'
  f_nan      = !VALUES.F_NAN

  if ~keyword_set(skip_spectra_p) then begin
    print, 'Photometry computed on the following energy ranges:'
    forprint, eband_lo, eband_hi, F='(F4.1," -",F4.1," keV")', /SILENT
  endif

  if keyword_set(generic_rmf_fn) && ~file_test(generic_rmf_fn) then begin
    print, generic_rmf_fn, F='(%"\nERROR: The GENERIC_RMF_FN specified (%s) does not exist; aborting!")'
    GOTO, FAILURE
  endif

  ; Modify a copy of CIAO's dmmerge_header_lookup.txt file, which controls how dmmerge combines keywords.
  ;  * Keywords derived from the GTI tables ("calcGTI") are "skipped" because they used the *first* GTI table, which
  ;    may not represent the data when data from ACIS-I and ACIS-S aimpoints are combined.
  ;  * Other keywords are "skipped" because they cannote be sensibly combined.

  ;awk '/calcGTI|RA_|DEC_|ROLL_|BTIM|DS_IDENT|OBJECT|OBS_ID|SEQ_NUM|OBI_NUM|TITLE|DATAMODE|OBSERVER/{print $1 "  SKIP"; next}; {print $0}'

  cmd = 'awk ''/calcGTI|RA_|DEC_|ROLL_|BTIM|DS_IDENT|OBJECT|OBS_ID|SEQ_NUM|OBI_NUM|TITLE|DATAMODE|OBSERVER/{print $1 "  SKIP"; next}; {print $0}'' ' +getenv('ASCDS_INSTALL')+'/data/dmmerge_header_lookup.txt > '+dmmerge_header_lookup_fn
  
  run_command, /QUIET, cmd




  obsname_list = ''

  for ii = 0L, num_sources-1 do begin
    ; WARNING!  The SOURCE_NOT_OBSERVED input should NOT be used to ignore sources in a MERGE operation!
    ; The merge the caller has requested could already exist (e.g. from when the source had a different position).
    ; To avoid persistent stale merge products, the code below must be run for every source.
    ; If the ObsIDs specified for this merge (or all ObsIDs) did not extract this source, then the code below
    ; needs to record that fact (e.g. setting NUM_OBS=0 in source.stats, and removing any existing 
    ; stale merge data products) so that collations do not report stale information.

    basedir   = sourcename[ii] + '/' 
    sourcedir = basedir + merge_subdir[ii]
    named2generic_extraction_path = keyword_set(merge_subdir[ii]) ?  '../' : ''

    merged_list_fn       = sourcedir + 'ObsIDs_merged.fits'
    merged_env_events_fn = sourcedir + env_events_basename
    merged_src_events_fn = sourcedir + src_events_basename
    merged_region_fn     = sourcedir + src_region_basename
         
    composite_img_fn     = sourcedir + env_image_basename
    composite_psf_fn     = sourcedir + psf_basename
    rebinned_composite_psf_fn  = sourcedir + 'recon.psf'

    event_reg_fn         = sourcedir + evt_region_basename

    merged_src_spectrum_basename = sourcename[ii] + '.pi'
    merged_bkg_spectrum_basename = sourcename[ii] + '_bkg.pi'
    merged_arf_basename          = sourcename[ii] + '.arf'
    merged_rmf_basename          = sourcename[ii] + '.rmf'
    merged_sequenced_lc_basename = sourcename[ii] + '.sequenced_lc.ps'
    merged_stacked_lc_basename   = sourcename[ii] + '.stacked_lc.ps'

    unnamed_src_stats_fn   = sourcename[ii] + '/' + src_stats_basename
            src_stats_fn   = sourcedir            + src_stats_basename
    photometry_fn          = sourcedir + src_photometry_basename
    merged_src_spectrum_fn = sourcedir + merged_src_spectrum_basename
    merged_bkg_spectrum_fn = sourcedir + merged_bkg_spectrum_basename
    merged_arf_fn          = sourcedir + merged_arf_basename
    merged_rmf_fn          = sourcedir + merged_rmf_basename
    merged_lc_smooth_fn    = sourcedir + lc_smooth_basename
    merged_stacked_lc_fn   = sourcedir + merged_stacked_lc_basename
    merged_sequenced_lc_fn = sourcedir + merged_sequenced_lc_basename

   ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(basedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    ;; ------------------------------------------------------------------------
    ;; Read the fundamental source properties from the unnamed source.stats file.
    unnamed_src_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    
    if (NOT keyword_set(error)) then begin
      ra  = psb_xpar( unnamed_src_stats, 'RA')
      dec = psb_xpar( unnamed_src_stats, 'DEC')
    endif else begin
      print, error
      message, 'ERROR reading '+unnamed_src_stats_fn
    endelse
    
    ; These top-level properties form the basis of the named source.stats file.
    src_stats    = unnamed_src_stats
 
    is_diffuse = psb_xpar( src_stats, 'DIFFUSE')
 
    
    ;; ------------------------------------------------------------------------
    ;; Figure out how far this source has been processed by looking for the data
    ;; products produced by various stages.
    ;; We can NOT search for "neighborhood.evt" because an earlier run of this 
    ;; stage could have made a file <sourcename>/<merge_name>/neighborhood.evt 
    ;; which would be misinterpreted here as an observation! 
    ;; We must instead search for "obs.stats" which appears only in observation
    ;; directories, and then see which of those observations has a "neighborhood.evt".
    
    ; If the observer has specified a list of observations then look for them, otherwise look for any.
    pattern_base = sourcename[ii] + '/' + (keyword_set(obsname) ? obsname : '*') + '/'
    
    if keyword_set(extraction_subdir[ii]) then begin
      ; When EXTRACTION_NAME is supplied the observer desires to merge those extractions, ignoring any obsids that
      ; do not have an extraction of that name.
      ; The directory where the merged products are stored is controlled independently via MERGE_NAME.
      obs_stats_fn   = file_search( pattern_base + extraction_subdir[ii] + obs_stats_basename, COUNT=num_obs )
    endif else begin
      ; Look for unnamed extractions.
      obs_stats_fn   = file_search( pattern_base +                         obs_stats_basename, COUNT=num_obs )
    endelse    
        
    psb_xaddpar, src_stats, 'NUM_OBS',  num_obs, 'number of observations extracted'

    src_stats_exists = file_test(src_stats_fn)
    if (num_obs EQ 0) then begin
      ; If the (named) stats file already exists then we must update it.
      if src_stats_exists then GOTO, MERGE_IS_EMPTY $
      else continue
    endif else begin
      ; We are going to process this source, so create the named merge dir and start the messages
      if ~src_stats_exists then file_mkdir, sourcedir
      
      print, F='(%"\n===================================================================")'
      print, sourcename[ii], strtrim(psb_xpar( unnamed_src_stats,'LABEL'),2), F='(%"Source: %s (%s)")'      
    endelse

    ;; ------------------------------------------------------------------------
    ; Remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    run_command, /QUIET, /HEASOFT, ['pset addrmf clobber=yes', 'pset addarf chatter=0 clobber=yes']
    run_command, /QUIET, ['pset dmcopy clobber=yes', 'pset reproject_events clobber=yes', 'pset dmmerge clobber=yes lookupTab='+dmmerge_header_lookup_fn]

    
    ;; ------------------------------------------------------------------------
    ;; Make pathnames to all the files we use, e.g. PSFs, neighborhoods, source spectra, bkg spectra?
    obs_data_template =  { $
                         obsname            :'' , $
                         obs_stats_fn       :'' , $
                         obs_dir            :'' , $
                         unnamed_obs_dir    :'' , $
                         psf_fn             :'' , $
                         env_events_fn      :'' , $
                         psf_frac_fn        :'' , $
                         src_events_fn      :'' , $
                         src_spectrum_fn    :'' , $
                         rmf_fn             :'' , $
                         arf_fn             :'' , $
                         bkg_spectrum_fn1   :'' , $
                         bkg_spectrum_fn2   :'' , $
                         lc_smooth_fn       :'' , $
                         TSTART             :!VALUES.F_NAN, $
                         off_angle          :0.0, $
                         emap_avg           :0.0, $
                         s_filter           :'' , $
                         n_filter           :'' , $
                         src_radius         :0.0, $
                         src_area           :0.0, $
                         mask_radius        :0.0, $
                         fracspec           :0.0, $
                         psf_fraction       :0.0, $ 
                         emap_med           :0.0, $
                         emap_max           :0.0, $
                         reg_edit           :0B , $
                         overlap            :0.0, $
                         prob_ks            :0.0, $
                         bkg_counts         :0.0, $
                         src_counts         :0.0, $
                         xpos_catalog       :0.0, $
                         ypos_catalog       :0.0, $
                         time_on_detector   :0.0, $
                         RATE_3x3           :0.0, $
                         num_in_warning     :0L   $
                         }
    obs_data = replicate(obs_data_template, num_obs)
                       
    obs_data.obs_dir          = file_dirname(obs_stats_fn, /MARK_DIRECTORY)
    obs_data.unnamed_obs_dir  =       strmid(obs_stats_fn, 0, 1+reform(strpos(obs_stats_fn, '/', 1+strlen(sourcename[ii])), 1,num_obs))
    ; In the line below, delete the vector obs_stats_fn so it won't be mistakenly used after obs_data is pruned.
    obs_data.obs_stats_fn     =    temporary(obs_stats_fn)
    
    obs_data.psf_fn           = obs_data.unnamed_obs_dir + psf_basename
    obs_data.env_events_fn    =         obs_data.obs_dir + env_events_basename
    obs_data.psf_frac_fn      =         obs_data.obs_dir + obs_frac_basename
    obs_data.src_events_fn    =         obs_data.obs_dir + src_events_basename
    obs_data.src_spectrum_fn  =         obs_data.obs_dir + src_spectrum_basename
    obs_data.rmf_fn           =         obs_data.obs_dir + rmf_basename
    obs_data.arf_fn           =         obs_data.obs_dir + arf_basename
    obs_data.bkg_spectrum_fn1 =         obs_data.obs_dir + bkg_spectrum_basename
    obs_data.bkg_spectrum_fn2 = obs_data.unnamed_obs_dir + bkg_spectrum_basename
    obs_data.lc_smooth_fn     =         obs_data.obs_dir + lc_smooth_basename

    if keyword_set(generic_rmf_fn) then obs_data.rmf_fn = generic_rmf_fn

    ; We choose to perform merging operations only where no observation is missing files.
    psf_available          = (num_obs EQ total( file_test(obs_data.psf_fn)))
    neighborhood_available = (num_obs EQ total( file_test(obs_data.env_events_fn)))
    aperture_available     = (num_obs EQ total( file_test(obs_data.src_events_fn)))
    spectra_available      = (num_obs EQ total( file_test(obs_data.src_spectrum_fn)   AND $
                                                file_test(obs_data.rmf_fn)            AND $
                                                file_test(obs_data.arf_fn)))
    timing_available       = (num_obs EQ total(file_test(obs_data.lc_smooth_fn)))
    
    skip_psf          = keyword_set(skip_psf_p)          || ~psf_available 
    skip_neighborhood = keyword_set(skip_neighborhood_p) || ~neighborhood_available
    skip_aperture     = keyword_set(skip_aperture_p)     || ~aperture_available
    skip_spectra      = keyword_set(skip_spectra_p)      || ~spectra_available    
    skip_timing       = keyword_set(skip_timing_p)       || ~timing_available 

    
    ; NFS is not entirely reliable. Look again for any data products that appear to be missing.
    if (skip_spectra) && ~keyword_set(skip_spectra_p) then begin
      wait, 0.1
      spectra_available      = (num_obs EQ total( file_test(obs_data.src_spectrum_fn)   AND $
                                                  file_test(obs_data.rmf_fn)            AND $
                                                  file_test(obs_data.arf_fn)))
      skip_spectra =  ~spectra_available
      if ~skip_spectra then print, "SECOND attempt to find spectrum, ARF, RMF files was successful."
    endif

;    help, psf_available,neighborhood_available,aperture_available,spectra_available,timing_available
    
    ; WHY? WHY? WHY? am I using zeros to represent "no position estimate"?????
    ra_data      = 0
    dec_data     = 0
    
    ;; ------------------------------------------------------------------------
    ;; READ SPECTRA AND SOURCE PROPERTY KEYWORDS FROM EACH EXTRACTION
    ;; ------------------------------------------------------------------------    
    comment_psf_fraction = ''

    for jj = 0, num_obs-1 do begin
      row = obs_data[jj]  
     
      ;; Read FITS keywords in obs.stats.
      obs_stats = headfits(row.obs_stats_fn)
      row.obsname      =           strtrim( psb_xpar( obs_stats,'OBSNAME'), 2 )
      row.bkg_counts   =                    psb_xpar( obs_stats,'BKG_CNTS')
      row.src_counts   =                    psb_xpar( obs_stats,'SRC_CNTS')
      
      row.xpos_catalog =                    psb_xpar( obs_stats,'X_CAT')
      row.ypos_catalog =                    psb_xpar( obs_stats,'Y_CAT')
      if ~finite(row.xpos_catalog) || ~finite(row.ypos_catalog) then begin
        row.xpos_catalog =                  psb_xpar( obs_stats,'X_DATA')
        row.ypos_catalog =                  psb_xpar( obs_stats,'Y_DATA')
      endif
      
      row.off_angle    =                    psb_xpar( obs_stats,'THETA')
      row.emap_avg     =                    psb_xpar( obs_stats,'EMAP_AVG')
      
      ; If S_FILTER keyword is missing, we want '' not the number 0.
      temp             =                    psb_xpar( obs_stats,'S_FILTER', COUNT=count)
      s_filter_exists  = (count EQ 1)
      row.s_filter     = s_filter_exists ? strtrim(temp, 2) : ''
      
      ; If N_FILTER keyword is missing, we want '' not the number 0.
      temp             =                    psb_xpar( obs_stats,'N_FILTER', COUNT=count)
      n_filter_exists  = (count EQ 1)
      row.n_filter     = n_filter_exists ? strtrim(temp, 2) : ''
      
      row.src_radius   =                    psb_xpar( obs_stats,'SRC_RAD')
      row.src_area     =                    psb_xpar( obs_stats,'SRC_AREA')
      row.mask_radius  =                    psb_xpar( obs_stats,'MSK_RAD')
      row.fracspec     =                    psb_xpar( obs_stats,'FRACSPEC')
      row.psf_fraction =                    psb_xpar( obs_stats,'PSF_FRAC',COMMENT=comment_psf_fraction)
      row.emap_med     =                    psb_xpar( obs_stats,'EMAP_MED')
      row.emap_max     =                    psb_xpar( obs_stats,'EMAP_MAX')
      row.reg_edit     =                    psb_xpar( obs_stats,'REG_EDIT')
      row.overlap      =                    psb_xpar( obs_stats,'OVERLAP')
      row.prob_ks      =                    psb_xpar( obs_stats,'PROB_KS')
      row.time_on_detector =                psb_xpar( obs_stats,'EXPOSURE')*psb_xpar( obs_stats,'FRACEXPO')
      row.RATE_3x3     =                    psb_xpar( obs_stats,'RATE_3x3')
      row.num_in_warning   =                psb_xpar( obs_stats,'SRC_CNTS')*psb_xpar( obs_stats,'WARNFRAC')

      
      ;; Read information from spectral files and PSF fraction files.    
      if spectra_available then begin
        src_spectrum = mrdfits(row.src_spectrum_fn, 1, src_header, /SILENT, STATUS=status)
        if (status NE 0) then message, 'ERROR reading ' + row.src_spectrum_fn
        
        if (jj EQ 0) then begin
          ;; Store the data we've collected so far for the current "row" (extraction).
          obs_data[jj] = temporary(row)
      
          ; Expand the tags in the obs_data structure.
          src_channels = src_spectrum.CHANNEL
          
          ; Add tags to the template to hold spectra.
          obs_data_template = create_struct(obs_data_template, $
                                            'src_observed_counts'        , fltarr(n_elements(src_channels)), $
                                            'bkg_observed_counts'        , fltarr(n_elements(src_channels)), $
                                            'bkg_spectrum_fn', '', $ ; Empty string when no bkg spectrum available.
                                            'bkg_arf_fn'     , '', $
                                            'obsid'          , '', $
                                            'src_backscal'   , f_nan, $
                                            'bkg_backscal'   , f_nan, $
                                            'src_exposure'   , f_nan, $
                                            'bkg_exposure'   , f_nan, $
                                            
                                            'psf_x_var'      , f_nan, $
                                            'psf_y_var'      , f_nan, $
                                            'bkg_x_var'      , 0.0, $
                                            'bkg_y_var'      , 0.0 $
                                            )
          temp     = temporary(obs_data)
          obs_data = replicate(obs_data_template, num_obs)
          copy_struct, temp, obs_data

          ; Recreate the "row" structure we're populating.
          row = obs_data[jj]  
        endif ; jj EQ 0

        ; Replace any S_FILTER property read from obs.stats with what is stored in spectrum header.
        ; If S_FILTER keyword is missing, we want '' not the number 0.
        temp             =                    psb_xpar( src_header,'S_FILTER', COUNT=count)
        s_filter_exists  = (count EQ 1)
        row.s_filter     = s_filter_exists ? strtrim(temp, 2) : ''

        if (total(src_channels - src_spectrum.CHANNEL) NE 0) then $
            message, 'ERROR: spectra have different channel sets!'
        
        ; Since not all ObsIDs may have the same number of channels, we use the + operator instead of "=" to store columns from the spectrum into a "row" structure 
        row.src_observed_counts  = 0
        row.src_observed_counts += src_spectrum.COUNTS
        row.obsid        = strtrim(psb_xpar( src_header, 'OBS_ID'),2)
        row.src_backscal         = psb_xpar( src_header, 'BACKSCAL')
        ; "src_exposure" field MUST be exposure TIME, not TIME multiplied by Effective Area.
        ; See note below at "exposure_fraction" calculation.
        row.src_exposure         = psb_xpar( src_header, 'EXPOSURE') 
        
        hdr_vals = [row.src_backscal,row.src_exposure]
        dum = where((hdr_vals EQ 0) OR (~finite(hdr_vals)), count)
        if (count GT 0) then message, 'ERROR: BACKSCAL or EXPOSURE keyword is not valid in '+row.src_spectrum_fn

        
        ; Process the BKG spectrum, if available
        ; Use a bkg spectrum (and optional ARF) from the named extraction dir, if available,
        ; or use a bkg spectrum from the generic extraction dir.
        if            file_test(row.bkg_spectrum_fn1) then begin
          row.bkg_spectrum_fn     = row.bkg_spectrum_fn1
          row.bkg_arf_fn          = row.obs_dir         + bkg_arf_basename
        endif else if file_test(row.bkg_spectrum_fn2) then begin
          row.bkg_spectrum_fn     = row.bkg_spectrum_fn2
          row.bkg_arf_fn          = row.unnamed_obs_dir + bkg_arf_basename
        endif else begin
          row.bkg_spectrum_fn     = ''  ; Empty string when no bkg spectrum available.
          row.bkg_arf_fn          = ''  ; Empty string when no bkg spectrum available.
        endelse
  
        if keyword_set(row.bkg_spectrum_fn) then begin
          bkg_spectrum = mrdfits(row.bkg_spectrum_fn, 1, bkg_header, /SILENT, STATUS=status)
          if (status NE 0) then message, 'ERROR reading ' + row.bkg_spectrum_fn
          
          ; Compare the STATUS filters applied (by AE at least) to the src and bkg spectra.
          ; If S_FILTER keyword is missing, we want '' not the number 0.
          temp                     = psb_xpar( bkg_header,'S_FILTER', COUNT=count)
          s_filter_bkg_exists      = (count EQ 1)
          s_filter_bkg             = s_filter_bkg_exists ? strtrim(temp, 2) : ''

        
          if s_filter_exists && s_filter_bkg_exists && (s_filter_bkg EQ row.s_filter) then begin
            ; The same S_FILTER value in the src and bkg headers is evidence of consistent filtering.
            
          endif else if ~s_filter_bkg_exists then begin
            ; When S_FILTER is missing, we cannot make assumptions about what STATUS filter was applied (by the observer).
            print, row.bkg_spectrum_fn, F='(%"WARNING: %s is obsolete; it should be rebuilt so that AE can confirm that the same STATUS filter has been applied to src and bkg spectra.")'

          endif else if ~s_filter_exists then begin
            ; When S_FILTER is missing, we cannot make assumptions about what STATUS filter was applied (by the observer).
            print, row.src_spectrum_fn, F='(%"WARNING: %s is obsolete; it should be rebuilt so that AE can confirm that the same STATUS filter has been applied to src and bkg spectra.")'
            
          endif else begin
            ; Both spectra contain S_FILTER keyword, but they do not match.  This is a failure!
            print, row.s_filter, s_filter_bkg, row.src_spectrum_fn, row.bkg_spectrum_fn, F="(%'ERROR: Different STATUS filters (\'%s\' and \'%s\') have been applied to %s and %s.')"
            GOTO, FAILURE
          endelse
          
          
          
          ; Verify src and bkg spectra have the same set of channels defined.
          if (total(src_channels - bkg_spectrum.CHANNEL) NE 0) then $
              message, 'ERROR: src and bkg spectra have different channel sets!'
            
          row.bkg_observed_counts  = 0
          row.bkg_observed_counts += bkg_spectrum.COUNTS
          row.bkg_backscal         = psb_xpar( bkg_header, 'BACKSCAL')
          row.bkg_exposure         = psb_xpar( bkg_header, 'EXPOSURE')
          
          ; Look for improper keyword values.
          hdr_vals = [row.bkg_backscal]
          dum = where((hdr_vals EQ 0) OR (~finite(hdr_vals)), count)
          if (count GT 0) then message, 'ERROR: BACKSCAL keyword is not valid in '+row.bkg_spectrum_fn
          
        endif else print, 'WARNING!  Background spectrum for ObsId '+row.obsid+' is missing.'

        ; AE spectra should have bkg EXPOSURE = 0.
        ; EPIC spectra should have identical src and bkg EXPOSURE.
        if ((row.bkg_exposure NE 0) && (abs(row.bkg_exposure - row.src_exposure)/row.src_exposure GT 0.01)) then $
          message, 'ERROR: background spectrum does not follow AE conventions'
        

        if psf_available then begin
          ; Use the PSF fraction entry that's at the nominal energy.
          psf_fraction = mrdfits(row.psf_frac_fn, 1, /SILENT, STATUS=status)
          if (status NE 0) then message, 'ERROR reading ' + row.psf_frac_fn
          
          ind = where(abs(psf_fraction.energy - nominal_psf_energy) LT 0.1, count)
          if (count EQ 0) then ind=[0]
          
          this = psf_fraction[ind[0]]
    
          row.psf_x_var            = this.x_sdev^2
          row.psf_y_var            = this.y_sdev^2
                                            
          dum = where(tag_names(this) EQ 'BKG_X_SDEV', count)
          if (count EQ 0) then begin
            ; For backward compatibility (before BKG_X_SDEV columns existed) ...
            row.bkg_x_var = this.x_sdev^2
            row.bkg_y_var = this.y_sdev^2
          endif else begin
            row.bkg_x_var = this.bkg_x_sdev^2
            row.bkg_y_var = this.bkg_y_sdev^2
          endelse
        endif ; psf_available
      endif ; spectra_available
      
      
      ;; Store the data.
      obs_data[jj] = temporary(row)
    endfor ; jj
    
    unpruned_exposure = total(obs_data.emap_avg, /DOUBLE)

    ;; ------------------------------------------------------------------------
    ;; PRUNE THE OBSERVATIONS TO A RANGE OF OFF-AXIS ANGLES, IF SPECIFIED
    ;; ------------------------------------------------------------------------
    accepted_ind = where((obs_data.off_angle GE theta_range[0]) AND (obs_data.off_angle LE theta_range[1]), num_obs, COMPLEMENT=rejected_ind, NCOMPLEMENT=num_rejected)

    if (num_obs EQ 0) then begin
      print, theta_range, F="(%'No observations fall in the specified off-axis angle range (%0.1f\' : %0.1f\').')"
      GOTO, MERGE_IS_EMPTY
    endif else if (num_obs EQ 1) && keyword_set(skip_single_obsid_merges) then begin
      print, F="(%'Skipping a single-ObsID merge, as requested.')"
      GOTO, MERGE_IS_EMPTY
    endif else if (num_rejected GT 0) then begin
      print, num_rejected, theta_range, F="(%'WARNING: ignoring %d extractions outside the specified off-axis angle range (%0.1f\' : %0.1f\'):')"
      forprint, SUBSET=rejected_ind, obs_data.obsname, obs_data.off_angle, F="(%'  %12s %0.1f\'')"
      
      obs_data = obs_data[accepted_ind]
    end



    ;; ------------------------------------------------------------------------
    ;; PRUNE THE OBSERVATIONS TO ELIMINATE EXTRACTION REGIONS FALLING OFF THE DETECTOR EDGE
    ;; ------------------------------------------------------------------------
    
; 2009 Jan We cannot think of a clear reason to discard sources near the field edge.  Non-uniformity in the emap leads to inaccuracy in the ARF, but that's true anywhere, including chip gaps.  The use of a Chandra+ACIS PSF (e.g. from MARX) for aperture correction also seems not quite right, since aperture correction seems like a property of the HRMA, not of the dithering; again, though, this is an issue anywhere the emap is not flat.  Thus, it's hard to see what's special about the field edge and why we should whack these extractions.  However, I'm leaving the code below, commented out, in case we change our minds and want a mechanism for pruning edge extractions.

;    accepted_ind = where((obs_data.emap_med/obs_data.emap_max) GT emap_uniformity_limit, num_obs, NCOMPLEMENT=num_rejected)
;                          
;    if (num_obs EQ 0) then begin
;      print, 'WARNING: all apertures appear to be spanning a detector edge; no data left to merge.'
;      GOTO, MERGE_IS_EMPTY
;    endif else if (num_rejected GT 0) then begin
;      print, num_rejected, F="(%'WARNING: ignoring %d extractions with apertures that appear to be spanning a detector edge.')"
;      obs_data = obs_data[accepted_ind]
;    end
;
     

    ;; ------------------------------------------------------------------------
    ;; PRUNE THE OBSERVATIONS TO ELIMINATE SEVERE OVERLAP OF EXTRACTION REGIONS
    ;; We do not have confidence that backgrounds can be accurately estimated 
    ;; when two sources share counts via severly overlapping extraction regions.
    ;; ------------------------------------------------------------------------
    overlap_is_acceptable = (obs_data.overlap LT overlap_limit)
    
    ; When an aperture has been hand edited (reg_edit EQ 1) we assume that the observer wants that extraction to be merged, regardless of any overlap it has with a neighbor's aperture!
    ind = where(~overlap_is_acceptable AND obs_data.reg_edit, count)
    if (count GT 0) then begin
      overlap_is_acceptable[ind] = 1B
      print, count, F="(%'WARNING: accepting %d hand-edited apertures with excessive OVERLAP:')"
      forprint, SUBSET=ind, obs_data.obsname, obs_data.overlap, F="(%'  %12s OVERLAP=%0.3f')"
    endif
    
    accepted_ind = where(overlap_is_acceptable, num_obs, COMPLEMENT=rejected_ind, NCOMPLEMENT=num_rejected)
    
    if (num_obs EQ 0) then begin
      if (n_elements(obs_data) EQ 1) then begin
        accepted_ind = 0
        num_obs      = 1
        print, obs_data.overlap, F="(%'WARNING: the single extraction has excessive OVERLAP (%0.2f).')"      
      endif else begin
        ; Although all the extractions have excessive overlap, we still need to merge some subset of the data so that we'll
        ; have rough photometry of the source to help us decide later whether to prune this source or his neighbor.
        ; Merging only the _single_ ObsId with the minimum overlap proved to be a poor design, because there are sometimes
        ; cases where that one happens to have photometry very differerent from one or more other ObsIds with overlap values 
        ; comparable to the smallest one.
        ; Thus we arbitrarily stretch the overlap limit upward by 20% for this source.
        stretched_overlap_limit = 1.2 * min(obs_data.overlap)
        print, n_elements(obs_data), stretched_overlap_limit, F="(%'WARNING: all %d extractions have excessive OVERLAP; accepting only those with OVERLAP comparable to the best extraction (i.e. OVERLAP<=%0.2f).')"      
        accepted_ind = where(obs_data.overlap LT stretched_overlap_limit, num_obs)
      endelse
    endif else if (num_rejected GT 0) then begin
      print, num_rejected, F="(%'WARNING: ignoring %d extractions with excessive OVERLAP:')"
      forprint, SUBSET=rejected_ind, obs_data.obsname, obs_data.off_angle, F="(%'  %12s %0.1f\'')"
      
    endif
    
    obs_data = obs_data[accepted_ind]

    
    
    ; Use RMF to figure out the energy bounds for each spectral channel.
    ; We pass null ARF_FN because we have not yet loaded any OBS_DATA.
    if spectra_available then ae_photometry, RMF_FN=obs_data[0].rmf_fn, ARF_FN=''
    
    
    ;; ------------------------------------------------------------------------
    ;; IF DESIRED, PRUNE THE OBSERVATIONS TO OPTIMIZE SOMETHING
    ;; ------------------------------------------------------------------------
    unbiased_exposure = total(obs_data.emap_avg, /DOUBLE) ; Save the exposure we would have for an unbiased photometry merge.
    
    if spectra_available && (num_obs GT 1) && (keyword_set(merge_for_pb)       || $
                                               keyword_set(merge_for_position) || $
                                               keyword_set(merge_for_photometry)) then begin
                                             
      if keyword_set(merge_for_pb)+keyword_set(merge_for_position)+keyword_set(merge_for_photometry) GT 1 then begin
        print, 'ERROR: only one MERGE optimization may be specified!'
        goto, FAILURE
      endif
      
      if ~skip_timing then print,F='(%"\nWARNING! This optimized merge (/MERGE_FOR_*) may not be appropriate for assessing variability.")'

      ;; ------------MERGE_FOR_PB------------------------------------------------------------
      ;; When MERGE_FOR_POSITION is requested it's important to get rid of ObsIDs with high background, since the MERGE_FOR_POSITION optimization can't do that.  So, we run the MERGE_FOR_PB algorithm first and then run the MERGE_FOR_POSITION algorithm.
      if keyword_set(merge_for_pb) || keyword_set(merge_for_position) then begin
      
        ; Caller must supply MIN_NUM_CTS if MERGE_FOR_PB or MERGE_FOR_POSITION requested.
        if ~keyword_set(min_num_cts) then begin
          print, 'ERROR: MIN_NUM_CTS must be specified!'
          goto, FAILURE
        endif
        
        ;; The P_B calculation below is supposed to replicate that in the section "COMPUTE PHOTOMETRY".

        ;; The observer wishes to prune the observation to minimize Pb, computed over ENERGY_RANGE.
        ;; Pb is the significance of the observed SRC_CNTS as a disproof of the "null hypothesis" which is 
        ;; that there is no source, i.e. that all the observed counts are background.  
        ;; We use equation A7 from Weisskopf 2006 (astro-ph/0609585).
        Pb_single_obs       = dblarr(num_obs)
        SRC_CNTS_single_obs = lonarr(num_obs)
        flag_value = 100  ; A flag value larger than any Pb signifying that we have already considered an obsid.
        accepted_ind = -1 ; Defensive coding ...
        jj=0
        while 1 do begin          
          if (jj LT num_obs) then begin
            ; Compute Pb for each obsid on its own.
            proposed_ind = jj
          endif else begin
            ; Propose to add the best (smallest Pb) unused observation.
            temp = min(Pb_single_obs, this_ind)
            ; Stop the search when we have considered all obsids (Pb_single_obs is full of flag values).
            if (temp EQ flag_value) then break
            
            proposed_ind = [accepted_ind, this_ind]
          endelse
          
          ; Compute photometry, Pb on the proposed set of extractions.
          ae_photometry, OBS_DATA=obs_data[proposed_ind], SRC_CHANNELS=src_channels
          ae_photometry, energy_range[0], energy_range[1], photometry

          
          if (jj LT num_obs) then begin
            ; Compute Pb for each obsid on its own.
            SRC_CNTS_single_obs[jj  ] = photometry.SRC_CNTS
                  Pb_single_obs[jj++] = photometry.PROB_NO_SOURCE 
                  
            ; Once Pb_single_obs is complete, begin the optimization search.
            if (jj EQ num_obs) then begin
              ; Accept the best (smallest Pb) observation.
              best_PROB_NO_SOURCE = min(Pb_single_obs, accepted_ind)
              SRC_CNTS_accepted   = SRC_CNTS_single_obs[accepted_ind]
              print, F='(%"\nOptimizing Pb:")'
              
;             forprint, obs_data.obsname, SRC_CNTS_single_obs, Pb_single_obs

              ; Flag the obsid we just considered so it won't be considered again.
              Pb_single_obs[accepted_ind] = flag_value
            endif 
          endif else begin
            ; Accept the proposed set of obsids if:
            ; * We do not yet have the specified minimum number of SRC_CNTS
            ; * Pb is zero 
            ; * Pb has not increased
            if (SRC_CNTS_accepted LT min_num_cts) || (photometry.PROB_NO_SOURCE EQ 0) || $
                                                     (photometry.PROB_NO_SOURCE LE best_PROB_NO_SOURCE) then begin
              accepted_ind        = proposed_ind
              best_PROB_NO_SOURCE = photometry.PROB_NO_SOURCE
              SRC_CNTS_accepted   = photometry.SRC_CNTS
            endif else print, obs_data[this_ind].obsname,                      F='(%"  Discard       ObsID %s")'
            
            ; Flag the obsid we just considered so it won't be considered again.
            Pb_single_obs[this_ind] = flag_value
          endelse
         endwhile
;debug_pb[ii] = best_PROB_NO_SOURCE
        psb_xaddpar, src_stats, 'MERGPRUN', 'MERGE_FOR_PB', 'pruning algorithm applied' 
        psb_xaddpar, src_stats, 'MERGQUAL',            1.0, 'Pb was optimized'        
        
        ; Discard the ObsIDs we have rejected.
        num_obs  = n_elements(accepted_ind)
        obs_data =   obs_data[accepted_ind]
        
        ; Defensively check for duplicates in the accepted_ind vector since a mistake here is really bad!
        if (n_elements(UNIQ(accepted_ind, SORT(accepted_ind))) NE num_obs) then message, 'ERROR: BUG IN AE!!' 
        
      endif ; MERGE_FOR_PB
      
      
      ;; -----------------------MERGE_FOR_PHOTOMETRY-------------------------------------------------
      if keyword_set(merge_for_photometry) then begin
        if ~keyword_set(min_quality) then begin
          print, 'ERROR: MIN_QUALITY must be specified!'
          goto, FAILURE
        endif
        
        ;; The SNR calculation below is supposed to replicate that in the section "COMPUTE PHOTOMETRY".

        ;; The observer wishes to prune the observation to maximize SNR, computed over ENERGY_RANGE.
        SNR_single_obs      = dblarr(num_obs)
        flag_value = -1E10  ; A flag value smaller than any SNR signifying that we have already considered an obsid.
        accepted_ind = -1 ; Defensive coding ...
        jj=0
        while 1 do begin          
          if (jj LT num_obs) then begin
            ; Compute SNR for each obsid on its own.
            proposed_ind = jj
          endif else begin
            ; Propose to add the best (largest SNR) unused observation.
            temp = max(SNR_single_obs, this_ind)
            ; Stop the search when we have considered all obsids (SNR_single_obs is full of flag values).
            if (temp EQ flag_value) then break
            
            proposed_ind = [accepted_ind, this_ind]
          endelse
          
          ; Compute photometry, SNR on the proposed set of extractions.
          ae_photometry, OBS_DATA=obs_data[proposed_ind], SRC_CHANNELS=src_channels
          ae_photometry, energy_range[0], energy_range[1], photometry

          
          if (jj LT num_obs) then begin
            ; Compute SNR for each obsid on its own.
              SNR_single_obs[jj++] = photometry.SRC_SIGNIF 
                  
            ; Once SNR_single_obs is complete, begin the optimization search.
            if (jj EQ num_obs) then begin
              ; Accept the best (largest SNR) observation.
              best_SNR    = max(SNR_single_obs, accepted_ind)
              adopted_SNR = best_SNR
              print, min_quality, F='(%"\nOptimizing SNR (subject to MIN_QUALITY=%0.1f):")'
              print, adopted_SNR, strjoin(obs_data[accepted_ind].obsname,'+'), F='(%"  SNR=%5.1f for ObsID %s")'
              
              ; Flag the obsid we just considered so it won't be considered again.
              SNR_single_obs[accepted_ind] = flag_value
            endif 
          endif else begin
            ; Accept the proposed set of obsids if:
            ; * The proposed SNR is larger than best_SNR * min_quality  (case where best_SNR > 0)
            ; * The proposed SNR is larger than best_SNR                (case where best_SNR < 0)
            if (photometry.SRC_SIGNIF GE (best_SNR*min_quality)) || (photometry.SRC_SIGNIF GE best_SNR)  then begin
              accepted_ind        = proposed_ind
                 best_SNR        >= photometry.SRC_SIGNIF
              adopted_SNR         = photometry.SRC_SIGNIF
              print, adopted_SNR, strjoin(obs_data[accepted_ind].obsname,'+'), F='(%"  SNR=%5.1f for ObsID %s")'
            endif else print, obs_data[this_ind].obsname,                      F='(%"  Discard       ObsID %s")'
            
            ; Flag the obsid we just considered so it won't be considered again.
            SNR_single_obs[this_ind] = flag_value
          endelse
        endwhile
                
        psb_xaddpar, src_stats, 'MERGPRUN', 'MERGE_FOR_PHOTOMETRY', 'pruning algorithm applied' 
        psb_xaddpar, src_stats, 'MERGQUAL',   adopted_SNR/best_SNR, 'adopted SNR / optimal SNR' 
        
        
        ; Discard the ObsIDs we have rejected.
        num_obs  = n_elements(accepted_ind)
        obs_data =   obs_data[accepted_ind]
        
        ; Defensively check for duplicates in the accepted_ind vector since a mistake here is really bad!
        if (n_elements(UNIQ(accepted_ind, SORT(accepted_ind))) NE num_obs) then message, 'ERROR: BUG IN AE!!' 
      endif ; MERGE_FOR_PHOTOMETRY
      
      
      ;; -----------------------MERGE_FOR_POSITION-------------------------------------------------
      if keyword_set(merge_for_position) then begin
        ;; The observer wishes to prune the observation to minimize position error, computed over ENERGY_RANGE.
        ;; The position uncertainty calculation below is supposed to replicate that
        ;; in the section "ESTIMATE POSITION UNCERTAINTY".
      
        position_error_single_obs = fltarr(num_obs)
        SRC_CNTS_single_obs       = lonarr(num_obs)
        flag_value = 1E6  ; A large error value signifying that we have already considered an obsid.
        jj=0
        while 1 do begin
          if (jj LT num_obs) then begin
            ; Compute position error for each obsid on its own.
            proposed_ind = jj
          endif else begin
            ; Propose to add the best (smallest position error) unused observation.
            temp = min(/NAN, position_error_single_obs, this_ind)
            ; Stop the search when we have considered all obsids (position_error_single_obs is full of flag values).
            if (temp EQ flag_value) then break
            
            proposed_ind = [accepted_ind, this_ind]
          endelse
          
          ; Compute photometry on the proposed set of extractions.
          ae_photometry, OBS_DATA=obs_data[proposed_ind], SRC_CHANNELS=src_channels
          ae_photometry, energy_range[0], energy_range[1], photometry, x_distribution_variance, y_distribution_variance


          ; The expos_data,eypos_data calculations below must be skipped when any of the terms are not available.
          if (photometry.SRC_CNTS EQ 0) || (x_distribution_variance LE 0) || (y_distribution_variance LE 0)  then begin
            this_position_error = !VALUES.F_NAN    
          endif else begin
            expos_data = sqrt(x_distribution_variance) / SQRT(photometry.SRC_CNTS)
            eypos_data = sqrt(y_distribution_variance) / SQRT(photometry.SRC_CNTS)
            this_position_error = sqrt( expos_data^2 + eypos_data^2 ) 
          endelse
          
          if (jj LT num_obs) then begin
            ; Compute position error for each obsid on its own.
            SRC_CNTS_single_obs      [jj  ] = photometry.SRC_CNTS
            position_error_single_obs[jj++] = this_position_error
                           
            ; Once position_error_single_obs is complete, begin the optimization search.
            if (jj EQ num_obs) then begin
              ; Accept the best (smallest position error) observation.
              best_position_error = min(/NAN, position_error_single_obs, accepted_ind)
              SRC_CNTS_accepted   =                  SRC_CNTS_single_obs[accepted_ind]
              print, F='(%"\nOptimizing position uncertainty:")'
              
;             forprint, obs_data.obsname, SRC_CNTS_single_obs, position_error_single_obs
              
              print, best_position_error, SRC_CNTS_accepted, strjoin(obs_data[accepted_ind].obsname,'+'), F='(%"  ERR_DATA=%5.2f (%d ct) for ObsID %s")'
             
              ; Flag the obsid we just considered so it won't be considered again.
              position_error_single_obs[accepted_ind] = flag_value
            endif 
          endif else begin
            ; If we do not yet have MIN_NUM_CTS in the merge, then accept the proposed set of obsids.
            ; If position error is zero or has decreased, then accept the proposed set of obsids.
            if (SRC_CNTS_accepted   LT min_num_cts) || (this_position_error EQ 0) || $
                                                       (this_position_error LT best_position_error) then begin
              accepted_ind        = proposed_ind
              best_position_error = this_position_error
              SRC_CNTS_accepted   = photometry.SRC_CNTS
              print, best_position_error, SRC_CNTS_accepted, strjoin(obs_data[accepted_ind].obsname,'+'), F='(%"  ERR_DATA=%5.2f (%d ct) for ObsID %s")'
            endif else print, obs_data[this_ind].obsname,                      F='(%"  Discard       ObsID %s")'

            ; Flag the obsid we just considered so it won't be considered again.
            position_error_single_obs[this_ind] = flag_value
          endelse
         endwhile
        psb_xaddpar, src_stats, 'MERGPRUN', 'MERGE_FOR_POSITION', 'pruning algorithm applied' 
        psb_xaddpar, src_stats, 'MERGQUAL',                  1.0, 'position error was optimized'        
        
        ; Discard the ObsIDs we have rejected.
        num_obs  = n_elements(accepted_ind)
        obs_data =   obs_data[accepted_ind]
        
        ; Defensively check for duplicates in the accepted_ind vector since a mistake here is really bad!
        if (n_elements(UNIQ(accepted_ind, SORT(accepted_ind))) NE num_obs) then message, 'ERROR: BUG IN AE!!' 
      endif ; MERGE_FOR_POSITION
      
      
      ; For all types of optimized merges, save the energy band in which optimized quantity was calculated.
      psb_xaddpar, src_stats, 'BANDPRUN', string(energy_range, F="(%'%0.2f:%0.2f keV')")
      
      ; END OF BLOCK "PRUNE THE OBSERVATIONS TO OPTIMIZE SOMETHING"
    endif else begin
      ; pruning NOT attempted
      sxdelpar, src_stats, ['MERGPRUN','MERGQUAL','BANDPRUN']
    endelse

    
    ;; ------------------------------------------------------------------------
    ;; We have chosen a set of ObsIDs to merge.
    num_obs = n_elements(obs_data)
    psb_xaddpar, src_stats, 'MERGNUM',  num_obs,  'number of observations merged'
    bkg_spectra_available = spectra_available && array_equal( obs_data.bkg_spectrum_fn NE '', 1B )

    ;; ------------------------------------------------------------------------
    ;; Sort the ObsIDs by TSTART, because:
    ;;  - Sorting should make MERGE_KS more sensitive to variability on long timescales.
    ;;  - A sorted ObsIDs_merged.fits will be appreciated by a human reader.
    ;;  - Merged event lists will be time-sorted.
    ;;  - Merged source.lc table will be time-sorted.
    
    if ~skip_aperture || ~skip_timing then begin
      for jj = 0, num_obs-1 do begin
        obs_data[jj].TSTART = psb_xpar( headfits(obs_data[jj].src_events_fn, EXT=1), 'TSTART' )
      endfor ; jj
      obs_data = obs_data[sort(obs_data.TSTART)]
    endif
    


    ;; ------------------------------------------------------------------------
    ;  For both point and diffuse sources, combining extractions by *summing* source
    ;  bkg spectra (leaving the merged spectra in COUNT units) is legitimate only
    ;  when all the extractions have the same bkg scaling.
    ;  For point sources, our personal recipes design point source bkg regions
    ;  to maintain similar scaling.
    ;  But diffuse backgrounds generally come from stowed data in CALDB and their 
    ;  scaling is not under our direct control.
    ;  So, for diffuse sources, we will decimate each of the bkg spectrum to that they
    ;  all have the same remaining scaling.
    ;; ------------------------------------------------------------------------
    if (num_obs GT 1) && bkg_spectra_available && is_diffuse then begin
      print, F='(%"\nDiscarding counts from background spectra to achieve identical BACKSCAL values.")'
      ; We want every extraction to end up with the same background scaling
      ; ("area exposure ratio", "1/BACKSCAL" in XSPEC ).
      ; That value should be max(area_exposure_ratio) = min(BACKSCAL).
      area_exposure_ratio_original = float(obs_data.src_backscal)/obs_data.bkg_backscal
      area_exposure_ratio_target   = max(area_exposure_ratio_original)
      
      ; Scale each bkg spectrum (obs_data[jj].spectra.), retaining integer count values.
      ; Scale each "area-exposure" calibration (obs_data[jj].bkg_backscal) accordingly.
      for jj = 0, num_obs-1 do begin
        ; We scale bkg spectra DOWN (never up), equivalent to discarding bkg counts/area.
        this_scaling =  area_exposure_ratio_original[jj] / area_exposure_ratio_target
        if (this_scaling GT 1) then message, 'ERROR: BUG IN AE!!' 
        
        total_original_scaled_counts = total(/INTEGER, obs_data[jj].bkg_observed_counts) / obs_data[jj].bkg_backscal

        obs_data[jj].bkg_observed_counts = stochastic_round( this_scaling * obs_data[jj].bkg_observed_counts )
        obs_data[jj].bkg_backscal        =                   this_scaling * obs_data[jj].bkg_backscal

        total_revised_scaled_counts = total(/INTEGER, obs_data[jj].bkg_observed_counts) / obs_data[jj].bkg_backscal

        print, obs_data[jj].obsname, this_scaling, F='(%"\nObsID %s background scaled by %0.2f.")'
        if ~almost_equal(total_original_scaled_counts,  total_revised_scaled_counts, TOLERANCE=0.001) then begin
          print, total_original_scaled_counts,  total_revised_scaled_counts, F='(%"\nERROR: scaled background counts changed significantly: %g --> %g).")'
          stop
        endif
      endfor ;jj
; Confirm that revised area exposure ratios are identical.
;forprint, area_exposure_ratio_original, float(obs_data.src_backscal)/obs_data.bkg_backscal
;stop
    endif ;spectra_available && is_diffuse



    ;; ------------------------------------------------------------------------
    ; Load the extractions we've chosen to merge into the ae_photometry routine.
    ;; ------------------------------------------------------------------------
    if spectra_available then ae_photometry, OBS_DATA=obs_data, SRC_CHANNELS=src_channels
    
    mergfrac = total(obs_data.emap_avg, /DOUBLE)/unpruned_exposure
    if (mergfrac LT 0.99) then print, round(100*(1-mergfrac)), F="(%'\nPruning algorithms discarded %d%% of the exposure.\n')"
    psb_xaddpar, src_stats, 'MERGFRAC', mergfrac, 'fraction of extracted data merged'
    
    psb_xaddpar, src_stats, 'MERGBIAS', (unbiased_exposure - total(obs_data.emap_avg, /DOUBLE)) / unbiased_exposure, 'fraction of exposure discarded to optimize merge'
    
    
    ; Record the extractions we've chosen to merge in a FITS file, for use by other tools.
    ; To save disk space I'm removing tags that store vectors, using an undocumented option (REMOVE) to create_struct().
    fxbhmake, theader, num_obs, 'OBSIDS MERGED', /INITIALIZE, /DATE
    psb_xaddpar, theader, 'CREATOR' , creator_string
    psb_xaddpar, theader, 'FOR_PB'  , keyword_set(merge_for_pb)        , 'MERGE_FOR_PB'
    psb_xaddpar, theader, 'FOR_POSN', keyword_set(merge_for_position)  , 'MERGE_FOR_POSITION'
    psb_xaddpar, theader, 'FOR_PHOT', keyword_set(merge_for_photometry), 'MERGE_FOR_PHOTOMETRY'
    psb_xaddpar, theader, 'OLAP_LIM', overlap_limit                    , 'OVERLAP_LIMIT'
   ;psb_xaddpar, theader, 'UNIF_LIM', emap_uniformity_limit            , 'EMAP_UNIFORMITY_LIMIT'
    psb_xaddpar, theader, 'THETA_LO', theta_range[0]                   , 'THETA_RANGE'
    psb_xaddpar, theader, 'THETA_HI', theta_range[1]                   , 'THETA_RANGE'
    
    tag_names = tag_names(obs_data)
    remove_tag = strmatch(tag_names, 'SRC_OBSERVED_COUNTS') OR $
                 strmatch(tag_names, 'BKG_OBSERVED_COUNTS')
    if array_equal(remove_tag, 0B) then begin
      trimmed_obs_data = obs_data
    endif else begin
      ; NOTE that REMOVE is an undocumented feature of the intrinsic function create_struct().
      trimmed_obs_data = replicate(create_struct(obs_data[0], REMOVE=where(remove_tag)), num_obs)
      struct_assign, obs_data, trimmed_obs_data
    endelse
    
    mwrfits, trimmed_obs_data, merged_list_fn, theader, /CREATE
    

    
    ;; ------------------------------------------------------------------------
    ;; MERGE FITS KEYWORDS
    ;; ------------------------------------------------------------------------
    ;; Appropriately combine single-obsid source properties from multiple obs.stats files.
    total_src_counts = total(obs_data.src_counts)
    total_exposure   = spectra_available ? total(obs_data.src_exposure) : 1.0
    
    dum = max(obs_data.emap_avg, imax)
    psb_xaddpar, src_stats, 'PRIM_OBS', obs_data[imax].obsname,                   'deepest merged observation'
    
    psb_xaddpar, src_stats, 'EMAP_TOT', total(      obs_data.emap_avg),           '[s cm**2 count /photon] sum of EMAP_AVG values for merged observations'
    psb_xaddpar, src_stats, 'WARNFRAC', total(/INT, obs_data.num_in_warning)  / total_src_counts,       'fraction of events in merged warning regions'
    psb_xaddpar, src_stats, 'OBSNAME',  strjoin(obs_data.obsname,','),            'merged observations'
    psb_xaddpar, src_stats, 'SRC_CNTS', round(total_src_counts),                  '[count] source apertures, in-band, merged observations'

    if ~is_diffuse then begin
      psb_xaddpar, src_stats, 'FRACEXPO', total(      obs_data.time_on_detector)/ total_exposure,'time_on_detector/EXPOSURE'
      psb_xaddpar, src_stats, 'RATE_3x3',   max(/NAN, obs_data.RATE_3x3)                        ,'[count /frame] in 3x3 cell, largest for merged observations'
      
      ; For off-axis angles, use the average weighted by exposure.
      exposure_weight = obs_data.emap_avg * (obs_data.off_angle    GT 0)
      merged_off_angle       = total(obs_data.off_angle   * exposure_weight) / (total(exposure_weight) > 1)         
      psb_xaddpar, src_stats, 'THETA',    merged_off_angle,                       '[arcmin] average off-axis angle for merged observations'
      psb_xaddpar, src_stats, 'THETA_LO', min(obs_data.off_angle),                '[arcmin] smallest off-axis angle for merged observations'
      psb_xaddpar, src_stats, 'THETA_HI', max(obs_data.off_angle),                '[arcmin] largest off-axis angle for merged observations'
      
      ; For source extraction properties, use the average weighted by exposure.
      exposure_weight = obs_data.emap_avg * (obs_data.src_area     GT 0)
      merged_src_area        = total(obs_data.src_area    * exposure_weight) / (total(exposure_weight) > 1)
      psb_xaddpar, src_stats, 'SRC_AREA',  merged_src_area,                      '[skypixel**2] average aperture area for merged observations'
  
      exposure_weight = obs_data.emap_avg * (obs_data.src_radius   GT 0)
      merged_src_radius      = total(obs_data.src_radius  * exposure_weight) / (total(exposure_weight) > 1)
      psb_xaddpar, src_stats, 'SRC_RAD',  merged_src_radius,                      '[skypixel] average source radius for merged observations'
  
      psb_xaddpar, src_stats, 'REG_EDIT',  max(obs_data.reg_edit),                'T=1=one or more custom apertures'

      exposure_weight = obs_data.emap_avg * (obs_data.psf_fraction GT 0)
      merged_psf_fraction    = total(obs_data.psf_fraction * exposure_weight) / (total(exposure_weight) > 1)
      psb_xaddpar, src_stats, 'PSF_FRAC', merged_psf_fraction,                    'average '+comment_psf_fraction+' for merged observations'
  
      ; Report crowding information about the most-crowded observation merged.
      psb_xaddpar, src_stats, 'FRACSPEC', min(obs_data.fracspec),                 'smallest target PSF fraction for merged observations'
      psb_xaddpar, src_stats, 'OVRLP_LM', overlap_limit,                          'overlap limit used for pruning'
      psb_xaddpar, src_stats, 'OVRLP_LO', min(obs_data.overlap, imin),            'smallest overlap fraction for merged observations'
      psb_xaddpar, src_stats, 'OVRLP_HI', max(obs_data.overlap, imax),            'largest overlap fraction for merged observations'
      psb_xaddpar, src_stats, 'BESTOBS' , obs_data[imin].obsname,                 'ObsID corresponding to OVRLP_LO'
      psb_xaddpar, src_stats, 'WORSTOBS', obs_data[imax].obsname,                 'ObsID corresponding to OVRLP_HI'
    endif ; ~is_diffuse

    ; For mask radius, report the maximum value.
    psb_xaddpar, src_stats, 'MSK_RAD', max(obs_data.mask_radius),               '[skypixel] largest mask radius for merged observations'
    
    ; Report timing information from the ObsId showing the most variability.
    ; The CDF of the minimum function  PROB_KS = min(p1,...,pN) 
    ; where Pi are independent p-values (i.e. uniform on [0:1]) is p(PROB_KS <= x) = 1 - (1-x)^N
    ; http://www.di.fc.ul.pt/~jpn/r/prob/range.html
    ;
    ; We currently force the observer to do that calculation, if desired.
    ; "N_KS" records the number of non-NaN single-ObsID p-values participating in the min() function.
    finite_prob_ks = (obs_data.prob_ks)[where(/NULL, finite(obs_data.prob_ks))]
    if isa(finite_prob_ks) then begin
      psb_xaddpar, src_stats, 'PROB_KS',        min(finite_prob_ks), 'smallest p-value, single-ObsID variability'
      psb_xaddpar, src_stats,    'N_KS', n_elements(finite_prob_ks), '# single-ObsID variability p-values'
    endif else begin
      psb_xaddpar, src_stats, 'PROB_KS', !VALUES.F_NAN, 'smallest p-value, single-ObsID variability'
      psb_xaddpar, src_stats,    'N_KS',             0, '# single-ObsID variability p-values'
    endelse

; To understand the effect min() on a set of p-values, the plot below is helpful.
; .run
; num_bins = 1D5
; binsize  = 1D/num_bins
; X = dindgen(1 + num_bins)/num_bins
; X = X[1:*]
; foreach N, reverse([1,2,4,8,16,32,64]) do $
;   function_1d, id1, X, 1D - (1D-X)^N, DATASET='N='+strtrim(N,2), XTIT='X = min(p1, p2,..., pN)', YTIT='p-value', TIT='p-value inflation by multiple comparisions'
; end
; Change both axes to log.

; We could also compute a meta-analysis p-value using Fisher's Method.
; That method seems to be appropriate for detecting persistent variability (rather than infrequent flaring).  

;NOTE TO SELF:
;
;Back in 2017 we investigated Fisher's method for "meta analysis" of the multiple p-values (PROB_NO_SOURCE) we have for detection significance.  See 2017 Sept 2 notes in
;/Volumes/cochise1/targets/t-rex/data/extract_fullmerge/point_sources.noindex/validation_notes.txt
;
;At that time, we concluded that combining PROB_NO_SOURCE values in that way was fraught with problems, because the individual p-values did not have nice random (uniform) distributions.  For example a source could have zero counts (PROB_NO_SOURCE=1) in 53 ObsIDs, and be obviously detected in one ObsID.  Fisher's Method would use the non-detections to overwhelm the firm detection.  The method seems to be inappropriate for detecting an Alternate Hypothesis that appears in only one of many observations.
;
; However, the  min(p1,...pN) statistic also allows observations with no data to overwhelm a firm detection, because the distribution of  min(p1,...pN) depends on N.



    ; Use the most on-axis observation as both the template for the composite PSF and as the reference tangent plane for reprojected event lists.
    dum = min(obs_data.off_angle, reference_observation_index)

    
    ;; ------------------------------------------------------------------------
    ;; Delete files produced by the /CHECK_POSITIONS stage, because a fresh merge makes them stale/obsolete.
    file_delete, /ALLOW_NONEXISTENT, composite_img_fn, event_reg_fn


    ;; ------------------------------------------------------------------------
    ;; MERGE PSF IMAGES
    ;; ------------------------------------------------------------------------
    if (psb_xpar( src_stats, 'DIFFUSE')) then GOTO, MERGE_APERTURES
    
    ;; Construct composite PSF image by weighting single-observation normalized PSFs by exposure.
    ;; In case we SKIP_PSF below, delete the PSF files we're about to build.
    file_delete, /ALLOW_NONEXISTENT, composite_psf_fn, rebinned_composite_psf_fn     
    if (skip_psf) then begin
      if NOT keyword_set(skip_psf_p) then print, F='(%"\nWARNING!  PSFs missing; not merged.")'
      GOTO, MERGE_APERTURES
    endif
    
    
    exposure_weight   = obs_data.emap_avg / (total(obs_data.emap_avg) > 1)
    composite_psf_hdr = ''
    
    ; Use the most on-axis ObsID as the template (binsize, field of view, set of PSF energies) for the composite PSF.
    fits_open, obs_data[reference_observation_index].psf_fn, reference_fcb
    for kk =0, reference_fcb.NEXTEND do begin
      fits_read, reference_fcb, dum, composite_psf_hdr, /HEADER_ONLY, /NO_PDU, EXTEN_NO=kk
      
      hduname              = psb_xpar( composite_psf_hdr, 'HDUNAME') 
      composite_psf_energy = psb_xpar( composite_psf_hdr, 'ENERGY') 
      radius50             = psb_xpar( composite_psf_hdr, 'RADIUS50')
      print, strtrim(hduname,2), composite_psf_energy, F='(%"\n%s (%0.1f keV) =")'

      composite_psf     = 0
      for jj = 0, num_obs-1 do begin
        ; Search the PSFs from this observation for the one whose mono-energy is closest to the template (composite_psf_energy).
        fits_open, obs_data[jj].psf_fn, this_fcb
        this_psf_energy = fltarr(1+this_fcb.NEXTEND)
        for ll =0, this_fcb.NEXTEND do begin
          fits_read, this_fcb, dum, this_header, /HEADER_ONLY, /NO_PDU, EXTEN_NO=ll
          this_psf_energy[ll] = psb_xpar( this_header, 'ENERGY') 
        endfor; ll
        dum = min(abs(this_psf_energy - composite_psf_energy), imin)
        fits_read, this_fcb, psf_img, this_header, /NO_PDU, EXTEN_NO=imin
        fits_close, this_fcb
        
        ; Set all the NaN values to zero to keep future computations happy.
        ind = where(finite(psf_img) EQ 0, count)
        if (count GT 0) then psf_img[ind] = 0
        
        ; Resample the single-ObsID PSF to the pixel grid of the template.
        ; We have to be careful with the normalization of the resampled image!
        
        ; We begin by normalizing this single-ObsID PSF image by its total power, found in the header keyword PSF_TOTL.
        psf_total = psb_xpar( this_header, 'PSF_TOTL')
        if (psf_total EQ 0) then begin
          print, "WARNING: obsolete PSFs in "+obs_data[jj].unnamed_obs_dir+" have incorrect scaling."
          psf_total = total(psf_img, /DOUBLE) 
        endif 
          
        psf_img /= psf_total
;print, total(/DOUBLE, psf_img)
        
        ; Then, we resample this normalized PSF image onto the template.
        ; Note that hastrom.pro interpolates rather than "re-bins", and thus does NOT preserve the total "power" in the image.
        ; Note also that resampling may crop this PSF image, since the footprint of this PSF and the template may be different.
        hastrom, temporary(psf_img), this_header, resampled_psf_img, dum_hdr, composite_psf_hdr, MISSING=0
        
        ; Because the resampled PSF image now has an unknown cropping, its correct normalization cannot be calculated from the sum of its pixels.
        ; Instead, we calculate the pixel rescaling required based on the change in pixel area between the original PSF image and the template.
        degree_per_templatepixel = psb_xpar( composite_psf_hdr, 'CDELT2')
        degree_per_psfpixel      = psb_xpar(       this_header, 'CDELT2')

        resampled_psf_img *= (degree_per_templatepixel^2 / float(degree_per_psfpixel)^2)
;print, total(/DOUBLE, resampled_psf_img)
;help, resampled_psf_img
        
        ; Finally, we rescale the resampled PSF by the exposure fraction of the observation.
        ; We cast to FLOAT type to save disk space.
        composite_psf += resampled_psf_img * float(exposure_weight[jj]) 
        
print, 'obs'+obs_data[jj].obsname, psb_xpar( this_header, 'ENERGY'), exposure_weight[jj], F='(%"%10s: (PSF @ %0.1f keV) * %0.3f +")'

        ; Keep track of the smallest RADIUS50 value found.
        this_radius50 = psb_xpar( this_header, 'RADIUS50')
        if (this_radius50 GT 0) then radius50 <= this_radius50
      endfor ;jj, looping over observations
    
      ; It is vital that the header used to save the composite PSF is the one from the template image ("composite_psf_hdr"), which has untouched astrometry keywords and reports the energy of the PSF we're building (ENERGY keyword).
      ; The header modified by hastrom.pro ("this_header") can NOT be used because hastrom does NOT update the PHYSICAL coordinate system keywords.
      ;
      ; We DO NOT touch the RA and DEC keywords in the PSF header---they record the location of the imaginary point source that constructed the PSF, which may be slightly shifted from the current position assigned to the actual source (because in the CONSTRUCT stage we re-use existing PSFs if they are "close" to the current source position).
      sxdelpar, composite_psf_hdr,['EXTNAME', 'X_CAT', 'Y_CAT', 'SUMRCTS','PCOUNT','GCOUNT']
      psb_xaddpar, composite_psf_hdr, 'CREATOR', creator_string
      psb_xaddpar, composite_psf_hdr, 'RADIUS50', radius50, '[arcsec] smallest RADIUS50 value for merged observations'
      psb_xaddpar, composite_psf_hdr, 'PSF_TOTL',      1.0, 'normalization of this image'
      crop_fraction = (1 - total(/DOUBLE, composite_psf)) > 0 
    ; help, crop_fraction
      psb_xaddpar, composite_psf_hdr, 'CROPFRAC', crop_fraction, 'fraction of the PSF cropped'
      if (kk EQ 0) then begin
        writefits, composite_psf_fn, float(composite_psf), composite_psf_hdr
      endif else begin
        sxdelpar, composite_psf_hdr, ['SIMPLE','EXTEND']
        psb_xaddpar, composite_psf_hdr, 'XTENSION', 'IMAGE   ', BEFORE='BITPIX'
        psb_xaddpar, composite_psf_hdr, 'EXTNAME', hduname
        mwrfits, float(composite_psf), composite_psf_fn, composite_psf_hdr
      endelse
    endfor ; kk
    fits_close, reference_fcb




MERGE_APERTURES:
    file_delete, /ALLOW_NONEXISTENT, merged_src_events_fn

    if (skip_aperture) then begin
      if ~keyword_set(skip_aperture_p) then print, F='(%"\nWARNING!  source.evt files missing; not merged.")'
      GOTO, MERGE_NEIGHBORHOODS
    endif
    
    
    ;; ------------------------------------------------------------------------
    ;; MERGE SOURCE EVENT LISTS
    ;; ------------------------------------------------------------------------
    ;; Align & merge the source event lists.
    ;; Use the most on-axis observation (reference_observation_index) as the reference tangent plane for reprojected event lists.
    if (num_obs EQ 1) then begin
      path = named2generic_extraction_path+strmid(obs_data[0].src_events_fn,strlen(basedir))
      file_link, path, merged_src_events_fn
      if (verbose GT 0) then print, merge_subdir[ii]+src_events_basename, path, F='(%"              %s --> %s")' 
    endif else begin
      temp_events_fn    = tempdir + string(indgen(num_obs), F='(%"temp%d.evt")')
      ; Get rid of temp files from the previous source.
      file_delete, temp_events_fn, /ALLOW_NONEXISTENT
      
      for jj=0,num_obs-1 do begin
        ; Link to reference observation's data, or reproject other observations.
        if (jj EQ reference_observation_index) then begin
          file_link, cwd+obs_data[reference_observation_index].src_events_fn, temp_events_fn[jj]
        endif else begin
          cmd = string(obs_data[jj].src_events_fn, temp_events_fn[jj], obs_data[reference_observation_index].src_events_fn, $
                       F='(%"reproject_events %s %s match=%s aspect=none random=-1")')
          run_command, cmd, QUIET=(verbose EQ 0)
        endelse
      endfor
      
      ; We use columnList to remove the PHAS column so that 3x3 & 5x5 files can be merged,
      ; and to remove the PHA_RO column, which tg_resolve_events does not propagate.
      ; We whack the expno subspace to work around a bug in CIAO 3.4.
      ; We save the STATUS column for use by the ae_afterglow_report tool.
      forprint, temp_events_fn, F='(%"%s[subspace -expno,-sky]")', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
      cmd = string( temp_text_fn, merged_src_events_fn, $
                    F="(%'dmmerge ""@-%s"" columnList=""-phas,-pha_ro"" outfile=%s  ')")
      run_command, cmd, QUIET=(verbose EQ 0)
    endelse


    ;; ------------------------------------------------------------------------
    ;; Compute a mean position for all the in-band events in all observations.
    ;; ------------------------------------------------------------------------
    ;; The energy range is controlled by the ENERGY_RANGE keyword.
    
    ;; The position is easy, we just filter the composite event list by energy, 
    ;; find the mean event position, and convert to celestial coordinates.
    cmd = string(merged_src_events_fn, 1000*energy_range, inband_events_fn, $
                         F="(%'dmcopy ""%s[energy=%6.1f:%7.1f]"" %s')")
    run_command, cmd, QUIET=(verbose EQ 0)

    inband_events = mrdfits(inband_events_fn, 1, theader, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + inband_events_fn

    ; If the FITS table is empty, mrdfits will return a scalar zero.
    if NOT keyword_set(inband_events) then begin
      ;; There are no in-band data so we skip various statistics.
      print, 'WARNING: no in-band data found in source region.'
    endif else begin 
      ; Build astrometic structure from data header.
      fxbfind, theader, 'TTYPE', dum1, TTYPE, dum2, 'null'
      fxbfind, theader, 'TCTYP', dum1, TCTYP, dum2, 'null'
      fxbfind, theader, 'TCRVL', dum1, TCRVL, dum2, 0.0D
      fxbfind, theader, 'TCRPX', dum1, TCRPX, dum2, 0.0D
      fxbfind, theader, 'TCDLT', dum1, TCDLT, dum2, 0.0D
      colnames = strlowcase( strtrim(TTYPE,2) )
      x_ind    = where(strlowcase(colnames) EQ 'x')
      y_ind    = where(strlowcase(colnames) EQ 'y')
      make_astr, event2wcs_astr, DELTA=TCDLT[[x_ind,y_ind]], CTYPE=TCTYP[[x_ind,y_ind]], $
                                 CRPIX=TCRPX[[x_ind,y_ind]], CRVAL=TCRVL[[x_ind,y_ind]]

      ; Convert mean position to celestial coordinates.
      ; REMEMBER THAT THE xy2ad and ad2xy programs assume that (x,y) are 
      ; ZERO-BASED pixel indexes.  Thus we must subtract 1 from the (x,y) 
      ; positions when converting to RA,DEC.
      xy2ad, mean(inband_events.x, /DOUBLE)-1, mean(inband_events.y, /DOUBLE)-1, event2wcs_astr, ra_data, dec_data
    endelse

    psb_xaddpar, src_stats, 'RA_DATA',   ra_data, string(energy_range, F="(%'[deg] position, %0.2f:%0.2f keV data mean')"), F='(F10.6)'
    psb_xaddpar, src_stats, 'DEC_DATA', dec_data, string(energy_range, F="(%'[deg] position, %0.2f:%0.2f keV data mean')"), F='(F10.6)'

    if is_diffuse then begin
      ; Assign RA and DEC so that COLLATE stage can make sensible region files.
      psb_xaddpar, src_stats, 'RA',   ra_data,     '[deg] source position, data mean', F='(F10.6)'
      psb_xaddpar, src_stats, 'DEC', dec_data,     '[deg] source position, data mean', F='(F10.6)'
    endif



MERGE_NEIGHBORHOODS:
    file_delete, /ALLOW_NONEXISTENT, merged_env_events_fn,  merged_region_fn
    
    if (skip_neighborhood) then begin
      if NOT keyword_set(skip_neighborhood_p) then print, F='(%"\nWARNING!  Neighborhoods missing; not merged.")'
      GOTO, MERGE_SPECTRA
    endif
    
    ;; ------------------------------------------------------------------------
    ;; MERGE NEIGHBORHOOD EVENT LISTS
    ;; ------------------------------------------------------------------------
    if is_diffuse then begin
      ;; DIFFUSE SOURCE
      ;  Diffuse extractions have no neighborhood event list, so just make a symlink to the merge of the extracted data.
      file_link, src_events_basename, merged_env_events_fn
    endif else begin
      ;; POINT SOURCE
      ;; Align & merge the neighborhood event lists.
      
      ; Use the most on-axis observation (reference_observation_index) as the reference tangent plane for reprojected event lists.

      ; We apply to each extraction the STATUS=0 filter decision (S_FILTER) made for that extraction with the hope that this strategy produces better image reconstructions, on average.
      ; This is not an easy call in the case where the source we are merging is weak, but very close neighbor is very bright.
      ; The aggressive filtering chosen for such a source at the extraction stage will reduce the flat background seen by the reconstruction.
      ; However, the core of the bright neighbor may be suppressed by the aggressive cleaning, leading to a ring of artifacts, some of which might fall in the aperture of the source we are working on here.

      if (num_obs EQ 1) then begin
      
        cmd = string( obs_data[0].env_events_fn, obs_data[0].s_filter, merged_env_events_fn, $
                      F="(%'dmcopy ""%s[%s]"" %s')")
        run_command, cmd
      endif else begin
        temp_events_fn    = tempdir + string(indgen(num_obs), F='(%"temp%d.evt")')
        ; Get rid of temp files from the previous source.
        file_delete, temp_events_fn, /ALLOW_NONEXISTENT
        
        for jj=0,num_obs-1 do begin
          ; Link to reference observation's data, or reproject other observations.
          if (jj EQ reference_observation_index) then begin
            file_link, cwd+obs_data[reference_observation_index].env_events_fn, temp_events_fn[jj]
          endif else begin
            cmd = string(obs_data[jj].env_events_fn, temp_events_fn[jj], obs_data[reference_observation_index].env_events_fn, $
                         F='(%"reproject_events %s %s match=%s aspect=none random=-1")')
            run_command, cmd
          endelse
        endfor
        
        ; We whack the expno "subspace" to work around a bug in CIAO 3.4 that corrupts the GTI tables.
        forprint, temp_events_fn, obs_data.s_filter, F='(%"%s[subspace -expno,-sky][%s]")', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
        
        ; We use columnList to remove the PHAS column so that 3x3 & 5x5 files can be merged,
        ; and to remove the PHA_RO column, which tg_resolve_events does not propagate.
        ; We save the STATUS column for use by the ae_afterglow_report tool.
        cmd = string( temp_text_fn, merged_env_events_fn, $
                      F="(%'dmmerge ""@-%s"" columnList=""-phas,-pha_ro"" outfile=%s  ')")
        run_command, cmd
      endelse
    endelse ; point source



    ;; ------------------------------------------------------------------------
    ;; Adjust the TDMIN & TDMAX keywords to get ds9 to produce a nice default binning of the event data.
    ;; HEASARC standards say their datatype should be the same as that of the table column to which they refer.
    ;; These TDMIN/TDMAX values are going to influence the size of the image built in the CHECK_POSITIONS stage.
    ;; Each extraction can have a different neighborhood size.
    ;; We choose the LARGEST neighborhood, to protect the center of the merged image from reconstruction artifacts.
    ;; ------------------------------------------------------------------------

    ; Let's use the box that bounds all the neighborhoods (which can be very different sizes).
    neighborhood_xsize = replicate(!VALUES.F_NAN, num_obs) ; skypix
    neighborhood_ysize = replicate(!VALUES.F_NAN, num_obs) ; skypix

    for jj=0,num_obs-1 do begin
      this_filename = obs_data[jj].env_events_fn
      ; For diffuse source, neighborhood.evt is a symlink to the ObsID event list, so we get TD* values from source.evt.
      if file_test(/SYMLINK, this_filename) then this_filename = obs_data[jj].src_events_fn
      
      env_header = headfits(this_filename, EXT=1)

      ; Identify the X and Y columns in env_events_fn.
      fxbfind, env_header, 'TTYPE', dum1, TTYPE, dum2, 'null'
      colnames = strlowcase( strtrim(TTYPE,2) )
      x_colnum = 1+where(strlowcase(colnames) EQ 'x')
      y_colnum = 1+where(strlowcase(colnames) EQ 'y')
      
      ; Each ObsID's TDM* keywords is in a different SKY coordinate system ...
      TDMINx = psb_xpar( env_header, string(x_colnum,F='(%"TDMIN%d")'), COUNT=count1)
      TDMAXx = psb_xpar( env_header, string(x_colnum,F='(%"TDMAX%d")'), COUNT=count2)
      TDMINy = psb_xpar( env_header, string(y_colnum,F='(%"TDMIN%d")'), COUNT=count3)
      TDMAXy = psb_xpar( env_header, string(y_colnum,F='(%"TDMAX%d")'), COUNT=count4)
      if (count1 EQ 1) && (count2 EQ 1) && (count3 EQ 1) && (count4 EQ 1) then begin
        ; ... but we care only about the height and width of the neighborhoods.  :)
        neighborhood_xsize[jj] = TDMAXx - TDMINx
        neighborhood_ysize[jj] = TDMAXy - TDMINy
      endif
    endfor ; jj                                        
    
    neighborhood_xsize = max(neighborhood_xsize, /NaN)
    neighborhood_ysize = max(neighborhood_ysize, /NaN)
    
    ; The center of the merged neighborhood is the source's SKY position in the reference observation.
    TDMINx = (obs_data[reference_observation_index].xpos_catalog - neighborhood_xsize/2)
    TDMAXx = (obs_data[reference_observation_index].xpos_catalog + neighborhood_xsize/2)
    TDMINy = (obs_data[reference_observation_index].ypos_catalog - neighborhood_ysize/2)
    TDMAXy = (obs_data[reference_observation_index].ypos_catalog + neighborhood_ysize/2)
    
    if ~(finite(TDMINx) && finite(TDMAXx) && finite(TDMINy) && finite(TDMAXy)) then begin
      print, 'ERROR: all neighborhood.evt/source.evt files are missing TDMIN/TDMAX keywords.'
      goto, FAILURE
    endif

    ; Diffuse extractions do not have a "position", so we need TDMIN/TDMAX to describe the bounding box of the data.
    if is_diffuse then begin
      TDMINx = min(inband_events.x) - 50
      TDMAXx = max(inband_events.x) + 50
      TDMINy = min(inband_events.y) - 50
      TDMAXy = max(inband_events.y) + 50
    endif ; is_diffuse
       
    ; When source.evt is a symlink we skip trying to modify the header.
    if ~skip_aperture && file_test(merged_src_events_fn) && ~file_test(merged_src_events_fn, /SYMLINK)  then begin
      ; Figure out the column numbers for X and Y in the merged source.evt file.
      merged_header = headfits(merged_src_events_fn, EXT=1)

      ; Identify the X and Y columns in merged_src_events_fn
      fxbfind, merged_header, 'TTYPE', dum1, TTYPE, dum2, 'null'
      colnames = strlowcase( strtrim(TTYPE,2) )
      x_colnum = 1+where(strlowcase(colnames) EQ 'x')
      y_colnum = 1+where(strlowcase(colnames) EQ 'y')
      
      openw, unit, temp_text_fn, /GET_LUN
      printf, unit, x_colnum, TDMINx, x_colnum, TDMAXx, y_colnum, TDMINy, y_colnum, TDMAXy, F='(%"#add\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f")'
      free_lun, unit
      
      cmd = string(merged_src_events_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
      run_command, cmd, /QUIET
    endif ; Modify source.evt header.
      
    ; When neighborhood.evt is a symlink we skip trying to modify the header.
    if ~skip_neighborhood && file_test(merged_env_events_fn) && ~file_test(merged_env_events_fn, /SYMLINK)  then begin
      ; Figure out the column numbers for X and Y in the merged neighborhood.evt file.
      merged_header = headfits(merged_env_events_fn, EXT=1)

      ; Identify the X and Y columns in merged_env_events_fn
      fxbfind, merged_header, 'TTYPE', dum1, TTYPE, dum2, 'null'
      colnames = strlowcase( strtrim(TTYPE,2) )
      x_colnum = 1+where(strlowcase(colnames) EQ 'x')
      y_colnum = 1+where(strlowcase(colnames) EQ 'y')
      
      openw, unit, temp_text_fn, /GET_LUN
      printf, unit, x_colnum, TDMINx, x_colnum, TDMAXx, y_colnum, TDMINy, y_colnum, TDMAXy, strjoin(obs_data.s_filter,','), $
        F='(%"#add\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f\nTDMIN%d=%0.2f\nTDMAX%d=%0.2f\nS_FILTER=\"...\"\nS_FILTER=\"%s\"")'
                ; S_FILTER is assigned twice above to work around a dmhedit bug (in CIAO 4.3) that converts a whitespace value to the integer zero.
      free_lun, unit
      
      cmd = string(merged_env_events_fn, temp_text_fn, F="(%'dmhedit infile=%s filelist=%s')")
      run_command, cmd, /QUIET
    endif ; Modify neighborhood.evt header.


    ;; ------------------------------------------------------------------------
    ;; Construct composite region file showing extraction polygons for all observations.
    ;; ------------------------------------------------------------------------
    ;  The code below cannot handle multi-component extraction regions, e.g. the form polygon() - polygon()
    ;  that is often used for DIFFUSE extractions. 
    if ~is_diffuse then begin
      ; Open region file.
      ; CIAO filtering has very odd quirks.  It seems to produce fewer errors if
      ; we include the DS9 header comment line in the region file.
      openw,  region1_unit, merged_region_fn, /GET_LUN
      printf, region1_unit, "# Region file format: DS9 version 3.0"
      printf, region1_unit, 'global width=1 font="helvetica 12 normal"'
      
      for jj = 0, num_obs-1 do begin
        theader = headfits(obs_data[jj].env_events_fn, EXT=1, ERRMSG=error )
        if (keyword_set(error)) then begin
          print, error
          message, 'ERROR reading ' + obs_data[jj].env_events_fn
        endif
        
        ; Build astrometic structure from data header.
        fxbfind, theader, 'TTYPE', dum1, TTYPE, dum2, 'null'
        fxbfind, theader, 'TCTYP', dum1, TCTYP, dum2, 'null'
        fxbfind, theader, 'TCRVL', dum1, TCRVL, dum2, 0.0D
        fxbfind, theader, 'TCRPX', dum1, TCRPX, dum2, 0.0D
        fxbfind, theader, 'TCDLT', dum1, TCDLT, dum2, 0.0D
        colnames = strlowcase( strtrim(TTYPE,2) )
        x_ind    = where(strlowcase(colnames) EQ 'x')
        y_ind    = where(strlowcase(colnames) EQ 'y')
        make_astr, event2wcs_astr, DELTA=TCDLT[[x_ind,y_ind]], CTYPE=TCTYP[[x_ind,y_ind]], $
                                   CRPIX=TCRPX[[x_ind,y_ind]], CRVAL=TCRVL[[x_ind,y_ind]]
  
        
        ;; Convert polygon to WCS & write to the region file.      
        ;; REMEMBER THAT THE xy2ad and ad2xy programs assume that (x,y) are 
        ;; ZERO-BASED pixel indexes.  Thus we must subtract 1 from the sky (x,y) 
        ;; positions when converting to RA,DEC.
        ae_ds9_to_ciao_regionfile, obs_data[jj].obs_dir + src_region_basename, '/dev/null', $
                                   /IGNORE_BACKGROUND_TAG, POLYGON_X=polygon_x, POLYGON_Y=polygon_y
        
        xy2ad, polygon_x-1, polygon_y-1, event2wcs_astr, polygon_ra, polygon_dec
        
        polygon = dblarr(2,n_elements(polygon_ra))
        polygon[0,*] = polygon_ra
        polygon[1,*] = polygon_dec
    
        src_region = 'polygon(' + strcompress(strjoin(string(polygon,F='(F10.6)'),","), /REMOVE) + ')'
        
        ; Assign a color to the obsname.  The first element of obsname_list is ''.
  ;      obs_index = where(obs_data[jj].obsname EQ obsname_list, count)
  ;      if (count EQ 0) then begin
  ;        ; Append the obsname, and point to it.
  ;        obsname_list = [obsname_list,obs_data[jj].obsname]
  ;        obs_index    = n_elements(obsname_list)-1
  ;      endif
  ;      color = region_colors[(obs_index-1) mod n_elements(region_colors)]
  
        color = 'green'
        printf, region1_unit, src_region, obs_data[jj].obsname, color, F='(%"J2000;%s # tag={%s} color=%s")' 
      endfor ; loop over neighborhood event files
      free_lun, region1_unit
    endif ; ~is_diffuse

    

MERGE_SPECTRA:
    ;; ------------------------------------------------------------------------
    ;; MERGE SOURCE AND BACKGROUND SPECTRA
    ;; ------------------------------------------------------------------------
    ;; Process the component source and background spectra.

    ;; ------------------------------------------------------------------------
    ;; ESTIMATE POSITION UNCERTAINTY
    ;; ------------------------------------------------------------------------
    if (~spectra_available) then begin
      ; If spectra are not available, then we can't make the ae_photometry calls below..
      GOTO, MERGE_RESPONSES
    endif
    
    ;; Estimate the standard deviation (uncertainty) on the mean data position computed earlier.
    ;; The energy range is controlled by the ENERGY_RANGE keyword.

    ;; Estimating this position error is a bit tricky.  
    ;; For N observations, the composite event list is a MIXTURE of data from 2N
    ;; distributions, two from each obsid: 
    ;; * A PSF (truncated by the extraction region).
    ;; * A flat background footprint (truncated by the extraction region).
    ;;
    ;; The distribution of the mixture, p(x), is a weighted sum of these distributions:
    ;;   p(x) = w1*p1(x) + w2*p2(x) + w3*p3(x) + ...
    ;; where the weights sum to 1.0.
    ;;
    ;; We use single-obsid photometry for each obsid in the specified energy band to get
    ;; these weights.
    ;;
    ;; The variance of p(x), VARp, can be written as
    ;;   VARp = Ep[X^2] - u^2
    ;; where Ep[.] is the expectation operator using p(x), and u is the mean of p(x),
    ;; rather than in the usual form Ep[(X-u)^2].
    ;;
    ;; These expectation operators deal nicely with p(x) as a linear combination of distributions:
    ;;   VARp = w1*Ep1[X^2] + w2*Ep2[X^2] + ... - u^2
    ;;
    ;; The mean of p(x), u, is just the weighted sum of the individual means:
    ;;   u = w1*u1 + w2*u2 + ...
    ;;
    ;; IF we had earlier computed and saved the Ep?[X^2] and u? quantities for each observation 
    ;; (in a common coordinate system, .e.g centered on the source position) then we would
    ;; compute VARp as shown above. 
    ;;
    ;; However, for now we have (in obs.psffrac) only the standard deviations without the
    ;; means.  So, here we will make the simplifying assumption that the means are identical.
    ;; Then we can write the variance in the standard way:
    ;;   VARp = Ep[(X-u)^2]
    ;;        = w1*Ep1[(X-u)^2] + w2*Ep2[(X-u)^2] + ...
    ;;        = w1*VARp1        + w2*VARp2        + ...    
    
    ae_photometry, energy_range[0], energy_range[1], photometry, x_distribution_variance, y_distribution_variance
           
            
    ; The position error calculations below must be skipped when any of the terms are not available.
    if photometry.SRC_CNTS EQ 0 || (ra_data  EQ 0) || (x_distribution_variance LE 0)  ||$
                                   (dec_data EQ 0) || (y_distribution_variance LE 0)  then begin
      position_error = !VALUES.F_NAN
      expos_data     = !VALUES.F_NAN
      eypos_data     = !VALUES.F_NAN
    endif else begin
      ; Compute the "standard error of the mean" by dividing by sqrt(N).
      ; That formula is asymptotically correct as N gets large.  I don't know what to do for small N.
      ; We do NOT use the Student's T distribution for this calculation.  The Student's T is applied in the case where the variance of the parent distribution is (poorly) estimated from the N data points. In our case we've estimated the variance of the parent distribution of the composite event data.
      expos_data = sqrt(x_distribution_variance) / SQRT(photometry.SRC_CNTS)
      eypos_data = sqrt(y_distribution_variance) / SQRT(photometry.SRC_CNTS)
   
      ; For N=100 counts, this "standard error of the mean" does indeed enclose 68% of the probability, 
      ; even when the parent distribution is not Gaussian, as can be demonstrated by the simulation below:

;  .run
;  ; Parent distribution is uniform over [0:1].
;  large_parent_sample = random(1E7)
;  dataset_1d, id0, large_parent_sample
;  
;  parent_sigma = stddev(large_parent_sample)
;  parent_mean  = 0.5
;  
;  help, parent_mean, parent_sigma
;  
;  N_counts = 100
;  N_trials = 10000.
;  
;  sample_mean = fltarr(N_trials)
;  for ii=0,N_trials-1 do  sample_mean[ii] = mean(random(N_counts))
;  
;  dataset_1d, id, sample_mean
;  print, stddev(sample_mean)
;  
;  standard_error_of_the_mean = parent_sigma / sqrt(N_counts)
;  
;  print, standard_error_of_the_mean
;  print, total(/INT, abs(parent_mean - sample_mean) LT standard_error) / N_trials
;  end
      
      
      ; These X and Y position errors describe an "error ellipse" around our estimated position.
      ; It's convenient and apparently traditional to approximate this ellipse by an
      ; "error circle" whose radius is the root mean square of the X & Y standard deviations, 
      ; often called a "dRMS" or "1DRMS" error (short for "1 deviation RMS").
      ; The integral of the error ellipse inside this dRMS radius (i.e. significance of this
      ; circular confidence region) varies from 63% for equal errors in X & Y (a circular ellipse)
      ; to 68% for a highly eccentric ellipse.
      
      ; I don't know a good astronomical reference for this practice, but see:
     
      ; Section 2.2 of "A Standardized Algorithm for the Determination of Position Errors 
      ; by the Example of GPS with and without 'Selective Availability'" 
      ; Ingo Harre, published in 'The International Hydrographic Review', Vol. 2, No. 1 (New Series), June 2001
      ; http://www.mar-it.de/NavGen/final_text3.pdf
           
      ; Section 4-10 of ``Engineering and Design - Hydrographic Surveying'', engineering manual, US Army Corps of Engineers
      ; http://publications.usace.army.mil/publications/eng-manuals/em1110-2-1003/
  
      ; Section 4.5 of``Engineering and Design: Navstar Global Positioning System Surveying'', engineering manual, US Army Corps of Engineers
      ; http://publications.usace.army.mil/publications/eng-manuals/EM_1110-1-1003_2011Feb28/}
  
      ; Section 2.4.3 of Principles and Practice of GPS Surveying
      ; http://www.gmat.unsw.edu.au/snap/gps/gps_survey/chap2/243.htm
  
  
      position_error = sqrt( expos_data^2 + eypos_data^2 ) 
      
    endelse
    ; And we'll also convert from skypixel to arcsec for the convenience of the source table generator.
    psb_xaddpar, src_stats, 'ERX_DATA', expos_data     * arcsec_per_skypixel, '[arcsec] 1-sigma uncertainty around (RA_DATA,DEC_DATA)', F='(F10.3)'              
    psb_xaddpar, src_stats, 'ERY_DATA', eypos_data     * arcsec_per_skypixel, '[arcsec] 1-sigma uncertainty around (RA_DATA,DEC_DATA)', F='(F10.3)'
    psb_xaddpar, src_stats, 'ERR_DATA', position_error * arcsec_per_skypixel, '[arcsec] 63% error circle around (RA_DATA,DEC_DATA), aka dRMS error', F='(F10.3)'


    
    ;; ------------------------------------------------------------------------
    ;; CREATE COMPOSITE SPECTRA AND RESPONSES

    ;; NOTE that to support two approaches to spectral fitting---simultaneously modeling of the background spectrum, 
    ;; and traditional background subtraction---we are storing the background scaling information in the AREASCAL column, 
    ;; rather than the traditional BACKSCAL column.  Section 2.3 of the XSPEC manual shows that AREASCAL will scale the
    ;; background spectrum in both styles of analysis.
    ;;
    ;; Note that this trick---making a single background spectral file that can either be declared as the BACKFILE to
    ;; the source spectrum, OR can be loaded as a second observed spectrum for simultaneous fitting---can only be done
    ;; because we are using the convention that both BACKSCAL=1 and AREASCAL=1 in the source spectrum.

    ;; We are careful to set both src & bkg EXPOSURE keywords to the same total exposure so that XSPEC
    ;; will not do any more scaling of the background beyond what we've specified via AREASCAL.
    ;; ------------------------------------------------------------------------
MERGE_RESPONSES:
    file_delete, /ALLOW_NONEXISTENT, merged_src_spectrum_fn, merged_bkg_spectrum_fn, merged_rmf_fn, merged_arf_fn, photometry_fn 
    
    if (skip_spectra) then begin
      if NOT keyword_set(skip_spectra_p) then print, F='(%"\nWARNING!  Spectra or responses missing; not merged.")'
      GOTO, MERGE_TIMING
    endif
    
    ; Retrieve the multi-obsid spectra for all channels.
    ae_photometry, SRC_CHANNELS=src_channels, SRC_CNTS_spectrum=SRC_CNTS_spectrum, BKG_CNTS_spectrum=BKG_CNTS_spectrum
    num_channels = n_elements(src_channels)

    ;; ------------------------------------------------------------------------
    ;; Build the FITS header parts that are common to src & bkg.
    theader = 0   &   dum=temporary(theader)
    ; WE DO NOT RELY ON the EXPOSURE keyword value in source.evt, because the dmmerge command does not correctly calculate EXPOSURE when ObsIDs from different aimpoints are combined!!!
    psb_xaddpar, theader, 'EXPOSURE', total_exposure, '[s] total exposure in merged ObsIDs'
    psb_xaddpar, theader, 'EXTNAME' ,  'SPECTRUM'
    psb_xaddpar, theader, 'CORRFILE',  'none'
    psb_xaddpar, theader, 'CORRSCAL',  1.0
    psb_xaddpar, theader, 'RESPFILE',  merged_rmf_basename
    psb_xaddpar, theader, 'ANCRFILE',  merged_arf_basename
    psb_xaddpar, theader, 'HDUCLASS',  'OGIP'
    psb_xaddpar, theader, 'HDUCLAS1',  'SPECTRUM'
    psb_xaddpar, theader, 'HDUVERS' ,   '1.2.0'
    psb_xaddpar, theader, 'POISSERR', 'T' 
    psb_xaddpar, theader, 'SYS_ERR' ,  0  
    psb_xaddpar, theader, 'QUALITY' ,  0  
    psb_xaddpar, theader, 'DETCHANS', num_channels  
    psb_xaddpar, theader, 'OBS_ID'  ,   strjoin(obs_data.obsid,',')
    psb_xaddpar, theader, 'GROUPING', 0
    psb_xaddpar, theader, 'CREATOR' , creator_string  
    psb_xaddpar, theader, 'S_FILTER', strjoin(obs_data.s_filter,','), 'STATUS filter applied by AE'
    
    keynames = ['TELESCOP','INSTRUME','FILTER','CHANTYPE','OBJECT']
    for jj=0,n_elements(keynames)-1 do begin
      psb_xaddpar, theader, keynames[jj], psb_xpar( src_header, keynames[jj])
    endfor
    
    
    ;; ------------------------------------------------------------------------
    ;; Write the source spectrum.
    fxhmake,  pheader, /INITIALIZE, /EXTEND, /DATE
    psb_xaddpar, pheader, 'CREATOR', creator_string
    psb_xaddpar, pheader, "FNFITS", merged_src_spectrum_fn

    writefits, merged_src_spectrum_fn, 0, pheader
    
    row = { CHANNEL: 0, COUNTS: 0L }
    bin_table = replicate(row, num_channels)
    bin_table.CHANNEL = src_channels 
    bin_table.COUNTS  = SRC_CNTS_spectrum
    
    src_theader = theader
    psb_xaddpar, src_theader, 'BACKFILE',  merged_bkg_spectrum_basename
    psb_xaddpar, src_theader, 'BACKSCAL',  1.0, 'background scaling completely specified in BACKFILE'
    psb_xaddpar, src_theader, 'AREASCAL',  1.0, 'background scaling completely specified in BACKFILE'
    psb_xaddpar, src_theader, 'HDUCLAS2',  'TOTAL'
    psb_xaddpar, src_theader, 'HDUCLAS3',  'COUNT'
    mwrfits, bin_table, merged_src_spectrum_fn, src_theader
    if (verbose GT 0) then print, 'Src spectrum: ', merge_subdir[ii]+merged_src_spectrum_basename
    
    
    ;; ------------------------------------------------------------------------
    ;; Write the background spectrum.
    ;; The single (not energy-dependent) background scaling we will use for this multi-obsid spectrum is that which produces the correct photometry over ENERGY_RANGE.
    ae_photometry, energy_range[0], energy_range[1], photometry
    
    fxhmake, pheader, /INITIALIZE, /EXTEND, /DATE
    psb_xaddpar, pheader, 'CREATOR', creator_string
    psb_xaddpar, pheader, "FNFITS", merged_bkg_spectrum_fn

    writefits, merged_bkg_spectrum_fn, 0, pheader
    
    row = { CHANNEL: 0, COUNTS: 0L}
    bin_table = replicate(row, num_channels)
    bin_table.CHANNEL = src_channels 
    bin_table.COUNTS  = BKG_CNTS_spectrum

    bkg_theader = theader
    psb_xaddpar, bkg_theader, 'BACKFILE',  'none'
    psb_xaddpar, bkg_theader, 'BACKSCAL',  1.0,                 'background scaling specified by AREASCAL'
    psb_xaddpar, bkg_theader, 'AREASCAL',  photometry.BACKSCAL, 'background scaling'
    psb_xaddpar, bkg_theader, 'HDUCLAS2',  'BKG'
    psb_xaddpar, bkg_theader, 'HDUCLAS3',  'COUNT'
    mwrfits, bin_table, merged_bkg_spectrum_fn, bkg_theader
    if (verbose GT 0) then print, 'Bkg spectrum: ', merge_subdir[ii]+merged_bkg_spectrum_basename
    
    
    ;; ------------------------------------------------------------------------
    ;; Compare the shapes of the source & background spectra over ENERGY_RANGE using a 2-sample Kolmogorov-Smirnov statistic.
    band_index = where((photometry.CHAN_LO  LE src_channels)   AND $
                       (src_channels        LE photometry.CHAN_HI))
    
    inband_SRC_CNTS_spectrum = SRC_CNTS_spectrum[band_index]
    inband_BKG_CNTS_spectrum = BKG_CNTS_spectrum[band_index]
    
    total_src_observed_counts = total(inband_SRC_CNTS_spectrum)
    total_bkg_observed_counts = total(inband_BKG_CNTS_spectrum)
    src_cum_distn             = total(inband_SRC_CNTS_spectrum, /CUMULATIVE) / total_src_observed_counts
    bkg_cum_distn             = total(inband_BKG_CNTS_spectrum, /CUMULATIVE) / total_bkg_observed_counts

    n_eff = (total_src_observed_counts*total_bkg_observed_counts) / float(total_src_observed_counts + total_bkg_observed_counts)

    ks = max( abs( src_cum_distn - bkg_cum_distn ) )
    prob_ks, ks, n_eff, ks_spect


    ;; ------------------------------------------------------------------------
    ;; The final RMF is a weighted average of the constituent RMFs.
    ;; The final ARF is a weighted average of the constituent ARFs.
    ;; We run the addrmf,addarf commands using an ASCII list file because the command line
    ;; parameters seem to have limiting length restrictions.
    rmf_fn        = strmid(obs_data.rmf_fn, strlen(basedir))
    arf_fn        = strmid(obs_data.arf_fn, strlen(basedir))
    rmf_result_fn = strmid(merged_rmf_fn,   strlen(basedir))
    arf_result_fn = strmid(merged_arf_fn,   strlen(basedir))

    ; In the weighted average used to combine ARFs, the weighting factor must be EXPOSURE time,
    ; NOT some other statistic!!!
    ; I think that stems from the fact that EXPOSURE keywords are summed in the merge.
    ; Suppose two extractions with exposure E1, E1 and effective area A1,A2 are merged.
    ; The "grasp" (s cm**2 count /photon) of the merged observations is E1*A1 + E2*A2.
    ; But the convention is to store that merged grasp in two separate terms:
    ;   E_merged (in header of merged spectrum) = E1 + E2
    ;   A_merged (in ARF for merged spectrum)   = E1*A1/(E1 + E2) + E2*A2/(E1 + E2)
    ; Then E_merged * A_merged = E1*A1 + E2*A2
    
    exposure_fraction = obs_data.src_exposure / total_exposure

    ; Use the supplied generic RMF, or link to a single obsid's RMF, or average multiple RMFs.
    use_longest_observation = 0
    dum = max(obs_data.src_exposure,imax)

    if keyword_set(generic_rmf_fn) then begin
      ; Don't create any merged RMF file.
    endif else if (num_obs EQ 1) then begin
      ; We use a symbolic link to avoid copying the single RMF & ARF.
      path = named2generic_extraction_path+rmf_fn
      file_link, path, basedir+rmf_result_fn
      if (verbose GT 0) then print, rmf_result_fn, path, F='(%"              %s --> %s")' 
    endif else begin
      ; Use addrmf to combine the RMFs weighting by exposure_fraction.
      forprint, rmf_fn, exposure_fraction, F='(A0,1x,G0.3)', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
      cmd = string( temp_text_fn, rmf_result_fn, F="(%'addrmf ""@%s"" rmffile=%s')")

      run_command, /HEASOFT, DIRECTORY=basedir, cmd, STATUS=status
      
      if keyword_set(status) then begin
        ; Sometimes the RMFs don't have the same structure and addrmf fails.
        ; In this case we simply link to the primary obsid's RMF.
        print, 'WARNING: addrmf call failed, probably because we are trying to combine RMFs with different energy bins.'
        print, 'WARNING: we will use the RMF & ARF from the longest observation.'
        file_delete, merged_rmf_fn, /ALLOW_NONEXISTENT
        path = named2generic_extraction_path+rmf_fn[imax]
        file_link, path, basedir+rmf_result_fn
        if (verbose GT 0) then print, rmf_result_fn, path, F='(%"              %s --> %s")' 
        use_longest_observation = 1
      endif 
    endelse ;(num_obs GT 1)
    
    
    ; Link to a single ARF, or link to the longest ARF if we had an addrmf failure above, or average multiple ARFs.
    if (num_obs EQ 1) then begin
      ; We use a symbolic link to avoid copying the single  ARF.
      path = named2generic_extraction_path+arf_fn
      file_link, path, basedir+arf_result_fn
      if (verbose GT 0) then print, arf_result_fn, path, F='(%"              %s --> %s")' 
    endif else if use_longest_observation then begin
      ;!!!!!! As of March 05, when addrmf fails above due to mismatched energy bins then addarf will fail with "ERROR: Energy bins mismatch", but will NOT return a non-zero exit status.  Thus, we must use the flag "use_longest_observation" to signal that we must link to the longest observation's ARF here rather than calling addarf which would likely suffer an undetected failure.
      path = named2generic_extraction_path+arf_fn[imax]
      file_link, path, basedir+arf_result_fn
      if (verbose GT 0) then print, arf_result_fn, path, F='(%"              %s --> %s")'
    endif else begin
      ; Use addarf to combine the ARFs weighting by exposure_fraction.
      forprint, arf_fn, exposure_fraction, F='(A0,1x,G0.3)', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
      cmd = string( temp_text_fn, arf_result_fn, F="(%'addarf ""@%s"" out_ARF=%s')")

      run_command, /HEASOFT, DIRECTORY=basedir, cmd, STATUS=status 

      if keyword_set(status) then begin
        print, 'WARNING: addrmf call failed, probably because we are trying to combine ARFs with different energy bins.'
        print, 'WARNING: we will use the ARF from the longest observation.'
        file_delete, merged_arf_fn, /ALLOW_NONEXISTENT
        path = named2generic_extraction_path+arf_fn[imax]
        file_link, path, basedir+arf_result_fn
        if (verbose GT 0) then print, arf_result_fn, path, F='(%"              %s --> %s")'
      endif else if is_diffuse then begin
        ; When addarf ran ok and this is a diffuse source, we want to fix the units of SPECRESP.
        run_command,  DIRECTORY=basedir, string(arf_result_fn, 'arcsec**2 cm**2 count /photon',$
                 F="(%'dmhedit infile=""%s[2]"" filelist=none operation=add key=TUNIT3 value=""\'%s\'"" comment=""surface brightness ARF""')")
      endif
    endelse
    
    
    ;; ------------------------------------------------------------------------
    ;; COMPUTE PHOTOMETRY
    ;; ------------------------------------------------------------------------

    ;; ------------------------------------------------------------------------
    ;; Integrate the spectrum and ARF over a variety of interesting bands and write a photometry table
    
    ; Load the merged response files into the ae_photometry object.
    ae_photometry, RMF_FN=keyword_set(generic_rmf_fn) ? generic_rmf_fn : merged_rmf_fn, ARF_FN=merged_arf_fn 

    flux_table = replicate( photometry, n_elements(eband_lo) )
    for jj = 0, n_elements(flux_table)-1 do begin
      ae_photometry, eband_lo[jj], eband_hi[jj], photometry
      
      flux_table[jj] = photometry
    endfor ;jj, looping over energy bands in flux table
    


    fxhmake,  header, /INITIALIZE, /EXTEND, /DATE
    psb_xaddpar, header     , 'CREATOR', creator_string
    writefits, photometry_fn, 0, header
    
    ae_photometry, TABLE_HEADER=table_header, DIFFUSE=is_diffuse
    psb_xaddpar, table_header, 'CREATOR', creator_string
    mwrfits, flux_table, photometry_fn, table_header
    if (verbose GT 0) then print, 'Photometry,Pb: ', merge_subdir[ii]+src_photometry_basename

    
    ;; ------------------------------------------------------------------------
    ;; Save summary information about merged spectra.
    ; Recall that the values in the vector obs_data.bkg_backscal are integrals of the exposure map over each bkg region.
    ; Recall that this exposure map integral has units of (skypixel**2 s cm**2 count /photon),
    if bkg_spectra_available then begin
      psb_xaddpar, src_stats, 'BACKGRND', total_bkg_observed_counts / total(obs_data.bkg_backscal) , $
                       string(energy_range, F="(%'[photon /cm**2 /s /skypixel**2] background SB, %0.1f:%0.1f keV, merged observations')")
  
      area_exposure_ratio = float(obs_data.src_backscal)/obs_data.bkg_backscal
      psb_xaddpar, src_stats, 'SCAL_MAX', 1. / min(area_exposure_ratio), 'largest bkg scaling among merged observations'
      psb_xaddpar, src_stats, 'SCAL_MIN', 1. / max(area_exposure_ratio), 'smallest bkg scaling among merged observations'
      psb_xaddpar, src_stats, 'KS_SPECT', ks_spect, string(energy_range, F="(%'p-value, src/bkg spectra, %0.1f:%0.1f keV')")
    endif else sxdelpar, src_stats, [ 'BACKGRND', 'SCAL_MAX', 'SCAL_MIN', 'KS_SPECT']
    
    psb_xaddpar, src_stats, 'EXPOSURE',  total_exposure, '[s] total exposure in merged observations'

    ae_photometry, nominal_psf_energy, nominal_psf_energy, photometry

    psb_xaddpar, src_stats, 'EFFAREA',  photometry.MEAN_ARF, $
                                     string(nominal_psf_energy, F='(%"[cm**2 count /photon] ARF value @%6.4f keV")')  
       
    
    
MERGE_TIMING:    
    ;; ------------------------------------------------------------------------
    ;; MERGE TIMING INFORMATION
    ;; ------------------------------------------------------------------------
    file_delete, /ALLOW_NONEXISTENT, merged_stacked_lc_fn, merged_sequenced_lc_fn
    
    if (skip_timing) then begin
      if NOT keyword_set(skip_timing_p) then print, F='(%"\nWARNING!  Lightcurves missing; not merged.")'
      GOTO, MERGE_SAVE_STATS
    endif

    ;; Merge the light curve tables and the corresponding GTI tables.
    ;; At some point, dmmerge could not handle the long column name "grp_median_energy_err" so below we had to temporarily rename it, do the merge, and restore its name.
    forprint, obs_data.lc_smooth_fn, F='(%"%s")', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
    cmd = string( temp_text_fn, temp_lc_fn, $
                  F="(%'dmmerge ""@-%s[cols *,gme_err=grp_median_energy_err]"" columnList="""" outfile=%s  ')")
    run_command, cmd

    run_command, /QUIET, string( temp_lc_fn, merged_lc_smooth_fn, F="(%'dmcopy ""%s[cols *,grp_median_energy_err=gme_err]"" %s ')")



    ;; Gather the single-obsid light curves that we wish to plot.
    ;; The first num_obs pointers are for the binned & grouped light curves.
    ;; The "time" values are the centers of the bins.
    ;; The second num_obs pointers are for the smoothed light curves.
    times                = ptrarr(2*num_obs)
    halfbin_sizes        = ptrarr(2*num_obs)
    rates                = ptrarr(2*num_obs)
    rate_errors          = ptrarr(2*num_obs)
    median_energies      = ptrarr(2*num_obs)
    median_energy_errors = ptrarr(2*num_obs)
    tstart          = dblarr(num_obs)
    tstop           = dblarr(num_obs)
    variability_probs = fltarr(num_obs)
    rate_span  = 0
    min_energy = !VALUES.F_INFINITY
    max_energy = 0
    abutted_exposure = 0
    abutted_counts   = 0
    num_groups_all_obs = 0L

    ; Even pointers are binned LC, odd are smooth LC.
    for jj = 0, num_obs-1 do begin
      lc_fn = obs_data[jj].lc_smooth_fn
      if (NOT file_test(lc_fn)) then continue

      tb = mrdfits(lc_fn, 1, lc_header, /SILENT, STATUS=status)
      if (status NE 0) then message, 'ERROR reading ' + lc_fn
      
      variability_probs[jj] = psb_xpar( lc_header, 'PROB_KS')
      
      tstart[jj] = min(tb.TIME_MIN)
      tstop [jj] = max(tb.TIME_MAX)
      
      ; Extract the binned time series and integrate over the groups.
      ; Recall from TIMING stage that the EXPOSURE column has units of "s cm**2 count /photon",
      ; and accounts for effective area, FRACEXPO, and the finite aperture.
      counts      = tb.COUNTS
      exposure    = tb.EXPOSURE
      group_codes = tb.GROUPING
      
      abutted_counts   = [abutted_counts,   counts]
      abutted_exposure = [abutted_exposure, exposure]
      
;help, counts, exposure, group_codes
      ; Integrate counts and exposure over each group.
      ind = [where(group_codes EQ 1, num_groups), n_elements(tb)]
      
      num_groups_all_obs = num_groups_all_obs + num_groups

      time           = dblarr(num_groups)
      halfbin_size   = fltarr(num_groups)
      group_counts   = fltarr(num_groups)
      group_exposure = fltarr(num_groups)
      median_energy       = fltarr(num_groups)
      median_energy_error = fltarr(num_groups)
      
      for kk=0,num_groups-1 do begin
        ind_left  = ind[kk]
        ind_right = ind[kk+1] - 1
        
        time_left          = ((tb.time)[ind_left]  - tstart[jj]) / 1000.  ; to get ks offset from tstart
        time_right         = ((tb.time)[ind_right] - tstart[jj]) / 1000.  ; to get ks offset from tstart
        time          [kk] = mean([time_left,time_right])
        halfbin_size  [kk] = 0.5 * (time_right - time_left)
  
        group_counts  [kk] = total(counts  [ind_left : ind_right])
        group_exposure[kk] = total(exposure[ind_left : ind_right])
        median_energy      [kk] = tb[ind_left].GRP_MEDIAN_ENERGY      / 1000.  ; to get keV 
        median_energy_error[kk] = tb[ind_left].GRP_MEDIAN_ENERGY_ERR  / 1000.  ; to get keV
      endfor ;kk

      aa = total(counts) & bb = total(group_counts)
      if (abs(aa-bb)/aa GT 0.01) then message, 'ERROR: looks like a code bug'
      
      aa = total(exposure) & bb = total(group_exposure)
      if (abs(aa-bb)/aa GT 0.01) then message, 'ERROR: looks like a code bug'
      
;help, time, group_counts, group_exposure, halfbin_size    

      ; Keep track of the span of the data.
      min_energy = min_energy < min([10,median_energy],/NAN)
      max_energy = max_energy > max([ 0,median_energy],/NAN)
      
      ; Change NaN values to zero in median_energy_error to keep oploterror happy later.
      ind = where(finite(median_energy_error) EQ 0, num_nan)
      if (num_nan GT 0) then median_energy_error[ind] = 0

      rate  =      group_counts  / group_exposure * 1000. ; photon /ks /cm**2
      error = sqrt(group_counts) / group_exposure * 1000. ; photon /ks /cm**2
      times               [jj] = ptr_new(time)
      halfbin_sizes       [jj] = ptr_new(halfbin_size)
      rates               [jj] = ptr_new(rate)
      rate_errors         [jj] = ptr_new(error)
      median_energies     [jj] = ptr_new(median_energy)      
      median_energy_errors[jj] = ptr_new(median_energy_error)      
      rate_span = rate_span > max([0,rate+error],/NAN)

;save, /COMPRESS, time, rate, error, FILE=sourcedir+'lightcurve.sav'

      ; Extract the smoothed time series information.  Convert time to ks relative to TSTART.
      time                = (tb.TIME - tstart[jj]) / 1000.  ; to get ks
      rate                =  tb.COUNT_RATE         * 1000.
      error               =  tb.RATE_ERR           * 1000.
      median_energy       =  tb.GRP_MEDIAN_ENERGY      / 1000.  ; to get keV
      median_energy_error =  tb.GRP_MEDIAN_ENERGY_ERR  / 1000.  ; to get keV

      ; Keep track of the span of the data.
      rate_span  = rate_span > max([0,rate+error],/NAN)

      ; Trim off the NaN tails.
      ind = where(finite(rate), count)
      if (count EQ 0) then begin
        count = 1
        ind   = [0]
      endif
      first_index = ind[0]
      last_index  = ind[count-1]
      
      time                = time               [first_index:last_index]
      rate                = rate               [first_index:last_index]
      error               = error              [first_index:last_index]
      median_energy       = median_energy      [first_index:last_index]
      median_energy_error = median_energy_error[first_index:last_index]
;help, time, rate, error, median_energy     

      times               [num_obs+jj] = ptr_new(time)
      rates               [num_obs+jj] = ptr_new(rate)
      rate_errors         [num_obs+jj] = ptr_new(error)
     ; The plotting code below is set up to plot smoothed median energy curves, but smoothed median energy values are not currently computed earlier in the AE workflow.
     ;median_energies     [num_obs+jj] = ptr_new(median_energy      )
     ;median_energy_errors[num_obs+jj] = ptr_new(median_energy_error)
    endfor ;jj
    

    ;; ------------------------------------------------------------------------
    ;; Use the K-S statistic to assess variability in the composite data. 
    ;  The model assumes a constant photon flux from the source; abutted_exposure is supposed to account for
    ;  each ObsID's intregration time, effective area, FRACEXPO, and the finite aperture.
    ;
    ;  WARNING!  This analysis knows nothing about backgrounds---all counts in the apertures are assumed to
    ;  come from the source.  
    ;  Varying background levels (e.g. due to varying off-axis angles) can mimick source variability.
    ;
    ;  WARNING!  MERGE_KS probably depends on the order in which ObsIDs are abutted, which is currently based on obsname (not on observing date).
    ;
    ; The MERG_CHI metric accounts for background, and may be superior to this metric. 
    
    uniform_flux_model = total(abutted_exposure, /DOUBLE, /CUMULATIVE) / total(abutted_exposure, /DOUBLE)
    inband_src_counts  = total(abutted_counts,   /DOUBLE)
    cum_count_distn    = total(abutted_counts,   /DOUBLE, /CUMULATIVE) / inband_src_counts
   
    if (n_elements(uniform_flux_model) NE n_elements(cum_count_distn)) then message, 'BUG!'
    
    ; Keep in mind that the time bins we're using here were constructed (in /TIMING) 
    ; so that events fall on the bin boundaries.  The event is counted (abutted_counts) in 
    ; the bin to the LEFT of the boundary.
    ; Thus the stair steps in the cumulative count distribution occur at bin boundaries.
    ; At each of those steps, the K-S test requires that we compute two distances:
    ; 1. From the model distribution to the base of the stair step.
    ;    This is done by comparing uniform_flux_model[i] to cum_count_distn[i-1].
    ; 2. From the model distribution to the top of the stair step.
    ;    This is done by comparing uniform_flux_model[i] to cum_count_distn[i].
    ks_distance = max( abs( uniform_flux_model - [0,cum_count_distn] ) ) > $
                  max( abs( uniform_flux_model -    cum_count_distn  ) )
    
    if (inband_src_counts LT 4) then begin
      ; We need at least 4 counts for KS probability to be meaningful.
      probks = !VALUES.F_NAN
      print, 'KS variability analysis skipped -- too few in-band counts.'
    endif else prob_ks, ks_distance, inband_src_counts, probks
      
    psb_xaddpar, src_stats, 'MERGE_KS',  probks, 'p-value, variability, band defined by TIMING stage'

    if (verbose GE 10) then begin
      function_1d, id_ks, lindgen(n_elements(uniform_flux_model)), uniform_flux_model, DATASET='exposure'
      function_1d, id_ks, lindgen(n_elements(cum_count_distn   )), cum_count_distn   , DATASET='events', XTIT='time bin (variable width)', YTIT='Cumulative Distribution'
    endif

    ;; ------------------------------------------------------------------------
    psb_xaddpar, src_stats, 'MERG_CHI', !VALUES.F_NAN, 'not available'
    if spectra_available && (num_obs GT 1) then begin
      ;; Use the chi-square statistic to assess variability in ObsID-averaged fluxes
      ;; The BIG advantage of this metric over the MERGE_KS metric is that these fluxes are background-subtracted.
      ;.
      ; References include:
      ;   Sokolovsky et. al, MNRAS 464, 274–292 (2017)
      ;   Numerical Recipes, 2nd Edition, Chapter 15 (Modeling of Data), Section 1 (Chi-Square Fitting)
      ;
      ; Let N be the number of flux measurements (number of ObsIDs in this merge).
      ; Our null hypothesis is that the source flux is constant.
      ; Our model for the data is a constant (one parameter), estimated by the weighted mean.
      ; Assuming the flux measurements have Gaussian uncertainties, of the size stated by ae_photometry, then
      ; the chi-square statistic follows the "chi-square distribution for v=N-M=N-1 degrees of freedom".
      ; p-values for that distribution can be computed by the IDL function 1 - CHISQR_PDF(chisquare, v).
      ; As always, small p-values suggest we should 'reject the null hypothesis' (source is constant).
      
      ; Use the ae_photometry tool to compute photon flux for each ObsID.
      for jj = 0, num_obs-1 do begin
        ; The table of extraction quantities should be loaded FIRST, 
        ae_photometry, OBS_DATA=obs_data[jj], SRC_CHANNELS=src_channels
        ; followed by loading the response files
        ae_photometry, RMF_FN=obs_data[jj].rmf_fn, ARF_FN=obs_data[jj].arf_fn 
        ; Then compute photometry values
        ae_photometry, energy_range[0], energy_range[1], photometry
        
        if (jj EQ 0) then flux_table = replicate( photometry, num_obs )
        flux_table[jj] = photometry
      endfor ;jj, looping over ObsIDs in this merge
  
      ; Calculate photon flux and uncertainty from NET_COUNTS.
      ; We do not use NET_CNTS_SIGMA_LOW because it is NaN when SRC_CNTS or BKG_CNTS is zero.
      photon_flux          = flux_table.NET_CNTS          / flux_table.MEAN_ARF / obs_data.src_exposure
      photon_flux_sigma_up = flux_table.NET_CNTS_SIGMA_UP / flux_table.MEAN_ARF / obs_data.src_exposure
;     forprint, obs_data.obsname, photon_flux, photon_flux_sigma_up
      
      ; Calculate a weighted mean flux, as our 1-parameter model for a constant source.
      weight        = 1. / photon_flux_sigma_up^2                      ; Bevington (2nd Edition) 4.17
      weight_total  = total(/DOUBLE, weight) 
      flux_weighted = total(/DOUBLE, weight*photon_flux) / weight_total
      
      ; Calculate the "standard score" or "z-score" for the residuals between the "data" (flux measurement) and model (weighted mean flux).
      Z = (photon_flux - flux_weighted) / photon_flux_sigma_up
      
      ; Calculate chi-square, and it's p-value.
      chisquare = total(/DOUBLE, Z^2 )
      
      dof = num_obs - 1
      p_value = 1 - CHISQR_PDF(chisquare, dof)
      
      psb_xaddpar, src_stats, 'MERG_CHI', p_value, string(energy_range, F="(%'p-value, variability, %0.1f:%0.1f keV')")
    endif ; spectra_available
    
    
    ;; ------------------------------------------------------------------------
    ;; Make a plot for each obsid, stacked on the same page.
    if (num_groups_all_obs LE 1) then goto, LC_CLEANUP
    ; Put a little margin on the axis ranges.
    time_span       = max(tstop-tstart) / 1000.
    xrange          = [-0.02*time_span, 1.02*time_span]
    rate_axis_range = [0, 1.02*rate_span]

    energy_span = max_energy - min_energy
    if (energy_span LT 1.0) then begin
      min_energy = 0 > (min_energy - 0.5 * (1.0-energy_span))
      max_energy = min_energy + 1.0
    endif
    energy_axis_range = [min_energy, max_energy]
    
    ;; Set up the Postscript page.
    if ~keyword_set(page_long_dim ) then page_long_dim  =  11   ; inches
    if ~keyword_set(page_short_dim) then page_short_dim =  8.5 ; inches
    
    ; Center the "plot region" on the page.
    if (num_obs LE 2) then begin
      xsize = page_long_dim  - 1 ; horizontal inches
      ysize = page_short_dim - 1 ; vertical inches
      xoffset = (page_short_dim - ysize) / 2.
      yoffset =  page_long_dim - (page_long_dim - xsize) / 2.
    endif else begin
      xsize   = page_short_dim - 1
      ysize   = page_long_dim  - 1
      xoffset = (page_short_dim - xsize) / 2.
      yoffset = (page_long_dim  - ysize) / 2.
    endelse
    
    color_manager, /PS_PSEUDO, FILENAME=merged_stacked_lc_fn, LANDSCAPE=0, $
            XOFFSET=xoffset, YOFFSET=yoffset, XSIZE=xsize, YSIZE=ysize, /INCHES, $
            RED=red, BLUE=blue, WHITE=white, /ENCAPSULATED
    !P.FONT =  0   ; "hardware" font
    color_manager, 'orange', orange
    
    !X.THICK=3
    !Y.THICK=3
    xticklen =  0.02*num_obs
    yticklen =  0.01
    charsize = (1 + (num_obs GT 2)) / 1.25

    ; Since we already sorted obs_data by TSTART, the observations will be plotted in time order.
    
    !P.MULTI   = [0,0,num_obs]
    !X.OMARGIN = [0,1]
    !Y.OMARGIN = [1,1]
    for kk = 0, num_obs-1 do begin
      tit    = string(obs_data[kk].obsname, variability_probs[kk], F='(%"ObsID %s (P!DKS!N=%0.2g)")')
      xtit   = ''
      xstyle = 1+4
      ymargin= [0,2]
      ytit   = ''
      ytit2  = ''
      
      if (kk EQ 0) then begin
        tit  = sourcename[ii] + '; ' + tit
      endif
      
      if (kk EQ num_obs/2) then begin
        ytit    = 'Gross (src+bkg) Photon Flux (photon /ks /cm**2)'
        ytit2   = 'Median Energy (keV)'
      endif
      
      if (kk EQ (num_obs-1)) then begin
        xtit    = 'Time (ks)'
        xstyle  = 1+8
        ymargin = [3,2]
      endif
       
      ; Plot the lightcurve axes.
      plot, [0], /NODATA,   TITLE=tit ,  CHARSIZE=charsize, $   
            XMARGIN=[11,6],  XTIT=xtit, XCHARSIZE=1.25, XSTYLE=xstyle, XRANGE=xrange,          XTICKLEN=xticklen, $
            YMARGIN=ymargin, YTIT=ytit, YCHARSIZE=1.25, YSTYLE=1+8,    YRANGE=rate_axis_range, YTICKLEN=yticklen, $
            THICK=3                                                                           
      
      jj = kk
      if ptr_valid(times[jj]) then begin
        ; Plot the binned/grouped lightcurve. We CANNOT USE PSYM=10 with unequal bin sizes!
        num_groups   = n_elements(*times[jj])
        nskip_binned = 1 ;  > floor(num_groups/40.)
        zerovec      = replicate(0,num_groups)
        oploterror, *times[jj], *rates[jj], *halfbin_sizes[jj],          zerovec, COLOR=white, ERRCOLOR=white, PSYM=3, ERRTHICK=3, /NOHAT   
        oploterror, *times[jj], *rates[jj],            zerovec, *rate_errors[jj], COLOR=white, ERRCOLOR=white, PSYM=3, ERRTHICK=3, /NOHAT, NSKIP=nskip_binned   
                                                                                                                         
        if ptr_valid(rates[num_obs+jj]) then begin
          ; Plot the smooth lightcurve
          time       = congrid(*times      [num_obs+jj], num_groups*20)
          rate       = congrid(*rates      [num_obs+jj], num_groups*20)
          rate_error = congrid(*rate_errors[num_obs+jj], num_groups*20)

          ; Since the smooth lightcurves are NaN at the ends (where the kernel falls off the data)
          ; the range of times left in the plot may be too small to be worth showing.
          if ( (max(time)-min(time))/time_span LT 0.001 ) then begin 
;            oploterror, mean(time,/NAN), mean(rate,/NAN), 0.025*time_span, mean(rate_error,/NAN), /NOHAT
          endif else begin
;            oplot, time, rate-rate_error
;            oplot, time, rate+rate_error
;           oploterror, time, rate, rate_error, NSKIP=1 > floor(n_elements(time)/10.), COLOR=??, ERRCOLOR=??, THICK=0.5
            oplot, time, rate+rate_error, COLOR=blue, THICK=0.5
            oplot, time, rate-rate_error, COLOR=blue, THICK=0.5
          endelse
        endif ; ptr_valid(rates[jj])
        
        ; Plot the median energy time series.  We CANNOT USE PSYM=10 with unequal bin sizes!
        !Y.WINDOW = !Y.WINDOW - [0, 0.5*(!Y.WINDOW[1]-!Y.WINDOW[0])]
        axis, YAXIS=1, /SAVE, YRANGE=energy_axis_range, YTIT=ytit2, YTICKLEN=yticklen, COLOR=red, CHARSIZE=charsize, YCHARSIZE=1.25
        
        oploterror, *times[jj], *median_energies[jj], *halfbin_sizes[jj],                   zerovec, COLOR=red, ERRCOLOR=red, PSYM=3, THICK=1, ERRTHICK=1, /NOHAT   
        oploterror, *times[jj], *median_energies[jj],            zerovec, *median_energy_errors[jj], COLOR=red, ERRCOLOR=red, PSYM=3, THICK=1, ERRTHICK=1, /NOHAT, NSKIP=nskip_binned   
        
        if ptr_valid(median_energies[num_obs+jj]) then begin
          ; Plot the smooth Emedian timeseries.
          time                = congrid(*times               [num_obs+jj], num_groups*20)
          median_energy       = congrid(*median_energies     [num_obs+jj], num_groups*20)
          median_energy_error = congrid(*median_energy_errors[num_obs+jj], num_groups*20)
          
          ; Since the smooth timeseries is NaN at the ends (where the kernel falls off the data)
          ; the range of times left in the plot may be too small to be worth showing.
          if ( (max(time)-min(time))/time_span LT 0.001 ) then begin 
;            oploterror, mean(time,/NAN), mean(median_energy,/NAN), 0.025*time_span, mean(median_energy_error,/NAN), /NOHAT
          endif else begin
;            oplot, time, median_energy-median_energy_error
;            oplot, time, median_energy+median_energy_error
;           oploterror, time, median_energy, median_energy_error, NSKIP=1 > floor(n_elements(time)/10.), COLOR=??, ERRCOLOR=??, THICK=0.5
;           oplot, time, median_energy, COLOR=orange, THICK=0.5
          endelse
        endif ; ptr_valid(median_energies[jj])
      endif else begin                                 
      endelse
    endfor ;kk, looping over obsids
        
    device, /close
    !P.MULTI   = 0
    !X.OMARGIN = [0,0]
    !Y.OMARGIN = [0,0]

    ;; ------------------------------------------------------------------------
    ;; Make a single plot with a "broken" time axis showing all the light curves.
    ; Find the time span of all the lightcurves plus breaks in the time axis.
    time_span    = total(tstop-tstart) / 1000.
    break_length = time_span * 0.05/num_obs
    time_span    = time_span + (break_length * (num_obs-1))
    xrange       = [-0.02*time_span, 1.02*time_span]
    
    ; Define position and length of the energy axis wrt the flux axis.
    ; Extend the upper limit of the flux axis as needed to accomodate the specified placement of the energy axis.
    energy_axis_window = [1.0,1.25]  ; endpoints of energy axis, in the normalized coordinates of the flux axis
    rate_axis_range    = [0, (1.02 > energy_axis_window[1]) * rate_span] ; endpoints of flux axis in flux units
    
    energy_axis_window *= rate_span / rate_axis_range[1]


    xsize = page_long_dim  - 1 ; horizontal inches
    ysize = page_short_dim - 1 ; vertical inches
    xoffset = (page_short_dim - ysize) / 2.
    yoffset =  page_long_dim - (page_long_dim - xsize) / 2.
    color_manager, /PS_PSEUDO, FILENAME=merged_sequenced_lc_fn, LANDSCAPE=0, $
            XOFFSET=xoffset, YOFFSET=yoffset, XSIZE=xsize, YSIZE=ysize, /INCHES, $
            RED=red, BLUE=blue, /ENCAPSULATED
    !P.FONT =  0   ; "hardware" font

    ; Plot the lightcurve axes.
    xstyle  = 1
    ymargin = [3,2]
    plot, [0], /NODATA, TITLE=string(sourcename[ii], float(probks), F='(%"%s (P!DKS!N=%0.2g)")'), $
          XMARGIN=[11,6],  XTIT='Accumulated Time (ks) (time-ordered ObsIDs separated by vertical dashed lines)',       XSTYLE=xstyle, XRANGE=xrange,          XTICKLEN=yticklen, $
          YMARGIN=ymargin, YTIT='Gross (src+bkg) Photon Flux (photon /ks /cm**2)', YSTYLE=1+8,    YRANGE=rate_axis_range, YTICKLEN=yticklen, $
          CHARSIZE=1, THICK=3 
    
    num_groups = 0L
    for kk = 0, num_obs-1 do begin
      jj = kk
      if ptr_valid(times[jj]) then num_groups = num_groups + n_elements(*times[jj])
    endfor ;kk, looping over obsids
    nskip_binned = 1 ; > floor(num_groups/40.)
    
;    num_times = 0L
;    for kk = 0, num_obs-1 do begin
;      jj = num_obs+kk
;      if ptr_valid(times[jj]) then num_times = num_times + n_elements(*times[jj])
;    endfor ;kk, looping over obsids
;    nskip_smoothed = 1 > floor(num_times/10.)

    ; Plot the binned and smoothed light curves.
    time_offset = 0.
    for kk = 0, num_obs-1 do begin
      jj = kk
      if ptr_valid(times[jj]) then begin
        ; Plot the binned/grouped lightcurve. We CANNOT USE PSYM=10 with unequal bin sizes!
        num_groups = n_elements(*times[jj])
        zerovec    = replicate(0,num_groups)
        oploterror, *times[jj] + time_offset, *rates[jj], *halfbin_sizes[jj],          zerovec, COLOR=white, ERRCOLOR=white, PSYM=3, ERRTHICK=3, /NOHAT   
        oploterror, *times[jj] + time_offset, *rates[jj],            zerovec, *rate_errors[jj], COLOR=white, ERRCOLOR=white, PSYM=3, ERRTHICK=3, /NOHAT, NSKIP=nskip_binned   
      
        if ptr_valid(rates[num_obs+jj]) then begin
          ; Plot the smooth lightcurve
          time       = congrid(*times      [num_obs+jj] + time_offset, num_groups*20)
          rate       = congrid(*rates      [num_obs+jj]              , num_groups*20)
          rate_error = congrid(*rate_errors[num_obs+jj]              , num_groups*20)
           
          ; Since the smooth lightcurves are NaN at the ends (where the kernel falls off the data)
          ; the range of times left in the plot may be too small to be worth showing.
          if ( (max(time)-min(time))/time_span LT 0.001 ) then begin
          endif else begin
            oplot, time, rate+rate_error, COLOR=blue, THICK=0.5
            oplot, time, rate-rate_error, COLOR=blue, THICK=0.5
          endelse
        endif ; ptr_valid(rates[jj])
      endif else begin
      endelse

      time_offset = time_offset + ((tstop[kk] - tstart[kk]) / 1000.)

      if (kk LT (num_obs-1)) then begin
        ; Make a graphic indicating a break in the time axis.
        oplot, [time_offset,time_offset], rate_axis_range, LINE=1
        time_offset = time_offset + break_length
        oplot, [time_offset,time_offset], rate_axis_range, LINE=1
      endif
    endfor ;kk, looping over obsids
    
    
    ; Plot the median energy time series.
    !Y.WINDOW = !Y.WINDOW[0] + (!Y.WINDOW[1]-!Y.WINDOW[0])*energy_axis_window
    axis, YAXIS=1, /SAVE, YRANGE=energy_axis_range, YTIT='Median Energy (keV)', YTICKLEN=yticklen, COLOR=red, CHARSIZE=1
    time_offset = 0.
    for kk = 0, num_obs-1 do begin
      jj = kk
      if ptr_valid(times[jj]) then begin
        ; Plot the median energy time series.  We CANNOT USE PSYM=10 with unequal bin sizes!
        num_groups = n_elements(*times[jj])
        zerovec    = replicate(0,num_groups)
        oploterror, *times[jj] + time_offset, *median_energies[jj], *halfbin_sizes[jj],                   zerovec, COLOR=red, ERRCOLOR=red, PSYM=3, THICK=1, ERRTHICK=1, /NOHAT
        oploterror, *times[jj] + time_offset, *median_energies[jj],            zerovec, *median_energy_errors[jj], COLOR=red, ERRCOLOR=red, PSYM=3, THICK=1, ERRTHICK=1, /NOHAT, NSKIP=nskip_binned   
      
        if ptr_valid(median_energies[num_obs+jj]) then begin
          ; Plot the smooth lightcurve
          time                = congrid(*times               [num_obs+jj] + time_offset, num_groups*20)
          median_energy       = congrid(*median_energies     [num_obs+jj], num_groups*20)
          median_energy_error = congrid(*median_energy_errors[num_obs+jj], num_groups*20)
           
          ; Since the smooth timeseries is NaN at the ends (where the kernel falls off the data)
          ; the range of times left in the plot may be too small to be worth showing.
          if ( (max(time)-min(time))/time_span LT 0.001 ) then begin
          endif else begin
;           oplot, time, median_energy, COLOR=orange, THICK=0.5
          endelse
        endif ; ptr_valid(median_energies[jj])
      endif
      
      time_offset = time_offset + ((tstop[kk] - tstart[kk]) / 1000.) + break_length
    endfor ;kk, looping over obsids

    device, /close
    color_manager, /X_PSEUDO
    if (verbose GT 0) then print, merged_stacked_lc_fn, merged_sequenced_lc_fn, F='(%"Binned light curve, smooth light curve, & median energy timeseries are available:\n  gv -swap %s &\n  gv -swap %s &")'

    if (verbose GE 10) then stop

LC_CLEANUP:
    ptr_free, times, halfbin_sizes, rates, rate_errors, median_energies, median_energy_errors    
    

MERGE_SAVE_STATS:    
    psb_xaddpar, src_stats, 'CREATOR',  creator_string
    writefits, src_stats_fn, 0, src_stats
    continue
    
MERGE_IS_EMPTY:
    file_delete, /ALLOW_NONEXISTENT, merged_list_fn, photometry_fn
    file_delete, /ALLOW_NONEXISTENT, composite_psf_fn
    file_delete, /ALLOW_NONEXISTENT, merged_env_events_fn, merged_src_events_fn, merged_region_fn
    file_delete, /ALLOW_NONEXISTENT, merged_src_spectrum_fn, merged_bkg_spectrum_fn, merged_rmf_fn, merged_arf_fn, photometry_fn 
    file_delete, /ALLOW_NONEXISTENT, merged_stacked_lc_fn, merged_sequenced_lc_fn
    sxdelpar, src_stats, ['MERGPRUN', 'MERGQUAL', 'BANDPRUN', 'PRIM_OBS', 'EMAP_TOT', 'FRACEXPO', 'WARNFRAC', 'OBSNAME', 'SRC_CNTS', 'THETA', 'THETA_LO', 'THETA_HI', 'SRC_AREA', 'SRC_RAD', 'PSF_FRAC', 'MSK_RAD', 'FRACSPEC', 'OVRLP_LM', 'OVRLP_LO', 'OVRLP_HI', 'BESTOBS', 'WORSTOBS', 'RA_DATA', 'DEC_DATA', 'BACKGRND', 'SCAL_MAX', 'SCAL_MIN', 'KS_SPECT', 'EXPOSURE', 'EFFAREA', 'PROB_KS', 'MERGE_KS', 'MERG_CHI', 'ERR_DATA', 'ERX_DATA', 'ERY_DATA' ]
    
    psb_xaddpar, src_stats, 'MERGNUM',  0, 'number of observations merged'
    psb_xaddpar, src_stats, 'MERGFRAC', 0, 'fraction of extracted data merged'
    psb_xaddpar, src_stats, 'MERGBIAS', 0, 'fraction of exposure discarded to optimize merge'
    GOTO, MERGE_SAVE_STATS

  endfor ;ii, looping over sources

;save, /COMPRESS, debug_pb, FILE='debug_pb.sav'

endif ;keyword_set(merge_observations)



;; =============================================================================
if keyword_set(check_positions) then begin
;; =============================================================================
  energy_range_label = string(energy_range, F='(%"%0.2f:%0.2f keV ")')

  if ~keyword_set(skip_event_regions) then begin
    ;; Set up the Postscript page using lengths in inches for 12 images per page.
    page_long_dim  = 11   ; inches
    page_short_dim =  8.5 ; inches
    
    image_size = 2.16
    gap_size   = 0.25
    window_xsize = 4*image_size + 3*gap_size
    window_ysize = 3*image_size + 3*gap_size
        
    window_xoffset = (page_short_dim - window_ysize) / 2.
    window_yoffset = page_long_dim - (page_long_dim - window_xsize) / 2.
        
    color_manager, /PS_GREY, FILENAME='mugshots.ps', /LANDSCAPE, /INCHES,$
                   XOFFSET=window_xoffset, YOFFSET=window_yoffset, $
                   XSIZE=window_xsize,     YSIZE=window_ysize
    !P.FONT =  0   ; "hardware" font
                                                         
    temp = convert_coord( !D.X_CH_SIZE, !D.Y_CH_SIZE, /DEV, /TO_NORM)
    x_ch_size = temp[0]
    y_ch_size = temp[1]
  endif ; ~skip_event_regions


  for ii = 0L, num_sources-1 do begin
  ; Skip sources already determined to be not extracted.
    if source_not_observed[ii] then continue

    basedir   = sourcename[ii] + '/' 
    sourcedir = basedir + merge_subdir[ii]
 
    merged_env_events_fn  = sourcedir + env_events_basename
    merged_src_events_fn  = sourcedir + src_events_basename
    merged_region_fn      = sourcedir + src_region_basename

    merged_list_obsolete_fn= sourcedir + 'ObsIDs_merged.txt'    
    merged_list_fn         = sourcedir + 'ObsIDs_merged.fits'    
         composite_psf_fn  = sourcedir + psf_basename
    rebinned_composite_psf_fn  = sourcedir + 'recon.psf'
         composite_emap_fn = sourcedir + env_emap_basename
         composite_img_fn  = sourcedir + env_image_basename
    temp_composite_img_fn  = tempdir   + env_image_basename
    
    event_reg_fn          = sourcedir + evt_region_basename

    unnamed_src_stats_fn   = sourcename[ii] + '/' + src_stats_basename
    src_stats_fn           = sourcedir            + src_stats_basename

   ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(basedir)
    is_writable = ~temp.EXISTS || (temp.WRITE && ~temp.SYMLINK)
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    ;; ------------------------------------------------------------------------
    ;; Read the fundamental source properties from the unnamed source.stats file.
    unnamed_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    
    if (NOT keyword_set(error)) then begin
      src_label = strtrim(psb_xpar( unnamed_stats,'LABEL'),2)

       ra_src = psb_xpar( unnamed_stats, 'RA')
      dec_src = psb_xpar( unnamed_stats, 'DEC')
    endif else begin
      print, error
      message, 'ERROR reading '+unnamed_src_stats_fn
    endelse

    ;; ------------------------------------------------------------------------
    ;; Read stats about merged observations of source.
    src_stats    = headfits(src_stats_fn, ERRMSG=error)
    
    if (NOT keyword_set(error)) then begin
      ra_data     = psb_xpar( src_stats, 'RA_DATA')
      dec_data    = psb_xpar( src_stats, 'DEC_DATA')
      off_angle   = psb_xpar( src_stats, 'THETA')
      src_radius  = psb_xpar( src_stats, 'SRC_RAD')
      mask_radius = psb_xpar( src_stats, 'MSK_RAD')
    endif else begin
      print, sourcedir, src_label, F='(%"\nMerge %s (%s) not found.")'
      continue
    endelse
    
    if (off_angle LT theta_range[0]) OR (off_angle GT theta_range[1]) then begin
      print, sourcename[ii], src_label, F='(%"\nSkipping merge not in THETA_RANGE: %s (%s)")'      
      continue
    endif else $
      print, sourcename[ii], src_label, F='(%"\n===================================================================\nSource: %s (%s)")'      



    ;; ------------------------------------------------------------------------
    if (NOT (file_test(merged_env_events_fn) AND file_test(composite_psf_fn))) then begin
      print, merge_subdir[ii], F='(%"\nMerge %s is empty.")'
      file_delete, /ALLOW_NONEXISTENT, composite_img_fn, event_reg_fn
      continue
    endif
    
    ; Defensively, remove any temp files and CIAO parameter files used by the previous source. 
    list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
    if (count GT 0) then file_delete, list
    
    run_command, /QUIET, 'pset dmcopy clobber=yes'
    
    ;; Read merged neighborhood event list and extract astrometry.
    bt = mrdfits(merged_env_events_fn, 1, theader, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + merged_env_events_fn
    
    xpos_events = bt.x  &  ypos_events = bt.y
    
    ; The dmmerge calls we used to build merged event lists remove one column (phas).
    ; In CIAO 4.3 that tool fails to renumber the TDMIN/TDMAX keywords.
    ; The result is that when we get here, the header contains a TDMIN/TDMAX pair for the column N+1, where N is the number of actual columns in the table.
    ; Due to IDL's default truncation of vector array subscripts, the fxbfind.pro tool returns the wrong values when run on such a file.
    ; We cannot remove these extra keywords using the existing dmhedit call back in the MERGE stage, where merged_env_events_fn, is built because dmhedit will complain someday in the future when the extra keywords are no longer present.
    ; Thus, we will remove this extra TDMIN/TDMAX keyword from the header.
    num_cols = psb_xpar( theader, 'TFIELDS')
    sxdelpar, theader, ['TDMIN', 'TDMAX'] + string(num_cols+1, F='(%"%d")') 
    
    fxbfind, theader, 'TTYPE', dum1, TTYPE, dum2, 'null'
    fxbfind, theader, 'TCTYP', dum1, TCTYP, dum2, 'null'
    fxbfind, theader, 'TCRVL', dum1, TCRVL, dum2, 0.0D
    fxbfind, theader, 'TCRPX', dum1, TCRPX, dum2, 0.0D
    fxbfind, theader, 'TCDLT', dum1, TCDLT, dum2, 0.0D
    fxbfind, theader, 'TDMIN', dum1, TDMIN, dum2, 0.0
    fxbfind, theader, 'TDMAX', dum1, TDMAX, dum2, 0.0
    colnames = strlowcase( strtrim(TTYPE,2) )
    x_ind    = where(strlowcase(colnames) EQ 'x')
    y_ind    = where(strlowcase(colnames) EQ 'y')
    make_astr, event2wcs_astr, DELTA=TCDLT[[x_ind,y_ind]], CTYPE=TCTYP[[x_ind,y_ind]], $
                               CRPIX=TCRPX[[x_ind,y_ind]], CRVAL=TCRVL[[x_ind,y_ind]]

    ;; ------------------------------------------------------------------------
    if ~keyword_set(skip_event_regions) then begin
      ;; Convert in-band neighborhood event list to RA/DEC & make a "point" region for each event, useful
      ;; for marking the events on a smoothed or reconstructed image.
      ;; REMEMBER THAT THE xy2ad and ad2xy programs assume that (x,y) are 
      ;; ZERO-BASED pixel indexes.  Thus we must subtract 1 from the sky (x,y) 
      ;; positions when converting to RA,DEC.
      ind = where((1000*energy_range[0] LE bt.energy) AND (bt.energy LE 1000*energy_range[1]), count)
      if (count GT 0) then begin
        xy2ad, xpos_events[ind]-1, ypos_events[ind]-1, event2wcs_astr, ra_evt, dec_evt
        
        openw,  region2_unit, event_reg_fn, /GET_LUN
        printf, region2_unit, "# Region file format: DS9 version 3.0"
        printf, region2_unit, "# "+energy_range_label
        printf, region2_unit, 'global width=1 font="helvetica 12 normal"'
        !TEXTUNIT = region2_unit
        forprint, TEXTOUT=5, ra_evt, dec_evt, F='("J2000;cross point ",F10.6,1x,F10.6," #")', /NoCOMMENT
        free_lun, region2_unit
      endif
    endif ; ~skip_event_regions

    ;; ------------------------------------------------------------------------
    ;; Construct composite neighborhood image ON THE SAME BIN GRID PHASE (not range) 
    ;; used in the composite PSF image, and with the source position at the center.
    
    ; Read composite PSF.  
    ; If there are multiple PSF images, we select the one closest to the NOMINAL_PSF_ENERGY specified by the caller.
    fits_open, composite_psf_fn, fcb
    psf_energy = fltarr(1+fcb.NEXTEND)
    for kk =0, fcb.NEXTEND do begin
      fits_read, fcb, dum, this_header, /HEADER_ONLY, /NO_PDU, EXTEN_NO=kk
      psf_energy[kk] = psb_xpar( this_header, 'ENERGY') 
    endfor
    dum = min(abs(psf_energy - nominal_psf_energy), extension_number)
    fits_read, fcb, composite_psf, composite_psf_hdr, /NO_PDU, EXTEN_NO=extension_number
    fits_close, fcb

    print, psb_xpar( composite_psf_hdr, 'ENERGY'), energy_range_label, F='(%"PSF model is for %0.1f keV photons; image from %s events.")'
    
    ; Extract conversion from PSF image index coordinates to RA/DEC (WCS) coordinates.
    extast, composite_psf_hdr, psf2wcs_astr
    
    ; Read the PSF position on the sky, which may be slightly different from the current position estimate for the actual source.
     ra_psf = psb_xpar( composite_psf_hdr, 'RA')
    dec_psf = psb_xpar( composite_psf_hdr, 'DEC')
    
    ; Find the 0-based index coordinates of the PSF pixel EDGE nearest the PSF origin.
    ; Centers of pixels are x.0, edges are x.5 in index coordinates.
    ad2xy, ra_psf, dec_psf, psf2wcs_astr, src_column, src_row
    edge_near_src_column = round(src_column+0.5) - 0.5  ; pixel coordinates in PSF image
    edge_near_src_row    = round(src_row   +0.5) - 0.5  ; pixel coordinates in PSF image
    
    ; Perform linear conversion from PSF image coordinates (column,row) to sky coordinates (x,y).
    ; We cannot use xy2ad.pro/ad2xy.pro for conversions between array index and PHYSICAL (sky) coordinate systems.
    crvalP = [psb_xpar( composite_psf_hdr, 'CRVAL1P'), psb_xpar( composite_psf_hdr, 'CRVAL2P')]
    crpixP = [psb_xpar( composite_psf_hdr, 'CRPIX1P'), psb_xpar( composite_psf_hdr, 'CRPIX2P')]
    cdeltP = [psb_xpar( composite_psf_hdr, 'CDELT1P'), psb_xpar( composite_psf_hdr, 'CDELT2P')]
    ctype  = [psb_xpar( composite_psf_hdr, 'CTYPE1P'), psb_xpar( composite_psf_hdr, 'CTYPE2P')]

    edge_near_src_x = crvalP[0] + cdeltP[0] * (edge_near_src_column+1 - crpixP[0])  ; sky coordinates
    edge_near_src_y = crvalP[1] + cdeltP[1] * (edge_near_src_row   +1 - crpixP[1])  ; sky coordinates
    
    ; Find the half-dimensions from the center (edge_near_src_x, edge_near_src_y) of the 
    ; data image that will match the FOV of neighborhood.evt, which is recorded by the TDMIN/TDMAX keywords.
    x_halfdim = [(TDMAX[x_ind] - edge_near_src_x), (edge_near_src_x - TDMIN[x_ind])] ; sky pixels
    y_halfdim = [(TDMAX[y_ind] - edge_near_src_y), (edge_near_src_y - TDMIN[y_ind])] ; sky pixels
    
    if (min(x_halfdim) LT 0) || (min(y_halfdim) LT 0) then begin
      ; edge_near_src does NOT lie in the interval [TDMIN:TDMAX]!
      print, 'ERROR: the TDMIN/TDMAX keywords in neighborhood.evt seem to be wrong.'
      GOTO, FAILURE
    endif
    
    x_halfdim = ceil( max(abs( x_halfdim/cdeltP[0] )) ) ; integer # of PSF/image pixels
    y_halfdim = ceil( max(abs( y_halfdim/cdeltP[1] )) ) ; integer # of PSF/image pixels

    ; Make the image square.
    x_halfdim >= y_halfdim ; integer # of PSF/image pixels
    y_halfdim  = x_halfdim ; integer # of PSF/image pixels
    
    ; And make the image, which should have dimensions that are EVEN.
    ; We write to the temp_dir to reduce NFS traffic.
    cmd = string( merged_env_events_fn, 1000*energy_range, $
                  edge_near_src_x - x_halfdim*cdeltP[0], $              ; sky coordinates
                  edge_near_src_x + x_halfdim*cdeltP[0], 2*x_halfdim, $ ; sky coordinates
                  edge_near_src_y - y_halfdim*cdeltP[1], $              ; sky coordinates
                  edge_near_src_y + y_halfdim*cdeltP[1], 2*y_halfdim, $ ; sky coordinates
                  (psb_xpar( theader, 'NAXIS2') LT 32767L) ? 'i2' : 'i4', $ ; prevent overflow of INT2 output image
                  temp_composite_img_fn, $
    F="(%'dmcopy ""%s[energy=%6.1f:%7.1f][bin x=%9.4f:%9.4f:#%d,y=%9.4f:%9.4f:#%d][opt type=%s]"" %s')")
    run_command, cmd

    composite_img = readfits(temp_composite_img_fn, composite_img_hdr, /SILENT)
    
    ; We must overwrite composite data image file to get rid of GTI tables that dmcopy keeps around
    ; so that we can put the reconstructed image in the second HDU.
    ; There seems to be no CIAO option to suppress GTI tables in the dmcopy above.
    writefits, temp_composite_img_fn, composite_img, composite_img_hdr
    extast, composite_img_hdr, compimg2wcs_astr 

    
    ; Defensively check that composite neighborhood image and PSF binnings match!
    pixsize_mismatch = (compimg2wcs_astr.CDELT - psf2wcs_astr.CDELT) / compimg2wcs_astr.CDELT
    if (max(pixsize_mismatch) GT 0.01) then begin
      help, compimg2wcs_astr, /ST
      help, psf2wcs_astr,     /ST
      message, 'ERROR: data image and PSF binning do not match!'
    endif

   
    
    ;; ------------------------------------------------------------------------
    ;; Correlate PSF & data images to refine the source position estimate.
    ;; The range over which we search is arbitrarily set to SRC_RAD/2.  
    ;; In /CONSTRUCT_REGIONS, SRC_RAD is determined from the polygon, and later in 
    ;; /EXTRACT_EVENTS it is recomputed.

    ; Convert src_radius from skypixel to PSF image pixels.
    search_radius = 1 > ceil((src_radius / 2.0) / psb_xpar( composite_psf_hdr, 'CDELT1P')) 

    ra_corr = !VALUES.F_NAN  &  dec_corr = !VALUES.F_NAN  &  quantization_corr = !VALUES.F_NAN  &  arcsec_per_reconpixel = !VALUES.F_NAN
    if keyword_set(skip_correlation) then $
      GOTO, END_OF_CORRELATION
    
    ; Compute size of 1 PSF pixel in arcsec.
    quantization_corr = psf2wcs_astr.CDELT[1] * 3600

    print, search_radius * quantization_corr, F='(%"Correlating PSF & data over +- %0.1f arcsec ...")'

    ; Find offset between two images.
    xy2ad, 0, 0, psf2wcs_astr, ra_corner, dec_corner
    ad2xy, ra_corner, dec_corner, compimg2wcs_astr, xoffset_b, yoffset_b

    xoffset_b = round(xoffset_b)
    yoffset_b = round(yoffset_b)
    
    corrmat = correl_images( double(composite_img), composite_psf, XOFFSET_B=xoffset_b, YOFFSET_B=yoffset_b, $
                             XSHIFT=search_radius, YSHIFT=search_radius )
    corrmat_analyze, corrmat, xoffset_optimum, yoffset_optimum, XOFF_INIT=xoffset_b, YOFF_INIT=yoffset_b
    
    if ((abs(xoffset_optimum) > abs(yoffset_optimum)) EQ search_radius) then begin
      print, 'WARNING!  Correlation postion cannot be computed; peak lies on search boundary.'
    endif else begin
      col_offset = xoffset_optimum - xoffset_b
      row_offset = yoffset_optimum - yoffset_b
      
      print, col_offset, row_offset, [col_offset,row_offset] * psf2wcs_astr.CDELT * 3600, $
             F='(%"  X,Y offset of correlation peak: %d, %d (pixels); %0.2f, %0.2f (arcsec)")'
          
      ;; Convert catalog position (RA,DEC) into PSF pixel indexes (0-based).
      ad2xy, ra_src, dec_src, psf2wcs_astr, column, row
      
      ;; Add optimum pixel offsets & convert back to RA,DEC.
      xy2ad, column+col_offset, row+row_offset, psf2wcs_astr, ra_corr, dec_corr
    endelse

END_OF_CORRELATION:

    ;; ------------------------------------------------------------------------
    ;; The composite image, and its reconstruction, are intended to be sized such that 
    ;; they are suitable for printing to mugshots.ps below, or inclusion in an "atlas"
    ;; page for the source.  Also, these reconstructions are intended to be quick &
    ;; dirty glimpses into whether there are missed point sources, or extended sources.
    ;; For both reasons, composite images with excessive spatial resolution are not desirable so we
    ;; will rebin as much as possible.
    
    rebin_factor = 1
    radius50 = psb_xpar( composite_psf_hdr, 'RADIUS50')
    if NOT finite(radius50) then begin
      print, 'Keyword RADIUS50 not found; using on-axis value of 0.85 skypix.'
      radius50 = 0.85 * arcsec_per_skypixel
    endif
    
    ; RADIUS50 is radius (arcsec) enclosing 50% PSF
    ; Let's rebin so that this radius is spanned by at least 2 pixels.
    ; Forcing the recon pixel size to be small can cause a source's power to be divided among several closely-spaced peaks; our simple-minded ML position algorithm then focusses on the brightest peak, ignoring the others.
    arcsec_per_psfpixel = psf2wcs_astr.CDELT[1] * 3600
    
    num_across          = radius50 / (arcsec_per_psfpixel)
    
    rebin_factor        = 1 > floor(num_across/2.)  
    
    
    if (rebin_factor GT 1) then begin
      ; Rebin data & PSF images.  
      print, rebin_factor*arcsec_per_psfpixel, F='(%"\nRebinning composite data and composite PSF images to %4.1f arcsec/pixel.")'
      
      search_radius = !VALUES.F_NAN

      ; The rebinned data image is resaved.  
      ; We choose to crop the data image at the top and right so the dimensions are a multiple of the rebin_factor, to avoid incomplete pixels in the rebinned image.
      proposed_xdim = floor((size(composite_img, /DIM))[0] / rebin_factor)
      proposed_ydim = floor((size(composite_img, /DIM))[1] / rebin_factor) 

      file_move, /OVERWRITE, temp_composite_img_fn, temp_image_fn
      cmd = string(temp_image_fn, proposed_xdim*rebin_factor, rebin_factor, proposed_ydim*rebin_factor, rebin_factor, temp_composite_img_fn, F="(%'dmcopy ""%s[bin #1=1:%d:%d,#2=1:%d:%d]"" %s')")
      run_command, cmd      
      composite_img = readfits(temp_composite_img_fn, composite_img_hdr, /SILENT)
      
      ; The rebinned PSF image is saved.
      ; We choose to let the upper and right edges of the result be incomplete rather than to crop the PSF.
      cmd = string(composite_psf_fn, 1+extension_number, rebin_factor, rebin_factor, rebinned_composite_psf_fn, F="(%'dmcopy ""%s[%d][bin #1=::%d,#2=::%d]"" %s')")
      run_command, cmd      
      composite_psf = readfits(rebinned_composite_psf_fn, composite_psf_hdr, /SILENT)

      print, psb_xpar( composite_psf_hdr, 'ENERGY'), energy_range_label, F='(%"PSF model is for %0.1f keV photons; image from %s events.")'
      
      ; Update the FITS astrometry structures.
      extast, composite_img_hdr, compimg2wcs_astr 
      extast, composite_psf_hdr, psf2wcs_astr
      arcsec_per_psfpixel = psf2wcs_astr.CDELT[1] * 3600
    endif ; (rebin_factor GT 1)

    
    ; To speed up the reconstruction, trim the PSF footprint until it retains no more than 96% of the total PSF power.
    psf_total = psb_xpar( composite_psf_hdr, 'PSF_TOTL')
    psf_xdim  = psb_xpar( composite_psf_hdr, 'NAXIS1')
    psf_ydim  = psb_xpar( composite_psf_hdr, 'NAXIS2')
    
    crop_size = 0
    while (total(/DOUBLE, composite_psf) GT (0.96 * psf_total)) do begin
      composite_psf[crop_size           ,*                   ] = 0
      composite_psf[                   *,crop_size           ] = 0
      composite_psf[psf_xdim-1-crop_size,*                   ] = 0
      composite_psf[                   *,psf_ydim-1-crop_size] = 0
      crop_size++
    endwhile
    
    if (crop_size GT 0) then begin
      print, crop_size, F='(%"\nTrimming a border of %d pixels from the composite PSF image to speed up reconstruction.")'

      ; The cropped PSF image is saved.
      file_move, /OVERWRITE, rebinned_composite_psf_fn, temp_image_fn
      cmd = string(temp_image_fn, 1+crop_size, psf_xdim-crop_size, 1+crop_size, psf_ydim-crop_size, rebinned_composite_psf_fn, $
                   F="(%'dmcopy ""%s[bin #1=%d:%d,#2=%d:%d]"" %s')")
      run_command, cmd      
      composite_psf = readfits(rebinned_composite_psf_fn, composite_psf_hdr, /SILENT)

      print, psb_xpar( composite_psf_hdr, 'ENERGY'), energy_range_label, F='(%"PSF model is for %0.1f keV photons; image from %s events.")'
      
      ; Update the FITS astrometry structures.
      extast, composite_psf_hdr, psf2wcs_astr
      arcsec_per_psfpixel = psf2wcs_astr.CDELT[1] * 3600
    endif
   
        
    ; Defensively check that composite neighborhood image and PSF binnings match!
    pixsize_mismatch = (compimg2wcs_astr.CDELT - psf2wcs_astr.CDELT) / compimg2wcs_astr.CDELT
    if (max(pixsize_mismatch) GT 0.01) then begin
      help, compimg2wcs_astr, /ST
      help, psf2wcs_astr,     /ST
      message, 'ERROR: data image and PSF binning do not match!'
    endif

    
;    ;; ------------------------------------------------------------------------
;    ;; Estimate the background level of the image and save to a header keyword.
;    ;; We coarsely rebin the image to push most of the PSF into ~1 pixel, but we
;    ;; ensure that the rebinned image retains a reasonable number (100) of pixels.
;    num_across          = radius50 / (arcsec_per_psfpixel)
;    
;    for rebin_factor= (1 > ceil(num_across * 4.)), 2, -1 do begin
;      proposed_xdim = floor((size(composite_img, /DIM))[0] / rebin_factor)
;      proposed_ydim = floor((size(composite_img, /DIM))[1] / rebin_factor) 
;      if (proposed_xdim*proposed_ydim GE 100) then break
;    endfor
;    
;    print, rebin_factor*arcsec_per_psfpixel, F='(%"\nEstimating background in composite data image rebinned to %4.1f arcsec/pixel.")'
;    
;     ; We choose to crop the data image to avoid incomplete pixels in the rebinned image.
;    cmd = string(temp_composite_img_fn, proposed_xdim*rebin_factor,rebin_factor, proposed_ydim*rebin_factor,rebin_factor, temp_image_fn, F="(%'dmcopy ""%s[bin #1=1:%d:%d,#2=1:%d:%d]"" %s')")
;    run_command, cmd      
;    bigpixel_image = readfits(temp_image_fn, bigpixel_hdr, /SILENT)
;    help, bigpixel_image
;    
;    ; Find the pixel size.
;    extast, bigpixel_hdr, bigpixel_astr 
;    arcsec_per_bigpixel = bigpixel_astr.CDELT[1] * 3600
;    
;    estimate_poisson_background, bigpixel_image, bkg_estimate, SIGNIFICANCE=0.99, /VERBOSE
;
;    cmd = string(temp_composite_img_fn, bkg_estimate * (arcsec_per_skypixel / arcsec_per_bigpixel)^2, $
;                 F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=BACKGRND value=""%f"" comment=""background (count /skypixel**2), total-band, merged observations""')")
;
;    run_command, cmd
; 
    
    ;; ------------------------------------------------------------------------
    ;; Try to build an exposure map that corresponds to the composite image.
    ;; The current design records the ObsIDs merged in ObsIDs_merged.fits, but older AE versions wrote ObsIDs_merged.txt.
    obs_dir_list = ''
    if file_test(merged_list_fn) then begin
      obs_data     = mrdfits(merged_list_fn,1, /SILENT)
      obs_dir_list = strtrim(obs_data.obs_dir,2) ; FITS string columns can end up with trailing blanks!
      
    endif else if file_test(merged_list_obsolete_fn) then begin
      readcol, merged_list_obsolete_fn, obs_dir_list, FORMAT='A', COMMENT=';', /SILENT
    endif
    
    if keyword_set(obs_dir_list) then begin
      num_obs = n_elements(obs_dir_list)
      if array_equal(file_test(obs_dir_list+env_emap_basename),1) then begin
        ;; Reproject the emaps onto the grid of neighborhood.img (temp_composite_img_fn).
 
        ; Note that the exposure map is a 2-D FUNCTION representing a physical quantity (typically, effective area
        ; multiplied by exposure time, in units of sec*cm^2*counts/photon), not a 2-D HISTOGRAM that is counting something
        ; with "per pixel" units. Thus, when rebinning we choose method=average so that the scale of the exposure map is
        ; unchanged.
        reproj_obs_emap_fn  = tempdir + string(indgen(num_obs), F='(%"reproj%d.emap")')
        resolution = 1 
        for kk=0, num_obs-1 do $
          run_command, string(obs_dir_list[kk]+env_emap_basename, temp_composite_img_fn, reproj_obs_emap_fn[kk], resolution, $
                             F="(%'reproject_image  infile=%s  matchfile=%s  outfile=%s  method=average resolution=%d')")

        
        ;; Sum the reprojected exposure maps.
        terms   = string(1+indgen(num_obs),F="(%'img%d')")
        formula = strjoin(terms, '+')
      
        forprint,        reproj_obs_emap_fn, F='(%"%s")', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
        
        run_command, string(temp_text_fn, composite_emap_fn, formula, $
                            F="(%'dmimgcalc infile=""@-%s"" infile2=none outfile=%s operation=""imgout=((float)(%s))"" clob+')")          
      endif
    endif ; keyword_set(obs_dir_list)


    ;; ------------------------------------------------------------------------
    ; Finally, now that we've finished the heavy I/O involving the composite image, we can move it from 
    ; the temp_dir to the source directory.
    file_move, /OVERWRITE, temp_composite_img_fn, composite_img_fn
    
    ;; ------------------------------------------------------------------------
    if keyword_set(skip_reconstruction) then begin
      ra_ml = !VALUES.F_NAN  &  dec_ml = !VALUES.F_NAN  
      GOTO, END_OF_RECONSTRUCTION
    endif
    
    ; The convolutions (convolve.pro in the AstroLib) performed by the reconstruction code (Max_Likelihood.pro in the AstroLib) make certain assumptions regarding where the conceptual "center" of the PSF is within the actual 2-D array used to represent the PDF.  
    ; Another way to think of this is as a convention for where in the array the "point source" that produces that PSF is located.
    ; Correct astrometry can be assigned to the reconstructed image only if you can determine the offset from the point source location assumed by convolve.pro to the actual location in your particular PSF (governed by how the PSF image was constructed, and how it's been subsequently rebinned and trimmed). 
    
    ; Since the convolve.pro routine has evolved over time, it is wise to verify that the LOCAL version behaves as we have assumed in the code below.
    ; We will do this check once per catalog, for both an odd-dimension PSF and an even-dimenson one, by reconstructing with a fake image and PSF.

    if (ii EQ 0) then begin
      print, 'Testing ae_Max_Likelihood procedure ...'
      ; Generate a fake "data image" consisting of a delta function.
      test_image = dblarr(10,10)
      test_image[3,3] = 1
    
      for jj=0,1 do begin
        case jj of 
          0: dim=5
          1: dim=6
        endcase
        
        ; Generate a PSF (with odd dimensions in the first pass, even dimensions in the second pass) consisting of a delta function, located where we think convolve.pro expects it to be..
        psf        = fltarr(dim,dim)
        psf_spec   = size(   psf  , /STRUCTURE)
        psf_origin = floor( (psf_spec.DIMENSIONS[0:1] - 1) / 2.0  )

        psf[psf_origin[0], psf_origin[1]] = 1
      
        ; Reconstruct the test image; verify that a single-pixel peak appears in the same pixel as the input signal.
        maxlik_img = 0
        ae_Max_Likelihood, test_image, psf, maxlik_img
        
        peak_val = maxlik_img[3,3] 
                   maxlik_img[3,3] = 0
        
        if (peak_val LT 10*max(maxlik_img)) then begin
          print, 'ERROR: AE assumptions about PSF conventions do not match those in convolve.pro!'
          print, 'A test image that looks like this:'
          print, fix(round(test_image)) 
          print, 'produced a reconstructed image that is shifted:'
          print, fix(round(maxlik_img))
          GOTO, FAILURE
        endif
      endfor ;jj
    endif ; (ii EQ 0)

    
    
    ; To assign meaningful astrometry to the reconstructed image we must account for the offset from the location in the PSF image where max_likelihood.pro (really, convolve.pro) assumes the point source is located, and the actual PSF position (which may be slightly different from the source position).

    ; The location (in 0-based image coordinates) assumed by max_likelihood.pro (verified by the test above) is calculated like this:
    psf_spec            = size( composite_psf, /STRUCTURE)
    psf_origin_standard = floor( (psf_spec.DIMENSIONS[0:1] - 1) / 2.0  )

    
    ; And the actual PSF location (in 0-based image coordinates) is calculated from the FITS astrometry.
    ad2xy, ra_psf, dec_psf, psf2wcs_astr, column, row
    
    psf_origin_actual = [column, row]
    
    print, (psf_origin_actual - psf_origin_standard) * arcsec_per_psfpixel, F='(%"\nReconstruction astrometry has been adjusted for the (%0.2f, %0.2f) arcsec offset between the assumptions made by the PSF model and by the AstroLib regarding where in the PSF image the source is located.")'


    ; Build from scratch a header for the reconstructed image.  WCS is taken from composite_img_hdr, with the adjustment discussed above.
    ; I do not copy composite_img_hdr itself because I can't figure out how the LTV* & LTM* keywords have to be manipulated in order to define the correct PHYSICAL (sky) coordinate system.
    mkhdr, maxlik_hdr, composite_img, /IMAGE

    maxlik2wcs_astr       = compimg2wcs_astr
    maxlik2wcs_astr.CRPIX = maxlik2wcs_astr.CRPIX - (psf_origin_actual - psf_origin_standard)
    putast, maxlik_hdr, maxlik2wcs_astr, CD_TYPE=1  ;CD_TYPE=1 ensures CDELT keywords carry plate scale   
    
    psb_xaddpar, maxlik_hdr, 'HDUNAME', 'Max_Likelihood'
    psb_xaddpar, maxlik_hdr, 'CREATOR', creator_string
    
     
    ;; Perform reconstruction of composite image.
    
    ; We cast composite_img & normalized_psf to single precision to help speed in Max_Likelihood on 32-bit processors.  
    ; On 64-bit processors it seems to make no difference.
    composite_img  = float(composite_img)
      
    normalized_psf = float(composite_psf / total(composite_psf, /DOUBLE))
    
    ; To avoid the left/right and up/down wrap-around effects inherent in convolution a margin (padding) must be put on the data image.  We do that here, and tell ae_Max_Likelihood to NOT pad.
    
    ; Discontinuities between the image and this padding will lead to Gibbs artifacts in the reconstruction.  There are many published strategies for designing the padding to be a "natural" extrapolation of the data.  Here we take the simple approach of setting the padding to an estimated image background level.

    estimate_poisson_background, composite_img, bkg_per_psfpixel, SIGNIFICANCE=0.99, VERBOSE=0
    
    image_spec = size(  composite_img, /STRUCTURE)
      psf_spec = size( normalized_psf, /STRUCTURE)

    padded_dim = image_spec.DIMENSIONS[0:1] + psf_spec.DIMENSIONS[0:1]
    
    padded_composite_img      = make_array( TYPE=image_spec.TYPE, DIMENSION=padded_dim, VALUE=bkg_per_psfpixel )
    padded_composite_img[0,0] = composite_img
    
    print, max(maxlikelihood_iterations), F='(%"\nreconstructing with %d ML iterations ...")'
    ; Destroy any existing data structures in max_likelihood call that are not supposed to exist yet.
    psf_ft            = 0
    maxlik_img        = 0
    padded_maxlik_img = 0
    
    for mm=1, max(maxlikelihood_iterations) do begin
      ; The /NO_FT option (direct convolution and correlation computations instead of via FFTs) may run faster in some cases, but that's hard to predict.
      ae_Max_Likelihood, padded_composite_img, normalized_psf, padded_maxlik_img, FT_PSF=psf_ft, /NO_PAD
      
      ; Save reconstruction in extension of composite_img_fn.   Cast to float() to save disk space.
      if (total(mm EQ maxlikelihood_iterations) GT 0) then begin
        maxlik_img = padded_maxlik_img[0:image_spec.DIMENSIONS[0]-1, 0:image_spec.DIMENSIONS[1]-1 ]
        psb_xaddpar, maxlik_hdr, 'ML_ITERS', mm
        mwrfits, float(maxlik_img), composite_img_fn, maxlik_hdr, /SILENT
      endif
    endfor

    help, composite_img, normalized_psf, maxlik_img
    print
    
;mwrfits, padded_composite_img, composite_img_fn    
;mwrfits, padded_maxlik_img, composite_img_fn    
    
    ; Verify that global flux has been preserved, since we've had trouble with this before!
    data_sum  = total(composite_img)
    recon_sum = total(maxlik_img)
    if (data_sum GT 0) && (abs(data_sum-recon_sum)/data_sum GT 0.10) then begin
      print, src_label, data_sum, recon_sum, F='(%"WARNING! %s: Global flux in the data (%0.3f) and recon (%0.3f) images differs by >10%%.")'
    endif 
    
    ;; ------------------------------------------------------------------------
    ;; Print composite data image & reconstructed image, 6 sources to a page.
    if ~keyword_set(skip_event_regions) then begin
      if (ii mod 6 EQ 0) then erase  ; Add a page break.
      
      xoffset = 2*(ii mod 2)*(image_size+gap_size)
      yoffset = (ii/2 mod 3)*(image_size+gap_size) + gap_size
      
      tv, 255 - bytscl(composite_img), /INCHES, xoffset,  yoffset, XSIZE=image_size, YSIZE=image_size
      xoffset2 = xoffset+image_size+gap_size
      tv, 255 - bytscl(maxlik_img),    /INCHES, xoffset2, yoffset, XSIZE=image_size, YSIZE=image_size
     
      xyouts, (xoffset+image_size+gap_size/2)/window_xsize, $
               yoffset/window_ysize - y_ch_size, sourcename[ii], /NORMAL, ALIGN=0.5
  
      plots, (xoffset + [0,0,2*image_size+gap_size,2*image_size+gap_size,0])/window_xsize, $
             (yoffset + [0,image_size,image_size,0,0])/window_ysize, /NORMAL
    endif ; ~skip_event_regions
 
    ;; ------------------------------------------------------------------------
    ;; Try to identify the position of the reconstruction peak in order to obtain
    ;; an independent estimate of the source's position.
    
    ;; The range over which we search is arbitrarily set to SRC_RAD/2.  
    ;; In /CONSTRUCT_REGIONS, SRC_RAD is determined from the polygon, and later in 
    ;; /EXTRACT_EVENTS it is recomputed.

    ; Convert src_radius from skypixel to recon image pixels.
    search_radius = 1 > ((src_radius / 2.0) / psb_xpar( composite_psf_hdr, 'CDELT1P')) 

    ; Compute size of 1 recon pixel in arcsec.
    arcsec_per_reconpixel = maxlik2wcs_astr.CDELT[1] * 3600

;   print, off_angle, F='(%"Mean off-axis angle: %4.1f arcmin")'
    print, search_radius * arcsec_per_reconpixel, F='(%"Finding reconstruction peak within +- %0.1f arcsec of catalog position ...")'
    
    ; Convert catalog position to 0-based array index coordinates in maxlik_img.
    ad2xy, ra_src, dec_src, maxlik2wcs_astr, src_column, src_row
    
    ; Extract a sub-image of the appropriate size around the catalog source position.
    search_minx = floor(src_column-search_radius) > 0
    search_miny = floor(src_row   -search_radius) > 0
    search_maxx =  ceil(src_column+search_radius) < ((size(maxlik_img, /DIM))[0]-1)
    search_maxy =  ceil(src_row   +search_radius) < ((size(maxlik_img, /DIM))[1]-1)
    
    search_img = maxlik_img[search_minx:search_maxx, search_miny:search_maxy]
    
    ; Find the array index coordinates of the largest pixel in the search region.
    dum  = max(search_img, imax)
    index_to_point, imax, xindex, yindex, size(search_img)
    xindex += search_minx
    yindex += search_miny
    
    ; If that pixel is a local maximum, then compute the centroid of the 3x3 neighborhood.
    ll_neighbor_ind = [0,1,2,3]
    ur_neighbor_ind = [5,6,7,8]
    central_ind     = 4
    centroid_island = maxlik_img[xindex-1:xindex+1, yindex-1:yindex+1]

    local_max_found = (centroid_island[central_ind] GE max(centroid_island[ll_neighbor_ind])) AND $
                      (centroid_island[central_ind] GT max(centroid_island[ur_neighbor_ind]))
    
    if local_max_found then begin
      offset = [-1,0,1]
      
      make_2d, offset, offset, x_offset, y_offset
      
      xy2ad, xindex + total(centroid_island*x_offset) / total(centroid_island), $
             yindex + total(centroid_island*y_offset) / total(centroid_island), maxlik2wcs_astr, ra_ml, dec_ml
    endif else begin
      print, src_label, F='(%"WARNING! %s: Brightest recon pixel near source is not a local maximum; ML position estimate skipped.")'
      ra_ml = !VALUES.F_NAN  &  dec_ml = !VALUES.F_NAN
    endelse

END_OF_RECONSTRUCTION:

    ;; ------------------------------------------------------------------------
    ;; Append markers showing all three positions for each source to merged extraction region file.
    openw,  region1_unit, merged_region_fn, /GET_LUN, /APPEND

    printf, region1_unit, F='(%"# Catalog (crosses), Data Mean (diamonds), Reconstruction Peak (boxes), and Correlation Peak (circles) positions")'
  
    color = 'magenta'

    printf,   region1_unit, ra_src,  dec_src,  color, F='(%"J2000;cross   point %10.6f %10.6f # tag={cat}  color=%s")'
    printf,   region1_unit, ra_data, dec_data, color, F='(%"J2000;diamond point %10.6f %10.6f # tag={data} color=%s")'
    if finite(ra_corr) AND finite(dec_corr) then $
      printf, region1_unit, ra_corr, dec_corr, color, F='(%"J2000;circle  point %10.6f %10.6f # tag={corr} color=%s")'
    if finite(ra_ml) AND finite(dec_ml) then $
      printf, region1_unit, ra_ml,   dec_ml,   color, F='(%"J2000;box     point %10.6f %10.6f # tag={ml}   color=%s")'

    free_lun, region1_unit
    
    if NOT keyword_set(skip_reconstruction) then begin
      prefix   = '"'+env_image_basename+'['
      suffix   = ']" '
      hdu_args = strjoin(prefix+strtrim(indgen(1+n_elements(maxlikelihood_iterations)),2)+suffix, ' ')
      print, sourcedir, hdu_args, src_region_basename, F="(%'\nTo display reconstructed image run:\n  (cd %s; ds9 -log %s -region load all %s) &')"  
    endif
        

    ;; ------------------------------------------------------------------------
    ;; Save information to source stats file.    
    psb_xaddpar, src_stats, 'CREATOR',  creator_string

    comment = '[deg] position, '+energy_range_label+' reconstruction peak'
    psb_xaddpar, src_stats, 'RA_ML',     ra_ml, comment, F='(F10.6)'
    psb_xaddpar, src_stats, 'DEC_ML',   dec_ml, comment, F='(F10.6)'

    psb_xaddpar, src_stats, 'RECONRES', arcsec_per_reconpixel, '[arcsec] pixel size in recon image'    , F='(F7.4)'
    psb_xaddpar, src_stats, 'QUANTCOR', quantization_corr    , '[arcsec] quantization of CORR position', F='(F7.4)'

    comment = '[deg] position, '+energy_range_label+' correlation peak'
    psb_xaddpar, src_stats, 'RA_CORR',   ra_corr, comment, F='(F10.6)'
    psb_xaddpar, src_stats, 'DEC_CORR', dec_corr, comment, F='(F10.6)'

    writefits, src_stats_fn, 0, src_stats
  endfor ; loop over sources 
    
  print, '============================================================================='
  if ~keyword_set(skip_event_regions) then begin
    device, /close
    color_manager, /X_PSEUDO
    print, 'Wrote composite source images to mugshots.ps'
  endif ; ~skip_event_regions

  print, 'Catalog (crosses), Data Mean (diamonds), Reconstruction Peak (boxes), and Correlation Peak (circles) positions written to {sourcename}/{extraction_name}/' + src_region_basename
  print, '============================================================================='
endif ;keyword_set(check_positions)





;; =============================================================================
if keyword_set(fit_spectra) then begin
;; =============================================================================
case n_elements(grouped_spectrum_pattern) of
 0: grouped_spectrum_pattern = strarr(num_sources)
 1: grouped_spectrum_pattern = replicate(grouped_spectrum_pattern,num_sources)
 num_sources:
 else: begin
      print, 'ERROR: parameter GROUPED_SPECTRUM_FILENAME must be a scaler or a vector as long as the catalog'
      GOTO, FAILURE
      end
endcase



  result = routine_info( 'acis_extract', /SOURCE )
  fdecomp, result.PATH, disk, codedir
  
  if NOT keyword_set(model_filename) then begin
    print, 'ERROR:Parameter MODEL_FILENAME not supplied!'
    GOTO, FAILURE
  endif
  
  if     keyword_set(min_num_cts) then begin
    print, 'ERROR: keyword MIN_NUM_CTS no longer used by /FIT_SPECTRA; see the manual.'
    GOTO, FAILURE
  endif

 
  if ~keyword_set(fit_timeout) then fit_timeout = 600
  
  
;; Loop until all sources have been processed.
processed_flag        = replicate(0B,num_sources)
minutes_blocked       = 0
num_to_process_prev   = num_sources+1

repeat begin
source_index = where(processed_flag EQ 0, num_to_process)

; When an historic lock file exists we must take steps to prevent infinite looping.
if (num_to_process GE num_to_process_prev) then begin
  if (num_to_process EQ 1) then begin
    pause = 0.5
    print, pause*60, F='(%"\nWaiting %d seconds ...")'
  endif else begin
    pause = 5
    print, pause   , F='(%"\nWaiting %d minutes ...")'
  endelse
  wait, pause*60
  minutes_blocked += pause
endif
num_to_process_prev = num_to_process

  for kk = 0, num_to_process-1 do begin
    ii = source_index[kk]
    processed_flag[ii] = 1
    
    basedir   = sourcename[ii] + '/' 
    sourcedir = basedir + merge_subdir[ii]
    
    photometry_fn          = sourcedir + src_photometry_basename
    merged_src_spectrum_fn = sourcedir + sourcename[ii] + '.pi'
    merged_bkg_spectrum_fn = sourcedir + sourcename[ii] + '_bkg.pi'
    merged_arf_fn          = sourcedir + sourcename[ii] + '.arf'
    merged_rmf_fn          = sourcedir + sourcename[ii] + '.rmf'
    lock_fn                = sourcedir + modelsubdir + 'ae_lock'

    unnamed_src_stats_fn   = sourcename[ii] + '/' + src_stats_basename

   ; We assume that an existing source directory that is a symbolic link should not be written to.
    temp = file_info(basedir)
    is_writable = ~temp.EXISTS || $
                  (temp.WRITE && ~temp.SYMLINK) || $
                  (temp.WRITE && temp.SYMLINK && keyword_set(allow_linked_source_directories))
    if ~is_writable then begin
      print, sourcename[ii], F='(%"\nSource %s is protected; skipping ...")'
      continue
    endif 

    
    print, F='(%"\n===================================================================")'
    unnamed_stats = headfits(unnamed_src_stats_fn, ERRMSG=error)
    if keyword_set(error) then begin
      print, error
      message, 'ERROR reading '+unnamed_src_stats_fn
    endif
    
    print, sourcename[ii], strtrim(psb_xpar( unnamed_stats,'LABEL'),2), merge_subdir[ii], F='(%"Source: %s (%s) %s")'

    if (NOT file_test(merged_src_spectrum_fn)) then begin
      print, 'WARNING! SOURCE SKIPPED: could not find spectrum ', merged_src_spectrum_fn
      continue
    endif
    
    file_mkdir, sourcedir + modelsubdir
    
    ;; Wait if we cannot create a lock file.
    catch, error_code
    if   (error_code NE 0) || file_test(lock_fn)  then begin
      if (error_code NE 0) then begin
        print, !ERROR_STATE.MSG
        if (!ERROR_STATE.NAME EQ 'IDL_M_FILE_CNTSYMLINK') then print, 'Error name is IDL_M_FILE_CNTSYMLINK'
      endif else print, 'Blocked by lock file ('+lock_fn+') ...  Will return to this source later.'
      
      processed_flag[ii] = 0
      catch, /CANCEL
      continue
    endif 
    file_link, session_name, lock_fn
    catch, /CANCEL

    minutes_blocked       = 0


    ;; Decide which fit statistic will be used, based on CSTAT_EXPRESSION passed by caller.
    ; CSTAT_EXPRESSION in an IDL expression, which can reference the source.photometry table as "bt".
    cstat = 0B
    if keyword_set(cstat_expression) then begin
      bt = mrdfits(photometry_fn, 1)
      ; Select the first energy band.
      bt = bt[0]
      ; Assign a value to "cstat" by evaluating the expression passed.
      if  ~execute( 'cstat ='+string(cstat_expression) ) then message,  'CSTAT_EXPRESSION cannot be evaluated.'
    endif ;CSTAT_EXPRESSION passed



    ;; Build a name for the model using the basename of MODEL_FILENAME and if supplied
    ;; appending the basename of MODEL_CHANGES_FILENAME.
    fdecomp, model_filename, disk, dir, base_model_name
    
    model_name = base_model_name
  
    if keyword_set(model_changes_filename) then begin
      fdecomp, model_changes_filename, disk, dir, model_changes_name, model_changes_qual
      
      custom_model_name = strjoin([model_name,model_changes_name], '_')

      ; Allow local files in the source directory to override the pathnames specified in MODEL_CHANGES_FILENAME.
      local_override_filename = sourcedir + modelsubdir + model_changes_name + '.' + model_changes_qual
      
      model_changes_to_run = strarr(n_elements(model_changes_filename))
      for jj=0,n_elements(model_changes_filename)-1 do begin
        if                 file_test(local_override_filename[jj]) then begin
          model_changes_to_run[jj] = local_override_filename[jj]
        endif else if      file_test(model_changes_filename[jj]) then begin
          model_changes_to_run[jj] = model_changes_filename[jj]
        endif
      endfor
      
      ; If any change scripts cannot be found, then skip the fitting run.
      ind = where(model_changes_to_run EQ '', count)
      if (count GT 0) then begin
        print, 'WARNING! SOURCE SKIPPED because script '+model_changes_filename[ind]+' could not be found.'
        file_delete, lock_fn, /ALLOW_NONEXISTENT
        continue
      endif
      
      if (n_elements(model_changes_filename) EQ 1) then begin
        fit_custom_fn = model_changes_to_run 
      endif else begin
        ; Concatenate all the model_changes_to_run files.
        fit_custom_fn = tempdir + 'model_changes.xcm'
        cmd = string(strjoin(model_changes_to_run, ' '), fit_custom_fn, F='(%"cat %s >! %s")')
        run_command, /UNIX, cmd, /QUIET
      endelse
      
      if file_test(fit_custom_fn) then begin
        model_name = custom_model_name
        print, 'CUSTOMIZATIONS to xcm script:'
        run_command, /UNIX, 'cat '+fit_custom_fn, result, /QUIET
        forprint, result, F="(%'  > %s')"
        print
      endif else fit_custom_fn = ''
    endif else fit_custom_fn = ''

    if keyword_set(interactive) then model_name = model_name + '_interactive'
    
    
    if keyword_set(grouped_spectrum_pattern[ii]) then begin
      ;; Use an existing grouped spectrum specified by caller (e.g. a pileup-corrected spectrum).
      grouped_spectrum_fn = file_search(sourcedir + grouped_spectrum_pattern[ii], COUNT=count)
      if (count EQ 0) then begin
        print, 'WARNING! SOURCE SKIPPED because grouped spectrum '+grouped_spectrum_pattern[ii]+' could not be found.'
        file_delete, lock_fn, /ALLOW_NONEXISTENT
        continue
      endif
      
      if (count GT 1) then begin
        print, 'WARNING! SOURCE SKIPPED because more than one grouped spectrum matching '+grouped_spectrum_pattern[ii]+' were found.'
        file_delete, lock_fn, /ALLOW_NONEXISTENT
        continue
      endif
      
      theader = headfits(grouped_spectrum_fn, EXT=1)
      this_snr_goal = psb_xpar(theader, 'SNR_GOAL')
      num_groups    = psb_xpar(theader, 'NUMGRPS' )
      grp_name = strcompress(string(this_snr_goal, F='(%"grp%8.1f")'),/REMOVE_ALL)
      inband_src_counts        = 0             ; Photometry is not relevant for pileup-corrected spectra.
      inband_scaled_bkg_counts = !VALUES.F_NAN ; Photometry is not relevant for pileup-corrected spectra.
      
      ; We're fitting a grouped spectrum, so we can NOT use Cstat as the fit statistic.
      cstat = 0B
      
      print, 'Using existing grouped spectrum '+grouped_spectrum_fn
      
    endif else begin
      ;; Group the spectrum.
      ;; Even if /CSTAT we will use this grouped spectrum for plotting.

      ; SNR_RANGE[1] is the user's goal for defining groups; SNR_RANGE[0] is the lower limit allowed before we abort the grouping attempt
      case n_elements(snr_range) of
       0: this_snr_range = (keyword_set(cstat)) ? [1,3] : [1,3]
       2: this_snr_range = snr_range
       else:begin
            print, 'ERROR: keyword SNR_RANGE should be a 2-element vector giving the range of SNR allowed for each spectral group, e.g. [2.5,5].'
            GOTO, FAILURE
            end
      endcase
      if (this_snr_range[1] LT 0) then begin
        print, 'ERROR: minimum SNR value (SNR_RANGE[1]) must be positive'
        GOTO, FAILURE
      endif
      
      ; If NUM_GROUPS_RANGE parameter not passed, adopt default values.
      ; We generally find it convenient to have the *same* defaults for CHISQ and CSTAT fitting,
      ; to facilitate comparison when both styles of fit are run on the same data.
      case n_elements(num_groups_range) of
       0: this_num_groups_range = (keyword_set(cstat)) ? [2+8,250] : [2+8,250]
       2: this_num_groups_range = num_groups_range
       else:begin
        print, 'ERROR: keyword NUM_GROUPS_RANGE should be a 2-element vector specifying how many spectral groups are desired, e.g. [2+8,250].'
            GOTO, FAILURE
            end
      endcase

      ;; We let ae_group_spectrum assign the filename grouped_spectrum_fn.
      grouped_spectrum_fn = ''
  
      ae_group_spectrum, merged_src_spectrum_fn, $
                         (keyword_set(group_without_background) ? '' : merged_bkg_spectrum_fn), $
                         grouped_spectrum_fn, $
                         CHANNEL_RANGE=channel_range, $
                         SNR_RANGE=this_snr_range, NUM_GROUPS_RANGE=this_num_groups_range, $
                         CREATOR_STRING=creator_string, $
                         this_snr_goal, grp_name, channel_starting_group, num_groups, inband_src_counts, inband_scaled_bkg_counts
      
   
      if (num_groups LT this_num_groups_range[0]) then begin
        print, num_groups-2, F='(%"WARNING! only %d unignored groups in spectrum.")'
        
        if NOT keyword_set(cstat) then begin
          print, 'WARNING! SOURCE SKIPPED: You might wish to try the C-statistic option for this source.'
          file_delete, lock_fn, /ALLOW_NONEXISTENT
          continue
        endif
      endif
    endelse
  
    
    if keyword_set(cstat) then begin
      ; C-stat fitting on raw spectrum; plot grouped spectrum via extra_spectrum_filename.
      fit_result_root         = 'nogrp_' + model_name
      merged_src_spectrum_fn  = file_basename(merged_src_spectrum_fn)
      ignore_spec            = string(channel_range+[-1,1], F='(%"1-%i,%i-**")')
      
      extra_spectrum_filename = file_basename(grouped_spectrum_fn)
      extra_ignore_spec      = string(num_groups, F='(%"1,%d")')
    endif else begin
      ; Chi^2 fitting on grouped spectrum; plot UNgrouped spectrum via extra_spectrum_filename.
      extra_spectrum_filename = file_basename(merged_src_spectrum_fn)
      extra_ignore_spec      = string(channel_range+[-1,1], F='(%"1-%i,%i-**")')

      fit_result_root         = grp_name + '_' + model_name
      merged_src_spectrum_fn  = file_basename(grouped_spectrum_fn)
      ignore_spec            = string(num_groups, F='(%"1,%d")')
    endelse
    
    fit_xcm_fn             = modelsubdir +fit_result_root + '.xcm'

        
    ; Define the energy range over which the background spectrum will influence the fit.
    if keyword_set(cstat) then begin
      ; Read RMF so that we can convert between detector channels and energy.
      ae_channel_energy_and_arf, merged_rmf_fn, merged_arf_fn, $
        channel_number, channel_lowenergy, channel_highenergy, channel_midenergy

      ; Originally, we decided that the bkg fitting range should be WIDER than that used for the source spectrum (channel_range), since the instrumental background extends beyond the limits of Chandra sensitivity to X-rays, and any bkg data lying just beyond the limits on the source spectrum help constrain the bkg at/near those energies.
      ; However doubts about that strategy have arisen:
      ; 1. Since the (scaled) background model is part of the model for the source spectrum, when those two
      ;    use different energy ranges the icounts plots in XSPEC are damaged.  I'm not sure if the fit 
      ;    itself is damaged, but the resulting icounts plots are definitely misleading.
      ; 2. As Figure 6.25 in the POG shows, instrumental background has strong gradients outside the 
      ;    standard 0.5:8 keV range for the source spectrum.  I worry that trying to model those slopes might
      ;    compromise modeling inside the energy range we care about!
      ; 3. When the background spectrum is dominated by X-rays (not instrumental events), as in T-ReX,
      ;    modeling those spectral features is possibly more important than modeling the ends of the 
      ;    instrumental background.  An expanded energy range for the bkg spectrum might "waste" precious
      ;    cplinear verices.

      bkg_channel_range = channel_range
      bkg_ignore_spec   = string(bkg_channel_range+[-1,1], F='(%"1-%i,%i-**")')
    endif else begin
      ; This is the chi^2 case.
      bkg_channel_range = channel_range
      bkg_ignore_spec   = extra_ignore_spec
    endelse
      
    
    ; Read the bkg spectrum.
    bin_table = mrdfits(merged_bkg_spectrum_fn, 1, theader, /SILENT, STATUS=status)
    if (status NE 0) then message, 'ERROR reading ' + merged_bkg_spectrum_fn
       
    channels            = bin_table.CHANNEL
    bkg_observed_counts = bin_table.COUNTS
    
    ; Remove background counts outside the energy band used for fitting the background, and count them.
    ind = where((channels LT bkg_channel_range[0]) OR (channels GT bkg_channel_range[1]))
    bkg_observed_counts[ind] = 0
    inband_bkg_counts = total(/INT, bkg_observed_counts)
    
    
    if keyword_set(cstat) then begin
      ; Information about fitting ungrouped Poisson spectrum with Poisson background spectrum can be found in:
      ;   Appendix B of XSPEC manual
      ;   https://giacomov.github.io/Bias-in-profile-poisson-likelihood/
      ;   http://heasarc.gsfc.nasa.gov/docs/xanadu/xspec/wstat.ps

      max_cplinear_vertices = 10
      ; We need to choose up to max_cplinear_vertices energies at which the cplinear bkg model will place a vertex.
      ; In an earlier version (3394 2009-04-01) we tried placing the vertices at energies midway *between* the 
      ; observed events (using ae_group_spectrum), however we ultimately decided to try placing them at observed
      ; energies in hopes of reducing the chance that XSPEC would peg a cplinear rate parameter at zero, which can 
      ; wreak havoc with the fitting process.  (The cplinear model cannot be driven to zero at the observed energies
      ; because Cstat would blow up to infinity.)
            
      ; The first/last vertices are placed at the first/last non-zero channels in the bkg spectrum's fitting range.
      ; Note that cplinear (cplinear_model.cxx) implements flat extrapolation beyond the first/last vertex.
      
      ; The concerns above about about cplinear being driven to zero at a vertex are probably obsolete, 
      ; because the 10 rate parameters are logarithms, and zero rates cannot be specified.
      
      ; Identify which channels in the energy band have data; these are candidate locations for cplinear vertex energies.
      ind = where(bkg_observed_counts GT 0, num_nonzero_channels)
      
      if (num_nonzero_channels LE 1) then begin
        ; The normal algorithm below won't work for cases with 0 or 1 nonzero channels.
        ; When there are zero bkg counts, AE's fitting scripts will not attempt to build a background model. 
        ; When there is one bkg count, we arbitrarily define vertices at each end of the bkg energy range.
        vertex_channels = bkg_channel_range


      endif else begin
        ; Choose the energies (channels) for a set of cplinear vertices to model the background spectrum.
        ; We believe that those vertices should be concentrated at the brightest regions of the spectrum.
        f_nan = !VALUES.F_NAN

        ; We have a historical preference for placing cplinear vertices in channels that have non-zero counts,
        ; so from here on we work with channel/count vectors that exclude channels with zero counts.
        ind = where(bkg_observed_counts GT 0, num_nonzero_channels)
        
        nonzero_channels            = channels           [ind]
        nonzero_bkg_observed_counts = bkg_observed_counts[ind]
        
        ; Defensively destroy the vectors channels and bkg_observed_counts so that we cannot 
        ; mistakenly use them with indexes that refer to vectors nonzero_channels and nonzero_bkg_observed_counts.
        channels            = f_nan
        bkg_observed_counts = f_nan
    
    
        ; The vertex selection algorithm below is based on the cumulative distribution of counts in the
        ; background spectrum.  We refer to those distribution values as "percentiles".
        nonzero_channel_percentile = total(/CUMULATIVE, nonzero_bkg_observed_counts) / $
                               float(total(/INTEGER   , nonzero_bkg_observed_counts))
        
        num_vertices = 10 < num_nonzero_channels
    
        ; Below, vertices we be referenced by their indexes into the following arrays:
        ;  nonzero_channels
        ;  nonzero_bkg_observed_counts
        ;  nonzero_channel_percentile
        
        ; Define vertices at the first and last non-zero channel.
        vertex_indexes = [0, num_nonzero_channels-1]
    
        ; Define additional vertices one at a time, so that each new vertex is 
        ; maximally distant (in count percentile) from the existing vertices.
        while (n_elements(vertex_indexes) LT num_vertices) do begin 
          ; For each channel not already hosting a vertex, calculate shortest "distance" to any defined vertex.
          ; Here, "distance" means difference in channel percentile.
          max_distance_to_a_vertex = replicate(99.9, num_nonzero_channels)
          max_distance_to_a_vertex  [vertex_indexes] = f_nan 
          
          masked_percentiles = nonzero_channel_percentile
          masked_percentiles[vertex_indexes] = f_nan
          
          
          ; Loop over existing vertices.
          foreach this_vertex_percentile, nonzero_channel_percentile[vertex_indexes]  do begin
            max_distance_to_a_vertex = $
              max_distance_to_a_vertex < abs( replicate(this_vertex_percentile, num_nonzero_channels) - masked_percentiles)
          endforeach
          
          ; Find the remaining vertex that is farthest from all adopted vertexes.
          max_d = max(max_distance_to_a_vertex,imax)
          if ~finite(max_d) || (max_d GT 1) || (max_d EQ 0) then message, 'Bug in cplinear vertex algorithm!'
          
          ; Define a vertex.
          vertex_indexes = [vertex_indexes, imax]
        endwhile
        ; Sort vertex_indexes.
        vertex_indexes = vertex_indexes[sort(vertex_indexes)]

        ; Look up vertex channel numbers.
        vertex_channels = nonzero_channels[vertex_indexes]
      endelse ; num_nonzero_channels GT 1
      
      ; Convert vertex channel numbers to units of energy.
      ; The odd construct below ensures that vertex_energy has the length expected by cplinear,
      ; even if we have defined fewer vertices for this spectrum.
      vertex_energy    = replicate(-1.0, max_cplinear_vertices) 
      vertex_energy[0] = interpol(channel_midenergy, channel_number, vertex_channels)
    
      if (num_nonzero_channels LE 1) then begin
        print, F='(%"\nWARNING: only one channel in bkg spectrum has counts; bkg will be modeled by cplinear vertices at the ends of its energy range.")'
      endif else begin
        print, inband_bkg_counts, num_vertices, F='(%"\n%d in-band bkg counts will be modeled by %d cplinear vertices at the folowing count percentiles & energies:")'
        forprint, nonzero_channel_percentile[vertex_indexes], vertex_energy, F='(%"  %0.2f  %0.2f keV")'
      endelse

    endif ; keyword_set(cstat)


    ;; Create XSPEC script.
    ;; Prefix contains source-specific commands.
    openw,  xcm_unit, sourcedir + fit_xcm_fn, /GET_LUN
    printf, xcm_unit, merged_src_spectrum_fn,       F='(%"set spectrum_filename       \"%s\"")'
    if (extra_spectrum_filename NE "") then $
      printf, xcm_unit, extra_spectrum_filename,    F='(%"set extra_spectrum_filename \"%s\"")'
    printf, xcm_unit,     ignore_spec,              F='(%"set     ignore_spec         \"%s\"")'
    printf, xcm_unit, bkg_ignore_spec,              F='(%"set bkg_ignore_spec         \"%s\"")'
    if (extra_spectrum_filename NE "") then $
      printf, xcm_unit, extra_ignore_spec,          F='(%"set extra_ignore_spec       \"%s\"")'
    printf, xcm_unit, fit_result_root,              F='(%"set model_name              \"%s\"")'
    printf, xcm_unit, keyword_set(cstat),           F='(%"set c_stat_flag               %d")'
    
    if (psb_xpar( unnamed_stats, 'DIFFUSE')) then begin
      printf, xcm_unit, inband_src_counts-inband_scaled_bkg_counts, psb_xpar( unnamed_stats, 'SRC_AREA')/3600., $
                                                      F='(%"set spectrum_description    \"NC=%d in %0.1f sq''\"")'
    endif else begin
      printf, xcm_unit, inband_src_counts, inband_scaled_bkg_counts, $
                                                      F='(%"set spectrum_description    \"NC=%d-%0.1f\"")'
    endelse
    
    if keyword_set(cstat) then begin
      printf, xcm_unit, strjoin(string(vertex_energy,F='(%"%0.2f")'),' '), $
                                                    F='(%"set cplinear_energies        {%s}")'
    endif
    printf, xcm_unit, codedir+'xspec_scripts',      F='(%"set model_directory         \"%s\"")'

    printf, xcm_unit, keyword_set(interactive),     F='(%"set interactive_flag          %d")'
    
    free_lun, xcm_unit

    ; Append MODEL_FILENAME to XSPEC script prefix using "sed" to insert any user-supplied customizations to the model.
    if (fit_custom_fn EQ '') then begin
      cmd = string(model_filename, sourcedir + fit_xcm_fn, F="(%'cat %s >>! %s')")
    endif else begin
      cmd = string(fit_custom_fn, model_filename, sourcedir + fit_xcm_fn, F="(%'sed -e ""/AE CUSTOMIZATIONS/r %s"" %s >>! %s')")
    endelse
    run_command, /UNIX, cmd
    
    
    ;; Perform the fit.
    ae_perform_fit, sourcedir, fit_result_root, INTERACTIVE=keyword_set(interactive), TEMPDIR=tempdir, FIT_TIMEOUT=fit_timeout

    file_delete, lock_fn, /ALLOW_NONEXISTENT
  endfor ;kk
endrep until (num_to_process EQ 0) OR (minutes_blocked GE 30)

if (num_to_process GT 0) then begin
  print
  print, 'WARNING!  These sources could not be processed.  Perhaps lock files remain from a previous session ...'
  forprint, SUBSET=source_index, sourcename
endif
endif ;keyword_set(fit_spectra)


print, F='(%"\nAE exited normally.")'

CLEANUP:
if file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif

if (exit_code EQ 0) then return $
else begin
  print
  print, 'acis_extract: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP

END  ; END of acis_extract


; =============================================================================
;;; Tool that constructs rectangles around the readout streaks of each source supplied.
;;; The code uses a simple fixed width for the rectangle; something based on the PSF would be better!

;;; ASPECT_FN is an aspect file for the observation; required by dmcoords.

; THE CIAO TOOL acis_streak_map MAY MAKE THIS TOOL OBSOLETE ....

PRO ae_make_streak_regions, catalog_or_srclist, obsname, obsdata_filename, region_filename, ASPECT_FN=aspect_fn, $
                  EXTRACTION_NAME=extraction_name


obs_stats_basename       = 'obs.stats'

temproot = temporary_directory( 'ae_make_streak_regions', VERBOSE=0, SESSION_NAME=session_name)
tempdir = temproot

run_command, PARAM_DIR=tempdir

ymargin = 32
xsize   = 10

  ;; Input catalog should be an ascii file with source names. 
  ;; 
  readcol, catalog_or_srclist, sourcename, FORMAT='A', COMMENT=';'
  num_sources = n_elements(sourcename)

if keyword_set(extraction_name) then extraction_subdir = extraction_name + '/' $
                                else extraction_subdir = ''
if (n_elements(extraction_subdir) EQ 1) then extraction_subdir = replicate(extraction_subdir,num_sources>1)

  openw, region_unit, region_filename, /GET_LUN
  for ii = 0L, num_sources-1 do begin
    print, F='(%"\n===================================================================")'
    print, sourcename[ii], F='(%"\nSource: %s")'
    obsdir    = sourcename[ii] + '/' + obsname + '/' + extraction_subdir[ii]
    stats_fn  = obsdir + obs_stats_basename
    stats = headfits(stats_fn, ERRMSG=error)
    if keyword_set(error) then begin
      print, sourcename[ii] + ' was not observed.'
      continue
    endif

    ; Find the nominal src position in chip coordinates.
    dmcoords_cmd = string(obsdata_filename, aspect_fn, psb_xpar( stats, 'X_CAT'), psb_xpar( stats, 'Y_CAT'),  F="(%'dmcoords %s asolfile=%s opt=sky x=%f y=%f')")
    run_command, dmcoords_cmd

    run_command, /QUIET, 'pget dmcoords chipx chipy chip_id', dmcoords_result
    ; Parse the string returned by pget with the ON_IOERROR mechanism enabled in order to find type conversion errors.
    ON_IOERROR, TYPE_CONVERSION_ERROR1
    chipx   = float(dmcoords_result[0])
    chipy   = float(dmcoords_result[1])
    chip_id = float(dmcoords_result[2])
    if (0) then begin
      TYPE_CONVERSION_ERROR1:
      print, !ERROR_STATE.MSG
      print, 'ERROR: dmcoords results could not be parsed.'
      forprint, ['  The dmcoords call was : ', '    '+dmcoords_cmd   ]
      forprint, ['  The output of pget was: ', '    '+dmcoords_result]
      GOTO, CLEANUP
    endif 
    ON_IOERROR, NULL
help, chip_id
    
    ; Compute the center of the ROTBOX.
    cmd = string(obsdata_filename, aspect_fn, chipx, 512.5,  F="(%'dmcoords %s asolfile=%s opt=chip chipx=%f chipy=%f')")
    run_command, cmd

    run_command, /QUIET, 'pget dmcoords x y', result
    xcenter = float(result[0])
    ycenter = float(result[1])
    
    ; Compute angle of chip system wrt sky system.
    cmd = string(obsdata_filename, aspect_fn, chipx, 1,  F="(%'dmcoords %s asolfile=%s opt=chip chipx=%f chipy=%f')")
    run_command, cmd

    run_command, /QUIET, 'pget dmcoords x y', result
    x0 = float(result[0])
    y0 = float(result[1])
    
    angle = !RADEG * atan(x0-xcenter,ycenter - y0)
help, angle
    
    printf, region_unit, xcenter, ycenter, xsize, 1024+ymargin, angle, F='(%"rotbox(%0.1f,%0.1f,%0.1f,%0.1f,%0.1f)")'
  endfor
    
  free_lun, region_unit
  
CLEANUP:
if file_test(temproot) then begin
  list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, temproot
endif
return
end ; ae_make_streak_regions



; ; =============================================================================
; ; Obsolete tool to estimate the cropping in the CIAO PSF library.
; PRO estimate_cropping
; 
; repeat begin
;   session_name = string(random()*1E7, F='(I7.7)')
;   temproot = 'AE' + session_name +'.noindex/'
;   temproot = filepath(temproot, /TMP)
; endrep until (NOT file_test(temproot))
; file_mkdir, temproot
; tempdir = temproot
; 
; run_command, PARAM_DIR=tempdir
; 
; arcsec_per_skypixel = 0.492 
; 
; f2lib = "$CALDB/data/chandra/acis/cpf/2dpsf/acisi1998-11-052dpsf2N0002.fits"
; temp_image_fn = "f2.psf"
; 
; f2_pixsiz = 12/24.0
; 
; energy_table = mrdfits(f2lib, 2, energy_header, /SILENT, STATUS=status)
; psf_energy = energy_table.energy
; 
; 
; for ii = 0,1 do begin
;   case ii of
;     0: lib = { name:'acisi1998-11-052dpsf1N0002.fits', dim:256, pixsiz:6/24.0, $
;                        elevation_samples:[0,5,10], crop_fraction:fltarr(3,n_elements(psf_energy)) }
; 
;     1: lib = { name:'acisi1998-11-052dpsf3N0002.fits', dim:512, pixsiz:2/24.0, $
;                        elevation_samples:[0,5], crop_fraction:fltarr(2,n_elements(psf_energy)) }
;   endcase
;   
;   print, lib
;   
;   for jj=0, n_elements(lib.elevation_samples)-1 do begin
;     dety  = 4096.5 - lib.elevation_samples[jj] * (60/arcsec_per_skypixel)  ; arcmin
;     
;     for kk=0, n_elements(psf_energy)-1 do begin
;       ; Make a full-sized F2 PSF, using its natural binning.
;       cmd = string( f2_pixsiz, f2_pixsiz, 512, 512, psf_energy[kk], 4096.5, dety, $
;                     f2lib, temp_image_fn, $
;                     F="(%'mkpsf coord=DET binspax=%6.4f binspay=%6.4f sizeoutx=%d sizeouty=%d energy=%f x=%7.2f y=%7.2f psflibfile=""%s"" outfile=%s outpsffile="""" ')")
; 
;       run_command, cmd
; 
;       psf_img = readfits(temp_image_fn, psf_header, /SILENT)
;       
;       ; Extract the section of the F2 image covered by the library under test.
;       field_of_view = lib.dim * lib.pixsiz / f2_pixsiz   ; F2 image pixels
;       dim           = round(field_of_view)
;       corner        = (psb_xpar( psf_header,'NAXIS1') / 2) - (dim/2)
;       
;       lib_img = psf_img[corner:corner+dim-1, corner:corner+dim-1]
;       tvscl, lib_img < max(lib_img)/10.
;       
;       central_power = total(lib_img, /DOUBLE)
;       total_power   = total(psf_img, /DOUBLE)
;       
;       lib.crop_fraction[jj,kk] = (total_power - central_power) / total_power
; 
; help, dim, central_power, total_power
;     endfor ;kk
; 
;   endfor ;jj
;   
;   fxhmake,  pheader, /EXTEND, /DATE, /INITIALIZE
;   psb_xaddpar, pheader, 'CREATOR', "estimate_cropping, $Revision: 5658 $"  
;   psb_xaddpar, pheader, "FNFITS",  lib.name
;   psb_xaddpar, pheader, "HDUNAME", 'CROP_FRACTION'
;   
;   writefits, lib.name, lib.crop_fraction, pheader
;   
;   mwrfits, energy_table, lib.name, energy_header
; 
;   elevation_table = {elevation: lib.elevation_samples}
;   fxbhmake, theader, n_elements(elevation_table), 'ELEVATIONS', /DATE
;   psb_xaddpar, theader, 'CREATOR', "estimate_cropping, $Revision: 5658 $"  
;   
;   mwrfits, elevation_table, lib.name, theader
; 
; endfor ;ii
; 
; CLEANUP:
; if file_test(temproot) then begin
;   list = reverse(file_search(temproot,'*',/MATCH_INITIAL_DOT,COUNT=count))
;   if (count GT 0) then file_delete, list
;   file_delete, temproot
; endif
; 
; return
; end



;; DESIGN NOTES

; Are we content to linearly extrapolate the PSF_FRAC(energy)?

; HEXTRACT.pro does NOT adjust PHYSICAL coordinates.  Here's how to do that manually:
; See http://iraf.noao.edu/projects/ccdmosaic/Imagedef/imagedef.html for definition
; of LTV? and LTM?_? keywords.
;          crpix = [psb_xpar( this_header, 'CRPIX1P'), psb_xpar( this_header, 'CRPIX2P')] - trim
;          ltv   = [psb_xpar( this_header, 'LTV1'),    psb_xpar( this_header, 'LTV2')]    - trim
;          
;          hextract, this_psf_img, this_header, trim, xdim-1-trim, trim, ydim-1-trim, /SILENT
;          psb_xaddpar, this_header, 'CRPIX1P', crpix[0]
;          psb_xaddpar, this_header, 'CRPIX2P', crpix[1]
;          psb_xaddpar, this_header, 'LTV1',    ltv[0]
;          psb_xaddpar, this_header, 'LTV2',    ltv[1]

;offaxis angle:              ~0:5'  ~6:7'  ~8:11'  ~12:12.4' ~13:20'
;ACIS-I library:             F3     F1     F1      F1,F2     F2
;PSF pix size (skypixel):  0.0833 0.25   0.5     0.5       1.0
;PSF dimension:              515    259    130     130,515   259
;
;Corners of I-array are 12.4' off axis.

