;;; $Id: tara_smooth.pro 5639 2021-12-11 19:05:41Z psb6 $

;;; This script is closely tied to the workflow shown in the recipe diffuse_procedure.txt.

;;; The parameter 'scene_name' is a scalar string specifying the first part of the filenames of the images you want to smooth.

;;; The parameter 'significance' is a scalar or vector number specifying the desired SNR of the smoothed flux image. 
;;; The significance computations assume there is no error on the background map!!

;;; By default, adaptive kernel smoothing with a Gaussian kernel is used.  Specify /TOPHAT to use a tophat kernel.

; If you wish to smooth all images using the same set of kernels that were  used in a previous run, then supply the pathname to that map (e.g. 'soft_med/sig015/tophat/fullfield.diffuse_filled.radius') via the optional parameter FIXED_RADIUS_MAP_FN.
; You can preserve the nominal smoothing runs (rather than overwriting them) by supplying RUN_NAME, e.g. RUN_NAME='kernels_from_soft_med'.


;---------------------------------------------------------------------

;;; Specify /WVT_BINNING to use the WVT Binning algorithm (Diehl 2006).
;     KEEPFIXED: [Input] Vector (size 2 x n_fixed) containing x and y image coordinates (0-based) of the bin generators 
;                that you want to keep fixed in their position, e.g. KEEPFIXED=[ [100,100], [200,400], [500,300] ]. 
;                The binning algorithm will move all other bins around as usual. 
;                Example use: Keep one bin fixed on the center of a galaxy.

;;; For example,
;;;    tara_smooth, 'iarray', [5,10,15], /TOPHAT
;;; will smooth images named "*/iarray.diffuse.img" to a SNR of 15 using a tophat kernel.

;;; Optionally, you can specify the bands (directories) you want to smooth.
;;;   BAND_NAME=['soft_band','hard_band','full_band','scale_band']
;;;   BAND_NAME='s*_band'
;;; The default is
;;;   BAND_NAME='*'
@acis_extract

PRO tara_smooth, scene_name, SUFFIX=suffix, significance, BAND_NAME=band_name, RUN_NAME=run_name, $
             MASK_FILENAME=mask_filename,$
             PLOT_ONLY=plot_only, DISCARD_EXISTING=discard_existing, SHOW=show, $
             TOPHAT=tophat, MAX_RADIUS=max_radius, FIXED_RADIUS_MAP_FN=fixed_radius_map_fn, $
             WVT_BINNING=wvt_binning, KEEPFIXED=keepfixed

creator_string = "tara_smooth, version " +strmid("$Rev:: 5639  $",7,5) +strmid("$Date: 2021-12-11 12:05:41 -0700 (Sat, 11 Dec 2021) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()

if ~keyword_set(band_name) then band_name = '*'

if n_elements(suffix) EQ 0 then suffix = '.diffuse'

ds9_params = '-geometry 1400x1000 -view magnifier no -view panner no -view info yes -view filename yes -view physical no -view image no'

target_name = getenv('TARGET')
if ~keyword_set(target_name) then target_name = 'Unknown Target'


; Choose a smoothing kernel, 1 for Gaussian kernal, 0 for tophat.
gaussian_not_tophat =  ~keyword_set(tophat)

; Find the observed data images to be smoothed.
obs_img_fn  = file_search(band_name, scene_name+suffix+'.img', COUNT=num_bands)
if (num_bands EQ 0) then begin
  print, 'tara_smooth: ERROR: No files were found matching ', band_name+'/'+scene_name+suffix+'.img'
  return
endif

signal_fn   = file_dirname(obs_img_fn)+'/'+scene_name+'.signal'
 noise_fn   = file_dirname(obs_img_fn)+'/'+scene_name+'.noise'

; Look for corresponding exposure maps.
emap_fn        = file_dirname(obs_img_fn)+'/'+scene_name+suffix+'.emap'
emap_term      = emap_fn
emap_available = file_test(emap_fn)

; Look for corresponding background images.
bkg_fn        = file_dirname(obs_img_fn)+'/'+scene_name+'.bkg.img'
bkg_term      = bkg_fn
bkg_available = file_test(bkg_fn)

ind = where(~bkg_available, count)
if (count GT 0) then begin
  print, 'WARNING: the following exposure map and/or background files are missing:'
  forprint, SUBSET=ind, bkg_fn, F='(%"%s")'
  
  answer=dialog_message(/QUESTION,/CENTER, TITLE='tara_smooth', 'WARNING: the listed background files are missing.  If you want to smooth without those backgrounds then answer ''yes''. \nIf you want to abort the smoothing run answer ''no''.')
  
  if (answer EQ 'No') then retall
  
  bkg_fn   [ind] = ''
  bkg_term [ind] = '0'
endif

print, 'Smoothing: '
forprint, obs_img_fn, bkg_term, emap_term, F="(%'(%s - %s) / %s')"
print

; Read the boolean mask file that depicts the combined field of view of the observations.
; Point source masking is NOT depicted in this image.
; For backward compatibility we try two different names for the mask file.
mask_fn = keyword_set(mask_filename) ? mask_filename : scene_name+'.mask'
!ERROR_STATE.CODE = 0
default_field_mask   = (readfits(mask_fn, /SILENT) GT 0)
if (!ERROR_STATE.CODE LT 0) then begin
  mask_fn     = scene_name+'_mask.img'
  !ERROR_STATE.CODE = 0
  default_field_mask   = (readfits(mask_fn, /SILENT) GT 0)
  if (!ERROR_STATE.CODE LT 0) then begin
    print, 'ERROR: could not read field mask from '+scene_name+'.mask'+' or '+mask_fn
    retall
  endif else print, 'Read field mask from '+mask_fn+' instead.'
endif




;---------------------------------------------------------------------
; Make directories to hold the output, such as sig11/gauss/
; Read any kernel radius map supplied.
smooth_dirs = (keyword_set(run_name) ? run_name+'/' : '')

if keyword_set(fixed_radius_map_fn) then begin
  !ERROR_STATE.CODE = 0
  fixed_radius_map = readfits(fixed_radius_map_fn)
  if (!ERROR_STATE.CODE LT 0) then begin
    print, 'ERROR: could not read '+fixed_radius_map_fn
    retall
  endif

  significance = 0 ; flag that kernels are pre-determined
endif else begin
  fixed_radius_map = 0
  smooth_dirs += string(significance, F='(%"sig%3.3d/")')
endelse

if keyword_set(wvt_binning) then begin
  smooth_dirs += 'wvt/'
endif else begin
  smooth_dirs += (gaussian_not_tophat ? 'gauss/' : 'tophat/') 
endelse


for ii=0,num_bands-1 do begin

  
  ;---------------------------------------------------------------------
  ; Read the masked emap (=0 near point sources).
  !ERROR_STATE.CODE = 0
  obs_emap    = readfits(emap_fn[ii])
  if (!ERROR_STATE.CODE LT 0) then begin
    print, 'ERROR: could not read '+emap_fn[ii]
    retall
  endif
  
  ; Defensively, verify that default_field_mask has the same dimensions as emap.
  if ~ARRAY_EQUAL(size(obs_emap, /DIMEN), size(default_field_mask, /DIMEN)) then begin
    print, emap_fn[ii], mask_fn, F='(%"ERROR: emap and field mask (%s, %s) must have the same dimensions.")'
    help, obs_emap, default_field_mask
    retall
  endif
  
  ;---------------------------------------------------------------------
  ; Modify the default field mask as needed for this energy band.
  field_mask = default_field_mask
  
  if keyword_set(wvt_binning) then begin
    ; Since the ultimate input to WVT_BINNING.pro is a flux image (not separate data and emap images), we MUST distinguish pixels that have zero flux because no counts happened to be observed there from pixels that have zero flux because no observation was made there (off-field, or masked regions).
    ; All unobserved pixels must be zero in the field_mask map---we can NOT "smooth over" small holes!!  
    
    ; I choose to also exclude pixels where the emap is very small, because flux calculations there will be very noisy.
    ; This is an arbitrary precaution whose benefit I have not quantified.
    typical_emap_value = median(obs_emap[where(obs_emap GT 0)])
    emap_threshold = 0.01 * typical_emap_value
    print, emap_threshold, F='(%"Ignoring pixels where exposure map is < %0.2g")'
    ind = where(obs_emap LT emap_threshold, count)
    if (count GT 0) then begin
      field_mask[ind] = 0
    endif
    
    suffix = ''
  endif else begin
    ; The inputs to the adaptive smoothing code are able to represent two separate concepts:
    ; 1. pixels where there has been no observation (emap <= 0 OR emap == NaN)
    ; 2. pixels where we do NOT wish to compute an answer (field_mask == 0), i.e. the pixels that should be null in the output image.
  
    ; Extrapolate over any small gaps in the field mask between pointings.
    ; NOTE THAT THIS OPTION WILL REDUCE THE SIZE OF ANY MASKING YOU HAVE DONE BY HAND IN fullfield.mask.
    ;boxcar_width = 5
    ;threshold    = 1/float(boxcar_width)^2

    ;field_mask = field_mask OR (smooth(float(field_mask), boxcar_width, /EDGE_TRUNCATE) GT threshold)
    
    ; Decide how you want to handle the holes in the data.  
    ; OPTION I: retain the holes.
    ;  field_mask = field_mask AND (obs_emap GT 0)
    ;  suffix     = '_nofill'
    
    ; OPTION II: extrapolate over the small holes in the emap remaining from source masking.
    ;boxcar_width = 5
    ;threshold    = 1/float(boxcar_width)^2
    ;field_mask = field_mask AND (smooth(float((obs_emap GT 0)), boxcar_width, /EDGE_TRUNCATE) GT threshold)
    ;suffix     = '_fill5'
    
    ; OPTION III: extrapolate over all the holes.
    field_mask = field_mask 
    suffix     = '_filled'
    
    ;tvscl, field_mask
  endelse ; adaptive smoothing

;---------------------------------------------------------------------
  ; Read observed and stowed background images.
  fdecomp, obs_img_fn[ii], disk, path, name, qual
  
  out_dirs = path+smooth_dirs
  file_mkdir, out_dirs
  
  !ERROR_STATE.CODE = 0
  obs_img = readfits(obs_img_fn[ii], obsdata_header)
  if (!ERROR_STATE.CODE LT 0) then begin
    print, 'ERROR: could not read '+obs_img_fn[ii]
    retall
  endif

  extast, obsdata_header, img2wcs_astr
  arcsec_per_imgpixel = abs(img2wcs_astr.CDELT[0] * img2wcs_astr.CD[0,0])*3600.0  

  if bkg_available[ii] then begin
    !ERROR_STATE.CODE = 0
    bkg_img = readfits(bkg_fn[ii])
    if (!ERROR_STATE.CODE LT 0) then begin
      print, 'ERROR: could not read '+obs_img_fn[ii]
      retall
    endif
  endif else bkg_img = make_array(DIMENSION=size(obs_img, /DIMENSION), /FLOAT)
  
  ; Sometimes data appear in pixels with zero exposure, presumably due to
  ; differences between the reprojection of event data and the reprojection
  ; of masked exposure map images.
  ind = where( ((obs_emap LE 0) OR ~finite(obs_emap)) AND (obs_img GT 0), count)
  if (count GT 0) then begin
    print, total(obs_img[ind]), F='(%"WARNING: %d observed events removed from pixels with zero exposure.")'
    obs_img   [ind] = 0
  endif

  ind = where( ((obs_emap LE 0) OR ~finite(obs_emap)) AND (bkg_img GT 0), count)
  if (count GT 0) then begin
    print, total(bkg_img[ind]), F='(%"WARNING: %d scaled stowed events removed from pixels with zero exposure.")'
    bkg_img[ind] = 0
  endif
  

;---------------------------------------------------------------------
  if keyword_set(wvt_binning) then begin
    ; Run the WVT Binning tool.
    
    for jj=0,n_elements(smooth_dirs)-1 do begin
      perform_smoothing = ~keyword_set(plot_only)
    
      tesselate_dir= out_dirs[jj] + 'tesselates/'
      file_mkdir, tesselate_dir
      out_fn       = out_dirs[jj] + name + suffix + ['.flux', '.signif', '.binnumber','.sqrt.flux']
      region_fn    = out_dirs[jj] + name + suffix +                            ".wvtbin.reg"
      tesselate_fn = tesselate_dir+ name + suffix + string(1+indgen(9000),F='(%".wvtbin%4.4d.reg")' )
      print
      if file_test(out_fn[0]) && ~keyword_set(discard_existing) then begin
        print, out_fn[0], F='(%"%s already exists; skipping this smoothing run.")'
        perform_smoothing = 0
      endif
      
      if perform_smoothing then begin
        ; Remove any existing region files.
        file_delete, tesselate_fn, /QUIET

        print, 'Building ' + out_fn[0]
        help,obs_img,bkg_img,obs_emap,field_mask
        
        ; On April 3, 2006 Steven Diehl recommended using the MAX_AREA input to limit the size of bins in regions with little signal.
        ; This is also discussed in the manual, S6.1.8.
        max_area = n_elements(obs_img) * 0.05
    
        ; Make sure the input images are zero in masked pixels!
        mask_ind = where(/NULL, field_mask LE 0)
        obs_img [mask_ind] = 0
        bkg_img [mask_ind] = 0
        obs_emap[mask_ind] = 0
        
        
        psb_xaddpar, obsdata_header, 'CREATOR' , creator_string
        psb_xaddpar, obsdata_header, 'TARGETSN', significance[jj]
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , 'photon /cm**2 /s /ImagePixel**2', 'photon surface brightness'

        if keyword_set(keepfixed) then $
          print, n_elements(keepfixed)/2, F='(%"\n%d fixed tessellate centers have been declared.")'
        
        WVT_OBSERVED_AND_BACKGROUND_IMAGES, obs_img, bkg_img, obs_emap, significance[jj], CTSIMAGE=obs_img, MASK=field_mask, MAX_AREA=max_area, KEEPFIXED=keepfixed, binned_data, xnode, ynode, weight, BINNUMBER=binnumber, SNBIN=snbin, PLOTIT=2

        flux_map           = binned_data
        flux_map[mask_ind] = !VALUES.F_NAN
        
        ; For reasons unknown, the WVT code returns BINNUMBER as a DOUBLE array.  In several places below our code  assumes it has integer values, so let's round it to integers now.
        binnumber           = round(binnumber)
        binnumber[mask_ind] = -1
        
        
        ; The default_field_mask specifies the set of pixels that the caller would like to end up in tessellates.
        ; The WVT call above was given a more restrictive mask (field_mask) that excludes pixels with small or zero exposure, in an effort to avoid numerical problems.
        ; Here, we expand the tessellate boundaries so that every pixel selected by default_field_mask is in some tessellate.
        xdim = (size(/DIMENSIONS, binnumber))[0]
        ydim = (size(/DIMENSIONS, binnumber))[1]
        repeat begin
          ; Find the bin numbers in the four neighboring pixels.
          ; Our convention is that -1 is the "null" value (pixel is not in a tessellate).
          neighborA           = shift(binnumber, 0, 1)
          neighborA[*,0]      = -1
          neighborB           = shift(binnumber, 1, 0)
          neighborB[0,*]      = -1
          neighborC           = shift(binnumber, 0,-1)
          neighborC[*,ydim-1] = -1
          neighborD           = shift(binnumber,-1, 0)
          neighborD[xdim-1,*] = -1
        
          foreach kk, permute(4) do begin
            case kk of
              0: begin
                 ind = where(/NULL, default_field_mask AND (binnumber EQ -1) AND (neighborA NE -1), count1)
                 binnumber[ind] = neighborA[ind]
                 end
        
              1: begin
                 ind = where(/NULL, default_field_mask AND (binnumber EQ -1) AND (neighborB NE -1), count2)
                 binnumber[ind] = neighborB[ind]
                 end
        
              2: begin
                 ind = where(/NULL, default_field_mask AND (binnumber EQ -1) AND (neighborC NE -1), count3)
                 binnumber[ind] = neighborC[ind]
                 end
        
              3: begin
                 ind = where(/NULL, default_field_mask AND (binnumber EQ -1) AND (neighborD NE -1), count4)
                 binnumber[ind] = neighborD[ind]
                 end
            endcase
          endforeach
        
          print, count1,count2,count3,count4, F='(%"Filling holes in tessellates (%d+%d+%d+%d pixels).")'
        endrep until (count1+count2+count3+count4 EQ 0)

        ; Create a significance map by applying the bin significance values (vector SNBIN) to the appropriate image pixes, as described by the BINNUMBER map.
        snr_map  = make_array(DIMENSION=size(obs_img, /DIMENSION), /FLOAT)
        nbins    = n_elements(snbin)
        bin_area = HISTOGRAM(binnumber, REVERSE_INDICES=r, min=1, max=nbins)
        
        save, /COMPRESS, FILE=out_dirs[jj] + name + suffix + '.sav'

        print, 'Building WVT region files ...'
        ; Open a master region file that will contain all the tesselate regions.
        openw,  unit1, region_fn, /GET_LUN
        printf, unit1, "# Region file format: DS9 version 4.1"
        printf, unit1, 'global width=1 font="helvetica 12 normal"'
        printf, unit1, "fk5"
        
        if keyword_set(keepfixed) then begin
          ; Make regions marking any bin generators specified by the caller.
          !TEXTUNIT = unit1  
          ; The xy2ad routine expects 0-based pixel coordinates.
          xy2ad, keepfixed[0,*], keepfixed[1,*], img2wcs_astr, ra_pt, dec_pt
          forprint, TEXTOUT=5, /NoCOM, ra_pt, dec_pt, F='(%"cross point %10.6f %10.6f # tag={bin generator} color=red")'
        endif
        
        for kk=0L, nbins-1 do begin
          if (bin_area[kk] GT 0) then begin
            ind = r[r[kk]:r[kk+1]-1]
          
            snr_map[ind] = snbin[kk]
            
            ; Create a ds9 region file that outlines this bin.
            openw,  unit2, tesselate_fn[kk], /GET_LUN
            printf, unit2, "# Region file format: DS9 version 4.1"
            printf, unit2, 'global width=1 font="helvetica 12 normal"'
            printf, unit2, "fk5"

            this_binnumber = kk+1
            contour, binnumber EQ this_binnumber, /CLOSED, LEVELS=[1],PATH_XY=xy, PATH_INFO=info, /PATH_DATA_COORDS

            ; Process each contour returned above (because a tesselate may not be a contiguous region).
            num_contours = n_elements(info)
            for cc=0, 0 > (num_contours-1) do begin
            
              if (num_contours EQ 0) then begin
                ; The contour routine could not deal with this bin (e.g. it's very small).
                print, this_binnumber, F='(%"WARNING! Contour algorithm failed on bin %d; using bounding box, which may not be perfect.")'
                index_to_point, where(binnumber EQ this_binnumber), X, Y, size(binnumber)
                
                polygon_x = [min(X)-0.5, max(X)+0.5, max(X)+0.5, min(X)-0.5]
                polygon_y = [min(Y)-0.5, min(Y)-0.5, max(Y)+0.5, max(Y)+0.5]

              endif else begin
  
                ; Extract the 0-based pixel coordinates of the contour polygon.
                ind = info[cc].offset + indgen(info[cc].N)
                polygon_x = float(round(reform(xy[0, ind])))
                polygon_y = float(round(reform(xy[1, ind])))
                
                ; Enlarge the polygons so that they touch each other.
                ; The best way to do this seems to be to move horizontal segments up/down and vertical segments left/right by 0.5 pixel.
                for this=0,n_elements(polygon_x)-1 do begin
                  next = (this+1) MOD n_elements(polygon_x)
                  
                  if (polygon_x[this] EQ polygon_x[next]) then begin
                    ; Determine whether this vertical segment defines the right or left edge of the tesselate.
                    midpoint_y        = round(0.5*(polygon_y[this] + polygon_y[next]))
                    left_of_polygon_x = polygon_x[this] - 2
                    
                    if (left_of_polygon_x LT 0) then begin
                      is_left_edge = 1
                    endif else begin
                      is_left_edge = (binnumber[left_of_polygon_x, midpoint_y] NE this_binnumber) 
                    endelse
                    
                    polygon_x[[this,next]] += is_left_edge ? (-0.5) : 0.5 
                  endif
                  
                  if (polygon_y[this] EQ polygon_y[next]) then begin
                    ; Determine whether this horizontal segment defines the lower or upper edge of the tesselate.
                    midpoint_x      = round(0.5*(polygon_x[this] + polygon_x[next]))
                    below_polygon_y = polygon_y[this] - 2
                    
                    if (below_polygon_y LT 0) then begin
                      is_bottom_edge = 1
                    endif else begin
                      is_bottom_edge = (binnumber[midpoint_x, below_polygon_y] NE this_binnumber) 
                    endelse
                    
                    polygon_y[[this,next]] += is_bottom_edge ? (-0.5) : 0.5 
                  endif
                endfor
              endelse
              
              ; The xy2ad routine expects 0-based pixel coordinates.
              xy2ad, polygon_x, polygon_y, img2wcs_astr, polygon_ra, polygon_dec
  
              ; We use sexagesimal format in the region file so they can be a CIAO filter.
              sexagesimal_coords = strjoin(adstring_formatted( polygon_ra, polygon_dec, PRECISION=2), ', ')
  
              region = 'polygon(' + sexagesimal_coords + ')'
  
              ; Write this region both to its own file, and to the master region file.
              printf, unit1, region, this_binnumber, this_binnumber, F='(%"%s # move=0 text={%4.4d} tag={wvtbin} tag={wvtbin%4.4d}")' 
              printf, unit2, region, this_binnumber, this_binnumber, F='(%"%s # move=0 text={%4.4d} tag={wvtbin} tag={wvtbin%4.4d}")' 
  
            endfor ; cc
            free_lun, unit2
            if (num_contours GT 1) then print, this_binnumber, num_contours,$
              F='(%"\nINFORMATION: Tesselate %d spans %d separate regions.")'
          endif
        endfor ;kk
        free_lun, unit1
        snr_map[mask_ind] = !VALUES.F_NAN
      
        ; Save flux, error, & radius images.

        ; Scale the flux image to convert the units from photon /cm**2 /s /ImagePixel**2 to photon /cm**2 /s /arcsec**2.
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , 'photon /cm**2 /s /arcsec**2', 'photon surface brightness'
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'flux_map'
        writefits, out_fn[0], float(     flux_map*(1.0 / arcsec_per_imgpixel)^2), obsdata_header
        
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , 'sqrt(photon /cm**2 /s /arcsec**2)', 'photon surface brightness'
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'flux_map'
        writefits, out_fn[3], float(sqrt(flux_map*(1.0 / arcsec_per_imgpixel)^2)), obsdata_header
        
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , ''
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'snr_map'
        writefits, out_fn[1], float(      snr_map), obsdata_header
        
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , ''
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'binnumber'
        writefits, out_fn[2], binnumber, obsdata_header
        
        bt = replicate({xnode:xnode[0], ynode:ynode[0], weight:weight[0], snbin:snbin[0]}, nbins)
        bt.xnode  = xnode
        bt.ynode  = ynode
        bt.weight = weight
        bt.snbin  = snbin

        ; Unit specifications follow the standard in "Specification of Physical Units within OGIP FITS files" at
        ; http://heasarc.gsfc.nasa.gov/docs/heasarc/ofwg/docs/general/ogip_93_001/ogip_93_001.html
        fxbhmake, theader, nbins, 'WVT_BINS', 'outputs from wvt_image.pro', /DATE, /INIT
        psb_xaddpar, theader, 'CREATOR' , creator_string
        psb_xaddpar, theader, 'TARGETSN', significance[jj]
        psb_xaddpar, theader, 'TUNIT1', "", 'location of the WVT bin generators, 0-based pixel index'
        psb_xaddpar, theader, 'TUNIT2', "", 'location of the WVT bin generators, 0-based pixel index'
        psb_xaddpar, theader, 'TUNIT3', "", 'weight of the WVT bin generators'
        psb_xaddpar, theader, 'TUNIT4', "", 'actual bin SNR'
        mwrfits, bt, out_fn[2], theader
        
        print, nbins, F='(%"\nINFORMATION: WVT Binning produced %d tesselates.")'
      endif ;perform_smoothing
      
      cmd = string(ds9_params, target_name, out_dirs[jj], bkg_fn[ii], obs_img_fn[ii], out_fn[2], region_fn, emap_fn[ii], out_fn[0], out_fn[1], region_fn, F="(%'ds9 %s -title ""%s: %s"" -lock frame wcs -scale mode 99.0 %s -linear %s -log %s -linear  -region %s  %s %s -log %s -region %s -linear -scale mode minmax -zoom to fit  >& /dev/null &')")

      if keyword_set(show) then run_command, cmd
      print, cmd, F="(%'\n Review WVT images with:\n  %s')"
    endfor ;jj
    
;---------------------------------------------------------------------
  endif else begin
    ;; Run our adaptive smoothing tool.
    
    if ~keyword_set(max_radius) then begin
      ; We arbitrarily set the maximum kernel to cover a certain fraction of the scene, 
      if gaussian_not_tophat then max_area = n_elements(obs_img) * 0.064 $
      else                        max_area = n_elements(obs_img) * 0.06
      
      max_radius = ceil(sqrt(max_area/!PI))
    endif
    
    for jj=0,n_elements(smooth_dirs)-1 do begin
      perform_smoothing = ~keyword_set(plot_only)
    
      out_fn    = out_dirs[jj] + name + suffix + ['.flux', '.signif', '.radius','.sqrt.flux']
      region_fn = out_dirs[jj] + name + suffix +  '.reg'
      print
      if file_test(out_fn[0]) && ~keyword_set(discard_existing) then begin
        print, out_fn[0], F='(%"%s already exists; skipping this smoothing run.")'
        perform_smoothing = 0
      endif
      
      if perform_smoothing then begin
        print, 'Building ' + out_fn[0]
        help,obs_img,bkg_img,obs_emap,field_mask,max_radius,fixed_radius_map
    
        adaptive_density_2d, obs_img, /ACCEPT_FLOAT_DATA, significance[jj], MAX_RADIUS=max_radius, MAX_NUM_KERNELS=(max_radius<100), FIXED_RADIUS_MAP=fixed_radius_map, EMAP=obs_emap, BACKGROUND_MAP=bkg_img, FIELD_MASK=field_mask, GAUSS=gaussian_not_tophat, flux_map, error_map, radius_map
        
        significance_map = float(flux_map/error_map)

        ; Warn user if many pixels failed to achieve requested significance.
        percentile = 10
        num_fluxes = total(/INT, finite(significance_map))
        value_at_percentile = significance_map[ (sort(significance_map))[ num_fluxes * (percentile/100.) ] ] 
        
        if (value_at_percentile LT significance[jj]) then begin
          note = keyword_set(fixed_radius_map) ? '' : string(max_radius, F='(%"  Setting MAX_RADIUS >%d may help.")')
          print, percentile, '"'+band_name[ii]+'"', value_at_percentile, median(significance_map), note, F='(%"\nWARNING: %d%% of %15s fluxes have SNR < %4.1f; median SNR = %4.1f. %s")'
        endif
        
        
        ; Save flux, error, & radius images.
        psb_xaddpar, obsdata_header, 'CREATOR' , creator_string
        psb_xaddpar, obsdata_header, 'TARGETSN', significance[jj]

        ; Scale the flux image to convert the units from photon /cm**2 /s /ImagePixel**2 to photon /cm**2 /s /arcsec**2.
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , 'photon /cm**2 /s /arcsec**2', 'photon surface brightness'
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'flux_map'
        writefits, out_fn[0], float(     flux_map*(1.0 / arcsec_per_imgpixel)^2), obsdata_header

        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , 'sqrt(photon /cm**2 /s /arcsec**2)', 'photon surface brightness'
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'flux_map'
        writefits, out_fn[3], float(sqrt(flux_map*(1.0 / arcsec_per_imgpixel)^2)), obsdata_header
        
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , ''
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'snr_map'
        writefits, out_fn[1], significance_map, obsdata_header
 
        psb_xaddpar, obsdata_header, 'BSCALE'  , 1.0
        psb_xaddpar, obsdata_header, 'BUNIT'   , 'ImagePixel'
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'radius_map'
        writefits, out_fn[2], radius_map        , obsdata_header

        ; Report median kernel radius, and create a few circular regions to depict kernels across the field.
        median_kernel_radius = median(radius_map[where(radius_map NE -1)]) * arcsec_per_imgpixel ; arcsec
        
        print, median_kernel_radius, F='(%"\nMedian kernel radius is %d arcsec.\n")'


        ; Find radius_map values for a set of evenly-spaced percentiles.
        percentiles = [0,10,20,30,40,50,60,70,80,90,100]

        num_radius_values = total(/INT, finite(radius_map))
        value_at_percentile = radius_map[ (sort(radius_map))[ (num_radius_values-1) * (percentiles/100.) ] ]

;       value_at_percentile = [min(/NAN,radius_map), value_at_percentile, max(/NAN,radius_map)]
        
        ind_unique = uniq(value_at_percentile)
        num_regions = n_elements(ind_unique)
        value_at_percentile  = value_at_percentile [ind_unique]
                 percentiles =          percentiles[ind_unique]

        ; Among the set of radius_map pixels close to each percentile, choose one pixel to be the site of
        ; a region depicting the corresponding kernel region.
        ind_regions = lonarr(num_regions)
        for ii=0, num_regions-1 do begin
          ind_this_group = where( abs(radius_map - value_at_percentile[ii]) LT 1, num_this_group )
          
          ind_regions[ii] = ind_this_group[ (num_this_group-1)*random() ]
        endfor ;ii

        temp = array_indices(radius_map, ind_regions)

        xy2ad, temp[0,*], temp[1,*], img2wcs_astr, ra_pt, dec_pt
 
        forprint, TEXTOUT=region_fn, ra_pt, dec_pt, arcsec_per_imgpixel * radius_map[ind_regions], percentiles, F='(%"fk5;circle %12.8f %12.8f %f\" # tag={kernel region} text={%dth percentile}")', /NoCOMMENT

      
        ;---------------------------------------------------------------------
        ;; Build a pair of images that represent 90% confidence intervals for the flux image, to visually represent its uncertainty.
        ;; Store them as additional HDUs in the flux FITS file.
        sxdelpar, obsdata_header, ['SIMPLE','EXTEND']
        psb_xaddpar, obsdata_header, 'XTENSION', 'IMAGE   ', BEFORE='BITPIX'
        
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'flux down limit'
        writefits, out_fn[0], /APPEND, float(flux_map - 1.644*error_map), obsdata_header
        psb_xaddpar, obsdata_header, 'HDUNAME' , 'flux up limit'
        writefits, out_fn[0], /APPEND, float(flux_map + 1.644*error_map), obsdata_header

        cmd = string(ds9_params, target_name, out_dirs[jj] + ' 90% flux confidence interval', max(flux_map, /NAN)/10, out_fn[0], out_fn[0], F="(%'ds9 %s -title ""%s: %s"" -scale limits 0 %g -log ""%s[1]"" ""%s[2]""  -zoom to fit -lock scale yes -lock colorbar yes -lock frame image  >& /dev/null &')")
        ;if keyword_set(show) then run_command, cmd
        print, cmd, F="(%'\n Review 90%% flux confidence interval images with:\n  %s')"

        
;        confidence_interval_image = make_array(DIMENSION=size( /DIMENSIONS, flux_map), /FLOAT)
;  
;        ; Build 2-D arrays holding the x and y indexes of the images.
;        xindex = lindgen( (size(/DIMENSIONS, flux_map))[0] )
;        yindex = lindgen( (size(/DIMENSIONS, flux_map))[1] )
;        make_2d, xindex, yindex
;        
;        ; 90% confidence interval for a Gaussian is +-1.644*sigma.
;        ind1 = where( (xindex+yindex) MOD 2, count1, COMPLEMENT=ind2, NCOMPLEMENT=count2 )
;        
;        confidence_interval_image[ind1] = float(flux_map[ind1] + 1.644*error_map[ind1])
;        confidence_interval_image[ind2] = float(flux_map[ind2] - 1.644*error_map[ind2])
;        
          
      endif ;perform_smoothing
      
      cmd = string(ds9_params, target_name, out_dirs[jj], bkg_fn[ii], obs_img_fn[ii],region_fn, out_fn[2], emap_fn[ii], out_fn[0],region_fn, out_fn[1], significance[jj]-1,significance[jj]+1, F="(%'ds9 %s -title ""%s: %s"" -lock frame wcs -scale mode 99.0 %s -linear %s -region %s -log %s -linear %s %s -region %s -log %s -linear -scale limits %0.1f %0.1f -zoom to fit  >& /dev/null &')")

      if keyword_set(show) then run_command, cmd
      print, cmd, F="(%'\n Review smoothed images with:\n  %s')"
    endfor ;jj
  endelse ; adaptive smoothing
endfor ;ii

return
end



;  idl |& tee -a patch_flux_image.log
;  .r tara_smooth
;  .run
;  foreach file, file_search('*/data/extract/adaptive_smoothing/*/*/*/*.flux') do begin
;    print
;    print, file
;    patch_flux_image, file
;  endforeach
;  end


PRO patch_flux_image, filename
  !ERROR_STATE.CODE = 0
  flux_map = readfits(filename, obsdata_header)
  if (!ERROR_STATE.CODE LT 0) then begin
    print, 'ERROR: could not read '+filename
    retall
  endif

  extast, obsdata_header, img2wcs_astr
  arcsec_per_imgpixel = abs(img2wcs_astr.CDELT[0] * img2wcs_astr.CD[0,0])*3600.0  

  ; Scale the flux image to convert the units from photon /cm**2 /s /ImagePixel**2 to photon /cm**2 /s /arcsec**2.
  cmd1 = string(filename, (1.0 / arcsec_per_imgpixel)^2,$
                 F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=BSCALE value=""%0.4f"" comment=""pix /arcsec**2""')")

  cmd2 = string(filename, 'photon /cm**2 /s /arcsec**2',$
                 F="(%'dmhedit infile=""%s[1]"" filelist=none operation=add key=BUNIT value=""\'%s\'"" comment=""photon surface brightness""')")

  print, cmd1
  print, cmd2
  run_command, cmd1 +' ; '+ cmd2

  ;psb_xaddpar, obsdata_header, 'BSCALE'  , (1.0 / arcsec_per_imgpixel)^2, 'pix /arcsec**2'
  ;psb_xaddpar, obsdata_header, 'BUNIT'   , 'photon /cm**2 /s /arcsec**2', 'photon surface brightness'
  ;psb_xaddpar, obsdata_header, 'HDUNAME' , 'flux_map'
  ;writefits, filename,  flux_map, obsdata_header

return
end

