;;; This is an automated version of the Townsley ACIS analysis recipe image_construction.txt.

;;; $Id: build_scene.pro 5572 2020-12-14 15:25:11Z psb6 $

;;; The event data to be merged is specified either by a wildcard pattern passed as PATTERN (default 'obsid*/acis.validation.evt2'), 
;;; or by an explicit set of pathnames passes as OBS_EVENT_FN.

;;; Normally, each input event list will have a corresponding exposure map with a name corresponding to the template_name.
;;; An alternate name for the emaps can be specified by EMAP_BASENAME.
;;; If no emaps are available, supply EMAP_BASENAME=''.

;;; The input template_name must be a string scalar containing any ONE of these scene names:  
;;;   'fullfield','sarray','s3','FI','iarray','central_1', 'core_0.5'.
;;; These names refer to pre-defined fields of view and binning defined with respect to tangentplane_reference_fn.

;;; The individual ObsID data can be filtered using CIAO filter syntax passed via the SPATIAL_FILTER, which is applied to the individual ObsID data BEFORE reprojection.  For example, a filter expressed in PHYSICAL coordinates can select off-axis ranges.  A filter expressed in celestial coordinates using sexagesimal format MAY work to specify a region on the sky. 

;;; When the optional /CREATE_TEMPLATE is supplied, this tool constructs scene templates (FITS images) with names <scene name>_template.img.
;;; If /CREATE_TEMPLATE is omitted, then scene templates with those names are expected to already exist.

;;; When /CREATE_IMAGE is specified, a scene exposure map is created, with corresponding images of the data within energy bands
;;; specified by IMAGE_FILTER_SPEC and IMAGE_NAME, both of which have default values.
;;; The names of the emap and image files can be modified with the optional input SUFFIX.

;;; If MERGED_EVENTFILE is supplied then the reprojected event lists are combined and saved to the specified file.
;;; MERGED_COLUMNLIST can be used to supply a column filter that's applied to the merged event list.
;;; The merged event list is NOT spatially filtered by the scene.
;;; An energy filter for the merged event list can be supplied via MERGED_FILTERSPEC.

@acis_extract
@match_xy

PRO build_scene, template_name, SUFFIX=suffix, SPATIAL_FILTER=spatial_filter, $

                 PATTERN=event_file_pattern, OBS_EVENT_FN=obs_event_fn, $

                 EMAP_BASENAME=emap_basename_p, MASK_BASENAME=mask_basename_p, $

                 MERGED_EVENTFILE=merged_eventfile, MERGED_COLUMNLIST=merged_columnlist, MERGED_FILTERSPEC=merged_filterspec, $

                 CREATE_TEMPLATE=create_template, TEMPLATE_FN=template_fn,$
                 DESIRED_IMAGE_PIXELS=desired_image_pixels, $

                 CREATE_IMAGE=create_image, IMAGE_FILTER_SPEC=image_filter_spec, IMAGE_NAME=image_name, $

                 SUM_RATES_NOT_COUNTS=sum_rates_not_counts,$

                 ; Parameter RESOLUTION is passed to CIAO reproject_image to control the quality of projection.
                 ; RESOLUTION specifies "number of points per side of polygon" outlining each output pixel.
                 ; 0 -> center of output pixel mapped to a single input pixel
                 ; 1 (default) -> 4 corners of pixel
                 ; 2 -> 4 corners + 4 midpoints
                 RESOLUTION=resolution,$ 
                 DMOPT=dmopt, SHOW=show

creator_string = "build_scene, version"+strmid("$Date: 2020-12-14 08:25:11 -0700 (Mon, 14 Dec 2020) $", 6, 11)
print, creator_string, F='(%"\n\n%s")'
print, now()
exit_code = 0


tangentplane_reference_fn = 'tangentplane_reference.evt'

target_name = getenv('TARGET')
if ~keyword_set(target_name) then target_name = 'Unknown Target'

;; ------------------------------------------------------------------------
;; Create a unique scratch directory.
tempdir = temporary_directory( 'build_scene.', VERBOSE=1, SESSION_NAME=session_name)

temp_text_fn             = tempdir + 'temp.txt'
dmmerge_header_lookup_fn = tempdir + 'dmmerge_header_lookup.txt'

run_command, PARAM_DIR=tempdir

run_command, /QUIET, ['pset dmcopy clobber=yes','pset reproject_events clobber=yes','pset dmimgcalc lookupTab=none clobber=yes','pset reproject_image clobber=yes']
 
if ~keyword_set(suffix)             then suffix=''
if ~keyword_set(event_file_pattern) then event_file_pattern = 'obsid*/acis.validation.evt2'

if (n_elements(image_filter_spec) EQ 0) then image_filter_spec = ['energy=500:7000', 'energy=2000:7000', 'energy= 500:2000']
if (n_elements(image_name)        EQ 0) then image_name        = [      'full_band',        'hard_band',        'soft_band']

image_name = strtrim(image_name,2)

num_images = n_elements(image_name)
if (n_elements(image_filter_spec) NE num_images) then begin
  print, 'ERROR: IMAGE_FILTER_SPEC and IMAGE_NAME must have the same number of elements.'
  GOTO, FAILURE
endif


if ~keyword_set(template_fn) then $
  template_fn   =                template_name+       '_template.img'


if ~keyword_set(create_template) && ~file_test(template_fn) then begin
  print, template_fn, F="(%'\nERROR: cannot find template image %s.')"
  GOTO, FAILURE
endif

if ~keyword_set(desired_image_pixels) then desired_image_pixels = 1000L^2


case template_name of
 'fullfield': $
    begin
    reproject_filter = ''
    regrid_filter    = ''
    emap_basename    = 'fullfield_1.emap'
    end
 'sarray': $
    begin
    if keyword_set(create_template) then begin
      ; We want the sarray template to be defined using all the reprojected CCDs (empty reproject_filter, same as the fullfield template).
      ; We'll filter again later during the image generation, using regrid_filter.
      reproject_filter = ''
      regrid_filter    = 'ccd_id>3,'
    endif else begin
      reproject_filter = 'ccd_id>3,'
      regrid_filter    = ''
    endelse
    emap_basename      = 'sarray_1.emap'
    end
 'FI' : $
    begin
    reproject_filter = 'ccd_id=0,1,2,3,4,6,8,9' 
    regrid_filter    = ''
    emap_basename    = 'FI_1.emap'
    end
 'BI' : $
    begin
    reproject_filter = 'ccd_id=5,7' 
    regrid_filter    = ''
    emap_basename    = 'BI_1.emap'
    end
 's3' : $
    begin
    reproject_filter = 'ccd_id=7'
    regrid_filter    = ''
    emap_basename    = 's3_1.emap'
    end
 'iarray' : $
    begin
    reproject_filter = 'ccd_id=0:3'
    regrid_filter    = ''
    emap_basename    = 'iarray_1.emap'
    end
 'central_1'  : $
    begin
    reproject_filter = 'ccd_id=0:3'
    regrid_filter    = ''
    emap_basename    = 'iarray_1.emap'
    end
 'core_0.5'   : $
    begin
    reproject_filter = 'ccd_id=0:3'
    regrid_filter    = ''
    emap_basename    = 'iarray_1.emap'
    end
 else: $
    begin
    reproject_filter = ''
    regrid_filter    = ''
    emap_basename    = 'obs.AI.full.emap'
    end
endcase

; Look for an EMAP_BASENAME input, including the case EMAP_BASENAME='' that directs us to skip the emap construction.
emap_specified_for_each_band = 0B 
case n_elements(emap_basename_p) of
  0         : emap_basename = ''
  1         : emap_basename = replicate(emap_basename_p, num_images)
  num_images: begin
              emap_basename =           emap_basename_p
              emap_specified_for_each_band = 1B
              end
  else: message, 'emap_basename must be a scalar string or a string array matching IMAGE_NAME.'
endcase

case n_elements(mask_basename_p) of
  0: mask_basename = ''
  1: mask_basename = mask_basename_p
  else: message, 'MASK_BASENAME must be a scalar string'
endcase

if keyword_set(obs_event_fn) then begin
  num_obs = n_elements(obs_event_fn)
endif else begin
  ;; Find the event data and emaps we have to work with.
  obs_event_fn = file_search(event_file_pattern, COUNT=num_obs)
  if (num_obs EQ 0) then begin
    print, 'NO event data found!'
    GOTO, FAILURE
  endif
endelse


reproj_obs_event_fn  = tempdir + string(indgen(num_obs), F='(%"reproj%d.evt")')
reproj_obs_emap_fn   = tempdir + string(indgen(num_obs), F='(%"reproj%d.emap")')
reproj_stowed_img_fn = tempdir + string(indgen(num_obs), F='(%"reproj%d.img")')

;; Choose an ObsID that will define the SKY aka PHYSICAL coordinate system for the target.
if ~file_test(tangentplane_reference_fn) then begin
  ;; Use the longest observation as the root tangent plane.
  exposure = fltarr(num_obs)
  for ii=0, num_obs-1 do begin
    header = headfits(obs_event_fn[ii], EXT=1)
    exposure[ii] = psb_xpar( header,'EXPOSURE')
  endfor ; ii
  dum = max(exposure, imax)
  
  file_delete, /QUIET, tangentplane_reference_fn
  file_link, obs_event_fn[imax], tangentplane_reference_fn
endif 

;; ========================================================================
;; Reproject event data for the desired set of CCDs onto the root tangent plane.
;; Even the root observation is passed through the reproject command in order to get CCD and spatial filtering.
this_filter = []
if keyword_set(reproject_filter) then this_filter = [this_filter, reproject_filter]
if keyword_set(  spatial_filter) then this_filter = [this_filter,   spatial_filter]
this_filter = keyword_set(this_filter) ? '['+strjoin(this_filter,',')+']' : ''

for ii=0, num_obs-1 do begin
  ; Reproject the data onto the reference tangent plane.
  ; We use random=0 because random=-1 produced an error message for stowed files, which don't have a TIME column.
  run_command, string(obs_event_fn[ii], this_filter, reproj_obs_event_fn[ii], tangentplane_reference_fn, F="(%'reproject_events ""%s%s"" %s match=%s random=0 aspect=none')")
endfor ; ii


;; ========================================================================
;; Combine the reprojected event lists.
if keyword_set(merged_eventfile) then begin
  column_spec = keyword_set(merged_columnlist) ? '[cols '+merged_columnlist+']' : ''
  filter_spec = keyword_set(merged_filterspec) ? '['     +merged_filterspec+']' : ''


  ; Modify a copy of CIAO's dmmerge_header_lookup.txt file, which controls how dmmerge combines keywords.
  ;  * Keywords derived from the GTI tables ("calcGTI") are "skipped" because they used the *first* GTI table, which
  ;    may not represent the data when data from ACIS-I and ACIS-S aimpoints are combined.
  ;  * Other keywords are "skipped" because they cannote be sensibly combined.

  ;awk '/calcGTI|RA_|DEC_|ROLL_|BTIM|DS_IDENT|OBJECT|OBS_ID|SEQ_NUM|OBI_NUM|TITLE|DATAMODE|OBSERVER/{print $1 "  SKIP"; next}; {print $0}'

  cmd = 'awk ''/calcGTI|RA_|DEC_|ROLL_|BTIM|DS_IDENT|OBJECT|OBS_ID|SEQ_NUM|OBI_NUM|TITLE|DATAMODE|OBSERVER/{print $1 "  SKIP"; next}; {print $0}'' ' +getenv('ASCDS_INSTALL')+'/data/dmmerge_header_lookup.txt > '+dmmerge_header_lookup_fn
  
  run_command, /QUIET, [cmd, 'pset dmmerge clobber=yes lookupTab='+dmmerge_header_lookup_fn]



  ; There's a bug in dmmerge in CIAO 3.3, 3.4 that messes up GTI tables. 
  ; We follow the instructions on the dmmerge bugs page (http://cxc.harvard.edu/ciao/bugs/dmmerge.html) 
  ; to ensure that the GTIs are going to be combined: append [subspace -expno] to each file specification.
  infile_stack = reproj_obs_event_fn+filter_spec+column_spec+'[subspace -expno,-sky]'
  print, F="(%'\nMerging these data:')"
  forprint, infile_stack
  forprint, infile_stack, TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
  
  run_command, string( temp_text_fn, merged_eventfile, F="(%'dmmerge ""@-%s"" %s clob+')")                   
endif


;; ========================================================================
;; Figure out the field of view and binning of the scene.
if keyword_set(create_template) then begin
  xymin =  !values.F_INFINITY 
  xymax = -!values.F_INFINITY 
  
  ;; Determine span of the reprojected data on the tangent plane (sky coordinate system XY).
  for ii=0, num_obs-1 do begin
    run_command, string(reproj_obs_event_fn[ii], F="(%'dmstat ""%s[cols x,y]"" median=no sigma=no verbose=0')")  

    run_command, /QUIET, 'pget dmstat out_min out_max', result
    xymin <= float(strsplit(result[0],',', /EXTRACT)) 
    xymax >= float(strsplit(result[1],',', /EXTRACT)) 
  endfor ; ii
  
  ; Calculate the square image pixel size that will cover the data with approximately the specified number of pixels.
  fov_area_skypix = (xymax[0]-xymin[0]) * (xymax[1]-xymin[1])
  
  skypixel_per_imagepixel = sqrt(fov_area_skypix / desired_image_pixels)

  ; Pad the data's field of view by a couple of image pixels.
  xymin -= 3 * skypixel_per_imagepixel
  xymax += 3 * skypixel_per_imagepixel

  ; Recalculate pixel size.
  fov_area_skypix = (xymax[0]-xymin[0]) * (xymax[1]-xymin[1])
  
  skypixel_per_imagepixel = sqrt(fov_area_skypix / desired_image_pixels)


  ; Calculate the (integer) dimensions of the image array.
  xdim = ceil((xymax[0]-xymin[0]) / skypixel_per_imagepixel)
  ydim = ceil((xymax[1]-xymin[1]) / skypixel_per_imagepixel)

  ; Create a CIAO binning specification of the form "xmin:xmax:delx" to ensure square pixels, which keeps CIAO happy.
  ; Try to avoid a mostly-empty last column/row by shaving a little off xmax/ymax.
  xmin = round(xymin[0])
  ymin = round(xymin[1])
  xmax = xmin + skypixel_per_imagepixel * xdim  - 0.01
  ymax = ymin + skypixel_per_imagepixel * ydim  - 0.01

  binspec = string(xmin,xmax,skypixel_per_imagepixel,ymin,ymax,skypixel_per_imagepixel, F='(%"x=%0.4f:%0.4f:%0.6f,y=%0.4f:%0.4f:%0.6f")')
  
  
  case template_name of
   'fullfield': $
      begin
      cmd = string(tangentplane_reference_fn, binspec, template_fn, F="(%'dmcopy ""%s[#row=1:10][bin %s]"" %s')")
      end
   'sarray': $
      begin
      cmd = string(tangentplane_reference_fn, binspec, template_fn, F="(%'dmcopy ""%s[#row=1:10][bin %s]"" %s')")
      end
   'FI' : $
      begin
      cmd = string(tangentplane_reference_fn, binspec, template_fn, F="(%'dmcopy ""%s[#row=1:10][bin %s]"" %s')")
      end
   's3' : $
      begin
      cmd = string(tangentplane_reference_fn, binspec, template_fn, F="(%'dmcopy ""%s[#row=1:10][bin %s]"" %s')")
      end
   'iarray' : $
      begin
      cmd = string(tangentplane_reference_fn, binspec, template_fn, F="(%'dmcopy ""%s[#row=1:10][bin %s]"" %s')")
      end
      
;   'central_1'  : $
;      begin
;      ; The two central scenes are centered on the aimpoint.
;      cen_x = 4096
;      cen_y = 4096
;      cmd = string(tangentplane_reference_fn, cen_x-512,cen_x+512,dimension[0], cen_y-512,cen_y+512,dimension[1], template_fn, F="(%'dmcopy ""%s[#row=1:10][bin x=%d:%d:#%d,y=%d:%d:#%d]"" %s')")
;      end
;      
;   'core_0.5'   : $
;      begin
;      ; The two central scenes are centered on the aimpoint.
;      cen_x = 4096
;      cen_y = 4096
;      cmd = string(tangentplane_reference_fn, cen_x-256,cen_x+256,dimension[0], cen_y-256,cen_y+256,dimension[1], template_fn, F="(%'dmcopy ""%s[#row=1:10][bin x=%d:%d:#%d,y=%d:%d:#%d]""  %s')")
;      end
      
   else: $
      begin
      cmd = string(tangentplane_reference_fn, binspec, template_fn, F="(%'dmcopy ""%s[#row=1:10][bin %s]"" %s')")
      end
  endcase
 run_command, cmd
endif ; keyword_set(create_template)

if ~keyword_set(create_image) then GOTO, CLEANUP


; IMAGE_NAME specifies the directory where images and emap will be written.
image_and_emap_dir = replicate('./', num_images)

ind = where(/NULL, image_name NE '')
if isa(ind, /INTEGER) then begin
  image_and_emap_dir[ind] = image_name[ind] + '/'
  
  file_mkdir, image_and_emap_dir[ind]
endif

merged_image_fn = image_and_emap_dir+template_name+suffix+'.img'
merged_emap_fn  = image_and_emap_dir+template_name+suffix+'.emap'
merged_mask_fn  =                    template_name+suffix+'.mask'


;; Find the binning that matches the template.
header = headfits(template_fn, ERRMSG=error )
if keyword_set(error) then begin
  print, error
  print, 'ERROR reading ' + template_fn
  GOTO, FAILURE
endif
template_xdim = psb_xpar( header, 'NAXIS1')
template_ydim = psb_xpar( header, 'NAXIS2')

run_command, string(template_fn, F="(%'get_sky_limits %s verbose=0 precision=2')")
run_command, /QUIET, 'pget get_sky_limits dmfilter', bin_spec
bin_spec = '[bin '+bin_spec[0]+']'

opt_spec = keyword_set(dmopt) ? '[opt '+dmopt+']' : '' 


;; ========================================================================
;; Apply energy filter to each reprojected eventlist, bin into an image, and sum.
terms = string(1+indgen(num_obs),F="(%'img%d')")

review_list = ''

; Loop over energy bands.
for jj=0, num_images-1 do begin
  ; ------------------------------------------------------------------------
  ;; Reproject the exposure maps.
  if emap_basename[jj] then begin

    obs_mask_fn = file_dirname(obs_event_fn)+'/'+mask_basename
    obs_emap_fn = file_dirname(obs_event_fn)+'/'+emap_basename[jj]
  
    num_emaps = total(/INT, file_test(obs_emap_fn))
    if (num_emaps NE num_obs) then begin
      print, 'ERROR: these emaps are missing: ', obs_emap_fn[where(~file_test(obs_emap_fn))]
      stop
      ;GOTO, FAILURE
    endif
    
    if ~keyword_set(resolution) then resolution = 1
    
    ; Note that the exposure map is a 2-D FUNCTION representing a physical quantity (typically, effective area
    ; multiplied by exposure time, in units of s cm^2 count /photon), not a 2-D HISTOGRAM that is counting something
    ; with "per pixel" units. Thus, when rebinning we choose method=average so that the scale of the exposure map is
    ; unchanged.
    print, F='(%"\n============================================================")'  
    if mask_basename then print, emap_basename[jj], F="(%'Masking and Reprojecting the %s Emaps\n')" $
                     else print, emap_basename[jj], F="(%'Reprojecting the %s Emaps\n')"
    ;forprint, obs_emap_fn
    this_filter = keyword_set(spatial_filter) ? '['+spatial_filter+']' : ''

    if mask_basename then begin
      ; Defensively, verify that emaps and masks have the same dimensions.
      must_abort = 0B
      
      for ii=0, num_obs-1 do begin
        emap = readfits(obs_emap_fn[ii], emap_header) ; observation ii
        mask = readfits(obs_mask_fn[ii])              ; observation ii
      
        if ~ARRAY_EQUAL(size(emap, /DIMEN), size(mask, /DIMEN)) then begin
          must_abort = 1B
          print, obs_mask_fn[ii], obs_emap_fn[ii], F='(%"ERROR: masked and unmasked emaps (%s, %s) must have the same dimensions.")'
        endif
    
      endfor ;ii
      if must_abort then GOTO, FAILURE
    endif ; mask_basename


    ; Process each ObsID's emap.
    for ii=0, num_obs-1 do begin

      ; Apply any specified single-ObsID mask file to this single-ObsID emap before reprojecting.
      if mask_basename then begin
        emap = readfits(obs_emap_fn[ii], emap_header) ; observation ii
        mask = readfits(obs_mask_fn[ii])              ; observation ii
        
        print, obs_emap_fn[ii], psb_xpar( emap_header, 'ENERGY'), F='(%"Masking %s (mono-energy=%0.1f keV)")'
        
        masked_obs_emap_fn  = tempdir + 'masked_obs.emap'
      
        ind = where(mask LE 0, count)
        if (count GT 0) then emap[ind] = 0
        writefits, masked_obs_emap_fn, emap, emap_header
      endif else masked_obs_emap_fn = obs_emap_fn[ii]
    
      ; If the SAME emaps are used for every band, then we can save time by reprojecting that set only once.
      if (jj EQ 0) || emap_specified_for_each_band then $
        run_command, string(masked_obs_emap_fn, this_filter, template_fn, reproj_obs_emap_fn[ii], resolution, $
                            F="(%'reproject_image  infile=""%s%s""  matchfile=%s  outfile=%s  method=average resolution=%d')")

      if keyword_set(sum_rates_not_counts) then begin
        ; We later want to scale the images of the event data by these emaps (dmimgcalc call below).
        ; To avoid NaN results we need to set the emap's zeros to a large number.
        emap = readfits(reproj_obs_emap_fn[ii], emap_header)
        ind = where(emap LE 0, count)
        if (count GT 0) then emap[ind] = !VALUES.F_INFINITY
        writefits, reproj_obs_emap_fn[ii], emap, emap_header
      endif
      
    endfor ;ii
  endif ; emap_basename defined


  if (image_filter_spec[jj] NE '') then begin
  
    print, image_name[jj], F='(%"\n============================================================\nBuilding %s Image")'  
  
    ; ------------------------------------------------------------------------
    ; Combine the reprojected event images.
    this_filter_spec = '['+regrid_filter+image_filter_spec[jj]+']' 
    
    infile_stack = reproj_obs_event_fn+this_filter_spec+bin_spec+opt_spec
    formula = strjoin(terms, '+')
    
    if keyword_set(sum_rates_not_counts) then begin
      ; If we used a single call to dmimgcalc to both normalize each stowed image and combine those normalized images, then as few as 75 ObsIDs will cause the parameter "operation" to overflow its buffer.
      ; So, we will use separate dmimgcalc calls to do the normalizations.  
      for ii=0,num_obs-1 do begin
        run_command, string(infile_stack[ii], reproj_obs_emap_fn[ii], reproj_stowed_img_fn[ii], $
                        F="(%'dmimgcalc infile=""%s"" infile2=%s outfile=%s operation=div verbose=0 clob+')")
      endfor ;ii
      
      infile_stack = reproj_stowed_img_fn
    endif

    ; Although the output image could be INTEGER for the observation image, it must be FLOAT for the background image (/SUM_RATES_NOT_COUNTS option).  For simplicity, below I'm casting the output to FLOAT in both cases. 
    print, F="(%'\nCombining these data:')"
    forprint, terms, infile_stack, F='(%"%s = %s")'
    forprint,        infile_stack, F='(%"%s")', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
    
    run_command, string(temp_text_fn, merged_image_fn[jj], formula, $
                        F="(%'dmimgcalc infile=""@-%s"" infile2=none outfile=%s operation=""imgout=((float)(%s))"" verbose=1 clob+')")
                      
    review_list = [review_list, merged_image_fn[jj]]

  endif ; (image_name[jj] NE '')

  ; ------------------------------------------------------------------------
  ;; Sum the exposure maps.
  if emap_basename[jj] && ~keyword_set(sum_rates_not_counts) then begin
    print, F="(%'\nSumming the reprojected emaps:')"
    forprint, terms, reproj_obs_emap_fn, F='(%"%s = %s")'
    forprint,        reproj_obs_emap_fn, F='(%"%s")', TEXTOUT=temp_text_fn, /SILENT, /NOCOMMENT
    
    formula = strjoin(terms, '+')
  
    run_command, string(temp_text_fn, merged_emap_fn[jj], formula, $
                        F="(%'dmimgcalc infile=""@-%s"" infile2=none outfile=%s operation=""imgout=((float)(%s))"" verbose=1 clob+')")          
    review_list = [review_list,merged_emap_fn[jj]]
    
    ;; Create a field mask from the emap.             
    ;; We want the mask to be 1.0 on-field so it can be applied via multiplication (e.g. in dmimgcalc calls),
    ;; and we want the mask to be NaN off-field so that the output image will also be NaN off-field so that
    ;; ds9 can display the off-field pixels in a color chosen by the observer.
    emap = readfits(merged_emap_fn[jj], emap_header)
    mask = make_array( SIZE=size(emap), VALUE=!VALUES.f_nan )
    mask[where(emap GT 0)] = 1
    writefits, merged_mask_fn, mask, emap_header
  endif
endfor ; jj loop over band-limited images


;; ========================================================================
;; Display results.
cd, CURRENT=cwd
title = 'build_scene: ' + strjoin(image_and_emap_dir,',')
cmd = string(target_name, title, strjoin(review_list,' '), F="(%'ds9 -title ""%s: %s"" -lock frame wcs -linear  %s -single -zoom to fit  -blink interval 2 -blink yes >& /dev/null &')")
if keyword_set(show) then begin
  run_command, cmd
  ;ae_send_to_ds9, my_ds9, NAME=title, OPTION_STRING=''
  ;ae_send_to_ds9, my_ds9, review_list
endif else print, cmd, F="(%'\n Review images with:\n  %s')"


CLEANUP:
if file_test(tempdir) then begin
  list = reverse(file_search(tempdir,'*',/MATCH_INITIAL_DOT,COUNT=count))
  if (count GT 0) then file_delete, list
  file_delete, tempdir
endif

if (exit_code EQ 0) then return $
else begin
  print, 'build_scene: Returning to top level due to fatal error.'
  retall
endelse

FAILURE:
exit_code = 1
GOTO, CLEANUP
end


