"""
Author: Simon Kunze

Plots for the manuscript 

Kunze, S., Groll, R., Besser, B., Thöming, J. (2022). 
Molecular diameters of rarefied gases. 
Sci Rep 12, 5871. 
https://doi.org/10.1038/s41598-022-05871-y

"""

import numpy as np
import matplotlib.pyplot as plt
import matplotlib as mpl
from copy import deepcopy
import os

import lit_data as ld
from sed_model import sed_rect, sed_circ
import helpers as hlp


# --------------------------------
# general plotting parameters:

savefig = True

figwidth = 3.5    # inches, one column
figheight = figwidth * 0.75
# dpi = 1200 # for final submission
dpi = 300 # for initial submission
marker_size = 10   # for scatter symbols
linewidth = 3
plt.rcParams.update({'font.size': 7})

# custom colors:
mpl.rcParams['axes.prop_cycle'] = mpl.cycler(color=["#F43A0C",
                                                    "#A95AA1",
                                                    "#659DF7",
                                                    "#BBBBBB"])
                                                    
bottomlim = 0.4
toplim = 10

leftlim = 8e-3
rightlim = 40

if savefig:
    if not os.path.isdir("plots"):
        os.mkdir("plots")

# --------------------------------
# --------------------------------


# get the literature data

# circular channels:

Kn_data_he_circ = ld.Kn_perrier11_circ_He
G_data_he_circ = ld.G_perrier11_circ_He

Kn_data_n2_circ = np.concatenate((ld.Kn_perrier11_circ_N2,
                                 ld.Kn_varoutis09))
G_data_n2_circ = np.concatenate((ld.G_perrier11_circ_N2,
                                 ld.G_varoutis09))

Kn_data_ar_circ = ld.Kn_perrier11_circ_Ar
G_data_ar_circ = ld.G_perrier11_circ_Ar

Kn_data_co2_circ = ld.Kn_knudsen09_circ_co2
G_data_co2_circ = ld.G_knudsen09_circ_co2

# rectangular channels:

Kn_data_he_rect = np.concatenate((ld.Kn_ewart07,
                                  ld.Kn_graur09_rect_he))
G_data_he_rect = np.concatenate((ld.G_ewart07,
                                 ld.G_graur09_rect_he))

Kn_data_n2_rect = ld.Kn_graur09_rect_n2
G_data_n2_rect = ld.G_graur09_rect_n2

Kn_data_ar_rect = ld.Kn_graur09_rect_ar
G_data_ar_rect = ld.G_graur09_rect_ar

# --------------------------------
# --------------------------------

# Initalize the models.
# For the circular channels, the diameter is irrelevant 
# for the results and will be at a standard value of 1 m.

Kn_range = np.logspace(np.log10(0.008), np.log10(100))

sed_he_circ = sed_circ({"Kn": Kn_range,
                        "gas": "He"})
sed_n2_circ = sed_circ({"Kn": Kn_range,
                        "gas": "N2"})
sed_ar_circ = sed_circ({"Kn": Kn_range,
                        "gas": "Ar"})
sed_co2_circ = sed_circ({"Kn": Kn_range,
                         "gas": "CO2"})

sed_he_rect = sed_rect({"Kn": Kn_range,
                        "gas": "He",
                        "h": ld.H_ewart07,
                        "w": ld.W_ewart07})
sed_n2_rect = sed_rect({"Kn": Kn_range,
                        "gas": "N2",
                        "h": ld.H_graur09_rect_n2,
                        "w": ld.W_graur09_rect_n2})
sed_ar_rect = sed_rect({"Kn": Kn_range,
                        "gas": "Ar",
                        "h": ld.H_graur09_rect_ar,
                        "w": ld.W_graur09_rect_ar})


# create instances with different diameters
# for the superposition plot
sed_sup_small = deepcopy(sed_he_rect)
sed_sup_small.d_gas = 160e-12
sed_sup_small.calc_G()
sed_sup_medium = deepcopy(sed_he_rect)
sed_sup_medium.d_gas = 200e-12
sed_sup_medium.calc_G()
sed_sup_large = deepcopy(sed_he_rect)
sed_sup_large.d_gas = 250e-12
sed_sup_large.calc_G()

def mdot_g_helper(mdot, obj):
    # helper function to get dimensionless mass flow components
    return hlp.mdot_to_g(mdot, obj.L, 
                         obj.P, obj.A, 
                         obj.dp, obj.T, obj.gas)

# get components of the mass flow for the superposition plot
conv_sup_small = mdot_g_helper(sed_sup_small.mdot_slip, sed_sup_small)
diff_sup_small = mdot_g_helper(sed_sup_small.mdot_eff_diff, sed_sup_small)

conv_sup_medium = mdot_g_helper(sed_sup_medium.mdot_slip, sed_sup_medium)
diff_sup_medium = mdot_g_helper(sed_sup_medium.mdot_eff_diff, sed_sup_medium)

conv_sup_large = mdot_g_helper(sed_sup_large.mdot_slip, sed_sup_large)
diff_sup_large = mdot_g_helper(sed_sup_large.mdot_eff_diff, sed_sup_large)



# fit model to data of circular channels
d_he_transition, = sed_he_circ.fit_to_data(Kn_data_he_circ,
                                          G_data_he_circ,
                                          d_gas=300e-12)
pred_d_he, = sed_he_circ.get_pred_interv()

d_n2_transition, = sed_n2_circ.fit_to_data(Kn_data_n2_circ,
                                          G_data_n2_circ,
                                          d_gas=300e-12)
pred_d_n2, = sed_n2_circ.get_pred_interv()

d_ar_transition, = sed_ar_circ.fit_to_data(Kn_data_ar_circ,
                                          G_data_ar_circ,
                                          d_gas=300e-12)
pred_d_ar, = sed_ar_circ.get_pred_interv()

d_co2_transition, = sed_co2_circ.fit_to_data(Kn_data_co2_circ,
                                            G_data_co2_circ,
                                            d_gas=300e-12)
pred_d_co2, = sed_co2_circ.get_pred_interv()

print(f"d_he_transition = ({d_he_transition*1e12:.0f} "
      + f"+- {pred_d_he*1e12:.0f}) pm")
print(f"d_n2_transition = ({d_n2_transition*1e12:.0f} "
      + f"+- {pred_d_n2*1e12:.0f}) pm")
print(f"d_ar_transition = ({d_ar_transition*1e12:.0f} "
      + f"+- {pred_d_ar*1e12:.0f}) pm")
print(f"d_co2_transition = ({d_co2_transition*1e12:.0f} "
      + f"+- {pred_d_co2*1e12:.0f}) pm")

# apply transition diameter to models of rectangular channels
sed_he_rect.d_gas = d_he_transition
sed_he_rect.calc_G()

sed_n2_rect.d_gas = d_n2_transition
sed_n2_rect.calc_G()

sed_ar_rect.d_gas = d_ar_transition
sed_ar_rect.calc_G()


# copy the models, apply literature diameters and recalculate

d_he_visc = 215e-12
d_he_kinetic = 260e-12
d_he_vdW = 286e-12

d_n2_visc = 370e-12
d_n2_kinetic = 364e-12
d_n2_vdW_long = 477e-12
d_n2_vdW_trans = 332e-12

d_ar_visc = 358e-12
d_ar_kinetic = 340e-12
d_ar_vdW = 366e-12

d_co2_visc = 453e-12
d_co2_kinetic = 330e-12
d_co2_vdW_long = 586e-12
d_co2_vdW_trans = 354e-12

sed_he_circ_visc = deepcopy(sed_he_circ)
sed_he_circ_kinetic = deepcopy(sed_he_circ)
sed_he_circ_vdW = deepcopy(sed_he_circ)
sed_he_circ_visc.d_gas = d_he_visc
sed_he_circ_kinetic.d_gas = d_he_kinetic
sed_he_circ_vdW.d_gas = d_he_vdW
sed_he_circ_visc.calc_G()
sed_he_circ_kinetic.calc_G()
sed_he_circ_vdW.calc_G()

sed_he_rect_visc = deepcopy(sed_he_rect)
sed_he_rect_kinetic = deepcopy(sed_he_rect)
sed_he_rect_vdW = deepcopy(sed_he_rect)
sed_he_rect_visc.d_gas = d_he_visc
sed_he_rect_kinetic.d_gas = d_he_kinetic
sed_he_rect_vdW.d_gas = d_he_vdW
sed_he_rect_visc.calc_G()
sed_he_rect_kinetic.calc_G()
sed_he_rect_vdW.calc_G()

sed_n2_circ_visc = deepcopy(sed_n2_circ)
sed_n2_circ_kinetic = deepcopy(sed_n2_circ)
sed_n2_circ_vdW_long = deepcopy(sed_n2_circ)
sed_n2_circ_vdW_trans = deepcopy(sed_n2_circ)
sed_n2_circ_visc.d_gas = d_n2_visc
sed_n2_circ_kinetic.d_gas = d_n2_kinetic
sed_n2_circ_vdW_long.d_gas = d_n2_vdW_long
sed_n2_circ_vdW_trans.d_gas = d_n2_vdW_trans
sed_n2_circ_visc.calc_G()
sed_n2_circ_kinetic.calc_G()
sed_n2_circ_vdW_long.calc_G()
sed_n2_circ_vdW_trans.calc_G()

sed_n2_rect_visc = deepcopy(sed_n2_rect)
sed_n2_rect_kinetic = deepcopy(sed_n2_rect)
sed_n2_rect_vdW_long = deepcopy(sed_n2_rect)
sed_n2_rect_vdW_trans = deepcopy(sed_n2_rect)
sed_n2_rect_visc.d_gas = d_n2_visc
sed_n2_rect_kinetic.d_gas = d_n2_kinetic
sed_n2_rect_vdW_long.d_gas = d_n2_vdW_long
sed_n2_rect_vdW_trans.d_gas = d_n2_vdW_trans
sed_n2_rect_visc.calc_G()
sed_n2_rect_kinetic.calc_G()
sed_n2_rect_vdW_long.calc_G()
sed_n2_rect_vdW_trans.calc_G()

sed_ar_circ_visc = deepcopy(sed_ar_circ)
sed_ar_circ_kinetic = deepcopy(sed_ar_circ)
sed_ar_circ_vdW = deepcopy(sed_ar_circ)
sed_ar_circ_visc.d_gas = d_ar_visc
sed_ar_circ_kinetic.d_gas = d_ar_kinetic
sed_ar_circ_vdW.d_gas = d_ar_vdW
sed_ar_circ_visc.calc_G()
sed_ar_circ_kinetic.calc_G()
sed_ar_circ_vdW.calc_G()

sed_ar_rect_visc = deepcopy(sed_ar_rect)
sed_ar_rect_kinetic = deepcopy(sed_ar_rect)
sed_ar_rect_vdW = deepcopy(sed_ar_rect)
sed_ar_rect_visc.d_gas = d_ar_visc
sed_ar_rect_kinetic.d_gas = d_ar_kinetic
sed_ar_rect_vdW.d_gas = d_ar_vdW
sed_ar_rect_visc.calc_G()
sed_ar_rect_kinetic.calc_G()
sed_ar_rect_vdW.calc_G()

sed_co2_circ_visc = deepcopy(sed_co2_circ)
sed_co2_circ_kinetic = deepcopy(sed_co2_circ)
sed_co2_circ_vdW_long = deepcopy(sed_co2_circ)
sed_co2_circ_vdW_trans = deepcopy(sed_co2_circ)
sed_co2_circ_visc.d_gas = d_co2_visc
sed_co2_circ_kinetic.d_gas = d_co2_kinetic
sed_co2_circ_vdW_long.d_gas = d_co2_vdW_long
sed_co2_circ_vdW_trans.d_gas = d_co2_vdW_trans
sed_co2_circ_visc.calc_G()
sed_co2_circ_kinetic.calc_G()
sed_co2_circ_vdW_long.calc_G()
sed_co2_circ_vdW_trans.calc_G()



# ---------------------------------
# ---------------------------------


# Calculate the deviation of the model.
# For this, new instances of the model are created and
# the mass flow is calculated for the Kn numbers of the
# experimental data.

sed_he_circ_dev = deepcopy(sed_he_circ)
sed_he_circ_dev.Kn = Kn_data_he_circ
sed_he_circ_dev.calc_G()

# differentiate between the two sources:
sed_n2_circ_dev_perrier = deepcopy(sed_n2_circ)
sed_n2_circ_dev_perrier.Kn = ld.Kn_perrier11_circ_N2
sed_n2_circ_dev_perrier.calc_G()
sed_n2_circ_dev_varoutis = deepcopy(sed_n2_circ)
sed_n2_circ_dev_varoutis.Kn = ld.Kn_varoutis09
sed_n2_circ_dev_varoutis.calc_G()

sed_ar_circ_dev = deepcopy(sed_ar_circ)
sed_ar_circ_dev.Kn = Kn_data_ar_circ
sed_ar_circ_dev.calc_G()

sed_co2_circ_dev = deepcopy(sed_co2_circ)
sed_co2_circ_dev.Kn = Kn_data_co2_circ
sed_co2_circ_dev.calc_G()

sed_he_rect_dev = deepcopy(sed_he_rect)
sed_he_rect_dev.Kn = Kn_data_he_rect
sed_he_rect_dev.calc_G()

sed_n2_rect_dev = deepcopy(sed_n2_rect)
sed_n2_rect_dev.Kn = Kn_data_n2_rect
sed_n2_rect_dev.calc_G()

sed_ar_rect_dev = deepcopy(sed_ar_rect)
sed_ar_rect_dev.Kn = Kn_data_ar_rect
sed_ar_rect_dev.calc_G()

# now, differentiate between the pressure sensors

def get_sensor_Kn_dev(Kn_data, G_data, G_model, pin_data,
                      p_min=0, p_max=np.inf):
    # small helper function to get the sensor specific data
    dat = np.array([[Kn, dev] for (Kn, dev, p) 
                              in zip(Kn_data,
                                    (G_model-G_data)
                                     / G_model * 100,
                                    pin_data)
                              if p > p_min and p < p_max])
    try:
        return dat[:,0], dat[:,1]
    except IndexError:
        return None, None

def get_all_sensors_Kn_dev(Kn_data, G_data, G_model, pin_data):
    # another helper function to get data of all sensors
    all_dat = []
    for p_min, p_max in zip([13332, 1333.2, 133.32, 0],
                            [np.inf, 13332, 1333.2, 133.322]):
        all_dat.append(get_sensor_Kn_dev(Kn_data, G_data, 
                                         G_model, pin_data, 
                                         p_min=p_min, p_max=p_max))
    return all_dat

# Helium, rectangular.
pins_he_rect = np.concatenate((ld.pins_ewart07,
                               ld.pins_graur09_rect_he))

Kn_dev_he_rect = get_all_sensors_Kn_dev(Kn_data_he_rect,
                                        G_data_he_rect,
                                        sed_he_rect_dev.G,
                                        pins_he_rect)

Kn_he_rect_sensA, dev_he_rect_sensA = Kn_dev_he_rect[0]
Kn_he_rect_sensB, dev_he_rect_sensB = Kn_dev_he_rect[1]
Kn_he_rect_sensC, dev_he_rect_sensC = Kn_dev_he_rect[2]
Kn_he_rect_sensD, dev_he_rect_sensD = Kn_dev_he_rect[3]


# Nitrogen, rectangular.
pins_n2_rect = ld.pins_graur09_rect_n2

Kn_dev_n2_rect = get_all_sensors_Kn_dev(Kn_data_n2_rect,
                                        G_data_n2_rect,
                                        sed_n2_rect_dev.G,
                                        pins_n2_rect)

Kn_n2_rect_sensA, dev_n2_rect_sensA = Kn_dev_n2_rect[0]
Kn_n2_rect_sensB, dev_n2_rect_sensB = Kn_dev_n2_rect[1]
Kn_n2_rect_sensC, dev_n2_rect_sensC = Kn_dev_n2_rect[2]
Kn_n2_rect_sensD, dev_n2_rect_sensD = Kn_dev_n2_rect[3]


# Argon, rectangular.
pins_ar_rect = ld.pins_graur09_rect_ar

Kn_dev_ar_rect = get_all_sensors_Kn_dev(Kn_data_ar_rect,
                                        G_data_ar_rect,
                                        sed_ar_rect_dev.G,
                                        pins_ar_rect)

Kn_ar_rect_sensA, dev_ar_rect_sensA = Kn_dev_ar_rect[0]
Kn_ar_rect_sensB, dev_ar_rect_sensB = Kn_dev_ar_rect[1]
Kn_ar_rect_sensC, dev_ar_rect_sensC = Kn_dev_ar_rect[2]
Kn_ar_rect_sensD, dev_ar_rect_sensD = Kn_dev_ar_rect[3]


# ------------------------------------
# actual plotting
# ------------------------------------


# --------------------------------------------
# superposition plot
# --------------------------------------------

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ls = []
fill_between_alpha = 0.3
ax.semilogx(sed_sup_medium.Kn, 
            conv_sup_medium+diff_sup_medium, 
            label="Superposition",
            linewidth=linewidth, color="k")
ax.fill_between(sed_sup_medium.Kn, 
                conv_sup_small+diff_sup_small,
                conv_sup_large+diff_sup_large,
                color="k", alpha=fill_between_alpha,
                linewidth=0)
ax.semilogx(sed_sup_medium.Kn, conv_sup_medium, 
            label="Convection",
            linewidth=linewidth, color="#444444",
            linestyle=":")
ax.fill_between(sed_sup_medium.Kn, 
                conv_sup_small, conv_sup_large,
                color="#444444", alpha=fill_between_alpha*0.8,
                linewidth=0)
ax.semilogx(sed_sup_medium.Kn, diff_sup_medium,
            label="Diffusion",
            linewidth=linewidth, color="#444444",
            linestyle="--")
ax.fill_between(sed_sup_medium.Kn, 
                diff_sup_small, diff_sup_large,
                color="#444444", alpha=fill_between_alpha*0.8,
                linewidth=0)

from matplotlib.patches import Rectangle
ax.add_patch(Rectangle((0.1, 0), 9.9, 4.5,
                        edgecolor = "None",
                        facecolor = "k",
                        alpha = 0.1))
ax.text(1, 2.5, "Transitional\nregime", horizontalalignment="center",
        multialignment="center")

ax.legend(frameon=False)
ax.text(0.0135, 4.25, "b", fontweight='bold', fontsize=9)
ax.set_ylim([0, 4.5])
ax.set_xlim([0.02, 100])
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel("Dimensionless mass flow (-)")
plt.tick_params(
    axis='y',
    which='both',
    left=False,
    labelleft=False) 
plt.tight_layout()
if savefig: plt.savefig("plots/superposition.png", dpi=dpi)


# --------------------------------------------
# plot as background for Kn range compilation figure
# --------------------------------------------

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ls = []

ax.semilogx(sed_sup_medium.Kn, 
            conv_sup_medium+diff_sup_medium, 
            linewidth=linewidth, color="#FFFFFF")
ax.text(0.0135, 4.25, "a", fontweight='bold', fontsize=9)
ax.set_ylim([0, 4.5])
ax.set_xlim([0.02, 100])
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel(" ")
plt.tick_params(
    axis='y',
    which='both',
    left=False,
    labelleft=False) 
plt.tight_layout()
if savefig: plt.savefig("plots/kn_range_background.png", dpi=dpi)

# --------------------------------------------
# mass flow plots for circular channels
# --------------------------------------------

# Helium, circular channels

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1
l_exp = ax.scatter(Kn_data_he_circ, G_data_he_circ,
           marker="o", color="k", alpha=alpha, s=marker_size,
           zorder=0, label="Experiment (Perrier 11)")
alpha = 0.8

l1, = ax.plot(sed_he_circ_visc.Kn, sed_he_circ_visc.G, label="From viscosity",
        linewidth=linewidth, alpha=alpha, color="C1")
l2, = ax.plot(sed_he_circ_kinetic.Kn, sed_he_circ_kinetic.G, label="Kinetic",
        linewidth=linewidth, alpha=alpha, color="C2")
l3, = ax.plot(sed_he_circ_vdW.Kn, sed_he_circ_vdW.G, label="Van der Waals",
        linewidth=linewidth, alpha=alpha, color="C3")

l4, = ax.plot(sed_he_circ.Kn, sed_he_circ.G, label="Transition",
        linewidth=linewidth, alpha=alpha, zorder=9, color="C0")

labels = [l1, l2, l3, l_exp, l4]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="Helium diameter", frameon=False)

ax.text(0.0022, 9, "a", fontweight='bold', fontsize=9)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel(" ")
ax.set_ylabel("Dimensionless mass flow (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/G_he_circ.png", dpi=dpi)


# Nitrogen, circular channels

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1
l_exp = ax.scatter(Kn_data_n2_circ, G_data_n2_circ,
           marker="o", color="k", alpha=alpha, s=marker_size,
           zorder=0, label="Experiment \n(Perrier 11, Varoutis 09)")

alpha = 0.8
l1, = ax.plot(sed_n2_circ.Kn, sed_n2_circ.G, label="Transition",
              linewidth=linewidth, alpha=alpha, zorder=9, color="C0")
l2, = ax.plot(sed_n2_circ_visc.Kn, sed_n2_circ_visc.G,
              label="From viscosity",
              linewidth=linewidth, alpha=alpha, color="C1")
l3, = ax.plot(sed_n2_circ_kinetic.Kn, sed_n2_circ_kinetic.G,
              label="Kinetic",
              linewidth=linewidth, alpha=alpha, color="C2")
l4, = ax.plot(sed_n2_circ_vdW_long.Kn, sed_n2_circ_vdW_long.G,
              label="Long. van der Waals",
              linewidth=linewidth, alpha=alpha, color="C3")
l5, = ax.plot(sed_n2_circ_vdW_trans.Kn, sed_n2_circ_vdW_trans.G,
              label="Trans. van der Waals",
              linewidth=linewidth, alpha=alpha, color="C3", linestyle="--")

labels = [l2, l3, l4, l5, l_exp, l1]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="Nitrogen diameter", frameon=False)

ax.text(0.0022, 9, "c", fontweight='bold', fontsize=9)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel("Dimensionless mass flow (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/G_n2_circ.png", dpi=dpi)


# Argon, circular channels

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1
l_exp = ax.scatter(Kn_data_ar_circ, G_data_ar_circ,
           marker="o", color="k", alpha=alpha, s=marker_size,
           label="Experiment (Perrier 11)")

alpha = 0.8
l1, = ax.plot(sed_ar_circ.Kn, sed_ar_circ.G, label="Transition",
        linewidth=linewidth, alpha=alpha, zorder=9)
l2, = ax.plot(sed_ar_circ_visc.Kn, sed_ar_circ_visc.G, label="From viscosity",
        linewidth=linewidth, alpha=alpha)
l3, = ax.plot(sed_ar_circ_kinetic.Kn, sed_ar_circ_kinetic.G, label="Kinetic",
        linewidth=linewidth, alpha=alpha)
l4, = ax.plot(sed_ar_circ_vdW.Kn, sed_ar_circ_vdW.G, label="Van der Waals",
        linewidth=linewidth, alpha=alpha)

labels = [l2, l3, l4, l_exp, l1]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="Argon diameter", frameon=False)

ax.text(0.0022, 9, "b", fontweight='bold', fontsize=9)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel(" ")
ax.set_ylabel(" ")
plt.tight_layout()
if savefig: plt.savefig("plots/G_ar_circ.png", dpi=dpi)


# Carbon dioxide, circular channels

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1
l_exp = ax.scatter(Kn_data_co2_circ, G_data_co2_circ,
           marker="o", color="k", alpha=alpha, s=marker_size,
           label="Experiment (Knudsen 09)")

alpha = 0.8
l1, = ax.plot(sed_co2_circ.Kn, sed_co2_circ.G,
              label="Transition",
              linewidth=linewidth, alpha=alpha, zorder=9)
l2, = ax.plot(sed_co2_circ_visc.Kn, sed_co2_circ_visc.G,
              label="From viscosity",
              linewidth=linewidth, alpha=alpha)
l3, = ax.plot(sed_co2_circ_kinetic.Kn, sed_co2_circ_kinetic.G,
              label="Kinetic",
              linewidth=linewidth, alpha=alpha)
l4, = ax.plot(sed_co2_circ_vdW_long.Kn, sed_co2_circ_vdW_long.G,
              label="Long. van der Waals",
              linewidth=linewidth, alpha=alpha)
l5, = ax.plot(sed_co2_circ_vdW_trans.Kn, sed_co2_circ_vdW_trans.G,
              label="Trans. van der Waals",
              linewidth=linewidth, alpha=alpha, color="C3", linestyle="--")

labels = [l2, l3, l4, l5, l_exp, l1]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="Carbon dioxide diameter", frameon=False)

ax.text(0.0022, 9, "d", fontweight='bold', fontsize=9)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel(" ")
plt.tight_layout()
if savefig: plt.savefig("plots/G_co2_circ.png", dpi=dpi)




# --------------------------------------------
# mass flow plots for rectangular channels
# --------------------------------------------

# Helium, rectangular channels

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1

l_exp = ax.scatter(Kn_data_he_rect, G_data_he_rect,
                   marker="s", color="k", alpha=alpha, s=marker_size,
                   label="Experiment (Ewart 07, Graur 09)")

alpha = 0.8
l1, = ax.plot(sed_he_rect.Kn, sed_he_rect.G, label="Transition",
              linewidth=linewidth, alpha=alpha, zorder=9)
l2, = ax.plot(sed_he_rect_visc.Kn, sed_he_rect_visc.G, label="From viscosity",
              linewidth=linewidth, alpha=alpha)
l3, = ax.plot(sed_he_rect_kinetic.Kn, sed_he_rect_kinetic.G, label="Kinetic",
              linewidth=linewidth, alpha=alpha)
l4, = ax.plot(sed_he_rect_vdW.Kn, sed_he_rect_vdW.G, label="Van der Waals",
              linewidth=linewidth, alpha=alpha)

labels = [l2, l3, l4, l_exp, l1]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="Helium diameter", frameon=False)

ax.text(0.0022, 9, "a", fontweight='bold', fontsize=9)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel(" ")
ax.set_ylabel("Dimensionless mass flow (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/G_he_rect.png", dpi=dpi)


# Nitrogen, rectangular channels

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1
l_exp = ax.scatter(Kn_data_n2_rect, G_data_n2_rect,
                   marker="s", color="k", alpha=alpha, s=marker_size,
                   label="Exp. (Graur 09)")

alpha = 0.8
l1, = ax.plot(sed_n2_rect.Kn, sed_n2_rect.G, label="Transition",
              linewidth=linewidth, alpha=alpha, zorder=3)
l2, = ax.plot(sed_n2_rect_visc.Kn, sed_n2_rect_visc.G,
              label="From viscosity", linewidth=linewidth, alpha=alpha)
l3, = ax.plot(sed_n2_rect_kinetic.Kn, sed_n2_rect_kinetic.G,
              label="Kinetic", linewidth=linewidth, alpha=alpha)
l4, = ax.plot(sed_n2_rect_vdW_long.Kn, sed_n2_rect_vdW_long.G,
              label="Long. vdW", linewidth=linewidth, alpha=alpha)
l5, = ax.plot(sed_n2_rect_vdW_trans.Kn, sed_n2_rect_vdW_trans.G,
              label="Trans. vdW", linewidth=linewidth,
              alpha=alpha, color="C3", linestyle="--")

labels = [l2, l3, l4, l5, l_exp, l1]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="Nitrogen diameter", ncol=2, frameon=False)

ax.text(0.0022, 9, "c", fontweight='bold', fontsize=9)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel("Dimensionless mass flow (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/G_n2_rect.png", dpi=dpi)


# Argon, rectangular channels

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1
l_exp = ax.scatter(Kn_data_ar_rect, G_data_ar_rect,
                   marker="s", color="k", alpha=alpha, s=marker_size,
                   label="Experiment (Graur 09)")

alpha = 0.8
l1, = ax.plot(sed_ar_rect.Kn, sed_ar_rect.G, label="Transition",
              linewidth=linewidth, alpha=alpha, zorder=9)
l2, = ax.plot(sed_ar_rect_visc.Kn, sed_ar_rect_visc.G, label="From viscosity",
              linewidth=linewidth, alpha=alpha)
l3, = ax.plot(sed_ar_rect_kinetic.Kn, sed_ar_rect_kinetic.G, label="Kinetic",
              linewidth=linewidth, alpha=alpha)
l4, = ax.plot(sed_ar_rect_vdW.Kn, sed_ar_rect_vdW.G, label="Van der Waals",
              linewidth=linewidth, alpha=alpha)

labels = [l2, l3, l4, l_exp, l1]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="Argon diameter", frameon=False)

ax.text(0.0022, 9, "b", fontweight='bold', fontsize=9)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel(" ")
plt.tight_layout()
if savefig: plt.savefig("plots/G_ar_rect.png", dpi=dpi)




# --------------------------------------------
# plot of molecular diameters
# --------------------------------------------

diam_names = ["Transition (this work)", "From viscosity",
              "Kinetic", "Long. van der Waals"]
zorders = [9, 1, 1, 1]

He_diams = np.array([d_he_transition, d_he_visc, d_he_kinetic, d_he_vdW])*1e12
He_lbl = np.repeat(["He"], 4)

N2_diams = np.array([d_n2_transition, d_n2_visc, d_n2_kinetic,
                     d_n2_vdW_long])*1e12
N2_trans_diam = d_n2_vdW_trans*1e12
N2_trans_lbl = "N2"
N2_lbl = np.repeat("N2", 4)

Ar_diams = np.array([d_ar_transition, d_ar_visc, d_ar_kinetic, d_ar_vdW])*1e12
Ar_lbl = np.repeat("Ar", 4)

CO2_diams = np.array([d_co2_transition, d_co2_visc, d_co2_kinetic,
                      d_co2_vdW_long])*1e12
CO2_trans_diam = d_co2_vdW_trans*1e12
CO2_trans_lbl = "CO2"
CO2_lbl = np.repeat("CO2", 4)

# prediction intervals of determined diameters:
pred_interv = np.array([pred_d_he, pred_d_ar, pred_d_n2, pred_d_co2])*1e12
pred_interv_lbls = ["He", "Ar", "N2", "CO2"]

all_lbls = np.array([He_lbl, Ar_lbl, N2_lbl, CO2_lbl]).T
all_diams = np.array([He_diams, Ar_diams, N2_diams, CO2_diams]).T

fig, ax = plt.subplots(figsize=(figwidth, figheight))
labels = []
for lbl, diam, diam_name, zorder in zip(all_lbls,
                                        all_diams,
                                        diam_names,
                                        zorders):
    ll = ax.scatter(lbl, diam, label=diam_name, zorder=zorder)
    labels.append(ll)
# additional points for the second vdW diameters of N2 and CO2:
labels.append(ax.scatter(N2_trans_lbl, N2_trans_diam, zorder=zorder,
                         color="C3", marker="^",
                         label="Trans. van der Waals"))
ax.scatter(CO2_trans_lbl, CO2_trans_diam, zorder=zorder, 
           color="C3", marker="^")
ax.errorbar(pred_interv_lbls, all_diams[0], pred_interv,
            fmt="none", capsize=6)

# re-ordering of the labels:
labels = [labels[1], labels[2], labels[3], labels[4], labels[0]]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, frameon=False)

ax.text(-0.65, 590, "d", fontweight='bold', fontsize=9)
ax.set_ylabel("Molecular diameter (pm)")
plt.tight_layout()
if savefig: plt.savefig("plots/mol_diams_comp.png", dpi=dpi)


# --------------------------------------------
# deviation plots
# --------------------------------------------

dev_lim_top = 20
dev_lim_bottom = -20

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.scatter(sed_he_circ_dev.Kn, 
          (sed_he_circ_dev.G-G_data_he_circ)
           /sed_he_circ_dev.G * 100,
           marker="o", color="k", alpha=1, s=marker_size,
           zorder=0, label="Helium, circular (Perrier 11)")
ax.axhline(linestyle="--", color="k")
ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(top=dev_lim_top, bottom=dev_lim_bottom)
ax.legend(frameon=False)
ax.text(0.002, 18.5, "a", fontweight='bold', fontsize=9)
ax.set_xlabel(" ")
ax.set_ylabel("Percentage of deviation (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/dev_he_circ.png", dpi=dpi)


fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.scatter(sed_n2_circ_dev_varoutis.Kn, 
          (sed_n2_circ_dev_varoutis.G-ld.G_varoutis09)
           /sed_n2_circ_dev_varoutis.G * 100,
           marker="o", color="k", alpha=1, s=marker_size,
           zorder=0, label="Nitrogen, circular (Varoutis 09)")
ax.scatter(sed_n2_circ_dev_perrier.Kn, 
          (sed_n2_circ_dev_perrier.G-ld.G_perrier11_circ_N2)
           /sed_n2_circ_dev_perrier.G * 100,
           marker="o", color="r", alpha=1, s=marker_size,
           zorder=0, label="Nitrogen, circular (Perrier 11)")
ax.axhline(linestyle="--", color="k")
ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(top=dev_lim_top, bottom=dev_lim_bottom)
ax.legend(frameon=False)
ax.text(0.002, 18.5, "c", fontweight='bold', fontsize=9)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel("Percentage of deviation (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/dev_n2_circ.png", dpi=dpi)


fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.scatter(sed_ar_circ_dev.Kn, 
          (sed_ar_circ_dev.G-G_data_ar_circ)
           /sed_ar_circ_dev.G * 100,
           marker="o", color="k", alpha=1, s=marker_size,
           zorder=0, label="Argon, circular (Perrier 11)")
ax.axhline(linestyle="--", color="k")
ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(top=dev_lim_top, bottom=dev_lim_bottom)
ax.legend(frameon=False)
ax.text(0.002, 18.5, "b", fontweight='bold', fontsize=9)
ax.set_xlabel(" ")
ax.set_ylabel(" ")
plt.tight_layout()
if savefig: plt.savefig("plots/dev_ar_circ.png", dpi=dpi)


fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.scatter(sed_co2_circ_dev.Kn, 
          (sed_co2_circ_dev.G-G_data_co2_circ)
           /sed_co2_circ_dev.G * 100,
           marker="o", color="k", alpha=1, s=marker_size,
           zorder=0, label="Carbon dioxide, circular (Knudsen 09)")
ax.axhline(linestyle="--", color="k")
ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(top=dev_lim_top, bottom=dev_lim_bottom)
ax.legend(frameon=False)
ax.text(0.002, 18.5, "d", fontweight='bold', fontsize=9)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel(" ")
plt.tight_layout()
if savefig: plt.savefig("plots/dev_co2_circ.png", dpi=dpi)


fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.scatter(Kn_he_rect_sensA, 
           dev_he_rect_sensA,
           marker="s", color="k", alpha=1, s=marker_size,
           label="Sensor A")
ax.scatter(Kn_he_rect_sensB, 
           dev_he_rect_sensB,
           marker="s", color="r", alpha=1, s=marker_size,
           label="Sensor B")
ax.scatter(Kn_he_rect_sensC, 
           dev_he_rect_sensC,
           marker="s", color="b", alpha=1, s=marker_size,
           label="Sensor C")
ax.scatter(Kn_he_rect_sensD, 
           dev_he_rect_sensD,
           marker="s", color="y", alpha=1, s=marker_size,
           label="Sensor D")
ax.axhline(linestyle="--", color="k")
ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(top=dev_lim_top, bottom=dev_lim_bottom)
ax.legend(title="Helium, rectangular\n(Ewart 07, Graur 09)", frameon=False)
ax.text(0.002, 18.5, "a", fontweight='bold', fontsize=9)
ax.set_xlabel(" ")
ax.set_ylabel("Percentage of deviation (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/dev_he_rect.png", dpi=dpi)


fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.scatter(Kn_n2_rect_sensA, 
           dev_n2_rect_sensA,
           marker="s", color="k", alpha=1, s=marker_size,
           label="Sensor A")
ax.scatter(Kn_n2_rect_sensB, 
           dev_n2_rect_sensB,
           marker="s", color="r", alpha=1, s=marker_size,
           label="Sensor B")
ax.scatter(Kn_n2_rect_sensC, 
           dev_n2_rect_sensC,
           marker="s", color="b", alpha=1, s=marker_size,
           label="Sensor C")
ax.scatter(Kn_n2_rect_sensD, 
           dev_n2_rect_sensD,
           marker="s", color="y", alpha=1, s=marker_size,
           label="Sensor D")
ax.axhline(linestyle="--", color="k")
ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(top=dev_lim_top, bottom=dev_lim_bottom)
ax.legend(title="Nitrogen, rectangular (Graur 09)", frameon=False)
ax.text(0.002, 18.5, "c", fontweight='bold', fontsize=9)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel("Percentage of deviation (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/dev_n2_rect.png", dpi=dpi)


fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.scatter(Kn_ar_rect_sensB, 
           dev_ar_rect_sensB,
           marker="s", color="r", alpha=1, s=marker_size,
           label="Sensor B")
ax.scatter(Kn_ar_rect_sensC, 
           dev_ar_rect_sensC,
           marker="s", color="b", alpha=1, s=marker_size,
           label="Sensor C")
ax.scatter(Kn_ar_rect_sensD, 
           dev_ar_rect_sensD,
           marker="s", color="y", alpha=1, s=marker_size,
           label="Sensor D")
ax.axhline(linestyle="--", color="k")
ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(top=dev_lim_top, bottom=dev_lim_bottom)
ax.legend(title="Argon, rectangular (Graur 09)", frameon=False)
ax.text(0.002, 18.5, "b", fontweight='bold', fontsize=9)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel(" ")
plt.tight_layout()
if savefig: plt.savefig("plots/dev_ar_rect.png", dpi=dpi)



# --------------------------------------------
# supplementary plots
# --------------------------------------------

# helium, circular channels, Kn calculated from viscosity 
# and from kinetic diameter

d_He_kinetic = 260e-12
Kn_data_he_circ_d_kinetic = hlp.mfp(
    ld.T_perrier11_circ_He, d_He_kinetic,
    ld.p_m_perrier11_circ_He) / (2*ld.r_perrier11_circ_He)

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1
l_exp1 = ax.scatter(Kn_data_he_circ, G_data_he_circ,
           marker="o", color="k", alpha=alpha, s=marker_size,
           zorder=0, label="Kn from viscosity")
l_exp2 = ax.scatter(Kn_data_he_circ_d_kinetic, G_data_he_circ,
           marker="o", color="r", alpha=alpha, s=marker_size,
           zorder=0, label="Kn from kinetic diameter")

labels = [l_exp1, l_exp2]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, frameon=False)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel("Dimensionless mass flow (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/G_he_circ_d_var.png", dpi=dpi)


# TMAC sensitivity for rectangular channels

sed_he_rect_tmac_1 = deepcopy(sed_he_rect)
sed_he_rect_tmac_1.sigma_slip = 1

sed_he_rect_tmac_1.calc_G()
sed_he_rect.calc_G()

sed_he_rect_tmac_09_FM = deepcopy(sed_he_rect)
sed_he_rect_tmac_09_FM.sigma = 0.9
sed_he_rect_tmac_09_FM.calc_G()

fig, ax = plt.subplots(figsize=(figwidth, figheight))
ax.set_xscale("log")
ax.set_yscale("log")
alpha = 1

l_exp = ax.scatter(Kn_data_he_rect, G_data_he_rect,
                   marker="s", color="k", alpha=alpha, s=marker_size,
                   label="Experiment (Ewart 07, Graur 09)")

alpha = 0.8
l1, = ax.plot(sed_he_rect.Kn, sed_he_rect.G, label="TMAC=0.9 only for slip",
              linewidth=linewidth, alpha=alpha, zorder=10)
l2, = ax.plot(sed_he_rect_tmac_09_FM.Kn, sed_he_rect_tmac_09_FM.G, 
    label="TMAC=0.9 for both slip and FM term",
              linewidth=linewidth, alpha=alpha, zorder=9)
l3, = ax.plot(sed_he_rect_tmac_1.Kn, sed_he_rect_tmac_1.G, label="TMAC=1",
              linewidth=linewidth, alpha=alpha, zorder=9)

labels = [l_exp, l1, l2, l3]
labs = [l.get_label() for l in labels]
ax.legend(labels, labs, title="", frameon=False)

ax.set_xlim(left=leftlim, right=rightlim)
ax.set_ylim(bottom=bottomlim, top=toplim)
ax.set_xlabel("Knudsen number (-)")
ax.set_ylabel("Dimensionless mass flow (-)")
plt.tight_layout()
if savefig: plt.savefig("plots/TMAC_sens.png", dpi=dpi)
