"""
Author: Simon Kunze

In this file experimental data from the literature
for rarefied gas flow is collected.

Important Note:
The Kn number is calculated using the mean free path resulting from viscosity!
For rectangular channels, h as the characteristic length for the Knudsen
number is used.

"""

import numpy as np

import helpers as hlp


"""
-------------------------------------------------------
Data for circular channels.
-------------------------------------------------------

"""


"""
Helium:
    
Perrier, P., Graur, I., Ewart, T., Méolans, J. (2011). Mass flow rate 
measurements in microtubes: From hydrodynamic to near free molecular regime 
Physics of Fluids  23(4), 042004. https://dx.doi.org/10.1063/1.3562948
"""

perrier11_circ_He = np.loadtxt("csv/perrier11_circ_He.csv",
                       skiprows=1, delimiter=";")
delta_perrier11_circ_He = perrier11_circ_He[:,0]
Q_perrier11_circ_He = perrier11_circ_He[:,1]  # this is the G used in the
                                                # paper which is different
                                                # to the G used in this
                                                # manuscript

r_perrier11_circ_He = 49.6e-6 / 2
L_perrier11_circ_He = 1.82e-2
T_perrier11_circ_He = 296.5
p_r_perrier11_circ_He = 5

perrier11_circ_He = "He"

p_m_perrier11_circ_He = ( delta_perrier11_circ_He 
                         * hlp.get_mu("He", T_perrier11_circ_He)
                         * np.sqrt(2*hlp.R/hlp.get_M("He")
                                   *T_perrier11_circ_He) 
                         / r_perrier11_circ_He )

Kn_perrier11_circ_He = hlp.mfp_visc(T_perrier11_circ_He,
                                    p_m_perrier11_circ_He,
                                    "He") / (2*r_perrier11_circ_He)

(pin_perrier11_circ_He,
 pout_perrier11_circ_He) = hlp.get_pin_pout(Kn_perrier11_circ_He,
                                            2*r_perrier11_circ_He,
                                            T_perrier11_circ_He,
                                            "He",
                                            ratio=p_r_perrier11_circ_He)
dp_perrier11_circ_He = pin_perrier11_circ_He - pout_perrier11_circ_He

# here, the mass flow is calculated from the dimensionless mass flow as used
# in the paper of Perrier
mdot_perrier11_circ_He = ( Q_perrier11_circ_He
                          * np.pi * r_perrier11_circ_He**3 
                          * dp_perrier11_circ_He
                          / (L_perrier11_circ_He
                             * np.sqrt(2*hlp.R/hlp.get_M("He")
                                       * T_perrier11_circ_He)) )
G_perrier11_circ_He = hlp.mdot_to_g(mdot_perrier11_circ_He, 
                                    L_perrier11_circ_He,
                                    2*np.pi*r_perrier11_circ_He,
                                    np.pi*r_perrier11_circ_He**2,
                                    dp_perrier11_circ_He, 
                                    T_perrier11_circ_He,
                                    perrier11_circ_He)


"""
Nitrogen:
    
Perrier, P., Graur, I., Ewart, T., Méolans, J. (2011).
Mass flow rate measurements in microtubes: From hydrodynamic to 
near free molecular regime 
Physics of Fluids  23(4), 042004. https://dx.doi.org/10.1063/1.3562948
"""

perrier11_circ_N2 = np.loadtxt("csv/perrier11_circ_N2.csv",
                       skiprows=1, delimiter=";")
delta_perrier11_circ_N2 = perrier11_circ_N2[:,0]
Q_perrier11_circ_N2 = perrier11_circ_N2[:,1] # this is the G used in the
                                                # paper which is different
                                                # to the G used in this
                                                # manuscript

r_perrier11_circ_N2 = 49.6e-6 / 2
L_perrier11_circ_N2 = 1.82e-2
T_perrier11_circ_N2 = 296.5
p_r_perrier11_circ_N2 = 5

p_m_perrier11_circ_N2 = ( delta_perrier11_circ_N2 
                         * hlp.get_mu("N2", T_perrier11_circ_N2)
                         * np.sqrt(2 * hlp.R / hlp.get_M("N2")
                                   * T_perrier11_circ_N2) 
                         / r_perrier11_circ_N2 )

Kn_perrier11_circ_N2 = hlp.mfp_visc(T_perrier11_circ_N2,
                                    p_m_perrier11_circ_N2,
                                    "N2") / (2*r_perrier11_circ_N2)

(pin_perrier11_circ_N2,
 pout_perrier11_circ_N2) = hlp.get_pin_pout(Kn_perrier11_circ_N2,
                                            2*r_perrier11_circ_N2,
                                            T_perrier11_circ_N2,
                                            "N2",
                                            ratio=p_r_perrier11_circ_N2)
dp_perrier11_circ_N2 = pin_perrier11_circ_N2 - pout_perrier11_circ_N2

# here, the mass flow is calculated from the dimensionless mass flow as used
# in the paper of Perrier
mdot_perrier11_circ_N2 = ( Q_perrier11_circ_N2
                          * np.pi * r_perrier11_circ_N2**3 
                          * dp_perrier11_circ_N2
                          / (L_perrier11_circ_N2 
                             * np.sqrt(2 * hlp.R / hlp.get_M("N2")
                                       * T_perrier11_circ_N2)) )
G_perrier11_circ_N2 = hlp.mdot_to_g(mdot_perrier11_circ_N2, 
                                    L_perrier11_circ_N2,
                                    2*np.pi*r_perrier11_circ_N2,
                                    np.pi*r_perrier11_circ_N2**2,
                                    dp_perrier11_circ_N2, 
                                    T_perrier11_circ_N2,
                                    "N2")


"""
Also nitrogen:
    
experimental data for circular tubes from
Varoutis, S., Naris, S., Hauer, V., Day, C., Valougeorgis, D. (2009).
Computational and experimental study of gas flows through long channels 
of various cross sections in the whole range of the Knudsen number
Journal of Vacuum Science & Technology A: Vacuum, Surfaces,
and Films  27(1), 89-100. https://dx.doi.org/10.1116/1.3043463

raw data
"""

varoutis09 = np.loadtxt("csv/varoutis09.csv",
                       skiprows=1, delimiter=";")

D_varoutis09 = 15.96e-3
L_varoutis09 = 1277e-3
gas_varoutis09 = "N2"

sccm_varoutis09 = varoutis09[:,0] # standard cubic centimers per minute
# transform sccm to kg/s
mdot_varoutis09 = sccm_varoutis09 / 60 * 1.251e-6


pin_varoutis09 = varoutis09[:,1] * 100  # for mbar to Pa
pout_varoutis09 = varoutis09[:,2] * 100  # for mbar to Pa
T_varoutis09 = varoutis09[:,3]

p_m_varoutis09 = (pin_varoutis09 + pout_varoutis09)/2

mfp_varoutis09 = hlp.mfp_visc(T_varoutis09,
                                       p_m_varoutis09,
                                       gas_varoutis09)

Kn_varoutis09 = mfp_varoutis09 / D_varoutis09

G_varoutis09 = hlp.mdot_to_g(mdot_varoutis09,
                             L_varoutis09,
                             np.pi * D_varoutis09,
                             np.pi/4 * D_varoutis09**2,
                             pin_varoutis09 - pout_varoutis09,
                             T_varoutis09,
                             gas_varoutis09)



"""
Argon:
    
Perrier, P., Graur, I., Ewart, T., Méolans, J. (2011). Mass flow rate 
measurements in microtubes: From hydrodynamic to near free molecular regime 
Physics of Fluids  23(4), 042004. https://dx.doi.org/10.1063/1.3562948

"""

perrier11_circ_Ar = np.loadtxt("csv/perrier11_circ_Ar.csv",
                       skiprows=1, delimiter=";")
delta_perrier11_circ_Ar = perrier11_circ_Ar[:,0]
Q_perrier11_circ_Ar = perrier11_circ_Ar[:,1]  # this is the G used in the
                                              # paper which is different
                                              # to the G used in this
                                              # manuscript

r_perrier11_circ_Ar = 49.6e-6 / 2
L_perrier11_circ_Ar = 1.82e-2
T_perrier11_circ_Ar = 296.5
gas_perrier11_circ_Ar = "Ar"
p_r_perrier11_circ_Ar = 5

p_m_perrier11_circ_Ar = ( delta_perrier11_circ_Ar 
                         * hlp.get_mu(gas_perrier11_circ_Ar, 
                                      T_perrier11_circ_Ar)
                         * np.sqrt(2*hlp.R
                                   / hlp.get_M(gas_perrier11_circ_Ar)
                                   *T_perrier11_circ_Ar) 
                         / r_perrier11_circ_Ar )

Kn_perrier11_circ_Ar = ( hlp.mfp_visc(T_perrier11_circ_Ar,
                                     p_m_perrier11_circ_Ar,
                                     gas_perrier11_circ_Ar)
                         / (2*r_perrier11_circ_Ar) )

(pin_perrier11_circ_Ar, 
 pout_perrier11_circ_Ar) = hlp.get_pin_pout(Kn_perrier11_circ_Ar,
                                            2 * r_perrier11_circ_Ar,
                                            T_perrier11_circ_Ar,
                                            gas_perrier11_circ_Ar,
                                            ratio=p_r_perrier11_circ_Ar)
dp_perrier11_circ_Ar = pin_perrier11_circ_Ar - pout_perrier11_circ_Ar
mdot_perrier11_circ_Ar = ( Q_perrier11_circ_Ar
                          * np.pi * r_perrier11_circ_Ar**3 
                          * dp_perrier11_circ_Ar
                          / (L_perrier11_circ_Ar 
                             * np.sqrt(2 * hlp.R
                                       / hlp.get_M(gas_perrier11_circ_Ar)
                                       * T_perrier11_circ_Ar)) )
G_perrier11_circ_Ar = hlp.mdot_to_g(mdot_perrier11_circ_Ar, 
                                    L_perrier11_circ_Ar,
                                    2*np.pi*r_perrier11_circ_Ar,
                                    np.pi*r_perrier11_circ_Ar**2,
                                    dp_perrier11_circ_Ar, 
                                    T_perrier11_circ_Ar,
                                    gas_perrier11_circ_Ar)


"""
Carbon dioxide:

Knudsen, M. (1909). Die Gesetze der Molekularströmung und der inneren
Reibungsströmung der Gase durch Röhren
Annalen der Physik  333(1), 75-130.
https://dx.doi.org/10.1002/andp.19093330106

"Roehre Nr. 4"

"""

knudsen09_circ_co2 = np.loadtxt("csv/knudsen09.csv",
                       skiprows=1)

T_degrC_knudsen09_circ_co2 = knudsen09_circ_co2[:,0]

p_m_cm_Hg_knudsen09_circ_co2 = knudsen09_circ_co2[:,1]

X_0_knudsen09_circ_co2 = knudsen09_circ_co2[:,2]  # called T_0 in 
                                                  # Knudsen's paper; 
                                                  # this is for all 24 tubes

# 1 cm Hg = 1333 Pa
p_m_knudsen09_circ_co2 = p_m_cm_Hg_knudsen09_circ_co2 * 1333
T_knudsen09_circ_co2 = T_degrC_knudsen09_circ_co2 + 273.15


# p. 98:
d_knudsen09_circ_co2 = 2 * 0.00333e-2
L_knudsen09_circ_co2 = 2e-2
n_knudsen09_circ_co2 = 24    # 24 tubes in paralllel

# assuption:
p_r_knudsen09_circ_co2 = 5

Kn_knudsen09_circ_co2 = ( hlp.mfp_visc(T_knudsen09_circ_co2, 
                                       p_m_knudsen09_circ_co2, "CO2")
                          / d_knudsen09_circ_co2 )
(pin_knudsen09_circ_co2,
 pout_knudsen09_circ_co2) = hlp.get_pin_pout(Kn_knudsen09_circ_co2,
                                        d_knudsen09_circ_co2,
                                        T_knudsen09_circ_co2,
                                        "CO2",
                                        ratio=p_r_knudsen09_circ_co2)

pin_cm_Hg_knudsen09_circ_co2 = pin_knudsen09_circ_co2 / 1333
pout_cm_Hg_knudsen09_circ_co2 = pout_knudsen09_circ_co2 / 1333


# according to p. 84; this is the mass flow trough one (!) tube
mdot_notSI_knudsen09_circ_co2 = (X_0_knudsen09_circ_co2 
                            * (pin_cm_Hg_knudsen09_circ_co2
                               - pout_cm_Hg_knudsen09_circ_co2)
                            * hlp.get_M("CO2") 
                            / hlp.R 
                            / T_knudsen09_circ_co2) / n_knudsen09_circ_co2

# convert to SI units:
mdot_knudsen09_circ_co2 = mdot_notSI_knudsen09_circ_co2 * 1333 * 1e-6


G_knudsen09_circ_co2 = hlp.mdot_to_g(mdot_knudsen09_circ_co2,
                               L_knudsen09_circ_co2,
                               np.pi*d_knudsen09_circ_co2,
                               np.pi/4*d_knudsen09_circ_co2**2, 
                               pin_knudsen09_circ_co2-pout_knudsen09_circ_co2,
                               T_knudsen09_circ_co2,
                               "CO2")

# here, the data is truncated in the viscous regime:
Kn_knudsen09_circ_co2 = Kn_knudsen09_circ_co2[8:]
G_knudsen09_circ_co2 = G_knudsen09_circ_co2[8:]

"""
-------------------------------------------------------
Data for rectangular channels.
-------------------------------------------------------

"""


"""
Helium:
    
EWART, T., PERRIER, P., GRAUR, I., MÉOLANS, J. (2007). 
Mass flow rate measurements in a microchannel, from hydrodynamic 
to near free molecular regimes 
Journal of Fluid Mechanics  584(), 337. 
https://dx.doi.org/10.1017/s0022112007006374

data coming from table 3 in the paper

"""

ewart07 = np.loadtxt("csv/ewart07.csv",
                       skiprows=1, delimiter=";")
mdot_ewart07 = ewart07[:,0]
pins_ewart07 = ewart07[:,1]
pouts_ewart07 = ewart07[:,2]
T_ewart07 = ewart07[:,3]

# data from the paper:
H_ewart07 = 9.38e-6
W_ewart07 = 492e-6
L_ewart07 = 9.39e-3

gas_ewart07 = "He"

mfp_ewart07 = hlp.mfp_visc(T_ewart07,
                           (pins_ewart07+pouts_ewart07)/2,
                           gas_ewart07)

Kn_ewart07 = mfp_ewart07 / H_ewart07

G_ewart07 = hlp.mdot_to_g(mdot_ewart07,
                         L_ewart07,
                         2*(H_ewart07+W_ewart07),
                         H_ewart07*W_ewart07,
                         pins_ewart07-pouts_ewart07,
                         T_ewart07,
                         gas_ewart07)


"""
Again helium:

Graur, I., Perrier, P., Ghozlani, W., Méolans, J. (2009).
Measurements of tangential momentum accommodation coefficient for 
various gases in plane microchannel
Physics of Fluids  21(10), 102004. https://dx.doi.org/10.1063/1.3253696
"""

graur09_rect_he = np.loadtxt("csv/graur09_rect_he.csv",
                       skiprows=1, delimiter=";")

delta_graur09_rect_he = graur09_rect_he[:,0]
Q_graur09_rect_he = graur09_rect_he[:,1]  # this is the G used in the
                                          # paper which is different
                                          # to the G used in this
                                          # manuscript

H_graur09_rect_he = 9.38e-6
W_graur09_rect_he = 492e-6
L_graur09_rect_he = 9.39e-3

# from table I:
T_graur09_rect_he = (294.2 + 297.7) / 2  # mean value of min and max
p_r_graur09_rect_he = 2.99
gas_graur09_rect_he = "He"

# for this, see the definition of delta just below eq. (9) in the paper
Kn_graur09_rect_he = np.sqrt(np.pi)/2 / delta_graur09_rect_he

(pins_graur09_rect_he,
 pouts_graur09_rect_he) = hlp.get_pin_pout(Kn_graur09_rect_he,
                                           H_graur09_rect_he,
                                           T_graur09_rect_he,
                                           gas_graur09_rect_he,
                                           ratio=p_r_graur09_rect_he)
dp_graur09_rect_he = pins_graur09_rect_he - pouts_graur09_rect_he

# eq. (10):
mdot_graur09_rect_he = ( Q_graur09_rect_he
                        * H_graur09_rect_he**2 * W_graur09_rect_he
                        * dp_graur09_rect_he
                        / (L_graur09_rect_he
                           * np.sqrt(2*hlp.R
                           / hlp.get_M(gas_graur09_rect_he) 
                           * T_graur09_rect_he)) )

G_graur09_rect_he = hlp.mdot_to_g(mdot_graur09_rect_he, 
                                  L_graur09_rect_he,
                                  2*(W_graur09_rect_he+H_graur09_rect_he), 
                                  W_graur09_rect_he*H_graur09_rect_he,
                                  dp_graur09_rect_he,
                                  T_graur09_rect_he,
                                  gas_graur09_rect_he)



"""
Nitrogen:
    
Graur, I., Perrier, P., Ghozlani, W., Méolans, J. (2009).
Measurements of tangential momentum accommodation coefficient 
for various gases in plane microchannel
Physics of Fluids  21(10), 102004. https://dx.doi.org/10.1063/1.3253696

"""

graur09_rect_n2 = np.loadtxt("csv/graur09_rect_n2.csv",
                       skiprows=1, delimiter=";")

delta_graur09_rect_n2 = graur09_rect_n2[:,0]
Q_graur09_rect_n2 = graur09_rect_n2[:,1]  # this is the G used in the
                                          # paper which is different
                                          # to the G used in this
                                          # manuscript

H_graur09_rect_n2 = 9.38e-6
W_graur09_rect_n2 = 492e-6
L_graur09_rect_n2 = 9.39e-3

# from table I:
T_graur09_rect_n2 = (291.6 + 293.8) / 2  # mean value of min and max
p_r_graur09_rect_n2 = 2.02
gas_graur09_rect_n2 = "N2"

# for this, see the definition of delta just below eq. (9) in the paper
Kn_graur09_rect_n2 = np.sqrt(np.pi)/2 / delta_graur09_rect_n2

(pins_graur09_rect_n2,
 pouts_graur09_rect_n2) = hlp.get_pin_pout(Kn_graur09_rect_n2,
                                           H_graur09_rect_n2,
                                           T_graur09_rect_n2,
                                           gas_graur09_rect_n2,
                                           ratio=p_r_graur09_rect_n2)
dp_graur09_rect_n2 = pins_graur09_rect_n2 - pouts_graur09_rect_n2

# eq. (10):
mdot_graur09_rect_n2 = ( Q_graur09_rect_n2
                        * H_graur09_rect_n2**2 * W_graur09_rect_n2 
                        * dp_graur09_rect_n2
                        / (L_graur09_rect_n2 
                           * np.sqrt(2*hlp.R
                                     / hlp.get_M(gas_graur09_rect_n2) 
                                     * T_graur09_rect_n2)) )

G_graur09_rect_n2 = hlp.mdot_to_g(mdot_graur09_rect_n2, 
                                  L_graur09_rect_n2,
                                  2*(W_graur09_rect_n2+H_graur09_rect_n2), 
                                  W_graur09_rect_n2*H_graur09_rect_n2,
                                  dp_graur09_rect_n2,
                                  T_graur09_rect_n2,
                                  gas_graur09_rect_n2)


"""
Argon:
    
Graur, I., Perrier, P., Ghozlani, W., Méolans, J. (2009).
Measurements of tangential momentum accommodation coefficient for 
various gases in plane microchannel
Physics of Fluids  21(10), 102004. https://dx.doi.org/10.1063/1.3253696

"""

graur09_rect_ar = np.loadtxt("csv/graur09_rect_ar.csv",
                       skiprows=1, delimiter=";")

delta_graur09_rect_ar = graur09_rect_ar[:,0]
Q_graur09_rect_ar = graur09_rect_ar[:,1]  # this is the G used in the
                                          # paper which is different
                                          # to the G used in this
                                          # manuscript

H_graur09_rect_ar = 9.38e-6
W_graur09_rect_ar = 492e-6
L_graur09_rect_ar = 9.39e-3

# from table I:
T_graur09_rect_ar = (291.6 + 293.8) / 2  # mean value of min and max
p_r_graur09_rect_ar = 2.02
gas_graur09_rect_ar = "Ar"

# for this, see the definition of delta just below eq. (9) in the paper
Kn_graur09_rect_ar = np.sqrt(np.pi)/2 / delta_graur09_rect_ar

(pins_graur09_rect_ar,
 pouts_graur09_rect_ar) = hlp.get_pin_pout(Kn_graur09_rect_ar,
                                           H_graur09_rect_ar,
                                           T_graur09_rect_ar, 
                                           gas_graur09_rect_ar,
                                           ratio=p_r_graur09_rect_ar)
dp_graur09_rect_ar = pins_graur09_rect_ar - pouts_graur09_rect_ar

# eq. (10):
mdot_graur09_rect_ar = ( Q_graur09_rect_ar
                        * H_graur09_rect_ar**2 * W_graur09_rect_ar
                        * dp_graur09_rect_ar
                        / (L_graur09_rect_ar
                           * np.sqrt(2*hlp.R
                                     / hlp.get_M(gas_graur09_rect_ar) 
                                     * T_graur09_rect_ar)) )

G_graur09_rect_ar = hlp.mdot_to_g(mdot_graur09_rect_ar, 
                                 L_graur09_rect_ar,
                                  2*(W_graur09_rect_ar+H_graur09_rect_ar), 
                                  W_graur09_rect_ar*H_graur09_rect_ar,
                                  dp_graur09_rect_ar,
                                  T_graur09_rect_ar,
                                  gas_graur09_rect_ar)