"""
Author: Simon Kunze

This file contains helper functions used by the model,
the script for loading the literature data as well as the
plotting script.

Physicial constants and properties of matter are taken from:
Lide, D. (1995). CRC handbook of chemistry and physics:
a ready-reference book of chemical and physical data
(in short: HCnP)
"""

import numpy as np

# ----------------------------------------
# physical constants, source: HCnP, p. 1-1
R = 8.3144598           # universal gas constant [kg m^2 / (s^2 mol K)] 
kB = 1.38065e-23        # boltzmann constant [J/K]

def get_mu(gas, T):
    """Returns the dynamic viscosity of given gas at given temperature
    by linear interpolation.
    Source: HCnP, pp. 6-242f
    
    Parameters
    ----------
    gas : str
        choices are "He", "CO2" and "N2".
    T : float, array_like
        temperature in K between 20 and 100 °C.
    
    Returns
    -------
    visc : float, array_like
       dynamic viscosity of gas at temp T in Pa*s
    """
    temps = [200, 300, 400]  # [K]
    if np.any(T < min(temps)) or np.any(T > max(temps)):
        raise ValueError("Temperature out of range")
        
    if gas == "He":
        viscs = [15.1e-6, 19.9e-6, 24.3e-6]  # [Pa*s]
    elif gas == "CO2":
        viscs = [10.1e-6, 15.0e-6, 19.7e-6]   # [Pa*s]
    elif gas == "N2":
        viscs = [12.9e-6, 17.9e-6, 22.2e-6]   # [Pa*s]
    elif gas == "Ar":
        viscs = [15.9e-6, 22.7e-6, 28.6e-6]   # [Pa*s]
    else:
        raise ValueError("no valid gas given")

    
    visc = np.interp(T, temps, viscs)
    return visc

def get_M(gas):
    """Returns the molar mass of given gas.
    Source: HCnP, pp. 4-4ff
    
    Parameters
    ----------
    gas : str or array of str
        choices are "He", "CO2" and "N2"
    
    Returns
    -------
    M : float
       Molar mass of given gas(es).
    """
    if isinstance(gas, str):
        # single string
        if gas == "He":
            M = 4.002602e-3         # molar mass of helium [kg/mol]
        elif gas == "CO2":
            M = 44.008e-3           # molar mass of carbon dioxide [kg/mol]
        elif gas == "N2":  
            M = 28.014e-3           # molar mass of nitrogen [kg/mol]
        elif gas == "Ar":
            M = 39.948e-3          # [kg/mol]
        else:
            raise ValueError("no valid gas given")
    else:
        # iterable
        M = np.array([get_M(entry) for entry in gas])

    return M


# ----------------------------------------


# ----------------------------------------
# functions calculating things based of physics:
    
def get_pin_pout(Kn_m, D_C, T, gas, d_gas=None, ratio=1000):
    """
    Calculates the inlet and outlet pressure using the Kn number.

    Parameters
    ----------
    Kn_m : float, ndarray
        Mean Knudsen numbers.
    D_C : float
        Characteristic length used calculating the Kn number.
    T : float, ndarray
        Temperature.
    gas : str
        Gas species.
    d_gas : float, optional
        Molecular diameter of the gas. If this is set, the diameter
        is used to calculate the pressures from the mean free path.
        Otherwise, the viscosity is used.
        The default is None.
    ratio : float, optional
        Ratio between inlet and outlet pressure. The default is 1000.

    Returns
    -------
    pin : float, ndarray
        Inlet pressures.
    pout : float, ndarray
        Outlet pressures.

    """
    mfp_m = Kn_m * D_C
    if d_gas:
        p_m = kB*T/(np.sqrt(2)*np.pi*d_gas**2*mfp_m)
    else:
        p_m = mfp_to_p_visc(mfp_m, T, gas)
    pout = 2*p_m / (ratio+1)
    pin = pout * ratio
    return pin, pout

def mfp_to_p_visc(mfp, T, gas):
    """
    returns the pressure in Pa
    """
    p = get_mu(gas, T) / mfp * np.sqrt(np.pi * R * T 
                                       / (2*get_M(gas)))
    return p

def mfp_visc(T, p, gas, M = None, mu = None):
    """
    Mean free path using kinematic viscosity. 
    Source: Sharipov, F. (1999). Rarefied gas flow through 
    a long rectangular channel 
    Journal of Vacuum Science & Technology A: Vacuum, 
    Surfaces, and Films  17(5), 3062-3066.
    https://dx.doi.org/10.1116/1.582006 [eq. (4)]
    (just using R and M instead of kB and m)
    
    Parameters
    ----------
    T : float
        temperature in K
    p : float
        pressure of gas in Pa
    gas : string
        gas name. If you want to specify M and mu manually, 
        set this to None.
    M : float, optional   
        molar mass in kg/mol
    mu : float, optional
        dynamic viscosity of gas in Pa*s
        
    Returns
    -------
    mfp : float
        mean free path in m
        
    """
    
    if gas:
        M = get_M(gas)
        mu = get_mu(gas, T)
    return mu/p * np.sqrt(np.pi * R * T / (2*M))

def mfp(T,d,p):
    """
    mean free path using particle diameter
    source: Brodkey, Hershey: Transport Phenomena, Volume 2, p. 716
    
    Parameters
    ----------
    T : float, ndarray
        temperature in K
    d : float
        gas molecule diameter in m
    p : float, ndarray
        gas pressure in Pa
        
    Returns
    -------
    mfp : float, ndarray
        mean free path in m
    """        
    return kB*T/(np.sqrt(2)*np.pi*d**2*p)


def get_Kn(pin, pout, T, d_h, gas, d_gas=None):
    if d_gas:
        mfp_m = mfp(T, d_gas, (pin+pout)/2)
    else:
        mfp_m = mfp_visc(T, (pin+pout)/2, gas)
    Kn_m = mfp_m / d_h
    return Kn_m

def mdot_to_g(mdot, L, P, A, dp, T, gas):
    """
    Calculates the dimensionless mass flow.    

    Parameters
    ----------
    mdot : float, ndarray
        Mass flow.
    L : float
        Length of the channel.
    P : float
        Perimeter of the channel.
    A : float
        Cross-sectional area of the channel.
    dp : float, ndarray
        Pressure difference between inlet and outlet.
    T : float, ndarray
        Temperature.
    gas : str
        Gas species.

    Returns
    -------
    G : float, ndarray
        Dimensionless mass flow.

    """
    M = get_M(gas)
    G = mdot * 3*P*L / (8*A**2*dp) * np.sqrt(np.pi*R*T/(2*M))
    return G



# ----------------------------------------



# ----------------------------------------
# other helper functions:

def convert_to_array(x):
    if not type(x) == np.ndarray:
        if type(x) in [int, float, list]:
            return np.array(x)
        else:
            raise ValueError("no valid value")
    else:
        return x
    
def get_opt_kwargs(to_extract, **kwargs):
    """Extracts variables from a dictionary to_extract, 
    if they are present. Otherwise, apply standard values.
    
    The variables to extract are given in **kwargs, together
    with their standard values.
    
    Returns a dict_values list. Access it in the form of
    a, b, c = get_opt_kwargs(to_extract, a=1, b=2, c=3)
    For a single value:
    a, = get_opt_kwargs(to_extract, a=1)
    """
    
    for var in kwargs:
        if var in to_extract:
            kwargs[var] = to_extract[var]
    return kwargs.values()


# ----------------------------------------
    