library(luplot)
library(ggplot2)
library(luscale)
library(data.table)
library(ggrepel)
library(patchwork)
library(quitte)
library(RColorBrewer)
library(mip)
library(ggiraph)
library(ggiraphExtra)
source("theme_my.R")

### Contact
# Florian Humpenöder
# Potsdam Institute for Climate Impact Research
# humpenoeder@pik-potsdam.de
# January 2022

# load MAgPIE results as data.table
rep <- as.data.table(readRDS("../model_runs/report_all_SCP44.rds"))

# write MAgPIE variables to file
rep[,varunit:=paste0(variable," (",unit,")")]
write.csv(unique(rep$varunit),"vars_magpie.csv",row.names = FALSE,quote = FALSE)
rep[,varunit:=NULL]

# load validation data
hist <- readRDS(paste0("validation.rds"))

# write validation variables to file
hist[,varunit:=paste0(variable," (",unit,")")]
write.csv(unique(hist$varunit),"vars_validation.csv",row.names = FALSE,quote = FALSE)
hist[,varunit:=NULL]

# protein to kcal ratio
protein_to_kcal_ratio <- readRDS("protein_to_kcal_ratio.rds")

# rename GLO to World
rep[region=="GLO",region:="World"]

#split up the scenario name by "_". You have to adjust this for your scenario name pattern
rep[, c("rev","scen") := tstrsplit(scenario, "_", fixed=TRUE)]
#remove columns rev and pcost. not needed
rep[,c("rev"):=NULL]
#remove rev from scenario name
rep[,scenario:=scen]
rep[,scen:=NULL]

scen_sel <- c("SSP2-Ref-MP0","SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80")
rep <- rep[scenario %in% scen_sel,]


##### Main figures #####

### Fig1 ruminant meat and MP demand
# Fig1a
var <- c("Nutrition|Protein Supply|Secondary products|+|Microbial protein","Nutrition|Protein Supply|Livestock products|+|Ruminant meat")
names(var) <- c("Microbial Protein (MP) from Fermentation","Ruminant Meat from Livestock")
b <- rep[variable %in% var & region %in% c("World") & period >= 2020 & period <= 2050,]
b <- droplevels(b)
b$variable <- factor(b$variable,levels = var,labels = names(var))
Fig1a <- b
#write.csv(Fig1a,file="Fig1a.csv",row.names = FALSE)

color <- c("#2171b5","#6e016b","#88419d","#8c6bb1","#8c96c6","#9ebcda")

unit <- 'bold("g protein" ~ "cap"^{-1} ~ "day"^{-1})'
p1 <- ggplot(b,aes(x=period,y=value))+facet_wrap(vars(scenario),ncol = 2)+theme_my(rotate_x = 90)# + labs(caption = paste(Sys.Date()))
p1 <- p1+ylab(parse(text = unit))+xlim(2020,2050)+xlab(NULL)+scale_fill_manual("Source of Protein",values=color,guide = guide_legend(title.position = "left"))+theme(legend.position="bottom")+ggtitle(expression(bold(NULL)))
p1 <- p1+geom_area(aes(fill=variable),alpha=1) + guides(fill=guide_legend(ncol=1,title.position = "top", byrow = TRUE,reverse=TRUE))+ggtitle("Per-Capita Ruminant Meat and MP Consumption")

# Fig1b
var <- "Demand|Livestock products|+|Ruminant meat"
a <- hist[variable==var & region=="World" & period >= 1975,]
a <- droplevels(a)
a[,variable:="Ruminant meat"]
levels(a$scenario) <- "historical"

b <- rep[variable==var & region=="World" & period <= 2050,]
b[,variable:="Ruminant meat"]
b <- droplevels(b)

b1 <- b
a1 <- a

var <- "Demand|Secondary products|+|Microbial protein"
a <- hist[variable==var & region=="World" & period >= 1975,]
a <- droplevels(a)
a[,variable:="Microbial Protein (MP)"]
levels(a$scenario) <- "historical"

b <- rep[variable==var & region=="World" & period <= 2050,]
b[,variable:="Microbial Protein (MP)"]
b <- droplevels(b)

b2 <- b
a2 <- a

bb2 <- rbind(b1,b2)
Fig1b <- rbind(bb2,a1)
#write.csv(Fig1b,file="Fig1b.csv",row.names = FALSE)

unit <- 'bold("Mt DM" ~ "yr"^{-1})'

p2 <- mipLineHistorical(bb2,a1,ylab = parse(text = unit),title = "Total Demand",facet.dim = "variable",color.dim.name = "Future projection",facet.ncol = 2,xlab = NULL)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

# Fig1c
var <- "Demand|Feed|+|Pasture"
a <- hist[variable==var & region=="World" & period >= 1975,]
a <- droplevels(a)
a[,variable:="Pasture"]
levels(a$scenario) <- "historical"
a <- a[0,]

b <- rep[variable==var & region=="World" & period <= 2050,]
b[,variable:="Pasture"]

b <- droplevels(b)
b3 <- b
a3 <- a

var <- c("Demand|Feed|+|Crops","Demand|Feed|+|Crop residues","Demand|Feed|+|Forage","Demand|Processing|Breeding|Crops|Sugar crops|Sugar cane")
b <- rep[variable %in% var & region == "World" & period <= 2050,]
b <- b[,.(value=sum(value)), by = .(model,scenario,region,period,unit,scenario)]
b[,variable:="Cropland"]
b <- b[,c("model","scenario","region","variable","unit","period","value")]
b <- droplevels(b)

a <- hist[variable %in% var & region=="World" & period >= 1985,]
a <- droplevels(a)
a <- a[,.(value=sum(value)), by = .(model,scenario,region,period,unit)]
a[,variable:="Cropland"]
a <- a[,c("model","scenario","region","variable","unit","period","value")]
a <- a[0,]

b4 <- b
a4 <- a

bb3 <- rbind(b3,b4)
aa3 <- rbind(a3,a4)

Fig1c <- rbind(bb3,aa3)
#write.csv(Fig1c,file="Fig1c.csv",row.names = FALSE)

writexl::write_xlsx(list(Fig1a = Fig1a,Fig1b = Fig1b,Fig1c = Fig1c),"Fig1_demand.xlsx")

unit <- 'bold("Mt DM" ~ "yr"^{-1})'

p3 <- mipLineHistorical(bb3,aa3,ylab = parse(text = unit),title = "Overall Feed and Feedstock Demand",facet.dim = "variable",color.dim.name = "Future projection",facet.ncol = 2,xlim = c(1975,2050))+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p3 <- p3+theme_my()+theme(legend.position="bottom")+guides(color="none")#+xlab(NULL)

combined <- p1 | (p2 / p3)# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_annotation(tag_levels = 'a') + plot_layout(guides = "collect",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "Fig1_demand.pdf",combined,width = 10,height = 6,scale=1)


### Fig 2 Land cover change
#subset global
b <- rep[region %in% c("World") & period <= 2050,]
b <- droplevels(b)
#define variables and names
var <- c("Resources|Land Cover|+|Cropland","Resources|Land Cover|+|Pastures and Rangelands","Resources|Land Cover|+|Forest","Resources|Land Cover|+|Other Land")
names(var) <- c("Cropland","Pasture","Forest","Other Land")
#subset vars
b <- b[variable %in% var,]
b <- droplevels(b)
#rename and re-order
b$variable <- factor(b$variable,levels = rev(var),labels = names(rev(var)))
levels(b$variable)
#calc difference wrt to 2020
b <- b[period >= 2020,]
b[, value:=value-value[period==2020], by = .(model,scenario,region,variable)]
levels(b$unit) <- "Change in Mha compared to 2020"
#write.csv(b,file="Fig2_LandCoverChange.csv",row.names = FALSE)
writexl::write_xlsx(list(Fig2 = b),"Fig2_LandCoverChange.xlsx")
b$positive <- ifelse(b$value >= 0, b$value, 0)
b$negative <- ifelse(b$value < 0, b$value, -1e-36)
p <- ggplot(b,aes(x=period))+facet_wrap(vars(scenario),ncol = 4)+theme_my(rotate_x = 90)+ylab("Change in Mha compared to 2020")
p <- p + geom_hline(yintercept = 0,linetype="dotted")
p <- p + geom_area(aes(y=positive,fill=variable),position="stack") + geom_area(aes(y=negative,fill=variable),position="stack") + scale_fill_manual("Land Type",values=rev(c("chocolate4", "#E6AB02", "darkgreen", "#66A61E")))
p <- p+theme(legend.position = "bottom")+guides(fill=guide_legend(nrow=1,title.position = "left", byrow = TRUE,reverse=TRUE))+xlab(NULL)
ggsave("Fig2_LandCoverChange.pdf",p,scale=1.5,height = 2.5,width = 6.7)


### Fig 3 relative change
#Fig3g
var <- c("Resources|NetForestChange","Emissions|CO2|Land|+|Land-use Change","Emissions|CH4|Land|+|Agriculture","Emissions|N2O|Land|+|Agriculture","SDG|SDG15|Industrial and intentional biological fixation of N","Resources|Water|Withdrawal|Agriculture")
names(var) <- c('"Deforestation (SDG15)"','CO[2]*" emissions land (SDG13)"','CH[4]*" emissions agricult. (SDG13)"','N[2]*O*" emissions agricult. (SDG13)"','"Nitrogen fixation (SDG15)"','"Agricultural water use (SDG06)"')

b <- rep[variable %in% var & region %in% c("World") & period %in% c(2050),]
b <- droplevels(b)
b$positive <- ifelse(b$value >= 0, b$value, 0)
b$negative <- ifelse(b$value < 0, b$value, -1e-36)
b[variable %in% var[c(1,2)], group:="Non-Linear",]
b[variable %in% var[c(3,4,5,6)], group:="Linear",]
b$variable <- factor(b$variable,levels = var,labels=names(var))
#write.csv(b,file="Fig3g_SDG_absolute.csv",row.names = FALSE)

#calculate relative change compared to SSP2-Ref-MP0
b[, value2:=value/value[scenario=="SSP2-Ref-MP0"]-1, by = .(model,region,period,variable)]
b2 <- copy(b)
b2$positive <- NULL
b2$negative <- NULL
b2$value2 <- round(b2$value2,2)
b2 <- droplevels(b2)
Fig3g <- b2
levels(Fig3g$unit) <- rep("Relative change compared to SSP2-Ref-MP0",6)
Fig3g$value <- Fig3g$value2
Fig3g$value2 <- NULL
levels(Fig3g$variable) <- c("Deforestation (SDG15)","CO2 emissions land (SDG13)","CH4 emissions agricult. (SDG13)","N2O emissions agricult. (SDG13)","Nitrogen fixation (SDG15)","Agricultural water use (SDG06)")
#write.csv(b2,file="Fig3g_SDG_relative.csv",row.names = FALSE)


p1 <- ggplot(b,aes(x=scenario,y=value2,label=paste0(round(value2*100),"%")))+theme_my()+geom_point(aes(color=variable))+geom_line(aes(color=variable,group=variable,linetype=group))+theme(legend.position=c(0.16,0.4))+guides(col = guide_legend(reverse = TRUE,ncol=1,title.position = "top",title = "Variable",order = 2),linetype=guide_legend(order = 1,title = "Type of Relationship"))+ylab("Relative Change")+xlab("Scenario")+scale_y_continuous(labels = scales::percent)+scale_color_brewer(palette = "Dark2",labels=function(x) parse(text=x))#+facet_wrap(vars(group),ncol = 2)#scales = "free_y"#labeller(.cols = label_parsed, .multi_line = FALSE)
p1 <- p1+geom_label_repel(aes(group=variable,colour=variable),show.legend = F)+theme(legend.text = element_text(hjust = 0, debug = F))# + labels=function(x) parse(text=x)
p11 <- p1
#ggsave(filename = "Fig3g_SDG.pdf",p1,width = 8,height = 4.5,scale=1.3)

#Fig3b
title <- 'bold(CO[2]*" emissions land (SDG13)")'
unit <- 'bold("Mt" ~ CO[2] ~ "yr"^{-1})'
var <- "Emissions|CO2|Land|+|Land-use Change"
a <- hist[variable==var & region=="World" & period >= 1975 & model == "Gasser et al 2020",]
a <- droplevels(a)
b <- rep[variable==var & region=="World" & period >= 2000  & period <= 2050,]
b <- droplevels(b)
Fig3b <- rbind(b,a)
levels(Fig3b$variable) <- "CO2 emissions land (SDG13)"
#write.csv(rbind(b,a),file="Fig3b_SDG.csv",row.names = FALSE)
b$model <- ""
b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = parse(text = title),facet.dim = NULL,color.dim.name = "Scenario",xlim = c(1975,2050),legend.ncol = 4)
p1 <- p1+theme_my()+theme(legend.position=c(0.1,0.1))+guides(fill=F)+guides(fill=F,color=guide_legend(title.position = "left",nrow=1))

#Fig3d
title <- 'bold(N[2]*O*" emissions agricult. (SDG13)")'
unit <- 'bold("Mt" ~ N[2]*O* ~ "yr"^{-1})'
var <- "Emissions|N2O|Land|+|Agriculture"
a <- hist[variable==var & region=="World" & period >= 1975 & model=="EDGAR_LU",]
a <- droplevels(a)
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
Fig3d <- rbind(b,a)
levels(Fig3d$variable) <- "N2O emissions agricult. (SDG13)"
#write.csv(rbind(b,a),file="Fig3d_SDG.csv",row.names = FALSE)
b$model <- ""
b <- droplevels(b)
p2 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = parse(text = title),facet.dim = NULL,color.dim.name = "Scenario",xlim = c(1975,2050),legend.ncol = 4)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+theme_my()+theme(legend.position=c(0.1,0.1))+guides(fill=F)+guides(fill=F,color=guide_legend(title.position = "left",nrow=1))

#Fig3c
title <- 'bold(CH[4]*" emissions agricult. (SDG13)")'
unit <- 'bold("Mt" ~ CH[4] ~ "yr"^{-1})'
var <- "Emissions|CH4|Land|+|Agriculture"
a <- hist[variable==var & region=="World" & period >= 1975 & model=="EDGAR_LU",]
a <- droplevels(a)
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
Fig3c <- rbind(b,a)
levels(Fig3c$variable) <- "CH4 emissions agricult. (SDG13)"
#write.csv(rbind(b,a),file="Fig3c_SDG.csv",row.names = FALSE)
b$model <- ""
b <- droplevels(b)
p3 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = parse(text = title),facet.dim = NULL,color.dim.name = "Scenario",xlim = c(1975,2050),legend.ncol = 4)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p3 <- p3+theme_my()+theme(legend.position=c(0.1,0.1))+guides(fill=F,color=guide_legend(title.position = "left",nrow=1))

#Fig3f
title <- "Nitrogen fixation (SDG15)"
unit <- 'bold("Mt Nr yr"^{-1})'
var <- "Resources|Nitrogen|Cropland Budget|Inputs"
a <- hist[variable==var & region=="World" & period >= 1975,]
a <- droplevels(a)
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
Fig3f <- rbind(b,a)
levels(Fig3f$variable) <- title
#write.csv(rbind(b,a),file="Fig3f_SDG.csv",row.names = FALSE)
b$model <- ""
b <- droplevels(b)
p4 <- mipLineHistorical(b,a,ylab = parse(text = unit),title =title,facet.dim = NULL,color.dim.name = "Scenario",xlim = c(1975,2050),legend.ncol = 4)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p4 <- p4+theme_my()+theme(legend.position=c(0.7,0.5))+guides(fill=F)+guides(fill=F,color=guide_legend(title.position = "left",nrow=1))

#Fig3e
title <- "Agricultural water use (SDG06)"
unit <- 'bold(km^3 ~ "yr"^{-1})'
var <- "Resources|Water|Withdrawal|Agriculture"
a <- hist[variable==var & region=="World" & period >= 1975 & model %in% c("foley_2011","wada_2011","wisser_2008"),]
a <- droplevels(a)
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
Fig3e <- rbind(b,a)
levels(Fig3e$variable) <- title
#write.csv(rbind(b,a),file="Fig3e_SDG.csv",row.names = FALSE)
b$model <- ""
b <- droplevels(b)
p5 <- mipLineHistorical(b,a,ylab = parse(text = unit),title =title,facet.dim = NULL,color.dim.name = "Scenario",ylim = c(2500,4500),xlim = c(1975,2050),legend.ncol = 4)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p5 <- p5+theme_my()+theme(legend.position=c(0.2,0.7))+guides(fill=F)+guides(fill=F,color=guide_legend(title.position = "left",nrow=1))

#Fig3a
title <- "Deforestation (SDG15)"
unit <- 'bold(Mha ~ "yr"^{-1})'
var <- "Resources|Land Cover|+|Forest"
a <- hist[variable==var & region=="World" & period >= 1975 & model=="FAO_forest",]
a <- droplevels(a)
levels(a$model) <- "FAOSTAT"
a$value <- -a$value
a <- as.magpie(a)
a <- as.magpie(apply(a, c(1, 3), diff))
a <- a/5
a <- as.data.table(as.quitte(a))
levels(a$unit) <- "Mha/yr"
levels(a$variable) <- "Resources|NetForestChange"
levels(a$region) <- "World"
a$region1 <- NULL
var <- "Resources|NetForestChange"
b <- rep[variable==var & region=="World" & period >= 2000 & period <= 2050,]
b$value <- -b$value
b <- droplevels(b)
Fig3a <- rbind(b,a)
levels(Fig3a$variable) <- title
#write.csv(rbind(b,a),file="Fig3a_SDG.csv",row.names = FALSE)
b$model <- ""
b <- droplevels(b)
p6 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = title,facet.dim = NULL,color.dim.name = "Scenario",xlim = c(1975,2050),ylim = c(0,8),legend.ncol = 4)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p6 <- p6+theme_my()+theme(legend.position=c(0.2,0.7))+guides(fill=FALSE)+guides(fill=F,color=guide_legend(title.position = "left",nrow=1))

p22 <- p6 + p1 + p3 + p2 + p5 + p4 & theme(legend.position = "bottom",legend.justification='left')#  + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- (p22+plot_layout(guides = "collect",ncol = 3,tag_level = 'keep'))/(p11+plot_layout(guides = "keep",ncol = 1,tag_level = "keep"))+ plot_annotation(tag_levels = c('a'))
ggsave(filename = "Fig3_SDG.pdf",combined,width = 10,height = 10,scale=1)

writexl::write_xlsx(list(Fig3a = Fig3a, Fig3b = Fig3b, Fig3c = Fig3c, Fig3d = Fig3d, Fig3e = Fig3e, Fig3f = Fig3f, Fig3g = Fig3g),"Fig3_SDG.xlsx")


##### Extended Data Figures #####

### Popuation and Income

ssp <- "SSP2"
scen_sel <- "SSP2-Ref-MP0"

var <- "Population"
a <- hist[variable==var & region!="World" & period >= 1975 & model=="WDI",]
a <- droplevels(a)
b <- rep[variable==var & region!="World" & period <=2050,]
b[,scenario:=ssp]
b$scenario <- factor(b$scenario)

b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+theme_my()

var <- "Income"
a <- hist[variable==var & region!="World" & period >= 1975 & model=="James" & unit=="US$05 MER/cap/yr",]
a <- droplevels(a)
b <- rep[variable==var & region!="World" & period <= 2050,]
b[,scenario:=ssp]
b$scenario <- factor(b$scenario)

b <- droplevels(b)
p2 <- mipLineHistorical(b,a,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})+theme_my()#+guides(color="none")#+labs(fill="WDI")


combined <- p1 + p2 + plot_annotation(tag_levels = 'a') + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "keep",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig1_pop_income_reg.pdf",combined,width = 10,height = 6,scale=1.3)
ggsave(filename = "ExtDataFig1_pop_income_reg.eps",combined,width = 10,height = 6,scale=1.3,dpi = 300,device=cairo_ps)

### per-capita demand ruminant meat and MP
ssp <- "SSP2"

title <- "Nutrition|Livestock products|Ruminant meat"
var <- "Nutrition|Calorie Supply|Livestock products|+|Ruminant meat"
a <- hist[variable==var & region!="World" & period >= 1975 & period <= 2005 & scenario=="SSP2",]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = b$unit,title = title,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p1 <- p1+scale_y_continuous(name = b$unit,sec.axis = sec_axis( trans=~.*as.numeric(protein_to_kcal_ratio[,,"livst_rum"]), name="g protein/cap/day"))+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

title <- "Nutrition|Fermentation|Microbial protein"
var <- "Nutrition|Calorie Supply|Secondary products|+|Microbial protein"
a <- hist[variable==var & region!="World" & period >= 1975 & period <= 2005 & scenario=="SSP2",]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]

ref <- b[scenario=="SSP2-Ref-MP80",]
ref[,value:=0]
ref[,scenario:="SSP2-Ref-MP0"]
b <- rbind(ref,b)

b <- droplevels(b)
p2 <- mipLineHistorical(b,a,ylab = b$unit,title = title,facet.dim = "region",color.dim.name = "Future projection")+xlim(c(1975,2050))+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+scale_y_continuous(name = b$unit,sec.axis = sec_axis( trans=~.*as.numeric(protein_to_kcal_ratio[,,"scp"]), name="g protein/cap/day"))+theme_my()+theme(legend.position="bottom")#+xlab(NULL)
p2 <- p2+guides(color="none")

bb <- rep[variable %in% c("Nutrition|Calorie Supply|Livestock products|+|Dairy","Nutrition|Calorie Supply|Livestock products|+|Ruminant meat") & period <= 2050,]
bb <- bb[,1:7]
#write.csv(bb,"ExtDataFig2_kcal_rum_MP_reg.csv",row.names = FALSE)

bb[variable=="Nutrition|Calorie Supply|Secondary products|+|Microbial protein",value:=value*as.numeric(protein_to_kcal_ratio[,,"scp"])]
bb[variable=="Nutrition|Calorie Supply|Livestock products|+|Ruminant meat",value:=value*as.numeric(protein_to_kcal_ratio[,,"livst_rum"])]
bb[,unit:="g protein/cap/day"]
#write.csv(bb,"ExtDataFig2_protein_rum_MP_reg.csv",row.names = FALSE)

combined <- p1 + p2 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "collect",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig2_kcal_rum_MP_reg.pdf",combined,width = 10,height = 6,scale=1.3)
ggsave(filename = "ExtDataFig2_kcal_rum_MP_reg.eps",combined,width = 10,height = 6,scale=1.3,dpi = 300,device=cairo_ps)

### total demand ruminant meat and MP
ssp <- "SSP2"

var <- "Demand|Livestock products|+|Ruminant meat"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p1 <- p1+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

var <- "Demand|Secondary products|+|Microbial protein"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p2 <- mipLineHistorical(b,NULL,ylab = b$unit,title = "Demand|Fermentation|+|Microbial protein",facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})+xlim(1975,2050)#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+theme_my()+theme(legend.position="bottom")#+xlab(NULL)
p2 <- p2+guides(color="none")


combined <- p1 + p2 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "collect",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig3_dem_rum_MP_reg.pdf",combined,width = 10,height = 6,scale=1.3)
ggsave(filename = "ExtDataFig3_dem_rum_MP_reg.eps",combined,width = 10,height = 6,scale=1.3,dpi = 300,device=cairo_ps)

### dairy and ruminant production

b <- rep[scenario %in% c("SSP2-Ref-MP0","SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80") & variable %in% c("Production|Livestock products|+|Dairy") & region == "World" & period <= 2050,]
b <- droplevels(b)
var <- c("Production|Livestock products|+|Dairy")
a <- hist[variable %in% var & region=="World" & period >= 1985,]
a <- droplevels(a)
p1_glo <- mipLineHistorical(b,a,title = "Production|Livestock products|+|Dairy",facet.dim = "region",color.dim.name = "Future projection",facet.ncol = 4,ylab = b$unit)+theme_my()+theme(legend.position = "bottom")

b <- rep[scenario %in% c("SSP2-Ref-MP0","SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80") & variable %in% c("Production|Livestock products|+|Ruminant meat") & region == "World" & period <= 2050,]
b <- droplevels(b)
var <- c("Production|Livestock products|+|Ruminant meat")
a <- hist[variable %in% var & region=="World" & period >= 1985,]
a <- droplevels(a)
p2_glo <- mipLineHistorical(b,a,title = "Production|Livestock products|+|Ruminant Meat",facet.dim = "region",color.dim.name = "Future projection",facet.ncol = 4,ylab = b$unit)+theme_my()+theme(legend.position = "bottom")

combined <- p2_glo + p1_glo + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "collect",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig4_dairy_rummeat_production.pdf",combined,width = 10,height = 5,scale=1.3)
ggsave(filename = "ExtDataFig4_dairy_rummeat_production.eps",combined,width = 10,height = 5,scale=1.3,dpi = 300,device=cairo_ps)

### Feed and Feedstock demand for ruminant meat and MP

#demand crop SCP
crop <- rep[scenario %in% c("SSP2-Ref-MP0","SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80") & variable %in% c("Demand|Feed|Feed for Ruminant meat|+|Crops","Demand|Feed|Feed for Ruminant meat|+|Crop residues","Demand|Feed|Feed for Ruminant meat|+|Forage") & region == "World" & period <= 2050,]
crop <- crop[,.(value=sum(value)), by = .(model,scenario,region,period,unit,scenario)]
crop[,variable:="Demand|Feed|Ruminant meat|Cropland"]

pasture <- rep[scenario %in% c("SSP2-Ref-MP0","SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80") & variable %in% c("Demand|Feed|Feed for Ruminant meat|+|Pasture") & region == "World" & period <= 2050,]
pasture <- pasture[,.(value=sum(value)), by = .(model,scenario,region,period,unit,scenario)]
pasture[,variable:="Demand|Feed|Ruminant meat|Pasture"]
b <- rbind(crop,pasture)
b <- b[,c("model","scenario","region","variable","unit","period","value")]
b <- droplevels(b)

var <- c("Demand|Feed|Feed for Ruminant meat|+|Crops","Demand|Feed|Feed for Ruminant meat|+|Crop residues","Demand|Feed|Feed for Ruminant meat|+|Forage")
a <- hist[variable %in% var & region=="World" & period >= 1985,]
a <- droplevels(a)
a <- a[,.(value=sum(value)), by = .(model,scenario,region,period,unit)]
a[,variable:="Demand|Feed|Ruminant meat|Cropland"]
a1 <- a

var <- c("Demand|Feed|Feed for Ruminant meat|+|Pasture")
a <- hist[variable %in% var & region=="World" & period >= 1985,]
a <- droplevels(a)
a[,variable:="Demand|Feed|Ruminant meat|Pasture"]
a2 <- a
a <- rbind(a1,a2)
a <- a[,c("model","scenario","region","variable","unit","period","value")]
p0 <- mipLineHistorical(b,title = "Feed Demand for Ruminant Meat production",facet.dim = "variable",color.dim.name = "Scenario",facet.ncol = 4,ylab = b$unit)+theme_my()#+theme(legend.position = "bottom")

b <- rep[scenario %in% c("SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80") & variable %in% c("Demand|Processing|Breeding|Crops|Sugar crops|Sugar cane") & region == "World" & period <= 2050,]
b <- droplevels(b)
bb <- b[scenario %in% c("SSP2-Ref-MP20"),]
bb <- droplevels(bb)
bb$scenario <- "SSP2-Ref-MP0"
bb[,value:=0]
b <- rbind(bb,b)
b$scenario <- factor(b$scenario)
b$variable <- factor(b$variable,levels = c("Demand|Processing|Breeding|Crops|Sugar crops|Sugar cane"),labels = c("Demand|Feedstock|Fermentation|Sugar Cane"))
b <- droplevels(b)
p1 <- mipLineHistorical(b,title = "Feedstock Demand for MP production",facet.dim = "variable",color.dim.name = "Scenario",facet.ncol = 4,ylab = b$unit)+theme_my()#+theme(legend.position = "none")
p1 <- p1+guides(color="none")

b <- rep[scenario %in% c("SSP2-Ref-MP0","SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80") & variable %in% c("Resources|Land Cover|+|Cropland","Resources|Land Cover|+|Pastures and Rangelands") & region == "World" & period >= 2020 & period <= 2050,]
b <- droplevels(b)
b$variable <- factor(b$variable,levels = c("Resources|Land Cover|+|Cropland","Resources|Land Cover|+|Pastures and Rangelands"),labels = c("Land-use Change|Cropland","Land-use Change|Pasture"))
#calc difference wrt to 2020
b <- b[period >= 2020,]
b[, value:=value-value[period==2020], by = .(model,scenario,region,variable)]
b <- droplevels(b)
p2 <- mipLineHistorical(b,title = "System-wide Land-use Change",facet.dim = "variable",color.dim.name = "Scenario",facet.ncol = 4,ylab = "Change in Mha compared to 2020")+theme_my()#+theme(legend.position = "none")
p2 <- p2+guides(color="none")

combined <- (p0 + p1 + plot_layout(widths = c(2/3, 1/3))) / p2 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "collect",ncol = 1) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig5_MP_dem_land.pdf",combined,width = 9,height = 10,scale=1.2)
ggsave(filename = "ExtDataFig5_MP_dem_land.eps",combined,width = 9,height = 10,scale=1.2,dpi = 300,device=cairo_ps)

### regional land-use change 

title <- "Land Cover|Cropland"
var <- "Resources|Land Cover|+|Cropland"
a <- hist[variable==var & region!="World" & period >= 1975 & model == "FAO_crop_past",]
a <- droplevels(a)
levels(a$model) <- "FAOSTAT"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = b$unit,title = title,facet.dim = "region",color.dim.name = "Future projection",scales="free_y")
p1 <- p1+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

title <- "Land Cover|Pasture"
var <- "Resources|Land Cover|+|Pastures and Rangelands"
a <- hist[variable==var & region!="World" & period >= 1975 & model == "FAO_crop_past",]
a <- droplevels(a)
levels(a$model) <- "FAOSTAT"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p2 <- mipLineHistorical(b,a,ylab = b$unit,title = title,facet.dim = "region",color.dim.name = "Future projection",scales = "free_y")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

title <- "Land Cover|Forest"
var <- "Resources|Land Cover|+|Forest"
a <- hist[variable==var & region!="World" & period >= 1975 & model=="FAO_forest",]
a <- droplevels(a)
levels(a$model) <- "FAOSTAT"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p3 <- mipLineHistorical(b,a,ylab = b$unit,title = title,facet.dim = "region",color.dim.name = "Future projection",scales = "free_y",xlim=c(1975,2050))+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p3 <- p3+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

title <- "Land Cover|Other"
var <- "Resources|Land Cover|+|Other Land"
a <- hist[variable==var & region!="World" & period >= 1975 & model=="MAgPIEown",]
a <- droplevels(a)
levels(a$model) <- "LUH2-based"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p4 <- mipLineHistorical(b,ylab = b$unit,title =title,facet.dim = "region",color.dim.name = "Future projection",scales = "free_y")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p4 <- p4+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

combined <- p1 + p2 + p3 + p4 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "keep",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig6_lu_reg.pdf",combined,width = 10,height = 12,scale=1.3)
ggsave(filename = "ExtDataFig6_lu_reg.eps",combined,width = 10,height = 12,scale=1.3,dpi = 300,device=cairo_ps)

### regional land-use change 2050 bar plot

var <- c("Resources|Land Cover|+|Cropland","Resources|Land Cover|+|Pastures and Rangelands","Resources|Land Cover|+|Forest","Resources|Land Cover|+|Other Land")
names(var) <- c("Cropland","Pasture","Forest","Other Land")
b <- rep[variable %in% var & region != "World" & period %in% c(2020,2050),]
b <- droplevels(b)
b$variable <- factor(b$variable,levels = var,labels = names(var))
b$region <- factor(b$region,levels = rev(levels(b$region)))
#calc difference wrt to 2020
b[, value:=value-value[period==2020], by = .(model,scenario,region,variable)]
b <- b[period == 2050,]
#write.csv(b,file="ExtDataFig7_lu_reg_2050_bar.csv",row.names = FALSE)
b$positive <- ifelse(b$value >= 0, b$value, 0)
b$negative <- ifelse(b$value < 0, b$value, -1e-36)

p <- ggplot(b,aes(y=region))+facet_wrap(vars(scenario),ncol=4,scales = "fixed")+theme_my(rotate_x = FALSE)+ylab(NULL)
p <- p+geom_bar(aes(x=positive,fill=variable),position="stack",stat = "identity")
p <- p+geom_bar(aes(x=negative,fill=variable,),position="stack",stat = "identity")
p <- p + scale_fill_manual("Land Type",values=c("chocolate4", "#E6AB02", "darkgreen", "#66A61E"))
p <- p+theme(legend.position = "bottom")+guides(color=guide_legend("Scenario",nrow=1,title.position = "left", byrow = TRUE,reverse=F))+xlab("Change in Mha between 2020 and 2050")# + labs(caption = paste(Sys.Date()))
ggsave(filename = "ExtDataFig7_lu_reg_2050_bar.pdf",p,width = 8,height = 4,scale=1.3)
ggsave(filename = "ExtDataFig7_lu_reg_2050_bar.eps",p,width = 8,height = 4,scale=1.3,dpi = 300,device=cairo_ps)

### SDGs regional: deforestation and water

title <- "Agricultural water use (SDG06)"
unit <- 'bold(km^3 ~ "yr"^{-1})'
var <- "Resources|Water|Withdrawal|Agriculture"
a <- hist[variable==var & region=="World" & period >= 1975 & model %in% c("foley_2011","wada_2011","wisser_2008"),]
a <- droplevels(a)
b <- rep[variable==var & region=="World" & period <= 2050,]
b$model <- ""
b <- droplevels(b)
p2 <- mipLineHistorical(b,a,ylab = parse(text = unit),title =title,facet.dim = NULL,color.dim.name = "Scenario",ylim = c(2500,4500),xlim = c(1975,2050),legend.ncol = 1)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})+theme_my()#+guides(color="none")#+labs(fill="WDI")

title <- "Deforestation (SDG15)"
unit <- 'bold(Mha ~ "yr"^{-1})'
var <- "Resources|Land Cover|+|Forest"
a <- hist[variable==var & region!="World" & period >= 1975 & model=="FAO_forest",]
a <- droplevels(a)
levels(a$model) <- "FAOSTAT"
#calc annual deforestation rate
a$value <- -a$value
a <- as.magpie(a)
a <- as.magpie(apply(a, c(1, 3), diff))
a <- a/5
a <- as.data.table(as.quitte(a))
levels(a$unit) <- "Mha/yr"
levels(a$variable) <- "Resources|NetForestChange"
var <- "Resources|NetForestChange"
b <- rep[variable==var & region!="World" & period <= 2050,]
b$model <- ""
b$value <- -b$value
b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = title,facet.dim = "region",color.dim.name = "Scenario",xlim = c(1975,2050),legend.ncol = 1)+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})+theme_my()#+guides(color="none")#+labs(fill="WDI")

combined <- p1 + p2 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "keep",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig8_SDG_water_defor.pdf",combined,width = 10,height = 6,scale=1.3)
ggsave(filename = "ExtDataFig8_SDG_water_defor.eps",combined,width = 10,height = 6,scale=1.3,dpi = 300,device=cairo_ps)

### SDGs regional: GHG emissios and nitrogen

ssp <- "SSP2"

title <- 'bold(CO[2]*" emissions land (SDG13)")'
unit <- 'bold("Mt" ~ CO[2] ~ "yr"^{-1})'
var <- "Emissions|CO2|Land|+|Land-use Change"
a <- hist[variable==var & region!="World" & period >= 1975 & model == "Gasser et al 2020",]
a <- droplevels(a)
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = parse(text = title),facet.dim = "region",color.dim.name = "Future projection")
p1 <- p1+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

title <- 'bold(N[2]*O*" emissions agricult. (SDG13)")'
unit <- 'bold("Mt" ~ N[2]*O* ~ "yr"^{-1})'
unit2 <- 'bold("Mt" ~ CO[2]*eq ~ "yr"^{-1})'
var <- "Emissions|N2O|Land|+|Agriculture"
a <- hist[variable==var & region!="World" & period >= 1975 & model=="EDGAR_LU",]
a <- droplevels(a)
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p2 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = parse(text = title),facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+scale_y_continuous(name = parse(text = unit),sec.axis = sec_axis( trans=~.*265, name=parse(text = unit2)))+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

title <- 'bold(CH[4]*" emissions agricult. (SDG13)")'
unit <- 'bold("Mt" ~ CH[4] ~ "yr"^{-1})'
var <- "Emissions|CH4|Land|+|Agriculture"
a <- hist[variable==var & region!="World" & period >= 1975 & model=="EDGAR_LU",]
a <- droplevels(a)
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p3 <- mipLineHistorical(b,a,ylab = parse(text = unit),title = parse(text = title),facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p3 <- p3+scale_y_continuous(name = parse(text = unit),sec.axis = sec_axis( trans=~.*28, name=parse(text = unit2)))+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

title <- "Nitrogen fixation (SDG15)"
unit <- 'bold("Mt Nr yr"^{-1})'
var <- "Resources|Nitrogen|Cropland Budget|Inputs"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p4 <- mipLineHistorical(b,a,ylab = parse(text = unit),title =title,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p4 <- p4+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

combined <- p1 + p2 + p3 + p4 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "keep",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "ExtDataFig9_SDG_reg.pdf",combined,width = 10,height = 12,scale=1.3)
ggsave(filename = "ExtDataFig9_SDG_reg.eps",combined,width = 10,height = 12,scale=1.3,dpi = 300,device=cairo_ps)

##### Supplementary Figures ######

### Total Food demand
ssp <- "SSP2"

var <- "Demand|Food|+|Crops"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p1 <- mipLineHistorical(b,a,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")
p1 <- p1+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

var <- "Demand|Food|+|Livestock products"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p2 <- mipLineHistorical(b,a,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

combined <- p1 + p2 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "collect",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "SupFig2_dem_food_reg.pdf",combined,width = 10,height = 6,scale=1.3)


### Bioenergy and Afforestation

var <- "Demand|Bioenergy|++|1st generation"
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
p1 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p1 <- p1+theme_my()+theme(legend.position="bottom")+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))

var <- "Demand|Bioenergy|++|1st generation"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p2 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p2 <- p2+theme_my()+theme(legend.position="bottom")+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))


var <- "Demand|Bioenergy|++|2nd generation"
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
p3 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p3 <- p3+theme_my()+theme(legend.position="bottom")+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))

var <- "Demand|Bioenergy|++|2nd generation"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p4 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p4 <- p4+theme_my()+theme(legend.position="bottom")+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))

var <- "Resources|Land Cover Change|Forest|+|Managed Forest"
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
p5 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p5 <- p5+theme_my()+theme(legend.position="bottom")+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))

var <- "Resources|Land Cover Change|Forest|+|Managed Forest"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p6 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p6 <- p6+theme_my()+theme(legend.position="bottom")+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))

combined <- p1 + p2 + p3 + p4 + p5 + p6 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "collect",ncol = 2) & theme(legend.position = "none")
ggsave(filename = "SupFig3_bioen_aff.pdf",combined,width = 10,height = 12,scale=1.3)


### Feed demand detail

ssp <- "SSP2"

var <- "Demand|Feed|+|Crops"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p0 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")
p0 <- p0+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

var <- "Demand|Feed|+|Crop residues"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p1 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")
p1 <- p1+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

var <- "Demand|Feed|+|Forage"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p2 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")
p2 <- p2+theme_my()+theme(legend.position="bottom")#+xlab(NULL)

var <- "Demand|Feed|+|Pasture"
a <- hist[variable==var & region!="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region!="World" & period <= 2050,]
b <- droplevels(b)
p3 <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p3 <- p3+theme_my()+theme(legend.position="bottom")+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))

var <- "Demand|Feed|+|Pasture"
a <- hist[variable==var & region=="World" & period >= 1975,]
a <- droplevels(a)
levels(a$scenario) <- "historical"
b <- rep[variable==var & region=="World" & period <= 2050,]
b <- droplevels(b)
p3_glo <- mipLineHistorical(b,ylab = b$unit,title = b$variable,facet.dim = "region",color.dim.name = "Future projection")+scale_y_continuous(labels=function(n){format(n, scientific = FALSE)})#+guides(color="none")#+labs(fill="WDI")
p3_glo <- p3_glo+theme_my()+theme(legend.position="bottom")#+xlab(NULL)


#demand crop SCP
b <- rep[scenario %in% c("SSP2-Ref-MP0","SSP2-Ref-MP20","SSP2-Ref-MP50","SSP2-Ref-MP80") & variable %in% c("Demand|Feed|+|Crops","Demand|Feed|+|Crop residues","Demand|Feed|+|Forage") & region == "World" & period <= 2050,]
b <- b[,.(value=sum(value)), by = .(model,scenario,region,period,unit,scenario)]
b[,variable:="Demand|Feed|Cropland"]
b <- b[,c("model","scenario","region","variable","unit","period","value")]
b <- droplevels(b)
var <- c("Demand|Feed|+|Crops","Demand|Feed|+|Crop residues","Demand|Feed|+|Forage")
a <- hist[variable %in% var & region=="World" & period >= 1985,]
a <- droplevels(a)
a <- a[,.(value=sum(value)), by = .(model,scenario,region,period,unit)]
a[,variable:="Demand|Feed|Cropland"]
a <- a[,c("model","scenario","region","variable","unit","period","value")]

p1_glo <- mipLineHistorical(b,title = "Demand|Feed|Cropland",facet.dim = "region",color.dim.name = "Future projection",facet.ncol = 4,ylab = b$unit)+theme_my()+theme(legend.position = "bottom")

##reg
#demand crop SCP
b <- rep[variable %in% c("Demand|Feed|+|Crops","Demand|Feed|+|Crop residues","Demand|Feed|+|Forage") & region != "World" & period <= 2050,]
b <- b[,.(value=sum(value)), by = .(model,scenario,region,period,unit,scenario)]
b[,variable:="Demand|Feed|Cropland"]
b <- b[,c("model","scenario","region","variable","unit","period","value")]
b <- droplevels(b)
var <- c("Demand|Feed|+|Crops","Demand|Feed|+|Crop residues","Demand|Feed|+|Forage")
a <- hist[variable %in% var & region!="World" & period >= 1985,]
a <- droplevels(a)
a <- a[,.(value=sum(value)), by = .(model,scenario,region,period,unit)]
a[,variable:="Demand|Feed|Cropland"]
a <- a[,c("model","scenario","region","variable","unit","period","value")]

p1_reg <- mipLineHistorical(b,title = "Demand|Feed|Cropland",facet.dim = "region",color.dim.name = "Future projection",facet.ncol = 3,ylab = b$unit)+theme_my()+theme(legend.position = "bottom")
p1_reg <- p1_reg+scale_x_continuous(guide = guide_axis(check.overlap = TRUE))

combined <- p1_glo + p1_reg + p3_glo + p3 + plot_annotation(tag_levels = 'a')# + labs(caption = paste(Sys.Date()))
combined <- combined + plot_layout(guides = "collect",ncol = 2) & theme(legend.position = "bottom")
ggsave(filename = "SupFig4_dem_feed_reg.pdf",combined,width = 10,height = 10,scale=1.3)

### Map

#data for map
ref <- read.magpie("../model_runs/SCP44_SSP2-Ref-MP0/cell.land_0.5_share.mz")
pol <- read.magpie("../model_runs/SCP44_SSP2-Ref-MP50/cell.land_0.5_share.mz")
diff <- pol-ref
diff <- dimSums(diff[,,c("primforest","secdforest","other")],dim=3)
getNames(diff) <- "NatVeg"
write.magpie(diff,"SupFig5_map.nc")
#unit: Difference in cell share
#title: Forest and other natural land cover in 2050
#subtitle: Difference between SSP2-Ref-MP50 and SSP2-Ref-MP0
#Plot map in SupFig5_map.nc with Panoply: https://www.giss.nasa.gov/tools/panoply/

