
# function returning a list of length 5 containing the hydraulic equation of the
# five fictive cases based on the Mercier hydrometric station
#  * rc0: Flat weir
#  * rc1: Rectangular notch (L=3m)
#  * rc2: Rectangular notch (L=1m)
#  * rc3: Triangular notch (alpha=90deg)
#  * rc4: Triangular notch (alpha=120deg)
# Each item in the list is a list with the following components:
#  * f: rating curve equation ==> a function taking stage values as input to compute 
#    the corresponding streamflow values
#  * lab: a label describing the case (usefull for plotting)
#  * lab_fr: a label describing the case in french
hydraulicEquations <- function() {
    ###---------------------------------------------------------------###
    # defining fictive theoretical rating curves                        #
    ###---------------------------------------------------------------###
    rcs = list()
    
    #####################################################################
    ### case 1: wide rectangulaire weir
    rcname = "rc1"
    rclab_fr = "Déversoir rectangulaire sans encoche (L=5m)"
    rclab_en = "Flat weir"
    f <- function(h){
        Kr = 0.4*sqrt(2*9.81); Kt = 0.31*sqrt(2*9.81); Kc = 25*sqrt(0.005)
        Cr = 1.5; Ct = 2.5; Cc = 1.67
        if (h<=1){
            0
        }else if (h>1 & h<=2){
            Kr*5*(h-1)^Cr
        }else if (h>2){
            b2 = 2-((Kr*5*(2-1)^Cr)/(Kc*5))^(1/Cc)
            Kc*5*(h-b2)^Cc
        }
    }
    rcs[[rcname]] = list("f"=function(h) {
        return(sapply(h, f))
    },"lab"=rclab_en, "lab_fr"=rclab_fr)
    
    #####################################################################
    ### case 2: two rectangular weirs including a wide and shallow one
    rcname = "rc2"
    rclab_fr = "Encoche rectangulaire (L=3m)"
    rclab_en = "Rectangular notch (L=3m)"
    f <- function(h){
        Kr = 0.4*sqrt(2*9.81); Kt = 0.31*sqrt(2*9.81); Kc = 25*sqrt(0.005)
        Cr = 1.5; Ct = 2.5; Cc = 1.67
        if (h<=0.72){
            0
        }else if (h>0.72 & h<=1){
            Kr*3*(h-0.72)^Cr
        }else if (h>1 & h<=2){
            b2 = 1
            Kr*3*(h-0.72)^Cr + Kr*2*(h-b2)^Cr
        }else if (h>2){
            b2 = 1
            b3 = 2-((Kr*3*(2-0.72)^Cr + Kr*2*(2-b2)^Cr)/(Kc*5))^(1/Cc)
            Kc*5*(h-b3)^Cc
        }
    }
    rcs[[rcname]] = list("f"=function(h) {
        return(sapply(h, f))
    },"lab"=rclab_en, "lab_fr"=rclab_fr)
    
    
    #####################################################################
    ### case 3: two rectangular weirs including a narrow and deep one
    rcname = "rc3"
    rclab_fr = "Encoche rectangulaire (L=1m)"
    rclab_en = "Rectangular notch (L=1m)"
    f <- function(h){
        Kr = 0.4*sqrt(2*9.81); Kt = 0.31*sqrt(2*9.81); Kc = 25*sqrt(0.005)
        Cr = 1.5; Ct = 2.5; Cc = 1.67
        if (h<=0.41){
            0
        }else if (h>0.41 & h<=1){
            Kr*1*(h-0.41)^Cr
        }else if (h>1 & h<=2){
            b2 = 1
            Kr*1*(h-0.41)^Cr + Kr*4*(h-b2)^Cr
        }else if (h>2){
            b2 = 1
            b3 = 2-((Kr*1*(2-0.41)^Cr + Kr*4*(2-b2)^Cr)/(Kc*5))^(1/Cc)
            Kc*5*(h-b3)^Cc
        }
    }
    rcs[[rcname]] = list("f"=function(h) {
        return(sapply(h, f))
    },"lab"=rclab_en, "lab_fr"=rclab_fr)
    
    #####################################################################
    ### case 4: on rectangular weir with a 90° triangular weir within
    rcname = "rc4"
    rclab_fr = "Encoche triangulaire (Alpha=90°)"
    rclab_en = "Triangular notch (Alpha=90°)"
    f <- function(h){
        Kr = 0.4*sqrt(2*9.81); Kt = 0.31*sqrt(2*9.81); Kc = 25*sqrt(0.005)
        Cr = 1.5; Ct = 2.5; Cc = 1.67
        a = 90*pi/180
        if (h<=0.195){
            0
        }else if (h>0.195 & h<=1){
            Kt*tan(a/2)*(h-0.195)^Ct
        }else if (h>1 & h<=2){
            b2 = 1
            Kt*tan(a/2)*(h-0.195)^Ct + Kr*3.86*(h-b2)^Cr - Kt*tan(a/2)*(h-b2)^Ct
        }else if (h>2){
            b2 = 1
            b3 = 2-((Kt*tan(a/2)*(2-0.195)^Ct + Kr*3.86*(2-b2)^Cr - Kt*tan(a/2)*(2-b2)^Ct)/(Kc*5))^(1/Cc)
            Kc*5*(h-b3)^Cc
        }
    }
    rcs[[rcname]] = list("f"=function(h) {
        return(sapply(h, f))
    },"lab"=rclab_en, "lab_fr"=rclab_fr)
    
    #####################################################################
    ### case 5: on rectangular weir with a 120° triangular weir within
    rcname = "rc5"
    rclab_fr = "Encoche triangulaire (Alpha=120°)"
    rclab_en = "Triangular notch (Alpha=120°)"
    f <- function(h){
        Kr = 0.4*sqrt(2*9.81); Kt = 0.31*sqrt(2*9.81); Kc = 25*sqrt(0.005)
        Cr = 1.5; Ct = 2.5; Cc = 1.67
        a = 120*pi/180
        if (h<=0.353){
            0
        }else if (h>0.353 & h<=1){
            Kt*tan(a/2)*(h-0.353)^Ct
        }else if (h>1 & h<=2){
            b2 = 1
            Kt*tan(a/2)*(h-0.353)^Ct + Kr*3.88*(h-b2)^Cr - Kt*tan(a/2)*(h-b2)^Ct
        }else if (h>2){
            b2 = 1
            b3 = 2-((Kt*tan(a/2)*(2-0.353)^Ct + Kr*3.88*(2-b2)^Cr - Kt*tan(a/2)*(2-b2)^Ct)/(Kc*5))^(1/Cc)
            Kc*5*(h-b3)^Cc
        }
    }
    rcs[[rcname]] = list("f"=function(h) {
        return(sapply(h, f))
    },"lab"=rclab_en, "lab_fr"=rclab_fr)
    
    
    return(rcs)
}

# This functions is used to invert a rating curve equation. It returns a new
# function that can be used to compute the stage values from the streamflow
# values.
# It is not an exact solution, only an approximation based on a large
# table of values matching stage with streamflow. The user is required to
# enter a start and end stage values as well as the resolution (length) of 
# the approximation (the larger the number, the more precise).
hydraulicEquationsInverter <- function(eq, from=0, to=10, length=10000) {
    href=seq(from, to, length.out=length)
    Qref=eq(href)
    # suppress warnings which occure due to different values of 
    # stage giving 0 streamflow values
    return(function(Q) {
        return(suppressWarnings(approx(
            x=Qref, y=href,
            xout=Q, 
        )$y))
    })
}


# function that compute the AM30 from a daily streamflow time series
# returns a data.frame withe a first column "year" (factor) and a column 
# data containing the corresponding AM30 data
# the required packages for this function are: 
# * dplyr: install.packages("dplyr")
# * RcppRoll: install.packages("RcppRoll")
require(dplyr)
require(RcppRoll)
computeAM30 <- function(time, data) {
  AM30 <- function(x) {
    if (length(x) < 330) {
      return(NA)
    }
    return(min(RcppRoll::roll_mean(x, 30), na.rm = TRUE))
  }
  A <- data.frame(time=time, data=data)
  B <- dplyr::mutate(A, year=as.factor(format(time, format="%Y")))
  C <- dplyr::group_by(B, year)
  D <- dplyr::summarize(C, am30=AM30(data))
  return(as.data.frame(D))
}

# generate a matrix of non systematic errors given
# a standard deviation (std) and a matrix size (nr x nc)
generate_nonsyst_errors <- function(std, nr, nc) {
    err <- matrix(NA_real_, nr, nc)
    for (k in 1:nc) {
        err[, k] <- rnorm(nr, 0, std)
    }
    return(err)
}

# generate a matrix of systematic errors following the approach
# of Horner et al. 2018. It requires the following:
# * at: the indices of the rows where the systematic error must
#   be resampled (e.g. seq(1, nr, 30))
# * std: the standard deviation to use in the normal distribution
#   in which errors are sampled.
# * nr, nc: the number of rows and columns of the error matrix
#   to generate.
generate_syst_errors <- function(at, std, nr, nc) {
    err <- matrix(NA_real_, nr, nc)
    # initial errors for each column
    cur <- rnorm(nc, 0, std)
    for (k in 1:nr) {
        if (k%in%at) {
            # generate new errors for each column
            cur <- rnorm(nc, 0, std)
        }
        err[k, ] <- cur
    }
    return(err)
}
# get the resampling indices from a time vector and 
# a periodicity in days
get_resampling_indices_from_periodicity <- function(time, p) {
    days <- seq(from=time[1], to=time[length(time)], by="days")
    days <- days[seq(1, length(days), by=p)]
    indices <- c()
    for (k in seq_along(days)) {
        d <- abs(time - days[k])
        indices[k] <- which(d==min(d))[1]
    }
    return(indices)
}

