#####################################################################
# installing the needed packages
# remotes::install_github("IvanHeriver/tAgg")
# install.packages(c("dplyr", "RcppRoll"))

# loading the needed packages / functions
library(tAgg)
library(dplyr)
library(RcppRoll)
source("functions.R")

# importing an actual streamflow time series
Q <- read.table("Q_Craponne.txt", sep=";", header=TRUE, stringsAsFactors=FALSE)
Q$Time <- as.POSIXct(Q$Time, format="%Y-%m-%d %H:%M", tz="UTC")
summary(Q)

#####################################################################
# create the inverse function of rating curve #2 (Rectangular notch (L=3m)):
RC_h_to_Q <- hydraulicEquations()[[2]]$f
RC_Q_to_h <- hydraulicEquationsInverter(RC_h_to_Q, from=0, to=4, length=10000)

h <- RC_Q_to_h(Q$Streamflow_m3_per_s)
Q2 <- RC_h_to_Q(h)

head(Q$Streamflow_m3_per_s)
head(Q2)
head(h)

#####################################################################
# compute the streamflow time series at different time intervals + AM30:

Q_daily <- tAgg(Q$Time, Q$Streamflow_m3_per_s, by="days", na.action="propagate")
Q_monthly <- tAgg(Q$Time, Q$Streamflow_m3_per_s, by="months", na.action="propagate")
Q_yearly <- tAgg(Q$Time, Q$Streamflow_m3_per_s, by="years", na.action="propagate")
Q_am30 <- computeAM30(Q$Time, Q$Streamflow_m3_per_s)

#####################################################################
# have a look at the results
time_range <- as.POSIXct(
    c("2010-01-01", "2011-01-01"), 
    format="%Y-%m-%d", tz="UTC"
)
# instantaneous, daily and monthly values
plot(Q[Q$Time>time_range[1] & Q$Time<time_range[2], ],
    type="l", col="black", lwd=2, log="")
lines(Q_daily, type="s", col="green", lwd=2)
lines(Q_monthly, type="s", col="darkorange", lwd=2)

# daily, monthly, yearly time series and AM30
Q_am30$time <- as.POSIXct(
    paste0(as.character(Q_am30$year), "-01-01"),
    format="%Y-%m-%d", tz="UTC"
)
time_range <- as.POSIXct(
    c("2000-01-01", "2011-01-01"), 
    format="%Y-%m-%d", tz="UTC"
)
plot(Q_daily[Q_daily$time>time_range[1] & Q_daily$time<time_range[2], ], 
    type="l", col="black", lwd=2, log="y")
lines(Q_monthly, type="s", col="darkorange", lwd=2)
lines(Q_yearly, type="s", col="lightblue", lwd=2)
lines(Q_am30[, c("time", "am30")], type="s", col="violet", lwd=2)


#####################################################################
# creating stage errors

# let's consider stage time series "h" (see above)...
summary(h)
# ... to generate the non-systematic errors with 
# a standard deviation of 5mm (0.005m) with 200 samples ...
nonsyste_errors <- generate_nonsyst_errors(0.005, length(h), 200)
# ... and the systematic errors with the same 
# standard deviation and resampling periodicity of 30 days
# (every 30 days the errors are resampled) with 200 samples
at <- get_resampling_indices_from_periodicity(Q$Time, 30)
syst_errors <- generate_syst_errors(at, 0.005, length(h), 200)

# let's have a look at the first 100 values for a few randomly chosen samples:
plot(NA, xlim=c(1, 100), ylim=c(-0.01, 0.01), type="n")
lines(nonsyste_errors[1:100, runif(1, 1, 101)], col="red")
lines(nonsyste_errors[1:100, runif(1, 1, 101)], col="green")
lines(nonsyste_errors[1:100, runif(1, 1, 101)], col="green")

plot(NA, xlim=c(1, 1000), ylim=c(-0.01, 0.01), type="n")
lines(syst_errors[, runif(1, 1, 101)], col="red")
lines(syst_errors[, runif(1, 1, 101)], col="green")
lines(syst_errors[, runif(1, 1, 101)], col="blue")

# both error matrix can easily be combined (provided they
# have the same dimensions)
stage_errors <- syst_errors + nonsyste_errors

plot(NA, xlim=c(1, 100), ylim=c(-0.02, 0.02), type="n")
lines(stage_errors[, runif(1, 1, 101)], col="red")
lines(stage_errors[, runif(1, 1, 101)], col="green")
lines(stage_errors[, runif(1, 1, 101)], col="blue")

# stage errors can be added to the stage time series itself
h_with_errors <- h + stage_errors

time_range <- as.POSIXct( 
    c("2010-01-01", "2010-04-01"), 
    format="%Y-%m-%d", tz="UTC"
)
is_to_plot <- Q$Time>time_range[1] & Q$Time<time_range[2]
plot(Q$Time[is_to_plot], h[is_to_plot], type="l", col="black", lwd=2, log="")
abline(v=Q$Time[at])
lines(Q$Time, h_with_errors[, runif(1, 1, 101)], col="red")
lines(Q$Time, h_with_errors[, runif(1, 1, 101)], col="green")
lines(Q$Time, h_with_errors[, runif(1, 1, 101)], col="blue")

