import logging

from mysql.connector import connect, MySQLConnection
from semantic_version import Version
from tqdm import tqdm

from utils.core import Status
from utils.runner import Runner
from tld import is_tld


class DictDiffer(object):
    """
    Calculate the difference between two dictionaries as:
    (1) items added
    (2) items removed
    (3) keys same in both but changed values
    (4) keys same in both and unchanged values
    """
    def __init__(self, current_dict, past_dict):
        self.current_dict, self.past_dict = current_dict, past_dict
        self.set_current, self.set_past = set(current_dict.keys()), set(past_dict.keys())
        self.intersect = self.set_current.intersection(self.set_past)
    def added(self):
        return self.set_current - self.intersect
    def removed(self):
        return self.set_past - self.intersect
    def changed(self):
        return set(o for o in self.intersect if self.past_dict[o] != self.current_dict[o])
    def unchanged(self):
        return set(o for o in self.intersect if self.past_dict[o] == self.current_dict[o])


class VulnerabilityFinder(Runner):
    def __init__(self, vulnerability_db: MySQLConnection, updates_db: MySQLConnection, logger: logging.Logger, batch_size: int):
        self._vulnerabilities_conn = vulnerability_db
        self.batch_size = batch_size
        super().__init__(updates_db, logger)

    def _is_vulnerable(self, vendor, product, version, update_id):
        cpe_ids = []
        with self._vulnerabilities_conn.cursor(buffered=True) as cursor:
            cursor.execute("SELECT id, vendor, product FROM cpeEntry WHERE vendor LIKE %s AND product LIKE %s", [
                f'%{vendor}%', f'%{product}%'
            ])
            for (id, matched_vendor, matched_product) in cursor:
                logging.getLogger().info('Match with {} from (vendor: {}, product {})'.format(id, matched_vendor, matched_product))
                cpe_ids.append(id)
            if not cpe_ids:
                return {}
            cpe_ids = ', '.join([str(cpe_id) for cpe_id in cpe_ids])

            cursor.execute(f'SELECT cveid, versionEndExcluding, versionEndIncluding, versionStartExcluding, versionStartIncluding FROM software WHERE cpeEntryId IN ({cpe_ids})')
            current = Version.coerce(version)
            current.prerelease = ()
            result = {}
            for (cveid, end_ex, end_in, start_ex, start_in) in cursor:
                start = Version.coerce(start_ex) if start_ex else Version.coerce(start_in) if start_in else None
                end = Version.coerce(end_ex) if end_ex else Version.coerce(end_in) if end_in else None
                if end:
                    end.prerelease = ()
                start_inclusive = True if start_in else False
                end_inclusive = True if end_in else False
                if start:
                    start.prerelease = ()
                    if start_inclusive:
                        vulnerable = start <= current <= end if end_inclusive else start <= current < end
                    else:
                        vulnerable = start < current <= end if end_inclusive else start < current < end
                elif end:
                    vulnerable = current <= end if end_inclusive else current < end
                else:
                    vulnerable = True

                if vulnerable:
                    with self._vulnerabilities_conn.cursor(dictionary=True) as sub_cursor:
                        sub_cursor.execute(f'SELECT * FROM vulnerability WHERE id={cveid}')
                        vulnerability = sub_cursor.fetchone()
                    logging.getLogger().info('Found vulnerability {} in update with id: {}'.format(vulnerability['cve'], update_id))
                    logging.getLogger().info(
                        'Scan info for {} - {} (Start: {}, End: {}, Current: {})'.format(vendor, product, start, end,
                                                                                        current))
                    if 'id' in vulnerability:
                        del vulnerability['id']
                    result[vulnerability['cve']] = vulnerability
        return result

    def _write_cvss_score_to_db(self, update_id, vulnerability: dict):
        column_names = ', '.join(vulnerability.keys())
        placeholders = ', '.join(['%s'] * len(vulnerability))
        with self._connection.cursor(prepared=True) as cursor:
            cursor.execute(f"INSERT INTO cves ({column_names}) VALUES({placeholders}) ON DUPLICATE KEY UPDATE id=id;", list(vulnerability.values()))
        self._connection.commit()
        with self._connection.cursor(buffered=True) as cursor:
            cursor.execute('SELECT id FROM cves WHERE cve = %(cve)s', vulnerability)
            (vulnerability_id,) = cursor.fetchone()
            cursor.execute('UPDATE updates SET cvss_v2=%s, cvss_v3=%s, fixes_cve_id=%s WHERE id=%s', [vulnerability['v2Score'], vulnerability['v3BaseScore'], vulnerability_id, update_id])
        self._connection.commit()

    def _update_status(self, update_id, status: Status):
        with self._connection.cursor() as cursor:
            cursor.execute(f'UPDATE updates SET status="{status}" WHERE id=%s', [update_id])
        self._connection.commit()

    def run_once(self):
        with self._connection.cursor(buffered=True) as cursor:
            cursor.execute(f"""
                SELECT id, package, old_version, new_version
                FROM updates as u
                WHERE status = '{Status.NEW}' OR status = '{Status.CHANGED}' LIMIT {self.batch_size}
            """)
            updates = []
            for (id, package, old_version, new_version) in tqdm(cursor):
                vendor, product = package.split(':')
                vendor_name = vendor.split('.')[1] if is_tld(vendor.split('.')[0]) else vendor.split('.')[0]
                old_vulns = self._is_vulnerable(vendor_name, product, old_version, id)
                new_vulns = self._is_vulnerable(vendor_name, product, new_version, id)
                diff = DictDiffer(new_vulns, old_vulns)
                if diff.removed():
                    fixed_cve = list(diff.removed())[0]
                    if fixed_cve:
                        logging.getLogger().info('Found fixed vulnerability (CVE: {}) for update: {}'.format(fixed_cve, id))
                        vulnerability = old_vulns[fixed_cve]
                        updates.append((id, vulnerability))
                        self._write_cvss_score_to_db(id, vulnerability)
                self._update_status(id, Status.DONE)
        print('FOUND UPDATES:')
        print(updates)
        logging.getLogger().info('No more vulnerabilities to find repos for')


if __name__ == '__main__':
    logging.getLogger().setLevel(logging.INFO)
    logging.basicConfig(filename='vulnerabilities.log', level=logging.INFO)
    with connect(host='127.0.0.1', port=33068, user='dc', database='dependencycheck', password='dc') as vuln_db:
        with connect(host='127.0.0.1', port=33062, user='vulnerability-history', database='vulnerability-history',
                     password='secret') as updates_db:
            vulnerability_finder = VulnerabilityFinder(vuln_db, updates_db, logging.getLogger(), 50)
            vulnerability_finder.run_once()
