import logging
import os
import shutil

from pydriller import RepositoryMining
from git import Git, Repo
from tqdm import tqdm

from config import CLONE_DIR, BASE_DIR, OUT_DIR, get_out_dir_for
from utils.core import Update
from utils.parse_dependencies import determine_list_of_versions, get_maven_dependencies


def ensure_repo_dir():
    if not os.path.exists(CLONE_DIR):
        os.mkdir(CLONE_DIR)


def ensure_out_dir(project_name=None):
    if not os.path.exists(OUT_DIR):
        os.mkdir(OUT_DIR)
    if project_name and not os.path.exists(get_out_dir_for(project_name)):
        os.mkdir(get_out_dir_for(project_name))


def rm_r(path):
    if os.path.isdir(path) and not os.path.islink(path):
        shutil.rmtree(path)
    elif os.path.exists(path):
        os.remove(path)


def delete_repository(project_full_name):
    repo_path = CLONE_DIR + '/' + project_full_name[project_full_name.index('/'):]
    rm_r(repo_path)


def clone_repository(project_full_name):
    if not os.path.exists(CLONE_DIR + '/'):
        os.mkdir(CLONE_DIR)
    repo_path = CLONE_DIR + '/' + project_full_name[project_full_name.index('/'):]
    if os.path.exists(repo_path):
        rm_r(repo_path)
    Git(CLONE_DIR).clone('git://github.com/{}.git'.format(project_full_name))
    return Repo(repo_path)


def get_pom_commits(git_dir, pom_path):
    return list(RepositoryMining(git_dir, filepath=pom_path).traverse_commits())


def find_fix_commit(repo: Repo, vulnerable_package_name, cve, pom_path='pom.xml'):
    print(repo)
    repo.git.bisect('start', '--term-old', 'vulnerable', '--term-new', 'patched', '--', pom_path)
    repo.git.bisect('patched')
    commits = get_pom_commits(repo.working_dir, pom_path)
    repo.git.bisect('vulnerable', commits[0].hash)
    repo.git.bisect('run', BASE_DIR + '/checkdependencies/check_version_for_vulnerabilities.sh', BASE_DIR, vulnerable_package_name, cve)
    update_commit_hash = repo.git.rev_parse('HEAD')
    fix_commit = [commit for commit in commits if commit.hash == update_commit_hash][0]
    print(fix_commit.hash)
    return fix_commit


def traverse_pom_history(repo: Repo, package_coords, fixed_date):
    repo_name = repo.working_dir.split('/')[-1]
    print('running for ', repo_name)
    ensure_out_dir(repo_name)
    pom_commits = get_pom_commits(repo.working_dir, 'pom.xml')
    dependency_versions = {}
    delays = []
    version_cache = {}
    fix_delay = None
    logger = logging.getLogger(__name__)
    for commit in tqdm(pom_commits):
        repo.git.checkout('-f', commit.hash)
        dependencies = get_maven_dependencies(repo_name, commit.hash)
        for dependency in dependencies:
            key = '{}:{}'.format(dependency['group'], dependency['artifact'])
            if key not in version_cache:
                version_cache[key] = determine_list_of_versions(dependency)
            if key not in version_cache or dependency['version'] not in version_cache[key]:
                logger.info('skipping {}'.format(dependency))
                continue
            release_date = version_cache[key][dependency['version']]
            if key not in dependency_versions:
                dependency_versions[key] = (dependency['version'], release_date)
            try:
                previous_version, previous_release_date = dependency_versions[key]
                is_update = dependency['version'] != previous_version and release_date > previous_release_date
            except TypeError:
                logger.error('Date comparison failure between new: {}, old: {}'.format(
                    release_date,
                    dependency_versions[key]
                ))
                continue
            if is_update:
                print('found update')
                patch_delay = commit.committer_date - version_cache[key][dependency['version']]
                is_fix = key == package_coords and release_date >= fixed_date and fix_delay is None
                update = Update(
                    package=key,
                    delay=patch_delay,
                    commit=commit,
                    is_fix_update=is_fix,
                    old_version=previous_version,
                    old_release_date=previous_release_date,
                    new_version=dependency['version'],
                    new_release_date=release_date,
                )
                if patch_delay.days > 0:
                    logger.info(update)
                    delays.append(update)
                    dependency_versions[key] = (dependency['version'], release_date)
                    if is_fix:
                        fix_delay = update
    return delays, fix_delay
