import logging
import os
import re
import shutil
import subprocess
from datetime import datetime

import requests
from semantic_version import Version

from config import get_repo_dir_for, get_out_dir_for


def clean_deps_directory(deps_dir_path):
    if os.path.isfile(deps_dir_path) or os.path.isdir(deps_dir_path):
        shutil.rmtree(deps_dir_path)
    os.mkdir(deps_dir_path)


def get_maven_dependencies(project_name, commit_hash):
    clean_deps_directory(get_out_dir_for(project_name, 'deps'))
    raw_output = ''
    try:
        raw_output = subprocess.check_output(['mvn', 'dependency:tree'], cwd=get_repo_dir_for(project_name)) \
            .decode('utf-8')
    except subprocess.CalledProcessError as e:
        logging.getLogger().warning('Some problems occurred with: {}'.format(e.cmd))
        logging.getLogger().warning('Return code: {}'.format(e.returncode))
        with open(get_out_dir_for(project_name, 'error.{}.log'.format(commit_hash)), 'w') as error_file:
            error_file.write(e.output.decode('utf-8'))
            logging.getLogger().warning('Check error.log for details')
        raw_output = e.output.decode('utf-8')
    if not re.search('\[INFO\].*SUCCESS', raw_output):
        return []
    output = []
    with open(get_out_dir_for(project_name, 'deps/mvn-dep-tree-output.txt'), 'w') as mvn_dep_tree_output:
        for line in raw_output.split('\n'):
            if '[INFO]' in line or '[WARNING]' in line:
                mvn_dep_tree_output.write(line)
                mvn_dep_tree_output.write('\n')
                output.append(line)
            if 'Total time' in line:
                break
    dependency_tree = _build_dependency_tree_from_maven_output(project_name, output)
    unique_dependencies = _get_unique_dependencies(project_name, dependency_tree)

    return unique_dependencies


misleading_gavs = """
commons-beanutils:commons-beanutils:dev
commons-beanutils:commons-beanutils:200
commons-io:commons-io:200
commons-lang:commons-lang:2003
com.google.guava:guava:r0
"""

misleading_gav_list = misleading_gavs.split("\n")


def misleading(group, artifact, version):
    gav = ":".join([group, artifact, version])
    for misleading_gav in misleading_gav_list:
        if misleading_gav != "" and misleading_gav in gav:
            return True
    return False


def determine_list_of_versions(dependency):
    if type(dependency) == str:
        dependency = dict(zip(['group', 'artifact'], dependency.split(':')))
    list_of_versions = {}
    group_dir = dependency['group'].replace('.', '/')
    logging.getLogger().info('Requesting maven information for: {}'.format(dependency))
    r = requests.get('https://repo1.maven.org/maven2/' + group_dir + '/' + dependency['artifact'])
    req_lines = r.text.split('\n')
    req_lines = list(map(lambda x: re.sub('\-         \-', 'unknown-date unknown:time', x), req_lines))
    req_lines = list(map(lambda x: re.sub('\W+-\W+$', '', x), req_lines))
    req_lines = list(filter(lambda x: 'href' in x and not 'maven-metadata' in x and not '>../</' in x, req_lines))
    for line2 in req_lines:
        p = re.findall('.*>(.*)/</a>(.*)', line2)
        if len(p) == 0:
            continue
        p = list(p[0])
        p = list(map(lambda x: x.strip(), p))
        if not misleading(dependency['group'], dependency['artifact'], p[0]):
            try:
                date = datetime.strptime(p[1] + ' Z', '%Y-%m-%d %H:%M %z')
            except ValueError:
                logging.getLogger().info('Could not parse date for {} {}'.format(p[0], p[1]))
                continue
            list_of_versions[p[0]] = date
    return list_of_versions


def _build_dependency_tree_from_maven_output(project_name, maven_output):
    dependency_tree = []
    with open(get_out_dir_for(project_name, 'deps/dependencies-tree.txt'), 'w') as f:
        for line in maven_output:
            if 'Reactor Summary:' in line:
                break
            s = line.replace('|', ' ').replace('+-', '  ').replace('\\-', '  ').replace('[INFO]', '')
            if '---' not in s and '--<' not in s and not s.startswith(' Building ') and not s.endswith(
                    '[jar]') and not s.endswith('[pom]'):
                if len(re.findall(':', s)) == 4:
                    if re.search(':$', s) is None:
                        if 'Total time' not in s and '[WARNING]' not in s:
                            s = s.replace('   ', ' ')
                            s = re.sub('^ ', '', s)
                            if '::' not in s:
                                s = re.sub(r'maven-dependency-plugin:[0-9.]*:tree \(default-cli\) @ ', '', s)
                                f.write(s)
                                f.write('\n')
                                dependency_tree.append(s)
    return dependency_tree


def validate_version(version_string, group, artifact):
    """Validates a version string againt the SemVer specification."""
    try:
        Version.parse(version_string, partial=True)
        return True
    except ValueError:
        logging.getLogger().warning(
            'Could not parse version ({}) for package {}:{}'.format(version_string, group, artifact))
        return False


def _get_unique_dependencies(project_name, dependency_tree):
    flattened_unique_gavs = [':'.join(s.replace(' ', '').split(':')[:4]) for s in dependency_tree if
                             ':' in s and re.search('-+<.*>-+', s) is None]
    flattened_unique_gavs = sorted(list(set(flattened_unique_gavs)))

    with open(get_out_dir_for(project_name, 'deps/flattened-unique-gavs.txt'), 'w') as f:
        f.write('\n'.join(flattened_unique_gavs))

    unique_dependencies = [
        {'group': group, 'artifact': artifact, 'type': typ, 'version': version}
        for group, artifact, typ, version in (gav.split(':') for gav in flattened_unique_gavs)
        # if validate_version(version, group, artifact)
    ]
    #  'version': Version(version, partial=True)
    return unique_dependencies
